#ifndef _KVI_IRC_USERLIST_H_INCLUDED_
#define _KVI_IRC_USERLIST_H_INCLUDED_

// =============================================================================
//
//      --- kvi_irc_userlist.h ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

class KviStr;
class KviIrcUser;

/**
 * Shared list of users with reference count.<br>
 * Small memory usage and relatively fast access.<br>
 * This is the GLOBAL LIST and should be UNIQUE (PER CONNECTION).<br>
 */
typedef struct _KviIrcUserListNode
{
	_KviIrcUserListNode *prev;
	KviIrcUser          *pUser;
	int                  nRefs;
	_KviIrcUserListNode *next;
} KviIrcUserListNode;

class KviIrcUserList
{
	friend class KviIrcUserChanList;
public:
	KviIrcUserList();
	~KviIrcUserList();
private:
	KviIrcUserListNode *m_pHead;
	KviIrcUserListNode *m_pTail;
	KviIrcUserListNode *m_pCur; // This is defined ONLY after a call to first(), firstNode(), next(),...
protected:
	void insertNode(KviIrcUserListNode *node);
	void removeNode(KviIrcUserListNode *node); // Deletes the node!
	KviIrcUserListNode *findNode(const KviIrcUser &user);
	KviIrcUserListNode *findNode(const char *nick);

	/**
	 * Add a user to the list.<br>
	 * Making a deep copy of data if the user is not already there or increasing
	 * the reference count and updating the host/username data (if possible and necessary).<br>
	 * Returns the pointer to the internal KviIrcUser user class (that should never be changed!).<br>
	 * You can NOT add directly a user here, you must do it through the client class.
	 */
	KviIrcUserListNode *addUser(const KviIrcUser &user);
	void killUserByNode(KviIrcUserListNode *node);
public:
	void clearList();
	// CAUTION!!
	// firstNode(), lastNode(), first() or next() MUST be called before
	// any iteration attempt with nextNode(), next(), prevNode() and prev().
	KviIrcUserListNode *firstNode();
	KviIrcUserListNode *lastNode();
	KviIrcUserListNode *nextNode();
	KviIrcUserListNode *prevNode();

	KviIrcUser *first();
	KviIrcUser *last();
	KviIrcUser *next();
	KviIrcUser *prev();

	void getFirstFiveNicksToUpdate(KviStr &buffer);
	/**
	 * Updates the mask of the user (assuming he is in the list!).<br>
	 * Returns true if the user was in the list.
	 */
	bool updateUser(const KviIrcUser &user);
	KviIrcUser *findUser(const KviIrcUser &user);
	KviIrcUser *findUser(const char *nick);
};

#endif // _KVI_IRC_USERLIST_H_INCLUDED_
