// =============================================================================
//
//      --- kvi_script_textedit.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Robin Verduijn <robin@debian.org>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptTextEdit"

#include <qtextedit.h>

#include "kvi_error.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_textedit.h"

/*
	@class: textedit
	@short:
		A text editor widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:

		!fn: $text()
		Returns the text currently displayed.<br>

		!fn: $setText(&lt;text&gt;)
		Sets the text to be edited.<br>

		!fn: $setReadOnly(&lt;bReadOnly&gt;)
		Sets this textedit to read only mode or read/write mode.<br>
		if &lt;bReadOnly&gt; is '1' then the read-only mode is set,
		if it is '0' then the read/write mode is set.<br>

		!fn: $readOnly()
		Returns '1' if this widget is in read only mode, '0' otherwise.<br>

		!fn: $setCursorPosition(&lt;paragraph&gt;,&lt;index&gt;)
		Moves the cursor to the specified position

		!fn: $cursorPosition()
		Returns the current cursor position in the form "paragraph, index"

	@description:

	@examples:

	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		class [class]lineedit[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * TEXTEDIT class
 */
void KviScriptTextEdit::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",           (scriptObjectFunction) &KviScriptTextEdit::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",              (scriptObjectFunction) &KviScriptTextEdit::builtinFunction_TEXT);
	d->addBuiltinFunction("readOnly",          (scriptObjectFunction) &KviScriptTextEdit::builtinFunction_READONLY);
	d->addBuiltinFunction("setReadOnly",       (scriptObjectFunction) &KviScriptTextEdit::builtinFunction_SETREADONLY);
	d->addBuiltinFunction("cursorPosition",    (scriptObjectFunction) &KviScriptTextEdit::builtinFunction_CURSORPOSITION);
	d->addBuiltinFunction("setCursorPosition", (scriptObjectFunction) &KviScriptTextEdit::builtinFunction_SETCURSORPOSITION);
}

KviScriptTextEdit::KviScriptTextEdit(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptTextEdit::~KviScriptTextEdit()
{
	// Nothing here
}

bool KviScriptTextEdit::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new QTextEdit(((KviScriptWidget *) parent())->m_pWidget, name());
		}
	}
	if( !m_pWidget )
		m_pWidget = new QTextEdit(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			((QTextEdit *) m_pWidget)->setText(pS->ptr());
	}
	return true;
}

int KviScriptTextEdit::builtinFunction_TEXT(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(((QTextEdit *) m_pWidget)->text());
	return KVI_ERROR_Success;
}

int KviScriptTextEdit::builtinFunction_CURSORPOSITION(QPtrList<KviStr> *, KviStr &buffer)
{
	int para, index;
	((QTextEdit *) m_pWidget)->getCursorPosition(&para, &index);
	buffer.append(KviStr::Format, "%d,%d", para, index);
	return KVI_ERROR_Success;
}

int KviScriptTextEdit::builtinFunction_SETCURSORPOSITION(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int para = pS->toInt(&bOk);
			if( !bOk )
				return KVI_ERROR_InvalidParameter;
			pS = params->next();
			if( pS ) {
				int index = pS->toInt(&bOk);
				if( !bOk )
					return KVI_ERROR_InvalidParameter;
				((QTextEdit *) m_pWidget)->setCursorPosition(para, index);
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTextEdit::builtinFunction_SETTEXT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((QTextEdit *) m_pWidget)->setText(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	((QTextEdit *) m_pWidget)->setText("");
	return KVI_ERROR_Success;
}

int KviScriptTextEdit::builtinFunction_READONLY(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(((QTextEdit *) m_pWidget)->isReadOnly() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptTextEdit::builtinFunction_SETREADONLY(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((QTextEdit *) m_pWidget)->setReadOnly(kvi_strEqualCI(pS->ptr(), "1"));
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

#include "m_kvi_script_textedit.moc"
