/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT6MPreferences.h"
#include "JT6MFrame.h"
#include "JT6MApp.h"

#include "jt6m/JT6MDefs.h"

#include <wx/tglbtn.h>

const int BORDER_SIZE = 5;

enum {
	Menu_File_Open = 100,
	Menu_File_Saving,
	Menu_File_Delete,
	Menu_File_Quit,
	Menu_Edit_Preferences,
	Menu_Help_About
};

BEGIN_DECLARE_EVENT_TYPES()
	DECLARE_EVENT_TYPE(CARD_LEVELS,   2651)
	DECLARE_EVENT_TYPE(FILE_LEVELS,   2652)
	DECLARE_EVENT_TYPE(CARD_MESSAGE,  2653)
	DECLARE_EVENT_TYPE(FILE_MESSAGE,  2654)
        DECLARE_EVENT_TYPE(ERROR_EVENT,   2657)
END_DECLARE_EVENT_TYPES()

DEFINE_EVENT_TYPE(CARD_LEVELS)
DEFINE_EVENT_TYPE(FILE_LEVELS)
DEFINE_EVENT_TYPE(CARD_MESSAGE)
DEFINE_EVENT_TYPE(FILE_MESSAGE)
DEFINE_EVENT_TYPE(ERROR_EVENT)

BEGIN_EVENT_TABLE(CJT6MFrame, wxFrame)
	EVT_MENU(Menu_File_Open, CJT6MFrame::onOpen)
	EVT_MENU(Menu_File_Saving, CJT6MFrame::onSaving)
	EVT_MENU(Menu_File_Delete, CJT6MFrame::onDelete)
	EVT_MENU(Menu_File_Quit, CJT6MFrame::onQuit)
	EVT_MENU(Menu_Edit_Preferences, CJT6MFrame::onPreferences)
	EVT_MENU(Menu_Help_About, CJT6MFrame::onAbout)

	EVT_TOGGLEBUTTON(Button_Input_Record, CJT6MFrame::onRecord)
	EVT_CHOICE(Choice_TX_Order, CJT6MFrame::onTXOrder)
	EVT_CHECKBOX(Button_TX_Enable, CJT6MFrame::onTXEnable)
	EVT_BUTTON(Button_Remote, CJT6MFrame::onUpdate)
	EVT_SPINCTRL(Spin_Minimum_Length, CJT6MFrame::onMinLength)

	EVT_TEXT(Text_Messages_Begin + 0, CJT6MFrame::onText)
	EVT_TEXT(Text_Messages_Begin + 1, CJT6MFrame::onText)
	EVT_TEXT(Text_Messages_Begin + 2, CJT6MFrame::onText)
	EVT_TEXT(Text_Messages_Begin + 3, CJT6MFrame::onText)
	EVT_TEXT(Text_Messages_Begin + 4, CJT6MFrame::onText)
	EVT_TEXT(Text_Messages_Begin + 5, CJT6MFrame::onText)
	EVT_TEXT(Text_Messages_Begin + 6, CJT6MFrame::onText)
	EVT_TEXT(Text_Messages_Begin + 7, CJT6MFrame::onText)
	EVT_TEXT(Text_Messages_Begin + 8, CJT6MFrame::onText)
	EVT_TEXT(Text_Messages_Begin + 9, CJT6MFrame::onText)

	EVT_RADIOBUTTON(Button_Messages_Begin + 0, CJT6MFrame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 1, CJT6MFrame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 2, CJT6MFrame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 3, CJT6MFrame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 4, CJT6MFrame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 5, CJT6MFrame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 6, CJT6MFrame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 7, CJT6MFrame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 8, CJT6MFrame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 9, CJT6MFrame::onMessage)

	EVT_CUSTOM(CARD_LEVELS,   -1, CJT6MFrame::onCardLevels)
	EVT_CUSTOM(FILE_LEVELS,   -1, CJT6MFrame::onFileLevels)
	EVT_CUSTOM(CARD_MESSAGE,  -1, CJT6MFrame::onCardMessage)
	EVT_CUSTOM(FILE_MESSAGE,  -1, CJT6MFrame::onFileMessage)
	EVT_CUSTOM(ERROR_EVENT,   -1, CJT6MFrame::onError)
END_EVENT_TABLE()

CJT6MFrame::CJT6MFrame() :
wxFrame(NULL, -1, title),
m_cardDisplay(NULL),
m_fileDisplay(NULL),
m_record(NULL),
m_transmit(NULL),
m_remote(NULL),
m_clock(NULL),
m_settings(NULL),
m_message(NULL),
m_messageNo(-1)
{
	SetMenuBar(createMenuBar());

	CreateStatusBar(2);

	// The top level sizer, graph, list box and then controls
	wxBoxSizer* mainSizer = new wxBoxSizer(wxVERTICAL);

	// Spectrum graphs
	wxNotebook* noteBook = new wxNotebook(this, -1);
	wxNotebookSizer* noteSizer = new wxNotebookSizer(noteBook);

	m_cardDisplay = new CJT6MDisplay(noteBook);
	noteBook->AddPage(m_cardDisplay, wxT("Sound Card"), true);

	m_fileDisplay = new CJT6MDisplay(noteBook);
	noteBook->AddPage(m_fileDisplay, wxT("WAV Files"), false);

	mainSizer->Add(noteSizer, 0, wxBOTTOM, BORDER_SIZE);

	// The three input sections, left two are control, right is messages
	wxBoxSizer* inputSizer = new wxBoxSizer(wxHORIZONTAL);

	// Handle the recording and transmit control boxes
	wxBoxSizer* leftSizer = new wxBoxSizer(wxVERTICAL);

	m_record = new CRecordControl(this);
	leftSizer->Add(m_record);

	bool txFirst = ::wxGetApp().getTXFirst();
	bool txEnable = ::wxGetApp().getTXEnable();
	m_transmit = new CTransmitControl(this, -1, txFirst, txEnable);
	leftSizer->Add(m_transmit);

	int minLength = ::wxGetApp().getMinLength();
	m_cardDisplay->setMinLength(minLength);
	m_fileDisplay->setMinLength(minLength);
	m_settings = new CJT6MSettingsControl(this, -1, minLength);
	leftSizer->Add(m_settings);

	inputSizer->Add(leftSizer);

	wxBoxSizer* middleSizer = new wxBoxSizer(wxVERTICAL);

	m_remote = new CRemoteControl(this);
	middleSizer->Add(m_remote);

	m_clock = new CClockPanel(this, -1, wxDefaultPosition, wxSize(145, 80));
	middleSizer->Add(m_clock);

	inputSizer->Add(middleSizer);

	wxString* messages = new wxString[JT6M_MAX_MESSAGES];
	for (int i = 0; i < JT6M_MAX_MESSAGES; i++)
		messages[i] = ::wxGetApp().getReplacedMessage(i);
	m_message = new CMessageControl(this, -1, messages, JT6M_MAX_MESSAGES, JT6M_MAX_MESSAGE_LENGTH);
	inputSizer->Add(m_message);
	delete[] messages;

	m_messageNo = 0;
	wxString message = m_message->getMessage(m_messageNo);
	::wxGetApp().sendMessage(message);

	mainSizer->Add(inputSizer);

	SetAutoLayout(true);

	mainSizer->Fit(this);
	mainSizer->SetSizeHints(this);

	SetSizer(mainSizer);
}

CJT6MFrame::~CJT6MFrame()
{
}

wxMenuBar* CJT6MFrame::createMenuBar()
{
	wxMenu* fileMenu = new wxMenu();
	fileMenu->Append(Menu_File_Open, wxT("&Open sound file\tCtrl-O"));
	fileMenu->AppendSeparator();
	fileMenu->Append(Menu_File_Saving, wxT("&Save received text\tCtrl-A"), wxEmptyString, true);
	fileMenu->Append(Menu_File_Delete, wxT("&Remove received text\tCtrl-R"));
	fileMenu->AppendSeparator();
	fileMenu->Append(Menu_File_Quit, wxT("E&xit\tAlt-X"));

	wxMenu* editMenu = new wxMenu();
	editMenu->Append(Menu_Edit_Preferences, wxT("&Preferences...\tCtrl-P"));

	wxMenu* helpMenu = new wxMenu();
	helpMenu->Append(Menu_Help_About, wxT("&About JT6M\tF1"));

	wxMenuBar* menuBar = new wxMenuBar();
	menuBar->Append(fileMenu, wxT("&File"));
	menuBar->Append(editMenu, wxT("&Edit"));
	menuBar->Append(helpMenu, wxT("&Help"));

	return menuBar;
}

void CJT6MFrame::showCardLevels(CJT6MLevels* levels)
{
	wxASSERT(levels != NULL);

	CJT6MLevelEvent event(levels, CARD_LEVELS);

	AddPendingEvent(event);
}

void CJT6MFrame::showFileLevels(CJT6MLevels* levels)
{
	wxASSERT(levels != NULL);

	CJT6MLevelEvent event(levels, FILE_LEVELS);

	AddPendingEvent(event);
}

void CJT6MFrame::showCardMessage(CJT6MMessage* message)
{
	wxASSERT(message != NULL);

	CJT6MMessageEvent event(message, CARD_MESSAGE);

	AddPendingEvent(event);
}

void CJT6MFrame::showFileMessage(CJT6MMessage* message)
{
	wxASSERT(message != NULL);

	CJT6MMessageEvent event(message, FILE_MESSAGE);

	AddPendingEvent(event);
}

void CJT6MFrame::error(const wxString& text)
{
	CErrorEvent event(text, ERROR_EVENT);

	AddPendingEvent(event);
}

void CJT6MFrame::onOpen(const wxCommandEvent& event)
{
	static wxString s_dir;
	static wxString s_file;

	wxFileDialog dialog(this, wxT("Open sound file"), s_dir, s_file, wxT("WAV files (*.wav;*.WAV)|*.wav;*.WAV"));

	if (dialog.ShowModal() != wxID_OK)
		return;

	s_dir  = dialog.GetDirectory();
	s_file = dialog.GetFilename();
	wxString fileName = dialog.GetPath();

	wxGetApp().readFile(fileName);
}

void CJT6MFrame::onSaving(const wxCommandEvent& event)
{
	bool saving = event.IsChecked();

	::wxGetApp().setSaving(saving);
}

void CJT6MFrame::onDelete(const wxCommandEvent& event)
{
	::wxGetApp().deleteMessages();
}

void CJT6MFrame::onQuit(const wxCommandEvent& event)
{
	Close(true);
}

void CJT6MFrame::onPreferences(const wxCommandEvent& event)
{
	CJT6MPreferences dialog(this);

	if (dialog.ShowModal() != wxID_OK)
		return;

	for (int i = 0; i < JT6M_MAX_MESSAGES; i++)
		m_message->setMessage(i, ::wxGetApp().getReplacedMessage(i));
}

void CJT6MFrame::onAbout(const wxCommandEvent& event)
{
	::wxMessageBox(title + wxT("\n\nWritten by Jonathan Naylor  HB9DRD\n\nLots of advice from Tomi Manninen  OH2BNS"), wxT("About JT6M"), wxICON_INFORMATION);
}

void CJT6MFrame::onRecord(const wxCommandEvent& event)
{
	bool recording = event.IsChecked();

	::wxGetApp().setRecording(recording);
}

void CJT6MFrame::onTXOrder(const wxCommandEvent& event)
{
	bool txFirst = int(event.GetClientData()) == Transmit_First;

	::wxGetApp().setTXFirst(txFirst);
}

void CJT6MFrame::onTXEnable(const wxCommandEvent& event)
{
	bool txEnabled = event.IsChecked();

	::wxGetApp().setTXEnable(txEnabled);
}

void CJT6MFrame::onUpdate(const wxCommandEvent& event)
{
        wxASSERT(m_remote != NULL);
	wxASSERT(m_message != NULL);

	wxString callsign = m_remote->getCallsign();
	wxString report   = m_remote->getReport();

	::wxGetApp().setRemote(callsign, report);

	for (int i = 0; i < JT6M_MAX_MESSAGES; i++)
		m_message->setMessage(i, ::wxGetApp().getReplacedMessage(i));
}

void CJT6MFrame::onMinLength(const wxSpinEvent& event)
{
	wxASSERT(m_cardDisplay != NULL);
	wxASSERT(m_fileDisplay != NULL);

	int length = event.GetPosition();

	::wxGetApp().setMinLength(length);

	m_cardDisplay->setMinLength(length);
	m_fileDisplay->setMinLength(length);
}

void CJT6MFrame::onText(const wxCommandEvent& event)
{
	int messageNo = event.GetId() - Text_Messages_Begin;

	wxASSERT(messageNo >= 0 && messageNo < JT6M_MAX_MESSAGES);

	if (messageNo == m_messageNo) {
		wxString message = m_message->getMessage(m_messageNo);
		::wxGetApp().sendMessage(message);
	}
}

void CJT6MFrame::onMessage(const wxCommandEvent& event)
{
	m_messageNo = event.GetId() - Button_Messages_Begin;

	wxASSERT(m_messageNo >= 0 && m_messageNo < JT6M_MAX_MESSAGES);

	wxString message = m_message->getMessage(m_messageNo);

	::wxGetApp().sendMessage(message);
}

void CJT6MFrame::onCardLevels(const CJT6MLevelEvent& event)
{
	wxASSERT(m_cardDisplay != NULL);

	CJT6MLevels* levels = event.getLevels();

	int level = levels->getLevel();

	wxString text;
	switch (level) {
		case Audio_High:
			text = wxT("Sound card level is too high");
			break;
		case Audio_OK:
			text = wxT("Sound card level is OK");
			break;
		case Audio_Low:
			text = wxT("Sound card level is too low");
			break;
	}

	SetStatusText(text, 0);

	m_cardDisplay->showLevels(levels);

	delete levels;
}

void CJT6MFrame::onFileLevels(const CJT6MLevelEvent& event)
{
	wxASSERT(m_fileDisplay != NULL);

	CJT6MLevels* levels = event.getLevels();

	int level = levels->getLevel();

	wxString text;
	switch (level) {
		case Audio_High:
			text = wxT("WAV file audio level is too high");
			break;
		case Audio_OK:
			text = wxT("WAV file audio level is OK");
			break;
		case Audio_Low:
			text = wxT("WAV file audio level is too low");
			break;
	}

	SetStatusText(text, 1);

	m_fileDisplay->showLevels(levels);

	delete levels;
}

void CJT6MFrame::onCardMessage(const CJT6MMessageEvent& event)
{
	wxASSERT(m_cardDisplay != NULL);

	CJT6MMessage* message = event.getMessage();

	m_cardDisplay->showMessage(message);
}

void CJT6MFrame::onFileMessage(const CJT6MMessageEvent& event)
{
	wxASSERT(m_fileDisplay != NULL);

	CJT6MMessage* message = event.getMessage();

	m_fileDisplay->showMessage(message);
}

void CJT6MFrame::onError(const CErrorEvent& event)
{
	wxString text = event.getText();

	wxMessageDialog dialog(this, text, wxT("JT6M Error"), wxICON_ERROR);

	dialog.ShowModal();
}
