package Lire::Config::ReferenceSpec;

use strict;

use Lire::Config::Reference;

=pod

=head1 NAME

Lire::Config::ReferenceSpec - SelectSpec which takes its options from an Index.

=head1 DESCRIPTION

This Lire::Config::TypeSpec is a SelectSpec which defines an option
for each available entry in the Index named on its index property..

=head2 new( 'name' => $name, 'index' => $index, ... )

Creates a new ReferenceSpec object. The available options will be
queried at run time under Index.

=cut

use base qw/ Lire::Config::SelectSpec /;

use Carp;

use Lire::Config::Index;
use Lire::Utils qw/check_param/;

sub new {
    my $self = shift->SUPER::new( @_ );

    my %args = @_;

    $self->index( $args{'index'} );


    return $self;
}

sub instance_class {
    return 'Lire::Config::Reference';
}

=pod

=head2 index( [ $new_index )

Returns (and optionnally change) the setting of the index
attributes.

=cut

sub index {
    my ($self, $index) = @_;

    if ( @_ == 2 ) {
        check_param( $index, 'index' );
        $self->{'_index'} = $_[1];
    }

    return $self->{'_index'};
}

sub add {
    croak( 'Lire::Config::ReferenceSpec cannot contain any components' );
}

sub normalize {
    my ( $self, $val ) = @_;

    return undef unless defined $val;
    return undef unless Lire::Config::Index->has_index( $self->{'_index'} );

    return Lire::Config::Index->get_index( $self->{'_index'} )->has_entry( $val ) ? $val : undef;
}

sub options {
    my $self = $_[0];

    return () unless Lire::Config::Index->has_index( $self->{'_index'} );

    my $index = Lire::Config::Index->get_index( $self->{'_index'} );
    my @options = ();
    foreach my $name ( @{$index->entries()} ) {
        push @options,
          Lire::Config::OptionSpec->new( 'name' => $name );
    }
    return @options;
}

sub as_value {
    my ( $self, $name ) = @_;

    return undef unless defined $name;
    return undef unless Lire::Config::Index->has_index( $self->{'_index'} );

    my $index = Lire::Config::Index->get_index( $self->{'_index'} );
    return undef unless $index->has_entry( $name );

    return $index->get_ref( $name );;
}

1;

__END__

=pod

=head1 SEE ALLO

 Lire::Config::ReportConfigIndex(3pm) Lire::Config::Index(3pm)

=head1 VERSION

$Id: ReferenceSpec.pm,v 1.1 2004/07/10 23:39:53 flacoste Exp $

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2004  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
