package Lire::ReportJob;

use strict;

use Carp;

use Lire::DlfSchema;
use Lire::Utils qw/ check_param check_object_param /;

=pod

=head1 NAME

Lire::ReportJob - Object used to configure periodical report generation

=head1 SYNOPSIS

  use Lire::ReportJob;
  use Lire::ReportSchedule;

  my $store = Lire::DlfStore->open( 'aStore' );
  my $job = new Lire::ReportJob( "webServer" );
  my $cfg = $store->get_report_config( 'my_report' );
  $job->add_schedule( new Lire::ReportSchedule( 'daily', $cfg );
  $job->run( $store, 'daily' );

=head1 DESCRIPTION

The Lire::ReportJob object is used to configure and execute report
generation jobs. These objects are usually configured in a store
through the lire(1) command.

=head2 new( $name, );

Create a new Lire::ReportJob(). The constructor takes one parameter
which is an identifier.


=cut

sub new {
    my ( $class, $name ) = @_;

    check_param( $name, 'name', qr/^[\w-]+$/ );
    my $self = bless { '_name' => $name,
                       '_schedules' => [],
                     }, $class;

    return $self;
}

=pod

=head2 name()

Returns the name of this ReportJob.

=cut

sub name { $_[0]{'_name'} };


=pod

=head2 add_schedule( $schedule, ... )

Adds one or more Lire::ReportSchedule to the ReportJob objects.

=cut

sub add_schedule {
    my ( $self, @schedules ) = @_;

    croak "missing one or more 'schedule' parameters"
      unless @schedules;

    foreach my $s ( @schedules ) {
        check_object_param( $s, 'schedule', 'Lire::ReportSchedule' );
        push @{$self->{'_schedules'}}, $s;
    }

    return;
}

=pod

=head2 schedules() 

Returns the Lire::ReportSchedule related to this object.

=cut

sub schedules {
    return @{$_[0]{'_schedules'}};
}

=pod

=head2 run( $store, $period, [$time]  )

Executes all the configured generation with a period of 'period' using
the $store DlfStore. Once the method returns, XML report will have
been generated in the DlfStore and OutputJob associated with the
ScheduleJob will have been sent. The $time parameter will be used to
determine the time window coverted by period. It defaults to the
current time.

=cut

sub run {
    my ( $self, $store, $period, $time ) = @_;

    $time ||= time();

    foreach my $sched ( $self->schedules() ) {
        next unless $sched->period() eq $period;
        $sched->run( $store, $self, $time );
    }

    return;
}

sub new_from_config {
    my ( $pkg, $dict ) = @_;

    $dict = $dict->Lire::Config::Dictionary::as_value();
    my $job = new Lire::ReportJob( $dict->{'name'} );
    $job->add_schedule( @{$dict->{'schedules'}} )
      if @{$dict->{'schedules'}};

    return $job;
}

1;

__END__

=pod

=head1 SEE ALSO

Lire::DlfStore(3pm) Lire::ImportJob(3pm) Lire::ReportSchedule(3pm)
lire(1) Lire::OutputJob(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: ReportJob.pm,v 1.7 2004/07/13 20:35:15 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

