/* $Id: cmd-offl.c,v 1.19 2001/08/19 04:47:53 linus Exp $
 *
 * Copyright (C) 1999  Lysator Computer Club,
 *                     Linkoping University, Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author: Linus Tolke
 */

#include <config.h>

#include <sys/param.h>

#include <sys/types.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>

#include <libintl.h>

#include <zmalloc.h>

#include <kom-types.h>
#include <services.h>
#include <kom-errno.h>

#include <misc-parser.h>

#include "xoutput.h"
#include "copy.h"
#include "error.h"
#include "cache.h"
#include "quit.h"
#include "main.h"
#include "offline.h"
#include "cmds.h"
#include "internal.h"

#define Export

/* Static function prototypes: */
static void trace_go(void);
static void trace_reset(void);
static int already_read(Local_text_no lno, Membership * ms);
static int before(Time * first, Time * second);

/*
 * These operations are probably slow so let the user know what is going on 
 * by giving him a counter to look at.
 */
static int counter;
static
void trace_go(void)
{
    if (counter == 0)
	xprintf("      \r");
    else
    {
	xprintf("\r%5d \b", counter);
    }
    counter ++;
    xflush(kom_outstream);
}

static
void trace_reset(void)
{
    counter = 0;
    trace_go(); 
}

/*
 * Functions to keep track of what was recently fetched.
 * They implement a binary tree.
 */
#define NEWLY_FUNCTIONS_FOR_TYPE(type)					\
struct type ## _struct							\
{									\
    type			no;					\
    struct type ## _struct * 	left;					\
    struct type ## _struct * 	right;					\
};									\
									\
static void free_fetched_ ## type ## _struct(struct type ## _struct *p);\
									\
static									\
void									\
free_fetched_ ## type ## _struct(struct type ## _struct * p)		\
{									\
    if (p == NULL)							\
	return;								\
    free_fetched_ ## type ## _struct(p->left);				\
    free_fetched_ ## type ## _struct(p->right);				\
    free(p);								\
}									\
									\
static int insert_into_ ## type ## _list(type no,			\
					 struct type ## _struct ** r);	\
									\
static									\
int									\
insert_into_ ## type ## _list(type no, struct type ## _struct ** r)	\
{									\
    if (*r == NULL)							\
    {									\
	*r = (struct type ## _struct *)					\
		malloc(sizeof(struct type ## _struct));			\
									\
	(*r)->no = no;							\
	(*r)->left = NULL;						\
	(*r)->right = NULL;						\
									\
	return FALSE;							\
    }									\
									\
    if ((*r)->no > no)							\
	return insert_into_ ## type ## _list(no, &(*r)->left);		\
    else if ((*r)->no < no)						\
	return insert_into_ ## type ## _list(no, &(*r)->right);		\
    else								\
	return TRUE;							\
}									\
									\
static int newly_fetched_ ## type(type no);				\
									\
static									\
int									\
newly_fetched_ ## type(type no)						\
{									\
    static struct type ## _struct * root = NULL;			\
									\
    if (no == 0)							\
    {									\
	free_fetched_ ## type ## _struct(root);				\
	root = NULL;							\
	return 0;							\
    }									\
									\
    return insert_into_ ## type ## _list(no, &root);			\
}

NEWLY_FUNCTIONS_FOR_TYPE(Conf_no);
NEWLY_FUNCTIONS_FOR_TYPE(Text_no);

/*
 * Compares times.
 * Returns true if the first time is before the second one else false.
 */
static
int
before(Time * first, Time * second)
{
#define CMP(kind)				\
    if (first->kind < second->kind)		\
	return TRUE;				\
    else if (first->kind > second->kind)	\
	return FALSE;

    CMP(tm_year);
    CMP(tm_mon);
    CMP(tm_mday);
    CMP(tm_hour);
    CMP(tm_min);
    CMP(tm_sec);
    return FALSE;
#undef CMP
}

/*
 * Returns true if the text with LOCAL NO is read in the MEMBERSHIP.
 */
static
int
already_read(Local_text_no lno, Membership * ms)
{
    int i;

    if (lno < ms->last_text_read)
	return TRUE;
    
    if (ms->read_texts == NULL)
	return FALSE;
    for (i = 0; i < ms->no_of_read; i++)
    {
	if (lno == ms->read_texts[i])
	    return TRUE;
    }
    return FALSE;
}

/*
 * Download all unread articles for later reading.
 *
 * Download is performed like this:
 * 1. Download a fresh membership. Save it.
 * 2. Download conf status of all the conferences in the membership.
 * 3. Download complete or missing part of the maps of the conferences that 
 *    have changed since last time.
 * 4. Step through the membership fetching texts and textstats that were not
 *    already there.
 * 5. For each textstat in the membership, refetch referenced textstats. 
 *    (This is probably a comment added to an existing text and must thus 
 *    be refetched.)
 * 6. For each textstat fetched in 4 or 5 fetch the confstat of the author if
 *    not already present in the database.
 *
 * Steps 2 - 6 can be done per conf. Step 5 and 6 can be done immediatly 
 * following the previous operation instead of saving them until last as
 * suggested by the list above.
 */
Export Success
cmd_download(String argument)
{
#define FUNCTION "cmd_download()"
    Success		retval;
    Membership_list	memb_list = EMPTY_MEMBERSHIP_LIST_i;
    unsigned short	mi;

    Conference		conf = EMPTY_CONFERENCE_i;

    Text_list		text_list = EMPTY_TEXT_LIST_i;
    Time		timestamp = NULL_TIME_i;

    String		textmass = EMPTY_STRING_i;
    Text_stat		textstat = EMPTY_TEXT_STAT_i;
    Text_stat		textstat2 = EMPTY_TEXT_STAT_i;
    unsigned int ti;

#define CLEAN_UP()				\
    do {					\
	release_membership_list(&memb_list);	\
	release_conf(&conf);			\
	release_text_list(&text_list);		\
	release_text_stat(&textstat);		\
	release_text_stat(&textstat2);		\
    } while (0)

    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_NO_ARGUMENTS();

#ifdef TRACE
    if (TRACE(5))
	xprintf("cmd_download()\n");
#endif

    trace_reset();
    trace_go();

    store_person_number(cpn);

#ifdef TRACE
    if (TRACE(5))
	xprintf("cmd_download() fetching membership for %lu.\n", cpn);
#endif

    retval = kom_get_membership(cpn, 0, MAX_CONF_NO, TRUE, &memb_list);
    trace_go();

    if (retval != OK)
    {
	fatal4(CLIENT_UNEXPECTED_SERVER_ERROR,
	       "get_membership(pers=%d, next_to_fetch=%d,chunk=%d)",
	       cpn, 0, MAX_CONF_NO);
	/*NOTREACHED*/
    }

    if (store_Membership_list(&memb_list) != OK)
    {
	RETURN(FAILURE);
    }

    newly_fetched_Text_no(0);	/* Empty the list. */
    newly_fetched_Conf_no(0);	/* Empty the list. */

    /*
     * The newly_fetched_Text_no list is used when a text is fetched during
     * this session with (at least) its author.
     */

    for (mi = 0; mi < memb_list.no_of_confs; mi++)
    {
	int confno = memb_list.confs[mi].conf_no;

	if (confno == 0)
	    continue;

	if (user_has_typed())
	{
	    RETURN(FAILURE);
	}

	handle_all_asyncs();

#ifdef TRACE
	if (TRACE(5))
	    xprintf("cmd_download() looking at conf %lu.\n", confno);
#endif

	if (!newly_fetched_Conf_no(confno))
	{
	    retval = kom_get_conf_stat(confno, &conf);
	    trace_go();

	    if (retval != OK)
	    {
#ifdef TRACE
		if (TRACE(5))
		    xprintf("cmd_download() cannot fetch conf %lu.\n", confno);
#endif

		continue;
	    }

	    if (store_Conference(confno, &conf) != OK)
	    {
		RETURN(FAILURE);
	    }
	}

	/* If there are no unread texts in the conf, we don't need the map. */
	if (memb_list.confs[mi].last_text_read >=
	    conf.first_local_no + conf.no_of_texts - 1)
	{
#ifdef TRACE
	    if (TRACE(5))
		xprintf("cmd_download() no unread in conf %lu.\n", confno);
#endif

	    continue;
	}

#ifdef TRACE
	if (TRACE(5))
	    xprintf("cmd_download() looking for map for conf %lu.\n", confno);
#endif

	if (fetch_map(confno, &text_list, &timestamp) != OK
	    || before(&timestamp, &conf.last_written))
	{
	    /* We need to refetch this map. */
	    /* It would be better just to fetch the increment. */
#ifdef TRACE
	    if (TRACE(5))
		xprintf("cmd_download() fetching map for conf %lu.\n", confno);
#endif

	    retval = kom_get_map(confno, 0, LONG_MAX, &text_list);
	    trace_go();

	    if (retval != OK)
	    {
		continue;
	    }

	    if (store_map(confno, &text_list, &conf.last_written) != OK)
	    {
		RETURN(FAILURE);
	    }
	}

#ifdef TRACE
	if (TRACE(5))
	    xprintf("cmd_download() looking for "
		    "unread unfetched texts in conf %lu.\n", confno);
#endif

#ifdef TRACE
	if (TRACE(0))
	    xprintf("Downloading from %S.\n", conf.name);
#endif

	for (ti = 0; ti < text_list.no_of_texts; ti++)
	{
	    Text_no		tno;
	    const Misc_info *	misc;
	    Misc_info_group	misc_data;

	    /* Skip over read texts. */
	    if (already_read(text_list.first_local_no + ti, 
			     &memb_list.confs[mi]))
	    {
		continue;
	    }

	    tno = text_list.texts[ti];

	    if (tno == 0)
		continue;

	    /* Fetch the text data. */
#ifdef TRACE
	    if (TRACE(5))
		xprintf("cmd_download() looking for text %lu localno %lu "
			"in conf %lu.\n", 
			tno, text_list.first_local_no + ti, confno);
#endif


	    /*
	     * The text never changes. 
	     * Therefore an old copy of the text is just as good as a new
	     * one.
	     */
	    if (fetch_text(tno, &textmass) != OK)
	    {
		/* It is not already fetched. */
		retval = kom_get_text(tno, 0, END_OF_STRING, &textmass);
		trace_go();

		if (retval != OK)
		{
		    continue;
		}

		if (store_text(tno, &textmass) != OK)
		{
		    RETURN(FAILURE);
		}
	    }

	    /*
	     * Now fetch the text stat.
	     * This might be new, this might be old. We don't care here.
	     */
	    if (fetch_Text_stat(tno, &textstat) != OK)
	    {
		/* It is not already fetched. */
		newly_fetched_Text_no(tno);

		retval = kom_get_text_stat(tno, &textstat);
		trace_go();

		if (retval != OK)
		{
		    continue;
		}

		if (store_Text_stat(tno, &textstat) != OK)
		{
		    RETURN(FAILURE);
		}
	    }

	    /* Verify that the author is fetched. */
#ifdef TRACE
	    if (TRACE(5))
		xprintf("cmd_download() looking for author of text %lu\n", 
			tno);
#endif

	    /* If we have an old copy, we don't need to fetch it again.
	     * The name might have changed but that is not crutial.
	     */
	    if (textstat.author == 0
		|| fetch_Conference(textstat.author, &conf) == OK)
	    {
		/* This author is either secret or newly fetched. */
	    }
	    else
	    {
		/* However if we fetch it, it is "new". */
		newly_fetched_Conf_no(textstat.author);

		retval = kom_get_conf_stat(textstat.author, &conf);
		trace_go();

		if (retval == OK)
		{
		    if (store_Conference(textstat.author, &conf) != OK)
			RETURN(FAILURE);
		}
	    }

#ifdef TRACE
	    if (TRACE(5))
		xprintf("cmd_download() going through misc_items "
			"of text %lu.\n", 
			tno);
#endif

	    for (misc = textstat.misc_items;
		 (misc_data = parse_next_misc(&misc,
					      textstat.misc_items +
					      textstat.no_of_misc)).type
		     != m_end_of_list;
		)
	    {
		Text_no ntno = 0;
		Conf_no ncno = 0;

		switch (misc_data.type)
		{
		case m_footn_in:
		case m_comm_in:
		case m_footn_to:
		case m_comm_to:
		    ntno = misc_data.text_ref;
		    break;
		case m_recpt:
		case m_cc_recpt:
		    ncno = misc_data.recipient;
		    break;
		default:
		    fprintf(stderr,
			    "Unknown misc-parser type returned: %u\n",
			    misc_data.type);
		    continue;
		}

		if (ncno != 0)
		{
		    /* This was a misc item referencing a person.
		     * Fetch that persons conf-stat. We need it
		     * to show the name.
		     * If we have an old copy, that will do.
		     */
#ifdef TRACE
		    if (TRACE(5))
			xprintf("cmd_download() looking for conf %lu from "
				"misc_items of text %lu.\n", 
				ncno, tno);
#endif

		    if (ncno == confno /* Optimize for the current conf. */
			|| fetch_Conference(ncno, &conf) == OK)
		    {
		    }
		    else
		    {
			/* If we fetch it, it is "new". */
			newly_fetched_Conf_no(ncno);

			retval = kom_get_conf_stat(ncno, &conf);
			trace_go();

			if (retval == OK)
			{
			    if (store_Conference(ncno, &conf) != OK)
				RETURN(FAILURE);
			}
		    }

		    if (misc_data.sender != 0)
		    {
			/* Check sender. */
#ifdef TRACE
			if (TRACE(5))
			    xprintf("cmd_download() looking for sender %lu.\n",
				    misc_data.sender);
#endif

			if (fetch_Conference(misc_data.sender, &conf) == OK)
			{
			}
			else
			{
				/* If we fetch it, it is "new". */
			    newly_fetched_Conf_no(misc_data.sender);

			    retval = kom_get_conf_stat(misc_data.sender,
						       &conf);
			    trace_go();

			    if (retval == OK)
			    {
				if (store_Conference(misc_data.sender,
						     &conf) != OK)
				    RETURN(FAILURE);
			    }
			}
		    }
		}

		if (ntno == 0)
		    continue;

		/*
		 * Some other text was referenced.
		 * Fetch that text-stat.
		 */
#ifdef TRACE
		if (TRACE(5))
		    xprintf("cmd_download() looking for referenced "
			    "text_stat %lu from misc_items of "
			    "text %lu.\n", 
			    ntno, tno);
#endif

		if (fetch_Text_stat(ntno, &textstat2) != OK)
		{
		    /*
		     * The text-stat is not known.
		     */
		    newly_fetched_Text_no(ntno);

		    retval = kom_get_text_stat(ntno, &textstat2);
		    trace_go();

		    if (retval != OK)
		    {
			continue;
		    }

		    if (store_Text_stat(ntno, &textstat2) != OK)
		    {
			RETURN(FAILURE);
		    }
		}

		/* Fetch the author. */
		/*
		 * If we have an old copy, we don't need to fetch it again.
		 * The name might have changed but that is not crutial.
		 */
#ifdef TRACE
		if (TRACE(5))
		    xprintf("cmd_download() looking for author %lu of "
			    "referenced text_stat %lu\n", 
			    textstat2.author, ntno);
#endif

		if (textstat2.author == 0
		    || fetch_Conference(textstat2.author, &conf) == OK)
		{
		    /* This author is either secret or newly fetched. */
		}
		else
		{
		    newly_fetched_Conf_no(textstat2.author);

		    retval = kom_get_conf_stat(textstat2.author, &conf);
		    trace_go();

		    if (retval == OK)
		    {
			if (store_Conference(textstat2.author, &conf) 
			    != OK)
			    RETURN(FAILURE);
		    }
		}

		/*
		 * During the fetching/verifying of the current text
		 * (text 1) we did reference another text (text 2)
		 * (number currently in ntno and text-stat in
		 * textstat2).  Lets verify that text 2 has a
		 * reference back to the current text (text 1).  If it
		 * does not, then the most plausible case is that text
		 * 2 was fetched in a previous session and the current
		 * text (text 1) was later added as a comment to text
		 * 2 and that we havn't refetched the text-stat of
		 * text 2 since. Simple enough solution, refetch
		 * text-stat of text 2.
		 *
		 * BUG: If the refetched text-stat contains newly
		 * added recipients (or senders) those won't be
		 * verified. This is probably not a problem since most
		 * of all recipients and senders are fetched anyway.
		 * */
		{
		    int found = FALSE;
		    const Misc_info * 	misc2;
		    Misc_info_group		misc_data2;

		    for (misc2 = textstat2.misc_items;
			 (misc_data2 =
			  parse_next_misc(&misc2,
					  textstat2.misc_items +
					  textstat2.no_of_misc)).type
			     != m_end_of_list;
			)
		    {
			Text_no ntno_back = 0;
					      
			switch (misc_data2.type)
			{
			case m_footn_in:
			case m_comm_in:
			case m_footn_to:
			case m_comm_to:
			    ntno_back = misc_data2.text_ref;
			    break;
			default:
			    continue;
			}

			if (ntno_back == tno)
			{
			    found = TRUE;
			    break;
			}
		    }

		    if (!found)
		    {
#ifdef TRACE
			if (TRACE(5))
			    xprintf("cmd_download() text_stat %lu referenced "
				    "from text %lu did not point back.\n",
				    ntno, tno);
#endif

			newly_fetched_Text_no(ntno);
			/*
			 * Observer that the author of a text cannot
			 * change, therefore we need not verify that
			 * the author is fetched because it is verified
			 * above.
			 */

			retval = kom_get_text_stat(ntno, &textstat2);
			trace_go();

			if (retval != OK)
			{
			    continue;
			}

			if (store_Text_stat(ntno, &textstat2) != OK)
			{
			    RETURN(FAILURE);
			}
		    }
		}
	    }
	}
    }

    RETURN(OK);
#undef CLEAN_UP
#undef FUNCTION
}

/*
 * Upload everything that was cached locally.
 *
 * Upload is performed like this:
 * 1. Upload read marks
 *      Walk through the membership.
 * 2. Upload written texts.
 */
Export Success
cmd_upload(String argument)
{
#define FUNCTION "cmd_upload()"

    Membership_list	memb_list = EMPTY_MEMBERSHIP_LIST_i;
    unsigned short	mi;
    String		message = EMPTY_STRING;
    unsigned short	no_of_miscs;
    Misc_info *		misc = NULL;
    int			index;

#define CLEAN_UP()				\
    do {					\
        s_clear(&argument); 			\
	release_membership_list(&memb_list);	\
	s_clear(&message);			\
	zfree(misc);				\
    } while (0)

    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_NO_ARGUMENTS();

#ifdef TRACE
    if (TRACE(4))
	xprintf("cmd_upload()\n");
#endif

    trace_reset();

    if (fetch_Membership_list(&memb_list) != OK)
    {
	RETURN (FAILURE);
    }

    for (mi = 0; mi < memb_list.no_of_confs; mi++)
    {
	int len;
	Local_text_no * lot;

	Conf_no cno = memb_list.confs[mi].conf_no;

	if (fetch_marks_in_conf(cno, &len, &lot) != FAILURE)
	{
	    if (len > 0)
	    {
		int pos = 0;
		int rest;

		/* Now we have a list of texts to be marked as read. */
#ifdef TRACE
		if (TRACE(4))
		    xprintf("cmd_upload() marking %d texts in conf %lu\n",
			    len, cno);
#endif

		/* BUG: We don't handle the problems.
		 * A serious problem is that the lyskom server has a limit on
		 * the number of texts that we are allowed to mark in each
		 * request. If this is a problem, then this will fail.
		 * Let us assume that the limit is 43 and only send 43 marks
		 * at a time.
		 */
		do
		{
		    rest = len - pos;
		    if (rest > 43)
			rest = 43;
		    kom_mark_as_read(cno, rest, &lot[pos]);
		    trace_go();
		    pos += rest;
		} while (pos < len);
	    }

	    zfree(lot);
	    done_with_marks_in_conf(memb_list.confs[mi].conf_no);
	}
    }

#ifdef TRACE
    if (TRACE(4))
    {
	xprintf("cmd_upload() done with marks. Beginning texts.\n");
    }
#endif

    while ((index = fetch_created_text(&message, &no_of_miscs, &misc)))
    {
	Text_no number_of_new_text = 
	    kom_create_text(message, no_of_miscs, misc);
	trace_go();

#ifdef TRACE
	if (TRACE(4))
	{
	    xprintf("cmd_upload() added text %lu.\n", 
		    number_of_new_text);
	}
#endif

	if (number_of_new_text == 0)
	{
	    xprintf(gettext("\nTexten kunde inte lggas in. Felet var\n"));
	    switch (kom_errno)
	    {
	    case KOM_NO_CONNECT:
		fatal1(CLIENT_SERVER_ERROR,
		       "create_text() failed");
		break;

	    case KOM_INDEX_OUT_OF_RANGE:
		xprintf(gettext("Du har inte rttigheter att skriva hr"));
		RETURN (FAILURE);

	    default:
		fatal3(CLIENT_UNEXPECTED_SERVER_ERROR,
		       "create_text() failed. Lyskom-errno: %d %s",
		       kom_errno, kom_errno_string());
		break;
	    }
	    continue;
	}

	done_with_created_text(index);

	mark_global_as_read(number_of_new_text, 0);
	trace_go();

	xprintf(gettext("\nText %d inlagd.\n"), number_of_new_text);
    }

#ifdef TRACE
    if (TRACE(4))
    {
	xprintf("cmd_upload() done with texts.\n");
    }
#endif

    RETURN (OK);
}
