/***************************************************************************
 $RCSfile: ctcore_public.c,v $
                             -------------------
    cvs         : $Id: ctcore_public.c,v 1.7 2003/02/17 04:17:48 aquamaniac Exp $
    begin       : Sun Dec 08 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif


#include "ctcore_public.h"
#include "chameleon/debug.h"
#include "chameleon/error.h"
#include "chameleon/directory.h"

#include <stdlib.h>
#include <assert.h>
#include <string.h>


CTREADERDESCRIPTION *CTCore_ReaderDescr_new(){
  CTREADERDESCRIPTION *rd;

  DBG_ENTER;
  rd=(CTREADERDESCRIPTION*)malloc(sizeof(CTREADERDESCRIPTION));
  assert(rd);
  memset(rd,0,sizeof(CTREADERDESCRIPTION));
  DBG_LEAVE;
  return rd;
}


void CTCore_ReaderDescr_free(CTREADERDESCRIPTION *rd){
  DBG_ENTER;
  if (rd)
    free(rd);
  DBG_LEAVE;
}


void CTCore_ReaderDescr_Add(CTREADERDESCRIPTION *cmd,
			    CTREADERDESCRIPTION **head) {
  CTREADERDESCRIPTION *curr;

  DBG_ENTER;
  assert(cmd);
  assert(head);

  curr=*head;
  if (!curr) {
    *head=cmd;
  }
  else {
    /* find last */
    while(curr->next) {
      curr=curr->next;
    } /* while */
    curr->next=cmd;
  }
  DBG_LEAVE;
}


void CTCore_ReaderDescr_Remove(CTREADERDESCRIPTION *cmd,
			       CTREADERDESCRIPTION **head) {
  CTREADERDESCRIPTION *curr;

  DBG_ENTER;
  assert(cmd);
  assert(head);

  curr=*head;
  if (curr) {
    if (curr==cmd) {
      *head=curr->next;
    }
    else {
      /* find predecessor */
      while(curr->next!=cmd) {
	curr=curr->next;
      } /* while */
      if (curr)
	curr->next=cmd->next;
    }
  }
  DBG_LEAVE;
}


CTREADERDESCRIPTION *CTCore_ReaderDescr_Duplicate(CTREADERDESCRIPTION *ds){
  CTREADERDESCRIPTION *nds;

  nds=CTCore_ReaderDescr_new();
  *nds=*ds;
  nds->next=0;
  return nds;
}


CTREADERDESCRIPTION *CTCore_ReadReaderDescr(CONFIGGROUP *grp) {
  CTREADERDESCRIPTION *ds;
  const char *p;
  int i;

  DBG_ENTER;
  ds=CTCore_ReaderDescr_new();

  /* get the reader flags
   */
  i=0;
  ds->readerFlags=0;
  while((p=Config_GetValue(grp, "flags", 0, i))!=0){
    if (strcasecmp(p,"keypad")==0)
      ds->readerFlags|=CTREADERFLAGS_KEYPAD;
    else if (strcasecmp(p,"display")==0)
      ds->readerFlags|=CTREADERFLAGS_DISPLAY;
    else {
      int j;

      j=Config_GetIntValue(grp,"flags",-1,i);
      if (j!=-1)
	ds->readerFlags|=j;
      else {
	DBG_ERROR("Unknown reader flag \"%s\"",p);
      }
    }
    i++;
  } /* while */

  p=Config_GetValue(grp,"readertype","other",0);
  if (p) {
    if (strlen(p)+1>sizeof(ds->typeName)) {
      DBG_ERROR("Typename too long, using \"other\"");
      p="other";
    }
    strcpy(ds->typeName,p);
  }

  p=Config_GetValue(grp,"drivertype","unknown",0);
  if (Config_Compare(p, "pcsc")==0)
    ds->driverType=DriverTypePCSC;
  else if (Config_Compare(p, "ctapi")==0)
    ds->driverType=DriverTypeCTAPI;
  else
    ds->driverType=DriverTypeUnknown;

  p=Config_GetValue(grp, "port", "COM1", 0);
  if (p) {
    if (strlen(p)+1>sizeof(ds->portName)) {
      DBG_ERROR("Portname too long, using \"COM1\"");
      p="COM1";
    }
    strcpy(ds->portName,p);

  }

  p=Config_GetValue(grp, "driver", 0, 0);
  if (p) {
    if ((strlen(p)+1)>sizeof(ds->driverName)) {
      DBG_ERROR("Drivername too long");
      CTCore_ReaderDescr_free(ds);
      DBG_LEAVE;
      return 0;
    }
    strcpy(ds->driverName,p);
  }

  p=Config_GetValue(grp, "name", 0, 0);
  if (p) {
    if ((strlen(p)+1)>sizeof(ds->name)) {
      DBG_ERROR("Name too long");
      CTCore_ReaderDescr_free(ds);
      DBG_LEAVE;
      return 0;
    }
    strcpy(ds->name,p);
  }

  DBG_LEAVE;
  return ds;
}


int CTCore_WriteReaderDescr(CONFIGGROUP *grp,
			    CTREADERDESCRIPTION *ds){
  const char *p;
  ERRORCODE err;

  /* reader type */
  err=Config_SetValue(grp,
		      CONFIGMODE_VARIABLE |
		      CONFIGMODE_NAMECREATE_VARIABLE |
		      CONFIGMODE_OVERWRITE_VARS,
		      "readertype",
		      ds->typeName);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    return 1;
  }

  switch(ds->driverType) {
  case DriverTypePCSC:
    p="pcsc";
    break;
  case DriverTypeCTAPI:
    p="ctapi";
    break;
  default:
    p="unknown";
    break;
  } /* switch */
  err=Config_SetValue(grp,
		      CONFIGMODE_VARIABLE |
		      CONFIGMODE_NAMECREATE_VARIABLE |
		      CONFIGMODE_OVERWRITE_VARS,
		      "drivertype",
		      p);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    return 1;
  }

  /* driver */
  err=Config_SetValue(grp,
		      CONFIGMODE_VARIABLE |
		      CONFIGMODE_NAMECREATE_VARIABLE |
		      CONFIGMODE_OVERWRITE_VARS,
		      "driver",
		      ds->driverName);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    return 1;
  }

  /* name */
  err=Config_SetValue(grp,
		      CONFIGMODE_VARIABLE |
		      CONFIGMODE_NAMECREATE_VARIABLE |
		      CONFIGMODE_OVERWRITE_VARS,
		      "name",
		      ds->name);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    return 1;
  }

  /* port */
  err=Config_SetValue(grp,
		      CONFIGMODE_VARIABLE |
		      CONFIGMODE_NAMECREATE_VARIABLE |
		      CONFIGMODE_OVERWRITE_VARS,
		      "port",
		      ds->portName);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    return 1;
  }

  /* reader flags */
  err=Config_ClearVariable(grp,
			   CONFIGMODE_VARIABLE |
			   CONFIGMODE_NAMECREATE_VARIABLE,
			   "flags");
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    return 1;
  }
  if (ds->readerFlags & CTREADERFLAGS_KEYPAD) {
    err=Config_AddValue(grp,
			CONFIGMODE_VARIABLE,
			"flags",
			"keypad");
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      return 1;
    }
  }
  if (ds->readerFlags & CTREADERFLAGS_DISPLAY) {
    err=Config_AddValue(grp,
			CONFIGMODE_VARIABLE,
			"flags",
			"display");
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      return 1;
    }
  }

  return 0;
}


int CTCore_ReadDriverDescriptions(const char *configdir,
				  CONFIGGROUP *cfg) {
  DIRECTORYDATA *d;
  char buffer[512];
  int bufferlen;
  int goon=1;

  d=Directory_new();
  if (Directory_Open(d,configdir)) {
    DBG_ERROR("Could not open directory \"%s\"", configdir);
    Directory_free(d);
    return 1;
  }

  DBG_INFO("Opened dir \"%s\", reading.", configdir);

  do {
    const char *tmp;

    bufferlen=sizeof(buffer);
    buffer[0]=0;
    if (Directory_Read(d,buffer,bufferlen)) {
      DBG_DEBUG("No file left");
      goon=0;
    }
    else {
      CONFIGGROUP *cmdgroup;
      int i;
      char fnbuffer[256];

      goon=1;
      DBG_INFO("Found file \"%s\"",buffer);
      i=strlen(buffer);
      if (i>4) {
	if (strcmp(buffer+(i-4),".dsc")==0) {
	  if (strlen(configdir+i+2)>sizeof(fnbuffer)) {
	    DBG_ERROR("Path too long");
	  }
	  else {
            /* filename is ok, read file */
	    strcpy(fnbuffer, configdir);
            strcat(fnbuffer, "/");
	    strcat(fnbuffer,buffer);
	    DBG_DEBUG("Really reading file \"%s\"",fnbuffer);
	    cmdgroup=Config_new();
	    if (Config_ReadFile(cmdgroup,
				fnbuffer,
				CONFIGMODE_REMOVE_STARTING_BLANKS |
				CONFIGMODE_REMOVE_TRAILING_BLANKS |
				CONFIGMODE_REMOVE_QUOTES|
				CONFIGMODE_ALLOW_GROUPS)) {
	      DBG_ERROR("Could not read file \"%s\"",fnbuffer);
	    }
	    else {
	      tmp=Config_GetValue(cmdgroup,
				  "readertype",
				  0,0);
	      if (!tmp) {
		DBG_ERROR("Readertype not given, will not add this driver");
	      }
	      else {
		CONFIGGROUP *destgroup;

		destgroup=Config_GetGroup(cfg,
					  tmp,
					  0);
		if (!destgroup) {
                  DBG_ERROR("Could not create group");
		}
		else {
		  DBG_INFO("Adding drivers from file \"%s\"",fnbuffer);
		  Config_ClearGroup(destgroup);
		  Config_ImportTreeChildren(destgroup,
					    cmdgroup);
		}
	      }
	    }
            Config_free(cmdgroup);
	  }
	}
      }
    }
  }
  while(goon);

  if (Directory_Close(d)) {
    DBG_WARN("Error closing directory \"%s\"",configdir);
  }
  Directory_free(d);

#if DEBUGMODE>10
  Config_DumpGroup(stdout, cfg, 2);
#endif
  return 0;
}



