/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  component.cc - AtkComponent C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "component.h"
#include "private/component_p.h"
#include "object.h"

using namespace Inti;

/*  Atk::Component
 */

Atk::Component::~Component()
{
}
	
AtkComponentIface* 
Atk::Component::atk_component_iface() const 
{ 
	return peek<AtkComponentIface>(ATK_TYPE_COMPONENT);
}

Atk::Component::operator AtkComponent* () const 
{ 
	return this ? atk_component() : 0; 
}
	
Atk::Object*
Atk::Component::ref_accessible_at_point(int x, int y, CoordType coord_type) const
{
	AtkObject *object = atk_component_ref_accessible_at_point(atk_component(), x, y, (AtkCoordType)coord_type);
	return object ? G::Object::wrap<Atk::Object>(object) : 0;
}

void 
Atk::Component::get_extents(int *x, int *y, int *width, int *height, CoordType coord_type) const
{
	atk_component_get_extents(atk_component(), x, y, width, height, (AtkCoordType)coord_type);
}

void 
Atk::Component::get_position(int *x, int *y, CoordType coord_type) const
{
	atk_component_get_position(atk_component(), x, y, (AtkCoordType)coord_type);

}

void 
Atk::Component::get_size(int *width, int *height) const
{
	atk_component_get_size(atk_component(), width, height);
}

Atk::Layer
Atk::Component::get_layer() const
{
	return (Layer)atk_component_get_layer(atk_component());
}

int
Atk::Component::get_mdi_zorder() const
{
	return atk_component_get_mdi_zorder(atk_component());
}

unsigned int
Atk::Component::add_focus_handler(AtkFocusHandler handler)
{
	return atk_component_add_focus_handler(atk_component(), handler);
}

bool 
Atk::Component::contains(int x, int y, CoordType coord_type) const
{
	return atk_component_contains(atk_component(), x, y, (AtkCoordType)coord_type);
}

bool 
Atk::Component::grab_focus()
{
	return atk_component_grab_focus(atk_component());
}

void 
Atk::Component::remove_focus_handler(unsigned int handler_id)
{
	atk_component_remove_focus_handler(atk_component(), handler_id);
}

bool
Atk::Component::set_extents(int x, int y, int width, int height, CoordType coord_type)
{
	return atk_component_set_extents(atk_component(), x, y, width, height, (AtkCoordType)coord_type);
}

bool 
Atk::Component::set_position(int x, int y, CoordType coord_type)
{
	return atk_component_set_position(atk_component(), x, y, (AtkCoordType)coord_type);
}

bool 
Atk::Component::set_size(int width, int height)
{
	return atk_component_set_size(atk_component(), width, height);
}

/*  Atk::ComponentIface
 */

void
Atk::ComponentIface::init(AtkComponentIface *g_iface)
{
	g_iface->add_focus_handler = &add_focus_handler_proxy;
	g_iface->contains = &contains_proxy;
	g_iface->ref_accessible_at_point = &ref_accessible_at_point_proxy;
	g_iface->get_extents = &get_extents_proxy;
	g_iface->get_position = &get_position_proxy;
	g_iface->get_size = &get_size_proxy;
	g_iface->grab_focus = &grab_focus_proxy;
	g_iface->remove_focus_handler = &remove_focus_handler_proxy;
	g_iface->set_extents = &set_extents_proxy;
	g_iface->set_position = &set_position_proxy;
	g_iface->set_size = &set_size_proxy;
	g_iface->get_layer = &get_layer_proxy;
	g_iface->get_mdi_zorder = &get_mdi_zorder_proxy;
}

guint
Atk::ComponentIface::add_focus_handler_proxy(AtkComponent *component, AtkFocusHandler handler)
{
	guint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		result = tmp_component->do_add_focus_handler(handler);
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->add_focus_handler)
			result = g_iface->add_focus_handler(component, handler);
	}
	return result;
}

gboolean
Atk::ComponentIface::contains_proxy(AtkComponent *component, gint x, gint y, AtkCoordType coord_type)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		result = tmp_component->do_contains(x, y, coord_type);
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->contains)
			result = g_iface->contains(component, x, y, coord_type);
	}
	return result;
}

AtkObject*
Atk::ComponentIface::ref_accessible_at_point_proxy(AtkComponent *component, gint x, gint y, AtkCoordType coord_type)
{
	AtkObject *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		result = tmp_component->do_ref_accessible_at_point(x, y, coord_type);
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->ref_accessible_at_point)
			result = g_iface->ref_accessible_at_point(component, x, y, coord_type);
	}
	return result;
}

void
Atk::ComponentIface::get_extents_proxy(AtkComponent *component, gint *x, gint *y, gint *width, gint *height, AtkCoordType coord_type)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		tmp_component->do_get_extents(x, y, width, height, coord_type);
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->get_extents)
			g_iface->get_extents(component, x, y, width, height, coord_type);
	}
}

void
Atk::ComponentIface::get_position_proxy(AtkComponent *component, gint *x, gint *y, AtkCoordType coord_type)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		tmp_component->do_get_position(x, y, coord_type);
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->get_position)
			g_iface->get_position(component, x, y, coord_type);
	}
}

void
Atk::ComponentIface::get_size_proxy(AtkComponent *component, gint *width, gint *height)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		tmp_component->do_get_size(width, height);
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->get_size)
			g_iface->get_size(component, width, height);
	}
}

gboolean
Atk::ComponentIface::grab_focus_proxy(AtkComponent *component)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		result = tmp_component->do_grab_focus();
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->grab_focus)
			result = g_iface->grab_focus(component);
	}
	return result;
}

void
Atk::ComponentIface::remove_focus_handler_proxy(AtkComponent *component, guint handler_id)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		tmp_component->do_remove_focus_handler(handler_id);
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->remove_focus_handler)
			g_iface->remove_focus_handler(component, handler_id);
	}
}

gboolean
Atk::ComponentIface::set_extents_proxy(AtkComponent *component, gint  x, gint y, gint width, gint height, AtkCoordType coord_type)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		result = tmp_component->do_set_extents(x, y, width, height, coord_type);
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->set_extents)
			result = g_iface->set_extents(component, x, y, width, height, coord_type);
	}
	return result;
}

gboolean
Atk::ComponentIface::set_position_proxy(AtkComponent *component, gint  x, gint y, AtkCoordType coord_type)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		result = tmp_component->do_set_position(x, y, coord_type);
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->set_position)
			result = g_iface->set_position(component, x, y, coord_type);
	}
	return result;
}

gboolean
Atk::ComponentIface::set_size_proxy(AtkComponent *component, gint width, gint height)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		result = tmp_component->do_set_size(width, height);
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->set_size)
			result = g_iface->set_size(component, width, height);
	}
	return result;
}

AtkLayer
Atk::ComponentIface::get_layer_proxy(AtkComponent *component)
{
	AtkLayer result = ATK_LAYER_INVALID;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		result = tmp_component->do_get_layer();
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->get_layer)
			result = g_iface->get_layer(component);
	}
	return result;
}

gint
Atk::ComponentIface::get_mdi_zorder_proxy(AtkComponent *component)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(component));
	Atk::Component *tmp_component = dynamic_cast<Atk::Component*>(object);
	if (tmp_component)
		result = tmp_component->do_get_mdi_zorder();
	else
	{
		AtkComponentIface *g_iface = static_cast<AtkComponentIface*>(g_type_interface_peek_parent(ATK_COMPONENT_GET_IFACE(component)));
		if (g_iface->get_mdi_zorder)
			result = g_iface->get_mdi_zorder(component);
	}
	return result;
}


/*  Overridable methods
 */

unsigned int
Atk::Component::do_add_focus_handler(AtkFocusHandler handler)
{
	unsigned int result = 0;
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->add_focus_handler)
		result = g_iface->add_focus_handler(atk_component(), handler);
	return result;
}

bool
Atk::Component::do_contains(int x, int y, AtkCoordType coord_type)
{
	bool result = false;
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->contains)
		result = g_iface->contains(atk_component(), x, y, coord_type);
	return result;
}

AtkObject*
Atk::Component::do_ref_accessible_at_point(int x, int y, AtkCoordType coord_type)
{
	AtkObject *result = 0;
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->ref_accessible_at_point)
		result = g_iface->ref_accessible_at_point(atk_component(), x, y, coord_type);
	return result;
}

void
Atk::Component::do_get_extents(int *x, int *y, int *width, int *height, AtkCoordType coord_type)
{
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->get_extents)
		g_iface->get_extents(atk_component(), x, y, width, height, coord_type);
}

void
Atk::Component::do_get_position(int *x, int *y, AtkCoordType coord_type)
{
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->get_position)
		g_iface->get_position(atk_component(), x, y, coord_type);
}

void
Atk::Component::do_get_size(int *width, int *height)
{
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->get_size)
		g_iface->get_size(atk_component(), width, height);
}

bool
Atk::Component::do_grab_focus()
{
	bool result = false;
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->grab_focus)
		result = g_iface->grab_focus(atk_component());
	return result;
}

void
Atk::Component::do_remove_focus_handler(unsigned int handler_id)
{
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->remove_focus_handler)
		g_iface->remove_focus_handler(atk_component(), handler_id);
}

bool
Atk::Component::do_set_extents(int  x, int y, int width, int height, AtkCoordType coord_type)
{
	bool result = false;
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->set_extents)
		result = g_iface->set_extents(atk_component(), x, y, width, height, coord_type);
	return result;
}

bool
Atk::Component::do_set_position(int  x, int y, AtkCoordType coord_type)
{
	bool result = false;
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->set_position)
		result = g_iface->set_position(atk_component(), x, y, coord_type);
	return result;
}

bool
Atk::Component::do_set_size(int width, int height)
{
	bool result = false;
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->set_size)
		result = g_iface->set_size(atk_component(), width, height);
	return result;
}

AtkLayer
Atk::Component::do_get_layer()
{
	AtkLayer result = ATK_LAYER_INVALID;
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->get_layer)
		result = g_iface->get_layer(atk_component());
	return result;
}

int
Atk::Component::do_get_mdi_zorder()
{
	int result = 0;
	AtkComponentIface *g_iface = peek_parent<AtkComponentIface>(atk_component_iface());
	if (g_iface->get_mdi_zorder)
		result = g_iface->get_mdi_zorder(atk_component());
	return result;
}

