/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *  Copyright (C) 2000 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/connection.h
//! @brief C++ signal-slot connection interface.

#ifndef INTI_CONNECTION_H
#define INTI_CONNECTION_H

#ifndef INTI_OBJECT_H
#include <inti/object.h>
#endif

namespace Inti {
	
class Slot;

//! @class Node connection.h inti/connection.h
//! @brief Base class for classes managing slots.
//!
//! A node connects a slot to its Connection class,disconnec the class returned
//! from a signal's connect() method.

class Node : public ReferencedObject
{
	Slot *slot_;
	
protected:
	Node(Slot *slot);
	//!< Constructor.

	virtual ~Node();
	//!< Destructor.

public:
	
	Slot* slot() { return slot_; }
	//!< Returns a pointer to the slot held by this node.
	
	virtual void block() = 0;
	//!< Block signal emission to the slot until unblock is called.

	virtual void unblock() = 0;
	//!< Unblock the slot so signal emmissions can be received.

	virtual void disconnect() = 0;
	//!< Disconnect the slot. The slot will no longer recieve signal emissions.
};

//! @class Connection connection.h inti/connection.h
//! @brief A signal connection class.
//!
//! A Connection class is returned by value from a signal's connect()
//! method. Using this class you can block, unblock and disconnect a
//! signal connection. When re-assigning a signal connection to another
//! Connection you don't need to disconnect the original connection
//! first. The asssignment operator disconnects the original 
//! connection before re-assigning the new connection.

class Connection
{
	Node *node_;

public:
//! @name Constructors
//! @{

	Connection();
	//!< Default constructor.

	Connection(Node *node);
	//!< Construct a connection object for node.
	//!< @param node A pointer to the Node class for this connection.

	Connection(const Connection& src);
	//!< Copy constructor.

	~Connection();
	//!< Destructor.

	Connection& operator=(const Connection& src);
	//!< Assignment operator.
	
//! @}
//! @name Accessors
//! @{

	bool is_connected() const;
	//!< Determines whether the connection is connected to a slot.
	//!< @return <EM>true</EM> if connected to a slot.
	//!<
	//!< <BR>This method is useful when connecting and disconnecting a
	//!< connection multiple times within a program.

//! @}
//! @name Methods
//! @{

	void block();
	//!< Block signal transmission to a slot.
	//!< The slot will not be called during any signal emissions unless it is unblocked again.

	void unblock();
	//!< Unblock a previously blocked slot.
	//!< A blocked slot is skipped during signal emissions and will not be invoked, unblocking
	//!< it (for exactly the number of times it has been blocked before) reverts its "blocked"
	//!< state, so the slot will be recognized by the signal system and is called upon future
	//!< or currently ongoing signal emissions.

	void disconnect();
	//!< Disconnect a slot.
	//!< The slot will not be called during any future or currently ongoing
	//!< emissions of the signal it has been connected to.
	
//! @}
};

} // namespace Inti

#endif // INTI_CONNECTION_H

