/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  container.cc - GtkContainer C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "container.h"
#include "private/container_p.h"

using namespace Inti;

/*  Gtk::Container
 */

Gtk::Container::Container(GtkContainer *container, bool reference)
: Widget((GtkWidget*)container, reference) 
{
}

Gtk::Container::Container()
: Widget((GtkWidget*)ContainerClass::create())
{
}

Gtk::Container::~Container()
{
}

GtkContainerClass*
Gtk::Container::gtk_container_class() const
{ 
	return get_class<GtkContainerClass>(); 
}
	
Gtk::Container::operator GtkContainer* () const 
{
	return this ? gtk_container() : 0; 
}
	
bool 
Gtk::Container::is_resize_container() const 
{ 
	return is_a(GTK_TYPE_CONTAINER) && gtk_container()->resize_mode != GTK_RESIZE_PARENT; 
}

GType
Gtk::Container::child_type() const
{
	return gtk_container_child_type(gtk_container());
}

Gtk::Widget*
Gtk::Container::focus_child() const
{
	return G::Object::wrap<Widget>(gtk_container()->focus_child);
}

bool
Gtk::Container::get_children(std::vector<Widget*>& child_list) const
{
	g_return_val_if_fail(child_list.empty(), false);
	GList *first = gtk_container_get_children(gtk_container());
	GList *next = first;

	while (next != 0)
	{
		child_list.push_back(G::Object::wrap<Widget>((GtkWidget*)next->data));
		next = g_list_next(next);
	}

	g_list_free(first);
	return !child_list.empty();
}

unsigned int
Gtk::Container::get_border_width() const
{
	return gtk_container_get_border_width(gtk_container());
}

Gtk::ResizeMode
Gtk::Container::get_resize_mode() const
{
	return (ResizeMode)gtk_container_get_resize_mode(gtk_container());
}

bool
Gtk::Container::get_focus_chain(std::vector<Widget*>& focusable_widgets) const
{
	GList *first = 0;
	bool result = gtk_container_get_focus_chain(gtk_container(), &first);
	GList *next = first;

	while (next != 0)
	{
		focusable_widgets.push_back(G::Object::wrap<Widget>((GtkWidget*)next->data));
		next = g_list_next(next);
	}

	g_list_free(first);
	return result;
}

Gtk::Adjustment*
Gtk::Container::get_focus_vadjustment() const
{
	return G::Object::wrap<Adjustment>(gtk_container_get_focus_vadjustment(gtk_container()));
}

Gtk::Adjustment*
Gtk::Container::get_focus_hadjustment() const
{
	return G::Object::wrap<Adjustment>(gtk_container_get_focus_hadjustment(gtk_container()));
}

void
Gtk::Container::add(Widget& widget)
{
	gtk_container_add(gtk_container(), widget.gtk_widget());
}

void
Gtk::Container::remove(Widget&widget)
{
	gtk_container_remove(gtk_container(), widget.gtk_widget());
}

void
Gtk::Container::check_resize()
{
	gtk_container_check_resize(gtk_container());
}

void
Gtk::Container::set_focus_child(Widget& widget)
{
	gtk_container_set_focus_child(gtk_container(), widget.gtk_widget());
}

void
Gtk::Container::set_border_width(unsigned int border_width)
{
	gtk_container_set_border_width(gtk_container(), border_width);
}

void 
Gtk::Container::set_resize_mode(ResizeMode resize_mode)
{	
	gtk_container_set_resize_mode(gtk_container(), (GtkResizeMode)resize_mode);
}

namespace { // ForeachSlot callback

void foreach_slot_callback(GtkWidget *widget, gpointer data)
{
	Gtk::Container::ForeachSlot *slot = static_cast<Gtk::Container::ForeachSlot*>(data);
	slot->call(*G::Object::wrap<Gtk::Widget>(widget));
}

} // ForeachSlot callback

void 
Gtk::Container::foreach(const ForeachSlot *each)
{
	gtk_container_foreach(gtk_container(), &foreach_slot_callback, (void*)each);
}

void 
Gtk::Container::set_reallocate_redraws(bool needs_redraws)
{
	gtk_container_set_reallocate_redraws(gtk_container(), needs_redraws);
}

void 
Gtk::Container::set_focus_vadjustment(Adjustment *adjustment)
{
	gtk_container_set_focus_vadjustment(gtk_container(), adjustment->gtk_adjustment());
}

void 
Gtk::Container::set_focus_hadjustment(Adjustment *adjustment)
{
	gtk_container_set_focus_hadjustment(gtk_container(), adjustment->gtk_adjustment());
}

void 
Gtk::Container::resize_children()
{
	gtk_container_resize_children(gtk_container());
}

void
Gtk::Container::propagate_expose(Widget& child, const Gdk::EventExpose& event)
{
	gtk_container_propagate_expose(gtk_container(), child.gtk_widget(), event.gdk_event_expose());
}

void 
Gtk::Container::set_focus_chain(const std::vector<Widget*>& focusable_widgets)
{
	g_return_if_fail(!focusable_widgets.empty());
	GList *tmp_widgets = 0;
	int count = focusable_widgets.size();
	
	int i = 0;
	while (i < count)
	{
		tmp_widgets = g_list_append(tmp_widgets, focusable_widgets[i]->gtk_widget());
		++i;
	}

	gtk_container_set_focus_chain(gtk_container(), tmp_widgets);
	g_list_free(tmp_widgets);
}

void 
Gtk::Container::unset_focus_chain()
{
	gtk_container_unset_focus_chain(gtk_container());
}

/*  Gtk::ContainerClass
 */

void
Gtk::ContainerClass::init(GtkContainerClass *g_class)
{
	WidgetClass::init((GtkWidgetClass*)g_class);
	g_class->add = &add_proxy;
	g_class->remove = &remove_proxy;
	g_class->check_resize = &check_resize_proxy;
	g_class->set_focus_child = &set_focus_child_proxy;
	g_class->forall = &forall_proxy;
	g_class->child_type = &child_type_proxy;
	g_class->composite_name = &composite_name_proxy;
	g_class->set_child_property = &set_child_property_proxy;
	g_class->get_child_property = &get_child_property_proxy;
}

GType
Gtk::ContainerClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_CONTAINER, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::ContainerClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::ContainerClass::forall_proxy(GtkContainer *container, gboolean include_internals, GtkCallback callback, gpointer callback_data)
{
	Container *tmp_container = G::Object::pointer<Container>(container);
	if (tmp_container)
		tmp_container->do_forall(include_internals, callback, callback_data);
	else
	{
		GtkContainerClass *g_class = G::TypeInstance::class_peek_parent<GtkContainerClass>(GTK_CONTAINER_GET_CLASS(container));
		if (g_class->forall)
			g_class->forall(container, include_internals, callback, callback_data);
	}
}

GtkType
Gtk::ContainerClass::child_type_proxy(GtkContainer *container)
{
	GtkType result = GTK_TYPE_INVALID;
	Container *tmp_container = G::Object::pointer<Container>(container);
	if (tmp_container)
		result = tmp_container->do_child_type();
	else
	{
		GtkContainerClass *g_class = G::TypeInstance::class_peek_parent<GtkContainerClass>(GTK_CONTAINER_GET_CLASS(container));
		if (g_class->child_type)
			result = g_class->child_type(container);
	}
	return result;
}

gchar*
Gtk::ContainerClass::composite_name_proxy(GtkContainer	*container, GtkWidget *child)
{
	gchar *result = 0;
	Container *tmp_container = G::Object::pointer<Container>(container);
	if (tmp_container)
		result = tmp_container->do_composite_name(child);
	else
	{
		GtkContainerClass *g_class = G::TypeInstance::class_peek_parent<GtkContainerClass>(GTK_CONTAINER_GET_CLASS(container));
		if (g_class->composite_name)
			result = g_class->composite_name(container, child);
	}
	return result;
}

void
Gtk::ContainerClass::set_child_property_proxy(GtkContainer *container, GtkWidget *child, guint property_id, const GValue *value, GParamSpec *pspec)
{
	Container *tmp_container = G::Object::pointer<Container>(container);
	if (tmp_container)
		tmp_container->do_set_child_property(child, property_id, value, pspec);
	else
	{
		GtkContainerClass *g_class = G::TypeInstance::class_peek_parent<GtkContainerClass>(GTK_CONTAINER_GET_CLASS(container));
		if (g_class->set_child_property)
			g_class->set_child_property(container, child, property_id, value, pspec);
	}
}

void
Gtk::ContainerClass::get_child_property_proxy(GtkContainer *container, GtkWidget *child, guint property_id, GValue *value, GParamSpec *pspec)
{
	Container *tmp_container = G::Object::pointer<Container>(container);
	if (tmp_container)
		tmp_container->do_get_child_property(child, property_id, value, pspec);
	else
	{
		GtkContainerClass *g_class = G::TypeInstance::class_peek_parent<GtkContainerClass>(GTK_CONTAINER_GET_CLASS(container));
		if (g_class->get_child_property)
			g_class->get_child_property(container, child, property_id, value, pspec);
	}
}

void
Gtk::ContainerClass::add_proxy(GtkContainer *container, GtkWidget *widget)
{
	Container *tmp_container = G::Object::pointer<Container>(container);
	if (tmp_container)
		tmp_container->on_add(*G::Object::wrap<Widget>(widget));
	else
	{
		GtkContainerClass *g_class = G::TypeInstance::class_peek_parent<GtkContainerClass>(GTK_CONTAINER_GET_CLASS(container));
		if (g_class->add)
			g_class->add(container, widget);
	}
}

void
Gtk::ContainerClass::remove_proxy(GtkContainer *container, GtkWidget *widget)
{
	Container *tmp_container = G::Object::pointer<Container>(container);
	if (tmp_container)
		tmp_container->on_remove(*G::Object::wrap<Widget>(widget));
	else
	{
		GtkContainerClass *g_class = G::TypeInstance::class_peek_parent<GtkContainerClass>(GTK_CONTAINER_GET_CLASS(container));
		if (g_class->remove)
			g_class->remove(container, widget);
	}
}

void
Gtk::ContainerClass::check_resize_proxy(GtkContainer *container)
{
	Container *tmp_container = G::Object::pointer<Container>(container);
	if (tmp_container)
		tmp_container->on_check_resize();
	else
	{
		GtkContainerClass *g_class = G::TypeInstance::class_peek_parent<GtkContainerClass>(GTK_CONTAINER_GET_CLASS(container));
		if (g_class->check_resize)
			g_class->check_resize(container);
	}
}

void
Gtk::ContainerClass::set_focus_child_proxy(GtkContainer *container, GtkWidget *widget)
{
	Container *tmp_container = G::Object::pointer<Container>(container);
	if (tmp_container)
	{
		tmp_container->on_set_focus_child(G::Object::wrap<Widget>(widget));
	}
	else
	{
		GtkContainerClass *g_class = G::TypeInstance::class_peek_parent<GtkContainerClass>(GTK_CONTAINER_GET_CLASS(container));
		if (g_class->set_focus_child)
			g_class->set_focus_child(container, widget);
	}
}

/*  Overridable methods
 */

void
Gtk::Container::do_forall(bool include_internals, GtkCallback callback, void *callback_data)
{
	GtkContainerClass *g_class = class_peek_parent<GtkContainerClass>(gtk_container_class());
	if (g_class->forall)
		g_class->forall(gtk_container(), include_internals, callback, callback_data);
}

GtkType
Gtk::Container::do_child_type()
{
	GtkType result = GTK_TYPE_INVALID;
	GtkContainerClass *g_class = class_peek_parent<GtkContainerClass>(gtk_container_class());
	if (g_class->child_type)
		result = g_class->child_type(gtk_container());
	return result;
}

char*
Gtk::Container::do_composite_name(GtkWidget *child)
{
	char *result = 0;
	GtkContainerClass *g_class = class_peek_parent<GtkContainerClass>(gtk_container_class());
	if (g_class->composite_name)
		result = g_class->composite_name(gtk_container(), child);
	return result;
}

void
Gtk::Container::do_set_child_property(GtkWidget *child, unsigned int property_id, const GValue *value, GParamSpec *pspec)
{
	GtkContainerClass *g_class = class_peek_parent<GtkContainerClass>(gtk_container_class());
	if (g_class->set_child_property)
		g_class->set_child_property(gtk_container(), child, property_id, value, pspec);
}

void
Gtk::Container::do_get_child_property(GtkWidget *child, unsigned int property_id, GValue *value, GParamSpec *pspec)
{
	GtkContainerClass *g_class = class_peek_parent<GtkContainerClass>(gtk_container_class());
	if (g_class->get_child_property)
		g_class->get_child_property(gtk_container(), child, property_id, value, pspec);
}

/*  Signal handlers
 */

void
Gtk::Container::on_add(Widget& widget)
{
	GtkContainerClass *g_class = class_peek_parent<GtkContainerClass>(gtk_container_class());
	if (g_class->add)
		g_class->add(gtk_container(), widget.gtk_widget());
}

void
Gtk::Container::on_remove(Widget& widget)
{
	GtkContainerClass *g_class = class_peek_parent<GtkContainerClass>(gtk_container_class());
	if (g_class->remove)
		g_class->remove(gtk_container(), widget.gtk_widget());
}

void
Gtk::Container::on_check_resize()
{
	GtkContainerClass *g_class = class_peek_parent<GtkContainerClass>(gtk_container_class());
	if (g_class->check_resize)
		g_class->check_resize(gtk_container());
}

void
Gtk::Container::on_set_focus_child(Widget *widget)
{
	GtkContainerClass *g_class = class_peek_parent<GtkContainerClass>(gtk_container_class());
	if (g_class->set_focus_child)
		g_class->set_focus_child(gtk_container(), *widget);
}

/* Properties
 */

const Gtk::Container::ResizeModePropertyType Gtk::Container::resize_mode_property("resize_mode");

const Gtk::Container::BorderWidthPropertyType Gtk::Container::border_width_property("border_width");

const Gtk::Container::ChildPropertyType Gtk::Container::child_property("child");

/*  Signals
 */

const Gtk::Container::AddSignalType Gtk::Container::add_signal("add");

const Gtk::Container::RemoveSignalType Gtk::Container::remove_signal("remove");

const Gtk::Container::CheckResizeSignalType Gtk::Container::check_resize_signal("check_resize");

const Gtk::Container::SetFocusChildSignalType Gtk::Container::set_focus_child_signal("set_focus_child");


