/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package test.javax.management.monitor;

import java.util.Set;

import junit.framework.TestCase;
import junit.framework.Assert;
import javax.management.MBeanServerFactory;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.Attribute;
import javax.management.NotificationListener;
import javax.management.Notification;
import javax.management.monitor.StringMonitor;
import javax.management.monitor.MonitorNotification;
import mx4j.log.*;

/**
 * Class StringMonitorTest, tests the String monitoring service
 *
 * @author <a href="mailto:tibu@users.sourceforge.net">Carlos Quiroz</a>
 * @version $Revision: 1.6 $
 */
public class StringMonitorTest extends TestCase {
    private MBeanServer server;

    /**
     * Dummy test interface
     */
    public interface TestClassMBean {
        public String getStr();
        public void setStr(String str);
    }

    public static class TestClass implements TestClassMBean {
        private String str;

        public TestClass(String str) {
            this.str = str;
        }

        public String getStr() {
            return str;
        }

        public void setStr(String str) {
            this.str = str;
        }
    }

    public static class MonitorListener implements NotificationListener {
        MonitorNotification lastNotification;

        public void handleNotification(Notification notification, Object handback) {
            if (notification instanceof MonitorNotification) {
                lastNotification = (MonitorNotification)notification;
            }
        }

        public MonitorNotification getLastNotification() {
            return lastNotification;
        }
    }

    /**
     * Constructor requested by the JUnit framework
     */
    public StringMonitorTest() {
        super("StringMonitor Test");
    }

    /**
     * Constructor requested by the JUnit framework
     */
    public StringMonitorTest(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        server = MBeanServerFactory.createMBeanServer("StringMonitor");
    }

    public void testLegacyMonitor() throws Exception {
        ObjectName name = new ObjectName("StringMonitor:name=monitorTarget");
        ObjectName monitorName = new ObjectName("StringMonitor:name=monitor1");
        try {
            TestClass a = new TestClass("test");
            server.registerMBean(a, name);
            StringMonitor monitor = new StringMonitor();
            server.createMBean("javax.management.monitor.StringMonitor", monitorName, null);
            server.setAttribute(monitorName, new Attribute("ObservedObject", name));
            server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Str"));
            server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
            server.setAttribute(monitorName, new Attribute("StringToCompare", "test"));
            server.setAttribute(monitorName, new Attribute("NotifyMatch", new Boolean(true)));
            server.setAttribute(monitorName, new Attribute("NotifyDiffer", new Boolean(true)));
            MonitorListener listener = new MonitorListener();
            server.addNotificationListener(monitorName, listener, null, null);
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_MATCHED, listener.getLastNotification().getType());
            long sequence = listener.getLastNotification().getSequenceNumber();
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_MATCHED, listener.getLastNotification().getType());
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(monitorName, new Attribute("StringToCompare", "test2"));
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_DIFFERED, listener.getLastNotification().getType());
            server.setAttribute(monitorName, new Attribute("NotifyMatch", new Boolean(false)));
            server.setAttribute(monitorName, new Attribute("StringToCompare", "test"));
            sequence = listener.getLastNotification().getSequenceNumber();
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_DIFFERED, listener.getLastNotification().getType());
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(monitorName, new Attribute("NotifyMatch", new Boolean(true)));
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_MATCHED, listener.getLastNotification().getType());
            assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
        } finally {
            server.unregisterMBean(name);
            server.unregisterMBean(monitorName);
        }
    }

    public void testMonitor() throws Exception {
        ObjectName name1 = new ObjectName("StringMonitor:name=monitorTarget1");
        ObjectName name2 = new ObjectName("StringMonitor:name=monitorTarget2");
        ObjectName monitorName = new ObjectName("StringMonitor:name=monitor1");
        try {
            TestClass a = new TestClass("test");
            server.registerMBean(a, name1);
            TestClass b = new TestClass("test");
            server.registerMBean(b, name2);

            StringMonitor monitor = new StringMonitor();

            server.createMBean("javax.management.monitor.StringMonitor", monitorName, null);
            server.invoke(monitorName, "addObservedObject", new Object [] {name1}, new String [] {"javax.management.ObjectName"});
            server.invoke(monitorName, "addObservedObject", new Object [] {name2}, new String [] {"javax.management.ObjectName"});
            server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Str"));
            server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
            server.setAttribute(monitorName, new Attribute("StringToCompare", "test"));
            server.setAttribute(monitorName, new Attribute("NotifyMatch", new Boolean(true)));
            server.setAttribute(monitorName, new Attribute("NotifyDiffer", new Boolean(true)));
            MonitorListener listener = new MonitorListener();
            server.addNotificationListener(monitorName, listener, null, null);
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_MATCHED, listener.getLastNotification().getType());
            long sequence = listener.getLastNotification().getSequenceNumber();
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_MATCHED, listener.getLastNotification().getType());
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(monitorName, new Attribute("StringToCompare", "test2"));
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_DIFFERED, listener.getLastNotification().getType());

            // check match on second monitored object
            server.setAttribute(name2, new Attribute("Str","test2"));
            synchronized(this) {
                wait(1500);
            }
            System.out.println("second test");
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_MATCHED,listener.getLastNotification().getType());

            server.setAttribute(name2, new Attribute("Str","test2n"));
            synchronized(this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_DIFFERED,listener.getLastNotification().getType());

            server.setAttribute(name1,new Attribute("Str","test"));
            server.setAttribute(name2,new Attribute("Str","test"));
            server.setAttribute(monitorName, new Attribute("NotifyMatch", new Boolean(false)));
            server.setAttribute(monitorName, new Attribute("StringToCompare", "test"));

            sequence = listener.getLastNotification().getSequenceNumber();
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_DIFFERED, listener.getLastNotification().getType());
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(monitorName, new Attribute("NotifyMatch", new Boolean(true)));
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.STRING_TO_COMPARE_VALUE_MATCHED, listener.getLastNotification().getType());
            assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
        } finally {
            server.unregisterMBean(name1);
            server.unregisterMBean(name2);
            server.unregisterMBean(monitorName);
        }
    }

    public void testErrors() throws Exception {
        ObjectName name = new ObjectName("StringMonitor:name=monitorTarget");
        ObjectName monitorName = new ObjectName("StringMonitor:name=monitor1");
        try {
            TestClass a = new TestClass("test");
            StringMonitor monitor = new StringMonitor();
            server.createMBean("javax.management.monitor.StringMonitor", monitorName, null);
            server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
            MonitorListener listener = new MonitorListener();
            server.addNotificationListener(monitorName, listener, null, null);
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});
            synchronized (this) {
                wait(1500);
            }
            long sequence = listener.getLastNotification().getSequenceNumber();
            // thrown if no MBean has been registered to the name or no name
            assertEquals(MonitorNotification.OBSERVED_OBJECT_ERROR, listener.getLastNotification().getType());
            synchronized (this) {
                wait(1500);
            }

            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(monitorName, new Attribute("ObservedObject", name));
            synchronized (this) {
                wait(1500);
            }
            assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
            server.registerMBean(a, name);
            synchronized (this) {
                wait(1500);
            }
            // objectname is ok but no attribute set
            assertEquals(MonitorNotification.OBSERVED_ATTRIBUTE_ERROR, listener.getLastNotification().getType());
            sequence = listener.getLastNotification().getSequenceNumber();
            synchronized (this) {
                wait(1500);
            }
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            sequence = listener.getLastNotification().getSequenceNumber();
            // wrong attribute name
            server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Str2"));
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.OBSERVED_ATTRIBUTE_ERROR, listener.getLastNotification().getType());
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            sequence = listener.getLastNotification().getSequenceNumber();
            // right attribute name but no string to compare
            server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Str"));
            synchronized (this) {
                wait(1500);
            }
            assertEquals(MonitorNotification.OBSERVED_ATTRIBUTE_ERROR, listener.getLastNotification().getType());
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
        } finally {
            //server.unregisterMBean(name);
            //server.unregisterMBean(monitorName);
        }
    }

    public void testStartStopSequence() throws Exception {
        ObjectName name = new ObjectName("StringMonitor:name=monitorTarget");
        ObjectName monitorName = new ObjectName("StringMonitor:name=monitor1");
        try {
            TestClass a = new TestClass("test");
            StringMonitor monitor = new StringMonitor();
            server.registerMBean(a, name);
            server.createMBean("javax.management.monitor.StringMonitor", monitorName, null);
            server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
            MonitorListener listener = new MonitorListener();
            server.addNotificationListener(monitorName, listener, null, null);
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});
            server.invoke(monitorName, "stop", new Object[] {}, new String[] {});
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});
        }
        finally {
            server.unregisterMBean(name);
            server.unregisterMBean(monitorName);
        }
    }
    public void testAddRemove() throws Exception{
        ObjectName name1 = new ObjectName("StringMonitor:name=monitorTarget1");
        ObjectName name2 = new ObjectName("StringMonitor:name=monitorTarget2");
        ObjectName name3 = new ObjectName("StringMonitor:name=monitorTarget3");
        ObjectName monitorName = new ObjectName("StringMonitor:name=monitor1");
        try {
            TestClass a = new TestClass("test1");
            server.registerMBean(a, name1);
            TestClass b = new TestClass("test2");
            server.registerMBean(b, name2);
            TestClass c = new TestClass("test3");
            server.registerMBean(c, name3);
            server.createMBean("javax.management.monitor.StringMonitor", monitorName, null);
            //server.setAttribute(monitorName, new Attribute("ObservedObject", name));
            server.invoke(monitorName, "addObservedObject", new Object[] {name1}, new String [] {"javax.management.ObjectName"});
            server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
            MonitorListener listener = new MonitorListener();
            server.addNotificationListener(monitorName, listener, null, null);
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});

            // should have one observed object
            ObjectName [] names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 1);
            assertEquals(names[0], name1);

            // add another, make sure we have two
            server.invoke(monitorName, "addObservedObject", new Object[] {name2}, new String [] {"javax.management.ObjectName"});
            names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 2);
            assertEquals(names[0],name1);
            assertEquals(names[1],name2);

            // remove one, make sure we're back to one
            server.invoke(monitorName, "removeObservedObject", new Object[] {name2}, new String [] {"javax.management.ObjectName"});
            names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 1);
            assertEquals(names[0],name1);

            // now add back a different one
            server.invoke(monitorName, "addObservedObject", new Object[] {name3}, new String [] {"javax.management.ObjectName"});
            names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 2);
            assertEquals(names[0],name1);
            assertEquals(names[1],name3);

            // try to remove one that doesn't exist.  Should NOT get IllegalArgumentException since JMX 1.2
            boolean errorFound = false;
            try{
                server.invoke(monitorName, "removeObservedObject", new Object[] {name2}, new String [] {"javax.management.ObjectName"});
            }catch(javax.management.RuntimeMBeanException mbe){
                assertTrue(mbe.getTargetException() instanceof IllegalArgumentException);
                errorFound = true;
            }
            assertFalse(errorFound);

            // try to add one that already exists.  Should NOT get IllegalArgumentException since JMX 1.2
            errorFound = false;
            try{
                server.invoke(monitorName, "addObservedObject", new Object[] {name3}, new String [] {"javax.management.ObjectName"});
            }catch(javax.management.RuntimeMBeanException mbe){
                assertTrue(mbe.getTargetException() instanceof IllegalArgumentException);
                errorFound = true;
            }
            assertFalse(errorFound);
        }finally {
            server.unregisterMBean(name1);
            server.unregisterMBean(name2);
            server.unregisterMBean(name3);
            server.unregisterMBean(monitorName);
        }
    }
}

