/*
 * Copyright (c) 2002, 2003, 2004 SuSE GmbH Nuernberg, Germany.
 * Author: Thorsten Kukuk <kukuk@suse.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define _GNU_SOURCE

#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <security/_pam_macros.h>
#include <security/pam_modules.h>

#include "public.h"
#include "logindefs.h"
#include "read-files.h"

char *files_etc_dir = "/etc";

static void add_other_module(options_t *options, const char *name);

static void
parse_option (const char *argv, const char *type, options_t *options)
{
  if (argv == NULL || argv[0] == '\0')
    return;

  if (strcasecmp (argv, "nullok") == 0)
    options->nullok = 1;
  else if (strcasecmp (argv, "not_set_pass") == 0)
    options->not_set_pass = 1;
  else if (strcasecmp (argv, "use_first_pass") == 0)
    options->use_first_pass = 1;
  else if (strcasecmp (argv, "use_authtok") == 0)
    options->use_authtok = 1;
  else if (strcasecmp (argv, "set_secrpc") == 0)
    options->secrpc_flag = 1;
  else if (strcasecmp (argv, "md5") == 0)
    {
      __pam_log (LOG_DEBUG, "md5 option is obsolete, please use /etc/default/passwd");
      options->use_crypt = MD5;
    }
  else if (strcasecmp (argv, "bigcrypt") == 0)
    {
      __pam_log (LOG_DEBUG, "bigcrypt option is obsolete, please use /etc/default/passwd");
#if defined(HAVE_BIGCRYPT)
      options->use_crypt = BIGCRYPT;
#else
      __pam_log (LOG_ERR, "No bigcrypt support compiled in");
#endif
    }
  else if (strcasecmp (argv, "bf") == 0 ||
	   strcasecmp (argv, "blowfish") == 0)
    {
      __pam_log (LOG_DEBUG, "blowfish option is obsolete, please use /etc/default/passwd");
#if defined(HAVE_CRYPT_GENSALT_RN)
      options->use_crypt = BLOWFISH;
#else
      __pam_log (LOG_ERR, "No blowfish support compiled in");
#endif
    }
  else if (strncasecmp (argv, "rounds=", 7) == 0)
    {
      __pam_log (LOG_DEBUG, "rounds option is obsolete, please use /etc/default/passwd");
#if defined(HAVE_CRYPT_GENSALT_RN)
      options->crypt_rounds = atoi (&argv[7]);
#else
      __pam_log (LOG_ERR, "Option \"%s\": No blowfish support compiled in",
		 argv);
#endif
    }
  else if (strncasecmp (argv, "crypt_rounds=", 13) == 0)
    {
#if defined(HAVE_CRYPT_GENSALT_RN)
      options->crypt_rounds = atoi (&argv[13]);
#else
      __pam_log (LOG_ERR, "Option \"%s\": No blowfish support compiled in",
		 argv);
#endif
    }
  else if (strncasecmp (argv, "nisdir=", 7) == 0)
    options->nisdir = strdup (&argv[7]);
  else if (strcasecmp (argv, "debug") == 0)
    {
      if (strcasecmp (type, "session") == 0)
	options->log_level = LOG_DEBUG;
      else
	options->debug = 1;
    }
  else if (strcasecmp (argv, "trace") == 0 &&
	   strcasecmp (type, "session") == 0)
    options->log_level = LOG_NOTICE;
  else if (strcasecmp (argv, "none") == 0 &&
	   strcasecmp (type, "session") == 0)
    options->log_level = -1;
  else if (strcasecmp (argv, "use_ldap") == 0)
    {
      add_other_module(options, "ldap");
    }
  else if (strcasecmp (argv, "use_krb5") == 0)
    {
      add_other_module(options, "krb5");
    }
  else if (strncasecmp (argv, "call_modules=", 13) == 0)
    {
      char *copy = strdup (&argv[13]), *arg;

      arg = copy;
      do
	{
	  char *cp = strchr (arg, ',');
	  if (cp)
	    *cp++ = '\0';

	  add_other_module (options, arg);
	  arg = cp;
	}
      while (arg);
      free (copy);
    }
  else
    __pam_log (LOG_ERR, "Unknown option: `%s'", argv);
}

static void
add_other_module(options_t *options, const char *name)
{
  unsigned int i = 0;

  if (options->use_other_modules)
    {
      while (options->use_other_modules[i])
        i++;
    }

  options->use_other_modules = realloc(options->use_other_modules,
  				(i + 2) * sizeof(char *));
  options->use_other_modules[i++] = strdup(name);
  options->use_other_modules[i] = NULL;
}

/* Load the config file (/etc/security/pam_unix2.conf)  */
static int
load_config (const char *configfile, const char *type, options_t *options)
{
  FILE *fp;
  char *buf = NULL;
  size_t buflen = 0;

  /* Set some default values, which could be overwritten later.  */
  options->use_crypt = NONE;

  fp = fopen (configfile, "r");
  if (NULL == fp)
    return -1;

  while (!feof (fp))
    {
      char *cp, *tmp;
#if defined(HAVE_GETLINE)
      ssize_t n = getline (&buf, &buflen, fp);
#elif defined (HAVE_GETDELIM)
      ssize_t n = getdelim (&buf, &buflen, '\n', fp);
#else
      ssize_t n;

      if (buf == NULL)
        {
          buflen = 8096;
          buf = malloc (buflen);
        }
      buf[0] = '\0';
      fgets (buf, buflen - 1, fp);
      if (buf != NULL)
        n = strlen (buf);
      else
        n = 0;
#endif /* HAVE_GETLINE / HAVE_GETDELIM */
      cp = buf;

      if (n < 1)
        break;

      tmp = strchr (cp, '#');  /* remove comments */
      if (tmp)
        *tmp = '\0';
      while (isspace ((int)*cp))    /* remove spaces and tabs */
        ++cp;
      if (*cp == '\0')        /* ignore empty lines */
        continue;

      if (cp[strlen (cp) - 1] == '\n')
        cp[strlen (cp) - 1] = '\0';

      if (strncasecmp (cp, type, strlen (type)) == 0 &&
	  (isspace ((int)cp[strlen (type)]) || cp[strlen (type)] == ':'))
	{
	  char *c;
          /* We found the line we needed */

	  c = strchr (cp, ':');
	  if (c != NULL)
	    ++c;

	  while (c != NULL && strlen (c) > 0)
	    {
	      /* remove all spaces */
	      while (isspace (*c))
		++c;

	      cp = c;
	      while (!isspace (*c) && *c != '\0')
		++c;

	      if (*c == '\0')
		parse_option (cp, type, options);
	      else
		{
		  *c = '\0';
		  parse_option (cp, type, options);
		  ++c;
		  cp = c;
		}
	    }
        }
    }
  fclose (fp);

  if (buf)
    free (buf);

  return 0;
}



int
get_options (options_t *options, const char *type,
	     int argc, const char **argv)
{
  if (load_config (SYSCONFDIR"/pam_unix2.conf", type, options) < 0)
    return -1;

  /* Parse parameters for module */
  for ( ; argc-- > 0; argv++)
    parse_option (*argv, type, options);

  return 0;
}
