/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: vorbis_meta.cpp,v 1.18 2004/03/12 23:33:55 robert Exp $

----------------------------------------------------------------------------*/
//---------------------------------------------------------------------------
// This code is based on vorbis.cpp and vorbis.cpp from FreeAmp. EMusic.com
// has released this code into the Public Domain. 
// (Thanks goes to Brett Thomas, VP Engineering Emusic.com)
//---------------------------------------------------------------------------
// Portions (c) Copyright Kristian G. Kvilekval, and permission to use in LGPL
// library granted on February 25th, 2003
#include "../config.h"
#ifdef HAVE_OGGVORBIS

#ifdef WIN32
#	if _MSC_VER == 1200
#		pragma warning(disable:4786)
#	endif
#else
#	include <unistd.h>
#endif

#include <stdio.h>
#include <math.h>
#include <assert.h>
#include <string>
#include <locale.h>
#include <ctype.h>
#include <map>
#include <algorithm>

using namespace std;

#include <musicbrainz/mb_c.h>

#include "metadata.h"
#include "vorbis_meta.h"
#include "vorbis/vorbisfile.h"
#include "vcedit.h"
#include "utf8/utf8.h"

typedef multimap<string,string> tagmap_t;

static bool 
add_comment(tagmap_t & tagmap, 
            const string& tag, const string&val, bool singleton = true)
{
    if (val.size() == 0) return false;
    // case 1: when a singleton, erase all existing values.
    if (singleton) 
        tagmap.erase (tag);

    tagmap.insert (pair<string,string>(tag,val));
    return true;
}

/** Find a value from in the tag table returning true when it exists 
 *  
 */
static bool
get_comment(tagmap_t& tagmap, const string& tag, string&val)
{
    tagmap_t::iterator it;
    it = tagmap.find (tag);
    if (it != tagmap.end()) 
    {
        val = (*it).second;
        return true;
    }
    return false;
}

/** Load all tags from the comment structure into the given map. */
static void 
load_tags (vorbis_comment *vc, tagmap_t& tagmap) 
{
    char *decoded;
    string entry;
    string key;
    string val;

    for (int i=0; i<vc->comments; i++) 
    {
        entry = vc->user_comments[i];

        string::size_type sep = entry.find ('=');
        if (sep != string::npos)
        {
            key = entry.substr(0, sep);

            utf8_decode (entry.substr(sep+1, string::npos).c_str(), &decoded);
            val = decoded;
            if (decoded)
            {
                 free (decoded);
                 decoded = NULL;
            }

            transform(key.begin(),key.end(),key.begin(),(int(*)(int))&tolower);
            tagmap.insert (pair<string,string>(key, val));
        }
    }
}

/** Save the tags in the map to the given comment structure.
 *  NOTE:  This will not enforce vorbis tag recommenation singleton
 * or values, so use with care 
 */
static void 
save_tags (vorbis_comment *vc, tagmap_t& tagmap) 
{

    string comment;
    string key;
    for (tagmap_t::iterator it = tagmap.begin(); it != tagmap.end(); it++) 
    {
        char *encoded;

        key = (*it).first;
        transform(key.begin(),key.end(),key.begin(),(int(*)(int))&toupper);

        utf8_encode ((*it).second.c_str(), &encoded);
        comment = key + '=' + encoded;
        free (encoded);
        
        vorbis_comment_add (vc, (char*)comment.c_str());
    }
}

/** Write out vorbis metadata.    */
bool Vorbis::write(const string &fileName, const Metadata & metadata, bool clear)
{
    char      dummy[20];
    char     *ptr;
    bool     writetags;
    string   temp;
    // We will support only id3-like tags.  For a more complete list see
    //  http://reactor-core.org/ogg-tag-standard.html

    ptr = strrchr(fileName.c_str(), '.');
    if (ptr == NULL)
        return false;

    if (strcmp(ptr, ".ogg"))
        return false;

    // Rewrite the file with the updated tags.
    vcedit_state *state;
    vorbis_comment *vc;

    state = vcedit_new_state();
    FILE *in = fopen (fileName.c_str(), "rb");
    if (in == NULL)
    {
        errString = string("Failed to open ") + fileName;
        vcedit_clear(state);
        return false;
    }
	if(vcedit_open(state, in) < 0)
    {
        errString = string("Failed to open ") + fileName +
                    string("as vorbis ") + string(vcedit_error(state));
        fclose (in);
        vcedit_clear(state);
        return false;
    }
    tagmap_t tagmap;

    vc = vcedit_comments(state);
    if (!clear)
        load_tags (vc, tagmap);

    add_comment(tagmap, "title", metadata.track.c_str());
    add_comment(tagmap, "artist", metadata.artist.c_str());
    add_comment(tagmap, "album", metadata.album.c_str());
    add_comment(tagmap, "musicbrainz_sortname", metadata.sortName.c_str());
    add_comment(tagmap, "musicbrainz_trackid", metadata.trackId.c_str());
    add_comment(tagmap, "musicbrainz_albumid", metadata.albumId.c_str());
    if (metadata.albumType != eAlbumType_Error)
    {
        convertFromAlbumType(metadata.albumType, temp);
        add_comment(tagmap, "musicbrainz_albumtype", temp);
    }
    if (metadata.albumStatus != eAlbumStatus_Error)
    {
        convertFromAlbumStatus(metadata.albumStatus, temp);
        add_comment(tagmap, "musicbrainz_albumstatus", temp);
    }
    add_comment(tagmap, "musicbrainz_artistid", metadata.artistId.c_str());
    add_comment(tagmap, "musicbrainz_trmid", metadata.fileTrm.c_str());
    if (metadata.variousArtist || !metadata.albumArtistId.empty())
    {
        if (metadata.variousArtist)
             add_comment(tagmap, "musicbrainz_albumartistid", 
                         MBI_VARIOUS_ARTIST_ID);
        else
             add_comment(tagmap, "musicbrainz_albumartistid", 
                         metadata.albumArtistId.c_str());
    }
    if (metadata.trackNum > 0)
    {
        sprintf (dummy, "%d", metadata.trackNum);
        add_comment(tagmap, "tracknumber", dummy);
    }
    if (metadata.releaseYear > 0)
    {
        char temp[16];
    
        sprintf(temp, "%04d-%02d-%02d", metadata.releaseYear, metadata.releaseMonth, metadata.releaseDay);
        add_comment(tagmap, "date", temp);
    }
    if (metadata.releaseCountry.length())
        add_comment(tagmap, "releasecountry", metadata.releaseCountry);

    vorbis_comment_clear (vc);

    save_tags (vc, tagmap);

    char newpath[1025];
    strncpy (newpath, fileName.c_str(), 1025);
    FILE *out = NULL;

    writetags = false;
    ptr = strrchr(newpath, '.');
    if (ptr)
    {
        strcpy(ptr, "XXXXX");
        out = fopen (newpath, "wb");

        if(out == NULL || vcedit_write(state, out) < 0)
        {
            errString = string("Failed to write comments to output file: ") +
                        string(vcedit_error(state));
        }
        else
            writetags = true;
    }

	if (in) fclose (in);
    if (out) fclose (out);

    vcedit_clear(state);

    if (writetags)
    {
        unlink(fileName.c_str());
        if (rename (newpath, fileName.c_str()) < 0)
        {
            errString = string("Failed rename output file: ") +
                        string(newpath);
            return false;
        }
    }
    return true;
}

bool Vorbis::read(const string &fileName, Metadata &metadata)
{
    char     *ptr;
    // We will support only id3-like tags.  For a more complete list see
    //  http://reactor-core.org/ogg-tag-standard.html

    ptr = strrchr(fileName.c_str(), '.');
    if (ptr == NULL)
        return false;

    if (strcmp(ptr, ".ogg"))
        return false;

    vcedit_state *state;
    vorbis_comment *vc;
    state = vcedit_new_state();

    FILE *in = fopen (fileName.c_str(), "rb");
    if (in == NULL)
    {
        vcedit_clear(state);
        return false;
    }
    if(vcedit_open(state, in) < 0)
    {
        fclose (in);
        vcedit_clear(state);
        return false;
    }

    tagmap_t tagmap;
    string val;

    vc = vcedit_comments(state);
    load_tags (vc, tagmap);

    if (get_comment(tagmap, "title", val))
        metadata.track = string(val.c_str());
    if (get_comment(tagmap, "artist", val))
        metadata.artist = string(val.c_str());
    if (get_comment(tagmap, "album", val))
        metadata.album = string(val.c_str());
    if (get_comment(tagmap, "tracknumber", val))
        metadata.trackNum = atoi(val.c_str());
    if (get_comment(tagmap, "musicbrainz_trackid", val))
        metadata.trackId = string(val.c_str());
    if (get_comment(tagmap, "musicbrainz_artistid", val))
        metadata.artistId = string(val.c_str());
    if (get_comment(tagmap, "musicbrainz_albumid", val))
        metadata.albumId = string(val.c_str());
    if (get_comment(tagmap, "musicbrainz_albumtype", val))
        metadata.albumType = convertToAlbumType(val.c_str());
    if (get_comment(tagmap, "musicbrainz_albumstatus", val))
        metadata.albumStatus = convertToAlbumStatus(val.c_str());
    if (get_comment(tagmap, "musicbrainz_sortname", val))
        metadata.sortName = string(val.c_str());
    if (get_comment(tagmap, "musicbrainz_albumartistid", val))
    {
        metadata.variousArtist = strcasecmp(val.c_str(), MBI_VARIOUS_ARTIST_ID) == 0;
        metadata.albumArtistId = string(val.c_str());
    }
    if (get_comment(tagmap, "musicbrainz_trmid", val))
        metadata.fileTrm = string(val.c_str());

    if (get_comment(tagmap, "date", val))
    {
        int month, day, year;
    
        if (sscanf(val.c_str(), "%d-%d-%d", &year, &month, &day) == 3)
        {
             metadata.releaseYear = year;
             metadata.releaseMonth = month;
             metadata.releaseDay = day;
        }
    }
    if (get_comment(tagmap, "releasecountry", val))
        metadata.releaseCountry = string(val.c_str());

    vcedit_clear(state);
    fclose(in);

    metadata.fileFormat = "ogg";

    return true;
}
#endif
