/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

package com.sap.dbtech.jdbc.translators;

import java.io.*;
import java.sql.*;
import com.sap.dbtech.jdbc.exceptions.*;
import com.sap.dbtech.util.*;
import com.sap.dbtech.vsp00.DataType;

/**
 *
 */
abstract class StreamTranslator
    extends BinaryDataTranslator
{
    /**
     * LongTranslator constructor comment.
     * @param mode int
     * @param ioType int
     * @param dataType int
     * @param len int
     * @param ioLen int
     * @param bufpos int
     */
    protected StreamTranslator(int mode, int ioType, int dataType, int len, int ioLen, int bufpos) {
        super(mode, ioType, dataType, len, ioLen, bufpos);
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public java.io.InputStream getAsciiStream (
        SQLParamController controller,
        StructuredMem mem,
        StructuredMem longData) throws SQLException
    {
        throw new ConversionExceptionSapDB (
            MessageTranslator.translate(MessageKey.ERROR_ASCIIREADFROMLONG)
         );
    }

    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public java.io.InputStream getBinaryStream (
        SQLParamController controller,
        StructuredMem mem,
        StructuredMem longData) throws SQLException
    {
        throw new ConversionExceptionSapDB (
            MessageTranslator.translate(MessageKey.ERROR_BINARYREADFROMLONG)
         );
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public java.sql.Blob getBlob (
        SQLParamController controller,
        StructuredMem mem,
        StructuredMem longData) throws SQLException
    {
        return this.getLob (controller, mem, longData);
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public java.io.Reader getCharacterStream (
        SQLParamController controller,
        StructuredMem mem,
        StructuredMem longData) throws SQLException
    {
        java.io.InputStream byteStream = this.getAsciiStream(controller, mem, longData);
        if (byteStream == null) {
            return null;
        }
        else {
            return new RawByteReader(byteStream);
        }
    }

    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public java.sql.Clob getClob (
        SQLParamController controller,
        StructuredMem mem,
        StructuredMem longData) throws SQLException
    {
        return this.getLob (controller, mem, longData);
    }
    /**
     *
     * @return boolean
     */
    public int getColumnDisplaySize ()
    {
      switch (this.dataType) {
         case DataType.Dstruni:
         case DataType.Dlonguni:
            return 1073741824;
         default:
            return 2147483647;
        }
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public GetvalLob getLob (
        SQLParamController controller,
        StructuredMem mem,
        StructuredMem longData) throws SQLException
    {
        GetvalLob result = null;
        byte [] descriptor;

        if (!this.isNull (controller, mem)) {
            descriptor = mem.getBytes (this.bufpos, this.logicalLength);
            result = new GetvalLob (controller.getConnectionSapDB (),
                descriptor, longData, this.dataType);
        }
        return result;
    }
    /**
     *
     * @return boolean
     */
    public int getPrecision ()
    {
        return Integer.MAX_VALUE;
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public java.io.InputStream getStream (
        SQLParamController controller,
        StructuredMem mem,
        StructuredMem longData) throws SQLException
    {
        java.io.InputStream result = null;
        AbstractGetval getval = null;
        byte [] descriptor;

        if (!this.isNull (controller, mem)) {
            descriptor = mem.getBytes (this.bufpos, this.logicalLength);
            if (com.sap.dbtech.jdbc.DriverSapDB.isJDBC20) {
                getval = new GetvalLob (controller.getConnectionSapDB (),
                    descriptor, longData, this.dataType);
            }
            else {
                getval = new Getval (controller.getConnectionSapDB (),
                    descriptor, longData, this.dataType);
            }
            result = getval.getAsciiStream ();
        }
        return result;
    }

    // JDBC Standard Conformance requires that TIMESTAMP values can be
    // retrieved from here, too.
    public Timestamp getTimestamp (
        SQLParamController controller,
        StructuredMem mem,
        java.util.Calendar cal)
    throws SQLException
    {
        String strValue = this.getString (controller, mem);
        if (strValue == null) {
            return null;
        }
        try {
            Timestamp ts = Timestamp.valueOf (strValue);
            if (cal == null)
              return ts;
            else {
              cal.setTime(ts);
              Timestamp result = DBTechTranslator.CalendarUtil.getTimestampFromCalendar(cal, ts.getNanos());
              return result;
            }
        }
        catch (IllegalArgumentException tsExc) {
          // ignore and try as date
          try {
              Date dateValue = Date.valueOf (strValue);
            if (cal == null)
              return new Timestamp (dateValue.getTime ());
            else {
              cal.setTime(dateValue);
              return DBTechTranslator.CalendarUtil.getTimestampFromCalendar(cal,0);
            }
          }
          catch (IllegalArgumentException dateExc) {
              throw newParseException (strValue, "Timestamp");
          }
       }
    }

    /**
     *
     * @return java.lang.String
     */
    public String getString (
        SQLParamController controller,
        StructuredMem mem) throws SQLException
    {
        final int bufSizeC = 4096;
        Reader reader;
        StringBuffer result = new StringBuffer ();

        reader = this.getCharacterStream (controller, mem, controller.getReplyData());
        if (reader == null) {
            return null;
        }
        try {
            char [] buf = new char [4096];
            int charsRead;

            while ((charsRead = reader.read (buf)) > 0) {
                result.append (new String (buf, 0, charsRead));
                if (charsRead < bufSizeC) {
                    break;
                }
            }
        }
        catch (StreamIOException streamExc) {
            throw streamExc.getSqlException ();
        }
        catch (IOException exc) {
            throw new JDBCDriverException (exc.getMessage (), controller);
        }
        return result.toString ();
    }
    /**
     *
     * @return boolean
     */
    public boolean isCaseSensitive () {
        return true;
    }
    /**
     * putSpecific method comment.
     */
    protected void putSpecific(
        com.sap.dbtech.jdbc.packet.DataPart dataPart,
        Object data)
    {
        Putval putval = (Putval) data;
        putval.putDescriptor (dataPart, this.bufpos);
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public Object transAsciiStreamForInput (
        java.io.InputStream stream, int length) throws SQLException
    {
        throw new ConversionExceptionSapDB (
            MessageTranslator.translate(MessageKey.ERROR_ASCIIPUTTOLONG)
        );

    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public Object transBinaryStreamForInput(
    java.io.InputStream stream, int length) throws SQLException {
        throw new ConversionExceptionSapDB(
        MessageTranslator.translate(MessageKey.ERROR_BINARYPUTTOLONG)
        );
    }

    /**
     * Translates a byte array. This is done only in derived classes
     * that accept byte arrays (this one may be a BLOB or a CLOB,
     * and so does not decide about it).
     * @param val The byte array to bind.
     * @return The Putval instance created for this one.
     */
    public Object  transBytesForInput (byte [] val)
        throws SQLException
    {
		throw new ConversionExceptionSapDB
			(MessageTranslator.translate
				(MessageKey.ERROR_BINARYPUTTOLONG));

    }
    
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public Object transBlobForInput (
        java.sql.Blob blob) throws SQLException
    {
        if (blob == null) {
            return null;
        }
        return new Putval (blob.getBinaryStream(),-1, bufpos);
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public Object transClobForInput (
        java.sql.Clob clob) throws SQLException
    {
        if (clob == null) {
            return null;
        }
        return new Putval (clob.getAsciiStream(),-1, bufpos);
    }
    /**
     *
     * @param obj java.lang.Object
     */
    public Object transSpecificForInput (Object obj)
        throws SQLException
    {
        Object result = null;

        if (obj instanceof InputStream) {
            result = this.transAsciiStreamForInput ((InputStream) obj, -1);
        }
        return result;
    }
    /**
     *
     * @return java.io.InputStream
     * @exception java.sql.SQLException The exception description.
     */
    public Object transStreamForInput (
        java.io.InputStream stream, int length) throws SQLException
    {
        if (stream == null) {
            return null;
        }
        else {
            return new Putval (stream, length, bufpos);
        }
    }
   
    public Object transStringForInput (String val)
        throws SQLException
    {
		throw new ConversionExceptionSapDB (
			MessageTranslator.translate(MessageKey.ERROR_ASCIIPUTTOLONG)
		);
    }
    
	public Object cloneObjectForBatch(Object object)
	{
		if(object == null) {
			return null;
		} else {
			try {
				Putval pv = (Putval)object;
				return pv.cloneForBatch();				
			} catch(ClassCastException ccex) {
				return object;
			}
		}
	}
    
}
