/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

package com.sap.dbtech.jdbc.translators;

import java.sql.*;
import com.sap.dbtech.jdbc.*;
import com.sap.dbtech.jdbc.exceptions.NotImplemented;
import com.sap.dbtech.vsp00.*;
import com.sap.dbtech.util.MessageKey;
import com.sap.dbtech.util.MessageTranslator;

/**
 *
 */
public abstract class TranslatorFactory
{
/**
 *
 * @return com.sap.dbtech.jdbc.translators.DBTechTranslator
 * @param mode int
 * @param ioType int
 * @param dataType int
 * @param frac int
 * @param len int
 * @param ioLen int
 * @param bufpos int
 * @param isDBProcedure
 * @exception java.sql.SQLException The exception description.
 */
public static DBTechTranslator create (
    int mode,
    int ioType,
    int dataType,
    int frac,
    int len,
    int ioLen,
    int bufpos,
    boolean spaceoption,
    boolean isDBProcedure, 
	DBProcParameterInfo procParamInfo
    )
        throws SQLException
{
    DBTechTranslator result = null;
    
    switch (dataType) {
        case DataType.Dcha:
        case DataType.Dche:
        case DataType.Dvarchara:
        case DataType.Dvarchare: {
          if (spaceoption)
            result = new SpaceoptionStringTranslator (mode, ioType, dataType, len, ioLen, bufpos);
          else
            result = new StringTranslator (mode, ioType, dataType, len, ioLen, bufpos);
            break;
        }
        case DataType.Dchb: {
			if(procParamInfo != null && procParamInfo.getType() == DBProcParameterInfo.STRUCTURE) {
				result = new StructureTranslator(mode, ioType, dataType, len, ioLen, bufpos, false);
			} else {        	
				result = new BytesTranslator (mode, ioType, dataType, len, ioLen, bufpos);
			}
			break; 
        }      
        case DataType.Dvarcharb:
            result = new BytesTranslator (mode, ioType, dataType, len, ioLen, bufpos);
            break;
        case DataType.Dboolean:
            result = new BooleanTranslator (mode, ioType, dataType, len, ioLen, bufpos);
            break;
        case DataType.Dtime:
            result = new TimeTranslator (mode, ioType, dataType, len, ioLen, bufpos);
            break;
        case DataType.Ddate:
            result = new DateTranslator (mode, ioType, dataType, len, ioLen, bufpos);
            break;
        case DataType.Dtimestamp:
            result = new TimestampTranslator (mode, ioType, dataType, len, ioLen, bufpos);
            break;
        case DataType.Dfixed:
        case DataType.Dfloat:
        case DataType.Dvfloat:
        case DataType.Dsmallint:
        case DataType.Dinteger:
            result = new NumericTranslator (mode, ioType, dataType, len, frac, ioLen, bufpos);
            break;
        case DataType.Dstra:
        case DataType.Dstre:
        case DataType.Dlonga:
        case DataType.Dlonge:
        	if(isDBProcedure) {
        		result = new ProcedureStreamTranslator (mode, ioType, dataType, len, ioLen, bufpos);
        	} else {
        		result = new AsciiStreamTranslator (mode, ioType, dataType, len, ioLen, bufpos);
        	}
            break;
        case DataType.Dstrb:
        case DataType.Dlongb:
            if(isDBProcedure) {
                result = new ProcedureStreamTranslator (mode, ioType, dataType, len, ioLen, bufpos);
            } else {
                result = new BinaryStreamTranslator (mode, ioType, dataType, len, ioLen, bufpos);
            }
            break;
        case DataType.Dunicode:
        case DataType.Dvarcharuni:{
          if (spaceoption)
             result = new SpaceoptionUnicodeStringTranslator (mode, ioType, dataType, len, ioLen, bufpos);
          else
             result = new UnicodeStringTranslator (mode, ioType, dataType, len, ioLen, bufpos);
          break;
        }
    case DataType.Dlonguni:
    case DataType.Dstruni:
        if(isDBProcedure) {
            result = new UnicodeProcedureStreamTranslator (mode, ioType, dataType, len, ioLen, bufpos);
        } else {
            result = new UnicodeStreamTranslator (mode, ioType, dataType, len, ioLen, bufpos);
        }
        break;
    case DataType.Dabaptabhandle:
        result = new ABAPStreamTranslator(mode, ioType, dataType, len, ioLen, bufpos);
        break;
    default:
            if (DriverSapDB.catchAllNotImplemented_C) {
                throw new NotImplemented(
                    MessageTranslator.translate(MessageKey.ERROR_NOTRANSLATOR,
                                                Integer.toString(dataType))
                    );
            }
            else {
                result = new BytesTranslator (mode, ioType, dataType, len, ioLen, bufpos);
            }
    }
    return result;
}
    /**
     *
     * @param mode
     *
     * @param ioType
     *
     * @param dataType
     *
     * @param frac
     *
     * @param len
     *
     * @param ioLen
     *
     * @param bufpos
     *
     * @return
     *
     * @exception SQLException
     */
    public static DBTechTranslator createForUnicode (
            int mode,
            int ioType,
            int dataType,
            int frac,
            int len,
            int ioLen,
            int bufpos,
            boolean spaceoption,
			boolean isDBProcedure,
			DBProcParameterInfo procParamInfo)
    throws SQLException
    {
        DBTechTranslator result = null;
      
       
        switch (dataType) {
            case DataType.Dcha:
            case DataType.Dche:
            case DataType.Dvarchara:
            case DataType.Dvarchare:{
              if (spaceoption)
                 result = new SpaceoptionUnicodeStringTranslator (mode, ioType, dataType, len, ioLen, bufpos);
              else
                 result = new UnicodeStringTranslator (mode, ioType, dataType, len, ioLen, bufpos);
              break;
            }
			case DataType.Dchb: {
				if(procParamInfo != null && procParamInfo.getType() == DBProcParameterInfo.STRUCTURE) {
					result = new StructureTranslator(mode, ioType, dataType, len, ioLen, bufpos, true);
				} else {        	
					result = new BytesTranslator (mode, ioType, dataType, len, ioLen, bufpos);
				}
				break; 
			}      
            case DataType.Dstra:
            case DataType.Dstre:
            case DataType.Dlonga:
            case DataType.Dlonge:
			        	if(isDBProcedure) {
			        		result = new UnicodeProcedureStreamTranslator (mode, ioType, dataType, len, ioLen, bufpos);
			        	} else {
	                result = new UnicodeStreamTranslator (mode, ioType, dataType, len, ioLen, bufpos);
	              } 
                break;
            case DataType.Dtime:
                result = new UnicodeTimeTranslator (mode, ioType, dataType, len, ioLen, bufpos);
                break;
            case DataType.Dtimestamp:
                result = new UnicodeTimestampTranslator (mode, ioType, dataType, len, ioLen, bufpos);
                break;
            case DataType.Ddate:
                result = new UnicodeDateTranslator (mode, ioType, dataType, len, ioLen, bufpos);
                break;
            case DataType.Dabaptabhandle:
				        result = new ABAPStreamTranslator(mode, ioType, dataType, len, ioLen, bufpos, true);
				break;
            default:
                result = TranslatorFactory.create (mode, ioType,
                        dataType, frac, len, ioLen, bufpos, spaceoption, isDBProcedure, procParamInfo);
                break;
        }
        return result;
    }

}
