/* ======================================================================
 * meshset.cc
 * 
 * This file is part of MeshIO, the general and extensible 3D mesh I/O
 * library.
 * Copyright (c) 1999, 2000 Niklas Elmqvist. All rights reserved.
 *
 * File created 1999-06-05 by Niklas Elmqvist <d97elm@dtek.chalmers.se>.
 * 
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * ======================================================================
 */

// -- System Includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// -- Local Includes
#include "meshio.hh"

using namespace MeshIO;

// -- Code Segment

void MeshIO::mioMSetInit(MeshSet *mset)
{
    // Initialize data members
    mset->list         = NULL;
    mset->materialList = NULL;
    mset->meshNum      = 0;
    mset->materialNum  = 0;
}

void MeshIO::mioMSetComputeNormals(MeshSet *mset)
{
    MeshNode *node;
    
    // Traverse all meshes in set
    for (node = mset->list; node != NULL; node = node->next) {
	
	// Compute normals
	mioMeshComputeNormals(node->mesh);
    }
}

void MeshIO::mioMSetAddMesh(MeshSet *mset, Mesh *mesh)
{
    MeshNode *node = (MeshNode *) malloc(sizeof(MeshNode));
    
    // Add the mesh to the list
    node->mesh = mesh;
    node->next = mset->list;
    mset->list = node;

    // Update counter
    mset->meshNum++;
}

void MeshIO::mioMSetDelMesh(MeshSet *mset, Mesh *mesh)
{
    MeshNode *node, *temp;

    // Sanity check
    if (mset->list == NULL)
	return;
    
    // Check the first element (special case) 
    if (mset->list->mesh == mesh) {
	
	// Extract element from list
	temp = mset->list;
	mset->list = temp->next;
    }   
    else {
	
	// Initialize traversal
	node = mset->list;
	
	// Look for the mesh in the list
	while (node->next != NULL && node->next->mesh != mesh)
	    node = node->next;
	
	// Did we find it?
	if (node->next == NULL)
	    return;
	
	// Extract element from list
	temp = node->next;
	node->next = temp->next;
    }
    
    // Remove it
    free(temp);

    // Update counter 
    mset->meshNum--;
}

int MeshIO::mioMSetComputeVertexNum(MeshSet *mset)
{
    MeshNode *node;
    int vertexNum = 0;

    // Traverse the mesh list and sum vertex counts
    for (node = mset->list; node != NULL; node = node->next) 
	vertexNum += node->mesh->vertexNum;

    // Return with result
    return vertexNum;
}

int MeshIO::mioMSetComputeFaceNum(MeshSet *mset)
{
    MeshNode *node;
    int faceNum = 0;

    // Traverse the mesh list and sum vertex counts
    for (node = mset->list; node != NULL; node = node->next) 
	faceNum += node->mesh->faceNum;

    // Return with result
    return faceNum;
}

void MeshIO::mioMSetClearMeshes(MeshSet *mset)
{
    MeshNode *mesh;

    // Step through list deallocating all nodes
    while (mset->list != NULL) {

	// Deallocate this node
	mesh = mset->list;
	mset->list = mesh->next;
	
	// Get rid of mesh
	mioMeshClear(mesh->mesh);
	free(mesh->mesh);
	free(mesh);
    }    
    
    // Zero mesh counter
    mset->meshNum = 0;
}

void MeshIO::mioMSetClearMaterials(MeshSet *mset)
{
    // Clear out all materials
    for (int i = 0; i < mset->materialNum; i++)
	mioMaterialClear(&mset->materialList[i]);
    
    // Deallocate the material array
    free(mset->materialList);
    
    // Zero material counter
    mset->materialNum = 0;
}

void MeshIO::mioMSetClear(MeshSet *mset)
{
    // First, clear out all meshes
    mioMSetClearMeshes(mset);

    // Now, clear the materials as well
    mioMSetClearMaterials(mset);
}

MeshSet *MeshIO::mioMSetCopy(MeshSet *mset)
{
    MeshSet *newSet = (MeshSet *) malloc(sizeof(MeshSet));
    MeshNode *mesh;
    Mesh *newMesh;

    // Initialize mesh set
    mioMSetInit(newSet);

    // Step through all meshes
    for (mesh = mset->list; mesh != NULL; mesh = mesh->next) {
	
	// Copy this mesh
	newMesh = mioMeshCopy(mesh->mesh);
	
	// Add this mesh to the new mesh set
	mioMSetAddMesh(newSet, newMesh);
    }
    
    // Create new material array
    newSet->materialList = (Material *)
	malloc(sizeof(Material) * mset->materialNum);

    // Step through all materials
    for (int i = 0; i < mset->materialNum; i++) {
	
	// Copy this material
	mioMaterialCopy(&newSet->materialList[i], &mset->materialList[i]);
    }

    // Return with new mesh set pointer
    return newSet;
}

void MeshIO::mioMSetDump(MeshSet *mset, bool verbose)
{
    MeshNode *node;
    int ndx;

    // General statistics about the mesh set
    printf("Number of meshes in set: %d.\n", mset->meshNum);
    printf("Number of materials in set: %d.\n", mset->materialNum);

    // Do we want a detailed dump?
    if (verbose == true) {

	Material *mat;
	int i;

	// All right, then, print all the materials as well
	for (int j = 0; j < mset->materialNum; j++) {

	    mat = &mset->materialList[j];
	    
	    printf("Material name: \"%s\".\n", 
		   mat->name != NULL ? mat->name : "<no name>");
	    printf("Texture file name: \"%s\".\n", 
		   mat->texFile != NULL ? mat->texFile : "<no texture>");
	    printf("Ambient color: ");
	    for (i = 0; i < 3; i++) printf("%0.2f ", mat->ambient[i]);
	    printf("\nDiffuse color: ");
	    for (i = 0; i < 3; i++) printf("%0.2f ", mat->diffuse[i]);
	    printf("\nSpecular color: ");
	    for (i = 0; i < 3; i++) printf("%0.2f ", mat->specular[i]);
	    printf("\n");
	}
    }

    // Traverse the mesh list and print the statistics for each mesh
    for (ndx = 0, node = mset->list; node != NULL; node = node->next, ndx++) {
	printf("Mesh #%d:\n", ndx);
	mioMeshDump(node->mesh, verbose);
    }
}

void MeshIO::mioMSetScale(MeshSet *mset, float xScale, float yScale, float zScale)
{
    MeshNode *node;

    // Step through all meshes
    for (node = mset->list; node != NULL; node = node->next) {

        // Scale this mesh
        mioMeshScale(node->mesh, xScale, yScale, zScale);
    }
}

