/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 event.h  -  Event and event distribution for messages (Event Log)

    begin                : Sun Jan 6 2002
    copyright            : (C) 2002 by 
    email                : 
 ***************************************************************************/

#ifndef EVENT_H
#define EVENT_H

#include "rcobject.h"

enum EventType{
	ET_NONE         = 0,
	ET_ERROR        = 1,
	ET_WARNING      = 2,
	ET_MESSAGE      = 3,
	ET_STATUSUPDATE = 4
};

enum EventSeverity{
	ES_NONE        = 0,
	ES_DEBUG       = 1,
	ES_MINIMAL     = 2,
	ES_UNIMPORTANT = 3,
	ES_JUSTINCASE  = 4,
	ES_GOODTOKNOW  = 5,
	ES_IMPORTANT   = 6,
	ES_CRITICAL    = 8
};

#define POST_MESSAGE(__severity, __string) ED().PostEvent(new MStringEvent(ET_MESSAGE, __severity, __string));
#define POST_WARNING(__severity, __string) ED().PostEvent(new MStringEvent(ET_WARNING, __severity, __string));
#define POST_ERROR(__severity, __string) ED().PostEvent(new MStringEvent(ET_ERROR, __severity, __string));
#define POST_EVENT(__event) ED().PostEvent(new __event);

#ifdef _DEBUG
#define POST_DMESSAGE(__string) ED().PostEvent(new MStringEvent(ET_MESSAGE, ES_DEBUG, __string));
#else
#define POST_DMESSAGE(__string)
#endif //_DEBUG


struct SEventNotifier{
    MMutex mutex;
    MWaitCondition wc;
};

class MEvent : public MRCObject  {
friend class MEventDispatcher;
public: 
	MEvent(int type, int severity, DWORD ID = 0, DWORD SourceID = 0);
	virtual ~MEvent();
	// pure virtual functions
	virtual int GetMemUsage() = 0;
	virtual CString Format()  = 0;
	// not so pure virtual functions
	virtual CString FormatLog(){ return Format(); }
	// general service
	int GetType(){return m_nType;}
	int GetSeverity(){return m_nSeverity;}
	int GetTime(){return m_nTime;}
	DWORD GetID(){return m_dwID;}
	DWORD GetSourceID(){return m_dwSourceID;}
protected:
	int m_nType;
	int m_nSeverity;
	int m_nTime;
	DWORD m_dwID; // have to think of the method for generating such IDs
	DWORD m_dwSourceID; // have to think of the method for generating such IDs
	SEventNotifier* m_pNotifier;
private:
	// no default constructor
	MEvent();
	MEvent(const MEvent&);
	const MEvent& operator=(const MEvent&);
};

template<class T>
class TSimpleEvent : public MEvent {
public:
	TSimpleEvent(int type, int severity, const T& value, DWORD ID = 0, DWORD SourceID = 0) : MEvent(type, severity, ID, SourceID), m_value(value) {}
	virtual ~TSimpleEvent(){}
	// pure virtual functions
	virtual int GetMemUsage(){return sizeof(TSimpleEvent<T>);}
	// utility
	inline const T& GetValue() const {return m_value;}
protected:
	T m_value;
};

class MIntEvent : public TSimpleEvent<int>{
public:
	MIntEvent(int type, int severity, int nValue, DWORD ID = 0, DWORD SourceID = 0) : TSimpleEvent<int>(type, severity, nValue, ID, SourceID) {}
	virtual CString Format(){CString s; return s.format("%d", m_value);}
};

class MStringEvent : public TSimpleEvent<CString>{
public:
	MStringEvent(int type, int severity, const CString& value, DWORD ID = 0, DWORD SourceID = 0);
	virtual int GetMemUsage();
	virtual CString Format();
};

// event receiver interface
class MEventReceiver {
public:
	virtual ~MEventReceiver();
	virtual bool IsOfInterest(MEvent* pEvent) = 0;
	virtual bool IsSync() = 0;
	virtual void OnEvent(MEvent* pEvent, bool bSend) = 0;
};

// event queue
class MEventDispatcher {
	// Post event, Send event, Add-RemoveReceiver, EnumReceivers, etc.
	friend MEventDispatcher& ED();
public:
	//
	void PostEvent(MEvent* pEvent);
	bool SendEvent(MEvent* pEvent);// same as
	     // post but waits for async receivers to receive
	     // returns true if the event was delivered
	void AddReceiver(MEventReceiver*);
	void RemoveReceiver(MEventReceiver*);
protected:
	list<MEventReceiver*> m_receivers;
	MMutex m_mutex;
private:
	MEventDispatcher();
public: // workaround for egcs bug
	~MEventDispatcher();
protected:
	static MEventDispatcher ms_EventDispatcher;
};
extern MEventDispatcher& ED();

class MSyncEventReceiver : public MEventReceiver
{
public:
private:
	virtual bool IsSync(){return true;}
};

class MAsyncEventReceiver : public MEventReceiver
{
public:
	// typedefs
	typedef std::deque<TSmartPtr<MEvent> > queue_type;
	typedef std::deque<TSmartPtr<MEvent> >::iterator iterator;
	typedef std::deque<TSmartPtr<MEvent> >::reverse_iterator reverse_iterator;
public:
	MAsyncEventReceiver(int nMaxTime, int nMaxMem = INT_MAX);
	virtual ~MAsyncEventReceiver();
	// owner thread (which exclusively calls poll)
	void SetPollingThread(); // think of it, whether to set a callback 'a handler' here as well
	                         // and what will call this handler then
	// getting events
	bool HaveEvents();
	int QueuedEvents();
	int MissedEvents();
	void ResetMissedEvents();
	int PriorityEvents();
	// queue access
	MEvent* LockFront(); // the queue will not be reduced until UnlockFront() is called; Pop() will still work
	MEvent* UnsafeFront(); // this function creates a crash race condition when used without LockFront() if other thread pushes many events at the same time
	void UnlockFront();
	TSmartPtr<MEvent> SafeFront(); // this does not create a crash race condition if other thread pushes many events at the same time
	void Pop();
	// make possible to lock the queue
	// thise methods are only supposed to be used together with
	inline void Lock(){ m_mutex.lock(); ASSERT(m_hThreadLock==(HANDLE)-1); m_hThreadLock=MThread::currentThread(); }
	inline void Unlock(){ ASSERT(m_hThreadLock==MThread::currentThread()); m_hThreadLock=(HANDLE)-1; m_mutex.unlock(); }
	// wait for an event
	bool Poll(u_long time = ULONG_MAX);
	// copy entire queue
	void CopyQueue(queue_type& copy);
protected:
	queue_type m_queue;
	std::queue< MEvent* > m_priorityEvents;
	MMutex m_mutex;
	MWaitCondition m_wait;
	HANDLE m_hPollingThread;
	HANDLE m_hThreadLock;
	int m_nFrontLocks;
	int m_nMaxTime;
	int m_nMaxMem;
	int m_nMemUsed;
	int m_nMissedEvents;
	// virtual methods
	virtual void OnEvent(MEvent* pEvent, bool bSend);
	virtual bool IsOfInterest(MEvent* pEvent){return true;}
	void FlushToFit(); // called internally to adjust the size of the queue
	// protected methods
	void PriorityEvent(MEvent* pEvent) { // must be called with m_mutex locked!
		m_priorityEvents.push(pEvent);
	}
private:
	virtual bool IsSync(){return false;}
};

#endif

