/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_figure.h 
 * @brief Implementation of the model element class.
 * 
 */

#ifndef __GC_FIGURE_H__
#define __GC_FIGURE_H__

#ifdef MAKEDLL
  #define GENERIC_CANVAS_API  __declspec(dllexport)
#else
  #define GENERIC_CANVAS_API  __declspec(dllimport)
#endif

#include <vector>

#include "myx_gc_layer.h"

//----------------------------------------------------------------------------------------------------------------------

class CGCModel;
class CLayer;
class CFigureInstance;
class CFigure;

#pragma warning(disable: 4251) // Disable warning about DLL interface for template classes.

typedef std::vector<CFigure*> CFigureList;
typedef std::vector<CFigure*>::iterator CFigureListIterator;

class GENERIC_CANVAS_API CFigure
{
  friend CGCModel;
  friend CFigureInstance;
private:
  GLuint FList;              // The display list that represents this figure in OpenGL.
  GLuint FTemplateList;      // The display list that represents the base OpenGL command for that figure type.
  CGCModel* FModel;          // The model to which this figure belongs.
  CFigure* FParent;          // The parent figure if any.
  char* FType;               // The name of the type to be used for display. This identifier gets converted to a display list.
  bool FDirty;               // True if any of the properties changed that affect the display list.
  double FScaling[3];        // The factors to scale the figure in each direction.
  double FTranslation[3];    // The factors to move the figure.
  double FRotation[4];       // The factors to rotate the figure. FRotation[0] is an angle in radians. Index 1-3 form a vector to rotate around.
                             // Note: Order of apllication of the 3 transformations is scaling first, then rotation, finally translation.
  CFigureList FChildren;     // A list of child figures.
  bool FDestroying;          // true if this figure is currently being destroyed. No need to update it when children are destroyed.
  CFigureInstances FNotificationList; // A list of class that want to get notfied when this figure class gets destroyed.
protected:
  void AddFreeNotification(CFigureInstance* Instance);
  void MakeDirty(void);
  void RemoveFreeNotification(CFigureInstance* Instance);
  void ValidateDisplayList(void);
public:
  CFigure(CGCModel* Owner, GLuint TemplateList);
  ~CFigure(void);

  virtual void __cdecl AddChild(CFigure* Child);
  virtual void __cdecl Clear(void);
  virtual void __cdecl InsertChild(int Index, CFigure* Child);
  virtual void __cdecl RemoveChild(CFigure* Child);
  virtual void __cdecl Render(void);
  virtual void __cdecl Rotate(double Angle, double Rx, double Ry, double Rz);
  virtual void __cdecl RotateV(double Angle, const double Axis[3]);
  virtual void __cdecl Scale(double Sx, double Sy, double Sz, bool Accumulative = false);
  virtual void __cdecl ScaleV(const double Factor[3], bool Accumulative = false);
  virtual void __cdecl Translate(double Tx, double Ty, double Tz, bool Accumulative = false);
  virtual void __cdecl TranslateV(const double Factor[3], bool Accumulative = false);
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * The figure instance class is a proxy for a figure on a particular layer. There can be more than one instances pointing to the same figure.
 */
class GENERIC_CANVAS_API CFigureInstance
{
  friend class CLayer;
  friend class CFigure;
  friend class CSelectionLayer;
private:
  GLuint FList;              // The OpenGL display list created for this instance.
  CLayer* FLayer;            // The layer on which this figure is displayed.
  CFigure* FFigure;          // The figure of which this class is an instance.
  bool FDirty;               // True if any of the properties changed that affect the display list.
  double FScaling[3];        // The factors to scale the figure in each direction.
  double FTranslation[3];    // The factors to move the figure.
  double FRotation[4];       // The factors to rotate the figure. FRotation[0] is an angle in radians. Index 1-3 form a vector to rotate around.
                             // Note: Order of apllication of the 3 transformations is scaling first, then rotation, finally translation.
  bool FValidBounds;         // True if the bounding box is valid.
  TBounds FCurrentBounds;    // Cached bounding box.
  bool FSelected;            // True if this instance is currently selected.
protected:
  void FreeNotification(CFigure* Figure);
  void MakeDirty(void);
  void RenderFeedback(void);
  void ValidateDisplayList(void);
public:
  CFigureInstance(CLayer* Owner, CFigure* Figure);
  ~CFigureInstance(void);

  virtual void __cdecl GetBounds(TBounds* Bounds);
  virtual void __cdecl Release(void);
  virtual void __cdecl Render(void);
  virtual void __cdecl Rotate(double Angle, double Rx, double Ry, double Rz);
  virtual void __cdecl RotateV(double Angle, const double Axis[3]);
  virtual void __cdecl Scale(double Sx, double Sy, double Sz, bool Accumulative = false);
  virtual void __cdecl ScaleV(const double Factor[3], bool Accumulative = false);
  virtual bool __cdecl Selected(void);
  virtual void __cdecl Translate(double Tx, double Ty, double Tz, bool Accumulative = false);
  virtual void __cdecl TranslateV(const double Factor[3], bool Accumulative = false);
};

//----------------------------------------------------------------------------------------------------------------------

#endif // __GC_FIGURE_H__
