/*
  filter.h

  Article filtering.

  $Id: filter.h,v 1.4 2002/05/14 14:21:13 mirkol Exp $
*/

#ifndef	FILTER_H
#define	FILTER_H

#include <sys/types.h>
#include <regex.h>
#include "fetchlist.h"
#include "over.h"

/* The possible actions in a filter. */
typedef enum {
    FILTER_FULL,
    FILTER_XOVER,
    FILTER_THREAD,
    FILTER_DISCARD,
    FILTER_DEFAULT
} FilterAction;

/* Representation of a rule. */
typedef enum {
    RULE_NEWSGROUP,					/* Wildmat data */
    RULE_SUBJECT,					/* Regex data */
    RULE_REFERENCE,
    RULE_FROM,
    RULE_MSGID,
    RULE_BYTES_LT, RULE_BYTES_EQ, RULE_BYTES_GT,	/* Number data */
    RULE_LINES_LT, RULE_LINES_EQ, RULE_LINES_GT,
    RULE_NOREFS_LT, RULE_NOREFS_EQ, RULE_NOREFS_GT,
    RULE_XPOSTS_LT, RULE_XPOSTS_EQ, RULE_XPOSTS_GT,
    RULE_DATE_LT, RULE_DATE_EQ, RULE_DATE_GT,
    RULE_POST_STATUS					/* 'y','n','m' */
} FilterRuleType;

/* Data for Date: header parsing. */

#define RULE_DATE_EQ_PRECISION ((time_t) (24*60*60))    /* +/- 24 hours precision */

typedef enum {
    NOW,                                /* beginning of fetch */
    LASTUPDATE,                         /* of last fetch */
    INVALID,                            /* invalid dates, only RULE_DATE_EQ */
    FIXED                               /* fixed time */
} FilterRuleDateEnumType;

typedef struct {
    time_t calctime;                    /* calctime = vartime + timeoffset */
    time_t timeoffset;
    FilterRuleDateEnumType vartime;
} FilterRuleDateType;


typedef union {
    regex_t regex;
    unsigned long amount;
    char *grp;
    char postAllow; /* 'y','n','m' */
    FilterRuleDateType reftime;
} FilterRuleData;

typedef struct {
    FilterRuleType type;
    FilterRuleData data;
} FilterRule;

/* A single filter is a collection of rules with an action. */
typedef struct {
    int nRules;
    int maxRules;
    FilterRule *rules;
    FilterAction action;
} Filter;

/* Add a filter to the list of filters. */
void
Flt_addFilter( const Filter *f );


/*
 * Called by client.c once before processing a batch of overviews
 * with Flt_checkFilters().
 */
void
Flt_init( const char *filename );

/*
 * Run the rules over the supplied overview. If a specific rule fires,
 * returns its action. If no rule fires, return the default read mode.
 */
FilterAction
Flt_checkFilters( const char *thisGrp, const char *newsgroups,
		  const Over *ov, FetchMode mode );

/*
 * Build and access a filter
 */
Filter *
new_Filter( void );

void
del_Filter( Filter *f );

FilterAction
Flt_action( const Filter *f );

int
Flt_nRules( const Filter *f );

Bool
Flt_getNewsgroups( void );

FilterRule
Flt_rule( const Filter *f, int ruleNo );

void
Flt_setAction( Filter *f, FilterAction action );

void
Flt_addRule( Filter *f, FilterRule rule );

#endif	/* FILTER_H */
