/*
 * audio-freebsd.cc --
 *
 *      Old audio driver for FreeBSD 2 and 3.  This has been superseded by
 *      audio-oss.cc.
 */

/*
 * Full Duplex audio module for the new sound driver and full duplex
 * cards. Luigi Rizzo, from original sources supplied by Amancio Hasty.
 *
 * This includes some enhancements:
 * - limit the maximum size of the playout queue to approx 4 frames;
 *   this is necessary if the write channel is slower than expected;
 *   the fix is based on two new ioctls, AIOGCAP and AIONWRITE,
 *   but the code should compile with the old driver as well.
 * - use whatever format is available from the card (included split
 *   format e.g. for the sb16);
 */


#include <fcntl.h>
#include <machine/soundcard.h>
#include "audio.h"
#include "mulaw.h"
#include "Tcl.h"

#define ULAW_ZERO 0x7f

class FreeBSDAudio : public Audio {
    public:
	FreeBSDAudio();
	virtual int FrameReady();
	virtual u_char* Read();
	virtual	void Write(u_char *);
	virtual void SetRGain(int);
	virtual void SetPGain(int);
	virtual void OutputPort(int);
	virtual void InputPort(int);
	virtual void Obtain();
	virtual void Release();
/*	virtual int HalfDuplex() const; */
    protected:

	u_char* readbuf;
	u_short *s16_buf;

        int mixerfd;

#if defined(AIOGCAP) /* new sound driver */
	int play_fmt, rec_fmt ; /* the sb16 has split format... */
	snd_capabilities soundcaps;
#endif
};

static class FreeBSDAudioClass : public TclClass {
public:
        FreeBSDAudioClass() : TclClass("Audio/FreeBSD") {}
        TclObject* create(int, const char*const*) {
	    return (new FreeBSDAudio);

        }
} freebsd_audio_class;

FreeBSDAudio::FreeBSDAudio() : mixerfd(-1)
{
    readbuf = new u_char[blksize_];
    s16_buf = new u_short[blksize_];

    memset(readbuf, ULAW_ZERO, blksize_);

    /*
     * The only way to determine if the device is full duplex
     * or not is by actually opening it.  Unfortunately, we might
     * not be able to open it because some other process is
     * using it.  Assume half-duplex.  Obtain() will override
     * if appropriate.
     */
    duplex_ = 0;

    input_names_ = "mike linein";
    output_names_ = "speaker lineout";
}

void
FreeBSDAudio::Obtain()
{
    char *thedev;
    char buf[64];
    int d = -1;

    if (haveaudio())
	abort();
    thedev=getenv("AUDIODEV");
    if (thedev==NULL)
	thedev="/dev/audio";
    else if (thedev[0]>='0') {
	d = atoi(thedev);
	sprintf(buf,"/dev/audio%d", d);
	thedev = buf ;
    }
    fd_ = open(thedev, O_RDWR );
    thedev=getenv("MIXERDEV");
    if (thedev == NULL)
	if (d < 0)
	    thedev = "/dev/mixer";
	else {
	    sprintf(buf,"/dev/mixer%d", d);
	    thedev = buf ;
	}

    mixerfd = open(thedev, O_RDWR);
    if (fd_ >= 0) {
	snd_chan_param pa;
	struct snd_size sz;

	ioctl(fd_, AIOGCAP, &soundcaps);

	// ErikM
	printf("FreeBSDAudio::Obtain: soundcaps.\n");
	printf("\tleft=%d\n\tright=%d\n", soundcaps.left, soundcaps.right);
	printf("\tformats=0x%lx\n", soundcaps.formats);
	printf("\tinputs=0x%lx\n", soundcaps.inputs);
	printf("\tmixers=0x%lx\n", soundcaps.mixers);
	printf("\trate_min=%ld\n", soundcaps.rate_min);
	printf("\trate_max=%ld\n", soundcaps.rate_max);

	pa.play_rate = pa.rec_rate = 8000 ;
	pa.play_format = pa.rec_format = AFMT_MU_LAW ;
	switch (soundcaps.formats & (AFMT_FULLDUPLEX | AFMT_WEIRD)) {
	case AFMT_FULLDUPLEX :

	  // ErikM
	  printf("FreeBSDAudio::Obtain: in AFMT_FULLDUPLEX\n");

	    /*
	     * this entry for cards with decent full duplex. Use s16
	     * preferably (some are broken in ulaw) or ulaw or u8 otherwise.
	     */
	    if (soundcaps.formats & AFMT_S16_LE)
		pa.play_format = pa.rec_format = AFMT_S16_LE ;
	    else if (soundcaps.formats & AFMT_MU_LAW)
		pa.play_format = pa.rec_format = AFMT_MU_LAW ;
	    else if (soundcaps.formats & AFMT_U8)
		pa.play_format = pa.rec_format = AFMT_U8 ;
	    else {
		printf("sorry, no supported formats\n");
		close(fd_);
		fd_ = -1 ;
		return;
	    }
	    break ;
	case AFMT_FULLDUPLEX | AFMT_WEIRD :

	  // ErikM
	  printf("FreeBSDAudio::Obtain: in AFMT_FULLDUPLEX | AFMT_WEIRD\n");

	  /* this is the sb16... */
	    if (soundcaps.formats & AFMT_S16_LE) {
		pa.play_format = AFMT_S8 ;
		pa.rec_format = AFMT_S16_LE;
	    } else {
		printf("sorry, no supported formats\n");
		close(fd_);
		fd_ = -1 ;
		return;
	    }
	    break ;
	default :
#if 0
	    printf("sorry don't know how to deal with this card\n");
	    close (fd_);
	    fd_ = -1;
#endif
	    break;
	}

	// ErikM
	printf("FreeBSDAudio::Obtain: pa.play_format=0x%lx, pa.rec_format=0x%lx\n",
	       pa.play_format, pa.rec_format);


	ioctl(fd_, AIOSFMT, &pa);
	play_fmt = pa.play_format ;
	rec_fmt = pa.rec_format ;
	sz.play_size = (play_fmt == AFMT_S16_LE) ? 2*blksize_ : blksize_;
	sz.rec_size = (rec_fmt == AFMT_S16_LE) ? 2*blksize_ : blksize_;
	ioctl(fd_, AIOSSIZE, &sz);

        /*
         * note: this uses a modified function of the new driver,
         * which will return AFMT_FULLDUPLEX set in SNDCTL_DSP_GETFMTS
         * for full-duplex devices. In the old driver this was 0 so
         * the default is to use half-duplex for them. Note also that I have
         * not tested half-duplex operation.
         */
        int i;
        ioctl(fd_, SNDCTL_DSP_GETFMTS, &i);
	// ErikM
#if 1
        printf("SNDCTL_DSP_GETFMTS returns 0x%08x %s duplex\n",
               i, i & AFMT_FULLDUPLEX ? "full":"half");
#endif
        duplex_ =  (i & AFMT_FULLDUPLEX) ? 1 : 0 ;

// ErikM
	printf("FreeBSDAudio::Obtain: duplex_ = %d\n", duplex_);


	/*
	 * Set the line input level to 0 to shut
	 * off the analog side-tone gain between
	 * the line-in and line-out.  This would otherwise
	 * wreak havoc on an echo canceler, for example,
	 * plugged into the audio adaptor.
	 */
	int v = 0;
	(void)ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_LINE), &v);

        /*
         * Restore the hardware settings in case
         * some other vat changed them.
         */
        InputPort(iport_);
        SetRGain(rgain_);
        SetPGain(pgain_);

        if (duplex_)
                Audio::Obtain();
        else
                notify();
    }
}


void FreeBSDAudio::Release()
{
    if (haveaudio()) {
            if (mixerfd > 0) {
                    close(mixerfd);
            }
            mixerfd = -1;
            Audio::Release();
    }
}

void FreeBSDAudio::Write(u_char *cp)
{
    long i = blksize_, l;

#if 0 && defined(AIOGCAP)
    int queued;
#endif

    if (play_fmt == AFMT_S16_LE) {
	for (i=0; i< (long) blksize_; i++)
	    s16_buf[i] = mulawtolin[cp[i]] ;
	cp = (u_char *)s16_buf;
        i = 2 *blksize_ ;
    }
    else if (play_fmt == AFMT_S8) {
	for (i=0; i< (long) blksize_; i++) {
	    int x = mulawtolin[cp[i]] ;
	    x =  (x >> 8 ) & 0xff;
	    cp[i] = (u_char)x ;
	}
	i = blksize_ ;
    } else if (play_fmt == AFMT_U8) {
	for (i=0; i< (long) blksize_; i++) {
	    int x = mulawtolin[cp[i]] ;
	    x =  (x >> 8 ) & 0xff;
	    x = (x ^ 0x80) & 0xff ;
	    cp[i] = (u_char)x ;
	}
	i = blksize_ ;
    }
#if 0 && defined(AIOGCAP)
    ioctl(fd_, AIONWRITE, &queued);
    queued = soundcaps.bufsize - queued ;
    if (play_fmt == AFMT_S16_LE) {
	if (queued > 8*blksize_)
	    i -= 8 ;
    } else {
	if (queued > 4*blksize_)
	    i -= 4 ;
    }
#endif
    for ( ; i > 0 ; i -= l) {
	l = write(fd_, cp, i);
	cp += l;
    }
}

u_char* FreeBSDAudio::Read()
{
    u_char* cp;
    long l=0, l0 = blksize_,  i = blksize_;

    cp = readbuf;

    if (rec_fmt == AFMT_S16_LE) {
	cp = (u_char *)s16_buf;
        l0 = i = 2 *blksize_ ;
    }
    for ( ; i > 0 ; i -= l ) {
	l = read(fd_, cp, i);
        cp += l ;
    }
    if (rec_fmt == AFMT_S16_LE) {
	for (i=0; i< (long) blksize_; i++)
	    readbuf[i] = lintomulaw[ s16_buf[i] & 0xffff ] ;
    }
    else if (rec_fmt == AFMT_S8) {
	for (i=0; i< (long) blksize_; i++)
	    readbuf[i] = lintomulaw[ readbuf[i]<<8 ] ;
    }
    else if (rec_fmt == AFMT_U8) {
	for (i=0; i< (long) blksize_; i++)
	    readbuf[i] = lintomulaw[ (readbuf[i]<<8) ^ 0x8000 ] ;
    }
    return readbuf;
}

/*
 * should check that I HaveAudio() before trying to set gain.
 *
 * In most mixer devices, there is only a master volume control on
 * the capture channel, so the following code does not really work
 * as expected. The only (partial) exception is the MIC line, where
 * there is generally a 20dB boost which can be enabled or not
 * depending on the type of device.
 */
void FreeBSDAudio::SetRGain(int level)
{
    rgain_ = level;
    float x = level;
    level = (int) (x/2.56);
    int foo = (level<<8) | level;
    switch (iport_) {
    case 2:
	if (ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_IGAIN), &foo) == -1)
	   perror("failed set input line volume");
	break;
    case 1:
	if (ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_IGAIN), &foo) == -1)
	   perror("failed set input line volume");
	break;
    case 0:
	if (ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_MIC), &foo) == -1)
	    perror("failed to set mic volume");
	if (ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_IGAIN), &foo) == -1)
	   perror("failed set input line volume");
	break;
    }
}

void FreeBSDAudio::SetPGain(int level)
{
    pgain_ = level;
    float x = level;
    level = (int) (x/2.56);
    int foo = (level<<8) | level;
    if (mixerfd >= 0) {
            if (ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_PCM), &foo) == -1) {
                    perror("failed to output level");
            }
    }
}

void FreeBSDAudio::OutputPort(int p)
{
    oport_ = p;
}

void FreeBSDAudio::InputPort(int p)
{
    int   zero = 0;

    switch(p) {
    case 2:
	zero = 1 << SOUND_MIXER_CD;
	break;
    case 1:
        zero = 1 << SOUND_MIXER_LINE;
	break;
    case 0 :
	zero = 1 << SOUND_MIXER_MIC;
	break;
    }
    if ( ioctl(mixerfd, SOUND_MIXER_WRITE_RECSRC, &zero) == -1 ) {
	perror("failed to select input");
        p = 0;
    }
    iport_ = p;
}

/*
 * FrameReady must return 0 every so often, or the system will keep
 * processing mike data and not other events.
 */
int FreeBSDAudio::FrameReady()
{
    int i = 0;
    int lim = blksize_;

    ioctl(fd_, FIONREAD, &i );
    if (rec_fmt == AFMT_S16_LE) lim = 2*blksize_;
    return (i >= lim) ? 1 : 0 ;
}

/*** end of file ***/
