/*
 * rep-decoder-jpeg2uncompressed.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "inet.h"
#include "rtp.h"
#include "rep-decoder.h"
#include "bsd-endian.h"
#include "tclcl.h"
#include "jpeg/jpeg.h"
#include "renderer.h"
#include "pktbuf.h"
#include "vidreps.h"

class JpegReassembler {
public:
	JpegReassembler();
	~JpegReassembler();
	u_char* reassemble(const rtphdr* rh, const u_char* bp, int& len);
	inline int badoff() const { return (badoff_); }
	inline int hugefrm() const { return (hugefrm_); }
protected:
	int decimate_;
	int ndec_;

	/*
	 * Reassembly buffer.  This should be a dma buffer in the
	 * jvdriver case but it's not clear if jvdriver allows buffers
	 * to be shared across sockets. FIXME ask Lance
	 * If it does, then we can pass dma buffers betwee
	 * the decoder and renderers.
	 */
#define JPEG_SLOTS 64
#define JPEG_SLOTMASK (JPEG_SLOTS - 1)
	struct slot {
		int seqno;
		int eof;	/* nbytes in last pkt of frame, o.w. 0 */
		u_int32_t off;
		u_int32_t ts;
	} slots_[JPEG_SLOTS];

	/*
	 * Reassembly buffers.  We do double-buffering, which allows
	 * packets to arrive out of order across frame boundaries,
	 * but not across an entire frame (i.e., we don't want to see
	 * any packets from frame k+2 until we're done with frame k).
	 * We use RTP timestamps to keep track of which frame is
	 * allocated to which buffer.  (RTP guarantees that the
	 * timestamps are constant across a frame, and increase
	 * between succesive frames.  FIXME is latter really true?)
	 */
	struct rbuf {
		int drop;
		u_int32_t ts;
		u_char* bp;
	};
	rbuf rb0_;
	rbuf rb1_;
	int rbsize_;
	int hugefrm_;
	int badoff_;
};

/*
 * Initial size of each reassembly buffer.
 */
#define JPEG_BUFSIZE (16*1024)

class JpegToUncompressedDecoder : public RepDecoder {

public:
  JpegToUncompressedDecoder();
  virtual ~JpegToUncompressedDecoder();
  virtual int command(int argc, const char*const* argv);
  virtual void resize(int inw, int inh);
  virtual void post_delay_recv(pktbuf *pb);
  virtual void set_frame_buffer(VidRep *fb) {
    output_ = (Uncompressed *) fb;
  };

protected:
  int inq_;		/* input quantization */
  int type_;		/* JPEG/RTP parameters type code */
  int decimation_;
  u_char *rvts_;

  void configure();
  JpegPixelDecoder* codec_;
  JpegDecoder::config config_;
  JpegReassembler reasm_;
  Uncompressed *output_;
};

static class JpegToUncompressedDecoderClass : public TclClass {
public:
  JpegToUncompressedDecoderClass() : TclClass("Module/VideoDecoder/JPEGToUncompressed") {}
  TclObject* create(int argc, const char*const* argv) {
    return (new JpegToUncompressedDecoder());
  }
} swd_jpeg_to_uncompressed_;

JpegToUncompressedDecoder::JpegToUncompressedDecoder()
  : RepDecoder(sizeof(jpeghdr)), rvts_(0), codec_(0)
{
  JpegDecoder::defaults(config_);

  inq_ = 0;
  /* guess type 0 */
  type_ = 0;
  decimation_ = 422;
  output_ = 0;
}

JpegToUncompressedDecoder::~JpegToUncompressedDecoder()
{
  delete codec_;
}

int
JpegToUncompressedDecoder::command(int argc, const char*const* argv)
{
  return (RepDecoder::command(argc,argv));
}

void JpegToUncompressedDecoder::configure()
{
  config_.comp[0].hsf = 2;
  int old_decimation = decimation_;
  if (type_ == 1) {
    decimation_ = 411;
    config_.comp[0].vsf = 2;
  } else {
    decimation_ = 422;
    config_.comp[0].vsf = 1;
  }
  config_.comp[1].hsf = 1;
  config_.comp[1].vsf = 1;
  config_.comp[2].hsf = 1;
  config_.comp[2].vsf = 1;
  config_.width = inw_;
  config_.height = inh_;
  JpegDecoder::quantizer(config_, inq_);

  delete codec_;
  codec_ = JpegPixelDecoder::create(config_, inw_, inh_);

  /*
   * If we have been given name of Uncompressed object
   * to put result in, set up byte images correctly.
   */
  if (output_ != 0) {
    u_char* frm = codec_->frame();
    int off = inw_ * inh_;

    if (decimation_ == 411) {
      output_->init(inw_, inh_, 2, 2, inw_, inh_, 0, 0, frm,
		    frm + off, frm + off + (off >> 2));
    } else {
      output_->init(inw_, inh_, 2, 1, inw_, inh_, 0, 0, frm,
		    frm + off, frm + off + (off >> 1));
    }
  }



  Tcl& tcl = Tcl::instance();
  int q;
  q = JpegDecoder::q_to_thresh(inq_);
  codec_->thresh(q);

#ifdef notyet
  int ct = atoi(tcl.attr("softJPEGcthresh"));
#else
  int ct = 6;
#endif

  codec_->cthresh(ct);

  if (old_decimation != decimation_)
    tcl.evalf("%s parameters_changed", name());
}

void JpegToUncompressedDecoder::resize(int inw, int inh) {
  output_->resized_ = 1;
  rvts_ = new u_char[(inw*inh)/64];
  RepDecoder::resize(inw, inh);
}

void JpegToUncompressedDecoder::post_delay_recv(pktbuf* pb)
{
  rtphdr* rh = (rtphdr*)pb->dp;
  const jpeghdr* p = (const jpeghdr*)(rh + 1);
  int needConfig = 0;
  if (p->q != inq_ || p->type != type_) {
    type_ = p->type;
    inq_ = p->q;
    needConfig = 1;
  }
  int inw = p->width << 3;
  int inh = p->height << 3;
  if (inw_ !=  inw || inh_ != inh) {
    resize(inw, inh);
    needConfig = 1;
  }
  if (needConfig)
    configure();

  u_int8_t* bp = (u_int8_t*)(p + 1);
  int cc = pb->len - (sizeof(*rh) + sizeof(*p));
  bp = reasm_.reassemble(rh, bp, cc);
  if (bp != 0) {
    if (output_ != 0) {
      output_->ts_ = ntohl(rh->rh_ts);
      output_->ssrc_ = ntohl(rh->rh_ssrc);
    }
    codec_->decode(bp, cc, rvts_, 0);
    codec_->resetndblk();

    if (output_ != 0) {
      if (target_ != 0) {
	target_->sched_exec(output_);
      }
      if (callback_ != 0) {
	Tcl& tcl = Tcl::instance();
	tcl.eval(callback_);
      }
    }
  }
  pb->release();
}

