/*
 * session-rtp.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/rtp/session-rtp.cc,v 1.33 2002/02/03 04:15:47 lim Exp $";

#include "config.h"
#include <math.h>
#include <errno.h>
#include <string.h>
#include "source.h"
#include "tclcl.h"
#include "media-timer.h"
#include "crypt.h"
#include "timer.h"
#include "ntp-time.h"
#include "session-rtp.h"

static class AudioSessionClass : public TclClass {
    public:
	AudioSessionClass() : TclClass("Session/RTP/Audio") {}
	TclObject* create(int, const char*const*) {
		return (new AudioSession);
	}
} audio_class;

static class VideoSessionClass : public TclClass {
    public:
	VideoSessionClass() : TclClass("Session/RTP/Video") {}
	TclObject* create(int, const char*const*) {
		return (new VideoSession);
	}
} video_class;

int RTP_Session::check_format(int fmt) const
{
	switch(fmt) {
	case RTCP_PT_SR:
	case RTCP_PT_RR:
	case RTCP_PT_SDES:
	case RTCP_PT_BYE:
	case RTCP_PT_APP:
		return (0);
	}
	return (1);
}

RTP_Session::RTP_Session()
	: badversion_(0), badoptions_(0), badfmt_(0), badext_(0), nrunt_(0),
	  last_np_(0),
	  sdes_seq_(0),
	  confid_(-1),
	  pool_(0)
{
	for (int i = 0; i < NLAYER; ++i) {
		dh_[i].manager(this);
		ch_[i].manager(this);
	}

	bind("badversion_", (int*)&badversion_);
	bind("badoptions_", (int*)&badoptions_);
	bind("badfmt_", (int*)&badfmt_);
	bind("badext_", (int*)&badext_);
	bind("nrunt_", (int*)&nrunt_);
}

RTP_Session::~RTP_Session()
{
	if (pool_ != 0) {
		Tcl::instance().evalf("%s destroy", pool_->name());
	}
}

u_int32_t RTP_Session::alloc_srcid(u_int32_t addr) const
{
	timeval tv;
	::gettimeofday(&tv, 0);
	u_int32_t srcid = u_int32_t(tv.tv_sec + tv.tv_usec);
	srcid += (u_int32_t)getuid();
	srcid += (u_int32_t)getpid();
	srcid += addr;
	return (srcid);
}

extern char* onestat(char* cp, const char* name, u_long v);

#ifdef notyet
char* RTP_Session::stats(char* cp) const
{
	cp = onestat(cp, "Bad-RTP-version", badversion_);
	cp = onestat(cp, "Bad-RTPv1-options", badoptions_);
	cp = onestat(cp, "Bad-Payload-Format", badfmt_);
	cp = onestat(cp, "Bad-RTP-Extension", badext_);
	cp = onestat(cp, "Runts", nrunt_);
	Crypt* p = dh_[0].net()->crypt();
	if (p != 0) {
		cp = onestat(cp, "Crypt-Bad-Length", p->badpktlen());
		cp = onestat(cp, "Crypt-Bad-P-Bit", p->badpbit());
	}
	/*FIXME*/
	if (ch_[0].DataHandler::net() != 0) {
		Crypt* p = ch_[0].DataHandler::net()->crypt();
		if (p != 0) {
			cp = onestat(cp, "Crypt-Ctrl-Bad-Length", p->badpktlen());
			cp = onestat(cp, "Crypt-Ctrl-Bad-P-Bit", p->badpbit());
		}
	}
	*--cp = 0;
	return (cp);
}
#endif

int RTP_Session::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	char* cp = tcl.buffer();
	if (argc == 2) {
		if (strcmp(argv[1], "sm") == 0) {
			strcpy(cp, sm_->name());
			tcl.result(cp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "data-net") == 0) {
			strcpy(cp, dh_[0].net()->name());
			tcl.result(cp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "ctrl-net") == 0) {
			strcpy(cp, ch_[0].DataHandler::net()->name());
			tcl.result(cp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "data-bandwidth") == 0) {
			tcl.resultf("%d", bps());
			return (TCL_OK);
		}
		if (strcmp(argv[1], "exit") == 0) {
			if (sm_->localsrc() != 0)
				send_bye();
			return (TCL_OK);
		}
		if (strcmp(argv[1], "active") == 0) {
			sm_->sortactive(cp);
			tcl.result(cp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "local-addr-heuristic") == 0) {
			strcpy(cp, intoa(LookupLocalAddr()));
			tcl.result(cp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "nb") == 0) {
			sprintf(cp, "%u", 8 * nb_);
			tcl.result(cp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "nf") == 0) {
			sprintf(cp, "%u", nf_);
			tcl.result(cp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "np") == 0 ||
		    strcmp(argv[1], "ns") == 0) {
			sprintf(cp, "%u", np_);
			tcl.result(cp);
			return (TCL_OK);
		}
	} else if (argc == 3) {
		if (strcmp(argv[1], "sm") == 0) {
			sm_ = (SourceManager*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "buffer-pool") == 0) {
			pool_ = (BufferPool*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "name") == 0) {
			Source* s = sm_->localsrc();
			s->sdes(RTCP_SDES_NAME, argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "email") == 0) {
			Source* s = sm_->localsrc();
			s->sdes(RTCP_SDES_EMAIL, argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "random-srcid") == 0) {
			sprintf(cp, "%u", alloc_srcid(inet_addr(argv[2])));
			tcl.result(cp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "data-net") == 0) {
			dh_[0].net((Network*)TclObject::lookup(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "ctrl-net") == 0) {
			ch_[0].net((Network*)TclObject::lookup(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "max-bandwidth") == 0) {
			double bw = atof(argv[2]) / 8.;
			for (int i = 0; i < NLAYER; ++i)
				ch_[i].bandwidth(bw);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "confid") == 0) {
			confid_ = atoi(argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "data-bandwidth") == 0) {
			/*FIXME assume value in range */
			bps(atoi(argv[2]));
			return (TCL_OK);
		}
	} else if (argc == 4) {
		if (strcmp(argv[1], "data-net") == 0) {
			u_int layer = atoi(argv[3]);
			if (layer >= NLAYER)
				abort();
			if (*argv[2] == 0) {
				dh_[layer].net(0);
				return (TCL_OK);
			}
			Network* net = (Network*)TclObject::lookup(argv[2]);
			if (net == 0) {
				tcl.resultf("no network %s", argv[2]);
				return (TCL_ERROR);
			}
			if (net->rchannel() < 0) {
				tcl.resultf("network not open");
				return (TCL_ERROR);
			}
			dh_[layer].net(net);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "ctrl-net") == 0) {
			u_int layer = atoi(argv[3]);
			if (layer >= NLAYER)
				abort();
			if (*argv[2] == 0) {
				ch_[layer].net(0);
				return (TCL_OK);
			}
			Network* net = (Network*)TclObject::lookup(argv[2]);
			if (net == 0) {
				tcl.resultf("no network %s", argv[2]);
				return (TCL_ERROR);
			}
			if (net->rchannel() < 0) {
				tcl.resultf("network not open");
				return (TCL_ERROR);
			}
			ch_[layer].net(net);
			return (TCL_OK);
		}
	}
	return (RTP_Transmitter::command(argc, argv));
}

void RTP_Session::transmit(pktbuf* pb)
{
	Network* n = dh_[pb->layer].net();
	/*
	 * FIXME check for null network: can happen because sender/receiver
	 * subscriptions aren't yet completely separated.
	 * THIS NEEDS TO BE FIXED.
	 */
	if (n != 0)
		n->send(pb);
}

void RTP_Session::transmit_ctrl(pktbuf* pb)
{
	Network* n = ch_[pb->layer].DataHandler::net();
	/*
	 * FIXME check for null network: can happen because sender/receiver
	 * subscriptions aren't yet completely separated.
	 * THIS NEEDS TO BE FIXED.
	 */
	if (n != 0)
		n->send(pb);
}

u_char* RTP_Session::build_sdes_item(u_char* p, int code, Source& s)
{
	const char* value = s.sdes(code);
	if (value != 0) {
		int len = strlen(value);
		*p++ = code;
		*p++ = len;
		memcpy(p, value, len);
		p += len;
	}
	return (p);
}

int RTP_Session::build_sdes(rtcphdr* rh, Source& ls)
{
	int flags = RTP_VERSION << 14 | 1 << 8 | RTCP_PT_SDES;
	rh->rh_flags = htons(flags);
	rh->rh_ssrc = ls.srcid();  // Source::srcid returns net order
	u_char* p = (u_char*)(rh + 1);
	p = build_sdes_item(p, RTCP_SDES_CNAME, ls);

	/*
	 * We always send a cname plus one other sdes
	 * There's a schedule for what we send sequenced by sdes_seq_:
	 *   - send 'email' every 0th & 4th packet
	 *   - send 'note' every 2nd packet
	 *   - send 'tool' every 6th packet
	 *   - send 'name' in all the odd slots
	 * (if 'note' is not the empty string, we switch the roles
	 *  of name & note)
	 */
	int nameslot, noteslot;
	const char* note = ls.sdes(RTCP_SDES_NOTE);
	if (note) {
		if (*note) {
			nameslot = RTCP_SDES_NOTE;
			noteslot = RTCP_SDES_NAME;
		} else {
			nameslot = RTCP_SDES_NAME;
			noteslot = RTCP_SDES_NOTE;
		}
	} else {
		nameslot = RTCP_SDES_NAME;
		noteslot = RTCP_SDES_NAME;
	}
	u_int seq = (++sdes_seq_) & 0x7;
	switch (seq) {

	case 0:  case 4:
		p = build_sdes_item(p, RTCP_SDES_EMAIL, ls);
		break;

	case 2:
		p = build_sdes_item(p, noteslot, ls);
		break;
	case 6:
		p = build_sdes_item(p, RTCP_SDES_TOOL, ls);
		break;
	case 5:
		p = build_sdes_item(p, RTCP_SDES_LOC, ls);
		break;
	default:
		p = build_sdes_item(p, nameslot, ls);
	}
	int len = p - (u_char*)rh;
	int pad = 4 - (len & 3);
	len += pad;
	rh->rh_len = htons((len >> 2) - 1);
	while (--pad >= 0)
		*p++ = 0;

	return (len);
}

void RTP_Session::announce(CtrlHandler* ch)
{
	send_report(ch, 0);
}

/*FIXME check for buffer overflow*/
/*
 * Send an RTPv2 report packet.
 */
void RTP_Session::send_report(CtrlHandler* ch, int bye, int app)
{
	static u_int8_t buf[2*RTP_MTU];
	Source& s = *sm_->localsrc();
	rtcphdr* rh = (rtcphdr*)buf;
	rh->rh_ssrc = s.srcid();   // Source::srcid returns net order
	int flags = RTP_VERSION << 14;
	timeval now = unixtime();
	int layer = ch - ch_;
	Source::Layer& sl = s.layer(layer);
	sl.lts_ctrl(now);
	int we_sent = 0;
	rtcp_rr* rr;
	/*
	 * If we've sent data since our last sender report send a
	 * new report.  The MediaTimer check is to make sure we still
	 * have a grabber -- if not, we won't be able to interpret the
	 * media timestamps so there's no point in sending an SR.
	 */
	MediaTimer* mt = MediaTimer::instance();
	if (sl.np() != last_np_ && (mt || (sl.ref_ntp_sec() != 0))) {
		last_np_ = sl.np();
		we_sent = 1;
		flags |= RTCP_PT_SR;
		rtcp_sr* sr = (rtcp_sr*)(rh + 1);
		sr->sr_ntp = ntp64time(now);
		HTONL(sr->sr_ntp.upper);
		HTONL(sr->sr_ntp.lower);
		if (sl.ref_ntp_sec() != 0) {
		  /* Generate ts from reference ntp and mts
		   * FIXME - 90kHz clock is assumed here, but needs
		   *       to be configurable.
		   */
		  double ntp_now;
		  double ntp_then;

		  ntp_now = ((double) ntohl(sr->sr_ntp.upper)) +
		    (((double) ntohl(sr->sr_ntp.lower))/4294967296.0);
		  ntp_then = ((double) sl.ref_ntp_sec()) +
		    (((double) sl.ref_ntp_fsec())/4294967296.0);

		  sr->sr_ts = htonl(((int) ((ntp_now - ntp_then) * 90000.0)) +
				    sl.ref_mts());
		} else {
		  sr->sr_ts = htonl(mt->ref_ts());
		}
		sr->sr_np = htonl(sl.np());
		sr->sr_nb = htonl(sl.nb());
		rr = (rtcp_rr*)(sr + 1);
	} else {
		flags |= RTCP_PT_RR;
		rr = (rtcp_rr*)(rh + 1);
	}
	int nrr = 0;
	int nsrc = 0;
	/*
	 * we don't want to inflate report interval if user has set
	 * the flag that causes all sources to be 'kept' so we
	 * consider sources 'inactive' if we haven't heard a control
	 * msg from them for ~32 reporting intervals.
	 */
	u_int inactive = u_int(ch->rint() * (32./1000.));
	if (inactive < 2)
		inactive = 2;
	for (Source* sp = sm_->sources(); sp != 0; sp = sp->next_) {
		++nsrc;
		Source::Layer& sl = sp->layer(layer);
		int received = sl.np() - sl.snp();
		if (received == 0) {
			if (u_int(now.tv_sec - sl.lts_ctrl().tv_sec) > inactive)
				--nsrc;
			continue;
		}
		sl.snp(sl.np());
		rr->rr_srcid = sp->srcid();
		int expected = sl.ns() - sl.sns();
		sl.sns(sl.ns());
		u_int32_t v;
		int lost = expected - received;
		if (lost <= 0)
			v = 0;
		else
			/* expected != 0 if lost > 0 */
			v = ((lost << 8) / expected) << 24;
		/* FIXME should saturate on over/underflow */
		v |= (sl.ns() - sl.np()) & 0xffffff;
		rr->rr_loss = htonl(v);
		rr->rr_ehsr = htonl(sl.ehs());
		PacketHandler* handler = (PacketHandler*)sp->handler();
		rr->rr_dv = (handler != 0) ? htonl(handler->delvar()) : 0;
		rr->rr_lsr = htonl(sl.sts_ctrl());
		if (sl.lts_ctrl().tv_sec == 0)
			rr->rr_dlsr = 0;
		else {
			u_int32_t ntp_now = ntptime(now);
			u_int32_t ntp_then = ntptime(sl.lts_ctrl());
			rr->rr_dlsr = htonl(ntp_now - ntp_then);
		}
		++rr;
		if (++nrr >= 31)
			break;
	}
	flags |= nrr << 8;
	rh->rh_flags = htons(flags);
	int len = (u_char*)rr - buf;
	rh->rh_len = htons((len >> 2) - 1);

	int plen;
	u_char* p = (u_char*)rr;
	if (!bye) {
		plen = build_sdes((rtcphdr*)p, s);
		p += plen;
		len += plen;
	}
	if (app) {
		plen = build_app((rtcphdr*)p, s, we_sent);
		p += plen;
		len += plen;
	}
	if (bye) {
		 plen = build_bye((rtcphdr*)p, s);
		 len += plen;
	}

	ch->send(buf, len);
	ch->adapt(nsrc, nrr, we_sent);
	ch->sample_size(len);
	if (layer == 0)
		sm_->CheckActiveSources(ch->rint());
}

int RTP_Session::build_bye(rtcphdr* rh, Source& ls)
{
	int flags = RTP_VERSION << 14 | 1 << 8 | RTCP_PT_BYE;
	rh->rh_flags = ntohs(flags);
	rh->rh_len = htons(1);
	rh->rh_ssrc = ls.srcid();   // Source::srcid returns net order
	return (8);
}

void RTP_Session::recv(DataHandler* dh)
{
	int layer = dh - dh_;
	pktbuf* pb = pool_->alloc(layer);
	u_int32_t addr;
	int port;
	int cc = dh->recv(pb->data, sizeof(pb->data), addr, port);
	if (cc <= 0) {
		pb->release();
		return;
	}

	int version = pb->data[0] >> 6;
	if (version != 2) {
		++badversion_;
		pb->release();
		return;
	}
	if ((size_t) cc < sizeof(rtphdr)) {
		++nrunt_;
		pb->release();
		return;
	}
	pb->len = cc;
	demux(pb, addr);
}

void RTP_Session::demux(pktbuf* pb, u_int32_t addr)
{
	rtphdr* rh = (rtphdr*)pb->data;
	u_int32_t srcid = rh->rh_ssrc;
	int flags = ntohs(rh->rh_flags);
#if 0 // MattD: BAD behavior: make compliant server but flexible client!!!
	if ((flags & RTP_X) != 0) {
		/*
		 * the minimal-control audio/video profile
		 * explicitly forbids extensions
		 */
		++badext_;
		pb->release();
		return;
	}
#endif
	/*
	 * Check for illegal payload types.  Most likely this is
	 * a session packet arriving on the data port.
	 */
	int fmt = flags & 0x7f;
	if (!check_format(fmt)) {
		++badfmt_;
		pb->release();
		return;
	}

	u_int16_t seqno = ntohs(rh->rh_seqno);
	Source* s = sm_->demux(this, srcid, addr, seqno, pb->layer);
	if (s == 0) {
		/*
		 * Takes a pair of validated packets before we will
		 * believe the source.  This prevents a runaway
		 * allocation of Source data structures for a
		 * stream of garbage packets.
		 */
		pb->release();
		return;
	}

	Source::Layer& sl = s->layer(pb->layer);
	timeval now = unixtime();
	sl.lts_data(now);
	sl.sts_data(rh->rh_ts);

	// extract CSRC count (CC field); increment pb-dp data pointer
	// & adjust length accordingly
	int cnt = (flags >> 8) & 0xf;
	if (cnt > 0) {
		/*FIXME move header up so it's contiguous with data*/
		rtphdr hdr = *rh;
		pb->dp += (cnt << 2);
		pb->len -= (cnt << 2);
		u_int32_t* bp = (u_int32_t*)(rh + 1);
		while (--cnt >= 0) {
			u_int32_t csrc = *(u_int32_t*)bp;
			bp += 4;
			Source* cs = sm_->lookup(this, csrc, srcid, addr);
			cs->layer(pb->layer).lts_data(now);
			cs->action();
		}
		rh = (rtphdr*)pb->dp;
		*rh = hdr;
	} else
		s->action();

	// MattD: be compliant and skip over extension headers
	if ((flags & RTP_X) != 0) {
	        rtphdr hdr = *rh;
		/*
		 * the minimal-control audio/video profile
		 * explicitly forbids extensions
		 */
	        // if no data present after standard RTP header, stop
   	        if(pb->len<4) {
		  pb->release();
		  return;
 	        }
  	        //unsigned short* sh=(unsigned short*)( (char*)(rh+1) /*pb->dp+2*/);
		//int x_type=(int)(ntohs(*sh));
		unsigned short * sh=(unsigned short*)( (char*)(rh+1)+2 /*pb->dp+2*/);
		int x_length=(int)(ntohs(*sh))<<2;
		if(pb->len< 4 + x_length ) {
		  pb->release();
		  return;
		}
		// printf("type=%d, advacing pointers by %d\n", x_type, 4+x_length);
	        pb->dp += (4 + x_length);
		pb->len -= (4 + x_length);
		// ++badext_;
		// pb->release();
		// return;
		rh = (rtphdr*)pb->dp;
		*rh = hdr;
	}

	/*
	 * This is a data packet.  If the source needs activation,
	 * or the packet format has changed, deal with this.
	 * Then, hand the packet off to the packet handler.
	 * FIXME might want to be careful about flip-flopping
	 * here when format changes due to misordered packets
	 * (easy solution -- keep rtp seqno of last fmt change).
	 */
	PacketModule* h = s->handler();
	if (h == 0)
		h = s->activate(fmt);
	else if (s->format() != fmt)
		h = s->change_format(fmt);

	/*
	 * FIXME bit rate doesn't include rtpv1 options;
	 * but v1 is going away anyway.
	 */
	int dup = sl.cs(seqno, s);
	sl.np(1);
	sl.nb(pb->len);
	if (dup) {
		pb->release();
		return;
	}
	if (flags & RTP_M)
		sl.nf(1);

#ifdef notdef
	/* This should move to the handler */
	/*FIXME could get rid of hdrlen and move run check into recv method*/
	int hlen = h->hdrlen();
	if (pb->len < hlen + sizeof(rtphdr)) {
		sl.runt(1);
		pb->release();
		return;
	}
#endif
	if (s->mute()) {
		pb->release();
		return;
	}
	h->recv(pb);
}

void RTP_Session::parse_rr_records(u_int32_t, rtcp_rr*, int,
				      const u_char*, u_int32_t)
{
}

void RTP_Session::parse_sr(rtcphdr* rh, int flags, u_char*ep,
			      Source* ps, u_int32_t addr, int layer)
{
	rtcp_sr* sr = (rtcp_sr*)(rh + 1);
	Source* s;
	u_int32_t ssrc = rh->rh_ssrc;
	if (ps->srcid() != ssrc)
		s = sm_->lookup(this, ssrc, ssrc, addr);
	else
		s = ps;

	Source::Layer& sl = s->layer(layer);
	sl.lts_ctrl(unixtime());
	sl.sts_ctrl(ntohl(sr->sr_ntp.upper) << 16 |
		    ntohl(sr->sr_ntp.lower) >> 16);
	sl.ntp_ts_sec(ntohl(sr->sr_ntp.upper));
	sl.ntp_ts_fsec(ntohl(sr->sr_ntp.lower));
	sl.mts(ntohl(sr->sr_ts));

	int cnt = flags >> 8 & 0x1f;
	s->trigger_sr();
	parse_rr_records(ssrc, (rtcp_rr*)(sr + 1), cnt, ep, addr);
}

void RTP_Session::parse_rr(rtcphdr* rh, int flags, u_char* ep,
			      Source* ps, u_int32_t addr, int layer)
{
	Source* s;
	u_int32_t ssrc = rh->rh_ssrc;
	if (ps->srcid() != ssrc)
		s = sm_->lookup(this, ssrc, ssrc, addr);
	else
		s = ps;

	s->layer(layer).lts_ctrl(unixtime());
	int cnt = flags >> 8 & 0x1f;
	parse_rr_records(ssrc, (rtcp_rr*)(rh + 1), cnt, ep, addr);
}

int RTP_Session::sdesbody(u_int32_t* p, u_char* ep, Source* ps,
			     u_int32_t addr, u_int32_t ssrc, int layer)
{
	Source* s;
	u_int32_t srcid = *p;
	if (ps->srcid() != srcid)
		s = sm_->lookup(this, srcid, ssrc, addr);
	else
		s = ps;
	if (s == 0)
		return (0);
	/*
	 * Note ctrl packet since we will never see any direct ctrl packets
	 * from a source through a mixer (and we don't want the source to
	 * time out).
	 */
	s->layer(layer).lts_ctrl(unixtime());

	u_char* cp = (u_char*)(p + 1);
	while (cp < ep) {
		char buf[256];

		u_int type = cp[0];
		if (type == 0) {
			/* end of chunk */
			return (((cp - (u_char*)p) >> 2) + 1);
		}
		u_int len = cp[1];
		u_char* eopt = cp + len + 2;
		if (eopt > ep)
			return (0);

		if (type >= RTCP_SDES_MIN && type <= RTCP_SDES_MAX) {
			memcpy(buf, (char*)&cp[2], len);
			buf[len] = 0;
			s->sdes(type, buf);
		} else
			/*FIXME*/;

		cp = eopt;
	}
	return (0);
}

void RTP_Session::parse_sdes(rtcphdr* rh, int flags, u_char* ep, Source* ps,
				u_int32_t addr, u_int32_t ssrc, int layer)
{
	int cnt = flags >> 8 & 0x1f;
	u_int32_t* p = (u_int32_t*)&rh->rh_ssrc;
	while (--cnt >= 0 && (u_char*)p < ep) {
		int n = sdesbody(p, ep, ps, addr, ssrc, layer);
		if (n == 0)
			break;
		p += n;
	}
	if (cnt >= 0)
		ps->badsdes(1);
}

void RTP_Session::parse_bye(rtcphdr* rh, int flags, u_char* ep, Source* ps)
{
	int cnt = flags >> 8 & 0x1f;
	u_int32_t* p = (u_int32_t*)&rh->rh_ssrc;

	while (--cnt >= 0) {
		if (p >= (u_int32_t*)ep) {
			ps->badbye(1);
			return;
		}
		Source* s;
		if (ps->srcid() != rh->rh_ssrc)
			s = sm_->consult(*p);
		else
			s = ps;
		if (s != 0)
			s->lts_done(unixtime());
		++p;
	}
}

/*
 * Receive an rtcp packet (from the control port).
 */
void RTP_Session::recv(CtrlHandler* ch)
{
	int layer = ch - ch_;
	pktbuf* pb = pool_->alloc(layer);
	u_int32_t src;
	int port;
	int cc = ch->recv(pb->data, sizeof(pb->data), src, port);
	if (cc <= 0) {
		pool_->release(pb);
		return;
	}

	rtcphdr* rh = (rtcphdr*)pb->data;
	if (cc < int(sizeof(*rh))) {
		++nrunt_;
		pool_->release(pb);
		return;
	}
	/*
	 * try to filter out junk: first thing in packet must be
	 * sr, rr or bye & version number must be correct.
	 */
	switch(ntohs(rh->rh_flags) & 0xc0ff) {
	case RTP_VERSION << 14 | RTCP_PT_SR:
	case RTP_VERSION << 14 | RTCP_PT_RR:
	case RTP_VERSION << 14 | RTCP_PT_BYE:
		break;
	default:
		/*
		 * FIXME should further categorize this error -- it is
		 * likely that people mis-implement applications that
		 * don't put something other than SR,RR,BYE first.
		 */
		++badversion_;
		pool_->release(pb);
		return;
	}
	/*
	 * at this point we think the packet's valid.  Update our average
	 * size estimator.  Also, there's valid ssrc so charge errors to it
	 */
	ch->sample_size(cc);
	u_int32_t addr = src;

	/*
	 * First record in compound packet must be the ssrc of the
	 * sender of the packet.  Pull it out here so we can use
	 * it in the sdes parsing, since the sdes record doesn't
	 * contain the ssrc of the sender (in the case of mixers).
	 */
	u_int32_t ssrc = rh->rh_ssrc;
	Source* ps = sm_->lookup(this, ssrc, ssrc, addr);
	if (ps == 0) {
		pool_->release(pb);
		return;
	}

	/*
	 * Outer loop parses multiple RTCP records of a "compound packet".
	 * There is no framing between records.  Boundaries are implicit
	 * and the overall length comes from UDP.
	 */

	u_char* epack = (u_char*)rh + cc;
	int valid = 1;
	while ((u_char*)rh < epack) {
		u_int len = (ntohs(rh->rh_len) << 2) + 4;
		u_char* ep = (u_char*)rh + len;
		if (ep > epack) {
			ps->badsesslen(1);
			pool_->release(pb);
			return;
		}
		u_int flags = ntohs(rh->rh_flags);
		if (flags >> 14 != RTP_VERSION) {
			ps->badsessver(1);
			pool_->release(pb);
			return;
		}
		switch (flags & 0xff) {

		case RTCP_PT_SR:
			parse_sr(rh, flags, ep, ps, addr, layer);
			break;

		case RTCP_PT_RR:
			parse_rr(rh, flags, ep, ps, addr, layer);
			break;

		case RTCP_PT_SDES:
			parse_sdes(rh, flags, ep, ps, addr, ssrc, layer);
			break;

		case RTCP_PT_BYE:
			parse_bye(rh, flags, ep, ps);
			break;

		case RTCP_PT_APP:
			parse_app(rh, flags, ep, ps);
			break;

		default:
		        valid = 0;
			ps->badsessopt(1);
			break;
		}
		rh = (rtcphdr*)ep;
	}
	if (valid && ps->ctrl_handler() != 0)
		ps->ctrl_handler()->recv(pb);
	else
		pool_->release(pb);
}

void RTP_Session::parse_app(rtcphdr*, int, u_char*, Source*)
{
}

int RTP_Session::build_app(rtcphdr*, Source&, int)
{
	return (0);
}

int VideoSession::build_app(rtcphdr* rr, Source& ls, int we_sent)
{
	rtcphdr* rh = rr;
	if (rtcp_thumbnail_ && we_sent)
		rh = (rtcphdr*)build_rtcp_thumbnail(rh, ls);

	return ((u_char*)rh - (u_char*)rr);
}

u_char* VideoSession::build_rtcp_thumbnail(rtcphdr* rh, Source& ls)
{
	/* FIXME right now, all we do is signal that source is active */
	int flags = RTP_VERSION << 14 | 1 << 8 | RTCP_PT_APP;
	rh->rh_flags = htons(flags);
	rh->rh_ssrc = ls.srcid();  // Source::srcid returns net order
	rtcp_thumbnail* tn = (rtcp_thumbnail*)(rh + 1);

	/* Just so we don't get confused if some application happens to send
	 * an APP packet.
	 */
	tn->magic = htons(0xbeef);
	tn->format = htons(ls.format());
	u_char* ep = ((u_char*)(tn + 1));
	int len = ep - (u_char*)rh;
	rh->rh_len = htons((len >> 2) - 1);
	return (ep);
}

void VideoSession::parse_app(rtcphdr* rh, int, u_char*, Source* ps)
{
	u_int32_t* p = (u_int32_t*)&rh->rh_ssrc;
	Source* s;
	if (ps->srcid() != rh->rh_ssrc)
		s = sm_->consult(*p);
	else
		s = ps;
	/* right now just check for rtcp_thumbnail */
	rtcp_thumbnail* tn = (rtcp_thumbnail*)(rh + 1);
	if (tn->magic != ntohs(0xbeef)) {
		return;
	}
	PacketModule* h = s->handler();
	if (h == 0)
		h = s->activate(ntohs(tn->format));
}

int VideoSession::command(int argc, const char*const* argv)
{
	if (argc == 3) {
		if (strcmp(argv[1], "rtcp-thumbnail") == 0) {
			rtcp_thumbnail_ = atoi(argv[2]);
			return (TCL_OK);
		}
	}

	return (RTP_Session::command(argc, argv));
}

void VideoSession::announce(CtrlHandler* ch)
{
	send_report(ch, 0, rtcp_thumbnail_);
}

VideoSession::VideoSession()
	:rtcp_thumbnail_(0)
{
}
