# control.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import AnnounceListenManager ZTrace MTrace URL_Get_Timer URL_Resp_Timer

set mtrace [MTrace init {trcWC}]
$mtrace toggle_window

set ztrace [ZTrace init]

#
# A control object for the web cache. It takes care of
# communications amoung all caches in a multicast session.
#
Class WebCacheControl -superclass AnnounceListenManager

#
# The web cache control control. <i>netspec</i> is the multicast
# address passed to the announce listen manager, and <i>wc</i> is
# the web cache component.
#
WebCacheControl public init { d1 d2 e1 e2 netspec } {
	$self next $netspec
	$self instvar resp_hit_c1_ resp_hit_c2_ resp_hit_d_
	$self instvar resp_miss_c1_ resp_miss_c2_ resp_miss_d_
	$self instvar get_c1_ get_c2_ get_d_

	# default random delay values; set to wait a minimum of
	# two times the estimated rtt (two times the  d values below)
	# to neighbors and choose a delay from an interval that is twice the
	# "length" of the estimated rtt.
	set get_c1_ 2.0
	set get_c2_ 2.0
	set resp_hit_c1_ $d1
	set resp_hit_c2_ $d2
	set resp_miss_c1_ $e1
	set resp_miss_c2_ $e2
	set get_d_ .25
	set resp_hit_d_ .25
	set resp_miss_d_ .25
}

#
# Create URL_Get_Timer for <i>url</i>.
#
WebCacheControl public create_get_timer { url } {
	$self instvar gettimer_table_ get_c1_ get_c2_ get_d_ cache_

	# create a URL_Get_Timer
	if ![info exists gettimer_table($url)] {
		mtrace trcWC "control: create_get_timer $url"
		set gettimer [new URL_Get_Timer $url $get_c1_ $get_c2_ $get_d_ $self]
		set gettimer_table_($url) $gettimer
	}
}

#
# Called when data arrived from the multicast session thereby
# the get and response timers associated with the url need to
# be canceled.
#
WebCacheControl public cancel_all_timers { url } {
	$self instvar cache_ gettimer_table_ resptimer_table_

	mtrace trcWC "control: cancel all timers $url"

	# cancel all timers and pending messages associated with the url
	if { [info exists gettimer_table_($url)] } {
		$gettimer_table_($url) destroy
		unset gettimer_table_($url)
	}
	if { [info exists resptimer_table_($url)] } {
		$resptimer_table_($url) destroy
		unset resptimer_table_($url)
	}
}


WebCacheControl public win_resp_timer { url } {
	$self instvar cache_ gettimer_table_ resptimer_table_

	mtrace trcWC "control: win_resp_timer $url"

	# cancel the URL_Resp_Timer
	if [info exists resptimer_table_($url)] {
		$resptimer_table_($url) destroy
		unset resptimer_table_($url)
	}

	# backoff the URL_Get_Timer
	if [info exists gettimer_table_($url)] {
		$gettimer_table_($url) backoff
	}

	# send a reply pending message if we don't have the
	# data.
	if { [$cache_ hit $url] == "" } {
		$self send_announcement WC_RESP_PEND $url
	}

	# ask the cache to send data associated with the url
	$cache_ send_data $url
}

#
# Send messages when timers expire. If a get message is
# being sent, this method also set a local resp timer. If
# a pending message is being sent, this method also cancel
# the assocaited resp timer.
#
WebCacheControl public send_announcement { type url { args "" } } {
	$self instvar resptimer_table_ resp_miss_c1_ resp_miss_c2_ \
		resp_miss_d_

	mtrace trcWC "control: send_announcement $type"

	switch -- $type {
		WC_URL_GET {
			$self announce "WC_URL_GET $url"

			# should not create multiple get timers for the
			# same url. this is ok because we forget about
			# the timer in the array once the srm receiver
			# all the data.
			if ![info exists resptimer_table_($url)] {
				set resptimer_table_($url) \
					[new URL_Resp_Timer $url \
					$resp_miss_c1_ $resp_miss_c2_ \
					$resp_miss_d_ $self]
			}
		}
		WC_RESP_PEND {
			$self announce "WC_RESP_PEND $url"
		}
	}
}

#
# Receive messages from other caches. There are currently the
# following messages:
# WC_URL_GET means that another cache has won the timer war so
# this cache should backoff its get timer. At the same time,
# it initiates a response timer, which decides which cache will
# send the data back to the session.
# WC_RESP_PEND means that another cache has won the response
# timer war and is currently getting/sending the data. This
# cancels the response timer, and backoffs the get timer.
#
WebCacheControl public recv_announcement { addr port mesg len } {
	$self instvar gettimer_table_ resptimer_table_ cache_ \
		resp_hit_c1_ resp_hit_c2_ resp_miss_c1_ resp_miss_c2_ \
		resp_hit_d_ resp_miss_d_

	mtrace trcWC "control: recv_announcement $mesg"

	set type [lindex $mesg 0]
	set url [lindex $mesg 1]

	switch -- $type {
		WC_URL_GET {
			# backoff its own URL_Get_Timer
			if [info exists gettimer_table_($url)] {
				$gettimer_table_($url) backoff
			}

			# check whether it has data in the local cache
			if { [$cache_ hit $url] != "" } {
				set c1 $resp_hit_c1_
				set c2 $resp_hit_c2_
				set d $resp_hit_d_
			} else {
				set c1 $resp_miss_c1_
				set c2 $resp_miss_c2_
				set d $resp_miss_d_
			}

			# again, do not create multiple timers, because it
			# confuses the destroy timer operation.
			if ![info exists resptimer_table_($url)] {
				set resptimer [new URL_Resp_Timer $url $c1 $c2 $d $self]
				set resptimer_table_($url) $resptimer
			}
		}
		WC_RESP_PEND {

			# cancel its URL_Resp_Timer if it doesn't have the
			# data since some other node has requested the data
			# from the origin server. if it does have the data,
			# do nothing.

			if { [$cache_ hit $url] == "" } {
				if [info exists resptimer_table_($url)] {
					$resptimer_table_($url) destroy
					unset resptimer_table_($url)
				}

				# backoff the URL_Get_Timer
				if [info exists gettimer_table_($url)] {
					$gettimer_table_($url) backoff
				}
			}
		}
	}
}

