# application-specialfx.tcl --
#
#       SDS app that provides interface to the Special FX functions
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import Application
import SSAC_Client
import CSdsService
import CServiceManager
import CService

Class CSpecialFxApplication -superclass Application

##############################################################################
#
# CSpecialFxApplication instproc InitArgs { options } {
#
# Input:
# options - the options object
#
# Output:
# none
#
# Description:
# Registers all options before the command line argument is parsed
#
##############################################################################
CSpecialFxApplication instproc InitArgs { options } {
    # for the service discovery service
    $options register_option -sa optServiceAddress
    $options register_option -sp optServicePort
    $options register_option -st optServiceTTL

    # for the service discovery service
    $options register_option -rp optComPort
}

##############################################################################
#
# CSpecialFxApplication instproc InitResources { options } {
#
# Input:
# options - the options object
#
# Output:
# none
#
# Description:
# Gives defaults for options
#
##############################################################################
CSpecialFxApplication instproc InitResources { options } {

    # for the service discovery service
    $options add_default optServiceAddress "224.4.6.8"
    $options add_default optServicePort "12344"
    $options add_default optServiceTTL "16"

    # for the service discovery service
    $options add_default optComPort "11412"
}


##############################################################################
#
# CSpecialFxApplication instproc init { argv } {
#
# Input:
# argv - command line input
#
# Output:
# none
#
# Description:
# Constructor for the object.
#
##############################################################################
CSpecialFxApplication instproc init { argv } {
    $self next Replay


    # Initiailization of options
    set options [$self options]
    $self InitArgs $options
    $self InitResources $options
    $options load_preferences "specialfx"
    set argv [$options parse_args $argv]


    # create the sds session object
    set inetServiceAddr [$options get_option optServiceAddress]
    set iServicePort [$options get_option optServicePort]
    set iServiceTTL [$options get_option optServiceTTL]

    $self instvar m_sdsService
    set m_sdsService [new CSdsService $self $inetServiceAddr $iServicePort \
	    $iServicePort $iServiceTTL]

    # setup the service objects
    $self instvar m_serviceManager
    $self instvar m_service

    set iPort [$options get_option optComPort]
    set m_serviceManager [new CServiceManager "ServiceApp" "SpecialFx" $iPort]
    $m_serviceManager Attach $self

}

##############################################################################
#
# CSpecialFxApplication public GetSdsServiceData { } {
#
# Input:
# none
#
# Output:
# the data that the will go out to the sds system
#
# Description:
# This is a callback function for the service discovery system.  Need to
# return the data that will go out to the sds system.  So far there are
# three fields with their values
#
##############################################################################
CSpecialFxApplication public GetSdsServiceData { } {
    set options [$self options]
    set iComPort [$options get_option optComPort]
    set hostname [exec hostname]

    set location "Locationless"
    set type "Special FX"

    set data [list "UPDATE:" [concat $hostname $iComPort] \
	    [list [list "LOCATION:" $location] \
	    	  [list "TYPE:" $type]]]

    return $data
}

##############################################################################
#
# CSpecialFxApplication instproc NewConnection { service }
#
# Input:
# service - the new service that got connected
#
# Output:
# 1 - if handling this service
# 0 - otherwise
#
# Description:
# A call back function for the service manager.  This function will be called
# when a new connection has just been noticed by the service manager
#
##############################################################################
CSpecialFxApplication instproc NewConnection { service } {
    $self instvar m_service

    set m_service $service

    # Add the service calls
    $m_service MapMessage "SYN_SERVICE_IMAGE" $self SynServiceImage
    $m_service MapMessage "GET_UI_WINDOW" $self GetUIWindow
    $m_service MapMessage "START_FX" $self StartFx
    $m_service MapMessage "CLOSE_LINK" $self CloseService

    return 1
}

##############################################################################
#
# CSpecialFxApplication public SynServiceImage { service arguments } {
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client when it wants the image or text
# to display the service.  The simple protocol goes as such:
# type type_dependent_arguments.  The type could be text and it's argument
# the ascii of title.  It could also be image and be a bitmap of the image
#
##############################################################################
CSpecialFxApplication public SynServiceImage { service arguments } {
    $service Send "SYN_SERVICE_IMAGE" "text SpecialFx"
}


##############################################################################
#
# CSpecialFxApplication public GetUIWindow { service arguments } {
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client when it wants the code
# to make the window.  It should send back the code to build the window.
# There are two things that are assumed to be set by the client.  One is
# winFrame the window to draw into.  Another is service, which is the
# m_service object to send back to yourself.
#
##############################################################################
CSpecialFxApplication public GetUIWindow { service arguments } {
    set cmd ""

    #################################################################
    #
    # Irene fill code here
    #
    #################################################################
    append cmd "regsub -all -- {\\\.} \$winFrame {_} __name \n"

    append cmd "frame \$winFrame.label \n"
    append cmd "pack \$winFrame.label -side top -fill x -expand 1\n"

    append cmd "label \$winFrame.label.label -text \"Please choose a SpecialFx:\" \n"
    append cmd "pack \$winFrame.label.label -side left \n"

    append cmd "frame \$winFrame.specialfx \n"
    append cmd "pack \$winFrame.specialfx -side top -fill x -expand 1\n"

    append cmd "global specialfx\$__name \n"
    append cmd "set specialfx\$__name H261FadeSubprogram \n"

    # Source 1 frames
    append cmd "frame \$winFrame.source1 \n"
    append cmd "label \$winFrame.source1.label -text \"SourceId 1:\" \n"
    append cmd "menubutton \$winFrame.source1.choice -textvariable __source1 -menu \
            \$winFrame.source1.choice.menu -relief raised\n"
    append cmd "menu \$winFrame.source1.choice.menu \n"

    # Source 2 frames
    append cmd "frame \$winFrame.source2 \n"
    append cmd "label \$winFrame.source2.label -text \"SourceId 2:\" \n"
    append cmd "menubutton \$winFrame.source2.choice -textvariable __source2 -menu \
            \$winFrame.source2.choice.menu -relief raised\n"
    append cmd "menu \$winFrame.source2.choice.menu \n"

    # Fading radio button
    append cmd "radiobutton \$winFrame.specialfx.fade -variable specialfx\$__name \
	    -text Fading -value H261FadeSubprogram -command \"\$winFrame.source2.label configure \
            -fg black; \$winFrame.source2.choice configure -state normal\" \n"
    append cmd "pack \$winFrame.specialfx.fade -side top -fill x -expand 1\n"

    # Titling radio button
    append cmd "radiobutton \$winFrame.specialfx.title -variable specialfx\$__name \
            -text Titling -value H261TitleSubprogram -command \"\$winFrame.source2.label configure \
            -fg grey; \$winFrame.source2.choice configure -state disabled\" \n"
    append cmd "pack \$winFrame.specialfx.title -side bottom -fill x -expand 1\n"

    # Source Id 1
    append cmd "global __source1 \n"
append cmd "global value_irene \n"
append cmd "set value_irene \[llength \$liSourceId\] \n"
append cmd "puts \"value is \$value_irene\n\" \n"
    append cmd "if {\[llength \$liSourceId\] == 0} {\n"
append cmd "puts \"here \n\" \n"
       append cmd "set __source1 \"N\/A\" \n"
    append cmd "} else {\n"
       append cmd "set __source1 \[lindex \$liSourceId 0\] \n"
    append cmd "}\n"
    append cmd "pack \$winFrame.source1 -side top -fill x -expand 1\n"
    append cmd "pack \$winFrame.source1.label -side left -fill x -expand 1\n"

    append cmd "pack \$winFrame.source1.choice -side right -fill x -expand 1\n"
    append cmd "foreach item \$liSourceId { \n"
    append cmd "\$winFrame.source1.choice.menu add radio -label \$item  \
            -variable __source1 \n"
    append cmd "} \n"

    # Source Id 2
    append cmd "global __source2 \n"
    append cmd "if {\[llength \$liSourceId\] == 0} {\n"
       append cmd "set __source2 \"N\/A\" \n"
    append cmd "} else {\n"
       append cmd "set __source2 \[lindex \$liSourceId 0\] \n"
    append cmd "}\n"
    append cmd "pack \$winFrame.source2 -side top -fill x -expand 1\n"
    append cmd "pack \$winFrame.source2.label -side left -fill x -expand 1\n"

    append cmd "pack \$winFrame.source2.choice -side right -fill x -expand 1\n"
    append cmd "foreach item \$liSourceId { \n"
    append cmd "\$winFrame.source2.choice.menu add radio -label \$item  \
            -variable __source2 \n"
    append cmd "} \n"


    append cmd "frame \$winFrame.button \n"
    append cmd "pack \$winFrame.button -side top -fill x -expand 1\n"

    # Close button
    append cmd "button \$winFrame.button.close -text Close \
            -command \"destroy \$winFrame\" \n"
    append cmd "pack \$winFrame.button.close -side right\n"

    # Start button
    append cmd "button \$winFrame.button.start -text Start \
            -command \"global specialfx\$__name; global __source1; global __source2; destroy \$winFrame; \$service Send START_FX \\\"\$inetMStudioAddr \$iMStudioPort \$inetClientAddr \$iClientPort \\\$specialfx\$__name \\\$__source1 \\\$__source2 \\\\\\\"\\\{\$liSourceId\\\} \\\\\\\" \\\" \" \n"
    append cmd "pack \$winFrame.button.start -side right\n"

    $service Send "GET_UI_WINDOW" $cmd
}


##############################################################################
#
# CSpecialFxApplication public StartFx { service arguments } {
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client when it wants the start
# an effect
#
##############################################################################
CSpecialFxApplication public StartFx { service arguments } {

    # set up the arguments and launch the replay proxy
    set strExec "dc-specialfxproxy-5.0b3"
    append strExec " $arguments"

    puts "strExec - $strExec"

    if { [catch "eval exec $strExec &" pid] != 0 } {
	puts "special effext got an error starting the proxy: check the path"
	exit 0
    }
}


##############################################################################
#
# CSpecialFxApplication public CloseService { service arguments }
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client closes the service link.
#
##############################################################################
CSpecialFxApplication public CloseService { service arguments } {
    exit
}

