# ui-dropdown.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/ui_tools/ui-dropdown.tcl,v 1.20 2002/02/03 04:30:23 lim Exp $


import WidgetClass Icons

#
# This file defines the set of classes used for typical drop-down objects
#   DropDown: basic DropDown class as defined in Tk's optMenu.tcl
#   DropDown/Color: class for selecting one of many colors
#   DropDown/Font: class for selecting one of many fonts
#   DropDown/Text: this class allows you to either select one of many options
#                  or type in your own value
#
# Methods:
#   init: The constructor. PArams for this method are:
#         path - the pathname for the main widget for this object
#                (init builds the entire widget, but does not pack it)
#         default - the default startup value
#         values - list of value-command pairs: { {val1 cmd1} {val2 cmd2} ... }
#                  the command is optional - it is useful only if you wish
#                  to do something esoteric when the user selects that
#                  particular value
#                  For DropDown/Color objects, the values are Tk colors; to
#                  actually embed text such as "custom" (for custom colors)
#                  or something else, prefix the value by "/" i.e. use a value
#                  of "/custom". The DropDown/Color class already has a
#                  built-in command method for the "/custom" value, so you
#                  don't need to specify one yourself
#         entry_options: (only for DropDown/Text class) any special options
#                  for the entry widget inside this object (e.g. {-width 10})
#
#   build: this method actually builds the widgets inside the object
#
#   add: add a new value to the list
#
#   current: set the current value, which is displayed in the main widget
#
#   get: get the current value select in the object
#

WidgetClass DropDown -configspec {
	{ -variable variable Variable {} config_var cget_var }
	{ -value value Value {} config_value cget_value }
	{ -state state State {normal} config_state }
	{ -label label Label {} config_label }
} -alias {
	{ -var -variable }
} -default {
	{ *button.relief raised }
	{ *button.indicatorOn 1 }
	{ *button.highlightThickness 2 }
	{ *button.takeFocus 1 }
	{ *button.padX 2 }
	{ *button.padY 1 }
	{ *menu.tearOff 0 }
	{ *menu*borderWidth 1 }
	{ *menu*activeBorderWidth 1 }
}


DropDown instproc build_widget { path } {
	$self set label_ {}
	menubutton $path.button -menu $path.button.menu
	pack $path.button -fill both -expand 1 -padx 0 -pady 0 -side bottom
	menu $path.button.menu
	$self set_subwidget menu $path.button.menu
	$self config_var -variable {}

	set script [bind Menubutton <Key-space>]
	bind $path.button <Key-Down> $script
}


DropDown instproc config_var { option var } {
	$self instvar var_

	# remove the previous trace
	if { [info exists var_] && $var_!="" } {
		upvar #0 $var_ global_var
		catch { trace vdelete global_var w "$self var_trace" }
	}

	if { $var=="" } {
		set var_ [$self tkvarname defvar_]
	} else {
		set var_ $var
	}

	upvar #0 $var_ global_var
	trace variable global_var w "$self var_trace"
	if { ![info exists global_var] || $global_var=="" } {
		$self set_default_var
	} else {
		$self var_trace $var_ "" w
	}
}


DropDown instproc cget_var { option } {
	$self instvar var_
	if { $var_==[$self tkvarname defvar_] } {
		return ""
	} else {
		return $var_
	}
}


DropDown instproc config_value { option value } {
	$self set_var $value
}


DropDown instproc cget_value { option } {
	upvar #0 [$self set var_] global_var
	if [info exists global_var] {
		return $global_var
	} else {
		return ""
	}
}


DropDown instproc config_label { option args } {
	$self instvar label_
	if { [llength $args] == 0 } {
		return $label_
	} else {
		set label_ [lindex $args 0]
		set path [$self info path]
		if { $label_=={} } {
			if [winfo exists $path.label] {
				set bd [$path cget -bd]
				set relief [$path cget -relief]
				destroy $path.label
				$path.button configure -bd $bd -relief $relief
				$path configure -bd 0 -relief flat
			}
		} else {
			if ![winfo exists $path.label] {
				button $path.label -bd 0 -highlightthickness 0\
						-relief flat \
						-activebackground \
						[WidgetClass widget_default \
						-background]
				set bd [$path.button cget -bd]
				set relief [$path.button cget -relief]
				$path configure -bd $bd -relief $relief
				$path.button configure -bd 0 -relief flat
			}
			eval $path.label configure $label_
			pack $path.label -side top -padx 0 -pady 0 -fill x
		}
	}
}


DropDown instproc config_state { option {value {}} } {
	$self instvar label_
	if { $value=={} } {
		return [$self subwidget button cget -state]
	} else {
		$self subwidget button configure -state $value
		if { $label_ != {} } {
			$self subwidget label configure -state $value
		}
	}
}


DropDown instproc index { index } {
	if { $index=="end" } {
		set index [$self subwidget menu index $index]
		if { $index=="none" } {
			set index -1
		} else {
			incr index
		}
	} else {
		set index [$self subwidget menu index $index]
		if { $index=="none" } {
			set index -1
		}
	}

	return $index
}


DropDown instproc insert { index args } {
	set index [$self index $index]
	if { $index==-1 } {
		set index 0
	}

	foreach arg $args {
		$self insert_item $index $arg
		incr index
	}

	upvar #0 [$self set var_] global_var
	if { ![info exists global_var] || $global_var=="" } {
		$self set_default_var
	}
}


DropDown instproc insert_separator { index } {
	$self subwidget menu insert $index separator
}


DropDown instproc insert_item { index value } {
	if { [lindex $value 0] == "-image" } {
		$self subwidget menu insert $index command -image \
				[lindex $value 1] \
				-command "[list $self] set_var [list $value]"
	} else {
		$self subwidget menu insert $index command -label $value \
				-command "[list $self] set_var [list $value]"
	}
}


DropDown instproc delete { index1 {index2 {}} } {
	if { $index2=="" } { set index2 $index1 }
	$self subwidget menu delete $index1 $index2
}


DropDown instproc set_var { value } {
	upvar #0 [$self set var_] global_var
	set global_var $value
}


DropDown instproc set_default_var { } {
	set menu [$self subwidget menu]
	set last [$self index end]
	if { $last=="none" } { set last -1 }
	for { set idx 0 } { $idx < $last } { incr idx } {
		if { [$menu type $idx]=="command" } break
	}
	if { $idx < $last } {
		$menu invoke $idx
	} else {
		upvar #0 [$self set var_] global_var
		set global_var ""
	}
}


DropDown instproc var_trace { args } {
	upvar #0 [$self set var_] global_var
	if { [lindex $global_var 0] == "-image" } {
		$self subwidget button configure -image [lindex $global_var 1]
	} else {
		$self subwidget button configure -text $global_var
	}
}



WidgetClass DropDown/Color -superclass DropDown


DropDown/Color instproc insert_item { index value } {
	if { [string index $value 0]=="/" } {
		$self subwidget menu insert $index command \
				-label [list [string range $value 1 end]] \
				-command "[list $self] set_var [list $value]"
	} else {
		$self subwidget menu insert $index command -label {    } \
				-background [list $value] \
				-activebackground [list $value] \
				-command "[list $self] set_var [list $value]"
	}
}


DropDown/Color instproc var_trace { args } {
	upvar #0 [$self set var_] global_var
	if { $global_var=="" } return

	if { $global_var=="/custom" } {
		error "custom"
		set current_color [$self subwidget button cget -background]
		set color [tk_chooseColor -title "Choose color" \
				-initialcolor $current_color]
		if { $color=={} } return
		$self insert end $color
		set global_var $color
	}

	if { [string index $global_var 0]=="/" } {
		$self subwidget button configure -background \
				[WidgetClass widget_default -background] \
				-activebackground \
				[WidgetClass widget_default -background] \
				-text [string range $global_var 1 end]
	} else {
		$self subwidget button configure -background $global_var \
				-activebackground $global_var -text "    "
	}
}


WidgetClass DropDown/Font -superclass DropDown


DropDown/Font instproc insert_item { index value } {
	upvar \#0 [$self set var_] global_var

	$self subwidget menu insert $index command -label ABCabc \
			-font $value \
			-command "[list $self] set_var [list $value]"
	$self set index_($value) $index
	if { $global_var == $value } {
		$self var_trace
	}
}


DropDown/Font instproc var_trace { args } {
	upvar \#0 [$self set var_] global_var
	if { $global_var=="" } return

	#$self subwidget button configure -font $global_var -text "ABCabc"
	$self subwidget button configure -text ""
	$self instvar index_ last_
	if { [array exists last_] && $last_(index)!={} } {
		$self subwidget menu entryconfigure $last_(index) \
				-background $last_(color)
	}
	if ![info exists index_($global_var)] return
	if { ![array exists last_] || $last_(index) != $index_($global_var) } {
		set last_(index) $index_($global_var)
		set last_(color) [$self subwidget menu entrycget $last_(index)\
				-background]
	}
	$self subwidget menu entryconfigure $index_($global_var) \
			-background [WidgetClass widget_default -selectcolor]
}


# varChangedCmd gets called only when the drop down changes the
# value, or <Return> is pressed in the entry
#
WidgetClass DropDown/Text -superclass DropDown -configspec {
	{-entryVal entryVal EntryVal {} config_entryVal cget_entryVal}
} -default {
	{ .highlightThickness 2 }
	{ .takeFocus 0 }
	{ .relief sunken }
	{ .borderWidth 2 }
	{ *button.highlightThickness 0 }
	{ *button.takeFocus 0 }
	{ *button.padX 0 }
	{ *button.padY 0 }
	{ *entry.highlightThickness 0 }
	{ *entry.takeFocus 1 }
	{ *entry.relief flat }
	{ *entry.borderWidth 0 }
}

DropDown/Text instproc config_entryVal { option value} {
	$self subwidget entry delete 0 end
	$self subwidget entry insert 0 $value
}

DropDown/Text instproc cget_entryVal { option value} {
	return [$self subwidget entry get]
}

DropDown/Text instproc build_widget { path } {
	entry $path.entry
	pack $path.entry -side left -fill both -expand 1
	$self next $path
	pack configure $path.button -side right -fill y -expand 0

	set script [bind Menubutton <Key-space>]
	regsub -all -- {%W} $script $path.button new_script
	bind $path.entry <Key-Down> $new_script
	bind $path.entry <Return> "$self return_pressed \[%W get\]"
	bind $path.entry <FocusOut> "$self restore_entry"
}

DropDown/Text instproc return_pressed { text } {
	$self set_var $text
	set path [$self info path]
	$path.entry selection from 0
	$path.entry selection to end
	$path.entry icursor end
}


DropDown/Text instproc restore_entry {} {
	upvar #0 [$self set var_] global_var
	# use set_var here to trigger updates of entry values (if any)
	if [info exists global_var] {
		$self config_entryVal {} $global_var
	} else {
		$self config_entryVal {} {}
	}
}

DropDown/Text instproc config_state { option {value {}} } {
	if { $value=={} } {
		return [$self subwidget cget button -state]
	} else {
		$self subwidget button configure -state $value
		$self subwidget entry  configure -state $value
	}
}

DropDown/Text instproc set_var { value } {
	#$self config_entryVal {} $value
	$self next $value
}

DropDown/Text instproc clear { } {
	$self set_var ""
}

DropDown/Text instproc var_trace { args } {
	upvar #0 [$self set var_] global_var

	set path [$self info path]
	if { [$path.entry selection present] && [focus]==$path } {
		$self config_entryVal {} $global_var
		$path.entry selection from 0
		$path.entry selection to end
		$path.entry icursor end
	} else {
		$self config_entryVal {} $global_var
	}
}


DropDown/Text instproc ev_key_down_ { } {
	set button [$self subwidget button]
	set takefocus [$button cget -takefocus]
	$button configure -takefocus 1
	set oldfocus [focus]
	focus $button
	event generate $button <Key-space>
	focus $oldfocus
	$button configure -takefocus $takefocus
}


WidgetClass EntryWithHistory -superclass DropDown/Text -configspec {
	{ -maxhistory maxHistory MaxHistory 15 config_maxhist cget_maxhist }
}


EntryWithHistory instproc add_history { } {
	set value [$self cget -value]
	regsub -all -- {\*} $value {\*} val1
	regsub -all -- {\?} $val1  {\?} val2
	regsub -all -- {\[} $val2  {\[} val1
	regsub -all -- {\]} $val1  {\]} val2
	catch { $self subwidget menu delete "$val2" }
	$self insert 0 $value

	set index [$self subwidget menu index end]
	set maxhistory [$self cget_maxhist -maxhistory]
	if { $index!="none" && $index >= $maxhistory } {
		$self subwidget menu delete $maxhistory end
	}
}


EntryWithHistory instproc config_maxhist { option value } {
	$self instvar max_history_
	set max_history_ $value

	# delete any extra entries in the history list
	set index [$self subwidget menu index end]
	if { $index!="none" && $index >= $max_history_ } {
		$self subwidget menu delete $index end
	}
}


EntryWithHistory instproc cget_maxhist { option } {
	$self instvar max_history_
	if [info exists max_history_] { return $max_history_ } else {return 0}
}

