/*
 * tcl-api.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#ifndef lint
static const char rcsid[] = "@(#) $Header: /usr/mash/src/repository/srmv2/common/tcl-api.cc,v 1.25 2002/02/03 03:04:20 lim Exp $";
#endif


#include "srmv2-api.h"
#include <tcl.h>

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#ifdef WIN32
#  include <windows.h>
#  include <winsock.h>
#else
#  include <sys/param.h>
#  include <netdb.h>
#  include <sys/socket.h>
#  include <arpa/inet.h>
#endif



/*
 *  SRMv2 Tcl API
 *  -------------
 *
 *  set sess [srm_create_session addr sport rport ?ttl?]
 *  srm_destroy_session $sess
 *  srm_reset_session $sess addr sport rport ?ttl?
 *  srm_session_bandwidth $sess $bps
 *  set bps [srm_session_bandwidth $sess]
 *  srm_drop_probability $sess $dropProb
 *  set dropProb [srm_drop_probability $sess]
 *  set src  [srm_create_source $sess ?srcid?]
 *  srm_app_info $src $info
 *  set info [srm_app_info $src]
 *  set cid  [srm_calloc $src parent_cid ?name?]
 *  set name [srm_get_container_name $src $cid]
 *  set pcid [srm_get_parent_cid $src $cid]
 *  set seqno [srm_send $src $cid data]
 *  srm_recover $src $cid sseq eseq
 *  set srcid [srm_get_source_id $src]
 *  set sess [srm_get_session $src]
 *  srm_callbacks $sess recv should_recover read_adu source_update recv_cid
 *  set cb_list [srm_callbacks $sess]
 *       set recv_proc           [lindex $cb_list 0]
 *       set should_recover_proc [lindex $cb_list 1]
 *       set read_adu_proc       [lindex $cb_list 2]
 *       set source_update_proc  [lindex $cb_list 3]
 *       set recv_cid_proc       [lindex $cb_list 4]
 *
 *  SRMv2 Tcl Callbacks
 *  -------------------
 *
 *  proc srm_recv { src cid seqno data } {
 *      # returns nothing
 *  }
 *  proc srm_should_recover { src cid sseq eseq } {
 *      # returns boolean
 *  }
 *  proc srm_read_adu { src cid seqno } {
 *      # returns the data associated with (src,cid,seqno)
 *  }
 *  proc srm_source_update { src info } {
 *      # returns nothing
 *  }
 *  proc srm_recv_cid { src cid parent_cid name } {
 *      # returns nothing
 *  }
 *
 */

#define SRM2TCL(p) Tcl_CreateObjCommand(interp, #p, p ## _proc, NULL, NULL)


enum {
	CB_RECV,
	CB_SHOULD_RECOVER,
	CB_READ_ADU,
	CB_SOURCE_UPDATE,
	CB_RECV_CID,
	CB_NUM_CALLBACKS
};


struct Tcl_SRM_UserHook {
	char *callbacks[CB_NUM_CALLBACKS];
	Tcl_Interp *interp;
};


static unsigned int
lookup_host_addr(const char *s)
{
	if (isdigit(*s))
		return (unsigned int)inet_addr(s);
	else {
		struct hostent *hp = gethostbyname(s);
		if (hp == 0)
			/*FIXME*/
			return (0);
		return *((unsigned int **)hp->h_addr_list)[0];
	}
}


static int
usage(Tcl_Interp *interp, const char *msg)
{
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "invalid number of arguments: ", msg, NULL);
	return TCL_ERROR;
}


static void*
extract_ptr(Tcl_Interp *interp, Tcl_Obj *obj)
{
	int p;
	if (Tcl_GetIntFromObj(interp, obj, &p)!=TCL_OK) return NULL;
	else return (void*)p;
}


static char*
get_callback(srm_source_t source, int index, Tcl_Interp **interp_ptr)
{
	const srm_callbacks *cb = srm_get_callbacks(srm_get_session(source));
	*interp_ptr = ((Tcl_SRM_UserHook*)cb->user_hook)->interp;
	return ((Tcl_SRM_UserHook*)cb->user_hook)->callbacks[index];
}


static void
srm_recv(srm_source_t source, unsigned int cid, unsigned int seqno,
	 const unsigned char *data, int len, const srm_adu_info * /*i*/)//FIXME
{
	Tcl_Interp *interp;
	char *callback = get_callback(source, CB_RECV, &interp);
	if (callback==NULL || *callback=='\0') return;

	Tcl_Obj *cmd;
	cmd = Tcl_NewStringObj("", 0);
	Tcl_IncrRefCount(cmd);

	if (Tcl_ListObjAppendElement(interp, cmd, Tcl_NewStringObj("eval",-1))
	    != TCL_OK) {
	        Tcl_DecrRefCount(cmd);
		return;
	}
	if (Tcl_ListObjAppendElement(interp, cmd,Tcl_NewStringObj(callback,-1))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		return;
	}

	Tcl_Obj *params;
	params = Tcl_NewStringObj("", 0);
	Tcl_IncrRefCount(params);

	if (Tcl_ListObjAppendElement(interp, params,Tcl_NewIntObj((int)source))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}
	if (Tcl_ListObjAppendElement(interp, params, Tcl_NewIntObj((int)cid))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}
	if (Tcl_ListObjAppendElement(interp, params, Tcl_NewIntObj((int)seqno))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}
	if (Tcl_ListObjAppendElement(interp, params,
				     Tcl_NewStringObj((char*)data, len))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}

	if (Tcl_ListObjAppendElement(interp, cmd, params)
	    != TCL_OK) {
	        Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}

	if (Tcl_GlobalEvalObj(interp, cmd)!=TCL_OK) {
		fprintf(stderr, "Error in callback %s:\n%s\n%s\n",
			callback, Tcl_GetStringResult(interp),
			Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY));
	}
	Tcl_DecrRefCount(cmd);
	Tcl_DecrRefCount(params);
}


static int
srm_should_recover(srm_source_t source, unsigned int cid, unsigned int sseq,
		   unsigned int eseq)
{
	Tcl_Interp *interp;
	char *callback = get_callback(source, CB_SHOULD_RECOVER, &interp);
	if (callback==NULL || *callback=='\0') return 0;

	Tcl_Obj *cmd;
	cmd = Tcl_NewStringObj("", 0);
	Tcl_IncrRefCount(cmd);

	if (Tcl_ListObjAppendElement(interp, cmd, Tcl_NewStringObj("eval",-1))
	    != TCL_OK) {
	        Tcl_DecrRefCount(cmd);
		return 0;
	}
	if (Tcl_ListObjAppendElement(interp, cmd,Tcl_NewStringObj(callback,-1))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		return 0;
	}


	Tcl_Obj *params;
	params = Tcl_NewStringObj("", 0);
	Tcl_IncrRefCount(params);

	if (Tcl_ListObjAppendElement(interp, params,Tcl_NewIntObj((int)source))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return 0;
	}
	if (Tcl_ListObjAppendElement(interp, params, Tcl_NewIntObj((int)cid))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return 0;
	}
	if (Tcl_ListObjAppendElement(interp, params, Tcl_NewIntObj((int)sseq))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return 0;
	}
	if (Tcl_ListObjAppendElement(interp, params, Tcl_NewIntObj((int)eseq))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return 0;
	}
	

	if (Tcl_ListObjAppendElement(interp, cmd, params)
	    != TCL_OK) {
	        Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return 0;
	}
	
	if (Tcl_GlobalEvalObj(interp, cmd)!=TCL_OK) {
		fprintf(stderr, "Error in callback %s:\n%s\n%s\n",
			callback, Tcl_GetStringResult(interp),
			Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY));
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
	} else {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		int recover=0;
		if (Tcl_GetIntFromObj(interp, Tcl_GetObjResult(interp),
				      &recover)!=TCL_OK ||
		    (recover!=1 && recover!=0)) {
			fprintf(stderr, "Return value from callback "
				"%s must be a boolean (1 or 0)\n", callback);
		} else {
			return recover;
		}
	}
	return 0;
}


static void
srm_read_adu(srm_source_t source, unsigned int cid, unsigned int seqno,
	     unsigned char **data_ptr, unsigned int *len_ptr,
	     srm_free_proc *free_proc_ptr, srm_adu_info * /*info*/) //FIXME
{
	*data_ptr = NULL;
	*len_ptr = 0;
	*free_proc_ptr = NULL;
	
	Tcl_Interp *interp;
	char *callback = get_callback(source, CB_READ_ADU, &interp);
	if (callback==NULL || *callback=='\0') return;

	Tcl_Obj *cmd;
	cmd = Tcl_NewStringObj("", 0);
	Tcl_IncrRefCount(cmd);


	if (Tcl_ListObjAppendElement(interp, cmd, Tcl_NewStringObj("eval",-1))
	    != TCL_OK) {
	        Tcl_DecrRefCount(cmd);
		return;
	}
	if (Tcl_ListObjAppendElement(interp, cmd,Tcl_NewStringObj(callback,-1))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		return;
	}


	Tcl_Obj *params;
	params = Tcl_NewStringObj("", 0);
	Tcl_IncrRefCount(params);

	if (Tcl_ListObjAppendElement(interp, params,Tcl_NewIntObj((int)source))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}
	if (Tcl_ListObjAppendElement(interp, params, Tcl_NewIntObj((int)cid))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
			return;
	}
	if (Tcl_ListObjAppendElement(interp, params,Tcl_NewIntObj((int)seqno))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}
	
	if (Tcl_ListObjAppendElement(interp, cmd, params)
	    != TCL_OK) {
	        Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}

	if (Tcl_GlobalEvalObj(interp, cmd)!=TCL_OK) {
		fprintf(stderr, "Error in callback %s:\n%s\n%s\n",
			callback, Tcl_GetStringResult(interp),
			Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY));
	} else {
		char *result = Tcl_GetStringFromObj(Tcl_GetObjResult(interp),
						    (int*) len_ptr);
		*data_ptr = (unsigned char*)malloc(*len_ptr);
		if (*data_ptr) {
			memcpy(*data_ptr, result, *len_ptr);
			*free_proc_ptr = free;
		}
		else *len_ptr = 0;
	}
	Tcl_DecrRefCount(cmd);
	Tcl_DecrRefCount(params);
}


static void
srm_source_update(srm_source_t source, const unsigned char *info, int len)
{
	Tcl_Interp *interp;
	char *callback = get_callback(source, CB_SOURCE_UPDATE, &interp);
	if (callback==NULL || *callback=='\0') return;

	Tcl_Obj *cmd;
	cmd = Tcl_NewStringObj("", 0);
	Tcl_IncrRefCount(cmd);

	if (Tcl_ListObjAppendElement(interp, cmd, Tcl_NewStringObj("eval",-1))
	    != TCL_OK) {
	        Tcl_DecrRefCount(cmd);
		return;
	}
	if (Tcl_ListObjAppendElement(interp, cmd,Tcl_NewStringObj(callback,-1))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		return;
	}


	Tcl_Obj *params;
	params = Tcl_NewStringObj("", 0);
	Tcl_IncrRefCount(params);

	if (Tcl_ListObjAppendElement(interp, params,Tcl_NewIntObj((int)source))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}
	if (Tcl_ListObjAppendElement(interp, params,
				     Tcl_NewStringObj((char*)info, len))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}
	

	if (Tcl_ListObjAppendElement(interp, cmd, params)
	    != TCL_OK) {
	        Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}

	if (Tcl_GlobalEvalObj(interp, cmd)!=TCL_OK) {
		fprintf(stderr, "Error in callback %s:\n%s\n%s\n",
			callback, Tcl_GetStringResult(interp),
			Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY));
	}
	Tcl_DecrRefCount(cmd);
	Tcl_DecrRefCount(params);
}


static void
srm_recv_cid(srm_source_t source, unsigned int cid, unsigned int parent,
	     const unsigned char *name, int name_len)
{
	Tcl_Interp *interp;
	char *callback = get_callback(source, CB_RECV_CID, &interp);
	if (callback==NULL || *callback=='\0') return;

	Tcl_Obj *cmd;
	cmd = Tcl_NewStringObj("", 0);
	Tcl_IncrRefCount(cmd);

	if (Tcl_ListObjAppendElement(interp, cmd, Tcl_NewStringObj("eval",-1))
	    != TCL_OK) {
	        Tcl_DecrRefCount(cmd);
		return;
	}
	if (Tcl_ListObjAppendElement(interp, cmd,Tcl_NewStringObj(callback,-1))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		return;
	}


	Tcl_Obj *params;
	params = Tcl_NewStringObj("", 0);
	Tcl_IncrRefCount(params);

	if (Tcl_ListObjAppendElement(interp, params,Tcl_NewIntObj((int)source))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}
	if (Tcl_ListObjAppendElement(interp, params, Tcl_NewIntObj((int)cid))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
			return;
	}
	if (Tcl_ListObjAppendElement(interp, params,Tcl_NewIntObj((int)parent))
	    !=TCL_OK) {
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
			return;
	}
	if (Tcl_ListObjAppendElement(interp, params,
				     Tcl_NewStringObj((char*)name,
						      name_len))!=TCL_OK){
		Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}
	
	if (Tcl_ListObjAppendElement(interp, cmd, params)
	    != TCL_OK) {
	        Tcl_DecrRefCount(cmd);
		Tcl_DecrRefCount(params);
		return;
	}

	if (Tcl_GlobalEvalObj(interp, cmd)!=TCL_OK) {
		fprintf(stderr, "Error in callback %s:\n%s\n%s\n",
			callback, Tcl_GetStringResult(interp),
			Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY));
	}
	Tcl_DecrRefCount(cmd);
	Tcl_DecrRefCount(params);
}


static int
srm_create_session_proc(ClientData /*clientData*/,
			Tcl_Interp *interp,
			int objc, Tcl_Obj *CONST objv[])
{
	static srm_callbacks tcl_callbacks_ = {
		srm_recv,
		srm_should_recover,
		srm_read_adu,
		srm_source_update,
		srm_recv_cid, NULL
	};

	if (objc!=4 && objc!=5)
		return usage(interp, "srm_create_session addr sport rport ?ttl?");
	
	char *addr = Tcl_GetStringFromObj(objv[1], NULL);
	unsigned int sport, rport, ttl;
	if (Tcl_GetIntFromObj(interp, objv[2], (int*)&sport)!=TCL_OK)
		return TCL_ERROR;
	if (Tcl_GetIntFromObj(interp, objv[3], (int*)&rport)!=TCL_OK)
		return TCL_ERROR;

	if (objc>4) {
		if (Tcl_GetIntFromObj(interp, objv[4], (int*)&ttl)!=TCL_OK)
			return TCL_ERROR;
	} else ttl = 16;

	srm_session_t s=srm_create_session(lookup_host_addr(addr), sport, rport,
					   ttl);
	if (!s) {
		Tcl_SetObjResult(interp, Tcl_NewObj());
		return TCL_OK;
	}
	Tcl_SRM_UserHook *hook = new Tcl_SRM_UserHook;
	for (int i=0; i<CB_NUM_CALLBACKS; i++) {
		hook->callbacks[i] = NULL;
	}
	hook->interp = interp;
	tcl_callbacks_.user_hook = hook;
	srm_set_callbacks(s, &tcl_callbacks_);

	Tcl_SetObjResult(interp, Tcl_NewIntObj((int)s));
	return TCL_OK;
}


static int
srm_destroy_session_proc(ClientData /*clientData*/,
			 Tcl_Interp *interp,
			 int objc, Tcl_Obj *CONST objv[])
{
	if (objc!=2)
		return usage(interp, "srm_destroy_session session");
	
	srm_session_t session = (srm_session_t)extract_ptr(interp, objv[1]);
	if (session==NULL) return TCL_ERROR;
	srm_destroy_session(session);
	return TCL_OK;
}


static int
srm_reset_session_proc(ClientData /*clientData*/,
		       Tcl_Interp *interp,
		       int objc, Tcl_Obj *CONST objv[])
{
	if (objc!=5 && objc!=6)
		return usage(interp,
			     "srm_reset_session session addr sport rport ?ttl?");
	
	srm_session_t session = (srm_session_t)extract_ptr(interp, objv[1]);
	if (session==NULL) return TCL_ERROR;

	char *addr = Tcl_GetStringFromObj(objv[2], NULL);
	unsigned int sport, rport, ttl;
	if (Tcl_GetIntFromObj(interp, objv[3], (int*)&sport)!=TCL_OK)
		return TCL_ERROR;
	if (Tcl_GetIntFromObj(interp, objv[4], (int*)&rport)!=TCL_OK)
		return TCL_ERROR;

	if (objc>5) {
		if (Tcl_GetIntFromObj(interp, objv[5], (int*)&ttl)!=TCL_OK)
			return TCL_ERROR;
	} else ttl = 16;

	int status = srm_reset_session(session, lookup_host_addr(addr), 
				      sport, rport, ttl);
	Tcl_SetObjResult(interp, Tcl_NewIntObj(status));
	return TCL_OK;
}


static int
srm_delay_until_full_packet_proc(ClientData /*clientData*/, Tcl_Interp *interp,
				 int objc, Tcl_Obj *CONST objv[])
{
	int flag;
	
	if (objc<2 || objc>3)
		return usage(interp,
			     "srm_delay_until_full_packet session ?flag?");

	srm_session_t session = (srm_session_t)extract_ptr(interp, objv[1]);
	if (session==NULL) return TCL_ERROR;

	if (objc>=3) {
		if (Tcl_GetIntFromObj(interp, objv[2], &flag)!=TCL_OK)
			return TCL_ERROR;
		flag = srm_delay_until_full_packet(session, flag);
		Tcl_SetObjResult(interp, Tcl_NewIntObj(flag));
	} else {
		flag = srm_get_delay_until_full_packet(session);
		Tcl_SetObjResult(interp, Tcl_NewIntObj(flag));
	}

	return TCL_OK;
}


static int
srm_session_bandwidth_proc(ClientData /*clientData*/, Tcl_Interp *interp,
			   int objc, Tcl_Obj *CONST objv[])
{
	int bps;
	
	if (objc<2 || objc>3)
		return usage(interp,
			     "srm_session_bandwidth session ?bps?");

	srm_session_t session = (srm_session_t)extract_ptr(interp, objv[1]);
	if (session==NULL) return TCL_ERROR;

	if (objc>=3) {
		if (Tcl_GetIntFromObj(interp, objv[2], &bps)!=TCL_OK)
			return TCL_ERROR;
		srm_set_session_bandwidth(session, bps);
		Tcl_SetObjResult(interp, Tcl_NewStringObj("", -1));
	} else {
		bps = srm_get_session_bandwidth(session);
		Tcl_SetObjResult(interp, Tcl_NewIntObj(bps));
	}

	return TCL_OK;
}


static int
srm_drop_probability_proc(ClientData /*clientData*/, Tcl_Interp *interp,
			  int objc, Tcl_Obj *CONST objv[])
{
	double dropProb;
	
	if (objc<2 || objc>3)
		return usage(interp,
			     "srm_drop_probability session ?dropProb?");

	srm_session_t session = (srm_session_t)extract_ptr(interp, objv[1]);
	if (session==NULL) return TCL_ERROR;

	if (objc>=3) {
		if (Tcl_GetDoubleFromObj(interp, objv[2], &dropProb)!=TCL_OK)
			return TCL_ERROR;
		dropProb = srm_set_drop_probability(session, dropProb);
		Tcl_SetObjResult(interp, Tcl_NewDoubleObj(dropProb));
	} else {
		dropProb = srm_get_drop_probability(session);
		Tcl_SetObjResult(interp, Tcl_NewDoubleObj(dropProb));
	}

	return TCL_OK;
}


static int
srm_create_source_proc(ClientData /*clientData*/,
		       Tcl_Interp *interp,
		       int objc, Tcl_Obj *CONST objv[])
{
	const char *srcid=NULL;
	
	if (objc<2 || objc>3)
		return usage(interp,
			     "srm_create_source session ?srcid?");

	srm_session_t session = (srm_session_t)extract_ptr(interp, objv[1]);
	if (session==NULL) return TCL_ERROR;

	if (objc>=3) {
		srcid = Tcl_GetStringFromObj(objv[2], NULL);
		if (*srcid=='\0') srcid=NULL;
	}

	srm_source_t s=srm_create_source(session, srcid);
	Tcl_SetObjResult(interp, Tcl_NewIntObj((int)s));
	return TCL_OK;
}


static int
srm_app_info_proc(ClientData /*clientData*/, Tcl_Interp *interp,
		  int objc, Tcl_Obj *CONST objv[])
{
	const unsigned char *info=NULL;
	int len;
	
	if (objc<2 || objc>3)
		return usage(interp,
			     "srm_app_info src ?info?");

	srm_source_t source = (srm_source_t)extract_ptr(interp, objv[1]);
	if (source==NULL) return TCL_ERROR;

	if (objc>=3) {
		info = (const unsigned char*) Tcl_GetStringFromObj(objv[2],
								   &len);
		if (len==0) info = NULL;
		srm_set_app_info(source, info, len);
		Tcl_SetObjResult(interp, Tcl_NewStringObj("", -1));
	} else {
		srm_get_app_info(source, &info, &len);
		Tcl_SetObjResult(interp, Tcl_NewStringObj((char*)info, len));
	}

	return TCL_OK;
}


static int srm_calloc_proc(ClientData /*clientData*/, Tcl_Interp *interp,
			   int objc, Tcl_Obj *CONST objv[])
{
	if (objc<3 || objc>4)
		return usage(interp,
			     "srm_calloc source parent_cid ?name?");

	srm_source_t source = (srm_source_t)extract_ptr(interp, objv[1]);
	if (source==NULL) return TCL_ERROR;
	unsigned int parent;
	char *name=NULL;
	int len=0;
	
	if (Tcl_GetIntFromObj(interp, objv[2], (int*)&parent)!=TCL_OK)
		return TCL_ERROR;

	if (objc>=4) {
		name = Tcl_GetStringFromObj(objv[3], &len);
		if (*name=='\0') { name=NULL; len=0; }
	}

	unsigned int cid=srm_calloc(source, parent, (unsigned char*)name, len);
	Tcl_SetObjResult(interp, Tcl_NewIntObj((int)cid));
	return TCL_OK;
}


static int
srm_get_container_name_proc(ClientData /*clientData*/,
			    Tcl_Interp *interp,
			    int objc, Tcl_Obj *CONST objv[])
{
	if (objc!=3)
		return usage(interp,
			     "srm_get_container_name source cid");

	srm_source_t source = (srm_source_t)extract_ptr(interp, objv[1]);
	if (source==NULL) return TCL_ERROR;
	unsigned int cid;
	const unsigned char *name=NULL;
	int len=0;
	
	if (Tcl_GetIntFromObj(interp, objv[2], (int*)&cid)!=TCL_OK)
		return TCL_ERROR;

	srm_get_container_name(source, cid, &name, &len);
	if (name==NULL) { name = (const unsigned char*)""; len=-1; }
	Tcl_SetObjResult(interp, Tcl_NewStringObj((char*)name, len));
	return TCL_OK;
}


static int
srm_get_parent_cid_proc(ClientData /*clientData*/,
			Tcl_Interp *interp,
			int objc, Tcl_Obj *CONST objv[])
{
	if (objc!=3)
		return usage(interp, "srm_get_parent_cid source cid");

	srm_source_t source = (srm_source_t)extract_ptr(interp, objv[1]);
	if (source==NULL) return TCL_ERROR;
	unsigned int cid;
	
	if (Tcl_GetIntFromObj(interp, objv[2], (int*)&cid)!=TCL_OK)
		return TCL_ERROR;

	cid = srm_get_parent_cid(source, cid);
	Tcl_SetObjResult(interp, Tcl_NewIntObj((int)cid));
	return TCL_OK;
}


static int srm_send_proc(ClientData /*clientData*/, Tcl_Interp *interp,
			 int objc, Tcl_Obj *CONST objv[])
{
	if (objc!=4)
		return usage(interp, "srm_send source cid data");

	srm_source_t source = (srm_source_t)extract_ptr(interp, objv[1]);
	if (source==NULL) return TCL_ERROR;
	unsigned int cid, seqno;
	unsigned char *data=NULL;
	int len=0;
	
	if (Tcl_GetIntFromObj(interp, objv[2], (int*)&cid)!=TCL_OK)
		return TCL_ERROR;
	data = (unsigned char*)Tcl_GetStringFromObj(objv[3], &len);

	seqno = srm_send(source, cid, data, len, NULL); //FIXME
	Tcl_SetObjResult(interp, Tcl_NewIntObj((int)seqno));
	return TCL_OK;
}


static int srm_recover_proc(ClientData /*clientData*/, Tcl_Interp *interp,
			    int objc, Tcl_Obj *CONST objv[])
{
	if (objc!=5)
		return usage(interp, "srm_recover source cid sseq eseq");

	srm_source_t source = (srm_source_t)extract_ptr(interp, objv[1]);
	if (source==NULL) return TCL_ERROR;
	unsigned int cid, sseq, eseq;
	
	if (Tcl_GetIntFromObj(interp, objv[2], (int*)&cid)!=TCL_OK)
		return TCL_ERROR;
	if (Tcl_GetIntFromObj(interp, objv[3], (int*)&sseq)!=TCL_OK)
		return TCL_ERROR;
	if (Tcl_GetIntFromObj(interp, objv[4], (int*)&eseq)!=TCL_OK)
		return TCL_ERROR;

	srm_recover(source, cid, sseq, eseq);
	Tcl_SetResult(interp, "", TCL_STATIC);
	return TCL_OK;
}


static int
srm_callbacks_proc(ClientData /*clientData*/,Tcl_Interp *interp,
		       int objc, Tcl_Obj *CONST objv[])
{
	if (objc!=CB_NUM_CALLBACKS+2 && objc!=2)
		return usage(interp,
			     "srm_callbacks session ?recv should_recover "
			     "read_adu source_update recv_cid?");

	srm_session_t session = (srm_session_t)extract_ptr(interp, objv[1]);
	if (session==NULL) return TCL_ERROR;
	const srm_callbacks *callbacks = srm_get_callbacks(session);
	char **tcl_cb = ((Tcl_SRM_UserHook*)callbacks->user_hook)->callbacks;

	if (objc==2) {
		Tcl_Obj *list = Tcl_NewStringObj("", 0);
		char *cb;
		int len;
	
		for (int i=0; i<CB_NUM_CALLBACKS; i++) {
		  // ErikM - 08/13/00 - replaced commented out line with 
		  //   one below it - compiler was complaining, and it doesn't
		  //   really make sense
		  //			cb = (tcl_cb[i] ? tcl_cb[i] : "");
		        cb = tcl_cb[i];
			len = strlen(cb);
		
			if (Tcl_ListObjAppendElement(interp, list,
						     Tcl_NewStringObj(cb, len))
			    !=TCL_OK) {
				Tcl_DecrRefCount(list);
				return TCL_ERROR;
			}
		}
		
		Tcl_SetObjResult(interp, list);
	} else {
		char *cb;
		int len;
		
		for (int i=0; i<CB_NUM_CALLBACKS; i++) {
			cb = Tcl_GetStringFromObj(objv[i+2], &len);
			if (tcl_cb[i]!=NULL) delete [] tcl_cb[i];
			if (cb==NULL || len<=0) tcl_cb[i] = NULL;
			else {
				tcl_cb[i] = new char[len+1];
				strcpy(tcl_cb[i], cb);
			}
		}

		Tcl_SetResult(interp, "", TCL_STATIC);
	}
	return TCL_OK;
}


static int srm_get_source_id_proc(ClientData /*clientData*/, 
				  Tcl_Interp *interp,
				  int objc, Tcl_Obj *CONST objv[])
{
	if (objc != 2) 
		return usage(interp, "srm_get_source_id source");
	
	srm_source_t source = (srm_source_t)extract_ptr(interp, objv[1]);
	if (source==NULL) return TCL_ERROR;

	char srcid[50];
	
	srm_srcid2str(srm_get_source_id(source), srcid);
	
	Tcl_SetObjResult(interp, Tcl_NewStringObj(srcid, -1));
	return TCL_OK;
}


static int srm_get_session_proc(ClientData /*clientData*/, 
				Tcl_Interp *interp,
				int objc, Tcl_Obj *CONST objv[])
{
	if (objc != 2) 
		return usage(interp, "srm_get_session source");
	
	srm_source_t source = (srm_source_t)extract_ptr(interp, objv[1]);
	if (source==NULL) return TCL_ERROR;
	
	srm_session_t session = srm_get_session(source);
	
	Tcl_SetObjResult(interp, Tcl_NewIntObj((int)session));
	return TCL_OK;
}


#ifndef NDEBUG
static int srm_trace_flags_proc(ClientData /*clientData*/, 
				Tcl_Interp *interp,
				int objc, Tcl_Obj *CONST objv[])
{
	if (objc != 2) 
		return usage(interp, "srm_trace_flags flags");

	unsigned int flags=0;
	if (Tcl_GetIntFromObj(interp, objv[1], (int*)&flags)!=TCL_OK)
		return TCL_ERROR;
	srm_trace_flags(flags);
	return TCL_OK;
}
#endif


extern "C" {
}


int
SRMv2_Init(Tcl_Interp *interp)
{
	SRM2TCL(srm_create_session);
	SRM2TCL(srm_destroy_session);
	SRM2TCL(srm_reset_session);
	SRM2TCL(srm_delay_until_full_packet);
	SRM2TCL(srm_session_bandwidth);
	SRM2TCL(srm_drop_probability);
	SRM2TCL(srm_create_source);
	SRM2TCL(srm_app_info);
	SRM2TCL(srm_calloc);
	SRM2TCL(srm_get_container_name);
	SRM2TCL(srm_get_parent_cid);
	SRM2TCL(srm_send);
	SRM2TCL(srm_recover);
	SRM2TCL(srm_callbacks);
	SRM2TCL(srm_get_source_id);
	SRM2TCL(srm_get_session);

#ifndef NDEBUG
	SRM2TCL(srm_trace_flags);
#endif

	return TCL_OK;
}


int
Srmv2_Init(Tcl_Interp *interp)
{
	return SRMv2_Init(interp);
}


srm_session_t
srm_session_from_obj(Tcl_Interp *interp, Tcl_Obj *o)
{
	return (srm_session_t)extract_ptr(interp, o);
}


srm_source_t
srm_source_from_obj(Tcl_Interp *interp, Tcl_Obj *o)
{
	return (srm_source_t)extract_ptr(interp, o);
}


srm_session_t
srm_session_from_str(Tcl_Interp *interp, const char *str)
{
	Tcl_Obj *o = Tcl_NewStringObj((char*)str, -1);
	Tcl_IncrRefCount(o);
	srm_session_t s = srm_session_from_obj(interp, o);
	Tcl_DecrRefCount(o);
	return s;
}


srm_source_t
srm_source_from_str(Tcl_Interp *interp, const char *str)
{
	Tcl_Obj *o = Tcl_NewStringObj((char*)str, -1);
	Tcl_IncrRefCount(o);
	srm_source_t s = srm_source_from_obj(interp, o);
	Tcl_DecrRefCount(o);
	return s;
}
