/*
 * net-ip.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/srmv2/net/net-ip.cc,v 1.13 2002/02/03 03:03:19 lim Exp $ (LBL)";

#include "pktbuf.h"
#include "net-ip.h"

int SRMv2_IPNetwork::open(u_int32_t addr, int port, int ttl)
{
	return (open(addr, port, port, ttl));
}

void SRMv2_IPNetwork::sendto(const SRMv2_pktbuf* pb)
{
	struct sockaddr_in saddr;
	bzero((char*)&saddr, sizeof(saddr));
	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = pb->addr;
	saddr.sin_port = htons(pb->port);
	
	::sendto(ssock_, (char*)pb->dp, pb->len, 0, (sockaddr*)&saddr,
		 sizeof(saddr));
}
	
int SRMv2_IPNetwork::open(u_int32_t addr, int sport, int rport, int ttl)
{
	addr_ = addr;
	sport_ = sport;
	rport_ = rport;
	ttl_ = ttl;

	ssock_ = openssock(addr, sport, ttl);
	if (ssock_ < 0)
		return (-1);

	/*
	 * Open the receive-side socket if requested.
	 */
	if (rport != 0) {
		if (add_membership() < 0)
			return (-1);
	}
	last_reset_ = 0;
	return (0);
}

/* UDP unicast server */
int SRMv2_IPNetwork::open(int port)
{
	int fd;
	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		return (-1);
	}
	addr_ = INADDR_ANY;
	rport_ = port;
	ttl_ = 0;

	int on = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on,
			sizeof(on)) < 0) {
		perror("SO_REUSEADDR");
		return (-1);
	}
#ifdef SO_REUSEPORT
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEPORT, (char *)&on,
		       sizeof(on)) < 0) {
		perror("SO_REUSEPORT");
		exit(1);
	}
#endif
	struct sockaddr_in saddr;
	bzero((char*)&saddr, sizeof(saddr));
	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = INADDR_ANY;
	saddr.sin_port = port;
	if (::bind(fd, (struct sockaddr*)&saddr, sizeof(saddr)) < 0) {
		perror("bind");
		return(-1);
	}
	if (port == 0) {
		socklen_t slen = sizeof(saddr);
		bzero((char*)&saddr, sizeof(saddr));
		if (getsockname(fd, (struct sockaddr*)&saddr, &slen) < 0) {
			perror("getsockname");
			return (-1);
		} else {
			rport_ = saddr.sin_port;
		}
	}
	rsock_ = fd;
	last_reset_ = 0;

	/* Open send socket. */
	if ((ssock_ = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		return (-1);
	}
	return (0);
}

SRMv2_IPNetwork::~SRMv2_IPNetwork()
{
	close();
}

int SRMv2_IPNetwork::close()
{
	if (ssock_ >= 0) {
		::close(ssock_);
		ssock_ = -1;
	}
	if (rsock_ >= 0) {
		::close(rsock_);
		rsock_ = -1;
	}
	return (0);
}

/*
 * Control group membership by closing the receive socket.
 * There are too many potential problems across different
 * systems in using the add/drop membership ioctls.
 * (besides, group membership should be implicit in the
 * socket multicast address binding and we should leave
 * the igmp group by closing the socket anyway).
 */
void SRMv2_IPNetwork::drop_membership()
{
	::close(rsock_);
	rsock_ = -1;
}

int SRMv2_IPNetwork::add_membership()
{
	if (rsock_ >= 0)
		return (0);

	/*
	 * Connecting the send socket also bound the local address.
	 * On a multihomed host we need to bind the receive socket
	 * to the same local address the kernel has chosen to send on.
	 */
	sockaddr_in local;
	localname(&local);
	rsock_ = openrsock(addr_, rport_, local);
	if (rsock_ < 0) {
		if (ssock_ >= 0)
			(void)::close(ssock_);
		return (-1);
	}
	local_ = local.sin_addr.s_addr;
#if defined(sun) && defined(__svr4__)
	/*
	 * gethostname on solaris prior to 2.6 always returns 0 for
	 * udp sockets.  do a horrible kluge that often fails on
	 * multihomed hosts to get the local address (we don't use
	 * this to open the recv sock, only for the 'interface'
	 * tcl command).
	 */
	if (local_ == 0) {
		char myhostname[1024];
		int error;

		error = sysinfo(SI_HOSTNAME, myhostname, sizeof(myhostname));
		if (error == -1) {
			perror("Getting my hostname");
			exit(-1);
		}
		local_ = LookupHostAddr(myhostname);
	}
#endif
	lport_ = local.sin_port;

	return (0);
}

void SRMv2_IPNetwork::localname(sockaddr_in* p)
{
	memset((char *)p, 0, sizeof(*p));
	p->sin_family = AF_INET;
	socklen_t len = sizeof(*p);

#if defined(_AIX)
	if (getsockname(ssock_, (struct sockaddr *)p, (long unsigned int *)&len) < 0) {
#else
	if (getsockname(ssock_, (struct sockaddr *)p, &len) < 0) {
#endif
		perror("getsockname");
		p->sin_addr.s_addr = 0;
		p->sin_port = 0;
	}
#ifdef WIN32
	if (p->sin_addr.s_addr == 0) {
		char hostname[80];
		struct hostent *hp;

		if (gethostname(hostname, sizeof(hostname)) >= 0) {
			if ((hp = gethostbyname(hostname)) >= 0) {
				p->sin_addr.s_addr = ((struct in_addr *)hp->h_addr)->s_addr;
			}
		}
	}
#endif	
}

void SRMv2_IPNetwork::reset()
{
	time_t t = time(0);
	int d = int(t - last_reset_);
	if (d > 3) {
		last_reset_ = t;
		if (ssock_ >= 0) {
			(void)::close(ssock_);
			ssock_ = openssock(addr_, sport_, ttl_);
		}
	}
}

int SRMv2_IPNetwork::openrsock(u_int32_t addr, u_short port,
			    const struct sockaddr_in& local)
{
	int fd;
	struct sockaddr_in sin;

	fd = socket(AF_INET, SOCK_DGRAM, 0);
	if (fd < 0) {
		perror("socket");
		exit(1);
	}
	nonblock(fd);
	int on = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on,
			sizeof(on)) < 0) {
		perror("SO_REUSEADDR");
	}
#ifdef SO_REUSEPORT
	on = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEPORT, (char *)&on,
		       sizeof(on)) < 0) {
		perror("SO_REUSEPORT");
		exit(1);
	}
#endif
	memset((char *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_port = port;
#ifdef IP_ADD_MEMBERSHIP
	if (IN_CLASSD(ntohl(addr))) {
		/*
		 * Try to bind the multicast address as the socket
		 * dest address.  On many systems this won't work
		 * so fall back to a destination of INADDR_ANY if
		 * the first bind fails.
		 */
		sin.sin_addr.s_addr = addr;
		if (::bind(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
			sin.sin_addr.s_addr = INADDR_ANY;
			if (::bind(fd, (struct sockaddr*)&sin, sizeof(sin)) < 0) {
				perror("bind");
				return (-1);
			}
		}
		/* 
		 * FIXME This is bogus multicast setup that really
		 * shouldn't have to be done (group membership should be
		 * implicit in the IP class D address, route should contain
		 * ttl & no loopback flag, etc.).  Steve Deering has promised
		 * to fix this for the 4.4bsd release.  We're all waiting
		 * with bated breath.
		 */
		struct ip_mreq mr;

		mr.imr_multiaddr.s_addr = addr;
		mr.imr_interface.s_addr = INADDR_ANY;
		if (setsockopt(fd, IPPROTO_IP, IP_ADD_MEMBERSHIP, 
			       (char *)&mr, sizeof(mr)) < 0) {
			perror("IP_ADD_MEMBERSHIP");
			return (-1);
		}
	} else
#endif
	{
		/*
		 * bind the local host's address to this socket.  If that
		 * fails, another vic probably has the addresses bound so
		 * just exit.
		 */
		sin.sin_addr.s_addr = local.sin_addr.s_addr;
		if (::bind(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
			perror("bind");
			return (-1);
		}
		/*
		 * MS Windows currently doesn't compy with the Internet Host
		 * Requirements standard (RFC-1122) and won't let us include
		 * the source address in the receive socket demux state.
		 * (The consequence of this is that all conversations have
		 * to be assigned a unique local port so the "vat side
		 * conversations" --- initiated by middle-clicking on
		 * the site name --- doesn't work under windows.)
		 */
#ifndef WIN32
		/*
		 * (try to) connect the foreign host's address to this socket.
		 */
		sin.sin_port = 0;
		sin.sin_addr.s_addr = addr;
		connect(fd, (struct sockaddr *)&sin, sizeof(sin));
#endif
	}
	/*
	 * FIXME don't need this for the session socket.
	 */	
	int bufsize = 80 * 1024;
	if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, (char *)&bufsize,
			sizeof(bufsize)) < 0) {
		bufsize = 32 * 1024;
		if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, (char *)&bufsize,
				sizeof(bufsize)) < 0)
			perror("SO_RCVBUF");
	}
	return (fd);
}

int SRMv2_IPNetwork::openssock(u_int32_t addr, u_short port, int ttl)
{
	int fd;
	struct sockaddr_in sin;

	fd = socket(AF_INET, SOCK_DGRAM, 0);
	if (fd < 0) {
		perror("socket");
		exit(1);
	}
	nonblock(fd);

#ifdef WIN32
	memset((char *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_port = 0;
	sin.sin_addr.s_addr = INADDR_ANY;
	if (::bind(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		perror("bind");
		return (-1);
	}
#endif
	memset((char *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_port = port;
	sin.sin_addr.s_addr = addr;
	if (connect(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		perror("connect");
		return(-1);
	}
	if (IN_CLASSD(ntohl(addr))) {
#ifdef IP_ADD_MEMBERSHIP
		char c;
		/*
		 * Turn on loopback by default.
		 * Note that some network stacks don't allow
		 * loopback to be disabled, but that's okay
		 * because looped-back packets are filtered out
		 * on the recv path anyway.
		 */
		c = 1;
		(void)setsockopt(fd, IPPROTO_IP, IP_MULTICAST_LOOP, &c, 1);

		/* set the multicast TTL */
#ifdef WIN32
		u_int t;
#else
		u_char t;
#endif
		t = (ttl > 255) ? 255 : (ttl < 0) ? 0 : ttl;
		if (setsockopt(fd, IPPROTO_IP, IP_MULTICAST_TTL,
			       (char*)&t, sizeof(t)) < 0) {
			perror("IP_MULTICAST_TTL");
			exit(1);
		}
#else
		fprintf(stderr, "\
not compiled with support for IP multicast\n\
you must specify a unicast destination\n");
		exit(1);
#endif
	}
	/*
	 * FIXME don't need this for the session socket.
	 */
	int bufsize = 80 * 1024;
	if (setsockopt(fd, SOL_SOCKET, SO_SNDBUF, (char *)&bufsize,
		       sizeof(bufsize)) < 0) {
		bufsize = 48 * 1024;
		if (setsockopt(fd, SOL_SOCKET, SO_SNDBUF, (char *)&bufsize,
			       sizeof(bufsize)) < 0)
			perror("SO_SNDBUF");
	}
	return (fd);
}


int
SRMv2_IPNetwork::ssock_port()
{
  if (ssock_ <= 0) return 0;

  struct sockaddr_in sin;
  socklen_t len = sizeof(sin);
  if (getsockname(ssock_, (sockaddr*) &sin, &len) < 0) return 0;
  return htons(sin.sin_port);
}


int
SRMv2_IPNetwork::rsock_port()
{
  if (rsock_ <= 0) return 0;

  struct sockaddr_in sin;
  socklen_t len = sizeof(sin);
  if (getsockname(rsock_, (sockaddr*) &sin, &len) < 0) return 0;
  return ntohs(sin.sin_port);
}
