# Copyright (C) 2000-2001 The OpenRPG Project
#
#    openrpg-dev@lists.sourceforge.net
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# --
#
# File: mapper/miniatures.py
# Author: Chris Davis
# Maintainer:
# Version:
#   $Id: miniatures.py,v 1.20 2003/11/19 06:53:50 tdb30_ Exp $
#
# Description: This file contains some of the basic definitions for the chat
# utilities in the orpg project.
#
__version__ = "$Id: miniatures.py,v 1.20 2003/11/19 06:53:50 tdb30_ Exp $"

from base import *
#from images import *
import images


MIN_STICKY_BACK = -0XFFFFFF
MIN_STICKY_FRONT = 0xFFFFFF


##----------------------------------------
##  miniature object
##----------------------------------------

FACE_NONE = 0
FACE_NORTH = 1
FACE_NORTHEAST = 2
FACE_EAST = 3
FACE_SOUTHEAST = 4
FACE_SOUTH = 5
FACE_SOUTHWEST = 6
FACE_WEST = 7
FACE_NORTHWEST = 8

SNAPTO_ALIGN_CENTER = 0
SNAPTO_ALIGN_TL = 1

def cmp_zorder(first,second):

    f = first.zorder
    s = second.zorder

    if f == None:
        f = 0
    if s == None:
        s = 0

    if f == s:
        value = 0
    elif f < s:
        value = -1
    else:
        value = 1

    return value

class bmp_miniature(protectable_attributes):
    def __init__(self, id,path, bmp, pos=cmpPoint(0,0),
                 heading=FACE_NONE, face=FACE_NONE, label="", locked=false,
                 hide=false, snap_to_align=SNAPTO_ALIGN_CENTER,zorder = 0, width=0, height=0):
        protectable_attributes.__init__(self)
        self._protected_attr = ["heading",
                                "face", "label", "path", "pos", "locked",
                                "snap_to_align", "hide", "id", "zorder", "width", "height"]

        self.heading = heading
        self.face = face
        self.label = label
        self.path = path
        self.bmp = bmp
        self.pos = pos
        self.selected = false
        self.locked = locked
        self.snap_to_align = snap_to_align
        self.hide = hide
        self.id = id
        self.zorder = zorder
        self.left = 0
        if not width:
            self.width = 0
        else:
            self.width = width
        if not height:
            self.height = 0
        else:
            self.height = height
        self.right = bmp.GetWidth()
        self.top = 0
        self.bottom = bmp.GetHeight()
        #self._clean_all_attr()

    def __del__(self):
        #images.delete_img(self.bmp)
        del self.bmp
        self.bmp = None        

    def set_bmp(self,bmp):
        self.bmp = bmp                

    def set_min_props(self, heading=FACE_NONE, face=FACE_NONE, label="", locked=false, hide=false, width=0, height=0):
        self.heading = heading
        self.face = face
        self.label = label
        self.locked = locked
        self.hide = hide
        self.width = width
        self.height = height

    def hit_test(self, pt):
        rect = self.get_rect()
        result = None
        # fixes problem between wxPython version 2.4.0.7 and higher
        # and wxPython version 2.4.0.2 and lower --Snowdog
        try:
            #2.4.0.7 and up check
            result = rect.InsideXY(pt.x, pt.y)
        except:
            #2.4.0.2 and lower check
            result = rect.Inside(pt.x, pt.y)
        return result

    def get_rect(self):
        return wxRect(self.pos.x, self.pos.y,
                      self.bmp.GetWidth(), self.bmp.GetHeight())

    def draw(self, dc, op = wxCOPY):
        if self.bmp != None and self.bmp.Ok():
            # check if hidden
            if self.hide:
                return true

            dc.DrawBitmap(self.bmp,self.pos.x,self.pos.y,true)
            # Blit the miniature bmp
#            memDC = wxMemoryDC()
#            memDC.SelectObject(self.bmp)
#            dc.Blit(self.pos.x, self.pos.y,
#                    self.bmp.GetWidth(), self.bmp.GetHeight(),
#                    memDC, 0, 0, op, true)
#            memDC.SelectObject(wxNullBitmap)
#            del memDC

            # set the width and height of the image
            if self.width and self.height:
               tmp_image = wxImageFromBitmap(self.bmp)
               tmp_image.Rescale(int(self.width), int(self.height))
               self.bmp = wxBitmapFromImage(tmp_image)

            self.left = 0
            self.right = self.bmp.GetWidth()
            self.top = 0
            self.bottom = self.bmp.GetHeight()
            # Draw the facing marker if needed
            if self.face:
                x_mid = self.pos.x + (self.bmp.GetWidth()/2)
                x_right = self.pos.x + self.bmp.GetWidth()
                y_mid = self.pos.y + (self.bmp.GetHeight()/2)
                y_bottom = self.pos.y + self.bmp.GetHeight()

                dc.SetPen( wxWHITE_PEN )
                dc.SetBrush( wxRED_BRUSH )
                triangle = []

                # Figure out which direction to draw the marker!!
                if self.face == FACE_WEST:
                    triangle.append(cmpPoint(self.pos.x,self.pos.y))
                    triangle.append(cmpPoint(self.pos.x - 5, y_mid))
                    triangle.append(cmpPoint(self.pos.x, y_bottom))
                elif self.face ==  FACE_EAST:
                    triangle.append(cmpPoint(x_right, self.pos.y))
                    triangle.append(cmpPoint(x_right + 5, y_mid))
                    triangle.append(cmpPoint(x_right, y_bottom))
                elif self.face ==  FACE_SOUTH:
                    triangle.append(cmpPoint(self.pos.x, y_bottom))
                    triangle.append(cmpPoint(x_mid, y_bottom + 5))
                    triangle.append(cmpPoint(x_right, y_bottom))
                elif self.face ==  FACE_NORTH:
                    triangle.append(cmpPoint(self.pos.x, self.pos.y))
                    triangle.append(cmpPoint(x_mid, self.pos.y - 5))
                    triangle.append(cmpPoint(x_right, self.pos.y))
                elif self.face == FACE_NORTHEAST:
                    triangle.append(cmpPoint(x_mid, self.pos.y))
                    triangle.append(cmpPoint(x_right + 5, self.pos.y - 5))
                    triangle.append(cmpPoint(x_right, y_mid))
                    triangle.append(cmpPoint(x_right, self.pos.y))
                elif self.face == FACE_SOUTHEAST:
                    triangle.append(cmpPoint(x_right, y_mid))
                    triangle.append(cmpPoint(x_right + 5, y_bottom + 5))
                    triangle.append(cmpPoint(x_mid, y_bottom))
                    triangle.append(cmpPoint(x_right, y_bottom))
                elif self.face == FACE_SOUTHWEST:
                    triangle.append(cmpPoint(x_mid, y_bottom))
                    triangle.append(cmpPoint(self.pos.x - 5, y_bottom + 5))
                    triangle.append(cmpPoint(self.pos.x, y_mid))
                    triangle.append(cmpPoint(self.pos.x, y_bottom))
                elif self.face == FACE_NORTHWEST:
                    triangle.append(cmpPoint(self.pos.x, y_mid))
                    triangle.append(cmpPoint(self.pos.x - 5, self.pos.y - 5))
                    triangle.append(cmpPoint(x_mid, self.pos.y))
                    triangle.append(cmpPoint(self.pos.x, self.pos.y))

                dc.DrawPolygon(triangle)
                dc.SetBrush(wxNullBrush)
                dc.SetPen(wxNullPen)

            # Draw the heading if needed
            if self.heading:
                x_adjust = 0
                y_adjust = 4

                x_half = self.bmp.GetWidth()/2
                y_half = self.bmp.GetHeight()/2
                x_quarter = self.bmp.GetWidth()/4
                y_quarter = self.bmp.GetHeight()/4
                x_3quarter = x_quarter*3
                y_3quarter = y_quarter*3
                x_full = self.bmp.GetWidth()
                y_full = self.bmp.GetHeight()
                x_center = self.pos.x + x_half
                y_center = self.pos.y + y_half

                # Remember, the pen/brush must be a different color than the
                # facing marker!!!!  We'll use black/cyan for starters.
                # Also notice that we will draw the heading on top of the
                # larger facing marker.
                dc.SetPen( wxBLACK_PEN )
                dc.SetBrush( wxCYAN_BRUSH )
##                wxBrush = dc.GetBrush()
##                wxBrush.SetStyle( wxCROSSDIAG_HATCH )
##                dc.SetBrush( wxBrush )
                triangle = []

                # Figure out which direction to draw the marker!!
                if self.heading == FACE_NORTH:
                    triangle.append(cmpPoint(x_center - x_quarter, y_center - y_half ))
                    triangle.append(cmpPoint(x_center, y_center - y_3quarter ))
                    triangle.append(cmpPoint(x_center + x_quarter, y_center - y_half ))
                elif self.heading ==  FACE_SOUTH:
                    triangle.append(cmpPoint(x_center - x_quarter, y_center + y_half ))
                    triangle.append(cmpPoint(x_center, y_center + y_3quarter ))
                    triangle.append(cmpPoint(x_center + x_quarter, y_center + y_half ))
                elif self.heading == FACE_NORTHEAST:
                    triangle.append(cmpPoint(x_center + x_quarter, y_center - y_half ))
                    triangle.append(cmpPoint(x_center + x_3quarter, y_center - y_3quarter ))
                    triangle.append(cmpPoint(x_center + x_half, y_center - y_quarter ))
                elif self.heading == FACE_EAST:
                    triangle.append(cmpPoint(x_center + x_half, y_center - y_quarter ))
                    triangle.append(cmpPoint(x_center + x_3quarter, y_center ))
                    triangle.append(cmpPoint(x_center + x_half, y_center + y_quarter ))
                elif self.heading == FACE_SOUTHEAST:
                    triangle.append(cmpPoint(x_center + x_half, y_center + y_quarter ))
                    triangle.append(cmpPoint(x_center + x_3quarter, y_center + y_3quarter ))
                    triangle.append(cmpPoint(x_center + x_quarter, y_center + y_half ))
                elif self.heading == FACE_SOUTHWEST:
                    triangle.append(cmpPoint(x_center - x_quarter, y_center + y_half ))
                    triangle.append(cmpPoint(x_center - x_3quarter, y_center + y_3quarter ))
                    triangle.append(cmpPoint(x_center - x_half, y_center + y_quarter ))
                elif self.heading == FACE_WEST:
                    triangle.append(cmpPoint(x_center - x_half, y_center + y_quarter ))
                    triangle.append(cmpPoint(x_center - x_3quarter, y_center ))
                    triangle.append(cmpPoint(x_center - x_half, y_center - y_quarter ))
                elif self.heading == FACE_NORTHWEST:
                    triangle.append(cmpPoint(x_center - x_half, y_center - y_quarter ))
                    triangle.append(cmpPoint(x_center - x_3quarter, y_center - y_3quarter ))
                    triangle.append(cmpPoint(x_center - x_quarter, y_center - y_half ))

                dc.DrawPolygon(triangle)
                dc.SetBrush(wxNullBrush)
                dc.SetPen(wxNullPen)


            #selected outline
            if self.selected:
                dc.SetPen(wxRED_PEN)
                dc.SetBrush(wxTRANSPARENT_BRUSH)
                dc.DrawRectangle(self.pos.x, self.pos.y,
                    self.bmp.GetWidth(), self.bmp.GetHeight())
                dc.SetBrush(wxNullBrush)
                dc.SetPen(wxNullPen)
            # draw label
            if len(self.label):
                dc.SetTextForeground(wxRED)
                (textWidth,textHeight) = dc.GetTextExtent(self.label)
                x = self.pos.x +((self.bmp.GetWidth() - textWidth) /2) - 1
                y = self.pos.y + self.bmp.GetHeight() + 6
                dc.SetPen(wxWHITE_PEN)
                dc.SetBrush(wxWHITE_BRUSH)
                dc.DrawRectangle(x,y,textWidth+2,textHeight+2)
                if (textWidth+2>self.right):
                    self.right+=int((textWidth+2-self.right)/2)+1
                    self.left-=int((textWidth+2-self.right)/2)+1                
                self.bottom=y+textHeight+2-self.pos.y
                dc.SetPen(wxNullPen)
                dc.SetBrush(wxNullBrush)
                dc.DrawText(self.label,x+1,y+1)

            self.top-=5
            self.bottom+=5
            self.left-=5
            self.right+=5
            return true
        else:
            return false

    def toxml(self, action = "update",preserve_changed=0):


        xml_str = ""

        if preserve_changed:
            original_changed = self._changed_attr()

        if action == "del":
            xml_str = "<miniature action='del' id='" + self.id + "'/>"
            return xml_str
        if action == "new":
            self._dirty_all_attr()

        changed = self._changed_attr()

        if changed:

            #  if there are any changes, make sure id is one of them
            if not changed.has_key("id"):
                self._dirty_attr("id")
                changed = self._changed_attr()

            xml_str = "<miniature"

            xml_str += " action='" + action + "'"

            for a in changed.keys():
                if a == "pos":
                    if not(self.pos is None):
                        xml_str += " posx='" + str(self.pos.x) + "'"
                        xml_str += " posy='" + str(self.pos.y) + "'"

                elif a == "heading":
                    if not (self.heading is None):
                        xml_str += " heading='" + str(self.heading) + "'"
                elif a == "face":
                    if not (self.face is None):
                        xml_str += " face='" + str(self.face) + "'"
                elif a == "path":
                    if not (self.path is None):
                        xml_str += " path='" + self.path + "'"
                elif a == "locked":
                    if not (self.locked is None):
                        xml_str+= "  locked='" + str(self.locked) + "'"
                elif a == "hide":
                    if not (self.hide is None):
                        xml_str+= " hide='" + str(self.hide) + "'"
                elif a == "label":
                    if not (self.label is None):
                        xml_str+= " label='" + self.label + "'"
                elif a == "snap_to_align":
                    if not (self.snap_to_align is None):
                        xml_str+= " align='" + str(self.snap_to_align) + "'"
                elif a == "id":
                    if not (self.id is None):
                        xml_str+= " id='" + self.id + "'"
                elif a == "zorder":
                    if not(self.id is None):
                        xml_str+= " zorder='" + str(self.zorder) + "'"
                elif a == "width":
                    if not(self.width is None):
                        xml_str+= " width='" + str(self.width) + "'"
                elif a == "height":
                    if not(self.height is None):
                        xml_str+= " height='" + str(self.height) + "'"

            xml_str += "/>"

        self._clean_all_attr()

        if preserve_changed:
            for a in original_changed.keys():
                self._dirty_attr(a)


        return xml_str

    def takedom(self,xml_dom):
        posx = xml_dom.getAttribute("posx")
        if posx <> "":
            self.pos.x = int(posx)
            self._clean_attr("pos")

        posy = xml_dom.getAttribute("posy")
        if posy <> "":
            self.pos.y = int(posy)
            self._clean_attr("pos")

        heading = xml_dom.getAttribute("heading")
        if heading <> "":
            self.heading = int(heading)
            self._clean_attr("heading")

        face = xml_dom.getAttribute("face")
        if face <> "":
            self.face = int(face)
            self._clean_attr("face")

        path = xml_dom.getAttribute("path")
        if path <> "":
            self.path = path
##            self.bmp = wxBitmap("icons/fetching.png",wxBITMAP_TYPE_PNG)
##            load_img(path,"miniature",self.id)  # change the bitmap image
            # Sorry if I hosed this up...just didn't look right
            self.set_bmp(images.load_img(path,"miniature",self.id))  # change the bitmap image
            self._clean_attr("path")

        locked = xml_dom.getAttribute("locked")
        if locked <> "":
            self.locked = int(locked)
            self._clean_attr("locked")

        hide = xml_dom.getAttribute("hide")
        if hide <> "":
            self.hide = int(hide)
            self._clean_attr("hide")

        label = xml_dom.getAttribute("label")
        if label <> "":
            self.label = label
            self._clean_attr("label")

        zorder = xml_dom.getAttribute("zorder")
        if zorder <> "":
            self.zorder = int(zorder)
            self._clean_attr("zorder")

        snap_to_align = xml_dom.getAttribute("align")
        if snap_to_align <> "":
            self.snap_to_align = int(snap_to_align)
            self._clean_attr("snap_to_align")

        id = xml_dom.getAttribute("id")
        if id <> "":
            self.id = id
            self._clean_attr("id")

        width = xml_dom.getAttribute("width")
        if width <> "":
            self.width = int(width)
            self._clean_attr("width")

        height = xml_dom.getAttribute("height")
        if height <> "":
            self.height = int(height)
            self._clean_attr("height")

##-----------------------------
## miniature layer
##-----------------------------
class miniature_layer(layer_base):
    def __init__(self, canvas):
        layer_base.__init__(self)
        self._protected_attr = ["serial_number"]
        self.canvas = canvas
        self.id = -1
        self.miniatures = []
        self.serial_number = 0

        self._clean_all_attr()

    def next_serial( self ):
        self.serial_number += 1
        return self.serial_number

    def get_next_highest_z(self):
        z = len(self.miniatures)+1
        return z


    def cleanly_collapse_zorder(self):
        #  lock the zorder stuff

        sorted_miniatures = self.miniatures[:]
        sorted_miniatures.sort(cmp_zorder)
        i = 0
        for mini in sorted_miniatures:
            mini.zorder = i
            i = i + 1
            mini._clean_attr("zorder")

        #  unlock the zorder stuff



    def collapse_zorder(self):
        #  lock the zorder stuff

        sorted_miniatures = self.miniatures[:]
        sorted_miniatures.sort(cmp_zorder)
        i = 0
        for mini in sorted_miniatures:
            if (mini.zorder != MIN_STICKY_BACK) and (mini.zorder != MIN_STICKY_FRONT):
##                print "zorder is " + str(mini.zorder) + " and is being set to " + str(i)
                mini.zorder = i
            else:
                print "sticky item found having value of " + str( mini.zorder)
            i = i + 1
        #  unlock the zorder stuff



    def rollback_serial( self ):
        self.serial_number -= 1

    def add_miniature(self,id,path,pos=cmpPoint(0,0),label="",heading=FACE_NONE,face=FACE_NONE,width=0, height=0):
        print "Before mini creation:" + str(self.get_next_highest_z())
        bmp = images.load_img(path,"miniature",id)
        if bmp:
            mini = bmp_miniature(id,path,bmp,pos,heading,face,label,
                                 zorder = self.get_next_highest_z(),width = width,height = height)
            print "After mini creation:" + str(self.get_next_highest_z())
            self.miniatures.append(mini)
            print "After mini addition:" + str(self.get_next_highest_z())
            mini._dirty_all_attr()
            xml_str = "<map><miniatures>"
            xml_str += mini.toxml("new")
            xml_str += "</miniatures></map>"
            self.canvas.frame.session.send(xml_str)

        else:
            print "Invalid image " + path + " has been ignored!"

    def get_miniature_by_id(self,id):

        for mini in self.miniatures:
            if mini.id == id:
                return mini
        return None

    def del_miniature(self,min):
        xml_str = "<map><miniatures>"
        xml_str += min.toxml("del")
        xml_str += "</miniatures></map>"
        self.canvas.frame.session.send(xml_str)
        self.miniatures.remove(min)
        del min
        self.collapse_zorder()

    def del_all_miniatures(self):
        while len(self.miniatures):
            min = self.miniatures.pop()
            del min
        self.collapse_zorder()
        
    def draw(self,dc,topleft,size):
        #min = self.miniatures.keys()
        sorted_miniatures = self.miniatures[:]
        sorted_miniatures.sort(cmp_zorder)

        for m in sorted_miniatures:
            if (m.pos.x>topleft[0]-m.right and
                m.pos.y>topleft[1]-m.bottom and 
                m.pos.x<topleft[0]+size[0]-m.left and
                m.pos.y<topleft[1]+size[1]-m.top):
                m.draw(dc)



    def find_miniature(self, pt,only_locked=false):
        min_list = []
        for m in self.miniatures:
            if m.hit_test(pt):
                if only_locked and not m.locked:
                    min_list.append(m)
                elif not only_locked:
                    min_list.append(m)
                else:
                    continue
        if len(min_list) > 0:
            return min_list
        else:
            return None

    def toxml(self,action="update"):
        """ format  """

        attributes = ""
        if action == "new":
            self._dirty_all_attr()
        changed = self._changed_attr()

        if changed:
            for a in changed.keys():
                if a == "serial_number":
                    attributes += " serial='" + str(self.serial_number) + "'"
                    self._clean_attr("serial_number")

        minis_string = ""
        if self.miniatures:
            for m in self.miniatures:
                minis_string += m.toxml(action)

        if minis_string or changed:
            s = "<miniatures"
            s += attributes
            if minis_string:
                s += ">"
                s += minis_string
                s += "</miniatures>"
            else:
                s+="/>"

            return s

        else:
            return ""

    def takedom(self,xml_dom):

        serial_number = xml_dom.getAttribute('serial')
        if serial_number <> "":
            self.serial_number = int(serial_number)
            self._clean_attr("serial_number")

        children = xml_dom._get_childNodes()
        for c in children:

            action = c.getAttribute("action")
            id = c.getAttribute('id')

            if action == "del":
                mini = self.get_miniature_by_id(id)
                if mini:
                    self.miniatures.remove(mini)
                    del mini
                else:
                    wxMessageBox("Deletion of unknown mini attempted","Map Synchronization Error")


            elif action == "new":
                pos = cmpPoint(int(c.getAttribute('posx')),int(c.getAttribute('posy')))
                path = c.getAttribute('path')
                label = c.getAttribute('label')
                try:
                    height = int(c.getAttribute('height'))
                    width = int(c.getAttribute('width'))
                    locked = int(c.getAttribute('locked'))
                    hide = int(c.getAttribute('hide'))
                    heading = int(c.getAttribute('heading'))
                    face = int(c.getAttribute('face'))
                    snap_to_align = int(c.getAttribute('align'))

                except:
                    height = width = locked = hide = heading = face = snap_to_align = zorder = 0

                #  The following section is necessary because the zorder might not be sent
                #    e.g. 0.9.4 clients

                #  Assume everything is okay
                zorder_bad = 0
                #  If there's a problem getting a zorder (like it's not there)
                #    set the flag
                try:
                    zorder = int(c.getAttribute('zorder'))
                except:
                    zorder_bad = 1
                    zorder = 0


                min = bmp_miniature(id, path, images.load_img(path,"miniature",id),pos, heading,
                      face, label, locked, hide, snap_to_align, zorder, width, height)

                self.miniatures.append(min)


                #  collapse the zorder.  If the client behaved well, then nothing should change.
                #    Otherwise, this will ensure that there's some kind of z-order
                self.collapse_zorder()

                #  Do the normal clean of this minis attributes
                min._clean_all_attr()

                #  But dirty zorder if there was a problem with it.  A new zorder should now
                #     be in effect
                if zorder_bad:
                    min._dirty_attr('zorder')


            else:
                mini = self.get_miniature_by_id(id)
                if mini:
                    mini.takedom(c)
                else:
                    wxMessageBox("Update of unknown mini attempted","Map Synchronization Error")

        #self.canvas.send_map_data()
        
