# sqlsession.py
#
# Copyright 2003 Wichert Akkerman <wichert@deephackmode.org>
#
# This file is free software; you can redistribute it and/or modify it
# under the terms of version 2 of the GNU General Public License as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# Calculate shared library dependencies

"""Session storage class

When dealing with multiple simultaneous sessions as is common with web-based
programming it is very useful to store session information on the server. To
do this a special kind of database is needed which allows for multiple readers
and writers in parallel.

This module implements such a database using a SQL database. The tables have
to use this structure::

 CREATE TABLE paidnet(
        id              VARCHAR(16) NOT NULL,
        attribute       VARCHAR(16) NOT NULL,
        value           VARCHAR(255) NOT NULL,
        CONSTRAINT single_key UNIQUE (id, attribute));
"""

__docformat__	= "epytext en"

import sqldict


class SQLSession(sqldict.SQLDict):
	"""Session storage

	This class acts very much like a selve.Shelf class, except for two
	differences:
	
	  1. it uses a SQL database
	  2. You can only store dictionaries in the top dictionary
	
	It works very simple::

	  session=Session("table", sqlserver["database"])
	  del session[oldkey]
	  session[sessionkey]["hostname"]=os.environ["REMOTE_ADDR"]
	  session[sessionkey]["username"]="username"
	"""
	table	= "session"
	keycols	= [ "id" ]

	def __getitem__(self, key):
		return _SQLSubSession(self, key)


class _SQLSubSession:
	def __init__(self, session, id):
		self.session=session
		self.id=id


	def __getitem__(self, key):
		res=self.session.dbc.query(
			"SELECT value FROM %s WHERE id=%%s AND attribute=%%s;" % self.session.table,
			(self.id, key), "format")
		
		if len(res)==0:
			raise KeyError, key

		return res[0][0]


	def __setitem__(self, key, item):
		try:
			self.session.dbc.execute(
				"INSERT INTO %s (id, attribute, value) VALUES (%%s, %%s, %%s);" % self.session.table,
				(self.id, key, item), "format")
		except self.session.dbc.IntegrityError:
			self.session.dbc.execute(
				"UPDATE %s SET value=%%s WHERE id=%%s AND attribute=%%s;" % self.session.table,
				(item, self.id, key), "format")


	def __delitem__(self, key):
		c=self.session.dbc.execute("DELETE FROM %s WHERE id=%%s AND attribute=%%s;" % self.session.table,
			(self.id, key), "format")

		if c.rowcount==0:
			raise KeyError, key

	def has_key(self, key):
		return self.session.dbc.query(
			"SELECT COUNT(value) FROM %s WHERE id=%%s AND attribute=%%s;" % self.session.table,
			(self.id, key), "format")[0][0]>0
		

	def keys(self):
		return map(lambda x: x[0],
			self.session.dbc.query(
				"SELECT attribute FROM %s WHERE id=%%s;" % self.session.table,
				(self.id,), "format"))
	

	def clear(self):
		self.dbc.execute("DELETE FROM %%s WHERE id=%s;" % self.session.table,
			(self.id,), "format")

