/*
 * ossmodule.c - A Python module for the Open Sound System
 *
 * Public Domain 1997
 * Timothy Butler
 *
 * THIS DOCUMENT IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifdef MODITEM

/*
 * The MODITEM macro will both define and initialize module-level
 * constants. ossmodule.c includes itself using the C preprocessor
 * and sets MODITEM to either define variables or initialize 
 * variables (in initoss() ).
 */

MODITEM(SOUND_MIXER_NRDEVICES, "i")
MODITEM(SOUND_MIXER_VOLUME,    "i")
MODITEM(SOUND_MIXER_BASS,      "i")
MODITEM(SOUND_MIXER_TREBLE,    "i")
MODITEM(SOUND_MIXER_SYNTH,     "i")
MODITEM(SOUND_MIXER_PCM,       "i")
MODITEM(SOUND_MIXER_SPEAKER,   "i")
MODITEM(SOUND_MIXER_LINE,      "i")
MODITEM(SOUND_MIXER_MIC,       "i")
MODITEM(SOUND_MIXER_CD,        "i")
MODITEM(SOUND_MIXER_IMIX,      "i")
MODITEM(SOUND_MIXER_ALTPCM,    "i")
MODITEM(SOUND_MIXER_RECLEV,    "i")
MODITEM(SOUND_MIXER_IGAIN,     "i")
MODITEM(SOUND_MIXER_OGAIN,     "i")
MODITEM(SOUND_MIXER_LINE1,     "i")
MODITEM(SOUND_MIXER_LINE2,     "i")
MODITEM(SOUND_MIXER_LINE3,     "i")

MODITEM(SOUND_MASK_VOLUME,    "i")
MODITEM(SOUND_MASK_BASS,      "i")
MODITEM(SOUND_MASK_TREBLE,    "i")
MODITEM(SOUND_MASK_SYNTH,     "i")
MODITEM(SOUND_MASK_PCM,       "i")
MODITEM(SOUND_MASK_SPEAKER,   "i")
MODITEM(SOUND_MASK_LINE,      "i")
MODITEM(SOUND_MASK_MIC,       "i")
MODITEM(SOUND_MASK_CD,        "i")
MODITEM(SOUND_MASK_IMIX,      "i")
MODITEM(SOUND_MASK_ALTPCM,    "i")
MODITEM(SOUND_MASK_RECLEV,    "i")
MODITEM(SOUND_MASK_IGAIN,     "i")
MODITEM(SOUND_MASK_OGAIN,     "i")
MODITEM(SOUND_MASK_LINE1,     "i")
MODITEM(SOUND_MASK_LINE2,     "i")
MODITEM(SOUND_MASK_LINE3,     "i")


MODITEM(SOUND_CAP_EXCL_INPUT,  "i")

MODITEM(SYNTH_TYPE_FM,     "i")
MODITEM(SYNTH_TYPE_SAMPLE, "i")
MODITEM(SYNTH_TYPE_MIDI,   "i")

MODITEM(SNDCARD_ADLIB,      "i")
MODITEM(SNDCARD_SB,         "i")
MODITEM(SNDCARD_PAS,        "i")
MODITEM(SNDCARD_GUS,        "i")
MODITEM(SNDCARD_MPU401,     "i")
MODITEM(SNDCARD_SB16,       "i")
MODITEM(SNDCARD_SB16MIDI,   "i")
MODITEM(SNDCARD_UART6850,   "i")
MODITEM(SNDCARD_GUS16,      "i")
MODITEM(SNDCARD_MSS,        "i")
MODITEM(SNDCARD_PSS,        "i")
MODITEM(SNDCARD_SSCAPE,     "i")
MODITEM(SNDCARD_PSS_MPU,    "i")
MODITEM(SNDCARD_PSS_MSS,    "i")
MODITEM(SNDCARD_SSCAPE_MSS, "i")
MODITEM(SNDCARD_TRXPRO,     "i")
MODITEM(SNDCARD_TRXPRO_SB,  "i")
MODITEM(SNDCARD_TRXPRO_MPU, "i")
MODITEM(SNDCARD_AWE32,      "i")

MODITEM(AFMT_QUERY,     "i")
MODITEM(AFMT_MU_LAW,    "i")
MODITEM(AFMT_A_LAW,     "i")
MODITEM(AFMT_IMA_ADPCM, "i")
MODITEM(AFMT_U8,        "i")
MODITEM(AFMT_S16_LE,    "i")
MODITEM(AFMT_S16_BE,    "i")
MODITEM(AFMT_S8,        "i")
MODITEM(AFMT_U16_LE,    "i")
MODITEM(AFMT_U16_BE,    "i")
MODITEM(AFMT_MPEG,      "i")
/* AFMT_S16_NE ?? */

#else

static 
char ModVersion[] = "$Revision: 1.4 $";
/* 
 * $Source: /usr/local/cvsroot/ossmodule/ossmodule.c,v $ 
 */
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <ioctl.h>

#include "Python.h"
#include "structmember.h"

#include "soundcard.h"

#define MOD_MIXER_FILE_DEFAULT     "/dev/mixer"
#define MOD_SEQUENCER_FILE_DEFAULT "/dev/sequencer"
#define MOD_AUDIO_FILE_DEFAULT     "/dev/audio"
#define MOD_VELOCITY_DEFAULT       64

/*
 * ??? OSS docs use SOUND_XXXX, differ from my FreeBSD soundcard.h 
 */
#ifndef SOUND_MIXER_READ
#define SOUND_MIXER_READ MIXER_READ
#endif

#ifndef SOUND_MIXER_WRITE
#define SOUND_MIXER_WRITE MIXER_WRITE
#endif


/*
 * Many of the oss module objects have methods that translate
 * directly into ioctl() operations.
 * 
 * These macros define some common method functions that use
 * ioctl():
 *
 *   OP      -- simple ioctl operation with no argument
 *   RDINT   -- operation to read an integer 
 *   WRINT   -- operation to write an integer
 *   WRRDINT -- operation that writes and then returns an integer
 *              that may have been updated by the ioctl() call.
 *
 * RDINT sets its integer argument to zero because some operations
 * depend on this (such as SNDCTL_SEQ_CTRLRATE).
 */

#define RDINT(ext_type,  method, fd, op)        \
static                                          \
PyObject *                                      \
method(ext_type *self, PyObject *args)          \
{                                               \
                                                \
    int result = 0;                             \
                                                \
    if (!PyArg_ParseTuple(args, ""))            \
        return NULL;                            \
                                                \
                                                \
    if (ioctl((fd), (op), &result) == -1)       \
    {                                           \
        PyErr_SetFromErrno(PyExc_IOError);      \
        return NULL;                            \
    }                                           \
                                                \
    return Py_BuildValue("i", result);          \
                                                \
}

#define WRINT(ext_type, method, fd, op)         \
static                                          \
PyObject *                                      \
method(ext_type *self, PyObject *args)          \
{                                               \
                                                \
    int i;                                      \
                                                \
    if (!PyArg_ParseTuple(args, "i", & i))      \
        return NULL;                            \
                                                \
                                                \
    if (ioctl((fd), (op), &i) == -1)            \
    {                                           \
        PyErr_SetFromErrno(PyExc_IOError);      \
        return NULL;                            \
    }                                           \
                                                \
    Py_INCREF(Py_None);                         \
    return Py_None;                             \
                                                \
}

#define WRRDINT(ext_type, method, fd, op)       \
static                                          \
PyObject *                                      \
method(ext_type *self, PyObject *args)          \
{                                               \
                                                \
    int i;                                      \
                                                \
    if (!PyArg_ParseTuple(args, "i", & i))      \
        return NULL;                            \
                                                \
                                                \
    if (ioctl((fd), (op), &i) == -1)            \
    {                                           \
        PyErr_SetFromErrno(PyExc_IOError);      \
        return NULL;                            \
    }                                           \
                                                \
    return Py_BuildValue("i", i);               \
}

#define OP(ext_type, method, fd, op)            \
static                                          \
PyObject *                                      \
method(ext_type *self, PyObject *args)          \
{                                               \
                                                \
    int i;                                      \
                                                \
    if (!PyArg_ParseTuple(args, ""))            \
        return NULL;                            \
                                                \
                                                \
    if (ioctl((fd), (op), 0) == -1)             \
    {                                           \
        PyErr_SetFromErrno(PyExc_IOError);      \
        return NULL;                            \
    }                                           \
                                                \
    Py_INCREF(Py_None);                         \
    return Py_None;                             \
                                                \
}


/*************************************************************************
 *
 *  Mixer - Extension Type
 *
 *************************************************************************/
typedef struct
{
    PyObject_HEAD 
    int fd;
} MixerObject;

staticforward PyTypeObject MixerType; /* shared type descriptor */

static 
MixerObject *
MixerNew(int fd)
{
    MixerObject *self;
    
    self = PyObject_NEW(MixerObject, &MixerType);
    
    if (self == NULL)
        return NULL;

    self->fd = fd;
    return self;
}

/*
 * Mixer - Instance Methods
 */

static
PyObject *
Mixer_fileno(MixerObject *self, PyObject *args)
{
    if (! PyArg_ParseTuple(args, ""))
        return NULL;

    return Py_BuildValue("i", self->fd);

}

static
PyObject *
Mixer_close(MixerObject *self, PyObject *args)
{
    int result;

    if (! PyArg_ParseTuple(args, ""))
        return NULL;
    
    if (close(self->fd) == -1)
    {
        PyErr_SetFromErrno(PyExc_IOError);      
        return NULL;
    }

    self->fd = -1;

    Py_INCREF(Py_None);
    return Py_None;

}

RDINT(MixerObject, Mixer_devmask,   self->fd, SOUND_MIXER_READ_DEVMASK)
RDINT(MixerObject, Mixer_recmask,   self->fd, SOUND_MIXER_READ_RECMASK)
RDINT(MixerObject, Mixer_stereodevs,self->fd, SOUND_MIXER_READ_STEREODEVS)
RDINT(MixerObject, Mixer_caps,      self->fd, SOUND_MIXER_READ_CAPS)

RDINT(MixerObject, Mixer_read_recsrc,    self->fd, SOUND_MIXER_READ_RECSRC)
WRINT(MixerObject, Mixer_write_recsrc,   self->fd, SOUND_MIXER_WRITE_RECSRC)

static
PyObject *
Mixer_read_channel(MixerObject *self, PyObject *args)
{
    int channel;
    int level;
    int left, right;

    if (!PyArg_ParseTuple(args, "i", &channel))
        return NULL;

    if (ioctl(self->fd, SOUND_MIXER_READ(channel), &level) == -1)
    {
        PyErr_SetFromErrno(PyExc_IOError);      
        return NULL;
    }

    left  =  level       & 0x00ff;
    right = (level >> 8) & 0x00ff;

    return Py_BuildValue("(i,i)", left, right);
}

static
PyObject *
Mixer_write_channel(MixerObject *self, PyObject *args)
{

    int channel;
    int level;
    int left, right;

    if (!PyArg_ParseTuple(args, "i(ii)", &channel,  &left, &right))
    {
        PyErr_Clear();
        if (!PyArg_ParseTuple(args, "ii", &channel, &left))
            return NULL;
        else
            right = left;
    }
    
    level = (right << 8) | left;

    if (ioctl(self->fd, SOUND_MIXER_WRITE(channel), &level) == -1)
    {
        PyErr_SetFromErrno(PyExc_IOError);      
        return NULL;
    }

    left  =  level       & 0x00ff;
    right = (level >> 8) & 0x00ff;

    return Py_BuildValue("(i,i)", left, right);
    
}

static
struct PyMethodDef MixerMethods[] = 
{
    { "fileno",           (PyCFunction) Mixer_fileno,           1},
    { "close",            (PyCFunction) Mixer_close,            1},
    { "devmask",          (PyCFunction) Mixer_devmask,          1},
    { "recmask",          (PyCFunction) Mixer_recmask,          1},
    { "stereodevs",       (PyCFunction) Mixer_stereodevs,       1},
    { "caps",             (PyCFunction) Mixer_caps,             1},
    { "read_recsrc",      (PyCFunction) Mixer_read_recsrc,      1},
    { "write_recsrc",     (PyCFunction) Mixer_write_recsrc,     1},
    { "read_channel",     (PyCFunction) Mixer_read_channel,     1},
    { "write_channel",    (PyCFunction) Mixer_write_channel,    1},

    { NULL, NULL }

};

/*
 * Mixer - Basic type operations
 */
static
void
Mixer_tp_dealloc(MixerObject *self)
{

    if (self->fd != -1)
    {
        close(self->fd);
    }

    PyMem_DEL(self);
}

static
int 
Mixer_tp_print(MixerObject *self, FILE *fp, int flags)
{
    fprintf(fp, "<Mixer: fd == %d>\n", self->fd);
    return 0;
}

static 
PyObject *
Mixer_tp_getattr(MixerObject *self, char *name)
{
    return Py_FindMethod(MixerMethods, (PyObject *) self, name);
}
/*
 * Type Descriptor
 */

static
PyTypeObject MixerType = 
{
    PyObject_HEAD_INIT(&PyType_Type)
    0,
    "mixer",
    sizeof(MixerObject),
    0,

    /* standard methods */
    (destructor)  Mixer_tp_dealloc, /* tp_dealloc */
    (printfunc)   Mixer_tp_print,
    (getattrfunc) Mixer_tp_getattr,
    0,
    0,
    0,
    
    /* type categories */
    0,
    0,
    0,
    
};

/*************************************************************************
 *
 * SynthInfo - Type Extension
 *
 *************************************************************************/

typedef struct 
{
    PyObject_HEAD
    struct synth_info si;
} SynthInfoObject;


/*
 * SynthInfo - Basic Type Operations
 */

staticforward PyTypeObject SynthInfoType;

static
SynthInfoObject *
SynthInfoNew(int device)
{

    SynthInfoObject   *self;
    struct synth_info *sip;

    self = PyObject_NEW(SynthInfoObject, &SynthInfoType);
    if (self == NULL)
        return NULL;
    
    sip = &(self->si);
    (sip->name)[0]       = '\0';
    sip->device          = device;
    sip->synth_type      = 0;
    sip->synth_subtype   = 0;
    sip->perc_mode       = 0;
    sip->nr_voices       = 0;
    sip->nr_drums        = 0;
    sip->instr_bank_size = 0;
    sip->capabilities    = 0;

    return self;
}



static 
void
SynthInfo_tp_dealloc(SynthInfoObject *self)
{
    PyMem_DEL(self);
}

static
char *
SynthInfoSynthTypeName(int synth_type)
{
    switch(synth_type)
    {
      case SYNTH_TYPE_FM:     return "SYNTH_TYPE_FM";
      case SYNTH_TYPE_SAMPLE: return "SYNTH_TYPE_SAMPLE";
      case SYNTH_TYPE_MIDI:   return "SYNTH_TYPE_MIDI";
      default:                return "UNKNOWN";
    }
}

static
char *
SynthInfoSynthSubTypeName(int synth_type)
{
    switch(synth_type)
    {
      case FM_TYPE_ADLIB:     return "FM_TYPE_ADLIB";
      case FM_TYPE_OPL3:      return "FM_TYPE_OPL3";
      case SAMPLE_TYPE_GUS:   return "SAMPLE_TYPE_GUS";
      case SAMPLE_TYPE_AWE32: return "SAMPLE_TYPE_AWE32";
      default:                return "UNKNOWN";
    }
}

static
int
SynthInfo_tp_print(SynthInfoObject *self, FILE *fp, int flags)
{
    struct synth_info *sip = &(self->si);

    fprintf(fp, "synth_info:\n");
    fprintf(fp, "\tname:            %.30s\n",   sip->name);
    fprintf(fp, "\tdevice:          %d\n",      sip->device);
    fprintf(fp, "\tsynth_type:      %d     (%s)\n", sip->synth_type, 
            SynthInfoSynthTypeName(sip->synth_type));
    fprintf(fp, "\tsynth_subtype:   0x%04x (%s)\n", sip->synth_subtype,
            SynthInfoSynthSubTypeName(sip->synth_subtype));
    fprintf(fp, "\tperc_mode:       %d (unsupported)\n",      
                                              sip->perc_mode);
    fprintf(fp, "\tnr_voices:       %d\n",      sip->nr_voices);
    fprintf(fp, "\tnr_drums:        %d (obsolete)\n",      
                                              sip->nr_drums);
    fprintf(fp, "\tinstr_bank_size: %d\n",    sip->instr_bank_size);
    fprintf(fp, "\tcapabilities:    0x%04lx\n",  sip->capabilities);
    
    if (sip->capabilities & SYNTH_CAP_PERCMODE)
        fprintf(fp, "\t\tSYNTH_CAP_PERCMODE\n");
    if (sip->capabilities & SYNTH_CAP_OPL3)
        fprintf(fp, "\t\tSYNTH_CAP_OPL3\n");
    if (sip->capabilities & SYNTH_CAP_INPUT)
        fprintf(fp, "\t\tSYNTH_CAP_INPUT\n");

    return 0;

}

/*
 * SynthInfo - Members
 */

#define OFF(x) offsetof(struct synth_info, x) 
static
struct memberlist SynthInfoMemberList[] = 
{
    {"name",            T_STRING_INPLACE, OFF(name),           RO},
    {"device",          T_INT,            OFF(device)            },
    {"synth_type",      T_INT,            OFF(synth_type),     RO},
    {"synth_subtype",   T_INT,            OFF(synth_subtype),  RO},
    {"perc_mode",       T_INT,            OFF(perc_mode),      RO},
    {"nr_voices",       T_INT,            OFF(nr_voices),      RO},
    {"nr_drums",        T_INT,            OFF(nr_drums),       RO},
    {"instr_bank_size", T_INT,            OFF(instr_bank_size),RO},
    {"capabilities",    T_ULONG,          OFF(capabilities),   RO},

    {NULL, 0 , 0 }
};

static
PyObject *
SynthInfo_tp_getattr(SynthInfoObject *self, char *attr_name)
{
    return PyMember_Get((char *) &(self->si), SynthInfoMemberList, attr_name);
}

static
int
SynthInfo_tp_setattr(SynthInfoObject *self, char *attr_name, PyObject *v)
{

    if (v == NULL) 
    {
        PyErr_SetString(PyExc_TypeError, "can't delete synth_info attributes");
        return -1;
    }

    return PyMember_Set((char *) &(self->si), 
                        SynthInfoMemberList, attr_name, v);
}

/*
 * SynthInfo - Type Descriptor
 */

static 
PyTypeObject SynthInfoType = 
{
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,
    "synth_info",
    sizeof(SynthInfoObject),
    0,

    /* standard methods */
    (destructor)  SynthInfo_tp_dealloc,
    (printfunc)   SynthInfo_tp_print,
    (getattrfunc) SynthInfo_tp_getattr,
    (setattrfunc) SynthInfo_tp_setattr,
    0,
    0,
};


/*************************************************************************
 *
 * MidiInfo - Type Extension
 *
 *************************************************************************/

typedef struct 
{
    PyObject_HEAD
    struct midi_info mi;
} MidiInfoObject;


/*
 * MidiInfo - Basic Type Operations
 */

staticforward PyTypeObject MidiInfoType;

static
MidiInfoObject *
MidiInfoNew(int device)
{

    MidiInfoObject   *self;
    struct midi_info *mip;

    self = PyObject_NEW(MidiInfoObject, &MidiInfoType);
    if (self == NULL)
        return NULL;
    
    mip = &(self->mi);
    (mip->name)[0]       = '\0';
    mip->device          = device;
    mip->capabilities    = 0;
    mip->dev_type        = 0;

    return self;
}

static 
void
MidiInfo_tp_dealloc(MidiInfoObject *self)
{
    PyMem_DEL(self);
}


static
int
MidiInfo_tp_print(MidiInfoObject *self, FILE *fp, int flags)
{
    struct midi_info *mip = &(self->mi);

    fprintf(fp, "midi_info:\n");
    fprintf(fp, "\tname:            %.30s\n",   mip->name);
    fprintf(fp, "\tdevice:          %d\n",      mip->device);
    fprintf(fp, "\tdev_type:        %d\n",      mip->dev_type); 
    fprintf(fp, "\tcapabilities:    0x%04lx\n", mip->capabilities);
    
    if (mip->capabilities & MIDI_CAP_MPU401)
        fprintf(fp, "\t\tMIDI_CAP_MPU401\n");

    return 0;

}

/*
 * MidiInfo - Members
 */

#undef OFF
#define OFF(x) offsetof(struct midi_info, x) 
static
struct memberlist MidiInfoMemberList[] = 
{
    {"name",            T_STRING_INPLACE, OFF(name),           RO},
    {"device",          T_INT,            OFF(device)            },
    {"dev_type",        T_INT,            OFF(dev_type),       RO},
    {"capabilities",    T_ULONG,          OFF(capabilities),   RO},

    {NULL, 0 , 0 }
};


static
PyObject *
MidiInfo_tp_getattr(MidiInfoObject *self, char *attr_name)
{
    return PyMember_Get((char *) &(self->mi), MidiInfoMemberList, attr_name);
}

static
int
MidiInfo_tp_setattr(MidiInfoObject *self, char *attr_name, PyObject *v)
{

    if (v == NULL) 
    {
        PyErr_SetString(PyExc_TypeError, "can't delete midi_info attributes");
        return -1;
    }
    return PyMember_Set((char *) &(self->mi),
                        SynthInfoMemberList, attr_name, v);
}

/*
 * MidiInfo - Type Descriptor
 */

static 
PyTypeObject MidiInfoType = 
{
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,
    "midi_info",
    sizeof(MidiInfoObject),
    0,

    /* standard methods */
    (destructor)  MidiInfo_tp_dealloc,
    (printfunc)   MidiInfo_tp_print,
    (getattrfunc) MidiInfo_tp_getattr,
    (setattrfunc) MidiInfo_tp_setattr,
    0,
    0,
};



/**************************************************************************
 *
 * SoundTimerInfo - Extension Type ??? Is this used anywhere?
 *
 ***************************************************************************/

typedef struct 
{
    PyObject_HEAD
    struct sound_timer_info sti;
} SoundTimerInfoObject;


/*
 * SoundTimerInfo - Basic Type Operations
 */

staticforward PyTypeObject SoundTimerInfoType;

static
SoundTimerInfoObject *
SoundTimerInfoNew(int device)
{

    SoundTimerInfoObject    *self;
    struct sound_timer_info *stip;

    self = PyObject_NEW(SoundTimerInfoObject, &SoundTimerInfoType);
    if (self == NULL)
        return NULL;
    
    stip = &(self->sti);
    (stip->name)[0] = '\0';
    stip->caps      = 0;

    return self;
}

static 
void
SoundTimerInfo_tp_dealloc(SoundTimerInfoObject *self)
{
    PyMem_DEL(self);
}


static
int
SoundTimerInfo_tp_print(SoundTimerInfoObject *self, FILE *fp, int flags)
{
    struct sound_timer_info *stip = &(self->sti);

    fprintf(fp, "sound_timer_info:\n");
    fprintf(fp, "\tname:    %.30s\n",  stip->name);
    fprintf(fp, "\tcaps:    0x%04x\n", stip->caps);
    
    return 0;

}
/*
 * SoundTimerInfo - Members
 */

#undef OFF
#define OFF(x) offsetof(struct sound_timer_info, x) 
static
struct memberlist SoundTimerInfoMemberList[] = 
{
    {"name",    T_STRING_INPLACE, OFF(name),   RO},
    {"caps",    T_ULONG,          OFF(caps),   RO},

    {NULL, 0 , 0 }
};

static
PyObject *
SoundTimerInfo_tp_getattr(SoundTimerInfoObject *self, char *attr_name)
{
    return PyMember_Get((char *) &(self->sti), 
                        SoundTimerInfoMemberList, attr_name);
}

static 
int
SoundTimerInfo_tp_setattr(SoundTimerInfoObject *self, 
                          char *attr_name, PyObject *v)
{

    if (v == NULL) 
    {
        PyErr_SetString(PyExc_TypeError, 
                        "can't delete sound_timer_info attributes");
        return -1;
    }

    return PyMember_Set((char *) &(self->sti),
                        SoundTimerInfoMemberList, attr_name, v);
}

/*
 * SoundTimerInfo - Type Descriptor
 */

static 
PyTypeObject SoundTimerInfoType = 
{
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,
    "sound_timer_info",
    sizeof(SoundTimerInfoObject),
    0,

    /* standard methods */
    (destructor)  SoundTimerInfo_tp_dealloc,
    (printfunc)   SoundTimerInfo_tp_print,
    (getattrfunc) SoundTimerInfo_tp_getattr,
    (setattrfunc) SoundTimerInfo_tp_setattr,
    0,
    0,
};

/*
 * SoundTimerInfo - Instance Methods (none)
 */

static 
struct PyMethodDef SoundTimerInfoMethods[] = 
{
    { NULL, NULL }
};


/**************************************************************************
 * 
 * Sequencer - Extension Type
 *
 **************************************************************************/

/*
 * Only one sequencer may exist at a time, this is its file descriptor.
 * It needs to be global so the sequencer buffer macros can have access to
 * it. We keep a copy of it in the Sequencer object so 
 */
static int SequencerFD = -1;

typedef struct
{
    PyObject_HEAD 
} SequencerObject;

staticforward PyTypeObject SequencerType; /* shared type descriptor */
staticforward PyMethodDef  SequencerMethods[];

/*
 * Sequencer - Basic type operations
 */
static 
SequencerObject *
SequencerNew(int fd)
{
    SequencerObject *self;
    
    self = PyObject_NEW(SequencerObject, &SequencerType);
    
    if (self == NULL)
        return NULL;

    SequencerFD = fd;
    return self;
}

static
void
Sequencer_tp_dealloc(SequencerObject *self)
{
    if (SequencerFD != -1)
        close(SequencerFD);

    PyMem_DEL(self);
}

static
int 
Sequencer_tp_print(SequencerObject *self, FILE *fp, int flags)
{
    fprintf(fp, "<Sequencer: fd == %d>\n", SequencerFD);
    return 0;
}

static 
PyObject *
Sequencer_tp_getattr(SequencerObject *self, char *name)
{
    return Py_FindMethod(SequencerMethods, (PyObject *) self, name);
}

/*
 * Sequencer - Type Descriptor
 */

static
PyTypeObject SequencerType = 
{
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,
    "sequencer",
    sizeof(SequencerObject),
    0,

    /* standard methods */
    (destructor)  Sequencer_tp_dealloc, /* tp_dealloc */
    (printfunc)   Sequencer_tp_print,
    (getattrfunc) Sequencer_tp_getattr,
    0,
    0,
    0,
    
};
/*
 * Sequencer - Instance Methods
 */

static
PyObject *
Sequencer_fileno(SequencerObject *self, PyObject *args)
{
    if (! PyArg_ParseTuple(args, ""))
        return NULL;

    return Py_BuildValue("i", SequencerFD);

}

RDINT(SequencerObject, Sequencer_nrsynths, SequencerFD, SNDCTL_SEQ_NRSYNTHS)
RDINT(SequencerObject, Sequencer_nrmidis,  SequencerFD, SNDCTL_SEQ_NRMIDIS)
RDINT(SequencerObject, Sequencer_ctrlrate, SequencerFD, SNDCTL_SEQ_CTRLRATE)


/*
 * Takes either an integer device number or a synth_info.
 * If the argument is an integer, then it returns a new synth_info.
 * If the argument is a synth_info, then it is filled in and its
 * reference count is incremented.
 */
static
PyObject *
Sequencer_synth_info(SequencerObject *self, PyObject *args)
{

    int                device;
    int                is_new_synth_info = 0;
    SynthInfoObject   *synth_info_object;
    struct synth_info *sip;

    if (PyArg_ParseTuple(args, "i", &device))
    {
        /* Allocate a new SynthInfo and fill in device */
        synth_info_object = SynthInfoNew(device);
        if (synth_info_object == NULL)
        {
            PyErr_NoMemory();
            return NULL;
        }
        is_new_synth_info = 1;
        (synth_info_object->si).device = device;
    }
    else
    {
        /* Check for a SynthInfo argument */
        if (! PyArg_ParseTuple(args, "O!", SynthInfoType, &synth_info_object))
            return NULL;
        is_new_synth_info = 0;
    }

    sip = &(synth_info_object->si);
    if (ioctl(SequencerFD, SNDCTL_SYNTH_INFO, sip) == -1)
    {
        PyErr_SetFromErrno(PyExc_IOError);
        if (is_new_synth_info)
            Py_DECREF(synth_info_object);

        return NULL;
    }

    /* If we created a new return value, 
       then it _doesn't_ need a new reference. */
    if (! is_new_synth_info)
        Py_INCREF(synth_info_object);
    
    return (PyObject *) synth_info_object;
}
/*
 * Takes either an integer device number or a synth_info.
 * If the argument is an integer, then it returns a new synth_info.
 * If the argument is a synth_info, then it is filled in and its
 * reference count is incremented.
 */
static
PyObject *
Sequencer_midi_info(SequencerObject *self, PyObject *args)
{

    int                device;
    int                is_new_midi_info = 0;
    MidiInfoObject    *midi_info_object;
    struct midi_info  *mip;

    if (PyArg_ParseTuple(args, "i", &device))
    {
        /* Allocate a new MidiInfo and fill in device */
        midi_info_object = MidiInfoNew(device);
        if (midi_info_object == NULL)
        {
            PyErr_NoMemory();
            return NULL;
        }
        is_new_midi_info = 1;
        (midi_info_object->mi).device = device;
    }
    else
    {
        /* Check for a MidiInfo argument */
        if (! PyArg_ParseTuple(args, "O!", MidiInfoType, &midi_info_object))
            return NULL;
        is_new_midi_info = 0;
    }

    mip = &(midi_info_object->mi);
    if (ioctl(SequencerFD, SNDCTL_MIDI_INFO, mip) == -1)
    {
        PyErr_SetFromErrno(PyExc_IOError);
        if (is_new_midi_info)
            Py_DECREF(midi_info_object);

        return NULL;
    }

    /* If we created a new return value, 
       then it _doesn't_ need a new reference. */
    if (! is_new_midi_info)
        Py_INCREF(midi_info_object);
    
    return (PyObject *) midi_info_object;
}

static
PyObject *
Sequencer_close(SequencerObject *self, PyObject *args)
{
    int result;

    if (! PyArg_ParseTuple(args, ""))
        return NULL;
    
    if (close(SequencerFD) == -1)
    {
        PyErr_SetFromErrno(PyExc_IOError);      
        return NULL;
    }

    SequencerFD = -1;

    Py_INCREF(Py_None);
    return Py_None;

}

/*
 * Sequencer buffer macros required by the OSS API
 * Lets try to make these names static.
 */


SEQ_DEFINEBUF(2048);

void
seqbuf_dump(void)
{
    if (_seqbufptr)
    {
        if (write(SequencerFD, _seqbuf, _seqbufptr) == -1)
        {
            PyErr_SetFromErrno(PyExc_IOError);
            return;
        }
        _seqbufptr = 0;
    }
}

static
PyObject *
Sequencer_dumpbuf(SequencerObject *self, PyObject *args)
{

    if (! PyArg_ParseTuple(args, ""))
        return NULL;

    SEQ_DUMPBUF();
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_wait_time(SequencerObject *self, PyObject *args)
{
    int time;

    if (! PyArg_ParseTuple(args, "i", &time))
        return NULL;

    SEQ_WAIT_TIME(time);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

/* 
 * Accepts either an integer or a single character string for the
 * byte to send.
 */
static
PyObject *
Sequencer_midiout(SequencerObject *self, PyObject *args)
{
    int  midi_dev;
    int  byte;
    char charbyte;

    if (!PyArg_ParseTuple(args, "ii", &midi_dev, &byte))
    {
        PyErr_Clear();
        if (!PyArg_ParseTuple(args, "ic", &midi_dev, &charbyte))
            return NULL;
        
        byte = charbyte;
    }

    SEQ_MIDIOUT(midi_dev, byte);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_set_patch(SequencerObject *self, PyObject *args)
{
    int  dev;
    int  voice;
    char patch;

    if (!PyArg_ParseTuple(args, "iii", &dev, &voice, &patch))
        return NULL;
        
    SEQ_SET_PATCH(dev, voice, patch);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_start_note(SequencerObject *self, PyObject *args)
{
    int dev;
    int voice;
    int note;
    int vel;

    if (!PyArg_ParseTuple(args, "iiii", &dev, &voice, &note, &vel))
    {
        PyErr_Clear();
        if (! PyArg_ParseTuple(args, "iii", &dev, &voice, &note))
            return NULL;
        
        vel = MOD_VELOCITY_DEFAULT;
    }
        
    SEQ_START_NOTE(dev, voice, note, vel);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_stop_note(SequencerObject *self, PyObject *args)
{
    int dev;
    int voice;
    int note;
    int vel;

    if (!PyArg_ParseTuple(args, "iiii", &dev, &voice, &note, &vel))
    {
        PyErr_Clear();
        if (! PyArg_ParseTuple(args, "iii", &dev, &voice, &note))
            return NULL;
        
        vel = MOD_VELOCITY_DEFAULT;
    }
        
    SEQ_STOP_NOTE(dev, voice, note, vel);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_chn_pressure(SequencerObject *self, PyObject *args)
{
    int dev;
    int voice;
    int pressure;

    if (!PyArg_ParseTuple(args, "iii", &dev, &voice, &pressure))
        return NULL;
        
    SEQ_CHN_PRESSURE(dev, voice, pressure);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_panning(SequencerObject *self, PyObject *args)
{
    int dev;
    int voice;
    int pos;

    if (!PyArg_ParseTuple(args, "iii", &dev, &voice, &pos))
        return NULL;
        
    SEQ_PANNING(dev, voice, pos);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_control(SequencerObject *self, PyObject *args)
{
    int dev;
    int voice;
    int controller;
    int value;

    if (!PyArg_ParseTuple(args, "iiii", &dev, &voice, &controller, &value))
        return NULL;
        
    SEQ_CONTROL(dev, voice, controller, value);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_bender_range(SequencerObject *self, PyObject *args)
{
    int dev;
    int voice;
    int value;

    if (!PyArg_ParseTuple(args, "iii", &dev, &voice, &value))
        return NULL;
        
    SEQ_BENDER_RANGE(dev, voice, value);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_pitchbend(SequencerObject *self, PyObject *args)
{
    int dev;
    int voice;
    int value;

    if (!PyArg_ParseTuple(args, "iii", &dev, &voice, &value))
        return NULL;
        
    SEQ_PITCHBEND(dev, voice, value);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_expression(SequencerObject *self, PyObject *args)
{
    int dev;
    int voice;
    int value;

    if (!PyArg_ParseTuple(args, "iii", &dev, &voice, &value))
        return NULL;
        
    SEQ_EXPRESSION(dev, voice, value);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}

static
PyObject *
Sequencer_main_volume(SequencerObject *self, PyObject *args)
{
    int dev;
    int voice;
    int value;

    if (!PyArg_ParseTuple(args, "iii", &dev, &voice, &value))
        return NULL;
        
    SEQ_MAIN_VOLUME(dev, voice, value);
    if (PyErr_Occurred())
        return NULL;
    
    Py_INCREF(Py_None);
    return Py_None;
}


static
struct PyMethodDef SequencerMethods[] = 
{
    { "fileno",           (PyCFunction) Sequencer_fileno,      1},
    { "close",            (PyCFunction) Sequencer_close,       1},
    { "nrsynths",         (PyCFunction) Sequencer_nrsynths,    1},
    { "nrmidis",          (PyCFunction) Sequencer_nrmidis,     1},
    { "synth_info",       (PyCFunction) Sequencer_synth_info,  1},
    { "midi_info",        (PyCFunction) Sequencer_midi_info,   1},
    { "ctrlrate",         (PyCFunction) Sequencer_ctrlrate,    1},
    { "dumpbuf",          (PyCFunction) Sequencer_dumpbuf,     1},
    { "wait_time",        (PyCFunction) Sequencer_wait_time,   1},
    { "midiout",          (PyCFunction) Sequencer_midiout,     1},
    { "set_patch",        (PyCFunction) Sequencer_set_patch,   1},
    { "start_note",       (PyCFunction) Sequencer_start_note,  1},
    { "stop_note",        (PyCFunction) Sequencer_stop_note,   1},
    { "chn_pressure",     (PyCFunction) Sequencer_chn_pressure,1},
    { "panning",          (PyCFunction) Sequencer_panning,     1},
    { "control",          (PyCFunction) Sequencer_control,     1},
    { "bender_range",     (PyCFunction) Sequencer_bender_range,1},
    { "pitchbend",        (PyCFunction) Sequencer_pitchbend,   1},
    { "expression",       (PyCFunction) Sequencer_expression,  1},
    { "main_volume",      (PyCFunction) Sequencer_main_volume, 1},

    { NULL, NULL }
};


/**************************************************************************
 * 
 * Audio - Extension Type
 *
 **************************************************************************/


typedef struct
{
    PyObject_HEAD
    int fd;
} AudioObject;

staticforward PyTypeObject AudioType; /* shared type descriptor */
staticforward PyMethodDef  AudioMethods[];

/*
 * Audio - Basic type operations
 */

static 
AudioObject *
AudioNew(int fd)
{
    AudioObject *self;
    
    self = PyObject_NEW(AudioObject, &AudioType);
    
    if (self == NULL)
        return NULL;

    self->fd = fd;
    return self;
}

static
void
Audio_tp_dealloc(AudioObject *self)
{
    if (self->fd != -1)
        close(self->fd);

    PyMem_DEL(self);
}

static
int 
Audio_tp_print(AudioObject *self, FILE *fp, int flags)
{
    fprintf(fp, "<Audio: fd == %d>\n", self->fd);
    return 0;
}

static 
PyObject *
Audio_tp_getattr(AudioObject *self, char *name)
{
    return Py_FindMethod(AudioMethods, (PyObject *) self, name);
}

/*
 * Audio - Type Descriptor
 */

static
PyTypeObject AudioType = 
{
    /* type header */
    PyObject_HEAD_INIT(&PyType_Type)
    0,
    "audio",
    sizeof(AudioObject),
    0,

    /* standard methods */
    (destructor)  Audio_tp_dealloc, /* tp_dealloc */
    (printfunc)   Audio_tp_print,
    (getattrfunc) Audio_tp_getattr,
    0,
    0,
    0,
    
};

/*
 * Audio - Instance Methods
 */

static
PyObject *
Audio_fileno(AudioObject *self, PyObject *args)
{
    if (! PyArg_ParseTuple(args, ""))
        return NULL;

    return Py_BuildValue("i", self->fd);

}
static
PyObject *
Audio_close(AudioObject *self, PyObject *args)
{
    int result;

    if (! PyArg_ParseTuple(args, ""))
        return NULL;
    
    if (close(self->fd) == -1)
    {
        PyErr_SetFromErrno(PyExc_IOError);      
        return NULL;
    }

    self->fd = -1;

    Py_INCREF(Py_None);
    return Py_None;

}

OP   (AudioObject, Audio_reset,      self->fd, SNDCTL_DSP_RESET)
OP   (AudioObject, Audio_sync,       self->fd, SNDCTL_DSP_SYNC)
OP   (AudioObject, Audio_post,       self->fd, SNDCTL_DSP_POST)

WRRDINT(AudioObject, Audio_format,      self->fd, SNDCTL_DSP_SETFMT)
RDINT  (AudioObject, Audio_get_formats, self->fd, SNDCTL_DSP_GETFMTS)
WRRDINT(AudioObject, Audio_stereo,      self->fd, SNDCTL_DSP_STEREO)

#if 0
    /* Documented in OSS manual but not implemented in my version -twb */
WRRDINT(AudioObject, Audio_channels,    self->fd, SNDCTL_DSP_CHANNELS)
#endif 

WRRDINT(AudioObject, Audio_speed,       self->fd, SNDCTL_DSP_SPEED)

static
PyObject *
Audio_query_format(AudioObject *self, PyObject *args)
{
    int format=AFMT_QUERY;

    if (!PyArg_ParseTuple(args, ""))
        return NULL;

    if (ioctl(self->fd, SNDCTL_DSP_SETFMT, &format) == -1) 
    {                                     
        PyErr_SetFromErrno(PyExc_IOError);
        return NULL;                      
    }

    return Py_BuildValue("i", format);
}

static
PyObject *
Audio_read(AudioObject *self, PyObject *args)
{
    int        bytes_requested;
    int        bytes_read;
    char *     buffer;

    PyObject * rv;

    /*
     * Might consider adding an optional string argument to use as a buffer
     */
    
    if (!PyArg_ParseTuple(args, "i", &bytes_requested))
            return NULL;

    
    if (! (rv = PyString_FromStringAndSize(NULL, bytes_requested)))
        return NULL;
 
    buffer = PyString_AsString(rv);

    bytes_read = read(self->fd, buffer, bytes_requested);

    if (bytes_read == -1)
    {
        Py_DECREF(rv);
        PyErr_SetFromErrno(PyExc_IOError);
        return NULL;
    }
    
    if (bytes_read != bytes_requested)
        if (_PyString_Resize(&rv, bytes_read == -1))
            return NULL;

    return rv;

}

static
PyObject *
Audio_write(AudioObject *self, PyObject *args)
{
    int     buffer_length;
    char *  buffer;
    int     bytes_written;

    if (!PyArg_ParseTuple(args, "s#", &buffer, &buffer_length))
        return NULL;

    bytes_written = write(self->fd, buffer, buffer_length);

    if (bytes_written == -1)
        return NULL;

    return Py_BuildValue("i", bytes_written);

}

static
struct PyMethodDef AudioMethods[] = 
{
    { "write",           (PyCFunction) Audio_write,        1},
    { "read",            (PyCFunction) Audio_read,         1},
    { "fileno",          (PyCFunction) Audio_fileno,       1},
    { "close",           (PyCFunction) Audio_close ,       1},
    { "reset",           (PyCFunction) Audio_reset,        1},
    { "sync",            (PyCFunction) Audio_sync,         1},
    { "post",            (PyCFunction) Audio_post,         1},
    { "format",          (PyCFunction) Audio_format,       1},
    { "query_format",    (PyCFunction) Audio_query_format, 1},
    { "get_formats",     (PyCFunction) Audio_get_formats,  1},
    { "stereo",          (PyCFunction) Audio_stereo,       1},
    { "speed",           (PyCFunction) Audio_stereo,       1},
    { NULL, NULL }
};


/**************************************************************************
 *
 * OSS Module
 *
 **************************************************************************/

/*
 * OSS Module - Methods
 */


/*
 * Module 
 */

static 
PyObject *
Mod_open_mixer(PyObject *self, PyObject *args) /* self not used */
{
    int fd;

    char *filename = MOD_MIXER_FILE_DEFAULT;
    int flags = O_RDWR;
    int mode  = 0;

    if (!PyArg_ParseTuple(args, ""))
    {
        PyErr_Clear();
        if (!PyArg_ParseTuple(args, "s"), &filename)
        {
            PyErr_Clear();
            if (!PyArg_ParseTuple(args, "si", &filename, &flags))
                return NULL;
        }
        
    }

    fd = open(filename, flags, mode);
    if (fd == -1)
    {
        PyErr_SetFromErrno(PyExc_IOError);
        return NULL;
    }

    return (PyObject *) MixerNew(fd);

}

static 
PyObject *
Mod_synth_info(PyObject *self, PyObject *args) /* self not used */
{

    int device = 0;

    if (!PyArg_ParseTuple(args, ""))
    {
        PyErr_Clear();
        if (!PyArg_ParseTuple(args, "i", &device))
            return NULL;
    }

    return (PyObject *) SynthInfoNew(device);

}
static 
PyObject *
Mod_sound_timer_info(PyObject *self, PyObject *args) /* self not used */
{

    int device = 0;

    if (!PyArg_ParseTuple(args, ""))
    {
        PyErr_Clear();
        if (!PyArg_ParseTuple(args, "i", &device))
            return NULL;
    }

    return (PyObject *) SoundTimerInfoNew(device);
}

static 
PyObject *
Mod_midi_info(PyObject *self, PyObject *args) /* self not used */
{

    int device = 0;

    if (!PyArg_ParseTuple(args, ""))
    {
        PyErr_Clear();
        if (!PyArg_ParseTuple(args, "i", &device))
            return NULL;
    }

    return (PyObject *) MidiInfoNew(device);

}

static 
PyObject *
Mod_open_sequencer(PyObject *self, PyObject *args) /* self not used */
{
    int fd;

    char *filename = MOD_SEQUENCER_FILE_DEFAULT;
    int flags = O_RDWR;
    int mode  = 0;

    if (!PyArg_ParseTuple(args, ""))
    {
        PyErr_Clear();
        if (!PyArg_ParseTuple(args, "s"), &filename)
        {
            PyErr_Clear();
            if (!PyArg_ParseTuple(args, "si", &filename, &flags))
                return NULL;
        }
        
    }

    fd = open(filename, flags, mode);
    if (fd == -1)
    {
        PyErr_SetFromErrno(PyExc_IOError);
        return NULL;
    }

    return (PyObject *) SequencerNew(fd);

}

static 
PyObject *
Mod_open_audio(PyObject *self, PyObject *args) /* self not used */
{
    int fd;

    char *filename = MOD_AUDIO_FILE_DEFAULT;
    int flags = O_RDWR;
    int mode  = 0;


    if (!PyArg_ParseTuple(args, ""))
    {
        PyErr_Clear();
        if (!PyArg_ParseTuple(args, "s"), &filename)
        {
            PyErr_Clear();
            if (!PyArg_ParseTuple(args, "si", &filename, &flags))
                return NULL;
        }
        
    }

    fd = open(filename, flags, mode);
    if (fd == -1)
    {
        PyErr_SetFromErrno(PyExc_IOError);
        return NULL;
    }

    return (PyObject *) AudioNew(fd);

}

static 
struct PyMethodDef ModMethods[] =
{
    { "open_mixer",           Mod_open_mixer,           1 },
    { "synth_info",           Mod_synth_info,           1 },
    { "sound_timer_info",     Mod_sound_timer_info,     1 },
    { "midi_info",            Mod_midi_info,            1 },
    { "open_sequencer",       Mod_open_sequencer,       1 },
    { "open_audio",           Mod_open_audio,           1 },

    { NULL, NULL}

};

/*
 * Module -  Attributes
 */

#define MODITEM(name, format) \
static PyObject * Mod_##name;
#include __FILE__
#undef MODITEM


static PyObject * Mod_DEVICE_LABELS;
static PyObject * Mod_DEVICE_NAMES;

static ModInitMixerNames(void)
{
    int i;
    char *(labels[]) = SOUND_DEVICE_LABELS;
    char *(names[])  = SOUND_DEVICE_NAMES;

    Mod_DEVICE_LABELS = PyList_New(0);
    Mod_DEVICE_NAMES  = PyList_New(0);

    for(i = 0; i < SOUND_MIXER_NRDEVICES; i++)
    {
        PyObject *new_label;
        PyObject *new_name;

        new_label = PyString_FromString(labels[i]);
        new_name  = PyString_FromString(names[i]);

        PyList_Append(Mod_DEVICE_LABELS, new_label);
        PyList_Append(Mod_DEVICE_NAMES,  new_name);

        Py_DECREF(new_label);
        Py_DECREF(new_name);
    }

}
 
void 
initoss(void)
{

    PyObject *module;
    PyObject *dict;

    module = Py_InitModule("oss", ModMethods);
    dict   = PyModule_GetDict(module);

#define MODITEM(name, format)                       \
    Mod_##name = Py_BuildValue(format, name);       \
    PyDict_SetItemString(dict, #name, Mod_##name); 
#include __FILE__
#undef MODITEM

    ModInitMixerNames();
    PyDict_SetItemString(dict, "SOUND_DEVICE_LABELS", Mod_DEVICE_LABELS);
    PyDict_SetItemString(dict, "SOUND_DEVICE_NAMES",  Mod_DEVICE_NAMES);

    if (PyErr_Occurred())
        Py_FatalError("can't initialize module oss");

}

#endif
