#define _XOPEN_SOURCE 500

#include "common.h"

#ifdef WITH_X
#include <gnome-xml/parser.h>
#include <gnome.h>
#include <libgnomeui/gnome-window-icon.h>
#include <gdk/gdkprivate.h>
#include <gdk/gdkx.h>
#endif

#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <dirent.h>
#include <cstdlib>
#include <errno.h>

#include <fstream>
#include <string>
using namespace std;

// borrowed from licq (http://www.licq.org/)
#if __GLIBC__ == 2 && __GLIBC_MINOR__ >= 1
#define HAVE_BACKTRACE
#endif

#ifdef HAVE_BACKTRACE
#include <execinfo.h>
#endif
//

#include "plugin.h"
#include "root-portal.h"
#include "module_registry.h"
#include "utils.h"
#include "modules/builtin_plugin_loader.h"
#ifdef WITH_X
#include "rootwindow.h"
#include "drawing_area_registry.h"
#include "tree_builder_xml.h"
#include "tree_saver_xml.h"
#include "config/configuration.h"
#endif

#define DEFAULT_CONFIG CONFIG_DIR "/config.rdf"
#define USER_PLUGIN_DIR CONFIG_DIR "/plugins"
#define SYSTEM_CONFIG "/etc/root-portal.rdf"

#ifdef DEBUG
bool debug = false;
#endif

ofstream dev_null;
bool slave_mode = false;
ofstream output_command_file;
bool with_x = true;

bool ensureDir(const string& dirName)
{
        DIR* dir = opendir(dirName.c_str());
        if (dir == NULL) {
                if (verbose)
                        cout << "Creating directory \"" << dirName << "\"" << endl;
                if (mkdir(dirName.c_str(), 0755) == -1) {
                        cerr << "Error creating directory: " << dirName << endl;
                        return false;
                }
        } else
                closedir(dir);
        return true;
}

string getHomeDir()
{
        return getenv("HOME");
}       

string getBaseDir()
{
        return getHomeDir() + "/" + CONFIG_DIR;
}

/* returns 0 on success, -1 on failure, or the pid of the other process if it is already running */
int RootPortal::getLock(const string& displayName)
{
        fstream lockFile;
        
        string name = getBaseDir();
        if (!ensureDir(name))
                return -1;
        name += "/lock";
        if (displayName.length() > 0)
                name += "." + displayName;

        lockFile.open(name.c_str(), ios::in);
        if (lockFile.is_open()) {
                int otherPid;
                lockFile >> otherPid;
                lockFile.close();
                
                string otherExe = get_exe(otherPid);
                if (otherExe != "") {
                        string thisExe = get_exe(getpid());
                        if (otherExe == thisExe)
                                return otherPid;
                }
        }
        
        lockFile.open(name.c_str(), ios::out);
        if(!lockFile.is_open()) {
                cerr << "Error creating lock file \"" << name << "\"" << endl;
                return -1;
        }
        lockFile << getpid() << endl;
        lockFile.close();
        return 0;
}

void RootPortal::load_settings()
{
        ifstream settings_file;
        settings_file.open((getBaseDir() + "/settings").c_str());
        if (settings_file.is_open()) {
                settings_file >> load_configurator_default;
                settings_file.close();
        } else {
                load_configurator_default = true;
        }
}

void RootPortal::save_settings()
{
        ofstream settings_file;
        settings_file.open((getBaseDir() + "/settings").c_str());
        settings_file << load_configurator_default;
        settings_file.close();
}

void RootPortal::releaseLock()
{
        // TODO: delete file
}       

bool RootPortal::fileExists(const string& name) {
        /* this algorithm uses fopen(2)
           you could use stat(2) also, but you'd
           get a lot of data you don't need */

        ifstream result;

        result.open(name.c_str(), ios::in);

        if(!result.is_open())
                return false;
        result.close();
        return true;
}

void RootPortal::displayWelcomeMessage()
{
	cout << PACKAGE " v" VERSION << endl;
        cout << "Copyright (c) 1999-2000 Michael Lucas-Smith and David Price" << endl
             << "ich@driftwood.draconic.com, mantys@goldweb.com.au" << endl;
}

string RootPortal::getPluginDir()
{
        string ex = get_exe(getpid());
        string tmp;
        for (unsigned int i = 0; i < ex.length(); i++) {
                if (ex[i] == '/') {
                        if (tmp == "bin")
                                return string(ex, 0, i - 4) + "/lib/root-portal";
                        else
                                tmp = "";
                } else
                        tmp += ex[i];
        }
        return "";
}

RootPortal::RootPortal(int argc, char* argv[]) throw (StartupCancelled)
    :
#ifdef WITH_X
    configurator(NULL),
#endif
      remote_configure_request(false)
{
        string homeDir = getHomeDir();

        load_settings();

        /* configuration variables */
        int enableVerbose = -1;
        char *configuration_ = NULL;
        char *output_command_file_name = NULL;
        int help = -1, version = -1;
        int set_slave_mode = -1;
#ifdef WITH_X
        int ldconfigurator = -1, dontLdConfigurator = -1;
        char *display_ = NULL;
#endif
        
#ifdef DEBUG
        int debug_enable = -1;
#endif          

        // we need to check for this now, since gnome does the parsing of most
        // of our options, but we don't want to call it if we aren't using X.
        for (int i = 1; i < argc; i++) {
                if (!strcmp(argv[i], "--no-x")) {
                        with_x = false;
                }
        }

#ifdef WITH_X
        /* this tells popt how to do magic on our variables */
        static const struct poptOption options[] = {
                { "version", 'v', POPT_ARG_NONE, &version, 0,
                  "display version information", NULL },
                { "display", 'd', POPT_ARG_STRING, &display_, 0,
                  "set which display to open", "DISPLAY" },
                { "verbose", 'V', POPT_ARG_NONE, &enableVerbose, 0,
                  "say what we are doing", NULL },
                { "configfile", 'f', POPT_ARG_STRING, &configuration_, 0,
                  "specify the configuration file", "<root-portalrc>" },
                { "configure", 'c', POPT_ARG_NONE, &ldconfigurator, 0,
                  "start with configurator open (default)", NULL },
                { "dont-configure", 'n', POPT_ARG_NONE, &dontLdConfigurator, 0,
                  "start without configurator open", NULL },
                { "slave-mode", 's', POPT_ARG_NONE, &set_slave_mode, 0,
                  "run in slave mode", NULL },
                { "no-x", '\0', POPT_ARG_NONE, NULL, 0,
                  "run without X (usually used with slave mode)", NULL },
                { "output-commands", 'o', POPT_ARG_STRING, &output_command_file_name, 0,
                  "send initialisation commands to a file", "" },
                
#ifdef DEBUG
                { "debug", 'z', POPT_ARG_NONE, &debug_enable, 0,
                  "enable debugging output (in green)", NULL },
#endif          
                { NULL, '\0', 0, NULL, 0}
        };
#endif
        
        if (!with_x) {
                for (int i = 1; i < argc; i++) {
                        if (!strcmp(argv[i], "--help") || !strcmp(argv[i], "-h")) {
                                cout << "-h  --help         display this help message\n"
                                        "-s  --slave-mode   run in slave mode\n"
#ifdef DEBUG
                                        "-z  --debug        enable debugging output (in green)\n"
#endif
                                        "-v  --verbose      say what we are doing\n";
                        }
                        else if (!strcmp(argv[i], "--slave-mode") || !strcmp(argv[i], "-s")) {
                                slave_mode = true;
                        }
#ifdef DEBUG
                        else if (!strcmp(argv[i], "--debug") || !strcmp(argv[i], "-z")) {
                                debug_enable = 1;
                        }
#endif
                        else if (!strcmp(argv[i], "--verbose") || !strcmp(argv[i], "-v")) {
                                enableVerbose = 1;
                        }
                        
                }
        }
        
        string display;
        if (with_x) {
#ifdef WITH_X
                gnome_program_init(PACKAGE, VERSION, LIBGNOMEUI_MODULE,
                                   argc, argv,
                                   GNOME_PARAM_POPT_TABLE, options,
                                   GNOME_PARAM_APP_DATADIR, DATADIR, NULL);

                if (display_ != NULL) {
                        display = display_;
                        free(display_);
                } else {
                        const char* disp = getenv("DISPLAY");
                        if (disp)
                                display = disp;
                }
#else
                cerr << "Root-Portal was configured --without-x, but the --no-x option was not given.\n";
                throw StartupCancelled(-1);
#endif
        }
        
        string configuration;
        if (configuration_ != NULL) {
                configuration = configuration_;
                free(configuration_);
        }

#ifdef DEBUG
        if (debug_enable != -1) {
                debug = true;
                enableVerbose = 1;
                rpdbgmsg("Debugging output enabled");
        }
#endif          

        if (output_command_file_name != NULL) {
                output_command_file.open(output_command_file_name, ios::out);
        }
        
        base_node_plugin_startup();
        initialise_builtin_plugins();
                
                
        if (version != -1) {
                displayWelcomeMessage();
                throw StartupCancelled(0);
        }
        if (help != -1) {
                displayWelcomeMessage();
                cout << "Use root-portal --help to see the full list of options" << endl;
                throw StartupCancelled(0);
        }
        if (set_slave_mode != -1)
                slave_mode = true;

        if (enableVerbose != -1) {
                verbose = 1;
        } else {
                verbose = 0;
        }

        if (with_x) {
                int otherPid = getLock(display);
                if (otherPid > 0) {
                        if (verbose)
                                cout << "Already running -- signaling other"
                                     << " instance to load configurator" << endl;
                        kill(otherPid, SIGUSR2);
                        throw StartupCancelled(0);
                }
        }

#ifdef WITH_X
        setDisplay(display);
#endif
        
        if (configuration == "") {
                string rcfile = homeDir + "/" DEFAULT_CONFIG;
                writeConfig = rcfile;
                if (fileExists(rcfile)) {
                        configuration = rcfile;
                } else if (fileExists(SYSTEM_CONFIG)) {
                        configuration = SYSTEM_CONFIG;
                }
        } else {
                if (verbose)
                        cout << "Using specified configuration file" << endl;
                writeConfig = configuration;
	}

#ifdef TO_PORT
        if (with_x) {
                if (fcntl(ConnectionNumber(GDK_WINDOW_DISPLAY()), F_SETFD, FD_CLOEXEC) < 0) {
                        cerr << "Cannot detach fork." << endl;
                        throw StartupCancelled(-1);
                }
        }
#endif
        
        if (verbose)
                cout << "RP-" VERSION " using " << configuration << endl;

        ensureDir(homeDir + "/" CONFIG_DIR);
        ensureDir(homeDir + "/" CONFIG_DIR "/plugins");
        
        Plugins::instance()->load(homeDir + "/" USER_PLUGIN_DIR);
        string pl1 = getPluginDir();
        if (pl1 != PLUGIN_DIR && pl1 != "")
                Plugins::instance()->load(pl1);
        Plugins::instance()->load(PLUGIN_DIR);
	
        ModuleTree::instance()->init();
        
        if (!slave_mode) {
#ifdef WITH_X
                if (verbose)
                        cout << "reading configuration from \"" << configuration << "\"\n";
                TreeBuilderXml::instance()->load_file(configuration);
#endif
        }
        if (output_command_file.is_open()) {
                output_command_file.close();
        }

#ifdef WITH_X
        if (with_x) {
                configurator = new Configuration();
                bool tmp = load_configurator_default;
                if (dontLdConfigurator != -1)
                        tmp = false;
                if (ldconfigurator != -1)
                        tmp = true;
                if (tmp)
                        configure();
        }
#endif
}

RootPortal::~RootPortal()
{
        initialise_builtin_plugins();                
        base_node_plugin_shutdown();        
}

void RootPortal::service()
{
        ModuleTree::instance()->service();
#ifdef WITH_X
        if (remote_configure_request && configurator != NULL) {
                remote_configure_request = false;
                configurator->displayConfigurator();
        }
#endif
        //	usleep(50); // was 1000, change back before you release
}

void RootPortal::saveConfig()
{
#ifdef WITH_X
        if (!slave_mode)
                TreeSaverXml::instance()->save_to_file(writeConfig);
#endif
}

void RootPortal::preShutdown()
{
        save_settings();
        saveConfig();
        ModuleTree::instance()->shutdown();
}

RootPortal* rootPortal;
int verbose;
bool time_to_shutdown = false;
Plugins* plugins;

void sigHandler(int sigNum)
{
        rpdbgmsg("received signal " << sigNum);
        switch (sigNum) {
                case SIGUSR2:
                        rootPortal->configure();
                        break;
                case SIGSEGV:
                        cerr << "Root-Portal Segmentation Violation Detected.\n";
                
#ifdef CRASH_DEBUG
                        char buffer[50];
                        snprintf(buffer, sizeof(buffer), "program-crash %d", getpid());
                        system(buffer);
#else
#ifdef HAVE_BACKTRACE
                        // borrowed from licq (http://www.licq.org/)
                        cerr << "Backtrace:\n";
                        {
                                void *array[32];
                                int n = backtrace(array, 32);
                                char **res = backtrace_symbols(array, n);
                                for (int i = 0; i < n; i++)
                                        cerr << res[i] << endl;
                        }
#endif
#endif
                        abort();
                default:
                        time_to_shutdown = true;
        }
}
#ifndef WITH_X
typedef int gint;
typedef void *gpointer;
#endif

gint roundnroundnroundwego(gpointer data)
{
#ifdef WITH_X
        handle_x_events();
#endif
        try {
                rootPortal->service();
        } catch (...) {
                cerr << "uncaught exception\n";
        }
#ifdef WITH_X
        if (time_to_shutdown && with_x)
                gtk_main_quit();
#endif
        if (slave_mode) {
                set_blocking(0, false);
                static string text;
                while (true) {
                        char ch;
                        int nread = read(0, &ch, 1);
                        if (nread == 1) {
                                if (ch == '\n') {
                                        if (text == "shutdown") {
                                                time_to_shutdown = true;
                                        } else {
                                                try {
                                                        cout << "R" << ModuleTree::instance()->issue_string_command(text) << endl;
                                                } catch (ModuleError ex) {
                                                        cerr << "Module exception: " << ex.description << endl;
                                                }
                                        }
                                        text = "";
                                } else {
                                        text += ch;
                                }
                        } else {
                                break;
                        }
                }
                set_blocking(0, true);
        }
	return true;
}

int main(int argc, char* argv[])
{
	struct sigaction sig;
	sig.sa_handler = &sigHandler;
        sigemptyset(&sig.sa_mask);
	sig.sa_flags = SA_RESTART;
	sigaction(SIGTERM, &sig, 0);
	sigaction(SIGHUP, &sig, 0);
	sigaction(SIGQUIT, &sig, 0);
	sigaction(SIGINT, &sig, 0);
	sigaction(SIGUSR2, &sig, 0);
	sigaction(SIGSEGV, &sig, 0);

	try {
                dev_null.open("/dev/null", ios::app);

                rootPortal = new RootPortal(argc, argv);

                if (with_x) {
#ifdef WITH_X
                        gtk_timeout_add(15, &roundnroundnroundwego, NULL);
                        rpdbgmsg("entering gtk_main()");
                        gtk_main();
                        rpdbgmsg("exited gtk_main()");
#endif
                } else {
                        while (!time_to_shutdown) {
                                roundnroundnroundwego(NULL);
                                usleep(1000);
                        }
                }

                rootPortal->preShutdown();
                delete rootPortal;

                // destroy singletons
                ModuleTree::destroy_instance();
                Plugins::destroy_instance();
                ModuleRegistry::destroy_instance();
#ifdef WITH_X
                DrawingAreaRegistry::destroy_instance();
#ifdef WITH_XML
                TreeBuilderXml::destroy_instance();
                TreeSaverXml::destroy_instance();
#endif
#endif
                verbose = 0;
                dev_null.close();
                if (output_command_file.is_open())
                        output_command_file.close();

#ifdef WITH_X
                if (with_x)
                        gtk_exit(0);
#endif
                return 0;
        } catch (StartupCancelled ex) {
#ifdef WITH_X
                gtk_exit(ex.returnCode);
#endif
                return ex.returnCode;
        }
}

