/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */


#include <zlib.h>
#include <roy.h>


#define RBUF_GUNZIP_BUFFER 40960
#define RBUF_GZIP_BUFFER 4096


RBuf *
rbuf_gzip (RBuf *buf)
{
    z_stream c_stream;	/* Compression stream. */
    char *buffer;
    int bufsize;
    int err;
    int len = rbuf_len (buf);
    RBuf *newbuf;


    bufsize = RBUF_GZIP_BUFFER * 2;
    buffer = malloc (bufsize); 

    newbuf = rbuf_new ();

    c_stream.zalloc = (alloc_func) 0;
    c_stream.zfree = (free_func) 0;
    c_stream.opaque = (voidpf) 0;

    err = deflateInit (&c_stream, Z_DEFAULT_COMPRESSION);
    if (err != Z_OK) {
        fprintf (stderr, "deflateInit failed!\n");
        goto fail;
    }

    c_stream.next_in = (Bytef *) rbuf_str (buf);
    c_stream.avail_in = len;
    c_stream.next_out = buffer;
    c_stream.avail_out = bufsize;

    /* Compress the stream. */
    while (TRUE) {
        if (c_stream.total_in == (uLong) len)
            break;

        /* This is greedy, so I think it only needs called once... */
        err = deflate (&c_stream, Z_NO_FLUSH);

        if (err < 0) {
            fprintf (stderr, "deflate error: %d!\n", err);
            goto fail;
        }

        rbuf_append_data (newbuf, buffer, bufsize - c_stream.avail_out);
        c_stream.next_out = buffer;
        c_stream.avail_out = bufsize;
    }

    /* Finish the stream. */
    while (TRUE) {
        c_stream.next_out = buffer;
        c_stream.avail_out = bufsize;
        err = deflate (&c_stream, Z_FINISH);

        if (err < 0) {
            fprintf (stderr, "deflate error: %d!\n", err);
            goto fail;
        }

        rbuf_append_data (newbuf, buffer, bufsize - c_stream.avail_out);

        if (Z_STREAM_END == err)
            break;
    }

    deflateEnd (&c_stream);
    free (buffer);

    return (newbuf);

fail:
    free (buffer);
    rbuf_free (newbuf);
    return (rbuf_new_with_str (""));
}


RBuf *
rbuf_gunzip (RBuf *buf)
{ 
    z_stream d_stream;	/* Decompression stream. */
    char *buffer;
    int bufsize;
    int err;
    int len = rbuf_len (buf);
    RBuf *newbuf;


    newbuf = rbuf_new_sized (rbuf_len (buf) * 2);
    bufsize = RBUF_GUNZIP_BUFFER;
    buffer = malloc (bufsize);

    d_stream.zalloc = (alloc_func) 0;
    d_stream.zfree = (free_func) 0;
    d_stream.opaque = (voidpf) 0;

    d_stream.next_in = rbuf_str (buf);
    d_stream.avail_in = rbuf_len (buf);;
    d_stream.next_out = buffer;
    d_stream.avail_out = bufsize;

    err = inflateInit (&d_stream);
    if (err != Z_OK) {
        fprintf (stderr, "inflateInit failed!\n");
        goto fail;
    }

    while (TRUE) {
        if (d_stream.total_in == (uLong) len)
            break;

        d_stream.next_out = buffer;
        d_stream.avail_out = bufsize;

        err = inflate (&d_stream, Z_NO_FLUSH);
        if (err < 0) {
            fprintf (stderr, "deflate error: %d!\n", err);
            goto fail;
        }

        rbuf_append_data (newbuf, buffer, bufsize - d_stream.avail_out);

        if (Z_STREAM_END == err)
            break;
    } 

    free (buffer);
    return (newbuf);

fail:
    free (buffer);
    rbuf_free (newbuf);
    return (rbuf_new_with_str (""));
}

