#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc UpdateAnalysisMenu {} {
    global ds9
    global current
    global analysis
    global menu

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateAnalysisMenu"
    }

    if {$current(frame) != ""} {
	$ds9(mb) entryconfig $menu(analysis) -state normal
	for {set i 0} {$i<$analysis(menu,count)} {incr i} {

	    if {[$current(frame) has fits]} {
		set fn [$current(frame) get fits file name number 1]
	    } else {
		set fn {none}
	    }

	    # disable by default
	    $analysis(menu,$i,parent) entryconfig $analysis(menu,$i,item) \
		-state disabled

	    foreach t $analysis(menu,$i,template) {
		if {[regexp ".$t" $fn]} {
		    $analysis(menu,$i,parent) entryconfig \
			$analysis(menu,$i,item) -state normal
		    break
		}
	    }
	}
    } else {
	$ds9(mb) entryconfig $menu(analysis) -state disabled
    }
}

proc LoadAnalysisDialog {} {
    set filename [OpenFileDialog analysisfbox]
    if {$filename != {}} {
	ProcessAnalysisFile $filename
	# need the bindings
	LayoutFrames
    }
}

proc ClearAnalysisDialog {} {
    if {[tk_messageBox -type okcancel -default cancel \
	     -message "Clear External Analysis Commands?"] != "ok"} {
	return
    }
    ClearAnalysis
}

# Analysis Menu Delete
proc ClearAnalysis {} {
    global ds9
    global analysis
    global canvas

    # is something loaded?

    if {$analysis(menu,count) == 0} {
	return
    }

    # delete cascade menus

    for {set i 0} {$i<$analysis(menu,hmenu,count)} {incr i} {
	destroy [lindex $analysis(menu,hmenu) $i]
    }
    set analysis(menu,hmenu) {}
    set analysis(menu,hmenu,count) 0

    # clear menu

    $ds9(mb).analysis delete 25 end

    for {set i 0} {$i<$analysis(menu,count)} {incr i} {
	unset analysis(menu,$i,parent)
	unset analysis(menu,$i,item)
	unset analysis(menu,$i,template)
	unset analysis(menu,$i,cmd)
	unset analysis(menu,$i,inuse)
	unset analysis(menu,$i,var)
    }
    set analysis(menu,count) 0

    # clear all bindings, first

    foreach f $ds9(frames) {
	for {set i 0} {$i<$analysis(bind,count)} {incr i} {
	    $canvas(name) bind $f "$analysis(bind,$i,item)" {}
	}
    }

    # clear bindings

    for {set i 0} {$i<$analysis(bind,count)} {incr i} {
	unset analysis(bind,$i,item)
	unset analysis(bind,$i,template)
	unset analysis(bind,$i,cmd)
	unset analysis(bind,$i,inuse)
    }
    set analysis(bind,count) 0

    # clear params

    for {set i 0} {$i<$analysis(param,count)} {incr i} {
	for {set j 0} {$j<$analysis(param,$i,count)} {incr j} {
	    unset analysis(param,$i,$j,var)
	    unset analysis(param,$i,$j,type)
	    unset analysis(param,$i,$j,title)
	    unset analysis(param,$i,$j,default)
	    unset analysis(param,$i,$j,last)
	    unset analysis(param,$i,$j,value)
	    unset analysis(param,$i,$j,info)
	}
	unset analysis(param,$i,count)
	unset analysis(param,$i)
    }
    set analysis(param,count) 0
}

proc InitAnalysisFile {} {
    global ds9

    if {[file exists "./$ds9(analysis,file)"]} {
	ProcessAnalysisFile "./$ds9(analysis,file)"
    } elseif {[file exists "~/$ds9(analysis,file)"]} {
	ProcessAnalysisFile "~/$ds9(analysis,file)"
    } elseif {[file exists "./$ds9(analysis,alt)"]} {
	ProcessAnalysisFile "./$ds9(analysis,alt)"
    } elseif {[file exists "~/$ds9(analysis,alt)"]} {
	ProcessAnalysisFile "~/$ds9(analysis,alt)"
    }

    if {[file exists "$ds9(analysis,user)"]} {
	ProcessAnalysisFile "$ds9(analysis,user)"
    }

    if {[file exists "$ds9(analysis,user2)"]} {
	ProcessAnalysisFile "$ds9(analysis,user2)"
    }

    if {[file exists "$ds9(analysis,user3)"]} {
	ProcessAnalysisFile "$ds9(analysis,user3)"
    }

    if {[file exists "$ds9(analysis,user4)"]} {
	ProcessAnalysisFile "$ds9(analysis,user4)"
    }

    UpdateAnalysisMenu
}

proc ProcessAnalysisFile {fn} {
    global ds9
    global analysis
    global canvas
    global message

    if {[file exists "$fn"]} {
	set ch [open $fn r]
	set data [read $ch]
	ProcessAnalysis data
	close $ch
    } else {
	Error "$message(error,analysis,file) $fn"
    }
}

proc ProcessAnalysis {varname} {
    upvar $varname var

    global ds9
    global analysis
    global canvas
    global menu

    set state 1
    set baseparent $ds9(mb).analysis
    set currentparent $baseparent
    set parentstack $baseparent

    $baseparent add separator

    set lines [split $var \n]
    set l [llength $lines]

    for {set ii 0} {$ii<$l} {incr ii} {
	set line [string trim [lindex $lines $ii]]

	# eat empty lines and comments for all except help
	if {$state != 6} {
	    # empty line
	    if {[string length $line] == 0} continue

	    # comments	    
	    if {[string range $line 0 0] == "\#"} continue

	    # strip any end of line comments
	    set id [string first "\#" $line]
	    if {$id > 0} {
		set line [string range $line 0 [expr $id-1]]
	    }
	}

	switch -- $state {
	    1 {
		# param
		if {[lindex $line 0] == "param"} {
		    if {[lindex $line 1] != ""} {
			set analysis(param,$analysis(param,count)) \
			    [lindex $line 1]
			set analysis(param,$analysis(param,count),count) 0
			set state 5
		    }
		    continue
		}

		# help
		if {[lindex $line 0] == "help"} {
		    set id [string first " " $line]
		    if {$id > 0} {
			set item [string range $line [expr $id+1] end]
		    } else {
			set item Help
		    }

		    set i $analysis(menu,count)
		    set analysis(menu,$i,parent) $currentparent
		    set analysis(menu,$i,item) $item
		    set analysis(menu,$i,template) "*"
		    set analysis(menu,$i,cmd) {help}
		    set analysis(menu,$i,inuse) 0
		    set analysis(menu,$i,var) {}
		    $currentparent add command -label $item \
			-command "AnalysisTask $i menu {} 0 0"

		    set state 6
		    continue
		}

		# hmenu
		if {[lindex $line 0] == "hmenu"} {
		    set id [string first " " $line]
		    if {$id > 0} {
			set item [string range $line [expr $id+1] end]
		    } else {
			set item Tasks
		    }

		    # make the menu label unique
		    set nmenu "$currentparent.hmenu$analysis(menu,hmenu,count)"
		    lappend analysis(menu,hmenu) $nmenu
		    incr analysis(menu,hmenu,count)

		    menu $nmenu -tearoff $menu(tearoff) \
			-selectcolor $menu(selectcolor)
		    $currentparent add cascade -label "$item" -menu $nmenu

		    set currentparent $nmenu
		    lappend parentstack $currentparent
		    continue
		}

		# end hmenu
		if {[lindex $line 0] == "endhmenu" ||
		    [lindex $line 0] == "end"} {
		    set parentstack [lreplace $parentstack end end]
		    set currentparent [lindex $parentstack end]
		    continue
		}

		if {[lindex $line 0] == "---"} {
		    $currentparent add separator
		    continue
		}

		# assume new command

		set item "$line"
		set template ""
		set type ""
		set cmd ""
		set state 2
	    }

	    2 {
		set template "$line"
		set state 3
	    }

	    3 {
		set type "$line"
		set state 4
	    }

	    4 {
		set cmd "$line"
		if {$item != "" && $template != "" && 
		    $type != "" && $cmd != ""} {
		    switch -- [lindex $type 0] {
			bind {
			    set b [lindex $type 1]
			    if {$b != ""} {
				set i $analysis(bind,count)
				set analysis(bind,$i,item) "<$b>"
				set analysis(bind,$i,template) "$template"
				set analysis(bind,$i,cmd) "$cmd"
				set analysis(bind,$i,inuse) 0
				incr analysis(bind,count)
			    }
			}
			web {
			    set i $analysis(menu,count)
			    set analysis(menu,$i,parent) $currentparent
			    set analysis(menu,$i,item) $item
			    set analysis(menu,$i,template) "$template"
			    set analysis(menu,$i,cmd) {web}
			    set analysis(menu,$i,inuse) 0
			    set analysis(menu,$i,var) "$cmd"
			    $currentparent add command -label "$item" \
				-command "AnalysisTask $i menu {} 0 0"
			    incr analysis(menu,count)
			}
			default {
			    set i $analysis(menu,count)
			    set analysis(menu,$i,parent) $currentparent
			    set analysis(menu,$i,item) "$item"
			    set analysis(menu,$i,template) "$template"
			    set analysis(menu,$i,cmd) "$cmd"
			    set analysis(menu,$i,inuse) 0
			    set analysis(menu,$i,var) {}
			    $currentparent add check -label "$item" \
				-command "AnalysisTask $i menu {} 0 0" \
				-variable analysis(menu,$i,inuse) \
				-selectcolor green
			    incr analysis(menu,count)
			}
		    }
		}
		set state 1
	    }

	    5 {
		# end param
		if {[lindex $line 0] == "endparam" || 
		    [lindex $line 0] == "end"} {
		    incr analysis(param,count)
		    set state 1
		    continue
		}

		if {[string range $line 0 0] == "@"} {
		    ParseIRAFParam [string range $line 1 end]
		    continue
		}

		set i $analysis(param,count)
		set j $analysis(param,$i,count)
		set analysis(param,$i,$j,var) [lindex $line 0]
		set analysis(param,$i,$j,type) [lindex $line 1]
		set analysis(param,$i,$j,title) [lindex $line 2]

		# default can contain the full menu 'aaa|bbb|ccc'
		set analysis(param,$i,$j,default) [lindex $line 3]
		# set last to first item
		set analysis(param,$i,$j,last) \
		    [lindex [split [lindex $line 3] |] 0]
		# and set value to last
		set analysis(param,$i,$j,value) \
		    $analysis(param,$i,$j,last)

		set analysis(param,$i,$j,info) [lindex $line 4]
		incr analysis(param,$i,count)
	    }

	    6 {
		# end help
		if {[lindex $line 0] == "endhelp" ||
		    [lindex $line 0] == "end"} {
		    incr analysis(menu,count)
		    set state 1
		    continue
		}

		set i $analysis(menu,count)
		append analysis(menu,$i,var) "$line\n"
	    }
	}
    }
}

proc AnalysisTask {i which frame x y} {
    global analysis

    switch -- $analysis($which,$i,cmd) {
	help {
	    AnalysisText "at${which}${i}" $analysis($which,$i,item) \
		$analysis($which,$i,var) insert
	}
	web {AnalysisWebDoit $i $which $frame $x $y}
	default {AnalysisTaskDoit $i $which $frame $x $y}
    }
}

proc AnalysisWebDoit {i which frame x y} {
    global analysis

    set cmd "$analysis($which,$i,var)"

    # do select macro expansion

    # escaped macros
    SetEscapedMacros cmd

    # $xpa_method
    ParseXPAMethodMacro cmd

    # $xpa
    ParseXPAMacro cmd

    # $xdim,$ydim,$bitpix
    ParseXYBitpixMacro cmd

    # $filename[$regions]
    ParseFilenameRegionMacro cmd

    # $filename
    ParseFilenameMacro cmd

    # $regions
    ParseRegionMacro cmd

    # $env
    ParseEnvMacro cmd

    # $x,$y
    ParseXYMacro cmd $frame $x $y

    # escaped macros
    UnsetEscapedMacros cmd

    if {$analysis(log)} {
	SimpleTextDialog acmd "Analysis Commands" 80 20 append bottom "$cmd\n"
    }

    HVAnalysisCmd "at${which}${i}" "$analysis($which,$i,item)" "$cmd"
}

proc AnalysisTaskDoit {i which frame x y} {
    global analysis
    global ds9
    global current

    if {[info exists analysis($which,$i,pid)]} {
	set analysis($which,$i,inuse) 1

	if {[tk_messageBox -type okcancel -default cancel \
		 -message "This analysis task is already running. Do you wish to kill it?"] == "ok"} {
	    eval "exec kill -9 $analysis($which,$i,pid)"
	}
	return
    }

    # don't turn on til task has started
    set analysis($which,$i,inuse) 0

    set analysis($which,$i,start) {}
    set analysis($which,$i,start,fn) {}
    set analysis($which,$i,start,url) {}
    set analysis($which,$i,finish) {}
    set analysis($which,$i,result) {}
    set analysis($which,$i,plot,title) {}
    set analysis($which,$i,plot,xaxis) {}
    set analysis($which,$i,plot,yaxis) {}
    set analysis($which,$i,plot,dim) 2
    set analysis($which,$i,image) {}

    set cmd $analysis($which,$i,cmd)

    # escaped macros
    SetEscapedMacros cmd

    # $data
    ParseDataMacro cmd $which $i

    # $xpa_method
    ParseXPAMethodMacro cmd

    # $xpa
    ParseXPAMacro cmd

    # $xdim,$ydim,$bitpix
    ParseXYBitpixMacro cmd

    # $filename[$regions]
    ParseFilenameRegionMacro cmd

    # $filename
    ParseFilenameMacro cmd

    # $regions
    ParseRegionMacro cmd

    # $env
    ParseEnvMacro cmd

    # $x,$y
    ParseXYMacro cmd $frame $x $y

    # $message
    if {![ParseMessageMacro cmd]} {
	AnalysisTaskEnd $which $i
	return
    }

    # $entry
    if {![ParseEntryMacro cmd]} {
	AnalysisTaskEnd $which $i
	return
    }

    # $param
    if {![ParseParamMacro cmd]} {
	AnalysisTaskEnd $which $i
	return
    }

    # $text
    ParseTextMacro cmd $which $i
    
    # $plot
    ParsePlotMacro cmd $which $i

    # $null
    ParseNullMacro cmd $which $i

    # $url
    ParseURLMacro cmd $which $i

    # $geturl
    # do this next to last
    ParseGetURLMacro cmd $which $i

    # $image
    # do this last
    ParseImageMacro cmd $which $i

    # escaped macros
    UnsetEscapedMacros cmd

    # ok, we are off and running
    set analysis($which,$i,inuse) 1

    if {$analysis($which,$i,start) == "geturl"} {
	AnalysisGetURL $which $i
    } else {
	AnalysisPipe $which $i $cmd
    }
}

proc AnalysisPipe {which i cmd} {
    global analysis
    global current
    global message

    switch -- $analysis($which,$i,start) {
	data {
	    $current(frame) save fits image file $analysis($which,$i,start,fn)
	}
	url {
	    ParseURL $analysis($which,$i,start,url) r
	    switch -- $r(scheme) {
		ftp {AnalysisFTPLoadFile $r(authority) $r(path) \
			 $analysis($which,$i,start,fn)}
		http -
		default \
		    {AnalysisHTTPLoadFile $analysis($which,$i,start,url) \
			 $analysis($which,$i,start,fn)}
	    }
	}
    }

    # last step, change all '][' into ',' so that multiple filters work right
    regsub -all {\]\[} $cmd "," cmd

    if {$analysis(log)} {
	SimpleTextDialog acmd "Analysis Commands" 80 20 append bottom "$cmd\n"
    }

    switch -- $analysis($which,$i,finish) {
	null {
	    global errorInfo
	    # nothing is returned, so there is aways an error,
	    # however, the command will be executed.
	    catch {open "| $cmd"}
	    set errorInfo {}
	    AnalysisTaskEnd $which $i
	    return
	}
	default {
	    if {[catch {set ch [open "| $cmd"]}]} {
		Error "$message(error,analysis,task)"
		AnalysisTaskEnd $which $i
		return
	    }
	    set analysis($which,$i,pid) [pid $ch]

	    switch -- $analysis($which,$i,finish) {
		image {
		    if {$analysis($which,$i,image) == "new"} {
			CreateFrame
		    }
		    global loadParam
		    set loadParam(load,type) channel
		    set loadParam(channel,name) $ch
		    set loadParam(file,type) fits
		    set loadParam(file,mode) {}
		    set loadParam(file,name) \
		"[string tolower [lindex $analysis($which,$i,item) 0]].fits"

		    StartLoad
		    ProcessLoad
		    catch {close $ch}
		    FinishLoad

		    AnalysisTaskEnd $which $i
		}
		default {
		    fileevent $ch readable [list AnalysisReader $ch $which $i]
		    fconfigure $ch -blocking 0 -buffering none
		}
	    }
	}
    }
}

proc AnalysisHTTPLoadFile {url fn} {
    set r {}
    set ch [open $fn w]
    set token [http::geturl $url -channel $ch -binary 1 -headers "[ProxyHTTP]"]
    catch {close $ch}
    upvar #0 $token t
    HTTPLog $token
    set r $t(body)
    http::cleanup $token

    return $r
}

proc AnalysisFTPLoadFile {host path fn} {
    global debug

    set ftp [ftp::Open $host "ftp" "ds9@"]
    if {$ftp > -1} {
	set ftp::VERBOSE $debug(tcl,ftp)
	set "ftp::ftp${ftp}(Output)" FTPLog
	ftp::Type $ftp binary
	ftp::Get $ftp $path $fn
	ftp::Close $ftp
    }
}

proc AnalysisReader {ch which i} {
    global analysis
    global ds9

    if {[eof $ch]} {
	catch {close $ch}
    
	switch -- $analysis($which,$i,finish) {
	    null -
	    image -
	    text {}
	    plot {
		AnalysisPlot "at${which}${i}" \
		    $analysis($which,$i,item) \
		    $analysis($which,$i,plot,title) \
		    $analysis($which,$i,plot,xaxis) \
		    $analysis($which,$i,plot,yaxis) \
		    $analysis($which,$i,plot,dim) \
		    $analysis($which,$i,result)
	    }
	    plotstdin {
		AnalysisPlotStdin "at${which}${i}" $analysis($which,$i,item) \
		    analysis($which,$i,result)
	    }
	    default {puts $analysis($which,$i,result)}
	}

	AnalysisTaskEnd $which $i
	return
    }

    set r [read $ch]
    # for real-time update
    switch -- $analysis($which,$i,finish) {
	text {
	    AnalysisText "at${which}${i}" $analysis($which,$i,item) $r append

	    global debug
	    if {$debug(tcl,idletasks)} {
		puts "AnalysisReader"
	    }
	    update idletasks
	}
    }
    append analysis($which,$i,result) $r
}

proc AnalysisPlotStdin {w wtt resultname} {
    upvar $resultname result

    # check for $ERROR
    set id [string first {$ERROR} $result]
    if {$id >= 0} {
	AnalysisText "${w}e" $wtt \
	    [string range $result [expr $id+1] end] append
	return
    }

    # check for ERROR:
    set id [string first {ERROR:} $result]
    if {$id >= 0} {
	AnalysisText "${w}e" $wtt [string range $result $id end] append
	return
    }

    # check for $BEGINTEXT/$ENDTEXT
    # assume each is followed by a \n, so skip it
    if {[string range $result 0 9] == {$BEGINTEXT}} {
	set eid [string first {$ENDTEXT} $result]
	if {$eid > 0} {
	    AnalysisText "${w}t" $wtt \
		[string range $result 11 [expr $eid-1]] append
	    set result [string range $result [expr $eid+9] end]
	} else {
	    # looks like all text
	    AnalysisText "${w}t" $wtt [string range $result 11 end] append
	    return
	}
    }    

    # now find the title, x axis label, y axis label, and dimension
    set id [string first "\n" $result]
    set tt [string range $result 0 $id]
    set rr [string range $result [expr $id+1] end]

    set l [llength $tt]
    set t [join [lrange $tt 0 [expr $l-4]]]
    set x [lindex $tt [expr $l-3]]
    set y [lindex $tt [expr $l-2]]
    set d [lindex $tt [expr $l-1]]

    if {$d != {} && $rr != {}} {
 	return [AnalysisPlot $w $wtt $t $x $y $d $rr]
    } else {
	Error "Error: [string range $tt 0 40]"
    }
}

proc AnalysisGetURL {which i} {
    global analysis
    global message

    if {![ParseURL $analysis($which,$i,start,url) r]} {
	Error "$message(error,analysis,task)"
	AnalysisTaskEnd $which $i
	return
    }
	
    # format all spaces
    set url {}
    regsub -all { } $analysis($which,$i,start,url) "+" url

    if {$analysis(log)} {
	SimpleTextDialog acmd "Analysis Commands" 80 20 append bottom "$url\n"
    }

    switch -- $analysis($which,$i,finish) {
	text {
	    set analysis($which,$i,result) [AnalysisHTTPRead $url]
	    AnalysisText "at${which}${i}" $analysis($which,$i,item) \
		$analysis($which,$i,result) append
	}

	plot {
	    set analysis($which,$i,result) [AnalysisHTTPRead $url]
	    AnalysisPlot "at${which}${i}" \
		$analysis($which,$i,item) \
		$analysis($which,$i,plot,title) \
		$analysis($which,$i,plot,xaxis) \
		$analysis($which,$i,plot,yaxis) \
		$analysis($which,$i,plot,dim) \
		$analysis($which,$i,result)
	}

	plotstdin {
	    set analysis($which,$i,result) [AnalysisHTTPRead $url]
	    AnalysisPlotStdin "at${which}${i}" $analysis($which,$i,item) \
		analysis($which,$i,result)
	}

	image {
	    AnalysisHTTPLoadFile $url $analysis($which,$i,start,fn)
	    if {![catch {set ch [open "$analysis($which,$i,start,fn)"]}]} {
		if {$analysis($which,$i,image) == "new"} {
		    CreateFrame
		}
		global loadParam
		set loadParam(load,type) channel
		set loadParam(channel,name) $ch
		set loadParam(file,type) fits
		set loadParam(file,mode) {}
		set loadParam(file,name) \
		  "[string tolower [lindex $analysis($which,$i,item) 0]].fits"

		StartLoad
		ProcessLoad
		catch {close $ch}
		FinishLoad
	    } else {
		Error "$message(error,analysis,task)"
	    }
	}

	default {
	    set analysis($which,$i,result) [AnalysisHTTPRead $url]
	    Error $analysis($which,$i,result)
	}
    }
    AnalysisTaskEnd $which $i
}

proc AnalysisHTTPRead {url} {
    global http
    
    set r {}
    catch {
	set token [http::geturl $url -binary 1 -headers "[ProxyHTTP]"]
	upvar #0 $token t
	HTTPLog $token
	set r $t(body)
	http::cleanup $token
    }
    return $r
}

proc AnalysisTaskEnd {which i} {
    global analysis
    global ds9

    set analysis($which,$i,inuse) 0
    if {$analysis($which,$i,start,fn) != {}} {
	if {[file exists $analysis($which,$i,start,fn)]} {
	    catch {file delete -force $analysis($which,$i,start,fn)}
	}
    }

    if {[info exists analysis($which,$i,pid)]} {
	unset analysis($which,$i,pid)
    }
    unset analysis($which,$i,start)
    unset analysis($which,$i,start,fn)
    unset analysis($which,$i,start,url)
    unset analysis($which,$i,finish)
    unset analysis($which,$i,result)
    unset analysis($which,$i,plot,title)
    unset analysis($which,$i,plot,xaxis)
    unset analysis($which,$i,plot,yaxis)
    unset analysis($which,$i,plot,dim)
    unset analysis($which,$i,image)
}

proc SetEscapedMacros {cmdname} {
    upvar $cmdname cmd
    global ds9
    global xpa

    set seq "WaJaWaJaW"
    if {[regexp {\$\$} $cmd]} {
	# fill with tempory sequence
	regsub -all {\$\$} $cmd $seq cmd
    }
}

proc UnsetEscapedMacros {cmdname} {
    upvar $cmdname cmd
    global ds9
    global xpa

    set seq "WaJaWaJaW"
    if {[regexp $seq $cmd]} {
	# reset to $
	regsub -all $seq $cmd {\$} cmd
    }
}

proc ParseDataMacro {cmdname which i} {
    upvar $cmdname cmd
    global ds9
    global analysis

    set exp {\$data.?\|}
    if {[regexp $exp $cmd]} {
	set analysis($which,$i,start) data
	set analysis($which,$i,start,fn) [tmpnam ans ".fits"]

	regsub $exp $cmd "cat $analysis($which,$i,start,fn) |" cmd
    }
}

proc ParseXPAMethodMacro {cmdname} {
    upvar $cmdname cmd
    global ds9
    global xpa

    if {[regexp {\$xpa_method} $cmd]} {
	regsub -all {\$xpa_method} $cmd "[xparec $xpa method]" cmd
    }
}

proc ParseXPAMacro {cmdname} {
    upvar $cmdname cmd
    global ds9

    if {[regexp {\$xpa} $cmd]} {
	regsub -all {\$xpa} $cmd $ds9(title) cmd
    }
}

proc ParseXYBitpixMacro {cmdname} {
    upvar $cmdname cmd
    global current

    if {$current(frame) != ""} {
	if {[regexp {\$xdim} $cmd]} {
	    regsub -all {\$xdim} $cmd [$current(frame) get width] cmd
	}

	if {[regexp {\$ydim} $cmd]} {
	    regsub -all {\$ydim} $cmd [$current(frame) get height] cmd
	}

	if {[regexp {\$bitpix} $cmd]} {
	    regsub -all {\$bitpix} $cmd [$current(frame) get bitpix] cmd
	}
    }
}

proc ParseFilenameRegionMacro {cmdname} {
    upvar $cmdname cmd
    global current

    set exp {(\$filename)\[(\$regions\(([^)]*)\))\]}
    while {[regexp $exp $cmd foo fn reg pp]} {
	set type ds9
	set prop {}
	set sys physical
	set sky fk5
	set format degrees

	# default for mosaics
	if {$current(frame) != ""} {
	    if {[$current(frame) has fits mosaic]} {
		set sys wcs
	    }
	}

	foreach p [split $pp ,] {
	    switch -- $p {
		ds9 -
		ciao -
		saotng -
		saoimage -
		pros -
		xy {set type $p}

		include {append prop {include = yes }}
		exclude {append prop {include = no }}
		source {append prop {source = yes }}
		background {append prop {source = no }}

		image -
		physical -
		detector -
		amplifier
		wcs -
		wcsa -
		wcsb -
		wcsc -
		wcsd -
		wcse -
		wcsf -
		wcsg -
		wcsh -
		wcsi -
		wcsj -
		wcsk -
		wcsl -
		wcsm -
		wcsn -
		wcso -
		wcsp -
		wcsq -
		wcsr -
		wcss -
		wcst -
		wcsu -
		wcsv -
		wcsw -
		wcsx -
		wcsy -
		wcsz {set sys $p}

		fk4 -
		b1950 -
		fk5 -
		j2000 -
		icrs -
		galactic -
		ecliptic {
		    if {"$sys"=="physical"} {
			set sys wcs
		    }
		    set sky $p
		}

		hms {set format sexagesimal}
		sexagesimal -
		degrees {set format $p}
	    }
	}

	SubstFilenameRegion cmd $exp $type $prop $sys $sky $format
    }

    set exp {(\$filename)\[(\$regions)\]}
    while {[regexp $exp $cmd foo fn reg]} {
	set type ds9
	set prop {}
	set sys physical
	set sky fk5
	set format degrees

	# default for mosaics
	if {$current(frame) != ""} {
	    if {[$current(frame) has fits mosaic]} {
		set sys wcs
	    }
	}

	SubstFilenameRegion cmd $exp $type $prop $sys $sky $format
    }
}

proc SubstFilenameRegion {cmdname exp type prop sys sky format} {
    upvar $cmdname cmd
    global current
    global ds9
    global analysis

    set fn [$current(frame) get fits file name full]
    set region \
	[$current(frame) marker list $type $sys $sky $format yes no $prop]
    if {$region != ""} {
	set sub {}
	foreach f $fn {
	    append sub "$f\[$region\] "
	}
    } else {
	set sub $fn
    }

    # substitute
    # ok, we need to check the length
    if {[string length $region] > 256} {
	# since we are writing to a file, 
	# we don't have to worry about quoting
	# special characters
	set fn [tmpnam ans ".reg"]
	incr analysis(param,seq)

	if {![catch {set ch [open "$fn" w]}]} {
	    puts $ch "$sub"
	    close $ch
	}

	regsub $exp $cmd "\@$fn" cmd

    } else {
	CleanFileName sub
        regsub $exp $cmd $sub cmd
    }
}

proc ParseFilenameMacro {cmdname} {
    upvar $cmdname cmd
    global current

    set exp {\$filename\(root\)}
    if {[regexp $exp $cmd]} {
	set sub [join [$current(frame) get fits file name root base]]
	CleanFileName sub
	regsub -all $exp $cmd $sub cmd
    }

    set exp {\$filename}
    if {[regexp $exp $cmd]} {
	set sub [join [$current(frame) get fits file name full]]
	CleanFileName sub
	regsub -all $exp $cmd $sub cmd
    }
}

proc CleanFileName {varname} {
    upvar $varname sub

    # we have to quote {"}, else problems down the road
    regsub -all {\"} $sub {\\"} sub

    # we have to quote {&}, else problems down the road
    regsub -all {\&} $sub {\\&} sub
}

proc ParseRegionMacro {cmdname} {
    global current
    upvar $cmdname cmd

    set exp {\$regions\(([^)]*)\)}
    while {[regexp $exp $cmd foo pp]} {

	set type ds9
	set prop {}
	set sys physical
	set sky fk5
	set format degrees

	# default for mosaics
	if {$current(frame) != ""} {
	    if {[$current(frame) has fits mosaic]} {
		set sys wcs
	    }
	}

	foreach p [split $pp ,] {
	    switch -- $p {
		ds9 -
		ciao -
		saotng -
		saoimage -
		pros -
		xy {set type $p}

		include {append prop {include = yes }}
		exclude {append prop {include = no }}
		source {append prop {source = yes }}
		background {append prop {source = no }}

		image -
		physical -
		detector -
		amplifier -
		wcs -
		wcsa -
		wcsb -
		wcsc -
		wcsd -
		wcse -
		wcsf -
		wcsg -
		wcsh -
		wcsi -
		wcsj -
		wcsk -
		wcsl -
		wcsm -
		wcsn -
		wcso -
		wcsp -
		wcsq -
		wcsr -
		wcss -
		wcst -
		wcsu -
		wcsv -
		wcsw -
		wcsx -
		wcsy -
		wcsz {set sys $p}

		fk4 -
		b1950 -
		fk5 -
		j2000 -
		icrs -
		galactic -
		ecliptic {
		    if {"$sys"=="physical"} {
			set sys wcs
		    }
		    set sky $p
		}

		hms {set format sexagesimal}
		sexagesimal -
		degrees {set format $p}
	    }
	}

	SubstRegion cmd $exp $type $prop $sys $sky $format
    }
    
    # SAOtng format
    set exp {\$((|include|exclude|source|background)_)?regions(_(|degrees|hms|pixels))?}
    while {[regexp  $exp $cmd foo a prop b sys]} {

	# check valid props
	switch -- $prop {
	    include {set prop {include = yes}}
	    exclude {set prop {include = no}}
	    source {set prop {source = yes}}
	    background {set prop {source = no}}
	    default {set prop {}}
	}

	# check valid coordinate systems
	set sky fk5
	switch -- $sys {
	    degrees {set sys wcs; set format degrees}
	    hms {set sys wcs; set format sexagesimal}
	    pixels -
	    default {set sys physical; set format degrees}
	}

	SubstRegion cmd $exp ds9 $prop $sys $sky $format
    }
}

proc SubstRegion {cmdname exp type prop sys sky format} {
    upvar $cmdname cmd
    global current
    global ds9
    global analysis

    # get any regions
    set region \
	[$current(frame) marker list $type $sys $sky $format yes no $prop]

    # substitute
    # ok, we need to check the length
    if {[string length $region] > 256} {
	# since we are writing to a file, we don't have to worry about quoting
	# special characters
        set fn [tmpnam ans ".reg"]
        incr analysis(param,seq)

        if {![catch {set ch [open "$fn" w]}]} {
            puts $ch "$region"
            close $ch
        }

        regsub $exp $cmd "\@$fn" cmd

    } else {
        # we have to quote {"}, else problems down the road
        regsub -all {\"} $region {\\"} region

        # we have to quote {&}, else problems down the road
        regsub -all {\&} $region {\\&} region

        regsub $exp $cmd $region cmd
    }
}

proc ParseEnvMacro {cmdname} {
    upvar $cmdname cmd

    global ds9
    global env

    set exp {\$env\(([^)]*)\)}
    if {[regexp $exp $cmd foo ee]} {
	if [info exist env($ee)] {
	    regsub -all $exp $cmd "$env($ee)" cmd
	} else {
	    regsub -all $exp $cmd {} cmd
	}
    }
}

proc ParseXYMacro {cmdname frame x y} {
    upvar $cmdname cmd

    set exp1 {\$x\(([^)]*)\)}
    set exp2 {\$y\(([^)]*)\)}
    if {[regexp $exp1 $cmd foo pp] && [regexp $exp2 $cmd foo2 pp2]} {
	set sys physical
	set sky fk5
	set format degrees

	foreach p [split $pp ,] {
	    switch -- $p {
		image -
		physical -
		detector -
		amplifier -
		wcs -
		wcsa -
		wcsb -
		wcsc -
		wcsd -
		wcse -
		wcsf -
		wcsg -
		wcsh -
		wcsi -
		wcsj -
		wcsk -
		wcsl -
		wcsm -
		wcsn -
		wcso -
		wcsp -
		wcsq -
		wcsr -
		wcss -
		wcst -
		wcsu -
		wcsv -
		wcsw -
		wcsx -
		wcsy -
		wcsz {set sys $p}

		fk4 -
		fk5 -
		icrs -
		galactic -
		ecliptic {set sky $p; set sys wcs}

		hms {set format sexagesimal}
		sexagesimal -
		degrees {set format $p}
	    }
	}
	
	switch -- $sys {
	    image -
	    physical -
	    detector -
	    amplifier {set coord [$frame get coordinates canvas $x $y $sys]}
	    default {set coord [$frame get coordinates canvas $x $y $sys \
				    $sky $format]}
	}

	regsub -all $exp1 $cmd [lindex $coord 0] cmd
	regsub -all $exp2 $cmd [lindex $coord 1] cmd

        return
    }
	    
    # no args

    set exp1 {\$x}
    set exp2 {\$y}
    if {[regexp $exp1 $cmd foo1] && [regexp $exp2 $cmd foo2]} {
        set coord [$frame get coordinates canvas $x $y physical]

	regsub -all $exp1 $cmd [lindex $coord 0] cmd
	regsub -all $exp2 $cmd [lindex $coord 1] cmd
    }
}

proc ParseMessageMacro {cmdname} {
    upvar $cmdname cmd

    # two args
    set exp {\|?.?\$message\((ok|okcancel|yesno),([^)]*)\).?\|?}
    while {[regexp $exp $cmd foo type message]} {
	regsub $exp $cmd {} cmd
	if {![AnalysisMessage $type $message]} {
	    return 0
	}
    }

    # one args
    set exp {\|?.?\$message\(([^)]*)\).?\|?}
    while {[regexp $exp $cmd foo message]} {
	regsub $exp $cmd {} cmd
	AnalysisMessage ok $message
    }

    return 1
}

proc ParseEntryMacro {cmdname} {
    upvar $cmdname cmd
    
    # one args
    set exp {\|?.?\$entry\(([^)]*)\).?\|?}
    while {[regexp $exp $cmd foo message]} {
	set result {}
	if {![AnalysisEntry $message result]} {
	    return 0
	}
	regsub $exp $cmd $result cmd
    }

    return 1
}

proc ParseParamMacro {cmdname} {
    upvar $cmdname cmd
    global analysis

    set exp {\$param\(([^)]*)\).?;?}
    while {[regexp  $exp $cmd foo param]} {
	regsub $exp $cmd {} cmd
	if {![AnalysisParam cmd $param]} {
	    return 0
	}
    }
    return 1
}

proc ParseTextMacro {cmdname which i} {
    upvar $cmdname cmd
    global analysis

    set exp1 {\|.?\$text}
    set exp2 {\|\&.?\$text}
    if {[regexp $exp1 $cmd]} {
	regsub $exp1 $cmd {} cmd

	set analysis($which,$i,finish) text
    } elseif {[regexp $exp2 $cmd]} {
	regsub $exp2 $cmd { 2>@ stdout} cmd

	set analysis($which,$i,finish) text
    }
}

proc ParseNullMacro {cmdname which i} {
    upvar $cmdname cmd
    global analysis

    set exp {\|.?\$null}
    if {[regexp $exp $cmd]} {
	regsub $exp $cmd {} cmd

	set analysis($which,$i,finish) null
    }
}

proc ParsePlotMacro {cmdname which i} {
    upvar $cmdname cmd
    global analysis

    set exp {\|.?\$plot\(([^,]+),([^,]+),([^,]+),([^)]+)\)}
    if {[regexp $exp $cmd foo \
	     analysis($which,$i,plot,title) \
	     analysis($which,$i,plot,xaxis) \
	     analysis($which,$i,plot,yaxis) \
	     analysis($which,$i,plot,dim)]} {
        regsub $exp $cmd {} cmd

	set analysis($which,$i,finish) plot
    }

    set exp {\|.?\$plot\(stdin\)}
    if {[regexp $exp $cmd]} {
        regsub $exp $cmd {} cmd

	set analysis($which,$i,finish) plotstdin
    }

    set exp {\|.?\$plot}
    if {[regexp $exp $cmd]} {
        regsub $exp $cmd {} cmd

	set analysis($which,$i,finish) plot
    }
}

proc ParseURLMacro {cmdname which i} {
    upvar $cmdname cmd
    global ds9
    global analysis

    set exp {\$url\((.*)\)}
    if {[regexp $exp $cmd foo analysis($which,$i,start,url)]} {
	set analysis($which,$i,start) url
	set analysis($which,$i,start,fn) [tmpnam ans ".fits"]

	regsub $exp $cmd "cat $analysis($which,$i,start,fn) " cmd
    }
}

proc ParseGetURLMacro {cmdname which i} {
    upvar $cmdname cmd
    global ds9
    global analysis

    set exp {\$geturl\((.*)\)}
    if {[regexp $exp $cmd foo analysis($which,$i,start,url)]} {
	set analysis($which,$i,start) geturl
	set analysis($which,$i,start,fn) [tmpnam ans ".fits"]

	regsub $exp $cmd "" cmd
    }
}

proc ParseImageMacro {cmdname which i} {
    upvar $cmdname cmd
    global analysis

    set exp {\|.?\$image\(([^)]*)\)}
    if {[regexp $exp $cmd foo analysis($which,$i,image)]} {
        regsub $exp $cmd {} cmd
	
	set analysis($which,$i,finish) image
    }

    set exp {\|.?\$image}
    if {[regexp $exp $cmd]} {
        regsub $exp $cmd {} cmd

	set analysis($which,$i,finish) image
    }
}

proc AnalysisPlotTool {} {
    global ap

    set tt $ap(tt)
    AnalysisPlot $tt {Plot Tool} {} {} {} 2 {}
}

proc AnalysisText {tt title txt method} {
    if {$txt != {} && $txt != "\n"} {
	SimpleTextDialog $tt $title 80 20 $method bottom $txt
    }
}

proc AnalysisMessage {type message} {
    if {$type == ""} {
	set type ok
    }

    switch -- [tk_messageBox -message $message -type $type] {
	ok {return 1}
	yes {return 1}
	cancel {return 0}
	default {return 0}
    }
}

proc AnalysisEntry {message resultvar} {
    upvar $resultvar result

    return [EntryDialog "Entry" $message 60 result]
}

proc AnalysisPref {varname} {
    upvar $varname var
    global prefs
    global ed

    set w ".anspref"

    set size 60

    set ed(ok) 0
    set ed(value) $var

    DialogCreate $w "Analysis Pref" -borderwidth 2
    frame $w.ans  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ans $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.ans.title -text "Select Analysis File to be loaded at startup:"
    frame $w.ans.a
    entry $w.ans.a.value -textvariable ed(value) -width $size
    button $w.ans.a.browse -text "Browse" -command "AnalysisPrefOpen ed(value)"

    pack $w.ans.a.value -ipadx 2 -ipady 2
    pack $w.ans.a.value $w.ans.a.browse -side left -padx 4 -pady 4
    pack $w.ans.title $w.ans.a -side top -padx 4 -pady 4
    pack $w.ans.title -anchor w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.clear -text "Clear" -command [list set ed(value) {}]
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok $w.buttons.clear $w.buttons.cancel \
	-side left -expand 1 -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.ans.a.value select range 0 end
    DialogWait $w ed(ok) $w.ans.a.value
    DialogDismiss $w

    if {$ed(ok)} {
	set var $ed(value)
    }
    
    set r $ed(ok)
    unset ed
    return $r
}

proc AnalysisPrefOpen {varname} {
    upvar $varname var

    FileLast analysisfbox $var
    set var [OpenFileDialog analysisfbox]
}
