#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# Public Proceduress

proc HV {tt title url font extmenu init} {
    global hv
    global debug
    global ds9
    global menu

    if {$debug(tcl,hv)} {
	puts "HV $tt $title $url"
    }

    set w ".${tt}"

    # see if we already have a window visible
    
    if {[winfo exist $w]} {
	raise $w
    } else {
	set hv($tt,top) ".${tt}"
	set hv($tt,mb) ".${tt}mb"
	set hv($tt,widget) {}
	set hv($tt,frame) new
	set hv($tt,file,mode) { }
	set hv($tt,save) 0
	set hv($tt,title) "$title"

	set hv($tt,active) 0
	set hv($tt,index) 0
	set hv($tt,font,size) $font
	set hv($tt,init) $init

	HVClearAll $tt
	set hv($tt,delete) 0

	# create window

	# Note: we want this window to have its own colormap
	toplevel $w
	wm title $w $title
	wm iconname $w $title
	wm group $w $ds9(top)
	wm protocol $w WM_DELETE_WINDOW "HVDestroyCmd $tt"

	$w configure -menu $hv($tt,mb)

	menu $hv($tt,mb) -tearoff 0 -selectcolor $menu(selectcolor)
	$hv($tt,mb) add cascade -label File -menu $hv($tt,mb).file
	$hv($tt,mb) add cascade -label Edit -menu $hv($tt,mb).edit
	$hv($tt,mb) add cascade -label View -menu $hv($tt,mb).view
	$hv($tt,mb) add cascade -label Frame -menu $hv($tt,mb).frame
	$hv($tt,mb) add cascade -label "Open Fits As" -menu $hv($tt,mb).open

	menu $hv($tt,mb).file -tearoff 0 -selectcolor $menu(selectcolor)
	if {$debug(tcl,hv) || $extmenu} {
	    $hv($tt,mb).file add command -label "Open URL" \
		-command "HVURLDialogCmd $tt"
	    $hv($tt,mb).file add command -label "Open File" \
		-command "HVFileDialogCmd $tt"
	    $hv($tt,mb).file add separator
	    $hv($tt,mb).file add command -label Clear -command "HVClearCmd $tt"
	    $hv($tt,mb).file add command -label "Page Source" \
		-command "HVPageSourceCmd $tt"
	    $hv($tt,mb).file add separator
	}
	$hv($tt,mb).file add command -label Reload -command "HVReloadCmd $tt"
	$hv($tt,mb).file add separator
	$hv($tt,mb).file add command -label Close -command "HVDestroyCmd $tt"

	menu $hv($tt,mb).edit -tearoff 0 -selectcolor $menu(selectcolor)
	$hv($tt,mb).edit add command -label Cut -state disabled
	$hv($tt,mb).edit add command -label Copy -state disabled
	$hv($tt,mb).edit add command -label Paste -state disabled
	$hv($tt,mb).edit add command -label Clear -state disabled

	menu $hv($tt,mb).view -tearoff 0 -selectcolor $menu(selectcolor)
	$hv($tt,mb).view add command -label Back -command "HVBackCmd $tt"
	$hv($tt,mb).view add command -label Forward -command "HVForwardCmd $tt"
	$hv($tt,mb).view add separator
	$hv($tt,mb).view add cascade -label "Text Size" \
	    -menu $hv($tt,mb).view.font
	$hv($tt,mb).view add separator
	$hv($tt,mb).view add command -label Stop -command "HVStopCmd $tt" \
	    -state disabled

	menu $hv($tt,mb).view.font -tearoff 0 -selectcolor $menu(selectcolor)
	$hv($tt,mb).view.font add radiobutton -label Smaller \
	    -variable hv($tt,font,size) -value -2 -command "HVFontCmd $tt"
	$hv($tt,mb).view.font add radiobutton -label Normal \
	    -variable hv($tt,font,size) -value 0 -command "HVFontCmd $tt"
	$hv($tt,mb).view.font add radiobutton -label Larger \
	    -variable hv($tt,font,size) -value 2 -command "HVFontCmd $tt"
	$hv($tt,mb).view.font add radiobutton -label Largest \
	    -variable hv($tt,font,size) -value 4 -command "HVFontCmd $tt"

	menu $hv($tt,mb).frame -tearoff 0 -selectcolor $menu(selectcolor)
	$hv($tt,mb).frame add checkbutton -label "Save FITS on download" \
	    -variable hv($tt,save)
	$hv($tt,mb).frame add separator
	$hv($tt,mb).frame add radiobutton \
	    -label "Create New Frame on Download" \
	    -variable hv($tt,frame) -value new
	$hv($tt,mb).frame add radiobutton \
	    -label "Use Current Frame on Download" \
	    -variable hv($tt,frame) -value current

	menu $hv($tt,mb).open -tearoff 0 -selectcolor $menu(selectcolor)
	$hv($tt,mb).open add radiobutton -label "Fits" \
	    -variable hv($tt,file,mode) -value { }
	$hv($tt,mb).open add separator
	$hv($tt,mb).open add radiobutton -label "RGB Fits Image" \
	    -variable hv($tt,file,mode) -value {rbg image}
	$hv($tt,mb).open add radiobutton -label "RGB Fits Cube" \
	    -variable hv($tt,file,mode) -value {rbg cube}
	$hv($tt,mb).open add separator
	$hv($tt,mb).open add radiobutton -label "Multi Ext Data Cube" \
	    -variable hv($tt,file,mode) -value {data cube}
	$hv($tt,mb).open add separator
	$hv($tt,mb).open add radiobutton -label "Mosaic IRAF" \
	    -variable hv($tt,file,mode) -value {mosaic image iraf}
	$hv($tt,mb).open add radiobutton -label "Mosaic IRAF Segment" \
	    -variable hv($tt,file,mode) -value {mosaic iraf}
	$hv($tt,mb).open add separator
	$hv($tt,mb).open add radiobutton -label "Mosaic WCS" \
	    -variable hv($tt,file,mode) -value {mosaic image wcs}
	$hv($tt,mb).open add radiobutton -label "Mosaic WCS Next" \
	    -variable hv($tt,file,mode) -value {mosaic image next wcs}
	$hv($tt,mb).open add radiobutton -label "Mosaic WCS Segment" \
	    -variable hv($tt,file,mode) -value {mosaic wcs}
	$hv($tt,mb).open add separator
	$hv($tt,mb).open add radiobutton -label "Mosaic Image WFPC2" \
	    -variable hv($tt,file,mode) -value {mosaic image wfpc2}

	image create photo hv${tt}back -data {R0lGODlhDwANAKL/AM///8DAwJD//y/I/y+X/y9n/wAAAAAAACH5BAEAAAEALAAAAAAPAA0AAAM0GLq2/qE0+AqYVFmB6eZFKEoRIAyCaaYCYWxDLM9uYBAxoe/7dA8ug3AoZOg6mRsyuUxmEgA7}
	image create photo hv${tt}forward -data {R0lGODlhDwANAKL/AM///8DAwJD//y/I/y+X/y9n/wAAAAAAACH5BAEAAAEALAAAAAAPAA0AAAM3GLpa/K8YSMuYlBVwV/kgCAhdsAFoig7ktA1wLA9SQdw4DkuB4f8/Ag2TMRB4GYUBmewRm09FAgA7}
	image create photo hv${tt}stop -data {R0lGODlhDQANALP/AP///1Lq81I5Of+EhCEAAHsAAMYAAP+UQv9zCHuMjP8AMf8AKf+MnK1CSv8QIQAAACH5BAEAAAEALAAAAAANAA0AAARWMMjUTC1J6ubOQYdiCBuIIMuiiCT1OWu6Ys05AMPC4ItBGB8dYMdI+RoHR4qY6v1CwlvRcEQ4brndwFAgJAwIRdPIzVTEYiqXJBEU1FQCW5Mg2O0ZSQQAOw==}
	image create photo hv${tt}reload -data {R0lGODlhDAANALP/AP///zk5OVJSUoSEhKWlpcDAwP//1v//xr3erZTOezGcEFKtSimce3NzezkxOQAAACH5BAEAAAUALAAAAAAMAA0AAARRcJBJyRilEMC5AcjQaB1wHMYkCFuXLKDQONsBLIuynEBAGAcJAnYy0AyGBOLENPg4qGUISTMdEIoEg4A6ohK6BND4YyqBqCdyve453vB44BEBADs=}

	image create photo biggray -data {R0lGODdhPAA+APAAALi4uAAAACwAAAAAPAA+AAACQISPqcvtD6OctNqLs968+w+G4kiW5omm6sq27gvH8kzX9o3n+s73/g8MCofEovGITCqXzKbzCY1Kp9Sq9YrNFgsAO}

	frame $w.b -relief groove -borderwidth 2
	pack $w.b -side top -fill x

	button $w.b.back -image hv${tt}back -width 15 -height 13 \
	    -command "HVBackCmd $tt"
	button $w.b.forward -image hv${tt}forward -width 15 -height 13 \
	    -command "HVForwardCmd $tt"
	button $w.b.stop -image hv${tt}stop -width 15 -height 13 \
	    -command "HVStopCmd $tt"
	button $w.b.reload -image hv${tt}reload -width 15 -height 13 \
	    -command "HVReloadCmd $tt"
	pack $w.b.back $w.b.forward $w.b.stop $w.b.reload -side left

	frame $w.h
	grid rowconfigure $w.h 0 -weight 1
	grid columnconfigure $w.h 0 -weight 1
	pack $w.h -side top -fill both -expand true

	set hv($tt,widget) [html $w.h.html \
				-yscrollcommand "$w.h.yscroll set" \
				-xscrollcommand "$w.h.xscroll set" \
				-padx 5 \
				-pady 9 \
				-formcommand "HVFormCB $tt" \
				-imagecommand "HVImageCB $tt" \
				-scriptcommand "HVScriptCB $tt"\
				-appletcommand "HVAppletCB $tt" \
				-framecommand "HVFrameCB $tt" \
				-underlinehyperlinks 1 \
				-bg white \
				-width 640 \
				-height 512 \
				-fontcommand "HVFontCB $tt" \
				-tablerelief raised]

	$hv($tt,widget) token handler {NOSCRIPT} "HVNoScriptCB $tt"
	$hv($tt,widget) token handler {/NOSCRIPT} "HVNoScriptCB $tt"

	scrollbar $w.h.yscroll -orient vertical -command "$w.h.html yview"
	scrollbar $w.h.xscroll -orient horizontal -command "$w.h.html xview"

	grid $w.h.html $w.h.yscroll -sticky news
	grid $w.h.xscroll -stick news

	frame $w.s -relief groove -borderwidth 2
	pack $w.s -side bottom -fill x

	label $w.s.status -text "" -font {helvetica 10} -width 120 -anchor w
	pack $w.s.status -side left

	bind $hv($tt,widget).x <1> "HVLinkBind $tt %x %y"
	bind $hv($tt,widget).x <2> "HVLinkNewBind $tt %x %y"
	bind $hv($tt,widget).x <Motion> "HVMotionBind $tt %x %y" 

	# we have a problem with the html widget. first time thur, some
	# structures are not allocated/initialized. if we first display
	# a blank page, all seems ok
	$hv($tt,widget) clear
	$hv($tt,widget) parse "<html>\n<body>\n<form method=\"get\" action=\"foo\">\n</form>\n</body>\n</html>"

	global debug
	if {$debug(tcl,idletasks)} {
	    puts "HV"
	}
	update idletasks
    }

    if {$url != ""} {
	# no need to resolve
	HVLoadURL $tt $url {}
    }
}

# Commands

proc HVDestroyCmd {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVDestroyCmd"
    }

    if {$hv($tt,active)} {
	if {$debug(tcl,hv)} {
	    puts "HVDestroyCmd aborted-- still active"
	}
	return
    }

    # this sometimes will not cancel pending transactions
    HVCancel $tt

    # clear any pending tmp files
    HVClearTmpFile $tt

    # clear the widge and all images
    HVClearWidget $tt

    # delete all buttons
    image delete hv${tt}back
    image delete hv${tt}forward
    image delete hv${tt}stop
    image delete hv${tt}reload

    # destroy the window and menubar
    destroy $hv($tt,top)
    destroy $hv($tt,mb)

    # stop any refresh
    if {$hv($tt,refresh,id)>0} {
	after cancel $hv($tt,refresh,id)
	set hv($tt,refresh,id) 0
    }

    # unset all hv($tt,*)
    foreach x [array names hv] {
	set f [split $x ,]
	if {[lindex $f 0] == $tt} {
	    unset hv($x)
	}
    }
}

proc HVURLDialogCmd {tt} {
    global hv
    global debug

    set url "$hv($tt,url)"
    if {[EntryDialog "URL" "Enter World Wide Web Location (URL):" 80 url]} {
	if {[string length $url] == 0} {
	    return
	}

	ParseURL $url r
	switch -- $r(scheme) {
	    {} {
		# append 'http://' if needed
		if {[string range $r(path) 0 0] == "/"} {
		    set url "http:/$url"
		} else {
		    set url "http://$url"
		}
		
		if {$debug(tcl,hv)} {
		    puts "HVURLDialogCmd new $url"
		}
	    }
	}

	# clear the base
	$hv($tt,widget) config -base {}

	HVClearIndex $tt 0
	HVClearAll $tt
	# no need to resolve
	HVLoadURL $tt $url {}
    }
}

proc HVFileDialogCmd {tt} {
    global hv
    global debug

    set fn [OpenFileDialog hvhtmlfbox]
    if {"$fn" != ""} {
	# clear the base
	$hv($tt,widget) config -base {}

	HVClearIndex $tt 0
	HVClearAll $tt
	# no need to resolve
	HVLoadURL $tt "$fn" {}
    }
}

proc HVClearCmd {tt} {
    global hv
    global debug

    HVClearAll $tt
    set hv($tt,active) 0

    HVClearWidget $tt

    HVClearStatus $tt
    HVClearIndex $tt 0
    HVUpdateDialog $tt
}

proc HVBackCmd {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVBackCmd index $hv($tt,index)"
    }

    incr hv($tt,index) -1
    if {[info exists hv($tt,index,$hv($tt,index))]} {
	set url [lindex $hv($tt,index,$hv($tt,index)) 0]
	set query [lindex $hv($tt,index,$hv($tt,index)) 1]
	if {$debug(tcl,hv)} {
	    puts "HVBackCmd $hv($tt,index) $url $query"
	}
	# clear the base
	$hv($tt,widget) config -base {}

	# HVGotoHTML will incr the index again
	incr hv($tt,index) -1
	# no need to resolve
	HVLoadURL $tt $url $query
    } else {
	incr hv($tt,index)
    }
}

proc HVForwardCmd {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVForwardCmd $hv($tt,index)"
    }

    incr hv($tt,index)
    if {[info exists hv($tt,index,$hv($tt,index))]} {
	set url [lindex $hv($tt,index,$hv($tt,index)) 0]
	set query [lindex $hv($tt,index,$hv($tt,index)) 1]
	if {$debug(tcl,hv)} {
	    puts "HVForwardCmd $hv($tt,index) $url $query"
	}
	# clear the base
	$hv($tt,widget) config -base {}

	# HVGotoHTML will incr the index again
	incr hv($tt,index) -1
	# no need to resolve
	HVLoadURL $tt $url $query
    } else {
	incr hv($tt,index) -1
    }
}

proc HVReloadCmd {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVReloadCmd"
    }

    # clear the base
    $hv($tt,widget) config -base {}

    # clear previous
    set hv($tt,previous) {}

    # HVGotoHTML will incr the index again
    incr hv($tt,index) -1
    # no need to resolve
    HVLoadURL $tt $hv($tt,url) $hv($tt,query)
}

proc HVStopCmd {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "\n*** HVStopCmd ***\n"
    }

    set hv($tt,previous) {}
    HVClearStatus $tt
    HVCancel $tt
}

proc HVPageSourceCmd {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVPageSourceCmd"
    }

    SimpleTextDialog ${tt}st $hv($tt,url) 80 20 insert top $hv($tt,html)
}

proc HVFontCmd {tt} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVFontCmd"
    }

    HVRefresh $tt
}

proc HVArchUserCmd {tt title url} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVArchUserCmd"
    }

    if {[string length $url] == 0} {
	return
    }

    ParseURL $url r
    switch -- $r(scheme) {
	{} {
	    # append 'http://' if needed
	    if {[string range $r(path) 0 0] == "/"} {
		set url "http:/$url"
	    } else {
		set url "http://$url"
	    }
	    
	    if {$debug(tcl,hv)} {
		puts "HVArchUserCmd new $url"
	    }
	}
    }
    HV $tt $title $url 2 0 {}
}

proc HVAnalysisCmd {tt title url} {
    global hv
    global debug

    if {$debug(tcl,hv)} {
	puts "HVAnalysisCmd $tt $title $url"
    }

    if {[string length $url] == 0} {
	HV $tt "$title" {} 2 1 {}
    } else {
	ParseURL $url r
	switch -- $r(scheme) {
	    {} {
		# append 'http://' if needed
		if {[string range $r(path) 0 0] == "/"} {
		    set url "http:/$url"
		} else {
		    set url "http://$url"
		}
		
		if {$debug(tcl,hv)} {
		    puts "HVURLDialogCmd new $url"
		}
	    }
	}
	HV $tt "$title" $url 2 1 {} 
    }
}

proc ProcessWebCmd {varname iname} {
    global hv
    global debug

    upvar $varname var
    upvar $iname i

    set url [lindex $var $i]
    if {[string length $url] == 0} {
	HV web Web {} 2 1 {}
    } else {
	ParseURL $url r
	switch -- $r(scheme) {
	    {} {
		# append 'http://' if needed
		if {[string range $r(path) 0 0] == "/"} {
		    set url "http:/$url"
		} else {
		    set url "http://$url"
		}
		
		if {$debug(tcl,hv)} {
		    puts "HVURLDialogCmd new $url"
		}
	    }
	}
	HV web Web $url 2 1 {} 
    }
}

# Archive Servers
# Optical

proc HVArchMAST {} {
    global current

    set value {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set value "[lindex $coord 0], [lindex $coord 1]"
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "2 target \{$value\}"
    }

    HV mast MAST http://stdatu.stsci.edu 2 1 $l
}

proc HVArchNSA {} {
    global current

    set ra {}
    set dec {}
    set size {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    set ra [lindex $coord 0]
	    set dec [lindex $coord 1]

	    set s [$current(frame) get fits size wcs arcmin]
	    set size [lindex $s 0]
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 ra \{$ra\}"
	lappend l "1 dec \{$dec\}"
	lappend l "1 width \{$size\}"
    }

    HV nsa NSA http://archive.noao.edu/nsa/ 2 1 $l
}

proc HVArchSSS {} {
    global current

    set ra {}
    set dec {}
    set size {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} [lindex $coord 0] { } ra
	    regsub -all {:} [lindex $coord 1] { } dec

	    set s [$current(frame) get fits size wcs arcmin]
	    set size [lindex $s 0]
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 coords \{$ra $dec\}"
	lappend l "1 size \{$size\}"
    }

    HV sss {SuperCOSMOS Sky Survey} \
	http://www-wfau.roe.ac.uk/sss/pixel.html 2 1 $l
}

# Infrad

proc HVArchIRAS {} {
    global current

    set value {}
    set size {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set value [$current(frame) get cursor wcs fk5 degrees]
	    set size [lindex [$current(frame) get fits size wcs arcsec] 0]
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "2 locstr \{$value\}"
	lappend l "2 subsz \{$size\}"
	lappend l "2 objstr \{$value\}"
	lappend l "2 size \{$size\}"
    }

    HV mass {IRAS} http://irsa.ipac.caltech.edu/ 2 1 $l
}

# High Energy

proc HVArchChandraFTP {} {
    global current

    set ra {}
    set dec {}
    set wid {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    set ra [lindex $coord 0]
	    set dec [lindex $coord 1]

	    set wid [lindex [$current(frame) get fits size wcs degrees] 0]
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 ra \{$ra\}"
	lappend l "1 dec \{$dec\}"
	lappend l "1 wid \{$wid\}"
    }

    HV chandraftp {Chandra FTP} \
	http://cfa-www.harvard.edu/archive/chandra/search 2 1 $l
}

proc HVArchChandra {} {
    global current

    set coord {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set size \
		[expr [lindex [$current(frame) get fits size wcs arcmin] 0]/2.]
	}
    }

    set l {}
    if {[string length $coord] != 0} {
	lappend l "1 lon [lindex $coord 0]"
	lappend l "1 lat [lindex $coord 1]"
	lappend l "1 radius $size"
    }

    HV chandra Chandra \
	http://cda.harvard.edu:9011/chaser/mainEntry.do 2 1 $l
}

proc HVArchRosat {} {
    global current

    set ra {}
    set dec {}
    set cprd {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    set ra [split [lindex $coord 0] :]
	    set dec [split [lindex $coord 1] :]
	    set raa "[lindex $ra 0]h[lindex $ra 1]m[lindex $ra 2]s"
	    set decc "[lindex $dec 0]d[lindex $dec 1]m[lindex $dec 2]s"
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 lon \{$raa\}"
	lappend l "1 lat \{$decc\}"
    }
    lappend l "1 cprd im1 \{photon image 0.1-2.4 keV (fits)\}"

    HV rosat {Rosat All-Sky} \
	http://www.xray.mpe.mpg.de/cgi-bin/rosat/rosat-survey 2 1 $l
}

proc HVArchSkyView {} {
    global current

    set value {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set value "[lindex $coord 0], [lindex $coord 1]"
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 VCOORD \{$value\}"
    }

    HV skyview SkyView http://skyview.gsfc.nasa.gov/cgi-bin/skvbasic.pl 2 1 $l
}

proc HVArchW3Browse {} {
    global current

    set value {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set value "[lindex $coord 0], [lindex $coord 1]"
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "3 Entry \{$value\}"
    }

    HV w3browse W3Browse \
	http://heasarc.gsfc.nasa.gov/db-perl/W3Browse/w3browse.pl 2 1 $l
}

# Radio

proc HVArchNVSS {} {
    global current

    set ra {}
    set dec {}
    set sra 1
    set sdec 1

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} [lindex $coord 0] { } ra
	    regsub -all {:} [lindex $coord 1] { } dec

	    set s [$current(frame) get fits size wcs degrees]
	    set sra [lindex $s 0]
	    set sdec [lindex $s 1]
	    if {$sra > 2} {
		set sra 2
	    }
	    if {$sdec > 2} {
		set sdec 2
	    }
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 RA \{$ra\}"
	lappend l "1 Dec \{$dec\}"
	lappend l "1 Size \{$sra $sdec\}"
    }
    lappend l "1 Type \{image/x-fits\} \{FITS Image\}"

    HV nvss NVSS http://www.cv.nrao.edu/nvss/postage.shtml 2 1 $l
}

proc HVArch4MASS {} {
    global current

    set ra {}
    set dec {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} [lindex $coord 0] { } ra
	    regsub -all {:} [lindex $coord 1] { } dec
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 RA \{$ra\}"
	lappend l "1 Dec \{$dec\}"
    }

    HV mass4 4MASS http://www.cv.nrao.edu/4mass/findFITS.shtml 2 1 $l
}

proc HVArchSIRTF {} {
    global current

    set ra {}
    set dec {}
    set sra 1
    set sdec 1

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} [lindex $coord 0] { } ra
	    regsub -all {:} [lindex $coord 1] { } dec

	    set s [$current(frame) get fits size wcs arcmin]
	    set sra [lindex $s 0]
	    set sdec [lindex $s 1]
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 RA \{$ra\}"
	lappend l "1 Dec \{$dec\}"
	lappend l "1 Size \{$sra $sdec\}"
    }
    lappend l "1 Type \{image/x-fits\} \{FITS Image\}"

    HV sirtf SIRTF http://www.cv.nrao.edu/sirtf_fls/SFpostage.shtml 2 1 $l
}

proc HVArchFirst {} {
    global current

    set value {}
    set size {4.5}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set value [$current(frame) get cursor wcs fk5 sexagesimal]
	    set size [lindex [$current(frame) get fits size wcs arcmin] 0]
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 RA \{$value\}"
	lappend l "1 ImageSize \{$size\}"
	lappend l "1 ImageType \{FITS Image\}"
    }

    HV first First http://third.ucllnl.org/cgi-bin/firstcutout 2 1 $l
}

# Other

proc HVArchDIS {} {
    global current

    set value {}
    set size {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set value "[lindex $coord 0], [lindex $coord 1]"
	    set size [lindex [$current(frame) get fits size wcs degrees] 0]
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 position \{$value\}"
	lappend l "1 delta \{$size\}"
    }

    HV dis DIS http://heasarc.gsfc.nasa.gov/vo/data-inventory.html 2 1 $l
}

proc HVArchNED {} {
    global current

    set ra {}
    set dec {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} [lindex $coord 0] { } ra
	    regsub -all {:} [lindex $coord 1] { } dec
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 lon \{$ra\}"
	lappend l "1 lat \{$dec\}"
    }

    HV ned {NED} http://nedwww.ipac.caltech.edu/ 2 1 $l
}

proc HVArchSIMBADSAO {} {
    HVArchSIMBAD simbadsao {SIMBAD@SAO} http://simbad.harvard.edu/Simbad
}

proc HVArchSIMBADCDS {} {
    HVArchSIMBAD simbadcds {SIMBAD@CDS} http://simbad.u-strasbg.fr/Simbad
}

proc HVArchSIMBAD {tt title url} {
    global current

    set value {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set value [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} $value { } value
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 Ident \{$value\}"
    }

    HV $tt $title $url 2 1 $l
}

proc HVArchADSSAO {} {
    HVArchADS adssao {ADS@SAO} http://adswww.harvard.edu/
}

proc HVArchADSCDS {} {
    HVArchADS adscds {ADS@CDS} http://cdsads.u-strasbg.fr/
}

proc HVArchADS {tt title url} {
    global current

    set value {}
    set size {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set value [$current(frame) get cursor wcs fk5 sexagesimal]
	    set size [lindex [$current(frame) get fits size wcs arcmin] 0]
	    regsub -all {:} $value { } value
	    # limit size to 1 arcmin, otherwise you get too many responces
	    if {$size < .1} {
		append value " : $size"
	    } else {
		append value " : .1"
	    }
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 object \{$value\}"
    }

    HV $tt $title $url 2 1 $l
}

proc HVArchSAOTDC {} {
    global current

    set ra {}
    set dec {}
    set rad {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    set ra [lindex $coord 0]
	    set dec [lindex $coord 1]

	    set s [lindex [$current(frame) get fits size wcs arcsec] 0]
	    set rad [expr $s/2.]
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 ra \{$ra\}"
	lappend l "1 dec \{$dec\}"
	lappend l "1 rad \{$rad\}"
    }

    HV saotdc {SAO Telescope Data Center} \
	http://tdc-www.harvard.edu/archive/ 2 1 $l
}

proc HVArchUserMenu {} {
    global hv
    global ds9
    global debug

    # clear menu
    if {[$ds9(mb).analysis.arch index end]>24} {
	$ds9(mb).analysis.arch delete 25 end
    }

    foreach n {1 2 3 4} {
	if {[string length $hv(archive,url,$n)] > 0} {
	    $ds9(mb).analysis.arch add command -label "$hv(archive,menu,$n)" \
		-command "HVArchUserCmd user$n \{$hv(archive,menu,$n)\} $hv(archive,url,$n)"
	}
    }
}

# Preferences

proc HVArchPref {n} {
    global hv
    global ds9
    global ed

    set w ".hvpref"

    set ed(ok) 0
    set ed(label) $hv(archive,menu,$n)
    set ed(url) $hv(archive,url,$n)

    DialogCreate $w "Archive Preference" -borderwidth 2
    frame $w.ans  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ans $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.ans.title -text "Enter URL to added to Archive Menu:"

    label $w.ans.mtitle -text "Menu:"
    entry $w.ans.mvalue -textvariable ed(label) -width 30

    label $w.ans.utitle -text "URL:"
    entry $w.ans.uvalue -textvariable ed(url) -width 60

    grid rowconfigure $w.ans 0 -pad 4
    grid rowconfigure $w.ans 1 -pad 4
    grid rowconfigure $w.ans 2 -pad 4

    grid $w.ans.title - -padx 4 -sticky w
    grid $w.ans.mtitle $w.ans.mvalue -padx 4 -sticky w
    grid $w.ans.utitle $w.ans.uvalue -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.clear -text "Clear" \
	-command "set ed(label) {}; set ed(url) {}"
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok $w.buttons.clear $w.buttons.cancel \
	-side left -expand 1 -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.ans.mvalue select range 0 end
    DialogWait $w ed(ok) $w.ans.mvalue
    DialogDismiss $w

    if {$ed(ok)} {
	set hv(archive,menu,$n) $ed(label)
	set hv(archive,url,$n) $ed(url)

	HVArchUserMenu
    }
    
    set r $ed(ok)
    unset ed
    return $r
}
