#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc PandaDialog {frame id} {
    global marker
    global ds9

    # see if we already have a header window visible

    set w ".marker$id"
    set mb ".mb$id"

    if [winfo exist $w] {
	raise $w
	return
    }

    # Init common items

    InitMarkerDialog $frame $id
    PropMenuPandaDialog $frame $mb $id
    CoordMenuMarkerDialog $frame $mb $id CoordPandaCB
    CoordMarkerCB $frame $id

    # file menu items

    $mb.file add command -label "Apply" \
	-command "ApplyPandaDialog $frame $id"
    $mb.file add separator
    $mb.file add command -label "Close" \
	-command "ClosePandaDialog $frame $id"

    # panda menu items

    set marker($frame,$id,rcoord) $marker(dialog,dist,system)
    set marker($frame,$id,rlabel) $marker(dialog,dist,system)
    set marker($frame,$id,rformat) $marker(dialog,dist,format)

    DistMenuMarkerDialog $frame $mb $id DistPandaCB radius rcoord rformat
    $mb add cascade -label Radius -menu $mb.radius

    set marker($frame,$id,method) dist
    AnnuliMenuMarkerDialog $frame $mb $id method
    $mb add cascade -label Method -menu $mb.method

    # panda specific callbacks

    $frame marker $id callback edit EditPandaCB $frame
    $frame marker $id callback delete DeletePandaDialog $frame

    # Dialog

    set type [string toupper "[$frame get marker $id type]"]

    toplevel $w -colormap $ds9(main)
    wm title $w $type
    wm iconname $w $type
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW "ClosePandaDialog $frame $id"

    $w configure -menu $mb

    frame $w.ref
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ref -fill x
    pack $w.buttons -fill x -ipadx 4 -ipady 4

    frame $w.ref.left
    frame $w.ref.right
    pack $w.ref.left $w.ref.right -side left -fill y

    # Left

    frame $w.ref.left.top -relief groove -borderwidth 2
    frame $w.ref.left.bottom -relief groove -borderwidth 2
    pack $w.ref.left.top $w.ref.left.bottom -fill both -expand true

    # ID

    label $w.ref.left.top.idTitle -text "Id"
    label $w.ref.left.top.idValue -text "$id"

    # Text

    label $w.ref.left.top.textTitle -text "Text"
    entry $w.ref.left.top.textValue -textvariable marker($frame,$id,text) \
	-width 40

    grid $w.ref.left.top.idTitle $w.ref.left.top.idValue -padx 4 -sticky w
    grid $w.ref.left.top.textTitle $w.ref.left.top.textValue -padx 4 -sticky w

    # Center

    label $w.ref.left.bottom.centerTitle -text "Center"
    entry $w.ref.left.bottom.centerX -textvariable marker($frame,$id,x) \
	-width 13
    entry $w.ref.left.bottom.centerY -textvariable marker($frame,$id,y) \
	-width 13
    label $w.ref.left.bottom.centerCoord -relief groove -width 9 \
	-padx 4 -textvariable marker($frame,$id,system)

    # Radius

    label $w.ref.left.bottom.innerTitle -text "Inner"
    label $w.ref.left.bottom.outerTitle -text "Outer"
    label $w.ref.left.bottom.radiusTitle -text "Radius"
    entry $w.ref.left.bottom.inner -textvariable marker($frame,$id,inner) \
	-width 13
    entry $w.ref.left.bottom.outer -textvariable marker($frame,$id,outer) \
	-width 13
    label $w.ref.left.bottom.radiusformat \
	-textvariable marker($frame,$id,rlabel) \
	-relief groove -width 9 -padx 4
    label $w.ref.left.bottom.annuliTitle -text "Annuli"
    entry $w.ref.left.bottom.annuli -textvariable marker($frame,$id,annuli) \
	-width 13

    # Angles

    label $w.ref.left.bottom.ang1Title -text "Start"
    label $w.ref.left.bottom.ang2Title -text "End"
    label $w.ref.left.bottom.anglesTitle -text "Angles"
    entry $w.ref.left.bottom.ang1 -textvariable marker($frame,$id,ang1) \
	-width 13
    entry $w.ref.left.bottom.ang2 -textvariable marker($frame,$id,ang2) \
	-width 13
    label $w.ref.left.bottom.anglesformat -text "degrees" \
	-relief groove -width 9 -padx 4
    label $w.ref.left.bottom.angnumTitle -text "Number"
    entry $w.ref.left.bottom.angnum -textvariable marker($frame,$id,angnum) \
	-width 13

    grid $w.ref.left.bottom.centerTitle $w.ref.left.bottom.centerX \
	$w.ref.left.bottom.centerY $w.ref.left.bottom.centerCoord \
	-padx 4 -sticky w
    grid x $w.ref.left.bottom.innerTitle $w.ref.left.bottom.outerTitle  \
	-padx 4 -sticky w
    grid $w.ref.left.bottom.radiusTitle $w.ref.left.bottom.inner \
	$w.ref.left.bottom.outer $w.ref.left.bottom.radiusformat \
	-padx 4 -sticky w
    grid $w.ref.left.bottom.annuliTitle $w.ref.left.bottom.annuli \
	-padx 4 -sticky w
    grid x $w.ref.left.bottom.ang1Title $w.ref.left.bottom.ang2Title \
	-padx 4 -sticky w
    grid $w.ref.left.bottom.anglesTitle $w.ref.left.bottom.ang1 \
	$w.ref.left.bottom.ang2 $w.ref.left.bottom.anglesformat \
	-padx 4 -sticky w
    grid $w.ref.left.bottom.angnumTitle $w.ref.left.bottom.angnum \
	-padx 4 -sticky w

    # Right

    frame $w.ref.right.radius -relief groove -borderwidth 2
    frame $w.ref.right.angles -relief groove -borderwidth 2
    pack $w.ref.right.radius $w.ref.right.angles -side left

    # Radius

    frame $w.ref.right.radius.title
    frame $w.ref.right.radius.value
    pack $w.ref.right.radius.title $w.ref.right.radius.value \
	-side top -padx 4 -pady 4

    label $w.ref.right.radius.title.title -text "Radius"
    label $w.ref.right.radius.title.radiusformat \
	-textvariable marker($frame,$id,rlabel) -relief groove -width 9 -padx 4

    grid $w.ref.right.radius.title.title \
	$w.ref.right.radius.title.radiusformat \
	-padx 4 -sticky news

    set txtscr $w.ref.right.radius.value.yscroll
    set marker($frame,$id,annulitxt) $w.ref.right.radius.value.txt
    text $marker($frame,$id,annulitxt) -height 15 -width 15 -wrap none \
	-font {courier 12} -yscrollcommand "$txtscr set"
    scrollbar $txtscr -command [list $marker($frame,$id,annulitxt) yview] \
	-orient vertical

    grid $marker($frame,$id,annulitxt) $txtscr -sticky news

    # Angles

    frame $w.ref.right.angles.title
    frame $w.ref.right.angles.value
    pack $w.ref.right.angles.title $w.ref.right.angles.value \
	-side top -padx 4 -pady 4

    label $w.ref.right.angles.title.title -text "Angles"
    label $w.ref.right.angles.title.anglesformat -text "degrees" \
	-relief groove -width 9 -padx 4

    grid $w.ref.right.angles.title.title \
	$w.ref.right.angles.title.anglesformat \
	-padx 4 -sticky news

    set txtscr $w.ref.right.angles.value.yscroll
    set marker($frame,$id,angtxt) $w.ref.right.angles.value.txt
    text $marker($frame,$id,angtxt) -height 15 -width 15 -wrap none \
	-font {courier 12} -yscrollcommand "$txtscr set"
    scrollbar $txtscr -command [list $marker($frame,$id,angtxt) yview] \
	-orient vertical

    grid $marker($frame,$id,angtxt) $txtscr -sticky news

    # Buttons

    button $w.buttons.apply -text "Apply" \
	-command "ApplyPandaDialog $frame $id"
    button $w.buttons.generate -text "Generate" \
	-command "PandaUpdate $frame $id"
    button $w.buttons.close -text "Close" \
	-command "ClosePandaDialog $frame $id"
    pack $w.buttons.apply $w.buttons.generate $w.buttons.close \
	-side left -padx 10 -expand true

    # some window managers need a hint
    raise $w

    # Init panda variables

    EditPandaCB $frame $id
    DistPandaCB $frame $id
}

proc PropMenuPandaDialog {frame mb id} {
    global menu
    global marker

    menu $mb.properties -tearoff 0 -selectcolor $menu(selectcolor)
    $mb.properties add checkbutton -label "Can Edit" \
	-variable marker($frame,$id,edit) \
	-command "PropertyMarkerDialog $frame $id edit"
    $mb.properties add checkbutton -label "Can Move" \
	-variable marker($frame,$id,move) \
	-command "PropertyMarkerDialog $frame $id move"
    $mb.properties add checkbutton -label "Can Delete" \
	-variable marker($frame,$id,delete) \
	-command "PropertyMarkerDialog $frame $id delete"
    $mb.properties add checkbutton -label "Fixed in Size" \
	-variable marker($frame,$id,fixed) \
	-command "PropertyMarkerDialog $frame $id fixed"
    $mb.properties add separator
    $mb.properties add radiobutton -label Include \
	-variable marker($frame,$id,include) -value 1 \
	-command "PropertyMarkerDialog $frame $id include"
    $mb.properties add radiobutton -label Exclude \
	-variable marker($frame,$id,include) -value 0 \
	-command "PropertyMarkerDialog $frame $id include"
    $mb.properties add separator
    $mb.properties add radiobutton -label Source \
	-variable marker($frame,$id,source) -value 1 \
	-command "PropertyMarkerDialog $frame $id source"
    $mb.properties add radiobutton -label Background \
	-variable marker($frame,$id,source) -value 0 \
	-command "PropertyMarkerDialog $frame $id source"
}

proc ApplyPandaDialog {frame id} {
    EditPandaDialog $frame $id
    ApplyMarkerDialog $frame $id
}

proc ClosePandaDialog {frame id} {
    global marker

    $frame marker $id delete callback edit EditPandaCB
    $frame marker $id delete callback delete DeletePandaDialog
    DeleteMarkerCBs $frame $id

    DeletePandaDialog $frame $id
}

proc DeletePandaDialog {frame id} {
    global marker

    DeleteMarkerDialog $frame $id

    unset marker($frame,$id,inner)
    unset marker($frame,$id,outer)
    unset marker($frame,$id,annuli)
    unset marker($frame,$id,annulitxt)
    unset marker($frame,$id,rcoord)
    unset marker($frame,$id,rformat)
    unset marker($frame,$id,rlabel)
    unset marker($frame,$id,method)
    unset marker($frame,$id,angtxt)
    unset marker($frame,$id,ang1)
    unset marker($frame,$id,ang2)
    unset marker($frame,$id,angnum)
}

proc EditPandaDialog {frame id} {
    global marker

    set levels ""
    regsub -all "\n" "[$marker($frame,$id,annulitxt) get 1.0 end]" " " levels
    # and trim any trailing spaces
    set levels [string trimright $levels " "]

    set angles ""
    regsub -all "\n" "[$marker($frame,$id,angtxt) get 1.0 end]" " " angles
    # and trim any trailing spaces
    set angles [string trimright $angles " "]

    if {($levels != "") && ($angles != "")} {
	$frame marker $id panda "\{$angles\}" "\{$levels\}" \
	    $marker($frame,$id,system) \
	    $marker($frame,$id,rcoord) $marker($frame,$id,rformat)
    }
}

proc EditPandaCB {frame id} {
    global marker

    set t \
	[$frame get marker $id panda radius $marker($frame,$id,rcoord) \
	     $marker($frame,$id,rformat)]

    set last [expr [llength $t]-1]
    set marker($frame,$id,inner) [lindex $t 0]
    set marker($frame,$id,outer) [lindex $t $last]
    set marker($frame,$id,annuli) $last

    $marker($frame,$id,annulitxt) delete 1.0 end
    $marker($frame,$id,annulitxt) insert end "$t"

    set a [$frame get marker $id panda angle $marker($frame,$id,system)]

    set last [expr [llength $a]-1]
    set marker($frame,$id,ang1) [lindex $a 0]
    set marker($frame,$id,ang2) [lindex $a $last]
    set marker($frame,$id,angnum) $last

    $marker($frame,$id,angtxt) delete 1.0 end
    $marker($frame,$id,angtxt) insert end "$a"
}

proc CoordPandaCB {frame id} {
    CoordMarkerCB $frame $id

    MoveMarkerCB $frame $id
    EditPandaCB $frame $id
}

proc DistPandaCB {frame id} {
    global marker
    set mb ".mb$id"

    AdjustDist $frame marker($frame,$id,rcoord)

    set marker($frame,$id,rlabel) $marker($frame,$id,rcoord)
    switch -- $marker($frame,$id,rcoord) {
	image -
	physical -
	amplifier -
	detector {}
	default {
	    if [$frame has wcs $marker($frame,$id,rcoord)] {
		if [$frame has wcs equatorial $marker($frame,$id,rcoord)] {
		    set marker($frame,$id,rlabel) $marker($frame,$id,rformat)
		} else {
		    set name [$frame get wcs name $marker($frame,$id,rcoord)]
		    if {$name != ""} {
			set marker($frame,$id,rlabel) $name
		    }
		}
	    }
	}
    }

    MoveMarkerCB $frame $id
    EditPandaCB $frame $id
}

proc PandaUpdate {frame id} {
    PandaUpdateRadius $frame $id
    PandaUpdateAngles $frame $id
}

proc PandaUpdateRadius {frame id} {
    global marker

    $marker($frame,$id,annulitxt) delete 1.0 end

    set inner $marker($frame,$id,inner)
    set outer $marker($frame,$id,outer)
    set annuli $marker($frame,$id,annuli)

    if {($inner != "") && ($outer != "") && ($annuli != "")} {
	switch -- $marker($frame,$id,method) {
	    dist {
		for {set i 0} {$i<=$annuli} {incr i} {
		    $marker($frame,$id,annulitxt) insert end \
 		    "[expr ((($outer-$inner)/double($annuli))*$i)+$inner]\n"
		}
	    }

	    area {
		set pi 3.14159265358979323846
		set area [expr $pi*(($outer*$outer)-($inner*$inner))/$annuli]

		set r0 $inner
		$marker($frame,$id,annulitxt) insert end "$r0\n"
		for {set i 0} {$i<$annuli} {incr i} {
		    set r1 [expr sqrt(($area+($pi*$r0*$r0))/$pi)]
		    $marker($frame,$id,annulitxt) insert end \
			[format "%.4f\n" $r1]
		    set r0 $r1
		}
	    }
	}
    }
}

proc PandaUpdateAngles {frame id} {
    global marker

    $marker($frame,$id,angtxt) delete 1.0 end

    set ang1 $marker($frame,$id,ang1)
    set ang2 $marker($frame,$id,ang2)
    set angnum $marker($frame,$id,angnum)

    if {($ang1 != "") && ($ang2 != "") && ($angnum != "")} {
	set s1 $ang1
	set s2 $ang2

	if {$s1 < 0} {
	    set s1 [expr $s1+360]
	    set s2 [expr $s2+360]
	}
	if {$s1 > 360} {
	    set s1 [expr $s1-360]
	    set s2 [expr $s2-360]
	}

	if {$s1 >= $s2} {
	    set s2 [expr $s2+360]
	}
	for {set i 0} {$i<=$angnum} {incr i} {
	    set v [expr ((($s2-$s1)/double($angnum))*$i)+$s1]
	    if {$v > 360} {
		set v [expr $v-360]
	    }
	    $marker($frame,$id,angtxt) insert end "$v\n"
	}
    }
}

proc PandaDefaultDialog {} {
    global marker
    global ed

    set w ".panda"

    set ed(ok) 0
    set ed(inner) $marker(panda,inner)
    set ed(outer) $marker(panda,outer)
    set ed(annuli) $marker(panda,annuli)
    set ed(ang1) $marker(panda,ang1)
    set ed(ang2) $marker(panda,ang2)
    set ed(angnum) $marker(panda,angnum)

    DialogCreate $w "Default Panda" -borderwidth 2
    frame $w.ed  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ed $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.ed.innerTitle -text "Inner"
    label $w.ed.outerTitle -text "Outer"
    label $w.ed.radiusTitle -text "Radius"
    entry $w.ed.inner -textvariable ed(inner) -width 10
    entry $w.ed.outer -textvariable ed(outer) -width 10
    label $w.ed.unit -text "image" -relief groove -width 8
    label $w.ed.annuliTitle -text "Annuli"
    entry $w.ed.annuli -textvariable ed(annuli) -width 10
    
    grid x $w.ed.innerTitle $w.ed.outerTitle -padx 4 -sticky w
    grid $w.ed.radiusTitle $w.ed.inner $w.ed.outer $w.ed.unit -padx 4 -sticky w
    grid $w.ed.annuliTitle $w.ed.annuli -padx 4 -sticky w

    label $w.ed.ang1Title -text "Start"
    label $w.ed.ang2Title -text "Stop"
    label $w.ed.angTitle -text "Angles"
    entry $w.ed.ang1 -textvariable ed(ang1) -width 10
    entry $w.ed.ang2 -textvariable ed(ang2) -width 10
    label $w.ed.angunit -text "degrees" -relief groove -width 8
    label $w.ed.angnumTitle -text "Number"
    entry $w.ed.angnum -textvariable ed(angnum) -width 10
    
    grid x $w.ed.ang1Title $w.ed.ang2Title -padx 4 -sticky w
    grid $w.ed.angTitle $w.ed.ang1 $w.ed.ang2 $w.ed.angunit -padx 4 -sticky w
    grid $w.ed.angnumTitle $w.ed.angnum -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.ed.inner select range 0 end
    DialogWait $w ed(ok) $w.ed.inner
    DialogDismiss $w

    if {$ed(ok)} {
	set marker(panda,inner) $ed(inner)
	set marker(panda,outer) $ed(outer)
	set marker(panda,annuli) $ed(annuli)
	set marker(panda,ang1) $ed(ang1)
	set marker(panda,ang2) $ed(ang2)
	set marker(panda,angnum) $ed(angnum)
    }

    unset ed
}
