// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __marker_h__
#define __marker_h__

#include <string.h>

#if __GNUC__ >= 3
#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;
#else
#include <iostream.h>
#include <strstream.h>
#include <iomanip.h>
#endif

#include <tk.h>

#include "vector.h"
#include "callback.h"
#include "list.h"
#include "coord.h"

class FrameBase;
class FitsImage;

class Tag {
  char* tag_;

  Tag* previous_;
  Tag* next_;

 public:
  Tag(const char*);
  Tag(const Tag&);
  Tag& operator=(const Tag&);
  ~Tag();

  const char* tag() {return tag_;}
  void set(const char*);
  Tag* previous() {return previous_;}
  void setPrevious(Tag* t) {previous_ = t;}
  Tag* next() {return next_;}
  void setNext(Tag* t) {next_ = t;}
};

class Marker {
public:
  enum Property {NONE=0, SELECT=1, EDIT=2, MOVE=4, ROTATE=8, DELETE=16, 
		 FIXED=32, INCLUDE=64, SOURCE=128, HIGHLITE=256};
  // Select-- user may select the marker
  // Edit-- user may edit the marker
  // Move-- user may move the marker
  // Rotate-- user may rotate the marker
  // Delete-- user may delete the marker
  // Fixed-- marker is fixed in size (not scaled based on zoom)
  // Include-- marker is 'included' or 'excluded' ie '+' or '-'
  // Source-- marker is a 'source' or 'background' marker

  enum Callback {SELECTCB, UNSELECTCB, 
		 EDITBEGINCB, EDITCB, EDITENDCB, 
		 MOVEBEGINCB, MOVECB, MOVEENDCB, 
		 ROTATEBEGINCB, ROTATECB, ROTATEENDCB, 
		 DELETECB, 
		 TEXTCB, COLORCB, LINEWIDTHCB, PROPERTYCB, FONTCB, UPDATECB};
  enum RenderMode {SRC,XOR};

protected:
  int id;
  char type[16];
  Vector center;                // ref coordinates
  BBox bbox;                    // canvas coordinates
  BBox allBBox;                 // canvas coordinates

  double angle;                 // radians

  Vector* handle;               // canvas coordinates 
  int numHandle;

  int color;
  char* colorName;
  int lineWidth;
  unsigned short properties;
  int selected;

  char* text;
  Tk_Font font;              // Tk font
  char* comment;

  Display* display;
  GC gc;

  int doCB;

  List<Tag> tags;

  List<CallBack> selectCB;
  List<CallBack> unselectCB;
  List<CallBack> moveBeginCB;
  List<CallBack> moveCB;
  List<CallBack> moveEndCB;
  List<CallBack> editBeginCB;
  List<CallBack> editCB;
  List<CallBack> editEndCB;
  List<CallBack> rotateBeginCB;
  List<CallBack> rotateCB;
  List<CallBack> rotateEndCB;
  List<CallBack> deleteCB;
  List<CallBack> textCB;
  List<CallBack> colorCB;
  List<CallBack> lineWidthCB;
  List<CallBack> propertyCB;
  List<CallBack> fontCB;
  List<CallBack> updateCB;

  FrameBase* parent;
  Marker* previous_;
  Marker* next_;

protected:
  virtual void render(Drawable, const Matrix&, double, RenderMode) =0;
  
  Vector* arrow(const Vector&, const Vector&);
  void renderArrow(Drawable, const Vector&, const Vector&);
  void psArrow(const Vector&, const Vector&);

  void setGC(RenderMode);
  virtual void setLineDash();
  void setLineNoDash();
  virtual void psLineDash();
  void psLineNoDash();

  void doCallBack(List<CallBack>*);
  void psColor(int, const char*);
  void calcAllBBox();
  void renderHandles(Drawable);
  void renderText(Drawable, const Matrix&, RenderMode);
  virtual void listProperties(ostream&, char);

  void listProps(ostream&);
  void listProsCoordSystem(ostream&, CoordSystem, SkyFrame);

  void listPre(ostream&, CoordSystem, SkyFrame, FitsImage*);
  void listPre2(ostream&, CoordSystem, SkyFrame, FitsImage*);

  void listPost(ostream&, char);

  void listCiaoPre(ostream&);

  void listSAOtngPre(ostream&, char);
  void listSAOtngPost(ostream&, char);

public:
  Marker(const Marker&);
  Marker(FrameBase* p, const Vector& v, 
	 const char* clr, int w, const char* f, 
	 const char* t, unsigned short prop, const char* c, 
	 const List<Tag>& tag);
  virtual ~Marker();

  virtual Marker* dup() =0;
  void newIdentity();
  virtual void updateBBox() =0;
  virtual void updateCoords(const Matrix&);

  virtual void draw(Drawable);
  virtual void drawXOR();
  virtual void drawMagnifier(const Matrix&);

  virtual void moveTo(const Vector& v);

  virtual void moveBegin();
  virtual void move(const Vector& v);
  virtual void moveEnd();

  virtual void editBegin(int);
  virtual void edit(const Vector& v, int h);
  virtual void editEnd();

  virtual void rotateBegin();
  virtual void rotate(const Vector& v, int h);
  virtual void rotateEnd();

  virtual void ps(int);
  virtual int isIn(const Vector& v) {return bbox.isIn(v);}

  void setAngle(double);
  double getAngle() {return angle;}

  virtual int isPolygon() {return 0;}

  virtual void list(ostream&, CoordSystem, SkyFrame, SkyFormat, char) =0;
  virtual void listCiao(ostream&, CoordSystem, SkyFrame, SkyFormat, char) {}
  virtual void listSAOtng(ostream&,CoordSystem,SkyFrame,SkyFormat,char) {}
  virtual void listSAOimage(ostream&,CoordSystem,SkyFrame,SkyFormat,char) {}
  virtual void listPros(ostream&, CoordSystem, SkyFrame, SkyFormat, char) {}
  virtual void listXY(ostream&, CoordSystem, SkyFrame, SkyFormat, char);

  void listCoordSystem(ostream&, CoordSystem, SkyFrame, FitsImage*);

  const BBox& getBBox() {return bbox;}
  const BBox& getAllBBox() {return allBBox;}
  int isVisible(const BBox&);

  void select();
  void unselect();
  void toggleSelect();
  int isSelected() {return selected;}

  void setColor(const char*);
  const char* getColorName() {return colorName;}

  void setLineWidth(int);
  int getLineWidth() {return lineWidth;}

  void setText(const char*);
  const char* getText() {return text;}
  void setFont(const char*);
  const char* getFont();

  void addTag(const char*);
  void editTag(const char*, const char*);
  void deleteTags();
  void deleteTag(int);
  void deleteTag(const char*);
  const char* getTag();
  const char* getNextTag();
  const char* getTag(int);
  int hasTag(const char*);

  int onHandle(const Vector& v);

  int getId() {return id;}
  const char* getType() {return type;}
  virtual Vector getCenter() {return center;}
  double mapLenFromRef(double, CoordSystem, SkyFormat);

  int canSelect() {return (properties & SELECT) ? 1:0;}
  int canEdit() {return (properties & EDIT) ? 1:0;}
  int canMove() {return (properties & MOVE) ? 1:0;}
  int canRotate() {return (properties & ROTATE) ? 1:0;}
  int canDelete() {return (properties & DELETE) ? 1:0;}
  int canHighlite() {return (properties & HIGHLITE) ? 1:0;}
  int isFixed() {return (properties & FIXED) ? 1:0;}
  void setProperty(unsigned short, int);
  int getProperty(unsigned short);
  unsigned short getProperty() {return properties;}

  int setCallBack(Callback, const char*, const char*);
  int deleteCallBack(Callback, const char*);
  void deleteAllCallBack(Callback);
  void doCallBack(Callback);

  Marker* previous() {return previous_;}
  void setPrevious(Marker* m) {previous_ = m;}
  Marker* next() {return next_;}
  void setNext(Marker* m) {next_ = m;}

  void enableCB() {doCB = 1;}
  void disableCB() {doCB = 0;}
  void clearCB();
};

#endif
