/** @file scim_x11_frontend.cpp
 * implementation of class X11FrontEnd.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_x11_frontend.cpp,v 1.137.2.1 2004/12/30 04:51:53 suzhe Exp $
 *
 */

#define Uses_SCIM_CONFIG_PATH
#define Uses_SCIM_FRONTEND
#define Uses_SCIM_ICONV
#define Uses_SCIM_SOCKET
#define Uses_SCIM_SOCKET_TRANSACTION
#define Uses_SCIM_PANEL
#define Uses_STL_MAP
#define Uses_C_LOCALE
#define Uses_C_STDIO
#define Uses_C_STDLIB

#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <errno.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include "IMdkit/IMdkit.h"
#include "IMdkit/Xi18n.h"

#include "scim_private.h"
#include "scim.h"

#include "scim_x11_ic.h"
#include "scim_x11_frontend.h"

#define scim_module_init x11_LTX_scim_module_init
#define scim_module_exit x11_LTX_scim_module_exit
#define scim_frontend_module_init x11_LTX_scim_frontend_module_init
#define scim_frontend_module_run x11_LTX_scim_frontend_module_run

#define SCIM_CONFIG_FRONTEND_X11_BROKEN_WCHAR    "/FrontEnd/X11/BrokenWchar"
#define SCIM_CONFIG_FRONTEND_X11_DYNAMIC         "/FrontEnd/X11/Dynamic"
#define SCIM_CONFIG_FRONTEND_X11_SERVER_NAME     "/FrontEnd/X11/ServerName"
#define SCIM_CONFIG_FRONTEND_X11_ONTHESPOT       "/FrontEnd/X11/OnTheSpot"
#define SCIM_CONFIG_FRONTEND_X11_PANEL_PROGRAM   "/FrontEnd/X11/PanelProgram"

#define SCIM_KEYBOARD_ICON_FILE            (SCIM_ICONDIR "/keyboard.png")

using namespace scim;

//Local static data
static Pointer <X11FrontEnd> _scim_frontend (0);
static X11IC *_last_focused_ic; //Uses only in factory menu activate cb

static int _argc;
static char **_argv;

//Module Interface
extern "C" {
    void scim_module_init (void)
    {
        SCIM_DEBUG_FRONTEND(1) << "Initializing X11 FrontEnd module...\n";
    }

    void scim_module_exit (void)
    {
        SCIM_DEBUG_FRONTEND(1) << "Exiting X11 FrontEnd module...\n";
        _scim_frontend.reset ();
    }

    void scim_frontend_module_init (const BackEndPointer &backend,
                                    const ConfigPointer &config,
                                    int argc,
                                     char **argv)
    {
        if (_scim_frontend.null ()) {
            SCIM_DEBUG_FRONTEND(1) << "Initializing X11 FrontEnd module (more)...\n";
            _scim_frontend = new X11FrontEnd (backend, config);
            _argc = argc;
            _argv = argv;
        }
    }

    void scim_frontend_module_run (void)
    {
        if (!_scim_frontend.null ()) {
            SCIM_DEBUG_FRONTEND(1) << "Starting X11 FrontEnd module...\n";
            _scim_frontend->init (_argc, _argv);
            _scim_frontend->run ();
        }
    }
}

X11FrontEnd::X11FrontEnd (const BackEndPointer &backend,
                          const ConfigPointer &config,
                          const String& server_name)
    : FrontEndBase (backend),
      m_xims (0),
      m_display (0),
      m_xims_window (0),
      m_server_name (server_name),
      m_panel_socket_address (scim_get_default_panel_socket_address ()),
      m_panel_socket_timeout (scim_get_default_socket_timeout ()),
      m_focus_ic (0),
      m_xims_dynamic (true),
      m_wchar_ucs4_equal (scim_if_wchar_ucs4_equal ()),
      m_broken_wchar (false),
      m_valid_key_mask (0xFFFF),
      m_iconv (String ()),
      m_config (config),
      m_should_exit (false),
      m_old_x_error_handler (0)
{
    if (!_scim_frontend.null () && _scim_frontend != this) {
        throw FrontEndError (
            String ("X11 -- only one frontend can be created!"));
    }

    _scim_frontend = this;

    if (!m_server_name.length ())
        m_server_name = String ("SCIM");
}

X11FrontEnd::~X11FrontEnd ()
{
    if (m_xims) {
        if (validate_ic (m_focus_ic)) {

            socket_prepare_transaction (m_focus_ic);
            focus_out (m_focus_ic->si_id);
            socket_req_turn_off_panel (m_focus_ic);
            socket_send_request ();

            ims_sync_ic (m_focus_ic);
        }

        XSync(m_display, False);
        IMCloseIM (m_xims);
    }

    if (m_display && m_xims_window) {
        XDestroyWindow (m_display, m_xims_window);
        XCloseDisplay (m_display);
    }

    m_panel_socket.close ();

    if (!m_config.null () && m_default_factories.size ()) {
        std::map<String, String>::iterator it;
        for (it = m_default_factories.begin (); it != m_default_factories.end (); ++ it) {
            scim_global_config_write (String (SCIM_GLOBAL_CONFIG_DEFAULT_IMENGINE_FACTORY) + String ("/") + it->first,
                             it->second);
        }
    }
}

void
X11FrontEnd::show_preedit_string (int id)
{
    SCIM_DEBUG_FRONTEND(2) << " Show preedit string, id=" << id << "\n";

    if (is_forward_mode () || m_focus_ic->si_id != id)
        return;

    if (ims_is_preedit_callback_mode (m_focus_ic))
        ims_preedit_callback_start (m_focus_ic);
    else
        socket_req_show_preedit_string (m_focus_ic);
}

void
X11FrontEnd::show_aux_string (int id)
{
    SCIM_DEBUG_FRONTEND(2) << " Show aux string, id=" << id << "\n";

    if (is_forward_mode () || m_focus_ic->si_id != id)
        return;

    socket_req_show_aux_string (m_focus_ic);
}

void
X11FrontEnd::show_lookup_table (int id)
{
    SCIM_DEBUG_FRONTEND(2) << " Show lookup table, id=" << id << "\n";

    if (is_forward_mode () || m_focus_ic->si_id != id)
        return;

    socket_req_show_lookup_table (m_focus_ic);
}

void
X11FrontEnd::hide_preedit_string (int id)
{
    SCIM_DEBUG_FRONTEND(2) << " Hide preedit string, id=" << id << "\n";

    if (!validate_ic (m_focus_ic) || m_focus_ic->si_id != id)
        return;

    if (ims_is_preedit_callback_mode (m_focus_ic))
        ims_preedit_callback_done (m_focus_ic);
    else
        socket_req_hide_preedit_string (m_focus_ic);
}

void
X11FrontEnd::hide_aux_string (int id)
{
    SCIM_DEBUG_FRONTEND(2) << " Hide aux string, id=" << id << "\n";

    if (!validate_ic (m_focus_ic) || m_focus_ic->si_id != id)
        return;

    socket_req_hide_aux_string (m_focus_ic);
}

void
X11FrontEnd::hide_lookup_table (int id)
{
    SCIM_DEBUG_FRONTEND(2) << " Hide lookup table, id=" << id << "\n";

    if (!validate_ic (m_focus_ic) || m_focus_ic->si_id != id)
        return;

    socket_req_hide_lookup_table (m_focus_ic);
}

void
X11FrontEnd::update_preedit_caret (int id, int caret)
{
    SCIM_DEBUG_FRONTEND(2) << " Update preedit caret, id=" << id << " caret=" << caret << "\n";

    if (is_forward_mode () || m_focus_ic->si_id != id)
        return;

    if (ims_is_preedit_callback_mode (m_focus_ic))
        ims_preedit_callback_caret (m_focus_ic, caret);
    else
        socket_req_update_preedit_caret (m_focus_ic, caret);
}

void
X11FrontEnd::update_preedit_string (int id, const WideString & str, const AttributeList & attrs)
{
    SCIM_DEBUG_FRONTEND(2) << " Update preedit string, id=" << id << "\n";

    if (is_forward_mode () || m_focus_ic->si_id != id)
        return;

    if (ims_is_preedit_callback_mode (m_focus_ic))
        ims_preedit_callback_draw (m_focus_ic, str, attrs);
    else
        socket_req_update_preedit_string (m_focus_ic, str, attrs);
}

void
X11FrontEnd::update_aux_string (int id, const WideString & str, const AttributeList & attrs)
{
    SCIM_DEBUG_FRONTEND(2) << " Update aux string, id=" << id << "\n";

    if (is_forward_mode () || m_focus_ic->si_id != id)
        return;

    socket_req_update_aux_string (m_focus_ic, str, attrs);
}

void
X11FrontEnd::update_lookup_table (int id, const LookupTable & table)
{
    SCIM_DEBUG_FRONTEND(2) << " Update lookup table, id=" << id << "\n";

    if (is_forward_mode () || m_focus_ic->si_id != id)
        return;

    socket_req_update_lookup_table (m_focus_ic, table);
}

void
X11FrontEnd::commit_string (int id, const WideString & str)
{
    SCIM_DEBUG_FRONTEND(2) << " Commit string, id=" << id << "\n";

    X11IC *ic = m_ic_manager.find_ic_by_siid (id);

    if (validate_ic (ic))
        ims_commit_string (ic, str);
}

void
X11FrontEnd::forward_key_event (int id, const KeyEvent & key)
{
    SCIM_DEBUG_FRONTEND(2) << " Forward keyevent, id=" << id << "\n";

    X11IC *ic = m_ic_manager.find_ic_by_siid (id);

    if (validate_ic (ic))
        ims_forward_key_event (ic, key);
}

void
X11FrontEnd::register_properties (int id, const PropertyList &properties)
{
    SCIM_DEBUG_FRONTEND(2) << " Register properties, id=" << id << "\n";

    if (is_forward_mode () || m_focus_ic->si_id != id)
        return;

    socket_req_register_properties (m_focus_ic, properties);
}

void
X11FrontEnd::update_property (int id, const Property &property)
{
    SCIM_DEBUG_FRONTEND(2) << " Update property, id=" << id << "\n";

    if (is_forward_mode () || m_focus_ic->si_id != id)
        return;

    socket_req_update_property (m_focus_ic, property);
}

void
X11FrontEnd::init (int argc, char **argv)
{
    String str;

    if (!m_config.null ()) {
        SCIM_DEBUG_FRONTEND (1) << "X11 -- Loading configuration.\n";

        //Read settings.
        m_server_name =
            m_config->read (String (SCIM_CONFIG_FRONTEND_X11_SERVER_NAME),
                            m_server_name);

        scim_string_to_key_list (m_trigger_keys,
            m_config->read (String (SCIM_CONFIG_FRONTEND_KEYS_TRIGGER),
                            String ("Control+space")));

        scim_string_to_key_list (m_next_factory_keys, 
            m_config->read (String (SCIM_CONFIG_FRONTEND_KEYS_NEXT_FACTORY),
                            String ("Control+Alt+Down,Control+Shift_R,Control+Shift_L")));

        scim_string_to_key_list (m_previous_factory_keys, 
            m_config->read (String (SCIM_CONFIG_FRONTEND_KEYS_PREVIOUS_FACTORY),
                            String ("Control+Alt+Up,Shift+Control_R,Shift+Control_L")));

        scim_string_to_key_list (m_show_factory_menu_keys, 
            m_config->read (String (SCIM_CONFIG_FRONTEND_KEYS_SHOW_FACTORY_MENU),
                            String ("Control+Alt+l,Control+Alt+m,Control+Alt+s,Control+Alt+Right")));

        KeyEvent key;
        scim_string_to_key (key,
            m_config->read (String (SCIM_CONFIG_FRONTEND_KEYS_VALID_KEY_MASK),
                            String ("Shift+Control+Alt+Lock")));

        m_valid_key_mask = (key.mask > 0) ? key.mask : 0xFFFF;
        m_valid_key_mask |= SCIM_KEY_ReleaseMask;

        m_broken_wchar =
            m_config->read (String (SCIM_CONFIG_FRONTEND_X11_BROKEN_WCHAR),
                            false);

        m_xims_dynamic =
            m_config->read (String (SCIM_CONFIG_FRONTEND_X11_DYNAMIC),
                            true);

        m_config->signal_connect_reload (slot (this, &X11FrontEnd::reload_config_callback));
    }

    if (!m_server_name.length ())
        m_server_name = String ("SCIM");

    SCIM_DEBUG_FRONTEND (1) << "X11 -- Connecting to panel daemon: " << m_panel_socket_address << ".\n";

    if (!socket_connect_panel (argc, argv))
        throw FrontEndError (String ("X11 -- failed to connect to the panel daemon!"));

    m_fallback_factory = new ComposeKeyFactory ();
    m_fallback_instance = m_fallback_factory->create_instance (String ("UTF-8"), 0);
    m_fallback_instance->signal_connect_commit_string (slot (this, &X11FrontEnd::fallback_commit_string_cb));

    init_ims ();
}

void
X11FrontEnd::run ()
{
    if (!m_display || !m_xims_window || !m_xims || !m_panel_socket.is_connected ()) {
        SCIM_DEBUG_FRONTEND(1) << "X11 -- Cannot run without initialization!\n";
        return;
    }

    XEvent event;

    fd_set read_fds, active_fds;

    int panel_fd = m_panel_socket.get_id ();
    int xserver_fd = ConnectionNumber (m_display);
    int max_fd = (panel_fd > xserver_fd) ? panel_fd : xserver_fd;

    FD_ZERO (&active_fds);
    FD_SET (panel_fd, &active_fds);
    FD_SET (xserver_fd, &active_fds);

    // Process the events which are already send to me from the X Server.
    while (XPending (m_display)) {
        XNextEvent (m_display, &event);
        XFilterEvent (&event, None);
    }

    m_should_exit = false;

    // Select between the X Server and the Panel GUI.
    while (!m_should_exit) {
        read_fds = active_fds;

        if (select (max_fd + 1, &read_fds, NULL, NULL, NULL) < 0) {
            SCIM_DEBUG_FRONTEND(1) << "X11 -- Error when watching events!\n";
            return;
        }

        if (FD_ISSET (panel_fd, &read_fds)) {
            Socket panel_socket (panel_fd);
            if (!check_socket_connection (panel_socket)) {
                if (!socket_connect_panel (_argc, _argv)) {
                    SCIM_DEBUG_FRONTEND(1) << "X11 -- Lost connection with panel daemon!\n";
                    return;
                }

                panel_fd = m_panel_socket.get_id ();
                max_fd = (panel_fd > xserver_fd) ? panel_fd : xserver_fd;

                FD_ZERO (&active_fds);
                FD_SET (panel_fd, &active_fds);
                FD_SET (xserver_fd, &active_fds);
            } else {
                socket_receive_reply ();
            }
        }

        if (FD_ISSET (xserver_fd, &read_fds)) {
            while (XPending (m_display)) {
                XNextEvent (m_display, &event);
                XFilterEvent (&event, None);
            }
        }
    }
}

String
X11FrontEnd::get_factory (const String &locale)
{
    std::map<String, String>::iterator it
        = m_default_factories.find (scim_get_locale_language (locale));

    if (it != m_default_factories.end ())
        return it->second;

    std::vector<String> uuids;

    if (get_factory_list (uuids, scim_get_locale_encoding (locale))) {
        String def_uuid;

        def_uuid = scim_global_config_read (String (SCIM_GLOBAL_CONFIG_DEFAULT_IMENGINE_FACTORY) +
                                            String ("/") + scim_get_locale_language (locale),
                                            String (""));

        if (std::find (uuids.begin (), uuids.end (), def_uuid) == uuids.end ())
            def_uuid = uuids [0];

        set_factory (locale, def_uuid);

        return def_uuid;
    }

    char buf [128];
    snprintf (buf, 127, "X11 -- No IMEngine Factory for locale %s!", locale.c_str ());
    throw FrontEndError (buf);
}

void
X11FrontEnd::set_factory (const String &locale, const String &sf_uuid)
{
    m_default_factories [scim_get_locale_language (locale)] = sf_uuid;
}

void
X11FrontEnd::next_factory (const String &locale)
{
    String sf_uuid = get_factory (locale);

    std::vector<String> uuids;

    if (get_factory_list (uuids, scim_get_locale_encoding (locale))) {
        String new_uuid = uuids [0];

        for (size_t i=0; i<uuids.size ()-1; ++i) {
            if (sf_uuid == uuids [i]) {
                new_uuid = uuids [i+1];
                break;
            }
        }
        set_factory (locale, new_uuid);
    }
}

void
X11FrontEnd::previous_factory (const String &locale)
{
    String sf_uuid = get_factory (locale);

    std::vector<String> uuids;

    if (get_factory_list (uuids, scim_get_locale_encoding (locale))) {
        String new_uuid = uuids [uuids.size ()-1];

        for (size_t i=uuids.size ()-1; i>0; --i) {
            if (sf_uuid == uuids [i]) {
                new_uuid = uuids [i-1];
                break;
            }
        }
        set_factory (locale, new_uuid);
    }
}

static Time
get_time (void)
{
    int tint;
    struct timeval tv;
    struct timezone tz;           /* is not used since ages */
    gettimeofday (&tv, &tz); 
    tint = (int) tv.tv_sec * 1000;
    tint = tint / 1000 * 1000; 
    tint = tint + tv.tv_usec / 1000;
    return ((Time) tint);
}

XKeyEvent
X11FrontEnd::keyevent_scim_to_x11 (const KeyEvent& key)
{
    XKeyEvent xkey;

    xkey.type = (key.mask & SCIM_KEY_ReleaseMask) ? KeyRelease : KeyPress;
    xkey.display = m_display;
    xkey.serial = 0L;
    xkey.send_event = False;
    xkey.x = xkey.y = xkey.x_root = xkey.y_root = 0;
    xkey.time = get_time ();
    xkey.same_screen = False;
    xkey.subwindow = None;
    xkey.window = m_xims_window;
    xkey.root = DefaultRootWindow (m_display);
    xkey.keycode = XKeysymToKeycode (m_display, (KeySym) key.code);
    xkey.state = key.mask & (~SCIM_KEY_ReleaseMask);

    return xkey;
}

KeyEvent
X11FrontEnd::keyevent_x11_to_scim (const XKeyEvent& key)
{
    KeyEvent scimkey;
    KeySym keysym;
    XKeyEvent xkey = key;
    char buf [32];

    XLookupString (&xkey, buf, 32, &keysym, 0);

    scimkey.mask = xkey.state;
    scimkey.code = keysym;

    if (key.type == KeyPress)
        scimkey.mask &= ~SCIM_KEY_ReleaseMask;
    else
        scimkey.mask |= SCIM_KEY_ReleaseMask;

    return scimkey;
}

String
X11FrontEnd::get_supported_locales (void)
{
    std::vector <String> all_locales;
    std::vector <String> supported_locales;

    scim_split_string_list (all_locales, get_all_locales (), ',');

    String last = String (setlocale (LC_CTYPE, 0));

    for (size_t i = 0; i < all_locales.size (); ++i) {
        if (setlocale (LC_CTYPE, all_locales [i].c_str ()) && XSupportsLocale ())
            supported_locales.push_back (all_locales [i]);
    }

    setlocale (LC_CTYPE, last.c_str ());

    return scim_combine_string_list (supported_locales, ',');
}

void
X11FrontEnd::init_ims (void)
{
    XIMStyle ims_styles_overspot [] = {
        XIMPreeditPosition  | XIMStatusNothing,
        XIMPreeditNothing   | XIMStatusNothing,
        XIMPreeditPosition  | XIMStatusCallbacks,
        XIMPreeditNothing   | XIMStatusCallbacks,
        0
    };

    XIMStyle ims_styles_onspot [] = {
        XIMPreeditPosition  | XIMStatusNothing,
        XIMPreeditCallbacks | XIMStatusNothing,
        XIMPreeditNothing   | XIMStatusNothing,
        XIMPreeditPosition  | XIMStatusCallbacks,
        XIMPreeditCallbacks | XIMStatusCallbacks,
        XIMPreeditNothing   | XIMStatusCallbacks,
        0
    };

    XIMEncoding ims_encodings[] = {
        "COMPOUND_TEXT",
        0
    };

    XIMTriggerKey ims_keys[] = {
        {XK_space, ControlMask, ControlMask},
        {0L, 0L, 0L},
        {0L, 0L, 0L},
        {0L, 0L, 0L}
    };

    XIMStyles styles;
    XIMTriggerKeys keys;
    XIMEncodings encodings;
    uint32 count_keys;

    String locales;

    if (m_xims != (XIMS) 0) {
        throw FrontEndError (String ("X11 -- XIMS already initialized!"));
       }

    for (count_keys=0; count_keys<3 && count_keys<m_trigger_keys.size (); ++count_keys) {
        ims_keys [count_keys].keysym = m_trigger_keys[count_keys].code;
        ims_keys [count_keys].modifier = m_trigger_keys[count_keys].mask;
        ims_keys [count_keys].modifier_mask = m_trigger_keys[count_keys].mask;
    }

    if (!m_config.null () && 
        m_config->read (String (SCIM_CONFIG_FRONTEND_X11_ONTHESPOT), false)) {
        styles.count_styles = sizeof (ims_styles_onspot)/sizeof (XIMStyle) - 1;
        styles.supported_styles = ims_styles_onspot;
    } else {
        styles.count_styles = sizeof (ims_styles_overspot)/sizeof (XIMStyle) - 1;
        styles.supported_styles = ims_styles_overspot;
    }

    keys.count_keys = count_keys;
    keys.keylist = ims_keys;

    encodings.count_encodings = sizeof (ims_encodings)/sizeof (XIMEncoding) - 1;
    encodings.supported_encodings = ims_encodings;

    locales = get_supported_locales ();

    SCIM_DEBUG_FRONTEND(1) << "Initializing XIMS: "
            << m_server_name << " with locale (" << locales.length () << "): " << locales << " ...\n";

    m_display = XOpenDisplay (NULL);

    if (!m_display)
        throw FrontEndError (String ("X11 -- Cannot open Display!"));


    m_xims_window = XCreateSimpleWindow (m_display,
                                         DefaultRootWindow (m_display),
                                         -1, -1, 1, 1, 0, 0, 0);

    if (!m_xims_window)
        throw FrontEndError (String ("X11 -- Cannot create IMS Window!"));

    XSetWindowAttributes attrs;
    unsigned long attrmask;

    attrs.override_redirect = true;
    attrmask = CWOverrideRedirect;

    XChangeWindowAttributes (m_display, m_xims_window, attrmask, &attrs);
    XSelectInput (m_display, m_xims_window, KeyPressMask | KeyReleaseMask);

    m_old_x_error_handler = XSetErrorHandler (x_error_handler);

    m_xims = IMOpenIM(m_display,
            IMModifiers, "Xi18n",
            IMServerWindow, m_xims_window,
            IMServerName, m_server_name.c_str (),
            IMLocale, locales.c_str (),
            IMServerTransport, "X/",
            IMInputStyles, &styles,
            IMEncodingList, &encodings,
            IMProtocolHandler, ims_protocol_handler,
            IMFilterEventMask, KeyPressMask | KeyReleaseMask,
            NULL);

    if (m_xims == (XIMS)NULL)
        throw FrontEndError (String ("X11 -- failed to initialize XIM Server!"));

    if (m_xims_dynamic) {
            IMSetIMValues(m_xims,
                IMOnKeysList, &keys,
                NULL);
    }
}

int
X11FrontEnd::ims_open_handler (XIMS ims, IMOpenStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Open handler: LANG=" << call_data->lang.name
                 << " Connect ID=" << call_data->connect_id << "\n";

    m_ic_manager.new_connection (call_data);
    return 1;
}

int
X11FrontEnd::ims_close_handler (XIMS ims, IMCloseStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Close handler: Connect ID="
            << call_data->connect_id << "\n";

    m_ic_manager.delete_connection (call_data);
    return 1;
}

int
X11FrontEnd::ims_create_ic_handler (XIMS ims, IMChangeICStruct *call_data)
{
    String locale = m_ic_manager.get_connection_locale (call_data->connect_id);
    String encoding = scim_get_locale_encoding (locale);

    SCIM_DEBUG_FRONTEND(2) << " IMS Create handler: Encoding=" << encoding << "\n";

    if (locale.empty () || encoding.empty ())
        return 0;

    String sfid = get_factory (locale);
    int siid = new_instance (sfid, encoding);

    if (siid >= 0) {
        m_ic_manager.create_ic (call_data, siid);

        SCIM_DEBUG_FRONTEND(2) << " IMS Create handler OK: SIID="
            << siid << " Connect ID=" << call_data->connect_id  << "\n";
        return 1;
    } else {
        SCIM_DEBUG_FRONTEND(2) << " IMS Create handler Failed: "
            << " Connect ID=" << call_data->connect_id  << "\n";
    }
    return 0;
}

int
X11FrontEnd::ims_set_ic_values_handler (XIMS ims, IMChangeICStruct *call_data)
{
    uint32 changes;

    X11IC *ic = m_ic_manager.find_ic (call_data->icid);

    if (!validate_ic (ic)) {
        SCIM_DEBUG_FRONTEND(1) << "Cannot find IC for id " << call_data->icid << "\n";
        return 0;
    }

    changes = m_ic_manager.set_ic_values (call_data);

    if (changes & SCIM_X11_IC_ENCODING) {
        SCIM_DEBUG_FRONTEND(1) << "Cannot change IC encoding on the fly!\n";
        return 0;
    }

    SCIM_DEBUG_FRONTEND(2) << " IMS Set IC values handler, ICID="
                    << call_data->icid << " Connect ID="
                    << call_data->connect_id << " Changes="
                    << changes << "\n";

    socket_prepare_transaction (ic);

    //It's focus IC
    if (!is_forward_mode () && m_focus_ic->id == call_data->icid) {
        if (changes & SCIM_X11_IC_PRE_SPOT_LOCATION)
            socket_req_update_spot_location (ic);
    } else if (!m_focus_ic && ic->xims_on) {
        //In this case the previous focused IC
        //was just unsetted, so we should focus this IC.
        set_focus_ic (ic);
    }

    socket_send_request ();

    return 1;
}

int
X11FrontEnd::ims_get_ic_values_handler (XIMS ims, IMChangeICStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Get IC values handler, ICID="
                    << call_data->icid << " Connect ID="
                    << call_data->connect_id << "\n";

    m_ic_manager.get_ic_values (call_data);
    return 1;
}

int
X11FrontEnd::ims_destroy_ic_handler (XIMS ims, IMDestroyICStruct *call_data)
{
    X11IC *ic = m_ic_manager.find_ic (call_data->icid);

    SCIM_DEBUG_FRONTEND(2) << " IMS Destroy IC handler, ICID="
                    << call_data->icid << " Connect ID=" 
                    << call_data->connect_id << "\n";

    if (!validate_ic (ic)) {
        SCIM_DEBUG_FRONTEND(1) << "Cannot find IC for id " << call_data->icid << "\n";
        return 0;
    }

    delete_instance (ic->si_id);

    if (m_focus_ic && m_focus_ic->id == ic->id) {
        socket_prepare_transaction (ic);
        socket_req_turn_off_panel (ic);
        socket_send_request ();
        m_focus_ic = 0;
    }

    m_ic_manager.destroy_ic (call_data);
    return 1;
}

int
X11FrontEnd::ims_set_ic_focus_handler (XIMS ims, IMChangeFocusStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Set IC focus handler, ICID="
                    << call_data->icid << " Connect ID="
                    << call_data->connect_id << "\n";
    
    X11IC *ic =m_ic_manager.find_ic (call_data->icid);

    if (!validate_ic (ic)) {
        SCIM_DEBUG_FRONTEND(1) << "Cannot find IC for id " << call_data->icid << "\n";
        return 0;
    }

    socket_prepare_transaction (ic);
    set_focus_ic (ic);
    socket_send_request ();

    return 1;
}

int
X11FrontEnd::ims_unset_ic_focus_handler (XIMS ims, IMChangeFocusStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Unset IC focus handler, ICID="
                    << call_data->icid << " Connect ID="
                    << call_data->connect_id << "\n";

    X11IC *ic =m_ic_manager.find_ic (call_data->icid);

    if (!validate_ic (ic)) {
        SCIM_DEBUG_FRONTEND(1) << "Cannot find IC for id " << call_data->icid << "\n";
        return 0;
    }

    if (validate_ic (m_focus_ic) && m_focus_ic->id == ic->id) {
        socket_prepare_transaction (ic);
        unset_focus_ic ();
        socket_send_request ();
    }

    return 1;
}

int
X11FrontEnd::ims_reset_ic_handler (XIMS ims, IMResetICStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Reset IC handler, ICID="
                    << call_data->icid << " Connect ID="
                    << call_data->connect_id << "\n";
    
    X11IC *ic =m_ic_manager.find_ic (call_data->icid);

    if (!validate_ic (ic)) {
        SCIM_DEBUG_FRONTEND(1) << "Cannot find IC for id " << call_data->icid << "\n";
        return 0;
    }

    socket_prepare_transaction (ic);
    reset (ic->si_id);
    socket_send_request ();

    return 1;
}

int
X11FrontEnd::ims_trigger_notify_handler (XIMS ims, IMTriggerNotifyStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Trigger notify handler, Flag="
                    << call_data->flag << " KeyIndex="
                    << call_data->key_index << " EventMask="
                    << call_data->event_mask << "\n";

    X11IC *ic =m_ic_manager.find_ic (call_data->icid);

    if (!validate_ic (ic)) {
        SCIM_DEBUG_FRONTEND(1) << "Cannot find IC for id " << call_data->icid << "\n";
        return 0;
    }

    socket_prepare_transaction (ic);

    if (!call_data->flag) {
        ims_turn_on_ic (ic);
        socket_send_request ();
        return 1;
    } else {
        ims_turn_off_ic (ic);
        socket_send_request ();
        return 1;
    }

    return 0;
}

int
X11FrontEnd::ims_forward_event_handler (XIMS ims, IMForwardEventStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Forward event handler, ICID="
                    << call_data->icid << " Connect ID="
                    << call_data->connect_id << " SerialNo="
                    << call_data->serial_number << "EventType="
                    << call_data->event.type << "\n";
    
    if (call_data->event.type != KeyPress && call_data->event.type != KeyRelease)
        return 1;

    X11IC *ic = m_ic_manager.find_ic (call_data->icid);

    if (!validate_ic (ic)) {
        SCIM_DEBUG_FRONTEND(1) << "Cannot find IC for id " << call_data->icid << "\n";
        return 0;
    }

    XKeyEvent *event = (XKeyEvent*) &(call_data->event);
    KeyEvent scimkey = keyevent_x11_to_scim (*event);

    scimkey.mask &= m_valid_key_mask;

    SCIM_DEBUG_FRONTEND(3)  << "  KeyEvent:\n"
                            << "   Type=" << event->type << " Display=" << event->display << " Serial=" << event->serial << " Send=" << event->send_event << "\n"
                            << "      X=" << event->x << " Y=" << event->y << " XRoot=" << event->x_root << " YRoot=" << event->y_root << "\n"
                            << "   Time=" << event->time << " SameScreen=" << event->same_screen << " SubWin=" << event->subwindow << " Win=" << event->window << "\n"
                            << "   Root=" << event->root << " KeyCode=" << event->keycode << " State=" << event->state << "\n"
                            << "  scimKeyEvent=(" << scimkey.code << "," << scimkey.mask << ")\n";

    socket_prepare_transaction (ic);

    // If the ic is not focused, then focus it first.
    if (m_focus_ic != ic)
        set_focus_ic (ic);

    // No focused ic, just return.
    if (!validate_ic (m_focus_ic))
        return 1;

    //handle on/off key for static event flow
    if (match_key_event (m_trigger_keys, scimkey)) {
        if (!m_focus_ic->xims_on) {
            ims_turn_on_ic (m_focus_ic);
        } else { 
            ims_turn_off_ic (m_focus_ic);
        }
        socket_send_request ();
        return 1;
    }

    if (match_key_event (m_show_factory_menu_keys, scimkey)) {
        socket_req_show_factory_menu (m_focus_ic);
        socket_send_request ();
        return 1;
    }

    // If the ims is not turned on, just forward the event.
    if (!m_focus_ic->xims_on) {
        if (!m_fallback_instance->process_key_event (scimkey))
            IMForwardEvent (ims, (XPointer) call_data);
        return 1;
    }

    if (match_key_event (m_next_factory_keys, scimkey)) {
        next_factory (ic->locale);

        String sfid = get_factory (ic->locale);
        replace_instance (ic->si_id, sfid);

        set_focus_ic (ic);

        socket_send_request ();

        return 1;
    }

    if (match_key_event (m_previous_factory_keys, scimkey)) {
        previous_factory (ic->locale);

        String sfid = get_factory (ic->locale);
        replace_instance (ic->si_id, sfid);

        set_focus_ic (ic);

        socket_send_request ();

        return 1;
    }

    SCIM_DEBUG_FRONTEND(2) << " IMS Forward event handler -- forward keyevent to client.\n";

    if (!process_key_event (m_focus_ic->si_id, scimkey)) {
        if (!m_fallback_instance->process_key_event (scimkey))
            IMForwardEvent (ims, (XPointer) call_data);
    }

    socket_send_request ();

    return 1;
}

int
X11FrontEnd::ims_sync_reply_handler (XIMS ims, IMSyncXlibStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Sync reply handler.\n";
    return 1;
}

int
X11FrontEnd::ims_preedit_start_reply_handler (XIMS ims, IMPreeditCBStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Preedit start reply handler.\n";
    return 1;
}

int
X11FrontEnd::ims_preedit_caret_reply_handler (XIMS ims, IMPreeditCBStruct *call_data)
{
    SCIM_DEBUG_FRONTEND(2) << " IMS Preedit caret reply handler.\n";
    return 1;
}

void
X11FrontEnd::ims_commit_string (const X11IC *ic, const WideString& str)
{
    IMCommitStruct cms;
    XTextProperty tp;

    SCIM_DEBUG_FRONTEND(2) << " IMS Committing string.\n";

    if (ims_wcstocts (tp, ic, str)) {
        memset (&cms, 0, sizeof (cms));
        cms.major_code = XIM_COMMIT;
        cms.icid = ic->id;
        cms.connect_id = ic->connect_id;
        cms.flag = XimLookupChars;
        cms.commit_string = (char *) tp.value;
        IMCommitString (m_xims, (XPointer) & cms);
        XFree (tp.value);
    }
}

void
X11FrontEnd::ims_forward_key_event (const X11IC *ic, const KeyEvent &key)
{
    IMForwardEventStruct fe;
    XEvent xkp;

    XKeyEvent *event = (XKeyEvent*) (&xkp);

    //create event
    xkp.xkey = keyevent_scim_to_x11 (key);

    memset (&fe, 0, sizeof (fe));
    fe.major_code = XIM_FORWARD_EVENT;
    fe.icid = ic->id;
    fe.connect_id = ic->connect_id;
    fe.sync_bit = 0;
    fe.serial_number = 0L;

    if (ic->focus_win)
        xkp.xkey.window = ic->focus_win;
    else if (ic->client_win)
        xkp.xkey.window = ic->client_win;

    xkp.xkey.serial = 0L;
    xkp.xkey.time = get_time ();
    memcpy (&(fe.event), &xkp, sizeof (fe.event));
    IMForwardEvent (m_xims, (XPointer) & fe);
}

bool
X11FrontEnd::ims_wcstocts (XTextProperty &tp,const X11IC *ic, const WideString& src)
{
    if (!validate_ic (ic)) return false;

    String last = String (setlocale (LC_CTYPE, 0));

    if (!setlocale (LC_CTYPE, ic->locale.c_str ())) {
        SCIM_DEBUG_FRONTEND(3) << "  wcstocts -- unspported locale " << ic->locale.c_str () << "\n";

        setlocale (LC_CTYPE, last.c_str ());
        return false;
    }

    int ret;

    if (m_wchar_ucs4_equal && !m_broken_wchar) {
        wchar_t *wclist [1];

        SCIM_DEBUG_FRONTEND(3) << "  Convert WideString to COMPOUND_TEXT -- Using XwcTextListToTextProperty.\n";

        wclist [0] = new wchar_t [src.length () + 1];
        memcpy (wclist [0], src.data (), sizeof (wchar_t) * src.length ());
        wclist [0][src.length ()] = 0;
        ret = XwcTextListToTextProperty (m_display, wclist, 1, XCompoundTextStyle, &tp);
        delete [] wclist [0];
    } else {
        char *clist [1];
        String mbs;

        SCIM_DEBUG_FRONTEND(3) << "  Convert WideString to COMPOUND_TEXT -- Using XmbTextListToTextProperty.\n";

        if (!m_iconv.set_encoding (ic->encoding)) {
            SCIM_DEBUG_FRONTEND(3) << "  Convert WideString to COMPOUND_TEXT -- Cannot initialize iconv for encoding "
                      << ic->encoding << "\n";

            setlocale (LC_CTYPE, last.c_str ());
            return false;
        }

        m_iconv.convert (mbs, src);
        clist [0] = (char *) mbs.c_str ();
        ret = XmbTextListToTextProperty (m_display, clist, 1, XCompoundTextStyle, &tp);
    }

    setlocale (LC_CTYPE, last.c_str ());
    return ret >= 0;
}

bool
X11FrontEnd::ims_is_preedit_callback_mode (const X11IC *ic)
{
    if (validate_ic (ic) && (ic->input_style & XIMPreeditCallbacks))
        return true;
    return false;
}

void
X11FrontEnd::ims_preedit_callback_start (X11IC *ic)
{
    if (!validate_ic (ic) || ic->onspot_preedit_started) return;

    ic->onspot_preedit_started = true;

    SCIM_DEBUG_FRONTEND(2) << " Onspot preedit start, ICID="
            << ic->id << " Connect ID=" << ic->connect_id << "\n";

    IMPreeditCBStruct pcb;

    pcb.major_code        = XIM_PREEDIT_START;
    pcb.minor_code        = 0;
    pcb.connect_id        = ic->connect_id;
    pcb.icid              = ic->id;
    pcb.todo.return_value = 0;
    IMCallCallback (m_xims, (XPointer) & pcb);
}

void
X11FrontEnd::ims_preedit_callback_done (X11IC *ic)
{
    if (!validate_ic (ic) || !ic->onspot_preedit_started) return;

    SCIM_DEBUG_FRONTEND(2) << " Onspot preedit done, ICID="
            << ic->id << " Connect ID=" << ic->connect_id << "\n";

    // First clear the preedit string.
    ims_preedit_callback_draw (ic, WideString ());

    ic->onspot_preedit_started = false;

    IMPreeditCBStruct pcb;

    pcb.major_code        = XIM_PREEDIT_DONE;
    pcb.minor_code        = 0;
    pcb.connect_id        = ic->connect_id;
    pcb.icid              = ic->id;
    pcb.todo.return_value = 0;
    IMCallCallback (m_xims, (XPointer) & pcb);
}

void
X11FrontEnd::ims_preedit_callback_draw (X11IC *ic, const WideString& str, const AttributeList & attrs)
{
    if (!validate_ic (ic)) return;

    if (!ic->onspot_preedit_started)
        ims_preedit_callback_start (ic);

    SCIM_DEBUG_FRONTEND(2) << " Onspot preedit draw, ICID="
            << ic->id << " Connect ID=" << ic->connect_id << "\n";

    IMPreeditCBStruct pcb;
    XIMText text;
    XIMFeedback *feedback;
    XIMFeedback attr;
    XTextProperty tp;

    unsigned int i, j, len;

    len = str.length ();
    if (!len && !ic->onspot_preedit_length)
        return;

    feedback = new XIMFeedback [str.length () + 1];

    for (i = 0; i < len; ++i)
        feedback [i] = XIMUnderline;

    for (i = 0; i < attrs.size (); ++i) {
        attr = 0;
        if (attrs [i].get_type () == SCIM_ATTR_DECORATE) {
            if (attrs [i].get_value () == SCIM_ATTR_DECORATE_REVERSE)
                attr = XIMReverse;
            else if (attrs [i].get_value () == SCIM_ATTR_DECORATE_HIGHLIGHT)
                attr = XIMHighlight;
        }
        for (j = attrs [i].get_start (); j < attrs [i].get_end () && j < len; ++j)
            feedback [j] |= attr;
    }

    feedback [len] = 0;

    pcb.major_code = XIM_PREEDIT_DRAW;
    pcb.connect_id = ic->connect_id;
    pcb.icid = ic->id;

    pcb.todo.draw.caret = len;
    pcb.todo.draw.chg_first = 0;
    pcb.todo.draw.chg_length = ic->onspot_preedit_length;
    pcb.todo.draw.text = &text;

    text.feedback = feedback;

    if (len > 0 && ims_wcstocts (tp, ic, str)) {
        text.encoding_is_wchar = false;
        text.length = strlen ((char*)tp.value);
        text.string.multi_byte = (char*)tp.value;
        IMCallCallback (m_xims, (XPointer) & pcb);
        XFree (tp.value);
    } else {
        text.encoding_is_wchar = false;
        text.length = 0;
        text.string.multi_byte = "";
        IMCallCallback (m_xims, (XPointer) & pcb);
        len = 0;
    }

    ic->onspot_preedit_length = len;

    delete [] feedback;
}

void
X11FrontEnd::ims_preedit_callback_caret (X11IC *ic, int caret)
{
    if (!validate_ic (ic) || !ic->onspot_preedit_started || caret > ic->onspot_preedit_length || caret < 0)
        return;

    SCIM_DEBUG_FRONTEND(2) << " Onspot preedit caret, ICID="
            << ic->id << " Connect ID=" << ic->connect_id << "\n";

    //save the caret position for future usage when updating preedit string.
    ic->onspot_caret = caret;

    //client usually does not support this callback
    IMPreeditCBStruct pcb;

    pcb.major_code = XIM_PREEDIT_CARET;
    pcb.connect_id = ic->connect_id;
    pcb.icid       = ic->id;
    pcb.todo.caret.direction = XIMAbsolutePosition;
    pcb.todo.caret.position = caret;
    pcb.todo.caret.style = XIMIsPrimary;
    IMCallCallback (m_xims, (XPointer) & pcb);
}

void
X11FrontEnd::ims_turn_on_ic (X11IC *ic)
{
    if (validate_ic (ic)) {
        SCIM_DEBUG_FRONTEND(2) << " IMS Forward event handler -- turn on server.\n";
        ic->xims_on = true;

        if (m_xims_dynamic && validate_ic (ic)) {
            IMPreeditStateStruct ips;
            ips.major_code = 0;
            ips.minor_code = 0;
            ips.icid = ic->id;
            ips.connect_id = ic->connect_id;
            IMPreeditStart (m_xims, (XPointer) & ips);
        }

        set_focus_ic (ic);
    }
}

void
X11FrontEnd::ims_turn_off_ic (X11IC *ic)
{
    if (validate_ic (ic)) {
        SCIM_DEBUG_FRONTEND(2) << " IMS Forward event handler -- turn off server.\n";

        focus_out (ic->si_id);

        ic->xims_on = false;

        if (ims_is_preedit_callback_mode (ic))
            ims_preedit_callback_done (ic);

        socket_req_update_factory_info (ic);
        socket_req_turn_off_panel (ic);

        if (m_xims_dynamic && validate_ic (ic)) {
            IMPreeditStateStruct ips;
            ips.major_code = 0;
            ips.minor_code = 0;
            ips.icid = ic->id;
            ips.connect_id = ic->connect_id;
            IMPreeditEnd (m_xims, (XPointer) & ips);
        }
    }
}

void
X11FrontEnd::ims_sync_ic (X11IC *ic)
{
    if (validate_ic (ic)) {
        IMSyncXlibStruct data;

        data.major_code = XIM_SYNC;
        data.minor_code = 0;
        data.connect_id = ic->connect_id;
        data.icid = ic->id;

        IMSyncXlib(m_xims, (XPointer) &data);
    }
}

void
X11FrontEnd::set_focus_ic (X11IC *ic)
{
    if (!validate_ic (ic)) {
        unset_focus_ic ();
        return;
    }

    if (validate_ic (m_focus_ic) && m_focus_ic->id != ic->id) 
        focus_out (m_focus_ic->si_id);

    m_focus_ic = ic;

    socket_req_focus_in (ic);
    socket_req_update_display (ic);
    socket_req_update_screen (ic);
    socket_req_update_spot_location (ic);
    socket_req_update_factory_info (ic);

    if (validate_ic (ic) && ic->xims_on) {
        socket_req_turn_on_panel (ic);
        socket_req_hide_preedit_string (ic);
        socket_req_hide_aux_string (ic);
        socket_req_hide_lookup_table (ic);

        // Focus in to let IMEngine show necessary areas.
        focus_in (ic->si_id);
    } else {
        socket_req_turn_off_panel (ic);
    }
}

void
X11FrontEnd::unset_focus_ic ()
{
    if (validate_ic (m_focus_ic)) {
        focus_out (m_focus_ic->si_id);
        socket_req_update_factory_info (0);
        socket_req_turn_off_panel (m_focus_ic);
        socket_req_focus_out (m_focus_ic);
    }

    m_focus_ic = 0;
}

int
X11FrontEnd::ims_protocol_handler (XIMS ims, IMProtocol *call_data)
{
    if (!_scim_frontend || !call_data || ims != _scim_frontend->m_xims)
        return 0;

    switch (call_data->major_code) {
        case XIM_OPEN:
            return _scim_frontend->ims_open_handler (ims, (IMOpenStruct *) call_data);
        case XIM_CLOSE:
            return _scim_frontend->ims_close_handler (ims, (IMCloseStruct *) call_data);
        case XIM_CREATE_IC:
            return _scim_frontend->ims_create_ic_handler (ims, (IMChangeICStruct *) call_data);
        case XIM_DESTROY_IC:
            return _scim_frontend->ims_destroy_ic_handler (ims, (IMDestroyICStruct *) call_data);
        case XIM_SET_IC_VALUES:
            return _scim_frontend->ims_set_ic_values_handler (ims, (IMChangeICStruct *) call_data);
        case XIM_GET_IC_VALUES:
            return _scim_frontend->ims_get_ic_values_handler (ims, (IMChangeICStruct *) call_data);
        case XIM_FORWARD_EVENT:
            return _scim_frontend->ims_forward_event_handler (ims, (IMForwardEventStruct *) call_data);
        case XIM_SET_IC_FOCUS:
            return _scim_frontend->ims_set_ic_focus_handler (ims, (IMChangeFocusStruct *) call_data);
        case XIM_UNSET_IC_FOCUS:
            return _scim_frontend->ims_unset_ic_focus_handler (ims, (IMChangeFocusStruct *) call_data);
        case XIM_RESET_IC:
            return _scim_frontend->ims_reset_ic_handler (ims, (IMResetICStruct *) call_data);
        case XIM_TRIGGER_NOTIFY:
            return _scim_frontend->ims_trigger_notify_handler (ims, (IMTriggerNotifyStruct *) call_data);
        case XIM_PREEDIT_START_REPLY:
            return _scim_frontend->ims_preedit_start_reply_handler (ims, (IMPreeditCBStruct *) call_data);
        case XIM_PREEDIT_CARET_REPLY:
            return _scim_frontend->ims_preedit_caret_reply_handler (ims, (IMPreeditCBStruct *) call_data);
        case XIM_SYNC_REPLY:
            return _scim_frontend->ims_sync_reply_handler (ims, (IMSyncXlibStruct *) call_data);
        default:
            SCIM_DEBUG_FRONTEND(1) << "Unknown major code " << call_data->major_code << "\n";
            break;
    }
    return 1;
}

bool
X11FrontEnd::match_key_event (const std::vector <KeyEvent> &keys,
                               const KeyEvent &key)
{
    std::vector<KeyEvent>::const_iterator kit; 

    for (kit = keys.begin (); kit != keys.end (); ++kit) {
        if (key.code == kit->code &&
            (key.mask & kit->mask) == kit->mask &&
            (key.mask & SCIM_KEY_ReleaseMask) == (kit->mask & SCIM_KEY_ReleaseMask))
            return true;
    }
    return false;
}

int
X11FrontEnd::x_error_handler (Display *display, XErrorEvent *error)
{
#if ENABLE_DEBUG
    char buf [256];

    XGetErrorText (display, error->error_code, buf, 256);

    SCIM_DEBUG_FRONTEND (1)
            << "X Error occurred:\n"
            << "  Display     = " << display << "\n"
            << "  Type        = " << error->type << "\n"
            << "  Resourceid  = " << error->resourceid << "\n"
            << "  Serial      = " << error->serial << "\n"
            << "  ErrorCode   = " << (uint32) error->error_code << "\n"
            << "  RequestCode = " << (uint32) error->request_code << "\n"
            << "  MinorCode   = " << (uint32) error->minor_code << "\n"
            << "  Error Text  = " << buf << "\n";
#endif

    // trap all possible error for broken focus ic.
    if ((error->error_code == BadWindow ||
         error->error_code == BadMatch) &&
        (error->request_code == X_GetWindowAttributes ||
         error->request_code == X_GetProperty ||
         error->request_code == X_SendEvent ||
         error->request_code == X_TranslateCoords)) {
        SCIM_DEBUG_FRONTEND(1) << "Discard This Error\n";
    } else if (_scim_frontend && _scim_frontend->m_old_x_error_handler) {
        _scim_frontend->m_old_x_error_handler (display, error);
    }

    return 0;
}

bool
X11FrontEnd::socket_connect_panel (int argc, char **argv)
{
    SocketAddress address (m_panel_socket_address);

    if (address.valid ()) {
        if (!m_panel_socket.connect (address)) {
            launch_panel (argc, argv);
            for (int i=0; i<200; ++i) {
                if (m_panel_socket.connect (address))
                    return socket_open_connection ();
                scim_usleep (100000);
            }
            return false;
        }
        return socket_open_connection ();
    }
    return false;
}

bool
X11FrontEnd::socket_open_connection ()
{
    if (!scim_socket_trans_open_connection (m_socket_magic_key,
                                            String ("FrontEnd"),
                                            String ("Panel"),
                                            m_panel_socket,
                                            m_panel_socket_timeout)) {
        m_panel_socket.close ();
        return false;
    }

    return true;
}

bool
X11FrontEnd::socket_prepare_transaction (const X11IC *ic)
{
    m_send_transaction.clear ();
    m_send_transaction.put_command (SCIM_TRANS_CMD_REQUEST);
    m_send_transaction.put_data (m_socket_magic_key);

    if (ic)
        m_send_transaction.put_data ((uint32) ic->id);
    else
        m_send_transaction.put_data ((uint32) (-1));

    int cmd;
    uint32 data;

    // forward the get ptr to skip the request header.
    m_send_transaction.get_command (cmd);
    m_send_transaction.get_data (data);
    m_send_transaction.get_data (data);

    return true;
}

bool
X11FrontEnd::socket_send_request (void)
{
    if (m_panel_socket.is_connected () &&
        m_send_transaction.get_data_type () != SCIM_TRANS_DATA_UNKNOWN) {
        // Try to restart panel daemon if write to socket failed.
        if (m_send_transaction.write_to_socket (m_panel_socket, 0x4d494353))
            return true;

        if (socket_connect_panel (_argc, _argv))
            return m_send_transaction.write_to_socket (m_panel_socket, 0x4d494353);
    }

    return false;
}

bool
X11FrontEnd::socket_receive_reply (void)
{
    SocketTransaction receive;

    if (!m_panel_socket.is_connected () || !receive.read_from_socket (m_panel_socket, m_panel_socket_timeout))
        return false;

    int cmd;
    uint32 context;
    X11IC *ic;

    if (!receive.get_command (cmd) || cmd != SCIM_TRANS_CMD_REPLY)
        return false;

    // No context id available, so there will be some global command.
    if (receive.get_data_type () == SCIM_TRANS_DATA_COMMAND) {
        while (receive.get_command (cmd)) {
            switch (cmd) {
                case SCIM_TRANS_CMD_RELOAD_CONFIG:
                    {
                        if (!m_config.null ())
                            m_config->reload ();
                    }
                    break;
                case SCIM_TRANS_CMD_EXIT:
                    {
                        m_should_exit = true;
                        return true;
                    }
                default:
                    break;
            }
        }
        return true;
    }

    // Now for context related command.
    if (!receive.get_data (context))
        return false;

    ic = m_ic_manager.find_ic (context);

    if (!validate_ic (ic))
        return false;

    socket_prepare_transaction (ic);

    while (receive.get_command (cmd)) {
        switch (cmd) {
            case SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE_PAGE_SIZE:
                {
                    uint32 size;
                    if (receive.get_data (size))
                        update_lookup_table_page_size (ic->si_id, size);
                }
                break;
            case SCIM_TRANS_CMD_LOOKUP_TABLE_PAGE_UP:
                {
                    lookup_table_page_up (ic->si_id);
                }
                break;
            case SCIM_TRANS_CMD_LOOKUP_TABLE_PAGE_DOWN:
                {
                    lookup_table_page_down (ic->si_id);
                }
                break;
            case SCIM_TRANS_CMD_TRIGGER_PROPERTY:
                {
                    String property;
                    if (receive.get_data (property) && ic->xims_on)
                        trigger_property (ic->si_id, property);
                }
                break;
            case SCIM_TRANS_CMD_MOVE_PREEDIT_CARET:
                {
                    uint32 caret;
                    if (receive.get_data (caret))
                        move_preedit_caret (ic->si_id, caret);
                }
                break;
            case SCIM_TRANS_CMD_SELECT_CANDIDATE:
                {
                    uint32 item;
                    if (receive.get_data (item))
                        select_candidate (ic->si_id, item);
                }
                break;
            case SCIM_TRANS_CMD_PROCESS_KEY_EVENT:
                {
                    KeyEvent key;
                    if (receive.get_data (key)) {
                        if (m_focus_ic != ic) set_focus_ic (ic);
                        if (!process_key_event (ic->si_id, key)) {
                            if (!m_fallback_instance->process_key_event (key))
                                ims_forward_key_event (ic, key);
                        }
                    }
                }
                break;
            case SCIM_TRANS_CMD_COMMIT_STRING:
                {
                    String str;
                    if (receive.get_data (str))
                        ims_commit_string (ic, utf8_mbstowcs (str));
                }
                break;
            case SCIM_TRANS_CMD_FORWARD_KEY_EVENT:
                {
                    KeyEvent key;
                    if (receive.get_data (key))
                        ims_forward_key_event (ic, key);
                }
                break;
            case SCIM_TRANS_CMD_PANEL_REQUEST_HELP:
                {
                    socket_req_show_help (ic);
                }
                break;
            case SCIM_TRANS_CMD_PANEL_REQUEST_FACTORY_MENU:
                {
                    socket_req_show_factory_menu (ic);
                }
                break;
            case SCIM_TRANS_CMD_PANEL_CHANGE_FACTORY:
                {
                    String sfid;
                    if (receive.get_data (sfid)) {
                        if (!sfid.length () && ic->xims_on) {
                            ims_turn_off_ic (ic);
                        } else if (sfid.length ()) {
                            set_factory (ic->locale, sfid);
                            replace_instance (ic->si_id, sfid);
                            ims_turn_on_ic (ic);
                        }
                    }
                }
                break;
            default:
                break;
        }
    }
    socket_send_request ();
    return true;
}

void
X11FrontEnd::socket_req_turn_on_panel (const X11IC * /*ic*/)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_PANEL_TURN_ON);
}

void
X11FrontEnd::socket_req_turn_off_panel (const X11IC * ic)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_PANEL_TURN_OFF);
}

void
X11FrontEnd::socket_req_update_display (const X11IC * /*ic*/)
{
    if (m_display) {
        String name (DisplayString (m_display));
        if (name.length ()) {
            m_send_transaction.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_DISPLAY);
            m_send_transaction.put_data (name);
        }
    }
}

void
X11FrontEnd::socket_req_update_screen (const X11IC *ic)
{
    if (validate_ic (ic)) {
        Window target = ic->focus_win ? ic->focus_win : ic->client_win;
        XWindowAttributes xwa;
        if (target && 
            XGetWindowAttributes (m_display, target, &xwa) &&
            validate_ic (ic)) {
            int num = ScreenCount (m_display);
            int idx;
            for (idx = 0; idx < num; ++ idx) {
                if (ScreenOfDisplay (m_display, idx) == xwa.screen) {
                    m_send_transaction.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_SCREEN);
                    m_send_transaction.put_data ((uint32) idx);
                }
            }
        }
    }
}

void
X11FrontEnd::socket_req_show_help (const X11IC *ic)
{
    String help = get_help_info (ic);
    m_send_transaction.put_command (SCIM_TRANS_CMD_PANEL_SHOW_HELP);
    m_send_transaction.put_data (help);
}

void
X11FrontEnd::socket_req_show_factory_menu (const X11IC *ic)
{
    if (validate_ic (ic)) {
        std::vector<String> uuids;
        if (get_factory_list (uuids, ic->encoding)) {
            m_send_transaction.put_command (SCIM_TRANS_CMD_PANEL_SHOW_FACTORY_MENU);
            for (size_t i = 0; i < uuids.size (); ++ i) {
                m_send_transaction.put_data (uuids [i]);
                m_send_transaction.put_data (utf8_wcstombs (get_factory_name (uuids [i])));
                m_send_transaction.put_data (get_factory_language (uuids [i]));
                m_send_transaction.put_data (get_factory_icon_file (uuids [i]));
            }
        }
    }
}

void
X11FrontEnd::socket_req_focus_in (const X11IC * /*ic*/)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_FOCUS_IN);
}

void
X11FrontEnd::socket_req_focus_out (const X11IC * /*ic*/)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_FOCUS_OUT);
}

void
X11FrontEnd::socket_req_update_factory_info (const X11IC *ic)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_FACTORY_INFO);

    if (validate_ic (ic) && ic->xims_on) {
        m_send_transaction.put_data (utf8_wcstombs (get_instance_name (ic->si_id)));
        m_send_transaction.put_data (get_instance_icon_file (ic->si_id));
    } else {
        m_send_transaction.put_data (String (_("English/Keyboard")));
        m_send_transaction.put_data (String (SCIM_KEYBOARD_ICON_FILE));
    }
}

void
X11FrontEnd::socket_req_update_spot_location (const X11IC *ic)
{
    if (validate_ic (ic)) {
        Window target = ic->focus_win ? ic->focus_win : ic->client_win;
        XWindowAttributes xwa;

        if (target && 
            XGetWindowAttributes (m_display, target, &xwa) &&
            validate_ic (ic)) {

            static int original_x = 0, original_y = 0;
            int spot_x, spot_y;
            Window child;

            if (m_focus_ic->pre_attr.spot_location.x >= 0 &&
                m_focus_ic->pre_attr.spot_location.y >= 0) {
                XTranslateCoordinates (m_display, target,
                    xwa.root,
                    m_focus_ic->pre_attr.spot_location.x + 8,
                    m_focus_ic->pre_attr.spot_location.y + 8,
                    &spot_x, &spot_y, &child);
            } else {
                XTranslateCoordinates (m_display, target,
                    xwa.root,
                    0,
                    xwa.height,
                    &spot_x, &spot_y, &child);
            }
    	    if (original_x != spot_x || original_y != spot_y) {
                original_x = spot_x;
                original_y = spot_y;
                m_send_transaction.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_SPOT_LOCATION);
                m_send_transaction.put_data ((uint32) spot_x);
                m_send_transaction.put_data ((uint32) spot_y);
            }
        }
    }
}

void
X11FrontEnd::socket_req_show_preedit_string (const X11IC * /*ic*/)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_SHOW_PREEDIT_STRING);
}

void
X11FrontEnd::socket_req_hide_preedit_string (const X11IC * /*ic*/)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_HIDE_PREEDIT_STRING);
}

void
X11FrontEnd::socket_req_update_preedit_string (const X11IC * /*ic*/, const WideString &str, const AttributeList &attrs)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_PREEDIT_STRING);
    m_send_transaction.put_data (utf8_wcstombs (str));
    m_send_transaction.put_data (attrs);
}

void
X11FrontEnd::socket_req_update_preedit_caret (const X11IC * /*ic*/, int caret)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_PREEDIT_CARET);
    m_send_transaction.put_data ((uint32) caret);
}

void
X11FrontEnd::socket_req_show_aux_string (const X11IC * /*ic*/)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_SHOW_AUX_STRING);
}

void
X11FrontEnd::socket_req_hide_aux_string (const X11IC * /*ic*/)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_HIDE_AUX_STRING);
}

void
X11FrontEnd::socket_req_update_aux_string (const X11IC *ic, const WideString &str, const AttributeList &attrs)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_AUX_STRING);
    m_send_transaction.put_data (utf8_wcstombs (str));
    m_send_transaction.put_data (attrs);
}

void
X11FrontEnd::socket_req_show_lookup_table (const X11IC * /*ic*/)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_SHOW_LOOKUP_TABLE);
}

void
X11FrontEnd::socket_req_hide_lookup_table (const X11IC * /*ic*/)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_HIDE_LOOKUP_TABLE);
}

void
X11FrontEnd::socket_req_update_lookup_table (const X11IC * /*ic*/, const LookupTable &table)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE);
    m_send_transaction.put_data (table);
}

void
X11FrontEnd::socket_req_register_properties (const X11IC * /*ic*/, const PropertyList &properties)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_REGISTER_PROPERTIES);
    m_send_transaction.put_data (properties);
}

void
X11FrontEnd::socket_req_update_property (const X11IC * /*ic*/, const Property &property)
{
    m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_PROPERTY);
    m_send_transaction.put_data (property);
}

String
X11FrontEnd::get_help_info (const X11IC *ic)
{
    String help;
    String tmp;

    help =  String (_("Smart Common Input Method platform ")) +
            String (SCIM_VERSION) +
            String (_("\n(C) 2002-2004 James Su <suzhe@tsinghua.org.cn>\n\n"
                      "Hot keys:\n\n  "));

    scim_key_list_to_string (tmp, m_trigger_keys);
    help += tmp + String (_(":\n    open/close the input method.\n\n  "));
    
    scim_key_list_to_string (tmp, m_next_factory_keys);
    help += tmp + String (_(":\n    switch to the next input method.\n\n  "));

    scim_key_list_to_string (tmp, m_previous_factory_keys);
    help += tmp + String (_(":\n    switch to the previous input method.\n\n\n"));

    if (validate_ic (ic) && ic->xims_on) {
        help += utf8_wcstombs (get_instance_name (ic->si_id));
        help += String (_(":\n\n"));

        help += utf8_wcstombs (get_instance_authors (ic->si_id));
        help += String (_("\n\n"));

        help += utf8_wcstombs (get_instance_help (ic->si_id));
        help += String (_("\n\n"));

        help += utf8_wcstombs (get_instance_credits (ic->si_id));
    }
    return help;
}

void
X11FrontEnd::launch_panel (int argc, char **argv)
{
    char * my_argv [2] = {"--no-stay", 0};

    String config_module;

    // Check the config module option.
    for (int i=0; i<argc; ++i) {
        if ((String (argv [i]) == "-c" || String (argv [i]) == "--config") && i < (argc-1)) {
            config_module = argv [i+1];
            break;
        }
    }

    if (!config_module.length ())
        config_module = scim_global_config_read (SCIM_GLOBAL_CONFIG_DEFAULT_CONFIG_MODULE, String ("simple"));

    scim_launch_panel (true, config_module, 0, "none", my_argv);
}

void
X11FrontEnd::reload_config_callback (const ConfigPointer &config)
{
    SCIM_DEBUG_FRONTEND(1) << "Reload configuration.\n";

    // Reload some configurations
 
    // Reload trigger keys if it's not using dynamic XIM.
    if (!m_xims_dynamic) {
        scim_string_to_key_list (m_trigger_keys,
            config->read (String (SCIM_CONFIG_FRONTEND_KEYS_TRIGGER),
                          String ("Control+space")));
    }

    scim_string_to_key_list (m_next_factory_keys, 
        config->read (String (SCIM_CONFIG_FRONTEND_KEYS_NEXT_FACTORY),
                      String ("Control+Alt+Down,Control+Shift_R,Control+Shift_L")));

    scim_string_to_key_list (m_previous_factory_keys, 
        config->read (String (SCIM_CONFIG_FRONTEND_KEYS_PREVIOUS_FACTORY),
                      String ("Control+Alt+Up,Shift+Control_R,Shift+Control_L")));

    scim_string_to_key_list (m_show_factory_menu_keys, 
        config->read (String (SCIM_CONFIG_FRONTEND_KEYS_SHOW_FACTORY_MENU),
                      String ("Control+Alt+l,Control+Alt+m,Control+Alt+s,Control+Alt+Right")));

    KeyEvent key;
    scim_string_to_key (key,
        config->read (String (SCIM_CONFIG_FRONTEND_KEYS_VALID_KEY_MASK),
                      String ("Shift+Control+Alt+Lock")));

    m_valid_key_mask = (key.mask > 0)?(key.mask | SCIM_KEY_ReleaseMask):0xFFFF;

    m_broken_wchar =
        config->read (String (SCIM_CONFIG_FRONTEND_X11_BROKEN_WCHAR),
                      m_broken_wchar);
}

void
X11FrontEnd::fallback_commit_string_cb (IMEngineInstanceBase * si, const WideString & str)
{
    if (validate_ic (m_focus_ic))
        ims_commit_string (m_focus_ic, str);
}

bool
X11FrontEnd::check_socket_connection (const Socket &socket)
{
    unsigned char buf [sizeof(uint32)];

    int nbytes = socket.read_with_timeout (buf, sizeof(uint32), m_panel_socket_timeout);

    if (nbytes == sizeof (uint32))
        return true;

    return false;
}


/*
vi:ts=4:nowrap:expandtab
*/
