/* # skkinput (Simple Kana-Kanji Input)
 * lispeval.c --- Evaluate lisp code
 * This file is part of skkinput.
 * Copyright (C) 1997
 * Takashi SAKAMOTO (sakamoto@yajima.kuis.kyoto-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "commondef.h"
#include "lispeval.h"
#include "lispparse.h"
#if defined(MYEVAL_ONLY)
static char *skkinput_version = "myeval-test 1.00" ;
#else
#undef main_c
#include "version.h"
#endif

#undef DEBUG
/*
 * ץȥ
 */
static struct skkinputlisp_entity *stringIsString
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *stringIsAtom
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *stringIsInteger
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *stringIsVariable
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *stringIsArray
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *stringIsFunction
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop ) ;
static void free_entity( struct skkinputlisp_entity *entity ) ;
static struct skkinputlisp_entity *conslist2conspair
( struct conslist *node,
  struct skkinputlisp_variable *localVariableListTop ) ;
static void free_unreferred_entity
( struct skkinputlisp_entity *entity ) ;

static int isIntegerP( struct myChar *string ) ;
static int isStringP( struct myChar *string ) ;
static int isArrayP( struct myChar *string ) ;

static int isNilStringP( struct myChar *string ) ;

static struct skkinputlisp_entity *myevalLambdaForm
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;

struct skkinputlisp_entity *string2entity
( struct myChar *string, 
  struct skkinputlisp_variable *localVariableListTop ) ;

/*
 *
 */
static struct skkinputlisp_entity *skkinputlisp_quote
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_car
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_cdr
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_cons
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_set
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_setq
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_and
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_or
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_cond
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_if
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_concat
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_make_string
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_substring
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_current_time_string
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_j_date
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_skk_version
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_aref
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_aset
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_defun
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_lambda
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_multiply
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_plus
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_divide
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_minus
( struct conslist *top, 
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_string_to_int
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_int_to_string
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_progn
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;
static struct skkinputlisp_entity *skkinputlisp_load_file
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop ) ;

#if !defined(MYEVAL_ONLY)
extern int j_date_ad
( struct myChar *buffer, int bufsize, int date_ad, int number_style ) ;
extern unsigned char *expand_file_name( unsigned char *path ) ;
extern int check_skkinput_jisyo_code( unsigned char *path ) ;
#else
int j_date_ad
( struct myChar *buffer, int bufsize, int date_ad, int number_style ) ;
unsigned char *expand_file_name( unsigned char *path ) ;
int check_skkinput_jisyo_code( unsigned char *path ) ;
#endif

/* ΤĤʤꥹȡ*/
static struct skkinputlisp_entity *entityListTop = NULL ;
static struct skkinputlisp_variable *globalVariableListTop = NULL ;
static struct skkinputlisp_ufunction
*userDefinedFunctionListTop = NULL ;

static struct skkinputlisp_function skkinputlispFunctionTable[] = {
  { "quote",			skkinputlisp_quote },
  { "car",			skkinputlisp_car }, 
  { "cdr",			skkinputlisp_cdr }, 
  { "cons",			skkinputlisp_cons }, 
  { "set",			skkinputlisp_set }, 
  { "setq",			skkinputlisp_setq }, 
  { "and",			skkinputlisp_and }, 
  { "or",			skkinputlisp_or }, 
  { "cond",			skkinputlisp_cond },
  { "if",			skkinputlisp_if },
  { "concat",			skkinputlisp_concat },
  { "make-string",		skkinputlisp_make_string },
  { "substring",		skkinputlisp_substring },
  { "current-time-string",	skkinputlisp_current_time_string },
  { "j-date",			skkinputlisp_j_date },
  { "skk-version",		skkinputlisp_skk_version },
  { "aref",			skkinputlisp_aref },
  { "aset",			skkinputlisp_aset },
  { "defun",			skkinputlisp_defun },
  { "lambda",			skkinputlisp_lambda },
  { "*",			skkinputlisp_multiply },
  { "+",			skkinputlisp_plus },
  { "/",			skkinputlisp_divide },
  { "-",			skkinputlisp_minus },
  { "string-to-int",		skkinputlisp_string_to_int },
  { "int-to-string",		skkinputlisp_int_to_string },
  { "progn",			skkinputlisp_progn },
  { "load-file",		skkinputlisp_load_file },
  { NULL, 			NULL },
} ;

/*
 * ʸͤľؿ
 */
static int char_to_number( struct myChar chara )
{
  if( !MYCHAR_IS_ALPHABETIC( chara ) )
    return ERR ;
  /* a  f δ֤硣*/
  if( chara.chara >= 'a' && chara.chara <= 'f' )
    return ( ( chara.chara - 'a' ) + 10 ) ;
  /* A  F δ֤硣*/
  if( chara.chara >= 'A' && chara.chara <= 'F' )
    return ( ( chara.chara - 'A' ) + 10 ) ;
  /* ʳϿ͡*/
  return ( chara.chara - '9' ) ;
}

/*------------------------------------------------------------------------
 *  skklisp βԤʬǤ롣
 *-----------------------------------------------------------------------*/

/*
 * Τɽؿ
 */
static void print_entity( struct skkinputlisp_entity *entity )
{
  int i, length ;
  struct skkinputlisp_entity **ptr ;
  if( entity == NULL )
    return ;
  switch( entity->type ){
  case ENTITY_ATOM :
    myCharFputstring( stdout, entity->data.string ) ;
    break ;
  case ENTITY_STRING :
    fputc( 0x22, stdout ) ;
    myCharFputstring( stdout, entity->data.string ) ;
    fputc( 0x22, stdout ) ;
    break ;
  case ENTITY_INTEGER :
    fprintf( stdout, "%d", entity->data.integer ) ;
    break ;
  case ENTITY_CONS :
    fprintf( stdout, "(" ) ;
    print_entity( entity->data.conspair.left ) ;
    fprintf( stdout, " . " ) ;
    print_entity( entity->data.conspair.right ) ;
    fprintf( stdout, ")" ) ;
    break ;
  case ENTITY_ARRAY :
    fprintf( stdout, "[" ) ;
    ptr = entity->data.array.array ;
    length = entity->data.array.length - 1 ;
    if( length >= 0 ){
      for( i = 0 ; i < length ; i ++, ptr ++ ){
	print_entity( *ptr ) ;
	fprintf( stdout, " " ) ;
      }
      print_entity( *ptr ) ;
    }
    fprintf( stdout, "]" ) ;
    break ;
  default :
    fprintf( stdout, "unknown entity.\n" ) ;
    break ;
  }
  return ;
}

/*
 * ΤõΤѤؿ
 */
static struct skkinputlisp_entity *search_entity
( struct skkinputlisp_entity *entity )
{
  struct skkinputlisp_entity *node ;

  if( ( node = entityListTop ) == NULL )
    return NULL ;
  /* ꥹȤõ() */
  for( ; node != NULL ; node = node->next ){
    /* ۤʤäƤƤӤޤǤʤ*/
    if( node->type != entity->type )
      continue ;
    /* ˤäӤˡۤʤ롣*/
    switch( entity->type ){
    case ENTITY_ATOM :
    case ENTITY_STRING :
      if( !myCharStrcmp( node->data.string, entity->data.string ) )
	return node ;
      break ;
    case ENTITY_INTEGER :
      if( node->data.integer == entity->data.integer )
	return node ;
      break ;
    case ENTITY_CONS :
      /* ʤݥ󥿤ȤӤƤ뤱ɡĤΤʤ */
      if( node->data.conspair.left == entity->data.conspair.left &&
	  node->data.conspair.right == entity->data.conspair.right )
	return node ;
      break ;
    default :
      break ;
    }
  }
  return NULL ;
}

/*
 * ΤΤѤؿ
 */
static struct skkinputlisp_entity *create_newEntity( void )
{
  struct skkinputlisp_entity *sklentity ;
  sklentity = malloc( sizeof( struct skkinputlisp_entity ) ) ;
  if( sklentity == NULL ){
    fprintf( stdout, "(skkinput) ** Fatal ** Memory fault.\n" ) ;
    exit( 1 ) ;
  }
  sklentity->type  = ENTITY_EMPTY ;
  sklentity->prev = sklentity->next = NULL ;
  sklentity->count = 0 ;
  return sklentity ;
}

static void link_newEntity( struct skkinputlisp_entity *entity )
{
  /* Ƭ NULL Ǥ롣*/
  entity->prev = NULL ;
  /* ƬμϥƬäͤˤʤ롣*/
  entity->next = entityListTop ;
  /* ⤷ƬäͤºߤƤΤʤ顢οͤοͤˤʤ롣*/
  if( entityListTop != NULL ){
    entityListTop->prev = entity ;
  }
  /* ߤƬοͤФ롣*/
  entityListTop = entity ;
  return ;
}

/*
 * Τ¸ߤ뤫õ¸ߤʤмΤؿ
 */
static struct skkinputlisp_entity *searchIfNotFindCreateStringEntity
( struct myChar *dstring )
{
  struct skkinputlisp_entity *node, *ret ;
  /* ޤμΤ롣*/
  if( ( node = create_newEntity() ) == NULL )
    return NULL ;
  node->type = ENTITY_STRING ;
  node->data.string = dstring ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( ret = search_entity( node ) ) == NULL ){
    link_newEntity( node ) ;
    return node ;
  }
  /* פˤʤäʸ롣*/
  free( dstring ) ;
  /* פˤʤäΡɤ롣*/
  free( node ) ;
  /* ֤ͤ*/
  return ret ;
}

/*
 * Ϳ줿ʸᤷ entity Ѵؿ
 */
struct skkinputlisp_entity *stringIsString
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct myChar *dptr, *dstring ;
  int mode, num, tmpnum, len ;

  string ++ ;
  /* γݤ˼Ԥ顢NULL ֤*/
  dstring = dptr = malloc
    ( sizeof( struct myChar ) * ( myCharStrlen( string ) + 1 ) ) ;
  if( dstring == NULL )
    return NULL ;

  num = len = 0 ;
  mode = SKKLISP_STRING_NORMAL ;
  /* ʸΥԡ򳫻Ϥ롣*/
  while( !IS_END_OF_STRING( *string ) && !IS_ASCII_EQUAL( *string, 0x22 ) ){ 
    switch( mode ){
    case SKKLISP_STRING_NORMAL :
      if( IS_ASCII_EQUAL( *string, '\\' ) ){ 
	mode = SKKLISP_STRING_BACKSLASH ;
	string ++ ;
	num = 0 ;
	continue ;
      }
      *dptr ++ = *string ++ ;
      len ++ ;
      break ;
    case SKKLISP_STRING_BACKSLASH :
      if( IS_ASCII_CHARA( *string ) ){
	if( string->chara == 'x' ){
	  mode = SKKLISP_STRING_HEX ;
	} else if( ( string->chara >= 'a' && string->chara <= 'f' ) ||
		   ( string->chara >= 'A' && string->chara <= 'F' ) ){
	  /* ³ʸե٥åȤ a  f ˤС*/
	  mode = SKKLISP_STRING_HEX ;
	  num  = char_to_number( *string ) ;
	} else if( string->chara >= '0' && string->chara <= '7' ){
	  /* ³ʸäˤ 8 ʿȻפäƴꤹ롣*/
	  mode = SKKLISP_STRING_OCT ;
	  num  = string->chara - '0' ;
	} else {
	  /* ʳʸäν*/
	  switch( string->chara ){
	    /* ԥɤϻĤ*/
	  case 'r' :
	  case 'n' :
	    MYCHAR_SET_CHARA( *dptr, '\n' ) ;
	    dptr ++ ;
	    len ++ ;
	    mode = SKKLISP_STRING_NORMAL ;
	    break ;
	    /* ݤʤΤǥ֤ϥڡĤѴ롣*/
	  case 't' :
	    MYCHAR_SET_CHARA( *dptr, ' ' ) ;
	    dptr ++ ;
	    len ++ ;
	    mode = SKKLISP_STRING_NORMAL ;
	    break ;
	  default :
	    *dptr ++ = *string ;
	    len ++ ;
	    mode = SKKLISP_STRING_NORMAL ;
	    break ;
	  }
	}
      } else {
	*dptr ++ = *string ;
	len ++ ;
	mode = SKKLISP_STRING_NORMAL ;
      }
      string ++ ;
      break ;
    case SKKLISP_STRING_HEX :
      tmpnum = char_to_number( *string ) ;
      if( tmpnum > 0 ){
	tmpnum += ( num << 4 ) ;
	/* ˤʤ᤮ʤ褦դ롣*/
	if( tmpnum >= 0x80 ){
	  MYCHAR_SET_CHARA( *dptr, num ) ;
	  dptr ++ ;
	  len ++ ;
	  *dptr ++ = *string ;
	  len ++ ;
	  mode = SKKLISP_STRING_NORMAL ;
	} else {
	  num = tmpnum ;
	}
      } else {
	/* ͤǤʤʤäΤʤߤ롣*/
	MYCHAR_SET_CHARA( *dptr, num ) ;
	dptr ++ ;
	len ++ ;
	mode = SKKLISP_STRING_NORMAL ;
	continue ;
      }
      string ++ ;
      break ;
    case SKKLISP_STRING_OCT :
      if( IS_ASCII_CHARA( *string ) &&
	  string->chara >= '0' &&
	  string->chara <= '7' ){
	tmpnum = ( string->chara - '0' ) + num * 8 ;
	if( tmpnum >= 0x80 ){
	  /* ͤǤʤʤäΤʤߤ롣*/
	  MYCHAR_SET_CHARA( *dptr, num ) ;
	  dptr ++ ;
	  len ++ ;
	  *dptr ++ = *string ;
	  len ++ ;
	  mode = SKKLISP_STRING_NORMAL ;
	} else {
	  num = tmpnum ;
	}
      } else {
	/* ͤǤʤʤäΤʤߤ롣*/
	MYCHAR_SET_CHARA( *dptr, num ) ;
	dptr ++ ;
	len ++ ;
	mode = SKKLISP_STRING_NORMAL ;
	continue ;
      }
      string ++ ;
      break ;
    default :
      break ;
    }
  }
  if( mode == SKKLISP_STRING_OCT || mode == SKKLISP_STRING_HEX ){
    MYCHAR_SET_CHARA( *dptr, num ) ;
    dptr ++ ;
  }
  MYCHAR_SET_END_OF_STRING( *dptr ) ;
  return searchIfNotFindCreateStringEntity( dstring ) ;
}

/*
 * nil ɽ魯Τ롣
 */
struct skkinputlisp_entity *nilEntity( void )
{
  struct skkinputlisp_entity *node, *ret ;
  /* ޤμΤ롣*/
  if( ( node = create_newEntity() ) == NULL )
    return NULL ;
  /* ̾ nil Ȥȥ롣*/
  node->type = ENTITY_ATOM ;
  node->data.string = malloc( sizeof( struct myChar ) * 4 ) ;
  myCharCharStrcpy( node->data.string, "nil" ) ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( ret = search_entity( node ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( node ) ;
    return node ;
  }
  /* פˤʤäʸ롣*/
  free( node->data.string ) ;
  /* פˤʤäΡɤ롣*/
  free( node ) ;
  return ret ;
}

/*
 * t ɽ魯Τ롣
 */
static struct skkinputlisp_entity *tEntity( void )
{
  struct skkinputlisp_entity *node, *ret ;
  /* ޤμΤ롣*/
  if( ( node = create_newEntity() ) == NULL )
    return NULL ;
  /* ̾ nil Ȥȥ롣*/
  node->type = ENTITY_ATOM ;
  node->data.string = malloc( sizeof( struct myChar ) * 2 ) ;
  MYCHAR_SET_CHARA( node->data.string[ 0 ], 't' ) ;
  MYCHAR_SET_END_OF_STRING( node->data.string[ 1 ] ) ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( ret = search_entity( node ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( node ) ;
    return node ;
  }
  /* פˤʤäʸ롣*/
  free( node->data.string ) ;
  /* פˤʤäΡɤ롣*/
  free( node ) ;
  return ret ;
}

/*
 * nil ɽ魯Τ롣
 */
struct skkinputlisp_entity *emptyStringEntity( void )
{
  struct skkinputlisp_entity *node, *ret ;
  /* ޤμΤ롣*/
  if( ( node = create_newEntity() ) == NULL )
    return NULL ;
  /* ̾ nil Ȥȥ롣*/
  node->type = ENTITY_STRING ;
  node->data.string = NULL ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( ret = search_entity( node ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( node ) ;
    return node ;
  }
  /* פˤʤäΡɤ롣*/
  free( node ) ;
  return ret ;
}

/*
 * ͤȹͤƺġ
 */
static struct skkinputlisp_entity *stringIsInteger
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct myChar *sptr = string ;
  int value = 0, flag ;
  struct skkinputlisp_entity *node, *ret ;

  flag = 1 ;
  if( IS_ASCII_EQUAL( *sptr, '?' ) ){
    sptr ++ ;
    value = sptr->chara ;
  } else {
    flag = 1 ;
    if( IS_ASCII_EQUAL( *sptr, '+' ) ){
      sptr ++ ;
    } else if( IS_ASCII_EQUAL( *sptr, '-' ) ){
      sptr ++ ;
      flag = -1 ;
    }
    /* ʸĴ٤롣*/
    while( !IS_END_OF_STRING( *sptr ) ){
      value = value * 10 + ( sptr->chara - '0' ) ;
      sptr ++ ;
    }
    value = value * flag ;
  }
  /* ޤμΤ롣*/
  if( ( node = create_newEntity() ) == NULL )
    return NULL ;
  /* ̾ nil Ȥȥ롣*/
  node->type = ENTITY_INTEGER ;
  node->data.integer = value ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( ret = search_entity( node ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( node ) ;
    return node ;
  }
  /* פˤʤäΡɤ롣*/
  free( node ) ;
  return ret ;
}

static struct skkinputlisp_entity *stringIsAtom
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *node, *ret ;

  if( string == NULL )
    return NULL ;
  if( isStringP( string ) ){
    return stringIsString( string, localVariableListTop ) ;
  } else if( isIntegerP( string ) ){
    return stringIsInteger( string, localVariableListTop ) ;
  } else if( isNilStringP( string ) ){
    return nilEntity() ;
  } else if( IS_ASCII_EQUAL( string[ 0 ], 't' ) && 
	     IS_END_OF_STRING( string[ 1 ] ) ){
    return tEntity() ;
  } else if( isArrayP( string ) ){
    return stringIsArray
      ( string, localVariableListTop ) ;
  }
  /* ޤμΤ롣*/
  if( ( node = create_newEntity() ) == NULL )
    return NULL ;
  node->type = ENTITY_ATOM ;
  node->data.string = malloc
    ( sizeof( struct myChar ) * ( myCharStrlen( string ) + 1 ) ) ;
  /* ꤬­Ƥ롩 */
  if( node->data.string == NULL ){
    free( node ) ;
    return NULL ;
  }
  /* ȥ̾򥳥ԡ롣*/
  myCharStrcpy( node->data.string, string ) ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( ret = search_entity( node ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( node ) ;
    return node ;
  }
  free( node->data.string ) ;
  /* פˤʤäΡɤ롣*/
  free( node ) ;
  return ret ;
}

/*
 * ѿθԤؿ
 */
static struct skkinputlisp_variable *search_localVariable
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_variable *node ;
  node = localVariableListTop ;
  while( node != NULL ){
    if( !myCharStrcmp( node->name, string ) )
      return node ;
    node = node->next ;
  }
  return NULL ;
}

static struct skkinputlisp_variable *search_globalVariable
( struct myChar *string )
{
  return search_localVariable( string, globalVariableListTop ) ;
}

static struct skkinputlisp_variable *create_newVariable( void )
{
  struct skkinputlisp_variable *node ;
  /* γݤ˼Ԥޤ */
  if( ( node = malloc( sizeof( struct skkinputlisp_variable ) ) ) == NULL ){
    return NULL ;
  }
  node->name  = NULL ;
  node->value = NULL ;
  /* ǥꥹȤˤĤʤǤޤġ*/
  node->next  = globalVariableListTop ;
  globalVariableListTop = node ;
  return node ;
}

static void increment_entity_count
( struct skkinputlisp_entity *entity )
{
  if( entity->type == ENTITY_CONS ){
    increment_entity_count( entity->data.conspair.left ) ;
    increment_entity_count( entity->data.conspair.right ) ;
  } else if( entity->type == ENTITY_ARRAY ){
    int i ;
    struct skkinputlisp_entity **node ;
    node = entity->data.array.array ;
    for( i = 0 ; i < entity->data.array.length ; i ++, node ++ ){
      increment_entity_count( *node ) ;
    }
  }
  entity->count ++ ;
  return ;
}

static int link_variable_to_entity
( struct skkinputlisp_variable *variable,
  struct skkinputlisp_entity *entity )
{
  if( variable->value != NULL ){
    /* ΤȤƤȤ򸺤餹*/
    free_entity( variable->value ) ;
  }
  variable->value = entity ;
  /* ȤƤ䤹*/
  increment_entity_count( entity ) ;
  return True ;
}

static int isIntegerP( struct myChar *string )
{
  /* Ƭäˤϡ⤷ͤǤϤʤȵ*/
  if( !IS_ASCII_CHARA( *string ) )
    return False ;
  if( string->chara == '?' ) 
    return True ;
  if( string->chara == '+' || string->chara == '-' )
    string ++ ;
  if( IS_ASCII_CHARA( *string ) && 
      string->chara >= '0' && 
      string->chara <= '9' ){
    struct myChar *ptr = string ;
    /* äȿʤȿͤǤϤʤ*/
    while( !IS_END_OF_STRING( *ptr ) ){
      if( !IS_ASCII_CHARA( *ptr ) || ptr->chara < '0' || ptr->chara > '9' )
	break ;
      ptr ++ ;
    }
    /* 衢ͤ衣ͤϲ ưϰ
       ʤ͡*/
    if( IS_END_OF_STRING( *ptr ) )
      return True ;
  }
  return False ;
}

static int isStringP( struct myChar *string )
{
  if( IS_ASCII_EQUAL( *string, 0x22 ) )
    return True ;
  return False ;
}

static int isArrayP( struct myChar *string )
{
  if( IS_ASCII_EQUAL( *string, '[' ) ){
    return True ;
  }
  return False ;
}

static int isNilStringP( struct myChar *string )
{
  struct myChar *ptr ;
  if( !IS_ASCII_EQUAL( string[ 0 ], 'n' ) ||
      !IS_ASCII_EQUAL( string[ 1 ], 'i' ) ||
      !IS_ASCII_EQUAL( string[ 2 ], 'l' ) )
    return False ;
  ptr = string + 3 ;
  skip_space( &ptr ) ;
  if( IS_END_OF_STRING( *ptr ) )
    return True ;
  return False ;
}

/*
 * 礭Ѥؿ
 */
static int countArraySizeFromConslist
( struct conslist *top )
{
  int number_of_items ;
  struct conslist *node ;
  /* 0 ˤʤΤϤʤ 0 äΤߡ*/
  if( ( node = top ) == NULL )
    return 0 ;
  number_of_items = 1 ;
  /* ˿ͤ¤ʤࡣ*/
  while( node->right.value.next != NULL &&
	 node->right.type == TYPE_CONSLIST_ARRAY ){
    number_of_items ++ ;
    node = node->right.value.next ;
  }
  return number_of_items ;
}

static struct skkinputlisp_entity *conslist2array
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct conslist *node ;
  struct skkinputlisp_entity **eptr, *entity ;
  int number_of_items, count ;

  number_of_items = countArraySizeFromConslist( top ) ;
  if( ( entity = create_newEntity() ) == NULL ){
    return NULL ;
  }
  entity->type = ENTITY_ARRAY ;
  if( number_of_items <= 0 ){
    entity->data.array.length = 0 ;
    entity->data.array.array = NULL ;
    return entity ;
  } else {
    entity->data.array.length = number_of_items ;
    entity->data.array.array = 
      malloc( sizeof( struct skkinputlisp_entity *) * number_of_items ) ;
  }
  if( entity->data.array.array == NULL ){
    free_unreferred_entity( entity ) ;
    return NULL ;
  }
  eptr = entity->data.array.array ;
  count = 0 ;
  node = top ;
  while( node != NULL && number_of_items > count ){
#if 0
    if( node->right.type != TYPE_CONSLIST_ARRAY )
      break ;
#endif
    switch( node->left.type ){
    case TYPE_CONSLIST_ARRAY :
      *eptr = conslist2array
	( node->left.value.next, localVariableListTop ) ;
      if( *eptr == NULL )
	goto conslist2array_error ;
      increment_entity_count( *eptr ) ;
      break ;
    case TYPE_CONSLIST_CONSPAIR :
      if( node->left.value.next == NULL ){
	*eptr = nilEntity() ;
      } else {
	*eptr = conslist2conspair
	  ( node->left.value.next, localVariableListTop ) ;
      }
      if( *eptr == NULL )
	goto conslist2array_error ;
      increment_entity_count( *eptr ) ;
      break ;
    case TYPE_CONSLIST_STRING :
      *eptr = stringIsAtom
	( node->left.value.string, localVariableListTop ) ;
      if( *eptr == NULL )
	goto conslist2array_error ;
      increment_entity_count( *eptr ) ;
      break ;
    default :
#if defined(DEBUG)
      fprintf( stdout, "(conslist2array) unknown data type\n" ) ;
#endif
      break ;
    }
    eptr ++ ;
    count ++ ;
    node = node->right.value.next ;
  }
  return entity ;

conslist2array_error:
  eptr = entity->data.array.array ;
  while( count > 0 ){
    free_entity( *eptr ) ;
    eptr ++ ;
    count -- ;
  }
  return NULL ;
}

/*
 * Ϳ줿ʸ(٥ȥ롩)ȹͤơΤõؿ
 */
static struct skkinputlisp_entity *stringIsArray
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct conslist *top ;
  struct skkinputlisp_entity *entity ;

  if( !IS_ASCII_EQUAL( *string, '[' ) )
    string ++ ;
  if( ( top = string2conslist( &string ) ) == NULL )
    return NULL ;
#if defined(DEBUG)
  print_conslist( top ) ;
  fprintf( stdout, "\n" ) ;
#endif
  entity = conslist2array( top->left.value.next, localVariableListTop ) ;
  free_conslist( top ) ;
  return entity ;
}

/*
 * Ϳ줿ʸѿ̾ȹͤơΤõؿ
 */
static struct skkinputlisp_entity *stringIsVariable
( struct myChar *string,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_variable *node ;

  /* t  nil Ф atom äĤޤäͤ*/
  if( IS_ASCII_EQUAL( string[ 0 ], 't' ) &&
      IS_END_OF_STRING( string[ 1 ] ) ){
    return tEntity() ;
  } else if( isNilStringP( string ) ){
    return nilEntity() ;
  } else if( isIntegerP( string ) ){
    return stringIsInteger
      ( string, localVariableListTop ) ;
  } else if( isStringP( string ) ){
    return stringIsString
      ( string, localVariableListTop ) ;
  } else if( isArrayP( string ) ){
    return stringIsArray
      ( string, localVariableListTop ) ;
  }
  /* ޤƤѿꥹȤõ롣*/
  node = search_localVariable( string, localVariableListTop ) ;
  if( node == NULL ){
    node = search_globalVariable( string ) ;
  }
  /* ǡºݤƤѿǤ */
  if( node == NULL ){
#if defined(DEBUG)
    fprintf
      ( stdout, "Symbol's value as variable is void: " ) ;
    myCharFputString( stdout, string ) ;
    fprintf( stdout, "\n" ) ;
    fflush( stdout ) ;
#endif
    return NULL ;
  }
  /* ڥեǤ setq ʤ󤫤äˤ void ʤ *
   * ƸäƤϤʤΤĴؿåΤȤդǤ */
  return node->value ;
}


/*
 * ηݤʤõδؿ
 */
struct skkinputlisp_function *search_skkinputlispfunction
( struct myChar *name )
{
  struct skkinputlisp_function *node ;
  if( ( node = skkinputlispFunctionTable ) == NULL )
    return NULL ;
  while( node->name != NULL ){
    if( !myCharCharStrcmp( name, node->name ) )
      return node ;
    node ++ ;
  }
  return NULL ;
}

/*
 * 桼ƤؿõΤѤؿ
 */
static struct skkinputlisp_ufunction
*search_skkinputlispUserDefinedFunction( struct myChar *name )
{
  struct skkinputlisp_ufunction *node ;
  if( ( node = userDefinedFunctionListTop ) == NULL )
    return NULL ;
  while( node != NULL ){
    if( !myCharStrcmp( name, node->name ) )
      return node ;
    node = node->next ;
  }
  return NULL ;
}

static struct skkinputlisp_ufunction
 *create_newUserDefinedFunctionEntry( void )
{
  struct skkinputlisp_ufunction *node ;
  node = malloc( sizeof( struct skkinputlisp_ufunction ) ) ;
  if( node == NULL )
    return NULL ;
  return node ;
}

static void link_userDefinedFunctionEntry
( struct skkinputlisp_ufunction *node )
{
  node->next = userDefinedFunctionListTop ;
  userDefinedFunctionListTop = node ;
  return ;
}

/*
 * conslist ɾ¹ԤΤѤؿ
 */
struct skkinputlisp_entity *myevalConslist
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_function *node ;
  struct skkinputlisp_ufunction *unode ;

  /* conslist  car Ĵ٤롣conslist  car ؿ̾ˤʤ뤫顣*/
  if( top == NULL ){
    /* () ξˤ NULL ˤʤΤǡ*/
    return nilEntity() ;
  }
  if( top->left.type == TYPE_CONSLIST_CONSPAIR ){
    struct conslist *node = top->left.value.next ;
    /* lambda ɤȽꤹ롣*/
    if( node->left.type != TYPE_CONSLIST_STRING ||
	node->left.value.string == NULL )
      return NULL ;
    if( myCharCharStrcmp( node->left.value.string, "lambda" ) )
      return NULL ;
    /* lambda äɡġ*/
    return myevalLambdaForm( top, localVariableListTop ) ;
  } else if( top->left.type != TYPE_CONSLIST_STRING ){
    /* array äꤷ֤ʤ*/
    return NULL ;
  } else if( IS_END_OF_STRING( top->left.value.string[ 0 ] ) ){
    return nilEntity() ;
  }
  unode = search_skkinputlispUserDefinedFunction
    ( top->left.value.string ) ;
  if( unode != NULL ){
    free( top->left.value.string ) ;
    top->left.type = TYPE_CONSLIST_CONSPAIR ;
    /* ԡ˼Ԥˤ NULL ֤*/
    top->left.value.next = copy_conspair( unode->function ) ;
    if( top->left.value.next == NULL )
      return NULL ;
    /* copy ʬˤϡƤӽФ¦ conslist 뤳Ȥ
       ʤäƤȦġ*/
    /* ƵȤϤޤʤ ʬǼʬؿȤϡ 
     * Ƶοåɬפ뤬ġ*/
    return myevalConslist( top, localVariableListTop ) ;
  }
  node = search_skkinputlispfunction( top->left.value.string ) ;
  if( node == NULL ){
#if defined(DEBUG)
    fprintf
      ( stdout, "Symbol's value as function is void: %s\n", 
	top->left.value.string ) ;
#endif
    return NULL ;
  }
  return ( node->func )( top, localVariableListTop ) ;
}


/*
 * Ϳ줿ʸ󤬴ؿȹͤơΤؿ
 */
static struct skkinputlisp_entity *stringIsFunction
( struct myChar *string, 
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity = NULL ;
  struct conslist *top ;
#if defined(DEBUG)
  fprintf( stdout, "(string2conslist) %s\n", string ) ;
#endif
  if( !IS_ASCII_EQUAL( *string, 0x27 ) )
    string ++ ;
  top = string2conslist( &string ) ;
#if defined(DEBUG)
  print_conslist( top ) ;
  fprintf( stdout, "\n" ) ;
#endif
  entity = myevalConslist( top, localVariableListTop ) ;
  free_conslist( top ) ;
  return entity ;
}

static void free_local_entity( struct skkinputlisp_entity *entity )
{
  /* ˤäƽ㤦*/
  switch( entity->type ){
  case ENTITY_ATOM :
  case ENTITY_STRING :
    if( entity->data.string != NULL )
      free( entity->data.string ) ;
    break ;
  case ENTITY_CONS :
    break ;
  case ENTITY_INTEGER :
  default :
    break ;
  }
  /* ⤷οͤ顢οͤ˼ʬȴ뤫鼫ʬμοͤ
     Τȶ롣*/
  if( entity->prev != NULL ){
    entity->prev->next = entity->next ;
  } else {
    /* οͤʤȤȤϥܥƬäΤ顢
       Ƭοͤޤ롣*/
    entityListTop = entity->next ;
  }
  /* ⤷˿ͤ顢ܥȴ뤫麣ܥοͤ
     οͤˤʤΤȶ롣*/
  if( entity->next != NULL ){
    entity->next->prev = entity->prev ;
  }
  free( entity ) ;
  return ;
}

/*
 * Ѥؿ
 */
static void free_entity_array( struct skkinputlisp_entity *entity )
{
  struct skkinputlisp_entity **node ;
  int i ;
  node = entity->data.array.array ;
  for( i = 0 ; i < entity->data.array.length ; i ++, node ++ ){
    free_entity( *node ) ;
  }
  return ;
}

/*
 * ΤѤؿ
 */
static void free_entity( struct skkinputlisp_entity *entity )
{
  if( entity == NULL )
    return ;
  if( entity->type == ENTITY_CONS ){
    free_entity( entity->data.conspair.left  ) ;
    free_entity( entity->data.conspair.right ) ;
  } else if( entity->type == ENTITY_ARRAY ){
    free_entity_array( entity ) ;
  }
  /* ʬؤƤ륫󥿤򸺤餹*/
  if( entity->count > 0 ){
    entity->count -- ;
  }
  if( entity->count <= 0 ){
    free_local_entity( entity ) ;
  }
  return ;
}

static void free_unreferred_entity
( struct skkinputlisp_entity *entity )
{
  if( entity == NULL )
    return ;
  if( entity->count > 0 )
    return ;
  if( entity->type == ENTITY_CONS ){
    /* 餫˼ʬȤõˤҶ򸺤餵ʤ*/
    free_entity( entity->data.conspair.left  ) ;
    free_entity( entity->data.conspair.right ) ;
  } else if( entity->type == ENTITY_ARRAY ){
    free_entity_array( entity ) ;
  }
  free_local_entity( entity ) ;
}

/*
 * ʸΤľΤѤؿ
 */
struct skkinputlisp_entity *string2entity
( struct myChar *string, 
  struct skkinputlisp_variable *localVariableListTop )
{
  /* NULL ä顢ɾΤ褦ʤ*/
  if( string == NULL )
    return NULL ;
  /* ʸƬˤʸ򥹥åפ롣*/
  skip_space( &string ) ;
#if defined(DEBUG)
  fprintf( stdout, "(string2entity) " ) ;
  myCharFputstring( string ) ;
  fputc( '\n', fp ) ;
  fflush( stdout ) ;
#endif
  if( IS_ASCII_EQUAL( *string, '(' ) ||
      IS_ASCII_EQUAL( *string, 0x27  ) ){
    /* ξˤϴؿǤ롣ɾʤȤʤ*/
    return stringIsFunction( string, localVariableListTop ) ;
  }
#if 0
  /* Ƭäˤϡ⤷ͤǤϤʤȵ*/
  if( isIntegerP( string ) ){
    return stringIsInteger( string, localVariableListTop ) ;
    /* ⤷ȡʸ󤸤ʤ */
  } else if( isStringP( string ) ){
    /* 衢ʸ衣*/
    return stringIsString( string, localVariableListTop ) ;
  }
#endif
  /* ξˤѿǤ롣ѿȤɾʤФʤʤ*/
  return stringIsVariable( string, localVariableListTop ) ;
}

/*
 * ߼򤷤Ƥؿ̵ŤĤϡġȴ᤮
 * ⡣
 */
void skkinputlisp_garbagecollector( void )
{
  struct skkinputlisp_entity *node, *nextNode ;
  int type ;

  node = entityListTop ;
  while( node != NULL ){
    if( node->count <= 0 ){
      type = node->type ;
      nextNode = node->next ;
      free_entity( node ) ;
      /* ⤷ȼοͤϤʤʤäƤǽäơ
	 οͤ⤤ʤʤäƤǽäơ⼫ʬ⤤ʤʤ
	 ƤȤġ*/
      if( type == ENTITY_CONS ){
	node = entityListTop ;
      } else {
	node = nextNode ;
      }
    } else {
      node = node->next ;
    }
  }
  return ;
}

static int isNilEntity( struct skkinputlisp_entity *node )
{
  if( node->type == ENTITY_ATOM &&
      !myCharCharStrcmp( node->data.string, "nil" ) )
    return True ;
  return False ;
}

static int isEntityListp( struct skkinputlisp_entity *entity )
{
  if( entity->type == ENTITY_CONS )
    return True ;
  if( isNilEntity( entity ) )
    return True ;
  return False ;
}

/*
 * conslist įƥꥹȤɤȽǤؿ
 * "Wrong type argument: listp, hogehoge" ȽѤ롣
 */
static int isConslistNodeListP
( struct consnode *node )
{
  if( node->type == TYPE_CONSLIST_CONSPAIR )
    return True ;
  /* nil ̤*/
  if( node->value.string == NULL )
    return True ;
  if( !myCharCharStrcmp( node->value.string, "nil" ) ||
      IS_END_OF_STRING( node->value.string[ 0 ] ) )
    return True ;
  return False ;
}

#if 0
static int isConslistNodeSequenceP
( struct consnode *node )
{
  if( node->type == TYPE_CONSLIST_CONSPAIR ||
      node->type == TYPE_CONSLIST_ARRAY )
    return True ;
  /* nil ̤*/
  if( node->value.string == NULL )
    return True ;
  if( !strcmp( node->value.string, "nil" ) ||
      node->value.string[ 0 ] == '\0' )
    return True ;
  return False ;
}
#endif

/*
 *
 */
static struct skkinputlisp_entity *conslist2conspair
( struct conslist *node,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *ret ;

  if( ( entity = create_newEntity() ) == NULL )
    return NULL ;
  entity->type = ENTITY_CONS ;
  entity->data.conspair.left = entity->data.conspair.right = NULL ;
  /* ޤ꤫顣*/
  if( node->left.type == TYPE_CONSLIST_STRING ){
    entity->data.conspair.left =
      stringIsAtom( node->left.value.string, localVariableListTop ) ;
  } else {
    if( node->left.value.next != NULL ){
      entity->data.conspair.left =
	conslist2conspair( node->left.value.next, localVariableListTop ) ;
    } else {
      entity->data.conspair.left = nilEntity() ;
    }
  }
  /* ݼԡ*/
  if( entity->data.conspair.left == NULL ){
    free_entity( entity ) ;
    return NULL ;
  }
  /* cons ξϻؤƤ뤳ȤΤˤʤʤȤʤ */
  increment_entity_count( entity->data.conspair.left ) ;
  if( node->right.type == TYPE_CONSLIST_STRING ){
    entity->data.conspair.right =
      stringIsAtom( node->right.value.string, localVariableListTop ) ;
  } else {
    if( node->right.value.next != NULL ){
      entity->data.conspair.right =
	conslist2conspair( node->right.value.next, localVariableListTop ) ;
    } else {
      entity->data.conspair.right = nilEntity() ;
    }
  }
  /* ݼԡ*/
  if( entity->data.conspair.right == NULL ){
    free_entity( entity ) ;
    return NULL ;
  }
  /* cons ξϻؤƤ뤳ȤΤˤʤʤȤʤ */
  increment_entity_count( entity->data.conspair.right ) ;

  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( ret = search_entity( entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
    return entity ;
  }
  /* פˤʤäΡɤ롣󥯤뤫顢Τ˺
     ʤ*/
  free_entity( entity ) ;
  /* դäĤϤȻؤ褦ˤʤȤʤ*/
  increment_entity_count( ret ) ;
  return ret ;
}

/*
 * ()  car  eval ؿ
 */
static struct skkinputlisp_entity *skkinputlisp_eval_car
( struct conslist *node,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity ;

  if( node == NULL )
    return nilEntity() ;

  if( node->left.type == TYPE_CONSLIST_STRING ){
    entity = stringIsVariable
      ( node->left.value.string, localVariableListTop ) ;
  } else if( node->left.type == TYPE_CONSLIST_ARRAY ){
    entity = conslist2array
      ( node->left.value.next, localVariableListTop ) ;
  } else {
    /* cdr  car ǤϽ eval 롣*/
    entity = myevalConslist
      ( node->left.value.next, localVariableListTop ) ;
  }
  return entity ;
}

/*
 * (cdr node)  sequence ɤȽꤹؿ
 *----
 * sequence ǤȤΤϡꥹȤǤ뤫Ǥ뤫Τ줫ξ
 * ­Ǥ롣 
 */
static int skkinputlisp_sequencepCheck
( struct conslist *node )
{
  if( !isConslistNodeListP( &node->right ) && 
      node->right.type != TYPE_CONSLIST_ARRAY ){
#if defined(DEBUG)
    fprintf
      ( stdout,
	"Wrong type argument: sequencep, %s\n",
	node->right.value.string ) ;
#endif      
    return False ;
  }
  return True ;
}

/*
 * (cdr node)  list ɤȽꤹؿ
 *----
 * list ǤȤΤϡconspair Ǥ뤫 nil Ǥ뤫Τ줫ξ
 * ­Ǥ롣
 */
static int skkinputlisp_listpCheck
( struct conslist *node )
{
  if( !isConslistNodeListP( &node->right ) ){
#if defined(DEBUG)
    fprintf
      ( stdout,
	"Wrong type argument: listp, %s\n",
	node->right.value.string ) ;
#endif      
    return False ;
  }
  return True ;
}

static struct skkinputlisp_entity *skkinputlisp_evalCarAndCheckType
( struct conslist *node, 
  struct skkinputlisp_variable *localVariableListTop,
  int type )
{
  struct skkinputlisp_entity *entity ;
#if defined(DEBUG)
  const char *err_msgs[] = {
    "atomp", "stringp", "integerp", "listp", "arrayp", "listp", 
  } ;
#endif
  /* ɾƤ*/
  if( ( entity = skkinputlisp_eval_car
	( node, localVariableListTop ) ) == NULL ){
    return NULL ;
  }
  switch( type ){
  case ENTITY_LIST :
    if( entity->type != ENTITY_CONS && !isNilEntity( entity ) )
      goto type_error ;
    break ;
  default :
    /* Υå*/
    if( entity->type != type )
      goto type_error ;
    break ;
  }
  return entity ;

type_error:
#if defined(DEBUG)
  if( type >= 0 )
    fprintf
      ( stdout, "Wrong type argument: %s, ", err_msgs[ type ] ) ;
  print_entity( entity ) ;
  fprintf( stdout, "\n" ) ;
#endif    
  free_unreferred_entity( entity ) ;
  return NULL ;
}

/*
 * conslist ηǼͿ줿˰οؿ
 */
static int count_argument_number( struct conslist *node )
{
  int i = 0 ;
  while( node != NULL ){
    if( node->right.type != TYPE_CONSLIST_CONSPAIR ){
      if( node->right.value.string == NULL ||
	  IS_END_OF_STRING( node->right.value.string[ 0 ] ) )
	break ;
    }
    i ++ ;
    node = node->right.value.next ;
  }
  return i ;
}

/*
 * 桼줿ؿϿΤѤؿ
 *----
 * ʴؿäƤΤ Ѳ
 * ƤޤΤġ
 */
static int skkinputlisp_registerUserDefinedFunction
( struct myChar *name, struct conslist *top )
{
  struct skkinputlisp_ufunction *fnode ;
  fnode = search_skkinputlispUserDefinedFunction( name ) ;
  if( fnode == NULL ){
    /* Ͽξ硣*/
    if( ( fnode = create_newUserDefinedFunctionEntry() ) == NULL )
      return False ;
    /* ̾ФƤ*/
    fnode->name = malloc
      ( sizeof( struct myChar ) * ( myCharStrlen( name ) + 1 ) ) ;
    if( fnode->name == NULL ){
      free( fnode ) ;
      return False ;
    }
    myCharStrcpy( fnode->name, name ) ;
    link_userDefinedFunctionEntry( fnode ) ;
  } else {
    /* ʤơˤΤ񤭤硣*/
    free_conslist( fnode->function ) ;
  }
  fnode->function = top ;
  return True ;
}

/*
 * ޥǤȤ quote ̤˼¹Ԥؿ
 *---
 * defmacro ̵ΤǡȤ򤷤Ƥ롣defmacro 
 * Τ 
 */
static struct skkinputlisp_entity *skkinputlisp_quote
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity ;
  struct conslist *node ;

  if( top->right.type != TYPE_CONSLIST_CONSPAIR ||
      top->right.value.next == NULL ){
#if defined(DEBUG)
    fprintf( stdout, "Wrong number of arguments: #<subr quote>, 0\n" ) ;
#endif
    return NULL ;
  }
  node = ( top->right.value.next ) ;
  if( node->left.type == TYPE_CONSLIST_CONSPAIR ){
    if( node->left.value.next == NULL ){
      entity = nilEntity() ;
    } else {
      entity = conslist2conspair
	( node->left.value.next, localVariableListTop ) ;
    }
  } else if( node->left.type == TYPE_CONSLIST_ARRAY ){
    entity = conslist2array
      ( node->left.value.next, localVariableListTop ) ;
  } else {
    /* atom 롣*/
    entity = stringIsAtom
      ( node->left.value.string, localVariableListTop ) ;
  }
  return entity ;
}

/*
 * (car) ¸ؿ
 */
static struct skkinputlisp_entity *skkinputlisp_car
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity = NULL ;
  struct conslist *node ;

  if( top->right.type != TYPE_CONSLIST_CONSPAIR ||
      top->right.value.next == NULL ){
#if defined(DEBUG)
    fprintf( stdout, "Wrong number of arguments: #<subr car>, 0\n" ) ;
#endif
    return NULL ;
  }
  node = top->right.value.next ;
  while( node != NULL ){
    entity = skkinputlisp_eval_car
      ( node, localVariableListTop ) ;
    if( entity == NULL )
      return NULL ;
    if( !isEntityListp( entity ) ){
#if defined(DEBUG)
      fprintf
	( stdout, "Wrong type argument: listp, " ) ;
      print_entity( entity ) ;
      fprintf( stdout, "\n" ) ;
#endif
      free_unreferred_entity( entity ) ;
      return NULL ;
    }
    node = node->right.value.next ;
    if( node != NULL ){
#if defined(DEBUG)
      fprintf( stdout, "Wrong number of arguments: #<subr car>, 2\n" ) ;
#endif
      free_unreferred_entity( entity ) ;
      return NULL ;
    } else {
      break ;
    }
  }
  /*
   *  nil ǤäΤˡcons ɤȽꤷƤ롣
   */
  return ( ( entity->type != ENTITY_CONS )? entity : 
    ( entity->data.conspair.left ) ) ;
}

/*
 * (cdr) ¸ؿ
 */
static struct skkinputlisp_entity *skkinputlisp_cdr
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity = NULL ;
  struct conslist *node ;

  if( top->right.type != TYPE_CONSLIST_CONSPAIR ||
      top->right.value.next == NULL ){
#if defined(DEBUG)
    fprintf( stdout, "Wrong number of arguments: #<subr cdr>, 0\n" ) ;
#endif
    return NULL ;
  }
  node = top->right.value.next ;
  while( node != NULL ){
    entity = skkinputlisp_eval_car
      ( node, localVariableListTop ) ;
    if( entity == NULL )
      return NULL ;
    if( !isEntityListp( entity ) ){
#if defined(DEBUG)
      fprintf
	( stdout, "Wrong type argument: listp, " ) ;
      print_entity( entity ) ;
      fprintf( stdout, "\n" ) ;
#endif
      free_unreferred_entity( entity ) ;
      return NULL ;
    }
    node = node->right.value.next ;
    if( node != NULL ){
#if defined(DEBUG)
      fprintf( stdout, "Wrong number of arguments: #<subr cdr>, 2\n" ) ;
#endif
      free_unreferred_entity( entity ) ;
      return NULL ;
    } else {
      break ;
    }
  }
  /*
   *  nil ǤäΤˡcons ɤȽꤷƤ롣
   */
  return ( ( entity->type != ENTITY_CONS )? entity : 
	   ( entity->data.conspair.right ) ) ;
}

/*
 * (cons) ¸ؿ
 */
static struct skkinputlisp_entity *skkinputlisp_cons
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *lEntity, *rEntity, *tmpEntity ;
  struct conslist *node ;
  int i ;

  node = top ;
  i = 0 ;
  while( node != NULL ){
    if( node->right.type != TYPE_CONSLIST_CONSPAIR ||
	node->right.value.next == NULL )
      break ;
    i ++ ;
    node = node->right.value.next ;
  }
  if( i != 2 ){
#if defined(DEBUG)
    fprintf
      ( stdout, "Wrong number of arguments: #<subr cons>, %d\n", i ) ;
#endif
    return NULL ;
  }
  node = top->right.value.next ;
  lEntity = skkinputlisp_eval_car
    ( node, localVariableListTop ) ;
  node = node->right.value.next ;
  rEntity = skkinputlisp_eval_car
    ( node, localVariableListTop ) ;
  if( lEntity == NULL || rEntity == NULL ){
    goto skkinputlisp_cons_error ;
  }
  if( ( entity = create_newEntity() ) == NULL ){
    goto skkinputlisp_cons_error ;
  }
  entity->type = ENTITY_CONS ;
  entity->data.conspair.left  = lEntity ;
  increment_entity_count( lEntity ) ;
  entity->data.conspair.right = rEntity ;
  increment_entity_count( rEntity ) ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( tmpEntity = search_entity( entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( entity ) ;
    entity = tmpEntity ;
  }
  return ( entity ) ;

skkinputlisp_cons_error:
  free_unreferred_entity( lEntity ) ;
  free_unreferred_entity( rEntity ) ;
  return NULL ;
}

/*
 * (set) ¸ؿ
 */
static struct skkinputlisp_entity *skkinputlisp_set
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity ;
  struct skkinputlisp_variable *varnode ;
  struct conslist *node ;
  int argc ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  /* οå롣*/
  argc = count_argument_number( top->right.value.next ) ;
  if( argc != 2 ){
#if defined(DEBUG)
    fprintf
      ( stdout, "Wrong number of arguments: #<subr set>, %d\n", argc ) ;
#endif    
    return NULL ;
  }
  node = top->right.value.next ;
  entity = skkinputlisp_eval_car( node, localVariableListTop ) ;
  if( entity == NULL )
    return NULL ;
  if( entity->type != ENTITY_ATOM ){
#if defined(DEBUG)
    fprintf
      ( stdout, "Wrong type argument: symbolp, " ) ;
    print_entity( entity ) ;
    fprintf
      ( stdout, "\n" ) ;
#endif
    free_unreferred_entity( entity ) ;
    return NULL ;
  }
  /* ޤƤѿꥹȤõ롣*/
  varnode = search_localVariable
    ( entity->data.string, localVariableListTop ) ;
  if( varnode == NULL ){
    varnode = search_globalVariable( entity->data.string ) ;
  }
  if( varnode == NULL ){
    if( ( varnode = create_newVariable() ) == NULL )
      return NULL ;
    varnode->name = malloc
      ( sizeof( struct myChar ) *
	( myCharStrlen( entity->data.string ) + 1 ) ) ;
    if( varnode->name == NULL ){
      free( varnode ) ;
      return NULL ;
    }
    myCharStrcpy( varnode->name, entity->data.string ) ;
  }
  free_unreferred_entity( entity ) ;
  node = node->right.value.next ;
  entity = skkinputlisp_eval_car( node, localVariableListTop ) ;
  if( entity == NULL )
    return NULL ;
  link_variable_to_entity( varnode, entity ) ;
  return entity ;
}

/*
 * (setq) ¸ؿ
 */
static struct skkinputlisp_entity *skkinputlisp_setq
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity = NULL ;
  struct skkinputlisp_variable *varnode ;
  struct conslist *node ;

  if( top->right.type != TYPE_CONSLIST_CONSPAIR ||
      top->right.value.next == NULL ){
    return nilEntity() ;
  }
  node = top->right.value.next ;
  while( node != NULL ){
    if( node->left.type != TYPE_CONSLIST_STRING ){
      return NULL ;
    } else {
      /* ޤƤѿꥹȤõ롣*/
      varnode = search_localVariable
	( node->left.value.string, localVariableListTop ) ;
      if( varnode == NULL ){
	varnode = search_globalVariable( node->left.value.string ) ;
      }
      if( varnode == NULL ){
	if( ( varnode = create_newVariable() ) == NULL )
	  return NULL ;
	varnode->name = node->left.value.string ;
	node->left.value.string = NULL ;
      }
    }
    if( node->right.type == TYPE_CONSLIST_STRING ){
      entity = nilEntity() ;
      link_variable_to_entity( varnode, entity ) ;
      return entity ;
    }
    node = node->right.value.next ;
    entity = skkinputlisp_eval_car( node, localVariableListTop ) ;
    if( entity == NULL )
      return NULL ;
    link_variable_to_entity( varnode, entity ) ;
    if( node->right.type != TYPE_CONSLIST_STRING ){ 
      node = node->right.value.next ;
    } else {
      free_unreferred_entity( entity ) ;
      return NULL ;
    }
  }
  return entity ;
}

/*
 * (and ...) ¸ؿ
 *----
 * ٤ưޤǤϳǧǤƤʤ褪...
 */
static struct skkinputlisp_entity *skkinputlisp_and
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity = NULL ;
  struct conslist *node ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  } else if( top->right.type == TYPE_CONSLIST_STRING ||
	     top->right.value.next == NULL ){
    return tEntity() ;
  }
  node = top->right.value.next ;
  while( node != NULL ){
    if( entity != NULL )
      free_unreferred_entity( entity ) ;

    entity = skkinputlisp_eval_car( node, localVariableListTop ) ;
    if( entity == NULL )
      return NULL ;
    if( node->right.type != TYPE_CONSLIST_CONSPAIR ){
#if defined(DEBUG)
      fprintf
	( stdout,
	  "Wrong type argument: listp, %s\n",
	  node->right.value.string ) ;
#endif      
      free_unreferred_entity( entity ) ;
      return NULL ;
    }
    /* nil ʤߤ롣*/
    if( isNilEntity( entity ) ){
      return entity ;
    }
    node = node->right.value.next ;
  }
  return entity ;
}

/*
 * (or) ¸ؿ
 */
static struct skkinputlisp_entity *skkinputlisp_or
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity = NULL ;
  struct conslist *node ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  } else if( top->right.type == TYPE_CONSLIST_STRING || 
	     top->right.value.next == NULL ){
    return nilEntity() ;
  }
  node = top->right.value.next ;
  while( node != NULL ){
    entity = skkinputlisp_eval_car( node, localVariableListTop ) ;
    if( entity == NULL )
      return NULL ;
    if( node->right.type != TYPE_CONSLIST_CONSPAIR ){
#if defined(DEBUG)
      fprintf
	( stdout,
	  "Wrong type argument: listp, %s\n",
	  node->right.value.string ) ;
#endif      
      free_unreferred_entity( entity ) ;
      return NULL ;
    }
    /* nil ʤߤ롣*/
    if( !isNilEntity( entity ) ){
      return entity ;
    }
    free_unreferred_entity( entity ) ;
    node = node->right.value.next ;
  }
  return nilEntity() ;
}

/*
 * cond ¸ؿ
 * (cond ( ... ) ... ( ... )) ȤˤʤäƤơ硹˥ꥹȤɾ
 * 롣
 */
static struct skkinputlisp_entity *skkinputlisp_cond
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity = NULL ;
  struct conslist *node, *lnode ;

  if( top->right.type != TYPE_CONSLIST_CONSPAIR ){
#if defined(DEBUG)
    fprintf
      ( stdout,
	"Wrong type argument: listp, %s\n",
	top->right.value.string ) ;
#endif      
    return NULL ;
  }
#if 0
  else if( top->right.value.next == NULL ){
    /*
     * (cond . nil)  nil ֤
     */ 
    return nilEntity() ;
  }
#endif
  node = top->right.value.next ;
  while( node != NULL ){
    /* ǤꥹȤɾƤ*/
    if( !isConslistNodeListP( &node->left ) ){
#if defined(DEBUG)
      fprintf
	( stdout,
	  "Wrong type argument: listp, %s\n",
	  node->left.value.string ) ;
#endif      
      return NULL ;
    }
    if( node->left.type == TYPE_CONSLIST_CONSPAIR ){
      lnode = node->left.value.next ;
      /* ɾƤ*/
      entity = skkinputlisp_eval_car
	( lnode, localVariableListTop ) ;
      /* ɾˤȤȤԤޤ */
      if( entity == NULL )
	return NULL ;
      /* ɾͤޤ */
      if( !isNilEntity( entity ) ){
	if( !isConslistNodeListP( &lnode->right ) ){
#if defined(DEBUG)
	  fprintf
	    ( stdout,
	      "Wrong type argument: listp, %s\n",
	      lnode->right.value.string ) ;
#endif
	  free_unreferred_entity( entity ) ;
	  return NULL ;
	}
	/* cond ΰǤꥹȤ硹ɾƤ*/
	while( lnode->right.value.next != NULL ){
	  entity = skkinputlisp_eval_car
	    ( lnode->right.value.next, localVariableListTop ) ;
	  lnode = lnode->right.value.next ;
	}
	/* ֺǸη̤֤롣*/
	return entity ;
      }
      free_unreferred_entity( entity ) ;
    } else {
      /* ξˤϤɤ nil ʤΤǡɾ̵̤Ǥ롣*/
    }
    /* cond μΰɾ롣*/
    if( node->right.type != TYPE_CONSLIST_CONSPAIR ){
#if defined(DEBUG)
      fprintf
	( stdout,
	  "Wrong type argument: listp, %s\n",
	  node->right.value.string ) ;
#endif      
      return NULL ;
    }
    node = node->right.value.next ;
  }
  return nilEntity() ;
}

/*
 * if ¸ؿ
 * (if a b c d) ȤˤʤäƤ롣
 */
static struct skkinputlisp_entity *skkinputlisp_if
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity = NULL ;
  struct conslist *node ;
  int argc ;

  if( !skkinputlisp_sequencepCheck( top ) )
    return NULL ;
  /* οå롣*/
  argc = count_argument_number( top->right.value.next ) ;
  if( argc < 2 ){
#if defined(DEBUG)
    fprintf
      ( stdout, "Wrong number of arguments: #<subr if>, %d\n", argc ) ;
#endif    
    return NULL ;
  }
  node = top->right.value.next ;
  /* ɾ롣*/
  entity = skkinputlisp_eval_car
    ( node, localVariableListTop ) ;
  /* ɾˤȤȤԤޤ */
  if( entity == NULL )
    return NULL ;
  node = node->right.value.next ;
  /* ơ ɤǤ */
  if( !isNilEntity( entity ) ){
    return skkinputlisp_eval_car( node, localVariableListTop ) ;
  } else {
    if( node->right.type == TYPE_CONSLIST_STRING || 
	node->right.value.next == NULL ){
      return nilEntity() ;
    } else {
      node = node->right.value.next ;
      return skkinputlisp_eval_car( node, localVariableListTop ) ;
    }
  }
  return nilEntity() ;
}

/*
 * ɤʬʤؿ
 *      (f2c) (j-stay)
 * Τɤʴؿ
 *      (j-input-by-code-or-menu) (j-clock)
 * ưפ˼ǽȻפؿ
 *      (skk-version) (j-date-and-time) (current-time-string)
 *      (j-date) (substring) (make-string)  (concat)
 */

static struct skkinputlisp_entity *skkinputlisp_concat
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *tmpEntity ;
  struct conslist *node ;
  struct myChar *string ;
  int length, slen ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  } else if( top->right.type == TYPE_CONSLIST_STRING || 
	     top->right.value.next == NULL ){
    return emptyStringEntity() ;
  }
  length = 0 ;
  string = NULL ;
  node = top->right.value.next ;
  while( node != NULL ){
    /* ɾƤ*/
    if( ( entity = skkinputlisp_eval_car
	  ( node, localVariableListTop ) ) == NULL ){
      goto myconcat_error ;
    }
    if( entity->type != ENTITY_STRING ){
#if defined(DEBUG)
      fprintf
	( stdout,
	  "Wrong type argument: sequencep, " ) ;
      print_entity( entity ) ;
      fprintf( stdout, "\n" ) ;
#endif
      goto myconcat_error ;
    }
    /* ʸ¸ߤʤСġ*/
    if( entity->data.string != NULL ){
      slen = myCharStrlen( entity->data.string ) ;
      if( string == NULL ){
	if( ( string = 
	      malloc( sizeof( struct myChar ) * ( slen + 1 ) ) ) == NULL )
	  goto myconcat_error ;
	length = slen ;
	MYCHAR_SET_END_OF_STRING( *string ) ;
      } else {
	struct myChar *prev_string = string ;
	string = malloc
	  ( sizeof( struct myChar ) * ( slen + length + 1 ) ) ;
	if( string == NULL ){
	  free( prev_string ) ;
	  goto myconcat_error ;
	}
	myCharStrcpy( string, prev_string ) ;
	free( prev_string ) ;
	length += slen ;
      }
      myCharStrcat( string, entity->data.string ) ;
      MYCHAR_SET_END_OF_STRING( string[ length ] ) ;
      free_unreferred_entity( entity ) ;
    }
    /* cond μΰɾ롣*/
    if( node->right.type != TYPE_CONSLIST_CONSPAIR ){
#if defined(DEBUG)
      fprintf
	( stdout, "Wrong type argument: listp, " ) ;
      myCharFputstring( node->right.value.string ) ;
      fputc( '\n', stdout ) ;
      fflush( stdout ) ;
#endif      
      goto myconcat_error ;
    }
    node = node->right.value.next ;
  }
  if( ( entity = create_newEntity() ) == NULL ){
    goto myconcat_error ;
  }
  entity->type = ENTITY_STRING ;
  entity->data.string = string ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( tmpEntity = search_entity( entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( entity ) ;
    entity = tmpEntity ;
  }
  return ( entity ) ;

myconcat_error:
  if( string != NULL )
    free( string ) ;
  free_unreferred_entity( entity ) ;
  return NULL ;
}

static struct skkinputlisp_entity *skkinputlisp_make_string
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *tmpEntity ;
  struct conslist *node ;
  struct myChar *ptr ;
  int argc, len, chara ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  argc = count_argument_number( top->right.value.next ) ;
  if( argc != 2 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr make-string>, %d\n", argc ) ;
#endif      
    return NULL ;
  }
  node = top->right.value.next ;
  /* ɾƤ*/
  if( ( entity = skkinputlisp_eval_car
	( node, localVariableListTop ) ) == NULL ){
    return NULL ;
  }
  if( entity->type != ENTITY_INTEGER ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong type argument: wholenump, " ) ;
    print_entity( entity ) ;
    fprintf( stdout, "\n" ) ;
#endif      
    goto mymakestring_error ;
  }
  len = entity->data.integer ;
  free_unreferred_entity( entity ) ;

  node = node->right.value.next ;
  /* ɾƤ*/
  if( ( entity = skkinputlisp_eval_car
	( node, localVariableListTop ) ) == NULL ){
    return NULL ;
  }
  if( entity->type != ENTITY_INTEGER ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong type argument: integerp, " ) ;
    print_entity( entity ) ;
    fprintf( stdout, "\n" ) ;
#endif      
    goto mymakestring_error ;
  }
  chara = entity->data.integer ;
  free_unreferred_entity( entity ) ;

  if( ( entity = create_newEntity() ) == NULL ){
    goto mymakestring_error ;
  }
  entity->type = ENTITY_STRING ;
  entity->data.string = malloc
    ( ( len + 1 ) * sizeof( unsigned char ) ) ;
  if( entity->data.string == NULL )
    goto mymakestring_error ;
  ptr = entity->data.string ;
  while( len > 0 ){
    MYCHAR_SET_CHARA( *ptr, chara ) ;
    ptr ++ ;
    len -- ;
  }
  MYCHAR_SET_END_OF_STRING( *ptr ) ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( tmpEntity = search_entity( entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( entity ) ;
    entity = tmpEntity ;
  }
  return ( entity ) ;

mymakestring_error:
  free_unreferred_entity( entity ) ;
  return NULL ;
}

static struct skkinputlisp_entity *skkinputlisp_substring
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *tmpEntity ;
  struct conslist *node ;
  int argc, len, startpoint, endpoint ;

  entity = tmpEntity = NULL ;
  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  argc = count_argument_number( top->right.value.next ) ;
  if( argc != 2 && argc != 3 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr substring>, %d\n", argc ) ;
#endif      
    return NULL ;
  }
  node = top->right.value.next ;
  /* ɾƤ*/
  if( ( entity = skkinputlisp_eval_car
	( node, localVariableListTop ) ) == NULL ){
    return NULL ;
  }
  if( entity->type != ENTITY_STRING ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong type argument: stringp, " ) ;
    print_entity( entity ) ;
    fprintf( stdout, "\n" ) ;
#endif      
    goto mysubstring_error ;
  }
  if( entity->data.string != NULL ){
    len = myCharStrlen( entity->data.string ) ;
  } else {
    len = 0 ;
  }
  node = node->right.value.next ;
  /* ɾƤ*/
  if( ( tmpEntity = skkinputlisp_eval_car
	( node, localVariableListTop ) ) == NULL ){
    return NULL ;
  }
  if( tmpEntity->type != ENTITY_INTEGER ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong type argument: integerp, " ) ;
    print_entity( tmpEntity ) ;
    fprintf( stdout, "\n" ) ;
#endif      
    goto mysubstring_error ;
  }
  startpoint = tmpEntity->data.integer ;
  if( startpoint > len || startpoint < 0 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Args out of range: \"%s\", %d\n",
	entity->data.string, startpoint ) ;
#endif      
    goto mysubstring_error ;
  }
  free_unreferred_entity( tmpEntity ) ;
  if( argc == 3 ){
    node = node->right.value.next ;
    /* ɾƤ*/
    if( ( tmpEntity = skkinputlisp_eval_car
	  ( node, localVariableListTop ) ) == NULL ){
      return NULL ;
    }
    if( tmpEntity->type != ENTITY_INTEGER ){
#if defined(DEBUG)
      fprintf
	( stdout, 
	  "Wrong type argument: integerp, " ) ;
      print_entity( tmpEntity ) ;
      fprintf( stdout, "\n" ) ;
#endif      
      goto mysubstring_error ;
    }
    endpoint = tmpEntity->data.integer ;
    if( len < endpoint || startpoint > endpoint ){
#if defined(DEBUG)
      fprintf
	( stdout, 
	  "Args out of range: \"%s\", %d, %d\n",
	  entity->data.string, startpoint, tmpEntity->data.integer ) ;
#endif      
      goto mysubstring_error ;
    }
  } else {
    endpoint = len ;
  }
  if( ( tmpEntity = create_newEntity() ) == NULL ){
    goto mysubstring_error ;
  }
  tmpEntity->type = ENTITY_STRING ;
  tmpEntity->data.string = malloc
    ( ( len + 1 ) * sizeof( struct myChar ) ) ;
  if( tmpEntity->data.string == NULL )
    goto mysubstring_error ;
  myCharStrncpy
    ( tmpEntity->data.string, entity->data.string + startpoint,
      endpoint - startpoint ) ;
  MYCHAR_SET_END_OF_STRING
    ( tmpEntity->data.string[ endpoint - startpoint ] ) ;
  free_unreferred_entity( entity ) ;

  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( entity = search_entity( tmpEntity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( tmpEntity ) ;
    entity = tmpEntity ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( tmpEntity ) ;
  }
  return ( entity ) ;

mysubstring_error:
  free_unreferred_entity( tmpEntity ) ;
  free_unreferred_entity( entity ) ;
  return NULL ;
}

/*
 * (current-time-string) ¹Ԥؿ
 *----
 * current-time-string  emacs-lisp ¸ߤؿǤꡢ
 * Ȥ뤳ȤƤ롣ΰɬ (a . b) η conspair 
 * ʤäƤʤФʤʤ1970/01/01 09:00:00 ηвÿ 
 * (a * 65536 + b) ηǻꤷƤΤǤ롣άȡߤޤǤ
 * вÿͿȤˤʤ롣
 */
static struct skkinputlisp_entity *skkinputlisp_current_time_string
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *tmpEntity ;
  struct conslist *node ;
  unsigned char *string ;
  int argc, len ;
  time_t secs ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  entity = NULL ;
  node = top->right.value.next ;
  argc = count_argument_number( node ) ;
  if( argc > 1 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr current-time-string>, %d\n",
	argc ) ;
#endif      
    return NULL ;
  }
  if( argc == 0 ){
    /* 1970/01/01 09:00:00 ηв֤롣*/
    secs = time( NULL ) ;
  } else {
    /* ɾƤ*/
    if( ( entity = skkinputlisp_eval_car
	  ( node, localVariableListTop ) ) == NULL ){
      return NULL ;
    }
    if( entity->type != ENTITY_CONS ){
#if defined(DEBUG)
      fprintf
	( stdout, "Wrong type argument: listp, " ) ;
      print_entity( entity ) ;
      fprintf( stdout, "\n" ) ;
#endif      
      free_unreferred_entity( entity ) ;
      return NULL ;
    }
    if( entity->data.conspair.left == NULL ||
	entity->data.conspair.left->type != ENTITY_INTEGER ){
#if defined(DEBUG)
      fprintf
	( stdout, "Wrong type argument: integerp, " ) ;
      print_entity( entity->data.conspair.left ) ;
      fprintf( stdout, "\n" ) ;
#endif      
      free_unreferred_entity( entity ) ;
      return NULL ;
    }
    if( entity->data.conspair.right == NULL ||
	entity->data.conspair.right->type != ENTITY_INTEGER ){
#if defined(DEBUG)
      fprintf
	( stdout, "Wrong type argument: integerp, " ) ;
      print_entity( entity->data.conspair.right ) ;
      fprintf( stdout, "\n" ) ;
#endif      
      free_unreferred_entity( entity ) ;
      return NULL ;
    }
    secs = entity->data.conspair.left->data.integer * 0x10000 +
      entity->data.conspair.right->data.integer ;
  }
  free_unreferred_entity( entity ) ;
  if( ( entity = create_newEntity() ) == NULL )
    return NULL ;
  entity->type = ENTITY_STRING ;
  string = ctime( &secs ) ;
  len    = strlen( string ) ;
  entity->data.string =
    malloc( sizeof( struct myChar ) * ( len + 1 ) ) ;
  if( entity->data.string == NULL ){
    free_unreferred_entity( entity ) ;
    return NULL ;
  }
  myCharCharStrcpy( entity->data.string, string ) ;
  if( IS_ASCII_EQUAL( entity->data.string[ len - 1 ], '\n' ) )
    MYCHAR_SET_END_OF_STRING( entity->data.string[ len - 1 ] ) ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( tmpEntity = search_entity( entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( entity ) ;
    entity = tmpEntity ;
  }
  return ( entity ) ;
}

#if defined(MYEVAL_ONLY)
int j_date_ad( struct myChar *buffer, int bufsize, int date_ad, int num_style )
{
  return False ;
}
#endif

/*
 * skk ͿƤؿǤ롢j-date ¹Ԥؿ
 *----
 * j-date ˤİƤꡢ
 *     ǽΰϡǯ椫񤫡
 *     ΰϡ  ͤɽ뤫Ȥͤɽˡ
 */
static struct skkinputlisp_entity *skkinputlisp_j_date
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *entities[ 2 ], *tmpEntity ;
  struct conslist *node ;
  int argc, i, local_date_ad, local_number_style, len ;
  struct myChar buffer[ TEXTBUFSIZE ] ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  entity = NULL ;
  node = top->right.value.next ;
  argc = count_argument_number( node ) ;
  if( argc > 2 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr j-date>, %d\n", argc ) ;
#endif      
    return NULL ;
  }
  for( i = 0 ; i < argc && node != NULL ; i ++ ){
    /* ɾƤ*/
    if( ( entity = skkinputlisp_eval_car
	  ( node, localVariableListTop ) ) == NULL ){
      return NULL ;
    }
    entities[ i ] = entity ;
    node = node->right.value.next ;
  }
  if( argc > 0 ){
    if( !isNilEntity( entities[ 0 ] ) ){
      local_date_ad = True ;
    } else {
      local_date_ad = False ;
    }
  } else {
    local_date_ad = -1 ;
  }
  if( argc == 2 ){
    if( entities[ 1 ]->type != ENTITY_INTEGER ){
      local_number_style = 0 ;
    } else {
      local_number_style = entities[ 1 ]->data.integer ;
    }
  } else {
    local_number_style = -1 ;
  }
  for( i = 0 ; i < argc ; i ++ )
    free_unreferred_entity( entities[ i ] ) ;
  if( !j_date_ad( buffer, TEXTMAXLEN, local_date_ad, local_number_style ) )
    return NULL ;
  if( ( entity = create_newEntity() ) == NULL )
    return NULL ;
  entity->type = ENTITY_STRING ;
  len = myCharStrlen( buffer ) ;
  entity->data.string =
    malloc( sizeof( struct myChar ) * ( len + 1 ) ) ;
  if( entity->data.string == NULL ){
    free_unreferred_entity( entity ) ;
    return NULL ;
  }
  myCharStrcpy( entity->data.string, buffer ) ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( tmpEntity = search_entity( entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( entity ) ;
    entity = tmpEntity ;
  }
  return ( entity ) ;
}

/*
 * (skk-version)
 */
static struct skkinputlisp_entity *skkinputlisp_skk_version
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *tmpEntity ;
  struct conslist *node ;
  int argc, len ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  entity = NULL ;
  node = top->right.value.next ;
  argc = count_argument_number( node ) ;
  if( argc > 1 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr skk-version>, %d\n", argc ) ;
#endif      
    return NULL ;
  }
  if( ( entity = create_newEntity() ) == NULL )
    return NULL ;
  entity->type = ENTITY_STRING ;
  len = strlen( skkinput_version ) ;
  entity->data.string = 
    malloc( sizeof( struct myChar ) * ( len + 1 ) ) ;
  if( entity->data.string == NULL ){
    free_unreferred_entity( entity ) ;
    return NULL ;
  }
  myCharCharStrcpy( entity->data.string, skkinput_version ) ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( tmpEntity = search_entity( entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( entity ) ;
    entity = tmpEntity ;
  }
  return ( entity ) ;
}

/*
 * (aref  ֹ) ¸ΤѤؿ
 */
static struct skkinputlisp_entity *skkinputlisp_aref
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *ar_entity, *i_entity ;
  struct skkinputlisp_entity *entity, *tmpEntity ;
  struct conslist *node ;
  int argc ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  entity = NULL ;
  node = top->right.value.next ;
  argc = count_argument_number( node ) ;
  if( argc != 2 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr aref>, %d\n", argc ) ;
#endif      
    return NULL ;
  }
  /* ɾƤ*/
  if( ( ar_entity = skkinputlisp_evalCarAndCheckType
	( node, localVariableListTop, ENTITY_ARRAY ) ) == NULL ){
    return NULL ;
  }
  node = node->right.value.next ;
  /* ɾƤ*/
  if( ( i_entity = skkinputlisp_evalCarAndCheckType
	( node, localVariableListTop, ENTITY_INTEGER ) ) == NULL ){
    return NULL ;
  }
  if( ar_entity->data.array.length <= i_entity->data.integer ||
      i_entity->data.integer < 0 ){
#if defined(DEBUG)
    fprintf( stdout, "Args out of range: " ) ;
    print_entity( ar_entity ) ;
    fprintf( stdout, " " ) ;
    print_entity( i_entity ) ;
    fprintf( stdout, "\n" ) ;
#endif    
    free_unreferred_entity( ar_entity ) ;
    free_unreferred_entity( i_entity ) ;
    return NULL ;
  }
  entity = ar_entity->data.array.array[ i_entity->data.integer ] ;
  free_unreferred_entity( ar_entity ) ;
  free_unreferred_entity( i_entity ) ;
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( tmpEntity = search_entity( entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( entity ) ;
    entity = tmpEntity ;
  }
  return ( entity ) ;
}

/*
 * (aset  ֹ ) ¸ΤѤؿ
 */
static struct skkinputlisp_entity *skkinputlisp_aset
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *ar_entity, *i_entity ;
  struct skkinputlisp_entity *s_entity ;
  struct conslist *node ;
  int argc ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  node = top->right.value.next ;
  if( ( argc = count_argument_number( node ) ) != 3 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr aset>, %d\n", argc ) ;
#endif      
    return NULL ;
  }
  /* ɾƤ(ǽΰɾ̤ɬǤʤ
     Фʤʤ*/
  if( ( ar_entity = skkinputlisp_evalCarAndCheckType
	( node, localVariableListTop, ENTITY_ARRAY ) ) == NULL ){
    return NULL ;
  }
  /* ɾƤ(裲ɾ̤ɬǤʤ
     Фʤʤ*/
  node = node->right.value.next ;
  if( ( i_entity = skkinputlisp_evalCarAndCheckType
	( node, localVariableListTop, ENTITY_INTEGER ) ) == NULL ){
    return NULL ;
  }
  /* ǰ˻ꤵ줿ͤΤǤ뤫ɤ礳äȽ
     Ǥޤ*/
  if( ar_entity->data.array.length <= i_entity->data.integer ||
      i_entity->data.integer < 0 ){
#if defined(DEBUG)
    fprintf( stdout, "Args out of range: " ) ;
    print_entity( ar_entity ) ;
    fprintf( stdout, " " ) ;
    print_entity( i_entity ) ;
    fprintf( stdout, "\n" ) ;
#endif    
    free_unreferred_entity( ar_entity ) ;
    free_unreferred_entity( i_entity ) ;
    return NULL ;
  }
  /* ɾƤ(裳ϲǤ褤Ρ) */
  node = node->right.value.next ;
  if( ( s_entity = skkinputlisp_eval_car
	( node, localVariableListTop ) ) == NULL ){
    /* ξ֤װȤƤϡsymbol's value as ۤۤ 롣*/
    free_unreferred_entity( ar_entity ) ;
    free_unreferred_entity( i_entity ) ;
    return NULL ;
  }
  /* ȤȻؤƤΤ˴롣*/
  free_entity
    ( ar_entity->data.array.array[ i_entity->data.integer ] ) ;
  /* ؤ뤳ȤˤʤΤΥ󥿤䤹*/
  increment_entity_count( s_entity ) ;
  ar_entity->data.array.array[ i_entity->data.integer ] = s_entity ;
  /* ⤷äȤ⤳äݤäؤ뤳Ȥ̵Τʤ
     */
  free_unreferred_entity( ar_entity ) ;
  free_unreferred_entity( i_entity ) ;
  return ( s_entity ) ;
}

/*
 * (defun) ¹Ԥؿ
 *
 * (defun ܥ ( ) ̿(ȥ)  )
 */
static struct skkinputlisp_entity *skkinputlisp_defun
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *s_entity ;
  struct conslist *node ;
  int argc ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  node = top->right.value.next ;
  if( ( argc = count_argument_number( node ) ) < 2 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr defun>, %d\n", argc ) ;
#endif      
    return NULL ;
  }
  /* ηå裱ϥܥǤʤФʤʤ*/
  if( node->left.type != TYPE_CONSLIST_STRING )
    return NULL ;
  s_entity = stringIsAtom
    ( node->left.value.string, localVariableListTop ) ;
  if( s_entity == NULL ){
#if defined(DEBUG)
    fprintf( stdout, "Wrong type argument: symbolp, " ) ;
    print_entity( s_entity ) ;
    fprintf( stdout, "\n" ) ;
#endif      
    return NULL ;
  }
  /* ꥹȤ򤹤Ͽ롣ؿ̾ lambda ˺ؤ롣*/
  node = top->right.value.next ;
  if( node->left.value.string != NULL ){
    free( node->left.value.string ) ;
  }
  /* strlcn( "lambda" ) == 6 */
  node->left.value.string = malloc( sizeof( struct myChar ) * 7 ) ;
  if( node->left.value.string == NULL ){
    free_unreferred_entity( s_entity ) ;
    return NULL ;
  }
  myCharCharStrcpy( node->left.value.string, "lambda" ) ;
  if( !skkinputlisp_registerUserDefinedFunction
      ( s_entity->data.string, node ) ){
    free_unreferred_entity( s_entity ) ;
    return NULL ;
  }
  top->right.value.next = NULL ;
  return s_entity ;
}

/*
 * (lambda) ¹Ԥؿ
 *
 * (lambda ( ) ̿(ȥ)  )
 * ΰ֤ˤʤȻפΤĤ󡣤Τ̵
 * 뤹롩
 */
static struct skkinputlisp_entity *skkinputlisp_lambda
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  /* ³ list ǤʤȤʤ*/
  if( !skkinputlisp_listpCheck( top ) )
    return NULL ;
  /* lambda ϥߡؿʤΤǡ⤷ʤ*/
  return conslist2conspair( top, localVariableListTop ) ;
}

/*
 * ɽѿѰդؿ
 */
static struct skkinputlisp_variable *alloc_localVariable
( struct myChar *name, struct skkinputlisp_entity *entity )
{
  struct skkinputlisp_variable *vnode ;
  if( name == NULL || entity == NULL )
    return NULL ;
  /* γݡ*/
  vnode = malloc( sizeof( struct skkinputlisp_variable ) ) ;
  if( vnode == NULL )
    return NULL ;
  vnode->name = malloc
    ( sizeof( struct myChar ) * ( myCharStrlen( name ) + 1 ) ) ;
  if( vnode->name == NULL ){
    free( vnode ) ;
    return NULL ;
  }
  myCharStrcpy( vnode->name, name ) ;
  vnode->value = NULL ;
  /* ΤĤʤ*/
  link_variable_to_entity( vnode, entity ) ;
  return vnode ;
}

/*
 * ɽѿ˴ΤѤؿ
 */
static void destroy_localVariable
( struct skkinputlisp_variable *variableListTop )
{
  struct skkinputlisp_variable *varnode, *nextVarnode ;
  varnode = variableListTop ;
  while( varnode != NULL ){
    nextVarnode = varnode->next ;
    /* ѿ̾롣*/
    if( varnode->name != NULL )
      free( varnode->name ) ;
    /* Τ롣*/
    free_entity( varnode->value ) ;
    varnode = nextVarnode ;
  }
  return ;
}

/*
 * ա  let ˤϻȤʤlet Ϻ줿ͤ˱ƶ
 * 뤬ϤȤʤΤǡ
 */
static struct skkinputlisp_variable *create_localVariableForLambda
( struct conslist *varNode, struct conslist *valueNode,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_variable *newListTop, *vnode ;
  struct skkinputlisp_entity *entity ;

  newListTop = NULL ;
  while( varNode != NULL ){
    if( varNode->left.type != TYPE_CONSLIST_STRING )
      goto create_localv_error ;
    entity = skkinputlisp_eval_car( valueNode, localVariableListTop ) ;
    if( entity == NULL )
      goto create_localv_error ;
    vnode = alloc_localVariable
      ( varNode->left.value.string, entity ) ;
    if( vnode == NULL ){
      free_unreferred_entity( entity ) ;
      goto create_localv_error ;
    } else {
      vnode->next = newListTop ;
      newListTop = vnode ;
    }
    varNode = varNode->right.value.next ;
    valueNode = valueNode->right.value.next ;
  }
  return newListTop ;

create_localv_error:
  destroy_localVariable( newListTop ) ;
  return NULL ;
}

/*
 * (lambda) ¹Բ᤹ؿ
 *-----
 * ơ(빽ݤ)
 * (car (car top)) == "lambda"
 * (car (cdr (car top))) == "(Υꥹ)"
 * (cdr (cdr (car top))) == "(¹ԡɾ̿)"
 * (cdr top) == "(ΥꥹȤ٤)"
 * ޤ(cdr top)  (car (cdr (car top))) 򸫤ơ(cdr (cdr (car
 * top))) áޤʤФʤʤθϡ 
 * (cdr (cdr (car top)))  eval Ƥ
 */
static struct skkinputlisp_entity *myevalLambdaForm
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  int argcl, argca, argc1 ;
  struct conslist *node, *vNode ;
  struct skkinputlisp_variable *newLocalVariableListTop ;
  struct skkinputlisp_variable *pVarListNode ;
  struct skkinputlisp_entity *entity ;

  /* (lambda ... ) Ĺꤹ롣*/
  node = top->left.value.next ;
  argcl = count_argument_number( node->right.value.next ) ;
  argc1 = 0 ;
  /* (lambda ǽΰ ... ) ȺǽΰС lambda ؿ
     ؤΰΥꥹȤɽƤ롣*/
  vNode = NULL ;
  if( argcl > 0 ){
    node = node->right.value.next ;
    if( !isConslistNodeListP( &node->left ) ){
#if defined(DEBUG)
      fprintf( stdout, "Wrong type argument: listp, 1\n" ) ;
#endif
      return NULL ;
    } else if( node->left.type == TYPE_CONSLIST_CONSPAIR ){
      vNode = node->left.value.next ;
      /* ѿ˻äƤΤꤷƤ*/
      argc1 = count_argument_number( node->left.value.next ) ;
    }
  }
  /* lambda Ϥοꤹ롣*/
  if( !isConslistNodeListP( &top->right ) )
    return NULL ;
  if( top->right.type == TYPE_CONSLIST_CONSPAIR ){
    argca = count_argument_number( top->right.value.next ) ;
  } else {
    argca = 0 ;
  }
  /* οӤ롣*/
  if( argc1 != argca ){
#if defined(DEBUG)
    fprintf( stdout, "Wrong number of arguments: " ) ;
    print_conslist( top->left.value.next ) ;
    fprintf( stdout, ", %d\n", argca ) ;
#endif
    return NULL ;
  }
  if( argcl == 0 ){
    return nilEntity() ;
  }
  /* ɽѿѰդ롣̾ argc1 ˤΡμ() 
     argca ˤΤѤ롣*/
  pVarListNode = NULL ;
  if( vNode != NULL ){
    struct skkinputlisp_variable *tmpnode ;
    newLocalVariableListTop = create_localVariableForLambda
      ( vNode, top->right.value.next, localVariableListTop ) ;
    /* ɽѿѰդ˼Ԥν*/
    if( newLocalVariableListTop == NULL ){
#if defined(DEBUG)
      fprintf( stdout, "(create_local_variable) returns NULL.\n" ) ;
#endif
      return NULL ;
    }
    if( ( tmpnode = newLocalVariableListTop ) != NULL ){
      while( tmpnode->next != NULL )
	tmpnode = tmpnode->next ;
      pVarListNode = tmpnode ;
      tmpnode->next = localVariableListTop ;
    }
  } else {
    newLocalVariableListTop = localVariableListTop ;
  }
  /* ǽ硹ɾäƤȤ롣*/
  node = top->left.value.next->right.value.next->right.value.next ;
  entity = NULL ;
  while( node != NULL ){
    if( entity != NULL )
      free_unreferred_entity( entity ) ;
    entity = skkinputlisp_eval_car( node, newLocalVariableListTop ) ;
    /* 顼Ф顢ߤ롣*/
    if( entity == NULL )
      break ;
    node = node->right.value.next ;
  }

  /* ɽѿ˴롣*/
  if( pVarListNode != NULL ){
    pVarListNode->next = NULL ;
    destroy_localVariable( newLocalVariableListTop ) ;
  }
  return entity ;
}

/*
 * (*   ) ¹Ԥؿ
 */
static struct skkinputlisp_entity *skkinputlisp_multiply
( struct conslist *top, 
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *r_entity, *entity ;
  struct conslist *node ;
  int argc ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  if( ( r_entity = create_newEntity() ) == NULL )
    return NULL ;
  r_entity->type = ENTITY_INTEGER ;
  r_entity->data.integer = 1 ;
  node = top->right.value.next ;
  /* οƤ*/
  argc = count_argument_number( node ) ;
  while( argc > 0 && node != NULL ){
    entity = skkinputlisp_evalCarAndCheckType
      ( node, localVariableListTop, ENTITY_INTEGER ) ;
    if( entity == NULL ){
      free_unreferred_entity( r_entity ) ;
      return NULL ;
    }
    /* 軻Ԥ*/
    r_entity->data.integer = r_entity->data.integer *
      entity->data.integer ;
    free_unreferred_entity( entity ) ;
    argc -- ;
    node = node->right.value.next ;
  }
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( entity = search_entity( r_entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( r_entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( r_entity ) ;
    r_entity = entity ;
  }
  return r_entity ;
}

/*
 * (+   ) ¹Ԥؿ
 */
static struct skkinputlisp_entity *skkinputlisp_plus
( struct conslist *top, 
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *r_entity, *entity ;
  struct conslist *node ;
  int argc ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  if( ( r_entity = create_newEntity() ) == NULL )
    return NULL ;
  r_entity->type = ENTITY_INTEGER ;
  r_entity->data.integer = 0 ;
  node = top->right.value.next ;
  /* οƤ*/
  argc = count_argument_number( node ) ;
  while( argc > 0 && node != NULL ){
    entity = skkinputlisp_evalCarAndCheckType
      ( node, localVariableListTop, ENTITY_INTEGER ) ;
    if( entity == NULL ){
      free_unreferred_entity( r_entity ) ;
      return NULL ;
    }
    /* 軻Ԥ*/
    r_entity->data.integer = r_entity->data.integer +
      entity->data.integer ;
    free_unreferred_entity( entity ) ;
    argc -- ;
    node = node->right.value.next ;
  }
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( entity = search_entity( r_entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( r_entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( r_entity ) ;
    r_entity = entity ;
  }
  return r_entity ;
}

/*
 * (/   ) ¹Ԥؿ
 */
static struct skkinputlisp_entity *skkinputlisp_divide
( struct conslist *top, 
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *r_entity, *entity ;
  struct conslist *node ;
  int argc ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  node = top->right.value.next ;
  /* οƤ*/
  argc = count_argument_number( node ) ;
  if( argc < 2 ){
#if defined(DEBUG)
    fprintf
      ( stdout, "Wrong number of arguments: #<subr />, %d\n", argc ) ;
#endif
    return NULL ;
  }
  /* 롣*/
  entity = skkinputlisp_evalCarAndCheckType
    ( node, localVariableListTop, ENTITY_INTEGER ) ;
  if( entity == NULL )
    return NULL ;
  if( ( r_entity = create_newEntity() ) == NULL )
    return NULL ;
  r_entity->type = ENTITY_INTEGER ;
  r_entity->data.integer = entity->data.integer ;
  free_unreferred_entity( entity ) ;
  node = node->right.value.next ;
  argc -- ;
  /* ʲơäƤ*/
  while( argc > 0 && node != NULL ){
    entity = skkinputlisp_evalCarAndCheckType
      ( node, localVariableListTop, ENTITY_INTEGER ) ;
    if( entity == NULL ){
      free_unreferred_entity( r_entity ) ;
      return NULL ;
    }
    /* Ԥ*/
    r_entity->data.integer = r_entity->data.integer /
      entity->data.integer ;
    free_unreferred_entity( entity ) ;
    argc -- ;
    node = node->right.value.next ;
  }
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( entity = search_entity( r_entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( r_entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( r_entity ) ;
    r_entity = entity ;
  }
  return r_entity ;
}


/*
 * (-   ) ¹Ԥؿ
 */
static struct skkinputlisp_entity *skkinputlisp_minus
( struct conslist *top, 
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *r_entity, *entity ;
  struct conslist *node ;
  int argc, value ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  /* ֤ͤѰդƤ*/
  if( ( r_entity = create_newEntity() ) == NULL )
    return NULL ;
  r_entity->type = ENTITY_INTEGER ;
  /* οƤ*/
  node = top->right.value.next ;
  argc = count_argument_number( node ) ;
  if( argc <= 0 ){
    r_entity->data.integer = 0 ;
  } else {
    entity = skkinputlisp_evalCarAndCheckType
      ( node, localVariableListTop, ENTITY_INTEGER ) ;
    if( entity == NULL ){
      free_unreferred_entity( r_entity ) ;
      return NULL ;
    }
    value = entity->data.integer ;
    free_unreferred_entity( entity ) ;
    if( argc == 1 ){
      /* ĤʤСοͤؤ롣*/
      r_entity->data.integer = value * (-1) ;
    } else {
      /* ʣСǽοͤ˸Ԥ*/
      r_entity->data.integer = value ;

      node = node->right.value.next ;
      argc -- ;
      while( argc > 0 && node != NULL ){
	entity = skkinputlisp_evalCarAndCheckType
	  ( node, localVariableListTop, ENTITY_INTEGER ) ;
	if( entity == NULL ){
	  free_unreferred_entity( r_entity ) ;
	  return NULL ;
	}
	/* Ԥ*/
	r_entity->data.integer = r_entity->data.integer -
	  entity->data.integer ;
	free_unreferred_entity( entity ) ;
	argc -- ;
	node = node->right.value.next ;
      }
    }
  }
  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( entity = search_entity( r_entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( r_entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( r_entity ) ;
    r_entity = entity ;
  }
  return r_entity ;
}

/*
 * ʸͤѴؿ
 */
static struct skkinputlisp_entity *skkinputlisp_string_to_int
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *r_entity ;
  struct conslist *node ;
  int argc, value ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  node = top->right.value.next ;
  if( ( argc = count_argument_number( node ) ) != 1 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr string-to-int>, %d\n", argc ) ;
#endif      
    return NULL ;
  }
  entity = skkinputlisp_evalCarAndCheckType
    ( node, localVariableListTop, ENTITY_STRING ) ;
  if( entity == NULL )
    return NULL ;
  value = myCharAtoi( entity->data.string ) ;
  free_unreferred_entity( entity ) ;
  /* ֤ͤѰդƤ*/
  if( ( r_entity = create_newEntity() ) == NULL )
    return NULL ;
  r_entity->type = ENTITY_INTEGER ;
  r_entity->data.integer = value ;

  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( entity = search_entity( r_entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( r_entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( r_entity ) ;
    r_entity = entity ;
  }
  return r_entity ;
}

/*
 * ͤʸѴؿ
 */
static struct skkinputlisp_entity *skkinputlisp_int_to_string
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *r_entity ;
  struct conslist *node ;
  /*  64 Ȥͤ MAX_INTEGER 򸫤ƽʤȤʤ
     ݤʤ ʬ 64夢פȻפɤ͡*/
  unsigned char buffer[ 64 ] ;
  int argc ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  node = top->right.value.next ;
  if( ( argc = count_argument_number( node ) ) != 1 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr int-to-string>, %d\n", argc ) ;
#endif      
    return NULL ;
  }
  entity = skkinputlisp_evalCarAndCheckType
    ( node, localVariableListTop, ENTITY_INTEGER ) ;
  if( entity == NULL )
    return NULL ;
  sprintf( buffer, "%d", entity->data.integer ) ;
  free_unreferred_entity( entity ) ;
  /* ֤ͤѰդƤ*/
  if( ( r_entity = create_newEntity() ) == NULL )
    return NULL ;
  r_entity->type = ENTITY_STRING ;
  r_entity->data.string = malloc
    ( sizeof( struct myChar ) * ( strlen( buffer ) + 1 ) ) ;
  if( r_entity->data.string == NULL ){
    free_unreferred_entity( r_entity ) ;
    return NULL ;
  }
  myCharCharStrcpy( r_entity->data.string, buffer ) ;

  /* μΤΤ¸ߤ뤫Ĵ٤롣*/
  if( ( entity = search_entity( r_entity ) ) == NULL ){
    /* ΤˤĤʤǤ*/
    link_newEntity( r_entity ) ;
    /* ʬȤλȥ󥿤䤵ʤȡ*/
  } else {
    free_unreferred_entity( r_entity ) ;
    r_entity = entity ;
  }
  return r_entity ;
}

static struct skkinputlisp_entity *skkinputlisp_progn
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity ;
  struct conslist *node ;
  int argc ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  node = top->right.value.next ;
  argc = count_argument_number( node ) ;
  entity = nilEntity() ;
  while( node != NULL && argc > 0 ){
    if( entity != NULL )
      free_unreferred_entity( entity ) ;
    entity = skkinputlisp_eval_car( node, localVariableListTop ) ;
    if( entity == NULL )
      return NULL ;
    node = node->right.value.next ;
    argc -- ;
  } 
  return entity ;
}

/*
 * ʴؿġδؿ load-file ʳǤФ˻Ȥʤȡ
 *  malloc ޤؿ¸ߤȤȤҤɤ...
 * load-file ʥեФƤ櫓Ǥ
 */
static struct myChar *add_myCharString_myCharString
( struct myChar *oldstring, struct myChar *string )
{
  int length ;
  struct myChar *ret ;
  
  if( oldstring != NULL ){
    length = myCharStrlen( oldstring ) + 1 ;
  } else {
    length = 1 ;
  } 
  length += myCharStrlen( string ) ;
  if( ( ret = malloc( sizeof( struct myChar ) * length ) ) == NULL )
    return NULL ;
  if( oldstring != NULL ){
    myCharStrcpy( ret, oldstring ) ;
  } else {
    MYCHAR_SET_END_OF_STRING( *ret ) ;
  }
  myCharStrcat( ret, string ) ;
  return ret ;
}

/*
 * ʴؿġδؿ load-file ʳǤФ˻Ȥʤȡ
 *  malloc ޤؿ¸ߤȤȤҤɤ...
 * load-file ʥեФƤ櫓Ǥ
 */
static struct myChar *add_myCharString_string
( struct myChar *oldstring, unsigned char *string )
{
  int length ;
  struct myChar *ret ;
  
  if( oldstring != NULL ){
    length = myCharStrlen( oldstring ) + 1 ;
  } else {
    length = 1 ;
  } 
  length += strlen( string ) ;
  if( ( ret = malloc( sizeof( struct myChar ) * length ) ) == NULL )
    return NULL ;
  if( oldstring != NULL ){
    myCharStrcpy( ret, oldstring ) ;
  } else {
    MYCHAR_SET_END_OF_STRING( *ret ) ;
  }
  myCharCharStrcat( ret, string ) ;
  return ret ;
}

static struct myChar *skkinputlisp_load_file_sub
( unsigned char *path, int jcode )
{
  FILE *fp ;
  struct myChar *command_string, *newstring ;
  struct myChar *oneline ;

  command_string = NULL ;
  if( ( fp = fopen( path, "r" ) ) == NULL )
    return False ;
  
  command_string = add_myCharString_string
    ( NULL, "(progn " ) ;
  while( !feof( fp ) ){
    if( ( oneline = mychar_readOneLine( fp, jcode ) ) == NULL )
      break ;
    delete_comment_string( oneline ) ;
    if( IS_END_OF_STRING( *oneline ) ){
      free( oneline ) ;
      continue ;
    }
    if( ( newstring = add_myCharString_myCharString
	  ( command_string, oneline ) ) == NULL ){
      free( oneline ) ;
      free( command_string ) ;
      fclose( fp ) ;
      return NULL ;
    }
    free( oneline ) ;
    free( command_string ) ;
    command_string = newstring ;
  }
  newstring = add_myCharString_string
    ( command_string, ")" ) ;
  free( command_string ) ;
  fclose( fp ) ;
  return newstring ;
}

/*
 * (load-file "ե̾") ¹Ԥؿ
 */
static struct skkinputlisp_entity *skkinputlisp_load_file
( struct conslist *top,
  struct skkinputlisp_variable *localVariableListTop )
{
  struct skkinputlisp_entity *entity, *tmpentity ;
  struct conslist *node ;
  unsigned char *path, *asciiString ;
  struct myChar *command_string, *realcommand ;
  int argc, jcode ;

  if( !skkinputlisp_sequencepCheck( top ) ){
    return NULL ;
  }
  node = top->right.value.next ;
  if( ( argc = count_argument_number( node ) ) != 1 ){
#if defined(DEBUG)
    fprintf
      ( stdout, 
	"Wrong number of arguments: #<subr load-file>, %d\n", argc ) ;
#endif      
    return NULL ;
  }
  /* ηå*/
  if( ( entity = skkinputlisp_evalCarAndCheckType
	( node, localVariableListTop, ENTITY_STRING ) ) == NULL ){
    return NULL ;
  }
  /* ⤦ expand-file-name Ƥޤ...Τʤ */
  asciiString = myCharStringToAsciiString( entity->data.string ) ;
  if( asciiString == NULL )
    return NULL ;
  path = expand_file_name( asciiString ) ;
  free( asciiString ) ;
  free_unreferred_entity( entity ) ;
  /* ɤ߹եδɤȽ̤롣*/
  jcode = check_skkinput_jisyo_code( path ) ;
  command_string = skkinputlisp_load_file_sub( path, jcode ) ;
  free( path ) ;
  if( command_string == NULL ){
    return NULL ;
  }
  if( ( realcommand = convert_quote( command_string ) ) == NULL ){
    free( command_string ) ;
    return NULL ;
  }
  tmpentity = string2entity( realcommand, localVariableListTop ) ;
  free( command_string ) ;
  free( realcommand ) ;
  if( tmpentity == NULL )
    return NULL ;
  free_unreferred_entity( tmpentity ) ;
  return tEntity() ;
}

/*
 * skkinput θ䤬 (...) ä˸ƽФؿ
 */
int skkinputlisp_eval
( struct myChar *commandstring, struct myChar *buffer )
{
  struct skkinputlisp_entity *entity ;
  struct myChar *realcommand ;
  int length ;

  if( ( realcommand = convert_quote( commandstring ) ) == NULL )
    return False ;
  entity = string2entity( realcommand, NULL ) ;
  free( realcommand ) ;
  if( entity == NULL ){
    skkinputlisp_garbagecollector() ;
    return False ;
  }
  /* ʸǤʤФʤʤ*/
  if( entity->type != ENTITY_STRING ){
    skkinputlisp_garbagecollector() ;
    return False ;
  }
  if( entity->data.string == NULL ){
    MYCHAR_SET_END_OF_STRING( buffer[ 0 ] ) ;
  } else {
    if( ( length = myCharStrlen( entity->data.string ) ) > TEXTMAXLEN )
      length = TEXTMAXLEN ;
    myCharStrncpy( buffer, entity->data.string, length ) ;
    MYCHAR_SET_END_OF_STRING( buffer[ length ] ) ;
  }
  skkinputlisp_garbagecollector() ;
  return True ;
}

/*
 * ֤ͤʸǤʤƤˤʤʸ¹Դؿ
 */
int skkinputlisp_weak_eval
( struct myChar *commandstring )
{
  struct skkinputlisp_entity *entity ;
  struct myChar *realcommand ;
  int ret ;

  if( ( realcommand = convert_quote( commandstring ) ) == NULL )
    return False ;
  entity = string2entity( realcommand, NULL ) ;
  free( realcommand ) ;
  ret = ( entity == NULL )? False : True ;
  skkinputlisp_garbagecollector() ;
  return ret ;
}

/*
 * j-num-list ˿Ѵλͤؿ
 *----
 * j-num-list òƤޤäƤޤġؿɬפȻפ
 * ơġ
 */
int skkinput_create_j_num_list
( struct myChar *string )
{
  struct skkinputlisp_entity *entity ;
  struct myChar *liststr ;
  struct myChar *ptr ;
  int counter, length ;

  length = counter = 0 ;
  for( ptr = string ; !IS_END_OF_STRING( *ptr ) ; ptr ++ ){
    if( IS_ASCII_EQUAL( *ptr, '/' ) )
      counter ++ ;
    length ++ ;
  }
  if( counter <= 0 || length <= 0 )
    return True ;
  liststr = malloc
    ( sizeof( struct myChar ) * ( length + counter * 2 + 30 ) ) ;
  if( liststr == NULL )
    return False ;
  
  myCharCharStrcpy( liststr, "(setq j-num-list (quote (" ) ;
  ptr = liststr + 25 ;
  while( !IS_END_OF_STRING( *string ) ){
    MYCHAR_SET_CHARA( *ptr, 0x22 ) ;
    ptr ++ ;
    while( !IS_END_OF_STRING( *string ) &&
	   !IS_ASCII_EQUAL( *string, '/' ) ){
      *ptr ++ = *string ++ ;
    }
    if( IS_ASCII_EQUAL( *string, '/' ) )
      string ++ ;
    MYCHAR_SET_CHARA( *ptr, 0x22 ) ;
    ptr ++ ;
    MYCHAR_SET_CHARA( *ptr, ' ' ) ;
    ptr ++ ;
  }
  MYCHAR_SET_CHARA( *ptr, ')' ) ;
  ptr ++ ;
  MYCHAR_SET_CHARA( *ptr, ')' ) ;
  ptr ++ ;
  MYCHAR_SET_CHARA( *ptr, ')' ) ;
  ptr ++ ;
  MYCHAR_SET_END_OF_STRING( *ptr ) ;

#if defined(DEBUG)
  fprintf( stderr, "(j-num-list): " ) ;
  myCharFputstring( stderr, liststr ) ;
  fputc( '\n', stderr ) ;
  fflush( stderr ) ;
#endif
  if( ( entity = string2entity( liststr, NULL ) ) == NULL ){
#if defined(DEBUG)
    fprintf( stderr, "(j-num-list error): %s\n", liststr ) ;
#endif
    free( liststr ) ;
    skkinputlisp_garbagecollector() ;
    return False ;
  }
  free( liststr ) ;
  skkinputlisp_garbagecollector() ;
  return True ;
}

#if defined(MYEVAL_ONLY)
/*
 * ե뤫ʸɤ߹ߡJIS ȻפΤʤ EUC Ѵʸ
 * ؿ
 */
int euc_fgets
( unsigned char *buffer, int len, FILE *fp, int echara,
  int coding_system, int ecflag )
{
  int c, status = 0, kflag = False ;

  while( len > 0 ){
    if( ( c = fgetc( fp ) ) == EOF )
      break ;
    if( coding_system ){
      /* JIS ξˤϵǽʸĴ٤롣*/
      switch( c ){
	/* ǽʸγϥɡ*/
      case 0x1b :
	status = c ;
	continue ;
	/* ʸ 0x1b ³ˤϵǽʸǤ롣*/
      case 0x24 :
      case 0x28 :
	if( status == 0x1b ){
	  status = c ;
	  continue ;
	} 
	break ;
	/* ʸ 0x1b,0x24/0x28 ³ˤϵǽʸǤ롣*/
      case 0x42 :
	if( status == 0x28 ){
	  kflag = False ;
	  status = 0 ;
	  continue ;
	} else if( status == 0x24 ){
	  kflag = True ;
	  status = 0 ;
	  continue ;
	}
	break ;
	/* ʳʸġ*/
      default :
	break ;
      }
      /* ե饰ΩäƤ顢8bitܤ򤿤Ƥ롣*/
      if( kflag )
	c |= 0x80 ;
    }
    /* ԥɤ򸫤ƤޤäˤϽλ롣*/
    if( c == echara ){
      /* Υե饰äƤ顢END ʸäƤޤ*/
      if( ecflag ){
	*buffer ++ = echara ;
	len -- ;
      }
      break ;
    }
    /* ɤ߹ʸ򤷤ޤ*/
    *buffer ++ = c ;
    len -- ;
  }
  /* ;͵Сü롣*/
  if( len > 0 )
    *buffer ++ = '\0' ;
  /* ɤ߻Ĥʸ֤*/
  return len ;
}

/*
 * emacs-lisp ˤ expand-file-name ؿ
 *-----
 * ֺǽ "~" ä home directory Ÿ뵡ǽ
 * Ƥ롣wild-card ʤȤˤбƤʤ
 * δؿƤ̤ skk-local-jisyo-path 뤳Ȥˤʤ롣
 * skk-local-jisyo-path  NULL Ǥʤä顢δؿƤФʤ
 * ȻפƤ¦Ǥ¿ʬåǤʤΤǡǥå
 * 롣
 */
unsigned char *expand_file_name( unsigned char *path )
{
  unsigned char *ptr ;
  unsigned char *home ;
  int len = strlen( path ) ;

  /* "~" Ȥ home directory ɽ뵭椬äƤν*/
  if( *path == '~' ){
    /* HOME Directory 褫롣*/
    home = getenv( "HOME" ) ;
    /* Home directory + ĤΥѥĹ malloc 롣*/
    ptr = malloc( strlen( home ) + len + 1 ) ;
    if( ptr == NULL ){
      /* malloc ԡĤ̿Ūʥ顼ʤΤǷ³ưϤʤ*/
      fprintf( stderr, "Memory fault.\n" ) ;
      exit( 1 ) ;
    }
    /* "~" Ÿơľ*/
    strcpy( ptr, home ) ;
    strcat( ptr, path + 1 ) ;
  } else {
    /* ɬפ̵硣*/
    ptr = ( unsigned char *)malloc( len + 1 ) ;
    strcpy( ptr, path ) ;
  }
  return ptr ;
}

/*
 * եįơESC äƤ JIS8bit ܤΩäƤ륳
 * äƤ EUC ȤȽǤ뤫ʤŬʴؿ
 *----
 * skkinput-jisyo Ѥʴɤʤ褦ˡġ
 */
int check_skkinput_jisyo_code( unsigned char *path )
{
  FILE *fp ;
  int c, ret ;

  /* ե뤬¸ߤʤˤϡǾcoding system *
   * 򤹤롣*/
  if( ( fp = fopen( path, "rb" ) ) == NULL ){
    return CODING_SYSTEM_IS_EUC ;
  }
  ret = CODING_SYSTEM_IS_EUC ;
  while( ( c = fgetc( fp ) ) != EOF ){
    /* ʸդν*/
    if( c == 0x1b )
      ret = CODING_SYSTEM_IS_JIS ;
    /* 8bit ܤΩäƤν*/
    if( c & 0x80 )
      break ;
  }
  fclose( fp ) ;
  return ret ;
}

/*
 * ƥȤʿͤΥᥤǤ
 */
int main( int argc, char *argv[] )
{
  struct skkinputlisp_entity *hoge ;
  struct myChar *string ;
  struct myChar myChar_buffer[1024];
  unsigned char buffer[ 1024 ] ;
  int i, j;
  int len ;

  /* ᥤ롼ס*/
  while( 1 ){
    printf( "> " ) ;
    if( fgets( buffer, 1024, stdin ) == NULL )
      break ;
    len = strlen( buffer ) ;
    if( buffer[ len - 1 ] != '\n' || buffer[ 0 ] == '\n' ){
      /* Ĺ᤮ԤϵѲ롣*/
      continue ;
    } else {
      buffer[ len - 1 ] = '\0' ;
    }
    memset(myChar_buffer, 0, sizeof myChar_buffer);
    for (i = j= 0; i < len; i++, j++) {
	if (buffer[i] < 0x80) {
	    myChar_buffer[j].charset = CHARSET_ASCII;
	    myChar_buffer[j].chara = buffer[i];
	} else {
	    myChar_buffer[j].charset = CHARSET_JISX0208_1983;
	    myChar_buffer[j].chara = buffer[i] << 8 + buffer[i+1];
	    i++;
	}
    }
    myChar_buffer[j].charset = CHARSET_ASCII;
    myChar_buffer[j].chara = 0;

    string = convert_quote( myChar_buffer ) ;
    if( string != NULL ){
      hoge = string2entity( string, NULL ) ;
      if( hoge != NULL ){
	print_entity( hoge ) ;
	printf( "\n" ) ;
	/* ǥ߼ԤŤʾС*/
	skkinputlisp_garbagecollector() ;
      }
      free( string ) ;
    }
  }
  return 0 ;
}

#endif
