%{

(* A parser for policy.conf and mls files. (There are two entry points
   to the parser.)  This is a translation of Stephen Smalley's parser
   into ocamlyacc, and is supposed to be faithful to the original.
   Action routines marked with { () } are places you may want to add
   an action routine.  *)

open Formulas
open Identifier

type signed_id =
    Pos of identifier
  | Neg of identifier

(* The value returned by a name non-terminal *)
type names =
    Ids of signed_id list
  | Ids_not of signed_id list
  | Ids_all

(* Partition signed ids into a pair of lists *)
let collect_ids slist =
  let f (pos, neg) s =
    match s with
      Pos i -> (i :: pos, neg)
    | Neg i -> (pos, i :: neg) in
  let (pos, neg) = List.fold_left f ([], []) slist in
  (List.rev pos, List.rev neg)

(* Converts names to contexts *)
let names2ctxs build names =
  let build_ids ids =
    let (pos, neg) = collect_ids ids in
    if neg = [] then
      build pos
    else if pos = [] then
      Ctx_not (build neg)
    else
      Ctx_and (build pos, Ctx_not (build neg)) in
  match names with
    Ids ids -> build_ids ids
  | Ids_not ids -> Ctx_not (build_ids ids)
  | Ids_all -> Ctx_true

let names2init_types n = names2ctxs (fun ids -> Ctx_types ids) n
let names2init_roles n = names2ctxs (fun ids -> Ctx_roles ids) n

(* Converts names to transitions *)
let names2trans build names =
  let build_ids ids =
    let (pos, neg) = collect_ids ids in
    if neg = [] then
      build pos
    else if pos = [] then
      Tran_not (build neg)
    else
      Tran_and (build pos, Tran_not (build neg)) in
  match names with
    Ids ids -> build_ids ids
  | Ids_not ids -> Tran_not (build_ids ids)
  | Ids_all -> Tran_true

let names2types n = names2trans (fun ids -> Tran_types ids) n
let names2types' n = names2trans (fun ids -> Next_types ids) n
let names2roles n = names2trans (fun ids -> Tran_roles ids) n
let names2roles' n = names2trans (fun ids -> Next_roles ids) n
let names2users n = names2trans (fun ids -> Tran_users ids) n
let names2users' n = names2trans (fun ids -> Next_users ids) n
let names2classes n = names2trans (fun ids -> Tran_classes ids) n
let names2perms n = names2trans (fun ids -> Tran_permissions ids) n

%}

%token <string> PATH
%token CLONE
%token COMMON
%token CLASS
%token CONSTRAIN
%token INHERITS
%token SID
%token ROLE
%token ROLES
%token TYPEALIAS
%token TYPE
%token TYPES
%token ALIAS
%token ATTRIBUTE
%token TYPE_TRANSITION
%token TYPE_MEMBER
%token TYPE_CHANGE
%token ROLE_TRANSITION
%token SENSITIVITY
%token DOMINANCE
%token DOM DOMBY INCOMP
%token CATEGORY
%token LEVEL
%token RANGES
%token USER
%token NEVERALLOW
%token ALLOW
%token AUDITALLOW
%token AUDITDENY
%token DONTAUDIT
%token SOURCE
%token TARGET
%token SAMEUSER
%token FSCON PORTCON NETIFCON NODECON
%token FSUSEPSID FSUSETASK FSUSETRANS FSUSEXATTR
%token GENFSCON
%token U1 U2 R1 R2 T1 T2
%token NOT AND OR
%token <Identifier.identifier> IDENTIFIER
%token <Identifier.identifier> USER_IDENTIFIER
%token <int> NUMBER
%token EQUALS
%token NOTEQUAL
%token IPV6_ADDR
%token COMMA COLON SEMICOLON
%token LPAREN RPAREN
%token LBRACE RBRACE
%token LBRACK RBRACK
%token HYPHEN PERIOD STAR TILDE
%token EOF

%left OR
%left AND
%right NOT
%left EQUALS NOTEQUAL

%start policy mls
%type <unit> policy
%type <unit> mls

%%
policy:
  classes initial_sids access_vectors
  opt_mls te_rbac users opt_constraints
  initial_sid_contexts opt_fs_contexts fs_uses
  opt_genfs_contexts net_contexts EOF { }

mls:
  req_mls EOF                   { }
;

classes:
  class_def                     { }
| classes class_def             { }

class_def:
  CLASS IDENTIFIER              { class_def $2 }
;

initial_sids:
  initial_sid_def               { }
| initial_sids initial_sid_def  { }
;

initial_sid_def:
  SID IDENTIFIER                { () }
;

access_vectors:
  opt_common_perms av_perms     { }
;

opt_common_perms:
                                { }
| common_perms                  { }
;

common_perms:
  common_perms_def              { }
| common_perms common_perms_def { }
;

common_perms_def:
  COMMON IDENTIFIER LBRACE identifier_list RBRACE
    { common_perms_def $2 $4 }
;

identifier_list:
  IDENTIFIER                    { [$1] }
| identifier_list IDENTIFIER    { $2 :: $1 }
;

av_perms:
  av_perms_def                  { }
| av_perms av_perms_def         { }

av_perms_def:
  CLASS IDENTIFIER LBRACE identifier_list RBRACE
    { class_perms_def $2 None $4 }
| CLASS IDENTIFIER INHERITS IDENTIFIER
    { class_perms_def $2 (Some $4) [] }
| CLASS IDENTIFIER INHERITS IDENTIFIER LBRACE identifier_list RBRACE
    { class_perms_def $2 (Some $4) $6 }
;

opt_mls:
                                { }
| req_mls                       { }
;

req_mls:
  sensitivities dominance opt_categories levels base_perms { }
;

sensitivities:
  sensitivity_def               { }
| sensitivities sensitivity_def { }
;
			;
sensitivity_def:
  SENSITIVITY IDENTIFIER alias_def SEMICOLON { () }
| SENSITIVITY IDENTIFIER SEMICOLON { () }
;
alias_def:
  ALIAS names                   { $2 }
;

dominance:
  DOMINANCE IDENTIFIER          { () }
| DOMINANCE LBRACE identifier_list RBRACE { () }
;

opt_categories:
                                { }
| categories                    { }
;

categories:
  category_def                  { }
| categories category_def       { }
;

category_def:
  CATEGORY IDENTIFIER alias_def SEMICOLON { () }
| CATEGORY IDENTIFIER SEMICOLON { () }
;

levels:
  level_def                     { }
| levels level_def              { }
;

level_def:
  LEVEL IDENTIFIER COLON id_comma_list SEMICOLON { () }
| LEVEL IDENTIFIER SEMICOLON    { () }
;

base_perms:
  opt_common_base av_base       { }
;

opt_common_base:
                                { }
| common_base                   { }
;

common_base:
  common_base_def               { }
| common_base common_base_def   { }
;
			;
common_base_def:
  COMMON IDENTIFIER LBRACE perm_base_list RBRACE
    { mls_common_perms_def $2 $4 }
;

av_base:
  av_base_def                   { }
| av_base av_base_def           { }
;

av_base_def:
  CLASS IDENTIFIER LBRACE perm_base_list RBRACE
    { mls_class_perms_def $2 $4 }
| CLASS IDENTIFIER
    { mls_class_perms_def $2 [] }
;

perm_base_list:
  perm_base                     { [$1] }
| perm_base_list perm_base      { $2 :: $1 }
;

perm_base:
  IDENTIFIER COLON IDENTIFIER   { $1, [$3] }
| IDENTIFIER COLON LBRACE identifier_list RBRACE { $1, $4 }
;

te_rbac:
  te_rbac_decl                  { }
| te_rbac te_rbac_decl          { }
;

te_rbac_decl:
  te_decl                       { }
| rbac_decl                     { }
| SEMICOLON                     { }
;

rbac_decl:
  role_type_def                 { }
| role_dominance                { }
| role_trans_def                { }
| role_allow_def                { }
;

te_decl:
  attribute_def                 { }
| type_def                      { }
| typealias_def                 { }
| transition_def                { }
| te_avtab_def                  { }
;

attribute_def:
  ATTRIBUTE IDENTIFIER SEMICOLON
    { attribute_def $2 }
;

type_def:
  TYPE IDENTIFIER alias_def opt_attr_list SEMICOLON
    { type_attributes_def $2 $4 ; type_alias_def $2 (names2init_types $3) }
| TYPE IDENTIFIER opt_attr_list SEMICOLON
    { type_attributes_def $2 $3 }
;

typealias_def:
  TYPEALIAS IDENTIFIER alias_def SEMICOLON
    { type_alias_def $2 (names2init_types $3) }
;

opt_attr_list:
  COMMA id_comma_list           { $2 }
|                               { [] }
;

transition_def:
  TYPE_TRANSITION names names COLON names IDENTIFIER SEMICOLON { () }
| TYPE_MEMBER names names COLON names IDENTIFIER SEMICOLON { () }
| TYPE_CHANGE names names COLON names IDENTIFIER SEMICOLON { () }
;

te_avtab_def:
  allow_def                     { }
| auditallow_def                { }
| auditdeny_def                 { }
| dontaudit_def                 { }
| neverallow_def                { }
;

allow_def:
  ALLOW names names COLON names names SEMICOLON
    { allow_def (names2types $2) (names2types' $3)
                (names2classes $5) (names2perms $6) }
;

auditallow_def:
  AUDITALLOW names names COLON names names SEMICOLON { () }
;

auditdeny_def:
  AUDITDENY names names COLON names names SEMICOLON { () }
;

dontaudit_def:
  DONTAUDIT names names COLON names names SEMICOLON { () }
;

neverallow_def:
  NEVERALLOW names names COLON names names SEMICOLON
    { never_allow_def (names2types $2) (names2types' $3)
                (names2classes $5) (names2perms $6) }
;

role_type_def:
  ROLE IDENTIFIER TYPES names SEMICOLON
    { role_types_def $2 (names2init_types $4) }
;

role_dominance:
  DOMINANCE LBRACE roles RBRACE { () }
;

role_trans_def:
  ROLE_TRANSITION names names IDENTIFIER SEMICOLON { () }
;

role_allow_def:
  ALLOW names names SEMICOLON
    { role_allow_def (names2roles $2) (names2roles' $3) }
;

roles:
  role_def                      { }
| roles role_def                { }
;

role_def:
  ROLE IDENTIFIER SEMICOLON     { () }
| ROLE IDENTIFIER LBRACE roles RBRACE { () }
;

opt_constraints:
                                { }
| constraints                   { }
;

constraints:
  constraint_def                { }
| constraints constraint_def    { }
;

constraint_def:
   CONSTRAIN names names cexpr SEMICOLON
     { constrant_def (names2classes $2) (names2perms $3) $4 }
;

cexpr:
  LPAREN cexpr RPAREN           { $2 }
| NOT cexpr                     { Tran_not $2 }
| cexpr AND cexpr               { Tran_and ($1, $3) }
| cexpr OR cexpr                { Tran_or ($1, $3) }
| cexpr_prim                    { $1 }
;

cexpr_prim:
  U1 op U2                      { $2 Same_users }
| R1 roleop R2                  { $2 Same_roles }
| T1 op T2                      { $2 Same_types }
| U1 op user_names              { $2 (names2users $3) }
| U2 op user_names              { $2 (names2users' $3) }
| R1 op names                   { $2 (names2roles $3) }
| R2 op names                   { $2 (names2roles' $3) }
| T1 op names                   { $2 (names2types $3) }
| T2 op names                   { $2 (names2types' $3) }
| SAMEUSER                      { Same_users }
| SOURCE ROLE names             { names2roles $3 }
| TARGET ROLE names             { names2roles' $3 }
| ROLE roleop                   { $2 Same_roles }
| SOURCE TYPE names             { names2types $3 }
| TARGET TYPE names             { names2types' $3 }
;

op:
  EQUALS                        { fun x -> x }
| NOTEQUAL                      { fun x -> Tran_not x }

roleop:
  op                            { $1 }
| DOM                           { failwith "role op dom not supported" }
| DOMBY                         { failwith "role op domby not supported" }
| INCOMP                        { failwith "role op incomp not supported" }
;

users:
  user_def                      { }
| users user_def                { }
;

user_id:
  IDENTIFIER                    { $1 }
| USER_IDENTIFIER               { $1 }
;

user_def:
  USER user_id ROLES names opt_user_ranges SEMICOLON
    { user_roles_def $2 (names2init_roles $4) }
;

opt_user_ranges:
                                { [] }
| RANGES user_ranges            { $2 }
;

user_ranges:
  mls_range_def                 { [$1] }
| LBRACE user_range_def_list RBRACE { $2 }
;

user_range_def_list:
  mls_range_def                 { [$1] }
| user_range_def_list mls_range_def { $2 :: $1 }
;

initial_sid_contexts:
  initial_sid_context_def       { }
| initial_sid_contexts initial_sid_context_def { }
			;
initial_sid_context_def:
  SID IDENTIFIER security_context_def { () }
;

opt_fs_contexts:
                                { }
| fs_contexts                   { }
;

fs_contexts:
  fs_context_def                { }
| fs_contexts fs_context_def    { }
;

fs_context_def:
  FSCON NUMBER NUMBER security_context_def security_context_def { () }
;

net_contexts:
  opt_port_contexts opt_netif_contexts opt_node_contexts { () }
;

opt_port_contexts:
                                { }
| port_contexts                 { }
;

port_contexts:
  port_context_def              { }
| port_contexts port_context_def { }
;

port_context_def:
  PORTCON IDENTIFIER NUMBER security_context_def { () }
| PORTCON IDENTIFIER NUMBER HYPHEN NUMBER security_context_def { () }
;

opt_netif_contexts:
                                { }
| netif_contexts                { }
;

netif_contexts:
  netif_context_def             { }
| netif_contexts netif_context_def { }
;

netif_context_def:
  NETIFCON IDENTIFIER security_context_def security_context_def { () }
;

opt_node_contexts:
                                { }
| node_contexts                 { }
;

node_contexts:
  node_context_def              { }
| node_contexts node_context_def { }
;

node_context_def:
  NODECON ipv4_addr_def ipv4_addr_def security_context_def { () }
| NODECON IPV6_ADDR IPV6_ADDR security_context_def { () }
;

fs_uses:
  fs_use_def                    { }
| fs_uses fs_use_def            { }
;

fs_use_def:
  FSUSEPSID IDENTIFIER SEMICOLON { () }
| FSUSEXATTR IDENTIFIER security_context_def SEMICOLON { () }
| FSUSETASK IDENTIFIER security_context_def SEMICOLON { () }
| FSUSETRANS IDENTIFIER security_context_def SEMICOLON { () }
;

opt_genfs_contexts:
                                { }
| genfs_contexts                { }
;

genfs_contexts:
  genfs_context_def             { }
| genfs_contexts genfs_context_def { }
;

genfs_context_def:
  GENFSCON IDENTIFIER PATH HYPHEN IDENTIFIER security_context_def { () }
| GENFSCON IDENTIFIER PATH HYPHEN HYPHEN security_context_def { () }
| GENFSCON IDENTIFIER PATH security_context_def { () }
;

ipv4_addr_def:
  NUMBER PERIOD NUMBER PERIOD NUMBER PERIOD NUMBER { () }
;

security_context_def:
  user_id COLON IDENTIFIER COLON IDENTIFIER opt_mls_range_def { () }
;

opt_mls_range_def:
                                { }
| COLON mls_range_def           { }
;

mls_range_def:
  mls_level_def HYPHEN mls_level_def { () }
| mls_level_def                      { () }
;

mls_level_def:
  IDENTIFIER COLON id_comma_list { $1, $3 }
| IDENTIFIER                    { $1, [] }
;

id_comma_list:
  IDENTIFIER                    { [$1] }
| id_comma_list COMMA IDENTIFIER { $3 :: $1 }
;

names:
  IDENTIFIER                    { Ids [Pos $1] }
| nested_id_set                 { Ids $1 }
| STAR                          { Ids_all }
| TILDE IDENTIFIER              { Ids_not [Pos $2] }
| TILDE nested_id_set           { Ids_not $2 }
;

nested_id_set:
  LBRACE nested_id_list RBRACE  { $2 }
;

nested_id_list:
  nested_id_element             { $1 }
| nested_id_list nested_id_element { $2 @ $1 }
;

nested_id_element:
  IDENTIFIER                    { [Pos $1] }
| HYPHEN IDENTIFIER             { [Neg $2] }
| nested_id_set                 { $1 }
;

user_names:
  user_id                       { Ids [Pos $1] }
| nested_user_id_set            { Ids $1 }
| STAR                          { Ids_all }
| TILDE user_id                 { Ids_not [Pos $2] }
| TILDE nested_user_id_set      { Ids_not $2 }
;

nested_user_id_set:
  LBRACE nested_user_id_list RBRACE { $2 }
;

nested_user_id_list:
  nested_user_id_element        { $1 }
| nested_user_id_list nested_user_id_element { $2 @ $1 }
;

nested_user_id_element:
  user_id                       { [Pos $1] }
| nested_user_id_set            { $1 }
;
