%{

/* A grammar for the SLAT Labeled Transition System syntax. */

#include <stdlib.h>
#include <stdio.h>
#include "symbol.h"
#include "formula.h"
#include "action.h"

extern int yylex();
extern int yyerror(const char *);

static symbol_list_t one(symbol_t item)
{
  return mk_symbol_list(item, 0); 
}

%}

%union {
  symbol_t symbol;
  symbol_list_t symbol_list;
  int boolean;
  arrow_t arrow;
  arrow_list_t arrow_list;
  diagram_t diagram;
  diagram_list_t diagram_list;
  tran_t tran;
  tran_list_t tran_list;
  lts_t lts;
}

%token <symbol> SYMBOL
%token CLASSES PERMISSIONS TYPES ROLES USERS NEXT
%token LPAREN RPAREN LBRACK RBRACK LBRACE RBRACE
%token PLUS COMMA SEMICOLON IMPLY IFF OR AND NOT
%token IN NOTIN EQ NEQ TRUE FALSE
%token STATE ACTION INIT TRANS SPEC
%right IMPLY
%left IFF
%left OR
%left AND
%right NOT
%token UNUSED

%type <symbol_list> set set_elems non_null_set
%type <symbol_list> type_def role_def user_def class_def permission_def
%type <boolean> more
%type <arrow> arrow
%type <arrow_list> elems non_null_elems
%type <diagram> diagram
%type <diagram_list> diagrams
%type <tran> transitions states actions init trans
%type <tran_list> specs
%type <lts> lts
%start start
%%

start: 
  transitions                { parsed_transition($1); }
| diagrams opt_semicolon     { parsed_diagrams(diagram_list_reverse($1)); }
| lts                        { parsed_lts($1); }
;

transitions:
  FALSE                      { $$ = mk_tran_false(); }
| TRUE                       { $$ = mk_tran_true(); }
| TYPES IN set               { $$ = mk_tran_types($3); }
| ROLES IN set               { $$ = mk_tran_roles($3); }
| USERS IN set               { $$ = mk_tran_users($3); }
| TYPES NEXT IN set          { $$ = mk_next_types($4); }
| ROLES NEXT IN set          { $$ = mk_next_roles($4); }
| USERS NEXT IN set          { $$ = mk_next_users($4); }
| CLASSES IN set             { $$ = mk_tran_classes($3); }
| PERMISSIONS IN set         { $$ = mk_tran_permissions($3); }
| TYPES NOTIN set            { $$ = mk_tran_not(mk_tran_types($3)); }
| ROLES NOTIN set            { $$ = mk_tran_not(mk_tran_roles($3)); }
| USERS NOTIN set            { $$ = mk_tran_not(mk_tran_users($3)); }
| TYPES NEXT NOTIN set       { $$ = mk_tran_not(mk_next_types($4)); }
| ROLES NEXT NOTIN set       { $$ = mk_tran_not(mk_next_roles($4)); }
| USERS NEXT NOTIN set       { $$ = mk_tran_not(mk_next_users($4)); }
| CLASSES NOTIN set          { $$ = mk_tran_not(mk_tran_classes($3)); }
| PERMISSIONS NOTIN set      { $$ = mk_tran_not(mk_tran_permissions($3)); }
| TYPES EQ SYMBOL            { $$ = mk_tran_types(one($3)); }
| ROLES EQ SYMBOL            { $$ = mk_tran_roles(one($3)); }
| USERS EQ SYMBOL            { $$ = mk_tran_users(one($3)); }
| TYPES NEXT EQ SYMBOL       { $$ = mk_next_types(one($4)); }
| ROLES NEXT EQ SYMBOL       { $$ = mk_next_roles(one($4)); }
| USERS NEXT EQ SYMBOL       { $$ = mk_next_users(one($4)); }
| TYPES NEXT EQ TYPES        { $$ = mk_same_types(); }
| TYPES EQ TYPES NEXT        { $$ = mk_same_types(); }
| ROLES NEXT EQ ROLES        { $$ = mk_same_roles(); }
| ROLES EQ ROLES NEXT        { $$ = mk_same_roles(); }
| USERS NEXT EQ USERS        { $$ = mk_same_users(); }
| USERS EQ USERS NEXT        { $$ = mk_same_users(); }
| CLASSES EQ SYMBOL          { $$ = mk_tran_classes(one($3)); }
| PERMISSIONS EQ SYMBOL      { $$ = mk_tran_permissions(one($3)); }
| TYPES NEQ SYMBOL           { $$ = mk_tran_not(mk_tran_types(one($3))); }
| ROLES NEQ SYMBOL           { $$ = mk_tran_not(mk_tran_roles(one($3))); }
| USERS NEQ SYMBOL           { $$ = mk_tran_not(mk_tran_users(one($3))); }
| TYPES NEXT NEQ SYMBOL      { $$ = mk_tran_not(mk_next_types(one($4))); }
| ROLES NEXT NEQ SYMBOL      { $$ = mk_tran_not(mk_next_roles(one($4))); }
| USERS NEXT NEQ SYMBOL      { $$ = mk_tran_not(mk_next_users(one($4))); }
| TYPES NEXT NEQ TYPES       { $$ = mk_tran_not(mk_same_types()); }
| TYPES NEQ TYPES NEXT       { $$ = mk_tran_not(mk_same_types()); }
| ROLES NEXT NEQ ROLES       { $$ = mk_tran_not(mk_same_roles()); }
| ROLES NEQ ROLES NEXT       { $$ = mk_tran_not(mk_same_roles()); }
| USERS NEXT NEQ USERS       { $$ = mk_tran_not(mk_same_users()); }
| USERS NEQ USERS NEXT       { $$ = mk_tran_not(mk_same_users()); }
| CLASSES NEQ SYMBOL         { $$ = mk_tran_not(mk_tran_classes(one($3))); }
| PERMISSIONS NEQ SYMBOL { $$ = mk_tran_not(mk_tran_permissions(one($3))); }
| NOT transitions            { $$ = mk_tran_not($2); }
| transitions AND transitions { $$ = mk_tran_and($1, $3); }
| transitions OR transitions { $$ = mk_tran_or($1, $3); }
| transitions IFF transitions { $$ = mk_tran_iff($1, $3); }
| transitions IMPLY transitions { $$ = mk_tran_imply($1, $3); }
| LPAREN transitions RPAREN  { $$ = $2; }
;

states:
  FALSE                      { $$ = mk_tran_false(); }
| TRUE                       { $$ = mk_tran_true(); }
| TYPES IN set               { $$ = mk_tran_types($3); }
| ROLES IN set               { $$ = mk_tran_roles($3); }
| USERS IN set               { $$ = mk_tran_users($3); }
| TYPES NOTIN set            { $$ = mk_tran_not(mk_tran_types($3)); }
| ROLES NOTIN set            { $$ = mk_tran_not(mk_tran_roles($3)); }
| USERS NOTIN set            { $$ = mk_tran_not(mk_tran_users($3)); }
| TYPES EQ SYMBOL            { $$ = mk_tran_types(one($3)); }
| ROLES EQ SYMBOL            { $$ = mk_tran_roles(one($3)); }
| USERS EQ SYMBOL            { $$ = mk_tran_users(one($3)); }
| TYPES NEQ SYMBOL           { $$ = mk_tran_not(mk_tran_types(one($3))); }
| ROLES NEQ SYMBOL           { $$ = mk_tran_not(mk_tran_roles(one($3))); }
| USERS NEQ SYMBOL           { $$ = mk_tran_not(mk_tran_users(one($3))); }
| NOT states                 { $$ = mk_tran_not($2); }
| states AND states          { $$ = mk_tran_and($1, $3); }
| states OR states           { $$ = mk_tran_or($1, $3); }
| states IFF states          { $$ = mk_tran_iff($1, $3); }
| states IMPLY states        { $$ = mk_tran_imply($1, $3); }
| LPAREN states RPAREN       { $$ = $2; }
;

actions:
  FALSE                      { $$ = mk_tran_false(); }
| TRUE                       { $$ = mk_tran_true(); }
| CLASSES IN set             { $$ = mk_tran_classes($3); }
| PERMISSIONS IN set         { $$ = mk_tran_permissions($3); }
| CLASSES NOTIN set          { $$ = mk_tran_not(mk_tran_classes($3)); }
| PERMISSIONS NOTIN set      { $$ = mk_tran_not(mk_tran_permissions($3)); }
| CLASSES EQ SYMBOL          { $$ = mk_tran_classes(one($3)); }
| PERMISSIONS EQ SYMBOL      { $$ = mk_tran_permissions(one($3)); }
| CLASSES NEQ SYMBOL         { $$ = mk_tran_not(mk_tran_classes(one($3))); }
| PERMISSIONS NEQ SYMBOL { $$ = mk_tran_not(mk_tran_permissions(one($3))); }
| NOT actions                { $$ = mk_tran_not($2); }
| actions AND actions        { $$ = mk_tran_and($1, $3); }
| actions OR actions         { $$ = mk_tran_or($1, $3); }
| actions IFF actions        { $$ = mk_tran_iff($1, $3); }
| actions IMPLY actions      { $$ = mk_tran_imply($1, $3); }
| LPAREN actions RPAREN      { $$ = $2; }
;

/* Sets */

set:
  LBRACE set_elems RBRACE    { $$ = $2; }
;

set_elems:
                             { $$ = 0; }
| non_null_set               { $$ = symbol_list_reverse($1); }
;

non_null_set:
  SYMBOL                     { $$ = one($1); }
| non_null_set COMMA SYMBOL  { $$ = mk_symbol_list($3, $1); }
;

/* Diagrams */

diagrams:
  diagram                    { $$ = mk_diagram_list($1, 0); }
| diagrams SEMICOLON diagram { $$ = mk_diagram_list($3, $1); }
;

diagram:
  LBRACK elems RBRACK states
  { $$ = mk_diagram($2, $4, 0, 0); }
| LBRACK elems RBRACK states LBRACK states SEMICOLON actions RBRACK
  { $$ = mk_diagram($2, $4, $6, $8); }
;

elems:
                             { $$ = 0; }
| non_null_elems opt_semicolon { $$ = arrow_list_reverse($1); }
;

opt_semicolon:

| SEMICOLON
;

non_null_elems:
  arrow                      { $$ = mk_arrow_list($1, 0); }
| non_null_elems SEMICOLON arrow  { $$ = mk_arrow_list($3, $1); }
;

arrow:
  states COMMA actions more  { $$ = mk_arrow($1, $3, $4); }
;

more:
                             { $$ = 0; }
| PLUS                       { $$ = 1; }
;

/* Labled Transition System */

lts:
  type_def role_def user_def class_def permission_def init trans specs
  { $$ = mk_lts($1, $2, $3, $4, $5, $6, $7, tran_list_reverse($8)); }
;

type_def:
  STATE TYPES IN set         { $$ = $4; }
;

role_def:
  STATE ROLES IN set         { $$ = $4; }
;

user_def:
  STATE USERS IN set         { $$ = $4; }
;

class_def:
  ACTION CLASSES IN set      { $$ = $4; }
;

permission_def:
  ACTION PERMISSIONS IN set  { $$ = $4; }
;

init:
  INIT states                { $$ = $2; }
;

trans:
  TRANS transitions          { $$ = $2; }
;

specs:
                             { $$ = 0; }
| specs SPEC transitions     { $$ = mk_tran_list($3, $1); }
;

%%
