%{

/* A lexical analyzer for formula syntax. */

#include <stdio.h>
#include <string.h>
#include "symbol.h"
#include "intern.h"
#include "formula.h"
#include "parser.h"

static int lineno = 1;

%}

ID		[A-Za-z_$][A-Za-z_$0-9]*

%%

[ \t\r\f]*	{  }
"\n"		{ lineno++; }
"#"		{ for (;;) {  /* delete comments */
                    int c = input();
                    if (c == EOF)
                      return YY_NULL;
                    else if (c == '\n')
                      break;
                  }
                  lineno++; }
"("             { return LPAREN; }
")"             { return RPAREN; }
"["             { return LBRACK; }
"]"             { return RBRACK; }
"{"             { return LBRACE; }
"}"             { return RBRACE; }
"+"             { return PLUS; }
","             { return COMMA; }
";"             { return SEMICOLON; }
"->"            { return IMPLY; }
"<->"           { return IFF; }
"|"             { return OR; }
"&"             { return AND; }
"!"             { return NOT; }
"="             { return EQ; }
"!="            { return NEQ; }
":"             { return IN; }
"!:"            { return NOTIN; }
"'"             { return NEXT; }
"c"             { return CLASSES; }
"p"             { return PERMISSIONS; }
"t"             { return TYPES; }
"r"             { return ROLES; }
"u"             { return USERS; }
"TRUE"          { return TRUE; }
"FALSE"         { return FALSE; }
"STATE"         { return STATE; }
"ACTION"        { return ACTION; }
"INIT"          { return INIT; }
"TRANS"         { return TRANS; }
"SPEC"          { return SPEC; }
{ID}		{ yylval.symbol = intern(yytext); return SYMBOL; }
.		{ return UNUSED; }

%%

static const char *filename = "-";

int 
setfile(const char *path)
{
  if (!path) {
    yyin = stdin;
    return 0;
  }
  filename = path;
  yyin = fopen(path, "r");
  return yyin ? 0 : -1;
}

int
yyerror(const char *msg) 
{
  return fprintf(stderr, "%s:%d: %s at token %s\n", 
                 filename, lineno, msg, yytext);
}
