/*
 * BufferWriter.cc -- Buffered Output
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>

#include "BufferWriter.h"

BufferWriter::BufferWriter( int _size ) : 
	size( _size ),
	used( 0 )
{
	buffer = new uint8_t[ _size ];
}

BufferWriter::~BufferWriter( )
{
	delete buffer;
}

int BufferWriter::FlushBuffer( )
{
	int bytes = Write( buffer, used );
	if ( bytes > 0 )
		used -= bytes;
	return used == 0;
}

int BufferWriter::PutBuffer( uint8_t *data, int length )
{
	int bytes = 0;
	int error = 0;
	
	while( !error && length )
	{
		if ( ( used + length ) < size )
		{
			memcpy( buffer + used, data + bytes, length );
			bytes += length;
			used += length;
			length = 0;
		}
		else if ( used != size )
		{
			int usable = size - used;
			memcpy( buffer + used, data + bytes, usable );
			length -= usable;
			bytes += usable;
			used += usable;
		}
			
		if ( used == size )
			error = !FlushBuffer( );
	}
	
	return bytes;
}

int BufferWriter::PutBuffer( uint8_t value )
{
	int bytes = 1;
	*( buffer + used ++ ) = value;
	if ( used == size )
		if ( !FlushBuffer( ) )
			bytes = 0;
	return bytes;
}

int BufferWriter::PutBuffer( unsigned long value )
{
	return PutBuffer( ( uint8_t ) value ) + 
		   PutBuffer( ( uint8_t )( value >> 8 ) ) + 
		   PutBuffer( ( uint8_t )( value >> 16 ) ) + 
		   PutBuffer( ( uint8_t )( value >> 24 ) );
}

int BufferWriter::PutBuffer( int16_t value )
{
	return PutBuffer( ( uint8_t ) value ) + 
		   PutBuffer( ( uint8_t )( value >> 8 ) );
}

int BufferWriter::PutBuffer( int16_t *values, int length )
{
	int index = 0;
	int bytes = 0;
	
	for ( index = 0; index < length; index ++ )
		bytes += PutBuffer( ( uint8_t ) values[ index ] ) + 
				 PutBuffer( ( uint8_t )( values[ index ] >> 8 ) );

	return bytes;
}


