/* xm.c: Guile/Ruby bindings for X/Xt/Xpm/Xm/Xp/Xext
 *   needs xen.h
 *   for tests and examples see snd-motif.scm, bess.scm, bess.rb, and snd-test.scm
 */

#include <config.h>

#define XM_DATE "30-Dec-04"

/* HISTORY: 
 *
 *   30-Dec:    plug various memory leaks.
 *   23-Nov:    resource type lookup indexing bugfix.
 *   22-Sep:    various minor cleanups.
 *   23-Aug:    more changes for new Guile.
 *   12-Aug:    some changes to accommodate new Guile names.
 *   22-June:   XmColorSelector resources.
 *   19-May:    plug several memory leaks.
 *   21-Apr:    XmMultiList, XmTabStack.
 *   19-Apr:    XmDataField.
 *   12-Apr:    XmDropDown, XmColumn.
 *   7-Apr:     XmButtonBox.
 *   22-Mar:    added feature 'Xp to indicate that the Xp stuff is included.
 *   8-Mar:     XtAppAddActionHook arity bugfix.
 *   12-Jan:    resources for XmFontSelector.
 *   8-Jan:     various changes for the SGI C compiler, thanks to Avi Bercovich.
 *   5-Jan-04:  added (Motif 2.2.3) XmCreateFontSelector, XmCreateColorSelector.
 *   --------
 *   1-Dec:     XShapeGetRectangles XRectangle array needed local allocation.
 *              removed (unusable) XtCallCallbackList.
 *   25-Nov:    more GC protection. Removed XInitThreads, XLockDisplay, XUnlockDisplay. Fixed XmTextBlock allocation bug.
 *   15-Oct:    XFontsOfFontSet indexing bugfix and several more struct field accessors from MS.
 *   14-Oct:    XShapeQueryExtension from Michael Scholz, plus other extensions/shape.h functions and constants.
 *              Also XSizeHints, XSet[Standard|WM]Properties and accessors for input and initial_state.
 *   3-Oct:     removed some macros that were intended only for testing, added XmTOP_LEFT etc.
 *   29-Sep:    changed: XSetWindowBorder, XSetWindowBackground, XCreatePixmapFromBitmapData, XSetWindowAttributes,
 *                 .backing_pixel, .border_pixel, .background_pixel, .base_pixel; these now use the "wrapped"
 *                 form of pixels, not the bare unsigned long.
 *   26-Sep:    .event -> #f if event is NULL.
 *   3-Sep:     XmRenderTableGetRenditions was incorrectly freeing the XmRenditions.
 *   20-Aug:    XtEventHandler *flag set to false if handler returns 'done.
 *   11-Aug:    int -> bool.
 *   17-July:   XpmAttributes .colorsymbols is a list.
 *   15-July:   type check cleanups.
 *   14-July:   .depths returns a list of Depth pointers; similar change for .visuals.
 *   11-July:   Several int->Dimension|Position|short resource type changes.
 *              removed several more undocumented resource names.
 *   23-June:   Motif 1 fixups.
 *   10-June:   added XmCvtByteStreamToXmString, XmCvtXmStringToByteStream, XmStringByteStreamLength.
 *              removed XFreeStringList (a no-op).
 *   20-May:    showValue resource is int (enumeration) in Motif 2.  resizeWidth|Height are booleans, not ints.
 *   9-May:     Ruby fixups (some functions accidentally omitted earlier).
 *   8-Apr:     XSetErrorHandler proc takes XErrorEvent, not XEvent 2nd arg (thanks Friedrich Delgado Friedrichs)
 *   7-Apr:     more changes for the WITH_GTK_AND_X11 switch.
 *   1-Apr:     XGetWindowProperty uses mem2string if not XA_STRING.
 *   31-Mar:    added WITH_GTK_AND_X11 switch for xg+local X funcs.
 *   4-Mar:     xm-ruby XM_DEFINE_ACCESSOR quoted SetName bugfix (Michael Scholz).
 *   1-Feb-03:  XChangeProperty data (arg7) can be list of ints as well as string.
 *   ----------
 *   19-Dec:    more Ruby fixups.
 *   6-Nov:     Ruby XmWMProtocols bugfixes thanks to Michael Scholz.
 *   17-Oct:    XtAppSetFallbackResources and fallbacks added to XtAppInitialize etc.
 *   15-Oct:    XtGetResourceList.
 *   11-Oct:    xm-ruby XM_DEFINE* cleaned up (thanks to Michael Scholz).
 *              removed all (ignored) XrmOptionDesc args (XtAppInitialize etc).
 *   8-Oct:     added WITH_EDITRES to include _XEditResCheckMessages.
 *   1-Oct:     some int args are now KeyCodes (Modifiermap stuff).
 *   23-Sep:    X ScreenSaver constants omitted earlier.
 *   18-Sep:    properties XFontStruct field now returns list of all properties.
 *              removed XmFontListCreate_r, XmFontListEntryCreate_r
 *   9-Sep:     added Motif WMProtocol convenience macros.
 *   30-Aug:    added add-resource for user-defined extensions to the resources.
 *   29-Aug:    added ten resources accidentally omitted earlier (thanks to Michael Scholz).
 *   2-Aug:     some Lesstif-related compile-time switches.
 *   26-Jul:    removed wrappers Widget Pixel GC and XtAppContext.
 *   24-Jul:    removed "|" prefix, use "." as default struct field prefix.
 *   19-Jul:    XM_FIELD_PREFIX for change from using vertical-bar ("|" is reserved in R5RS).
 *   17-Jun:    removed XtSetWMColormapWindows.
 *   29-Apr:    minor 64-bit fixups.
 *   29-Mar:    XmParseProc.
 *   20-Mar:    XpmGetErrorString omitted inadvertently earlier.
 *   4-Mar:     XWindowChanges and XSetWindowAttributes struct creators.
 *   1-Mar:     XmTabListFree, various ->* conversions (->strings etc).
 *   25-Feb:    XmTextBlock fields
 *   22-Feb:    #f = NULL and vice-versa throughout
 *   21-Feb:    added various callback struct makers, changed XtCallCallbacks to be compatible with them.
 *   18-Feb:    removed undocumented functions: XmCvtFromHorizontalPixels, XmCvtFromVerticalPixels, XmCvtToHorizontalPixels, XInitImage
 *                XmCvtToVerticalPixels, XmGetIconFileName, XmStringCreateFontList, XmStringCreateFontList_r, XmStringLtoRCreate
 *              added XM_DISABLE_DEPRECATED which affects:
 *                XtError, XtSetErrorHandler, XtSetWarningHandler, XtSetErrorMsgHandler, XtSetWarningMsgHandler, XtWarningMsg, XtAppWarning, XtErrorMsg
 *                XtSetSelectionTimeout, XtInitialize , XtAddActions, XtAddInput, XtAddTimeout, XtAddWorkProc, XtCreateApplicationShell
 *                XtNextEvent, XtPeekEvent, XtPending, XtProcessEvent, XtMainLoop, XtGetSelectionTimeout, XtWarning,
 *                XmNdefaultFontList, XmNshellUnitType, XmNfontList, XmFontList*, XmNchildType, XmGetMenuCursor, 
 *                XmNstringDirection, XmMainWindowSep1, XmMainWindowSep2, XmMainWindowSep3, XmMainWindowSetAreas, 
 *                XmNwhichButton, XmScrolledWindowSetAreas, XmSetFontUnit, XmSetFontUnits, XmNdefaultFontList, 
 *                XmSetMenuCursor, XmStringByteCompare, XmStringCreateLtoR, XmSTRING_COMPONENT_CHARSET, XmSTRING_COMPONENT_FONTLIST_ELEMENT_TAG,
 *                XmStringCreateSimple, XmStringGetLtoR, XmStringGetNextComponent, XmStringGetNextSegment, XmStringLength, XmStringNConcat
 *                XmStringNCopy, XmStringPeekNextComponent, XmStringSegmentCreate, XmTrackingLocate, XmRemoveFrom|AddTo|PostFromList
 *                XGet|SetStandardColormap
 *              added XmWidgetGetDisplayRect inadvertently omitted earlier.
 *   14-Feb:    XUniqueContext added, XExtentsOfFontSet and XGetErrorDatabaseText deleted.
 *              X save-set and X host-address stuff deleted.
 *              XVisualInfo fields added. 
 *              Cursor type added.
 *   1-Feb:     Motif 2.2 additions (tooltip).
 *   21-Jan:    Ruby fixups (XEN_COPY_ARG to protect lists)
 *   7-Jan-02:  XEvent fields settable. added XtCallCallbacks-raw.
 *   ----------
 *   12-Sep:    xm-version.
 *   13-Aug:    Xp bindings, X11 predefined Atoms.
 *   6-Aug:     XmTransfer functions inadvertently omitted earlier.
 *   3-Aug:     type checks in XtSetValues.
 *   23-Jul-01: use lists rather than vectors.
 */

#if UNDEF_USE_SND
  #undef USE_SND
  #define USE_SND 0
#endif

#if HAVE_EXTENSION_LANGUAGE

#if HAVE_MOTIF
  #include <Xm/XmAll.h>
#else
  #include <X11/Xlib.h>
  #include <X11/Intrinsic.h>
  #include <X11/Shell.h>
  #include <X11/Xatom.h>
#endif

#include <X11/keysym.h>
#include <X11/cursorfont.h>
#include <stdio.h>

#if HAVE_XSHAPEQUERYEXTENSION
#include <X11/extensions/shape.h>
#endif

/* compile-time flags are HAVE_XPM HAVE_MOTIF HAVE_XM_XP HAVE_GUILE|HAVE_RUBY XM_DISABLE_DEPRECATED LESSTIF_VERSION WITH_EDITRES WITH_GTK_AND_X11 */

/* if you're using g++ and it complains about XmRemoveFromPostFromList, update Motif (you need 2.1.30) */

#if USE_SND
  /* USE_SND causes xm to use Snd's error handlers which are much smarter than xen's fallback versions */
  #include "snd.h"
#else
  #include "xen.h"
  /* in snd-0.h: */
  #if HAVE_RUBY
    #define S_setB "set_"
  #else
    #define S_setB "set! "
  #endif
#endif

#ifndef CALLOC
  #define CALLOC(a, b)  calloc((size_t)(a), (size_t)(b))
  #define MALLOC(a)     malloc((size_t)(a))
  #define FREE(a)       free(a)
  #define REALLOC(a, b) realloc(a, (size_t)(b))
#endif

#if HAVE_XPM
  #include <X11/xpm.h>
#endif
#if HAVE_XmCreateDataField
  #include <Xm/DataF.h>
#endif
#if HAVE_XmCreateMultiList
  #include <Xm/MultiList.h>
#endif
#if HAVE_XmCreateTabStack
  #include <Xm/TabStack.h>
#endif
#if HAVE_XmCreateButtonBox
  #include <Xm/ButtonBox.h>
#endif
#if HAVE_XmCreateColumn
  #include <Xm/Column.h>
#endif
#if HAVE_XmCreateDropDown
  #include <Xm/DropDown.h>
#endif
#if HAVE_XmCreateFontSelector
  #include <Xm/FontS.h>
#endif
#if HAVE_XmCreateColorSelector
  #include <Xm/ColorS.h>
#endif
#if WITH_EDITRES
  #include <X11/Xmu/Editres.h>
#endif

/* prefix for all names */
#if HAVE_GUILE
  #define XM_PREFIX ""
  #define XM_POSTFIX ""
  #define XM_FIELD_PREFIX "."
#else
/* for Ruby, XM PREFIX needs to be uppercase */
  #define XM_PREFIX "R"
  #define XM_POSTFIX ""
  #define XM_FIELD_PREFIX "R"
#endif

#define XM_FIELD_ASSERT_TYPE(Assertion, Arg, Position, Caller, Correct_Type) \
  XEN_ASSERT_TYPE(Assertion, Arg, Position, XM_FIELD_PREFIX Caller XM_POSTFIX, Correct_Type)

#if HAVE_RUBY
#define XM_SET_FIELD_ASSERT_TYPE(Assertion, Arg, Position, Caller, Correct_Type) \
  XEN_ASSERT_TYPE(Assertion, Arg, Position, XM_FIELD_PREFIX S_setB Caller XM_POSTFIX, Correct_Type)
#else
#define XM_SET_FIELD_ASSERT_TYPE(Assertion, Arg, Position, Caller, Correct_Type) \
  XEN_ASSERT_TYPE(Assertion, Arg, Position, S_setB XM_FIELD_PREFIX Caller XM_POSTFIX, Correct_Type)
#endif

#if HAVE_MOTIF
  #define MOTIF_2 (XmVERSION >= 2)
  /* 2.1 really -- I don't have a test system for 2.0 -- I'm using LessTif's headers to check 1.2 */
#endif

#if WITH_GTK_AND_X11
  #include "xg-x11.h"
#endif

#if (XmVERSION == 1) && (HAVE_XM_XP)
  #undef HAVE_XM_XP
  #define HAVE_XM_XP 0
#endif

#define XtIsSubClass XtIsSubclass
/* kludge around a bug in some Motif versions */

/* layout of file:
 *    preliminaries
 *    type manglers
 *    arglist handlers
 *    Motif procedures
 *    X procedures
 *    Xt procedures
 *    Xp procedures
 *    Xpm procedures
 *    struct handlers
 *    string constants
 *    integer constants
 *    pointer constants
 *    atom constants
 */

/* --------------------------------------------------------------------------------
 * a sample program

(let* ((shell-app (XtVaOpenApplication 
		    "Test" 0 '() applicationShellWidgetClass
		    (list XmNallowShellResize #t)))
       (app (cadr shell-app))
       (shell (car shell-app))
       (black (BlackPixelOfScreen 
		(DefaultScreenOfDisplay 
		  (XtDisplay shell)))))
  (if (not (XtIsApplicationShell shell))
      (display "not appshell"?))
  (XtSetValues shell (list XmNtitle "Hi!"))
  (let* ((main-pane 
	  (XtVaCreateManagedWidget 
	    "main-pane" xmFormWidgetClass shell
	    (list XmNforeground       black
		  XmNtopAttachment    XmATTACH_FORM
		  XmNbottomAttachment XmATTACH_FORM
		  XmNleftAttachment   XmATTACH_FORM
		  XmNrightAttachment  XmATTACH_FORM
		  XmNallowResize      #t)))
	 (button (XtCreateManagedWidget 
		   "push me" xmPushButtonWidgetClass main-pane '() 0)))
    (XtAddCallback button XmNactivateCallback 
		    (lambda (widget context event-info)
		      (display widget)
		      (display (.reason event-info))
		      (display context))
		    123)
    (XtRealizeWidget shell)
    (XtAppMainLoop app)))
 */


/* --------------------------------------------------------------------------------
 * differences from C:
 *
 *    Arg list is lisp list of name/value pairs and the "len" arg associated with it is optional
 *    ref args are usually returned by proc, and not passed in unless init val is needed
 *    array args are passed as lists, returned as lists
 *    pointers to structs are '(type val) where val is opaque except via accessors 
 *      that is, all non-simple types are lists in xm where the first element is a symbol describing the type
 *      and the second is usually the C value stored directly as an unsigned long
 *    "Va" args are passed and returned as lists
 *    XtCallback procedure args are passed by value
 *    various "client data" args are optional
 *    XtCallbackLists are passed as lists of procedure/data pairs
 *    where explicit NULL is needed as arg, use #f (or '() for list args)
 *
 * omitted:       
 *
 *    Xwc*, Xrm*, XIM*, XOM*, XIC*, XOC*, Xcms*, 
 *    "resource" handlers and "quark" stuff (XtAppSetTypeConverter etc)
 *    all XTextProperty, XClassHint, XSizeHints support, XSetIconSizes
 *    XConnection internal stuff, XVaCreateNestedList, XSetAuthorization
 *    XtQueryGeometry and XtMakeGeometryRequest, all text16 stuff
 *    XtGetKeySymTable, XtChangeManagedSet (undocumented), XtAppGet|SetExitFlag
 *    XtSetEventDispatcher (undoc), XtSignal stuff (undoc), XtGetErrorDatabaseText
 *    XtBlockHook stuff (undoc), XtRegisterExtensionSelector (undoc)
 *    XmResolvePartOffsets, XmResolveAllPartOffsets
 *    XpSet|GetLocaleHinter, XFreeStringList
 *    XtHooksOfDisplay, XtRegiserDrawable, XtUnregisterDrawable
 *
 * added:
 *
 *    XGCValues -> a blank XGCValues struct (for XCreateGC etc)
 *    XColor pixel red green blue flags pad
 *    XArc XRectangle XPoint XSegment XEvent XWindowChanges XSetWindowAttributes
 *    XTextItem XpmImage XpmColorSymbol
 *    XDrawLinesDirect same as XDrawLines but takes (opaque) ptr to XPoint array
 *    vector->XPoints vect packages point data in vector as (opaque) array of XPoints 
 *    freeXPoints to free (opaque) XPoint array created by vector->Xpoints
 *    <name>? -> #t if arg is of type <name>
 *    <name> -> empty struct of type <name>
 *
 * Structs are accessed by the field name and the lisp variable (which contains the struct type)
 *   (.pixel color) for example, or (.foreground gcvalue)
 */
 


/* -------------------------------- smob for GC -------------------------------- */

/* "smob" for locally created stuff (X graphics entities mainly) */

static XEN_OBJECT_TYPE xm_obj_tag;

#if HAVE_GUILE
static size_t xm_obj_free(XEN obj)
{
  void *val;
  val = (void *)XEN_OBJECT_REF(obj);
  FREE(val);
  return(0);
}
#endif
#if HAVE_RUBY
static void *xm_obj_free(XEN obj)
{
  FREE((void *)obj);
  return(NULL);
}
#endif

static XEN make_xm_obj(void *ptr)
{
  XEN_MAKE_AND_RETURN_OBJECT(xm_obj_tag, ptr, 0, xm_obj_free);
}

static void define_xm_obj(void)
{
  xm_obj_tag = XEN_MAKE_OBJECT_TYPE("XmObj", sizeof(void *));
#if HAVE_GUILE
  scm_set_smob_free(xm_obj_tag, xm_obj_free);
#endif
}  


/* -------------------------------- type manglers -------------------------------- */

/* most non-simple vars are handled as a list: (type ptr) where ptr is the (un-interpreted) C value
 *   the _OBJ form is used where we need to eventually free the memory
 *   the _PTR form treats NULL as #f and vice-versa
 */

#define WRAP_FOR_XEN(Name, Value) XEN_LIST_2(C_STRING_TO_XEN_SYMBOL(Name), C_TO_XEN_ULONG((unsigned long)Value))
#define WRAP_FOR_XEN_OBJ(Name, Value) XEN_LIST_3(C_STRING_TO_XEN_SYMBOL(Name), C_TO_XEN_ULONG((unsigned long)Value), make_xm_obj(Value))
#define WRAP_P(Name, Value) (XEN_LIST_P(Value) && \
                            (XEN_LIST_LENGTH(Value) >= 2) && \
                            (XEN_SYMBOL_P(XEN_CAR(Value))) && \
                            (strcmp(Name, XEN_SYMBOL_TO_C_STRING(XEN_CAR(Value))) == 0))

#define XM_TYPE(Name, XType) \
  static XEN C_TO_XEN_ ## Name (XType val) {return(WRAP_FOR_XEN(#Name, val));} \
  static XType XEN_TO_C_ ## Name (XEN val) {return((XType)XEN_TO_C_ULONG(XEN_CADR(val)));} \
  static bool XEN_ ## Name ## _P(XEN val) {return(WRAP_P(#Name, val));} \
  static XEN XEN_ ## Name ## _p(XEN val) {return(C_TO_XEN_BOOLEAN(WRAP_P(#Name, val)));}

#define XM_TYPE_NO_p(Name, XType) \
  static XEN C_TO_XEN_ ## Name (XType val) {return(WRAP_FOR_XEN(#Name, val));} \
  static XType XEN_TO_C_ ## Name (XEN val) {return((XType)XEN_TO_C_ULONG(XEN_CADR(val)));} \
  static bool XEN_ ## Name ## _P(XEN val) {return(WRAP_P(#Name, val));}

#define XM_TYPE_PTR(Name, XType) \
  static XEN C_TO_XEN_ ## Name (XType val) {if (val) return(WRAP_FOR_XEN(#Name, val)); return(XEN_FALSE);} \
  static XType XEN_TO_C_ ## Name (XEN val) {if (XEN_FALSE_P(val)) return((XType)NULL); return((XType)XEN_TO_C_ULONG(XEN_CADR(val)));} \
  static bool XEN_ ## Name ## _P(XEN val) {return(WRAP_P(#Name, val));} \
  static XEN XEN_ ## Name ## _p(XEN val) {return(C_TO_XEN_BOOLEAN(WRAP_P(#Name, val)));}

#define XM_TYPE_PTR_NO_p(Name, XType) \
  static XEN C_TO_XEN_ ## Name (XType val) {if (val) return(WRAP_FOR_XEN(#Name, val)); return(XEN_FALSE);} \
  static XType XEN_TO_C_ ## Name (XEN val) {if (XEN_FALSE_P(val)) return((XType)NULL); return((XType)XEN_TO_C_ULONG(XEN_CADR(val)));} \
  static bool XEN_ ## Name ## _P(XEN val) {return(WRAP_P(#Name, val));}

#define XM_TYPE_PTR_NO_p_NO_P(Name, XType) \
  static XEN C_TO_XEN_ ## Name (XType val) {if (val) return(WRAP_FOR_XEN(#Name, val)); return(XEN_FALSE);} \
  static XType XEN_TO_C_ ## Name (XEN val) {if (XEN_FALSE_P(val)) return((XType)NULL); return((XType)XEN_TO_C_ULONG(XEN_CADR(val)));}

#define XM_TYPE_PTR_NO_C2X(Name, XType) \
  static XType XEN_TO_C_ ## Name (XEN val) {if (XEN_FALSE_P(val)) return((XType)NULL); return((XType)XEN_TO_C_ULONG(XEN_CADR(val)));} \
  static bool XEN_ ## Name ## _P(XEN val) {return(WRAP_P(#Name, val));} \
  static XEN XEN_ ## Name ## _p(XEN val) {return(C_TO_XEN_BOOLEAN(WRAP_P(#Name, val)));}

#define XM_TYPE_PTR_NO_C2X_NO_p(Name, XType) \
  static XType XEN_TO_C_ ## Name (XEN val) {if (XEN_FALSE_P(val)) return((XType)NULL); return((XType)XEN_TO_C_ULONG(XEN_CADR(val)));} \
  static bool XEN_ ## Name ## _P(XEN val) {return(WRAP_P(#Name, val));}

#define XM_TYPE_PTR_OBJ(Name, XType) \
  static XEN C_TO_XEN_ ## Name (XType val) {if (val) return(WRAP_FOR_XEN_OBJ(#Name, val)); return(XEN_FALSE);} \
  static XType XEN_TO_C_ ## Name (XEN val) {if (XEN_FALSE_P(val)) return(NULL); return((XType)XEN_TO_C_ULONG(XEN_CADR(val)));} \
  static bool XEN_ ## Name ## _P(XEN val) {return(WRAP_P(#Name, val));} \
  static XEN XEN_ ## Name ## _p(XEN val) {return(C_TO_XEN_BOOLEAN(WRAP_P(#Name, val)));}


#define XEN_TO_C_Dimension(Arg)  (Dimension)(XEN_TO_C_INT(Arg))
#define C_TO_XEN_Dimension(Arg)  (C_TO_XEN_INT((Dimension)Arg))
#define XEN_Dimension_P(Arg)     (XEN_INTEGER_P(Arg))

#define XEN_TO_C_Position(Arg)   (Position)(XEN_TO_C_INT(Arg))
#define C_TO_XEN_Position(Arg)   (C_TO_XEN_INT((Position)Arg))
#define XEN_Position_P(Arg)      (XEN_INTEGER_P(Arg))

#define XEN_TO_C_char(Arg)       (char)(XEN_TO_C_INT(Arg))
#define C_TO_XEN_char(Arg)       (C_TO_XEN_INT(Arg))
#define XEN_char_P(Arg)          (XEN_INTEGER_P(Arg))

#define XEN_TO_C_Modifiers(Arg)  (Modifiers)(XEN_TO_C_ULONG(Arg))
#define C_TO_XEN_Modifiers(Arg)  (C_TO_XEN_ULONG(Arg))
#define XEN_Modifiers_P(Arg)     (XEN_ULONG_P(Arg))

XM_TYPE(Cursor, Cursor)
XM_TYPE_PTR(Screen, Screen *)
XM_TYPE_PTR_OBJ(XRectangle, XRectangle *)
XM_TYPE_PTR_OBJ(XArc, XArc *)
XM_TYPE_PTR_OBJ(XPoint, XPoint *)
XM_TYPE_PTR_OBJ(XSegment, XSegment *)
XM_TYPE_PTR_OBJ(XColor, XColor *)
XM_TYPE(Atom, Atom)
XM_TYPE_PTR(Colormap, Colormap)
XM_TYPE_PTR_NO_C2X(Depth, Depth *)
XM_TYPE_PTR(Display, Display *)
XM_TYPE(Font, Font)
XM_TYPE(GC, GC)
XM_TYPE(KeySym, KeySym)
XM_TYPE(Pixel, Pixel)
XM_TYPE(Pixmap, Pixmap)
XM_TYPE(Region, Region)
XM_TYPE(Time, Time)
XM_TYPE_PTR(Visual, Visual *)
XM_TYPE_PTR(Window, Window)
XM_TYPE_PTR(XCharStruct, XCharStruct *)
XM_TYPE_PTR(XFontProp, XFontProp *)
XM_TYPE(XFontSet, XFontSet)
XM_TYPE_PTR(XFontStruct, XFontStruct *)
XM_TYPE_PTR(XGCValues, XGCValues *)
XM_TYPE_PTR_NO_C2X(XTextItem, XTextItem *)
XM_TYPE_PTR(XModifierKeymap, XModifierKeymap *) /* opaque in this implementation */
XM_TYPE_PTR(XImage, XImage *)
static XAnyEvent *XEN_TO_C_XAnyEvent(XEN val) {return((XAnyEvent *)XEN_TO_C_ULONG(XEN_CADR(val)));}
XM_TYPE_PTR_NO_C2X(XButtonEvent, XButtonEvent *)
XM_TYPE_PTR_NO_C2X(XCirculateEvent, XCirculateEvent *)
XM_TYPE_PTR_NO_C2X(XCirculateRequestEvent, XCirculateRequestEvent *)
XM_TYPE_PTR_NO_C2X(XClientMessageEvent, XClientMessageEvent *)
XM_TYPE_PTR_NO_C2X(XColormapEvent, XColormapEvent *)
XM_TYPE_PTR_NO_C2X(XConfigureEvent, XConfigureEvent *)
XM_TYPE_PTR_NO_C2X(XConfigureRequestEvent, XConfigureRequestEvent *)
XM_TYPE_PTR_NO_C2X(XCreateWindowEvent, XCreateWindowEvent *)
XM_TYPE_PTR_NO_C2X(XCrossingEvent, XCrossingEvent *)
XM_TYPE_PTR_NO_C2X(XDestroyWindowEvent, XDestroyWindowEvent *)
XM_TYPE_PTR(XErrorEvent, XErrorEvent *)
XM_TYPE_PTR_NO_C2X(XExposeEvent, XExposeEvent *)
XM_TYPE_PTR_NO_C2X(XFocusChangeEvent, XFocusChangeEvent *)
XM_TYPE_PTR_NO_C2X(XGraphicsExposeEvent, XGraphicsExposeEvent *)
XM_TYPE_PTR_NO_C2X(XGravityEvent, XGravityEvent *)
XM_TYPE_PTR_NO_C2X(XKeyEvent, XKeyEvent *)
XM_TYPE_PTR_NO_C2X(XKeymapEvent, XKeymapEvent *)
XM_TYPE_PTR_NO_C2X(XMapEvent, XMapEvent *)
XM_TYPE_PTR_NO_C2X(XMapRequestEvent, XMapRequestEvent *)
XM_TYPE_PTR_NO_C2X(XMappingEvent, XMappingEvent *)
XM_TYPE_PTR_NO_C2X(XMotionEvent, XMotionEvent *)
XM_TYPE_PTR_NO_C2X(XNoExposeEvent, XNoExposeEvent *)
XM_TYPE_PTR_NO_C2X(XPropertyEvent, XPropertyEvent *)
XM_TYPE_PTR_NO_C2X(XReparentEvent, XReparentEvent *)
XM_TYPE_PTR_NO_C2X(XResizeRequestEvent, XResizeRequestEvent *)
XM_TYPE_PTR_NO_C2X(XSelectionClearEvent, XSelectionClearEvent *)
XM_TYPE_PTR_NO_C2X(XSelectionEvent, XSelectionEvent *)
XM_TYPE_PTR(XSelectionRequestEvent, XSelectionRequestEvent *)
XM_TYPE_PTR_NO_C2X(XUnmapEvent, XUnmapEvent *)
XM_TYPE_PTR_NO_C2X(XVisibilityEvent, XVisibilityEvent *)
XM_TYPE_PTR_OBJ(XSetWindowAttributes, XSetWindowAttributes *)
XM_TYPE_PTR(XVisualInfo, XVisualInfo *)
XM_TYPE_PTR(XWMHints, XWMHints *)
XM_TYPE_PTR_NO_C2X_NO_p(XSizeHints, XSizeHints *)
XM_TYPE_PTR(XWindowAttributes, XWindowAttributes *)
XM_TYPE_PTR_OBJ(XWindowChanges, XWindowChanges *)
XM_TYPE_PTR(XStandardColormap, XStandardColormap *)
XM_TYPE(KeyCode, KeyCode)
XM_TYPE(XContext, XContext)
XM_TYPE_PTR(XIconSize, XIconSize *)

#if HAVE_XM_XP
XM_TYPE(XPContext, XPContext)
#endif

#if HAVE_MOTIF
XM_TYPE_PTR(Widget, Widget)
XM_TYPE(WidgetClass, WidgetClass)
XM_TYPE(XtAppContext, XtAppContext)
XM_TYPE(XtRequestId, XtRequestId)
XM_TYPE(XtWorkProcId, XtWorkProcId)
XM_TYPE(XtInputId, XtInputId)
XM_TYPE(XtIntervalId, XtIntervalId)
XM_TYPE_NO_p(XtActionHookId, XtActionHookId)
XM_TYPE_NO_p(XtTranslations, XtTranslations) /* opaque */
XM_TYPE_PTR(XmString, XmString)
XM_TYPE_PTR_NO_p_NO_P(XmAnyCallbackStruct, XmAnyCallbackStruct *)
XM_TYPE_PTR_NO_p(XmArrowButtonCallbackStruct, XmArrowButtonCallbackStruct *)
XM_TYPE_PTR_NO_p(XmCommandCallbackStruct, XmCommandCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDragDropFinishCallbackStruct, XmDragDropFinishCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDragMotionCallbackStruct, XmDragMotionCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDragProcCallbackStruct, XmDragProcCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDrawingAreaCallbackStruct, XmDrawingAreaCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDrawnButtonCallbackStruct, XmDrawnButtonCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDropFinishCallbackStruct, XmDropFinishCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDropProcCallbackStruct, XmDropProcCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDropSiteEnterCallbackStruct, XmDropSiteEnterCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDropSiteLeaveCallbackStruct, XmDropSiteLeaveCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDropStartCallbackStruct, XmDropStartCallbackStruct *)
XM_TYPE_PTR_NO_p(XmFileSelectionBoxCallbackStruct, XmFileSelectionBoxCallbackStruct *)
XM_TYPE_PTR_NO_p(XmListCallbackStruct, XmListCallbackStruct *)
#if MOTIF_2
XM_TYPE(XmTab, XmTab) /* opaque */
XM_TYPE_PTR_NO_p(XmDragStartCallbackStruct, XmDragStartCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDisplayCallbackStruct, XmDisplayCallbackStruct *)
XM_TYPE_PTR_NO_p(XmDestinationCallbackStruct, XmDestinationCallbackStruct *)
XM_TYPE_PTR_NO_p(XmConvertCallbackStruct, XmConvertCallbackStruct *)
XM_TYPE_PTR_NO_p(XmComboBoxCallbackStruct, XmComboBoxCallbackStruct *)
XM_TYPE_PTR_NO_p(XmContainerOutlineCallbackStruct, XmContainerOutlineCallbackStruct *)
XM_TYPE_PTR_NO_p(XmContainerSelectCallbackStruct, XmContainerSelectCallbackStruct *)
XM_TYPE_PTR_NO_p(XmNotebookCallbackStruct, XmNotebookCallbackStruct *)
XM_TYPE_PTR_NO_p(XmNotebookPageInfo, XmNotebookPageInfo *)
XM_TYPE_PTR(XmRenderTable, XmRenderTable)
XM_TYPE_PTR(XmRendition, XmRendition)
XM_TYPE_PTR_NO_p(XmSpinBoxCallbackStruct, XmSpinBoxCallbackStruct *)
XM_TYPE_PTR_NO_p(XmTraverseObscuredCallbackStruct, XmTraverseObscuredCallbackStruct *)
XM_TYPE_PTR_NO_p(XmTopLevelLeaveCallbackStruct, XmTopLevelLeaveCallbackStruct *)
XM_TYPE_PTR_NO_p(XmTopLevelEnterCallbackStruct, XmTopLevelEnterCallbackStruct *)
#if HAVE_XM_XP
XM_TYPE_PTR_NO_p(XmPrintShellCallbackStruct, XmPrintShellCallbackStruct *)
#endif
XM_TYPE_PTR_NO_p(XmPopupHandlerCallbackStruct, XmPopupHandlerCallbackStruct *)
XM_TYPE_PTR_NO_p(XmSelectionCallbackStruct, XmSelectionCallbackStruct *)
XM_TYPE_PTR_NO_C2X_NO_p(XmTransferDoneCallbackStruct, XmTransferDoneCallbackStruct *)
XM_TYPE_PTR(XmTabList, XmTabList) /* opaque */
XM_TYPE(XmParseMapping, XmParseMapping)
#endif
XM_TYPE_PTR_NO_p(XmOperationChangedCallbackStruct, XmOperationChangedCallbackStruct *)
XM_TYPE_PTR_NO_p(XmPushButtonCallbackStruct, XmPushButtonCallbackStruct *)
XM_TYPE_PTR_NO_p(XmRowColumnCallbackStruct, XmRowColumnCallbackStruct *)
XM_TYPE_PTR_NO_p(XmScaleCallbackStruct, XmScaleCallbackStruct *)
XM_TYPE_PTR_NO_p(XmScrollBarCallbackStruct, XmScrollBarCallbackStruct *)
XM_TYPE_PTR_NO_p(XmSelectionBoxCallbackStruct, XmSelectionBoxCallbackStruct *)
XM_TYPE_PTR_NO_p(XmTextVerifyCallbackStruct, XmTextVerifyCallbackStruct *)
XM_TYPE_PTR_NO_C2X_NO_p(XmTextBlock, XmTextBlock)
XM_TYPE_PTR_NO_p(XmToggleButtonCallbackStruct, XmToggleButtonCallbackStruct *)
#if HAVE_XmCreateDataField
XM_TYPE_PTR_NO_p(XmDataFieldCallbackStruct, XmDataFieldCallbackStruct *)
#endif
#if HAVE_XmCreateMultiList
XM_TYPE_PTR_NO_p(XmMultiListCallbackStruct, XmMultiListCallbackStruct *)
XM_TYPE_PTR_NO_p(XmMultiListRowInfo, XmMultiListRowInfo *)
#endif
#if HAVE_XmCreateTabStack
XM_TYPE_PTR_NO_p(XmTabStackCallbackStruct, XmTabStackCallbackStruct *)
#endif
#if (!XM_DISABLE_DEPRECATED)
  XM_TYPE(XmFontContext, XmFontContext) /* opaque */
  XM_TYPE(XmFontList, XmFontList) /* opaque, obsolete == XmRenderTable in motif 2 */
  XM_TYPE(XmFontListEntry, XmFontListEntry) /* opaque */
#else
  #define XEN_TO_C_XmFontList(Arg) XEN_TO_C_XmRenderTable(Arg)
#endif
XM_TYPE(XmTextSource, XmTextSource)
XM_TYPE(XmStringContext, XmStringContext)

static int XEN_XmFontList_or_XmRenderTable_P(XEN arg)
{
#if (!XM_DISABLE_DEPRECATED)
#if MOTIF_2
  return(XEN_XmFontList_P(arg) || XEN_XmRenderTable_P(arg));
#else
  return(XEN_XmFontList_P(arg));
#endif
#else
  return(XEN_XmRenderTable_P(arg));
#endif
}
#endif

static XEN type_to_event_symbol(int utype)
{
  char *type;
  type = "XErrorEvent"; /* -1 for an error event? */
  switch (utype)
    {
    case 0:                type = "XAnyEvent"; break;
    case KeyPress:
    case KeyRelease:       type = "XKeyEvent"; break;
    case ButtonPress:
    case ButtonRelease:    type = "XButtonEvent"; break;
    case MotionNotify:     type = "XMotionEvent"; break;
    case EnterNotify:
    case LeaveNotify:      type = "XCrossingEvent"; break;
    case FocusIn:
    case FocusOut:         type = "XFocusChangeEvent"; break;
    case KeymapNotify:     type = "XKeymapEvent"; break;
    case Expose:           type = "XExposeEvent"; break;
    case GraphicsExpose:   type = "XGraphicsExposeEvent"; break;
    case NoExpose:         type = "XNoExposeEvent"; break;
    case VisibilityNotify: type = "XVisibilityEvent"; break;
    case CreateNotify:     type = "XCreateWindowEvent"; break;
    case DestroyNotify:    type = "XDestroyWindowEvent"; break;
    case UnmapNotify:      type = "XUnmapEvent"; break;
    case MapNotify:        type = "XMapEvent"; break;
    case MapRequest:       type = "XMapRequestEvent"; break;
    case ReparentNotify:   type = "XReparentEvent"; break;
    case ConfigureNotify:  type = "XConfigureEvent"; break;
    case ConfigureRequest: type = "XConfigureRequestEvent"; break;
    case GravityNotify:    type = "XGravityEvent"; break;
    case ResizeRequest:    type = "XResizeRequestEvent"; break;
    case CirculateNotify:  type = "XCirculateEvent"; break;
    case CirculateRequest: type = "XCirculateRequestEvent"; break;
    case PropertyNotify:   type = "XPropertyEvent"; break;
    case SelectionClear:   type = "XSelectionClearEvent"; break;
    case SelectionRequest: type = "XSelectionRequestEvent"; break;
    case SelectionNotify:  type = "XSelectionEvent"; break;
    case ColormapNotify:   type = "XColormapEvent"; break;
    case ClientMessage:    type = "XClientMessageEvent"; break;
    case MappingNotify:    type = "XMappingEvent"; break;
    }
  return(C_STRING_TO_XEN_SYMBOL(type));
}

static XEN C_TO_XEN_XEvent_1(XEvent *e, int need_free)
{
  if (e == NULL) return(XEN_FALSE); /* synthetic callback may have no event */
  if (need_free)
    return(XEN_LIST_4(type_to_event_symbol(e->type),
		      C_TO_XEN_ULONG((unsigned long)e),
		      make_xm_obj(e),
		      C_STRING_TO_XEN_SYMBOL("XEvent")));
  return(XEN_LIST_4(type_to_event_symbol(e->type),
		    C_TO_XEN_ULONG((unsigned long)e),
		    XEN_FALSE,
		    C_STRING_TO_XEN_SYMBOL("XEvent")));
}


#define C_TO_XEN_XEvent(e)     C_TO_XEN_XEvent_1(e, false)
#define C_TO_XEN_XEvent_OBJ(e) C_TO_XEN_XEvent_1(e, true)
#define XEN_TO_C_XEvent(Arg)   (XEvent *)XEN_TO_C_ULONG(XEN_CADR(Arg))
#define XEN_XEvent_P(Value)    (XEN_LIST_P(Value) &&\
                               (XEN_LIST_LENGTH(Value) == 4) &&\
                               (XEN_SYMBOL_P(XEN_CADDDR(Value))) &&\
                               (strcmp("XEvent", XEN_SYMBOL_TO_C_STRING(XEN_CADDDR(Value))) == 0))

static XEN XEN_XEvent_p(XEN val) 
{
  return(C_TO_XEN_BOOLEAN(XEN_XEvent_P(val)));
}

static XEN gxm_XEvent(XEN type)
{
  XEvent *e;
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(type), type, XEN_ONLY_ARG, "XEvent", "an X event type (integer)");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  if (XEN_INTEGER_P(type))
    e->type = XEN_TO_C_INT(type);
  return(XEN_LIST_4(type_to_event_symbol(e->type),
		    C_TO_XEN_ULONG((unsigned long)e),
		    make_xm_obj(e),
		    C_STRING_TO_XEN_SYMBOL("XEvent")));
}

static XEN gxm_XGCValues(void) 
{
  XGCValues *e;
  e = (XGCValues *)CALLOC(1, sizeof(XGCValues));
  return(WRAP_FOR_XEN_OBJ("XGCValues", e));
}

#if HAVE_MOTIF
#if HAVE_RUBY
#define XM_Make(Name) \
  static XEN gxm_ ## Name (void) {Name *e; e = (Name *)CALLOC(1, sizeof(Name)); return(WRAP_FOR_XEN_OBJ(#Name, e));} \
  XEN_NARGIFY_0(gxm_ ## Name ## _w, gxm_ ## Name)
#define XM_Declare(Name) \
  XEN_DEFINE_PROCEDURE(XM_PREFIX #Name XM_POSTFIX, gxm_ ## Name ## _w, 0, 0, 0, "Make an " #Name " struct")
#else
#define XM_Make(Name) \
  static XEN gxm_ ## Name (void) {Name *e; e = (Name *)CALLOC(1, sizeof(Name)); return(WRAP_FOR_XEN_OBJ(#Name, e));}
#define XM_Declare(Name) \
  XEN_DEFINE_PROCEDURE(XM_PREFIX #Name XM_POSTFIX, gxm_ ## Name, 0, 0, 0, "Make an " #Name " struct")
#endif

XM_Make(XmAnyCallbackStruct)
XM_Make(XmArrowButtonCallbackStruct)
XM_Make(XmCommandCallbackStruct)
XM_Make(XmDragDropFinishCallbackStruct)
XM_Make(XmDragMotionCallbackStruct)
XM_Make(XmDragProcCallbackStruct)
XM_Make(XmDrawingAreaCallbackStruct)
XM_Make(XmDrawnButtonCallbackStruct)
XM_Make(XmDropFinishCallbackStruct)
XM_Make(XmDropProcCallbackStruct)
XM_Make(XmDropSiteEnterCallbackStruct)
XM_Make(XmDropSiteLeaveCallbackStruct)
XM_Make(XmDropStartCallbackStruct)
XM_Make(XmFileSelectionBoxCallbackStruct)
XM_Make(XmListCallbackStruct)
XM_Make(XmOperationChangedCallbackStruct)
XM_Make(XmPushButtonCallbackStruct)
XM_Make(XmRowColumnCallbackStruct)
XM_Make(XmScaleCallbackStruct)
XM_Make(XmScrollBarCallbackStruct)
XM_Make(XmSelectionBoxCallbackStruct)
XM_Make(XmTextVerifyCallbackStruct)
XM_Make(XmToggleButtonCallbackStruct)
#if MOTIF_2
XM_Make(XmDestinationCallbackStruct)
XM_Make(XmConvertCallbackStruct)
XM_Make(XmComboBoxCallbackStruct)
XM_Make(XmContainerOutlineCallbackStruct)
XM_Make(XmContainerSelectCallbackStruct)
XM_Make(XmNotebookCallbackStruct)
XM_Make(XmSpinBoxCallbackStruct)
XM_Make(XmTraverseObscuredCallbackStruct)
XM_Make(XmTopLevelLeaveCallbackStruct)
XM_Make(XmTopLevelEnterCallbackStruct)
XM_Make(XmPopupHandlerCallbackStruct)
XM_Make(XmSelectionCallbackStruct)
XM_Make(XmTransferDoneCallbackStruct)
XM_Make(XmDisplayCallbackStruct)
XM_Make(XmDragStartCallbackStruct)
#endif
#if HAVE_XmCreateDataField
XM_Make(XmDataFieldCallbackStruct)
#endif
#if HAVE_XmCreateMultiList
XM_Make(XmMultiListCallbackStruct)
#endif
#if HAVE_XmCreateTabStack
XM_Make(XmTabStackCallbackStruct)
#endif

static XEN gxm_XmTextBlock(void) 
{
  XmTextBlockRec *e; 
  e = (XmTextBlockRec *)CALLOC(1, sizeof(XmTextBlockRec)); 
  return(WRAP_FOR_XEN_OBJ("XmTextBlock", e));
}

#if HAVE_RUBY
XEN_NARGIFY_0(gxm_XmTextBlock_w, gxm_XmTextBlock)
#else
#define gxm_XmTextBlock_w gxm_XmTextBlock
#endif
  
static void define_makes(void)
{
  XM_Declare(XmAnyCallbackStruct);
  XM_Declare(XmArrowButtonCallbackStruct);
  XM_Declare(XmCommandCallbackStruct);
  XM_Declare(XmDragDropFinishCallbackStruct);
  XM_Declare(XmDragMotionCallbackStruct);
  XM_Declare(XmDragProcCallbackStruct);
  XM_Declare(XmDrawingAreaCallbackStruct);
  XM_Declare(XmDrawnButtonCallbackStruct);
  XM_Declare(XmDropFinishCallbackStruct);
  XM_Declare(XmDropProcCallbackStruct);
  XM_Declare(XmDropSiteEnterCallbackStruct);
  XM_Declare(XmDropSiteLeaveCallbackStruct);
  XM_Declare(XmDropStartCallbackStruct);
  XM_Declare(XmFileSelectionBoxCallbackStruct);
  XM_Declare(XmListCallbackStruct);
  XM_Declare(XmOperationChangedCallbackStruct);
  XM_Declare(XmPushButtonCallbackStruct);
  XM_Declare(XmRowColumnCallbackStruct);
  XM_Declare(XmScaleCallbackStruct);
  XM_Declare(XmScrollBarCallbackStruct);
  XM_Declare(XmSelectionBoxCallbackStruct);
  XM_Declare(XmTextVerifyCallbackStruct);
  XM_Declare(XmToggleButtonCallbackStruct);
  XEN_DEFINE_PROCEDURE(XM_PREFIX "XmTextBlock" XM_POSTFIX, gxm_XmTextBlock_w, 0, 0, 0, "Make an XmTextBlock struct");
#if MOTIF_2
  XM_Declare(XmDestinationCallbackStruct);
  XM_Declare(XmConvertCallbackStruct);
  XM_Declare(XmComboBoxCallbackStruct);
  XM_Declare(XmContainerOutlineCallbackStruct);
  XM_Declare(XmContainerSelectCallbackStruct);
  XM_Declare(XmNotebookCallbackStruct);
  XM_Declare(XmSpinBoxCallbackStruct);
  XM_Declare(XmTraverseObscuredCallbackStruct);
  XM_Declare(XmTopLevelLeaveCallbackStruct);
  XM_Declare(XmTopLevelEnterCallbackStruct);
  XM_Declare(XmPopupHandlerCallbackStruct);
  XM_Declare(XmSelectionCallbackStruct);
  XM_Declare(XmTransferDoneCallbackStruct);
  XM_Declare(XmDisplayCallbackStruct);
  XM_Declare(XmDragStartCallbackStruct);
#endif
#if HAVE_XmCreateDataField
  XM_Declare(XmDataFieldCallbackStruct);
#endif
#if HAVE_XmCreateMultiList
  XM_Declare(XmMultiListCallbackStruct);
#endif
#if HAVE_XmCreateTabStack
  XM_Declare(XmTabStackCallbackStruct);
#endif
}

static int its_a_callbackstruct(char *name)
{
  if (name)
    {
      int len;
      len = strlen(name);
      if (len > 16)
	{
	  char *end_name;
	  end_name = (char *)(name + len - 14);
	  return(strcmp(end_name, "CallbackStruct") == 0);
	}
    }
  return(0);
}

#define XEN_AnyCallbackStruct_P(Value) (XEN_LIST_P(Value) && \
                                       (XEN_LIST_LENGTH(Value) >= 2) && \
                                       (XEN_SYMBOL_P(XEN_CAR(Value))) && \
                                       (its_a_callbackstruct(XEN_SYMBOL_TO_C_STRING(XEN_CAR(Value)))))
#endif

static int xm_protect(XEN obj);
static void xm_unprotect_at(int ind);

#if HAVE_MOTIF

/* in XtGetValues we need to return tagged-types (etc) for arbitrarily named resources,
 *   and in XtSetValues we need to do type checks, so resources are hashed by
 *   name and type:
 */

typedef enum {XM_INT, XM_ULONG, XM_UCHAR, XM_FLOAT, XM_STRING, XM_XMSTRING, XM_STRING_TABLE,
	      XM_INT_TABLE, XM_RENDER_TABLE, XM_TAB_LIST, XM_WIDGET, XM_WIDGET_LIST, 
	      XM_BOOLEAN, XM_CALLBACK, XM_PIXEL, XM_PIXMAP, XM_XFONTSTRUCT, XM_DIMENSION,
	      XM_ATOM, XM_ATOM_LIST, XM_STRING_LIST, XM_CHARSET_TABLE, XM_TEXT_SOURCE,
	      XM_FONTLIST, XM_COLORMAP, XM_KEYSYM, XM_KEYSYM_TABLE, XM_SCREEN, XM_WINDOW,
	      XM_VISUAL, XM_RECTANGLE_LIST, XM_WIDGET_CLASS, XM_STRING_OR_INT,
	      XM_TRANSFER_CALLBACK, XM_CONVERT_CALLBACK, XM_SEARCH_CALLBACK, XM_ORDER_CALLBACK,
	      XM_QUALIFY_CALLBACK, XM_ALLOC_COLOR_CALLBACK, XM_POPUP_CALLBACK, XM_SCREEN_COLOR_CALLBACK,
	      XM_DROP_CALLBACK, XM_TRANSFER_ENTRY_LIST, XM_DRAG_CALLBACK, XM_STRING_OR_XMSTRING, XM_PARSE_CALLBACK,
	      XM_BOOLEAN_OR_INT, XM_POSITION, XM_SHORT, XM_ROW_INFO, XM_CURSOR
} xm_resource_t;

static xm_resource_t resource_type(char *resource);

static XEN C_TO_XEN_Widgets(Widget *array, int len)
{
  XEN lst = XEN_EMPTY_LIST;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_Widget(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

static XEN C_TO_XEN_XmStringTable(XmStringTable array, int len)
{
  XEN lst = XEN_EMPTY_LIST;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_XmString(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

static Widget *XEN_TO_C_Widgets(XEN lst_1, int n)
{
  Widget *ws;
  int i;
  XEN lst;
  lst = XEN_COPY_ARG(lst_1);
  ws = (Widget *)CALLOC(n, sizeof(Widget));
  for (i = 0; (i < n) && (XEN_NOT_NULL_P(lst)); i++, lst = XEN_CDR(lst))
    if (XEN_Widget_P(XEN_CAR(lst)))
      ws[i] = XEN_TO_C_Widget(XEN_CAR(lst));
    else 
      {
	FREE(ws);
	ws = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(lst), i, c__FUNCTION__, "a Widget");
	break;
      }
  return(ws);
}

static XmString *XEN_TO_C_XmStrings(XEN v_1, int len)
{
  XmString *str;
  int i;
  XEN v;
  v = XEN_COPY_ARG(v_1);
  str = (XmString *)CALLOC(len, sizeof(XmString));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    if (XEN_XmString_P(XEN_CAR(v)))
      str[i] = (XmString)XEN_TO_C_XmString(XEN_CAR(v));
    else 
      {
	FREE(str);
	str = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "an XmString");
	break;
      }
  return(str);
}

static XmDropTransferEntryRec *XEN_TO_C_XmDropTransferEntryRecs(XEN v_1, int len)
{
  XmDropTransferEntryRec *ps;
  int i;
  XEN v;
  v = XEN_COPY_ARG(v_1);
  ps = (XmDropTransferEntryRec *)CALLOC(len, sizeof(XmDropTransferEntryRec));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    {
      if (XEN_Atom_P(XEN_CAR(v)))
	ps[i].target = XEN_TO_C_Atom(XEN_CAR(v));
      else 
	{
	  if (XEN_LIST_P(XEN_CAR(v)))
	    {
	      if (XEN_Atom_P(XEN_CAR(XEN_CAR(v))))
		ps[i].target = XEN_TO_C_Atom(XEN_CAR(XEN_CAR(v)));
	      else XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "an Atom");
	      ps[i].client_data = (XtPointer)XEN_CADR(XEN_CAR(v));
	    }
	}
    }
  return(ps);
}

static XmStringTable XEN_TO_C_XmStringTable(XEN v_1, int len)
{
  XEN v;
  XmStringTable str;
  int i;
  v = XEN_COPY_ARG(v_1);
  str = (XmStringTable)CALLOC(len + 1, sizeof(XmString));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    if (XEN_XmString_P(XEN_CAR(v)))
      str[i] = XEN_TO_C_XmString(XEN_CAR(v));
    else 
      {
	FREE(str);
	str = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "an XmString");
	break;
      }
  return(str);
}

#endif

static XEN C_TO_XEN_Ints(int *array, int len)
{
  XEN lst = XEN_EMPTY_LIST;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_INT(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

#if HAVE_GUILE
static XEN c_to_xen_ints(XEN array, XEN len)
{
  return(C_TO_XEN_Ints((int *)XEN_TO_C_ULONG(array), XEN_TO_C_INT(len)));
}
#endif

static XEN C_TO_XEN_Atoms(Atom *array, int len)
{
  XEN lst = XEN_EMPTY_LIST;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_Atom(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

#if HAVE_GUILE
static XEN c_to_xen_atoms(XEN array, XEN len)
{
  return(C_TO_XEN_Atoms((Atom *)XEN_TO_C_ULONG(array), XEN_TO_C_INT(len)));
}
#endif

static XEN C_TO_XEN_Strings(char **array, int len)
{
  XEN lst = XEN_EMPTY_LIST;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_STRING(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

#if HAVE_GUILE
static XEN c_to_xen_strings(XEN array, XEN len)
{
  return(C_TO_XEN_Strings((char **)XEN_TO_C_ULONG(array), XEN_TO_C_INT(len)));
}

static XEN c_to_xen_string(XEN str)
{
  char *tmp;
  tmp = (char *)XEN_TO_C_ULONG(str);
  if (tmp)
    return(C_TO_XEN_STRING(tmp));
  return(XEN_FALSE);
}
#endif

static XEN copy_xrectangle(XRectangle *old_r)
{
  XRectangle *r;
  r = (XRectangle *)CALLOC(1, sizeof(XRectangle));
  r->x = old_r->x;
  r->y = old_r->y;
  r->width = old_r->width;
  r->height = old_r->height;
  return(C_TO_XEN_XRectangle(r));
}

static XEN C_TO_XEN_XRectangles(XRectangle *array, int len)
{
  XEN lst = XEN_EMPTY_LIST;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(copy_xrectangle(&(array[i])), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

#if HAVE_GUILE
static XEN c_to_xen_xrectangles(XEN array, XEN len)
{
  return(C_TO_XEN_XRectangles((XRectangle *)XEN_TO_C_ULONG(array), XEN_TO_C_INT(len)));
}
#endif

static XEN C_TO_XEN_KeySyms(KeySym *array, int len)
{
  XEN lst = XEN_EMPTY_LIST;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_KeySym(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

static Window *XEN_TO_C_Windows(XEN lst_1, int n)
{
  Window *ws;
  int i;
  XEN lst;
  if (n == 0) return(NULL);
  lst = XEN_COPY_ARG(lst_1);
  ws = (Window *)CALLOC(n, sizeof(Window));
  for (i = 0; (i < n) && (XEN_NOT_NULL_P(lst)); i++, lst = XEN_CDR(lst))
    if (XEN_Window_P(XEN_CAR(lst)))
      ws[i] = XEN_TO_C_Window(XEN_CAR(lst));
    else 
      {
	FREE(ws);
	ws = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(lst), i, c__FUNCTION__, "a Window");
	break;
      }
  return(ws);
}

#if MOTIF_2
static XmRendition *XEN_TO_C_XmRenditions(XEN lst_1, int n)
{
  XEN lst;
  XmRendition *ws;
  int i;
  if (n == 0) return(NULL);
  lst = XEN_COPY_ARG(lst_1);
  ws = (XmRendition *)CALLOC(n, sizeof(XmRendition));
  for (i = 0; (i < n) && (XEN_NOT_NULL_P(lst)); i++, lst = XEN_CDR(lst))
    if (XEN_XmRendition_P(XEN_CAR(lst)))
      ws[i] = XEN_TO_C_XmRendition(XEN_CAR(lst));
    else 
      {
	FREE(ws);
	ws = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(lst), i, c__FUNCTION__, "an XmRendition");
	break;
      }
  return(ws);
}

static XmTab *XEN_TO_C_XmTabs(XEN v_1, int len)
{
  XEN v;
  XmTab *str;
  int i;
  if (len == 0) return(NULL);
  v = XEN_COPY_ARG(v_1);
  str = (XmTab *)CALLOC(len, sizeof(XmTab));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    if (XEN_XmTab_P(XEN_CAR(v)))
      str[i] = (XmTab)XEN_TO_C_XmTab(XEN_CAR(v));
    else 
      {
	FREE(str);
	str = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "an XmTab");
	break;
      }
  return(str);
}
#endif

static Atom *XEN_TO_C_Atoms(XEN v_1, int len)
{
  XEN v;
  Atom *str;
  int i;
  if (len == 0) return(NULL);
  v = XEN_COPY_ARG(v_1);
  str = (Atom *)CALLOC(len, sizeof(Atom));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    if (XEN_Atom_P(XEN_CAR(v)))
      str[i] = (Atom)XEN_TO_C_Atom(XEN_CAR(v));
    else 
      {
	FREE(str);
	str = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "an Atom");
	break;
      }
  return(str);
}

static Pixel *XEN_TO_C_Pixels(XEN v_1, int len)
{
  XEN v;
  Pixel *str;
  int i;
  if (len == 0) return(NULL);
  v = XEN_COPY_ARG(v_1);
  str = (Pixel *)CALLOC(len, sizeof(Pixel));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    if (XEN_Pixel_P(XEN_CAR(v)))
      str[i] = (Pixel)XEN_TO_C_Pixel(XEN_CAR(v));
    else 
      {
	FREE(str);
	str = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "a Pixel");
	break;
      }
  return(str);
}

static KeySym *XEN_TO_C_KeySyms(XEN v_1, int len)
{
  XEN v;
  KeySym *str;
  int i;
  if (len == 0) return(NULL);
  v = XEN_COPY_ARG(v_1);
  str = (KeySym *)CALLOC(len, sizeof(KeySym));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    if (XEN_KeySym_P(XEN_CAR(v)))
      str[i] = (KeySym)XEN_TO_C_KeySym(XEN_CAR(v));
    else 
      {
	FREE(str);
	str = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "a KeySym");
	break;
      }
  return(str);
}

static char **XEN_TO_C_Strings(XEN v_1, int len)
{
  XEN v;
  char **str;
  int i;
  if (len == 0) return(NULL);
  v = XEN_COPY_ARG(v_1);
  str = (char **)CALLOC(len, sizeof(char *));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    if (XEN_STRING_P(XEN_CAR(v)))
      str[i] = XEN_TO_C_STRING(XEN_CAR(v)); /* should this be protected? */
    else 
      {
	FREE(str);
	str = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "a char*");
	break;
      }
  return(str);
}

static int *XEN_TO_C_Ints(XEN v_1, int len)
{
  XEN v;
  int *ps;
  int i;
  if (len == 0) return(NULL);
  v = XEN_COPY_ARG(v_1);
  ps = (int *)CALLOC(len, sizeof(int));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    if (XEN_INTEGER_P(XEN_CAR(v)))
      ps[i] = XEN_TO_C_INT(XEN_CAR(v));
    else 
      {
	FREE(ps);
	ps = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "an int");
	break;
      }
  return(ps);
}

static Cardinal *XEN_TO_C_Cardinals(XEN v_1, int len)
{
  XEN v;
  Cardinal *ps;
  int i;
  if (len == 0) return(NULL);
  v = XEN_COPY_ARG(v_1);
  ps = (Cardinal *)CALLOC(len, sizeof(int));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    if (XEN_INTEGER_P(XEN_CAR(v)))
      ps[i] = (Cardinal)XEN_TO_C_INT(XEN_CAR(v));
    else 
      {
	FREE(ps);
	ps = NULL;
	XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "a Cardinal");
	break;
      }
  return(ps);
}

static XRectangle *XEN_TO_C_XRectangles(XEN v_1, int len)
{
  XEN v;
  XRectangle *str, *dat;
  int i;
  if (len == 0) return(NULL);
  v = XEN_COPY_ARG(v_1);
  str = (XRectangle *)CALLOC(len, sizeof(XRectangle));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(v)); i++, v = XEN_CDR(v))
    {
      if (XEN_XRectangle_P(XEN_CAR(v)))
	dat = (XRectangle *)XEN_TO_C_XRectangle(XEN_CAR(v));
      else 
	{
	  FREE(str);
	  str = NULL;
	  XEN_ASSERT_TYPE(0, XEN_CAR(v), i, c__FUNCTION__, "an XRectangle");
	  break;
	}
      str[i].x = dat->x;
      str[i].y = dat->y;
      str[i].width = dat->width;
      str[i].height = dat->height;
    }
  return(str);
}


#if HAVE_MOTIF

/* -------- arglists -------- */

static XEN wrap_callback_struct(int type, XtPointer info);
static int map_over_protected_elements(bool (*func)(XEN val, int loc, unsigned long fid), unsigned long id);
static XEN xm_protected_element(int loc);
static int callback_struct_type(Widget w, char *name);
static void xm_unprotect(XEN obj);

enum {CALLBACK_TYPE, CALLBACK_FUNC, CALLBACK_DATA, CALLBACK_STRUCT_TYPE, CALLBACK_GC_LOC};

#define C_TO_XEN_XM_XtCallback(Code, Context) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("XtCallback"), Code, Context, XEN_ZERO, XEN_ZERO)
#define XM_XtCallback_P(Arg) WRAP_P("XtCallback", Arg)

static void gxm_XtCallbackProc(Widget w, XtPointer context, XtPointer info)
{
  XEN descr = (XEN)context;
  /* XEN_ASSERT_TYPE(XM_XtCallback_P(descr) && (XEN_LIST_LENGTH(descr) == 5), descr, XEN_ONLY_ARG, "XtCallback", "list of 5 elements"); */
  XEN_CALL_3(XEN_LIST_REF(descr, CALLBACK_FUNC),    /* descr: (list "XtCallback" func user-data struct-type gc-loc) */
	     C_TO_XEN_Widget(w),
	     XEN_LIST_REF(descr, CALLBACK_DATA),
	     wrap_callback_struct(XEN_TO_C_INT(XEN_LIST_REF(descr, CALLBACK_STRUCT_TYPE)), info),
	     c__FUNCTION__);
}

#define C_TO_XEN_XM_Drop_Callback(Code) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("Drop_Callback"), Code, XEN_FALSE, XEN_ZERO, XEN_ZERO)
#define XM_Drop_Callback_P(Arg) WRAP_P("Drop_Callback", Arg)

static bool find_dropproc(XEN val, int loc, unsigned long w)
{
  return((XM_Drop_Callback_P(val)) &&
	 (((XEN_FALSE_P((XEN)w)) && 
	   (XEN_FALSE_P(XEN_LIST_REF(val, CALLBACK_DATA)))) ||
	  ((XEN_Widget_P(XEN_LIST_REF(val, CALLBACK_DATA))) &&
	   (XEN_TO_C_ULONG(XEN_CADR(XEN_LIST_REF(val, CALLBACK_DATA))) == w))));
}

static void gxm_Drop_Callback(Widget w, XtPointer context, XtPointer info)
{
  int i;
  XmDropProcCallbackStruct *cb = (XmDropProcCallbackStruct *)info;
  i = map_over_protected_elements(find_dropproc, (unsigned long)w);
  if (i >= 0)
    {
      XEN code;
      code = XEN_LIST_REF(xm_protected_element(i), CALLBACK_FUNC);
      if (XEN_PROCEDURE_P(code))
	XEN_CALL_3(code,
		   C_TO_XEN_Widget(w),
		   XEN_FALSE,
		   C_TO_XEN_XmDropProcCallbackStruct(cb),
		   c__FUNCTION__);
    }
  cb->dropSiteStatus = XmINVALID_DROP_SITE; /* try to exit cleanly from on-going drop */
}

#define C_TO_XEN_XM_Drag_Callback(Code) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("Drag_Callback"), Code, XEN_FALSE, XEN_ZERO, XEN_ZERO)
#define XM_Drag_Callback_P(Arg) WRAP_P("Drag_Callback", Arg)

static bool find_dragproc(XEN val, int loc, unsigned long w)
{
  return((XM_Drag_Callback_P(val)) &&
	 (((XEN_FALSE_P((XEN)w)) && 
	   (XEN_FALSE_P(XEN_LIST_REF(val, CALLBACK_DATA)))) ||
	  ((XEN_Widget_P(XEN_LIST_REF(val, CALLBACK_DATA))) &&
	   (XEN_TO_C_ULONG(XEN_CADR(XEN_LIST_REF(val, CALLBACK_DATA))) == w))));
}

static void gxm_Drag_Callback(Widget w, XtPointer context, XtPointer info)
{
  int i;
  XmDragProcCallbackStruct *cb = (XmDragProcCallbackStruct *)info;
  i = map_over_protected_elements(find_dragproc, (unsigned long)w);
  if (i >= 0)
    {
      XEN code;
      code = XEN_LIST_REF(xm_protected_element(i), CALLBACK_FUNC);
      if (XEN_PROCEDURE_P(code))
	XEN_CALL_3(code,
		   C_TO_XEN_Widget(w),
		   XEN_FALSE,
		   C_TO_XEN_XmDragProcCallbackStruct(cb),
		   c__FUNCTION__);
    }
}

#define C_TO_XEN_XM_XtPopupChild(Code) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("XtPopupChild"), Code, XEN_FALSE, XEN_ZERO, XEN_ZERO)
#define XM_XtPopupChild_P(Arg) WRAP_P("XtPopupChild", Arg)

static bool find_popupchild(XEN val, int loc, unsigned long w)
{
  return((XM_XtPopupChild_P(val)) &&
	 (((XEN_FALSE_P((XEN)w)) && 
	   (XEN_FALSE_P(XEN_LIST_REF(val, CALLBACK_DATA)))) ||
	  ((XEN_Widget_P(XEN_LIST_REF(val, CALLBACK_DATA))) &&
	   (XEN_TO_C_ULONG(XEN_CADR(XEN_LIST_REF(val, CALLBACK_DATA))) == w))));
}

static void gxm_XtPopupChild(Widget w)
{
  /* since all we have to go by is the widget, we have to scan the gc table for any associated code */
  int i;
  i = map_over_protected_elements(find_popupchild, (unsigned long)w);
  if (i >= 0)
    {
      XEN code;
      code = XEN_LIST_REF(xm_protected_element(i), CALLBACK_FUNC);
      if (XEN_PROCEDURE_P(code))
	XEN_CALL_1(code,
		   C_TO_XEN_Widget(w),
		   c__FUNCTION__);
    }
}

#define C_TO_XEN_XM_XmSearchProc(Code) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("XmSearchProc"), Code, XEN_FALSE, XEN_ZERO, XEN_ZERO)
#define XM_XmSearchProc_P(Arg) WRAP_P("XmSearchProc", Arg)

static bool find_searchproc(XEN val, int loc, unsigned long w)
{
  return((XM_XmSearchProc_P(val)) &&
	 (((XEN_FALSE_P((XEN)w)) && 
	   (XEN_FALSE_P(XEN_LIST_REF(val, CALLBACK_DATA)))) ||
	  ((XEN_Widget_P(XEN_LIST_REF(val, CALLBACK_DATA))) &&
	   (XEN_TO_C_ULONG(XEN_CADR(XEN_LIST_REF(val, CALLBACK_DATA))) == w))));
}

static void gxm_XmSearchProc(Widget w, XmFileSelectionBoxCallbackStruct *info)
{
  /* here we again have to go by the widget */
  int i;
  i = map_over_protected_elements(find_searchproc, (unsigned long)w);
  if (i >= 0)
    {
      XEN code;
      code = XEN_LIST_REF(xm_protected_element(i), CALLBACK_FUNC);
      if (XEN_PROCEDURE_P(code))
	XEN_CALL_2(code,
		   C_TO_XEN_Widget(w),
		   C_TO_XEN_XmFileSelectionBoxCallbackStruct(info),
		   c__FUNCTION__);
    }
}

#define C_TO_XEN_XM_XmQualifyProc(Code) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("XmQualifyProc"), Code, XEN_FALSE, XEN_ZERO, XEN_ZERO)
#define XM_XmQualifyProc_P(Arg) WRAP_P("XmQualifyProc", Arg)

static bool find_qualifyproc(XEN val, int loc, unsigned long w)
{
  return((XM_XmQualifyProc_P(val)) &&
	 (((XEN_FALSE_P((XEN)w)) && 
	   (XEN_FALSE_P(XEN_LIST_REF(val, CALLBACK_DATA)))) ||
	  ((XEN_Widget_P(XEN_LIST_REF(val, CALLBACK_DATA))) &&
	   (XEN_TO_C_ULONG(XEN_CADR(XEN_LIST_REF(val, CALLBACK_DATA))) == w))));
}

static void gxm_XmQualifyProc(Widget w, XtPointer indata, XtPointer outdata)
{
  /* here we again have to go by the widget */
  int i;
  i = map_over_protected_elements(find_qualifyproc, (unsigned long)w);
  if (i >= 0)
    {
      XEN code;
      code = XEN_LIST_REF(xm_protected_element(i), CALLBACK_FUNC);
      if (XEN_PROCEDURE_P(code))
	XEN_CALL_3(code,
		   C_TO_XEN_Widget(w),
		   C_TO_XEN_XmFileSelectionBoxCallbackStruct((XmFileSelectionBoxCallbackStruct *)indata),
		   C_TO_XEN_XmFileSelectionBoxCallbackStruct((XmFileSelectionBoxCallbackStruct *)outdata),
		   c__FUNCTION__);
    }
}

#define C_TO_XEN_XM_XtOrderProc(Code) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("XtOrderProc"), Code, XEN_FALSE, XEN_ZERO, XEN_ZERO)
#define XM_XtOrderProc_P(Arg) WRAP_P("XtOrderProc", Arg)

static bool find_orderproc(XEN val, int loc, unsigned long w)
{
  return((XM_XtOrderProc_P(val)) &&
	 (((XEN_FALSE_P((XEN)w)) && 
	   (XEN_FALSE_P(XEN_LIST_REF(val, CALLBACK_DATA)))) ||
	  ((XEN_Widget_P(XEN_LIST_REF(val, CALLBACK_DATA))) &&
	   (XEN_TO_C_ULONG(XEN_CADR(XEN_LIST_REF(val, CALLBACK_DATA))) == w))));
}

static Cardinal gxm_XtOrderProc(Widget w)
{
  /* here we again have to go by the widget */
  int i, result = 0;
  i = map_over_protected_elements(find_orderproc, (unsigned long)w);
  if (i >= 0)
    {
      XEN code;
      code = XEN_LIST_REF(xm_protected_element(i), CALLBACK_FUNC);
      if (XEN_PROCEDURE_P(code))
	result = XEN_TO_C_INT(XEN_CALL_1(code,
					 C_TO_XEN_Widget(w),
					 c__FUNCTION__));
    }
  return((Cardinal)result);
}

#if MOTIF_2
#define C_TO_XEN_XM_Parse_Callback(Code) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("Parse_Callback"), Code, XEN_FALSE, XEN_ZERO, XEN_ZERO)
#define XM_Parse_Callback_P(Arg) WRAP_P("Parse_Callback", Arg)

static bool find_parseproc(XEN val, int loc, unsigned long w)
{
  return((XM_Parse_Callback_P(val)) &&
	 (((XEN_FALSE_P((XEN)w)) && 
	   (XEN_FALSE_P(XEN_LIST_REF(val, CALLBACK_DATA)))) ||
	  ((XEN_XmParseMapping_P(XEN_LIST_REF(val, CALLBACK_DATA))) &&
	   (XEN_TO_C_ULONG(XEN_CADR(XEN_LIST_REF(val, CALLBACK_DATA))) == w))));
}

static XmIncludeStatus gxm_Parse_Callback(XtPointer *in_out, XtPointer text_end, XmTextType type, XmStringTag locale_tag,
					  XmParseMapping entry, int pattern_length, XmString *str_include, XtPointer call_data)
{
  int i;
  i = map_over_protected_elements(find_parseproc, (unsigned long)entry);
  if (i >= 0)
    {
      XEN code;
      code = XEN_LIST_REF(xm_protected_element(i), CALLBACK_FUNC);
      if (XEN_PROCEDURE_P(code))
	return(XEN_TO_C_INT(XEN_APPLY(code,
				      XEN_LIST_8(C_TO_XEN_STRING((char *)(*in_out)),
						 C_TO_XEN_ULONG(text_end), /* can't work... */
						 C_TO_XEN_INT(type),
						 C_TO_XEN_STRING(locale_tag),
						 C_TO_XEN_XmParseMapping(entry),
						 C_TO_XEN_INT(pattern_length),
						 C_TO_XEN_XmString((*str_include)), /* can't work... */
						 (XEN)call_data),
				      c__FUNCTION__)));
    }
  return(0);
}
#endif

static XEN xm_XmColorAllocationProc = XEN_FALSE;

static void gxm_XmAllocColorProc(Display *dpy, Colormap color, XColor *bs)
{
  /* DIFF: XmAllocColorProc should return new XColor value
   */
  XEN val;
  val = XEN_CALL_3(xm_XmColorAllocationProc,
		   C_TO_XEN_Display(dpy),
		   C_TO_XEN_Colormap(color),
		   C_TO_XEN_XColor(bs),
		   c__FUNCTION__);
  bs = XEN_TO_C_XColor(val);
}

static XEN xm_XmColorCalculationProc = XEN_FALSE;

static void gxm_XmColorCalculationProc(Screen *scr, XColor *bg, XColor *fg, XColor *sel, XColor *ts, XColor *bs)
{
  /* DIFF: XmColorCalculationProc takes 2 args, returns list of 4 colors
   */
  XEN lst;
  int loc;
  lst = XEN_CALL_2(xm_XmColorCalculationProc,
		   C_TO_XEN_Screen(scr),
		   C_TO_XEN_XColor(bg),
		   c__FUNCTION__);
  loc = xm_protect(lst);
  if (XEN_LIST_P(lst))
    {
      fg = XEN_TO_C_XColor(XEN_LIST_REF(lst, 0)); 
      sel = XEN_TO_C_XColor(XEN_LIST_REF(lst, 1));
      ts = XEN_TO_C_XColor(XEN_LIST_REF(lst, 2));
      bs = XEN_TO_C_XColor(XEN_LIST_REF(lst, 3));
   }
  xm_unprotect_at(loc);
}

static XEN xm_XmColorProc = XEN_FALSE; /* XmColorProc is not the same as XmScreen color calculation proc */

static void gxm_XmColorProc(XColor *bg, XColor *fg, XColor *sel, XColor *ts, XColor *bs)
{
  XEN lst;
  int loc;
  lst = XEN_CALL_1(xm_XmColorProc,
		   C_TO_XEN_XColor(bg),
		   c__FUNCTION__);
  loc = xm_protect(lst);
  if (XEN_LIST_P(lst))
    {
      fg = XEN_TO_C_XColor(XEN_LIST_REF(lst, 0)); 
      sel = XEN_TO_C_XColor(XEN_LIST_REF(lst, 1));
      ts = XEN_TO_C_XColor(XEN_LIST_REF(lst, 2));
      bs = XEN_TO_C_XColor(XEN_LIST_REF(lst, 3));
   }
  xm_unprotect_at(loc);
}

static XtCallbackList XEN_TO_C_XtCallbackList(XEN call_list1)
{
  XEN call_list;
  int call_i, call_len;
  XtCallbackRec *cl = NULL;
  call_list = XEN_COPY_ARG(call_list1);
  call_len = XEN_LIST_LENGTH(call_list) / 2;
  if (call_len == 0) return(NULL);
  cl = (XtCallbackRec *)CALLOC(call_len + 1, sizeof(XtCallbackRec));
  for (call_i = 0; call_i < call_len; call_i++, call_list = XEN_CDDR(call_list))
    {
      XEN func, data;
      func = XEN_CAR(call_list);
      if (XEN_LIST_LENGTH(call_list) == 2)
	data = XEN_CADR(call_list);
      else data = XEN_FALSE;
      if ((XEN_PROCEDURE_P(func)) && (XEN_REQUIRED_ARGS_OK(func, 3)))
	{
	  XEN descr;
	  cl[call_i].callback = gxm_XtCallbackProc;
	  descr = C_TO_XEN_XM_XtCallback(XEN_CAR(call_list), data);
	  cl[call_i].closure = (XtPointer)descr;
	  XEN_LIST_SET(descr, CALLBACK_GC_LOC, C_TO_XEN_INT(xm_protect(descr)));
	}
    }
  return(cl);
}

static XEN C_TO_XEN_STRING_WITH_TERMINATION(char *str, unsigned long len)
{
  if ((len == 0) || (str == NULL)) return(XEN_FALSE);
  str[len] = '\0';
  return(C_TO_XEN_STRING(str));
}

static XEN xm_XtSelectionCallback_Descr = XEN_FALSE;
static void gxm_XtSelectionCallbackProc(Widget w, XtPointer x, Atom *a1, Atom *a2, XtPointer x1, unsigned long *l, int *i)
{
  XEN_APPLY(XEN_CAR(xm_XtSelectionCallback_Descr),
	    XEN_LIST_7(C_TO_XEN_Widget(w),
		       XEN_CADR(xm_XtSelectionCallback_Descr),
		       C_TO_XEN_Atom(*a1),
		       C_TO_XEN_Atom(*a2),
		       C_TO_XEN_STRING_WITH_TERMINATION((char *)x1, *l), /* should we handle Atom -> Lisp type conversions? */
		       C_TO_XEN_ULONG(*l),
		       C_TO_XEN_INT(*i)),
	    c__FUNCTION__);
}

static XEN xm_XtConvertSelectionIncr_Descr = XEN_FALSE;
static Boolean gxm_XtConvertSelectionIncrProc(Widget w, Atom *selection, Atom *target, 
					   Atom *type_return, XtPointer *value_return, 
					   unsigned long *length_return, int *format_return, 
					   unsigned long *max_length, XtPointer client_data, 
					   XtRequestId *request_id)
{
  /* DIFF: user callback here takes 6 args (w selection target max_length client_data request_id)
   *       should return (list ...) if ok, #f if not
   *       the list should be (type value length format)
   */
  XEN result;
  result = XEN_APPLY(xm_XtConvertSelectionIncr_Descr,
		     XEN_LIST_6(C_TO_XEN_Widget(w),
				C_TO_XEN_Atom(*selection),
				C_TO_XEN_Atom(*target),
				C_TO_XEN_INT(*max_length),
				C_TO_XEN_ULONG(client_data),
				C_TO_XEN_ULONG(*request_id)), /* XtRequestId is XtPointer */
		     c__FUNCTION__);
  if (XEN_FALSE_P(result))
    return(0);
  (*type_return) = XEN_TO_C_Atom(XEN_LIST_REF(result, 0));
  (*value_return) = (XtPointer)XEN_TO_C_ULONG(XEN_LIST_REF(result, 1));
  (*length_return) = (unsigned long)XEN_TO_C_INT(XEN_LIST_REF(result, 2));
  (*format_return) = XEN_TO_C_INT(XEN_LIST_REF(result, 3));
  return(1);
}

static Arg *protect_args(Arg *args, int len)
{
  /* probably too clever... trying to put off the copy until after all error checking to reduce memory leakage */
  int i;
  for (i = 0; i < len; i++)
    if (args[i].name)
      args[i].name = strdup(args[i].name); /* XEN_TO_C_STRING will eventually gc, so protect against that until we're done with args */
  return(args);
}

static Arg *free_args(Arg *args, int len)
{
  int i;
  for (i = 0; i < len; i++)
    if (args[i].name)
      free(args[i].name); /* free what we strdup'd above */
  FREE(args);
  return(NULL);
}

static Arg *XEN_TO_C_Args(XEN inargl)
{
  /* an Arg array in xm is a list of name value pairs */
  Arg *args = NULL;
  int i, len, gcloc;
  XEN descr, inarg;
  /* if XtVaNestedList supported, scan for it here, and increase length as needed,
   *   then make recursive call to XEN_TO_C_Args in that branch, unloading afterwards
   *   this is not actually needed in xm -- just use append!
   */
  inarg = XEN_COPY_ARG(inargl);
  len = XEN_LIST_LENGTH(inarg) / 2;
  if (len == 0) return(NULL);
  gcloc = xm_protect(inarg);
  args = (Arg *)CALLOC(len, sizeof(Arg));
  for (i = 0; i < len; i++, inarg = XEN_CDDR(inarg))
    {
      XtCallbackRec *cl = NULL;
      xm_resource_t type;
      XEN xname, value;
      char *name;
      xname = XEN_CAR(inarg);
      XEN_ASSERT_TYPE(XEN_STRING_P(xname), xname, 0, c__FUNCTION__, "string");
      name = XEN_TO_C_STRING(xname);
      type = resource_type(name);
      value = XEN_CADR(inarg);
      switch (type)
	{
	  /* here the XtSetArg call wants an XtCallbackList, the incoming XEN type is a list of callback data pairs
	   *   the new callback rec.callback = gxm_XtCallback (etc -- chosen by resource type)
	   *   the rec.closure will be the wrapped func/data info
	   *   we don't have all the data we need for the actual wrapped list here since we need the widget to choose the callback struct type
	   *   so fixup_args should always follow if there's a possible callback list in the resource settings 
	   * also, which callback is invoked may depend on callback type 
	   */
	case XM_CALLBACK:
	  cl = XEN_TO_C_XtCallbackList(value);
	  if (cl) XtSetArg(args[i], name, cl);
	  break;
#if MOTIF_2
	case XM_PARSE_CALLBACK:
	  if ((XEN_PROCEDURE_P(value)) && (XEN_REQUIRED_ARGS_OK(value, 8)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_Parse_Callback);
	      descr = C_TO_XEN_XM_Parse_Callback(value);
	      XEN_LIST_SET(descr, CALLBACK_GC_LOC, C_TO_XEN_INT(xm_protect(descr)));
	    }
	  else 
	    {
	      if (XEN_FALSE_P(value))
		XtSetArg(args[i], name, NULL);
	      else XEN_ASSERT_TYPE(0, value, 0, name, "procedure of 8 args");
	    }
	  break;
#endif
	case XM_DROP_CALLBACK:
	  if ((XEN_PROCEDURE_P(value)) && (XEN_REQUIRED_ARGS_OK(value, 3)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_Drop_Callback);
	      descr = C_TO_XEN_XM_Drop_Callback(value);
	      XEN_LIST_SET(descr, CALLBACK_GC_LOC, C_TO_XEN_INT(xm_protect(descr)));
	    }
	  else 
	    {
	      if (XEN_FALSE_P(value))
		XtSetArg(args[i], name, NULL);
	      else XEN_ASSERT_TYPE(0, value, 0, name, "procedure of 3 args");
	    }
	  break;
	case XM_DRAG_CALLBACK:
	  if ((XEN_PROCEDURE_P(value)) && (XEN_REQUIRED_ARGS_OK(value, 3)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_Drag_Callback);
	      descr = C_TO_XEN_XM_Drag_Callback(value);
	      XEN_LIST_SET(descr, CALLBACK_GC_LOC, C_TO_XEN_INT(xm_protect(descr)));
	    }
	  else 
	    {
	      if (XEN_FALSE_P(value))
		XtSetArg(args[i], name, NULL);
	      else XEN_ASSERT_TYPE(0, value, 0, name, "procedure of 3 args");
	    }
	  break;
	case XM_SEARCH_CALLBACK:    /* XmNfileSearchProc and XmNdirSearchProc, XmSearchProc XmFileSelectionBox 756 */
	  if ((XEN_PROCEDURE_P(value)) && (XEN_REQUIRED_ARGS_OK(value, 2)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XmSearchProc);
	      descr = C_TO_XEN_XM_XmSearchProc(value);
	      XEN_LIST_SET(descr, CALLBACK_GC_LOC, C_TO_XEN_INT(xm_protect(descr)));
	    }
	  else 
	    {
	      if (XEN_FALSE_P(value))
		XtSetArg(args[i], name, NULL);
	      else XEN_ASSERT_TYPE(0, value, 0, name, "procedure of 2 args");
	    }
	  break;
	case XM_QUALIFY_CALLBACK:   /* XmNqualifySearchDataProc, XmQualifyProc */
	  if ((XEN_PROCEDURE_P(value)) && (XEN_REQUIRED_ARGS_OK(value, 3)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XmQualifyProc);
	      descr = C_TO_XEN_XM_XmQualifyProc(value);
	      XEN_LIST_SET(descr, CALLBACK_GC_LOC, C_TO_XEN_INT(xm_protect(descr)));
	    }
	  else 
	    {
	      if (XEN_FALSE_P(value))
		XtSetArg(args[i], name, NULL);
	      else XEN_ASSERT_TYPE(0, value, 0, name, "procedure of 3 args");
	    }
	  break;
	case XM_ORDER_CALLBACK:
	  if ((XEN_PROCEDURE_P(value)) && (XEN_REQUIRED_ARGS_OK(value, 1)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XtOrderProc);
	      descr = C_TO_XEN_XM_XtOrderProc(value);
	      XEN_LIST_SET(descr, CALLBACK_GC_LOC, C_TO_XEN_INT(xm_protect(descr)));
	    }
	  else 
	    {
	      if (XEN_FALSE_P(value))
		XtSetArg(args[i], name, NULL);
	      else XEN_ASSERT_TYPE(0, value, 0, name, "procedure of 1 arg");
	    }
	  break;
	case XM_TRANSFER_CALLBACK:  /* XmNtransferProc, XtSelectionCallbackProc, XmDropTransfer */
	  /* for now I'll assume no collisions here */
	  xm_XtSelectionCallback_Descr = XEN_LIST_2(value, XEN_FALSE);
	  xm_protect(xm_XtSelectionCallback_Descr);
	  XtSetArg(args[i], name, (unsigned long)gxm_XtSelectionCallbackProc);
	  break;
	case XM_CONVERT_CALLBACK:   /* XmNconvertProc, XtConvertSelectionIncrProc, XmDragContext */
	  xm_XtConvertSelectionIncr_Descr = value;
	  xm_protect(xm_XtConvertSelectionIncr_Descr);
	  XtSetArg(args[i], name, (unsigned long)gxm_XtConvertSelectionIncrProc);
	  break;
	case XM_ALLOC_COLOR_CALLBACK:     /* XmNcolorAllocationProc, XmAllocColorProc XmScreen 921 */
	  if ((XEN_PROCEDURE_P(value)) && (XEN_REQUIRED_ARGS_OK(value, 3)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XmAllocColorProc);
	      if (XEN_PROCEDURE_P(xm_XmColorAllocationProc)) xm_unprotect(xm_XmColorAllocationProc);
	      xm_protect(value);
	      xm_XmColorAllocationProc = value;
	    }
	  else 
	    {
	      if (XEN_FALSE_P(value))
		XtSetArg(args[i], name, NULL);
	      else XEN_ASSERT_TYPE(0, value, 0, name, "procedure of 3 args");
	    }
	  break;
	case XM_SCREEN_COLOR_CALLBACK:     /* XmNcolorCalculationProc, XmScreen 921 */
	  if ((XEN_PROCEDURE_P(value)) && (XEN_REQUIRED_ARGS_OK(value, 2)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XmColorCalculationProc);
	      if (XEN_PROCEDURE_P(xm_XmColorCalculationProc)) xm_unprotect(xm_XmColorCalculationProc);
	      xm_protect(value);
	      xm_XmColorCalculationProc = value;
	    }
	  else 
	    {
	      if (XEN_FALSE_P(value))
		XtSetArg(args[i], name, NULL);
	      else XEN_ASSERT_TYPE(0, value, 0, name, "procedure of 2 args");
	    }
	  break;
	case XM_POPUP_CALLBACK:     /* XmNcreatePopupChildProc, XtCreatePopupChildProc */
	  if ((XEN_PROCEDURE_P(value)) && (XEN_REQUIRED_ARGS_OK(value, 1)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XtPopupChild);
	      descr = C_TO_XEN_XM_XtPopupChild(value);
	      XEN_LIST_SET(descr, CALLBACK_GC_LOC, C_TO_XEN_INT(xm_protect(descr)));
	    }
	  else 
	    {
	      if (XEN_FALSE_P(value))
		XtSetArg(args[i], name, NULL);
	      else XEN_ASSERT_TYPE(0, value, 0, name, "procedure of 1 arg");
	    }
	  break;

	  /* the rest are just doing type checks before the conversion to C */

	case XM_INT:
	  XEN_ASSERT_TYPE(XEN_INTEGER_P(value), value, XEN_ONLY_ARG, name, "an integer");
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_INT(value)));
	  break;
	case XM_FLOAT:
	  XEN_ASSERT_TYPE(XEN_DOUBLE_P(value), value, XEN_ONLY_ARG, name, "a float");
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_DOUBLE(value)));
	  break;
	case XM_STRING:	      
	  XEN_ASSERT_TYPE(XEN_STRING_P(value), value, XEN_ONLY_ARG, name, "a string");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_STRING(value)));
	  break;
	case XM_STRING_OR_INT:
	  XEN_ASSERT_TYPE(XEN_STRING_P(value) || XEN_INTEGER_P(value), value, XEN_ONLY_ARG, name, "an integer or a string");      
	  if (XEN_STRING_P(value))
	    XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_STRING(value)));
	  else XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_INT(value)));
	  break;
	case XM_XMSTRING:
	  XEN_ASSERT_TYPE(XEN_XmString_P(value), value, XEN_ONLY_ARG, name, "an XmString");      	      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_XmString(value)));
	  break;
	case XM_STRING_OR_XMSTRING:
	  XEN_ASSERT_TYPE(XEN_XmString_P(value) || XEN_STRING_P(value), value, XEN_ONLY_ARG, name, "a string or an XmString");  
	  if (XEN_STRING_P(value))
	    XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_STRING(value)));
	  else XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_XmString(value)));
	  break;
	case XM_STRING_TABLE:
	  XEN_ASSERT_TYPE(XEN_LIST_P(value), value, XEN_ONLY_ARG, name, "an XmStringTable");      	           
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_XmStringTable(value, XEN_LIST_LENGTH(value))));
	  break;
	case XM_TRANSFER_ENTRY_LIST:
	  XEN_ASSERT_TYPE(XEN_LIST_P(value), value, XEN_ONLY_ARG, name, "a list");      	           
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_XmDropTransferEntryRecs(value, XEN_LIST_LENGTH(value))));
	  break;
#if MOTIF_2
	case XM_RENDER_TABLE: 
	  XEN_ASSERT_TYPE(XEN_XmRenderTable_P(value), value, XEN_ONLY_ARG, name, "an XmRenderTable");
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_XmRenderTable(value)));
	  break;
	case XM_TAB_LIST:
	  XEN_ASSERT_TYPE(XEN_XmTabList_P(value), value, XEN_ONLY_ARG, name, "an XmTabList"); 
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_XmTabList(value)));
	  break;
#endif
	case XM_WIDGET:
	  XEN_ASSERT_TYPE(XEN_Widget_P(value), value, XEN_ONLY_ARG, name, "a Widget"); 
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Widget(value)));
	  break;
	case XM_BOOLEAN:
	  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(value), value, XEN_ONLY_ARG, name, "a boolean"); 
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_BOOLEAN(value)));
	  break;
	case XM_BOOLEAN_OR_INT:
	  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(value) || XEN_INTEGER_P(value), value, XEN_ONLY_ARG, name, "a boolean or int"); 
	  if (XEN_BOOLEAN_P(value))
	    XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_BOOLEAN(value)));
	  else XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_INT(value)));
	  break;
	case XM_PIXEL:
	  XEN_ASSERT_TYPE(XEN_Pixel_P(value), value, XEN_ONLY_ARG, name, "a pixel"); 
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Pixel(value)));
	  break;
	case XM_PIXMAP:
	  XEN_ASSERT_TYPE(XEN_Pixmap_P(value), value, XEN_ONLY_ARG, name, "a pixmap"); 
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Pixmap(value)));
	  break;
	case XM_DIMENSION:
	  XEN_ASSERT_TYPE(XEN_INTEGER_P(value), value, XEN_ONLY_ARG, name, "a Dimension (integer)");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Dimension(value)));
	  break;
	case XM_POSITION:
	  XEN_ASSERT_TYPE(XEN_INTEGER_P(value), value, XEN_ONLY_ARG, name, "a Position (integer)");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Position(value)));
	  break;
	case XM_SHORT:
	  XEN_ASSERT_TYPE(XEN_INTEGER_P(value), value, XEN_ONLY_ARG, name, "a short");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_INT(value)));
	  break;
	case XM_ATOM:
	  XEN_ASSERT_TYPE(XEN_Atom_P(value), value, XEN_ONLY_ARG, name, "an Atom");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Atom(value)));
	  break;
	case XM_TEXT_SOURCE:
	  XEN_ASSERT_TYPE(XEN_XmTextSource_P(value), value, XEN_ONLY_ARG, name, "an XmTextSource");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_XmTextSource(value)));
	  break;
#if (!XM_DISABLE_DEPRECATED)
	case XM_FONTLIST:
	  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(value), value, XEN_ONLY_ARG, name, "an XmFontList");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_XmFontList(value)));
	  break;
#endif
	case XM_COLORMAP:
	  XEN_ASSERT_TYPE(XEN_Colormap_P(value), value, XEN_ONLY_ARG, name, "a Colormap");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Colormap(value)));
	  break;
	case XM_KEYSYM:
	  XEN_ASSERT_TYPE(XEN_KeySym_P(value), value, XEN_ONLY_ARG, name, "a KeySym");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_KeySym(value)));
	  break;
	case XM_SCREEN:
	  XEN_ASSERT_TYPE(XEN_Screen_P(value), value, XEN_ONLY_ARG, name, "a Screen");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Screen(value)));
	  break;
	case XM_WINDOW:
	  XEN_ASSERT_TYPE(XEN_Window_P(value), value, XEN_ONLY_ARG, name, "a Window");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Window(value)));
	  break;
	case XM_VISUAL:
	  XEN_ASSERT_TYPE(XEN_Visual_P(value), value, XEN_ONLY_ARG, name, "a Visual");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Visual(value)));
	  break;
	case XM_WIDGET_CLASS:
	  XEN_ASSERT_TYPE(XEN_WidgetClass_P(value), value, XEN_ONLY_ARG, name, "a WidgetClass");      
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_WidgetClass(value)));
	  break;

	case XM_ATOM_LIST:
	  XEN_ASSERT_TYPE(XEN_LIST_P(value), value, XEN_ONLY_ARG, name, "a list of Atoms");
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Atoms(value, XEN_LIST_LENGTH(value))));
	  break;
	case XM_INT_TABLE:
	  XEN_ASSERT_TYPE(XEN_LIST_P(value), value, XEN_ONLY_ARG, name, "a list of ints");
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Ints(value, XEN_LIST_LENGTH(value))));
	  break;
	case XM_WIDGET_LIST:
	  XEN_ASSERT_TYPE(XEN_LIST_P(value), value, XEN_ONLY_ARG, name, "a list of Widgets");
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Widgets(value, XEN_LIST_LENGTH(value))));
	  break;
	case XM_KEYSYM_TABLE:
	  XEN_ASSERT_TYPE(XEN_LIST_P(value), value, XEN_ONLY_ARG, name, "a list of KeySyms");
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_KeySyms(value, XEN_LIST_LENGTH(value))));
	  break;
	case XM_STRING_LIST:
	case XM_CHARSET_TABLE:
	  XEN_ASSERT_TYPE(XEN_LIST_P(value), value, XEN_ONLY_ARG, name, "a list of char *");
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Strings(value, XEN_LIST_LENGTH(value))));
	  break;
	case XM_RECTANGLE_LIST:
	  XEN_ASSERT_TYPE(XEN_LIST_P(value), value, XEN_ONLY_ARG, name, "a list of XRectangles");
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_XRectangles(value, XEN_LIST_LENGTH(value))));
	  break;
#if HAVE_XmCreateMultiList
	case XM_ROW_INFO:
	  XEN_ASSERT_TYPE(XEN_XmMultiListRowInfo_P(value), value, XEN_ONLY_ARG, name, "XmMultiListRowInfo*");
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_XmMultiListRowInfo(value)));
	  break;
#endif
	case XM_CURSOR:
	  XEN_ASSERT_TYPE(XEN_Cursor_P(value), value, XEN_ONLY_ARG, name, "a cursor"); 
	  XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_Cursor(value)));
	  break;

	default:
	  if (XEN_ULONG_P(value))
	    XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_ULONG(value)));
	  else if (XEN_INTEGER_P(value))
	    XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_INT(value)));
	  else if (XEN_BOOLEAN_P(value))
	    XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_BOOLEAN(value)));
	  else if (XEN_STRING_P(value))
	    XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_STRING(value)));
	  else if (XEN_LIST_P(value))
	    XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_ULONG(XEN_CADR(value))));  /* all tagged types */
	  else 
	    XtSetArg(args[i], name, (XtArgVal)(XEN_TO_C_ULONG(value)));
	  break;
	}
    }
  xm_unprotect_at(gcloc);
  return(protect_args(args, len));
}

static void fixup_args(Widget w, Arg *args, int len)
{
  /* search for and fill-in incompletely specified callback info now that we have the widget type */
  int i;
  for (i = 0; i < len; i++)
    {
      char *name;
      name = args[i].name;
      if (name)
	{
	  XtCallbackRec *cl = NULL;
	  int j;
	  XEN data;
	  switch (resource_type(name))
	    {
	    case XM_STRING_TABLE:
	      FREE((XmStringTable *)(args[i].value));
	      break;
	    case XM_WIDGET_LIST:
	      FREE((Widget *)(args[i].value));
	      break;
	    case XM_CALLBACK:
	      cl = (XtCallbackRec *)(args[i].value);
	      for (j = 0 ;; j++)
		{
		  if (cl[j].callback == NULL) break;
		  data = (XEN)(cl[j].closure);
		  XEN_LIST_SET(data, CALLBACK_STRUCT_TYPE, C_TO_XEN_INT(callback_struct_type(w, name)));
		}
	      FREE(cl);
	      break;

	    case XM_DROP_CALLBACK:
	      j = map_over_protected_elements(find_dropproc, (unsigned long)w);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(XEN_TO_C_INT(XEN_LIST_REF(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_dropproc, (unsigned long)XEN_FALSE);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  XEN_LIST_SET(data, CALLBACK_DATA, C_TO_XEN_Widget(w));
		}
	      else fprintf(stderr,"can't fixup drop proc!");
	      break;
	    case XM_DRAG_CALLBACK:
	      j = map_over_protected_elements(find_dragproc, (unsigned long)w);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(XEN_TO_C_INT(XEN_LIST_REF(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_dragproc, (unsigned long)XEN_FALSE);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  XEN_LIST_SET(data, CALLBACK_DATA, C_TO_XEN_Widget(w));
		}
	      else fprintf(stderr,"can't fixup drag proc!");
	      break;
	    case XM_QUALIFY_CALLBACK:
	      j = map_over_protected_elements(find_qualifyproc, (unsigned long)w); /* see if one already exists */
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(XEN_TO_C_INT(XEN_LIST_REF(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_qualifyproc, (unsigned long)XEN_FALSE); /* i.e. find the unset one */
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  XEN_LIST_SET(data, CALLBACK_DATA, C_TO_XEN_Widget(w));
		}
	      break;
	    case XM_SEARCH_CALLBACK:
	      j = map_over_protected_elements(find_searchproc, (unsigned long)w);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(XEN_TO_C_INT(XEN_LIST_REF(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_searchproc, (unsigned long)XEN_FALSE);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  XEN_LIST_SET(data, CALLBACK_DATA, C_TO_XEN_Widget(w));
		}
	      break;
	    case XM_ORDER_CALLBACK:
	      j = map_over_protected_elements(find_orderproc, (unsigned long)w);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(XEN_TO_C_INT(XEN_LIST_REF(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_orderproc, (unsigned long)XEN_FALSE);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  XEN_LIST_SET(data, CALLBACK_DATA, C_TO_XEN_Widget(w));
		}
	      break;
	    case XM_POPUP_CALLBACK:
	      j = map_over_protected_elements(find_popupchild, (unsigned long)w);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(XEN_TO_C_INT(XEN_LIST_REF(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_popupchild, (unsigned long)XEN_FALSE);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  XEN_LIST_SET(data, CALLBACK_DATA, C_TO_XEN_Widget(w));
		}
	      break;
	    default:
	      break;
	    }
	}
    }
}

static int xmstringtable_length(Widget w, char *name)
{
  /* it's not safe here to search for 0 as end of table, so where possible we'll match resources with counts */
  int len = 0;
  if (strcmp(name, XmNitems) == 0) XtVaGetValues(w, XmNitemCount, &len, NULL);
  else if (strcmp(name, XmNlistItems) == 0) XtVaGetValues(w, XmNlistItemCount, &len, NULL);
  else if (strcmp(name, XmNselectedItems) == 0) XtVaGetValues(w, XmNselectedItemCount, &len, NULL);
  else if (strcmp(name, XmNhistoryItems) == 0) XtVaGetValues(w, XmNhistoryItemCount, &len, NULL);
  else if (strcmp(name, XmNfileListItems) == 0) XtVaGetValues(w, XmNfileListItemCount, &len, NULL);
  else if (strcmp(name, XmNdirListItems) == 0) XtVaGetValues(w, XmNdirListItemCount, &len, NULL);
#if MOTIF_2
  else if (strcmp(name, XmNdetail) == 0) XtVaGetValues(w, XmNdetailCount, &len, NULL);
  else if (strcmp(name, XmNvalues) == 0) XtVaGetValues(w, XmNnumValues, &len, NULL);
#endif
  else XtVaGetValues(w, XmNbuttonCount, &len, NULL);
  return(len);
}

static XEN C_TO_XEN_ANY(Widget w, Arg arg)
{
  /* XtGetValues -- a list of pairs: resource-name place-holder where we fill in the 2nd element */
  /* this has to wait until the last moment to decide what arg.value is -- if we assume unsigned long*
   *   (which works on the Pentium/Linux), the Sun gets confused by Dimensions (unsigned short)
   */
  Arg a[1];
  int j, ilen = 0;
  switch (resource_type(arg.name))
    {
    case XM_INT:	      return(C_TO_XEN_INT((*((int *)(arg.value)))));
    case XM_ULONG:	      return(C_TO_XEN_ULONG((*((unsigned long *)(arg.value)))));
    case XM_UCHAR:	      return(C_TO_XEN_INT((*((unsigned char *)(arg.value)))));
    case XM_FLOAT:	      return(C_TO_XEN_DOUBLE((*((float *)(arg.value))))); /* the resource values are floats */
    case XM_STRING:	      return(C_TO_XEN_STRING((char *)(*((char **)(arg.value)))));
    case XM_STRING_OR_XMSTRING: /* fileselectionbox here , not parsemapping */
    case XM_XMSTRING:	      return(C_TO_XEN_XmString((XmString)(*((XmString *)(arg.value)))));
    case XM_STRING_TABLE:     return(C_TO_XEN_XmStringTable((XmStringTable)(*((XmStringTable *)(arg.value))), xmstringtable_length(w, arg.name)));
#if MOTIF_2
    case XM_RENDER_TABLE:     return(C_TO_XEN_XmRenderTable((XmRenderTable)(*((XmRenderTable *)(arg.value)))));
    case XM_TAB_LIST:	      return(C_TO_XEN_XmTabList((XmTabList)(*((XmTabList *)(arg.value)))));
#endif
    case XM_WIDGET:	      return(C_TO_XEN_Widget((Widget)(*((Widget *)(arg.value)))));
    case XM_WIDGET_LIST:      /* (XtGetValues c1 (list XmNchildren 0) 1) */
#if MOTIF_2
      if (strcmp(arg.name, XmNchildren) == 0)
	XtSetArg(a[0], XmNnumChildren, &ilen);             /* Composite */
      else XtSetArg(a[0], XmNselectedObjectCount, &ilen);  /* Container */
#else
      XtSetArg(a[0], XmNnumChildren, &ilen);
#endif
      XtGetValues(w, a, 1);
      return(C_TO_XEN_Widgets((Widget *)(*((Widget **)(arg.value))), ilen));
      break;
    case XM_BOOLEAN:	      return(C_TO_XEN_BOOLEAN((*((Boolean *)(arg.value)))));
    case XM_BOOLEAN_OR_INT:   return(C_TO_XEN_INT((*((int *)(arg.value)))));
    case XM_SEARCH_CALLBACK:
      j = map_over_protected_elements(find_searchproc, (unsigned long)w);
      if (j >= 0) return(XEN_LIST_REF(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_DROP_CALLBACK:
      j = map_over_protected_elements(find_dropproc, (unsigned long)w);
      if (j >= 0) return(XEN_LIST_REF(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_DRAG_CALLBACK:
      j = map_over_protected_elements(find_dragproc, (unsigned long)w);
      if (j >= 0) return(XEN_LIST_REF(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_ORDER_CALLBACK:
      j = map_over_protected_elements(find_orderproc, (unsigned long)w);
      if (j >= 0) return(XEN_LIST_REF(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_QUALIFY_CALLBACK:
      j = map_over_protected_elements(find_qualifyproc, (unsigned long)w);
      if (j >= 0) return(XEN_LIST_REF(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_ALLOC_COLOR_CALLBACK:
      return(xm_XmColorAllocationProc);
      break;
    case XM_SCREEN_COLOR_CALLBACK:
      return(xm_XmColorCalculationProc);
      break;
    case XM_POPUP_CALLBACK:
      j = map_over_protected_elements(find_popupchild, (unsigned long)w);
      if (j >= 0) return(XEN_LIST_REF(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_TRANSFER_CALLBACK:
      return(xm_XtSelectionCallback_Descr);
    case XM_CONVERT_CALLBACK:
      return(xm_XtConvertSelectionIncr_Descr);
    case XM_CALLBACK:	   
      /* this can't work because we don't know what the desired callback was, so we can't search the gc table */
      /*   and if the callback was a C procedure, what good would its address be in this context? */
      /*   (XtGetValues_1 would need to pass the entire Arg or something) */
      return(C_TO_XEN_ULONG((*((unsigned long *)(arg.value)))));
      break;

    case XM_PIXEL:	      return(C_TO_XEN_Pixel((*((Pixel *)(arg.value)))));
    case XM_PIXMAP:	      return(C_TO_XEN_Pixmap((*((Pixmap *)(arg.value)))));
    case XM_XFONTSTRUCT:      return(C_TO_XEN_XFontStruct((XFontStruct *)(*((XFontStruct **)(arg.value)))));
    case XM_DIMENSION:	      return(C_TO_XEN_Dimension((*((Dimension *)(arg.value)))));
    case XM_POSITION:	      return(C_TO_XEN_Position((*((Position *)(arg.value)))));
    case XM_SHORT:	      return(C_TO_XEN_INT((*((short *)(arg.value)))));
    case XM_ATOM:	      return(C_TO_XEN_Atom((*((Atom *)(arg.value)))));
    case XM_TEXT_SOURCE:      return(C_TO_XEN_XmTextSource((XmTextSource)(*((XmTextSource *)(arg.value)))));
    case XM_ATOM_LIST:	      
      if (strcmp(arg.name, XmNexportTargets) == 0)            /* DragContext */
	XtSetArg(a[0], XmNnumExportTargets, &ilen);        /* DropSite */
      else XtSetArg(a[0], XmNnumImportTargets, &ilen);
      XtGetValues(w, a, 1);
      if ((ilen > 0) && (ilen < 100))
	return(C_TO_XEN_Atoms((Atom *)(*((Atom **)(arg.value))), ilen));
      else return(XEN_FALSE);
      break;
    case XM_STRING_LIST:                                  /* ApplicationShell */
      XtSetArg(a[0], XmNargc, &ilen);
      XtGetValues(w, a, 1);
      return(C_TO_XEN_Strings((char **)(*((char ***)(arg.value))), ilen));
      break;
    case XM_CHARSET_TABLE:    
      XtSetArg(a[0], XmNbuttonCount, &ilen); /* may not be long enough... */
      XtGetValues(w, a, 1);
      return(C_TO_XEN_Strings((char **)(*((char ***)(arg.value))), ilen));
      break;
#if (!XM_DISABLE_DEPRECATED)
    case XM_FONTLIST:	      return(C_TO_XEN_XmFontList((XmFontList)(*((XmFontList *)(arg.value)))));
#endif
#if HAVE_XmCreateMultiList
    case XM_ROW_INFO:         return(C_TO_XEN_XmMultiListRowInfo((XmMultiListRowInfo *)(*((XmMultiListRowInfo **)(arg.value)))));
#endif
    case XM_COLORMAP:	      return(C_TO_XEN_Colormap((*((Colormap *)(arg.value)))));
    case XM_KEYSYM:	      return(C_TO_XEN_KeySym((*((KeySym *)(arg.value)))));
    case XM_KEYSYM_TABLE:     
      XtSetArg(a[0], XmNbuttonCount, &ilen);
      XtGetValues(w, a, 1);
      return(C_TO_XEN_KeySyms((KeySym *)(*((KeySym **)(arg.value))), ilen));
      break;
    case XM_SCREEN:	      return(C_TO_XEN_Screen((Screen *)(*((Screen **)(arg.value)))));
    case XM_WINDOW:	      return(C_TO_XEN_Window((Window)(*((Window *)(arg.value)))));
    case XM_VISUAL:	      return(C_TO_XEN_Visual((Visual *)(*((Visual **)(arg.value)))));
    case XM_INT_TABLE:	      
#if MOTIF_2
      if (strcmp(arg.name, XmNdetailOrder) == 0) 
	XtSetArg(a[0], XmNdetailOrderCount, &ilen); 
      else
	{
	  if (strcmp(arg.name, XmNselectionArray) == 0) 
	    XtSetArg(a[0], XmNselectionArrayCount, &ilen); 
	  else XtSetArg(a[0], XmNselectedPositionCount, &ilen);
	}
      XtGetValues(w, a, 1);
      return(C_TO_XEN_Ints((int *)(*((int **)(arg.value))), ilen));
#endif
      break;
    case XM_RECTANGLE_LIST:                              /* XmNrectangles exists but is not documented */
      XtSetArg(a[0], XmNnumDropRectangles, &ilen);
      XtGetValues(w, a, 1);
      return(C_TO_XEN_XRectangles((XRectangle *)(*((XRectangle **)(arg.value))), ilen));
      break;
    case XM_WIDGET_CLASS:     return(C_TO_XEN_WidgetClass((WidgetClass)(*((WidgetClass *)(arg.value)))));
    case XM_STRING_OR_INT:
      if ((w) &&
	  ((XmIsText(w)) || (XmIsTextField(w))))
	return(C_TO_XEN_STRING((char *)(*((char **)(arg.value)))));
      else return(C_TO_XEN_INT((int)(*((int *)(arg.value)))));
      break;
    default:
      break;
    }
  return(C_TO_XEN_ULONG((*((unsigned long *)(arg.value))))); /* fallback */
}

static XEN C_TO_XEN_Args(Widget w, Arg *args, int len)
{
  /* here XtGetValues (or equivalent) has filled in the resource values and we need to pass them back
   *   to scheme properly wrapped
   */
  int i, loc;
  XEN lst = XEN_EMPTY_LIST;
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--) 
    lst = XEN_CONS_2(C_TO_XEN_STRING(args[i].name),
		     C_TO_XEN_ANY(w, args[i]),
		     lst);
  xm_unprotect_at(loc);
  return(lst);
}

/* (XtGetValues c1 (list XmNx 0) 1) */

static XEN gxm_XtGetValues_1(XEN arg1, XEN larg2, int len)
{
  Arg *args;
  unsigned long *locs;
  Widget w;
  XEN val, arg2;
  int i, gcloc;
  /* here we need to make sure the ref args are ok from C's point of view */
  if (len <= 0) return(XEN_FALSE);
  w = XEN_TO_C_Widget(arg1);
  arg2 = XEN_COPY_ARG(larg2);
  gcloc = xm_protect(arg2);
  args = (Arg *)CALLOC(len, sizeof(Arg));
  locs = (unsigned long *)CALLOC(len, sizeof(unsigned long));
  for (i = 0; i < len; i++, arg2 = XEN_CDDR(arg2))
    {
      char *name;
      name = strdup(XEN_TO_C_STRING(XEN_CAR(arg2)));
      XtSetArg(args[i], name, &(locs[i]));
    }
  XtGetValues(w, args, len);
  val = C_TO_XEN_Args(w, args, len);
  free_args(args, len);
  FREE(locs);
  xm_unprotect_at(gcloc);
  return(xen_return_first(val, larg2));
}

#endif


/* -------------------------------- gc protection -------------------------------- */

static XEN xm_protected = XEN_FALSE;
static int xm_protected_size = 0;
static XEN xm_gc_table = XEN_FALSE;
static int last_xm_unprotect = -1;

static int xm_protect(XEN obj)
{
  int i, new_size;
  XEN new_table;
  XEN *older, *newer;
  older = XEN_VECTOR_ELEMENTS(xm_protected);
  /* VECTOR_SET here and below */
  if (last_xm_unprotect >= 0)
    {
      i = last_xm_unprotect;
      if (XEN_FALSE_P(older[i]))
	{
	  older[i] = obj;
	  last_xm_unprotect = -1;
	  return(i);
	}
      last_xm_unprotect = -1;
    }
  for (i = 0; i < xm_protected_size; i++)
    if (XEN_FALSE_P(older[i]))
      {
	older[i] = obj;
	return(i);
      }
  new_size = xm_protected_size * 2;
  new_table = XEN_MAKE_VECTOR(new_size, XEN_FALSE);
  newer = XEN_VECTOR_ELEMENTS(new_table);
  for (i = 0; i < xm_protected_size; i++)
    {
      newer[i] = older[i];
      older[i] = XEN_FALSE;
    }
  newer[xm_protected_size] = obj;
  XEN_VECTOR_SET(xm_gc_table, 0, new_table);
  i = xm_protected_size;
  xm_protected_size = new_size;
  xm_protected = new_table;
  return(i);
}

static void xm_unprotect(XEN obj)
{
  int i;
  XEN *velts;
  velts = XEN_VECTOR_ELEMENTS(xm_protected);
  for (i = 0; i < xm_protected_size; i++)
    if (XEN_EQ_P(velts[i], obj))
      {
	velts[i] = XEN_FALSE;
	last_xm_unprotect = i;
	return;
      }
}

static void xm_unprotect_at(int ind)
{
  XEN *velts;
  velts = XEN_VECTOR_ELEMENTS(xm_protected);
  velts[ind] = XEN_FALSE;
  last_xm_unprotect = ind;
}

static int map_over_protected_elements(bool (*func)(XEN val, int loc, unsigned long fid), unsigned long id)
{
  int i;
  XEN *velts;
  velts = XEN_VECTOR_ELEMENTS(xm_protected);
  for (i = 0; i < xm_protected_size; i++)
    if (func(velts[i], i, id))
      return(i);
  return(-1);
}

static XEN xm_protected_element(int loc)
{
  XEN *velts;
  velts = XEN_VECTOR_ELEMENTS(xm_protected);
  return(velts[loc]);
}


/* ----------------------------------------------------------------------------------------------------
 *
 *                            *         *  *         *
 *                             *       *   **       **
 *                              *     *    * *     * *
 *                               *   *     *  *   *  *
 *                                * *      *   * *   *
 *                                 *       *    *    *
 *                                * *      *         *
 *                               *   *     *         *
 *                              *     *    *         *
 *                             *       *   *         *
 *                            *         *  *         *
 * ----------------------------------------------------------------------------------------------------
 */

/* Motif */
#if HAVE_MOTIF

/* weird order of procedures here and throughout caused by the C-header scripts --
 *   basically I forgot to reverse the main lists before pouring out the code,
 *   so everything is slightly backwards
 */

static int XEN_TO_C_INT_DEF(XEN len, XEN lst) 
{
  if (XEN_INTEGER_P(len))
    return(XEN_TO_C_INT(len));
  else
    {
      int list_len;
      list_len = XEN_LIST_LENGTH(lst);
      return((int)(list_len / 2));
    }
}

#if MOTIF_2
static XEN gxm_XmTransferDone(XEN arg1, XEN arg2)
{
  #define H_XmTransferDone "void XmTransferDone(XtPointer transfer_id, XmTransferStatus status) completes a data transfer"
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, 1, "XmTransferDone", "XtPointer");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTransferDone", "XmTransferStatus");
  XmTransferDone((XtPointer)XEN_TO_C_ULONG(arg1), (XmTransferStatus)XEN_TO_C_INT(arg2));
  return(arg1);
}

static XEN gxm_XmTransferSendRequest(XEN arg1, XEN arg2)
{
  #define H_XmTransferSendRequest "void XmTransferSendRequest(XtPointer transfer_id, Time time) transfers a MULTIPLE request"
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, 1, "XmTransferSendRequest", "XtPointer");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmTransferSendRequest", "Time");
  XmTransferSendRequest((XtPointer)XEN_TO_C_ULONG(arg1), XEN_TO_C_Time(arg2));
  return(arg1);
}

static XEN gxm_XmTransferStartRequest(XEN arg1)
{
  #define H_XmTransferStartRequest "void XmTransferStartRequest(XtPointer transfer_id) begins a MULTIPLE transfer"
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, 1, "XmTransferStartRequest", "XtPointer");
  XmTransferStartRequest((XtPointer)XEN_TO_C_ULONG(arg1));
  return(arg1);
}

static XEN gxm_XmTransferSetParameters(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmTransferSetParameters "void XmTransferSetParameters(XtPointer transfer_id, XtPointer parm, int parm_fmt, \
unsigned long parm_length, Atom parm_type)  establishes parameters to be passed by the next call to XmTransferValue"
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, 1, "XmTransferSetParameters", "XtPointer");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XmTransferSetParameters", "XtPointer");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTransferSetParameters", "an integer");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XmTransferSetParameters", "unsigned long");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg5), arg5, 5, "XmTransferSetParameters", "an Atom");
  XmTransferSetParameters((XtPointer)XEN_TO_C_ULONG(arg1), 
			  (XtPointer)XEN_TO_C_ULONG(arg2), 
			  XEN_TO_C_INT(arg3),
			  XEN_TO_C_ULONG(arg4), 
			  XEN_TO_C_Atom(arg5));
  return(arg1);
}

static void gxm_TransferValueProc(Widget w, XtPointer context, XtPointer info)
{
  XEN descr = (XEN)context;
  XEN_CALL_3(XEN_LIST_REF(descr, 0),
	     C_TO_XEN_Widget(w),
	     XEN_LIST_REF(descr, 1),
	     C_TO_XEN_XmSelectionCallbackStruct((XmSelectionCallbackStruct *)info),
	     c__FUNCTION__);
}

static XEN gxm_XmTransferValue(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmTransferValue "void XmTransferValue(XtPointer transfer_id, Atom target, XtCallbackProc proc, XtPointer client_data, Time time) \
transfers data to a destination"
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, 1, "XmTransferValue", "XtPointer");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XmTransferValue", "an Atom");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg3) && (XEN_REQUIRED_ARGS_OK(arg3, 3)), arg3, 3, "XmTransferValue", "XtCallbackProc (3 args)");
  XEN_ASSERT_TYPE(XEN_Time_P(arg5), arg5, 5, "XmTransferValue", "Time");
  XmTransferValue((XtPointer)XEN_TO_C_ULONG(arg1), 
		  XEN_TO_C_Atom(arg2), 
		  gxm_TransferValueProc,
		  (XtPointer)XEN_LIST_2(arg3, arg4),
		  XEN_TO_C_Time(arg5));
  return(arg1);
}
#endif

static XEN gxm_new_widget(const char *caller, Widget (*func)(Widget parent, char *name, ArgList al, Cardinal ac), XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, caller, "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, caller, "char*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, caller, "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg4), arg4, 4, caller, "int");
  {
    Arg *args;
    int arglen;
    args = XEN_TO_C_Args(arg3);
    w = (*func)(XEN_TO_C_Widget(arg1), 
		XEN_TO_C_STRING(arg2), 
		args, 
		arglen = XEN_TO_C_INT_DEF(arg4, arg3));
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XmCreateMenuShell(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateMenuShell "Widget XmCreateMenuShell(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MenuShell widget creation function"
  return(gxm_new_widget("XmCreateMenuShell", XmCreateMenuShell, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmProcessTraversal(XEN arg1, XEN arg2)
{
  #define H_XmProcessTraversal "Boolean XmProcessTraversal(Widget widget, XmTraversalDirection direction) determines which \
component receives keyboard events when a widget has the focus"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmProcessTraversal", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmProcessTraversal", "XmTraversalDirection");
  return(C_TO_XEN_BOOLEAN(XmProcessTraversal(XEN_TO_C_Widget(arg1), (XmTraversalDirection)XEN_TO_C_INT(arg2))));
}

static XEN gxm_XmGetFocusWidget(XEN arg1)
{
  #define H_XmGetFocusWidget "Widget XmGetFocusWidget(Widget widget) returns the ID of the widget that has keyboard focus"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmGetFocusWidget", "Widget");
  return(C_TO_XEN_Widget(XmGetFocusWidget(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmGetTabGroup(XEN arg1)
{
  #define H_XmGetTabGroup "Widget XmGetTabGroup(Widget widget) returns the widget ID of a tab group"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmGetTabGroup", "Widget");
  return(C_TO_XEN_Widget(XmGetTabGroup(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmGetVisibility(XEN arg1)
{
  #define H_XmGetVisibility "XmVisibility XmGetVisibility(Widget widget) determines if a widget is visible"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmGetVisibility", "Widget");
  return(C_TO_XEN_INT(XmGetVisibility(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmIsTraversable(XEN arg1)
{
  #define H_XmIsTraversable "Boolean XmIsTraversable(Widget widget) identifies whether a widget can be traversed"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmIsTraversable", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsTraversable(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmGetDestination(XEN arg1)
{
  #define H_XmGetDestination "Widget XmGetDestination(Display *display) gets the current destination widget for paste etc"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmGetDestination", "Display*");
  return(C_TO_XEN_Widget(XmGetDestination(XEN_TO_C_Display(arg1))));
}

#if MOTIF_2
static XEN gxm_XmRenderTableAddRenditions(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmRenderTableAddRenditions "XmRenderTable XmRenderTableAddRenditions(XmRenderTable oldtable, \
XmRendition *renditions, Cardinal rendition_count, XmMergeMode merge_mode) adds renditions to a render table"
  /* DIFF: XmRenderTableAddRenditions arg2 is list of Renditions, arg1 can be #f = NULL
   */
  int len, listlen;
  XmRendition *rs;
  XmRenderTable res;
  XEN_ASSERT_TYPE(XEN_XmRenderTable_P(arg1) || XEN_FALSE_P(arg1), arg1, 1, "XmRenderTableAddRenditions", "XmRenderTable");
  XEN_ASSERT_TYPE(XEN_LIST_P_WITH_LENGTH(arg2, listlen), arg2, 2, "XmRenderTableAddRenditions", "list of XmRendition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmRenderTableAddRenditions", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmRenderTableAddRenditions", "XmMergeMode");
  len = XEN_TO_C_INT(arg3);
  if (len > listlen) len = listlen;
  if (len <= 0) return(XEN_FALSE);
  rs = XEN_TO_C_XmRenditions(arg2, len);
  res = XmRenderTableAddRenditions(XEN_FALSE_P(arg1) ? NULL : XEN_TO_C_XmRenderTable(arg1), 
				   rs, (Cardinal)len, 
				   (XmMergeMode)XEN_TO_C_INT(arg4));
  FREE(rs);
  return(C_TO_XEN_XmRenderTable(res));
}

static XEN gxm_XmRenderTableRemoveRenditions(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmRenderTableRemoveRenditions "XmRenderTable XmRenderTableRemoveRenditions(XmRenderTable oldtable, XmStringTag *tags, int tag_count) \
removes renditions"
  /* DIFF: XmRenderTableRemoveRenditions arg2 is list of strings
   */
  int len = 0;
  XmStringTag *tags = NULL;
  XmRenderTable rt = NULL;
  if (XEN_NOT_BOUND_P(arg1) || XEN_FALSE_P(arg1)) return(XEN_FALSE);
  XEN_ASSERT_TYPE(XEN_XmRenderTable_P(arg1) || XEN_FALSE_P(arg1), arg1, 1, "XmRenderTableRemoveRenditions", "XmRenderTable");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2) || XEN_FALSE_P(arg2) || XEN_NOT_BOUND_P(arg2), arg2, 2, "XmRenderTableRemoveRenditions", "XmStringTag*");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmRenderTableRemoveRenditions", "int");
  if (XEN_INTEGER_P(arg3)) 
    {
      len = XEN_TO_C_INT(arg3); 
      if (len <= 0) return(XEN_FALSE);
    }
  else 
    {
      if (XEN_LIST_P(arg2))
	len = XEN_LIST_LENGTH(arg2);
    }
  if (XEN_BOUND_P(arg2)) tags = (XmStringTag *)XEN_TO_C_Strings(arg2, len);
  rt = XmRenderTableRemoveRenditions(XEN_TO_C_XmRenderTable(arg1), tags, len);
  if (tags) FREE(tags);
  return(xen_return_first(C_TO_XEN_XmRenderTable(rt), arg2));
}

static XEN gxm_XmRenderTableCopy(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmRenderTableCopy "XmRenderTable XmRenderTableCopy(XmRenderTable table, XmStringTag *tags, int tag_count) copies renditions"
  /* DIFF: XmRenderTableCopy arg2 is list of strings
   */
  int len = 0;
  XmStringTag *tags = NULL;
  XmRenderTable rt = NULL;
  if (XEN_NOT_BOUND_P(arg1) || XEN_FALSE_P(arg1)) return(XEN_FALSE);
  XEN_ASSERT_TYPE(XEN_XmRenderTable_P(arg1), arg1, 1, "XmRenderTableCopy", "XmRenderTable");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2) || XEN_FALSE_P(arg2) || XEN_NOT_BOUND_P(arg2), arg2, 2, "XmRenderTableCopy", "XmStringTag*");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmRenderTableCopy", "int");
  if (XEN_INTEGER_P(arg3)) 
    {
      len = XEN_TO_C_INT(arg3); 
      if (len <= 0) return(XEN_FALSE);
    }
  else 
    {
      if (XEN_LIST_P(arg2))
	len = XEN_LIST_LENGTH(arg2);
    }
  if (XEN_BOUND_P(arg2)) tags = (XmStringTag *)XEN_TO_C_Strings(arg2, len);
  rt = XmRenderTableCopy(XEN_TO_C_XmRenderTable(arg1), tags, len);
  if (tags) FREE(tags);
  return(xen_return_first(C_TO_XEN_XmRenderTable(rt), arg2));
}

static XEN gxm_XmRenderTableFree(XEN arg1)
{
  #define H_XmRenderTableFree "void XmRenderTableFree(XmRenderTable table)  recovers memory"
  XEN_ASSERT_TYPE(XEN_XmRenderTable_P(arg1), arg1, 1, "XmRenderTableFree", "XmRenderTable");
  XmRenderTableFree(XEN_TO_C_XmRenderTable(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmRenderTableGetTags(XEN arg1)
{
  #define H_XmRenderTableGetTags "int XmRenderTableGetTags(XmRenderTable table) gets rendition tags (list of strings)"
  /* DIFF: XmRenderTableGetTags omits arg2, returns list of strings
   */
  int len, loc;
  XEN lst = XEN_EMPTY_LIST;
  XmStringTag *str;
  XEN_ASSERT_TYPE(XEN_XmRenderTable_P(arg1), arg1, 1, "XmRenderTableGetTags", "XmRenderTable");
  loc = xm_protect(lst);
  len = XmRenderTableGetTags(XEN_TO_C_XmRenderTable(arg1), &str);
  if (str)
    {
      int i;
      for (i = len - 1; i >= 0; i--)
	{
	  lst = XEN_CONS(C_TO_XEN_STRING(str[i]), lst);
	  free(str[i]);
	}
      free(str);
    }
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XmRenderTableGetRendition(XEN arg1, XEN arg2)
{
  #define H_XmRenderTableGetRendition "XmRendition XmRenderTableGetRendition(XmRenderTable table, XmStringTag tag) \
matches a rendition tag"
  XEN_ASSERT_TYPE(XEN_XmRenderTable_P(arg1), arg1, 1, "XmRenderTableGetRendition", "XmRenderTable");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmRenderTableGetRendition", "XmStringTag");
  return(C_TO_XEN_XmRendition(XmRenderTableGetRendition(XEN_TO_C_XmRenderTable(arg1), XEN_TO_C_STRING(arg2))));
}

static XEN gxm_XmRenderTableGetRenditions(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmRenderTableGetRenditions "XmRendition *XmRenderTableGetRenditions(XmRenderTable table, XmStringTag *tags, Cardinal tag_count) \
matches rendition tags"
  /* DIFF: XmRenderTableGetRenditions returns list of XmRenditions, arg2 is list of strings
   */
  int i, len = 0, loc;
  XEN lst = XEN_EMPTY_LIST;
  XmRendition *rs;
  XmStringTag *tags;
  if ((XEN_NOT_BOUND_P(arg1)) || XEN_NOT_BOUND_P(arg2) || XEN_FALSE_P(arg2)) return(XEN_FALSE);
  XEN_ASSERT_TYPE(XEN_XmRenderTable_P(arg1), arg1, 1, "XmRenderTableGetRenditions", "XmRenderTable");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmRenderTableGetRenditions", "list of String");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmRenderTableGetRenditions", "int");
  if (XEN_BOUND_P(arg3))
    len = XEN_TO_C_INT(arg3);
  else len = XEN_LIST_LENGTH(arg2);
  if (len <= 0) return(XEN_FALSE);
  loc = xm_protect(lst);
  tags = (XmStringTag *)XEN_TO_C_Strings(arg2, len);
  rs = XmRenderTableGetRenditions(XEN_TO_C_XmRenderTable(arg1), tags, (Cardinal)len);
  FREE(tags);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_XmRendition(rs[i]), lst);
  free(rs);
  xm_unprotect_at(loc);
  return(xen_return_first(lst, arg2));
}

static XEN gxm_XmRenditionCreate(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmRenditionCreate "XmRendition XmRenditionCreate(Widget widget, XmStringTag tag, ArgList arglist, Cardinal argcount) \
creates a rendition"
  XmRendition w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmRenditionCreate", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmRenditionCreate", "XmStringTag");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XmRenditionCreate", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg4), arg4, 4, "XmRenditionCreate", "int");
  {
    Arg *args;
    int arglen;
    args = XEN_TO_C_Args(arg3);
    arglen = XEN_TO_C_INT_DEF(arg4, arg3);
    w = XmRenditionCreate(XEN_TO_C_Widget(arg1), XEN_TO_C_STRING(arg2), args, arglen);
    if (args) free_args(args, arglen);
  }
  return(C_TO_XEN_XmRendition(w));
}

static XEN gxm_XmRenditionFree(XEN arg1)
{
  #define H_XmRenditionFree "void XmRenditionFree(XmRendition rendition) frees a rendition"
  XEN_ASSERT_TYPE(XEN_XmRendition_P(arg1), arg1, 1, "XmRenditionFree", "XmRendition");
  XmRenditionFree(XEN_TO_C_XmRendition(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmRenditionRetrieve(XEN arg1, XEN larg2, XEN arg3)
{
  #define H_XmRenditionRetrieve "void XmRenditionRetrieve(XmRendition rendition, ArgList arglist, Cardinal argcount) \
retrieves rendition resources"
  XEN_ASSERT_TYPE(XEN_XmRendition_P(arg1), arg1, 1, "XmRenditionRetrieve", "XmRendition");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg2), larg2, 2, "XmRenditionRetrieve", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmRenditionRetrieve", "int");
  {
    /* this is a kind of XtGetvalues, not set
     */
    Arg *args;
    unsigned long *locs;
    XEN val;
    int i, len, gcloc;
    XmRendition r;
    char *name;
    XEN arg2;
    arg2 = XEN_COPY_ARG(larg2);
    gcloc = xm_protect(arg2);
    /* here we need to make sure the ref args are ok from C's point of view */
    r = XEN_TO_C_XmRendition(arg1);
    len = XEN_TO_C_INT_DEF(arg3, arg2);
    if (len <= 0) XEN_ASSERT_TYPE(0, arg3, 3, "XmRenditionRetrieve", "positive integer");
    args = (Arg *)CALLOC(len, sizeof(Arg));
    locs = (unsigned long *)CALLOC(len, sizeof(unsigned long));
    for (i = 0; i < len; i++, arg2 = XEN_CDDR(arg2))
      {
	name = strdup(XEN_TO_C_STRING(XEN_CAR(arg2)));
	XtSetArg(args[i], name, &(locs[i]));
      }
    XmRenditionRetrieve(r, args, len);
    val = C_TO_XEN_Args((Widget)r, args, len);
    free_args(args, len);
    FREE(locs);
    xm_unprotect_at(gcloc);
    return(xen_return_first(val, larg2));
  }
}

static XEN gxm_XmRenditionUpdate(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmRenditionUpdate "void XmRenditionUpdate(XmRendition rendition, ArgList arglist, Cardinal argcount) \
modifies resources"
  XEN_ASSERT_TYPE(XEN_XmRendition_P(arg1), arg1, 1, "XmRenditionUpdate", "XmRendition");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmRenditionUpdate", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmRenditionUpdate", "int");
  {
    Arg *args;
    int arglen;
    args = XEN_TO_C_Args(arg2);
    arglen = XEN_TO_C_INT_DEF(arg3, arg2);
    XmRenditionUpdate(XEN_TO_C_XmRendition(arg1), args, arglen);
    if (args) free_args(args, arglen);
  }
  return(XEN_FALSE);
}

static XEN gxm_XmRenderTableCvtToProp(XEN arg1, XEN arg2)
{
  #define H_XmRenderTableCvtToProp "unsigned int XmRenderTableCvtToProp(Widget widget, XmRenderTable table) \
converts a render table to a string representation -> (val props)"
  /* DIFF: XmRenderTableCvtToProp omits and rtns arg3
   */
  char *buf;
  unsigned int val;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmRenderTableCvtToProp", "Widget");
  XEN_ASSERT_TYPE(XEN_XmRenderTable_P(arg2), arg2, 2, "XmRenderTableCvtToProp", "XmRenderTable");
  val = XmRenderTableCvtToProp(XEN_TO_C_Widget(arg1), XEN_TO_C_XmRenderTable(arg2), &buf);
  return(XEN_LIST_2(C_TO_XEN_ULONG(val),
		    C_TO_XEN_STRING(buf)));
}

static XEN gxm_XmRenderTableCvtFromProp(XEN arg1, XEN arg2, XEN arg3)
{
  char *str;
  int len;
  #define H_XmRenderTableCvtFromProp "XmRenderTable XmRenderTableCvtFromProp(Widget widget, char *property, unsigned int length) \
converts from a string representation to a render table"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmRenderTableCvtFromProp", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmRenderTableCvtFromProp", "char*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XmRenderTableCvtFromProp", "unsigned int");
  str = XEN_TO_C_STRING(arg2);
  len = XEN_TO_C_ULONG(arg3);
  if ((str) || ((int)strlen(str) == len))
    return(C_TO_XEN_XmRenderTable(XmRenderTableCvtFromProp(XEN_TO_C_Widget(arg1), str, len)));
  return(XEN_FALSE);
}

static XEN gxm_XmTabListInsertTabs(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmTabListInsertTabs "XmTabList XmTabListInsertTabs(XmTabList oldlist, XmTab *tabs, Cardinal tab_count, int position) \
inserts tabs into a tab list"
  /* DIFF: XmTabListInsertTabs arg2 is list of XmTabs (can be #f = NULL)
   */
  int len, listlen;
  XmTab *tabs;
  XmTabList tl;
  XEN_ASSERT_TYPE(XEN_XmTabList_P(arg1) || XEN_FALSE_P(arg1), arg1, 1, "XmTabListInsertTabs", "XmTabList");
  XEN_ASSERT_TYPE(XEN_LIST_P_WITH_LENGTH(arg2, listlen), arg2, 2, "XmTabListInsertTabs", "list of XmTab");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTabListInsertTabs", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmTabListInsertTabs", "int");
  len = XEN_TO_C_INT(arg3);
  if (len > listlen) len = listlen;
  if (len <= 0) return(XEN_FALSE);
  tabs = XEN_TO_C_XmTabs(arg2, len);
  tl = XmTabListInsertTabs(XEN_FALSE_P(arg1) ? NULL : XEN_TO_C_XmTabList(arg1), 
			   tabs, len, XEN_TO_C_INT(arg4));
  FREE(tabs);
  return(C_TO_XEN_XmTabList(tl));
}

static XEN gxm_XmTabListCopy(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmTabListCopy "XmTabList XmTabListCopy(XmTabList tablist, int offset, Cardinal count) creates \
a new tab list from an existing list"
  /* Motif documentation incorrectly calls this "XmTabListTabCopy" */
  XEN_ASSERT_TYPE(XEN_XmTabList_P(arg1), arg1, 1, "XmTabListCopy", "XmTabList");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTabListCopy", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTabListCopy", "int");
  return(C_TO_XEN_XmTabList(XmTabListCopy(XEN_TO_C_XmTabList(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XmTabListTabCount(XEN arg1)
{
  #define H_XmTabListTabCount "Cardinal XmTabListTabCount(XmTabList tablist) returns length of tablist"
  XEN_ASSERT_TYPE(XEN_XmTabList_P(arg1), arg1, 1, "XmTabListTabCount", "XmTabList");  
  return(C_TO_XEN_INT(XmTabListTabCount(XEN_TO_C_XmTabList(arg1))));
}


static XEN gxm_XmTabListGetTab(XEN arg1, XEN arg2)
{
  #define H_XmTabListGetTab "XmTab XmTabListGetTab(XmTabList tablist, Cardinal position) returns a copy of a tab"
  XEN_ASSERT_TYPE(XEN_XmTabList_P(arg1), arg1, 1, "XmTabListGetTab", "XmTabList");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTabListGetTab", "int");
  return(C_TO_XEN_XmTab(XmTabListGetTab(XEN_TO_C_XmTabList(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XmTabListReplacePositions(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmTabListReplacePositions "XmTabList XmTabListReplacePositions(XmTabList oldlist, Cardinal *position_list, XmTab *tabs, Cardinal tab_count) \
creates a new tab list with replacement tabs"
  /* DIFF: XmTabListReplacePositions arg2 is list of ints, arg3 is list of XmTabs
   */
  Cardinal *ts;
  int len;
  XEN res;
  XmTab *tabs;
  XEN_ASSERT_TYPE(XEN_XmTabList_P(arg1), arg1, 1, "XmTabListReplacePositions", "XmTabList");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmTabListReplacePositions", "list of ints");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XmTabListReplacePositions", "list of XmTab");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg4), arg4, 4, "XmTabListReplacePositions", "int");
  if (XEN_INTEGER_P(arg4)) len = XEN_TO_C_INT(arg4); else len = XEN_LIST_LENGTH(arg3);
  if (len <= 0) return(XEN_FALSE);
  ts = XEN_TO_C_Cardinals(arg2, len);
  tabs = XEN_TO_C_XmTabs(arg3, len);
  res = C_TO_XEN_XmTabList(XmTabListReplacePositions(XEN_TO_C_XmTabList(arg1), ts, tabs, len));
  FREE(ts);
  FREE(tabs);
  return(res);
}

static XEN gxm_XmTabListRemoveTabs(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmTabListRemoveTabs "XmTabList XmTabListRemoveTabs(XmTabList oldlist, Cardinal *position_list, Cardinal position_count) \
removes noncontiguous tabs"
  /* DIFF: XmTabListRemoveTabs arg2 is list of ints
   */
  Cardinal *ts;
  int len;
  XEN res;
  XEN_ASSERT_TYPE(XEN_XmTabList_P(arg1), arg1, 1, "XmTabListRemoveTabs", "XmTabList");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmTabListRemoveTabs", "list of int");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmTabListRemoveTabs", "int");
  if (XEN_INTEGER_P(arg3)) len = XEN_TO_C_INT(arg3); else len = XEN_LIST_LENGTH(arg2);
  if (len <= 0) return(XEN_FALSE);
  ts = XEN_TO_C_Cardinals(arg2, len);
  res = C_TO_XEN_XmTabList(XmTabListRemoveTabs(XEN_TO_C_XmTabList(arg1), ts, len));
  FREE(ts);
  return(res);
}

static XEN gxm_XmTabCreate(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmTabCreate "XmTab XmTabCreate(float value, unsigned char units, XmOffsetModel offset_model, unsigned char alignment, char *decimal) \
creates a tab stop"
  XEN_ASSERT_TYPE(XEN_DOUBLE_P(arg1), arg1, 1, "XmTabCreate", "float");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTabCreate", "unsigned char");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTabCreate", "XmOffsetModel"); 
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmTabCreate", "unsigned char");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg5), arg5, 5, "XmTabCreate", "char*");
  return(C_TO_XEN_XmTab(XmTabCreate((float)XEN_TO_C_DOUBLE(arg1), 
				    (unsigned char)(XEN_TO_C_INT(arg2)),
				    (XmOffsetModel)XEN_TO_C_INT(arg3), 
				    (unsigned char)(XEN_TO_C_INT(arg4)),
				    XEN_TO_C_STRING(arg5))));
}

static XEN gxm_XmTabListFree(XEN arg1)
{
  #define H_XmTabListFree "void XmTabListFree(XmTabList tab) frees a tab list"
  XEN_ASSERT_TYPE(XEN_XmTabList_P(arg1), arg1, 1, "XmTabListFree", "XmTabList");
  XmTabListFree(XEN_TO_C_XmTabList(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmTabFree(XEN arg1)
{
  #define H_XmTabFree "void XmTabFree(XmTab tab) frees a tab"
  XEN_ASSERT_TYPE(XEN_XmTab_P(arg1), arg1, 1, "XmTabFree", "XmTab");
  XmTabFree(XEN_TO_C_XmTab(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmTabGetValues(XEN arg1)
{
  #define H_XmTabGetValues "float XmTabGetValues(XmTab tab) returns tab values"
  /* DIFF: XmTabGetValues tab [units offset align decimal] -> (list val units ofset align decimal)
     arg2/3/4/5 omit rtn with float 
  */
  XmOffsetModel off;
  unsigned char a1, a2;
  char **a3;
  float res;
  XEN val;
  XEN_ASSERT_TYPE(XEN_XmTab_P(arg1), arg1, 1, "XmTabGetValues", "XmTab");
  a3 = (char **)CALLOC(1, sizeof(char *));
  res = XmTabGetValues(XEN_TO_C_XmTab(arg1), &a1, &off, &a2, a3);
  val = XEN_LIST_5(C_TO_XEN_DOUBLE((double)res),
		   C_TO_XEN_INT((int)a1),
		   C_TO_XEN_INT((int)off),
		   C_TO_XEN_INT((int)a2),
		   C_TO_XEN_STRING(a3[0]));
  FREE(a3);
  return(val);
}

static XEN gxm_XmTabSetValue(XEN arg1, XEN arg2)
{
  #define H_XmTabSetValue "void XmTabSetValue(XmTab tab, float value) sets a tab stop"
  XEN_ASSERT_TYPE(XEN_XmTab_P(arg1), arg1, 1, "XmTabSetValue", "XmTab");
  XEN_ASSERT_TYPE(XEN_DOUBLE_P(arg2), arg2, 2, "XmTabSetValue", "float");
  XmTabSetValue(XEN_TO_C_XmTab(arg1), (float)XEN_TO_C_DOUBLE(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmStringTableProposeTablist(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmStringTableProposeTablist "XmTabList XmStringTableProposeTablist(XmStringTable strings, Cardinal num_strings, Widget widget, \
float pad_value, XmOffsetModel offset_model) returns a tab list"
  /* DIFF: XmStringTableProposetablist 1st arg is list of XmStrings
   */
  XmStringTable tab;
  XmTabList tabl;
  XEN_ASSERT_TYPE(XEN_LIST_P(arg1), arg1, 1, "XmStringTableProposeTablist", "XmStringTable");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmStringTableProposeTablist", "int");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg3), arg3, 3, "XmStringTableProposeTablist", "Widget");
  XEN_ASSERT_TYPE(XEN_DOUBLE_P(arg4), arg4, 4, "XmStringTableProposeTablist", "float");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XmStringTableProposeTablist", "XmOffsetModel");
  tab = XEN_TO_C_XmStringTable(arg1, XEN_TO_C_INT(arg2)); 
  tabl = XmStringTableProposeTablist(tab,
				     XEN_TO_C_INT(arg2), 
				     XEN_TO_C_Widget(arg3),
				     (float)XEN_TO_C_DOUBLE(arg4), 
				     (XmOffsetModel)XEN_TO_C_INT(arg5));
  FREE(tab);
  return(C_TO_XEN_XmTabList(tabl));
}

#define XEN_XmParseTable_P(Arg) XEN_LIST_P(Arg)

static XmParseTable XEN_TO_C_XmParseTable(XEN lst, int size)
{
  int i;
  XmParseTable pt;
  if (size <= 0) return(NULL);
  pt = (XmParseTable)XtCalloc(size, sizeof(XmParseMapping));
  for (i = 0; i < size; i++)
    {
      XEN val;
      val = XEN_LIST_REF(lst, i);
      if (XEN_XmParseMapping_P(val))
	pt[i] = XEN_TO_C_XmParseMapping(val);
      else XEN_ASSERT_TYPE(0, val, i, c__FUNCTION__, "an XmParseMapping");
    }
  return(pt);
}

static XEN gxm_XmParseTableFree(XEN arg1, XEN arg2)
{
  #define H_XmParseTableFree "void XmParseTableFree(XmParseTable parse_table, Cardinal count) recovers memory"
  XEN_ASSERT_TYPE(XEN_XmParseTable_P(arg1), arg1, 1, "XmParseTableFree", "XmParseTable");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg2), arg2, 2, "XmParseTableFree", "int");
  /* can't happen -- a no-op */
  return(XEN_FALSE);
}

static XEN gxm_XmParseMappingFree(XEN arg1)
{
  #define H_XmParseMappingFree "void XmParseMappingFree(XmParseMapping parse_mapping) frees a parse mapping"
  XEN_ASSERT_TYPE(XEN_XmParseMapping_P(arg1), arg1, 1, "XmParseMappingFree", "XmParseMapping");
  XmParseMappingFree(XEN_TO_C_XmParseMapping(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmParseMappingGetValues(XEN arg1, XEN larg2, XEN arg3)
{
  #define H_XmParseMappingGetValues "void XmParseMappingGetValues(XmParseMapping parse_mapping, ArgList arglist, Cardinal argcount) \
retrieves attributes of a parse mapping"
  XEN_ASSERT_TYPE(XEN_XmParseMapping_P(arg1), arg1, 1, "XmParseMappingGetValues", "XmParseMapping");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg2), larg2, 2, "XmParseMappingGetValues", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmParseMappingGetValues", "int");
  {
    Arg *args;
    unsigned long *locs;
    int len, gcloc;
    XEN val, arg2;
    int i;
    len = XEN_TO_C_INT_DEF(arg3, larg2);
    arg2 = XEN_COPY_ARG(larg2);
    gcloc = xm_protect(arg2);
    args = (Arg *)CALLOC(len, sizeof(Arg));
    locs = (unsigned long *)CALLOC(len, sizeof(unsigned long));
    for (i = 0; i < len; i++, arg2 = XEN_CDDR(arg2))
      {
	char *name;
	name = strdup(XEN_TO_C_STRING(XEN_CAR(arg2)));
	XtSetArg(args[i], name, &(locs[i]));
      }
    XmParseMappingGetValues(XEN_TO_C_XmParseMapping(arg1), args, len);
    val = C_TO_XEN_Args(NULL, args, len);
    free_args(args, len);
    FREE(locs);
    xm_unprotect_at(gcloc);
    return(xen_return_first(val, larg2));
  }
}

static XEN gxm_XmParseMappingSetValues(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmParseMappingSetValues "void XmParseMappingSetValues(XmParseMapping parse_mapping, ArgList arglist, Cardinal argcount) \
sets attributes of a parse mapping"
  XEN_ASSERT_TYPE(XEN_XmParseMapping_P(arg1), arg1, 1, "XmParseMappingSetValues", "XmParseMapping");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmParseMappingSetValues", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmParseMappingSetValues", "int");
  {
    Arg *args;
    int arglen;
    args = XEN_TO_C_Args(arg2);
    arglen = XEN_TO_C_INT_DEF(arg3, arg2);
    XmParseMappingSetValues(XEN_TO_C_XmParseMapping(arg1), args, arglen);
    if (args) free_args(args, arglen);
    return(XEN_FALSE);
  }
}

static XEN gxm_XmParseMappingCreate(XEN arg1, XEN arg2)
{
  #define H_XmParseMappingCreate "XmParseMapping XmParseMappingCreate(ArgList arglist, Cardinal argcount) create a parse mapping"
  XmParseMapping w;
  XEN_ASSERT_TYPE(XEN_LIST_P(arg1), arg1, 1, "XmParseMappingCreate", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg2), arg2, 2, "XmParseMappingCreate", "int");
  {
    Arg *args;
    int arglen;
    args = XEN_TO_C_Args(arg1);
    arglen = XEN_TO_C_INT_DEF(arg2, arg1);
    w = XmParseMappingCreate(args, arglen);
    if (args) free_args(args, arglen);
  }
  return(C_TO_XEN_XmParseMapping(w));
}

static XEN gxm_XmCvtXmStringToByteStream(XEN str)
{
  #define H_XmCvtXmStringToByteStream "int XmCvtXmStringToByteStream(XmString str) converts an XmString into a byte stream."
  unsigned char *prop = NULL;
  int res = 0; /* #bytes in returned val */
  XmString xstr;
  XEN_ASSERT_TYPE(XEN_XmString_P(str), str, XEN_ONLY_ARG, "XmCvtXmStringToByteStream", "XmString");
  xstr = XEN_TO_C_XmString(str);
  if (!(XmStringEmpty(xstr)))
    res = XmCvtXmStringToByteStream(xstr, &prop);
  /* here there is apparently no trailing 0, so this function worries Valgrind */
  return(XEN_LIST_2(C_TO_XEN_INT(res), C_TO_XEN_STRING((const char *)prop)));
}

static XEN gxm_XmCvtByteStreamToXmString(XEN str)
{
  #define H_XmCvtByteStreamToXmString "XmString XmCvtByteStreamToXmString(char *str) converts a byte stream into an XmString."
  XmString res = NULL;
  char *bstr;
  XEN_ASSERT_TYPE(XEN_STRING_P(str), str, XEN_ONLY_ARG, "XmCvtByteStreamToXmString", "char *");
  bstr = XEN_TO_C_STRING(str);
  if (bstr)
    res = XmCvtByteStreamToXmString((unsigned char *)(XEN_TO_C_STRING(str)));
  return(C_TO_XEN_XmString(res));
}

static XEN gxm_XmStringByteStreamLength(XEN str)
{
  #define H_XmStringByteStreamLength "int XmStringByteStreamLength(char *str) returns the length of the byte stream."
  int res;
  XEN_ASSERT_TYPE(XEN_STRING_P(str), str, XEN_ONLY_ARG, "XmStringByteStreamLength", "char *");
  res = (int)XmStringByteStreamLength((unsigned char *)(XEN_TO_C_STRING(str)));
  return(C_TO_XEN_INT(res));
}

static XEN gxm_XmStringPutRendition(XEN arg1, XEN arg2)
{
  #define H_XmStringPutRendition "XmString XmStringPutRendition(XmString string, XmStringTag rendition) places \
renditions around strings"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringPutRendition", "XmString");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmStringPutRendition", "XmStringTag");
  return(C_TO_XEN_XmString(XmStringPutRendition(XEN_TO_C_XmString(arg1), XEN_TO_C_STRING(arg2))));
}

static XEN gxm_XmStringGenerate(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmStringGenerate "XmString XmStringGenerate(XtPointer text, XmStringTag tag, XmTextType type, XmStringTag rendition) \
generates a compound string"

  XmTextType type;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmStringGenerate", "XtPointer");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2) || XEN_FALSE_P(arg2), arg2, 2, "XmStringGenerate", "XmStringTag");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmStringGenerate", "XmTextType");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XmStringGenerate", "XmStringTag");
  type = (XmTextType)XEN_TO_C_INT(arg3);
  if (type > 1) XEN_ASSERT_TYPE(0, arg3, 3, "XmStringGenerate", "XmTextType");
  return(C_TO_XEN_XmString(XmStringGenerate((XtPointer)XEN_TO_C_STRING(arg1), 
					    XEN_FALSE_P(arg2) ? NULL : XEN_TO_C_STRING(arg2), 
					    type,
					    XEN_TO_C_STRING(arg4))));
}

static XEN gxm_XmStringDirectionToDirection(XEN arg1)
{
  #define H_XmStringDirectionToDirection "XmDirection XmStringDirectionToDirection(XmStringDirection direction) converts \
from XmStringDirection to XmDirection"
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 1, "XmStringDirectionToDirection", "XmStringDirection");
  return(C_TO_XEN_INT(XmStringDirectionToDirection(XEN_TO_C_INT(arg1))));
}

static XEN gxm_XmDirectionToStringDirection(XEN arg1)
{
  #define H_XmDirectionToStringDirection "XmStringDirection XmDirectionToStringDirection (dir) converts an XmDirection \
value to an XmStringDirection value"
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 1, "XmDirectionToStringDirection", "XmDirection");
  return(C_TO_XEN_INT(XmDirectionToStringDirection(XEN_TO_C_INT(arg1))));
}

static XEN gxm_XmStringTableParseStringArray(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XmStringTableParseStringArray "XmStringTable XmStringTableParseStringArray(XtPointer *strings, Cardinal count, XmStringTag tag, \
XmTextType type, XmParseTable parse, Cardinal parse_count, XtPointer call_data) converts an array of strings \
to a compound string table"
  /* DIFF: XmStringTableParseStringArray arg1 is list of strings
   */
  char **strs;
  int len;
  XmStringTable val;
  XmParseTable pt = NULL;
  XEN lst;
  XmTextType type;
  XEN_ASSERT_TYPE(XEN_LIST_P(arg1), arg1, 1, "XmStringTableParseStringArray", "list of strings");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmStringTableParseStringArray", "int");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg3) || XEN_STRING_P(arg3), arg3, 3, "XmStringTableParseStringArray", "XmStringTag");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmStringTableParseStringArray", "XmTextType");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg5) || XEN_XmParseTable_P(arg5), arg5, 5, "XmStringTableParseStringArray", "XmParseTable");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XmStringTableParseStringArray", "int");
  type = (XmTextType)XEN_TO_C_INT(arg4);
  if (type > 1) XEN_ASSERT_TYPE(0, arg4, 4, "XmStringTableParseStringArray", "XmTextType");
  len = XEN_TO_C_INT(arg2);
  if (len <= 0) return(XEN_FALSE);
  strs = XEN_TO_C_Strings(arg1, len);
  if (XEN_XmParseTable_P(arg5)) pt = XEN_TO_C_XmParseTable(arg5, len);
  val = XmStringTableParseStringArray((XtPointer *)strs, 
				      (Cardinal)len, 
				      (XEN_FALSE_P(arg3)) ? NULL : XEN_TO_C_STRING(arg3),
				      type,
				      pt,
				      XEN_TO_C_INT(arg6),
				      (XtPointer)arg7);
  FREE(strs);
  lst = C_TO_XEN_XmStringTable(val, len);
  free(val);
  if (pt) free(pt); 
  return(xen_return_first(lst, arg1));
}

static XEN gxm_XmStringTableUnparse(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8)
{
  #define H_XmStringTableUnparse "XtPointer *XmStringTableUnparse(XmStringTable table, Cardinal count, XmStringTag tag, XmTextType tag_type, \
XmTextType output_type, XmParseTable parse, Cardinal parse_count, XmParseModel parse_model) converts a table of \
compound strings to an array of text"
  /* DIFF: XmStringTableUnparse returns list of strings, 1st arg is list of XmStrings
   */
  XEN lst = XEN_EMPTY_LIST;
  char **tab;
  XmStringTable tb;
  int i, len, loc;
  XmTextType type1, type2;
  XmParseTable pt = NULL;
  XEN_ASSERT_TYPE(XEN_LIST_P(arg1), arg1, 1, "XmStringTableUnparse", "XmStringTable");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmStringTableUnparse", "int");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg3) || XEN_STRING_P(arg3), arg3, 3, "XmStringTableUnparse", "XmStringTag");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmStringTableUnparse", "XmTextType");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XmStringTableUnparse", "XmTextType");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg6) || XEN_XmParseTable_P(arg6), arg6, 6, "XmStringTableUnparse", "XmParseTable");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XmStringTableUnparse", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XmStringTableUnparse", "XmParseModel");
  type1 = (XmTextType)XEN_TO_C_INT(arg4);
  if (type1 > 1) XEN_ASSERT_TYPE(0, arg4, 4, "XmStringTableUnparse", "XmTextType");
  type2 = (XmTextType)XEN_TO_C_INT(arg5);
  if (type2 > 1) XEN_ASSERT_TYPE(0, arg5, 5, "XmStringTableUnparse", "XmTextType");
  len = XEN_TO_C_INT(arg2);
  if (len <= 0) return(XEN_FALSE);
  loc = xm_protect(lst);
  tb = XEN_TO_C_XmStringTable(arg1, len);
  if (XEN_XmParseTable_P(arg6)) pt = XEN_TO_C_XmParseTable(arg6, len);
  tab = (char **)XmStringTableUnparse(tb,
				      len, 
				      (XEN_FALSE_P(arg3)) ? NULL : XEN_TO_C_STRING(arg3), 
				      type1,
				      type2,
				      pt,
				      XEN_TO_C_INT(arg7), 
				      (XmParseModel)XEN_TO_C_INT(arg8));
  FREE(tb);
  for (i = len - 1; i >= 0; i--)
    {
      lst = XEN_CONS(C_TO_XEN_STRING(tab[i]), lst);
      free(tab[i]);
    }
  free(tab);
  if (pt) free(pt);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XmStringTableToXmString(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmStringTableToXmString "XmString XmStringTableToXmString(XmStringTable table, Cardinal count, XmString break_component) \
converts a compound string table to a single compound string"
  /* DIFF: XmStringTableToXmString 1st arg is list of XmStrings
   */
  int count;
  XmStringTable tab;
  XmString val;
  XEN_ASSERT_TYPE(XEN_LIST_P(arg1), arg1, 1, "XmStringTableToXmString", "XmStringTable");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmStringTableToXmString", "int");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg3) || XEN_FALSE_P(arg3), arg3, 3, "XmStringTableToXmString", "XmString");
  count = XEN_TO_C_INT(arg2);
  if (count <= 0) return(XEN_FALSE);
  tab = XEN_TO_C_XmStringTable(arg1, count);
  val = XmStringTableToXmString(tab,
				count,
				(XEN_XmString_P(arg3)) ? XEN_TO_C_XmString(arg3) : NULL);
  FREE(tab);
  return(C_TO_XEN_XmString(val));
}

static XEN gxm_XmStringToXmStringTable(XEN arg1, XEN arg2)
{
  #define H_XmStringToXmStringTable "Cardinal XmStringToXmStringTable(XmString string, XmString break_component) \
converts a single compound string to a table of compound strings"
  /* DIFF: XmStringToXmStringTable omits and rtns (list len table)
   */
  XmStringTable tab;
  Cardinal val;
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringToXmStringTable", "XmString");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2) || XEN_FALSE_P(arg2), arg2, 2, "XmStringToXmStringTable", "XmString");
  val = XmStringToXmStringTable(XEN_TO_C_XmString(arg1), 
				(XEN_XmString_P(arg2)) ? XEN_TO_C_XmString(arg2) : NULL,
				&tab);
  return(XEN_LIST_2(C_TO_XEN_INT(val), C_TO_XEN_XmStringTable(tab, val)));
}

static XEN gxm_XmStringParseText(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XmStringParseText "XmString XmStringParseText(XtPointer text, XtPointer *text_end, XmStringTag tag, XmTextType type, \
XmParseTable parse_table, Cardinal parse_count, XtPointer call_data) converts a character string to a compound string"
  /* DIFF: XmStringParseText arg1 is string, arg2 is int
   */
  int loc, len;
  char *str, *tag = NULL;
  XmTextType type;
  XtPointer *intext = NULL;
  XmParseTable pt = NULL;
  XEN rtn;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmStringParseText", "string");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2) || XEN_FALSE_P(arg2), arg2, 2, "XmStringParseText", "int");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3) || XEN_FALSE_P(arg3), arg3, 3, "XmStringParseText", "XmStringTag");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmStringParseText", "XmTextType");
  XEN_ASSERT_TYPE(XEN_XmParseTable_P(arg5), arg5, 5, "XmStringParseText", "XmParseTable");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XmStringParseText", "int");
  type = (XmTextType)XEN_TO_C_INT(arg4);
  if (type > 1) XEN_ASSERT_TYPE(0, arg4, 4, "XmStringParseText", "XmTextType");
  str = XEN_TO_C_STRING(arg1);
  if (XEN_INTEGER_P(arg2)) 
    {
      loc = XEN_TO_C_INT(arg2);
      intext = (XtPointer *)(str + loc);
    }
  if (XEN_STRING_P(arg3)) tag = XEN_TO_C_STRING(arg3);
  len = XEN_TO_C_INT(arg6);
  if (XEN_XmParseTable_P(arg5)) pt = XEN_TO_C_XmParseTable(arg5, len);
  rtn = C_TO_XEN_XmString(XmStringParseText(str, intext, tag, type, pt, len, (XtPointer)arg7));
  if (pt) free(pt);
  return(xen_return_first(rtn, arg1, arg2));
}

static XEN gxm_XmStringUnparse(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XmStringUnparse "XtPointer XmStringUnparse(XmString string, XmStringTag tag, XmTextType tag_type, XmTextType output_type, \
XmParseTable parse_table, Cardinal parse_count, XmParseModel parse_model) unparses text"
  XmTextType type1, type2;
  XEN rtn;
  char *str;
  XmParseTable pt = NULL;
  int len;
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringUnparse", "XmString");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2) || XEN_FALSE_P(arg2), arg2, 2, "XmStringUnparse", "XmStringTag or #f");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmStringUnparse", "XmTextType");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmStringUnparse", "XmTextType");
  XEN_ASSERT_TYPE(XEN_XmParseTable_P(arg5) || XEN_FALSE_P(arg5), arg5, 5, "XmStringUnparse", "XmParseTable");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XmStringUnparse", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XmStringUnparse", "XmParseModel");
  type1 = (XmTextType)XEN_TO_C_INT(arg3);
  if (type1 > 1) XEN_ASSERT_TYPE(0, arg3, 3, "XmStringUnparse", "XmTextType");
  type2 = (XmTextType)XEN_TO_C_INT(arg4);
  if (type2 > 1) XEN_ASSERT_TYPE(0, arg4, 4, "XmStringUnparse", "XmTextType");
  len = XEN_TO_C_INT(arg6);
  if (XEN_XmParseTable_P(arg5)) pt = XEN_TO_C_XmParseTable(arg5, len);
  str = (char *)XmStringUnparse(XEN_TO_C_XmString(arg1), 
				(XEN_STRING_P(arg2)) ? XEN_TO_C_STRING(arg2) : NULL,
				type1, type2, pt, len,
				(XmParseModel)XEN_TO_C_INT(arg7));
  rtn = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  if (pt) free(pt);
  return(rtn);
}

static XEN gxm_XmStringComponentCreate(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmStringComponentCreate "XmString XmStringComponentCreate(XmStringComponentType c_type, unsigned int length, XtPointer value) \
creates arbitrary components"
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 1, "XmStringComponentCreate", "XmStringComponentType");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XmStringComponentCreate", "unsigned int");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3) || XEN_FALSE_P(arg3), arg3, 3, "XmStringComponentCreate", "XtPointer");
  return(C_TO_XEN_XmString(XmStringComponentCreate(XEN_TO_C_INT(arg1), 
						   XEN_TO_C_ULONG(arg2), 
						   XEN_FALSE_P(arg3) ? NULL : XEN_TO_C_STRING(arg3))));
}

static XEN gxm_XmStringGetNextTriple(XEN arg1)
{
  #define H_XmStringGetNextTriple "XmStringComponentType XmStringGetNextTriple(XmStringContext context) \
returns the type, length, and value of the next component in the compound string"
  /* DIFF: XmStringGetNextTriple omits last 2 args, returns val
   */
  unsigned int len;
  int val;
  XtPointer *ptr = NULL;
  XEN_ASSERT_TYPE(XEN_XmStringContext_P(arg1), arg1, 1, "XmStringGetNextTriple", "XmStringContext");
  val = XmStringGetNextTriple(XEN_TO_C_XmStringContext(arg1), &len, ptr);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_INT((int)len),
		    (val == XmSTRING_COMPONENT_TEXT) ? C_TO_XEN_STRING((char *)(*ptr)) : C_TO_XEN_ULONG((unsigned long)ptr)));
}

static XEN gxm_XmStringPeekNextTriple(XEN arg1)
{
  #define H_XmStringPeekNextTriple "XmStringComponentType XmStringPeekNextTriple(XmStringContext context) returns the \
component type of the next component"
  XEN_ASSERT_TYPE(XEN_XmStringContext_P(arg1), arg1, 1, "XmStringPeekNextTriple", "XmStringContext");
  return(C_TO_XEN_INT(XmStringPeekNextTriple(XEN_TO_C_XmStringContext(arg1))));
}
#endif
/* MOTIF_2 */

static XEN gxm_XmStringDrawUnderline(XEN args)
{
  #define H_XmStringDrawUnderline "void XmStringDrawUnderline(Display *d, Window w, XmRenderTable rendertable, XmString string, GC gc, \
Position x, Position y, Dimension width, unsigned char alignment, unsigned char layout_direction, XRectangle *clip, XmString underline) \
underlines a string drawn in an X Window"
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  arg10 = XEN_LIST_REF(args, 9);
  arg11 = XEN_LIST_REF(args, 10);
  arg12 = XEN_LIST_REF(args, 11);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmStringDrawUnderline", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmStringDrawUnderline", "Window");
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg3), arg3, 3, "XmStringDrawUnderline", "XmFontList");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg4), arg4, 4, "XmStringDrawUnderline", "XmString");
  XEN_ASSERT_TYPE(XEN_GC_P(arg5), arg5, 5, "XmStringDrawUnderline", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XmStringDrawUnderline", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XmStringDrawUnderline", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XmStringDrawUnderline", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg9), arg9, 9, "XmStringDrawUnderline", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg10), arg10, 10, "XmStringDrawUnderline", "unsigned int");
  XEN_ASSERT_TYPE(XEN_XRectangle_P(arg11), arg11, 11, "XmStringDrawUnderline", "XRectangle");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg12), arg12, 12, "XmStringDrawUnderline", "XmString");
  XmStringDrawUnderline(XEN_TO_C_Display(arg1), 
			XEN_TO_C_Window(arg2), 
			XEN_TO_C_XmFontList(arg3), 
			XEN_TO_C_XmString(arg4), 
			XEN_TO_C_GC(arg5), 
			XEN_TO_C_INT(arg6), XEN_TO_C_INT(arg7), XEN_TO_C_INT(arg8), 
			XEN_TO_C_ULONG(arg9), XEN_TO_C_ULONG(arg10), 
			XEN_TO_C_XRectangle(arg11), 
			XEN_TO_C_XmString(arg12));
  return(XEN_FALSE);
}

static XEN gxm_XmStringDrawImage(XEN args)
{
  #define H_XmStringDrawImage "void XmStringDrawImage(Display *d, Window w, XmRenderTable rendertable, XmString string, GC gc, Position x, \
Position y, Dimension width, unsigned char alignment, unsigned char layout_direction, XRectangle *clip) \
draws a compound string in an X Window and creates an image"
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  arg10 = XEN_LIST_REF(args, 9);
  arg11 = XEN_LIST_REF(args, 10);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmStringDrawImage", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmStringDrawImage", "Window");
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg3), arg3, 3, "XmStringDrawImage", "XmFontList");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg4), arg4, 4, "XmStringDrawImage", "XmString");
  XEN_ASSERT_TYPE(XEN_GC_P(arg5), arg5, 5, "XmStringDrawImage", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XmStringDrawImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XmStringDrawImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XmStringDrawImage", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg9), arg9, 9, "XmStringDrawImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg10), arg10, 10, "XmStringDrawImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_XRectangle_P(arg11), arg11, 11, "XmStringDrawImage", "XRectangle*");
  XmStringDrawImage(XEN_TO_C_Display(arg1), 
		    XEN_TO_C_Window(arg2), 
		    XEN_TO_C_XmFontList(arg3), 
		    XEN_TO_C_XmString(arg4), 
		    XEN_TO_C_GC(arg5), 
		    XEN_TO_C_INT(arg6), XEN_TO_C_INT(arg7), XEN_TO_C_INT(arg8), 
		    XEN_TO_C_ULONG(arg9), XEN_TO_C_ULONG(arg10), 
		    XEN_TO_C_XRectangle(arg11));
  return(XEN_FALSE);
}

static XEN gxm_XmStringDraw(XEN args)
{
  #define H_XmStringDraw "void XmStringDraw(Display *d, Window w, XmRenderTable rendertable, XmString string, GC gc, Position x, Position y, \
Dimension width, unsigned char alignment, unsigned char layout_direction, XRectangle *clip) draws a compound \
string in an X window"
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  arg10 = XEN_LIST_REF(args, 9);
  arg11 = XEN_LIST_REF(args, 10);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmStringDraw", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmStringDraw", "Window");
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg3), arg3, 3, "XmStringDraw", "XmFontList");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg4), arg4, 4, "XmStringDraw", "XmString");
  XEN_ASSERT_TYPE(XEN_GC_P(arg5), arg5, 5, "XmStringDraw", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XmStringDraw", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XmStringDraw", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XmStringDraw", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg9), arg9, 9, "XmStringDraw", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg10), arg10, 10, "XmStringDraw", "unsigned int");
  XEN_ASSERT_TYPE(XEN_XRectangle_P(arg11), arg11, 11, "XmStringDraw", "XRectangle*");
  XmStringDraw(XEN_TO_C_Display(arg1), 
	       XEN_TO_C_Window(arg2), 
	       XEN_TO_C_XmFontList(arg3), 
	       XEN_TO_C_XmString(arg4), 
	       XEN_TO_C_GC(arg5), 
	       XEN_TO_C_INT(arg6), XEN_TO_C_INT(arg7), XEN_TO_C_INT(arg8), 
	       XEN_TO_C_ULONG(arg9), XEN_TO_C_ULONG(arg10), 
	       XEN_TO_C_XRectangle(arg11));
  return(XEN_FALSE);
}

static XEN gxm_XmStringLineCount(XEN arg1)
{
  #define H_XmStringLineCount "int XmStringLineCount(XmString string) returns the number of separators plus one \
in the provided compound string"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringLineCount", "XmString");
  return(C_TO_XEN_INT(XmStringLineCount(XEN_TO_C_XmString(arg1))));
}

static XEN gxm_XmStringExtent(XEN arg1, XEN arg2)
{
  #define H_XmStringExtent "void XmStringExtent(XmRenderTable rendertable, XmString string) \
determines the size of the smallest rectangle that will enclose the compound string"
  /* DIFF: XmStringExtent omits and returns the last 2 args
   */
  Dimension w, h;
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg1), arg1, 1, "XmStringExtent", "XmFontList");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmStringExtent", "XmString");
  XmStringExtent(XEN_TO_C_XmFontList(arg1), XEN_TO_C_XmString(arg2), &w, &h);
  return(XEN_LIST_2(C_TO_XEN_Dimension(w),
		    C_TO_XEN_Dimension(h)));
}

static XEN gxm_XmStringHeight(XEN arg1, XEN arg2)
{
  #define H_XmStringHeight "Dimension XmStringHeight(XmRenderTable rendertable, XmString string) returns the line \
height of the given compound string"
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg1), arg1, 1, "XmStringHeight", "XmFontList");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmStringHeight", "XmString");
  return(C_TO_XEN_Dimension(XmStringHeight(XEN_TO_C_XmFontList(arg1), XEN_TO_C_XmString(arg2))));
}

static XEN gxm_XmStringWidth(XEN arg1, XEN arg2)
{
  #define H_XmStringWidth "Dimension XmStringWidth(XmRenderTable rendertable, XmString string) returns \
the width of the widest line in a compound string"
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg1), arg1, 1, "XmStringWidth", "XmFontList");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmStringWidth", "XmString");
  return(C_TO_XEN_Dimension(XmStringWidth(XEN_TO_C_XmFontList(arg1), XEN_TO_C_XmString(arg2))));
}

static XEN gxm_XmStringBaseline(XEN arg1, XEN arg2)
{
  #define H_XmStringBaseline "Dimension XmStringBaseline(XmRenderTable rendertable, XmString string) returns \
the number of pixels between \
the top of the character box and the baseline of the first line of text"
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg1), arg1, 1, "XmStringBaseline", "XmFontList");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmStringBaseline", "XmString");
  return(C_TO_XEN_Dimension(XmStringBaseline(XEN_TO_C_XmFontList(arg1), XEN_TO_C_XmString(arg2))));
}

static XEN gxm_XmStringFree(XEN arg1)
{
  #define H_XmStringFree "void XmStringFree(XmString string) conditionally deallocates memory"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringFree", "XmString");
  XmStringFree(XEN_TO_C_XmString(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmStringHasSubstring(XEN arg1, XEN arg2)
{
  #define H_XmStringHasSubstring "Boolean XmStringHasSubstring(XmString string, XmString substring) indicates \
whether one compound string is contained within another"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringHasSubstring", "XmString");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmStringHasSubstring", "XmString");
  return(C_TO_XEN_BOOLEAN(XmStringHasSubstring(XEN_TO_C_XmString(arg1), XEN_TO_C_XmString(arg2))));
}

#if MOTIF_2
static XEN gxm_XmStringIsVoid(XEN arg1)
{
  #define H_XmStringIsVoid "Boolean XmStringIsVoid(XmString s1) provides information on the existence of non-zero-length text \
components, tab components, or separator components"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringIsVoid", "XmString");
  return(C_TO_XEN_BOOLEAN(XmStringIsVoid(XEN_TO_C_XmString(arg1))));
}
#endif

static XEN gxm_XmStringEmpty(XEN arg1)
{
  #define H_XmStringEmpty "Boolean XmStringEmpty(XmString s1) provides information on the existence of \
non-zero-length text components"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringEmpty", "XmString");
  return(C_TO_XEN_BOOLEAN(XmStringEmpty(XEN_TO_C_XmString(arg1))));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XmStringLength(XEN arg1)
{
  #define H_XmStringLength "int XmStringLength(XmString s1) obtains the length of a compound string"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringLength", "XmString");
  return(C_TO_XEN_INT(XmStringLength(XEN_TO_C_XmString(arg1))));
}
#endif

static XEN gxm_XmStringCompare(XEN arg1, XEN arg2)
{
  #define H_XmStringCompare "Boolean XmStringCompare(XmString s1, XmString s2) compares two strings"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringCompare", "XmString");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmStringCompare", "XmString");
  return(C_TO_XEN_BOOLEAN(XmStringCompare(XEN_TO_C_XmString(arg1), XEN_TO_C_XmString(arg2))));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XmStringByteCompare(XEN arg1, XEN arg2)
{
  #define H_XmStringByteCompare "Boolean XmStringByteCompare(XmString s1, XmString s2) indicates the results of a \
byte-by-byte comparison"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringByteCompare", "XmString");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmStringByteCompare", "XmString");
  return(C_TO_XEN_BOOLEAN(XmStringByteCompare(XEN_TO_C_XmString(arg1), XEN_TO_C_XmString(arg2))));
}

static XEN gxm_XmStringNCopy(XEN arg1, XEN arg2)
{
  #define H_XmStringNCopy "XmString XmStringNCopy(XmString s1, int num_bytes) creates a copy of a compound string"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringNCopy", "XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmStringNCopy", "int");
  return(C_TO_XEN_XmString(XmStringNCopy(XEN_TO_C_XmString(arg1), XEN_TO_C_INT(arg2))));
}
#endif

static XEN gxm_XmStringCopy(XEN arg1)
{
  #define H_XmStringCopy "XmString XmStringCopy(XmString s1)  makes a copy of a string"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringCopy", "XmString");
  return(C_TO_XEN_XmString(XmStringCopy(XEN_TO_C_XmString(arg1))));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XmStringNConcat(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmStringNConcat "XmString XmStringNConcat(XmString s1, XmString s2, int num_bytes) appends a specified \
number of bytes to a compound string"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringNConcat", "XmString");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmStringNConcat", "XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmStringNConcat", "int");
  return(C_TO_XEN_XmString(XmStringNConcat(XEN_TO_C_XmString(arg1), XEN_TO_C_XmString(arg2), XEN_TO_C_INT(arg3))));
}
#endif

#if MOTIF_2
static XEN gxm_XmStringConcatAndFree(XEN arg1, XEN arg2)
{
  #define H_XmStringConcatAndFree "XmString XmStringConcatAndFree(XmString s1, XmString s2) appends one string to \
another and frees the original strings"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringConcatAndFree", "XmString");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmStringConcatAndFree", "XmString");
  return(C_TO_XEN_XmString(XmStringConcatAndFree(XEN_TO_C_XmString(arg1), XEN_TO_C_XmString(arg2))));
}
#endif

static XEN gxm_XmStringConcat(XEN arg1, XEN arg2)
{
  #define H_XmStringConcat "XmString XmStringConcat(XmString s1, XmString s2) appends one string to another"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringConcat", "XmString");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmStringConcat", "XmString");
  return(C_TO_XEN_XmString(XmStringConcat(XEN_TO_C_XmString(arg1), XEN_TO_C_XmString(arg2))));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XmFontListFreeFontContext(XEN arg1)
{
  #define H_XmFontListFreeFontContext "void XmFontListFreeFontContext(XmFontContext context) instructs the toolkit that \
the font list context is no longer needed"
  XEN_ASSERT_TYPE(XEN_XmFontContext_P(arg1), arg1, 1, "XmFontListFreeFontContext", "XmFontContext");
  XmFontListFreeFontContext(XEN_TO_C_XmFontContext(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmFontListGetNextFont(XEN arg1)
{
  #define H_XmFontListGetNextFont "Boolean XmFontListGetNextFont(XmFontContext context) \
allows applications to access the fonts and character sets in a font list"
  /* DIFF: XmFontListGetNextFont omits args 2 3, returns list
   */
  XmStringCharSet s;
  XFontStruct *f;
  int b;
  XEN_ASSERT_TYPE(XEN_XmFontContext_P(arg1), arg1, 1, "XmFontListGetNextFont", "XmFontContext");
  b = XmFontListGetNextFont(XEN_TO_C_XmFontContext(arg1), &s, &f);
  if (b) /* else "s" may be garbage */
    return(XEN_LIST_3(C_TO_XEN_BOOLEAN(b),
		      C_TO_XEN_STRING(s),
		      C_TO_XEN_XFontStruct(f)));
  return(XEN_FALSE);
}

static XEN gxm_XmFontListInitFontContext(XEN arg1)
{
  #define H_XmFontListInitFontContext "Boolean XmFontListInitFontContext(XmFontList fontlist) \
allows applications to access the entries in a font list"
  /* DIFF: XmFontListInitFontContext omits arg1 and rtns it
   */
  XmFontContext fc;
  Boolean b;
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg1), arg1, 1, "XmFontListInitFontContext", "XmFontList");
  b = XmFontListInitFontContext(&fc, XEN_TO_C_XmFontList(arg1));
  if (b)
    return(C_TO_XEN_XmFontContext(fc));
  return(XEN_FALSE);
}

static XEN gxm_XmFontListCopy(XEN arg1)
{
  #define H_XmFontListCopy "XmFontList XmFontListCopy(XmFontList fontlist) copies a font list"
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg1), arg1, 1, "XmFontListCopy", "XmFontList");
  return(C_TO_XEN_XmFontList(XmFontListCopy(XEN_TO_C_XmFontList(arg1))));
}

static XEN gxm_XmFontListAdd(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmFontListAdd "XmFontList XmFontListAdd(XmFontList oldlist, XFontStruct *font, XmStringCharSet charset) \
creates a new font list"
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg1), arg1, 1, "XmFontListAdd", "XmFontList");
  XEN_ASSERT_TYPE(XEN_XFontStruct_P(arg2), arg2, 2, "XmFontListAdd", "XFontStruct*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XmFontListAdd", "XmStringCharSet");
  return(C_TO_XEN_XmFontList(XmFontListAdd(XEN_TO_C_XmFontList(arg1), XEN_TO_C_XFontStruct(arg2), XEN_TO_C_STRING(arg3))));
}

static XEN gxm_XmFontListFree(XEN arg1)
{
  #define H_XmFontListFree "void XmFontListFree(XmFontList list) recovers memory used by a font list"
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg1), arg1, 1, "XmFontListFree", "XmFontList");
  XmFontListFree(XEN_TO_C_XmFontList(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmFontListCreate(XEN arg1, XEN arg2)
{
  #define H_XmFontListCreate "XmFontList XmFontListCreate(XFontStruct *font, XmStringCharSet charset) creates a font list"
  XEN_ASSERT_TYPE(XEN_XFontStruct_P(arg1), arg1, 1, "XmFontListCreate", "XFontStruct*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmFontListCreate", "XmStringCharSet");
  return(C_TO_XEN_XmFontList(XmFontListCreate(XEN_TO_C_XFontStruct(arg1), XEN_TO_C_STRING(arg2))));
}

static XEN gxm_XmFontListEntryLoad(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmFontListEntryLoad "XmFontListEntry XmFontListEntryLoad(Display *display, char *font_name, XmFontType type, char *tag) \
loads a font or creates a font set and creates an accompanying font list entry"

  XmFontType type;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmFontListEntryLoad", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmFontListEntryLoad", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmFontListEntryLoad", "XmFontType");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XmFontListEntryLoad", "char*");
  type = (XmFontType)XEN_TO_C_INT(arg3);
  if (type > 1) XEN_WRONG_TYPE_ARG_ERROR("XmFontListEntryLoad", 3, arg3, "XmFontType");
  return(C_TO_XEN_XmFontListEntry(XmFontListEntryLoad(XEN_TO_C_Display(arg1),
						      XEN_TO_C_STRING(arg2),
						      type,
						      XEN_TO_C_STRING(arg4))));
}

static XEN gxm_XmFontListRemoveEntry(XEN arg1, XEN arg2)
{
  #define H_XmFontListRemoveEntry "XmFontList XmFontListRemoveEntry(XmFontList oldlist, XmFontListEntry entry) removes a \
font list entry from a font list"
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg1), arg1, 1, "XmFontListRemoveEntry", "XmFontList");
  XEN_ASSERT_TYPE(XEN_XmFontListEntry_P(arg2), arg2, 2, "XmFontListRemoveEntry", "XmFontListEntry");
  return(C_TO_XEN_XmFontList(XmFontListRemoveEntry(XEN_TO_C_XmFontList(arg1), XEN_TO_C_XmFontListEntry(arg2))));
}

static XEN gxm_XmFontListNextEntry(XEN arg1)
{
  #define H_XmFontListNextEntry "XmFontListEntry XmFontListNextEntry(XmFontContext context) returns the next entry in a font list"
  XEN_ASSERT_TYPE(XEN_XmFontContext_P(arg1), arg1, 1, "XmFontListNextEntry", "XmFontContext");
  return(C_TO_XEN_XmFontListEntry(XmFontListNextEntry(XEN_TO_C_XmFontContext(arg1))));
}

static XEN gxm_XmFontListAppendEntry(XEN arg1, XEN arg2)
{
  #define H_XmFontListAppendEntry "XmFontList XmFontListAppendEntry(XmFontList oldlist, XmFontListEntry entry) appends an \
entry to a font list"
  XEN_ASSERT_TYPE(XEN_XmFontList_or_XmRenderTable_P(arg1) || XEN_FALSE_P(arg1), arg1, 1, "XmFontListAppendEntry", "XmFontList");
  XEN_ASSERT_TYPE(XEN_XmFontListEntry_P(arg2), arg2, 2, "XmFontListAppendEntry", "XmFontListEntry");
  return(C_TO_XEN_XmFontList(XmFontListAppendEntry((XEN_FALSE_P(arg1)) ? NULL : XEN_TO_C_XmFontList(arg1), 
						   XEN_TO_C_XmFontListEntry(arg2))));
}

static XEN gxm_XmFontListEntryGetTag(XEN arg1)
{
  char *str;
  XEN res;
  #define H_XmFontListEntryGetTag "char* XmFontListEntryGetTag(XmFontListEntry entry) retrieves the tag of a font list entry"
  XEN_ASSERT_TYPE(XEN_XmFontListEntry_P(arg1), arg1, 1, "XmFontListEntryGetTag", "XmFontListEntry");
  str = XmFontListEntryGetTag(XEN_TO_C_XmFontListEntry(arg1));
  res = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  return(res);
}

static XEN gxm_XmFontListEntryGetFont(XEN arg1)
{
  #define H_XmFontListEntryGetFont "XtPointer XmFontListEntryGetFont(XmFontListEntry entry) \
retrieves font information from a font list entry -> fontstruct"
  /* DIFF: XmFontListEntryGetFont omits arg2
   */
  XmFontType f;
  XtPointer ptr;
  XEN_ASSERT_TYPE(XEN_XmFontListEntry_P(arg1), arg1, 1, "XmFontListEntryGetFont", "XmFontListEntry");
  ptr  = XmFontListEntryGetFont(XEN_TO_C_XmFontListEntry(arg1), &f);
  if (f == XmFONT_IS_FONTSET)
    return(C_TO_XEN_XFontSet((XFontSet)ptr)); 
  return(C_TO_XEN_XFontStruct((XFontStruct *)ptr));
}

static XEN gxm_XmFontListEntryFree(XEN arg1)
{
  #define H_XmFontListEntryFree "void XmFontListEntryFree(XmFontListEntry *entry) recovers memory used by a font list entry"
  /* DIFF: XmFontListEntryFree takes FontListEntry (not ptr to it)
   */
  XmFontListEntry f;
  XEN_ASSERT_TYPE(XEN_XmFontListEntry_P(arg1), arg1, 1, "XmFontListEntryFree", "XmFontListEntry");
  f = XEN_TO_C_XmFontListEntry(arg1);
  XmFontListEntryFree(&f);
  return(XEN_FALSE);
}

static XEN gxm_XmFontListEntryCreate(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmFontListEntryCreate "XmFontListEntry XmFontListEntryCreate(char *tag, XmFontType type, XtPointer font) \
creates a font list entry"
  XmFontType type;
  XtPointer gad;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmFontListEntryCreate", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmFontListEntryCreate", "XmFontType");
  XEN_ASSERT_TYPE(XEN_XFontStruct_P(arg3) || XEN_XFontSet_P(arg3), arg3, 3, "XmFontListEntryCreate", "XFontSet or XFontStruct");
  type = (XmFontType)XEN_TO_C_INT(arg2);
  if (type > 1) XEN_WRONG_TYPE_ARG_ERROR("XmFontListEntryCreate", 2, arg2, "XmFontType");
  if (type == XmFONT_IS_FONTSET)
    gad = (XtPointer)XEN_TO_C_XFontSet(arg3);
  else gad = (XtPointer)XEN_TO_C_XFontStruct(arg3);
  return(C_TO_XEN_XmFontListEntry(XmFontListEntryCreate(XEN_TO_C_STRING(arg1), 
							type,
							gad)));
}

static XEN gxm_XmStringGetLtoR(XEN arg1, XEN arg2)
{
  #define H_XmStringGetLtoR "Boolean XmStringGetLtoR(XmString string, XmStringCharSet tag) searches \
for a text segment in the input compound string"
  /* DIFF: XmStringGetLtoR omits and rtns last arg
   */
  char *str;
  Boolean b;
  XEN val;
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmStringGetLtoR", "XmString");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmStringGetLtoR", "XmStringCharSet");
  b = XmStringGetLtoR(XEN_TO_C_XmString(arg1), XEN_TO_C_STRING(arg2), &str);
  val = C_TO_XEN_STRING(str);
  free(str);
  return(XEN_LIST_2(C_TO_XEN_BOOLEAN(b), val));
}

static XEN gxm_XmStringGetNextSegment(XEN arg1)
{
  #define H_XmStringGetNextSegment "Boolean XmStringGetNextSegment(XmStringContext context) fetches the bytes in the next segment of a compound string"
  /* DIFF: XmStringGetNextSegment omits last args, returns them
   */
  Boolean b; /* returned boolean omitted arg 5 */
  int val;
  unsigned char c;
  XmStringCharSet tag;
  char *text;
  XEN_ASSERT_TYPE(XEN_XmStringContext_P(arg1), arg1, 1, "XmStringGetNextSegment", "XmStringContext");
  val = XmStringGetNextSegment(XEN_TO_C_XmStringContext(arg1), 
			       &text, &tag, &c, &b);
  return(XEN_LIST_5(C_TO_XEN_BOOLEAN(val),
		    C_TO_XEN_STRING(text),
		    C_TO_XEN_STRING(tag),
		    C_TO_XEN_INT((int)c),
		    C_TO_XEN_BOOLEAN(b)));
}

static XEN gxm_XmStringPeekNextComponent(XEN arg1)
{
  #define H_XmStringPeekNextComponent "XmStringComponentType XmStringPeekNextComponent(XmStringContext context) \
returns the component type of the next component to be fetched"
  XEN_ASSERT_TYPE(XEN_XmStringContext_P(arg1), arg1, 1, "XmStringPeekNextComponent", "XmStringContext");
  return(C_TO_XEN_INT(XmStringPeekNextComponent(XEN_TO_C_XmStringContext(arg1))));
}

static XEN gxm_XmStringGetNextComponent(XEN arg1)
{
  #define H_XmStringGetNextComponent "XmStringComponentType XmStringGetNextComponent(XmStringContext context) \
returns the type and value of the next component in a compound string (list val text tag direction component len value)"
  /* DIFF: XmStringGetNextComponent omits all but 1st arg, returns list
   */
  unsigned char direction = 0;
  unsigned short len;
  unsigned char *value;
  unsigned char component;
  XmStringCharSet tag;
  char *text = NULL;
  int val;
  XEN xtext = XEN_FALSE, xdir = XEN_FALSE, xtag = XEN_FALSE;
  XEN_ASSERT_TYPE(XEN_XmStringContext_P(arg1), arg1, 1, "XmStringGetNextComponent", "XmStringContext");
  val = XmStringGetNextComponent(XEN_TO_C_XmStringContext(arg1), &text, &tag, &direction, &component, &len, &value);
  if ((val == XmSTRING_COMPONENT_TEXT) || (val == XmSTRING_COMPONENT_LOCALE_TEXT))
    xtext = C_TO_XEN_STRING(text);
  else
    {
      if (val == XmSTRING_COMPONENT_DIRECTION)
	xdir = C_TO_XEN_INT((int)direction);
      else 
	{
#if MOTIF_2
	  if ((val == XmSTRING_COMPONENT_FONTLIST_ELEMENT_TAG) || (val == XmSTRING_COMPONENT_CHARSET))
#else
	  if (val == XmSTRING_COMPONENT_CHARSET)
#endif
	    xtag = C_TO_XEN_STRING(tag);
	}
    }
  return(XEN_LIST_7(C_TO_XEN_INT(val),
		    xtext,
		    xtag,
		    xdir,
		    C_TO_XEN_INT((int)component),
		    C_TO_XEN_INT((int)len),
		    C_TO_XEN_STRING((const char *)value)));
}
#endif

static XEN gxm_XmStringFreeContext(XEN arg1)
{
  #define H_XmStringFreeContext "void XmStringFreeContext(XmStringContext context) releases the string scanning \
context data structure"
  XEN_ASSERT_TYPE(XEN_XmStringContext_P(arg1), arg1, 1, "XmStringFreeContext", "XmStringContext");
  XmStringFreeContext(XEN_TO_C_XmStringContext(arg1));
  return(XEN_FALSE);
}

static XmStringContext gxm_initxmsc;
static XEN gxm_XmStringInitContext(XEN arg2)
{
  #define H_XmStringInitContext "Boolean XmStringInitContext(XmString string) creates \
a data structure for scanning an XmString component by component"
  /* DIFF XmStringInitContext 1st arg omitted and rtn
   */
  int val;
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 1, "XmStringInitContext", "XmString");
  val = XmStringInitContext(&gxm_initxmsc, XEN_TO_C_XmString(arg2));
  return(XEN_LIST_2(C_TO_XEN_BOOLEAN(val), C_TO_XEN_XmStringContext(gxm_initxmsc)));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XmStringCreateLtoR(XEN arg1, XEN arg2)
{
  #define H_XmStringCreateLtoR "XmString XmStringCreateLtoR(char *text, char *tag) creates a compound string"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmStringCreateLtoR", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmStringCreateLtoR", "XmStringCharSet");
  return(C_TO_XEN_XmString(XmStringCreateLtoR(XEN_TO_C_STRING(arg1), XEN_TO_C_STRING(arg2))));
}

static XEN gxm_XmStringSegmentCreate(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmStringSegmentCreate "XmString XmStringSegmentCreate(char *text, XmStringTag tag, XmStringDirection direction, Boolean separator) \
creates a compound string"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmStringSegmentCreate", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmStringSegmentCreate", "XmStringCharSet");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmStringSegmentCreate", "int");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg4), arg4, 4, "XmStringSegmentCreate", "boolean");
  return(C_TO_XEN_XmString(XmStringSegmentCreate(XEN_TO_C_STRING(arg1), 
						 XEN_TO_C_STRING(arg2), 
						 XEN_TO_C_INT(arg3), 
						 XEN_TO_C_BOOLEAN(arg4))));
}
#endif

static XEN gxm_XmStringSeparatorCreate(void)
{
  #define H_XmStringSeparatorCreate "XmString XmStringSeparatorCreate(void) creates a compound string"
  return(C_TO_XEN_XmString(XmStringSeparatorCreate()));
}

static XEN gxm_XmStringDirectionCreate(XEN arg1)
{
  #define H_XmStringDirectionCreate "XmString XmStringDirectionCreate(XmStringDirection direction) creates a compound string"
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 1, "XmStringDirectionCreate", "int");
  return(C_TO_XEN_XmString(XmStringDirectionCreate(XEN_TO_C_INT(arg1))));
}

static XEN gxm_XmStringCreateLocalized(XEN arg1)
{
  #define H_XmStringCreateLocalized "XmString XmStringCreateLocalized(char *text) creates a compound string in the current locale"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmStringCreateLocalized", "String");
  return(C_TO_XEN_XmString(XmStringCreateLocalized(XEN_TO_C_STRING(arg1))));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XmStringCreateSimple(XEN arg1)
{
  #define H_XmStringCreateSimple "XmString XmStringCreateSimple(char *text) creates a compound string in the \
language environment of a widget"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmStringCreateSimple", "char*");
  return(C_TO_XEN_XmString(XmStringCreateSimple(XEN_TO_C_STRING(arg1))));
}
#endif

static XEN gxm_XmStringCreate(XEN arg1, XEN arg2)
{
  #define H_XmStringCreate "XmString XmStringCreate(char *text, char *tag) creates a compound string"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmStringCreate", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmStringCreate", "XmStringCharSet");
  return(C_TO_XEN_XmString(XmStringCreate(XEN_TO_C_STRING(arg1), XEN_TO_C_STRING(arg2))));
}

static XEN gxm_XmChangeColor(XEN arg1, XEN arg2)
{
  #define H_XmChangeColor "void XmChangeColor(Widget widget, Pixel background) recalculates all associated colors of a widget"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmChangeColor", "Widget");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg2), arg2, 2, "XmChangeColor", "Pixel");
  XmChangeColor(XEN_TO_C_Widget(arg1), XEN_TO_C_Pixel(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmGetColors(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmGetColors "void XmGetColors(Screen *screen, Colormap colormap, Pixel background) generates foreground, select, and shadow colors"
  /* DIFF: XmGetColors omits trailing 4 args and returns them
   */
  Pixel fg,ts, bs, sr;
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XmGetColors", "Screen*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XmGetColors", "Colormap");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg3), arg3, 3, "XmGetColors", "Pixel");
  XmGetColors(XEN_TO_C_Screen(arg1), 
	      XEN_TO_C_Colormap(arg2), 
	      XEN_TO_C_Pixel(arg3),
	      &fg, &ts, &bs, &sr);
  return(XEN_LIST_4(C_TO_XEN_Pixel(fg),
		    C_TO_XEN_Pixel(ts),
		    C_TO_XEN_Pixel(bs),
		    C_TO_XEN_Pixel(sr)));
}

static XEN gxm_XmGetColorCalculation(void)
{
  #define H_XmGetColorCalculation "XmColorProc XmGetColorCalculation(void) get the procedure used for default color calculation"
  return(xm_XmColorProc);
}

static XEN gxm_XmSetColorCalculation(XEN arg1)
{
  #define H_XmSetColorCalculation "XmColorProc XmSetColorCalculation(XmColorProc color_proc) set the procedure used for default color calculation"
  /* DIFF: XmSetColorCalculation NULL -> #f
   */
  if (XEN_PROCEDURE_P(xm_XmColorProc)) xm_unprotect(xm_XmColorProc);
  if (XEN_FALSE_P(arg1))
    {
      xm_XmColorProc = XEN_FALSE;
      XmSetColorCalculation(NULL);
    }
  else
    {
      XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg1) && (XEN_REQUIRED_ARGS_OK(arg1, 1)), arg1, 1, "XmSetColorCalculation", "(XmColorProc but 1 arg)");
      xm_protect(arg1);
      xm_XmColorProc = arg1;
      XmSetColorCalculation((XmColorProc)gxm_XmColorProc);
    }
  return(arg1);
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XmTrackingLocate(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmTrackingLocate "Widget XmTrackingLocate(Widget widget, Cursor cursor, Boolean confine_to) allows for modal \
selection of a component (obsolete)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmTrackingLocate", "Widget");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg2), arg2, 2, "XmTrackingLocate", "Cursor");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XmTrackingLocate", "boolean");
  return(C_TO_XEN_Widget(XmTrackingLocate(XEN_TO_C_Widget(arg1), XEN_TO_C_ULONG(arg2), XEN_TO_C_BOOLEAN(arg3))));
}
#endif

static XEN gxm_XmTrackingEvent(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmTrackingEvent "Widget XmTrackingEvent(Widget widget, Cursor cursor, Boolean confine_to) -> (widget event)"
  /* DIFF: XmTrackingEvent widget cursor confine [event] -> (list widget event)
   */
  XEvent *e; /* do we need to allocate? */
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmTrackingEvent", "Widget");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg2), arg2, 2, "XmTrackingEvent", "Cursor");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XmTrackingEvent", "boolean");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  w = XmTrackingEvent(XEN_TO_C_Widget(arg1), XEN_TO_C_ULONG(arg2), XEN_TO_C_BOOLEAN(arg3), e);
  return(XEN_LIST_2(C_TO_XEN_Widget(w), C_TO_XEN_XEvent_OBJ(e)));
}


/* The various XmVaCreateSimple{object} require special arg handling --
     these are not "normal" resource lists, but can also contain special indicators
     -- not immediately clear how to handle these things!
     Apparently we need to run through the args, split out the special args,
     collect arglists, add to arglist buttons, buttonCount etc, then call the
     underlying simple creator. (see lesstif lib/Xm/VaSimple.c)

   rowcol = XmVaCreateSimpleCheckBox(toplevel, "checkBox", cb,
				     XmNspacing, 2,
				     XmNmarginHeight, 4,
				     XmVaCHECKBUTTON, s1, 0, NULL, NULL,
				     XmVaCHECKBUTTON, s2, 0, NULL, NULL,
				     NULL);

   option = XmVaCreateSimpleOptionMenu(form, "option", cs("MyLabel"), 
				      (KeySym) 'M', 1, SimpleCallbackProc,
				      XmVaPUSHBUTTON, cs("Red"), (int) 'R', 
				      "", XmNULL,
				      XmVaCASCADEBUTTON, cs("Help"), (int) 'H',
				      XmVaSEPARATOR,
				      XmVaPUSHBUTTON, cs("Green"), (int) 'G', 
				      "", XmNULL,
				      XmVaSEPARATOR,
				      XmVaPUSHBUTTON, cs("Blue"), (int) 'B', 
				      "", XmNULL,
				      NULL);				    
*/

static XEN gxm_XmVaCreateSimpleCheckBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmVaCreateSimpleCheckBox "Widget XmVaCreateSimpleCheckBox(Widget parent, String name, XtCallbackProc callback, args)"
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmVaCreateSimpleCheckBox", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmVaCreateSimpleCheckBox", "String");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg3) && (XEN_REQUIRED_ARGS_OK(arg3, 3)), arg3, 3, "XmVaCreateSimpleCheckBox", "XtCallbackProc (3 args)");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XmVaCreateSimpleCheckBox", "List");
  arg4 = XEN_CONS_2(C_TO_XEN_STRING(XmNvalueChangedCallback),
		    XEN_LIST_2(arg3, XEN_FALSE), /* XtCallbackList technically */
		    arg4);
  {
    int arglen;
    Arg *args;
    args = XEN_TO_C_Args(arg4);
    arglen = XEN_LIST_LENGTH(arg4) / 2;
    w = XmCreateSimpleCheckBox(XEN_TO_C_Widget(arg1), 
			       XEN_TO_C_STRING(arg2), 
			       args,
			       arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));

}

static XEN gxm_XmVaCreateSimpleRadioBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmVaCreateSimpleRadioBox "Widget XmVaCreateSimpleRadioBox(Widget parent, String name, int button_set, XtCallbackProc callback, args)"
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmVaCreateSimpleRadioBox", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmVaCreateSimpleRadioBox", "String");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmVaCreateSimpleRadioBox", "int");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 3)), arg4, 4, "XmVaCreateSimpleRadioBox", "XtCallbackProc (3 args");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg5), arg5, 5, "XmVaCreateSimpleRadioBox", "List");
  arg5 = XEN_CONS_2(C_TO_XEN_STRING(XmNvalueChangedCallback),
		    XEN_LIST_2(arg4, XEN_FALSE),
		    arg5);
  /* can't handle button set here since the menuHistory resource is the widget allocated later */
  {
    int arglen;
    Arg *args;
    args = XEN_TO_C_Args(arg5);
    arglen = XEN_LIST_LENGTH(arg5) / 2;
    w = XmCreateSimpleRadioBox(XEN_TO_C_Widget(arg1), 
			       XEN_TO_C_STRING(arg2), 
			       args,
			       arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XmVaCreateSimpleOptionMenu(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XmVaCreateSimpleOptionMenu "Widget XmVaCreateSimpleOptionMenu(Widget parent, String name, XmString option_label, \
KeySym option_mnemonic, int button_set, XtCallbackProc callback, args)"
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmVaCreateSimpleOptionMenu", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmVaCreateSimpleOptionMenu", "String");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg3), arg3, 3, "XmVaCreateSimpleOptionMenu", "XmString");
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg4), arg4, 4, "XmVaCreateSimpleOptionMenu", "KeySym");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XmVaCreateSimpleOptionMenu", "int");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg6) && (XEN_REQUIRED_ARGS_OK(arg6, 3)), arg6, 6, "XmVaCreateSimpleOptionMenu", "XtCallbackProc (3 args)");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg7), arg7, 7, "XmVaCreateSimpleOptionMenu", "List");
  arg7 = XEN_CONS_2(C_TO_XEN_STRING(XmNlabelString),
		    arg3,
		    XEN_CONS_2(C_TO_XEN_STRING(XmNmnemonic),
			       arg4,
			       XEN_CONS_2(C_TO_XEN_STRING(XmNvalueChangedCallback),
					  XEN_LIST_2(arg6, XEN_FALSE),
					  arg7)));
  {
    int arglen;
    Arg *args;
    args = XEN_TO_C_Args(arg7);
    arglen = XEN_LIST_LENGTH(arg7) / 2;
    w = XmCreateSimpleOptionMenu(XEN_TO_C_Widget(arg1), 
				 XEN_TO_C_STRING(arg2), 
				 args,
				 arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XmVaCreateSimplePulldownMenu(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmVaCreateSimplePulldownMenu "Widget XmVaCreateSimplePulldownMenu(Widget parent, String name, int post_from_button, XtCallbackProc callback, args)"
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmVaCreateSimplePulldownMenu", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmVaCreateSimplePulldownMenu", "String");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmVaCreateSimplePulldownMenu", "int");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 3)), arg4, 4, "XmVaCreateSimplePulldownMenu", "XtCallbackProc (3 args)");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg5), arg5, 5, "XmVaCreateSimplePulldownMenu", "List");
  arg5 = XEN_CONS_2(C_TO_XEN_STRING(XmNvalueChangedCallback),
		    XEN_LIST_2(arg4, XEN_FALSE),
		    arg5);
  {
    int arglen;
    Arg *args;
    args = XEN_TO_C_Args(arg5);
    arglen = XEN_LIST_LENGTH(arg5) / 2;
    w = XmCreateSimplePulldownMenu(XEN_TO_C_Widget(arg1), 
				   XEN_TO_C_STRING(arg2), 
				   args,
				   arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XmVaCreateSimplePopupMenu(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmVaCreateSimplePopupMenu "Widget XmVaCreateSimplePopupMenu(Widget parent, String name, XtCallbackProc callback, args)"
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmVaCreateSimplePopupMenu", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmVaCreateSimplePopupMenu", "String");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg3) && (XEN_REQUIRED_ARGS_OK(arg3, 3)), arg3, 3, "XmVaCreateSimplePopupMenu", "XtCallbackProc (3 args)");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XmVaCreateSimplePopupMenu", "List");
  arg4 = XEN_CONS_2(C_TO_XEN_STRING(XmNvalueChangedCallback),
		    XEN_LIST_2(arg3, XEN_FALSE),
		    arg4);
  {
    int arglen;
    Arg *args;
    args = XEN_TO_C_Args(arg4);
    arglen = XEN_LIST_LENGTH(arg4) / 2;
    w = XmCreateSimplePopupMenu(XEN_TO_C_Widget(arg1), 
				XEN_TO_C_STRING(arg2), 
				args,
				arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XmVaCreateSimpleMenuBar(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmVaCreateSimpleMenuBar "Widget XmVaCreateSimpleMenuBar(Widget parent, String name, args)"
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmVaCreateSimpleMenuBar", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmVaCreateSimpleMenuBar", "String");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XmVaCreateSimpleMenuBar", "List");
  {
    int arglen;
    Arg *args;
    args = XEN_TO_C_Args(arg3);
    arglen = XEN_LIST_LENGTH(arg3) / 2;
    w = XmVaCreateSimpleMenuBar(XEN_TO_C_Widget(arg1), 
				XEN_TO_C_STRING(arg2),
				args,
				arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XmCreateSimpleCheckBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSimpleCheckBox "Widget XmCreateSimpleCheckBox(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimpleCheckBox", XmCreateSimpleCheckBox, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateSimpleRadioBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSimpleRadioBox "Widget XmCreateSimpleRadioBox(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimpleRadioBox", XmCreateSimpleRadioBox, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateSimpleOptionMenu(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSimpleOptionMenu "Widget XmCreateSimpleOptionMenu(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimpleOptionMenu", XmCreateSimpleOptionMenu, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateSimplePulldownMenu(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSimplePulldownMenu "Widget XmCreateSimplePulldownMenu(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimplePulldownMenu", XmCreateSimplePulldownMenu, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateSimplePopupMenu(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSimplePopupMenu "Widget XmCreateSimplePopupMenu(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimplePopupMenu", XmCreateSimplePopupMenu, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateSimpleMenuBar(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSimpleMenuBar "Widget XmCreateSimpleMenuBar(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimpleMenuBar", XmCreateSimpleMenuBar, arg1, arg2, arg3, arg4));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XmGetMenuCursor(XEN arg1)
{
  #define H_XmGetMenuCursor "Cursor XmGetMenuCursor(Display *display) returns the cursor ID for the current menu cursor"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmGetMenuCursor", "Display*");
  return(C_TO_XEN_Cursor(XmGetMenuCursor(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XmSetMenuCursor(XEN arg1, XEN arg2)
{
  #define H_XmSetMenuCursor "void XmSetMenuCursor(Display *display, Cursor cursorId) modifies the menu cursor for a client"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmSetMenuCursor", "Display*");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg2), arg2, 2, "XmSetMenuCursor", "Cursor");
  XmSetMenuCursor(XEN_TO_C_Display(arg1), XEN_TO_C_Cursor(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmSetFontUnit(XEN arg1, XEN arg2)
{
  #define H_XmSetFontUnit "void XmSetFontUnit(Display *display, int font_unit_value) sets the font unit value for a display"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmSetFontUnit", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmSetFontUnit", "int");
  XmSetFontUnit(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmSetFontUnits(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmSetFontUnits "void XmSetFontUnits(Display *display, int h_value, int v_value) sets the font unit value for a display"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmSetFontUnits", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmSetFontUnits", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmSetFontUnits", "int");
  XmSetFontUnits(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3));
  return(XEN_FALSE);
}
#endif

static XEN gxm_XmConvertUnits(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmConvertUnits "int XmConvertUnits(Widget widget, int orientation, int from_unit_type, int from_value, int to_unit_type) \
converts a value in one unit type to another unit type"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmConvertUnits", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmConvertUnits", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmConvertUnits", "register");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmConvertUnits", "register");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XmConvertUnits", "register");
  return(C_TO_XEN_INT(XmConvertUnits(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5))));
}

#if MOTIF_2
static XEN gxm_XmConvertStringToUnits(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmConvertStringToUnits "int XmConvertStringToUnits(Screen *screen, String spec, int orientation, int to_type, XtEnum *parse_error) \
converts a string specification to a unit value"
  /* DIFF: XmConvertStringToUnits scr spec orient type [err] -> #f (err) or val
   */
  XtEnum err = 0;
  int val;
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XmConvertStringToUnits", "Screen*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmConvertStringToUnits", "String");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmConvertStringToUnits", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmConvertStringToUnits", "int");
  val = XmConvertStringToUnits(XEN_TO_C_Screen(arg1), XEN_TO_C_STRING(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), &err);
  if (err == 0)
    return(C_TO_XEN_INT(val));
  return(XEN_FALSE);
}
#endif

static XEN gxm_XmCvtXmStringToCT(XEN arg1)
{
  #define H_XmCvtXmStringToCT "char *XmCvtXmStringToCT(XmString string) converts a compound string to compound text"
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 1, "XmCvtXmStringToCT", "XmString");
  return(C_TO_XEN_STRING(XmCvtXmStringToCT(XEN_TO_C_XmString(arg1))));
}

static XEN gxm_XmCvtCTToXmString(XEN arg1)
{
  #define H_XmCvtCTToXmString "XmString XmCvtCTToXmString(char *text) converts compound text to a compound string"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmCvtCTToXmString", "char*");
  return(C_TO_XEN_XmString(XmCvtCTToXmString(XEN_TO_C_STRING(arg1))));
}

static XEN gxm_XmMapSegmentEncoding(XEN arg1)
{
  char *str;
  XEN res;
  #define H_XmMapSegmentEncoding "char *XmMapSegmentEncoding(char *fontlist_tag) returns the compound text \
encoding format associated with the specified font list tag"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmMapSegmentEncoding", "char*");
  str = XmMapSegmentEncoding(XEN_TO_C_STRING(arg1));
  res = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  return(res);
}

static XEN gxm_XmRegisterSegmentEncoding(XEN arg1, XEN arg2)
{
  char *str;
  XEN res;
  #define H_XmRegisterSegmentEncoding "char *XmRegisterSegmentEncoding(char *fontlist_tag, char *ct_encoding) \
registers a compound text encoding format for a specified font list element tag"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XmRegisterSegmentEncoding", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmRegisterSegmentEncoding", "char*");
  str = XmRegisterSegmentEncoding(XEN_TO_C_STRING(arg1), XEN_TO_C_STRING(arg2));
  res = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  return(res);
}

static XEN gxm_XmWidgetGetBaselines(XEN arg1)
{
  #define H_XmWidgetGetBaselines "Boolean XmWidgetGetBaselines(Widget widget) retrieves baseline information for a widget"
  /* DIFF: XmWidgetGetBaselines omits args 2 and 3, returns list of Dimensions
   */
  Dimension *ds;
  int len, i, loc;
  Boolean b;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmWidgetGetBaselines", "Widget");
  b = XmWidgetGetBaselines(XEN_TO_C_Widget(arg1), &ds, &len);
  if (!b)
    return(XEN_FALSE);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_Dimension(ds[i]), lst);
  free(ds);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XmWidgetGetDisplayRect(XEN arg1)
{
  #define H_XmWidgetGetDisplayRect "Boolean XmWidgetGetDisplayRect(Widget widget) returns widget's bounding box as XRectangle"
  XRectangle *r;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmWidgetGetDisplayRect", "Widget");
  r = (XRectangle *)CALLOC(1, sizeof(XRectangle));
  if (!(XmWidgetGetDisplayRect(XEN_TO_C_Widget(arg1), r)))
    {
      FREE(r);
      return(XEN_FALSE);
    }
  return(C_TO_XEN_XRectangle(r));
}

#if MOTIF_2
static XEN gxm_XmObjectAtPoint(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmObjectAtPoint "Widget XmObjectAtPoint(Widget widget, Position x, Position y) determines which child \
intersects or comes closest to a specified point"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmObjectAtPoint", "Widget");
  XEN_ASSERT_TYPE(XEN_Position_P(arg2), arg2, 2, "XmObjectAtPoint", "Position");
  XEN_ASSERT_TYPE(XEN_Position_P(arg3), arg3, 3, "XmObjectAtPoint", "Position");
  return(C_TO_XEN_Widget(XmObjectAtPoint(XEN_TO_C_Widget(arg1), XEN_TO_C_Position(arg2), XEN_TO_C_Position(arg3))));
}
#endif

static XEN gxm_XmUpdateDisplay(XEN arg1)
{
  #define H_XmUpdateDisplay "void XmUpdateDisplay (widget) processes all pending exposure events immediately"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmUpdateDisplay", "Widget");
  XmUpdateDisplay(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmDestroyPixmap(XEN arg1, XEN arg2)
{
  #define H_XmDestroyPixmap "Boolean XmDestroyPixmap(Screen *screen, Pixmap pixmap) removes a pixmap from the pixmap cache"
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XmDestroyPixmap", "Screen*");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg2), arg2, 2, "XmDestroyPixmap", "Pixmap");
  return(C_TO_XEN_BOOLEAN(XmDestroyPixmap(XEN_TO_C_Screen(arg1), 
					  XEN_TO_C_Pixmap(arg2))));
}

static XEN gxm_XmGetPixmapByDepth(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmGetPixmapByDepth "Pixmap XmGetPixmapByDepth(Screen *screen, char *image_name, Pixel foreground, Pixel background, int depth) \
generates a pixmap, stores it in a pixmap cache, and returns the pixmap"
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XmGetPixmapByDepth", "Screen*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmGetPixmapByDepth", "char*");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg3), arg3, 3, "XmGetPixmapByDepth", "Pixel");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg4), arg4, 4, "XmGetPixmapByDepth", "Pixel");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XmGetPixmapByDepth", "int");
  return(C_TO_XEN_Pixmap(XmGetPixmapByDepth(XEN_TO_C_Screen(arg1), 
					    XEN_TO_C_STRING(arg2), 
					    XEN_TO_C_Pixel(arg3), 
					    XEN_TO_C_Pixel(arg4), 
					    XEN_TO_C_INT(arg5))));
}

static XEN gxm_XmGetPixmap(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmGetPixmap "Pixmap XmGetPixmap(Screen *screen, char *image_name, Pixel foreground, Pixel background) A pixmap caching function \
that generates a pixmap, stores it in a pixmap cache, and returns the pixmap"
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XmGetPixmap", "Screen*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmGetPixmap", "char*");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg3), arg3, 3, "XmGetPixmap", "Pixel");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg4), arg4, 4, "XmGetPixmap", "Pixel");
  return(C_TO_XEN_Pixmap(XmGetPixmap(XEN_TO_C_Screen(arg1), 
				     XEN_TO_C_STRING(arg2), 
				     XEN_TO_C_Pixel(arg3), 
				     XEN_TO_C_Pixel(arg4))));
}

static XEN gxm_XmUninstallImage(XEN arg1)
{
  #define H_XmUninstallImage "Boolean XmUninstallImage(XImage *image) removes an image from the image cache"
  XEN_ASSERT_TYPE(XEN_XImage_P(arg1), arg1, 1, "XmUninstallImage", "XImage*");
  return(C_TO_XEN_BOOLEAN(XmUninstallImage(XEN_TO_C_XImage(arg1))));
}

static XEN gxm_XmInstallImage(XEN arg1, XEN arg2)
{
  #define H_XmInstallImage "Boolean XmInstallImage(XImage *image, char *image_name) adds an image to the image cache"
  XEN_ASSERT_TYPE(XEN_XImage_P(arg1), arg1, 1, "XmInstallImage", "XImage*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmInstallImage", "char*");
  return(C_TO_XEN_BOOLEAN(XmInstallImage(XEN_TO_C_XImage(arg1), 
					 XEN_TO_C_STRING(arg2))));
}

static XEN gxm_XmCreateMainWindow(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateMainWindow "Widget XmCreateMainWindow(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MainWindow widget creation function"
  return(gxm_new_widget("XmCreateMainWindow", XmCreateMainWindow, arg1, arg2, arg3, arg4));
}

#if (!XM_DISABLE_DEPRECATED)
#define XEN_MainWindowWidget_P(Arg) (XEN_Widget_P(Arg) && XmIsMainWindow(XEN_TO_C_Widget(Arg)))

static XEN gxm_XmMainWindowSep3(XEN arg1)
{
  #define H_XmMainWindowSep3 "Widget XmMainWindowSep3(Widget widget) returns the widget ID of the third Separator widget"
  XEN_ASSERT_TYPE(XEN_MainWindowWidget_P(arg1), arg1, 1, "XmMainWindowSep3", "MainWindow Widget");
  return(C_TO_XEN_Widget(XmMainWindowSep3(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmMainWindowSep2(XEN arg1)
{
  #define H_XmMainWindowSep2 "Widget XmMainWindowSep2(Widget widget) the widget ID of the second Separator widget"
  XEN_ASSERT_TYPE(XEN_MainWindowWidget_P(arg1), arg1, 1, "XmMainWindowSep2", "MainWindow Widget");
  return(C_TO_XEN_Widget(XmMainWindowSep2(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmMainWindowSep1(XEN arg1)
{
  #define H_XmMainWindowSep1 "Widget XmMainWindowSep1(Widget widget) returns the widget ID of the first Separator"
  XEN_ASSERT_TYPE(XEN_MainWindowWidget_P(arg1), arg1, 1, "XmMainWindowSep1", "MainWindow Widget");
  return(C_TO_XEN_Widget(XmMainWindowSep1(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmMainWindowSetAreas(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XmMainWindowSetAreas "void XmMainWindowSetAreas(Widget widget, Widget menu_bar, Widget command_window, Widget horizontal_scrollbar, \
Widget vertical_scrollbar, Widget work_region) identifies manageable children for each area"
  XEN_ASSERT_TYPE(XEN_MainWindowWidget_P(arg1), arg1, 1, "XmMainWindowSetAreas", "MainWindow Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2) || XEN_BOOLEAN_P(arg2), arg2, 2, "XmMainWindowSetAreas", "Widget or #f=NULL");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg3) || XEN_BOOLEAN_P(arg3), arg3, 3, "XmMainWindowSetAreas", "Widget or #f=NULL");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg4) || XEN_BOOLEAN_P(arg4), arg4, 4, "XmMainWindowSetAreas", "Widget or #f=NULL");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg5) || XEN_BOOLEAN_P(arg5), arg5, 5, "XmMainWindowSetAreas", "Widget or #f=NULL");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg6) || XEN_BOOLEAN_P(arg6), arg6, 6, "XmMainWindowSetAreas", "Widget or #f=NULL");
  XmMainWindowSetAreas(XEN_TO_C_Widget(arg1),
		       (XEN_Widget_P(arg2) ? XEN_TO_C_Widget(arg2) : NULL),
		       (XEN_Widget_P(arg3) ? XEN_TO_C_Widget(arg3) : NULL),
		       (XEN_Widget_P(arg4) ? XEN_TO_C_Widget(arg4) : NULL),
		       (XEN_Widget_P(arg5) ? XEN_TO_C_Widget(arg5) : NULL),
		       (XEN_Widget_P(arg6) ? XEN_TO_C_Widget(arg6) : NULL));
  return(XEN_FALSE);
}
#endif

static XEN gxm_XmTranslateKey(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmTranslateKey "void XmTranslateKey(Display *display, KeyCode keycode, Modifiers modifiers) \
The default keycode-to-keysym translator -> (modifiers keysym)"
  /* DIFF: XmTranslateKey omit and rtn last 2 args
   */
  Modifiers m;
  KeySym k;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmTranslateKey", "Display*");
  XEN_ASSERT_TYPE(XEN_KeyCode_P(arg2), arg2, 2, "XmTranslateKey", "KeyCode");
  XEN_ASSERT_TYPE(XEN_Modifiers_P(arg3), arg3, 3, "XmTranslateKey", "Modifiers");
  XmTranslateKey(XEN_TO_C_Display(arg1), XEN_TO_C_KeyCode(arg2), XEN_TO_C_Modifiers(arg3), &m, &k);
  return(XEN_LIST_2(C_TO_XEN_Modifiers(m),
		    C_TO_XEN_KeySym(k)));
}

static XEN gxm_XmCreateScrolledList(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateScrolledList "Widget XmCreateScrolledList(Widget parent, String name, ArgList arglist, Cardinal argcount) The List \
ScrolledList creation function"
  return(gxm_new_widget("XmCreateScrolledList", XmCreateScrolledList, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateList(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateList "Widget XmCreateList(Widget parent, String name, ArgList arglist, Cardinal argcount) The List widget creation function"
  return(gxm_new_widget("XmCreateList", XmCreateList, arg1, arg2, arg3, arg4));
}

#define XEN_ListWidget_P(Arg) (XEN_Widget_P(Arg) && XmIsList(XEN_TO_C_Widget(Arg)))

static XEN gxm_XmListPosSelected(XEN arg1, XEN arg2)
{
  #define H_XmListPosSelected "Boolean XmListPosSelected(Widget widget, int position) determines if the list item at a \
specified position is selected"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListPosSelected", "List Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmListPosSelected", "int");
  return(C_TO_XEN_BOOLEAN(XmListPosSelected(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XmListUpdateSelectedList(XEN arg1)
{
  #define H_XmListUpdateSelectedList "void XmListUpdateSelectedList(Widget widget) updates the XmNselectedItems resource"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListUpdateSelectedList", "List Widget");
  XmListUpdateSelectedList(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmListSetHorizPos(XEN arg1, XEN arg2)
{
  #define H_XmListSetHorizPos "void XmListSetHorizPos(Widget widget, int position) scrolls to the specified position in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListSetHorizPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmListSetHorizPos", "int");
  XmListSetHorizPos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmListGetSelectedPos(XEN arg1)
{
  #define H_XmListGetSelectedPos "Boolean XmListGetSelectedPos(Widget widget) A List function that returns the position of every selected item in the list"
  /* DIFF: XmListGetSelectedPos omits args 2 and 3, returns list of positions
   */
  int *ps;
  int i, len, loc;
  Boolean b;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListGetSelectedPos", "List Widget");
  b = XmListGetSelectedPos(XEN_TO_C_Widget(arg1), &ps, &len);
  if (!b)
    return(XEN_FALSE);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_INT(ps[i]), lst);
  free(ps);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XmListGetMatchPos(XEN arg1, XEN arg2)
{
  #define H_XmListGetMatchPos "Boolean XmListGetMatchPos(Widget widget, XmString item) returns all instances of an item in the list"
  /* DIFF: XmListGetSelectedPos omits args 3 and 4, returns list of positions
   */
  int *ps;
  int i, len, loc;
  Boolean b;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListGetMatchPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmListGetMatchPos", "XmString");
  b = XmListGetMatchPos(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2), &ps, &len);
  if (!b)
    return(XEN_FALSE);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_INT(ps[i]), lst);
  free(ps);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XmListPosToBounds(XEN arg1, XEN arg2)
{
  #define H_XmListPosToBounds "Boolean XmListPosToBounds(Widget widget, int position) returns the bounding box of an item at a specified position in a list"
  /* DIFF: XmListPosToBounds last 4 args omitted and returned 
   */
  Position x, y;
  Dimension w, h;
  int val;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListPosToBounds", "List Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmListPosToBounds", "int");
  val = XmListPosToBounds(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), &x, &y, &w, &h);
  return(XEN_LIST_5(C_TO_XEN_BOOLEAN(val),
		    C_TO_XEN_Position(x),
		    C_TO_XEN_Position(y),
		    C_TO_XEN_Dimension(w),
		    C_TO_XEN_Dimension(h)));
}


static XEN gxm_XmListYToPos(XEN arg1, XEN arg2)
{
  #define H_XmListYToPos "int XmListYToPos(Widget widget, Position y) returns the position of the item at a specified y-coordinate"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListYToPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_Position_P(arg2), arg2, 2, "XmListYToPos", "Position");
  return(C_TO_XEN_INT(XmListYToPos(XEN_TO_C_Widget(arg1), XEN_TO_C_Position(arg2))));
}

static XEN gxm_XmListSetKbdItemPos(XEN arg1, XEN arg2)
{
  #define H_XmListSetKbdItemPos "Boolean XmListSetKbdItemPos(Widget widget, int position) sets the location cursor at a specified position"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListSetKbdItemPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmListSetKbdItemPos", "int");
  return(C_TO_XEN_BOOLEAN(XmListSetKbdItemPos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XmListGetKbdItemPos(XEN arg1)
{
  #define H_XmListGetKbdItemPos "int XmListGetKbdItemPos(Widget widget) returns the position of the item at the location cursor"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListGetKbdItemPos", "List Widget");
  return(C_TO_XEN_INT(XmListGetKbdItemPos(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmListItemPos(XEN arg1, XEN arg2)
{
  #define H_XmListItemPos "int XmListItemPos(Widget widget, XmString item) returns the position of an item in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListItemPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmListItemPos", "XmString");
  return(C_TO_XEN_INT(XmListItemPos(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2))));
}

static XEN gxm_XmListItemExists(XEN arg1, XEN arg2)
{
  #define H_XmListItemExists "Boolean XmListItemExists(Widget widget, XmString item) checks if a specified item is in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListItemExists", "List Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmListItemExists", "XmString");
  return(C_TO_XEN_BOOLEAN(XmListItemExists(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2))));
}

static XEN gxm_XmListSetAddMode(XEN arg1, XEN arg2)
{
  #define H_XmListSetAddMode "void XmListSetAddMode(Widget widget, Boolean state) sets add mode in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListSetAddMode", "List Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XmListSetAddMode", "boolean");
  XmListSetAddMode(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmListSetBottomItem(XEN arg1, XEN arg2)
{
  #define H_XmListSetBottomItem "void XmListSetBottomItem(Widget widget, XmString item) makes an existing item the last \
visible item in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListSetBottomItem", "List Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmListSetBottomItem", "XmString");
  XmListSetBottomItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmListSetItem(XEN arg1, XEN arg2)
{
  #define H_XmListSetItem "void XmListSetItem(Widget widget, XmString item) makes an existing item the first visible item in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListSetItem", "List Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmListSetItem", "XmString");
  XmListSetItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmListSetBottomPos(XEN arg1, XEN arg2)
{
  #define H_XmListSetBottomPos "void XmListSetBottomPos(Widget widget, int position) makes a specified item the last visible item in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListSetBottomPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmListSetBottomPos", "int");
  XmListSetBottomPos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmListSetPos(XEN arg1, XEN arg2)
{
  #define H_XmListSetPos "void XmListSetPos(Widget widget, int position) makes the item at the given position the first \
visible position in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListSetPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmListSetPos", "int");
  XmListSetPos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmListDeselectAllItems(XEN arg1)
{
  #define H_XmListDeselectAllItems "void XmListDeselectAllItems(Widget widget) unhighlights and removes all items from the selected list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListDeselectAllItems", "List Widget");
  XmListDeselectAllItems(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmListDeselectPos(XEN arg1, XEN arg2)
{
  #define H_XmListDeselectPos "void XmListDeselectPos(Widget widget, int position) deselects an item at a specified position in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListDeselectPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmListDeselectPos", "int");
  XmListDeselectPos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmListDeselectItem(XEN arg1, XEN arg2)
{
  #define H_XmListDeselectItem "void XmListDeselectItem(Widget widget, XmString item) deselects the specified item from the selected list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListDeselectItem", "List Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmListDeselectItem", "XmString");
  XmListDeselectItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmListSelectPos(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmListSelectPos "void XmListSelectPos(Widget widget, int position, Boolean notify) selects an item at a specified \
position in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListSelectPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmListSelectPos", "int");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XmListSelectPos", "boolean");
  XmListSelectPos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_BOOLEAN(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmListSelectItem(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmListSelectItem "void XmListSelectItem(Widget widget, XmString item, Boolean notify) selects an item in the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListSelectItem", "List Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmListSelectItem", "XmString");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XmListSelectItem", "boolean");
  XmListSelectItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2), XEN_TO_C_BOOLEAN(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmListReplacePositions(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmListReplacePositions "void XmListReplacePositions(Widget widget, int *position_list, XmString *item_list, int item_count) \
replaces items in a list based on position"
  /* DIFF: XmListReplacePositions arg 2 is list of ints, arg3 is list of XmStrings
   */
  int *ps;
  XmString *str;
  int len;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListReplacePositions", "List Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmListReplacePositions", "list of int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XmListReplacePositions", "list of XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg4), arg4, 4, "XmListReplacePositions", "int");
  if (XEN_INTEGER_P(arg4)) len = XEN_TO_C_INT(arg4); else len = XEN_LIST_LENGTH(arg3);
  if (len <= 0) return(XEN_FALSE);
  ps = XEN_TO_C_Ints(arg2, len);
  str = XEN_TO_C_XmStrings(arg3, len);
  XmListReplacePositions(XEN_TO_C_Widget(arg1), ps, str, len);
  FREE(ps);
  FREE(str);
  return(XEN_FALSE);
}

static XEN gxm_XmListReplaceItemsPosUnselected(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmListReplaceItemsPosUnselected "void XmListReplaceItemsPosUnselected(Widget widget, XmString *new_items, int item_count, int position) \
replaces items in a list without selecting the replacement items"
  /* DIFF: XmListReplaceItemsPosUnselected arg 2 is list of XmStrings
   */
  XmString *str;
  int len;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListReplaceItemsPosUnselected", "List Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmListReplaceItemsPosUnselected", "list of XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmListReplaceItemsPosUnselected", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmListReplaceItemsPosUnselected", "int");
  len = XEN_TO_C_INT(arg3);
  if (len <= 0) return(XEN_FALSE);
  str = XEN_TO_C_XmStrings(arg2, len);
  XmListReplaceItemsPosUnselected(XEN_TO_C_Widget(arg1), str, len, XEN_TO_C_INT(arg4));
  FREE(str);
  return(XEN_FALSE);
}

static XEN gxm_XmListReplaceItemsUnselected(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmListReplaceItemsUnselected "void XmListReplaceItemsUnselected(Widget widget, XmString *old_items, int item_count, XmString *new_items) \
replaces items in a list"
  /* DIFF: XmListReplaceItemsUnselected args 2 and 4 are lists of XmStrings
   */
  XmString *str1, *str2;
  int len;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListReplaceItemsUnselected", "List Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmListReplaceItemsUnselected", "list of XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmListReplaceItemsUnselected", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XmListReplaceItemsUnselected", "list of XmString");
  len = XEN_TO_C_INT(arg3);
  if (len <= 0) return(XEN_FALSE);
  str1 = XEN_TO_C_XmStrings(arg2, len);
  str2 = XEN_TO_C_XmStrings(arg4, len);
  XmListReplaceItemsUnselected(XEN_TO_C_Widget(arg1), str1, len, str2);
  FREE(str1);
  FREE(str2);
  return(XEN_FALSE);
}

static XEN gxm_XmListReplaceItemsPos(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmListReplaceItemsPos "void XmListReplaceItemsPos(Widget widget, XmString *new_items, int item_count, int position) \
replaces the specified elements in the list"
  /* DIFF: XmListReplaceItemsPos arg 2 is list of XmStrings
   */
  XmString *str;
  int len;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListReplaceItemsPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmListReplaceItemsPos", "list of XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmListReplaceItemsPos", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmListReplaceItemsPos", "int");
  len = XEN_TO_C_INT(arg3);
  if (len <= 0) return(XEN_FALSE);
  str = XEN_TO_C_XmStrings(arg2, len);
  XmListReplaceItemsPos(XEN_TO_C_Widget(arg1), str, len, XEN_TO_C_INT(arg4));
  FREE(str);
  return(XEN_FALSE);
}

static XEN gxm_XmListReplaceItems(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmListReplaceItems "void XmListReplaceItems(Widget widget, XmString *old_items, int item_count, XmString *new_items) \
replaces the specified elements in the list"
  /* DIFF: XmListReplaceItems args 2 and 4 are lists of XmStrings
   */
  XmString *str1, *str2;
  int len;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListReplaceItems", "List Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmListReplaceItems", "list of XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmListReplaceItems", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XmListReplaceItems", "XmString*");
  len = XEN_TO_C_INT(arg3);
  if (len <= 0) return(XEN_FALSE);
  str1 = XEN_TO_C_XmStrings(arg2, len);
  str2 = XEN_TO_C_XmStrings(arg4, len);
  XmListReplaceItems(XEN_TO_C_Widget(arg1), str1, len, str2);
  FREE(str1);
  FREE(str2);
  return(XEN_FALSE);
}

static XEN gxm_XmListDeleteAllItems(XEN arg1)
{
  #define H_XmListDeleteAllItems "void XmListDeleteAllItems(Widget widget) deletes all items from the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListDeleteAllItems", "List Widget");
  XmListDeleteAllItems(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmListDeleteItemsPos(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmListDeleteItemsPos "void XmListDeleteItemsPos(Widget widget, int item_count, int position) deletes \
items from the list starting at the given position"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListDeleteItemsPos", "List Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmListDeleteItemsPos", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmListDeleteItemsPos", "int");
  XmListDeleteItemsPos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmListDeletePos(XEN arg1, XEN arg2)
{
  #define H_XmListDeletePos "void XmListDeletePos(Widget widget, int position) deletes an item from a list at a specified position"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListDeletePos", "List Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmListDeletePos", "int");
  XmListDeletePos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmListDeletePositions(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmListDeletePositions "void XmListDeletePositions(Widget widget, int *position_list, int position_count) deletes \
items from a list based on an array of positions"
  /* DIFF: XmListDeletePositions arg2 is list of ints
   */
  int *pos;
  int len;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListDeletePositions", "List Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmListDeletePositions", "list of int");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmListDeletePositions", "int");
  if (XEN_INTEGER_P(arg3)) len = XEN_TO_C_INT(arg3); else len = XEN_LIST_LENGTH(arg2);
  if (len <= 0) return(XEN_FALSE);
  pos = XEN_TO_C_Ints(arg2, len);
  XmListDeletePositions(XEN_TO_C_Widget(arg1), pos, len);
  FREE(pos);
  return(XEN_FALSE);
}

static XEN gxm_XmListDeleteItems(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmListDeleteItems "void XmListDeleteItems(Widget widget, XmString *items, int item_count) deletes items from the list"
  /* DIFF: XmListDeleteItems arg 2 is list of XmStrings
   */
  XmString *str;
  int len;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListDeleteItems", "List Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmListDeleteItems", "list of XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmListDeleteItems", "int");
  if (XEN_INTEGER_P(arg3)) len = XEN_TO_C_INT(arg3); else len = XEN_LIST_LENGTH(arg2);
  if (len <= 0) return(XEN_FALSE);
  str = XEN_TO_C_XmStrings(arg2, len);
  XmListDeleteItems(XEN_TO_C_Widget(arg1), str, len);
  FREE(str);
  return(XEN_FALSE);
}

static XEN gxm_XmListDeleteItem(XEN arg1, XEN arg2)
{
  #define H_XmListDeleteItem "void XmListDeleteItem(Widget widget, XmString item) deletes an item from the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListDeleteItem", "List Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmListDeleteItem", "XmString");
  XmListDeleteItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmListAddItemUnselected(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmListAddItemUnselected "void XmListAddItemUnselected(Widget widget, XmString item, int position) adds an item to the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListAddItemUnselected", "List Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmListAddItemUnselected", "XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmListAddItemUnselected", "int");
  XmListAddItemUnselected(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2), XEN_TO_C_INT(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmListAddItemsUnselected(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmListAddItemsUnselected "void XmListAddItemsUnselected(Widget widget, XmString *items, int item_count, int position) \
adds items to a list"
  /* DIFF: XmListAddItemsUnselected arg 2 is list of XmStrings
   */
  XmString *str;
  int len;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListAddItemsUnselected", "List Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmListAddItemsUnselected", "list of XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmListAddItemsUnselected", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmListAddItemsUnselected", "int");
  len = XEN_TO_C_INT(arg3);
  if (len <= 0) return(XEN_FALSE);
  str = XEN_TO_C_XmStrings(arg2, len);
  XmListAddItemsUnselected(XEN_TO_C_Widget(arg1), str, len, XEN_TO_C_INT(arg4));
  FREE(str);
  return(XEN_FALSE);
}

static XEN gxm_XmListAddItems(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmListAddItems "void XmListAddItems(Widget widget, XmString *items, int item_count, int position) adds items to the list"
  /* DIFF: XmListAddItems arg 2 is list of XmStrings
   */
  XmString *str;
  int len;
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListAddItems", "List Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmListAddItems", "XmString*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmListAddItems", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmListAddItems", "int");
  len = XEN_TO_C_INT(arg3);
  if (len <= 0) return(XEN_FALSE);
  str = XEN_TO_C_XmStrings(arg2, len);
  XmListAddItems(XEN_TO_C_Widget(arg1), str, len, XEN_TO_C_INT(arg4));
  FREE(str);
  return(XEN_FALSE);
}

static XEN gxm_XmListAddItem(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmListAddItem "void XmListAddItem(Widget widget, XmString item, int position) adds an item to the list"
  XEN_ASSERT_TYPE(XEN_ListWidget_P(arg1), arg1, 1, "XmListAddItem", "List Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmListAddItem", "XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmListAddItem", "int");
  XmListAddItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2), XEN_TO_C_INT(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmIsMotifWMRunning(XEN arg1)
{
  #define H_XmIsMotifWMRunning "Boolean XmIsMotifWMRunning(Widget shell)  determines whether the window manager is running"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmIsMotifWMRunning", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsMotifWMRunning(XEN_TO_C_Widget(arg1))));
}

/* DIFF: all XmCreate<Obj> arglist is a list of args 
 */
static XEN gxm_XmCreateLabel(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateLabel "Widget XmCreateLabel(Widget parent, String name, ArgList arglist, Cardinal argcount) The Label widget creation function"
  return(gxm_new_widget("XmCreateLabel", XmCreateLabel, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateLabelGadget(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateLabelGadget "Widget XmCreateLabelGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) The LabelGadget creation function"
  return(gxm_new_widget("XmCreateLabelGadget", XmCreateLabelGadget, arg1, arg2, arg3, arg4));
}

#if HAVE_XmToolTipGetLabel
static XEN gxm_XmToolTipGetLabel(XEN arg1)
{
  #define H_XmToolTipGetLabel "Widget XmToolTipGetLabel(Widget wid) apparently returns the tooltip label associated with its argument"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, XEN_ONLY_ARG, "XmToolTipGetLabel", "Widget");
  return(C_TO_XEN_Widget(XmToolTipGetLabel(XEN_TO_C_Widget(arg1))));
}
#endif

#if HAVE_XmCreateMultiList

#ifndef XmIsMultiList
#define XmIsMultiList(w) (XtIsSubclass(w, xmMultiListWidgetClass))
#endif

static XEN gxm_XmCreateMultiList(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateMultiList "Widget XmCreateMultiList(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MultiList widget creation function"
  /* see comment under XmCreateDropDown */
#ifdef _XmExt18List_h
  return(gxm_new_widget("XmCreateMultiList", XmCreateExtended18List, arg1, arg2, arg3, arg4));
#else
  return(gxm_new_widget("XmCreateMultiList", XmCreateMultiList, arg1, arg2, arg3, arg4));
#endif
}

static XEN gxm_XmIsMultiList(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsMultiList", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsMultiList(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmMultiListUnselectAllItems(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmMultiListUnselectAllItems", "Widget");
  XmMultiListUnselectAllItems(XEN_TO_C_Widget(arg));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListDeselectAllItems(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmMultiListDeselectAllItems", "Widget");
  XmMultiListDeselectAllItems(XEN_TO_C_Widget(arg));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListSelectAllItems(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmMultiListSelectAllItems", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg1), arg1, 2, "XmMultiListSelectAllItems", "boolean");
  XmMultiListSelectAllItems(XEN_TO_C_Widget(arg), XEN_TO_C_BOOLEAN(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListDeselectRow(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmMultiListDeselectRow", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 2, "XmMultiListDeselectRow", "int");
  XmMultiListDeselectRow(XEN_TO_C_Widget(arg), XEN_TO_C_INT(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListToggleRow(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmMultiListToggleRow", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 2, "XmMultiListToggleRow", "int");
  XmMultiListToggleRow(XEN_TO_C_Widget(arg), (short)XEN_TO_C_INT(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListMakeRowVisible(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmMultiListMakeRowVisible", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 2, "XmMultiListMakeRowVisible", "int");
  XmMultiListMakeRowVisible(XEN_TO_C_Widget(arg), XEN_TO_C_INT(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListSelectRow(XEN arg, XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmMultiListSelectRow", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 2, "XmMultiListSelectRow", "int");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 3, "XmMultiListSelectRow", "boolean");
  XmMultiListSelectRow(XEN_TO_C_Widget(arg), XEN_TO_C_INT(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListUnselectItem(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmMultiListUnselectItem", "Widget");
  XEN_ASSERT_TYPE(XEN_XmMultiListRowInfo_P(arg1), arg1, 2, "XmMultiListUnselectItem", "XmMultiListRowInfo*");
  XmMultiListUnselectItem(XEN_TO_C_Widget(arg), XEN_TO_C_XmMultiListRowInfo(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListDeselectItem(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmMultiListDeselectItem", "Widget");
  XEN_ASSERT_TYPE(XEN_XmMultiListRowInfo_P(arg1), arg1, 2, "XmMultiListDeselectItem", "XmMultiListRowInfo*");
  XmMultiListDeselectItem(XEN_TO_C_Widget(arg), XEN_TO_C_XmMultiListRowInfo(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListDeselectItems(XEN arg, XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmMultiListDeselectItems", "Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 2, "XmMultiListDeselectItems", "XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 3, "XmMultiListDeselectItems", "int");
  XmMultiListDeselectItems(XEN_TO_C_Widget(arg), XEN_TO_C_XmString(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListSelectItems(XEN arg, XEN arg1, XEN arg2, XEN arg3)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmMultiListSelectItems", "Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg1), arg1, 2, "XmMultiListSelectItems", "XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 3, "XmMultiListSelectItems", "int");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 4, "XmMultiListSelectItems", "boolean");
  XmMultiListSelectItems(XEN_TO_C_Widget(arg), XEN_TO_C_XmString(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_BOOLEAN(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmMultiListGetSelectedRows(XEN arg)
{
  XEN lst = XEN_EMPTY_LIST;
  int i;
  XmMultiListRowInfo **rows;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmMultiListGetSelectedRows", "Widget");
  rows = XmMultiListGetSelectedRows(XEN_TO_C_Widget(arg));
  if (rows)
    {
      int len;
      for (len = 0; (rows[len]) ;len++);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_XmMultiListRowInfo(rows[i]), lst);
    }
  /* free rows? */
  return(lst);
}
#endif

#if HAVE_XmCreateTabStack

#ifndef XmIsTabStack
#define XmIsTabStack(w) (XtIsSubclass(w, xmTabStackWidgetClass))
#endif

static XEN gxm_XmCreateTabStack(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateTabStack "Widget XmCreateTabStack(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The TabStack widget creation function"
  return(gxm_new_widget("XmCreateTabStack", XmCreateTabStack, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmIsTabStack(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsTabStack", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsTabStack(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmTabStackGetSelectedTab(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmTabStackGetSelectedTab", "Widget");
  return(C_TO_XEN_Widget(XmTabStackGetSelectedTab(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmTabStackSelectTab(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmTabStackSelectTab", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg1), arg1, 2, "XmTabStackSelectTab", "boolean");
  XmTabStackSelectTab(XEN_TO_C_Widget(arg), XEN_TO_C_BOOLEAN(arg1));
  return(XEN_FALSE);
}

#if HAVE_XmTabStackXYToWidget
static XEN gxm_XmTabStackIndexToWidget(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmTabStackIndexToWidget", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 2, "XmTabStackIndexToWidget", "int");
  return(XEN_TO_C_Widget(XmTabStackIndexToWidget(XEN_TO_C_Widget(arg), XEN_TO_C_INT(arg1))));
}

static XEN gxm_XmTabStackXYToWidget(XEN arg, XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmTabStackXYToWidget", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 2, "XmTabStackXYToWidget", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 3, "XmTabStackXYToWidget", "int");
  return(XEN_TO_C_Widget(XmTabStackXYToWidget(XEN_TO_C_Widget(arg), XEN_TO_C_INT(arg1), XEN_TO_C_INT(arg2))));
}
#endif
#endif

#if HAVE_XmCreateDataField

#ifndef XmIsDataField
#define XmIsDataField(w) (XtIsSubclass(w, xmDataFieldWidgetClass))
#endif

static XEN gxm_XmCreateDataField(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateDataField "Widget XmCreateDataField(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The DataField widget creation function"
  return(gxm_new_widget("XmCreateDataField", XmCreateDataField, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmIsDataField(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsDataField", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsDataField(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmDataFieldGetString(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmDataFieldGetString", "Widget");
  return(C_TO_XEN_STRING(XmDataFieldGetString(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmDataFieldGetSelection(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmDataFieldGetSelection", "Widget");
  return(C_TO_XEN_STRING(XmDataFieldGetSelection(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmDataFieldPaste(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmDataFieldPaste", "Widget");
  return(C_TO_XEN_BOOLEAN(XmDataFieldPaste(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmDataFieldCut(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmDataFieldCut", "Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg1), arg1, 2, "XmDataFieldCut", "Time");
  return(C_TO_XEN_BOOLEAN(XmDataFieldCut(XEN_TO_C_Widget(arg), XEN_TO_C_Time(arg1))));
}

static XEN gxm_XmDataFieldCopy(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmDataFieldCopy", "Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg1), arg1, 2, "XmDataFieldCopy", "Time");
  return(C_TO_XEN_BOOLEAN(XmDataFieldCopy(XEN_TO_C_Widget(arg), XEN_TO_C_Time(arg1))));
}

static XEN gxm_XmDataFieldSetString(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmDataFieldSetString", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 2, "XmDataFieldSetString", "char*");
  XmDataFieldSetString(XEN_TO_C_Widget(arg), XEN_TO_C_STRING(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmDataFieldSetEditable(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmDataFieldSetEditable", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg1), arg1, 2, "XmDataFieldSetEditable", "boolean");
  XmDataFieldSetEditable(XEN_TO_C_Widget(arg), XEN_TO_C_BOOLEAN(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmDataFieldSetAddMode(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmDataFieldSetAddMode", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg1), arg1, 2, "XmDataFieldSetAddMode", "boolean");
  XmDataFieldSetAddMode(XEN_TO_C_Widget(arg), XEN_TO_C_BOOLEAN(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmDataFieldSetInsertionPosition(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmDataFieldSetInsertionPosition", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 2, "XmDataFieldSetInsertionPosition", "int");
  XmDataFieldSetInsertionPosition(XEN_TO_C_Widget(arg), (XmTextPosition)XEN_TO_C_INT(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmDataFieldShowPosition(XEN arg, XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmDataFieldShowPosition", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 2, "XmDataFieldShowPosition", "int");
  XmDataFieldShowPosition(XEN_TO_C_Widget(arg), (XmTextPosition)XEN_TO_C_INT(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmDataFieldXYToPos(XEN arg, XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmDataFieldXYToPos", "Widget");
  XEN_ASSERT_TYPE(XEN_Position_P(arg1), arg1, 2, "XmDataFieldXYToPos", "Position");
  XEN_ASSERT_TYPE(XEN_Position_P(arg2), arg2, 3, "XmDataFieldXYToPos", "Position");
  return(C_TO_XEN_INT((int)(XmDataFieldXYToPos(XEN_TO_C_Widget(arg), XEN_TO_C_Position(arg1), XEN_TO_C_Position(arg2)))));
}

static XEN gxm_XmDataFieldSetHighlight(XEN arg, XEN arg1, XEN arg2, XEN arg3)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmDataFieldSetHighlight", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 2, "XmDataFieldSetHighlight", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 3, "XmDataFieldSetHighlight", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg3, 4, "XmDataFieldSetHighlight", "int (highligh mode)");
  XmDataFieldSetHighlight(XEN_TO_C_Widget(arg), (XmTextPosition)XEN_TO_C_INT(arg1), (XmTextPosition)XEN_TO_C_INT(arg2), 
			  (XmHighlightMode)XEN_TO_C_INT(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmDataFieldSetSelection(XEN arg, XEN arg1, XEN arg2, XEN arg3)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 1, "XmDataFieldSetSelection", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 2, "XmDataFieldSetSelection", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 3, "XmDataFieldSetSelection", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg3, 4, "XmDataFieldSetSelection", "Time");
  XmDataFieldSetSelection(XEN_TO_C_Widget(arg), (XmTextPosition)XEN_TO_C_INT(arg1), (XmTextPosition)XEN_TO_C_INT(arg2), XEN_TO_C_Time(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmDataFieldGetSelectionPosition(XEN arg)
{
  XmTextPosition pos1 = 0, pos2 = 0;
  Boolean val = False;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmDataFieldGetSelectionPosition", "Widget");
  val = XmDataFieldGetSelectionPosition(XEN_TO_C_Widget(arg), &pos1, &pos2);
  return(XEN_LIST_3(C_TO_XEN_BOOLEAN(val),
		    C_TO_XEN_INT((int)pos1),
		    C_TO_XEN_INT((int)pos2)));
}
#endif

#if HAVE_XmCreateButtonBox
static XEN gxm_XmCreateButtonBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateButtonBox "Widget XmCreateButtonBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ButtonBox widget creation function"
  return(gxm_new_widget("XmCreateButtonBox", XmCreateButtonBox, arg1, arg2, arg3, arg4));
}

#ifndef XmIsButtonBox
#define XmIsButtonBox(w) (XtIsSubclass(w, xmButtonBoxWidgetClass))
#endif

static XEN gxm_XmIsButtonBox(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsButtonBox", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsButtonBox(XEN_TO_C_Widget(arg))));
}
#endif

#if HAVE_XmCreateColumn
static XEN gxm_XmCreateColumn(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateColumn "Widget XmCreateColumn(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Column widget creation function"
  return(gxm_new_widget("XmCreateColumn", XmCreateColumn, arg1, arg2, arg3, arg4));
}

#ifndef XmIsColumn
#define XmIsColumn(w) (XtIsSubclass(w, xmColumnWidgetClass))
#endif

static XEN gxm_XmIsColumn(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsColumn", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsColumn(XEN_TO_C_Widget(arg))));
}

#if HAVE_XmColumnGetChildLabel
static XEN gxm_XmColumnGetChildLabel(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmColumnGetChildLabel", "Widget");
  return(C_TO_XEN_Widget(XmColumnGetChildLabel(XEN_TO_C_Widget(arg))));
}
#endif
#endif

#if HAVE_XmCreateDropDown
static XEN gxm_XmCreateDropDown(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateDropDown "Widget XmCreateDropDown(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The DropDown widget creation function"
  /* there is a problem here: the XmCreateDropDown macro in DropDown.h does not give us
     function argument access to the actual function.  Hence a short-term kludge...
  */
#ifdef _XmCominationBox2_h
  /* yes, it actually is spelled that way! */
return(gxm_new_widget("XmCreateDropDown", XmCreateCombinationBox2, arg1, arg2, arg3, arg4));
#else
  return(gxm_new_widget("XmCreateDropDown", XmCreateDropDown, arg1, arg2, arg3, arg4));
#endif
}

#ifndef XmIsDropDown
#define XmIsDropDown(w) (XtIsSubclass(w, xmDropDownWidgetClass))
#endif

static XEN gxm_XmIsDropDown(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsDropDown", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsDropDown(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmDropDownGetValue(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmDropDownGetValue", "Widget");
  return(C_TO_XEN_STRING((char *)XmDropDownGetValue(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmDropDownGetText(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmDropDownGetText", "Widget");
  return(C_TO_XEN_Widget(XmDropDownGetText(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmDropDownGetLabel(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmDropDownGetLabel", "Widget");
  return(C_TO_XEN_Widget(XmDropDownGetLabel(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmDropDownGetArrow(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmDropDownGetArrow", "Widget");
  return(C_TO_XEN_Widget(XmDropDownGetArrow(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmDropDownGetList(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmDropDownGetList", "Widget");
  return(C_TO_XEN_Widget(XmDropDownGetList(XEN_TO_C_Widget(arg))));
}
#endif

#if HAVE_XmCreateFontSelector
static XEN gxm_XmCreateFontSelector(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateFontSelector "Widget XmCreateFontSelector(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The FontSelector widget creation function"
  return(gxm_new_widget("XmCreateFontSelector", XmCreateFontSelector, arg1, arg2, arg3, arg4));
}
#endif

#if HAVE_XmCreateColorSelector
static XEN gxm_XmCreateColorSelector(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateColorSelector "Widget XmCreateColorSelector(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ColorSelector widget creation function"
  return(gxm_new_widget("XmCreateColorSelector", XmCreateColorSelector, arg1, arg2, arg3, arg4));
}
#endif

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmCreateIconHeader(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateIconHeader "Widget XmCreateIconHeader(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateIconHeader", XmCreateIconHeader, arg1, arg2, arg3, arg4));
}
#endif

static XEN gxm_XmCreateIconGadget(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateIconGadget "Widget XmCreateIconGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The IconGadget widget creation function"
  return(gxm_new_widget("XmCreateIconGadget", XmCreateIconGadget, arg1, arg2, arg3, arg4));
}
#endif

static XEN gxm_XmCreateToggleButton(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateToggleButton "Widget XmCreateToggleButton(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ToggleButton widget creation function"
  return(gxm_new_widget("XmCreateToggleButton", XmCreateToggleButton, arg1, arg2, arg3, arg4));
}

#define XEN_ToggleButtonWidget_P(Arg) (XEN_Widget_P(Arg) && (XmIsToggleButton(XEN_TO_C_Widget(Arg)) || XmIsToggleButtonGadget(XEN_TO_C_Widget(Arg))))

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmToggleButtonSetValue(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmToggleButtonSetValue "void XmToggleButtonSetValue(Widget widget, XmToggleButtonState state, Boolean notify) \
sets or changes the current state"
  XEN_ASSERT_TYPE(XEN_ToggleButtonWidget_P(arg1), arg1, 1, "XmToggleButtonSetValue", "ToggleButton Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmToggleButtonSetValue", "int (actually XmToggleButtonState)");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XmToggleButtonSetValue", "boolean");
  return(C_TO_XEN_BOOLEAN(XmToggleButtonSetValue(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_BOOLEAN(arg3))));
}
#endif
#endif

static XEN gxm_XmToggleButtonSetState(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmToggleButtonSetState "void XmToggleButtonSetState(Widget widget, Boolean state, Boolean notify) \
sets or changes the current state"
  XEN_ASSERT_TYPE(XEN_ToggleButtonWidget_P(arg1), arg1, 1, "XmToggleButtonSetState", "ToggleButton Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XmToggleButtonSetState", "boolean");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XmToggleButtonSetState", "boolean");
  XmToggleButtonSetState(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2), XEN_TO_C_BOOLEAN(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmToggleButtonGetState(XEN arg1)
{
  #define H_XmToggleButtonGetState "Boolean XmToggleButtonGetState(Widget widget) obtains the state of a ToggleButton"
  XEN_ASSERT_TYPE(XEN_ToggleButtonWidget_P(arg1), arg1, 1, "XmToggleButtonGetState", "ToggleButton Widget");
  return(C_TO_XEN_BOOLEAN(XmToggleButtonGetState(XEN_TO_C_Widget(arg1))));
}


static XEN gxm_XmCreateToggleButtonGadget(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateToggleButtonGadget "Widget XmCreateToggleButtonGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ToggleButtonGadget creation function"
  return(gxm_new_widget("XmCreateToggleButtonGadget", XmCreateToggleButtonGadget, arg1, arg2, arg3, arg4));
}

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmToggleButtonGadgetSetValue(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmToggleButtonGadgetSetValue "Boolean XmToggleButtonGadgetSetValue(Widget w, XmToggleButtonState newstate, Boolean notify)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmToggleButtonGadgetSetValue", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmToggleButtonGadgetSetValue", "int");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XmToggleButtonGadgetSetValue", "boolean");
  return(C_TO_XEN_BOOLEAN(XmToggleButtonGadgetSetValue(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_BOOLEAN(arg3))));
}
#endif

static XEN gxm_XmCreateGrabShell(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateGrabShell "Widget XmCreateGrabShell(Widget parent, char *name, ArgList al, Cardinal ac) -> a new GrabShell"
  return(gxm_new_widget("XmCreateGrabShell", XmCreateGrabShell, arg1, arg2, arg3, arg4));
}

#endif

static XEN gxm_XmToggleButtonGadgetSetState(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmToggleButtonGadgetSetState "void XmToggleButtonGadgetSetState(Widget widget, Boolean state, Boolean notify) \
sets or changes the current state"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmToggleButtonGadgetSetState", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XmToggleButtonGadgetSetState", "boolean");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XmToggleButtonGadgetSetState", "boolean");
  XmToggleButtonGadgetSetState(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2), XEN_TO_C_BOOLEAN(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmToggleButtonGadgetGetState(XEN arg1)
{
  #define H_XmToggleButtonGadgetGetState "Boolean XmToggleButtonGadgetGetState(Widget widget) obtains \
the state of a ToggleButtonGadget"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmToggleButtonGadgetGetState", "Widget");
  return(C_TO_XEN_BOOLEAN(XmToggleButtonGadgetGetState(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmCreateFrame(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateFrame "Widget XmCreateFrame(Widget parent, String name, ArgList arglist, Cardinal argcount) The Frame widget creation function"
  return(gxm_new_widget("XmCreateFrame", XmCreateFrame, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateFormDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateFormDialog "Widget XmCreateFormDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A Form FormDialog creation function"
  return(gxm_new_widget("XmCreateFormDialog", XmCreateFormDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateForm(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateForm "Widget XmCreateForm(Widget parent, String name, ArgList arglist, Cardinal argcount) The Form widget creation function"
  return(gxm_new_widget("XmCreateForm", XmCreateForm, arg1, arg2, arg3, arg4));
}

#define XEN_TextWidget_P(Arg) (XEN_Widget_P(Arg) && (XmIsText(XEN_TO_C_Widget(Arg)) || XmIsTextField(XEN_TO_C_Widget(Arg))))
#define XEN_JustTextWidget_P(Arg) (XEN_Widget_P(Arg) && XmIsText(XEN_TO_C_Widget(Arg)))

static XEN gxm_XmTextFindString(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmTextFindString "Boolean XmTextFindString(Widget widget, XmTextPosition start, char *string, XmTextDirection direction) \
finds the beginning position of a text string"
  /* DIFF: XmTextFindString widget start string dir [pos] -> pos or #f
  */
  XmTextPosition pos;
  int res;
  XEN_ASSERT_TYPE(XEN_JustTextWidget_P(arg1), arg1, 1, "XmTextFindString", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFindString", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XmTextFindString", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmTextFindString", "XmTextDirection");
  res = XmTextFindString(XEN_TO_C_Widget(arg1), 
			 (XmTextPosition)XEN_TO_C_INT(arg2), 
			 XEN_TO_C_STRING(arg3), 
			 (XmTextDirection)XEN_TO_C_INT(arg4), 
			 &pos);
  if (res)
    return(C_TO_XEN_INT(pos));
  return(XEN_FALSE);
}

static XEN gxm_XmTextEnableRedisplay(XEN arg1)
{
  #define H_XmTextEnableRedisplay "void XmTextEnableRedisplay(Widget widget) forces the visual update of a Text widget"
  XEN_ASSERT_TYPE(XEN_JustTextWidget_P(arg1), arg1, 1, "XmTextEnableRedisplay", "Text or TextField Widget");
  XmTextEnableRedisplay(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmTextDisableRedisplay(XEN arg1)
{
  #define H_XmTextDisableRedisplay "void XmTextDisableRedisplay(Widget widget) temporarily prevents visual update of the Text widget"
  XEN_ASSERT_TYPE(XEN_JustTextWidget_P(arg1), arg1, 1, "XmTextDisableRedisplay", "Text or TextField Widget");
  XmTextDisableRedisplay(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmTextGetCenterline(XEN arg1)
{
  #define H_XmTextGetCenterline "int XmTextGetCenterline(Widget widget) Return the height (length) of a character string when the writing direction is vertical"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetCenterline", "Text or TextField Widget");
  return(C_TO_XEN_INT(XmTextGetCenterline(XEN_TO_C_Widget(arg1))));
}
#endif
#endif

static XEN gxm_XmTextGetBaseline(XEN arg1)
{
  #define H_XmTextGetBaseline "int XmTextGetBaseline(Widget widget) accesses the y position of the baseline"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetBaseline", "Text or TextField Widget");
  return(C_TO_XEN_INT(XmTextGetBaseline(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextScroll(XEN arg1, XEN arg2)
{
  #define H_XmTextScroll "void XmTextScroll(Widget widget, int lines) scrolls text"
  XEN_ASSERT_TYPE(XEN_JustTextWidget_P(arg1), arg1, 1, "XmTextScroll", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextScroll", "int");
  XmTextScroll(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextShowPosition(XEN arg1, XEN arg2)
{
  #define H_XmTextShowPosition "void XmTextShowPosition(Widget widget, XmTextPosition position) forces text at a given position to be displayed"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextShowPosition", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextShowPosition", "XmTextPosition");
  XmTextShowPosition(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextSetSource(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmTextSetSource "void XmTextSetSource(Widget widget, XmTextSource source, XmTextPosition top_character, \
XmTextPosition cursor_position) sets the source of the widget"
  XEN_ASSERT_TYPE(XEN_JustTextWidget_P(arg1), arg1, 1, "XmTextSetSource", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_XmTextSource_P(arg2), arg2, 2, "XmTextSetSource", "XmTextSource");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextSetSource", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmTextSetSource", "XmTextPosition");
  XmTextSetSource(XEN_TO_C_Widget(arg1), XEN_TO_C_XmTextSource(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4));
  return(XEN_FALSE);
}

static XEN gxm_XmTextGetSource(XEN arg1)
{
  #define H_XmTextGetSource "XmTextSource XmTextGetSource(Widget widget) accesses the source of the widget"
  XEN_ASSERT_TYPE(XEN_JustTextWidget_P(arg1), arg1, 1, "XmTextGetSource", "Text or TextField Widget");
  return(C_TO_XEN_XmTextSource(XmTextGetSource(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextPosToXY(XEN arg1, XEN arg2)
{
  #define H_XmTextPosToXY "Boolean XmTextPosToXY(Widget widget, XmTextPosition position) A Text function \
that returns the x and y position of a character position"
  /* DIFF: XmTextPosToXY omits last 2 args and returns them
   */
  Position x, y;
  int val;
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextPosToXY", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextPosToXY", "XmTextPosition");
  val = XmTextPosToXY(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), &x, &y);
  return(XEN_LIST_3(C_TO_XEN_BOOLEAN(val),
		    C_TO_XEN_Position(x),
		    C_TO_XEN_Position(y)));
}

static XEN gxm_XmTextXYToPos(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmTextXYToPos "XmTextPosition XmTextXYToPos(Widget widget, Position x, Position y) accesses \
the character position nearest an x and y position"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextXYToPos", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextXYToPos", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextXYToPos", "int");
  return(C_TO_XEN_INT(XmTextXYToPos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XmTextGetSelectionPosition(XEN arg1)
{
  #define H_XmTextGetSelectionPosition "Boolean XmTextGetSelectionPosition(Widget widget) returns the position of the primary selection"
  /* DIFF: XmTextGetSelectionPosition widget [left right] -> (list res left right)
     no arg2 arg3, returns (owner left right) instead */
  XmTextPosition pos1, pos2;
  int res;
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetSelectionPosition", "Text or TextField Widget");
  res = XmTextGetSelectionPosition(XEN_TO_C_Widget(arg1), &pos1, &pos2);
  return(XEN_LIST_3(C_TO_XEN_BOOLEAN(res),
		    C_TO_XEN_INT(pos1),
		    C_TO_XEN_INT(pos2)));
}

static XEN gxm_XmTextClearSelection(XEN arg1, XEN arg2)
{
  #define H_XmTextClearSelection "void XmTextClearSelection(Widget widget, Time time) clears the primary selection"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextClearSelection", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmTextClearSelection", "Time");
  XmTextClearSelection(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextSetSelection(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmTextSetSelection "void XmTextSetSelection(Widget widget, XmTextPosition first, XmTextPosition last, Time time) \
sets the primary selection of the text"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextSetSelection", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextSetSelection", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextSetSelection", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_Time_P(arg4), arg4, 4, "XmTextSetSelection", "Time");
  XmTextSetSelection(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_Time(arg4));
  return(XEN_FALSE);
}

static XEN gxm_XmTextGetSelection(XEN arg1)
{
  char *str;
  XEN res;
  #define H_XmTextGetSelection "char *XmTextGetSelection(Widget widget) retrieves the value of the primary selection"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetSelection", "Text or TextField Widget");
  str = XmTextGetSelection(XEN_TO_C_Widget(arg1));
  res = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  return(res);
}

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmTextPasteLink(XEN arg1)
{
  #define H_XmTextPasteLink "Boolean XmTextPasteLink(Widget widget) inserts a link to the clipboard selection"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextPasteLink", "Text or TextField Widget");
  return(C_TO_XEN_BOOLEAN(XmTextPasteLink(XEN_TO_C_Widget(arg1))));
}
#endif
#endif

static XEN gxm_XmTextPaste(XEN arg1)
{
  #define H_XmTextPaste "Boolean XmTextPaste(Widget widget) inserts the clipboard selection"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextPaste", "Text or TextField Widget");
  return(C_TO_XEN_BOOLEAN(XmTextPaste(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextCut(XEN arg1, XEN arg2)
{
  #define H_XmTextCut "Boolean XmTextCut(Widget widget, Time time) copies the primary selection to the clipboard and deletes the selected text"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextCut", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmTextCut", "Time");
  return(C_TO_XEN_BOOLEAN(XmTextCut(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2))));
}

#if MOTIF_2
static XEN gxm_XmTextCopyLink(XEN arg1, XEN arg2)
{
  #define H_XmTextCopyLink "Boolean XmTextCopyLink(Widget widget, Time time) copies a link to the primary selection to the clipboard"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextCopyLink", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmTextCopyLink", "Time");
  return(C_TO_XEN_BOOLEAN(XmTextCopyLink(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2))));
}
#endif

static XEN gxm_XmTextCopy(XEN arg1, XEN arg2)
{
  #define H_XmTextCopy "Boolean XmTextCopy(Widget widget, Time time) copies the primary selection to the clipboard"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextCopy", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmTextCopy", "Time");
  return(C_TO_XEN_BOOLEAN(XmTextCopy(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2))));
}

static XEN gxm_XmTextRemove(XEN arg1)
{
  #define H_XmTextRemove "Boolean XmTextRemove(Widget widget) deletes the primary selection"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextRemove", "Text or TextField Widget");
  return(C_TO_XEN_BOOLEAN(XmTextRemove(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextSetCursorPosition(XEN arg1, XEN arg2)
{
  #define H_XmTextSetCursorPosition "void XmTextSetCursorPosition(Widget w, int position) sets the insertion cursor position"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextSetCursorPosition", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextSetCursorPosition", "XmTextPosition");
  XmTextSetCursorPosition(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextSetInsertionPosition(XEN arg1, XEN arg2)
{
  #define H_XmTextSetInsertionPosition "void XmTextSetInsertionPosition(Widget widget, XmTextPosition position) \
sets the position of the insert cursor"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextSetInsertionPosition", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextSetInsertionPosition", "XmTextPosition");
  XmTextSetInsertionPosition(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextGetInsertionPosition(XEN arg1)
{
  #define H_XmTextGetInsertionPosition "XmTextPosition XmTextGetInsertionPosition(Widget widget) accesses the \
position of the insert cursor"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetInsertionPosition", "Text or TextField Widget");
  return(C_TO_XEN_INT(XmTextGetInsertionPosition(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextGetCursorPosition(XEN arg1)
{
  #define H_XmTextGetCursorPosition "int XmTextGetCursorPosition(Widget w) presumably returns the widget's insertion cursor position"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetCursorPosition", "Text or TextField Widget");
  return(C_TO_XEN_INT(XmTextGetCursorPosition(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextSetTopCharacter(XEN arg1, XEN arg2)
{
  #define H_XmTextSetTopCharacter "void XmTextSetTopCharacter(Widget widget, XmTextPosition top_character) sets \
the position of the first character displayed"
  XEN_ASSERT_TYPE(XEN_JustTextWidget_P(arg1), arg1, 1, "XmTextSetTopCharacter", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextSetTopCharacter", "XmTextPosition");
  XmTextSetTopCharacter(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextGetTopCharacter(XEN arg1)
{
  #define H_XmTextGetTopCharacter "XmTextPosition XmTextGetTopCharacter(Widget widget) accesses the position of the first character displayed"
  XEN_ASSERT_TYPE(XEN_JustTextWidget_P(arg1), arg1, 1, "XmTextGetTopCharacter", "Text or TextField Widget");
  return(C_TO_XEN_INT(XmTextGetTopCharacter(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextSetMaxLength(XEN arg1, XEN arg2)
{
  #define H_XmTextSetMaxLength "void XmTextSetMaxLength(Widget widget, int max_length) sets the value of the current \
maximum allowable length of a text string entered from the keyboard"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextSetMaxLength", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextSetMaxLength", "int");
  XmTextSetMaxLength(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextGetMaxLength(XEN arg1)
{
  #define H_XmTextGetMaxLength "int XmTextGetMaxLength(Widget widget) accesses the value of the current maximum allowable \
length of a text string entered from the keyboard"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetMaxLength", "Text or TextField Widget");
  return(C_TO_XEN_INT(XmTextGetMaxLength(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextSetEditable(XEN arg1, XEN arg2)
{
  #define H_XmTextSetEditable "void XmTextSetEditable(Widget widget, Boolean editable) sets the edit permission"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextSetEditable", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XmTextSetEditable", "boolean");
  XmTextSetEditable(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextGetEditable(XEN arg1)
{
  #define H_XmTextGetEditable "Boolean XmTextGetEditable(Widget widget) accesses the edit permission state"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetEditable", "Text or TextField Widget");
  return(C_TO_XEN_BOOLEAN(XmTextGetEditable(XEN_TO_C_Widget(arg1))));
}

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmTextGetAddMode(XEN arg1)
{
  #define H_XmTextGetAddMode "Boolean XmTextGetAddMode(Widget w) presumably returns the widget's current add mode."
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetAddMode", "Text or TextField Widget");
  return(C_TO_XEN_BOOLEAN(XmTextGetAddMode(XEN_TO_C_Widget(arg1))));
}
#endif

static XEN gxm_XmTextSetAddMode(XEN arg1, XEN arg2)
{
  #define H_XmTextSetAddMode "void XmTextSetAddMode(Widget widget, Boolean state) sets the widget's add mode -- \
this determines whether you can move the insertion cursor without changing the primary selection"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextSetAddMode", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XmTextSetAddMode", "boolean");
  XmTextSetAddMode(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}
#endif

static XEN gxm_XmTextInsert(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmTextInsert "void XmTextInsert(Widget widget, XmTextPosition position, char *value) inserts a character \
string into a text string"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextInsert", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextInsert", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XmTextInsert", "char*");
  XmTextInsert(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_STRING(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmTextReplace(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmTextReplace "void XmTextReplace(Widget widget, XmTextPosition from_pos, XmTextPosition to_pos, char *value) \
replaces part of a text string"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextReplace", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextReplace", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextReplace", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XmTextReplace", "char*");
  XmTextReplace(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_STRING(arg4));
  return(XEN_FALSE);
}

static XEN gxm_XmTextSetString(XEN arg1, XEN arg2)
{
  #define H_XmTextSetString "void XmTextSetString(Widget widget, char *value) sets the string value"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextSetString", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmTextSetString", "char*");
  XmTextSetString(XEN_TO_C_Widget(arg1), XEN_TO_C_STRING(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextGetLastPosition(XEN arg1)
{
  #define H_XmTextGetLastPosition "XmTextPosition XmTextGetLastPosition(Widget widget) accesses the last position in the text"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetLastPosition", "Text or TextField Widget");
  return(C_TO_XEN_INT(XmTextGetLastPosition(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextGetString(XEN arg1)
{
  char *str;
  XEN res;
  #define H_XmTextGetString "char *XmTextGetString(Widget widget) accesses the string value"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetString", "Text or TextField Widget");
  str = XmTextGetString(XEN_TO_C_Widget(arg1));
  res = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  return(res);
}

static XEN gxm_XmTextGetSubstring(XEN arg1, XEN arg2, XEN arg3)
{
  /* DIFF: omit and rtn last 2 args */
  #define H_XmTextGetSubstring "int XmTextGetSubstring(Widget widget, XmTextPosition start, int num_chars) \
retrieves a copy of a portion of the internal text buffer"
  int rtn, len;
  char *buf;
  XEN str = XEN_FALSE;
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextGetSubstring", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextGetSubstring", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextGetSubstring", "int");
  len = XEN_TO_C_INT(arg3);
  buf = (char *)CALLOC(len + 1, sizeof(char));
  rtn = XmTextGetSubstring(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), len, len + 1, buf);
  if (rtn != XmCOPY_FAILED)
    str = C_TO_XEN_STRING(buf);
  FREE(buf);
  return(str);
}

static XEN gxm_XmCreateText(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateText "Widget XmCreateText(Widget parent, String name, ArgList arglist, Cardinal argcount) The Text widget creation function"
  return(gxm_new_widget("XmCreateText", XmCreateText, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateScrolledText(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateScrolledText "Widget XmCreateScrolledText(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Text ScrolledText creation function"
  return(gxm_new_widget("XmCreateScrolledText", XmCreateScrolledText, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmTextSetHighlight(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmTextSetHighlight "void XmTextSetHighlight(Widget widget, XmTextPosition left, XmTextPosition right, XmHighlightMode mode) \
highlights text"
  XEN_ASSERT_TYPE(XEN_TextWidget_P(arg1), arg1, 1, "XmTextSetHighlight", "Text or TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextSetHighlight", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextSetHighlight", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmTextSetHighlight", "XmHighlightMode");
  XmTextSetHighlight(XEN_TO_C_Widget(arg1), 
		     (XmTextPosition)XEN_TO_C_INT(arg2), 
		     (XmTextPosition)XEN_TO_C_INT(arg3), 
		     (XmHighlightMode)XEN_TO_C_INT(arg4));
  return(XEN_FALSE);
}

static XEN gxm_XmCreateFileSelectionDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateFileSelectionDialog "Widget XmCreateFileSelectionDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The FileSelectionBox FileSelectionDialog creation function"
  return(gxm_new_widget("XmCreateFileSelectionDialog", XmCreateFileSelectionDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateFileSelectionBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateFileSelectionBox "Widget XmCreateFileSelectionBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The FileSelectionBox widget creation function"
  return(gxm_new_widget("XmCreateFileSelectionBox", XmCreateFileSelectionBox, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmFileSelectionDoSearch(XEN arg1, XEN arg2)
{
  #define H_XmFileSelectionDoSearch "void XmFileSelectionDoSearch(Widget widget, XmString dirmask) \
initiates a directory search"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmFileSelectionDoSearch", "Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2) || XEN_FALSE_P(arg2), arg2, 2, "XmFileSelectionDoSearch", "XmString");
  XmFileSelectionDoSearch(XEN_TO_C_Widget(arg1), XEN_FALSE_P(arg2) ? NULL : XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmFileSelectionBoxGetChild(XEN arg1, XEN arg2)
{
  #define H_XmFileSelectionBoxGetChild "Widget XmFileSelectionBoxGetChild(Widget widget, unsigned char child) \
used to access a component"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmFileSelectionBoxGetChild", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XmFileSelectionBoxGetChild", "unsigned int");
  return(C_TO_XEN_Widget(XmFileSelectionBoxGetChild(XEN_TO_C_Widget(arg1), XEN_TO_C_ULONG(arg2))));
}

static XEN gxm_XmCreateTextField(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateTextField "Widget XmCreateTextField(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The TextField widget creation function"
  return(gxm_new_widget("XmCreateTextField", XmCreateTextField, arg1, arg2, arg3, arg4));
}

#define XEN_TextFieldWidget_P(Arg) (XEN_Widget_P(Arg) && XmIsTextField(XEN_TO_C_Widget(Arg)))

static XEN gxm_XmTextFieldGetBaseline(XEN arg1)
{
  #define H_XmTextFieldGetBaseline "int XmTextFieldGetBaseline(Widget widget) accesses the y position of the baseline"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetBaseline", "TextField Widget");
  return(C_TO_XEN_INT(XmTextFieldGetBaseline(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextFieldSetHighlight(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmTextFieldSetHighlight "void XmTextFieldSetHighlight(Widget widget, XmTextPosition left, XmTextPosition right, XmHighlightMode mode) \
highlights text"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldSetHighlight", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldSetHighlight", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextFieldSetHighlight", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmTextFieldSetHighlight", "XmHighlightMode");
  XmTextFieldSetHighlight(XEN_TO_C_Widget(arg1), 
			  (XmTextPosition)XEN_TO_C_INT(arg2), 
			  (XmTextPosition)XEN_TO_C_INT(arg3), 
			  (XmHighlightMode)XEN_TO_C_INT(arg4));
  return(XEN_FALSE);
}

static XEN gxm_XmTextFieldShowPosition(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldShowPosition "void XmTextFieldShowPosition(Widget widget, XmTextPosition position) \
forces text at a given position to be displayed"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldShowPosition", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldShowPosition", "XmTextPosition");
  XmTextFieldShowPosition(XEN_TO_C_Widget(arg1), 
			  (XmTextPosition)XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextFieldPosToXY(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldPosToXY "Boolean XmTextFieldPosToXY(Widget widget, XmTextPosition position) returns the x and y position of a character position"
  /* DIFF: XmTextFieldPosToXY omits last 2 args and returns them
   */
  Position x, y;
  int val;
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldPosToXY", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldPosToXY", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmTextPosToXY", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextPosToXY", "XmTextPosition");
  val = XmTextFieldPosToXY(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), &x, &y);
  return(XEN_LIST_3(C_TO_XEN_BOOLEAN(val),
		    C_TO_XEN_Position(x),
		    C_TO_XEN_Position(y)));
}

static XEN gxm_XmTextFieldXYToPos(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmTextFieldXYToPos "XmTextPosition XmTextFieldXYToPos(Widget widget, Position x, Position y) \
accesses the character position nearest an x and y position"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldXYToPos", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldXYToPos", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextFieldXYToPos", "int");
  return(C_TO_XEN_INT(XmTextFieldXYToPos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XmTextFieldSetSelection(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmTextFieldSetSelection "void XmTextFieldSetSelection(Widget widget, XmTextPosition first, XmTextPosition last, Time time) \
sets the primary selection of the text"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldSetSelection", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldSetSelection", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextFieldSetSelection", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_Time_P(arg4), arg4, 4, "XmTextFieldSetSelection", "Time");
  XmTextFieldSetSelection(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_Time(arg4));
  return(XEN_FALSE);
}

static XEN gxm_XmTextFieldClearSelection(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldClearSelection "void XmTextFieldClearSelection(Widget widget, Time time) clears the primary selection"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldClearSelection", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmTextFieldClearSelection", "Time");
  XmTextFieldClearSelection(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2));
  return(XEN_FALSE);
}

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmTextFieldPasteLink(XEN arg1)
{
  #define H_XmTextFieldPasteLink "Boolean XmTextFieldPasteLink(Widget widget) inserts a link to the clipboard selection"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldPasteLink", "TextField Widget");
  return(C_TO_XEN_BOOLEAN(XmTextFieldPasteLink(XEN_TO_C_Widget(arg1))));
}
#endif

static XEN gxm_XmTextFieldCopyLink(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldCopyLink "Boolean XmTextFieldCopyLink(Widget widget, Time time) copies a link to the \
primary selection to the clipboard"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldCopyLink", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmTextFieldCopyLink", "Time");
  return(C_TO_XEN_BOOLEAN(XmTextFieldCopyLink(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2))));
}
#endif

static XEN gxm_XmTextFieldPaste(XEN arg1)
{
  #define H_XmTextFieldPaste "Boolean XmTextFieldPaste(Widget widget) inserts the clipboard selection"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldPaste", "TextField Widget");
  return(C_TO_XEN_BOOLEAN(XmTextFieldPaste(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextFieldCut(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldCut "Boolean XmTextFieldCut(Widget widget, Time time) copies the primary selection \
to the clipboard and deletes the selected text"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldCut", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmTextFieldCut", "Time");
  return(C_TO_XEN_BOOLEAN(XmTextFieldCut(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2))));
}

static XEN gxm_XmTextFieldCopy(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldCopy "Boolean XmTextFieldCopy(Widget widget, Time time) copies the primary selection to the clipboard"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldCopy", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmTextFieldCopy", "Time");
  return(C_TO_XEN_BOOLEAN(XmTextFieldCopy(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2))));
}

static XEN gxm_XmTextFieldRemove(XEN arg1)
{
  #define H_XmTextFieldRemove "Boolean XmTextFieldRemove(Widget widget) deletes the primary selection"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldRemove", "TextField Widget");
  return(C_TO_XEN_BOOLEAN(XmTextFieldRemove(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextFieldGetSelection(XEN arg1)
{
  char *str;
  XEN res;
  #define H_XmTextFieldGetSelection "char *XmTextFieldGetSelection(Widget widget) retrieves the value of the primary selection"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetSelection", "TextField Widget");
  str = XmTextFieldGetSelection(XEN_TO_C_Widget(arg1));
  res = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  return(res);
}

static XEN gxm_XmTextFieldGetSelectionPosition(XEN arg1)
{
  #define H_XmTextFieldGetSelectionPosition "Boolean XmTextFieldGetSelectionPosition(Widget widget) \
returns the position of the primary selection"
  /* DIFF: XmTextFieldGetSelectionPosition widget [left right] -> (list res left right)
     no arg2 arg3, returns (owner left right) instead 
  */
  XmTextPosition pos1, pos2;
  int res;
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetSelectionPosition", "TextField Widget");
  res = XmTextFieldGetSelectionPosition(XEN_TO_C_Widget(arg1), &pos1, &pos2);
  return(XEN_LIST_3(C_TO_XEN_BOOLEAN(res),
		    C_TO_XEN_INT(pos1),
		    C_TO_XEN_INT(pos2)));
}

static XEN gxm_XmTextFieldSetInsertionPosition(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldSetInsertionPosition "void XmTextFieldSetInsertionPosition(Widget widget, XmTextPosition position) \
sets the position of the insertion cursor"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldSetInsertionPosition", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldSetInsertionPosition", "XmTextPosition");
  XmTextFieldSetInsertionPosition(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextFieldSetCursorPosition(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldSetCursorPosition "void XmTextFieldSetCursorPosition(Widget w, int position) sets the insertion cursor position"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldSetCursorPosition", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldSetCursorPosition", "XmTextPosition");
  XmTextFieldSetCursorPosition(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextFieldGetInsertionPosition(XEN arg1)
{
  #define H_XmTextFieldGetInsertionPosition "XmTextPosition XmTextFieldGetInsertionPosition(Widget widget) \
accesses the position of the insertion cursor"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetInsertionPosition", "TextField Widget");
  return(C_TO_XEN_INT(XmTextFieldGetInsertionPosition(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextFieldGetCursorPosition(XEN arg1)
{
  #define H_XmTextFieldGetCursorPosition "int XmTextFieldGetCursorPosition(Widget w) presumably returns the widget's insertion cursor position"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetCursorPosition", "TextField Widget");
  return(C_TO_XEN_INT(XmTextFieldGetCursorPosition(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextFieldSetMaxLength(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldSetMaxLength "void XmTextFieldSetMaxLength(Widget widget, int max_length) sets the \
value of the current maximum allowable length of a text string \
entered from the keyboard"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldSetMaxLength", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldSetMaxLength", "int");
  XmTextFieldSetMaxLength(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextFieldGetMaxLength(XEN arg1)
{
  #define H_XmTextFieldGetMaxLength "int XmTextFieldGetMaxLength(Widget widget) accesses the value of the \
current maximum allowable length of a text string \
entered from the keyboard"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetMaxLength", "TextField Widget");
  return(C_TO_XEN_INT(XmTextFieldGetMaxLength(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextFieldSetEditable(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldSetEditable "void XmTextFieldSetEditable(Widget widget, Boolean editable) sets the edit permission"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldSetEditable", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XmTextFieldSetEditable", "boolean");
  XmTextFieldSetEditable(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextFieldGetEditable(XEN arg1)
{
  #define H_XmTextFieldGetEditable "Boolean XmTextFieldGetEditable(Widget widget) accesses the edit permission state"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetEditable", "TextField Widget");
  return(C_TO_XEN_BOOLEAN(XmTextFieldGetEditable(XEN_TO_C_Widget(arg1))));
}

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmTextFieldGetAddMode(XEN arg1)
{
  #define H_XmTextFieldGetAddMode "Boolean XmTextFieldGetAddMode(Widget w) presumably returns the widget's current add mode."
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetAddMode", "TextField Widget");
  return(C_TO_XEN_BOOLEAN(XmTextFieldGetAddMode(XEN_TO_C_Widget(arg1))));
}
#endif

static XEN gxm_XmTextFieldSetAddMode(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldSetAddMode "void XmTextFieldSetAddMode(Widget widget, Boolean state) sets the state of Add mode"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldSetAddMode", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XmTextFieldSetAddMode", "boolean");
  XmTextFieldSetAddMode(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}
#endif

static XEN gxm_XmTextFieldInsert(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmTextFieldInsert "void XmTextFieldInsert(Widget widget, XmTextPosition position, char *value) \
inserts a character string into a text string"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldInsert", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldInsert", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XmTextFieldInsert", "char*");
  XmTextFieldInsert(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_STRING(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmTextFieldReplace(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmTextFieldReplace "void XmTextFieldReplace(Widget widget, XmTextPosition from_pos, XmTextPosition to_pos, char *value) \
replaces part of a text string"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldReplace", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldReplace", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextFieldReplace", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XmTextFieldReplace", "char*");
  XmTextFieldReplace(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_STRING(arg4));
  return(XEN_FALSE);
}

static XEN gxm_XmTextFieldSetString(XEN arg1, XEN arg2)
{
  #define H_XmTextFieldSetString "void XmTextFieldSetString(Widget widget, char *value) sets the string value"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldSetString", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmTextFieldSetString", "char*");
  XmTextFieldSetString(XEN_TO_C_Widget(arg1), XEN_TO_C_STRING(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmTextFieldGetLastPosition(XEN arg1)
{
  #define H_XmTextFieldGetLastPosition "XmTextPosition XmTextFieldGetLastPosition(Widget widget) accesses \
the position of the last text character"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetLastPosition", "TextField Widget");
  return(C_TO_XEN_INT(XmTextFieldGetLastPosition(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmTextFieldGetSubstring(XEN arg1, XEN arg2, XEN arg3)
{
  /* DIFF: omit and rtn last 2 args */
  #define H_XmTextFieldGetSubstring "int XmTextFieldGetSubstring(Widget widget, XmTextPosition start, int num_chars) \
retrieves a copy of a portion of the internal text buffer"
  int rtn, len;
  char *buf;
  XEN str = XEN_FALSE;
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetSubstring", "TextField Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmTextFieldGetSubstring", "XmTextPosition");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTextFieldGetSubstring", "int");
  len = XEN_TO_C_INT(arg3);
  buf = (char *)CALLOC(len + 1, sizeof(char));
  rtn = XmTextFieldGetSubstring(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), len, len + 1, buf);
  if (rtn != XmCOPY_FAILED)
    str = C_TO_XEN_STRING(buf);
  FREE(buf);
  return(str);
}

static XEN gxm_XmTextFieldGetString(XEN arg1)
{
  char *str;
  XEN res;
  #define H_XmTextFieldGetString "char *XmTextFieldGetString(Widget widget) accesses the string value"
  XEN_ASSERT_TYPE(XEN_TextFieldWidget_P(arg1), arg1, 1, "XmTextFieldGetString", "TextField Widget");
  str = XmTextFieldGetString(XEN_TO_C_Widget(arg1));
  res = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  return(res);
}

static XEN gxm_XmDropTransferAdd(XEN arg1, XEN arg2)
{
  #define H_XmDropTransferAdd "void XmDropTransferAdd(Widget drop_transfer, XmDropTransferEntryRec *transfers) \
enables additional drop transfer entries to be processed after initiating a drop transfer"
  /* DIFF: XmDropTransferAdd  arg2 is list of (atom data) lists, arg3 omitted
   */
  int len;
  XmDropTransferEntryRec *entries;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropTransferAdd", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmDropTransferAdd", "XmDropTransferEntry");
  len = XEN_LIST_LENGTH(arg2);
  if (len <= 0) return(XEN_FALSE);
  entries = XEN_TO_C_XmDropTransferEntryRecs(arg2, len);
  XmDropTransferAdd(XEN_TO_C_Widget(arg1), entries, len);
  FREE(entries);
  return(XEN_FALSE);
}

static XEN gxm_XmDropTransferStart(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmDropTransferStart "Widget XmDropTransferStart(Widget widget, ArgList arglist, Cardinal argcount) \
initiates a drop transfer"
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropTransferStart", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmDropTransferStart", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmDropTransferStart", "int");
  {
    Arg *args;
    int arglen;
    args = XEN_TO_C_Args(arg2);
    arglen = XEN_TO_C_INT_DEF(arg3, arg2);
    w = XmDropTransferStart(XEN_TO_C_Widget(arg1), args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XmDropSiteConfigureStackingOrder(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmDropSiteConfigureStackingOrder "void XmDropSiteConfigureStackingOrder(Widget widget, Widget sibling, Cardinal stack_mode) \
reorders a stack of widgets that are registered drop sites"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropSiteConfigureStackingOrder", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2), arg2, 2, "XmDropSiteConfigureStackingOrder", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmDropSiteConfigureStackingOrder", "int");
  XmDropSiteConfigureStackingOrder(XEN_TO_C_Widget(arg1), XEN_TO_C_Widget(arg2), XEN_TO_C_INT(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmDropSiteQueryStackingOrder(XEN arg1)
{
  #define H_XmDropSiteQueryStackingOrder "Status XmDropSiteQueryStackingOrder(Widget widget) -> (list parent child ...)"
  /* DIFF: XmDropSiteQueryStackingOrder widget [parent child numchild] -> (list parent child ...)
     no parent children num_children -- returned as '(parent child1...) or #f 
  */
  Widget parent;
  Widget *children;
  XEN lst = XEN_EMPTY_LIST;
  unsigned int num_children;
  int res, i, loc;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropSiteQueryStackingOrder", "Widget");
  res = XmDropSiteQueryStackingOrder(XEN_TO_C_Widget(arg1), &parent, &children, &num_children);
  if (res == 0)
    return(XEN_FALSE);
  loc = xm_protect(lst);
  for (i = num_children - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_Widget(children[i]), lst);
  xm_unprotect_at(loc);
  return(XEN_CONS(C_TO_XEN_Widget(parent), lst));
}

static XEN gxm_XmDropSiteRetrieve(XEN arg1, XEN larg2, XEN arg3)
{
  #define H_XmDropSiteRetrieve "void XmDropSiteRetrieve(Widget widget, ArgList arglist, Cardinal argcount) \
retrieves resource values set on a drop site"

  Arg *args;
  unsigned long *locs;
  XEN val = XEN_FALSE;
  int i, len, gcloc;
  XEN arg2;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropSiteRetrieve", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg2), larg2, 2, "XmDropSiteRetrieve", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmDropSiteRetrieve", "int");
  arg2 = XEN_COPY_ARG(larg2);
  gcloc = xm_protect(arg2);
  len = XEN_TO_C_INT_DEF(arg3, larg2);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg3, 3, "XmDropSiteRetrieve", "positive integer");
  args = (Arg *)CALLOC(len, sizeof(Arg));
  locs = (unsigned long *)CALLOC(len, sizeof(unsigned long));
  for (i = 0; i < len; i++, arg2 = XEN_CDDR(arg2))
    {
      char *name;
      name = strdup(XEN_TO_C_STRING(XEN_CAR(arg2)));
      XtSetArg(args[i], name, &(locs[i]));
    }
  XmDropSiteRetrieve(XEN_TO_C_Widget(arg1), args, len);
  val = C_TO_XEN_Args((Widget)(XEN_TO_C_Widget(arg1)), args, len);
  free_args(args, len);
  FREE(locs);
  xm_unprotect_at(gcloc);
  return(xen_return_first(val, larg2));
}

static XEN gxm_XmDropSiteEndUpdate(XEN arg1)
{
  #define H_XmDropSiteEndUpdate "void XmDropSiteEndUpdate(Widget widget) facilitates processing updates to multiple drop sites"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropSiteEndUpdate", "Widget");
  XmDropSiteEndUpdate(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmDropSiteUpdate(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmDropSiteUpdate "void XmDropSiteUpdate(Widget widget, ArgList arglist, Cardinal argcount) sets \
resource values for a drop site"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropSiteUpdate", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmDropSiteUpdate", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmDropSiteUpdate", "int");
  {
    Widget w;
    Arg *args;
    int arglen;
    w = XEN_TO_C_Widget(arg1);
    args = XEN_TO_C_Args(arg2);
    arglen = XEN_TO_C_INT_DEF(arg3, arg2);
    XmDropSiteUpdate(w, args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(XEN_FALSE);
}

static XEN gxm_XmDropSiteStartUpdate(XEN arg1)
{
  #define H_XmDropSiteStartUpdate "void XmDropSiteStartUpdate(Widget widget) facilitates processing updates \
to multiple drop sites"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropSiteStartUpdate", "Widget");
  XmDropSiteStartUpdate(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

#if MOTIF_2
static XEN gxm_XmDropSiteRegistered(XEN arg1)
{
  #define H_XmDropSiteRegistered "Boolean XmDropSiteRegistered(Widget widget) determines if a drop site has been registered"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropSiteRegistered", "Widget");
  return(C_TO_XEN_BOOLEAN(XmDropSiteRegistered(XEN_TO_C_Widget(arg1))));
}
#endif

static XEN gxm_XmDropSiteUnregister(XEN arg1)
{
  #define H_XmDropSiteUnregister "void XmDropSiteUnregister(Widget widget) frees drop site information"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropSiteUnregister", "Widget");
  XmDropSiteUnregister(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmDropSiteRegister(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmDropSiteRegister "void XmDropSiteRegister(Widget widget, ArgList arglist, Cardinal argcount) \
identifies a drop site and assigns resources that specify its behavior"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDropSiteRegister", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmDropSiteRegister", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XmDropSiteRegister", "int");
  {
    Widget w;
    Arg *args;
    int arglen;
    w = XEN_TO_C_Widget(arg1);
    args = XEN_TO_C_Args(arg2);
    arglen = XEN_TO_C_INT_DEF(arg3, arg2);
    XmDropSiteRegister(w, args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(XEN_FALSE);
}

#if MOTIF_2
static XEN gxm_XmSimpleSpinBoxSetItem(XEN arg1, XEN arg2)
{
  #define H_XmSimpleSpinBoxSetItem "void XmSimpleSpinBoxSetItem(Widget w, XmString item) set an item in the XmSimpleSpinBox list"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmSimpleSpinBoxSetItem", "Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmSimpleSpinBoxSetItem", "XmString");
  XmSimpleSpinBoxSetItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmSimpleSpinBoxDeletePos(XEN arg1, XEN arg2)
{
  #define H_XmSimpleSpinBoxDeletePos "void XmSimpleSpinBoxDeletePos(Widget w, int pos) delete a XmSimpleSpinBox item"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmSimpleSpinBoxDeletePos", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmSimpleSpinBoxDeletePos", "int");
  XmSimpleSpinBoxDeletePos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmSimpleSpinBoxAddItem(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmSimpleSpinBoxAddItem "void XmSimpleSpinBoxAddItem(Widget w, XmString item, int pos) add an item to the XmSimpleSpinBox"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmSimpleSpinBoxAddItem", "Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmSimpleSpinBoxAddItem", "XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmSimpleSpinBoxAddItem", "int");
  XmSimpleSpinBoxAddItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2), XEN_TO_C_INT(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmCreateSimpleSpinBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSimpleSpinBox "Widget XmCreateSimpleSpinBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The SimpleSpinBox widget creation function"
  return(gxm_new_widget("XmCreateSimpleSpinBox", XmCreateSimpleSpinBox, arg1, arg2, arg3, arg4));
}
#endif

static XEN gxm_XmCreateDrawnButton(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateDrawnButton "Widget XmCreateDrawnButton(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The DrawnButton widget creation function"
  return(gxm_new_widget("XmCreateDrawnButton", XmCreateDrawnButton, arg1, arg2, arg3, arg4));
}

#if MOTIF_2
static XEN gxm_XmSpinBoxValidatePosition(XEN arg1)
{
  #define H_XmSpinBoxValidatePosition "int XmSpinBoxValidatePosition(Widget textfield) translate the current value of \
the specified XmSpinBox child into a valid position"
  /* DIFF: XmSpinBoxValidatePosition omits arg2, returns pos
   */
  int pos;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmSpinBoxValidatePosition", "Widget");
  XmSpinBoxValidatePosition(XEN_TO_C_Widget(arg1), &pos);
  return(C_TO_XEN_INT(pos));
}

static XEN gxm_XmCreateSpinBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSpinBox "The SpinBox creation function"
  return(gxm_new_widget("XmCreateSpinBox", XmCreateSpinBox, arg1, arg2, arg3, arg4));
}
#endif

static XEN gxm_XmCreateDrawingArea(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateDrawingArea "Widget XmCreateDrawingArea(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The DrawingArea widget creation function"
  return(gxm_new_widget("XmCreateDrawingArea", XmCreateDrawingArea, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateSeparator(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSeparator "Widget XmCreateSeparator(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Separator widget creation function"
  return(gxm_new_widget("XmCreateSeparator", XmCreateSeparator, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateDragIcon(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateDragIcon "Widget XmCreateDragIcon(Widget widget, String name, ArgList arglist, Cardinal argcount) \
creates a DragIcon widget"
  return(gxm_new_widget("XmCreateDragIcon", XmCreateDragIcon, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateSeparatorGadget(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSeparatorGadget "Widget XmCreateSeparatorGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The SeparatorGadget creation function"
  return(gxm_new_widget("XmCreateSeparatorGadget", XmCreateSeparatorGadget, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmTargetsAreCompatible(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmTargetsAreCompatible "Boolean XmTargetsAreCompatible(Display *display, Atom *export_targets, Cardinal num_export_targets, \
Atom *import_targets, Cardinal num_import_targets)  tests whether the target types match between a drop site and source object"
  /* DIFF: XmTargetsAreCompatible arg2 and arg4 are lists of Atoms
   */
  Atom *outs, *ins;
  int val, len1, len2;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmTargetsAreCompatible", "Display*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmTargetsAreCompatible", "list of Atom");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmTargetsAreCompatible", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XmTargetsAreCompatible", "list of Atom");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XmTargetsAreCompatible", "int");
  len1 = XEN_TO_C_INT(arg3);
  outs = XEN_TO_C_Atoms(arg2, len1);
  len2 = XEN_TO_C_INT(arg5);
  ins = XEN_TO_C_Atoms(arg4, len2);
  val = XmTargetsAreCompatible(XEN_TO_C_Display(arg1), outs, len1, ins, len2);
  FREE(outs);
  FREE(ins);
  return(C_TO_XEN_BOOLEAN(val));
}

static XEN gxm_XmDragCancel(XEN arg1)
{
  #define H_XmDragCancel "void XmDragCancel(Widget dragcontext) terminates a drag transaction"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDragCancel", "Widget");
  XmDragCancel(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmDragStart(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmDragStart "Widget XmDragStart(Widget widget, XEvent *event, ArgList arglist, Cardinal argcount) \
initiates a drag and drop transaction"
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDragStart", "Widget");
  XEN_ASSERT_TYPE(XEN_XEvent_P(arg2), arg2, 2, "XmDragStart", "XEvent*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XmDragStart", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg4), arg4, 4, "XmDragStart", "int");
  {
    Arg *args;
    int arglen;
    args = XEN_TO_C_Args(arg3);
    arglen = XEN_TO_C_INT_DEF(arg4, arg3);
    w = XmDragStart(XEN_TO_C_Widget(arg1), 
		    XEN_TO_C_XEvent(arg2), 
		    args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XmCreatePromptDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreatePromptDialog "Widget XmCreatePromptDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The SelectionBox PromptDialog creation function"
  return(gxm_new_widget("XmCreatePromptDialog", XmCreatePromptDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateSelectionDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSelectionDialog "Widget XmCreateSelectionDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The SelectionBox SelectionDialog creation function"
  return(gxm_new_widget("XmCreateSelectionDialog", XmCreateSelectionDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateSelectionBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateSelectionBox "Widget XmCreateSelectionBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The SelectionBox widget creation function"
  return(gxm_new_widget("XmCreateSelectionBox", XmCreateSelectionBox, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmSelectionBoxGetChild(XEN arg1, XEN arg2)
{
  #define H_XmSelectionBoxGetChild "Widget XmSelectionBoxGetChild(Widget widget, unsigned char child) used to access a SelectionBox component"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmSelectionBoxGetChild", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XmSelectionBoxGetChild", "unsigned int");
  return(C_TO_XEN_Widget(XmSelectionBoxGetChild(XEN_TO_C_Widget(arg1), XEN_TO_C_ULONG(arg2))));
}

static XEN gxm_XmGetXmDisplay(XEN arg1)
{
  #define H_XmGetXmDisplay "Widget XmGetXmDisplay(Display *display) A Display function that returns the"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmGetXmDisplay", "Display*");
  return(C_TO_XEN_Widget(XmGetXmDisplay(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XmGetDragContext(XEN arg1, XEN arg2)
{
  #define H_XmGetDragContext "Widget XmGetDragContext(Widget refwidget, Time timestamp) retrieves \
the DragContext widget ID associated with a timestamp"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmGetDragContext", "Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmGetDragContext", "Time");
  return(C_TO_XEN_Widget(XmGetDragContext(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2))));
}

static XEN gxm_XmScrollVisible(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmScrollVisible "void XmScrollVisible(Widget scrollw_widget, Widget widget, Dimension left_right_margin, Dimension top_bottom_margin) \
makes an invisible  descendant of a ScrolledWindow work area visible"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmScrollVisible", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2), arg2, 2, "XmScrollVisible", "Widget");
  XEN_ASSERT_TYPE(XEN_Dimension_P(arg3), arg3, 3, "XmScrollVisible", "Dimension");
  XEN_ASSERT_TYPE(XEN_Dimension_P(arg4), arg4, 4, "XmScrollVisible", "Dimension");
  XmScrollVisible(XEN_TO_C_Widget(arg1), XEN_TO_C_Widget(arg2), 
		  XEN_TO_C_Dimension(arg3), XEN_TO_C_Dimension(arg4));
  return(XEN_FALSE);
}

static XEN gxm_XmCreateScrolledWindow(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateScrolledWindow "Widget XmCreateScrolledWindow(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ScrolledWindow widget creation function"
  return(gxm_new_widget("XmCreateScrolledWindow", XmCreateScrolledWindow, arg1, arg2, arg3, arg4));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XmScrolledWindowSetAreas(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmScrolledWindowSetAreas "void XmScrolledWindowSetAreas(Widget widget, Widget horizontal_scrollbar, Widget vertical_scrollbar, \
Widget work_region) adds or changes a window work region and a horizontal or vertical \
ScrollBar widget to the ScrolledWindow widget"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmScrolledWindowSetAreas", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2), arg2, 2, "XmScrolledWindowSetAreas", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg3), arg3, 3, "XmScrolledWindowSetAreas", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg4), arg4, 4, "XmScrolledWindowSetAreas", "Widget");
  XmScrolledWindowSetAreas(XEN_TO_C_Widget(arg1), XEN_TO_C_Widget(arg2), 
			   XEN_TO_C_Widget(arg3), XEN_TO_C_Widget(arg4));
  return(XEN_FALSE);
}
#endif

static XEN gxm_XmCreateDialogShell(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateDialogShell "Widget XmCreateDialogShell(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The DialogShell widget creation function"
  return(gxm_new_widget("XmCreateDialogShell", XmCreateDialogShell, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmScrollBarSetValues(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XmScrollBarSetValues "void XmScrollBarSetValues (widget, value, slider_size, increment, page_increment, notify) \
changes ScrollBar's increment values and the slider's size and position"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmScrollBarSetValues", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmScrollBarSetValues", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmScrollBarSetValues", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmScrollBarSetValues", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XmScrollBarSetValues", "int");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg6), arg6, 6, "XmScrollBarSetValues", "boolean");
  XmScrollBarSetValues(XEN_TO_C_Widget(arg1), 
		       XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), 
		       XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
		       XEN_TO_C_BOOLEAN(arg6));
  return(XEN_FALSE);
}

static XEN gxm_XmScrollBarGetValues(XEN arg1)
{
  #define H_XmScrollBarGetValues "void XmScrollBarGetValues (widget) returns the ScrollBar's increment values (list val size incr page)"
  /* DIFF: XmScrollBarGetValues omits and returns last 4 args
   */
  int val, size, incr, page;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmScrollBarGetValues", "Widget");
  XmScrollBarGetValues(XEN_TO_C_Widget(arg1), &val, &size, &incr, &page);
  return(XEN_LIST_4(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(size),
		    C_TO_XEN_INT(incr),
		    C_TO_XEN_INT(page)));
}

static XEN gxm_XmCreateScrollBar(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateScrollBar "Widget XmCreateScrollBar(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ScrollBar widget creation function"
  return(gxm_new_widget("XmCreateScrollBar", XmCreateScrollBar, arg1, arg2, arg3, arg4));
}

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmGetXmScreen(XEN arg1)
{
  #define H_XmGetXmScreen "Widget XmGetXmScreen(Screen *screen) returns the XmScreen object ID for a specified screen"
  /* this is the Motif Screen "widget" */
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XmGetXmScreen", "Screen*");
  return(C_TO_XEN_Widget(XmGetXmScreen(XEN_TO_C_Screen(arg1))));
}
#endif
#endif

static XEN gxm_XmClipboardRegisterFormat(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmClipboardRegisterFormat "int XmClipboardRegisterFormat (display, format_name, format_length) registers a new format"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardRegisterFormat", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmClipboardRegisterFormat", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmClipboardRegisterFormat", "int");
  return(C_TO_XEN_INT(XmClipboardRegisterFormat(XEN_TO_C_Display(arg1), 
						XEN_TO_C_STRING(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XmClipboardInquirePendingItems(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmClipboardInquirePendingItems "int XmClipboardInquirePendingItems (display, window, format_name) \
returns a list of data ID/private ID pairs"
  /* DIFF: XmClipboardInquirePendingItems omits last 2 args, returns list of pairs
   */
  unsigned long len;
  XmClipboardPendingList clst;
  int i, loc, rtn;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardInquirePendingItems", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardInquirePendingItems", "Window");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XmClipboardInquirePendingItems", "char*");
  rtn = XmClipboardInquirePendingItems(XEN_TO_C_Display(arg1), 
				       XEN_TO_C_Window(arg2), 
				       XEN_TO_C_STRING(arg3), 
				       &clst,
				       &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(XEN_LIST_2(C_TO_XEN_INT(clst[i].DataId), 
			      C_TO_XEN_INT(clst[i].PrivateId)),
		   lst);
  lst = XEN_CONS(C_TO_XEN_INT(rtn), lst);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XmClipboardInquireLength(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmClipboardInquireLength "int XmClipboardInquireLength (display, window, format_name) returns the length of the stored data"
  /* DIFF: XmClipboardInquireLength omit and rtn last arg
   */
  unsigned long len;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardInquireLength", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardInquireLength", "Window");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XmClipboardInquireLength", "char*");
  val = XmClipboardInquireLength(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_STRING(arg3), &len);
  return(XEN_LIST_2(C_TO_XEN_INT(val),
		    C_TO_XEN_ULONG(len)));
}

static XEN gxm_XmClipboardInquireFormat(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmClipboardInquireFormat "int XmClipboardInquireFormat (display, window, index, buffer_len) \
returns a specified format name"
  /* DIFF: XmClipboardInquireFormat omits arg4 (XtPointer buffer) and arg6, returns them
   */
  XEN res;
  XtPointer buf;
  unsigned long len, n;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardInquireFormat", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardInquireFormat", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmClipboardInquireFormat", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XmClipboardInquireFormat", "ulong");
  len = XEN_TO_C_ULONG(arg4);
  if (len == 0) return(XEN_FALSE);
  buf = (XtPointer)CALLOC(len + 1, sizeof(char));
  val = XmClipboardInquireFormat(XEN_TO_C_Display(arg1), 
				 XEN_TO_C_Window(arg2), 
				 XEN_TO_C_INT(arg3), 
				 buf, len, &n);
  res = C_TO_XEN_STRING((char *)buf);
  FREE(buf);
  return(XEN_LIST_2(C_TO_XEN_INT(val), res));
}

static XEN gxm_XmClipboardInquireCount(XEN arg1, XEN arg2)
{
  #define H_XmClipboardInquireCount "int XmClipboardInquireCount (display, window) returns the number of data item formats"
  /* DIFF: XmClipboardInquireCount omits and rtns last 2 args
   */
  int count, val;
  unsigned long len;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardInquireCount", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardInquireCount", "Window");
  val = XmClipboardInquireCount(XEN_TO_C_Display(arg1), 
				XEN_TO_C_Window(arg2),
				&count, &len);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(count),
		    C_TO_XEN_ULONG(len)));
}

static XEN gxm_XmClipboardRetrieve(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmClipboardRetrieve "int XmClipboardRetrieve (display, window, format_name, length) retrieves a data item from the clipboard"
  /* DIFF: XmClipboardRetrieve omits buf arg, and last 2, returning them and a list of ulongs
   */
  unsigned long n;
  long id;
  XtPointer buf;
  int len, val;
  XEN str;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardRetrieve", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardRetrieve", "Window");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XmClipboardRetrieve", "char*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XmClipboardRetrieve", "ulong");
  len = XEN_TO_C_ULONG(arg4);
  if (len <= 0) return(XEN_FALSE);
  buf = (XtPointer)MALLOC(len);
  val = XmClipboardRetrieve(XEN_TO_C_Display(arg1), 
			    XEN_TO_C_Window(arg2), 
			    XEN_TO_C_STRING(arg3), 
			    buf, len, &n, &id);
  str = C_TO_XEN_STRING((char *)buf);
  FREE(buf);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    str,
		    C_TO_XEN_ULONG(id)));
}

static XEN gxm_XmClipboardEndRetrieve(XEN arg1, XEN arg2)
{
  #define H_XmClipboardEndRetrieve "int XmClipboardEndRetrieve (display, window) completes retrieval of \
data from the clipboard"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardEndRetrieve", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardEndRetrieve", "Window");
  return(C_TO_XEN_INT(XmClipboardEndRetrieve(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XmClipboardStartRetrieve(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmClipboardStartRetrieve "int XmClipboardStartRetrieve (display, window, timestamp) \
prepares to retrieve data from the clipboard"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardStartRetrieve", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardStartRetrieve", "Window");
  XEN_ASSERT_TYPE(XEN_Time_P(arg3), arg3, 3, "XmClipboardStartRetrieve", "Time");
  return(C_TO_XEN_INT(XmClipboardStartRetrieve(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_Time(arg3))));
}

static XEN gxm_XmClipboardUnlock(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmClipboardUnlock "int XmClipboardUnlock (display, window, remove_all_locks) unlocks the clipboard"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardUnlock", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardUnlock", "Window");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XmClipboardUnlock", "boolean");
  return(C_TO_XEN_INT(XmClipboardUnlock(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_BOOLEAN(arg3))));
}

static XEN gxm_XmClipboardLock(XEN arg1, XEN arg2)
{
  #define H_XmClipboardLock "int XmClipboardLock (display, window) locks the clipboard"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardLock", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardLock", "Window");
  return(C_TO_XEN_INT(XmClipboardLock(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XmClipboardUndoCopy(XEN arg1, XEN arg2)
{
  #define H_XmClipboardUndoCopy "int XmClipboardUndoCopy (display, window) deletes the last item placed on the clipboard"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardUndoCopy", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardUndoCopy", "Window");
  return(C_TO_XEN_INT(XmClipboardUndoCopy(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XmClipboardCopyByName(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XmClipboardCopyByName "int XmClipboardCopyByName (display, window, data_id,  buf, len, id) copies a data item passed by name"
  /* DIFF: XmClipboardCopyByName arg4 is string
   */
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardCopyByName", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardCopyByName", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmClipboardCopyByName", "long");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XmClipboardCopyByName", "XtPointer");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XmClipboardCopyByName", "ulong");
  return(C_TO_XEN_INT(XmClipboardCopyByName(XEN_TO_C_Display(arg1), 
					    XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3), 
					    (XtPointer)XEN_TO_C_STRING(arg4), 
					    XEN_TO_C_ULONG(arg5), 
					    (unsigned long)arg6)));
}

static XEN gxm_XmClipboardWithdrawFormat(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmClipboardWithdrawFormat "int XmClipboardWithdrawFormat (display, window, data_id) \
indicates that the application no longer wants to supply a data item"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardWithdrawFormat", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardWithdrawFormat", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmClipboardWithdrawFormat", "long");
  return(C_TO_XEN_INT(XmClipboardWithdrawFormat(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XmClipboardCancelCopy(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmClipboardCancelCopy "int XmClipboardCancelCopy (display, window, item_id) cancels a copy to the clipboard"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardCancelCopy", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardCancelCopy", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmClipboardCancelCopy", "long");
  return(C_TO_XEN_INT(XmClipboardCancelCopy(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XmClipboardEndCopy(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmClipboardEndCopy "int XmClipboardEndCopy (display, window, item_id) completes the \
copying of data to the clipboard"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardEndCopy", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardEndCopy", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmClipboardEndCopy", "long");
  return(C_TO_XEN_INT(XmClipboardEndCopy(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XmClipboardCopy(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XmClipboardCopy "int XmClipboardCopy (display, window, item_id, format_name, buffer, len, id) copies a data item \
to temporary storage for later copying to clipboard"
  /* DIFF: XmClipboardCopy omits and returns last arg, arg5 is string
   */
  long id;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardCopy", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardCopy", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmClipboardCopy", "long");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XmClipboardCopy", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg5), arg5, 5, "XmClipboardCopy", "XtPointer");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XmClipboardCopy", "ulong");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XmClipboardCopy", "long");
  val = XmClipboardCopy(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
			XEN_TO_C_INT(arg3), XEN_TO_C_STRING(arg4), 
			(XtPointer)XEN_TO_C_STRING(arg5), XEN_TO_C_ULONG(arg6), 
			XEN_TO_C_INT(arg7), &id);
  return(XEN_LIST_2(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(id)));
}

/* There's just one clipboard, I think, so these callbacks must be globals */
static XEN xm_XmCutPasteProc = XEN_FALSE;

static void gxm_XmCutPasteProc(Widget w, long *data, long *privater, int *reason)
{
  XEN_CALL_4(xm_XmCutPasteProc, 
	     C_TO_XEN_Widget(w),
	     C_TO_XEN_ULONG(*data),
	     C_TO_XEN_ULONG(*privater),
	     C_TO_XEN_INT(*reason),
	     c__FUNCTION__);
}

static XEN gxm_XmClipboardStartCopy(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XmClipboardStartCopy "int XmClipboardStartCopy (display, window, clip_label, timestamp, widget, callback) \
sets up a storage and data structure, returns id"
  /* DIFF: XmClipboardStartCopy omits and returns last arg
   */
  long id;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardStartCopy", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardStartCopy", "Window");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg3), arg3, 3, "XmClipboardStartCopy", "XmString");
  XEN_ASSERT_TYPE(XEN_Time_P(arg4), arg4, 4, "XmClipboardStartCopy", "Time");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg5), arg5, 5, "XmClipboardStartCopy", "Widget");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg6) && (XEN_REQUIRED_ARGS_OK(arg6, 4)), arg6, 6, "XmClipboardStartCopy", "(XmCutPasteProc widget data priv reason)");
  xm_protect(arg6);
  if (XEN_PROCEDURE_P(xm_XmCutPasteProc)) xm_unprotect(xm_XmCutPasteProc);
  xm_XmCutPasteProc = arg6;
  val = XmClipboardStartCopy(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_XmString(arg3), 
			     XEN_TO_C_Time(arg4), XEN_TO_C_Widget(arg5), (XmCutPasteProc)gxm_XmCutPasteProc, &id);
  return(XEN_LIST_2(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(id)));
}

static XEN xm_XmVoidProc = XEN_FALSE;
static void gxm_XmVoidProc(Widget w, int *data, int *privater, int *reason)
{
  XEN_CALL_4(xm_XmVoidProc, 
	     C_TO_XEN_Widget(w),
	     C_TO_XEN_ULONG(*data),
	     C_TO_XEN_ULONG(*privater),
	     C_TO_XEN_INT(*reason),
	     c__FUNCTION__);
}

#ifndef LESSTIF_VERSION
static XEN gxm_XmClipboardBeginCopy(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmClipboardBeginCopy "int XmClipboardBeginCopy(display, window, XmString label, widget, callback)"
  /* DIFF: XmClipboardBeinCopy omits and returns last arg
   */
  long id;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmClipboardBeginCopy", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XmClipboardBeginCopy", "Window");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg3), arg3, 3, "XmClipboardBeginCopy", "XmString");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg4), arg4, 4, "XmClipboardBeginCopy", "Widget");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg5) && (XEN_REQUIRED_ARGS_OK(arg5, 4)), arg5, 5, "XmClipboardBeginCopy", "(XmVoidProc widget data priv reason)");
  xm_protect(arg5);
  if (XEN_PROCEDURE_P(xm_XmVoidProc)) xm_unprotect(xm_XmVoidProc);
  xm_XmVoidProc = arg5;
  val = XmClipboardBeginCopy(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
			     XEN_TO_C_XmString(arg3), XEN_TO_C_Widget(arg4), 
			     gxm_XmVoidProc, &id);
  return(XEN_LIST_2(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(id)));
}
#endif

#define XEN_ScaleWidget_P(Arg) (XEN_Widget_P(Arg) && XmIsScale(XEN_TO_C_Widget(Arg)))

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmScaleSetTicks(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XmScaleSetTicks "void XmScaleSetTicks(Widget scale, int big_every, Cardinal num_medium, Cardinal num_small, Dimension size_big, \
Dimension size_medium, Dimension size_small) controls Scale tick marks"
  XEN_ASSERT_TYPE(XEN_ScaleWidget_P(arg1), arg1, 1, "XmScaleSetTicks", "Scale Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmScaleSetTicks", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmScaleSetTicks", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XmScaleSetTicks", "int");
  XEN_ASSERT_TYPE(XEN_Dimension_P(arg5), arg5, 5, "XmScaleSetTicks", "Dimension");
  XEN_ASSERT_TYPE(XEN_Dimension_P(arg6), arg6, 6, "XmScaleSetTicks", "Dimension");
  XEN_ASSERT_TYPE(XEN_Dimension_P(arg7), arg7, 7, "XmScaleSetTicks", "Dimension");
  XmScaleSetTicks(XEN_TO_C_Widget(arg1), 
		  XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), 
		  XEN_TO_C_Dimension(arg5), XEN_TO_C_Dimension(arg6), XEN_TO_C_Dimension(arg7));
  return(XEN_FALSE);
}
#endif
#endif

static XEN gxm_XmCreateScale(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateScale "Widget XmCreateScale(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Scale widget creation function"
  return(gxm_new_widget("XmCreateScale", XmCreateScale, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmScaleGetValue(XEN arg1)
{
  #define H_XmScaleGetValue "void XmScaleGetValue(Widget widget) returns the current (scale) slider position"
  /* DIFF: XmScaleGetValue omits and returns arg2
   */
  int val;
  XEN_ASSERT_TYPE(XEN_ScaleWidget_P(arg1), arg1, 1, "XmScaleGetValue", "Scale Widget");
  XmScaleGetValue(XEN_TO_C_Widget(arg1), &val);
  return(C_TO_XEN_INT(val));
}

static XEN gxm_XmScaleSetValue(XEN arg1, XEN arg2)
{
  #define H_XmScaleSetValue "void XmScaleSetValue(Widget widget, int value) sets a Scale slider value"
  XEN_ASSERT_TYPE(XEN_ScaleWidget_P(arg1), arg1, 1, "XmScaleSetValue", "Scale Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmScaleSetValue", "int");
  XmScaleSetValue(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

#define XEN_ContainerWidget_P(Arg) (XEN_Widget_P(Arg) && XmIsContainer(XEN_TO_C_Widget(Arg)))

#if MOTIF_2
static XEN gxm_XmContainerPasteLink(XEN arg1)
{
  #define H_XmContainerPasteLink "Boolean XmContainerPasteLink(Widget container) container function to insert links from the clipboard"
  XEN_ASSERT_TYPE(XEN_ContainerWidget_P(arg1), arg1, 1, "XmContainerPasteLink", "Container Widget");
  return(C_TO_XEN_BOOLEAN(XmContainerPasteLink(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmContainerCopyLink(XEN arg1, XEN arg2)
{
  #define H_XmContainerCopyLink "Boolean XmContainerCopyLink(Widget container, Time timestamp) Container function to copy links to the clipboard"
  XEN_ASSERT_TYPE(XEN_ContainerWidget_P(arg1), arg1, 1, "XmContainerCopyLink", "Container Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmContainerCopyLink", "Time");
  return(C_TO_XEN_BOOLEAN(XmContainerCopyLink(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2))));
}

static XEN gxm_XmContainerPaste(XEN arg1)
{
  #define H_XmContainerPaste "Boolean XmContainerPaste(Widget container) Container function to insert items from the clipboard"
  XEN_ASSERT_TYPE(XEN_ContainerWidget_P(arg1), arg1, 1, "XmContainerPaste", "Container Widget");
  return(C_TO_XEN_BOOLEAN(XmContainerPaste(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmContainerCopy(XEN arg1, XEN arg2)
{
  #define H_XmContainerCopy "Boolean XmContainerCopy(Widget container, Time timestamp) Container function to copy primary selection to the clipboard"
  XEN_ASSERT_TYPE(XEN_ContainerWidget_P(arg1), arg1, 1, "XmContainerCopy", "Container Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmContainerCopy", "Time");
  return(C_TO_XEN_BOOLEAN(XmContainerCopy(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2))));
}

static XEN gxm_XmContainerCut(XEN arg1, XEN arg2)
{
  #define H_XmContainerCut "Boolean XmContainerCut(Widget container, Time timestamp) Container function to move items to the clipboard"
  XEN_ASSERT_TYPE(XEN_ContainerWidget_P(arg1), arg1, 1, "XmContainerCut", "Container Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XmContainerCut", "Time");
  return(C_TO_XEN_BOOLEAN(XmContainerCut(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2))));
}

static XEN gxm_XmContainerReorder(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmContainerReorder "void XmContainerReorder(Widget container, WidgetList widgets, int num_widgets) Container function to reorder children"
  /* DIFF: XmContainerReorder arg2 is list of Widgets
   */
  WidgetList ws;
  int len;
  XEN_ASSERT_TYPE(XEN_ContainerWidget_P(arg1), arg1, 1, "XmContainerReorder", "Container Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XmContainerReorder", "WidgetList");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmContainerReorder", "int");
  len = XEN_TO_C_INT(arg3);
  if (len > 0)
    {
      ws = XEN_TO_C_Widgets(arg2, len);
      XmContainerReorder(XEN_TO_C_Widget(arg1), ws, len);
      if (ws) FREE(ws);
    }
  return(XEN_FALSE);
}

static XEN gxm_XmContainerRelayout(XEN arg1)
{
  #define H_XmContainerRelayout "void XmContainerRelayout(Widget container) Container relayout function"
  XEN_ASSERT_TYPE(XEN_ContainerWidget_P(arg1), arg1, 1, "XmContainerRelayout", "Container Widget");
  XmContainerRelayout(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmContainerGetItemChildren(XEN arg1, XEN arg2)
{
  #define H_XmContainerGetItemChildren "int XmContainerGetItemChildren(Widget container, Widget item) \
returns a list of all children of an item"
  /* DIFF: XmContainerGetItemChildren omits arg3, rtns widget list
   */
  Widget *ws;
  int len;
  XEN_ASSERT_TYPE(XEN_ContainerWidget_P(arg1), arg1, 1, "XmContainerGetItemChildren", "Container Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2), arg2, 2, "XmContainerGetItemChildren", "Widget");
  len = XmContainerGetItemChildren(XEN_TO_C_Widget(arg1), XEN_TO_C_Widget(arg2), &ws);
  return(C_TO_XEN_Widgets(ws, len));
}

static XEN gxm_XmCreateContainer(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateContainer "Widget XmCreateContainer(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Container creation function"
  return(gxm_new_widget("XmCreateContainer", XmCreateContainer, arg1, arg2, arg3, arg4));
}
#endif

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XmRemoveFromPostFromList(XEN arg1, XEN arg2)
{
  #define H_XmRemoveFromPostFromList "void XmRemoveFromPostFromList(Widget menu, Widget post_from_widget) \
disables a menu for a particular widget"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmRemoveFromPostFromList", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2), arg2, 2, "XmRemoveFromPostFromList", "Widget");
  XmRemoveFromPostFromList(XEN_TO_C_Widget(arg1), XEN_TO_C_Widget(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmAddToPostFromList(XEN arg1, XEN arg2)
{
  #define H_XmAddToPostFromList "void XmAddToPostFromList(Widget menu, Widget post_from_widget) \
makes a menu accessible from more than one widget"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmAddToPostFromList", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2), arg2, 2, "XmAddToPostFromList", "Widget");
  XmAddToPostFromList(XEN_TO_C_Widget(arg1), XEN_TO_C_Widget(arg2));
  return(XEN_FALSE);
}
#endif

static XEN gxm_XmGetTearOffControl(XEN arg1)
{
  #define H_XmGetTearOffControl "Widget XmGetTearOffControl(Widget menu) obtains the widget ID for the tear-off control in a menu"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmGetTearOffControl", "Widget");
  return(C_TO_XEN_Widget(XmGetTearOffControl(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmGetPostedFromWidget(XEN arg1)
{
  #define H_XmGetPostedFromWidget "Widget XmGetPostedFromWidget(Widget menu) returns the widget from which a menu was posted"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmGetPostedFromWidget", "Widget");
  return(C_TO_XEN_Widget(XmGetPostedFromWidget(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmCreatePulldownMenu(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreatePulldownMenu "Widget XmCreatePulldownMenu(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A RowColumn widget creation function"
  return(gxm_new_widget("XmCreatePulldownMenu", XmCreatePulldownMenu, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreatePopupMenu(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreatePopupMenu "Widget XmCreatePopupMenu(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A RowColumn widget creation function"
  return(gxm_new_widget("XmCreatePopupMenu", XmCreatePopupMenu, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateMenuBar(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateMenuBar "Widget XmCreateMenuBar(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A RowColumn widget creation function"
  return(gxm_new_widget("XmCreateMenuBar", XmCreateMenuBar, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmOptionButtonGadget(XEN arg1)
{
  #define H_XmOptionButtonGadget "Widget XmOptionButtonGadget(Widget option_menu) obtains \
the widget ID for the CascadeButtonGadget in an OptionMenu"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmOptionButtonGadget", "Widget");
  return(C_TO_XEN_Widget(XmOptionButtonGadget(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmOptionLabelGadget(XEN arg1)
{
  #define H_XmOptionLabelGadget "Widget XmOptionLabelGadget(Widget option_menu) obtains the \
widget ID for the LabelGadget in an OptionMenu"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmOptionLabelGadget", "Widget");
  return(C_TO_XEN_Widget(XmOptionLabelGadget(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XmCreateOptionMenu(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateOptionMenu "Widget XmCreateOptionMenu(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A RowColumn widget creation function"
  return(gxm_new_widget("XmCreateOptionMenu", XmCreateOptionMenu, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateRadioBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateRadioBox "Widget XmCreateRadioBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A RowColumn widget creation function"
  return(gxm_new_widget("XmCreateRadioBox", XmCreateRadioBox, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateWorkArea(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateWorkArea "Widget XmCreateWorkArea(Widget parent, String name, ArgList arglist, Cardinal argcount) \
creates a RowColumn WorkArea"
  return(gxm_new_widget("XmCreateWorkArea", XmCreateWorkArea, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateRowColumn(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateRowColumn "Widget XmCreateRowColumn(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The RowColumn widget creation function"
  return(gxm_new_widget("XmCreateRowColumn", XmCreateRowColumn, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmMenuPosition(XEN arg1, XEN arg2)
{
  #define H_XmMenuPosition "void XmMenuPosition(Widget menu, XButtonPressedEvent *event) positions a Popup menu pane"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmMenuPosition", "Widget");
  XEN_ASSERT_TYPE(XEN_XButtonEvent_P(arg2), arg2, 2, "XmMenuPosition", "XButtonPressedEvent*");
  XmMenuPosition(XEN_TO_C_Widget(arg1), XEN_TO_C_XButtonEvent(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmCreateCommandDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateCommandDialog "Widget XmCreateCommandDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Command CommandDialog creation function"
  return(gxm_new_widget("XmCreateCommandDialog", XmCreateCommandDialog, arg1, arg2, arg3, arg4));
}

#define XEN_CommandWidget_P(Arg) (XEN_Widget_P(Arg) && XmIsCommand(XEN_TO_C_Widget(Arg)))

static XEN gxm_XmCommandError(XEN arg1, XEN arg2)
{
  #define H_XmCommandError "void XmCommandError(Widget widget, XmString error) Command function that displays an error message"
  XEN_ASSERT_TYPE(XEN_CommandWidget_P(arg1), arg1, 1, "XmCommandError", "Command Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmCommandError", "XmString");
  XmCommandError(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmCommandAppendValue(XEN arg1, XEN arg2)
{
  #define H_XmCommandAppendValue "void XmCommandAppendValue(Widget widget, XmString command) \
appends the passed XmString to the end of the string displayed in the command area of the widget"
  XEN_ASSERT_TYPE(XEN_CommandWidget_P(arg1), arg1, 1, "XmCommandAppendValue", "Command Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmCommandAppendValue", "XmString");
  XmCommandAppendValue(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmCommandSetValue(XEN arg1, XEN arg2)
{
  #define H_XmCommandSetValue "void XmCommandSetValue(Widget widget, XmString command) A Command function that replaces a displayed string"
  XEN_ASSERT_TYPE(XEN_CommandWidget_P(arg1), arg1, 1, "XmCommandSetValue", "Command Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmCommandSetValue", "XmString");
  XmCommandSetValue(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmCommandGetChild(XEN arg1, XEN arg2)
{
  #define H_XmCommandGetChild "Widget XmCommandGetChild(Widget widget, unsigned char child) A Command function that is used to access a component"
  XEN_ASSERT_TYPE(XEN_CommandWidget_P(arg1), arg1, 1, "XmCommandGetChild", "Command Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XmCommandGetChild", "unsigned int");
  return(C_TO_XEN_Widget(XmCommandGetChild(XEN_TO_C_Widget(arg1), XEN_TO_C_ULONG(arg2))));
}

static XEN gxm_XmCreateCommand(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateCommand "Widget XmCreateCommand(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Command widget creation function"
  return(gxm_new_widget("XmCreateCommand", XmCreateCommand, arg1, arg2, arg3, arg4));
}

#if MOTIF_2

#define XEN_ComboBoxWidget_P(Arg) (XEN_Widget_P(Arg) && XmIsComboBox(XEN_TO_C_Widget(Arg)))

static XEN gxm_XmComboBoxUpdate(XEN arg1)
{
  #define H_XmComboBoxUpdate "void XmComboBoxUpdate(Widget widget) A ComboBox function that resynchronizes data"
  XEN_ASSERT_TYPE(XEN_ComboBoxWidget_P(arg1), arg1, 1, "XmComboBoxUpdate", "ComboBox Widget");
  XmComboBoxUpdate(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmComboBoxSetItem(XEN arg1, XEN arg2)
{
  #define H_XmComboBoxSetItem "void XmComboBoxSetItem(Widget w, XmString item) set an item in the XmComboBox list"
  XEN_ASSERT_TYPE(XEN_ComboBoxWidget_P(arg1), arg1, 1, "XmComboBoxSetItem", "ComboBox Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmComboBoxSetItem", "XmString");
  XmComboBoxSetItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmComboBoxSelectItem(XEN arg1, XEN arg2)
{
  #define H_XmComboBoxSelectItem "void XmComboBoxSelectItem(Widget w, XmString item) select a XmComboBox item"
  XEN_ASSERT_TYPE(XEN_ComboBoxWidget_P(arg1), arg1, 1, "XmComboBoxSelectItem", "ComboBox Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmComboBoxSelectItem", "XmString");
  XmComboBoxSelectItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmComboBoxDeletePos(XEN arg1, XEN arg2)
{
  #define H_XmComboBoxDeletePos "void XmComboBoxDeletePos(Widget w, int pos)  Delete a XmComboBox item"
  XEN_ASSERT_TYPE(XEN_ComboBoxWidget_P(arg1), arg1, 1, "XmComboBoxDeletePos", "ComboBox Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmComboBoxDeletePos", "int");
  XmComboBoxDeletePos(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmComboBoxAddItem(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmComboBoxAddItem "void XmComboBoxAddItem(Widget w, XmString item, int pos, Boolean unique) add an item to the ComboBox widget"
  XEN_ASSERT_TYPE(XEN_ComboBoxWidget_P(arg1), arg1, 1, "XmComboBoxAddItem", "ComboBox Widget");
  XEN_ASSERT_TYPE(XEN_XmString_P(arg2), arg2, 2, "XmComboBoxAddItem", "XmString");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XmComboBoxAddItem", "int");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg4), arg4, 4, "XmComboBoxAddItem", "Boolean");
  XmComboBoxAddItem(XEN_TO_C_Widget(arg1), XEN_TO_C_XmString(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_BOOLEAN(arg4));
  return(XEN_FALSE);
}

static XEN gxm_XmCreateDropDownList(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateDropDownList "Widget XmCreateDropDownList(Widget parent, String name, ArgList arglist, Cardinal arg_count) \
The Drop-down list ComboBox widget creation function"
  return(gxm_new_widget("XmCreateDropDownList", XmCreateDropDownList, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateDropDownComboBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateDropDownComboBox "Widget XmCreateDropDownComboBox(Widget parent, String name, ArgList arglist, Cardinal arg_count) \
The Drop-down ComboBox widget creation function"
  return(gxm_new_widget("XmCreateDropDownComboBox", XmCreateDropDownComboBox, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateComboBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateComboBox "Widget XmCreateComboBox(Widget parent, String name, ArgList arglist, Cardinal arg_count) \
The default ComboBox widget creation function"
  return(gxm_new_widget("XmCreateComboBox", XmCreateComboBox, arg1, arg2, arg3, arg4));
}
#endif

static XEN gxm_XmCreatePushButton(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreatePushButton "Widget XmCreatePushButton(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The PushButton widget creation function"
  return(gxm_new_widget("XmCreatePushButton", XmCreatePushButton, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreatePushButtonGadget(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreatePushButtonGadget "Widget XmCreatePushButtonGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The PushButtonGadget creation function"
  return(gxm_new_widget("XmCreatePushButtonGadget", XmCreatePushButtonGadget, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCascadeButtonHighlight(XEN arg1, XEN arg2)
{
  #define H_XmCascadeButtonHighlight "void XmCascadeButtonHighlight(Widget cascadeButton, Boolean highlight) A CascadeButton and \
CascadeButtonGadget function that sets the highlight state"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmCascadeButtonHighlight", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XmCascadeButtonHighlight", "boolean");
  XmCascadeButtonHighlight(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmCreateCascadeButton(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateCascadeButton "Widget XmCreateCascadeButton(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The CascadeButton widget creation function"
  return(gxm_new_widget("XmCreateCascadeButton", XmCreateCascadeButton, arg1, arg2, arg3, arg4));
}

static void gxm_ProtocolProc(Widget w, XtPointer context, XtPointer info)
{
  XEN descr = (XEN)context;
  XEN_CALL_3(XEN_CADR(descr),
	     C_TO_XEN_Widget(w),
	     XEN_CADDR(descr),
	     C_TO_XEN_ULONG(info), /* what's this? */
	     c__FUNCTION__);
}

#define C_TO_XEN_XM_ProtocolHook(Code, Context, PropertyAtom, ProtocolAtom) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("ProtocolHook"), Code, Context, PropertyAtom, ProtocolAtom)
#define XM_ProtocolHook_P(Arg) WRAP_P("ProtocolHook", Arg)

#define C_TO_XEN_XM_ProtocolProc(Code, Context, PropertyAtom, ProtocolAtom) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("ProtocolProc"), Code, Context, PropertyAtom, ProtocolAtom)
#define XM_ProtocolProc_P(Arg) WRAP_P("ProtocolProc", Arg)

static XEN gxm_XmSetProtocolHooks(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XmSetProtocolHooks "void XmSetProtocolHooks(Widget shell, Atom property, Atom protocol, XtCallbackProc prehook, \
XtPointer pre_closure, XtCallbackProc posthook, XtPointer post_closure) A VendorShell function that allows preactions and postactions \
to be executed when a protocol message is received from MWM"
  XEN descr1, descr2;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmSetProtocolHooks", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XmSetProtocolHooks", "Atom");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XmSetProtocolHooks", "Atom");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 3)), arg4, 4, "XmSetProtocolHooks", "(XtCallbackProc widget data callb)");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg6) && (XEN_REQUIRED_ARGS_OK(arg6, 3)), arg6, 6, "XmSetProtocolHooks", "(XtCallbackProc widget data callb)");
  descr1 = C_TO_XEN_XM_ProtocolHook(arg4, arg5, arg2, arg3);
  descr2 = C_TO_XEN_XM_ProtocolHook(arg6, arg7, arg2, arg3);
  xm_protect(descr1);
  xm_protect(descr2);
  XmSetProtocolHooks(XEN_TO_C_Widget(arg1), XEN_TO_C_Atom(arg2), XEN_TO_C_Atom(arg3),
		     gxm_ProtocolProc, (XtPointer)descr1,
		     gxm_ProtocolProc, (XtPointer)descr2);
  return(XEN_FALSE);
}

static XEN gxm_XmDeactivateProtocol(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmDeactivateProtocol "void XmDeactivateProtocol(Widget shell, Atom property, Atom protocol) \
deactivates a protocol without removing it"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmDeactivateProtocol", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XmDeactivateProtocol", "Atom");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XmDeactivateProtocol", "Atom");
  XmDeactivateProtocol(XEN_TO_C_Widget(arg1), XEN_TO_C_Atom(arg2), XEN_TO_C_Atom(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XmActivateProtocol(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmActivateProtocol "oid XmActivateProtocol(Widget shell, Atom property, Atom protocol) activates a protocol"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmActivateProtocol", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XmActivateProtocol", "Atom");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XmActivateProtocol", "Atom");
  XmActivateProtocol(XEN_TO_C_Widget(arg1), XEN_TO_C_Atom(arg2), XEN_TO_C_Atom(arg3));
  return(XEN_FALSE);
}

static bool unprotect_protocolproc(XEN val, int loc, unsigned long udescr)
{
  XEN descr = (XEN)udescr; /* ('protocolproc func data propatom protoatom) */
  return((XM_ProtocolProc_P(val)) &&
	 (XEN_TO_C_ULONG(XEN_CADR(val)) == XEN_TO_C_ULONG(XEN_CADR(descr))) &&
	 (XEN_TO_C_ULONG(XEN_CADDR(val)) == XEN_TO_C_ULONG(XEN_CADDR(descr))) &&
	 (XEN_TO_C_Atom(XEN_LIST_REF(val, 3)) == XEN_TO_C_Atom(XEN_LIST_REF(val,3))) &&
	 (XEN_TO_C_Atom(XEN_LIST_REF(val, 4)) == XEN_TO_C_Atom(XEN_LIST_REF(val,4))));
}

static XEN gxm_XmRemoveProtocolCallback(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmRemoveProtocolCallback "void XmRemoveProtocolCallback(Widget shell, Atom property, Atom protocol, XtCallbackProc callback, \
XtPointer closure) removes a callback from the internal list"
  XEN descr;
  int loc, dloc;
  XEN *velts;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmRemoveProtocolCallback", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XmRemoveProtocolCallback", "Atom");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XmRemoveProtocolCallback", "Atom"); 
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 3)), arg4, 4, "XmRemoveProtocolCallback", "XtCallbackProc (3 args)");
  descr = C_TO_XEN_XM_ProtocolProc(arg4, arg5, arg2, arg3);
  dloc = xm_protect(descr);
  loc = map_over_protected_elements(unprotect_protocolproc, (unsigned long)descr);
  velts = XEN_VECTOR_ELEMENTS(xm_protected);
  XmRemoveProtocolCallback(XEN_TO_C_Widget(arg1), 
			   XEN_TO_C_Atom(arg2), 
			   XEN_TO_C_Atom(arg3), 
			   gxm_ProtocolProc,
			   (XtPointer)(velts[loc])); /* this was the original tag passed in */
  /* now unprotect the proc and our descr */
  xm_unprotect_at(dloc);
  xm_unprotect_at(loc);
  return(XEN_FALSE);
}

static XEN gxm_XmAddProtocolCallback(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmAddProtocolCallback "void XmAddProtocolCallback(Widget shell, Atom property, Atom protocol, XtCallbackProc callback, \
XtPointer closure) adds client callbacks for a protocol"
  XEN descr;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmAddProtocolCallback", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XmAddProtocolCallback", "Atom");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XmAddProtocolCallback", "Atom");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 3)), arg4, 4, "XmAddProtocolCallback", "(XtCallbackProc widget data callb)");
  descr = C_TO_XEN_XM_ProtocolProc(arg4, arg5, arg2, arg3);
  xm_protect(descr);
  XmAddProtocolCallback(XEN_TO_C_Widget(arg1), 
			XEN_TO_C_Atom(arg2),
			XEN_TO_C_Atom(arg3),
			gxm_ProtocolProc,
			(XtPointer)descr);
  return(XEN_FALSE);
}

static XEN gxm_XmRemoveProtocols(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmRemoveProtocols "void XmRemoveProtocols(Widget shell, Atom property, Atom *protocols, Cardinal num_protocols) \
removes the protocols from the protocol manager and deallocates the internal tables"
  /* DIFF: XmRemoveProtocols takes list of Atoms
   */
  Atom *outs;
  int len;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmRemoveProtocols", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XmRemoveProtocols", "Atom");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XmRemoveProtocols", "list of Atom");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg4), arg4, 4, "XmRemoveProtocols", "int");
  if (XEN_INTEGER_P(arg4)) len = XEN_TO_C_INT(arg4); else len = XEN_LIST_LENGTH(arg3);
  if (len <= 0) return(XEN_FALSE);
  outs = XEN_TO_C_Atoms(arg3, len);
  XmRemoveProtocols(XEN_TO_C_Widget(arg1), XEN_TO_C_Atom(arg2), outs, len);
  FREE(outs);
  return(XEN_FALSE);
}

static XEN gxm_XmAddProtocols(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmAddProtocols "void XmAddProtocols(Widget shell, Atom property, Atom *protocols, Cardinal num_protocols) \
adds the protocols to the protocol manager and allocates the internal tables"
  /* DIFF: XmAddProtocols takes list of Atoms
   */
  Atom *outs;
  int len;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmAddProtocols", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XmAddProtocols", "Atom");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XmAddProtocols", "list of Atom");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg4), arg4, 4, "XmAddProtocols", "int");
  if (XEN_INTEGER_P(arg4)) len = XEN_TO_C_INT(arg4); else len = XEN_LIST_LENGTH(arg3);
  if (len <= 0) return(XEN_FALSE);
  outs = XEN_TO_C_Atoms(arg3, len);
  XmAddProtocols(XEN_TO_C_Widget(arg1), XEN_TO_C_Atom(arg2), outs, len);
  FREE(outs);
  return(XEN_FALSE);
}

static XEN gxm_XmCascadeButtonGadgetHighlight(XEN arg1, XEN arg2)
{
  #define H_XmCascadeButtonGadgetHighlight "void XmCascadeButtonGadgetHighlight(Widget cascadeButtonGadget, Boolean highlight) \
A CascadeButtonGadget function that sets the highlight state"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmCascadeButtonGadgetHighlight", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XmCascadeButtonGadgetHighlight", "boolean");
  XmCascadeButtonGadgetHighlight(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XmCreateCascadeButtonGadget(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateCascadeButtonGadget "Widget XmCreateCascadeButtonGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The CascadeButtonGadget creation function"
  return(gxm_new_widget("XmCreateCascadeButtonGadget", XmCreateCascadeButtonGadget, arg1, arg2, arg3, arg4));
}

#if MOTIF_2 && HAVE_XM_XP
static XEN gxm_XmRedisplayWidget(XEN arg1)
{
  #define H_XmRedisplayWidget "voidXmRedisplayWidget(Widgetwidget) Synchronously activates the expose method of a widget to draw its content"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmRedisplayWidget", "Widget");
  XmRedisplayWidget(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XmPrintPopupPDM(XEN arg1, XEN arg2)
{
  #define H_XmPrintPopupPDM "XtEnum XmPrintPopupPDM(Widgetprint_shell, Widget video_transient_for) Send a notification for the PDM to be popped up"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmPrintPopupPDM", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2), arg2, 2, "XmPrintPopupPDM", "Widget");
  return(C_TO_XEN_INT(XmPrintPopupPDM(XEN_TO_C_Widget(arg1), XEN_TO_C_Widget(arg2))));
}

static void gxm_XPFinishProc(Display *display, XPContext context, XPGetDocStatus status, XPointer client_data);

static XEN gxm_XmPrintToFile(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmPrintToFile "XtEnumXmPrintToFile(Display*dpy, Stringfilename, XPFinishProcfinish_proc, XtPointerclient_data) \
retrieves and saves data that would normally be printed by the X Print Server"
  XEN descr;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmPrintToFile", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmPrintToFile", "char*");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg3) && (XEN_REQUIRED_ARGS_OK(arg3, 4)), arg3, 3, "XmPrintToFile", "XPFinishProc (4 args)");
  descr = XEN_LIST_3(XEN_FALSE, arg3, arg4);
  xm_protect(descr);
  return(C_TO_XEN_INT(XmPrintToFile(XEN_TO_C_Display(arg1), 
				    XEN_TO_C_STRING(arg2), 
				    (XPFinishProc)gxm_XPFinishProc,
				    (char *)descr)));
}

static XEN gxm_XmPrintSetup(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XmPrintSetup "Widget XmPrintSetup(Widget video_widget, Screen *print_screen, String print_shell_name, ArgList args, Cardinal num_args) \
setup and create a Print Shell widget"
  Widget w;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmPrintSetup", "Widget");
  XEN_ASSERT_TYPE(XEN_Screen_P(arg2), arg2, 2, "XmPrintSetup", "Screen*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XmPrintSetup", "String");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XmPrintSetup", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg5), arg5, 5, "XmPrintSetup", "int");
  {
    Arg *args;
    int arglen;
    args = XEN_TO_C_Args(arg4);
    arglen = XEN_TO_C_INT_DEF(arg5, arg4);
    w = XmPrintSetup(XEN_TO_C_Widget(arg1), 
		     XEN_TO_C_Screen(arg2), 
		     XEN_TO_C_STRING(arg3), 
		     args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));
}
#endif

static XEN gxm_XmCreateBulletinBoardDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateBulletinBoardDialog "Widget XmCreateBulletinBoardDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The BulletinBoard BulletinBoardDialog creation function"
  return(gxm_new_widget("XmCreateBulletinBoardDialog", XmCreateBulletinBoardDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateBulletinBoard(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateBulletinBoard "Widget XmCreateBulletinBoard(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The BulletinBoard widget creation function"
  return(gxm_new_widget("XmCreateBulletinBoard", XmCreateBulletinBoard, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreatePanedWindow(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreatePanedWindow "Widget XmCreatePanedWindow(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The PanedWindow widget creation function"
  return(gxm_new_widget("XmCreatePanedWindow", XmCreatePanedWindow, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmGetAtomName(XEN arg1, XEN arg2)
{
  char *str;
  XEN res;
  #define H_XmGetAtomName "String XmGetAtomName(Display *display, Atom atom)  returns the string representation for an atom"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmGetAtomName", "Display*");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XmGetAtomName", "Atom");
  str = XmGetAtomName(XEN_TO_C_Display(arg1), XEN_TO_C_Atom(arg2));
  res = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  return(res);
}

static XEN gxm_XmInternAtom(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XmInternAtom "A macro that returns an atom for a given name"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XmInternAtom", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XmInternAtom", "String");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XmInternAtom", "Boolean");
  return(C_TO_XEN_Atom(XmInternAtom(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2), XEN_TO_C_BOOLEAN(arg3))));
}

#if MOTIF_2
static XEN gxm_XmNotebookGetPageInfo(XEN arg1, XEN arg2)
{
  #define H_XmNotebookGetPageInfo "XmNotebookPageStatus XmNotebookGetPageInfo(Widget notebook, int page_number) \
A Notebook function that returns page information"
  XmNotebookPageInfo *info;
  int err;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmNotebookGetPageInfo", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XmNotebookGetPageInfo", "int");
  info = (XmNotebookPageInfo *)CALLOC(1, sizeof(XmNotebookPageInfo));
  err = XmNotebookGetPageInfo(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), info);
  return(XEN_LIST_2(C_TO_XEN_INT(err), C_TO_XEN_XmNotebookPageInfo(info)));
}

static XEN gxm_XmCreateNotebook(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateNotebook "void XmCreateNotebook(Widget parent, String name, ArgList arglist, Cardinal argcount) The Notebook widget creation function"
  return(gxm_new_widget("XmCreateNotebook", XmCreateNotebook, arg1, arg2, arg3, arg4));
}
#endif

static XEN gxm_XmCreateArrowButton(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateArrowButton "Widget XmCreateArrowButton(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ArrowButton widget creation function"
  return(gxm_new_widget("XmCreateArrowButton", XmCreateArrowButton, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateArrowButtonGadget(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateArrowButtonGadget "Widget XmCreateArrowButtonGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ArrowButtonGadget creation function"
  return(gxm_new_widget("XmCreateArrowButtonGadget", XmCreateArrowButtonGadget, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmMessageBoxGetChild(XEN arg1, XEN arg2)
{
  #define H_XmMessageBoxGetChild "Widget XmMessageBoxGetChild(Widget widget, unsigned char child) A MessageBox function that is used to access a component"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XmMessageBoxGetChild", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XmMessageBoxGetChild", "unsigned int");
  return(C_TO_XEN_Widget(XmMessageBoxGetChild(XEN_TO_C_Widget(arg1), XEN_TO_C_ULONG(arg2))));
}

static XEN gxm_XmCreateTemplateDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateTemplateDialog "Widget XmCreateTemplateDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A MessageBox TemplateDialog creation function"
  return(gxm_new_widget("XmCreateTemplateDialog", XmCreateTemplateDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateWorkingDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateWorkingDialog "Widget XmCreateWorkingDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox WorkingDialog creation function"
  return(gxm_new_widget("XmCreateWorkingDialog", XmCreateWorkingDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateWarningDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateWarningDialog "Widget XmCreateWarningDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox WarningDialog creation function"
  return(gxm_new_widget("XmCreateWarningDialog", XmCreateWarningDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateQuestionDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateQuestionDialog "Widget XmCreateQuestionDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox QuestionDialog creation function"
  return(gxm_new_widget("XmCreateQuestionDialog", XmCreateQuestionDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateInformationDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateInformationDialog "Widget XmCreateInformationDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox InformationDialog creation function"
  return(gxm_new_widget("XmCreateInformationDialog", XmCreateInformationDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateErrorDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateErrorDialog "Widget XmCreateErrorDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox ErrorDialog creation function"
  return(gxm_new_widget("XmCreateErrorDialog", XmCreateErrorDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateMessageDialog(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateMessageDialog "Widget XmCreateMessageDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox MessageDialog creation function"
  return(gxm_new_widget("XmCreateMessageDialog", XmCreateMessageDialog, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmCreateMessageBox(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XmCreateMessageBox "Widget XmCreateMessageBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox widget creation function"
  return(gxm_new_widget("XmCreateMessageBox", XmCreateMessageBox, arg1, arg2, arg3, arg4));
}

static XEN gxm_XmIsMessageBox(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsMessageBox", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsMessageBox(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsArrowButtonGadget(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsArrowButtonGadget", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsArrowButtonGadget(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsArrowButton(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsArrowButton", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsArrowButton(XEN_TO_C_Widget(arg))));
}

#if MOTIF_2
static XEN gxm_XmIsNotebook(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsNotebook", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsNotebook(XEN_TO_C_Widget(arg))));
}
#endif

static XEN gxm_XmIsPanedWindow(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsPanedWindow", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsPanedWindow(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsBulletinBoard(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsBulletinBoard", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsBulletinBoard(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsPrimitive(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsPrimitive", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsPrimitive(XEN_TO_C_Widget(arg))));
}

#if MOTIF_2 && HAVE_XM_XP
static XEN gxm_XmIsPrintShell(XEN arg)
{
  #define H_XmIsPrintShell "Boolean XmIsPrintShell(Widget)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsPrintShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsPrintShell(XEN_TO_C_Widget(arg))));
}
#endif

static XEN gxm_XmIsCascadeButtonGadget(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsCascadeButtonGadget", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsCascadeButtonGadget(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsCascadeButton(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsCascadeButton", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsCascadeButton(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsPushButtonGadget(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsPushButtonGadget", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsPushButtonGadget(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsPushButton(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsPushButton", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsPushButton(XEN_TO_C_Widget(arg))));
}

#if MOTIF_2
static XEN gxm_XmIsComboBox(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsComboBox", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsComboBox(XEN_TO_C_Widget(arg))));
}
#endif

static XEN gxm_XmIsCommand(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsCommand", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsCommand(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsRowColumn(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsRowColumn", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsRowColumn(XEN_TO_C_Widget(arg))));
}

#if MOTIF_2
static XEN gxm_XmIsContainer(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsContainer", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsContainer(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsScreen", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsScreen(XEN_TO_C_Widget(arg))));
}
#endif

static XEN gxm_XmIsScale(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsScale", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsScale(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsScrollBar(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsScrollBar", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsScrollBar(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsDialogShell(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsDialogShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsDialogShell(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsScrolledWindow(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsScrolledWindow", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsScrolledWindow(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsDisplay(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsDisplay", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsDisplay(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsSelectionBox(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsSelectionBox", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsSelectionBox(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsDragContext(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsDragContext", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsDragContext(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsSeparatorGadget(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsSeparatorGadget", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsSeparatorGadget(XEN_TO_C_Widget(arg))));
}

#if MOTIF_2
#ifndef LESSTIF_VERSION
static XEN gxm_XmIsDragIconObjectClass(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsDragIconObjectClass", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsDragIconObjectClass(XEN_TO_C_Widget(arg))));
}
#endif
#endif

static XEN gxm_XmIsSeparator(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsSeparator", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsSeparator(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsDrawingArea(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsDrawingArea", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsDrawingArea(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsDrawnButton(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsDrawnButton", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsDrawnButton(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsDropSiteManager(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsDropSiteManager", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsDropSiteManager(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsDropTransfer(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsDropTransfer", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsDropTransfer(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsTextField(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsTextField", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsTextField(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsFileSelectionBox(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsFileSelectionBox", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsFileSelectionBox(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsText(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsText", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsText(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsForm(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsForm", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsForm(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsFrame(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsFrame", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsFrame(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsGadget(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsGadget", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsGadget(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsToggleButtonGadget(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsToggleButtonGadget", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsToggleButtonGadget(XEN_TO_C_Widget(arg))));
}

#if MOTIF_2
static XEN gxm_XmIsGrabShell(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsGrabShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsGrabShell(XEN_TO_C_Widget(arg))));
}
#endif

static XEN gxm_XmIsToggleButton(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsToggleButton", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsToggleButton(XEN_TO_C_Widget(arg))));
}

#if MOTIF_2
static XEN gxm_XmIsIconGadget(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsIconGadget", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsIconGadget(XEN_TO_C_Widget(arg))));
}

#ifndef LESSTIF_VERSION
static XEN gxm_XmIsIconHeader(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsIconHeader", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsIconHeader(XEN_TO_C_Widget(arg))));
}
#endif
#endif

static XEN gxm_XmIsLabelGadget(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsLabelGadget", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsLabelGadget(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsLabel(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsLabel", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsLabel(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsVendorShell(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsVendorShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsVendorShell(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsList(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsList", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsList(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsMainWindow(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsMainWindow", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsMainWindow(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsManager(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsManager", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsManager(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XmIsMenuShell(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XmIsMenuShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XmIsMenuShell(XEN_TO_C_Widget(arg))));
}


#endif
/* end HAVE_MOTIF */

/* ----------------------------------------------------------------------------------------------------
 *
 *                            *         *
 *                             *       *
 *                              *     *
 *                               *   *
 *                                * *
 *                                 *
 *                                * *
 *                               *   *
 *                              *     *
 *                             *       *
 *                            *         *
 * ----------------------------------------------------------------------------------------------------
 */

static XEN gxm_XXorRegion(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XXorRegion "XXorRegion(sra, srb, dr_return) calculates the difference between the union and intersection of two regions."
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XXorRegion", "Region");
  XEN_ASSERT_TYPE(XEN_Region_P(arg2), arg2, 2, "XXorRegion", "Region");
  XEN_ASSERT_TYPE(XEN_Region_P(arg3), arg3, 3, "XXorRegion", "Region");
  return(C_TO_XEN_INT(XXorRegion(XEN_TO_C_Region(arg1), XEN_TO_C_Region(arg2), XEN_TO_C_Region(arg3))));
}

static XEN gxm_XUnionRegion(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XUnionRegion "XUnionRegion(sra, srb, dr_return) computes the union of two regions."
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XUnionRegion", "Region");
  XEN_ASSERT_TYPE(XEN_Region_P(arg2), arg2, 2, "XUnionRegion", "Region");
  XEN_ASSERT_TYPE(XEN_Region_P(arg3), arg3, 3, "XUnionRegion", "Region");
  return(C_TO_XEN_INT(XUnionRegion(XEN_TO_C_Region(arg1), XEN_TO_C_Region(arg2), XEN_TO_C_Region(arg3))));
}

static XEN gxm_XUnionRectWithRegion(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XUnionRectWithRegion "XUnionRectWithRegion(rectangle, src_region, dest_region_return) updates the destination region from a \
union of the specified rectangle and the specified source region."
  XEN_ASSERT_TYPE(XEN_XRectangle_P(arg1), arg1, 1, "XUnionRectWithRegion", "XRectangle*");
  XEN_ASSERT_TYPE(XEN_Region_P(arg2), arg2, 2, "XUnionRectWithRegion", "Region");
  XEN_ASSERT_TYPE(XEN_Region_P(arg3), arg3, 3, "XUnionRectWithRegion", "Region");
  return(C_TO_XEN_INT(XUnionRectWithRegion(XEN_TO_C_XRectangle(arg1), XEN_TO_C_Region(arg2), XEN_TO_C_Region(arg3))));
}

static XEN gxm_XSubtractRegion(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSubtractRegion "XSubtractRegion(sra, srb, dr_return) subtracts srb from sra and stores the results in dr_return."
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XSubtractRegion", "Region");
  XEN_ASSERT_TYPE(XEN_Region_P(arg2), arg2, 2, "XSubtractRegion", "Region");
  XEN_ASSERT_TYPE(XEN_Region_P(arg3), arg3, 3, "XSubtractRegion", "Region");
  return(C_TO_XEN_INT(XSubtractRegion(XEN_TO_C_Region(arg1), XEN_TO_C_Region(arg2), XEN_TO_C_Region(arg3))));
}

static XEN gxm_XShrinkRegion(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XShrinkRegion "XShrinkRegion(r, dx, dy) reduces the specified region by a specified amount."
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XShrinkRegion", "Region");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XShrinkRegion", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XShrinkRegion", "int");
  return(C_TO_XEN_INT(XShrinkRegion(XEN_TO_C_Region(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3))));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XSetStandardColormap(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XSetStandardColormap "XSetStandardColormap(dpy, window, colormap, atom) sets the standard colormap property"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetStandardColormap", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetStandardColormap", "Window");
  XEN_ASSERT_TYPE(XEN_XStandardColormap_P(arg3), arg3, 3, "XSetStandardColormap", "XStandardColormap*");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg4), arg4, 4, "XSetStandardColormap", "Atom");
  XSetStandardColormap(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_XStandardColormap(arg3), XEN_TO_C_Atom(arg4));
  return(XEN_FALSE);
}
static XEN gxm_XSetStandardProperties(XEN dpy, XEN win, XEN win_name, XEN icon_name, XEN icon_pixmap, XEN argv, XEN argc, XEN hints)
{
  #define H_XSetStandardProperties "XSetStandardProperties(dpy, win, win_name, icon_name, icon_pixmap, argv, argc, hints) sets the standard properties"
  int c_argc = 0;
  char **c_argv = NULL;
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XSetStandardProperties", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(win), win, XEN_ARG_2, "XSetStandardProperties", "Window");
  XEN_ASSERT_TYPE(XEN_STRING_P(win_name) || XEN_NULL_P(win_name) || XEN_FALSE_P(win_name), win_name, XEN_ARG_3, "XSetStandardProperties", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(icon_name) || XEN_NULL_P(icon_name) || XEN_FALSE_P(icon_name), icon_name, XEN_ARG_4, "XSetStandardProperties", "char*");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(icon_pixmap) || XEN_INTEGER_P(icon_pixmap), icon_pixmap, XEN_ARG_5, "XSetStandardProperties", "Pixmap or None");
  XEN_ASSERT_TYPE(XEN_LIST_P(argv), argv, XEN_ARG_6, "XSetStandardProperties", "list of char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(argc), argc, XEN_ARG_7, "XSetStandardProperties", "int");
  XEN_ASSERT_TYPE(XEN_FALSE_P(hints) || XEN_NULL_P(hints) || XEN_XSizeHints_P(hints), hints, XEN_ARG_8, "XSetStandardProperties", "XSizeHints* or false");
  c_argc = XEN_TO_C_INT(argc);
  if (c_argc > 0) c_argv = XEN_TO_C_Strings(argv, c_argc);
  XSetStandardProperties(XEN_TO_C_Display(dpy),
			 XEN_TO_C_Window(win),
			 (XEN_STRING_P(win_name)) ? XEN_TO_C_STRING(win_name): NULL,
			 (XEN_STRING_P(icon_name)) ? XEN_TO_C_STRING(icon_name) : NULL,
			 (XEN_Pixmap_P(icon_pixmap)) ? XEN_TO_C_Pixmap(icon_pixmap) : None,
			 c_argv,
			 c_argc,
			 (XEN_XSizeHints_P(hints)) ? XEN_TO_C_XSizeHints(hints) : NULL);
  if (c_argv) FREE(c_argv);
  return(xen_return_first(XEN_FALSE, argv));
}
#endif
static XEN gxm_XSetWMProperties(XEN dpy, XEN win, XEN win_name, XEN icon_name, XEN argv, XEN argc, XEN normal_hints, XEN wm_hints)
{
  /* last arg omitted -- XClassHint not supported */
  #define H_XSetWMProperties "XSetWMProperties(dpy, win, win_name, icon_name, argv, argc, normal_hints wm_hints) sets the window properties"
  int c_argc = 0;
  char **c_argv = NULL;
  XTextProperty w_name, i_name;
  bool use_w_name = false, use_i_name = false;
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XSetWMProperties", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(win), win, XEN_ARG_2, "XSetWMProperties", "Window");
  XEN_ASSERT_TYPE(XEN_STRING_P(win_name) || XEN_NULL_P(win_name) || XEN_FALSE_P(win_name), win_name, XEN_ARG_3, "XSetWMProperties", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(icon_name) || XEN_NULL_P(icon_name) || XEN_FALSE_P(icon_name), icon_name, XEN_ARG_4, "XSetWMProperties", "char*");
  XEN_ASSERT_TYPE(XEN_LIST_P(argv), argv, XEN_ARG_5, "XSetWMProperties", "list of char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(argc), argc, XEN_ARG_6, "XSetWMProperties", "int");
  XEN_ASSERT_TYPE(XEN_FALSE_P(normal_hints) || XEN_XSizeHints_P(normal_hints), normal_hints, XEN_ARG_7, "XSetWMProperties", "XSizeHints* or false");
  XEN_ASSERT_TYPE(XEN_FALSE_P(wm_hints) || XEN_XWMHints_P(wm_hints), wm_hints, XEN_ARG_8, "XSetWMProperties", "XWMHints* or false");
  c_argc = XEN_TO_C_INT(argc);
  if (c_argc > 0) c_argv = XEN_TO_C_Strings(argv, c_argc);
  if (XEN_STRING_P(win_name))
    {
      char *name;
      use_w_name = true;
      name = XEN_TO_C_STRING(win_name);
      XStringListToTextProperty(&name, 1, &w_name);
    }
  if (XEN_STRING_P(icon_name))
    {
      char *name;
      use_i_name = true; 
      name = XEN_TO_C_STRING(icon_name);
      XStringListToTextProperty(&name, 1, &i_name);
   }
  XSetWMProperties(XEN_TO_C_Display(dpy),
		   XEN_TO_C_Window(win),
		   (use_w_name) ? &w_name : NULL,
		   (use_i_name) ? &i_name : NULL,
		   c_argv,
		   c_argc,
		   XEN_TO_C_XSizeHints(normal_hints),
		   XEN_TO_C_XWMHints(wm_hints),
		   NULL);
  if (c_argv) FREE(c_argv);
  return(xen_return_first(XEN_FALSE, win_name, icon_name, argv));
}

static XEN gxm_XSetRegion(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetRegion "XSetRegion(display, gc, r) sets the clip-mask in the GC to the specified region."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetRegion", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetRegion", "GC");
  XEN_ASSERT_TYPE(XEN_Region_P(arg3), arg3, 3, "XSetRegion", "Region");
  return(C_TO_XEN_INT(XSetRegion(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_Region(arg3))));
}

static XEN gxm_XSetWMHints(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetWMHints "XSetWMHints(display, w, wmhints) sets the window manager hints that include icon information and location, the \
initial state of the window, and whether the application relies on the window manager to get keyboard input."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetWMHints", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetWMHints", "Window");
  XEN_ASSERT_TYPE(XEN_XWMHints_P(arg3), arg3, 3, "XSetWMHints", "XWMHints*");
  return(C_TO_XEN_INT(XSetWMHints(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_XWMHints(arg3))));
}

static XEN gxm_XSetRGBColormaps(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XSetRGBColormaps "void XSetRGBColormaps(display, w, std_colormap, count, property) replaces the RGB colormap definition in the \
specified property on the named window."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetRGBColormaps", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetRGBColormaps", "Window");
  XEN_ASSERT_TYPE(XEN_XStandardColormap_P(arg3), arg3, 3, "XSetRGBColormaps", "XStandardColormap*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XSetRGBColormaps", "int");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg5), arg5, 5, "XSetRGBColormaps", "Atom");
  XSetRGBColormaps(XEN_TO_C_Display(arg1), 
		   XEN_TO_C_Window(arg2), 
		   XEN_TO_C_XStandardColormap(arg3), 
		   XEN_TO_C_INT(arg4), 
		   XEN_TO_C_Atom(arg5));
  return(XEN_FALSE);
}

static XEN gxm_XUniqueContext(void)
{
  #define H_XUniqueContext "XContext XUniqueContext() creates a unique context type that may be used in subsequent calls to XSaveContext."
  return(C_TO_XEN_XContext(XUniqueContext()));
}

static XEN gxm_XSaveContext(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XSaveContext "XSaveContext(dpy, rid, context) saves a context"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSaveContext", "Display*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XSaveContext", "XID");
  XEN_ASSERT_TYPE(XEN_XContext_P(arg3), arg3, 3, "XSaveContext", "XContext");
  return(C_TO_XEN_INT(XSaveContext(XEN_TO_C_Display(arg1), XEN_TO_C_ULONG(arg2), XEN_TO_C_XContext(arg3), (caddr_t)arg4)));
}

static XEN gxm_XRectInRegion(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XRectInRegion "int XRectInRegion(r, x, y, width, height) returns RectangleIn if the rectangle is entirely in the specified region, \
RectangleOut if the rectangle is entirely out of the specified region, and RectanglePart if the rectangle is partially in the specified region. "
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XRectInRegion", "Region");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XRectInRegion", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XRectInRegion", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XRectInRegion", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XRectInRegion", "unsigned int");
  return(C_TO_XEN_INT(XRectInRegion(XEN_TO_C_Region(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_ULONG(arg4), XEN_TO_C_ULONG(arg5))));
}

static XEN gxm_XPolygonRegion(XEN larg1, XEN arg2, XEN arg3)
{
  #define H_XPolygonRegion "Region XPolygonRegion(points, n, fill_rule) returns a region for the polygon defined by the points list."
  /* DIFF: XPolygonRegion XPoint* arg (arg 1) is list of XPoints
   */
  XPoint *pt, *pt1;
  int i, len;
  Region res;
  XEN arg1;
  XEN_ASSERT_TYPE(XEN_LIST_P(larg1), larg1, 1, "XPolygonRegion", "list of XPoints");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XPolygonRegion", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XPolygonRegion", "int");
  arg1 = XEN_COPY_ARG(larg1);
  len = XEN_TO_C_INT(arg2);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg2, 2, "XPolygonRegion", "positive integer");
  pt = (XPoint *)CALLOC(len, sizeof(XPoint));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg1)); i++, arg1 = XEN_CDR(arg1))
    {
      XEN xp;
      xp = XEN_CAR(arg1);
      if (!(XEN_XPoint_P(xp))) {FREE(pt); XEN_ASSERT_TYPE(0, xp, i, "XPolygonRegion", "XPoint"); return(XEN_FALSE);}
      pt1 = XEN_TO_C_XPoint(XEN_CAR(arg1));
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
    }
  res = XPolygonRegion(pt, len, XEN_TO_C_INT(arg3));
  FREE(pt);
  return(C_TO_XEN_Region(res));
}

static XEN gxm_XPointInRegion(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XPointInRegion "Bool XPointInRegion(r, x, y) returns #t if the point (x, y) is contained in the region r."
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XPointInRegion", "Region");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XPointInRegion", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XPointInRegion", "int");
  return(C_TO_XEN_BOOLEAN(XPointInRegion(XEN_TO_C_Region(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XOffsetRegion(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XOffsetRegion "XOffsetRegion(r, dx, dy) moves the specified region by a specified amount."
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XOffsetRegion", "Region");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XOffsetRegion", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XOffsetRegion", "int");
  return(C_TO_XEN_INT(XOffsetRegion(XEN_TO_C_Region(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3))));
}

static XVisualInfo *match_visual_info;
static XEN gxm_XMatchVisualInfo(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XMatchVisualInfo "Status XMatchVisualInfo(display, screen, depth, class) returns the visual information for a \
visual that matches the specified depth and class for a screen."
  /* DIFF: XMatchVisualInfo dpy scr dep class [visual] -> #f or visual
   */
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XMatchVisualInfo", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XMatchVisualInfo", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XMatchVisualInfo", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XMatchVisualInfo", "int");
  match_visual_info = (XVisualInfo *)CALLOC(1, sizeof(XVisualInfo));
  val = XMatchVisualInfo(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), match_visual_info);
  if (val)
    return(C_TO_XEN_XVisualInfo(match_visual_info));
  return(XEN_FALSE);
}

static XEN gxm_XLookupString(XEN arg1)
{
  #define H_XLookupString "int XLookupString(event_struct) translates a key event to a KeySym and a string -> (len str keysym)."
  /* DIFF: XLookupString last arg is ignored, keyevent and all but last omitted -> (list len str keysym)
   */
  KeySym key;
  char *str;
  int len;
  XEN res;
  XEN_ASSERT_TYPE(XEN_XKeyEvent_P(arg1), arg1, 1, "XLookupString", "XKeyEvent*");/* user-created */
  str = (char *)CALLOC(16, sizeof(char));
  len = XLookupString(XEN_TO_C_XKeyEvent(arg1), str, 16, &key, NULL);
  res = XEN_LIST_3(C_TO_XEN_INT(len),
		   C_TO_XEN_STRING(str),
		   C_TO_XEN_KeySym(key));
  FREE(str);
  return(res);
}

static XEN gxm_XConvertCase(XEN arg1)
{
  #define H_XConvertCase "void XConvertCase(keysym) returns the uppercase and lowercase forms of the specified \
Keysym, if the KeySym is subject to case conversion; otherwise, the specified KeySym is returned to both lower_return and upper_return."
  /* DIFF: XConvertCase keysym [k1 k2] -> (list k1 k2)
   */
  KeySym k1, k2;
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg1), arg1, 1, "XConvertCase", "KeySym");
  XConvertCase(XEN_TO_C_KeySym(arg1), &k1, &k2);
  return(XEN_LIST_2(C_TO_XEN_KeySym(k1),
		    C_TO_XEN_KeySym(k2)));
}

static XEN gxm_XIntersectRegion(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XIntersectRegion "XIntersectRegion(sra, srb, dr_return) computes the intersection of two regions."
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XIntersectRegion", "Region");
  XEN_ASSERT_TYPE(XEN_Region_P(arg2), arg2, 2, "XIntersectRegion", "Region");
  XEN_ASSERT_TYPE(XEN_Region_P(arg3), arg3, 3, "XIntersectRegion", "Region");
  return(C_TO_XEN_INT(XIntersectRegion(XEN_TO_C_Region(arg1), XEN_TO_C_Region(arg2), XEN_TO_C_Region(arg3))));
}

static XEN gxm_XGetWMHints(XEN arg1, XEN arg2)
{
  #define H_XGetWMHints "XWMHints *XGetWMHints(display, w) reads the window manager hints and returns NULL if no WM_HINTS property was \
set on the window or returns a pointer to a XWMHints structure if it succeeds."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetWMHints", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetWMHints", "Window");
  return(C_TO_XEN_XWMHints(XGetWMHints(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XGetVisualInfo(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XGetVisualInfo "XVisualInfo *XGetVisualInfo(display, vinfo_mask, vinfo_template) returns a list of visual \
structures that have attributes equal to the attributes specified by vinfo_template."
  /* DIFF: XGetVisualInfo dpy mask template [nitems] -> '() or (list visual...)
   */
  XVisualInfo *v;
  int len;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetVisualInfo", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XGetVisualInfo", "long");
  XEN_ASSERT_TYPE(XEN_XVisualInfo_P(arg3), arg3, 3, "XGetVisualInfo", "XVisualInfo*");
  v = XGetVisualInfo(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_XVisualInfo(arg3), &len);
  if (v)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_XVisualInfo(v + i), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XGetStandardColormap(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XGetStandardColormap "XGetStandardColormap(dpy, window, atom) gets the standard colormap property"
  /* DIFF: XGetStandardColormap omits colormap arg and returns it or #f
   */
  XStandardColormap *c;
  Status val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetStandardColormap", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetStandardColormap", "Window");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XGetStandardColormap", "Atom");
  c = (XStandardColormap *)CALLOC(1, sizeof(XStandardColormap));
  val = XGetStandardColormap(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), c, XEN_TO_C_Atom(arg3));
  if (val != 0)
    return(C_TO_XEN_XStandardColormap(c));
  FREE(c);
  return(XEN_FALSE);
}
#endif

static XEN gxm_XGetRGBColormaps(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XGetRGBColormaps "Status XGetRGBColormaps(display, w, property) returns the RGB colormap \
definitions stored in the specified property on the named window."
  /* DIFF: XGetRGBColormaps omits arg 3 and 4, returns list of XStandardColormaps 
   */
  int len, i, loc;
  Status val;
  XEN lst = XEN_EMPTY_LIST;
  XStandardColormap **cs = NULL; /* do I allocate this?? */
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetRGBColormaps", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetRGBColormaps", "Window");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XGetRGBColormaps", "Atom");
  val = XGetRGBColormaps(XEN_TO_C_Display(arg1), 
			 XEN_TO_C_Window(arg2), 
			 cs, &len,
			 XEN_TO_C_Atom(arg3));
  if (val == 0)
    return(XEN_FALSE);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_XStandardColormap(cs[i]), lst);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XGetIconSizes(XEN arg1, XEN arg2)
{
  #define H_XGetIconSizes "Status XGetIconSizes(display, w) returns #f if the window manager has not \
set icon sizes; otherwise, it return nonzero and a list of XIconSize structs."
  /* DIFF: XGetIconSizes omit last 2 args, return list of XIconSizes
   */
  XIconSize *sizes;
  int i, len, val, loc;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetIconSizes", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetIconSizes", "Window");
  val = XGetIconSizes(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), &sizes, &len);
  if (val == 0)
    return(XEN_FALSE);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_XIconSize(&(sizes[i])), lst);
  XFree(sizes);
  xm_unprotect_at(loc);
  return(XEN_LIST_2(C_TO_XEN_INT(val),
		    lst));
}

static XEN gxm_XFindContext(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XFindContext "XFindContext(dpy, rid, context) gets data from the context manager"
  /* DIFF: XFindContext last arg omitted, val returned
   */
  caddr_t x;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFindContext", "Display*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XFindContext", "XID");
  XEN_ASSERT_TYPE(XEN_XContext_P(arg3), arg3, 3, "XFindContext", "XContext");
  val = XFindContext(XEN_TO_C_Display(arg1), XEN_TO_C_ULONG(arg2), XEN_TO_C_XContext(arg3), &x);
  return(XEN_LIST_2(C_TO_XEN_INT(val),
		    (XEN)x));
}

static XEN gxm_XEqualRegion(XEN arg1, XEN arg2)
{
  #define H_XEqualRegion "Bool XEqualRegion(r1, r2) returns #t if the two regions have the same offset, size, and shape."
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XEqualRegion", "Region");
  XEN_ASSERT_TYPE(XEN_Region_P(arg2), arg2, 2, "XEqualRegion", "Region");
  return(C_TO_XEN_BOOLEAN(XEqualRegion(XEN_TO_C_Region(arg1), XEN_TO_C_Region(arg2))));
}

static XEN gxm_XEmptyRegion(XEN arg1)
{
  #define H_XEmptyRegion "Bool XEmptyRegion(r) returns #t if the region is empty."
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XEmptyRegion", "Region");
  return(C_TO_XEN_BOOLEAN(XEmptyRegion(XEN_TO_C_Region(arg1))));
}

static XEN gxm_XDestroyRegion(XEN arg1)
{
  #define H_XDestroyRegion "XDestroyRegion(r) deallocates the storage associated with a specified region."
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XDestroyRegion", "Region");
  return(C_TO_XEN_INT(XDestroyRegion(XEN_TO_C_Region(arg1))));
}

static XEN gxm_XDeleteContext(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XDeleteContext "int XDeleteContext(display, rid, context) deletes the entry for the given resource ID and type from the data structure."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDeleteContext", "Display*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XDeleteContext", "XID");
  XEN_ASSERT_TYPE(XEN_XContext_P(arg3), arg3, 3, "XDeleteContext", "XContext");
  return(C_TO_XEN_INT(XDeleteContext(XEN_TO_C_Display(arg1), XEN_TO_C_ULONG(arg2), XEN_TO_C_XContext(arg3))));
}

static XEN gxm_XDefaultString(void)
{
  #define H_XDefaultString "char *XDefaultString() returns the default string used by Xlib for text conversion"
  return(C_TO_XEN_STRING(XDefaultString()));
}

static XEN gxm_XCreateRegion(void)
{
  #define H_XCreateRegion "Region XCreateRegion()"
  return(C_TO_XEN_Region(XCreateRegion()));
}

static XEN gxm_XClipBox(XEN arg1)
{
  #define H_XClipBox "XClipBox(r) returns the smallest rectangle enclosing the specified region."
  /* DIFF: XClipBox region [rectangle] -> (list val rectangle)
   */
  XRectangle *r;
  int val;
  XEN_ASSERT_TYPE(XEN_Region_P(arg1), arg1, 1, "XClipBox", "Region");
  r = (XRectangle *)CALLOC(1, sizeof(XRectangle));
  val = XClipBox(XEN_TO_C_Region(arg1), r);
  return(XEN_LIST_2(C_TO_XEN_INT(val), C_TO_XEN_XRectangle(r)));
}

static XEN gxm_XAllocWMHints(void)
{
  #define H_XAllocWMHints "XAllocWMHints() allocates a window manager hints structure"
  return(C_TO_XEN_XWMHints(XAllocWMHints()));
}

static XEN gxm_XAllocStandardColormap(void)
{
  #define H_XAllocStandardColormap "XStandardColormap *XAllocStandardColormap() allocates and returns a pointer to a XStandardColormap structure."
  return(C_TO_XEN_XStandardColormap(XAllocStandardColormap()));
}

static XEN gxm_XAllocIconSize(void)
{
  #define H_XAllocIconSize "XIconSize *XAllocIconSize() allocates and returns a pointer to a XIconSize structure."
  return(C_TO_XEN_XIconSize(XAllocIconSize()));
}

static XEN gxm_XFilterEvent(XEN arg1, XEN arg2)
{
  #define H_XFilterEvent "Bool XFilterEvent(event, w) passes the event to any filters registered for it in the given window"
  XEN_ASSERT_TYPE(XEN_XEvent_P(arg1), arg1, 1, "XFilterEvent", "XEvent*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XFilterEvent", "Window");
  return(C_TO_XEN_BOOLEAN(XFilterEvent(XEN_TO_C_XEvent(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XContextualDrawing(XEN arg1)
{
  #define H_XContextualDrawing "Bool XContextualDrawing(font_set) returns #t if text drawn with the font set might include context-dependent drawing."
  XEN_ASSERT_TYPE(XEN_XFontSet_P(arg1), arg1, 1, "XContextualDrawing", "XFontSet");
  return(C_TO_XEN_BOOLEAN(XContextualDrawing(XEN_TO_C_XFontSet(arg1))));
}

static XEN gxm_XDirectionalDependentDrawing(XEN arg1)
{
  #define H_XDirectionalDependentDrawing "Bool XDirectionalDependentDrawing(font_set) returns #t if the drawing functions implement \
implicit text directionality."
  XEN_ASSERT_TYPE(XEN_XFontSet_P(arg1), arg1, 1, "XDirectionalDependentDrawing", "XFontSet");
  return(C_TO_XEN_BOOLEAN(XDirectionalDependentDrawing(XEN_TO_C_XFontSet(arg1))));
}

static XEN gxm_XContextDependentDrawing(XEN arg1)
{
  #define H_XContextDependentDrawing "Bool XContextDependentDrawing(font_set) returns #t if the drawing functions implement implicit \
text directionality or if text drawn with the font_set might include context-dependent drawing."
  XEN_ASSERT_TYPE(XEN_XFontSet_P(arg1), arg1, 1, "XContextDependentDrawing", "XFontSet");
  return(C_TO_XEN_BOOLEAN(XContextDependentDrawing(XEN_TO_C_XFontSet(arg1))));
}

static XEN gxm_XLocaleOfFontSet(XEN arg1)
{
  #define H_XLocaleOfFontSet "char *XLocaleOfFontSet(font_set) returns the name of the locale bound to the specified XFontSet, as a \
null-terminated string."
  XEN_ASSERT_TYPE(XEN_XFontSet_P(arg1), arg1, 1, "XLocaleOfFontSet", "XFontSet");
  return(C_TO_XEN_STRING(XLocaleOfFontSet(XEN_TO_C_XFontSet(arg1))));
}

static XEN gxm_XBaseFontNameListOfFontSet(XEN arg1)
{
  #define H_XBaseFontNameListOfFontSet "char *XBaseFontNameListOfFontSet(font_set) returns the original base font name list supplied \
by the client when the  XFontSet was created."
  XEN_ASSERT_TYPE(XEN_XFontSet_P(arg1), arg1, 1, "XBaseFontNameListOfFontSet", "XFontSet");
  return(C_TO_XEN_STRING(XBaseFontNameListOfFontSet(XEN_TO_C_XFontSet(arg1))));
}

static XEN gxm_XFontsOfFontSet(XEN arg1)
{
  #define H_XFontsOfFontSet "int XFontsOfFontSet(font_set) returns a list of one or more \
XFontStructs and font names for the fonts used by the Xmb and Xwc layers, for the given font set."
  /* DIFF: XFontsOfFontSet omit last 2 args, return 2 lists
   */
  int i, len, loc1, loc2;
  char **names;
  XEN lst1 = XEN_EMPTY_LIST, lst2 = XEN_EMPTY_LIST;
  XFontStruct **fs;
  XEN_ASSERT_TYPE(XEN_XFontSet_P(arg1), arg1, 1, "XFontsOfFontSet", "XFontSet");
  len = XFontsOfFontSet(XEN_TO_C_XFontSet(arg1), &fs, &names);
  loc1 = xm_protect(lst1);
  loc2 = xm_protect(lst2);
  for (i = len - 1; i >= 0; i--)
    {
      lst1 = XEN_CONS(C_TO_XEN_XFontStruct(fs[i]), lst1);
      lst2 = XEN_CONS(C_TO_XEN_STRING(names[i]), lst2);
    }
  xm_unprotect_at(loc1);
  xm_unprotect_at(loc2);
  return(XEN_LIST_2(lst1, lst2));
}

static XEN gxm_XFreeFontSet(XEN arg1, XEN arg2)
{
  #define H_XFreeFontSet "void XFreeFontSet(display, font_set) frees the specified font set."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFreeFontSet", "Display*");
  XEN_ASSERT_TYPE(XEN_XFontSet_P(arg2), arg2, 2, "XFreeFontSet", "XFontSet");
  XFreeFontSet(XEN_TO_C_Display(arg1), XEN_TO_C_XFontSet(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XCreateFontSet(XEN arg1, XEN arg2)
{
  #define H_XCreateFontSet "XFontSet XCreateFontSet(display, base_font_name_list) creates a font set for the specified display."
  /* DIFF: XCreateFontSet ignores (omits) the 3 trailing missing glyph args
   */
  char **cs;
  int len;
  char *str;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreateFontSet", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XCreateFontSet", "char*");
  return(C_TO_XEN_XFontSet(XCreateFontSet(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2), &cs, &len, &str)));
}

static XEN gxm_XSetLocaleModifiers(XEN arg1)
{
  #define H_XSetLocaleModifiers "char *XSetLocaleModifiers(modifier_list) sets the X modifiers for the current locale setting."
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XSetLocaleModifiers", "char*");
  return(C_TO_XEN_STRING(XSetLocaleModifiers(XEN_TO_C_STRING(arg1))));
}

static XEN gxm_XSupportsLocale(void)
{
  #define H_XSupportsLocale "Bool XSupportsLocale() returns #t if Xlib functions are capable of operating under the current locale."
  return(C_TO_XEN_BOOLEAN(XSupportsLocale()));
}

static XEN gxm_XWriteBitmapFile(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XWriteBitmapFile "int XWriteBitmapFile(display, filename, bitmap, width, height, x_hot, y_hot) writes a bitmap out to a file in \
the X Version 11 format."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XWriteBitmapFile", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XWriteBitmapFile", "char*");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg3), arg3, 3, "XWriteBitmapFile", "Pixmap");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XWriteBitmapFile", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XWriteBitmapFile", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XWriteBitmapFile", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XWriteBitmapFile", "int");
  return(C_TO_XEN_INT(XWriteBitmapFile(XEN_TO_C_Display(arg1), 
				       XEN_TO_C_STRING(arg2), 
				       XEN_TO_C_Pixmap(arg3), 
				       XEN_TO_C_ULONG(arg4), XEN_TO_C_ULONG(arg5), 
				       XEN_TO_C_INT(arg6), XEN_TO_C_INT(arg7))));
}

static XEN gxm_XWindowEvent(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XWindowEvent "XWindowEvent(display, w, event_mask) searches the event queue for an event that matches both \
the specified window and event mask, and removes it or waits until it arrives."
  /* DIFF: XWindowEvent dpy win ev [evrtn] -> (list val evrtn)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XWindowEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XWindowEvent", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XWindowEvent", "long");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XWindowEvent(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3), e);
  return(XEN_LIST_2(C_TO_XEN_INT(val), C_TO_XEN_XEvent_OBJ(e)));
}

static XEN gxm_XWidthOfScreen(XEN arg1)
{
  #define H_WidthOfScreen "XWidthOfScreen(screen) returns the width of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XWidthOfScreen", "Screen*");
  return(C_TO_XEN_INT(XWidthOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XWidthMMOfScreen(XEN arg1)
{
  #define H_WidthMMOfScreen "XWidthMMOfScreen(screen) returns the width of the specified screen in millimeters."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XWidthMMOfScreen", "Screen*");
  return(C_TO_XEN_INT(XWidthMMOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XWarpPointer(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8, XEN arg9)
{
  #define H_XWarpPointer "XWarpPointer(display, src_w, dest_w, src_x, src_y, src_width, src_height, dest_x, dest_y)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XWarpPointer", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XWarpPointer", "Window");
  XEN_ASSERT_TYPE(XEN_Window_P(arg3), arg3, 3, "XWarpPointer", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XWarpPointer", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XWarpPointer", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XWarpPointer", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XWarpPointer", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XWarpPointer", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg9), arg9, 9, "XWarpPointer", "int");
  return(C_TO_XEN_INT(XWarpPointer(XEN_TO_C_Display(arg1), 
				   XEN_TO_C_Window(arg2), XEN_TO_C_Window(arg3), 
				   XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
				   XEN_TO_C_ULONG(arg6), XEN_TO_C_ULONG(arg7), 
				   XEN_TO_C_INT(arg8), XEN_TO_C_INT(arg9))));
}

static XEN gxm_XVendorRelease(XEN arg1)
{
  #define H_VendorRelease "VendorRelease(display) returns a number related to a vendor's release of the X server."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XVendorRelease", "Display*");
  return(C_TO_XEN_INT(XVendorRelease(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XUnmapWindow(XEN arg1, XEN arg2)
{
  #define H_XUnmapWindow "XUnmapWindow(display, w) unmaps the specified window and causes the X server to generate an UnmapNotify event."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XUnmapWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XUnmapWindow", "Window");
  return(C_TO_XEN_INT(XUnmapWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XUnmapSubwindows(XEN arg1, XEN arg2)
{
  #define H_XUnmapSubwindows "XUnmapSubwindows(display, w) unmaps all subwindows for the specified window in bottom-to-top stacking order."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XUnmapSubwindows", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XUnmapSubwindows", "Window");
  return(C_TO_XEN_INT(XUnmapSubwindows(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XUnloadFont(XEN arg1, XEN arg2)
{
  #define H_XUnloadFont "XUnloadFont(display, font) deletes the association between the font resource ID and the specified font."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XUnloadFont", "Display*");
  XEN_ASSERT_TYPE(XEN_Font_P(arg2), arg2, 2, "XUnloadFont", "Font");
  return(C_TO_XEN_INT(XUnloadFont(XEN_TO_C_Display(arg1), XEN_TO_C_Font(arg2))));
}

static XEN gxm_XUninstallColormap(XEN arg1, XEN arg2)
{
  #define H_XUninstallColormap "XUninstallColormap(display, colormap) removes the specified colormap from the required list for its screen."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XUninstallColormap", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XUninstallColormap", "Colormap");
  return(C_TO_XEN_INT(XUninstallColormap(XEN_TO_C_Display(arg1), XEN_TO_C_Colormap(arg2))));
}

static XEN gxm_XUngrabServer(XEN arg1)
{
  #define H_XUngrabServer "XUngrabServer(display) restarts processing of requests and close downs on other connections."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XUngrabServer", "Display*");
  return(C_TO_XEN_INT(XUngrabServer(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XUngrabPointer(XEN arg1, XEN arg2)
{
  #define H_XUngrabPointer "XUngrabPointer(display, time) releases the pointer and any queued events if this client has actively grabbed the \
pointer from XGrabPointer, XGrabButton,or from a normal button press."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XUngrabPointer", "Display*");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XUngrabPointer", "Time");
  return(C_TO_XEN_INT(XUngrabPointer(XEN_TO_C_Display(arg1), XEN_TO_C_Time(arg2))));
}

static XEN gxm_XUngrabKeyboard(XEN arg1, XEN arg2)
{
  #define H_XUngrabKeyboard "XUngrabKeyboard(display, time) releases the keyboard and any queued events if this client has it actively grabbed \
from either XGrabKeyboard or XGrabKey."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XUngrabKeyboard", "Display*");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XUngrabKeyboard", "Time");
  return(C_TO_XEN_INT(XUngrabKeyboard(XEN_TO_C_Display(arg1), XEN_TO_C_Time(arg2))));
}

static XEN gxm_XUngrabKey(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XUngrabKey "XUngrabKey(display, keycode, modifiers, grab_window) releases the key combination on the specified window if it was \
grabbed by this client."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XUngrabKey", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XUngrabKey", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XUngrabKey", "unsigned int");
  XEN_ASSERT_TYPE(XEN_Window_P(arg4), arg4, 4, "XUngrabKey", "Window");
  return(C_TO_XEN_INT(XUngrabKey(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_ULONG(arg3), XEN_TO_C_Window(arg4))));
}

static XEN gxm_XUngrabButton(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XUngrabButton "XUngrabButton(display, button, modifiers, grab_window) releases the passive button/key combination on the specified \
window if it was grabbed by this client."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XUngrabButton", "Display*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XUngrabButton", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XUngrabButton", "unsigned int");
  XEN_ASSERT_TYPE(XEN_Window_P(arg4), arg4, 4, "XUngrabButton", "Window");
  return(C_TO_XEN_INT(XUngrabButton(XEN_TO_C_Display(arg1), XEN_TO_C_ULONG(arg2), XEN_TO_C_ULONG(arg3), XEN_TO_C_Window(arg4))));
}

static XEN gxm_XUndefineCursor(XEN arg1, XEN arg2)
{
  #define H_XUndefineCursor "XUndefineCursor(display, w) undoes the effect of a previous XDefineCursor for this window."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XUndefineCursor", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XUndefineCursor", "Window");
  return(C_TO_XEN_INT(XUndefineCursor(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XTranslateCoordinates(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XTranslateCoordinates "Bool XTranslateCoordinates(display, src_w, dest_w, src_x, src_y) \
takes the src_x and src_y coordinates relative to the source window's origin and returns these coordinates to dest_x_return and dest_y_return \
relative to the destination window's origin -> (rtn x y win)."
  /* DIFF: XTranslateCoordinates omit last 3 args
   */
  Window w;
  int x, y, rtn;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XTranslateCoordinates", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XTranslateCoordinates", "Window");
  XEN_ASSERT_TYPE(XEN_Window_P(arg3), arg3, 3, "XTranslateCoordinates", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XTranslateCoordinates", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XTranslateCoordinates", "int");
  rtn = XTranslateCoordinates(XEN_TO_C_Display(arg1), 
			      XEN_TO_C_Window(arg2), 
			      XEN_TO_C_Window(arg3), 
			      XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
			      &x, &y, &w);
  return(XEN_LIST_4(C_TO_XEN_BOOLEAN(rtn),
		    C_TO_XEN_INT(x),
		    C_TO_XEN_INT(y),
		    C_TO_XEN_Window(w)));
}

static XEN gxm_XTextWidth(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XTextWidth "int XTextWidth(font_struct, string, count) returns the width of the specified 8-bit string."
  XEN_ASSERT_TYPE(XEN_XFontStruct_P(arg1), arg1, 1, "XTextWidth", "XFontStruct*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XTextWidth", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XTextWidth", "int");
  return(C_TO_XEN_INT(XTextWidth(XEN_TO_C_XFontStruct(arg1), XEN_TO_C_STRING(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XTextExtents(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XTextExtents "XTextExtents(font_struct, string, nchars) returns an XCharStruct structure describing the text."
  /* DIFF: XTextExtents omit final 4 args and returns them [Xcharset returned as embedded list)
   */
  int dir, fa, fd, rtn;
  XCharStruct val;
  XEN_ASSERT_TYPE(XEN_XFontStruct_P(arg1), arg1, 1, "XTextExtents", "XFontStruct*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XTextExtents", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XTextExtents", "int");
  rtn = XTextExtents(XEN_TO_C_XFontStruct(arg1), 
		     XEN_TO_C_STRING(arg2), 
		     XEN_TO_C_INT(arg3), 
		     &dir, &fa, &fd, &val);
  return(XEN_LIST_5(C_TO_XEN_INT(rtn),
		    C_TO_XEN_INT(dir),
		    C_TO_XEN_INT(fa),
		    C_TO_XEN_INT(fd),
		    XEN_LIST_6(C_TO_XEN_INT((int)(val.lbearing)),
			       C_TO_XEN_INT((int)(val.rbearing)),
			       C_TO_XEN_INT((int)(val.width)),
			       C_TO_XEN_INT((int)(val.ascent)),
			       C_TO_XEN_INT((int)(val.descent)),
			       C_TO_XEN_INT((int)(val.attributes)))));
}

static XEN gxm_XSync(XEN arg1, XEN arg2)
{
  #define H_XSync "XSync(display, discard) flushes the output buffer and then waits until all requests have been received and processed by the X server."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSync", "Display*");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XSync", "Bool");
  return(C_TO_XEN_INT(XSync(XEN_TO_C_Display(arg1), XEN_TO_C_BOOLEAN(arg2))));
}

static XEN gxm_XStoreNamedColor(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XStoreNamedColor "XStoreNamedColor(display, colormap, color, pixel, flags) looks up the named color with respect to the screen \
associated with the colormap and stores the result in the specified colormap."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XStoreNamedColor", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XStoreNamedColor", "Colormap");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XStoreNamedColor", "char*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XStoreNamedColor", "ulong"); /* this is explicitly an index into the colormap, so I'll leave it as is */
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XStoreNamedColor", "int");
  return(C_TO_XEN_INT(XStoreNamedColor(XEN_TO_C_Display(arg1), 
				       XEN_TO_C_Colormap(arg2), 
				       XEN_TO_C_STRING(arg3), XEN_TO_C_ULONG(arg4), XEN_TO_C_INT(arg5))));
}

static XEN gxm_XStoreName(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XStoreName "XStoreName(display, w, window_name) assigns the name passed to window_name to the specified window."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XStoreName", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XStoreName", "Window");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XStoreName", "char*");
  return(C_TO_XEN_INT(XStoreName(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_STRING(arg3))));
}

static XEN gxm_XStoreColors(XEN arg1, XEN arg2, XEN larg3, XEN arg4)
{
  #define H_XStoreColors "XStoreColors(display, colormap, color, ncolors) changes the colormap entries of the pixel values specified in the \
pixel members of the XColor structures."
  /* DIFF: XStoreColors arg 3 is list of XColor 
   */
  XColor *xc;
  int i, len;
  XEN arg3;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XStoreColors", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XStoreColors", "Colormap");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg3), larg3, 3, "XStoreColors", "list of XColor");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XStoreColors", "int");
  arg3 = XEN_COPY_ARG(larg3);
  len = XEN_TO_C_INT(arg4);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg4, 4, "XStoreColors", "positive integer");
  xc = (XColor *)CALLOC(len, sizeof(XColor));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg3)); i++, arg3 = XEN_CDR(arg3))
    {
      XColor *xc1;
      xc1 = XEN_TO_C_XColor(XEN_CAR(arg3));
      xc[i].pixel = xc1->pixel;
      xc[i].red = xc1->red;
      xc[i].green = xc1->green;
      xc[i].blue = xc1->blue;
      xc[i].flags = xc1->flags;
      xc[i].pad = xc1->pad;
    }
  XStoreColors(XEN_TO_C_Display(arg1), 
	       XEN_TO_C_Colormap(arg2),
	       xc, len);
  FREE(xc);
  return(C_TO_XEN_INT(len));
}

static XEN gxm_XStoreColor(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XStoreColor "XStoreColor(display, colormap, color) changes the colormap entry of the pixel value specified in the pixel member \
of the XColor structure."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XStoreColor", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XStoreColor", "Colormap");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg3), arg3, 3, "XStoreColor", "XColor"); 
  return(C_TO_XEN_INT(XStoreColor(XEN_TO_C_Display(arg1), XEN_TO_C_Colormap(arg2), XEN_TO_C_XColor(arg3))));
}

static XEN gxm_XStoreBytes(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XStoreBytes "XStoreBytes(display, bytes, nbytes)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XStoreBytes", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XStoreBytes", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XStoreBytes", "int");
  return(C_TO_XEN_INT(XStoreBytes(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2), XEN_TO_C_INT(arg3) + 1)));
}

static XEN gxm_XStoreBuffer(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XStoreBuffer "XStoreBuffer(display, bytes, nbytes, buffer)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XStoreBuffer", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XStoreBuffer", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XStoreBuffer", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XStoreBuffer", "int");
  return(C_TO_XEN_INT(XStoreBuffer(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2), XEN_TO_C_INT(arg3) + 1, XEN_TO_C_INT(arg4))));
}

static XEN gxm_XSetWindowColormap(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetWindowColormap "XSetWindowColormap(display, w, colormap) sets the specified colormap of the specified window."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetWindowColormap", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetWindowColormap", "Window");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg3), arg3, 3, "XSetWindowColormap", "Colormap");
  return(C_TO_XEN_INT(XSetWindowColormap(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_Colormap(arg3))));
}

static XEN gxm_XSetWindowBorderWidth(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetWindowBorderWidth "XSetWindowBorderWidth(display, w, width) sets the specified window's border width to the specified width."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetWindowBorderWidth", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetWindowBorderWidth", "Window");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XSetWindowBorderWidth", "unsigned int");
  return(C_TO_XEN_INT(XSetWindowBorderWidth(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_ULONG(arg3))));
}

static XEN gxm_XSetWindowBorderPixmap(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetWindowBorderPixmap "XSetWindowBorderPixmap(display, w, border_pixmap) sets the border pixmap of the window to the pixmap you specify."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetWindowBorderPixmap", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetWindowBorderPixmap", "Window");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg3) || XEN_INTEGER_P(arg3), arg3, 3, "XSetWindowBorderPixmap", "Pixmap");
  return(C_TO_XEN_INT(XSetWindowBorderPixmap(XEN_TO_C_Display(arg1), 
					     XEN_TO_C_Window(arg2),
					     (XEN_Pixmap_P(arg3)) ? XEN_TO_C_Pixmap(arg3) : CopyFromParent)));
}

static XEN gxm_XSetWindowBorder(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetWindowBorder "XSetWindowBorder(display, w, border_pixel) sets the border of the window to the pixel value you specify."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetWindowBorder", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetWindowBorder", "Window");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg3), arg3, 3, "XSetWindowBorder", "pixel"); 
  return(C_TO_XEN_INT(XSetWindowBorder(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_Pixel(arg3))));
}

static XEN gxm_XSetWindowBackgroundPixmap(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetWindowBackgroundPixmap "XSetWindowBackgroundPixmap(display, w, background_pixmap) sets the background pixmap of the window to \
the specified pixmap."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetWindowBackgroundPixmap", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetWindowBackgroundPixmap", "Window");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg3) || XEN_INTEGER_P(arg3), arg3, 3, "XSetWindowBackgroundPixmap", "Pixmap");
  return(C_TO_XEN_INT(XSetWindowBackgroundPixmap(XEN_TO_C_Display(arg1), 
						 XEN_TO_C_Window(arg2), 
						 (XEN_Pixmap_P(arg3)) ? XEN_TO_C_Pixmap(arg3) : XEN_TO_C_ULONG(arg3))));
}

static XEN gxm_XSetWindowBackground(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetWindowBackground "XSetWindowBackground(display, w, background_pixel) sets the background of the window to the specified pixel value."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetWindowBackground", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetWindowBackground", "Window");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg3), arg3, 3, "XSetWindowBackground", "pixel"); 
  return(C_TO_XEN_INT(XSetWindowBackground(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_Pixel(arg3))));
}

static XEN gxm_XSetTile(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetTile "XSetTile(display, gc, tile) sets the fill tile in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetTile", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetTile", "GC");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg3), arg3, 3, "XSetTile", "Pixmap");
  return(C_TO_XEN_INT(XSetTile(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_Pixmap(arg3))));
}

static XEN gxm_XSetTSOrigin(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XSetTSOrigin "XSetTSOrigin(display, gc, ts_x_origin, ts_y_origin) sets the tile/stipple origin in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetTSOrigin", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetTSOrigin", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetTSOrigin", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XSetTSOrigin", "int");
  return(C_TO_XEN_INT(XSetTSOrigin(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4))));
}

static XEN gxm_XSetSubwindowMode(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetSubwindowMode "XSetSubwindowMode(display, gc, subwindow_mode) sets the subwindow mode in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetSubwindowMode", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetSubwindowMode", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetSubwindowMode", "int");
  return(C_TO_XEN_INT(XSetSubwindowMode(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XSetStipple(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetStipple "XSetStipple(display, gc, stipple) sets the stipple in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetStipple", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetStipple", "GC");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg3), arg3, 3, "XSetStipple", "Pixmap");
  return(C_TO_XEN_INT(XSetStipple(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_Pixmap(arg3))));
}

static XEN gxm_XSetState(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XSetState "XSetState(display, gc, foreground, background, function, plane_mask) sets the foreground, background, plane mask, and \
function components for the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetState", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetState", "GC");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg3), arg3, 3, "XSetState", "Pixel");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg4), arg4, 4, "XSetState", "Pixel");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XSetState", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XSetState", "ulong");
  return(C_TO_XEN_INT(XSetState(XEN_TO_C_Display(arg1), 
				XEN_TO_C_GC(arg2), 
				XEN_TO_C_Pixel(arg3), XEN_TO_C_Pixel(arg4), 
				XEN_TO_C_INT(arg5), XEN_TO_C_ULONG(arg6))));
}

static XEN gxm_XSetSelectionOwner(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XSetSelectionOwner "XSetSelectionOwner(display, selection, owner, time) changes the owner and last-change time for the specified selection"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetSelectionOwner", "Display*");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XSetSelectionOwner", "Atom");
  XEN_ASSERT_TYPE(XEN_Window_P(arg3), arg3, 3, "XSetSelectionOwner", "Window");
  XEN_ASSERT_TYPE(XEN_Time_P(arg4), arg4, 4, "XSetSelectionOwner", "Time");
  return(C_TO_XEN_INT(XSetSelectionOwner(XEN_TO_C_Display(arg1), XEN_TO_C_Atom(arg2), XEN_TO_C_Window(arg3), XEN_TO_C_Time(arg4))));
}

static XEN gxm_XSetScreenSaver(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XSetScreenSaver "XSetScreenSaver(display, timeout, interval, prefer_blanking, allow_exposures) enables the screen saver."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetScreenSaver", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XSetScreenSaver", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetScreenSaver", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XSetScreenSaver", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XSetScreenSaver", "int");
  return(C_TO_XEN_INT(XSetScreenSaver(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5))));
}

static XEN gxm_XSetPointerMapping(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetPointerMapping "int XSetPointerMapping(display, map, nmap) sets the mapping of the pointer."
  int i, len, rtn;
  unsigned char *map;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetPointerMapping", "Display*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XSetPointerMapping", "list of ints");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XSetPointerMapping", "int");
  if (XEN_INTEGER_P(arg3)) len = XEN_TO_C_INT(arg3); else len = XEN_LIST_LENGTH(arg2);
  map = (unsigned char *)CALLOC(len, sizeof(unsigned char));
  for (i = 0; i < len; i++)
    map[i] = (unsigned char)XEN_TO_C_INT(XEN_LIST_REF(arg2, i));
  rtn = XSetPointerMapping(XEN_TO_C_Display(arg1), map, len);
  FREE(map);
  return(C_TO_XEN_INT(rtn));
}

static XEN gxm_XSetPlaneMask(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetPlaneMask "XSetPlaneMask(display, gc, plane_mask) sets the plane mask in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetPlaneMask", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetPlaneMask", "GC");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XSetPlaneMask", "ulong");
  return(C_TO_XEN_INT(XSetPlaneMask(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_ULONG(arg3))));
}

static XEN gxm_XSetModifierMapping(XEN arg1, XEN arg2)
{
  #define H_XSetModifierMapping "int XSetModifierMapping(display, modmap) specifies the KeyCodes of the keys (if any) that are to be used as modifiers."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetModifierMapping", "Display*");
  XEN_ASSERT_TYPE(XEN_XModifierKeymap_P(arg2), arg2, 2, "XSetModifierMapping", "XModifierKeymap*");
  return(C_TO_XEN_INT(XSetModifierMapping(XEN_TO_C_Display(arg1), XEN_TO_C_XModifierKeymap(arg2))));
}

static XEN gxm_XSetLineAttributes(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XSetLineAttributes "XSetLineAttributes(display, gc, line_width, line_style, cap_style, join_style) sets the line drawing components \
in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetLineAttributes", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetLineAttributes", "GC");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XSetLineAttributes", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XSetLineAttributes", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XSetLineAttributes", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XSetLineAttributes", "int");
  return(C_TO_XEN_INT(XSetLineAttributes(XEN_TO_C_Display(arg1), 
					 XEN_TO_C_GC(arg2), 
					 XEN_TO_C_ULONG(arg3), XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), XEN_TO_C_INT(arg6))));
}

static XEN gxm_XSetInputFocus(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XSetInputFocus "XSetInputFocus(display, focus, revert_to, time) changes the input focus and the last-focus-change time."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetInputFocus", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetInputFocus", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetInputFocus", "int");
  XEN_ASSERT_TYPE(XEN_Time_P(arg4), arg4, 4, "XSetInputFocus", "Time");
  return(C_TO_XEN_INT(XSetInputFocus(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_Time(arg4))));
}

static XEN gxm_XSetIconName(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetIconName "XSetIconName(display, w, icon_name) sets the name to be displayed in a window's icon."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetIconName", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetIconName", "Window");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XSetIconName", "char*");
  return(C_TO_XEN_INT(XSetIconName(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_STRING(arg3))));
}

static XEN gxm_XSetGraphicsExposures(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetGraphicsExposures "XSetGraphicsExposures(display, gc, graphics_exposures) sets the graphics-exposures flag in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetGraphicsExposures", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetGraphicsExposures", "GC");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XSetGraphicsExposures", "Bool");
  return(C_TO_XEN_INT(XSetGraphicsExposures(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_BOOLEAN(arg3))));
}

static XEN gxm_XSetFunction(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetFunction "XSetFunction(display, gc, function) sets a specified value in the specified GC"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetFunction", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetFunction", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetFunction", "int");
  return(C_TO_XEN_INT(XSetFunction(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XSetForeground(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetForeground "XSetForeground(display, gc, foreground) sets the foreground in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetForeground", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetForeground", "GC");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg3), arg3, 3, "XSetForeground", "Pixel");
  return(C_TO_XEN_INT(XSetForeground(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_Pixel(arg3))));
}

static XEN gxm_XSetFontPath(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetFontPath "XSetFontPath(display, directories, ndirs) defines the directory search path for font lookup."
  /* DIFF: XSetFontPath arg2 is list of strings
   */
  char **paths;
  int len, rtn;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetFontPath", "Display*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XSetFontPath", "list of char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetFontPath", "int");
  len = XEN_TO_C_INT(arg3);
  if (len > 0)
    paths = XEN_TO_C_Strings(arg2, len);
  else paths = NULL;
  rtn = XSetFontPath(XEN_TO_C_Display(arg1), paths, len);
  if (paths) FREE(paths);
  return(xen_return_first(C_TO_XEN_INT(rtn), arg2));
}

static XEN gxm_XSetFont(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetFont "XSetFont(display, gc, font) sets the current font in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetFont", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetFont", "GC");
  XEN_ASSERT_TYPE(XEN_Font_P(arg3), arg3, 3, "XSetFont", "Font");
  return(C_TO_XEN_INT(XSetFont(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_Font(arg3))));
}

static XEN gxm_XSetFillStyle(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetFillStyle "XSetFillStyle(display, gc, fill_style) sets the fill-style in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetFillStyle", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetFillStyle", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetFillStyle", "int");
  return(C_TO_XEN_INT(XSetFillStyle(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XSetFillRule(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetFillRule "XSetFillRule(display, gc, fill_rule) sets the fill-rule in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetFillRule", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetFillRule", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetFillRule", "int");
  return(C_TO_XEN_INT(XSetFillRule(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XSetDashes(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XSetDashes "XSetDashes(display, gc, dash_offset, dash_list, n) sets the dash-offset and dash-list attributes for dashed line styles \
in the specified GC."
  char *dashes;
  int i, len = 0, val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetDashes", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetDashes", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetDashes", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XSetDashes", "list of ints");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg5), arg5, 5, "XSetDashes", "optional length of list (int)");
  if (XEN_INTEGER_P(arg5)) len = XEN_TO_C_INT(arg5);
  if (len <= 0) len = XEN_LIST_LENGTH(arg4);
  dashes = (char *)CALLOC(len, sizeof(char));
  for (i = 0; i < len; i++) dashes[i] = (char)(XEN_TO_C_INT(XEN_LIST_REF(arg4, i)));
  val = XSetDashes(XEN_TO_C_Display(arg1), 
		   XEN_TO_C_GC(arg2), 
		   XEN_TO_C_INT(arg3),
		   dashes, len);
  FREE(dashes);
  return(C_TO_XEN_INT(val));
}

static XEN gxm_XSetCommand(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XSetCommand "XSetCommand(display, w, argv, argc) sets the command and arguments used to invoke the application."
  /* DIFF: XSetCommand argv is list of strings
   */
  int len, val;
  char **str;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetCommand", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetCommand", "Window");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XSetCommand", "list of char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XSetCommand", "int");
  len = XEN_TO_C_INT(arg4);
  if (len <= 0) return(XEN_FALSE);
  str = XEN_TO_C_Strings(arg3, len);
  val = XSetCommand(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), str, len);
  FREE(str);
  return(xen_return_first(C_TO_XEN_INT(val), arg3));
}

static XEN gxm_XSetCloseDownMode(XEN arg1, XEN arg2)
{
  #define H_XSetCloseDownMode "XSetCloseDownMode(display, close_mode) defines what will happen to the client's resources at connection close."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetCloseDownMode", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XSetCloseDownMode", "int");
  return(C_TO_XEN_INT(XSetCloseDownMode(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XSetClipRectangles(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN larg5, XEN arg6, XEN arg7)
{
  #define H_XSetClipRectangles "XSetClipRectangles(display, gc, clip_x_origin, clip_y_origin, rectangles, n, ordering) changes the clip-mask in \
the specified GC to the specified list of rectangles and sets the clip origin."
  /* DIFF: XSetClipRectangles XRectangle* arg (arg 5) is list of XRectangles
   */
  XRectangle *pt;
  int i, len;
  XEN arg5;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetClipRectangles", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetClipRectangles", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetClipRectangles", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XSetClipRectangles", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg5), larg5, 5, "XSetClipRectangles", "list of XRectangles");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XSetClipRectangles", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XSetClipRectangles", "int");
  arg5 = XEN_COPY_ARG(larg5);
  len = XEN_TO_C_INT(arg6);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg6, 6, "XSetClipRectangles", "positive integer");
  pt = (XRectangle *)CALLOC(len, sizeof(XRectangle));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg5)); i++, arg5 = XEN_CDR(arg5))
    {
      XRectangle *pt1;
      pt1 = XEN_TO_C_XRectangle(XEN_CAR(arg5));
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
      pt[i].width = pt1->width;
      pt[i].height = pt1->height;
    }
  XSetClipRectangles(XEN_TO_C_Display(arg1), 
		     XEN_TO_C_GC(arg2), 
		     XEN_TO_C_INT(arg3), 
		     XEN_TO_C_INT(arg4), 
		     pt, len,
		     XEN_TO_C_INT(arg7));
  FREE(pt);
  return(C_TO_XEN_INT(len));
}

static XEN gxm_XSetClipOrigin(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XSetClipOrigin "XSetClipOrigin(display, gc, clip_x_origin, clip_y_origin) sets the clip origin in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetClipOrigin", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetClipOrigin", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetClipOrigin", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XSetClipOrigin", "int");
  return(C_TO_XEN_INT(XSetClipOrigin(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4))));
}

static XEN gxm_XSetClipMask(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetClipMask "XSetClipMask(display, gc, pixmap) sets the clip-mask in the specified GC to the specified pixmap."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetClipMask", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetClipMask", "GC");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg3) || XEN_INTEGER_P(arg3), arg3, 3, "XSetClipMask", "Pixmap or None");
  return(C_TO_XEN_INT(XSetClipMask(XEN_TO_C_Display(arg1), 
				   XEN_TO_C_GC(arg2),
				   (XEN_Pixmap_P(arg3)) ? XEN_TO_C_Pixmap(arg3) : None)));
}

static XEN gxm_XSetBackground(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetBackground "XSetBackground(display, gc, background) sets the background in the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetBackground", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetBackground", "GC");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg3), arg3, 3, "XSetBackground", "Pixel");
  return(C_TO_XEN_INT(XSetBackground(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_Pixel(arg3))));
}

static XEN gxm_XSetArcMode(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetArcMode "XSetArcMode(display, gc, arc_mode)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetArcMode", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XSetArcMode", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSetArcMode", "int");
  return(C_TO_XEN_INT(XSetArcMode(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XSetAccessControl(XEN arg1, XEN arg2)
{
  #define H_XSetAccessControl "XSetAccessControl(display, mode) either enables or disables the use of the access control list at each connection setup."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetAccessControl", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XSetAccessControl", "int");
  return(C_TO_XEN_INT(XSetAccessControl(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XSendEvent(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XSendEvent "Status XSendEvent(display, w, propagate, event_mask, event_send) identifies the destination window, determines which \
clients should receive the specified events, "
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSendEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSendEvent", "Window");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XSendEvent", "Bool");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XSendEvent", "long");
  XEN_ASSERT_TYPE(XEN_XEvent_P(arg5), arg5, 5, "XSendEvent", "XEvent*");
  return(C_TO_XEN_INT(XSendEvent(XEN_TO_C_Display(arg1), 
				 XEN_TO_C_Window(arg2), 
				 XEN_TO_C_BOOLEAN(arg3), XEN_TO_C_INT(arg4), XEN_TO_C_XEvent(arg5))));
}

static XEN gxm_XSelectInput(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSelectInput "XSelectInput(dpy, window, event_mask) selects input events"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSelectInput", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSelectInput", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSelectInput", "long");
  return(C_TO_XEN_INT(XSelectInput(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XScreenCount(XEN arg1)
{
  #define H_ScreenCount "returns the number of available screens."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XScreenCount", "Display*");
  return(C_TO_XEN_INT(XScreenCount(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XRotateWindowProperties(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XRotateWindowProperties "XRotateWindowProperties(display, w, properties, num_prop, npositions) allows you to rotate properties on a \
window and causes the X server to generate PropertyNotify events."
  /* DIFF: XRotateWindowProperties arg3 is list of atoms
   */
  Atom *outs;
  int val, len;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XRotateWindowProperties", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XRotateWindowProperties", "Window");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XRotateWindowProperties", "list of Atom");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XRotateWindowProperties", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XRotateWindowProperties", "int");
  len = XEN_TO_C_INT(arg4);
  if (len <= 0) return(XEN_FALSE);
  outs = XEN_TO_C_Atoms(arg3, len);
  val = XRotateWindowProperties(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), outs, len, XEN_TO_C_INT(arg5));
  FREE(outs);
  return(C_TO_XEN_INT(val));
}

static XEN gxm_XRotateBuffers(XEN arg1, XEN arg2)
{
  #define H_XRotateBuffers "XRotateBuffers(display, rotate) rotates the cut buffers, such that buffer 0 becomes buffer n, buffer 1 becomes n + 1 \
mod 8, and so on. "
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XRotateBuffers", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XRotateBuffers", "int");
  return(C_TO_XEN_INT(XRotateBuffers(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XRestackWindows(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XRestackWindows "XRestackWindows(display, windows, nwindows) restacks the windows in the order specified, from top to bottom."
  /* DIFF: XRestackWindows arg2 is list of Windows
   */
  int len, rtn;
  Window *ws;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XRestackWindows", "Display*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XRestackWindows", "list of Windows");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XRestackWindows", "int");
  len = XEN_TO_C_INT(arg3);
  if (len <= 0) return(XEN_FALSE);
  ws = XEN_TO_C_Windows(arg2, len);
  rtn = XRestackWindows(XEN_TO_C_Display(arg1), ws, len);
  FREE(ws);
  return(C_TO_XEN_INT(rtn));
}

static XEN gxm_XResizeWindow(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XResizeWindow "XResizeWindow(display, w, width, height) changes the inside dimensions of the specified window, not including its borders."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XResizeWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XResizeWindow", "Window");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XResizeWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XResizeWindow", "unsigned int");
  return(C_TO_XEN_INT(XResizeWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_ULONG(arg3), XEN_TO_C_ULONG(arg4))));
}

static XEN gxm_XResetScreenSaver(XEN arg1)
{
  #define H_XResetScreenSaver "XResetScreenSaver(display) resets the screen saver."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XResetScreenSaver", "Display*");
  return(C_TO_XEN_INT(XResetScreenSaver(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XReparentWindow(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XReparentWindow "XReparentWindow(display, w, parent, x, y)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XReparentWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XReparentWindow", "Window");
  XEN_ASSERT_TYPE(XEN_Window_P(arg3), arg3, 3, "XReparentWindow", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XReparentWindow", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XReparentWindow", "int");
  return(C_TO_XEN_INT(XReparentWindow(XEN_TO_C_Display(arg1), 
				      XEN_TO_C_Window(arg2), 
				      XEN_TO_C_Window(arg3), 
				      XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5))));
}

static XEN gxm_XRefreshKeyboardMapping(XEN arg1)
{
  #define H_XRefreshKeyboardMapping "XRefreshKeyboardMapping(event_map) refreshes the stored modifier and keymap information."
  XEN_ASSERT_TYPE(XEN_XMappingEvent_P(arg1), arg1, 1, "XRefreshKeyboardMapping", "XMappingEvent*");
  return(C_TO_XEN_INT(XRefreshKeyboardMapping(XEN_TO_C_XMappingEvent(arg1))));
}

static XEN gxm_XRecolorCursor(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XRecolorCursor "XRecolorCursor(display, cursor, foreground_color, background_color) changes the color of the specified cursor, and if \
the cursor is being displayed on a screen, the change is visible immediately."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XRecolorCursor", "Display*");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg2), arg2, 2, "XRecolorCursor", "Cursor");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg3), arg3, 3, "XRecolorCursor", "XColor");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg4), arg4, 4, "XRecolorCursor", "XColor");
  return(C_TO_XEN_INT(XRecolorCursor(XEN_TO_C_Display(arg1), XEN_TO_C_Cursor(arg2), 
				     XEN_TO_C_XColor(arg3), XEN_TO_C_XColor(arg4))));
}

static XEN gxm_XRebindKeysym(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XRebindKeysym "XRebindKeysym(display, keysym, list, mod_count, string, num_bytes) can be used to rebind the meaning of a KeySym for the client."
  /* DIFF: XRebindKeysym mod_list is list of keysyms
   */
  KeySym *ks;
  int len, val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XRebindKeysym", "Display*");
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg2), arg2, 2, "XRebindKeysym", "KeySym");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XRebindKeysym", "list of KeySym");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XRebindKeysym", "int");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg5), arg5, 5, "XRebindKeysym", "string");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XRebindKeysym", "int");
  len = XEN_TO_C_INT(arg4);
  if (len <= 0) return(XEN_FALSE);
  ks = XEN_TO_C_KeySyms(arg3, len);
  val = XRebindKeysym(XEN_TO_C_Display(arg1), 
		      XEN_TO_C_KeySym(arg2),
		      ks, len, 
		      (unsigned char *)XEN_TO_C_STRING(arg5), XEN_TO_C_INT(arg6));
  FREE(ks);
  return(C_TO_XEN_INT(val));
}

static XEN gxm_XReadBitmapFileData(XEN arg1)
{
  #define H_XReadBitmapFileData "int XReadBitmapFileData(filename) reads in a \
file containing a bitmap, in the same manner as XReadBitmapFile, but returns the data directly rather than creating a pixmap in the server."
  /* DIFF: XReadBitmapFileData omits last 5 args, returns as list
   */
  unsigned int w, h, i, j;
  int x, y;
  unsigned char **str = NULL; /* allocated by X? */
  int val, loc;
  XEN bits = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XReadBitmapFileData", "char*");
  val = XReadBitmapFileData(XEN_TO_C_STRING(arg1), &w, &h, str, &x, &y);
  loc = xm_protect(bits);
  for (i = 0; i < h; i++)
    for (j = 0; j < w; j++)
      bits = XEN_CONS(C_TO_XEN_INT((int)(str[i][j])), bits);
  if (str) free(str);
  xm_unprotect_at(loc);
  return(XEN_LIST_6(C_TO_XEN_INT(val),
		    C_TO_XEN_INT((int)w),
		    C_TO_XEN_INT((int)h),
		    bits,
		    C_TO_XEN_INT((int)x),
		    C_TO_XEN_INT((int)y)));
}

static XEN gxm_XReadBitmapFile(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XReadBitmapFile "int XReadBitmapFile(display, d, filename) reads in a file containing a bitmap."
  /* DIFF: XReadBitmapFile omits last 5 args, returns as list
   */
  unsigned int w, h;
  int x, y;
  int val;
  Pixmap *p = NULL; /* allocated by X? */
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XReadBitmapFile", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XReadBitmapFile", "Drawable");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XReadBitmapFile", "char*");
  val = XReadBitmapFile(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_STRING(arg3), &w, &h, p, &x, &y);
  return(XEN_LIST_6(C_TO_XEN_INT(val),
		    C_TO_XEN_INT((int)w),
		    C_TO_XEN_INT((int)h),
		    C_TO_XEN_Pixmap(*p),
		    C_TO_XEN_INT((int)x),
		    C_TO_XEN_INT((int)y)));
}

static XEN gxm_XRaiseWindow(XEN arg1, XEN arg2)
{
  #define H_XRaiseWindow "XRaiseWindow(display, w)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XRaiseWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XRaiseWindow", "Window");
  return(C_TO_XEN_INT(XRaiseWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XQueryTree(XEN arg1, XEN arg2)
{
  #define H_XQueryTree "Status XQueryTree(display, w) returns the root ID, the \
parent window ID, a pointer to the list of children windows and the number of children in the list for the specified window."
  /* DIFF: XQueryTree last 4 arg omit, returns (list val root parent (list children))
   */
  unsigned int arrlen;
  Window *ws;
  Window root, parent;
  int i, val, loc;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryTree", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XQueryTree", "Window");
  val = XQueryTree(XEN_TO_C_Display(arg1), 
		   XEN_TO_C_Window(arg2), 
		   &root, &parent, &ws, &arrlen);
  loc = xm_protect(lst);
  for (i = arrlen - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_Window(ws[i]), lst);
  XFree(ws);
  xm_unprotect_at(loc);
  return(XEN_LIST_4(C_TO_XEN_INT(val),
		    C_TO_XEN_Window(root),
		    C_TO_XEN_Window(parent),
		    lst));
}

static XEN gxm_XQueryTextExtents(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XQueryTextExtents "XQueryTextExtents(display, font, string) returns the bounding box of the specified 8-bit string."
  /* DIFF: XQueryTextExtents omits last 5 args, returns list 
   */
  XCharStruct *c;
  int fa, fd, dr, val;
  char *str;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryTextExtents", "Display*");
  XEN_ASSERT_TYPE(XEN_Font_P(arg2), arg2, 2, "XQueryTextExtents", "Font");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XQueryTextExtents", "char*");
  str = XEN_TO_C_STRING(arg3);
  c = (XCharStruct *)CALLOC(1, sizeof(XCharStruct));
  val = XQueryTextExtents(XEN_TO_C_Display(arg1), 
			  XEN_TO_C_Font(arg2), 
			  str, strlen(str), &dr, &fa, &fd, c); 
  return(XEN_LIST_5(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(dr),
		    C_TO_XEN_INT(fa),
		    C_TO_XEN_INT(fd),
		    WRAP_FOR_XEN_OBJ("XCharStruct", c)));
}

static XEN gxm_XQueryPointer(XEN arg1, XEN arg2)
{
  #define H_XQueryPointer "Bool XQueryPointer(display, w) returns the root window the pointer is logically on and the pointer \
coordinates relative to the root window's origin."
  /* DIFF: XQueryPointer last 7 args omit and rtn
   */
  Window w1, w2;
  int rx, ry, wx, wy;
  unsigned int mask;
  int rtn;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryPointer", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XQueryPointer", "Window");
  rtn = XQueryPointer(XEN_TO_C_Display(arg1), 
		      XEN_TO_C_Window(arg2), 
		      &w1, &w2, &rx, &ry, &wx, &wy, &mask);
  return(XEN_LIST_8(C_TO_XEN_BOOLEAN(rtn),
		    C_TO_XEN_Window(w1),
		    C_TO_XEN_Window(w2),
		    C_TO_XEN_INT(rx),
		    C_TO_XEN_INT(ry),
		    C_TO_XEN_INT(wx),
		    C_TO_XEN_INT(wy),
		    C_TO_XEN_ULONG((unsigned long)mask)));
}

static XEN gxm_XQueryKeymap(XEN arg1)
{
  #define H_XQueryKeymap "XQueryKeymap(display) returns a bit vector for the logical state of the keyboard, where each bit \
set to 1 indicates that the corresponding key is currently pressed down."
  /* DIFF: XQueryKeymap omits keys -> (list val keys)
   */
  char keys[32];
  int val, i, loc;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryKeymap", "Display*");
  val = XQueryKeymap(XEN_TO_C_Display(arg1), keys);
  loc = xm_protect(lst);
  for (i = 31; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_INT((int)keys[i]), lst);
  xm_unprotect_at(loc);
  return(XEN_CONS(C_TO_XEN_INT(val), lst));
}

static XEN gxm_XQueryExtension(XEN arg1, XEN arg2)
{
  #define H_XQueryExtension "Bool XQueryExtension(dpy, name) gets extension version information"
  /* DIFF: XQueryExtension dpy name [op er er] -> (list val op er er)
   */
  int op, err1, err2, val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryExtension", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XQueryExtension", "char*");
  val = XQueryExtension(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2), &op, &err1, &err2);
  return(XEN_LIST_4(C_TO_XEN_BOOLEAN(val),
		    C_TO_XEN_INT(op),
		    C_TO_XEN_INT(err1),
		    C_TO_XEN_INT(err2)));
}

static XEN gxm_XQueryColors(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XQueryColors "XQueryColors(display, colormap, defs_in_out, ncolors)"
  int i, len, rtn;
  XEN lst;
  XColor *cols;
  XColor *col;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryColors", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XQueryColors", "Colormap");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XQueryColors", "XColor list");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg4), arg4, 4, "XQueryColors", "int");
  if (XEN_INTEGER_P(arg4)) len = XEN_TO_C_INT(arg4); else len = XEN_LIST_LENGTH(arg3);
  lst = XEN_COPY_ARG(arg3);
  cols = (XColor *)CALLOC(len, sizeof(XColor));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(lst)); i++, lst = XEN_CDR(lst))
    {
      if (XEN_XColor_P(XEN_CAR(lst)))
	{
	  col = XEN_TO_C_XColor(XEN_CAR(lst));
	  cols[i].pixel = col->pixel;
	}
      else XEN_ASSERT_TYPE(0, XEN_CAR(lst), i, c__FUNCTION__, "an XColor");
    }
  rtn = XQueryColors(XEN_TO_C_Display(arg1), XEN_TO_C_Colormap(arg2), cols, len);
  for (i = 0; i < len; i++)
    {
      col = XEN_TO_C_XColor(XEN_LIST_REF(arg3, i));
      col->red = cols[i].red;
      col->green = cols[i].green;
      col->blue = cols[i].blue;
      col->flags = cols[i].flags;
    }
  FREE(cols);
  return(C_TO_XEN_INT(rtn));
}

static XEN gxm_XQueryColor(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XQueryColor "XQueryColor(display, colormap, def_in_out) returns the current RGB value for the pixel in the XColor structure \
and sets the DoRed, DoGreen, DoBlue flags."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryColor", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XQueryColor", "Colormap");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg3), arg3, 3, "XQueryColor", "XColor*");
  return(C_TO_XEN_INT(XQueryColor(XEN_TO_C_Display(arg1), XEN_TO_C_Colormap(arg2), XEN_TO_C_XColor(arg3))));
}

static XEN gxm_XQueryBestTile(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XQueryBestTile "Status XQueryBestTile(display, which_screen, width, height)  returns the best or \
closest size, that is, the size that can be tiled fastest on the screen specified by which_screen."
  /* DIFF: XQueryBestTile dpy scr wid hgt [wd hg] -> (list status wd hg)
   */
  unsigned int w, h;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryBestTile", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XQueryBestTile", "Drawable");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XQueryBestTile", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XQueryBestTile", "unsigned int");
  val = XQueryBestTile(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
		       XEN_TO_C_ULONG(arg3), XEN_TO_C_ULONG(arg4), &w, &h);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_ULONG(w),
		    C_TO_XEN_ULONG(h)));
}

static XEN gxm_XQueryBestStipple(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XQueryBestStipple "Status XQueryBestStipple(display, which_screen, width, height)  returns the \
best or closest size, that is, the size that can be stippled fastest on the screen specified by which_screen."
  /* DIFF: XQueryBestStipple dpy scr wid hgt [wd hg] -> (list status wd hg)
   */
  unsigned int w, h;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryBestStipple", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XQueryBestStipple", "Drawable");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XQueryBestStipple", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XQueryBestStipple", "unsigned int");
  val = XQueryBestStipple(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
			  XEN_TO_C_ULONG(arg3), XEN_TO_C_ULONG(arg4), 
			  &w, &h);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_ULONG(w),
		    C_TO_XEN_ULONG(h)));
}

static XEN gxm_XQueryBestSize(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XQueryBestSize "Status XQueryBestSize(display, class, which_screen, width, height)  returns the \
best or closest size to the specified size."
  /* DIFF: XQueryBestSize dpy cls scr wid hgt [wd hg] -> (list status wd hg)
   */
  unsigned int w, h;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryBestSize", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XQueryBestSize", "int");
  XEN_ASSERT_TYPE(XEN_Window_P(arg3), arg3, 3, "XQueryBestSize", "Drawable");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XQueryBestSize", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XQueryBestSize", "unsigned int");
  val = XQueryBestSize(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_Window(arg3), 
		       XEN_TO_C_ULONG(arg4), XEN_TO_C_ULONG(arg5), 
		       &w, &h);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_ULONG(w),
		    C_TO_XEN_ULONG(h)));
}

static XEN gxm_XQueryBestCursor(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XQueryBestCursor "Status XQueryBestCursor(display, d, width, height) provides a way to find \
out what size cursors are actually possible on the display."
  /* DIFF: XQueryBestCursor dpy d wid hgt [wd hg] -> (list status wd hg)
   */
  unsigned int w, h;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryBestCursor", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XQueryBestCursor", "Drawable");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XQueryBestCursor", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XQueryBestCursor", "unsigned int");
  val = XQueryBestCursor(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
			 XEN_TO_C_ULONG(arg3), XEN_TO_C_ULONG(arg4), 
			 &w, &h);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_ULONG(w),
		    C_TO_XEN_ULONG(h)));
}

static XEN gxm_XQLength(XEN arg1)
{
  #define H_QLength "returns the length of the event queue for the connected display."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQLength", "Display*");
  return(C_TO_XEN_INT(XQLength(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XPutImage(XEN args)
{
  #define H_XPutImage "XPutImage(display, d, gc, image, src_x, src_y, dest_x, dest_y, width, height) combines an image with a rectangle \
of the specified drawable."
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  arg10 = XEN_LIST_REF(args, 9);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XPutImage", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XPutImage", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XPutImage", "GC");
  XEN_ASSERT_TYPE(XEN_XImage_P(arg4), arg4, 4, "XPutImage", "XImage*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XPutImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XPutImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XPutImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XPutImage", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg9), arg9, 9, "XPutImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg10), arg10, 10, "XPutImage", "unsigned int");
  return(C_TO_XEN_INT(XPutImage(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
				XEN_TO_C_GC(arg3), XEN_TO_C_XImage(arg4), XEN_TO_C_INT(arg5), XEN_TO_C_INT(arg6), 
				XEN_TO_C_INT(arg7), XEN_TO_C_INT(arg8), XEN_TO_C_ULONG(arg9), XEN_TO_C_ULONG(arg10))));
}

static XEN gxm_XPutBackEvent(XEN arg1, XEN arg2)
{
  #define H_XPutBackEvent "XPutBackEvent(display, event) pushes an event back onto the head of the display's event queue by copying the event into the queue."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XPutBackEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_XEvent_P(arg2), arg2, 2, "XPutBackEvent", "XEvent*");
  return(C_TO_XEN_INT(XPutBackEvent(XEN_TO_C_Display(arg1), XEN_TO_C_XEvent(arg2))));
}

static XEN gxm_XProtocolVersion(XEN arg1)
{
  #define H_ProtocolVersion "ProtocolVersion(display) returns the major version number (11) of the X protocol associated with the connected display."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XProtocolVersion", "Display*");
  return(C_TO_XEN_INT(XProtocolVersion(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XProtocolRevision(XEN arg1)
{
  #define H_ProtocolRevision "ProtocolRevision(display) returns the minor protocol revision number of the X server."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XProtocolRevision", "Display*");
  return(C_TO_XEN_INT(XProtocolRevision(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XPlanesOfScreen(XEN arg1)
{
  #define H_PlanesOfScreen "PlanesOfScreen(screen) returns the number of planes in the root window of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XPlanesOfScreen", "Screen*");
  return(C_TO_XEN_INT(XPlanesOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XPending(XEN arg1)
{
  #define H_XPending "int XPending(display) returns the number of events that have been received from the X server but have not been removed \
from the event queue."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XPending", "Display*");
  return(C_TO_XEN_INT(XPending(XEN_TO_C_Display(arg1))));
}

static XEN xm_XPeekIfEventProc = XEN_FALSE;
static Bool gxm_XPeekIfEventProc(Display *dpy, XEvent *e, XtPointer p)
{
  return(XEN_TO_C_BOOLEAN(XEN_CALL_3(xm_XPeekIfEventProc, C_TO_XEN_Display(dpy), C_TO_XEN_XEvent(e), (XEN)(p), c__FUNCTION__)));
}

static XEN gxm_XPeekIfEvent(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XPeekIfEvent "XPeekIfEvent(display, predicate, arg) returns only when the specified predicate procedure returns #t for an event."
  /* DIFF: XPeekIfEvent dpy [evrtn] proc ptr -> (list val evrtn)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XPeekIfEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg2) && (XEN_REQUIRED_ARGS_OK(arg2, 3)), arg2, 2, "XPeekIfEvent", "(Bool_Proc dpy ev data)");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XPeekIfEvent(XEN_TO_C_Display(arg1), 
		     e, 
		     (Bool (*)(Display *d, XEvent *ev, char *p))gxm_XPeekIfEventProc, /* C++ insists on the cast */
		     (XPointer)arg3);
  return(XEN_LIST_2(C_TO_XEN_INT(val), C_TO_XEN_XEvent_OBJ(e)));
}

static XEN gxm_XPeekEvent(XEN arg1)
{
  #define H_XPeekEvent "XPeekEvent(display) returns the first event from the event queue, but it does not remove the event from the queue."
  /* DIFF: XPeekEvent dpy [ev] -> (list int event)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XPeekEvent", "Display*");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XPeekEvent(XEN_TO_C_Display(arg1), e);
  return(XEN_LIST_2(C_TO_XEN_INT(val), C_TO_XEN_XEvent_OBJ(e)));
}

static XEN gxm_XParseGeometry(XEN arg1)
{
  #define H_XParseGeometry "int XParseGeometry(parsestring) returns a bitmask that \
indicates which of the four values (width, height, xoffset, and yoffset) were actually found in the string and whether the x and y values are negative. "
  /* DIFF: XParseGeometry str [x y w h] -> (list bit x y w h)
   */
  int x = 0, y = 0, val = 0;
  unsigned int w = 0, h = 0;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XParseGeometry", "char*");
  val = XParseGeometry(XEN_TO_C_STRING(arg1), &x, &y, &w, &h);
  return(XEN_LIST_5(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(x),
		    C_TO_XEN_INT(y),
		    C_TO_XEN_INT(w),
		    C_TO_XEN_INT(h)));
}

static XEN gxm_XParseColor(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XParseColor "Status XParseColor(display, colormap, spec, exact_def_return) looks up the string name of a color with respect \
to the screen associated with the specified colormap."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XParseColor", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XParseColor", "Colormap");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XParseColor", "char*");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg4), arg4, 4, "XParseColor", "XColor");
  return(C_TO_XEN_INT(XParseColor(XEN_TO_C_Display(arg1), XEN_TO_C_Colormap(arg2), XEN_TO_C_STRING(arg3), XEN_TO_C_XColor(arg4))));
}

static XEN gxm_XNoOp(XEN arg1)
{
  #define H_XNoOp "XNoOp(dpy) sends the server a no-op for exercise"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XNoOp", "Display*");
  return(C_TO_XEN_INT(XNoOp(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XNextEvent(XEN arg1)
{
  #define H_XNextEvent "XNextEvent(display) copies the first event from the event queue into the specified XEvent structure \
and then removes it from the queue."
  /* DIFF: XNextEvent dpy [ev] -> (list int event)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XNextEvent", "Display*");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XNextEvent(XEN_TO_C_Display(arg1), e);
  return(XEN_LIST_2(C_TO_XEN_INT(val), C_TO_XEN_XEvent_OBJ(e)));
}

static XEN gxm_XMoveWindow(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XMoveWindow "XMoveWindow(display, w, x, y) moves the specified window to the specified x and y coordinates, but it does not \
change the window's size, raise the window, or change the mapping state of the window."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XMoveWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XMoveWindow", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XMoveWindow", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XMoveWindow", "int");
  return(C_TO_XEN_INT(XMoveWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4))));
}

static XEN gxm_XMoveResizeWindow(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XMoveResizeWindow "XMoveResizeWindow(display, w, x, y, width, height) changes the size and location of the specified window without raising it."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XMoveResizeWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XMoveResizeWindow", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XMoveResizeWindow", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XMoveResizeWindow", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XMoveResizeWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XMoveResizeWindow", "unsigned int");
  return(C_TO_XEN_INT(XMoveResizeWindow(XEN_TO_C_Display(arg1), 
					XEN_TO_C_Window(arg2), 
					XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), 
					XEN_TO_C_ULONG(arg5), XEN_TO_C_ULONG(arg6))));
}

static XEN gxm_XMinCmapsOfScreen(XEN arg1)
{
  #define H_MinCmapsOfScreen "returns the minimum number of installed colormaps supported by the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XMinCmapsOfScreen", "Screen*");
  return(C_TO_XEN_INT(XMinCmapsOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XMaxCmapsOfScreen(XEN arg1)
{
  #define H_MaxCmapsOfScreen "returns the maximum number of installed colormaps supported by the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XMaxCmapsOfScreen", "Screen*");
  return(C_TO_XEN_INT(XMaxCmapsOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XMaskEvent(XEN arg1, XEN arg2)
{
  #define H_XMaskEvent "XMaskEvent(display, event_mask) searches the event queue for the events associated with the specified mask."
  /* DIFF: XMaskEvent dpy mask [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XMaskEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XMaskEvent", "long");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XMaskEvent(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), e);
  return(XEN_LIST_2(C_TO_XEN_INT(val), C_TO_XEN_XEvent_OBJ(e)));
}

static XEN gxm_XMapWindow(XEN arg1, XEN arg2)
{
  #define H_XMapWindow "XMapWindow(display, w) maps the window and all of its subwindows that have had map requests."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XMapWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XMapWindow", "Window");
  return(C_TO_XEN_INT(XMapWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XMapSubwindows(XEN arg1, XEN arg2)
{
  #define H_XMapSubwindows "XMapSubwindows(display, w) maps all subwindows for a specified window in top-to-bottom stacking order."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XMapSubwindows", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XMapSubwindows", "Window");
  return(C_TO_XEN_INT(XMapSubwindows(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XMapRaised(XEN arg1, XEN arg2)
{
  #define H_XMapRaised "XMapRaised(display, w) maps the window and all of its subwindows that have had map requests, and raises the \
specified window to the top of the stack."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XMapRaised", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XMapRaised", "Window");
  return(C_TO_XEN_INT(XMapRaised(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XLowerWindow(XEN arg1, XEN arg2)
{
  #define H_XLowerWindow "XLowerWindow(display, w) lowers the specified window to the bottom of the stack so that it does not obscure any sibling windows."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XLowerWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XLowerWindow", "Window");
  return(C_TO_XEN_INT(XLowerWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XLookupColor(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XLookupColor "Status XLookupColor(display, colormap, color_name, exact_def_return, screen_def_return) looks up the string name \
of a color with respect to the screen associated with the specified colormap."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XLookupColor", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XLookupColor", "Colormap");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XLookupColor", "char*");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg4), arg4, 4, "XLookupColor", "XColor*");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg5), arg5, 5, "XLookupColor", "XColor*");
  return(C_TO_XEN_INT(XLookupColor(XEN_TO_C_Display(arg1), 
				   XEN_TO_C_Colormap(arg2), 
				   XEN_TO_C_STRING(arg3), 
				   XEN_TO_C_XColor(arg4), XEN_TO_C_XColor(arg5))));
}

static XEN gxm_XKillClient(XEN arg1, XEN arg2)
{
  #define H_XKillClient "XKillClient(display, resource) forces a close-down of the client that created the resource"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XKillClient", "Display*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XKillClient", "XID");
  return(C_TO_XEN_INT(XKillClient(XEN_TO_C_Display(arg1), XEN_TO_C_ULONG(arg2))));
}

static XEN gxm_XKeysymToKeycode(XEN arg1, XEN arg2)
{
  #define H_XKeysymToKeycode "KeyCode XKeysymToKeycode(display, keysym)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XKeysymToKeycode", "Display*");
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg2), arg2, 2, "XKeysymToKeycode", "KeySym");
  return(C_TO_XEN_KeyCode(XKeysymToKeycode(XEN_TO_C_Display(arg1), XEN_TO_C_KeySym(arg2))));
}

static XEN gxm_XInstallColormap(XEN arg1, XEN arg2)
{
  #define H_XInstallColormap "XInstallColormap(display, colormap) installs the specified colormap for its associated screen."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XInstallColormap", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XInstallColormap", "Colormap");
  return(C_TO_XEN_INT(XInstallColormap(XEN_TO_C_Display(arg1), XEN_TO_C_Colormap(arg2))));
}

static XEN gxm_XImageByteOrder(XEN arg1)
{
  #define H_ImageByteOrder "specifies the required byte order for images for each scanline unit in XY format (bitmap) or for each pixel value in Z format."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XImageByteOrder", "Display*");
  return(C_TO_XEN_INT(XImageByteOrder(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XIfEvent(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XIfEvent "XIfEvent(display, predicate, arg) completes only when the specified predicate procedure returns #t for \
an event, which indicates an event in the queue matches."
  /* DIFF: XIfEvent dpy [ev] proc ptr -> (list val ev)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XIfEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg2) && (XEN_REQUIRED_ARGS_OK(arg2, 3)), arg2, 2, "XIfEvent", "(Bool_Proc dpy ev data)");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XIfEvent(XEN_TO_C_Display(arg1), 
		 e, 
		 (Bool (*)(Display *d1, XEvent *e1, char *p1))gxm_XPeekIfEventProc, 
		 (char *)arg3);
  return(XEN_LIST_2(C_TO_XEN_INT(val), C_TO_XEN_XEvent_OBJ(e)));
}

static XEN gxm_XHeightOfScreen(XEN arg1)
{
  #define H_HeightOfScreen "XHeightOfScreen(screen) returns the height of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XHeightOfScreen", "Screen*");
  return(C_TO_XEN_INT(XHeightOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XHeightMMOfScreen(XEN arg1)
{
  #define H_HeightMMOfScreen "XHeightMMOfScreen(screen) returns the height of the specified screen in millimeters."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XHeightMMOfScreen", "Screen*");
  return(C_TO_XEN_INT(XHeightMMOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XGrabServer(XEN arg1)
{
  #define H_XGrabServer "XGrabServer(display) disables processing of requests and close downs on all other connections than the one this request arrived on."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGrabServer", "Display*");
  return(C_TO_XEN_INT(XGrabServer(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XGrabPointer(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8, XEN arg9)
{
  #define H_XGrabPointer "int XGrabPointer(display, grab_window, owner_events, event_mask, pointer_mode, keyboard_mode, confine_to, cursor, time) \
actively grabs control of the pointer and returns GrabSuccess if the grab was successful."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGrabPointer", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGrabPointer", "Window");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XGrabPointer", "Bool");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XGrabPointer", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XGrabPointer", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XGrabPointer", "int");
  XEN_ASSERT_TYPE(XEN_Window_P(arg7), arg7, 7, "XGrabPointer", "Window");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg8), arg8, 8, "XGrabPointer", "Cursor");
  XEN_ASSERT_TYPE(XEN_Time_P(arg9), arg9, 9, "XGrabPointer", "Time");
  return(C_TO_XEN_INT(XGrabPointer(XEN_TO_C_Display(arg1), 
				   XEN_TO_C_Window(arg2), 
				   XEN_TO_C_BOOLEAN(arg3), 
				   XEN_TO_C_ULONG(arg4), XEN_TO_C_INT(arg5), XEN_TO_C_INT(arg6), 
				   XEN_TO_C_Window(arg7), XEN_TO_C_Cursor(arg8), 
				   XEN_TO_C_Time(arg9))));
}

static XEN gxm_XGrabKeyboard(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XGrabKeyboard "int XGrabKeyboard(display, grab_window, owner_events, pointer_mode, keyboard_mode, time) actively grabs control of \
the keyboard and generates FocusIn and FocusOut events."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGrabKeyboard", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGrabKeyboard", "Window");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XGrabKeyboard", "Bool");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XGrabKeyboard", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XGrabKeyboard", "int");
  XEN_ASSERT_TYPE(XEN_Time_P(arg6), arg6, 6, "XGrabKeyboard", "Time");
  return(C_TO_XEN_INT(XGrabKeyboard(XEN_TO_C_Display(arg1), 
				    XEN_TO_C_Window(arg2), 
				    XEN_TO_C_BOOLEAN(arg3), XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
				    XEN_TO_C_Time(arg6))));
}

static XEN gxm_XGrabKey(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XGrabKey "XGrabKey(display, keycode, modifiers, grab_window, owner_events, pointer_mode, keyboard_mode) establishes a passive \
grab on the keyboard."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGrabKey", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XGrabKey", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XGrabKey", "unsigned int");
  XEN_ASSERT_TYPE(XEN_Window_P(arg4), arg4, 4, "XGrabKey", "Window");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg5), arg5, 5, "XGrabKey", "Bool");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XGrabKey", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XGrabKey", "int");
  return(C_TO_XEN_INT(XGrabKey(XEN_TO_C_Display(arg1), 
			       XEN_TO_C_INT(arg2), XEN_TO_C_ULONG(arg3), 
			       XEN_TO_C_Window(arg4), 
			       XEN_TO_C_BOOLEAN(arg5), XEN_TO_C_INT(arg6), XEN_TO_C_INT(arg7))));
}

static XEN gxm_XGrabButton(XEN args)
{
  #define H_XGrabButton "XGrabButton(display, button, modifiers, grab_window, owner_events, event_mask, pointer_mode, keyboard_mode, confine_to, cursor) \
establishes a passive grab."
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  arg10 = XEN_LIST_REF(args, 9);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGrabButton", "Display*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XGrabButton", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XGrabButton", "unsigned int");
  XEN_ASSERT_TYPE(XEN_Window_P(arg4), arg4, 4, "XGrabButton", "Window");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg5), arg5, 5, "XGrabButton", "Bool");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XGrabButton", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XGrabButton", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XGrabButton", "int");
  XEN_ASSERT_TYPE(XEN_Window_P(arg9), arg9, 9, "XGrabButton", "Window");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg10), arg10, 10, "XGrabButton", "Cursor");
  return(C_TO_XEN_INT(XGrabButton(XEN_TO_C_Display(arg1), 
				  XEN_TO_C_ULONG(arg2), XEN_TO_C_ULONG(arg3), 
				  XEN_TO_C_Window(arg4), XEN_TO_C_BOOLEAN(arg5), 
				  XEN_TO_C_ULONG(arg6), XEN_TO_C_INT(arg7), 
				  XEN_TO_C_INT(arg8), 
				  XEN_TO_C_Window(arg9), XEN_TO_C_Cursor(arg10))));
}

static XEN gxm_XGetWindowAttributes(XEN arg1, XEN arg2)
{ 
  #define H_XGetWindowAttributes "Status XGetWindowAttributes(display, w) returns the current attributes for the \
specified window to an XWindowAttributes structure."
  /* DIFF: XGetWindowAttributes omits and rtns arg3 
   */
  XWindowAttributes *w;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetWindowAttributes", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetWindowAttributes", "Window");
  w = (XWindowAttributes *)CALLOC(1, sizeof(XWindowAttributes));
  XGetWindowAttributes(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), w);
  return(C_TO_XEN_XWindowAttributes(w));
}

static XEN gxm_XGetWindowProperty(XEN args)
{
  #define H_XGetWindowProperty "int XGetWindowProperty(display, w, property, long_offset, long_length, delete, req_type) \
returns the actual type of the property; the actual format of the property; the \
number of 8-bit, 16-bit, or 32-bit items transferred; the number of bytes remaining to be read in the property; and a pointer to the data \
actually returned."
  /* DIFF: XGetWindowProperty omit trailing 5 args, rtn as list
   */
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7; 
  XEN result = XEN_FALSE;
  Atom a;
  int ret, val;
  unsigned long len = 0, bytes;
  unsigned char *data;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetWindowProperty", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetWindowProperty", "Window");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XGetWindowProperty", "Atom");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XGetWindowProperty", "long");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XGetWindowProperty", "long");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg6), arg6, 6, "XGetWindowProperty", "Bool");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg7), arg7, 7, "XGetWindowProperty", "Atom");
  val = XGetWindowProperty(XEN_TO_C_Display(arg1), 
			   XEN_TO_C_Window(arg2), 
			   XEN_TO_C_Atom(arg3), 
			   XEN_TO_C_INT(arg4), 
			   XEN_TO_C_INT(arg5), 
			   XEN_TO_C_BOOLEAN(arg6), 
			   XEN_TO_C_Atom(arg7), 
			   &a, &ret, &len, &bytes, &data);
  if ((a != (Atom)None) && (len > 0))
    {
#if HAVE_GUILE && (HAVE_SCM_MEM2STRING || HAVE_SCM_C_MAKE_RECTANGULAR)
      if (a == XA_STRING)
	result = C_TO_XEN_STRING((char *)data);
      else result = C_TO_XEN_STRINGN((char *)data, len * ret / 8); /* is this a good idea? -- perhaps a void pointer here? */
#else
      result = C_TO_XEN_STRING((char *)data);
#endif
      if (data) XFree(data);
    }
  return(XEN_LIST_6(C_TO_XEN_INT(val),
		    C_TO_XEN_Atom(a),
		    C_TO_XEN_INT(ret),
		    C_TO_XEN_ULONG(len),
		    C_TO_XEN_ULONG(bytes),
		    result));
}

static XEN gxm_XGetTransientForHint(XEN arg1, XEN arg2)
{
  #define H_XGetTransientForHint "Status XGetTransientForHint(display, w) returns the WM_TRANSIENT_FOR property for the specified window."
  /* DIFF: XGetTransientForHint omit and rtn last arg
   */
  Window w;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetTransientForHint", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetTransientForHint", "Window");
  val = XGetTransientForHint(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), &w);
  return(XEN_LIST_2(C_TO_XEN_INT(val),
		    C_TO_XEN_Window(w)));
}

static XEN gxm_XGetScreenSaver(XEN arg1)
{
  #define H_XGetScreenSaver "XGetScreenSaver(display) gets the current screen saver values."
  /* DIFF: XGetScreenSaver omit and rtn last 4 args
   */
  int a, b, c ,d, val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetScreenSaver", "Display*");
  val = XGetScreenSaver(XEN_TO_C_Display(arg1), &a, &b, &c, &d);
  return(XEN_LIST_5(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(a),
		    C_TO_XEN_INT(b),
		    C_TO_XEN_INT(c),
		    C_TO_XEN_INT(d)));
}

static XEN gxm_XGetPointerMapping(XEN arg1, XEN ignore, XEN arg3)
{
  #define H_XGetPointerMapping "int XGetPointerMapping(display, ignored, len) returns the current mapping of the pointer."
  /* DIFF: XGetPointerMapping ignores arg2, returns list
   * (XGetPointerMapping (XtDisplay (cadr (main-widgets))) 0 3)
   */
  int i, len, loc, rtn;
  unsigned char *map;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetPointerMapping", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XGetPointerMapping", "int");
  len = XEN_TO_C_INT(arg3);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg3, 3, "XGetPointerMapping", "positive integer");
  map = (unsigned char *)CALLOC(len, sizeof(unsigned char));
  rtn = XGetPointerMapping(XEN_TO_C_Display(arg1), map, len);
  if (len > rtn) len = rtn;
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_INT((int)(map[i])), lst);
  FREE(map);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XGetPointerControl(XEN arg1)
{
  #define H_XGetPointerControl "XGetPointerControl(display) \
returns the pointer's current acceleration multiplier and acceleration threshold."
  /* DIFF: XGetPointerControl omits and return last 3 args
   */
  int val, num, den, thresh;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetPointerControl", "Display*");
  val = XGetPointerControl(XEN_TO_C_Display(arg1), &num, &den, &thresh);
  return(XEN_LIST_4(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(num),
		    C_TO_XEN_INT(den),
		    C_TO_XEN_INT(thresh)));
}

static XEN gxm_XGetKeyboardControl(XEN arg1)
{
  #define H_XGetKeyboardControl "XGetKeyboardControl(display) returns the current control values for the keyboard \
to the XKeyboardState structure."
  /* DIFF: XGetKeyboardControl omits arg2 and rtns list of fields
   */
  XKeyboardState ks;
  XEN v;
  XEN *velts;
  int i, loc;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetKeyboardControl", "Display*");
  XGetKeyboardControl(XEN_TO_C_Display(arg1), &ks);
  v = XEN_MAKE_VECTOR(32, XEN_ZERO);
  loc = xm_protect(v);
  velts = XEN_VECTOR_ELEMENTS(v);
  /* VECTOR_SET here */
  for (i = 0; i < 32; i++)
    velts[i] = C_TO_XEN_INT((int)(ks.auto_repeats[i]));
  xm_unprotect_at(loc);
  return(XEN_LIST_7(C_TO_XEN_INT(ks.key_click_percent),
		    C_TO_XEN_INT(ks.bell_percent),
		    C_TO_XEN_INT(ks.bell_pitch),
		    C_TO_XEN_INT(ks.bell_duration),
		    C_TO_XEN_INT(ks.led_mask),
		    C_TO_XEN_INT(ks.global_auto_repeat),
		    v));
}

static XEN gxm_XGetInputFocus(XEN arg1)
{
  #define H_XGetInputFocus "XGetInputFocus(display) returns the focus window and the current focus state."
  /* DIFF: XGetInputFocus omit and rtn last 2 args
   */
  Window w;
  int r, val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetInputFocus", "Display*");
  val = XGetInputFocus(XEN_TO_C_Display(arg1), &w, &r);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_Window(w),
		    C_TO_XEN_INT(r)));
}

static XEN gxm_XGetIconName(XEN arg1, XEN arg2)
{
  #define H_XGetIconName "Status XGetIconName(display, w) returns the name to be displayed in the specified window's icon."
  /* DIFF: XGetIconName omits and returns arg3
   */
  char *str;
  int val;
  XEN res = XEN_FALSE;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetIconName", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetIconName", "Window");
  val = XGetIconName(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), &str);
  if (val != 0)
    {
      res = C_TO_XEN_STRING(str);
      XFree(str);
    }
  return(res);
}

static XEN gxm_XGetGeometry(XEN arg1, XEN arg2)
{
  #define H_XGetGeometry "Status XGetGeometry(display, d) returns the root window and the current geometry of the drawable."
  /* DIFF: XGetGeometry omits last 7 args and returns list
   */
  unsigned int wr, hr, br, dr;
  int xr, yr, val;
  Window root;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetGeometry", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetGeometry", "Drawable");
  val = XGetGeometry(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
		     &root, &xr, &yr, &wr, &hr, &br, &dr);
  return(XEN_LIST_8(C_TO_XEN_INT(val),
		    C_TO_XEN_Window(root),
		    C_TO_XEN_INT(xr),
		    C_TO_XEN_INT(yr),
		    C_TO_XEN_ULONG(wr),
		    C_TO_XEN_ULONG(hr),
		    C_TO_XEN_ULONG(br),
		    C_TO_XEN_ULONG(dr)));
}

static XEN gxm_XGetGCValues(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XGetGCValues "Status XGetGCValues(display, gc, valuemask) returns the components specified by valuemask for the specified GC."
  /* DIFF: XGetGCValues omits and returns last arg
   */
  XGCValues *val;
  int rtn;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetGCValues", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XGetGCValues", "GC");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XGetGCValues", "ulong");
  val = (XGCValues *)CALLOC(1, sizeof(XGCValues));
  rtn = XGetGCValues(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_ULONG(arg3), val);
  return(XEN_LIST_2(C_TO_XEN_INT(rtn),
		    C_TO_XEN_XGCValues(val)));
}

static XEN gxm_XGetFontProperty(XEN arg1, XEN arg2)
{
  #define H_XGetFontProperty "Bool XGetFontProperty(font_struct, atom) returns the value of the specified font property. "
  /* DIFF: XGetFontProperty omits and rtns last arg
   */
  Bool val = False;
  unsigned long prop = 0;
  XEN_ASSERT_TYPE(XEN_XFontStruct_P(arg1), arg1, 1, "XGetFontProperty", "XFontStruct*");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XGetFontProperty", "Atom");
  val = XGetFontProperty(XEN_TO_C_XFontStruct(arg1), XEN_TO_C_Atom(arg2), &prop);
  return(XEN_LIST_2(C_TO_XEN_BOOLEAN(val),
		    C_TO_XEN_ULONG(prop)));
}

static XEN gxm_XGetErrorText(XEN arg1, XEN arg2, XEN ignore, XEN arg4)
{
  #define H_XGetErrorText "XGetErrorText(display, code, buffer_return, length) copies a null-terminated string describing the specified error \
code into the specified buffer."
  /* DIFF: XGetErrorText ignores arg3, returns text
   */
  char *buf;
  int len, val;
  XEN str;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetErrorText", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XGetErrorText", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XGetErrorText", "int");
  len = XEN_TO_C_INT(arg4);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg4, 4, "XGetErrorText", "positive integer");
  buf = (char *)CALLOC(len, sizeof(char));
  val = XGetErrorText(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), buf, len);
  str = C_TO_XEN_STRING(buf);
  FREE(buf);
  return(XEN_LIST_2(C_TO_XEN_INT(val),
		    str));
}

static XEN gxm_XGeometry(XEN args)
{
  #define H_XGeometry "int XGeometry(dpy, screen, position, default_position, bwidth, fwidth, fheight, xadder, yadder) calculates \
window geometry given user geometry string and default geometry"
  /* DIFF: XGetGeometry omits trailing 4 args and returns them
   */
  int x, y, w, h, val;
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGeometry", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XGeometry", "int");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XGeometry", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XGeometry", "char*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XGeometry", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XGeometry", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XGeometry", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XGeometry", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg9), arg9, 9, "XGeometry", "int");
  val = XGeometry(XEN_TO_C_Display(arg1), 
		  XEN_TO_C_INT(arg2), 
		  XEN_TO_C_STRING(arg3), 
		  XEN_TO_C_STRING(arg4), 
		  XEN_TO_C_ULONG(arg5), 
		  XEN_TO_C_ULONG(arg6), 
		  XEN_TO_C_ULONG(arg7), 
		  XEN_TO_C_INT(arg8), 
		  XEN_TO_C_INT(arg9), 
		  &x, &y, &w, &h);
  return(XEN_LIST_5(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(x),
		    C_TO_XEN_INT(y),
		    C_TO_XEN_INT(w),
		    C_TO_XEN_INT(h)));
}

static XEN gxm_XFreePixmap(XEN arg1, XEN arg2)
{
  #define H_XFreePixmap "XFreePixmap(display, pixmap) first deletes the association between the pixmap ID and the pixmap."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFreePixmap", "Display*");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg2), arg2, 2, "XFreePixmap", "Pixmap");
  return(C_TO_XEN_INT(XFreePixmap(XEN_TO_C_Display(arg1), XEN_TO_C_Pixmap(arg2))));
}

static XEN gxm_XFreeModifiermap(XEN arg1)
{
  #define H_XFreeModifiermap "XFreeModifiermap(modmap) frees the specified XModifierKeymap structure."
  XEN_ASSERT_TYPE(XEN_XModifierKeymap_P(arg1), arg1, 1, "XFreeModifiermap", "XModifierKeymap*");
  return(C_TO_XEN_INT(XFreeModifiermap(XEN_TO_C_XModifierKeymap(arg1))));
}

static XEN gxm_XFreeGC(XEN arg1, XEN arg2)
{
  #define H_XFreeGC "XFreeGC(display, gc) destroys the specified GC as well as all the associated storage."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFreeGC", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XFreeGC", "GC");
  return(C_TO_XEN_INT(XFreeGC(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2))));
}

static XEN gxm_XFreeFontPath(XEN ignore)
{
  #define H_XFreeFontPath "XFreeFontPath(list) frees the data allocated by XGetFontPath (a no-op in xm)."
  /* DIFF: XFreeFontPath is no-op
   */
  return(XEN_FALSE);
}

static XEN gxm_XFreeFontNames(XEN ignore)
{
  #define H_XFreeFontNames "XFreeFontNames(list) frees the array and strings returned by XListFonts or XListFontsWithInfo (a no-op in xm)."
  /* DIFF: XFreeFontNames is no-op
   */
  return(XEN_FALSE);
}

static XEN gxm_XFreeFontInfo(XEN ignore1, XEN ignore2, XEN ignore3)
{
  #define H_XFreeFontInfo "XFreeFontInfo(names, free_info, actual_count) frees a font structure or an array of font structures, and \
optionally an array of font names (a no-op in xm)."
  /* DIFF: XFreeFontInfo is a no-op 
   */
  return(XEN_FALSE);
}

static XEN gxm_XFreeFont(XEN arg1, XEN arg2)
{
  #define H_XFreeFont "XFreeFont(display, font_struct) deletes the association between the font resource ID and the specified font and \
frees the XFontStruct structure."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFreeFont", "Display*");
  XEN_ASSERT_TYPE(XEN_XFontStruct_P(arg2), arg2, 2, "XFreeFont", "XFontStruct*");
  return(C_TO_XEN_INT(XFreeFont(XEN_TO_C_Display(arg1), XEN_TO_C_XFontStruct(arg2))));
}

static XEN gxm_XFreeExtensionList(XEN ignore)
{
  /* DIFF: XFreeExtensionList is a no-op
   */
  return(XEN_FALSE);
}

static XEN gxm_XFreeCursor(XEN arg1, XEN arg2)
{
  #define H_XFreeCursor "XFreeCursor(display, cursor) deletes the association between the cursor resource ID and the specified cursor."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFreeCursor", "Display*");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg2), arg2, 2, "XFreeCursor", "Cursor");
  return(C_TO_XEN_INT(XFreeCursor(XEN_TO_C_Display(arg1), XEN_TO_C_Cursor(arg2))));
}

static XEN gxm_XFreeColors(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XFreeColors "XFreeColors(display, colormap, pixels, npixels, planes) frees the cells represented by pixels whose values are in the pixels array."
  /* DIFF: XFreeColors pixel array (arg3) is list of pixels
   */
  unsigned long *ps;
  int len, val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFreeColors", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XFreeColors", "Colormap");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XFreeColors", "list of pixel");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XFreeColors", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XFreeColors", "ulong");
  len = XEN_TO_C_INT(arg4);
  if (len <= 0) return(XEN_FALSE);
  ps = XEN_TO_C_Pixels(arg3, len);
  val = XFreeColors(XEN_TO_C_Display(arg1), 
		    XEN_TO_C_Colormap(arg2), 
		    ps, len, 
		    XEN_TO_C_ULONG(arg5));
  FREE(ps);
  return(C_TO_XEN_INT(val));
}

static XEN gxm_XFreeColormap(XEN arg1, XEN arg2)
{
  #define H_XFreeColormap "XFreeColormap(display, colormap) deletes the association between the colormap resource ID and the colormap and \
frees the colormap storage."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFreeColormap", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XFreeColormap", "Colormap");
  return(C_TO_XEN_INT(XFreeColormap(XEN_TO_C_Display(arg1), XEN_TO_C_Colormap(arg2))));
}

static XEN gxm_XFree(XEN arg1)
{
  #define H_XFree "XFree(data) is a general-purpose Xlib routine that frees the specified data."
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1) || XEN_LIST_P(arg1), arg1, 1, "XFree", "void* or xm entity");
  if (XEN_LIST_P(arg1))
    XFree((void *)XEN_TO_C_ULONG(XEN_CADR(arg1)));
  else XFree((void *)XEN_TO_C_ULONG(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XForceScreenSaver(XEN arg1, XEN arg2)
{
  #define H_XForceScreenSaver "XForceScreenSaver(display, mode) activates the screen saver"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XForceScreenSaver", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XForceScreenSaver", "int");
  return(C_TO_XEN_INT(XForceScreenSaver(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XFlush(XEN arg1)
{
  #define H_XFlush "XFlush(display) flushes the output buffer."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFlush", "Display*");
  return(C_TO_XEN_INT(XFlush(XEN_TO_C_Display(arg1))));
}


static XEN gxm_XFillRectangles(XEN arg1, XEN arg2, XEN arg3, XEN larg4, XEN arg5)
{
  #define H_XFillRectangles "XFillRectangles(display, d, gc, rectangles, nrectangles)"
  /* DIFF: XFillRectangles XRectangle* arg (arg 4) is list of XRectangles
   */
  XRectangle *pt;
  int i, len;
  XEN arg4;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFillRectangles", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XFillRectangles", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XFillRectangles", "GC");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg4), larg4, 4, "XFillRectangles", "list of XRectangle");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XFillRectangles", "int");
  arg4 = XEN_COPY_ARG(larg4);
  len = XEN_TO_C_INT(arg5);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg5, 5, "XFillRectangles", "positive integer");
  pt = (XRectangle *)CALLOC(len, sizeof(XRectangle));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg4)); i++, arg4 = XEN_CDR(arg4))
    {
      XRectangle *pt1;
      pt1 = XEN_TO_C_XRectangle(XEN_CAR(arg4));
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
      pt[i].width = pt1->width;
      pt[i].height = pt1->height;
    }
  XFillRectangles(XEN_TO_C_Display(arg1), 
		  XEN_TO_C_Window(arg2), 
		  XEN_TO_C_GC(arg3),
		  pt, len);
  FREE(pt);
  return(C_TO_XEN_INT(len));
}

static XEN gxm_XFillRectangle(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XFillRectangle "XFillRectangle(display, d, gc, x, y, width, height)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFillRectangle", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XFillRectangle", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XFillRectangle", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XFillRectangle", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XFillRectangle", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XFillRectangle", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XFillRectangle", "unsigned int");
  return(C_TO_XEN_INT(XFillRectangle(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
				     XEN_TO_C_GC(arg3), XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
				     XEN_TO_C_ULONG(arg6), XEN_TO_C_ULONG(arg7))));
}

static XEN gxm_XFillPolygon(XEN arg1, XEN arg2, XEN arg3, XEN larg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XFillPolygon "XFillPolygon(display, d, gc, points, npoints, shape, mode)"
  /* DIFF: XFillPolygon Point* arg (arg 4) is list of XPoint
   */
  XPoint *pt, *pt1;
  int i, len;
  XEN arg4;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFillPolygon", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XFillPolygon", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XFillPolygon", "GC");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg4), larg4, 4, "XFillPolygon", "list of XPoints");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XFillPolygon", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XFillPolygon", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XFillPolygon", "int");
  arg4 = XEN_COPY_ARG(larg4);
  len = XEN_TO_C_INT(arg5);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg5, 5, "XFillPolygon", "positive integer");
  pt = (XPoint *)CALLOC(len, sizeof(XPoint));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg4)); i++, arg4 = XEN_CDR(arg4))
    {
      XEN xp;
      xp = XEN_CAR(arg4);
      if (!(XEN_XPoint_P(xp))) XEN_ASSERT_TYPE(0, xp, i, "XFillRegion", "XPoint");
      pt1 = XEN_TO_C_XPoint(xp);
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
    }
  XFillPolygon(XEN_TO_C_Display(arg1), 
	       XEN_TO_C_Window(arg2), 
	       XEN_TO_C_GC(arg3), 
	       pt, len, 
	       XEN_TO_C_INT(arg6), 
	       XEN_TO_C_INT(arg7));
  FREE(pt);
  return(C_TO_XEN_INT(len));
}

static XEN gxm_XFillArcs(XEN arg1, XEN arg2, XEN arg3, XEN larg4, XEN arg5)
{
  #define H_XFillArcs "XFillArcs(display, d, gc, arcs, narcs)"
  /* DIFF: XFillArcs Arc* arg (arg 4) is list of XArcs
   */
  int i, len;
  Display *dpy;
  Drawable draw;
  GC gc;
  XEN arg4;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFillArcs", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XFillArcs", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XFillArcs", "GC");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg4), larg4, 4, "XFillArcs", "list of XArcs");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XFillArcs", "int");
  arg4 = XEN_COPY_ARG(larg4);
  len = XEN_TO_C_INT(arg5);
  if (len <= 0) return(XEN_FALSE);
  dpy = XEN_TO_C_Display(arg1);
  draw = XEN_TO_C_Window(arg2);
  gc = XEN_TO_C_GC(arg3);
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg4)); i++, arg4 = XEN_CDR(arg4))
    {
      XArc *arc;
      arc = XEN_TO_C_XArc(XEN_CAR(arg4));
      XFillArc(dpy, draw, gc, arc->x, arc->y, arc->width, arc->height, arc->angle1, arc->angle2);
    }
  return(C_TO_XEN_INT(len));
}

static XEN gxm_XFillArc(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8, XEN arg9)
{
  #define H_XFillArc "XFillArc(display, d, gc, x, y, width, height, angle1, angle2) fills the region described by the specified arc."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFillArc", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XFillArc", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XFillArc", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XFillArc", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XFillArc", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XFillArc", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XFillArc", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XFillArc", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg9), arg9, 9, "XFillArc", "int");
  return(C_TO_XEN_INT(XFillArc(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
			       XEN_TO_C_GC(arg3), XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
			       XEN_TO_C_ULONG(arg6), XEN_TO_C_ULONG(arg7), 
			       XEN_TO_C_INT(arg8), XEN_TO_C_INT(arg9))));
}

static XEN gxm_XFetchName(XEN arg1, XEN arg2)
{
  #define H_XFetchName "Status XFetchName(display, w) returns the name of the specified window."
  /* DIFF: XFetchName omits and rtns arg3
   */
  char *name;
  int val;
  XEN str;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFetchName", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XFetchName", "Window");
  val = XFetchName(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), &name);
  if (val == 0)
    return(XEN_FALSE);
  str = C_TO_XEN_STRING(name);
  free(name);
  return(str);
}

static XEN gxm_XEventsQueued(XEN arg1, XEN arg2)
{
  #define H_XEventsQueued "int XEventsQueued(display, mode)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XEventsQueued", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XEventsQueued", "int");
  return(C_TO_XEN_INT(XEventsQueued(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XEnableAccessControl(XEN arg1)
{
  #define H_XEnableAccessControl "XEnableAccessControl(display) enables the use of the access control list at each connection setup."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XEnableAccessControl", "Display*");
  return(C_TO_XEN_INT(XEnableAccessControl(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XDrawText(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XDrawText "XDrawText(display, d, gc, x, y, items, nitems) draws text"
  int i, len = 0, res;
  XTextItem *items;
  XEN lst;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawText", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawText", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawText", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XDrawText", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawText", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg6), arg6, 6, "XDrawText", "list of XTextItem");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg7), arg7, 7, "XDrawText", "int");
  lst = XEN_COPY_ARG(arg6);
  if (XEN_INTEGER_P(arg7)) len = XEN_TO_C_INT(arg7); else len = XEN_LIST_LENGTH(arg6);
  items = (XTextItem *)CALLOC(len, sizeof(XTextItem));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(lst)); i++, lst = XEN_CDR(lst))
    {
      XTextItem *val;
      val = XEN_TO_C_XTextItem(XEN_CAR(lst));
      items[i].chars = val->chars;
      items[i].nchars = val->nchars;
      items[i].delta = val->delta;
      items[i].font = val->font;
    }
  res = XDrawText(XEN_TO_C_Display(arg1), 
		  XEN_TO_C_Window(arg2), 
		  XEN_TO_C_GC(arg3), 
		  XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
		  items, len);
  FREE(items);
  return(C_TO_XEN_INT(res));
}

static XEN gxm_XDrawString(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XDrawString "XDrawString(display, d, gc, x, y, string, length)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawString", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawString", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawString", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XDrawString", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawString", "int");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg6), arg6, 6, "XDrawString", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XDrawString", "int");
  return(C_TO_XEN_INT(XDrawString(XEN_TO_C_Display(arg1), 
				  XEN_TO_C_Window(arg2), 
				  XEN_TO_C_GC(arg3), 
				  XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
				  XEN_TO_C_STRING(arg6), XEN_TO_C_INT(arg7))));
}

static XEN gxm_XDrawSegments(XEN arg1, XEN arg2, XEN arg3, XEN larg4, XEN arg5)
{
  #define H_XDrawSegments "XDrawSegments(display, d, gc, segments, nsegments) draws multiple, unconnected lines. "
  /* DIFF: XDrawSegments XSegment* arg (arg 4) is list of XSegments
   */
  XSegment *pt;
  int i, len;
  XEN arg4;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawSegments", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawSegments", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawSegments", "GC");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg4), larg4, 4, "XDrawSegments", "list of XSegments");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawSegments", "int");
  arg4 = XEN_COPY_ARG(larg4);
  len = XEN_TO_C_INT(arg5);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg5, 5, "XDrawSegments", "positive integer");
  pt = (XSegment *)CALLOC(len, sizeof(XSegment));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg4)); i++, arg4 = XEN_CDR(arg4))
    {
      XSegment *pt1;
      pt1 = XEN_TO_C_XSegment(XEN_CAR(arg4));
      pt[i].x1 = pt1->x1;
      pt[i].y1 = pt1->y1;
      pt[i].x2 = pt1->x2;
      pt[i].y2 = pt1->y2;
    }
  XDrawSegments(XEN_TO_C_Display(arg1), 
		XEN_TO_C_Window(arg2), 
		XEN_TO_C_GC(arg3),
		pt, len);
  FREE(pt);
  return(C_TO_XEN_INT(len));
}

static XEN gxm_XDrawRectangles(XEN arg1, XEN arg2, XEN arg3, XEN larg4, XEN arg5)
{
  #define H_XDrawRectangles "XDrawRectangles(display, d, gc, rectangles, nrectangles) draws the outlines of the specified rectangles."
  /* DIFF: XDrawRectangles XRectangle* arg (arg 4) is list of XRectangles
   */
  XRectangle *pt;
  int i, len;
  XEN arg4;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawRectangles", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawRectangles", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawRectangles", "GC");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg4), larg4, 4, "XDrawRectangles", "list of XRectangles");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawRectangles", "int");
  arg4 = XEN_COPY_ARG(larg4);
  len = XEN_TO_C_INT(arg5);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg5, 5, "XDrawRectangles", "positive integer");
  pt = (XRectangle *)CALLOC(len, sizeof(XRectangle));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg4)); i++, arg4 = XEN_CDR(arg4))
    {
      XRectangle *pt1;
      pt1 = XEN_TO_C_XRectangle(XEN_CAR(arg4));
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
      pt[i].width = pt1->width;
      pt[i].height = pt1->height;
    }
  XDrawRectangles(XEN_TO_C_Display(arg1), 
		  XEN_TO_C_Window(arg2), 
		  XEN_TO_C_GC(arg3),
		  pt, len);
  FREE(pt);
  return(C_TO_XEN_INT(len));
}

static XEN gxm_XDrawRectangle(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XDrawRectangle "XDrawRectangle(display, d, gc, x, y, width, height) draws the outlines of the specified rectangle."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawRectangle", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawRectangle", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawRectangle", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XDrawRectangle", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawRectangle", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XDrawRectangle", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XDrawRectangle", "unsigned int");
  return(C_TO_XEN_INT(XDrawRectangle(XEN_TO_C_Display(arg1), 
				     XEN_TO_C_Window(arg2), 
				     XEN_TO_C_GC(arg3), 
				     XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), XEN_TO_C_ULONG(arg6), XEN_TO_C_ULONG(arg7))));
}

static XEN gxm_XDrawPoints(XEN arg1, XEN arg2, XEN arg3, XEN larg4, XEN arg5, XEN arg6)
{
  #define H_XDrawPoints "XDrawPoints(display, d, gc, points, npoints, mode) draws multiple points."
  /* DIFF: XDrawPoints XPoint* arg (arg 4) is list of XPoints
   */
  XPoint *pt, *pt1;
  int i, len;
  XEN arg4;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawPoints", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawPoints", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawPoints", "GC");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg4), larg4, 4, "XDrawPoints", "list of XPoints");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawPoints", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XDrawPoints", "int");
  arg4 = XEN_COPY_ARG(larg4);
  len = XEN_TO_C_INT(arg5);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg5, 5, "XDrawPoints", "positive integer");
  pt = (XPoint *)CALLOC(len, sizeof(XPoint));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg4)); i++, arg4 = XEN_CDR(arg4))
    {
      XEN xp;
      xp = XEN_CAR(arg4);
      if (!(XEN_XPoint_P(xp))) XEN_ASSERT_TYPE(0, xp, i, "XDrawPoints", "XPoint");
      pt1 = XEN_TO_C_XPoint(xp);
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
    }
  XDrawPoints(XEN_TO_C_Display(arg1), 
	      XEN_TO_C_Window(arg2), 
	      XEN_TO_C_GC(arg3), 
	      pt, len, 
	      XEN_TO_C_INT(arg6));
  FREE(pt);
  return(C_TO_XEN_INT(len));
}

static XEN gxm_XDrawPoint(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XDrawPoint "XDrawPoint(display, d, gc, x, y) uses the foreground pixel and function components of the GC to draw a single \
point into the specified drawable."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawPoint", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawPoint", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawPoint", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XDrawPoint", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawPoint", "int");
  return(C_TO_XEN_INT(XDrawPoint(XEN_TO_C_Display(arg1), 
				 XEN_TO_C_Window(arg2), 
				 XEN_TO_C_GC(arg3), 
				 XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5))));
}

static XEN gxm_XDrawLines(XEN arg1, XEN arg2, XEN arg3, XEN larg4, XEN arg5, XEN arg6)
{
  #define H_XDrawLines "XDrawLines(display, d, gc, points, npoints, mode) uses the components of the specified GC to draw npoints lines \
between each pair of points (point[i], point[i+1]) in the array of XPoint structures."
  /* DIFF: XDrawLines XPoint* arg (arg 4) is list of XPoints
   */
  XPoint *pt, *pt1;
  int i, len;
  XEN arg4;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawLines", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawLines", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawLines", "GC");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg4), larg4, 4, "XDrawLines", "list of XPoints");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawLines", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XDrawLines", "int");
  arg4 = XEN_COPY_ARG(larg4);
  len = XEN_TO_C_INT(arg5);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg5, 5, "XDrawLines", "positive integer");
  pt = (XPoint *)CALLOC(len, sizeof(XPoint));
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg4)); i++, arg4 = XEN_CDR(arg4))
    {
      XEN xp;
      xp = XEN_CAR(arg4);
      if (!(XEN_XPoint_P(xp))) XEN_ASSERT_TYPE(0, xp, i, "XDrawLines", "XPoint");
      pt1 = XEN_TO_C_XPoint(xp);
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
    }
  XDrawLines(XEN_TO_C_Display(arg1), 
	     XEN_TO_C_Window(arg2), 
	     XEN_TO_C_GC(arg3), 
	     pt, len, 
	     XEN_TO_C_INT(arg6));
  FREE(pt);
  return(C_TO_XEN_INT(len));
}

static XEN gxm_XDrawLinesDirect(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XDrawLinesDirect "XDrawLinesDirect is the same as XDrawLines but takes an (opaque) pointer to an XPoint array"
  XPoint *pt;
  int len;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawLinesDirect", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawLinesDirect", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawLinesDirect", "GC");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XDrawLinesDirect", "array of XPoints");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawLines", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XDrawLines", "int");
  len = XEN_TO_C_INT(arg5);
  if (len <= 0) return(XEN_FALSE);
  pt = (XPoint *)XEN_TO_C_ULONG(arg4);
  XDrawLines(XEN_TO_C_Display(arg1), 
	     XEN_TO_C_Window(arg2), 
	     XEN_TO_C_GC(arg3), 
	     pt, len, 
	     XEN_TO_C_INT(arg6));
  return(C_TO_XEN_INT(len));
}

static XEN gxm_Vector2XPoints(XEN arg1)
{
  #define H_vector2XPoints "(vector->XPoints vect) packages point data in vect as (opaque) array of XPoints"
  int i, j, len;
  XEN *velts;
  /* vector assumed to be sequence of x y pairs (not XPoints from local view)
   */
  XPoint *pt;
  XEN_ASSERT_TYPE(XEN_VECTOR_P(arg1), arg1, XEN_ONLY_ARG, "vector->XPoints", "vector of x,y values");
  len = XEN_VECTOR_LENGTH(arg1) / 2;
  if (len <= 0) XEN_ASSERT_TYPE(0, arg1, 1, "vector->XPoints", "positive integer");
  velts = XEN_VECTOR_ELEMENTS(arg1);
  pt = (XPoint *)CALLOC(len, sizeof(XPoint));
  for (i = 0, j = 0; i < len; i++, j += 2)
    {
      pt[i].x = XEN_TO_C_INT(velts[j]);
      pt[i].y = XEN_TO_C_INT(velts[j + 1]);
    }
  return(C_TO_XEN_ULONG((unsigned long)pt));
}

static XEN gxm_FreeXPoints(XEN arg1)
{
  void *pts;
  #define H_freeXPoints "(freeXPoints vect) frees an (opaque) XPoint array created by vector->Xpoints"
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, XEN_ONLY_ARG, "freeXPoints", "opaque XPoint array");
  pts = (void *)(XEN_TO_C_ULONG(arg1));
  FREE(pts);
  return(XEN_FALSE);
}


static XEN gxm_XDrawLine(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XDrawLine "XDrawLine(display, d, gc, x1, y1, x2, y2) uses the components of the specified GC to draw a line between the \
specified set of points (x1, y1) and (x2, y2)."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawLine", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawLine", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawLine", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XDrawLine", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawLine", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XDrawLine", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XDrawLine", "int");
  return(C_TO_XEN_INT(XDrawLine(XEN_TO_C_Display(arg1), 
				XEN_TO_C_Window(arg2), 
				XEN_TO_C_GC(arg3), 
				XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), XEN_TO_C_INT(arg6), XEN_TO_C_INT(arg7))));
}

static XEN gxm_XDrawImageString(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XDrawImageString "XDrawImageString(display, d, gc, x, y, string, length)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawImageString", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawImageString", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawImageString", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XDrawImageString", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawImageString", "int");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg6), arg6, 6, "XDrawImageString", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XDrawImageString", "int");
  return(C_TO_XEN_INT(XDrawImageString(XEN_TO_C_Display(arg1), 
				       XEN_TO_C_Window(arg2), 
				       XEN_TO_C_GC(arg3), 
				       XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), XEN_TO_C_STRING(arg6), XEN_TO_C_INT(arg7))));
}

static XEN gxm_XDrawArcs(XEN arg1, XEN arg2, XEN arg3, XEN larg4, XEN arg5)
{
  #define H_XDrawArcs "XDrawArcs(display, d, gc, arcs, narcs) draws multiple circular or elliptical arcs."
  /* DIFF: XDrawArcs Arc* arg (arg 4) is list of XArcs
   */
  int i, len;
  Display *dpy;
  Drawable draw;
  GC gc;
  XEN arg4;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawArcs", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawArcs", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawArcs", "GC");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg4), larg4, 4, "XDrawArcs", "list of XArcs");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawArcs", "int");
  arg4 = XEN_COPY_ARG(larg4);
  len = XEN_TO_C_INT(arg5);
  dpy = XEN_TO_C_Display(arg1);
  draw = XEN_TO_C_Window(arg2);
  gc = XEN_TO_C_GC(arg3);
  for (i = 0; (i < len) && (XEN_NOT_NULL_P(arg4)); i++, arg4 = XEN_CDR(arg4))
    {
      XArc *arc;
      arc = XEN_TO_C_XArc(XEN_CAR(arg4));
      XDrawArc(dpy, draw, gc, arc->x, arc->y, arc->width, arc->height, arc->angle1, arc->angle2);
    }
  return(C_TO_XEN_INT(len));
}

static XEN gxm_XDrawArc(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8, XEN arg9)
{
  #define H_XDrawArc "XDrawArc(display, d, gc, x, y, width, height, angle1, angle2) draws a single circular or elliptical arc."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDrawArc", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDrawArc", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg3), arg3, 3, "XDrawArc", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XDrawArc", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XDrawArc", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XDrawArc", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XDrawArc", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XDrawArc", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg9), arg9, 9, "XDrawArc", "int");
  return(C_TO_XEN_INT(XDrawArc(XEN_TO_C_Display(arg1), 
			       XEN_TO_C_Window(arg2), 
			       XEN_TO_C_GC(arg3), 
			       XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
			       XEN_TO_C_ULONG(arg6), XEN_TO_C_ULONG(arg7), XEN_TO_C_INT(arg8), XEN_TO_C_INT(arg9))));
}

static XEN gxm_XDisplayWidthMM(XEN arg1, XEN arg2)
{
  #define H_DisplayWidthMM "DisplayWidthMM(display, screen_number) returns the width of the specified screen in millimeters."
  #define H_XDisplayWidthMM "XDisplayWidthMM(display, screen_number) returns the width of the specified screen in millimeters."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDisplayWidthMM", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XDisplayWidthMM", "int");
  return(C_TO_XEN_INT(XDisplayWidthMM(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XDisplayWidth(XEN arg1, XEN arg2)
{
  #define H_DisplayWidth "DisplayWidth(display, screen_number) returns the width of the screen in pixels."
  #define H_XDisplayWidth "XDisplayWidth(display, screen_number) returns the width of the screen in pixels."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDisplayWidth", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XDisplayWidth", "int");
  return(C_TO_XEN_INT(XDisplayWidth(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XDisplayPlanes(XEN arg1, XEN arg2)
{
  #define H_DisplayPlanes "DisplayPlanes(display, screen_number) returns the depth of the root window of the specified screen."
  #define H_XDisplayPlanes "XDisplayPlanes(display, screen_number) returns the depth of the root window of the specified screen."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDisplayPlanes", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XDisplayPlanes", "int");
  return(C_TO_XEN_INT(XDisplayPlanes(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XDisplayKeycodes(XEN arg1)
{
  #define H_XDisplayKeycodes "XDisplayKeycodes(display) returns the min-keycodes and max-keycodes supported by the specified display."
  /* DIFF: XDisplayKeycodes omit and rtn arg 2 and 3
   */
  int m1, m2, val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDisplayKeycodes", "Display*");
  val = XDisplayKeycodes(XEN_TO_C_Display(arg1), &m1, &m2);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_INT(m1),
		    C_TO_XEN_INT(m2)));
}

static XEN gxm_XDisplayHeightMM(XEN arg1, XEN arg2)
{
  #define H_DisplayHeightMM "DisplayHeightMM(display, screen_number) returns the height of the specified screen in millimeters."
  #define H_XDisplayHeightMM "XDisplayHeightMM(display, screen_number) returns the height of the specified screen in millimeters."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDisplayHeightMM", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XDisplayHeightMM", "int");
  return(C_TO_XEN_INT(XDisplayHeightMM(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XDisplayHeight(XEN arg1, XEN arg2)
{
  #define H_DisplayHeight "DisplayHeight(display, screen_number) returns the height of the specified screen in pixels."
  #define H_XDisplayHeight "XDisplayHeight(display, screen_number) returns the height of the specified screen in pixels."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDisplayHeight", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XDisplayHeight", "int");
  return(C_TO_XEN_INT(XDisplayHeight(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XDisplayCells(XEN arg1, XEN arg2)
{
  #define H_DisplayCells "DisplayCells(display, screen_number) returns the number of entries in the default colormap."
  #define H_XDisplayCells "XDisplayCells(display, screen_number) returns the number of entries in the default colormap."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDisplayCells", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XDisplayCells", "int");
  return(C_TO_XEN_INT(XDisplayCells(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XDisableAccessControl(XEN arg1)
{
  #define H_XDisableAccessControl "XDisableAccessControl(display) disables the use of the access control list at each connection setup."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDisableAccessControl", "Display*");
  return(C_TO_XEN_INT(XDisableAccessControl(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XDoesSaveUnders(XEN arg1)
{
  #define H_DoesSaveUnders "DoesSaveUnders(screen) returns a Boolean value indicating whether the screen supports save unders."
  #define H_XDoesSaveUnders "XDoesSaveUnders(screen) returns a Boolean value indicating whether the screen supports save unders."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XDoesSaveUnders", "Screen*");
  return(C_TO_XEN_BOOLEAN(XDoesSaveUnders(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XDoesBackingStore(XEN arg1)
{
  #define H_DoesBackingStore "DoesBackingStore(screen) returns WhenMapped, NotUseful,or Always,which indicate whether the screen supports backing stores."
  #define H_XDoesBackingStore "XDoesBackingStore(screen) returns WhenMapped, NotUseful,or Always,which indicate whether the screen supports backing stores."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XDoesBackingStore", "Screen*");
  return(C_TO_XEN_BOOLEAN(XDoesBackingStore(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XDestroySubwindows(XEN arg1, XEN arg2)
{
  #define H_XDestroySubwindows "XDestroySubwindows(display, w) destroys all inferior windows of the specified window, in bottom-to-top stacking order."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDestroySubwindows", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDestroySubwindows", "Window");
  return(C_TO_XEN_INT(XDestroySubwindows(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XDestroyWindow(XEN arg1, XEN arg2)
{
  #define H_XDestroyWindow "XDestroyWindow(display, w) destroys the specified window as well as all of its subwindows and causes the X server \
to generate a DestroyNotify event for each window."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDestroyWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDestroyWindow", "Window");
  return(C_TO_XEN_INT(XDestroyWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XDeleteProperty(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XDeleteProperty "XDeleteProperty(display, w, property) deletes the specified property only if the property was defined on the specified \
window and causes the X server to generate a PropertyNotify event on the window unless the property does not exist."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDeleteProperty", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDeleteProperty", "Window");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XDeleteProperty", "Atom");
  return(C_TO_XEN_INT(XDeleteProperty(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_Atom(arg3))));
}

static XEN gxm_XDefineCursor(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XDefineCursor "XDefineCursor(display, w, cursor)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDefineCursor", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XDefineCursor", "Window");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg3), arg3, 3, "XDefineCursor", "Cursor");
  return(C_TO_XEN_INT(XDefineCursor(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_Cursor(arg3))));
}

static XEN gxm_XDefaultScreen(XEN arg1)
{
  #define H_XDefaultScreen "XDefaultScreen(display)"
  #define H_DefaultScreenOfDisplay "returns the default screen of the specified display."
  #define H_XDefaultScreenOfDisplay "returns the default screen of the specified display."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDefaultScreen", "Display*");
  return(C_TO_XEN_INT(XDefaultScreen(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XDefaultDepthOfScreen(XEN arg1)
{
  #define H_DefaultDepthOfScreen "returns the default depth of the root window of the specified screen."
  #define H_XDefaultDepthOfScreen "returns the default depth of the root window of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XDefaultDepthOfScreen", "Screen*");
  return(C_TO_XEN_INT(XDefaultDepthOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XDefaultDepth(XEN arg1, XEN arg2)
{
  #define H_DefaultDepth "returns the depth (number of planes) of the default root window for the specified screen."
  #define H_XDefaultDepth "returns the depth (number of planes) of the default root window for the specified screen."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDefaultDepth", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XDefaultDepth", "int");
  return(C_TO_XEN_INT(XDefaultDepth(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XCopyPlane(XEN args)
{
  #define H_XCopyPlane "XCopyPlane(display, src, dest, gc, src_x, src_y, width, height, dest_x, dest_y, plane) uses a single bit plane of the \
specified source rectangle combined with the specified GC to modify the specified rectangle of dest."
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  arg10 = XEN_LIST_REF(args, 9);
  arg11 = XEN_LIST_REF(args, 10);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCopyPlane", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCopyPlane", "Drawable");
  XEN_ASSERT_TYPE(XEN_Window_P(arg3), arg3, 3, "XCopyPlane", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg4), arg4, 4, "XCopyPlane", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XCopyPlane", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XCopyPlane", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XCopyPlane", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg8), arg8, 8, "XCopyPlane", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg9), arg9, 9, "XCopyPlane", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg10), arg10, 10, "XCopyPlane", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg11), arg11, 11, "XCopyPlane", "ulong");
  return(C_TO_XEN_INT(XCopyPlane(XEN_TO_C_Display(arg1), 
				 XEN_TO_C_Window(arg2), 
				 XEN_TO_C_Window(arg3), 
				 XEN_TO_C_GC(arg4), 
				 XEN_TO_C_INT(arg5), XEN_TO_C_INT(arg6), XEN_TO_C_ULONG(arg7), 
				 XEN_TO_C_ULONG(arg8), XEN_TO_C_INT(arg9), 
				 XEN_TO_C_INT(arg10), XEN_TO_C_ULONG(arg11))));
}

static XEN gxm_XCopyGC(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XCopyGC "XCopyGC(display, src, valuemask, dest) copies the specified components from the source GC to the destination GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCopyGC", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XCopyGC", "GC");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XCopyGC", "ulong");
  XEN_ASSERT_TYPE(XEN_GC_P(arg4), arg4, 4, "XCopyGC", "GC");
  return(C_TO_XEN_INT(XCopyGC(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_ULONG(arg3), XEN_TO_C_GC(arg4))));
}

static XEN gxm_XCopyArea(XEN args)
{
  #define H_XCopyArea "XCopyArea(display, src, dest, gc, src_x, src_y, width, height, dest_x, dest_y) combines the specified rectangle of src \
with the specified rectangle of dest."
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  arg10 = XEN_LIST_REF(args, 9);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCopyArea", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCopyArea", "Drawable");
  XEN_ASSERT_TYPE(XEN_Window_P(arg3), arg3, 3, "XCopyArea", "Drawable");
  XEN_ASSERT_TYPE(XEN_GC_P(arg4), arg4, 4, "XCopyArea", "GC");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XCopyArea", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XCopyArea", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XCopyArea", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg8), arg8, 8, "XCopyArea", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg9), arg9, 9, "XCopyArea", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg10), arg10, 10, "XCopyArea", "int");
  return(C_TO_XEN_INT(XCopyArea(XEN_TO_C_Display(arg1), 
				XEN_TO_C_Window(arg2), 
				XEN_TO_C_Window(arg3), 
				XEN_TO_C_GC(arg4), 
				XEN_TO_C_INT(arg5), XEN_TO_C_INT(arg6), XEN_TO_C_ULONG(arg7), 
				XEN_TO_C_ULONG(arg8), XEN_TO_C_INT(arg9), XEN_TO_C_INT(arg10))));
}

static XEN gxm_XConvertSelection(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XConvertSelection "void XConvertSelection(display, selection, target, property, requestor, time)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XConvertSelection", "Display*");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XConvertSelection", "Atom");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XConvertSelection", "Atom");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg4), arg4, 4, "XConvertSelection", "Atom");
  XEN_ASSERT_TYPE(XEN_Window_P(arg5), arg5, 5, "XConvertSelection", "Window");
  XEN_ASSERT_TYPE(XEN_Time_P(arg6), arg6, 6, "XConvertSelection", "Time");
  return(C_TO_XEN_INT(XConvertSelection(XEN_TO_C_Display(arg1), 
					XEN_TO_C_Atom(arg2), XEN_TO_C_Atom(arg3), XEN_TO_C_Atom(arg4), 
					XEN_TO_C_Window(arg5), XEN_TO_C_Time(arg6))));
}

static XEN gxm_XConnectionNumber(XEN arg1)
{
  #define H_ConnectionNumber "returns a connection number for the specified display."
  #define H_XConnectionNumber "returns a connection number for the specified display."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XConnectionNumber", "Display*");
  return(C_TO_XEN_INT(XConnectionNumber(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XConfigureWindow(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XConfigureWindow "XConfigureWindow(display, w, value_mask, values) uses the values specified in the XWindowChanges structure to \
reconfigure a window's size, position, border, and stacking order."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XConfigureWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XConfigureWindow", "Window");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XConfigureWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_XWindowChanges_P(arg4), arg4, 4, "XConfigureWindow", "XWindowChanges*");
  return(C_TO_XEN_INT(XConfigureWindow(XEN_TO_C_Display(arg1), 
				       XEN_TO_C_Window(arg2), XEN_TO_C_ULONG(arg3), XEN_TO_C_XWindowChanges(arg4))));
}

static XEN gxm_XCloseDisplay(XEN arg1)
{
  #define H_XCloseDisplay "XCloseDisplay(display) closes the connection to the X server for the display specified in the Display structure and \
destroys all windows, resource IDs etc."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCloseDisplay", "Display*");
  return(C_TO_XEN_INT(XCloseDisplay(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XClearWindow(XEN arg1, XEN arg2)
{
  #define H_XClearWindow "XClearWindow(display, w) clears the entire area in the specified window."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XClearWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XClearWindow", "Window");
  return(C_TO_XEN_INT(XClearWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XClearArea(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XClearArea "XClearArea(display, w, x, y, width, height, exposures) paints a rectangular area in the specified window according to the \
specified dimensions with the window's background pixel or pixmap."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XClearArea", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XClearArea", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XClearArea", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XClearArea", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XClearArea", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XClearArea", "unsigned int");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg7), arg7, 7, "XClearArea", "Bool");
  return(C_TO_XEN_INT(XClearArea(XEN_TO_C_Display(arg1), 
				 XEN_TO_C_Window(arg2), 
				 XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), 
				 XEN_TO_C_ULONG(arg5), XEN_TO_C_ULONG(arg6), XEN_TO_C_BOOLEAN(arg7))));
}

static XEN gxm_XCirculateSubwindowsUp(XEN arg1, XEN arg2)
{
  #define H_XCirculateSubwindowsUp "XCirculateSubwindowsUp(display, w) raises the lowest mapped child of the specified window that is partially or \
completely occluded by another child."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCirculateSubwindowsUp", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCirculateSubwindowsUp", "Window");
  return(C_TO_XEN_INT(XCirculateSubwindowsUp(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XCirculateSubwindowsDown(XEN arg1, XEN arg2)
{
  #define H_XCirculateSubwindowsDown "XCirculateSubwindowsDown(display, w) lowers the highest mapped child of the specified window that partially or \
completely occludes another child."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCirculateSubwindowsDown", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCirculateSubwindowsDown", "Window");
  return(C_TO_XEN_INT(XCirculateSubwindowsDown(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XCirculateSubwindows(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XCirculateSubwindows "XCirculateSubwindows(display, w, direction) circulates children of the specified window in the specified direction."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCirculateSubwindows", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCirculateSubwindows", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XCirculateSubwindows", "int");
  return(C_TO_XEN_INT(XCirculateSubwindows(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XCheckWindowEvent(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XCheckWindowEvent "Bool XCheckWindowEvent(display, w, event_mask) searches the event queue and then the events available \
on the server connection for the first event that matches the specified window and event mask."
  /* DIFF: XCheckWindowEvent dpy win mask [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCheckWindowEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCheckWindowEvent", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XCheckWindowEvent", "long");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XCheckWindowEvent(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3), e);
  if (val)
    return(C_TO_XEN_XEvent_OBJ(e));
  else
    {
      FREE(e);
      return(XEN_FALSE);
    }
}

static XEN gxm_XCheckTypedWindowEvent(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XCheckTypedWindowEvent "Bool XCheckTypedWindowEvent(display, w, event_type) searches the event queue and then any events \
available on the server connection for the first event that matches the specified event mask"
  /* DIFF: XCheckTypedWindowEvent dpy win mask [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCheckTypedWindowEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCheckTypedWindowEvent", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XCheckTypedWindowEvent", "int");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XCheckTypedWindowEvent(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3), e);
  if (val)
    return(C_TO_XEN_XEvent_OBJ(e));
  else
    {
      FREE(e);
      return(XEN_FALSE);
    }
}

static XEN gxm_XCheckTypedEvent(XEN arg1, XEN arg2)
{
  #define H_XCheckTypedEvent "Bool XCheckTypedEvent(display, event_type) searches the event queue and then any events available  \
on the server connection for the first event that matches the specified type."
  /* DIFF: XCheckTypedEvent dpy mask [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCheckTypedEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XCheckTypedEvent", "int");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XCheckTypedEvent(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), e);
  if (val)
    return(C_TO_XEN_XEvent_OBJ(e));
  else
    {
      FREE(e);
      return(XEN_FALSE);
    }
}

static XEN gxm_XCheckMaskEvent(XEN arg1, XEN arg2)
{
  #define H_XCheckMaskEvent "Bool XCheckMaskEvent(display, event_mask) searches the event queue and then any events available on \
the server connection for the first event that matches the specified mask."
  /* DIFF: XCheckMaskEvent dpy mask [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCheckMaskEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XCheckMaskEvent", "long");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XCheckMaskEvent(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), e);
  if (val)
    return(C_TO_XEN_XEvent_OBJ(e));
  else
    {
      FREE(e);
      return(XEN_FALSE);
    }
}

static XEN gxm_XCheckIfEvent(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XCheckIfEvent "Bool XCheckIfEvent(display, predicate, arg)"
  /* DIFF: XCheckIfEvent dpy [ev] proc ptr -> (list val ev)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCheckIfEvent", "Display*");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg2) && (XEN_REQUIRED_ARGS_OK(arg2, 3)), arg2, 2, "XCheckIfEvent", "(Bool_Proc dpy ev data)");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XCheckIfEvent(XEN_TO_C_Display(arg1), 
		      e, 
		      (Bool (*)(Display *d, XEvent *ev, char *p))gxm_XPeekIfEventProc, 
		      (char*)arg3);
  if (val)
    return(C_TO_XEN_XEvent_OBJ(e));
  else
    {
      FREE(e);
      return(XEN_FALSE);
    }
}

static XEN gxm_XChangeWindowAttributes(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XChangeWindowAttributes "XChangeWindowAttributes(display, w, valuemask, attributes) uses the window attributes in the XSetWindowAttributes \
structure to change the specified window attributes."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XChangeWindowAttributes", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XChangeWindowAttributes", "Window");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XChangeWindowAttributes", "ulong");
  XEN_ASSERT_TYPE(XEN_XSetWindowAttributes_P(arg4), arg4, 4, "XChangeWindowAttributes", "XSetWindowAttributes*");
  return(C_TO_XEN_INT(XChangeWindowAttributes(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_ULONG(arg3), 
					      XEN_TO_C_XSetWindowAttributes(arg4))));
}

static XEN gxm_XChangeProperty(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8)
{
  #define H_XChangeProperty "XChangeProperty(display, w, property, type, format, mode, data, nelements) alters the property for the specified \
window and causes the X server to generate a PropertyNotify event on that window."
  unsigned char *command;
  int len;
  int *data = NULL;
  XEN rtn;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XChangeProperty", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XChangeProperty", "Window");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XChangeProperty", "Atom");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg4), arg4, 4, "XChangeProperty", "Atom");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XChangeProperty", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XChangeProperty", "int");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg7) || XEN_LIST_P(arg7), arg7, 7, "XChangeProperty", "string or list of int");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg8), arg8, 8, "XChangeProperty", "int");
  if (XEN_STRING_P(arg7))
    {
      command = (unsigned char *)(XEN_TO_C_STRING(arg7));
      if (XEN_INTEGER_P(arg8)) len = XEN_TO_C_INT(arg8); else len = strlen((const char *)command) + 1;
    }
  else 
    {
      if (XEN_INTEGER_P(arg8)) len = XEN_TO_C_INT(arg8); else len = XEN_LIST_LENGTH(arg7);
      data = XEN_TO_C_Ints(arg7, len);
      command = (unsigned char *)data;
    }
  rtn = C_TO_XEN_INT(XChangeProperty(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
				     XEN_TO_C_Atom(arg3), XEN_TO_C_Atom(arg4), 
				     XEN_TO_C_INT(arg5), XEN_TO_C_INT(arg6), 
				     (const unsigned char *)command, len));
  if (data) FREE(data);
  return(xen_return_first(rtn, arg7));
}

static XEN gxm_XChangePointerControl(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XChangePointerControl "XChangePointerControl(display, do_accel, do_threshold, accel_numerator, accel_denominator, threshold) \
defines how the pointing device moves."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XChangePointerControl", "Display*");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XChangePointerControl", "Bool");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XChangePointerControl", "Bool");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XChangePointerControl", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XChangePointerControl", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XChangePointerControl", "int");
  return(C_TO_XEN_INT(XChangePointerControl(XEN_TO_C_Display(arg1), 
					    XEN_TO_C_BOOLEAN(arg2), XEN_TO_C_BOOLEAN(arg3), 
					    XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), XEN_TO_C_INT(arg6))));
}

static XEN gxm_XChangeKeyboardMapping(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XChangeKeyboardMapping "XChangeKeyboardMapping(display, first_keycode, keysyms_per_keycode, keysyms, num_codes) defines the \
symbols for the specified number of KeyCodes starting with first_keycode."
  /* DIFF: XChangeKeyboardMapping takes list of KeySyms
   */
  KeySym *ks;
  int len, val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XChangeKeyboardMapping", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XChangeKeyboardMapping", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XChangeKeyboardMapping", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XChangeKeyboardMapping", "list of KeySym");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XChangeKeyboardMapping", "int");
  len = XEN_TO_C_INT(arg5);
  if (len <= 0) return(XEN_FALSE);
  ks = XEN_TO_C_KeySyms(arg4, len);
  val = XChangeKeyboardMapping(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), ks, len);
  FREE(ks);
  return(C_TO_XEN_INT(val));
}

static XEN gxm_XChangeKeyboardControl(XEN arg1, XEN arg2, XEN larg3)
{
  #define H_XChangeKeyboardControl "XChangeKeyboardControl(display, value_mask, values) controls the keyboard characteristics defined by \
the XKeyboardControl structure."
  /* DIFF: XChangeKeyboardControl arg3 is list of XKeyboardControl fields
   */
  XEN arg3;
  XKeyboardControl kc;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XChangeKeyboardControl", "Display*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XChangeKeyboardControl", "ulong");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg3), larg3, 3, "XChangeKeyboardControl", "XKeyboardControl*");
  arg3 = XEN_COPY_ARG(larg3);
  kc.key_click_percent = XEN_TO_C_INT(XEN_CAR(arg3)); arg3 = XEN_CDR(arg3);
  if (!(XEN_NULL_P(arg3))) 
    {
      kc.bell_percent = XEN_TO_C_INT(XEN_CAR(arg3)); arg3 = XEN_CDR(arg3);
      if (!(XEN_NULL_P(arg3)))
	{
	  kc.bell_pitch = XEN_TO_C_INT(XEN_CAR(arg3)); arg3 = XEN_CDR(arg3);
	  if (!(XEN_NULL_P(arg3)))
	    {
	      kc.bell_duration = XEN_TO_C_INT(XEN_CAR(arg3)); arg3 = XEN_CDR(arg3);
	      if (!(XEN_NULL_P(arg3)))
		{
		  kc.led = XEN_TO_C_INT(XEN_CAR(arg3)); arg3 = XEN_CDR(arg3);
		  if (!(XEN_NULL_P(arg3)))
		    {
		      kc.led_mode = XEN_TO_C_INT(XEN_CAR(arg3)); arg3 = XEN_CDR(arg3);
		      if (!(XEN_NULL_P(arg3)))
			{
			  kc.key = XEN_TO_C_INT(XEN_CAR(arg3)); arg3 = XEN_CDR(arg3);
			  if (!(XEN_NULL_P(arg3)))
			    {
			      kc.auto_repeat_mode = XEN_TO_C_INT(XEN_CAR(arg3));
			    }}}}}}}
  return(C_TO_XEN_INT(XChangeKeyboardControl(XEN_TO_C_Display(arg1), XEN_TO_C_ULONG(arg2), &kc)));
}

static XEN gxm_XChangeGC(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XChangeGC "XChangeGC(display, gc, valuemask, values) changes the components specified by valuemask for the specified GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XChangeGC", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XChangeGC", "GC");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XChangeGC", "ulong");
  XEN_ASSERT_TYPE(XEN_XGCValues_P(arg4), arg4, 4, "XChangeGC", "XGCValues*");
  return(C_TO_XEN_INT(XChangeGC(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2), XEN_TO_C_ULONG(arg3), XEN_TO_C_XGCValues(arg4))));
}

static XEN gxm_XChangeActivePointerGrab(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XChangeActivePointerGrab "XChangeActivePointerGrab(display, event_mask, cursor, time) changes the specified dynamic parameters \
if the pointer is actively grabbed by the client and if the specified time is no earlier than the last-pointer-grab time and no later than \
the current X server time."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XChangeActivePointerGrab", "Display*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XChangeActivePointerGrab", "unsigned int");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg3), arg3, 3, "XChangeActivePointerGrab", "Cursor");
  XEN_ASSERT_TYPE(XEN_Time_P(arg4), arg4, 4, "XChangeActivePointerGrab", "Time");
  return(C_TO_XEN_INT(XChangeActivePointerGrab(XEN_TO_C_Display(arg1), XEN_TO_C_ULONG(arg2), XEN_TO_C_Cursor(arg3), XEN_TO_C_Time(arg4))));
}

static XEN gxm_XCellsOfScreen(XEN arg1)
{
  #define H_CellsOfScreen "CellsOfScreen(screen) returns the number of colormap cells in the default colormap of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XCellsOfScreen", "Screen*");
  return(C_TO_XEN_INT(XCellsOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XBitmapUnit(XEN arg1)
{
  #define H_BitmapUnit "BitmapUnit(display) returns the size of a bitmap's scanline unit in bits."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XBitmapUnit", "Display*");
  return(C_TO_XEN_INT(XBitmapUnit(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XBitmapPad(XEN arg1)
{
  #define H_BitmapPad "BitmapPad(display) returns the number of bits that each scanline must be padded."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XBitmapPad", "Display*");
  return(C_TO_XEN_INT(XBitmapPad(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XBitmapBitOrder(XEN arg1)
{
  #define H_BitmapBitOrder "BitmapBitOrder(display) returns LSBFirst or MSBFirst to indicate whether the leftmost bit in the bitmap as \
displayed on the screen is the least or most significant bit in the unit."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XBitmapBitOrder", "Display*");
  return(C_TO_XEN_INT(XBitmapBitOrder(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XBell(XEN arg1, XEN arg2)
{
  #define H_XBell "XBell(display, percent) rings the bell on the keyboard on the specified display, if possible."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XBell", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XBell", "int");
  return(C_TO_XEN_INT(XBell(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XAutoRepeatOn(XEN arg1)
{
  #define H_XAutoRepeatOn "XAutoRepeatOn(display) turns on auto-repeat for the keyboard on the specified display."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XAutoRepeatOn", "Display*");
  return(C_TO_XEN_INT(XAutoRepeatOn(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XAutoRepeatOff(XEN arg1)
{
  #define H_XAutoRepeatOff "XAutoRepeatOff(display) turns off auto-repeat for the keyboard on the specified display."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XAutoRepeatOff", "Display*");
  return(C_TO_XEN_INT(XAutoRepeatOff(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XAllowEvents(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XAllowEvents "XAllowEvents(display, event_mode, time)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XAllowEvents", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XAllowEvents", "int");
  XEN_ASSERT_TYPE(XEN_Time_P(arg3), arg3, 3, "XAllowEvents", "Time");
  return(C_TO_XEN_INT(XAllowEvents(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_Time(arg3))));
}

static XEN gxm_XAllocNamedColor(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XAllocNamedColor "Status XAllocNamedColor(display, colormap, color_name, screen_def_return, exact_def_return) looks up the \
named color with respect to the screen that is associated with the specified colormap."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XAllocNamedColor", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XAllocNamedColor", "Colormap");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XAllocNamedColor", "char*");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg4), arg4, 4, "XAllocNamedColor", "XColor");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg5), arg5, 5, "XAllocNamedColor", "XColor");
  return(C_TO_XEN_INT(XAllocNamedColor(XEN_TO_C_Display(arg1), 
				       XEN_TO_C_Colormap(arg2), 
				       XEN_TO_C_STRING(arg3), 
				       XEN_TO_C_XColor(arg4), XEN_TO_C_XColor(arg5))));
}

static XEN gxm_XAllocColorPlanes(XEN args)
{
  #define H_XAllocColorPlanes "Status XAllocColorPlanes(display, colormap, contig, ncolors, nreds, ngreens, nblues)"
  /* DIFF: XAllocColorPlanes omits pixel array (arg4) and trailing 3 args, returns them and embedded list of pixels
   */
  unsigned long r,g,b;
  unsigned long *ps;
  int i, len, val;
  XEN lst = XEN_FALSE;
  XEN arg1, arg2, arg3, arg5, arg6, arg7, arg8;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg5 = XEN_LIST_REF(args, 3);
  arg6 = XEN_LIST_REF(args, 4);
  arg7 = XEN_LIST_REF(args, 5);
  arg8 = XEN_LIST_REF(args, 6);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XAllocColorPlanes", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XAllocColorPlanes", "Colormap");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XAllocColorPlanes", "Bool");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XAllocColorPlanes", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XAllocColorPlanes", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XAllocColorPlanes", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XAllocColorPlanes", "int");
  len = XEN_TO_C_INT(arg5);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg5, 5, "XAllocColorPlanes", "positive integer");
  ps = (unsigned long *)CALLOC(len, sizeof(unsigned long));
  val = XAllocColorPlanes(XEN_TO_C_Display(arg1), 
			  XEN_TO_C_Colormap(arg2), 
			  XEN_TO_C_BOOLEAN(arg3), 
			  ps, len, 
			  XEN_TO_C_INT(arg6), XEN_TO_C_INT(arg7), XEN_TO_C_INT(arg8), 
			  &r, &g, &b);
  if (val != 0)
    {
      XEN plist = XEN_EMPTY_LIST;
      int loc;
      loc = xm_protect(plist);
      for (i = len - 1; i >= 0; i--)
	plist = XEN_CONS(C_TO_XEN_ULONG(ps[i]), plist);
      xm_unprotect_at(loc);
      lst = XEN_LIST_5(C_TO_XEN_INT(val),
		       plist,
		       C_TO_XEN_ULONG(r),
		       C_TO_XEN_ULONG(g),
		       C_TO_XEN_ULONG(b));
    }
  FREE(ps);
  return(lst);
}

static XEN gxm_XAllocColorCells(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XAllocColorCells "Status XAllocColorCells(display, colormap, contig, nplanes, npixels) \
allocates read/write color cells."
  /* DIFF: XAllocColorCells arg 4 and 6 omitted and returned as (embedded) lists 
   */
  int mlen, plen, i, val;
  XEN mlst = XEN_EMPTY_LIST, plst = XEN_EMPTY_LIST;
  unsigned long *ms, *ps;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XAllocColorCells", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XAllocColorCells", "Colormap");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XAllocColorCells", "Bool");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XAllocColorCells", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XAllocColorCells", "unsigned int");
  mlen = XEN_TO_C_INT(arg4);
  if (mlen <= 0) return(XEN_FALSE);
  plen = XEN_TO_C_INT(arg5);
  if (plen <= 0) return(XEN_FALSE);
  ms = (unsigned long *)CALLOC(mlen, sizeof(unsigned long));
  ps = (unsigned long *)CALLOC(plen, sizeof(unsigned long));
  val = XAllocColorCells(XEN_TO_C_Display(arg1), 
			 XEN_TO_C_Colormap(arg2), 
			 XEN_TO_C_BOOLEAN(arg3), 
			 ms, mlen, 
			 ps, plen);
  if (val != 0)
    {
      int loc1, loc2;
      loc1 = xm_protect(mlst);
      loc2 = xm_protect(plst);
      for (i = mlen - 1; i >= 0; i--) 
	mlst = XEN_CONS(C_TO_XEN_ULONG(ms[i]), mlst);
      for (i = plen - 1; i >= 0; i--) 
	mlst = XEN_CONS(C_TO_XEN_ULONG(ps[i]), plst);
      xm_unprotect_at(loc1);
      xm_unprotect_at(loc2);
    }
  FREE(ms);
  FREE(ps);
  if (val != 0)
    return(XEN_LIST_3(C_TO_XEN_INT(val),
		      mlst,
		      plst));
  return(XEN_FALSE);
}

static XEN gxm_XAllocColor(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XAllocColor "Status XAllocColor(display, colormap, screen_in_out) allocates a read-only colormap entry corresponding to the \
closest RGB value supported by the hardware."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XAllocColor", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XAllocColor", "Colormap");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg3), arg3, 3, "XAllocColor", "XColor"); 
  return(C_TO_XEN_INT(XAllocColor(XEN_TO_C_Display(arg1), XEN_TO_C_Colormap(arg2), XEN_TO_C_XColor(arg3))));
}

static XEN gxm_XActivateScreenSaver(XEN arg1)
{
  #define H_XActivateScreenSaver "XActivateScreenSaver(display) activates the screen saver."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XActivateScreenSaver", "Display*");
  return(C_TO_XEN_INT(XActivateScreenSaver(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XSetTransientForHint(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XSetTransientForHint "XSetTransientForHint(display, w, prop_window) sets the WM_TRANSIENT_FOR property of the specified \
window to the specified prop_window."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetTransientForHint", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetTransientForHint", "Window");
  XEN_ASSERT_TYPE(XEN_Window_P(arg3), arg3, 3, "XSetTransientForHint", "Window");
  return(C_TO_XEN_INT(XSetTransientForHint(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_Window(arg3))));
}

static XEN gxm_XSetWMColormapWindows(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XSetWMColormapWindows "Status XSetWMColormapWindows(display, w, colormap_windows, count) replaces the WM_COLORMAP_WINDOWS property \
on the specified window with the list of windows specified by the colormap_windows argument."
  /* DIFF: XSetWMColormapWindows arg 3 is list of Windows
   */
  int len, rtn;
  Window *ws;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetWMColormapWindows", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetWMColormapWindows", "Window");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XSetWMColormapWindows", "list of Windows");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XSetWMColormapWindows", "int");
  len = XEN_TO_C_INT(arg4);
  if (len <= 0) return(XEN_FALSE);
  ws = XEN_TO_C_Windows(arg3, len);
  rtn = XSetWMColormapWindows(XEN_TO_C_Display(arg1), 
			      XEN_TO_C_Window(arg2),
			      ws, len);
  FREE(ws);
  return(C_TO_XEN_INT(rtn));
}

static XEN gxm_XGetWMColormapWindows(XEN arg1, XEN arg2)
{
  #define H_XGetWMColormapWindows "Status XGetWMColormapWindows(display, w) returns the list of \
window identifiers stored in the WM_COLORMAP_WINDOWS property on the specified window."
  /* DIFF: XGetWMColormapWindows omit last 2 args, return list of windows
   */
  XEN lst = XEN_EMPTY_LIST;
  int i, len, rtn, loc;
  Window *ws;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetWMColormapWindows", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetWMColormapWindows", "Window");
  rtn = XGetWMColormapWindows(XEN_TO_C_Display(arg1), 
			      XEN_TO_C_Window(arg2), 
			      &ws, &len);
  if (rtn == 0)
    return(XEN_FALSE);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_Window(ws[i]), lst);
  xm_unprotect_at(loc);
  XFree(ws);
  return(lst);
}

static XEN gxm_XGetCommand(XEN arg1, XEN arg2)
{
  #define H_XGetCommand "Status XGetCommand(display, w) reads the WM_COMMAND property from the specified window \
and returns a string list."
  /* DIFF: XGetCommand omits last 2 args, returns list
   */
  char **argv;
  int argc;
  XEN lst = XEN_EMPTY_LIST;
  Status err;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetCommand", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetCommand", "Window");
  err = XGetCommand(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), &argv, &argc);
  if (err != 0)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = argc - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_STRING(argv[i]), lst);
      XFreeStringList(argv);
      xm_unprotect_at(loc);
    }
  return(lst);
}

static XEN gxm_XWithdrawWindow(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XWithdrawWindow "Status XWithdrawWindow(display, w, screen_number) unmaps the specified window and sends a synthetic  \
UnmapNotify event to the root window of the specified screen."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XWithdrawWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XWithdrawWindow", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XWithdrawWindow", "int");
  return(C_TO_XEN_INT(XWithdrawWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XIconifyWindow(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XIconifyWindow "Status XIconifyWindow(display, w, screen_number) sends a WM_CHANGE_STATE  ClientMessage event with a format of \
32 and a first data element of IconicState to the root window of the specified screen with an event mask set to \
SubstructureNotifyMask | SubstructureRedirectMask."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XIconifyWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XIconifyWindow", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XIconifyWindow", "int");
  return(C_TO_XEN_INT(XIconifyWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XSetWMProtocols(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XSetWMProtocols "Status XSetWMProtocols(display, w, protocols, count) replaces the WM_PROTOCOLS property on the \
specified window with the list of atoms specified by the protocols argument."
  /* DIFF: XSetWMProtocols arg3 is list of atoms
   */
  Atom *outs;
  int val, len;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetWMProtocols", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XSetWMProtocols", "Window");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XSetWMProtocols", "list of Atom");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XSetWMProtocols", "int");
  len = XEN_TO_C_INT(arg4);
  if (len <= 0) return(XEN_FALSE);
  outs = XEN_TO_C_Atoms(arg3, len);
  val = XSetWMProtocols(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), outs, len);
  FREE(outs);
  return(C_TO_XEN_INT(val));
}

static XEN gxm_XGetWMProtocols(XEN arg1, XEN arg2)
{
  #define H_XGetWMProtocols "Status XGetWMProtocols(display, w) returns the list of atoms stored in the \
WM_PROTOCOLS property on the specified window."
  /* DIFF: XGetWMProtocols omits last 2 args, returns list
   */
  Atom *ats;
  int len, i, val, loc;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetWMProtocols", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetWMProtocols", "Window");
  val = XGetWMProtocols(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), &ats, &len);
  if (val == 0)
    return(XEN_FALSE);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_Atom(ats[i]), lst);
  xm_unprotect_at(loc);
  XFree((void *)ats);
  return(lst);
}

static XEN gxm_XReconfigureWMWindow(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XReconfigureWMWindow "Status XReconfigureWMWindow(display, w, screen_number, value_mask, values) issues a  ConfigureWindow \
request on the specified top-level window."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XReconfigureWMWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XReconfigureWMWindow", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XReconfigureWMWindow", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XReconfigureWMWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_XWindowChanges_P(arg5), arg5, 5, "XReconfigureWMWindow", "XWindowChanges*");
  return(C_TO_XEN_INT(XReconfigureWMWindow(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_ULONG(arg4), 
					   XEN_TO_C_XWindowChanges(arg5))));
}

static XEN gxm_XListDepths(XEN arg1, XEN arg2)
{
  #define H_XListDepths "int *XListDepths(display, screen_number) returns the array of depths that are available on the \
specified screen."
  /* DIFF: XListDepths omits last arg, returns list of depths
   */
  XEN lst = XEN_EMPTY_LIST;
  int i, len, loc;
  int *ds;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XListDepths", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XListDepths", "int");
  ds = XListDepths(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2), &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_INT(ds[i]), lst);
  XFree(ds);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XListPixmapFormats(XEN arg1)
{
  #define H_XListPixmapFormats "XPixmapFormatValues *XListPixmapFormats(display) returns an array of XPixmapFormatValues \
structures that describe the types of Z format images supported by the specified display."
  /* DIFF: XListPixmapFormats omits arg2, rtns list of lists, each holding XPixmapFormatValues data
   */
  XPixmapFormatValues *ps;
  int len, loc;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XListPixmapFormats", "Display*");
  ps = XListPixmapFormats(XEN_TO_C_Display(arg1), &len);
  loc = xm_protect(lst);
  if (ps)
    {
      int i;
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(XEN_LIST_3(C_TO_XEN_INT(ps[i].depth),
				  C_TO_XEN_INT(ps[i].bits_per_pixel),
				  C_TO_XEN_INT(ps[i].scanline_pad)),
		       lst);
      free(ps);
    }
  xm_unprotect_at(loc);
  return(lst);
}


static XEN xm_XIOErrorHandler = XEN_FALSE;
static int gxm_XIOErrorHandler(Display *dpy)
{
  XEN_CALL_1(xm_XIOErrorHandler, C_TO_XEN_Display(dpy), c__FUNCTION__);
  return(0); /* never happens */
}

static XEN gxm_XSetIOErrorHandler(XEN arg1)
{
  #define H_XSetIOErrorHandler "int (*XSetIOErrorHandler(handler))() sets the fatal I/O error handler. "
  XEN old_val;
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg1) || XEN_PROCEDURE_P(arg1), arg1, XEN_ONLY_ARG, "XSetIOErrorHandler", "#f=NULL or function of 1 arg");
  xm_protect(arg1);
  old_val = xm_XIOErrorHandler;
  xm_XIOErrorHandler = arg1;
  if (XEN_FALSE_P(arg1))
    XSetIOErrorHandler(NULL);
  else XSetIOErrorHandler(gxm_XIOErrorHandler);
  if (XEN_PROCEDURE_P(old_val)) xm_unprotect(old_val); /* hmmm... what if we're gc'd on the way back? */
  return(old_val);
}

static XEN xm_XErrorHandler = XEN_FALSE;
static int gxm_XErrorHandler(Display *dpy, XErrorEvent *e)
{
  XEN_CALL_2(xm_XErrorHandler, C_TO_XEN_Display(dpy), C_TO_XEN_XErrorEvent((XErrorEvent *)e), c__FUNCTION__);
  return(0); /* never happens */
}

static XEN gxm_XSetErrorHandler(XEN arg1)
{
  #define H_XSetErrorHandler "XSetErrorHandler(proc) causes proc to be called if an error occurs"
  XEN old_val;
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg1) || XEN_PROCEDURE_P(arg1), arg1, XEN_ONLY_ARG, "XSetErrorHandler", "#f=NULL or function of 2 args");
  xm_protect(arg1);
  old_val = xm_XErrorHandler;
  xm_XErrorHandler = arg1;
  if (XEN_FALSE_P(arg1))
    XSetErrorHandler(NULL);
  else XSetErrorHandler(gxm_XErrorHandler);
  if (XEN_PROCEDURE_P(old_val)) xm_unprotect(old_val); /* hmmm... what if we're gc'd on the way back? */
  return(old_val);
}

static XEN gxm_XScreenNumberOfScreen(XEN arg1)
{
  #define H_XScreenNumberOfScreen "int XScreenNumberOfScreen(screen) returns the screen index number of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XScreenNumberOfScreen", "Screen*");
  return(C_TO_XEN_INT(XScreenNumberOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XEventMaskOfScreen(XEN arg1)
{
  #define H_EventMaskOfScreen "EventMaskOfScreen(screen) returns the root event mask of the root window for the specified screen at connection setup."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XEventMaskOfScreen", "Screen*");
  return(C_TO_XEN_ULONG(XEventMaskOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XDefaultScreenOfDisplay(XEN arg1)
{
  /* #define DefaultScreenOfDisplay(display) */
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDefaultScreenOfDisplay", "Display*");
  return(C_TO_XEN_Screen(XDefaultScreenOfDisplay(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XScreenOfDisplay(XEN arg1, XEN arg2)
{
  #define H_ScreenOfDisplay "ScreenOfDisplay(display, screen_number) returns a pointer to the screen of the specified display."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XScreenOfDisplay", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XScreenOfDisplay", "int");
  return(C_TO_XEN_Screen(XScreenOfDisplay(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XDisplayOfScreen(XEN arg1)
{
  #define H_DisplayOfScreen "DisplayOfScreen(screen) returns the display of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XDisplayOfScreen", "Screen*");
  return(C_TO_XEN_Display(XDisplayOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XDefaultColormapOfScreen(XEN arg1)
{
  #define H_DefaultColormapOfScreen "DefaultColormapOfScreen(screen) returns the default colormap of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XDefaultColormapOfScreen", "Screen*");
  return(C_TO_XEN_Colormap(XDefaultColormapOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XDefaultColormap(XEN arg1, XEN arg2)
{
  #define H_DefaultColormap "DefaultColormap(display, screen_number) returns the default colormap ID for allocation on the specified screen."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDefaultColormap", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XDefaultColormap", "int");
  return(C_TO_XEN_Colormap(XDefaultColormap(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XDisplayString(XEN arg1)
{
  #define H_DisplayString "DisplayString(display) returns the string that was passed to XOpenDisplay when the current display was opened."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDisplayString", "Display*");
  return(C_TO_XEN_STRING(XDisplayString(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XServerVendor(XEN arg1)
{
  #define H_ServerVendor "ServerVendor(display) returns a pointer to a null-terminated string that provides some identification of the \
owner of the X server implementation."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XServerVendor", "Display*");
  return(C_TO_XEN_STRING(XServerVendor(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XLastKnownRequestProcessed(XEN arg1)
{
  #define H_LastKnownRequestProcessed "LastKnownRequestProcessed(display) extracts the full serial number of the last request known by Xlib \
to have been processed by the X server."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XLastKnownRequestProcessed", "Display*");
  return(C_TO_XEN_ULONG(XLastKnownRequestProcessed(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XNextRequest(XEN arg1)
{
  #define H_NextRequest "NextRequest(display) extracts the full serial number that is to be used for the next request."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XNextRequest", "Display*");
  return(C_TO_XEN_ULONG(XNextRequest(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XWhitePixelOfScreen(XEN arg1)
{
  #define H_WhitePixelOfScreen "WhitePixelOfScreen(screen) returns the white pixel value of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XWhitePixelOfScreen", "Screen*");
  return(C_TO_XEN_Pixel(XWhitePixelOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XBlackPixelOfScreen(XEN arg1)
{
  #define H_BlackPixelOfScreen "BlackPixelOfScreen(screen) returns the black pixel value of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XBlackPixelOfScreen", "Screen*");
  return(C_TO_XEN_Pixel(XBlackPixelOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XAllPlanes(void)
{
  #define H_AllPlanes "Allplanes() returns a value with all bits set to 1 suitable for use in a plane argument to a procedure."
  return(C_TO_XEN_ULONG(XAllPlanes()));
}

static XEN gxm_XWhitePixel(XEN arg1, XEN arg2)
{
  #define H_WhitePixel "WhitePixel(display, screen_number) returns the white pixel value for the specified screen."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XWhitePixel", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XWhitePixel", "int");
  return(C_TO_XEN_Pixel(XWhitePixel(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XBlackPixel(XEN arg1, XEN arg2)
{
  #define H_BlackPixel "BlackPixel(display, screen_number) returns the black pixel value for the specified screen."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XBlackPixel", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XBlackPixel", "int");
  return(C_TO_XEN_Pixel(XBlackPixel(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XDefaultGCOfScreen(XEN arg1)
{
  #define H_DefaultGCOfScreen "DefaultGCOfScreen(screen) returns the default GC of the specified screen, which has the same depth as the root \
window of the screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XDefaultGCOfScreen", "Screen*");
  return(C_TO_XEN_GC(XDefaultGCOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XDefaultGC(XEN arg1, XEN arg2)
{
  #define H_DefaultGC "DefaultGC(display, screen_number) returns the default GC for the root window of the "
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDefaultGC", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XDefaultGC", "int");
  return(C_TO_XEN_GC(XDefaultGC(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XDefaultVisualOfScreen(XEN arg1)
{
  #define H_DefaultVisualOfScreen "DefaultVisualOfScreen(screen) returns the default visual of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XDefaultVisualOfScreen", "Screen*");
  return(C_TO_XEN_Visual(XDefaultVisualOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XDefaultVisual(XEN arg1, XEN arg2)
{
  #define H_DefaultVisual "DefaultVisual(display, screen_number) returns the default visual type for the specified screen."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDefaultVisual", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XDefaultVisual", "int");
  return(C_TO_XEN_Visual(XDefaultVisual(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XRootWindowOfScreen(XEN arg1)
{
  #define H_RootWindowOfScreen "RootWindowOfScreen(screen) returns the root window of the specified screen."
  XEN_ASSERT_TYPE(XEN_Screen_P(arg1), arg1, 1, "XRootWindowOfScreen", "Screen*");
  return(C_TO_XEN_Window(XRootWindowOfScreen(XEN_TO_C_Screen(arg1))));
}

static XEN gxm_XDefaultRootWindow(XEN arg1)
{
  #define H_DefaultRootWindow "DefaultRootWindow(display) returns the root window for the default screen."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDefaultRootWindow", "Display*");
  return(C_TO_XEN_Window(XDefaultRootWindow(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XRootWindow(XEN arg1, XEN arg2)
{
  #define H_RootWindow "RootWindow(display, screen_number) returns the root window."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XRootWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XRootWindow", "int");
  return(C_TO_XEN_Window(XRootWindow(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XVisualIDFromVisual(XEN arg1)
{
  #define H_XVisualIDFromVisual "VisualID XVisualIDFromVisual(visual) returns the visual ID for the specified visual type."
  XEN_ASSERT_TYPE(XEN_Visual_P(arg1), arg1, 1, "XVisualIDFromVisual", "Visual*");
  return(C_TO_XEN_ULONG(XVisualIDFromVisual(XEN_TO_C_Visual(arg1))));
}

static XEN gxm_XDisplayMotionBufferSize(XEN arg1)
{
  #define H_XDisplayMotionBufferSize "unsigned long XDisplayMotionBufferSize(display)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XDisplayMotionBufferSize", "Display*");
  return(C_TO_XEN_ULONG(XDisplayMotionBufferSize(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XExtendedMaxRequestSize(XEN arg1)
{
  #define H_XExtendedMaxRequestSize "long XExtendedMaxRequestSize(display) returns zero if the specified display does not support an \
extended-length protocol encoding; otherwise, it returns the maximum request size (in 4-byte units) supported by the server using the \
extended-length encoding."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XExtendedMaxRequestSize", "Display*");
  return(C_TO_XEN_INT(XExtendedMaxRequestSize(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XMaxRequestSize(XEN arg1)
{
  #define H_XMaxRequestSize "long XMaxRequestSize(display) returns the maximum request size (in 4-byte units) supported by the server \
without using an extended-length protocol encoding."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XMaxRequestSize", "Display*");
  return(C_TO_XEN_INT(XMaxRequestSize(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XStringToKeysym(XEN arg1)
{
  #define H_XStringToKeysym "KeySym XStringToKeysym(string)"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XStringToKeysym", "char*");
  return(C_TO_XEN_KeySym(XStringToKeysym(XEN_TO_C_STRING(arg1))));
}

static XEN gxm_XGetKeyboardMapping(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XGetKeyboardMapping "KeySym *XGetKeyboardMapping(display, first_keycode, keycode_count) returns \
the symbols for the specified number of KeyCodes starting with first_keycode."
  /* DIFF: XGetKeyboardMapping omits last arg, returns list of keys
   */
  int n, i, len, count, loc;
  KeySym *keys;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetKeyboardMapping", "Display*");
  XEN_ASSERT_TYPE(XEN_KeyCode_P(arg2), arg2, 2, "XGetKeyboardMapping", "KeyCode");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XGetKeyboardMapping", "int");
  count = XEN_TO_C_INT(arg3);
  keys = XGetKeyboardMapping(XEN_TO_C_Display(arg1), XEN_TO_C_KeyCode(arg2), count, &n);
  len = count * n;
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_KeySym(keys[i]), lst);
  XFree(keys);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XLookupKeysym(XEN arg1, XEN arg2)
{
  #define H_XLookupKeysym "KeySym XLookupKeysym(key_event, index) uses a given keyboard event and the index you specified to return \
the KeySym from the list that corresponds to the KeyCode member in the XKeyPressedEvent or XKeyReleasedEvent structure."
  XEN_ASSERT_TYPE(XEN_XKeyEvent_P(arg1), arg1, 1, "XLookupKeysym", "XKeyEvent*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XLookupKeysym", "int");
  return(C_TO_XEN_KeySym(XLookupKeysym(XEN_TO_C_XKeyEvent(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XKeycodeToKeysym(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XKeycodeToKeysym "KeySym XKeycodeToKeysym(display, keycode, index) uses internal Xlib tables and returns the KeySym defined \
for the specified KeyCode and the element of the KeyCode vector."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XKeycodeToKeysym", "Display*");
  XEN_ASSERT_TYPE(XEN_KeyCode_P(arg2), arg2, 2, "XKeycodeToKeysym", "KeyCode");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XKeycodeToKeysym", "int");
  return(C_TO_XEN_KeySym(XKeycodeToKeysym(XEN_TO_C_Display(arg1), XEN_TO_C_KeyCode(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XListProperties(XEN arg1, XEN arg2)
{
  #define H_XListProperties "Atom *XListProperties(display, w) returns a pointer to an array of atom properties that \
are defined for the specified window or returns NULL if no properties were found."
  /* DIFF: XListProperties returns list, no arg3
   */
  int i, len, loc;
  XEN lst = XEN_EMPTY_LIST;
  Atom *ats;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XListProperties", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XListProperties", "Window");
  ats = XListProperties(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), &len);
  if (ats == NULL)
    return(XEN_FALSE);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_Atom(ats[i]), lst);
  xm_unprotect_at(loc);
  XFree(ats);
  return(lst);
}

static XEN gxm_XListExtensions(XEN arg1)
{
  #define H_XListExtensions "XListExtensions(dpy) -> list of strings describing available extensions"
  /* DIFF: XListExtensions omits arg2, returns list
   */
  int i, len, loc;
  XEN lst = XEN_EMPTY_LIST;
  char **str;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XListExtensions", "Display*");
  str = XListExtensions(XEN_TO_C_Display(arg1), &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_STRING(str[i]), lst);
  XFreeExtensionList(str);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XGetFontPath(XEN arg1)
{
  #define H_XGetFontPath "char **XGetFontPath(display) allocates and returns an array of strings containing the search path."
  /* DIFF: XGetFontPath omits arg2, returns list
   */
  int i, len, loc;
  XEN lst = XEN_EMPTY_LIST;
  char **str;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetFontPath", "Display*");
  str = XGetFontPath(XEN_TO_C_Display(arg1), &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_STRING(str[i]), lst);
  XFreeFontPath(str);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XListFontsWithInfo(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XListFontsWithInfo "char **XListFontsWithInfo(display, pattern, maxnames) returns a list of \
font names that match the specified pattern and their associated font information."
  /* DIFF: XListFontsWithInfo omit last 2 args, returns list of lists
   */
  int i, count, loc;
  XFontStruct *info;
  char **val;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XListFontsWithInfo", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XListFontsWithInfo", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XListFontsWithInfo", "int");
  val = XListFontsWithInfo(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2), XEN_TO_C_INT(arg3), &count, &info);
  loc = xm_protect(lst);
  for (i = count - 1; i >= 0; i--)
    lst = XEN_CONS(XEN_LIST_2(C_TO_XEN_STRING(val[i]), 
			      C_TO_XEN_XFontStruct(&(info[i]))),
		   lst);
  XFreeFontInfo(val, info, count);
  /* XFreeFontNames(val); */
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XListFonts(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XListFonts "char **XListFonts(display, pattern, maxnames) returns an array of available font names that match \
the string you passed to the pattern argument."
  /* DIFF: XListFonts omits arg4, returns list
   */
  int i, len, loc;
  XEN lst = XEN_EMPTY_LIST;
  char **str;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XListFonts", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XListFonts", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XListFonts", "int");
  str = XListFonts(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2), XEN_TO_C_INT(arg3), &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_STRING(str[i]), lst);
  XFreeFontNames(str);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XListInstalledColormaps(XEN arg1, XEN arg2)
{
  #define H_XListInstalledColormaps "Colormap *XListInstalledColormaps(display, w) returns a list of the currently installed \
colormaps for the screen of the specified window."
  /* DIFF: XListInstalledColormaps omits last arg, rtns list of XColormaps
   */
  Colormap *cm;
  int i, len, loc;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XListInstalledColormaps", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XListInstalledColormaps", "Window");
  cm = XListInstalledColormaps(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_Colormap(cm[i]), lst);
  free(cm);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XCreateWindow(XEN args)
{
  #define H_XCreateWindow "Window XCreateWindow(display, parent, x, y, width, height, border_width, depth, class, visual, valuemask, attributes) \
creates an unmapped subwindow for a specified parent window, returns the window ID of the created window, and causes the X server to generate \
a CreateNotify event."
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  arg10 = XEN_LIST_REF(args, 9);
  arg11 = XEN_LIST_REF(args, 10);
  arg12 = XEN_LIST_REF(args, 11);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreateWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCreateWindow", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XCreateWindow", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XCreateWindow", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XCreateWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XCreateWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XCreateWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XCreateWindow", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg9), arg9, 9, "XCreateWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_Visual_P(arg10), arg10, 10, "XCreateWindow", "Visual*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg11), arg11, 11, "XCreateWindow", "ulong");
  XEN_ASSERT_TYPE(XEN_XSetWindowAttributes_P(arg12), arg12, 12, "XCreateWindow", "XSetWindowAttributes*");
  return(C_TO_XEN_Window(XCreateWindow(XEN_TO_C_Display(arg1), 
				       XEN_TO_C_Window(arg2), 
				       XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), 
				       XEN_TO_C_ULONG(arg5), XEN_TO_C_ULONG(arg6), 
				       XEN_TO_C_ULONG(arg7), XEN_TO_C_INT(arg8), 
				       XEN_TO_C_ULONG(arg9), XEN_TO_C_Visual(arg10), 
				       XEN_TO_C_ULONG(arg11), 
				       XEN_TO_C_XSetWindowAttributes(arg12))));
}

static XEN gxm_XGetSelectionOwner(XEN arg1, XEN arg2)
{
  #define H_XGetSelectionOwner "Window XGetSelectionOwner(display, selection) returns the window ID associated with the window that \
currently owns the specified selection."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetSelectionOwner", "Display*");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XGetSelectionOwner", "Atom");
  return(C_TO_XEN_Window(XGetSelectionOwner(XEN_TO_C_Display(arg1), XEN_TO_C_Atom(arg2))));
}

static XEN gxm_XCreateSimpleWindow(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8, XEN arg9)
{
  #define H_XCreateSimpleWindow "Window XCreateSimpleWindow(display, parent, x, y, width, height, border_width, border, background) \
creates an unmapped InputOutput subwindow for a specified parent window, returns the window ID of the created window, and causes the X \
server to generate a CreateNotify event."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreateSimpleWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCreateSimpleWindow", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XCreateSimpleWindow", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XCreateSimpleWindow", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XCreateSimpleWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XCreateSimpleWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XCreateSimpleWindow", "unsigned int");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg8), arg8, 8, "XCreateSimpleWindow", "Pixel");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg9), arg9, 9, "XCreateSimpleWindow", "Pixel");
  return(C_TO_XEN_Window(XCreateSimpleWindow(XEN_TO_C_Display(arg1), 
					     XEN_TO_C_Window(arg2), 
					     XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), 
					     XEN_TO_C_ULONG(arg5), XEN_TO_C_ULONG(arg6), 
					     XEN_TO_C_ULONG(arg7), XEN_TO_C_Pixel(arg8), 
					     XEN_TO_C_Pixel(arg9))));
}

static XEN gxm_XCreatePixmapFromBitmapData(XEN arg1, XEN arg2, XEN larg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8)
{
  #define H_XCreatePixmapFromBitmapData "Pixmap XCreatePixmapFromBitmapData(display, d, data, width, height, fg, bg, depth) creates a \
pixmap of the given depth and then does a bitmap-format XPutImage of the data into it."
  /* DIFF: XCreatePixmapFromBitmapData takes list of chars as arg3 (not char *)
   */
  char *bits;
  int i, len;
  Pixmap p;
  XEN arg3;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreatePixmapFromBitmapData", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCreatePixmapFromBitmapData", "Drawable");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg3), larg3, 3, "XCreatePixmapFromBitmapData", "list of char");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XCreatePixmapFromBitmapData", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XCreatePixmapFromBitmapData", "unsigned int");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg6), arg6, 6, "XCreatePixmapFromBitmapData", "pixel");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg7), arg7, 7, "XCreatePixmapFromBitmapData", "pixel");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg8), arg8, 8, "XCreatePixmapFromBitmapData", "unsigned int");
  len = XEN_LIST_LENGTH(larg3);
  if (len <= 0) XEN_ASSERT_TYPE(0, larg3, 3, "XCreatePixmapFromBitmapData", "positive integer");
  arg3 = XEN_COPY_ARG(larg3);
  bits = (char *)CALLOC(len, sizeof(char));
  for (i = 0; i < len; i++, arg3 = XEN_CDR(arg3))
    bits[i] = (char)XEN_TO_C_INT(XEN_CAR(arg3));
  p = XCreatePixmapFromBitmapData(XEN_TO_C_Display(arg1), 
				  XEN_TO_C_Window(arg2), 
				  bits,
				  XEN_TO_C_ULONG(arg4), XEN_TO_C_ULONG(arg5), 
				  XEN_TO_C_Pixel(arg6), XEN_TO_C_Pixel(arg7), 
				  XEN_TO_C_ULONG(arg8));
  FREE(bits);
  return(C_TO_XEN_Pixmap(p));
}

static XEN gxm_XCreateBitmapFromData(XEN arg1, XEN arg2, XEN larg3, XEN arg4, XEN arg5)
{
  #define H_XCreateBitmapFromData "Pixmap XCreateBitmapFromData(display, d, data, width, height) allows you to include in your C \
program a bitmap file that was written out by XWriteBitmapFile"
  /* DIFF: XCreateBitmapFromData takes list of chars as arg3 (not char *)
   */
  char *bits;
  int i, len;
  Pixmap p;
  XEN arg3;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreateBitmapFromData", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCreateBitmapFromData", "Drawable");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg3), larg3, 3, "XCreateBitmapFromData", "list of char");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XCreateBitmapFromData", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XCreateBitmapFromData", "unsigned int");
  len = XEN_LIST_LENGTH(larg3);
  if (len <= 0) XEN_ASSERT_TYPE(0, larg3, 3, "XCreateBitmapFromData", "positive integer");
  arg3 = XEN_COPY_ARG(larg3);
  bits = (char *)CALLOC(len, sizeof(char));
  for (i = 0; i < len; i++, arg3 = XEN_CDR(arg3))
    bits[i] = (char)XEN_TO_C_INT(XEN_CAR(arg3));
  p = XCreateBitmapFromData(XEN_TO_C_Display(arg1), 
			    XEN_TO_C_Window(arg2),
			    bits, 
			    XEN_TO_C_ULONG(arg4), 
			    XEN_TO_C_ULONG(arg5));
  FREE(bits);
  return(C_TO_XEN_Pixmap(p));
}

static XEN gxm_XCreatePixmap(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XCreatePixmap "Pixmap XCreatePixmap(display, d, width, height, depth)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreatePixmap", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCreatePixmap", "Drawable");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XCreatePixmap", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XCreatePixmap", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XCreatePixmap", "unsigned int");
  return(C_TO_XEN_Pixmap(XCreatePixmap(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
				       XEN_TO_C_ULONG(arg3), XEN_TO_C_ULONG(arg4), XEN_TO_C_ULONG(arg5))));
}

static XEN gxm_XFlushGC(XEN arg1, XEN arg2)
{
  #define H_XFlushGC "XFlushGC(dpy, gc) forces cached GC changes to X server"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFlushGC", "Display*");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XFlushGC", "GC");
  XFlushGC(XEN_TO_C_Display(arg1), XEN_TO_C_GC(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XCreateGC(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XCreateGC "GC XCreateGC(display, d, valuemask, values) creates a graphics context and returns a GC."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreateGC", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCreateGC", "Drawable");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XCreateGC", "ulong");
  XEN_ASSERT_TYPE(XEN_XGCValues_P(arg4), arg4, 4, "XCreateGC", "XGCValues*");
  return(C_TO_XEN_GC(XCreateGC(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_ULONG(arg3), XEN_TO_C_XGCValues(arg4))));
}

static XEN gxm_XLoadFont(XEN arg1, XEN arg2)
{
  #define H_XLoadFont "Font XLoadFont(display, name) loads the specified font and returns its associated font ID."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XLoadFont", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XLoadFont", "char*");
  return(C_TO_XEN_Font(XLoadFont(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2))));
}

static XEN gxm_XCreateFontCursor(XEN arg1, XEN arg2)
{
  #define H_XCreateFontCursor "Cursor XCreateFontCursor(display, shape)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreateFontCursor", "Display*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XCreateFontCursor", "unsigned int");
  return(C_TO_XEN_Cursor(XCreateFontCursor(XEN_TO_C_Display(arg1), XEN_TO_C_ULONG(arg2))));
}

static XEN gxm_XCreateGlyphCursor(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XCreateGlyphCursor "Cursor XCreateGlyphCursor(display, source_font, mask_font, source_char, mask_char, foreground_color, \
background_color) is similar to XCreatePixmapCursor except that the source and mask bitmaps are obtained from the specified font glyphs."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreateGlyphCursor", "Display*");
  XEN_ASSERT_TYPE(XEN_Font_P(arg2), arg2, 2, "XCreateGlyphCursor", "Font");
  XEN_ASSERT_TYPE(XEN_Font_P(arg3) || XEN_INTEGER_P(arg3) || XEN_FALSE_P(arg3), arg3, 3, "XCreateGlyphCursor", "Font");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XCreateGlyphCursor", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XCreateGlyphCursor", "unsigned int");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg6), arg6, 6, "XCreateGlyphCursor", "XColor");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg7), arg7, 7, "XCreateGlyphCursor", "XColor");
  return(C_TO_XEN_Cursor(XCreateGlyphCursor(XEN_TO_C_Display(arg1), XEN_TO_C_Font(arg2), 
					    (XEN_Font_P(arg3)) ? XEN_TO_C_Font(arg3) : None,
					    XEN_TO_C_ULONG(arg4),
					    XEN_TO_C_ULONG(arg5), 
					    XEN_TO_C_XColor(arg6), XEN_TO_C_XColor(arg7))));
}

static XEN gxm_XCreatePixmapCursor(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XCreatePixmapCursor "Cursor XCreatePixmapCursor(display, source, mask, foreground_color, background_color, x, y) creates \
a cursor and returns the cursor ID associated with it."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreatePixmapCursor", "Display*");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg2), arg2, 2, "XCreatePixmapCursor", "Pixmap");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg3) || XEN_INTEGER_P(arg3) || XEN_FALSE_P(arg3), arg3, 3, "XCreatePixmapCursor", "Pixmap");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg4), arg4, 4, "XCreatePixmapCursor", "XColor");
  XEN_ASSERT_TYPE(XEN_XColor_P(arg5), arg5, 5, "XCreatePixmapCursor", "XColor");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XCreatePixmapCursor", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XCreatePixmapCursor", "unsigned int");
  return(C_TO_XEN_Cursor(XCreatePixmapCursor(XEN_TO_C_Display(arg1), 
					     XEN_TO_C_Pixmap(arg2), 
					     (XEN_Pixmap_P(arg3)) ? XEN_TO_C_Pixmap(arg3) : None,
					     XEN_TO_C_XColor(arg4), 
					     XEN_TO_C_XColor(arg5), 
					     XEN_TO_C_ULONG(arg6), XEN_TO_C_ULONG(arg7))));
}

static XEN gxm_XCreateColormap(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XCreateColormap "Colormap XCreateColormap(display, w, visual, alloc) creates a colormap of the specified visual type for \
the screen on which the specified window resides and returns the colormap ID associated with it."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreateColormap", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XCreateColormap", "Window");
  XEN_ASSERT_TYPE(XEN_Visual_P(arg3), arg3, 3, "XCreateColormap", "Visual*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XCreateColormap", "int");
  return(C_TO_XEN_Colormap(XCreateColormap(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_Visual(arg3), XEN_TO_C_INT(arg4))));
}

static XEN gxm_XCopyColormapAndFree(XEN arg1, XEN arg2)
{
  #define H_XCopyColormapAndFree "Colormap XCopyColormapAndFree(display, colormap) creates a colormap of the same visual type and \
for the same screen as the specified colormap and returns the new colormap ID."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCopyColormapAndFree", "Display*");
  XEN_ASSERT_TYPE(XEN_Colormap_P(arg2), arg2, 2, "XCopyColormapAndFree", "Colormap");
  return(C_TO_XEN_Colormap(XCopyColormapAndFree(XEN_TO_C_Display(arg1), XEN_TO_C_Colormap(arg2))));
}

static XEN gxm_XInternAtom(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XInternAtom "Atom XInternAtom(display, atom_name, only_if_exists) returns the atom identifier associated with the specified atom_name string."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XInternAtom", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XInternAtom", "char*");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XInternAtom", "Bool");
  return(C_TO_XEN_Atom(XInternAtom(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2), XEN_TO_C_BOOLEAN(arg3))));
}

static XEN xm_AfterFunction = XEN_FALSE;
static int default_after_function(Display *dpy) {return(0);}
static int gxm_AfterFunction(Display *dpy)
{
  return(XEN_TO_C_INT(XEN_CALL_1(xm_AfterFunction, 
				 C_TO_XEN_Display(dpy),
				 c__FUNCTION__)));
}

static XEN gxm_XSetAfterFunction(XEN arg1, XEN arg2)
{
  #define H_XSetAfterFunction "XSetAfterFunction(dpy, proc) sets a function (one arg: dpy) to be called after every X function call"
  XEN old_func;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSetAfterFunction", "Display*");
  xm_protect(arg2);
  old_func = xm_AfterFunction;
  xm_AfterFunction = arg2;
  if (XEN_PROCEDURE_P(arg2))
    XSetAfterFunction(XEN_TO_C_Display(arg1), gxm_AfterFunction);
  else XSetAfterFunction(XEN_TO_C_Display(arg1), default_after_function);
  if (XEN_PROCEDURE_P(old_func)) xm_unprotect(old_func);
  return(old_func);
}

static XEN gxm_XSynchronize(XEN arg1, XEN arg2)
{
  #define H_XSynchronize "int (*XSynchronize(display, onoff))() turns on/off synchronous behavior."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XSynchronize", "Display*");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XSynchronize", "boolean");
  XSynchronize(XEN_TO_C_Display(arg1),
	       XEN_TO_C_BOOLEAN(arg2));
  return(xm_AfterFunction);
}

static XEN gxm_XKeysymToString(XEN arg1)
{
  #define H_XKeysymToString "char *XKeysymToString(keysym)"
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg1), arg1, 1, "XKeysymToString", "KeySym");
  return(C_TO_XEN_STRING(XKeysymToString(XEN_TO_C_KeySym(arg1))));
}

static XEN gxm_XDisplayName(XEN arg1)
{
  #define H_XDisplayName "char *XDisplayName(string) returns the name of the display that  XOpenDisplay would attempt to use."
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XDisplayName", "char*");
  return(C_TO_XEN_STRING(XDisplayName(XEN_TO_C_STRING(arg1))));
}

static XEN gxm_XGetAtomName(XEN arg1, XEN arg2)
{
  #define H_XGetAtomName "char *XGetAtomName(display, atom) returns the name associated with the specified atom."
  char *str;
  XEN res;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetAtomName", "Display*");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XGetAtomName", "Atom");
  str = XGetAtomName(XEN_TO_C_Display(arg1), XEN_TO_C_Atom(arg2));
  res = C_TO_XEN_STRING(str);
  XFree(str);
  return(res);
}

static XEN gxm_XFetchBuffer(XEN arg1, XEN arg2)
{
  #define H_XFetchBuffer "char *XFetchBuffer(display, buffer) returns #f if there \
is no data in the buffer or if an invalid buffer is specified, otherwise a string."
  int len = 0;
  char *buf;
  XEN lst = XEN_FALSE;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFetchBuffer", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XFetchBuffer", "int");
  buf = XFetchBuffer(XEN_TO_C_Display(arg1), &len, XEN_TO_C_INT(arg2));
  if (len > 0) 
    {
      lst = C_TO_XEN_STRING(buf);
      free(buf);
    }
  return(lst);
}

static XEN gxm_XFetchBytes(XEN arg1)
{
  #define H_XFetchBytes "char *XFetchBytes(display) returns the string in cut buffer 0"
  /* DIFF: XFetchBytes returns string, omits arg2
   */
  int len = 0;
  char *buf;
  XEN lst = XEN_FALSE;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XFetchBytes", "Display*");
  buf = XFetchBytes(XEN_TO_C_Display(arg1), &len);
  if (len > 0) 
    {
      lst = C_TO_XEN_STRING(buf);
      free(buf);
    }
  return(lst);
}

static XEN gxm_XOpenDisplay(XEN arg1)
{
  #define H_XOpenDisplay "Display *XOpenDisplay(display_name) returns a Display structure that serves as the connection to the X server \
and that contains all the information about that X server."
  Display *dpy;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1) || XEN_FALSE_P(arg1), arg1, 1, "XOpenDisplay", "char*");
  dpy = XOpenDisplay(XEN_FALSE_P(arg1) ? NULL : XEN_TO_C_STRING(arg1));
  if (dpy)
    return(C_TO_XEN_Display(dpy));
  return(XEN_FALSE);
}

static XEN gxm_XGetSubImage(XEN args)
{
  #define H_XGetSubImage "XImage *XGetSubImage(display, d, x, y, width, height, plane_mask, format, dest_image, dest_x, dest_y) updates \
dest_image with the specified subimage in the same manner as "
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  arg10 = XEN_LIST_REF(args, 9);
  arg11 = XEN_LIST_REF(args, 10);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetSubImage", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetSubImage", "Drawable");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XGetSubImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XGetSubImage", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XGetSubImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XGetSubImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XGetSubImage", "ulong");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XGetSubImage", "int");
  XEN_ASSERT_TYPE(XEN_XImage_P(arg9), arg9, 9, "XGetSubImage", "XImage*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg10), arg10, 10, "XGetSubImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg11), arg11, 11, "XGetSubImage", "int");
  return(C_TO_XEN_XImage(XGetSubImage(XEN_TO_C_Display(arg1),
				      XEN_TO_C_Window(arg2), 
				      XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), 
				      XEN_TO_C_ULONG(arg5), XEN_TO_C_ULONG(arg6), 
				      XEN_TO_C_ULONG(arg7), XEN_TO_C_INT(arg8), 
				      XEN_TO_C_XImage(arg9), 
				      XEN_TO_C_INT(arg10), XEN_TO_C_INT(arg11))));
}

static XEN gxm_XGetImage(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8)
{
  #define H_XGetImage "XImage *XGetImage(display, d, x, y, width, height, plane_mask, format) returns a pointer to an XImage structure."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetImage", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetImage", "Drawable");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XGetImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XGetImage", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XGetImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XGetImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XGetImage", "ulong");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg8), arg8, 8, "XGetImage", "int");
  return(C_TO_XEN_XImage(XGetImage(XEN_TO_C_Display(arg1), 
				   XEN_TO_C_Window(arg2), 
				   XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), 
				   XEN_TO_C_ULONG(arg5), XEN_TO_C_ULONG(arg6), 
				   XEN_TO_C_ULONG(arg7), XEN_TO_C_INT(arg8))));
}

static XEN gxm_XCreateImage(XEN args)
{
  #define H_XCreateImage "XImage *XCreateImage(display, visual, depth, format, offset, data, width, height, bitmap_pad, bytes_per_line) \
allocates the memory needed for an XImage structure for the specified display but does not allocate space for the image itself."
  XEN arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10;
  arg1 = XEN_LIST_REF(args, 0);
  arg2 = XEN_LIST_REF(args, 1);
  arg3 = XEN_LIST_REF(args, 2);
  arg4 = XEN_LIST_REF(args, 3);
  arg5 = XEN_LIST_REF(args, 4);
  arg6 = XEN_LIST_REF(args, 5);
  arg7 = XEN_LIST_REF(args, 6);
  arg8 = XEN_LIST_REF(args, 7);
  arg9 = XEN_LIST_REF(args, 8);
  arg10 = XEN_LIST_REF(args, 9);
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XCreateImage", "Display*");
  XEN_ASSERT_TYPE(XEN_Visual_P(arg2), arg2, 2, "XCreateImage", "Visual*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XCreateImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XCreateImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XCreateImage", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XCreateImage", "pointer");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg7), arg7, 7, "XCreateImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg8), arg8, 8, "XCreateImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg9), arg9, 9, "XCreateImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg10), arg10, 10, "XCreateImage", "int");
  return(C_TO_XEN_XImage(XCreateImage(XEN_TO_C_Display(arg1), 
				      XEN_TO_C_Visual(arg2), 
				      XEN_TO_C_ULONG(arg3), XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
				      (char *)XEN_TO_C_ULONG(arg6),
				      XEN_TO_C_ULONG(arg7), XEN_TO_C_ULONG(arg8),
				      XEN_TO_C_INT(arg9), XEN_TO_C_INT(arg10))));
}

static XEN gxm_XNewModifiermap(XEN arg1)
{
  #define H_XNewModifiermap "XModifierKeymap *XNewModifiermap(max_keys_per_mod) returns a pointer to XModifierKeymap structure for later use."
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 1, "XNewModifiermap", "int");
  return(C_TO_XEN_XModifierKeymap(XNewModifiermap(XEN_TO_C_INT(arg1))));
}

static XEN gxm_XInsertModifiermapEntry(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XInsertModifiermapEntry "XModifierKeymap *XInsertModifiermapEntry(modmap, keycode_entry, modifier) adds the specified KeyCode to \
the set that controls the specified modifier and returns the resulting XModifierKeymap structure (expanded as needed)."
  XEN_ASSERT_TYPE(XEN_XModifierKeymap_P(arg1), arg1, 1, "XInsertModifiermapEntry", "XModifierKeymap*");
  XEN_ASSERT_TYPE(XEN_KeyCode_P(arg2), arg2, 2, "XInsertModifiermapEntry", "KeyCode");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XInsertModifiermapEntry", "int");
  return(C_TO_XEN_XModifierKeymap(XInsertModifiermapEntry(XEN_TO_C_XModifierKeymap(arg1), XEN_TO_C_KeyCode(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XGetModifierMapping(XEN arg1)
{
  #define H_XGetModifierMapping "XModifierKeymap *XGetModifierMapping(display) returns a pointer to a newly created XModifierKeymap structure \
that contains the keys being used as modifiers."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetModifierMapping", "Display*");
  return(C_TO_XEN_XModifierKeymap(XGetModifierMapping(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XDeleteModifiermapEntry(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XDeleteModifiermapEntry "XModifierKeymap *XDeleteModifiermapEntry(modmap, keycode_entry, modifier) deletes the specified KeyCode \
from the set that controls the specified modifier and returns a pointer to the resulting XModifierKeymap structure."
  XEN_ASSERT_TYPE(XEN_XModifierKeymap_P(arg1), arg1, 1, "XDeleteModifiermapEntry", "XModifierKeymap*");
  XEN_ASSERT_TYPE(XEN_KeyCode_P(arg2), arg2, 2, "XDeleteModifiermapEntry", "KeyCode");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XDeleteModifiermapEntry", "int");
  return(C_TO_XEN_XModifierKeymap(XDeleteModifiermapEntry(XEN_TO_C_XModifierKeymap(arg1), XEN_TO_C_KeyCode(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XGetMotionEvents(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XGetMotionEvents "XTimeCoord *XGetMotionEvents(display, w, start, stop) returns all events in the motion history \
buffer that fall between the specified start and stop times, inclusive, and that have coordinates that lie within the specified window \
(including its borders) at its present placement."
  /* DIFF: XGetMotionEvents omits last arg, return time coords as list of lists
   */
  int n, i, loc;
  XTimeCoord *tcs;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XGetMotionEvents", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XGetMotionEvents", "Window");
  XEN_ASSERT_TYPE(XEN_Time_P(arg3), arg3, 3, "XGetMotionEvents", "Time");
  XEN_ASSERT_TYPE(XEN_Time_P(arg4), arg4, 4, "XGetMotionEvents", "Time");
  tcs = XGetMotionEvents(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), XEN_TO_C_Time(arg3), XEN_TO_C_Time(arg4), &n);
  loc = xm_protect(lst);
  for (i = n - 1; i > 0; i--)
    lst = XEN_CONS(XEN_LIST_3(C_TO_XEN_Time(tcs->time),
			      C_TO_XEN_INT((int)(tcs->x)),
			      C_TO_XEN_INT((int)(tcs->y))),
		   lst);
  XFree(tcs); /* free each as well? */
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XQueryFont(XEN arg1, XEN arg2)
{
  #define H_XQueryFont "XFontStruct *XQueryFont(display, font) returns a pointer to the XFontStruct structure, which contains information \
associated with the font."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XQueryFont", "Display*");
  XEN_ASSERT_TYPE(XEN_Font_P(arg2), arg2, 2, "XQueryFont", "Font");
  return(C_TO_XEN_XFontStruct(XQueryFont(XEN_TO_C_Display(arg1), XEN_TO_C_Font(arg2))));
}

static XEN gxm_XLoadQueryFont(XEN arg1, XEN arg2)
{
  #define H_XLoadQueryFont "XFontStruct *XLoadQueryFont(display, name) provides the most common way for accessing a font. XLoadQueryFont \
both opens (loads) the specified font and returns a pointer to the appropriate XFontStruct structure."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XLoadQueryFont", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XLoadQueryFont", "char*");
  return(C_TO_XEN_XFontStruct(XLoadQueryFont(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2))));
}

static XEN gxm_DefaultScreen(XEN arg)
{
  #define H_DefaultScreen "returns the default screen number referenced in the XOpenDisplay routine."
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "DefaultScreen", "Display*");
  return(C_TO_XEN_INT(DefaultScreen(XEN_TO_C_Display(arg))));
}

static XEN gxm_DefaultRootWindow(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "DefaultRootWindow", "Display*");
  return(C_TO_XEN_Window(DefaultRootWindow(XEN_TO_C_Display(arg))));
}

static XEN gxm_QLength(XEN arg)
{
  /* QLength(display) */
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "QLength", "Display*");
  return(C_TO_XEN_INT(QLength(XEN_TO_C_Display(arg))));
}

static XEN gxm_ScreenCount(XEN arg1)
{
  /* ScreenCount(display) */
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "ScreenCount", "Display*");
  return(C_TO_XEN_INT(XScreenCount(XEN_TO_C_Display(arg1))));
}

static XEN gxm_ServerVendor(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "ServerVendor", "Display*");
  return(C_TO_XEN_STRING(ServerVendor(XEN_TO_C_Display(arg))));
}

static XEN gxm_ProtocolVersion(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "ProtocolVersion", "Display*");
  return(C_TO_XEN_INT(ProtocolVersion(XEN_TO_C_Display(arg))));
}

static XEN gxm_ProtocolRevision(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "ProtocolRevision", "Display*");
  return(C_TO_XEN_INT(ProtocolRevision(XEN_TO_C_Display(arg))));
}

static XEN gxm_VendorRelease(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "VendorRelease", "Display*");
  return(C_TO_XEN_INT(VendorRelease(XEN_TO_C_Display(arg))));
}

static XEN gxm_DisplayString(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "DisplayString", "Display*");
  return(C_TO_XEN_STRING(DisplayString(XEN_TO_C_Display(arg))));
}

static XEN gxm_BitmapUnit(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "BitmapUnit", "Display*");
  return(C_TO_XEN_INT(BitmapUnit(XEN_TO_C_Display(arg))));
}

static XEN gxm_BitmapBitOrder(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "BitmapBitOrder", "Display*");
  return(C_TO_XEN_INT(BitmapBitOrder(XEN_TO_C_Display(arg))));
}

static XEN gxm_BitmapPad(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "BitmapPad", "Display*");
  return(C_TO_XEN_INT(BitmapPad(XEN_TO_C_Display(arg))));
}

static XEN gxm_ImageByteOrder(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "ImageByteOrder", "Display*");
  return(C_TO_XEN_INT(ImageByteOrder(XEN_TO_C_Display(arg))));
}

static XEN gxm_NextRequest(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "NextRequest", "Display*");
  return(C_TO_XEN_ULONG(NextRequest(XEN_TO_C_Display(arg))));
}

static XEN gxm_LastKnownRequestProcessed(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "LastKnownRequestProcessed", "Display*");
  return(C_TO_XEN_ULONG(LastKnownRequestProcessed(XEN_TO_C_Display(arg))));
}

static XEN gxm_DefaultScreenOfDisplay(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg), arg, 0, "DefaultScreenOfDisplay", "Display");
  return(C_TO_XEN_Screen(DefaultScreenOfDisplay(XEN_TO_C_Display(arg))));
}

static XEN gxm_DisplayOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "DisplayOfScreen", "Screen");
  return(C_TO_XEN_Display(DisplayOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_RootWindowOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "RootWindowOfScreen", "Screen");
  return(C_TO_XEN_Window(RootWindowOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_BlackPixelOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "BlackPixelOfScreen", "Screen");
  return(C_TO_XEN_Pixel(BlackPixelOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_WhitePixelOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "WhitePixelOfScreen", "Screen");
  return(C_TO_XEN_Pixel(WhitePixelOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_DefaultColormapOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "DefaultColormapOfScreen", "Screen");
  return(C_TO_XEN_Colormap(DefaultColormapOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_DefaultDepthOfScreen(XEN arg)
{
  /* DefaultDepthOfScreen(screen) */
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "DefaultDepthOfScreen", "Screen");
  return(C_TO_XEN_INT(DefaultDepthOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_DefaultGCOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "DefaultGCOfScreen", "Screen");
  return(C_TO_XEN_GC(DefaultGCOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_DefaultVisualOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "DefaultVisualOfScreen", "Screen");
  return(C_TO_XEN_Visual(DefaultVisualOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_WidthOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "WidthOfScreen", "Screen");
  return(C_TO_XEN_INT(WidthOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_HeightOfScreen(XEN arg)
{
  /* HeightOfScreen(screen) */
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "HeightOfScreen", "Screen");
  return(C_TO_XEN_INT(HeightOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_WidthMMOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "WidthMMOfScreen", "Screen");
  return(C_TO_XEN_INT(WidthMMOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_HeightMMOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "HeightMMOfScreen", "Screen");
  return(C_TO_XEN_INT(HeightMMOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_PlanesOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "PlanesOfScreen", "Screen");
  return(C_TO_XEN_INT(PlanesOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_CellsOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "CellsOfScreen", "Screen");
  return(C_TO_XEN_INT(CellsOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_MinCmapsOfScreen(XEN arg)
{
  /* MinCmapsOfScreen(screen) */
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "MinCmapsOfScreen", "Screen");
  return(C_TO_XEN_INT(MinCmapsOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_MaxCmapsOfScreen(XEN arg)
{
  /* MaxCmapsOfScreen(screen) */
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "MaxCmapsOfScreen", "Screen");
  return(C_TO_XEN_INT(MaxCmapsOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_DoesSaveUnders(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "DoesSaveUnders", "Screen");
  return(C_TO_XEN_BOOLEAN(DoesSaveUnders(XEN_TO_C_Screen(arg))));
}

static XEN gxm_DoesBackingStore(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "DoesBackingStore", "Screen");
  return(C_TO_XEN_BOOLEAN(DoesBackingStore(XEN_TO_C_Screen(arg))));
}

static XEN gxm_EventMaskOfScreen(XEN arg)
{
  XEN_ASSERT_TYPE(XEN_Screen_P(arg), arg, 0, "EventMaskOfScreen", "Screen");
  return(C_TO_XEN_ULONG(EventMaskOfScreen(XEN_TO_C_Screen(arg))));
}

static XEN gxm_RootWindow(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "RootWindow", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "RootWindow", "int");
  return(C_TO_XEN_Window(RootWindow(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_DefaultVisual(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "DefaultVisual", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "DefaultVisual", "int");
  return(C_TO_XEN_Visual(DefaultVisual(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_DefaultGC(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "DefaultGC", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "DefaultGC", "int");
  return(C_TO_XEN_GC(DefaultGC(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_BlackPixel(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "BlackPixel", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "BlackPixel", "int");
  return(C_TO_XEN_Pixel(BlackPixel(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_WhitePixel(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "WhitePixel", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "WhitePixel", "int");
  return(C_TO_XEN_Pixel(WhitePixel(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_DisplayWidth(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "DisplayWidth", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "DisplayWidth", "int");
  return(C_TO_XEN_INT(DisplayWidth(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_DisplayHeight(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "DisplayHeight", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "DisplayHeight", "int");
  return(C_TO_XEN_INT(DisplayHeight(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_DisplayWidthMM(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "DisplayWidthMM", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "DisplayWidthMM", "int");
  return(C_TO_XEN_INT(DisplayWidthMM(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_DisplayHeightMM(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "DisplayHeightMM", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "DisplayHeightMM", "int");
  return(C_TO_XEN_INT(DisplayHeightMM(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_DisplayPlanes(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "DisplayPlanes", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "DisplayPlanes", "int");
  return(C_TO_XEN_INT(DisplayPlanes(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_DisplayCells(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "DisplayCells", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "DisplayCells", "int");
  return(C_TO_XEN_INT(DisplayCells(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_DefaultColormap(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "DefaultColormap", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "DefaultColormap", "int");
  return(C_TO_XEN_Colormap(DefaultColormap(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_ScreenOfDisplay(XEN arg1, XEN arg2)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "ScreenOfDisplay", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "ScreenOfDisplay", "int");
  return(C_TO_XEN_Screen(ScreenOfDisplay(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_DefaultDepth(XEN arg1, XEN arg2)
{
  /* DefaultDepth(display, screen_number) */
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "DefaultDepth", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "DefaultDepth", "int");
  return(C_TO_XEN_INT(DefaultDepth(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_IsKeypadKey(XEN arg)
{
  #define H_IsKeypadKey "IsKeypadKey(keysym) returns #t if the specified KeySym is a keypad key."
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg), arg, 0, "IsKeypadKey", "KeySym");
  return(C_TO_XEN_BOOLEAN(IsKeypadKey(XEN_TO_C_KeySym(arg))));
}

static XEN gxm_IsPrivateKeypadKey(XEN arg)
{
  #define H_IsPrivateKeypadKey "IsPrivateKeypadKey(keysym) returns #t if the specified KeySym is a vendor-private keypad key."
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg), arg, 0, "IsPrivateKeypadKey", "KeySym");
  return(C_TO_XEN_BOOLEAN(IsPrivateKeypadKey(XEN_TO_C_KeySym(arg))));
}

static XEN gxm_IsCursorKey(XEN arg)
{
  #define H_IsCursorKey "IsCursorKey(keysym) returns #t if the specified KeySym is a cursor key."
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg), arg, 0, "IsCursorKey", "KeySym");
  return(C_TO_XEN_BOOLEAN(IsCursorKey(XEN_TO_C_KeySym(arg))));
}

static XEN gxm_IsPFKey(XEN arg)
{
  #define H_IsPFKey "IsPFKey(keysym) returns #t if the specified KeySym is a PF key."
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg), arg, 0, "IsPFKey", "KeySym");
  return(C_TO_XEN_BOOLEAN(IsPFKey(XEN_TO_C_KeySym(arg))));
}

static XEN gxm_IsFunctionKey(XEN arg)
{
  #define H_IsFunctionKey "IsFunctionKey(keysym) returns #t if the KeySym is a function key."
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg), arg, 0, "IsFunctionKey", "KeySym");
  return(C_TO_XEN_BOOLEAN(IsFunctionKey(XEN_TO_C_KeySym(arg))));
}

static XEN gxm_IsMiscFunctionKey(XEN arg)
{
  #define H_IsMiscFunctionKey "IsMiscFunctionKey(keysym) returns #t if the specified KeySym is a miscellaneous function key."
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg), arg, 0, "IsMiscFunctionKey", "KeySym");
  return(C_TO_XEN_BOOLEAN(IsMiscFunctionKey(XEN_TO_C_KeySym(arg))));
}

static XEN gxm_IsModifierKey(XEN arg)
{
  #define H_IsModifierKey "IsModifierKey(keysym) returns #t if the specified KeySym is a modifier key."
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg), arg, 0, "IsModifierKey", "KeySym");
  return(C_TO_XEN_BOOLEAN(IsModifierKey(XEN_TO_C_KeySym(arg))));
}

static XEN gxm_XAddPixel(XEN arg1, XEN arg2)
{
  #define H_XAddPixel "XAddPixel(ximage, value) adds a constant value to every pixel in an image."
  XEN_ASSERT_TYPE(XEN_XImage_P(arg1), arg1, 1, "XAddPixel", "XImage*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XAddPixel", "long");
  return(C_TO_XEN_INT(XAddPixel(XEN_TO_C_XImage(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XSubImage(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XSubImage "XImage *XSubImage(ximage, x, y, subimage_width, subimage_height) creates a new image that is a subsection of an existing one."
  XEN_ASSERT_TYPE(XEN_XImage_P(arg1), arg1, 1, "XSubImage", "XImage*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XSubImage", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XSubImage", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg4), arg4, 4, "XSubImage", "unsigned int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XSubImage", "unsigned int");
  return(C_TO_XEN_XImage(XSubImage(XEN_TO_C_XImage(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_ULONG(arg4), XEN_TO_C_ULONG(arg5))));
}

static XEN gxm_XPutPixel(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XPutPixel "XPutPixel(ximage, x, y, pixel) overwrites the pixel in the named image with the specified pixel value."
  XEN_ASSERT_TYPE(XEN_XImage_P(arg1), arg1, 1, "XPutPixel", "XImage*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XPutPixel", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XPutPixel", "int");
  XEN_ASSERT_TYPE(XEN_Pixel_P(arg4), arg4, 4, "XPutPixel", "Pixel");
  return(C_TO_XEN_INT(XPutPixel(XEN_TO_C_XImage(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_Pixel(arg4))));
}

static XEN gxm_XGetPixel(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XGetPixel "Pixel XGetPixel(ximage, x, y) returns the specified pixel from the named image."
  XEN_ASSERT_TYPE(XEN_XImage_P(arg1), arg1, 1, "XGetPixel", "XImage*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XGetPixel", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XGetPixel", "int");
  return(C_TO_XEN_Pixel(XGetPixel(XEN_TO_C_XImage(arg1), XEN_TO_C_INT(arg2), XEN_TO_C_INT(arg3))));
}

static XEN gxm_XDestroyImage(XEN arg1)
{
  #define H_XDestroyImage "XDestroyImage(ximage) deallocates the memory associated with the XImage structure."
  XEN_ASSERT_TYPE(XEN_XImage_P(arg1), arg1, 1, "XDestroyImage", "XImage*");
  return(C_TO_XEN_INT(XDestroyImage(XEN_TO_C_XImage(arg1))));
}



/* ----------------------------------------------------------------------------------------------------
 *
 *                            *         *  ***********  *         *  ***********
 *                             *       *   *             *       *        *
 *                              *     *    *              *     *         *
 *                               *   *     *               *   *          *
 *                                * *      *                * *           *
 *                                 *       *******           *            *
 *                                * *      *                * *           *
 *                               *   *     *               *   *          *
 *                              *     *    *              *     *         *
 *                             *       *   *             *       *        *
 *                            *         *  ***********  *         *       *
 * ----------------------------------------------------------------------------------------------------
 */

#if HAVE_XSHAPEQUERYEXTENSION

/* thanks to Michael Scholz! */

static XEN gxm_XShapeQueryExtension(XEN dpy)
{
  #define H_XShapeQueryExtension "(XShapeQueryExtension dpy) returns list of (Bool event_base error_base)"
  /* DIFF: (proposal)
     [C] Bool XShapeQueryExtension(Display *dpy, int *event_base, int *error_base)
     [XEN] (XShapeQueryExtension dpy) -> (list Bool event_base error_base)
  */
  int event_base = 0, error_base = 0;
  Bool ret = False;
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XShapeQueryExtension", "Display*");
  ret = XShapeQueryExtension(XEN_TO_C_Display(dpy), &event_base, &error_base);
  return(XEN_LIST_3(C_TO_XEN_BOOLEAN(ret), C_TO_XEN_INT(event_base), C_TO_XEN_INT(error_base)));
}

static XEN gxm_XShapeQueryVersion(XEN dpy)
{
  #define H_XShapeQueryVersion "(XShapeQueryVersion dpy) returns list of (Bool major_version minor_version)"
  int major = 0, minor = 0;
  Bool ret = False;
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XShapeQueryVersion", "Display*");
  ret = XShapeQueryVersion(XEN_TO_C_Display(dpy), &major, &minor);
  return(XEN_LIST_3(C_TO_XEN_BOOLEAN(ret), C_TO_XEN_INT(major), C_TO_XEN_INT(minor)));
}

static XEN gxm_XShapeQueryExtents(XEN dpy, XEN win)
{
  #define H_XShapeQueryExtents "(XShapeQueryExtents dpy win) returns list of (status bounding_shaped x_bound y_bound w_bound \
h_bound clip_shaped x_clip y_clip w_clip h_clip"
  Bool bounding_shaped;
  int x_bounding;
  int y_bounding;
  unsigned int w_bounding;
  unsigned int h_bounding;
  Bool clip_shaped;
  int x_clip;
  int y_clip;
  unsigned int w_clip;
  unsigned int h_clip;
  Status ret;
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XShapeQueryExtents", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(win), win, XEN_ARG_2, "XShapeQueryExtents", "Window");
  ret = XShapeQueryExtents(XEN_TO_C_Display(dpy), XEN_TO_C_Window(win),
			   &bounding_shaped, &x_bounding, &y_bounding, &w_bounding, &h_bounding,
			   &clip_shaped, &x_clip, &y_clip, &w_clip, &h_clip);
  return(XEN_CONS(C_TO_XEN_INT(ret),
		  XEN_CONS(C_TO_XEN_INT(bounding_shaped),
			   XEN_LIST_9(C_TO_XEN_INT(x_bounding),
				      C_TO_XEN_INT(y_bounding),
				      C_TO_XEN_INT(w_bounding),
				      C_TO_XEN_INT(h_bounding),
				      C_TO_XEN_INT(clip_shaped),
				      C_TO_XEN_INT(x_clip),
				      C_TO_XEN_INT(y_clip),
				      C_TO_XEN_INT(w_clip),
				      C_TO_XEN_INT(h_clip)))));
}

static XEN gxm_XShapeGetRectangles(XEN dpy, XEN win, XEN kind)
{
  #define H_XShapeGetRectangles "(XShapeGetRectangles dpy win kind) returns list of (xrectangles ordering)"
  int count = 0, ordering = 0;
  XRectangle *res;
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XShapeGetRectangles", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(win), win, XEN_ARG_2, "XShapeGetRectangles", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(kind), kind, XEN_ARG_3, "XShapeGetRectangles", "int");
  res = XShapeGetRectangles(XEN_TO_C_Display(dpy), XEN_TO_C_Window(win), XEN_TO_C_INT(kind), &count, &ordering);
  return(XEN_LIST_2(C_TO_XEN_XRectangles(res, count), C_TO_XEN_INT(ordering)));
}

static XEN gxm_XShapeOffsetShape(XEN dpy, XEN win, XEN kind, XEN x, XEN y)
{
  #define H_XShapeOffsetShape "(XShapeOffsetShape dpy win kind x-off y-off)"
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XShapeOffsetShape", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(win), win, XEN_ARG_2, "XShapeOffsetShape", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(kind), kind, XEN_ARG_3, "XShapeOffsetShape", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(x), x, XEN_ARG_4, "XShapeOffsetShape", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(y), y, XEN_ARG_5, "XShapeOffsetShape", "int");
  XShapeOffsetShape(XEN_TO_C_Display(dpy), XEN_TO_C_Window(win), 
		    XEN_TO_C_INT(kind), XEN_TO_C_INT(x), XEN_TO_C_INT(y));
  return(XEN_FALSE);
}

static XEN gxm_XShapeCombineRegion(XEN dpy, XEN win, XEN kind, XEN x, XEN y, XEN reg, XEN op)
{
  #define H_XShapeCombineRegion "(XShapeCombineRegion dpy win kind x-off y-off region op)"
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XShapeCombineRegion", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(win), win, XEN_ARG_2, "XShapeCombineRegion", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(kind), kind, XEN_ARG_3, "XShapeCombineRegion", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(x), x, XEN_ARG_4, "XShapeCombineRegion", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(y), y, XEN_ARG_5, "XShapeCombineRegion", "int");
  XEN_ASSERT_TYPE(XEN_Region_P(reg), reg, XEN_ARG_6, "XShapeCombineRegion", "Region");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(op), op, XEN_ARG_6, "XShapeCombineRegion", "int");
  XShapeCombineRegion(XEN_TO_C_Display(dpy), XEN_TO_C_Window(win), 
		      XEN_TO_C_INT(kind), XEN_TO_C_INT(x), XEN_TO_C_INT(y),
		      XEN_TO_C_Region(reg),
		      XEN_TO_C_INT(op));
  return(XEN_FALSE);
}

static XEN gxm_XShapeCombineMask(XEN dpy, XEN win, XEN kind, XEN x, XEN y, XEN pix, XEN op)
{
  #define H_XShapeCombineMask "(XShapeCombineMask dpy win kind x-off y-off pixmap op)"
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XShapeCombineMask", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(win), win, XEN_ARG_2, "XShapeCombineMask", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(kind), kind, XEN_ARG_3, "XShapeCombineMask", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(x), x, XEN_ARG_4, "XShapeCombineMask", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(y), y, XEN_ARG_5, "XShapeCombineMask", "int");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(pix), pix, XEN_ARG_6, "XShapeCombineMask", "Pixmap");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(op), op, XEN_ARG_6, "XShapeCombineMask", "int");
  XShapeCombineMask(XEN_TO_C_Display(dpy), XEN_TO_C_Window(win), 
		    XEN_TO_C_INT(kind), XEN_TO_C_INT(x), XEN_TO_C_INT(y),
		    XEN_TO_C_Pixmap(pix),
		    XEN_TO_C_INT(op));
  return(XEN_FALSE);
}

static XEN gxm_XShapeCombineShape(XEN dpy, XEN win, XEN kind, XEN x, XEN y, XEN src, XEN src_kind, XEN op)
{
  #define H_XShapeCombineShape "(XShapeCombineShape dpy win kind x-off y-off src src_kind op)"
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XShapeCombineShape", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(win), win, XEN_ARG_2, "XShapeCombineShape", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(kind), kind, XEN_ARG_3, "XShapeCombineShape", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(x), x, XEN_ARG_4, "XShapeCombineShape", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(y), y, XEN_ARG_5, "XShapeCombineShape", "int");
  XEN_ASSERT_TYPE(XEN_Window_P(src), src, XEN_ARG_6, "XShapeCombineShape", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(src_kind), src_kind, XEN_ARG_7, "XShapeCombineShape", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(op), op, XEN_ARG_8, "XShapeCombineShape", "int");
  XShapeCombineShape(XEN_TO_C_Display(dpy), XEN_TO_C_Window(win), 
		     XEN_TO_C_INT(kind), XEN_TO_C_INT(x), XEN_TO_C_INT(y),
		     XEN_TO_C_Window(src), XEN_TO_C_INT(src_kind),
		     XEN_TO_C_INT(op));
  return(XEN_FALSE);
}

static XEN gxm_XShapeCombineRectangles(XEN dpy, XEN win, XEN kind, XEN x, XEN y, XEN rects, XEN n_rects, XEN op, XEN ordering)
{
  #define H_XShapeCombineRectangles "(XShapeCombineRectangles dpy win kind x-off y-off rectangles n-rects op ordering)"
  XRectangle *cr = NULL;
  XEN_ASSERT_TYPE(XEN_Display_P(dpy), dpy, XEN_ARG_1, "XShapeCombineRectangles", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(win), win, XEN_ARG_2, "XShapeCombineRectangles", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(kind), kind, XEN_ARG_3, "XShapeCombineRectangles", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(x), x, XEN_ARG_4, "XShapeCombineRectangles", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(y), y, XEN_ARG_5, "XShapeCombineRectangles", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(rects), rects, XEN_ARG_6, "XShapeCombineRectangles", "list of XRectangles");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(n_rects), n_rects, XEN_ARG_7, "XShapeCombineRectangles", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(op), op, XEN_ARG_8, "XShapeCombineRectangles", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(ordering), ordering, XEN_ARG_9, "XShapeCombineRectangles", "int");
  cr = XEN_TO_C_XRectangles(rects, XEN_TO_C_INT(n_rects));
  XShapeCombineRectangles(XEN_TO_C_Display(dpy), XEN_TO_C_Window(win), 
			  XEN_TO_C_INT(kind), XEN_TO_C_INT(x), XEN_TO_C_INT(y),
			  cr, XEN_TO_C_INT(n_rects),
			  XEN_TO_C_INT(op), XEN_TO_C_INT(ordering));
  if (cr) FREE(cr);
  return(XEN_FALSE);
}
#endif



/* ----------------------------------------------------------------------------------------------------
 *
 *                            *         *  ***********
 *                             *       *        *
 *                              *     *         *
 *                               *   *          *
 *                                * *           *
 *                                 *            *
 *                                * *           *
 *                               *   *          *
 *                              *     *         *
 *                             *       *        *
 *                            *         *       *
 * ----------------------------------------------------------------------------------------------------
 */

#if HAVE_MOTIF

typedef enum {CANCEL_CONVERT, CONVERT, LOSE, DONE, CONVERT_INCR, LOSE_INCR, DONE_INCR} xm_selmap_t;
/* need a way to map from widget to selection proc */

typedef struct {
  Widget w;
  xm_selmap_t type;
  XEN proc;
} selmap;

static selmap *selmaps = NULL;
static int selmap_size = 0;
static int selmap_ctr = 0;
static void add_selmap(Widget w, xm_selmap_t type, XEN proc)
{
  if (selmap_size == 0)
    selmaps = (selmap *)CALLOC(8, sizeof(selmap));
  else
    {
      if (selmap_size == selmap_ctr)
	{
	  int i;
	  selmap_size += 8;
	  selmaps = (selmap *)REALLOC(selmaps, selmap_size * sizeof(selmap));
	  for (i = selmap_ctr; i < selmap_size; i++) selmaps[i].w = NULL;
	}
    }
  selmaps[selmap_ctr].w = w;
  selmaps[selmap_ctr].type = type;
  xm_protect(proc);
  selmaps[selmap_ctr++].proc = proc;
}

static XEN unselmap(Widget w, xm_selmap_t type)
{
  int i;
  for (i = 0; i < selmap_ctr; i++)
    if ((selmaps[i].w == w) &&
	(selmaps[i].type == type))
      return(selmaps[i].proc);
  return(XEN_FALSE);
}

static void gxm_XtCancelConvertSelectionProc(Widget w, Atom *a1, Atom *a2, XtRequestId *id, XtPointer x) 
{
  XEN proc;
  proc = unselmap(w, CANCEL_CONVERT);
  if (XEN_PROCEDURE_P(proc))
    XEN_CALL_5(proc,
	       C_TO_XEN_Widget(w),
	       C_TO_XEN_Atom(*a1),
	       C_TO_XEN_Atom(*a2),
	       C_TO_XEN_XtRequestId(*id),
	       (XEN)x,
	       "CancelConvert");
} 

static Boolean gxm_XtConvertSelectionProc(Widget w, Atom *a1, Atom *a2, Atom *a3, XtPointer* x, unsigned long *l, int *i) 
{
  XEN proc, val;
  proc = unselmap(w, CONVERT);
  if (XEN_PROCEDURE_P(proc))
    {
      val = XEN_CALL_3(proc,
		       C_TO_XEN_Widget(w),
		       C_TO_XEN_Atom(*a1),
		       C_TO_XEN_Atom(*a2),
		       "ConvertSelection");
      (*a3) = XEN_TO_C_Atom(XEN_LIST_REF(val, 1));
      (*x) = (XtPointer)(XEN_LIST_REF(val, 2));
      (*l) = XEN_TO_C_ULONG(XEN_LIST_REF(val, 3));
      (*i) = XEN_TO_C_INT(XEN_LIST_REF(val, 4));
      return(XEN_TO_C_BOOLEAN(XEN_CAR(val)));
    }
  return(0);
}

static void gxm_XtLoseSelectionIncrProc(Widget w, Atom *a, XtPointer x) 
{
  XEN proc;
  proc = unselmap(w, LOSE_INCR);
  if (XEN_PROCEDURE_P(proc))
    XEN_CALL_3(proc,
	       C_TO_XEN_Widget(w),
	       C_TO_XEN_Atom(*a),
	       (XEN)x,
	       "LoseSelectionIncr");
}

static void gxm_XtLoseSelectionProc(Widget w, Atom *a) 
{
  XEN proc;
  proc = unselmap(w, LOSE);
  if (XEN_PROCEDURE_P(proc))
    XEN_CALL_2(proc,
	       C_TO_XEN_Widget(w),
	       C_TO_XEN_Atom(*a),
	       "LoseSelection");
}

static void gxm_XtSelectionDoneProc(Widget w, Atom *a1, Atom *a2) 
{
  XEN proc;
  proc = unselmap(w, DONE);
  if (XEN_PROCEDURE_P(proc))
    XEN_CALL_3(proc,
	       C_TO_XEN_Widget(w),
	       C_TO_XEN_Atom(*a1),
	       C_TO_XEN_Atom(*a2),
	       "DoneSelection");
}

static void gxm_XtSelectionDoneIncrProc(Widget w, Atom *a1, Atom *a2, XtRequestId *i, XtPointer x) 
{
  XEN proc;
  proc = unselmap(w, DONE_INCR);
  if (XEN_PROCEDURE_P(proc))
    XEN_CALL_5(proc,
	       C_TO_XEN_Widget(w),
	       C_TO_XEN_Atom(*a1),
	       C_TO_XEN_Atom(*a2),
	       C_TO_XEN_XtRequestId(*i),
	       (XEN)x,
	       "DoneSelectionIncr");
}

static XEN gxm_XtAppUnlock(XEN arg1)
{
  #define H_XtAppUnlock "void XtAppUnlock(app_context) unlocks the application context."
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppUnlock", "XtAppContext");
  XtAppUnlock(XEN_TO_C_XtAppContext(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtAppLock(XEN arg1)
{
  #define H_XtAppLock "void XtAppLock(app_context) locks the application context including all its related displays and widgets."
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppLock", "XtAppContext");
  XtAppLock(XEN_TO_C_XtAppContext(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtToolkitThreadInitialize(void)
{
  #define H_XtToolkitThreadInitialize "Boolean XtToolkitThreadInitialize()"
  return(C_TO_XEN_BOOLEAN(XtToolkitThreadInitialize()));
}

static XEN gxm_XtGetDisplays(XEN arg1)
{
  #define H_XtGetDisplays "void XtGetDisplays(app_context) -> list of displays"
  /* DIFF: XtGetDisplays not arg2 arg3 returns list
   */
  unsigned int x;
  Display **ds;
  int i, loc;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtGetDisplays", "XtAppContext");
  XtGetDisplays(XEN_TO_C_XtAppContext(arg1), &ds, &x);
  loc = xm_protect(lst);
  for (i = x - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_Display(ds[i]), lst);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XtGetApplicationNameAndClass(XEN arg1)
{
  #define H_XtGetApplicationNameAndClass "void XtGetApplicationNameAndClass(display) returns the application name \
and class passed to XtDisplayInitialize for the specified display."
  /* DIFF: XtGetApplicationNameAndClass omits and rtns args 2 and 3
   */
  char *name, *clas;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtGetApplicationNameAndClass", "Display*");
  XtGetApplicationNameAndClass(XEN_TO_C_Display(arg1), &name, &clas);
  return(XEN_LIST_2(C_TO_XEN_STRING(name),
		    C_TO_XEN_STRING(clas)));
}

static XEN gxm_XtUngrabPointer(XEN arg1, XEN arg2)
{
  #define H_XtUngrabPointer "void XtUngrabPointer(widget, time) calls XUngrabPointer with the specified time."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtUngrabPointer", "Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XtUngrabPointer", "Time");
  XtUngrabPointer(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtGrabPointer(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8)
{
  #define H_XtGrabPointer "int XtGrabPointer(widget, owner_events, event_mask, pointer_mode, keyboard_mode, confine_to, cursor, time) calls \
XGrabPointer specifying the widget's window as the grab window."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGrabPointer", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XtGrabPointer", "boolean");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XtGrabPointer", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XtGrabPointer", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XtGrabPointer", "int");
  XEN_ASSERT_TYPE(XEN_Window_P(arg6), arg6, 6, "XtGrabPointer", "Window");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg7), arg7, 7, "XtGrabPointer", "Cursor");
  XEN_ASSERT_TYPE(XEN_Time_P(arg8), arg8, 8, "XtGrabPointer", "Time");
  return(C_TO_XEN_INT(XtGrabPointer(XEN_TO_C_Widget(arg1), 
				    XEN_TO_C_BOOLEAN(arg2), XEN_TO_C_ULONG(arg3), 
				    XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5), 
				    XEN_TO_C_Window(arg6), XEN_TO_C_Cursor(arg7), 
				    XEN_TO_C_Time(arg8))));
}

static XEN gxm_XtUngrabButton(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtUngrabButton "void XtUngrabButton(widget, button, modifiers) calls XUngrabButton specifying the widget's window as the ungrab \
window if the widget is realized. "
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtUngrabButton", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtUngrabButton", "unsigned int");
  XEN_ASSERT_TYPE(XEN_Modifiers_P(arg3), arg3, 3, "XtUngrabButton", "Modifiers");
  XtUngrabButton(XEN_TO_C_Widget(arg1), XEN_TO_C_ULONG(arg2), XEN_TO_C_Modifiers(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XtGrabButton(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8, XEN arg9)
{
  #define H_XtGrabButton "void XtGrabButton(widget, button, modifiers, owner_events, event_mask, pointer_mode, keyboard_mode, confine_to, cursor) \
calls XGrabButton specifying the widget's window as the grab window if the widget is realized."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGrabButton", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XtGrabButton", "int");
  XEN_ASSERT_TYPE(XEN_Modifiers_P(arg3), arg3, 3, "XtGrabButton", "Modifiers");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg4), arg4, 4, "XtGrabButton", "boolean");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XtGrabButton", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XtGrabButton", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XtGrabButton", "int");
  XEN_ASSERT_TYPE(XEN_Window_P(arg8), arg8, 8, "XtGrabButton", "Window");
  XEN_ASSERT_TYPE(XEN_Cursor_P(arg9), arg9, 9, "XtGrabButton", "Cursor");
  XtGrabButton(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), 
	       XEN_TO_C_Modifiers(arg3), XEN_TO_C_BOOLEAN(arg4), XEN_TO_C_ULONG(arg5), XEN_TO_C_INT(arg6), XEN_TO_C_INT(arg7), 
	       XEN_TO_C_Window(arg8), XEN_TO_C_Cursor(arg9));
  return(XEN_FALSE);
}

static XEN gxm_XtUngrabKeyboard(XEN arg1, XEN arg2)
{
  #define H_XtUngrabKeyboard "void XtUngrabKeyboard(widget, time) calls XUngrabKeyboard with the specified time."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtUngrabKeyboard", "Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XtUngrabKeyboard", "Time");
  XtUngrabKeyboard(XEN_TO_C_Widget(arg1), XEN_TO_C_Time(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtGrabKeyboard(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtGrabKeyboard "int XtGrabKeyboard(widget, owner_events, pointer_mode, keyboard_mode, time)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGrabKeyboard", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XtGrabKeyboard", "boolean");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XtGrabKeyboard", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XtGrabKeyboard", "int");
  XEN_ASSERT_TYPE(XEN_Time_P(arg5), arg5, 5, "XtGrabKeyboard", "Time");
  return(C_TO_XEN_INT(XtGrabKeyboard(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2), XEN_TO_C_INT(arg3), XEN_TO_C_INT(arg4), XEN_TO_C_Time(arg5))));
}

static XEN gxm_XtUngrabKey(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtUngrabKey "void XtUngrabKey(widget, keycode, modifiers) calls XUngrabKey specifying the widget's window as the ungrab window \
if the widget is realized."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtUngrabKey", "Widget");
  XEN_ASSERT_TYPE(XEN_KeyCode_P(arg2), arg2, 2, "XtUngrabKey", "KeyCode");
  XEN_ASSERT_TYPE(XEN_Modifiers_P(arg3), arg3, 3, "XtUngrabKey", "Modifiers");
  XtUngrabKey(XEN_TO_C_Widget(arg1), XEN_TO_C_KeyCode(arg2), XEN_TO_C_Modifiers(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XtGrabKey(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XtGrabKey "void XtGrabKey(widget, keycode, modifiers, owner_events, pointer_mode, keyboard_mode) calls XGrabKey specifying the \
widget's window as the grab window if the widget is realized."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGrabKey", "Widget");
  XEN_ASSERT_TYPE(XEN_KeyCode_P(arg2), arg2, 2, "XtGrabKey", "KeyCode");
  XEN_ASSERT_TYPE(XEN_Modifiers_P(arg3), arg3, 3, "XtGrabKey", "Modifiers");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg4), arg4, 4, "XtGrabKey", "boolean");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XtGrabKey", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XtGrabKey", "int");
  XtGrabKey(XEN_TO_C_Widget(arg1), XEN_TO_C_KeyCode(arg2), XEN_TO_C_Modifiers(arg3), XEN_TO_C_BOOLEAN(arg4), XEN_TO_C_INT(arg5), XEN_TO_C_INT(arg6));
  return(XEN_FALSE);
}

static XEN gxm_XtCancelSelectionRequest(XEN arg1, XEN arg2)
{
  #define H_XtCancelSelectionRequest "void XtCancelSelectionRequest(requestor, selection)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtCancelSelectionRequest", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtCancelSelectionRequest", "Atom");
  XtCancelSelectionRequest(XEN_TO_C_Widget(arg1), XEN_TO_C_Atom(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtSendSelectionRequest(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtSendSelectionRequest "void XtSendSelectionRequest(requestor, selection, time)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtSendSelectionRequest", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtSendSelectionRequest", "Atom");
  XEN_ASSERT_TYPE(XEN_Time_P(arg3), arg3, 3, "XtSendSelectionRequest", "Time");
  XtSendSelectionRequest(XEN_TO_C_Widget(arg1), XEN_TO_C_Atom(arg2), XEN_TO_C_Time(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XtCreateSelectionRequest(XEN arg1, XEN arg2)
{
  #define H_XtCreateSelectionRequest "void XtCreateSelectionRequest(requestor, selection)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtCreateSelectionRequest", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtCreateSelectionRequest", "Atom");
  XtCreateSelectionRequest(XEN_TO_C_Widget(arg1), XEN_TO_C_Atom(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtGetSelectionValuesIncremental(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XtGetSelectionValuesIncremental "void XtGetSelectionValuesIncremental(w, selection, targets, count, callback, client_data, time) \
is similar to XtGetSelectionValueIncremental except that it takes a list of target types and a list of client data and obtains the current \
value of the selection converted to each of the targets."
  /* DIFF: XtGetSelectionValuesIncremental arg 3 is list of Atoms
   */
  Atom *outs;
  int len, loc;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGetSelectionValuesIncremental", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtGetSelectionValuesIncremental", "Atom");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XtGetSelectionValuesIncremental", "list of Atom");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XtGetSelectionValuesIncremental", "int");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg5) && (XEN_REQUIRED_ARGS_OK(arg5, 7)), arg5, 5, "XtGetSelectionValuesIncremental", "XtSelectionCallbackProc");
  XEN_ASSERT_TYPE(XEN_Time_P(arg7), arg7, 7, "XtGetSelectionValuesIncremental", "Time");
  len = XEN_TO_C_INT(arg4);
  if (len <= 0) return(XEN_FALSE);
  outs = XEN_TO_C_Atoms(arg3, len);
  xm_XtSelectionCallback_Descr = XEN_LIST_2(arg5, arg6);
  loc = xm_protect(xm_XtSelectionCallback_Descr);
  XtGetSelectionValuesIncremental(XEN_TO_C_Widget(arg1), 
				  XEN_TO_C_Atom(arg2), outs, len, 
				  gxm_XtSelectionCallbackProc, 
				  (XtPointer *)arg6, XEN_TO_C_Time(arg7));
  xm_unprotect_at(loc);
  FREE(outs);
  return(XEN_FALSE);
}  

static XEN gxm_XtGetSelectionValueIncremental(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XtGetSelectionValueIncremental "void XtGetSelectionValueIncremental(w, selection, target, callback, client_data, time) is similar \
to XtGetSelectionValue except that the selection_callback procedure will be called repeatedly upon delivery of multiple segments of the selection value."
  int loc;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGetSelectionValueIncremental", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtGetSelectionValueIncremental", "Atom");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XtGetSelectionValueIncremental", "Atom");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 7)), arg4, 4, "XtGetSelectionValueIncremental", "XtSelectionCallbackProc");
  XEN_ASSERT_TYPE(XEN_Time_P(arg6), arg6, 6, "XtGetSelectionValueIncremental", "Time");
  xm_XtSelectionCallback_Descr = XEN_LIST_2(arg4, arg5);
  loc = xm_protect(xm_XtSelectionCallback_Descr);
  XtGetSelectionValueIncremental(XEN_TO_C_Widget(arg1), 
				 XEN_TO_C_Atom(arg2), XEN_TO_C_Atom(arg3), 
				 gxm_XtSelectionCallbackProc, 
				 (XtPointer)arg5, XEN_TO_C_Time(arg6));
  xm_unprotect_at(loc);
  return(XEN_FALSE);
}

static XEN gxm_XtGetSelectionRequest(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtGetSelectionRequest "XSelectionRequestEvent* XtGetSelectionRequest(w, selection, request_id)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGetSelectionRequest", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtGetSelectionRequest", "Atom");
  XEN_ASSERT_TYPE(XEN_XtRequestId_P(arg3), arg3, 3, "XtGetSelectionRequest", "XtRequestId");
  return(C_TO_XEN_XSelectionRequestEvent(XtGetSelectionRequest(XEN_TO_C_Widget(arg1), XEN_TO_C_Atom(arg2), XEN_TO_C_XtRequestId(arg3))));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtGetSelectionTimeout(void)
{
  #define H_XtGetSelectionTimeout "unsigned long XtGetSelectionTimeout() has been superseded by XtAppGetSelectionTimeout."
  return(C_TO_XEN_ULONG(XtGetSelectionTimeout()));
}

static XEN gxm_XtSetSelectionTimeout(XEN arg1)
{
  #define H_XtSetSelectionTimeout "void XtSetSelectionTimeout(timeout) has been superseded by XtAppSetSelectionTimeout."
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, 1, "XtSetSelectionTimeout", "ulong");
  XtSetSelectionTimeout(XEN_TO_C_ULONG(arg1));
  return(XEN_FALSE);
}
#endif

static XEN gxm_XtAppGetSelectionTimeout(XEN arg1)
{
  #define H_XtAppGetSelectionTimeout "unsigned long XtAppGetSelectionTimeout(app_context) returns the current selection timeout value, in milliseconds."
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppGetSelectionTimeout", "XtAppContext");
  return(C_TO_XEN_ULONG(XtAppGetSelectionTimeout(XEN_TO_C_XtAppContext(arg1))));
}

static XEN gxm_XtAppSetSelectionTimeout(XEN arg1, XEN arg2)
{
  #define H_XtAppSetSelectionTimeout "void XtAppSetSelectionTimeout(app_context, timeout) sets the app's selection timeout mechanism."
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppSetSelectionTimeout", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtAppSetSelectionTimeout", "ulong");
  XtAppSetSelectionTimeout(XEN_TO_C_XtAppContext(arg1), XEN_TO_C_ULONG(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtGetSelectionValues(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XtGetSelectionValues "void XtGetSelectionValues(w, selection, targets, count, callback, client_data, time) is similar to \
XtGetSelectionValue except that it takes a list of target types and a list of client data and obtains the current value of the selection \
converted to each of the targets."
  /* DIFF: XtGetSelectionValues arg 3 is list of Atoms
   */
  Atom *outs;
  int len, loc;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGetSelectionValues", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtGetSelectionValues", "Atom");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XtGetSelectionValues", "list of Atom");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XtGetSelectionValues", "int");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg5) && (XEN_REQUIRED_ARGS_OK(arg5, 7)), arg5, 5, "XtGetSelectionValues", "XtSelectionCallbackProc");
  XEN_ASSERT_TYPE(XEN_Time_P(arg7), arg7, 7, "XtGetSelectionValues", "Time");
  len = XEN_TO_C_INT(arg4);
  if (len <= 0) return(XEN_FALSE);
  outs = XEN_TO_C_Atoms(arg3, len);
  xm_XtSelectionCallback_Descr = XEN_LIST_2(arg5, arg6);
  loc = xm_protect(xm_XtSelectionCallback_Descr);
  XtGetSelectionValues(XEN_TO_C_Widget(arg1), 
		       XEN_TO_C_Atom(arg2), outs, len, 
		       gxm_XtSelectionCallbackProc, 
		       (XtPointer *)arg6, XEN_TO_C_Time(arg7));
  xm_unprotect_at(loc);
  FREE(outs);
  return(XEN_FALSE);
}

static XEN gxm_XtGetSelectionValue(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XtGetSelectionValue "void XtGetSelectionValue(w, selection, target, callback, client_data, time) requests the value of the \
selection that has been converted to the target type. "
  int loc;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGetSelectionValue", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtGetSelectionValue", "Atom");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg3), arg3, 3, "XtGetSelectionValue", "Atom");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 7)), arg4, 4, "XtGetSelectionValue", "XtSelectionCallbackProc");
  XEN_ASSERT_TYPE(XEN_Time_P(arg6), arg6, 6, "XtGetSelectionValue", "Time");
  xm_XtSelectionCallback_Descr = XEN_LIST_2(arg4, arg5);
  loc = xm_protect(xm_XtSelectionCallback_Descr);
  XtGetSelectionValue(XEN_TO_C_Widget(arg1), 
		      XEN_TO_C_Atom(arg2), 
		      XEN_TO_C_Atom(arg3), 
		      gxm_XtSelectionCallbackProc, 
		      (XtPointer)arg5, XEN_TO_C_Time(arg6));
  xm_unprotect_at(loc);
  return(XEN_FALSE);
}

static XEN gxm_XtDisownSelection(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtDisownSelection "void XtDisownSelection(w, selection, time) informs the selection mechanism that the specified widget is to \
lose ownership of the selection."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtDisownSelection", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtDisownSelection", "Atom");
  XEN_ASSERT_TYPE(XEN_Time_P(arg3), arg3, 3, "XtDisownSelection", "Time");
  XtDisownSelection(XEN_TO_C_Widget(arg1), XEN_TO_C_Atom(arg2), XEN_TO_C_Time(arg3));
  return(XEN_FALSE);
}


/*-------- file predicate -------- */

/* a "Substitution" is a list '(char substitute), so pass a list of such lists below where a substitution array is required */

static XEN xm_filepredicate_proc;

static Boolean gxm_XtFilePredicate(String filename) 
{
  return(XEN_TO_C_BOOLEAN(XEN_CALL_1(xm_filepredicate_proc,
				     C_TO_XEN_STRING(filename),
				     c__FUNCTION__)));
}

static SubstitutionRec *gxm_make_subs(XEN lst_1)
{
  int len;
  SubstitutionRec *subs = NULL;
  len = XEN_LIST_LENGTH(lst_1);
  if (len > 0)
    {
      int i;
      XEN lst;
      lst = XEN_COPY_ARG(lst_1);
      subs = (SubstitutionRec *)CALLOC(len, sizeof(SubstitutionRec));
      for (i = 0; i < len; i++, lst = XEN_CDR(lst))
	{
	  if (!(XEN_LIST_P(XEN_CAR(lst))))
	    {
	      FREE(subs);
	      return(NULL);
	    }
	  subs[i].match = XEN_TO_C_CHAR(XEN_CAR(XEN_CAR(lst)));
	  subs[i].substitution = strdup(XEN_TO_C_STRING(XEN_CADR(XEN_CAR(lst))));
	}
    }
  return(subs);
}

/* (XtFindFile "/lib/%N:/usr/lib/%N:/usr/local/lib/%N" (list (list #\N "libxm.so")) 1 file-exists?) */

static XEN gxm_XtResolvePathname(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8)
{
  #define H_XtResolvePathname "String XtResolvePathname(display, type, filename, suffix, path, substitutions, num_substitutions, predicate)"
  /* DIFF: XtResolvePathname args use #f for NULL
   *       (XtResolvePathname (XtDisplay (cadr (main-widgets))) "app-defaults" #f #f #f #f 0 #f)
   */
  int arg8_loc = -1;
  XEN res;
  char *str;
  SubstitutionRec *subs = NULL;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtResolvePathname", "Display*");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg2) || XEN_STRING_P(arg2), arg2, 2, "XtResolvePathname", "char*");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg3) || XEN_STRING_P(arg3), arg3, 3, "XtResolvePathname", "char*");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg4) || XEN_STRING_P(arg4), arg4, 4, "XtResolvePathname", "char*");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg5) || XEN_STRING_P(arg5), arg5, 5, "XtResolvePathname", "char*");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg6) || XEN_LIST_P(arg6), arg6, 6, "XtResolvePathname", "Substitution list");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XtResolvePathname", "int");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg8) || 
		  (XEN_PROCEDURE_P(arg8) && (XEN_REQUIRED_ARGS_OK(arg8, 1))), 
		  arg8, 8, "XtResolvePathname", "XtFilePredicate (takes 1 arg)");
  if (XEN_LIST_P(arg6)) 
    {
      subs = gxm_make_subs(XEN_COPY_ARG(arg6));
      if (subs == NULL) return(XEN_FALSE); /* type error? */
    }
  if (XEN_PROCEDURE_P(arg8))
    {
      arg8_loc = xm_protect(arg8);
      xm_filepredicate_proc = arg8;
    }
  str = XtResolvePathname(XEN_TO_C_Display(arg1), 
			  (XEN_FALSE_P(arg2)) ? NULL : XEN_TO_C_STRING(arg2), 
			  (XEN_FALSE_P(arg3)) ? NULL : XEN_TO_C_STRING(arg3), 
			  (XEN_FALSE_P(arg4)) ? NULL : XEN_TO_C_STRING(arg4), 
			  (XEN_FALSE_P(arg5)) ? NULL : XEN_TO_C_STRING(arg5), 
			  subs,
			  XEN_TO_C_INT(arg7), 
			  (XEN_FALSE_P(arg8)) ? NULL : gxm_XtFilePredicate);
  if (XEN_PROCEDURE_P(arg8)) xm_unprotect_at(arg8_loc);
  if (subs) 
    {
      int i, len;
      len = XEN_LIST_LENGTH(arg6);
      for (i = 0; i < len; i++)
	if (subs[i].substitution) free(subs[i].substitution);
      FREE(subs);
    }
  res = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  return(res);
}

static XEN gxm_XtFindFile(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XtFindFile "String XtFindFile(path, substitutions, num_substitutions, predicate) \
searches for a file using substitutions in the path list"
  char *str;
  XEN res;
  int arg4_loc = -1;
  SubstitutionRec *subs = NULL;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtFindFile", "char*");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg2) || XEN_LIST_P(arg2), arg2, 2, "XtFindFile", "Substitution list");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XtFindFile", "int");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg4) || 
		  (XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 1))), 
		  arg4, 4, "XtFindFile", "XtFilePredicate (takes 1 arg)");
  if (XEN_LIST_P(arg2)) 
    {
      subs = gxm_make_subs(XEN_COPY_ARG(arg2));
      if (subs == NULL) return(XEN_FALSE); /* type error? */
    }
  if (XEN_PROCEDURE_P(arg4))
    {
      arg4_loc = xm_protect(arg4);
      xm_filepredicate_proc = arg4;
    }
  str = XtFindFile(XEN_TO_C_STRING(arg1), 
		   subs,
		   XEN_TO_C_INT(arg3),
		   (XEN_FALSE_P(arg4) ? NULL : gxm_XtFilePredicate));
  if (XEN_PROCEDURE_P(arg4)) xm_unprotect_at(arg4_loc);
  if (subs) 
    {
      int i, len;
      len = XEN_LIST_LENGTH(arg2);
      for (i = 0; i < len; i++)
	if (subs[i].substitution) free(subs[i].substitution);
      FREE(subs);
    }
  res = C_TO_XEN_STRING(str);
  if (str) XtFree(str);
  return(res);
}

static XEN gxm_XtReleaseGC(XEN arg1, XEN arg2)
{
  #define H_XtReleaseGC "void XtReleaseGC(w, gc) deallocate the specified shared GC."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtReleaseGC", "Widget");
  XEN_ASSERT_TYPE(XEN_GC_P(arg2), arg2, 2, "XtReleaseGC", "GC");
  XtReleaseGC(XEN_TO_C_Widget(arg1), XEN_TO_C_GC(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtDestroyGC(XEN arg1)
{
  #define H_XtDestroyGC "XtDestroyGC(gc) is obsolete -- use XtReleaseGC"
  XEN_ASSERT_TYPE(XEN_GC_P(arg1), arg1, 1, "XtDestroyGC", "GC");
  XtDestroyGC(XEN_TO_C_GC(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtAllocateGC(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XtAllocateGC "GC XtAllocateGC(w, depth, value_mask, values, dynamic_mask, unused_mask) returns a sharable GC that may be modified by the client."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtAllocateGC", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XtAllocateGC", "int");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XtAllocateGC", "XtGCMask");
  XEN_ASSERT_TYPE(XEN_XGCValues_P(arg4), arg4, 4, "XtAllocateGC", "XGCValues*");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg5), arg5, 5, "XtAllocateGC", "XtGCMask");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg6), arg6, 6, "XtAllocateGC", "XtGCMask");
  return(C_TO_XEN_GC(XtAllocateGC(XEN_TO_C_Widget(arg1), XEN_TO_C_INT(arg2), 
				  XEN_TO_C_ULONG(arg3), XEN_TO_C_XGCValues(arg4), 
				  XEN_TO_C_ULONG(arg5), XEN_TO_C_ULONG(arg6))));
}

static XEN gxm_XtGetGC(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtGetGC "GC XtGetGC(w, value_mask, values) returns a sharable, read-only GC."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGetGC", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtGetGC", "XtGCMask");
  XEN_ASSERT_TYPE(XEN_XGCValues_P(arg3), arg3, 3, "XtGetGC", "XGCValues*");
  return(C_TO_XEN_GC(XtGetGC(XEN_TO_C_Widget(arg1), XEN_TO_C_ULONG(arg2), XEN_TO_C_XGCValues(arg3))));
}


/* -------- background process -------- */

/* (80) XtWorkProc called (protected) until quits or removed explicitly */

#define C_TO_XEN_XM_Background(Code, Context) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("Background"), Code, Context, XEN_ZERO, XEN_ZERO)
#define XM_Background_P(Arg) WRAP_P("Background", Arg)

static Boolean gxm_XtWorkProc(XtPointer cdata)
{
  /* if true, quits */
  int val;
  XEN descr = (XEN)cdata;
  /* (list 'Background function context gc-loc id) */
  val = XEN_TO_C_BOOLEAN(XEN_CALL_1(XEN_CADR(descr), 
				    XEN_CADDR(descr),
				    c__FUNCTION__));
  if (val) xm_unprotect_at(XEN_TO_C_INT(XEN_LIST_REF(descr, 3)));
  return(val);
}

static bool unprotect_workproc(XEN val, int loc, unsigned long id)
{
  if ((XM_Background_P(val)) &&
      ((XtWorkProcId)XEN_TO_C_ULONG(XEN_LIST_REF(val, 4)) == id))
    {
      xm_unprotect_at(loc);
      return(true);
    }
  return(false);
}

static XEN gxm_XtRemoveWorkProc(XEN arg1)
{
  #define H_XtRemoveWorkProc "void XtRemoveWorkProc(id) explicitly removes the specified background work procedure."
  XtWorkProcId id;
  XEN_ASSERT_TYPE(XEN_XtWorkProcId_P(arg1), arg1, 1, "XtRemoveWorkProc", "XtWorkProcId");
  id = XEN_TO_C_XtWorkProcId(arg1);
  XtRemoveWorkProc(id);
  map_over_protected_elements(unprotect_workproc, id);
  return(XEN_FALSE);
}

static XEN gxm_XtAppAddWorkProc(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtAppAddWorkProc "XtWorkProcId XtAppAddWorkProc(app_context, proc, client_data) adds the specified work procedure for the \
application identified by app_context."
  XtWorkProcId id;
  int gc_loc;
  XEN descr;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppAddWorkProc", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg2) && (XEN_REQUIRED_ARGS_OK(arg2, 1)), arg2, 2, "XtAppAddWorkProc", "(XtWorkProc data)");
  
  descr = C_TO_XEN_XM_Background(arg2, (XEN_BOUND_P(arg3)) ? arg3 : XEN_FALSE);
  gc_loc = xm_protect(descr);
  id = XtAppAddWorkProc(XEN_TO_C_XtAppContext(arg1), 
			gxm_XtWorkProc, 
			(XtPointer)descr);
  XEN_LIST_SET(descr, 3, C_TO_XEN_INT(gc_loc));
  XEN_LIST_SET(descr, 4, C_TO_XEN_ULONG(id));
  return(C_TO_XEN_XtWorkProcId(id));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtAddWorkProc(XEN arg1, XEN arg2)
{
  #define H_XtAddWorkProc "XtWorkProcId XtAddWorkProc(proc, client_data) has been replaced by XtAppAddWorkProc."
  XtWorkProcId id;
  int gc_loc;
  XEN descr;
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg1) && (XEN_REQUIRED_ARGS_OK(arg1, 1)), arg1, 1, "XtAddWorkProc", "(XtWorkProc data)");
  descr = C_TO_XEN_XM_Background(arg1, (XEN_BOUND_P(arg2)) ? arg2 : XEN_FALSE);
  gc_loc = xm_protect(descr);
  id = XtAddWorkProc(gxm_XtWorkProc, 
		     (XtPointer)descr);
  XEN_LIST_SET(descr, 3, C_TO_XEN_INT(gc_loc));
  XEN_LIST_SET(descr, 4, C_TO_XEN_ULONG(id));
  return(C_TO_XEN_XtWorkProcId(id));
}
#endif

/* the next 4 are needed where the caller allocates a block of memory, but X frees it (XCreateImage) --
 *   can't use Scheme-allocated memory here etc
 */
static XEN gxm_XtFree(XEN arg1)
{
  #define H_XtFree "void XtFree(ptr)"
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, 1, "XtFree", "pointer");
  XtFree((char *)XEN_TO_C_ULONG(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtRealloc(XEN arg1, XEN arg2)
{
  #define H_XtRealloc "char *XtRealloc(ptr, num)"
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, 1, "XtRealloc", "pointer");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XtRealloc", "int");
  return(C_TO_XEN_ULONG(XtRealloc((char *)XEN_TO_C_ULONG(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XtCalloc(XEN arg1, XEN arg2)
{
  #define H_XtCalloc "char *XtCalloc(num, size)"
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 1, "XtCalloc", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XtCalloc", "int");
  return(C_TO_XEN_ULONG(XtCalloc(XEN_TO_C_INT(arg1), XEN_TO_C_INT(arg2))));
}

static XEN gxm_XtMalloc(XEN arg1)
{
  #define H_XtMalloc "char *XtMalloc(size)"
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 1, "XtMalloc", "int");
  return(C_TO_XEN_ULONG(XtMalloc(XEN_TO_C_INT(arg1))));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtWarning(XEN arg1)
{
  #define H_XtWarning "void XtWarning(message)"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtWarning", "char*");
  XtWarning(XEN_TO_C_STRING(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtAppWarning(XEN arg1, XEN arg2)
{
  #define H_XtAppWarning "void XtAppWarning(app_context, message) calls the installed nonfatal error procedure and passes the specified message."
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppWarning", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtAppWarning", "char*");
  XtAppWarning(XEN_TO_C_XtAppContext(arg1), XEN_TO_C_STRING(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtError(XEN arg1)
{
  #define H_XtError "void XtError(message)"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtError", "char*");
  XtError(XEN_TO_C_STRING(arg1));
  return(XEN_FALSE);
}
#endif

static XEN xm_XtErrorHandler = XEN_FALSE;
static XEN xm_XtWarningHandler = XEN_FALSE;

static void gxm_XtErrorHandler(String msg)
{
  if (XEN_PROCEDURE_P(xm_XtErrorHandler))
    XEN_CALL_1(xm_XtErrorHandler, 
	       C_TO_XEN_STRING(msg), 
	       c__FUNCTION__);
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtSetErrorHandler(XEN arg1)
{
  #define H_XtSetErrorHandler "void XtSetErrorHandler(handler) has been superseded by XtAppSetErrorHandler."
  XEN old_val;
  old_val = xm_XtErrorHandler;
  xm_protect(arg1);
  xm_XtErrorHandler = arg1;
  XtSetErrorHandler(gxm_XtErrorHandler);
  if (XEN_PROCEDURE_P(old_val)) xm_unprotect(old_val);
  return(old_val);
}
#endif

static XEN gxm_XtAppSetErrorHandler(XEN arg1, XEN arg2)
{
  #define H_XtAppSetErrorHandler "void XtAppSetErrorHandler(app_context, handler) registers the specified procedure, which is called when \
a fatal error condition occurs."
  XEN old_val;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppSetErrorHandler", "XtAppContext");
  old_val = xm_XtErrorHandler;
  xm_protect(arg2);
  xm_XtErrorHandler = arg2;
  XtAppSetErrorHandler(XEN_TO_C_XtAppContext(arg1), gxm_XtErrorHandler);
  if (XEN_PROCEDURE_P(old_val)) xm_unprotect(old_val);
  return(old_val);
}

static void gxm_XtWarningHandler(String msg)
{
  if (XEN_PROCEDURE_P(xm_XtWarningHandler))
    XEN_CALL_1(xm_XtWarningHandler, 
	       C_TO_XEN_STRING(msg), 
	       c__FUNCTION__);
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtSetWarningHandler(XEN arg1)
{
  #define H_XtSetWarningHandler "void XtSetWarningHandler(handler) has been superseded by XtAppSetWarningHandler."
  XEN old_val;
  old_val = xm_XtWarningHandler;
  xm_protect(arg1);
  xm_XtWarningHandler = arg1;
  XtSetWarningHandler(gxm_XtWarningHandler);
  if (XEN_PROCEDURE_P(old_val)) xm_unprotect(old_val);
  return(old_val);
}
#endif

static XEN gxm_XtAppSetWarningHandler(XEN arg1, XEN arg2)
{
  #define H_XtAppSetWarningHandler "void XtAppSetWarningHandler(app_context, handler) registers the specified procedure, which is called \
when a nonfatal error condition occurs."
  XEN old_val;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppSetWarningHandler", "XtAppContext");
  old_val = xm_XtWarningHandler;
  xm_protect(arg2);
  xm_XtWarningHandler = arg2;
  XtAppSetWarningHandler(XEN_TO_C_XtAppContext(arg1), gxm_XtWarningHandler);
  if (XEN_PROCEDURE_P(old_val)) xm_unprotect(old_val);
  return(old_val);
}

static XEN gxm_XtAppError(XEN arg1, XEN arg2)
{
  #define H_XtAppError "void XtAppError(app_context, message) calls the installed error procedure and passes the specified message."
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppError", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtAppError", "char*");
  XtAppError(XEN_TO_C_XtAppContext(arg1), XEN_TO_C_STRING(arg2));
  return(XEN_FALSE);
}


#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtWarningMsg(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XtWarningMsg "void XtWarningMsg(name, type, class, default, params, num_params)"
  /* DIFF: XtWarningMsg takes final int not int*, arg5 is list of strings
   */
  int size;
  Cardinal csize;
  char **pars;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtWarningMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtWarningMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XtWarningMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XtWarningMsg", "char*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg5), arg5, 5, "XtWarningMsg", "list of String");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XtWarningMsg", "int");
  size = XEN_TO_C_INT(arg6);
  if (size <= 0) return(XEN_FALSE);
  pars = XEN_TO_C_Strings(arg5, size);
  csize = (Cardinal)size;
  XtWarningMsg(XEN_TO_C_STRING(arg1), XEN_TO_C_STRING(arg2), XEN_TO_C_STRING(arg3), XEN_TO_C_STRING(arg4), pars, &csize);
  FREE(pars);
  return(xen_return_first(XEN_FALSE, arg5));
}
#endif

static XEN gxm_XtAppWarningMsg(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XtAppWarningMsg "void XtAppWarningMsg(app_context, name, type, class, default, params, num_params) calls the high-level error \
handler and passes the specified information."
  /* DIFF: XtAppWarningMsg takes final int not int*, arg6 is list of strings
   */
  int size;
  Cardinal csize;
  char **pars;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppWarningMsg", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtAppWarningMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XtAppWarningMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XtAppWarningMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg5), arg5, 5, "XtAppWarningMsg", "char*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg6), arg6, 6, "XtAppWarningMsg", "list of String");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XtAppWarningMsg", "int");
  size = XEN_TO_C_INT(arg7);
  if (size <= 0) return(XEN_FALSE);
  pars = XEN_TO_C_Strings(arg6, size);
  csize = (Cardinal)size;
  XtAppWarningMsg(XEN_TO_C_XtAppContext(arg1), XEN_TO_C_STRING(arg2), 
		  XEN_TO_C_STRING(arg3), XEN_TO_C_STRING(arg4), XEN_TO_C_STRING(arg5), 
		  pars, &csize);
  FREE(pars);
  return(xen_return_first(XEN_FALSE, arg6));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtErrorMsg(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XtErrorMsg "void XtErrorMsg(name, type, class, default, params, num_params) has been superseded by XtAppErrorMsg."
  /* DIFF: XtErrorMsg takes final int not int*, arg5 is list of strings
   */
  int size;
  Cardinal csize;
  char **pars;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtErrorMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtErrorMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XtErrorMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XtErrorMsg", "char*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg5), arg5, 5, "XtErrorMsg", "list of String");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XtErrorMsg", "int");
  size = XEN_TO_C_INT(arg6);
  if (size <= 0) return(XEN_FALSE);
  pars = XEN_TO_C_Strings(arg5, size);
  csize = (Cardinal)size;
  XtErrorMsg(XEN_TO_C_STRING(arg1), XEN_TO_C_STRING(arg2), XEN_TO_C_STRING(arg3), XEN_TO_C_STRING(arg4), pars, &csize);
  FREE(pars);
  return(xen_return_first(XEN_FALSE, arg5));
}
#endif

static XEN gxm_XtAppErrorMsg(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7)
{
  #define H_XtAppErrorMsg "void XtAppErrorMsg(app_context, name, type, class, default, params, num_params) calls the high-level error \
handler and passes the specified information."
  /* DIFF: XtAppErrorMsg takes final int not int*, arg6 is list of strings
   */
  int size;
  Cardinal csize;
  char **pars;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppErrorMsg", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtAppErrorMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XtAppErrorMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XtAppErrorMsg", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg5), arg5, 5, "XtAppErrorMsg", "char*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg6), arg6, 6, "XtAppErrorMsg", "list of String");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 7, "XtAppErrorMsg", "int");
  size = XEN_TO_C_INT(arg7);
  if (size <= 0) return(XEN_FALSE);
  pars = XEN_TO_C_Strings(arg6, size);
  csize = (Cardinal)size;
  XtAppErrorMsg(XEN_TO_C_XtAppContext(arg1), XEN_TO_C_STRING(arg2), 
		XEN_TO_C_STRING(arg3), XEN_TO_C_STRING(arg4), 
		XEN_TO_C_STRING(arg5), pars, &csize);
  FREE(pars);
  return(xen_return_first(XEN_FALSE, arg6));
}

static XEN xm_XtErrorMsgHandler = XEN_FALSE;
static XEN xm_XtWarningMsgHandler = XEN_FALSE;

static void gxm_XtErrorMsgHandler(String name, String type, String clas, String defp, String *pars, Cardinal *num)
{
  /* DIFF: XtErrorMsgHandler takes list of string pars
   */
  if ((XEN_PROCEDURE_P(xm_XtErrorMsgHandler)) && (num) && ((pars) || (*num == 0)))
    {
      XEN lst = XEN_EMPTY_LIST;
      int i, len, loc;
      loc = xm_protect(lst);
      len = (*num);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_STRING(pars[i]), lst);
      XEN_CALL_6(xm_XtErrorMsgHandler, 
		 C_TO_XEN_STRING(name),
		 C_TO_XEN_STRING(type),
		 C_TO_XEN_STRING(clas),
		 C_TO_XEN_STRING(defp),
		 lst,
		 C_TO_XEN_INT(*num),
		 c__FUNCTION__);
      xm_unprotect_at(loc);
    }
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtSetErrorMsgHandler(XEN arg1)
{
  #define H_XtSetErrorMsgHandler "void XtSetErrorMsgHandler(msg_handler) has been superseded by XtAppSetErrorMsgHandler."
  XEN old_val;
  old_val = xm_XtErrorMsgHandler;
  xm_protect(arg1);
  xm_XtErrorMsgHandler = arg1;
  XtSetErrorMsgHandler(gxm_XtErrorMsgHandler);
  if (XEN_PROCEDURE_P(old_val)) xm_unprotect(old_val);
  return(old_val);
}
#endif

static XEN gxm_XtAppSetErrorMsgHandler(XEN arg1, XEN arg2)
{
  #define H_XtAppSetErrorMsgHandler "void XtAppSetErrorMsgHandler(app_context, msg_handler) registers the specified procedure, which is called \
when a fatal error occurs."
  XEN old_val;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppSetErrorMsgHandler", "XtAppContext");
  old_val = xm_XtErrorMsgHandler;
  xm_protect(arg2);
  xm_XtErrorMsgHandler = arg2;
  XtAppSetErrorMsgHandler(XEN_TO_C_XtAppContext(arg1), gxm_XtErrorMsgHandler);
  if (XEN_PROCEDURE_P(old_val)) xm_unprotect(old_val);
  return(old_val);
}


static void gxm_XtWarningMsgHandler(String name, String type, String clas, String defp, String *pars, Cardinal *num)
{
  /* DIFF: XtWarningMsgHandler takes list of string pars
   */
  if ((XEN_PROCEDURE_P(xm_XtWarningMsgHandler)) && (num) && ((pars) || (*num == 0)))
    {
      XEN lst = XEN_EMPTY_LIST;
      int i, len, loc;
      loc = xm_protect(lst);
      len = (*num);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_STRING(pars[i]), lst);
      XEN_CALL_6(xm_XtWarningMsgHandler, 
		 C_TO_XEN_STRING(name),
		 C_TO_XEN_STRING(type),
		 C_TO_XEN_STRING(clas),
		 C_TO_XEN_STRING(defp),
		 lst,
		 C_TO_XEN_INT(*num),
		 c__FUNCTION__);
      xm_unprotect_at(loc);
    }
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtSetWarningMsgHandler(XEN arg1)
{
  #define H_XtSetWarningMsgHandler "void XtSetWarningMsgHandler(msg_handler) has been superseded by XtAppSetWarningMsgHandler."
  XEN old_val;
  old_val = xm_XtWarningMsgHandler;
  xm_protect(arg1);
  xm_XtWarningMsgHandler = arg1;
  XtSetWarningMsgHandler(gxm_XtWarningMsgHandler);
  if (XEN_PROCEDURE_P(old_val)) xm_unprotect(old_val);
  return(old_val);
}
#endif

static XEN gxm_XtAppSetWarningMsgHandler(XEN arg1, XEN arg2)
{
  #define H_XtAppSetWarningMsgHandler "void XtAppSetWarningMsgHandler(app_context, msg_handler) registers the specified procedure, which \
is called when a nonfatal error condition occurs."
  XEN old_val;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppSetWarningMsgHandler", "XtAppContext");
  old_val = xm_XtWarningMsgHandler;
  xm_protect(arg2);
  xm_XtWarningMsgHandler = arg2;
  XtAppSetWarningMsgHandler(XEN_TO_C_XtAppContext(arg1), gxm_XtWarningMsgHandler);
  if (XEN_PROCEDURE_P(old_val)) xm_unprotect(old_val);
  return(old_val);
}

static XEN gxm_XtGetValues(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtGetValues "void XtGetValues(Widget w, ArgList args, Cardinal num_args) returns the values of the \
resources specified for the widget w.  In xm, 'args' is a list of resource names followed by a placeholder \
(normally 0), and the length of the arglist is optional.  For example, to get the current height and width \
of a widget, (" XM_PREFIX "XtGetValues" XM_POSTFIX " w (list " XM_PREFIX "XmNheight" XM_POSTFIX " 0 \
" XM_PREFIX "XmNwidth" XM_POSTFIX " 0)) which returns the list with the values filled in: (list height 123 \
width 321).  If the resource value is an array in C, it is returned as a list."

  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGetValues", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XtGetValues", "List");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XtGetValues", "int");
  return(gxm_XtGetValues_1(arg1, arg2, XEN_TO_C_INT_DEF(arg3, arg2)));
}

static XEN gxm_XtVaGetValues(XEN arg1, XEN arg2)
{
  #define H_XtVaGetValues "void XtVaGetValues(w, ...) in xm is the same as XtGetValues."
  /* DIFF: XtVaGetValues -> returns original list with vals in place 
   */
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtVaGetValues", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XtVaGetValues", "List");
  return(gxm_XtGetValues_1(arg1, arg2, XEN_LIST_LENGTH(arg2) / 2));
}

static XEN gxm_XtVaSetValues(XEN arg1, XEN arg2)
{
  #define H_XtVaSetValues "void XtVaSetValues(w, ...) in xm is the same as XtSetValues."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtVaSetValues", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XtVaSetValues", "List");
  {
    Widget w;
    Arg *args;
    int arglen;
    args = XEN_TO_C_Args(arg2);
    w = XEN_TO_C_Widget(arg1);
    arglen = XEN_LIST_LENGTH(arg2) / 2;
    XtSetValues(w, args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(XEN_FALSE);
}

static XEN gxm_XtSetValues(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtSetValues "void XtSetValues(Widget w, ArgList args, Cardinal num_args) sets the values of the \
resources specified for the widget w.  In xm, 'args' is a list of resource names followed by the new value, \
and the length of the arglist is optional.  For example, to set the current height and width \
of a widget, (" XM_PREFIX "XtSetValues" XM_POSTFIX " w (list " XM_PREFIX "XmNheight" XM_POSTFIX " 123 \
" XM_PREFIX "XmNwidth" XM_POSTFIX " 321)).  If the resource value in C is an array, it is a list in xm."

  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtSetValues", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XtSetValues", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg3), arg3, 3, "XtSetValues", "int");
  {
    Arg *args;
    int arglen;
    Widget w;
    w = XEN_TO_C_Widget(arg1);
    args = XEN_TO_C_Args(arg2);
    arglen = XEN_TO_C_INT_DEF(arg3, arg2);
    XtSetValues(w, args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(XEN_FALSE);
}

static XEN gxm_XtCloseDisplay(XEN arg1)
{
  #define H_XtCloseDisplay "void XtCloseDisplay(display) closes the specified display as soon as it is safe to do so."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtCloseDisplay", "Display*");
  XtCloseDisplay(XEN_TO_C_Display(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtDisplayToApplicationContext(XEN arg1)
{
  #define H_XtDisplayToApplicationContext "XtAppContext XtDisplayToApplicationContext(dpy) returns the application context for the specified display."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtDisplayToApplicationContext", "Display*");
  return(C_TO_XEN_XtAppContext(XtDisplayToApplicationContext(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XtWidgetToApplicationContext(XEN arg1)
{
  #define H_XtWidgetToApplicationContext "XtAppContext XtWidgetToApplicationContext(w) returns the application context for the specified widget."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtWidgetToApplicationContext", "Widget");
  return(C_TO_XEN_XtAppContext(XtWidgetToApplicationContext(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtInitializeWidgetClass(XEN arg1)
{
  #define H_XtInitializeWidgetClass "void XtInitializeWidgetClass(object_class)"
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg1), arg1, 1, "XtInitializeWidgetClass", "WidgetClass");
  XtInitializeWidgetClass(XEN_TO_C_WidgetClass(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtDestroyApplicationContext(XEN arg1)
{
  #define H_XtDestroyApplicationContext "void XtDestroyApplicationContext(app_context) destroys the specified application context as soon as it is safe to do so."
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtDestroyApplicationContext", "XtAppContext");
  XtDestroyApplicationContext(XEN_TO_C_XtAppContext(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtCreateApplicationContext(void)
{
  #define H_XtCreateApplicationContext "XtAppContext XtCreateApplicationContext()"
  return(C_TO_XEN_XtAppContext(XtCreateApplicationContext()));
}

static XEN gxm_argv_to_list(XEN lst, int argc, char **argv)
{
  int i, loc;
  if (argc == 0) return(lst);
  loc = xm_protect(lst);
  for (i = argc - 1; i >= 0; i--)
    lst = XEN_CONS(C_TO_XEN_STRING(argv[i]), lst);
  FREE(argv);
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XtOpenDisplay(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg7, XEN arg8)
{
  #define H_XtOpenDisplay "Display *XtOpenDisplay(app_context, display_string, application_name, application_class, argc, argv) \
calls XOpenDisplay the specified display name."
  /* DIFF: XtOpenDisplay ignore arg5 6, argc is int, argv is list of strings, returns (list dpy argv ...)
   */
  char **argv = NULL;
  int argc;
  XEN lst = XEN_EMPTY_LIST;
  Display *dpy;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtOpenDisplay", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2) || XEN_FALSE_P(arg2), arg2, 2, "XtOpenDisplay", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XtOpenDisplay", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XtOpenDisplay", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 5, "XtOpenDisplay", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg8), arg8, 6, "XtOpenDisplay", "list of char*");
  argc = XEN_TO_C_INT(arg7);
  if (XEN_LIST_LENGTH(arg8) != argc) return(XEN_FALSE); /* error? */
  if (argc > 0) argv = XEN_TO_C_Strings(arg8, argc);
  dpy = XtOpenDisplay(XEN_TO_C_XtAppContext(arg1), 
		      (XEN_FALSE_P(arg2)) ? NULL : XEN_TO_C_STRING(arg2), 
		      XEN_TO_C_STRING(arg3), 
		      XEN_TO_C_STRING(arg4), 
		      NULL, 0, &argc, argv);
  if (dpy)
    lst = XEN_CONS(C_TO_XEN_Display(dpy), lst);
  else lst = XEN_CONS(XEN_FALSE, lst);
  return(gxm_argv_to_list(lst, argc, argv));
}

#if (!XM_DISABLE_DEPRECATED) 
static XEN gxm_XtInitialize(XEN arg1, XEN arg2, XEN arg5, XEN arg6)
{
  #define H_XtInitialize "Widget XtInitialize(shell_name, application_class, argc, argv) calls XtToolkitInitialize \
followed by XtOpenDisplay with display_string NULL and application_name NULL, and finally calls XtAppCreateShell with appcation_name NULL, \
widget_class applicationShellWidgetClass , and the specified args and num_args and returns the created shell. "
  /* DIFF: XtInitialize ignore arg 3 4, argc is int, argv is list of strings
   */
  char **argv = NULL;
  int argc;
  Widget w;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtInitialize", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtInitialize", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 3, "XtInitialize", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg6), arg6, 4, "XtInitialize", "list of char*");
  argc = XEN_TO_C_INT(arg5);
  if (XEN_LIST_LENGTH(arg6) != argc) return(XEN_FALSE); /* error? */
  if (argc > 0) argv = XEN_TO_C_Strings(arg6, argc);
  w = XtInitialize(XEN_TO_C_STRING(arg1), XEN_TO_C_STRING(arg2), NULL, 0, &argc, argv);
  return(gxm_argv_to_list(XEN_CONS(C_TO_XEN_Widget(w), XEN_EMPTY_LIST), argc, argv));
}
#endif

static XEN gxm_XtAppSetFallbackResources(XEN app, XEN specs)
{
  #define H_XtAppSetFallbackResources "XtAppSetFallbackResources(app, list-of-strings) sets the app's default resource values \
from the list of strings"
  char **fallbacks;
  int i, len;
  XEN lst;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(app), app, 1, "XtAppSetFallbackResources", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_LIST_P(specs), specs, 2, "XtAppSetFallbackResources", "list of char*");
  len = XEN_LIST_LENGTH(specs);
  lst = XEN_COPY_ARG(specs);
  fallbacks = (char **)CALLOC(len + 1, sizeof(char *)); /* +1 for null termination */
  for (i = 0; i < len; i++, lst = XEN_CDR(lst)) 
    fallbacks[i] = XEN_TO_C_STRING(XEN_CAR(lst));
  XtAppSetFallbackResources(XEN_TO_C_XtAppContext(app), fallbacks);
  FREE(fallbacks);
  return(xen_return_first(app, specs));
}

static XEN gxm_XtVaAppInitialize(XEN arg2, XEN arg5, XEN arg6, XEN arg8, XEN specs)
{
  #define H_XtVaAppInitialize "Widget XtVaAppInitialize(application_class, argc_in_out, argv_in_out, args, fallbacks) -- the order \
of the arguments is slightly different from the C Xt call.  The final arg is an (optional) list of strings."
  /* DIFF: XtVaAppInitialize [app] class {options numopts} {argc} argv resources -> (list widget app (new argv)), argc is int not int* options/num ignored
     Arg *args;
     app is returned not passed (list widget app) 
  */
  XtAppContext app;
  Arg *args;
  Widget res;
  int i, len = 0, argc, arglen;
  char **argv = NULL;
  char **fallbacks = NULL;
  XEN lst;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 1, "XtVaAppInitialize", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 2, "XtVaAppInitialize", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg6), arg6, 3, "XtVaAppInitialize", "list of String");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg8), arg8, 4, "XtVaAppInitialize", "arg list");
  XEN_ASSERT_TYPE(XEN_LIST_P(specs) || XEN_NOT_BOUND_P(specs), specs, 5, "XtVaAppInitialize", "list of char*");
  argc = XEN_TO_C_INT(arg5);
  if (XEN_LIST_LENGTH(arg6) != argc) return(XEN_FALSE); /* error? */
  if (argc > 0) argv = XEN_TO_C_Strings(arg6, argc);
  if (XEN_LIST_P(specs))
    {
      int gcloc;
      len = XEN_LIST_LENGTH(specs);
      lst = XEN_COPY_ARG(specs);
      gcloc = xm_protect(lst);
      fallbacks = (char **)CALLOC(len + 1, sizeof(char *)); /* +1 for null termination */
      for (i = 0; i < len; i++, lst = XEN_CDR(lst)) 
	fallbacks[i] = strdup(XEN_TO_C_STRING(XEN_CAR(lst)));
      xm_unprotect_at(gcloc);
    }
  args = XEN_TO_C_Args(arg8);
  arglen = XEN_LIST_LENGTH(arg8) / 2;
  res = XtAppInitialize(&app, 
			XEN_TO_C_STRING(arg2), 
			NULL,
			0,
			&argc, 
			argv,
			fallbacks,
			args,
			arglen);
  if (args)
    {
      fixup_args(res, args, arglen);
      free_args(args, arglen);
    }
  if (fallbacks) 
    {
      for (i = 0; i < len; i++)
	if (fallbacks[i]) free(fallbacks[i]);
      FREE(fallbacks);
    }
  return(XEN_LIST_3(C_TO_XEN_Widget(res), 
		    C_TO_XEN_XtAppContext(app),
		    gxm_argv_to_list(XEN_EMPTY_LIST, argc, argv)));
}


static XEN gxm_XtAppInitialize(XEN arg2, XEN arg5, XEN arg6, XEN arg8, XEN arg9)
{
  #define H_XtAppInitialize "Widget XtAppInitialize(application_class, argc_in_out, argv_in_out, \
args, num_args) calls XtToolkitInitialize followed by XtCreateApplicationContext ,then calls XtOpenDisplay with \
display_string NULL and application_name NULL, and finally calls XtAppCreateShell with appcation_name NULL, widget_class applicationShellWidgetClass , \
and the specified args and num_args and returns the created shell.  The num_args argument can be list of strings = fallback resources"
  /* DIFF: XtAppInitialize [app] class {options numopts} {argc} argv resources args numargs -> (list widget app), argc is int not int* options/num ignored
     Arg *args;
     app is returned not passed (list widget app) 
  */
  XtAppContext app;
  Arg *args;
  Widget res;
  int argc, arglen;
  char **argv = NULL;
  char **fallbacks = NULL;
  int i, len = 0;
  XEN lst;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 1, "XtAppInitialize", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 2, "XtAppInitialize", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg6), arg6, 3, "XtAppInitialize", "list of String*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg8), arg8, 4, "XtAppInitialize", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg9) || XEN_LIST_P(arg9) || XEN_NOT_BOUND_P(arg9), arg9, 5, "XtAppInitialize", "int or list of strings"); /* num_args */
  argc = XEN_TO_C_INT(arg5);
  if (XEN_LIST_LENGTH(arg6) != argc) return(XEN_FALSE); /* error? */
  if (argc > 0) argv = XEN_TO_C_Strings(arg6, argc);
  args = XEN_TO_C_Args(arg8);
  if (XEN_INTEGER_P(arg9)) arglen = XEN_TO_C_INT(arg9); else arglen = XEN_LIST_LENGTH(arg8) / 2;
  if (XEN_LIST_P(arg9))
    {
      int gcloc;
      len = XEN_LIST_LENGTH(arg9);
      lst = XEN_COPY_ARG(arg9);
      gcloc = xm_protect(lst);
      fallbacks = (char **)CALLOC(len + 1, sizeof(char *)); /* +1 for null termination */
      for (i = 0; i < len; i++, lst = XEN_CDR(lst)) 
	fallbacks[i] = strdup(XEN_TO_C_STRING(XEN_CAR(lst)));
      xm_unprotect_at(gcloc);
    }
  res = XtAppInitialize(&app,
			XEN_TO_C_STRING(arg2), 
			NULL,
			0,
			&argc,
			argv,
			fallbacks,
			args,
			arglen);
  if (args)
    {
      fixup_args(res, args, arglen);
      free_args(args, arglen);
    }
  if (fallbacks) 
    {
      for (i = 0; i < len; i++)
	if (fallbacks[i]) free(fallbacks[i]);
      FREE(fallbacks);
    }
  return(XEN_LIST_3(C_TO_XEN_Widget(res), 
		    C_TO_XEN_XtAppContext(app),
		    gxm_argv_to_list(XEN_EMPTY_LIST, argc, argv)));
}

static XEN gxm_XtVaOpenApplication(XEN arg1, XEN arg4, XEN arg5, XEN arg7, XEN arg8, XEN specs)
{
  #define H_XtVaOpenApplication "Widget XtVaOpenApplication(application_class, argc_in_out, argv_in_out, widget_class, args, fallbacks)"
  /* DIFF: XtVaOpenApplication [app] name {options numopts} {argc} argv resources class args -> (list widget app), argc is int not int* options/num ignored
     Arg *args;
     app is returned not passed (list widget app) 
  */
  XtAppContext app;
  Arg *args;
  Widget res;
  int argc, arglen;
  char **argv = NULL;
  char **fallbacks = NULL;
  int i, len = 0;
  XEN lst;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtVaOpenApplication", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 2, "XtVaOpenApplication", "int"); /* was arg3 by mistake, 11-Oct-02 */
  XEN_ASSERT_TYPE(XEN_LIST_P(arg5), arg5, 3, "XtVaOpenApplication", "list of String");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg7), arg7, 4, "XtVaOpenApplication", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg8), arg8, 5, "XtVaOpenApplication", "arg list");
  XEN_ASSERT_TYPE(XEN_LIST_P(specs) || XEN_NOT_BOUND_P(specs), specs, 5, "XtVaOpenApplication", "list of char*");
  argc = XEN_TO_C_INT(arg4);
  if (XEN_LIST_LENGTH(arg5) != argc) return(XEN_FALSE); /* error? */
  if (argc > 0) argv = XEN_TO_C_Strings(arg5, argc);
  if (XEN_LIST_P(specs))
    {
      int gcloc;
      len = XEN_LIST_LENGTH(specs);
      lst = XEN_COPY_ARG(specs);
      gcloc = xm_protect(lst);
      fallbacks = (char **)CALLOC(len + 1, sizeof(char *)); /* +1 for null termination */
      for (i = 0; i < len; i++, lst = XEN_CDR(lst)) 
	fallbacks[i] = strdup(XEN_TO_C_STRING(XEN_CAR(lst)));
      xm_unprotect_at(gcloc);
    }
  args = XEN_TO_C_Args(arg8);
  arglen = XEN_LIST_LENGTH(arg8) / 2;
  res = XtOpenApplication(&app, 
			  XEN_TO_C_STRING(arg1), 
			  NULL,
			  0,
			  &argc,
			  argv,
			  fallbacks,
			  XEN_TO_C_WidgetClass(arg7), 
			  args, 
			  arglen);
  if (args)
    {
      fixup_args(res, args, arglen);
      free_args(args, arglen);
    }
  if (fallbacks) 
    {
      for (i = 0; i < len; i++)
	if (fallbacks[i]) free(fallbacks[i]);
      FREE(fallbacks);
    }
  return(XEN_LIST_3(C_TO_XEN_Widget(res), 
		    C_TO_XEN_XtAppContext(app),
		    gxm_argv_to_list(XEN_EMPTY_LIST, argc, argv)));
}

static XEN gxm_XtOpenApplication(XEN arg1, XEN arg4, XEN arg5, XEN arg7, XEN arg8, XEN arg9)
{
  #define H_XtOpenApplication "Widget XtOpenApplication(application_class, argc_in_out, argv_in_out, \
widget_class, args, num_args) calls XtToolkitInitialize followed by XtCreateApplicationContext , then calls XtOpenDisplay \
with display_string NULL and application_name NULL, and finally calls XtAppCreateShell with appcation_name NULL, widget_class \
applicationShellWidgetClass ,and the specified args and num_args and returns the created shell.  num_args can also be a list \
of fallback resources."
  /* DIFF: XtOpenApplication [app] name {options numopts} {argc} argv resources class args argnum -> (list widget app), argc is int not int* options/num ignored
  */
  XtAppContext app;
  Arg *args;
  Widget res;
  int argc, arglen;
  char **argv;
  char **fallbacks = NULL;
  int i, len = 0;
  XEN lst;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtOpenApplication", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 2, "XtOpenApplication", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg5), arg5, 3, "XtOpenApplication", "list of String*");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg7), arg7, 4, "XtOpenApplication", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg8), arg8, 8, "XtOpenApplication", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg9) || XEN_LIST_P(arg9) || XEN_NOT_BOUND_P(arg9), arg9, 5, "XtOpenApplication", "int or list of strings"); /* num_args */
  argc = XEN_TO_C_INT(arg4);
  argv = XEN_TO_C_Strings(arg5, argc);
  args = XEN_TO_C_Args(arg8);
  if (XEN_INTEGER_P(arg9)) arglen = XEN_TO_C_INT(arg9); else arglen = XEN_LIST_LENGTH(arg8) / 2;
  if (XEN_LIST_P(arg9))
    {
      int gcloc;
      len = XEN_LIST_LENGTH(arg9);
      lst = XEN_COPY_ARG(arg9);
      gcloc = xm_protect(lst);
      fallbacks = (char **)CALLOC(len + 1, sizeof(char *)); /* +1 for null termination */
      for (i = 0; i < len; i++, lst = XEN_CDR(lst)) 
	fallbacks[i] = XEN_TO_C_STRING(XEN_CAR(lst));
      xm_unprotect_at(gcloc);
    }
  res = XtOpenApplication(&app, XEN_TO_C_STRING(arg1), 
			  NULL, 0, &argc,
			  argv, fallbacks, XEN_TO_C_WidgetClass(arg7), 
			  args, arglen);
  if (args)
    {
      fixup_args(res, args, arglen);
      free_args(args, arglen);
    }
  if (fallbacks) 
    {
      for (i = 0; i < len; i++)
	if (fallbacks[i]) free(fallbacks[i]);
      FREE(fallbacks);
    }
  return(XEN_LIST_3(C_TO_XEN_Widget(res), 
		    C_TO_XEN_XtAppContext(app),
		    gxm_argv_to_list(XEN_EMPTY_LIST, argc, argv)));
}

static XEN gxm_XtDisplayInitialize(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg7, XEN arg8)
{
  #define H_XtDisplayInitialize "void XtDisplayInitialize(app_context, display, application_name, application_class, argc, argv) \
builds the resource database, calls the Xlib XrmParseCommand to parse the command line, and performs other per display initialization."
  /* DIFF: XtDisplayInitialize arg 5 6 ignored, argc is normal int, argv is list of strings, returns argv
   */
  char **argv;
  int argc;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtDisplayInitialize", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_Display_P(arg2), arg2, 2, "XtDisplayInitialize", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XtDisplayInitialize", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XtDisplayInitialize", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg7), arg7, 5, "XtDisplayInitialize", "int");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg8), arg8, 6, "XtDisplayInitialize", "list of char*");
  argc = XEN_TO_C_INT(arg7);
  argv = XEN_TO_C_Strings(arg8, argc);
  XtDisplayInitialize(XEN_TO_C_XtAppContext(arg1), 
		      XEN_TO_C_Display(arg2), 
		      XEN_TO_C_STRING(arg3), 
		      XEN_TO_C_STRING(arg4), 
		      NULL, 0, 
		      &argc, argv);
  return(gxm_argv_to_list(XEN_EMPTY_LIST, argc, argv));
}

/* -------- XtLanguage callback -------- */
/* (456) a global */

static XEN xm_language_proc = XEN_FALSE;

static String gxm_XtLanguageProc(Display* d, String s, XtPointer context) 
{
  char *res;
  res = XEN_TO_C_STRING(XEN_CALL_3(xm_language_proc,
				   C_TO_XEN_Display(d),
				   C_TO_XEN_STRING(s),
				   (XEN)context,
				   c__FUNCTION__));
  if (res)
    return(strdup(res));
  return(NULL);
}

static XEN gxm_XtSetLanguageProc(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtSetLanguageProc "Widget XtSetLanguageProc(app_context, proc, client_data)"
  /* arg can be null -> use default */
  /* DIFF: XtSetLanguageProc args1 and 2 use #f for NULL, return of #f means none was set
   */
  XEN previous_proc = XEN_FALSE;
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg1) || XEN_XtAppContext_P(arg1), arg1, 1, "XtSetLanguageProc", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg2) || (XEN_PROCEDURE_P(arg2) && (XEN_REQUIRED_ARGS_OK(arg2, 3))), arg2, 2, "XtSetLanguageProc", "XtLanguageProc");
  previous_proc = xm_language_proc;
  if (XEN_PROCEDURE_P(previous_proc))
    xm_unprotect(previous_proc);
  if (XEN_PROCEDURE_P(arg2))
    xm_protect(arg2);
  if (XEN_FALSE_P(arg1))
    {
      if (XEN_FALSE_P(arg2))
	XtSetLanguageProc(NULL, NULL, NULL);
      else XtSetLanguageProc(NULL,
			     (XtLanguageProc)gxm_XtLanguageProc, 
			     (XtPointer)arg3);
    }
  else 
    {
      if (XEN_FALSE_P(arg2))
	XtSetLanguageProc(XEN_TO_C_XtAppContext(arg1), NULL, NULL);
      else  XtSetLanguageProc(XEN_TO_C_XtAppContext(arg1), 
			      (XtLanguageProc)gxm_XtLanguageProc, 
			      (XtPointer)arg3);
    }
  return(previous_proc);
}

static XEN gxm_XtToolkitInitialize(void)
{
  #define H_XtToolkitInitialize "void XtToolkitInitialize()"
  XtToolkitInitialize();
  return(XEN_FALSE);
}

static XEN gxm_XtVaAppCreateShell(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtVaAppCreateShell "Widget XtVaAppCreateShell(application_name, application_class, widget_class, display, ...)"
  Arg *args;
  Widget w;
  int arglen;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtVaAppCreateShell", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtVaAppCreateShell", "char*");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg3), arg3, 3, "XtVaAppCreateShell", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_Display_P(arg4), arg4, 4, "XtVaAppCreateShell", "Display*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg5), arg5, 5, "XtVaAppCreateShell", "List");
  args = XEN_TO_C_Args(arg5);
  arglen = XEN_LIST_LENGTH(arg5) / 2;
  w = XtAppCreateShell(XEN_TO_C_STRING(arg1), 
		       XEN_TO_C_STRING(arg2), 
		       XEN_TO_C_WidgetClass(arg3), 
		       XEN_TO_C_Display(arg4), 
		       args,
		       arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XtAppCreateShell(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XtAppCreateShell "Widget XtAppCreateShell(application_name, application_class, widget_class, display, args, num_args) saves \
the specified application name and application class for qualifying all widget resource specifiers."
  Arg *args;
  Widget w;
  int arglen;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtAppCreateShell", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtAppCreateShell", "char*");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg3), arg3, 3, "XtAppCreateShell", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_Display_P(arg4), arg4, 4, "XtAppCreateShell", "Display*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg5), arg5, 5, "XtAppCreateShell", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg6), arg6, 6, "XtAppCreateShell", "int");
  args = XEN_TO_C_Args(arg5);
  arglen = XEN_TO_C_INT_DEF(arg6, arg5);
  w = XtAppCreateShell(XEN_TO_C_STRING(arg1), 
		       XEN_TO_C_STRING(arg2), 
		       XEN_TO_C_WidgetClass(arg3), 
		       XEN_TO_C_Display(arg4), 
		       args, 
		       arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_TO_XEN_Widget(w));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtCreateApplicationShell(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XtCreateApplicationShell "Widget XtCreateApplicationShell(name, widget_class, args, num_args) calls XtAppCreateShell with the \
application NULL, the application class passed to XtInitialize , and the default application context created by XtInitialize. This \
routine has been replaced by XtAppCreateShell."
  Arg *args;
  Widget w;
  int arglen;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtCreateApplicationShell", "char*");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg2), arg2, 2, "XtCreateApplicationShell", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XtCreateApplicationShell", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg4), arg4, 4, "XtCreateApplicationShell", "int");
  args = XEN_TO_C_Args(arg3);
  arglen = XEN_TO_C_INT_DEF(arg4, arg3);
  w = XtCreateApplicationShell(XEN_TO_C_STRING(arg1), 
			       XEN_TO_C_WidgetClass(arg2), 
			       args, arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_TO_XEN_Widget(w));
}
#endif

static XEN gxm_XtVaCreateManagedWidget(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XtVaCreateManagedWidget "Widget XtVaCreateManagedWidget(name, widget_class, parent, ...)"
  Arg *args;
  Widget w;
  int arglen;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtVaCreateManagedWidget", "char*");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg2), arg2, 2, "XtVaCreateManagedWidget", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg3), arg3, 3, "XtVaCreateManagedWidget", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XtVaCreateManagedWidget", "List");
  args = XEN_TO_C_Args(arg4);
  arglen = XEN_LIST_LENGTH(arg4) / 2;
  w = XtCreateManagedWidget(XEN_TO_C_STRING(arg1),
			    XEN_TO_C_WidgetClass(arg2), 
			    XEN_TO_C_Widget(arg3), 
			    args,
			    arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XtVaCreateWidget(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XtVaCreateWidget "Widget XtVaCreateWidget(name, widget_class, parent, ...)"
  Arg *args;
  int arglen;
  Widget w;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtVaCreateWidget", "char*");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg2), arg2, 2, "XtVaCreateWidget", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg3), arg3, 3, "XtVaCreateWidget", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XtVaCreateWidget", "List");
  args = XEN_TO_C_Args(arg4);
  arglen = XEN_LIST_LENGTH(arg4) / 2;
  w = XtCreateWidget(XEN_TO_C_STRING(arg1), 
		     XEN_TO_C_WidgetClass(arg2), 
		     XEN_TO_C_Widget(arg3), 
		     args,
		     arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XtCreateManagedWidget(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtCreateManagedWidget "Widget XtCreateManagedWidget(name, widget_class, parent, args, num_args) is a routine \
that calls XtCreateWidget and XtManageChild."
  Arg *args;
  Widget w;
  int arglen;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtCreateManagedWidget", "char*");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg2), arg2, 2, "XtCreateManagedWidget", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg3), arg3, 3, "XtCreateManagedWidget", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XtCreateManagedWidget", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg5), arg5, 5, "XtCreateManagedWidget", "int");
  args = XEN_TO_C_Args(arg4);
  arglen = XEN_TO_C_INT_DEF(arg5, arg4);
  w = XtCreateManagedWidget(XEN_TO_C_STRING(arg1), 
			    XEN_TO_C_WidgetClass(arg2), 
			    XEN_TO_C_Widget(arg3), 
			    args, arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XtCreateWidget(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtCreateWidget "Widget XtCreateWidget(name, widget_class, parent, args, num_args) performs much of the boilerplate operations of widget creation."
  Arg *args;
  Widget w;
  int arglen;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtCreateWidget", "char*");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg2), arg2, 2, "XtCreateWidget", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg3), arg3, 3, "XtCreateWidget", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XtCreateWidget", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg5), arg5, 5, "XtCreateWidget", "int");
  args = XEN_TO_C_Args(arg4);
  arglen = XEN_TO_C_INT_DEF(arg5, arg4);
  w = XtCreateWidget(XEN_TO_C_STRING(arg1), 
		     XEN_TO_C_WidgetClass(arg2), 
		     XEN_TO_C_Widget(arg3), 
		     args, arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XtCallbackPopdown(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtCallbackPopdown "void XtCallbackPopdown(w, client_data, call_data) calls XtPopdown with the specified shell_widget and then \
calls XtSetSensitive to resensitize the enable_widget."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtCallbackPopdown", "Widget");
  XtCallbackPopdown(XEN_TO_C_Widget(arg1), (XtPointer)arg2, (XtPointer)arg3);
  return(XEN_FALSE);
}

static XEN gxm_XtPopdown(XEN arg1)
{
  #define H_XtPopdown "void XtPopdown(popup_shell)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtPopdown", "Widget");
  XtPopdown(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtCallbackExclusive(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtCallbackExclusive "void XtCallbackExclusive(w, client_data, call_data)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtCallbackExclusive", "Widget");
  XtCallbackExclusive(XEN_TO_C_Widget(arg1), (XtPointer)arg2, (XtPointer)arg3);
  return(XEN_FALSE);
}

static XEN gxm_XtCallbackNonexclusive(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtCallbackNonexclusive "void XtCallbackNonexclusive(w, client_data, call_data)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtCallbackNonexclusive", "Widget");
  XtCallbackNonexclusive(XEN_TO_C_Widget(arg1), (XtPointer)arg2, (XtPointer)arg3);
  return(XEN_FALSE);
}

static XEN gxm_XtCallbackNone(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtCallbackNone "void XtCallbackNone(w, client_data, call_data)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtCallbackNone", "Widget");
  XtCallbackNone(XEN_TO_C_Widget(arg1), (XtPointer)arg2, (XtPointer)arg3);
  return(XEN_FALSE);
}

static XEN gxm_XtPopupSpringLoaded(XEN arg1)
{
  #define H_XtPopupSpringLoaded "void XtPopupSpringLoaded(popup_shell)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtPopupSpringLoaded", "Widget");
  XtPopupSpringLoaded(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtPopup(XEN arg1, XEN arg2)
{
  #define H_XtPopup "void XtPopup(popup_shell, grab_kind)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtPopup", "Widget");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XtPopup", "XtGrabKind");
  XtPopup(XEN_TO_C_Widget(arg1), (XtGrabKind)XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtVaCreatePopupShell(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XtVaCreatePopupShell "Widget XtVaCreatePopupShell(name, widget_class, parent, ...)"
  Arg *args;
  int arglen;
  Widget w;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtVaCreatePopupShell", "char*");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg2), arg2, 2, "XtVaCreatePopupShell", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg3), arg3, 3, "XtVaCreatePopupShell", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XtVaCreatePopupShell", "List");
  args = XEN_TO_C_Args(arg4);
  arglen = XEN_LIST_LENGTH(arg4) / 2;
  w = XtVaCreatePopupShell(XEN_TO_C_STRING(arg1), 
			   XEN_TO_C_WidgetClass(arg2), 
			   XEN_TO_C_Widget(arg3),
			   args,
			   arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XtCreatePopupShell(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtCreatePopupShell "Widget XtCreatePopupShell(name, widget_class, parent, args, num_args) ensures that the specified \
class is a subclass of Shell and, rather than using insert_child to attach the widget to the parent's children list, attaches the shell \
to the parent's pop-ups list directly."
  Widget w;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtCreatePopupShell", "char*");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg2), arg2, 2, "XtCreatePopupShell", "WidgetClass");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg3), arg3, 3, "XtCreatePopupShell", "Widget");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4), arg4, 4, "XtCreatePopupShell", "ArgList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg5), arg5, 5, "XtCreatePopupShell", "int");
  {
    Arg *args;
    int arglen;
    args = XEN_TO_C_Args(arg4);
    arglen = XEN_TO_C_INT_DEF(arg5, arg4);
    w = XtCreatePopupShell(XEN_TO_C_STRING(arg1), 
			   XEN_TO_C_WidgetClass(arg2), 
			   XEN_TO_C_Widget(arg3), 
			   args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_TO_XEN_Widget(w));
}

static XEN gxm_XtHasCallbacks(XEN arg1, XEN arg2)
{
  #define H_XtHasCallbacks "XtCallbackStatus XtHasCallbacks(w, callback_name) first checks to see if the widget has a callback \
list identified by callback_name; returns XtCallbackNoList or XtCallbackHasNone if none, else XtCallbackHasSome"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtHasCallbacks", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtHasCallbacks", "char*");
  return(C_TO_XEN_INT(XtHasCallbacks(XEN_TO_C_Widget(arg1), XEN_TO_C_STRING(arg2))));
}

static XEN gxm_XtCallCallbacks(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtCallCallbacks "void XtCallCallbacks(w, callback_name, call_data) calls each procedure that is registered in the \
specified widget's callback list. The call_data arg is assumed to be a callback struct reference"
  XtPointer val = NULL;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtCallCallbacks", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtCallCallbacks", "char*");
  if (XEN_LIST_P(arg3)) val = (XtPointer)XEN_TO_C_ULONG(XEN_CADR(arg3));
  XtCallCallbacks(XEN_TO_C_Widget(arg1), XEN_TO_C_STRING(arg2), val);
  return(XEN_FALSE);
}

static XEN gxm_XtRemoveAllCallbacks(XEN arg1, XEN arg2)
{
  #define H_XtRemoveAllCallbacks "void XtRemoveAllCallbacks(w, callback_name) removes all the callback procedures from the specified widget's callback list."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtRemoveAllCallbacks", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtRemoveAllCallbacks", "char*");
  XtRemoveAllCallbacks(XEN_TO_C_Widget(arg1), XEN_TO_C_STRING(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtRemoveCallback(XEN arg1, XEN arg2, XEN arg4)
{
  #define H_XtRemoveCallback "void XtRemoveCallback(w, callback_name, client_data) removes a callback"
  /* DIFF: XtRemoveCallback omits proc arg and is passed whatever XtAddCallback returned
   */
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtRemoveCallback", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtRemoveCallback", "char*"); 
  xm_unprotect_at(XEN_TO_C_INT(XEN_LIST_REF(arg4, CALLBACK_GC_LOC)));
  XtRemoveCallback(XEN_TO_C_Widget(arg1), XEN_TO_C_STRING(arg2), gxm_XtCallbackProc, (XtPointer)arg4);
  return(XEN_FALSE);
}

static XEN gxm_XtRemoveCallbacks(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtRemoveCallbacks "void XtRemoveCallbacks(w, callback_name, callbacks) removes the specified callback procedures from the \
specified widget's callback list. (The 3rd arg is a list of descriptors returned by XtAddCallback)."
  /* DIFF: XtRemoveCallbacks takes list of descriptors as arg3
   */
  XEN lst;
  int i, len;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtRemoveCallbacks", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtRemoveCallbacks", "char*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XtRemoveCallbacks", "list of XtCallbacks");
  lst = XEN_COPY_ARG(arg3);
  len = XEN_LIST_LENGTH(lst);
  for (i = 0; i < len; i++, lst = XEN_CDR(lst))
    gxm_XtRemoveCallback(arg1, arg2, XEN_CAR(lst));
  return(XEN_FALSE);
}

/* need these for all callback proc types, and locations to store the list of them */

enum {
  GXM_Print, GXM_Any, GXM_Arrow, GXM_Combo, GXM_Command, GXM_Container_Outline, GXM_Container_Select,
  GXM_Convert, GXM_Destination, GXM_Display, GXM_Drag_Drop, GXM_Drag_Motion, GXM_Drag_Proc, GXM_Drag_Start,
  GXM_Drawing, GXM_Drawn, GXM_Drop_Finish, GXM_Drop_Proc, GXM_DropSite_Enter, GXM_DropSite_Leave, GXM_Drop_Start,
  GXM_File, GXM_List, GXM_Notebook, GXM_Operation, GXM_Popup, GXM_PushButton, GXM_RowColumn, GXM_Scale,
  GXM_ScrollBar, GXM_Selection, GXM_SpinBox, GXM_ToggleButton, GXM_TopLevel_Enter, GXM_TopLevel_Leave,
  GXM_Traverse, GXM_Verify, GXM_DataField, GXM_MultiList, GXM_TabStack
};

static XEN wrap_callback_struct(int type, XtPointer info)
{
  switch (type)
    {
    case GXM_Any: return(C_TO_XEN_XmAnyCallbackStruct((XmAnyCallbackStruct *)info));
    case GXM_Arrow: return(C_TO_XEN_XmArrowButtonCallbackStruct((XmArrowButtonCallbackStruct *)info));
    case GXM_Command: return(C_TO_XEN_XmCommandCallbackStruct((XmCommandCallbackStruct *)info));
    case GXM_Drag_Drop: return(C_TO_XEN_XmDragDropFinishCallbackStruct((XmDragDropFinishCallbackStruct *)info));
    case GXM_Drag_Motion: return(C_TO_XEN_XmDragMotionCallbackStruct((XmDragMotionCallbackStruct *)info));
    case GXM_Drag_Proc: return(C_TO_XEN_XmDragProcCallbackStruct((XmDragProcCallbackStruct *)info));
    case GXM_Drawing: return(C_TO_XEN_XmDrawingAreaCallbackStruct((XmDrawingAreaCallbackStruct *)info));
    case GXM_Drawn: return(C_TO_XEN_XmDrawnButtonCallbackStruct((XmDrawnButtonCallbackStruct *)info));
    case GXM_Drop_Finish: return(C_TO_XEN_XmDropFinishCallbackStruct((XmDropFinishCallbackStruct *)info));
    case GXM_Drop_Proc: return(C_TO_XEN_XmDropProcCallbackStruct((XmDropProcCallbackStruct *)info));
    case GXM_DropSite_Enter: return(C_TO_XEN_XmDropSiteEnterCallbackStruct((XmDropSiteEnterCallbackStruct *)info));
    case GXM_DropSite_Leave: return(C_TO_XEN_XmDropSiteLeaveCallbackStruct((XmDropSiteLeaveCallbackStruct *)info));
    case GXM_Drop_Start: return(C_TO_XEN_XmDropStartCallbackStruct((XmDropStartCallbackStruct *)info));
    case GXM_File: return(C_TO_XEN_XmFileSelectionBoxCallbackStruct((XmFileSelectionBoxCallbackStruct *)info));
    case GXM_List: return(C_TO_XEN_XmListCallbackStruct((XmListCallbackStruct *)info));
    case GXM_PushButton: return(C_TO_XEN_XmPushButtonCallbackStruct((XmPushButtonCallbackStruct *)info));
    case GXM_RowColumn: return(C_TO_XEN_XmRowColumnCallbackStruct((XmRowColumnCallbackStruct *)info));
    case GXM_Scale: return(C_TO_XEN_XmScaleCallbackStruct((XmScaleCallbackStruct *)info));
    case GXM_ScrollBar: return(C_TO_XEN_XmScrollBarCallbackStruct((XmScrollBarCallbackStruct *)info));
    case GXM_Selection: return(C_TO_XEN_XmSelectionBoxCallbackStruct((XmSelectionBoxCallbackStruct *)info));
    case GXM_ToggleButton: return(C_TO_XEN_XmToggleButtonCallbackStruct((XmToggleButtonCallbackStruct *)info));
    case GXM_Verify: return(C_TO_XEN_XmTextVerifyCallbackStruct((XmTextVerifyCallbackStruct *)info));
#if MOTIF_2
    case GXM_Popup: return(C_TO_XEN_XmPopupHandlerCallbackStruct((XmPopupHandlerCallbackStruct *)info));
    case GXM_Drag_Start: return(C_TO_XEN_XmDragStartCallbackStruct((XmDragStartCallbackStruct *)info));
    case GXM_Convert: return(C_TO_XEN_XmConvertCallbackStruct((XmConvertCallbackStruct *)info));
    case GXM_Container_Outline: return(C_TO_XEN_XmContainerOutlineCallbackStruct((XmContainerOutlineCallbackStruct *)info));
    case GXM_Container_Select: return(C_TO_XEN_XmContainerSelectCallbackStruct((XmContainerSelectCallbackStruct *)info));
    case GXM_Destination: return(C_TO_XEN_XmDestinationCallbackStruct((XmDestinationCallbackStruct *)info));
    case GXM_Display: return(C_TO_XEN_XmDisplayCallbackStruct((XmDisplayCallbackStruct *)info));
    case GXM_Combo: return(C_TO_XEN_XmComboBoxCallbackStruct((XmComboBoxCallbackStruct *)info));
    case GXM_Notebook: return(C_TO_XEN_XmNotebookCallbackStruct((XmNotebookCallbackStruct *)info));
    case GXM_Operation: return(C_TO_XEN_XmOperationChangedCallbackStruct((XmOperationChangedCallbackStruct *)info));
    case GXM_SpinBox: return(C_TO_XEN_XmSpinBoxCallbackStruct((XmSpinBoxCallbackStruct *)info));
#if HAVE_XM_XP
    case GXM_Print: return(C_TO_XEN_XmPrintShellCallbackStruct((XmPrintShellCallbackStruct *)info));
#endif
    case GXM_TopLevel_Enter: return(C_TO_XEN_XmTopLevelEnterCallbackStruct((XmTopLevelEnterCallbackStruct *)info));
    case GXM_TopLevel_Leave: return(C_TO_XEN_XmTopLevelLeaveCallbackStruct((XmTopLevelLeaveCallbackStruct *)info));
    case GXM_Traverse: return(C_TO_XEN_XmTraverseObscuredCallbackStruct((XmTraverseObscuredCallbackStruct *)info));
#endif
#if HAVE_XmCreateDataField
    case GXM_DataField: return(C_TO_XEN_XmDataFieldCallbackStruct((XmDataFieldCallbackStruct *)info));
#endif
#if HAVE_XmCreateMultiList
    case GXM_MultiList: return(C_TO_XEN_XmMultiListCallbackStruct((XmMultiListCallbackStruct *)info));
#endif
#if HAVE_XmCreateTabStack
    case GXM_TabStack: return(C_TO_XEN_XmTabStackCallbackStruct((XmTabStackCallbackStruct *)info));
#endif
    }
  return(XEN_FALSE);
}

static int callback_struct_type(Widget w, char *name)
{
#if MOTIF_2
  if (strcmp(name, XmNdestinationCallback) == 0) return(GXM_Destination);
  if (strcmp(name, XmNpopupHandlerCallback) == 0) return (GXM_Popup);
  if (strcmp(name, XmNconvertCallback) == 0) return(GXM_Convert);
  if (strcmp(name, XmNdragStartCallback) == 0) return(GXM_Drag_Start);
#endif
  if (strcmp(name, XmNdropProc) == 0) return(GXM_Drop_Proc);      
  if (strcmp(name, XmNdragProc) == 0) return(GXM_Drag_Proc);      

  if (XmIsArrowButton(w)) return(GXM_Arrow);
  if (XmIsPushButton(w)) return(GXM_PushButton);
  if (XmIsScale(w)) return(GXM_Scale);
  if (XmIsRowColumn(w)) return(GXM_RowColumn);
  if (XmIsScrollBar(w)) return(GXM_ScrollBar);
  if (XmIsToggleButton(w)) return(GXM_ToggleButton);
  if (XmIsList(w)) return(GXM_List);
  if (XmIsFileSelectionBox(w)) return(GXM_File);
  if (XmIsDrawingArea(w)) return(GXM_Drawing);
  if (XmIsScrolledWindow(w)) return(GXM_Traverse);
#if MOTIF_2
  if (XmIsNotebook(w)) return(GXM_Notebook);
  if (XmIsComboBox(w)) return(GXM_Combo);
#endif
  if ((XmIsText(w)) || (XmIsTextField(w)))
    {
      if ((strcmp(name, XmNlosingFocusCallback) == 0) ||
	  (strcmp(name, XmNmodifyVerifyCallback) == 0) ||
	  (strcmp(name, XmNmotionVerifyCallback) == 0))
	return(GXM_Verify);
      return(GXM_Any);
    }
#if MOTIF_2
  if (XmIsContainer(w))
    {
      if (strcmp(name, XmNoutlineChangedCallback) == 0) return(GXM_Container_Outline);
      if ((strcmp(name, XmNselectionCallback) == 0) ||
	  (strcmp(name, XmNdefaultActionCallback) == 0))
	return(GXM_Container_Select);
      return(GXM_Destination); 
    }
  /* how to recognize a SpinBox? */
#if HAVE_XM_XP
  if (XmIsPrintShell(w)) return(GXM_Print);
#endif
#endif
  if (XmIsCommand(w)) return(GXM_Command);
  if (XmIsDisplay(w)) return(GXM_Display);
  if (XmIsSelectionBox(w)) return(GXM_Selection);
  if (XmIsDragContext(w))
    {
      if (strcmp(name, XmNdragDropFinishCallback) == 0) return(GXM_Drag_Drop);
      if (strcmp(name, XmNdropFinishCallback) == 0) return(GXM_Drop_Finish);
      if (strcmp(name, XmNdragMotionCallback) == 0) return(GXM_Drag_Motion);
      if (strcmp(name, XmNdropSiteEnterCallback) == 0) return(GXM_DropSite_Enter);
      if (strcmp(name, XmNdropSiteLeaveCallback) == 0) return(GXM_DropSite_Leave);
      if (strcmp(name, XmNdropStartCallback) == 0) return(GXM_Drop_Start);
      if (strcmp(name, XmNoperationChangedCallback) == 0) return(GXM_Operation);
      if (strcmp(name, XmNtopLevelEnterCallback) == 0) return(GXM_TopLevel_Enter);
      if (strcmp(name, XmNtopLevelLeaveCallback) == 0) return(GXM_TopLevel_Leave);
    }
  if (XmIsDrawnButton(w)) return(GXM_Drawn);
#if HAVE_XmCreateDataField
  if (XmIsDataField(w)) return(GXM_DataField);
#endif
  return(GXM_Any);
}

static XEN gxm_XtAddCallback(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XtAddCallback "void XtAddCallback(w, callback_name, callback, client_data) adds the specified callback procedure to \
the specified widget's callback list.  In xm, the client-data is optional, defaulting to #f. The callback procedure takes \
3 args: widget client-data callback-info. Returns a description of the callback suitable for use with XtRemoveCallback."
  /* DIFF: XtAddCallback returns the C-side "client-data" (for subsequent XtRemoveCallback)
   */
  char *name;
  Widget w;
  int gc_loc;
  XEN call_descr = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtAddCallback", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtAddCallback", "char*");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg3) && (XEN_REQUIRED_ARGS_OK(arg3, 3)), arg3, 3, "XtAddCallback", "(XtCallbackProc widget data callb)");
  w = XEN_TO_C_Widget(arg1);
  name = XEN_TO_C_STRING(arg2);
  call_descr = C_TO_XEN_XM_XtCallback(arg3, (XEN_BOUND_P(arg4)) ? arg4 : XEN_FALSE);
  gc_loc = xm_protect(call_descr);
  XEN_LIST_SET(call_descr, CALLBACK_GC_LOC, C_TO_XEN_INT(gc_loc));
  XEN_LIST_SET(call_descr, CALLBACK_STRUCT_TYPE, C_TO_XEN_INT(callback_struct_type(w, name)));
  XtAddCallback(w, name, gxm_XtCallbackProc, (XtPointer)call_descr);
  return(xen_return_first(call_descr, arg2));
}

static XEN gxm_XtAddCallbacks(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtAddCallbacks "void XtAddCallbacks(w, callback_name, callbacks) add the specified list of callbacks to the specified widget's callback list. \
It returns a list of callback descriptors for use with XtRemoveCallback(s)."
  /* DIFF: XtAddCallbacks takes list of (func data) pairs as arg3
   */
  XEN res = XEN_EMPTY_LIST, lst;
  int i, len;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtAddCallbacks", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtAddCallbacks", "char*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XtAddCallbacks", "list of XtCallbacks");
  len = XEN_LIST_LENGTH(arg3);
  lst = XEN_COPY_ARG(arg3);
  for (i = len - 1; i >= 0; i--, lst = XEN_CDR(lst))
    res = XEN_CONS(gxm_XtAddCallback(arg1, arg2, XEN_CAR(XEN_CAR(lst)), XEN_CADR(XEN_CAR(lst))),
		   res);
  return(res);
}

static XEN gxm_XtParent(XEN arg1)
{
  #define H_XtParent "Widget XtParent(w) returns the widget's parent widget ID."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtParent", "Widget");
  return(C_TO_XEN_Widget(XtParent(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtClass(XEN arg1)
{
  #define H_XtClass "WidgetClass XtClass(w)"
  #define H_XtSuperclass "WidgetClass XtSuperclass(w) returns a pointer to the widget's superclass class structure."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtClass", "Widget");
  return(C_TO_XEN_WidgetClass(XtClass(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtSuperclass(XEN arg1)
{
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtSuperclass", "Widget");
  return(C_TO_XEN_WidgetClass(XtSuperclass(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtName(XEN arg1)
{
  #define H_XtName "Widget XtName(w) returns the widget's name."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtName", "Widget");
  return(C_TO_XEN_STRING(XtName(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtWindowOfObject(XEN arg1)
{
  #define H_XtWindowOfObject "Window XtWindowOfObject(object) returns the window of the specified object."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtWindowOfObject", "Widget");
  return(C_TO_XEN_Window(XtWindowOfObject(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtWindow(XEN arg1)
{
  #define H_XtWindow "Window XtWindow(w) returns the window of the specified widget."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtWindow", "Widget");
  return(C_TO_XEN_Window(XtWindow(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtScreenOfObject(XEN arg1)
{
  #define H_XtScreenOfObject "Screen *XtScreenOfObject(object) returns the screen pointer for the specified object."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtScreenOfObject", "Widget");
  return(C_TO_XEN_Screen(XtScreenOfObject(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtScreen(XEN arg1)
{
  #define H_XtScreen "Screen* XtScreen(w) returns the screen pointer for the specified widget."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtScreen", "Widget");
  return(C_TO_XEN_Screen(XtScreen(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtDisplayOfObject(XEN arg1)
{
  #define H_XtDisplayOfObject "Display *XtDisplayOfObject(object) returns the display pointer for the specified object."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtDisplayOfObject", "Widget");
  return(C_TO_XEN_Display(XtDisplayOfObject(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtDisplay(XEN arg1)
{
  #define H_XtDisplay "Display* XtDisplay(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtDisplay", "Widget");
  return(C_TO_XEN_Display(XtDisplay(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtVaCreateArgsList(XEN arg1, XEN ignore2)
{
  #define H_XtVaCreateArgsList "XtVarArgsList XtVaCreateArgsList(unused, ...) allocates memory and copies its arguments into a single \
list pointer, which may be used with XtVaNestedList."
  /* DIFF: XtVaCreateArgsList just returns its 1st arg
   */
  return(arg1);
}

static XEN gxm_XtMergeArgLists(XEN arg1, XEN ignore2, XEN arg3, XEN ignore4)
{
  #define H_XtMergeArgLists "ArgList XtMergeArgLists(args1, num_args1, args2, num_args2) allocates enough storage to hold the combined \
ArgList structures and copies them into it."
  /* just merges, not duplicate check, kinda dumb to drop into C for that */
  XEN_ASSERT_TYPE(XEN_LIST_P(arg1), arg1, 1, "XtMergeArgLists", "list");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg3), arg3, 3, "XtMergeArgLists", "list");
  return(XEN_APPEND(arg1, arg3));
}

static XEN gxm_XtWindowToWidget(XEN arg1, XEN arg2)
{
  #define H_XtWindowToWidget "Widget XtWindowToWidget(display, window) translates the specified window and display pointer into the appropriate widget instance."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtWindowToWidget", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XtWindowToWidget", "Window");
  return(C_TO_XEN_Widget(XtWindowToWidget(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XtNameToWidget(XEN arg1, XEN arg2)
{
  #define H_XtNameToWidget "Widget XtNameToWidget(reference, names) looks for a widget whose name is the first component in the specified names"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtNameToWidget", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtNameToWidget", "char*");
  return(C_TO_XEN_Widget(XtNameToWidget(XEN_TO_C_Widget(arg1), XEN_TO_C_STRING(arg2))));
}

static XEN gxm_XtSetSensitive(XEN arg1, XEN arg2)
{
  #define H_XtSetSensitive "void XtSetSensitive(w, sensitive)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtSetSensitive", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XtSetSensitive", "boolean");
  XtSetSensitive(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtDestroyWidget(XEN arg1)
{
  #define H_XtDestroyWidget "void XtDestroyWidget(w) provides the only method of destroying a widget, including widgets that need to destroy themselves."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtDestroyWidget", "Widget");
  XtDestroyWidget(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtUnrealizeWidget(XEN arg1)
{
  #define H_XtUnrealizeWidget "void XtUnrealizeWidget(w) destroys the windows of an existing widget and all of its children (recursively down the widget tree)."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtUnrealizeWidget", "Widget");
  XtUnrealizeWidget(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtRealizeWidget(XEN arg1)
{
  #define H_XtRealizeWidget "void XtRealizeWidget(w) maps the widget window."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtRealizeWidget", "Widget");
  XtRealizeWidget(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtAppPending(XEN arg1)
{
  #define H_XtAppPending "XtInputMask XtAppPending(app_context) returns a nonzero value if there are events pending from the X server, timer \
pending, or other input sources pending. "
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppPending", "XtAppContext");
  return(C_TO_XEN_ULONG(XtAppPending(XEN_TO_C_XtAppContext(arg1))));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtPending(void)
{
  #define H_XtPending "XtInputMask XtPending() has been replaced by XtAppPending."
  return(C_TO_XEN_BOOLEAN(XtPending()));
}
#endif

static XEN gxm_XtAppNextEvent(XEN arg1)
{
  #define H_XtAppNextEvent "void XtAppNextEvent(app_context) flushes the X output buffers of each Display in the application \
context and waits for an event while looking at the other input sources, timeout timeout values, and signal handlers and calling any callback \
procedures triggered by them -> event."
  /* DIFF: XtAppNextEvent app [ev] -> ev
   */
  XEvent *e;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppNextEvent", "XtAppContext");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  XtAppNextEvent(XEN_TO_C_XtAppContext(arg1), e);
  return(C_TO_XEN_XEvent_OBJ(e));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtNextEvent(void)
{
  #define H_XtNextEvent "void XtNextEvent() has been replaced by XtAppNextEvent."
  /* DIFF: XtNextEvent [ev] -> ev
   */
  XEvent *e;
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  XtNextEvent(e);
  return(C_TO_XEN_XEvent_OBJ(e));
}
#endif


/* -------- Input callback -------- */

/* (77) explicitly removed via XtRemoveInput */

#define C_TO_XEN_XM_Input(Code, Context) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("Input"), Code, Context, XEN_ZERO, XEN_ZERO)
#define XM_Input_P(Arg) WRAP_P("Input", Arg)

static void gxm_XtInputCallbackProc(XtPointer cdata, int *fileno, XtInputId *id) 
{
  XEN descr = (XEN)cdata;
  /* (list 'Input function context gc-loc id) */
  XEN_CALL_3(XEN_CADR(descr), 
	     XEN_CADDR(descr), 
	     C_TO_XEN_ULONG(*fileno), 
	     C_TO_XEN_XtInputId(*id), 
	     c__FUNCTION__);
}

static bool unprotect_inputproc(XEN val, int loc, unsigned long id)
{
  if ((XM_Input_P(val)) &&
      ((XtInputId)XEN_TO_C_ULONG(XEN_LIST_REF(val, 4)) == id))
    {
      xm_unprotect_at(loc);
      return(true);
    }
  return(false);
}

static XEN gxm_XtRemoveInput(XEN arg1)
{
  #define H_XtRemoveInput "void XtRemoveInput(id) causes the read routine to stop watching for input from the input source."
  XtInputId id;
  XEN_ASSERT_TYPE(XEN_XtInputId_P(arg1), arg1, 1, "XtRemoveInput", "XtInputId");
  id = XEN_TO_C_XtInputId(arg1);
  XtRemoveInput(id);
  map_over_protected_elements(unprotect_inputproc, id);
  return(XEN_FALSE);
}

static XEN gxm_XtAppAddInput(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtAppAddInput "XtInputId XtAppAddInput(app_context, source, condition, proc, client_data) registers with the read routine a \
new source of events, which is usually file input but can also be file output."
  XtInputId id;
  int gc_loc;
  XEN descr;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppAddInput", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XtAppAddInput", "int");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 3)), arg4, 4, "XtAppAddInput", "(XtInputCallbackProc data fileno id)");
  descr = C_TO_XEN_XM_Input(arg4, (XEN_BOUND_P(arg5)) ? arg5 : XEN_FALSE);
  gc_loc = xm_protect(descr);
  id = XtAppAddInput(XEN_TO_C_XtAppContext(arg1), 
		     XEN_TO_C_INT(arg2), 
		     (XtPointer)arg3, 
		     gxm_XtInputCallbackProc, 
		     (XtPointer)descr);
  XEN_LIST_SET(descr, 3, C_TO_XEN_INT(gc_loc));
  XEN_LIST_SET(descr, 4, C_TO_XEN_ULONG(id));
  return(C_TO_XEN_XtInputId(id));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtAddInput(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XtAddInput "XtInputId XtAddInput(source, condition, proc, client_data) has been replaced by XtAppAddInput."
  XtInputId id;
  int gc_loc;
  XEN descr;
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg1), arg1, 1, "XtAddInput", "int");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg3) && (XEN_REQUIRED_ARGS_OK(arg3, 3)), arg3, 3, "XtAddInput", "(XtInputCallbackProc data fileno id)");
  descr = C_TO_XEN_XM_Input(arg3, (XEN_BOUND_P(arg4)) ? arg4 : XEN_FALSE);
  gc_loc = xm_protect(descr);
  id = XtAddInput(XEN_TO_C_INT(arg1), 
		  (XtPointer)arg2, 
		  gxm_XtInputCallbackProc, 
		  (XtPointer)descr);
  XEN_LIST_SET(descr, 3, C_TO_XEN_INT(gc_loc));
  XEN_LIST_SET(descr, 4, C_TO_XEN_ULONG(id));
  return(C_TO_XEN_XtInputId(id));
}
#endif

/* -------- Timer Callback -------- */
/* (79) protect the function, then unprotect after invocation */

#define C_TO_XEN_XM_TimeOut(Code, Context) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("TimeOut"), Code, Context, XEN_ZERO, XEN_ZERO)
#define XM_TimeOut_P(Arg) WRAP_P("TimeOut", Arg)

static void gxm_XtTimerCallbackProc(XtPointer cdata, XtIntervalId* i) 
{
  XEN descr = (XEN)cdata;
  /* (list 'TimeOut function context gc-loc id) */
  int gc_loc;
  gc_loc = XEN_TO_C_INT(XEN_LIST_REF(descr, 3));
  XEN_CALL_2(XEN_CADR(descr), 
	     XEN_CADDR(descr),
	     C_TO_XEN_XtIntervalId(*i),
	     c__FUNCTION__);
  xm_unprotect_at(gc_loc);
}

static bool unprotect_timeoutproc(XEN val, int loc, unsigned long id)
{
  if ((XM_TimeOut_P(val)) &&
      ((XtIntervalId)XEN_TO_C_ULONG(XEN_LIST_REF(val, 4)) == id))
    {
      xm_unprotect_at(loc);
      return(true);
    }
  return(false);
}

static XEN gxm_XtRemoveTimeOut(XEN arg1)
{
  #define H_XtRemoveTimeOut "void XtRemoveTimeOut(timer) removes the timeout."
  XtIntervalId id;
  XEN_ASSERT_TYPE(XEN_XtIntervalId_P(arg1), arg1, 1, "XtRemoveTimeOut", "XtIntervalId");
  id = XEN_TO_C_XtIntervalId(arg1);
  XtRemoveTimeOut(id);
  map_over_protected_elements(unprotect_timeoutproc, id);
  return(XEN_FALSE);
}

static XEN gxm_XtAppAddTimeOut(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XtAppAddTimeOut "XtIntervalId XtAppAddTimeOut(app_context, interval, proc, client_data) creates a timeout and returns an identifier for it."
  XtIntervalId id;
  int gc_loc;
  XEN descr;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppAddTimeOut", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtAppAddTimeOut", "ulong");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg3) && (XEN_REQUIRED_ARGS_OK(arg3, 2)), arg3, 3, "XtAppAddTimeOut", "(XtTimerCallbackProc data id)");
  descr = C_TO_XEN_XM_TimeOut(arg3, (XEN_BOUND_P(arg4)) ? arg4 : XEN_FALSE);
  gc_loc = xm_protect(descr);
  id = XtAppAddTimeOut(XEN_TO_C_XtAppContext(arg1), 
		       XEN_TO_C_ULONG(arg2), 
		       gxm_XtTimerCallbackProc, 
		       (XtPointer)descr);
  XEN_LIST_SET(descr, 3, C_TO_XEN_INT(gc_loc));
  XEN_LIST_SET(descr, 4, C_TO_XEN_ULONG(id));
  return(C_TO_XEN_XtIntervalId(id));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtAddTimeOut(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtAddTimeOut "XtIntervalId XtAddTimeOut(interval, proc, client_data) has been replaced by XtAppAddTimeOut."
  XtIntervalId id;
  int gc_loc;
  XEN descr;
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, 1, "XtAddTimeOut", "ulong");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg2) && (XEN_REQUIRED_ARGS_OK(arg2, 2)), arg2, 2, "XtAddTimeOut", "(XtTimerCallbackProc data id)");
  descr = C_TO_XEN_XM_TimeOut(arg2, (XEN_BOUND_P(arg3)) ? arg3 : XEN_FALSE);
  gc_loc = xm_protect(descr);
  id = XtAddTimeOut(XEN_TO_C_ULONG(arg1), 
		    gxm_XtTimerCallbackProc, 
		    (XtPointer)descr);
  XEN_LIST_SET(descr, 3, C_TO_XEN_INT(gc_loc));
  XEN_LIST_SET(descr, 4, C_TO_XEN_ULONG(id));
  return(C_TO_XEN_XtIntervalId(id));
}
#endif

static XEN gxm_XtLastTimestampProcessed(XEN arg1)
{
  #define H_XtLastTimestampProcessed "Time XtLastTimestampProcessed(display) returns the timestamp of the last event"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtLastTimestampProcessed", "Display*");
  return(C_TO_XEN_Time(XtLastTimestampProcessed(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XtLastEventProcessed(XEN arg1)
{
  #define H_XtLastEventProcessed "XEvent* XtLastEventProcessed(display) returns the last event passed to XtDispatchEvent for the \
specified display and NULL if there has been no event. "
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtLastEventProcessed", "Display*");
  return(C_TO_XEN_XEvent(XtLastEventProcessed(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XtGetKeyboardFocusWidget(XEN arg1)
{
  #define H_XtGetKeyboardFocusWidget "Widget XtGetKeyboardFocusWidget(widget) returns the widget that would be the end result of keyboard \
event forwarding for a keyboard event for the specified widget."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtGetKeyboardFocusWidget", "Widget");
  return(C_TO_XEN_Widget(XtGetKeyboardFocusWidget(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtSetKeyboardFocus(XEN arg1, XEN arg2)
{
  #define H_XtSetKeyboardFocus "XtSetKeyboardFocus(subtree descendant) causes XtDispatchEvent to remap and send the event to the specified \
descendant widget."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtSetKeyboardFocus", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2), arg2, 2, "XtSetKeyboardFocus", "Widget");
  XtSetKeyboardFocus(XEN_TO_C_Widget(arg1), XEN_TO_C_Widget(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtAddExposureToRegion(XEN arg1, XEN arg2)
{
  #define H_XtAddExposureToRegion "void XtAddExposureToRegion(event, region) computes the union of the rectangle defined by the exposure event \
and the specified region. Then, it stores the results back in region."
  XEN_ASSERT_TYPE(XEN_XEvent_P(arg1), arg1, 1, "XtAddExposureToRegion", "XEvent*");
  XEN_ASSERT_TYPE(XEN_Region_P(arg2), arg2, 2, "XtAddExposureToRegion", "Region");
  XtAddExposureToRegion(XEN_TO_C_XEvent(arg1), XEN_TO_C_Region(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtAppMainLoop(XEN arg1)
{

  #define H_XtAppMainLoop "void XtAppMainLoop(app_context) first reads the next incoming X event by calling XtAppNextEvent and then it dispatches \
the event to the appropriate registered procedure by calling XtDispatchEvent."
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppMainLoop", "XtAppContext");
  XtAppMainLoop(XEN_TO_C_XtAppContext(arg1));
  return(XEN_FALSE);
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtMainLoop(void)
{
  #define H_XtMainLoop "void XtMainLoop() has been replaced by XtAppMainLoop."
  XtMainLoop();
  return(XEN_FALSE);
}
#endif

static XEN gxm_XtAppProcessEvent(XEN arg1, XEN arg2)
{
  #define H_XtAppProcessEvent "void XtAppProcessEvent(app_context, mask) processes one timer, alternate input, signal source, or X event. \
If there is nothing of the appropriate type to process, XtAppProcessEvent blocks until there is."
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppProcessEvent", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtAppProcessEvent", "XtInputMask");
  XtAppProcessEvent(XEN_TO_C_XtAppContext(arg1), XEN_TO_C_ULONG(arg2));
  return(XEN_FALSE);
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtProcessEvent(XEN arg1)
{
  #define H_XtProcessEvent "void XtProcessEvent(mask) has been replaced by XtAppProcessEvent."
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg1), arg1, 1, "XtProcessEvent", "XtInputMask");
  XtProcessEvent(XEN_TO_C_ULONG(arg1));
  return(XEN_FALSE);
}
#endif

static XEN gxm_XtRemoveGrab(XEN arg1)
{
  #define H_XtRemoveGrab "void XtRemoveGrab(w) removes widgets from the modal cascade starting at the most recent widget up to and \
including the specified widget. "
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtRemoveGrab", "Widget");
  XtRemoveGrab(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtAddGrab(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtAddGrab "void XtAddGrab(w, exclusive, spring_loaded) appends the widget (and associated parameters) to the modal cascade and \
checks that exclusive is #t if spring_loaded is #t."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtAddGrab", "Widget");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XtAddGrab", "boolean");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XtAddGrab", "boolean");
  XtAddGrab(XEN_TO_C_Widget(arg1), XEN_TO_C_BOOLEAN(arg2), XEN_TO_C_BOOLEAN(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XtBuildEventMask(XEN arg1)
{
  #define H_XtBuildEventMask "EventMask XtBuildEventMask(w) -> widget's event mask"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtBuildEventMask", "Widget");
  return(C_TO_XEN_ULONG(XtBuildEventMask(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtDispatchEventToWidget(XEN arg1, XEN arg2)
{
  #define H_XtDispatchEventToWidget "Boolean XtDispatchEventToWidget(widget, event) scans the list of registered event handlers for the \
specified widget and calls each handler that has been registered for the specified event type, subject to the continue_to_dispatch value \
returned by each handler."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtDispatchEventToWidget", "Widget");
  XEN_ASSERT_TYPE(XEN_XEvent_P(arg2), arg2, 2, "XtDispatchEventToWidget", "XEvent*");
  return(C_TO_XEN_BOOLEAN(XtDispatchEventToWidget(XEN_TO_C_Widget(arg1), XEN_TO_C_XEvent(arg2))));
}

enum {EVENT_HANDLER_TYPE, EVENT_HANDLER_FUNC, EVENT_HANDLER_DATA, EVENT_HANDLER_GC_LOC, EVENT_HANDLER_WIDGET, EVENT_HANDLER_MASK};

#define C_TO_XEN_XM_XtEventHandler(Code, Context, Widget, Mask) \
  XEN_LIST_6(C_STRING_TO_XEN_SYMBOL("XtEventHandler"), Code, Context, XEN_ZERO, Widget, Mask)
#define XM_XtEventHandler_P(Arg) WRAP_P("XtEventHandler", Arg)

static void gxm_XtEventHandler(Widget w, XtPointer context, XEvent *event, Boolean *flag)
{
  XEN result;
  XEN descr = (XEN)context;
  result = XEN_CALL_4(XEN_LIST_REF(descr, EVENT_HANDLER_FUNC),
		      C_TO_XEN_Widget(w),
		      XEN_LIST_REF(descr, EVENT_HANDLER_DATA),
		      C_TO_XEN_XEvent(event),
		      C_TO_XEN_BOOLEAN(*flag),
		      c__FUNCTION__);
  if ((XEN_SYMBOL_P(result)) && (strcmp("done", XEN_SYMBOL_TO_C_STRING(result)) == 0))
    (*flag) = false;
}

static bool find_xteventproc_1(XEN val, int loc, unsigned long wd)
{
  XEN lst = (XEN)wd;
  unsigned long w;
  XEN code;
  XEN data;
  w = XEN_TO_C_ULONG(XEN_CAR(lst));
  code = XEN_CADR(lst);
  data = XEN_CADDR(lst);
  return((XM_XtEventHandler_P(val)) &&
	 (XEN_Widget_P(XEN_LIST_REF(val, EVENT_HANDLER_WIDGET))) &&
	 (XEN_TO_C_ULONG(XEN_CADR(XEN_LIST_REF(val, EVENT_HANDLER_WIDGET))) == w) &&
	 (XEN_EQ_P(code, XEN_LIST_REF(val, EVENT_HANDLER_FUNC))) &&
	 (XEN_EQ_P(data, XEN_LIST_REF(val, EVENT_HANDLER_DATA))));
}

static XEN find_xteventproc(Widget w, XEN code, XEN data)
{
  /* here we again have to go by the widget */
  XEN lst;
  int i, loc;
  lst = XEN_LIST_3(C_TO_XEN_ULONG((unsigned long)w),code, data);
  loc = xm_protect(lst);
  i = map_over_protected_elements(find_xteventproc_1, (unsigned long)lst);
  xm_unprotect_at(loc);
  if (i >= 0)
    return(xm_protected_element(i));
  return(XEN_FALSE);
}

static XEN gxm_XtInsertRawEventHandler(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XtInsertRawEventHandler "void XtInsertRawEventHandler(w, event_mask, nonmaskable, proc, client_data, position) is similar to \
XtInsertEventHandler except that it does not modify the widget's event mask and never causes an XSelectInput for the specified events."
  XEN call_descr = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtInsertRawEventHandler", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtInsertRawEventHandler", "EventMask");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XtInsertRawEventHandler", "boolean");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 4)), arg4, 4, "XtInsertRawEventHandler", "XtEventHandler");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XtInsertRawEventHandler", "XtListPosition");
  call_descr = C_TO_XEN_XM_XtEventHandler(arg4, (XEN_BOUND_P(arg5)) ? arg5 : XEN_FALSE, arg1, arg2);
  XEN_LIST_SET(call_descr, EVENT_HANDLER_GC_LOC, C_TO_XEN_INT(xm_protect(call_descr)));
  XtInsertRawEventHandler(XEN_TO_C_Widget(arg1), 
			  XEN_TO_C_ULONG(arg2), 
			  XEN_TO_C_BOOLEAN(arg3), 
			  gxm_XtEventHandler, 
			  (XtPointer)call_descr, 
			  (XtListPosition)XEN_TO_C_INT(arg6));
  return(XEN_FALSE);
}

static XEN gxm_XtInsertEventHandler(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XtInsertEventHandler "void XtInsertEventHandler(w, event_mask, nonmaskable, proc, client_data, position) is identical to \
XtAddEventHandler with the additional position argument. "
  XEN call_descr = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtInsertEventHandler", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtInsertEventHandler", "EventMask");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XtInsertEventHandler", "boolean");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 4)), arg4, 4, "XtInsertEventHandler", "XtEventHandler");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg6), arg6, 6, "XtInsertEventHandler", "XtListPosition");
  call_descr = C_TO_XEN_XM_XtEventHandler(arg4, (XEN_BOUND_P(arg5)) ? arg5 : XEN_FALSE, arg1, arg2);
  XEN_LIST_SET(call_descr, EVENT_HANDLER_GC_LOC, C_TO_XEN_INT(xm_protect(call_descr)));
  XtInsertEventHandler(XEN_TO_C_Widget(arg1), 
		       XEN_TO_C_ULONG(arg2), 
		       XEN_TO_C_BOOLEAN(arg3), 
		       gxm_XtEventHandler, 
		       (XtPointer)call_descr, 
		       (XtListPosition)XEN_TO_C_INT(arg6));
  return(XEN_FALSE);
}

static XEN gxm_XtRemoveRawEventHandler(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtRemoveRawEventHandler "void XtRemoveRawEventHandler(w, event_mask, nonmaskable, proc, client_data) stops the specified \
procedure from receiving the specified events."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtRemoveRawEventHandler", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtRemoveRawEventHandler", "EventMask");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XtRemoveRawEventHandler", "boolean");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 4)), arg4, 4, "XtRemoveRawEventHandler", "XtEventHandler");
  XtRemoveRawEventHandler(XEN_TO_C_Widget(arg1), 
			  XEN_TO_C_ULONG(arg2), 
			  XEN_TO_C_BOOLEAN(arg3),
			  gxm_XtEventHandler, 
			  (XtPointer)find_xteventproc(XEN_TO_C_Widget(arg1), arg4, arg5));
  return(XEN_FALSE);
}

static XEN gxm_XtAddRawEventHandler(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtAddRawEventHandler "void XtAddRawEventHandler(w, event_mask, nonmaskable, proc, client_data) is similar to XtAddEventHandler \
except that it does not affect the widget's mask and never causes an XSelectInput for its events."
  XEN call_descr = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtAddRawEventHandler", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtAddRawEventHandler", "EventMask");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XtAddRawEventHandler", "boolean");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 4)), arg4, 4, "XtAddRawEventHandler", "XtEventHandler");
  call_descr = C_TO_XEN_XM_XtEventHandler(arg4, (XEN_BOUND_P(arg5)) ? arg5 : XEN_FALSE, arg1, arg2);
  XEN_LIST_SET(call_descr, EVENT_HANDLER_GC_LOC, C_TO_XEN_INT(xm_protect(call_descr)));
  XtAddRawEventHandler(XEN_TO_C_Widget(arg1), 
		       XEN_TO_C_ULONG(arg2), 
		       XEN_TO_C_BOOLEAN(arg3), 
		       gxm_XtEventHandler,
		       (XtPointer)call_descr);
  return(XEN_FALSE);
}

static XEN gxm_XtRemoveEventHandler(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtRemoveEventHandler "XtRemoveEventHandler(w, event_mask, nonmaskable, proc, client_data)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtRemoveEventHandler", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtRemoveEventHandler", "EventMask");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XtRemoveEventHandler", "boolean");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 4)), arg4, 4, "XtRemoveEventHandler", "XtEventHandler");
  XtRemoveEventHandler(XEN_TO_C_Widget(arg1), 
		       XEN_TO_C_ULONG(arg2), 
		       XEN_TO_C_BOOLEAN(arg3), 
		       gxm_XtEventHandler, 
		       (XtPointer)find_xteventproc(XEN_TO_C_Widget(arg1), arg4, arg5));
  return(XEN_FALSE);
}

static XEN gxm_XtAddEventHandler(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtAddEventHandler "void XtAddEventHandler(w, event_mask, nonmaskable, proc, client_data) registers a procedure with the dispatch \
mechanism that is to be called when an event that matches the mask occurs on the specified widget.  To set the 'continue_to_dispatch' \
flag to 'false', return the symbol 'done from the event handler."
  XEN call_descr = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtAddEventHandler", "Widget");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg2), arg2, 2, "XtAddEventHandler", "EventMask");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg3), arg3, 3, "XtAddEventHandler", "boolean");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 4)), arg4, 4, "XtAddEventHandler", "XtEventHandler");
  call_descr = C_TO_XEN_XM_XtEventHandler(arg4, (XEN_BOUND_P(arg5)) ? arg5 : XEN_FALSE, arg1, arg2);
  XEN_LIST_SET(call_descr, EVENT_HANDLER_GC_LOC, C_TO_XEN_INT(xm_protect(call_descr)));
  XtAddEventHandler(XEN_TO_C_Widget(arg1), 
		    XEN_TO_C_ULONG(arg2), 
		    XEN_TO_C_BOOLEAN(arg3), 
		    gxm_XtEventHandler,
		    (XtPointer)call_descr);
  return(XEN_FALSE);
}

static XEN gxm_XtConvertCase(XEN arg1, XEN arg2)
{
  #define H_XtConvertCase "void XtConvertCase(display, keysym) calls the appropriate converter and returns the results."
  /* DIFF: XtConvertCase dpy keysym [k1 k2] -> (list k1 k2)
   */
  KeySym k1, k2;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtConvertCase", "Display*");
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg2), arg2, 2, "XtConvertCase", "KeySym");
  XtConvertCase(XEN_TO_C_Display(arg1), XEN_TO_C_KeySym(arg2), &k1, &k2);
  return(XEN_LIST_2(C_TO_XEN_KeySym(k1),
		    C_TO_XEN_KeySym(k2)));
}

/* -------- case converter -------- */

/* (424) convert case XtRegisterCaseConverter global */

static XEN xm_XtCaseProc = XEN_FALSE;
static void gxm_XtCaseProc(Display* d, KeySym k1, KeySym* k2, KeySym* k3)
{
  XEN val;
  int loc;
  val = XEN_CALL_2(xm_XtCaseProc,
		   C_TO_XEN_Display(d),
		   C_TO_XEN_KeySym(k1),
		   c__FUNCTION__);
  loc = xm_protect(val);
  if (XEN_LIST_P(val))
    {
      (*k2) = XEN_TO_C_KeySym(XEN_CAR(val));
      (*k3) = XEN_TO_C_KeySym(XEN_CADR(val));
    }
  xm_unprotect_at(loc);
}

static XEN gxm_XtRegisterCaseConverter(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XtRegisterCaseConverter "void XtRegisterCaseConverter(display, proc, start, stop) registers the specified case converter."
  /* DIFF: XtRegisterCaseConverter user XtCaseProc should return the new KeySyms as a list (not as ref args)
   */
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtRegisterCaseConverter", "Display*");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg2) && (XEN_REQUIRED_ARGS_OK(arg2, 2)), arg2, 2, "XtRegisterCaseConverter", "(XtCaseProc dpy keysym)");
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg3), arg3, 3, "XtRegisterCaseConverter", "KeySym");
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg4), arg4, 4, "XtRegisterCaseConverter", "KeySym");
  if (XEN_PROCEDURE_P(xm_XtCaseProc)) xm_unprotect(xm_XtCaseProc);
  xm_protect(arg2);
  xm_XtCaseProc = arg2;
  XtRegisterCaseConverter(XEN_TO_C_Display(arg1), gxm_XtCaseProc, XEN_TO_C_KeySym(arg3), XEN_TO_C_KeySym(arg4));
  return(XEN_FALSE);
}


/* -------- keyproc -------- */
/* (454) XtSetKeyTranslator global */

static XEN xm_XtKeyProc = XEN_FALSE;
static void gxm_XtKeyProc(Display *dpy, KeyCode c, Modifiers m, Modifiers *mp, KeySym *sym)
{
  XEN val;
  int loc;
  val = XEN_CALL_3(xm_XtKeyProc,
		   C_TO_XEN_Display(dpy),
		   C_TO_XEN_KeyCode(c),
		   C_TO_XEN_Modifiers(m),
		   c__FUNCTION__);
  loc = xm_protect(val);
  if (XEN_LIST_P(val))
    {
      /* KeySym is long, Modifier(s) is int, so these can actually work, I guess */
      (*mp) = XEN_TO_C_Modifiers(XEN_CAR(val));
      (*sym) = XEN_TO_C_KeySym(XEN_CADR(val));
    }
  else XtTranslateKey(dpy, c, m, mp, sym);
  xm_unprotect_at(loc);
}

static XEN gxm_XtSetKeyTranslator(XEN arg1, XEN arg2)
{
  #define H_XtSetKeyTranslator "void XtSetKeyTranslator(display, proc) sets the specified procedure as the current key translator. "
  /* DIFF: XtSetKeyTranslator user XtKeyProc should return the new Modifiers and KeySym as a list (not as ref arg), set arg2 #f to get default proc
   */
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtSetKeyTranslator", "Display*");
  XEN_ASSERT_TYPE(XEN_FALSE_P(arg2) || (XEN_PROCEDURE_P(arg2) && (XEN_REQUIRED_ARGS_OK(arg2, 3))), arg2, 2, "XtSetKeyTranslator", "(XtKeyProc dpy key mod)");
  if (XEN_PROCEDURE_P(xm_XtKeyProc)) xm_unprotect(xm_XtKeyProc);
  xm_XtKeyProc = arg2;
  if (XEN_FALSE_P(arg2))
    XtSetKeyTranslator(XEN_TO_C_Display(arg1), XtTranslateKey);
  else
    {
      xm_protect(arg2);
      XtSetKeyTranslator(XEN_TO_C_Display(arg1), (XtKeyProc)gxm_XtKeyProc);
    }
  return(XEN_FALSE);
}

static XEN gxm_XtTranslateKey(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtTranslateKey "XtTranslateKey(Display *, XtKeyCode, Modifiers) -> modifiers and keysym"
  /* DIFF: XtTranslateKey omit and rtn last 2 args
   */
  Modifiers m;
  KeySym k;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtTranslateKey", "Display*");
  XEN_ASSERT_TYPE(XEN_KeyCode_P(arg2), arg2, 2, "XtTranslateKey", "KeyCode");
  XEN_ASSERT_TYPE(XEN_Modifiers_P(arg3), arg3, 3, "XtTranslateKey", "Modifiers");
  XtTranslateKey(XEN_TO_C_Display(arg1), XEN_TO_C_KeyCode(arg2), XEN_TO_C_Modifiers(arg3), &m, &k);
  return(XEN_LIST_2(C_TO_XEN_Modifiers(m),
		    C_TO_XEN_KeySym(k)));
}

static XEN gxm_XtTranslateKeycode(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtTranslateKeycode "void XtTranslateKeycode(display, keycode, modifiers) passes the specified \
arguments directly to the currently registered KeyCode to KeySym translator, returns (modifiers keysym)."
  /* DIFF: XtTranslateKeycode omit and rtn last 2 args
   */
  Modifiers m;
  KeySym k;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtTranslateKeycode", "Display*");
  XEN_ASSERT_TYPE(XEN_KeyCode_P(arg2), arg2, 2, "XtTranslateKeycode", "KeyCode");
  XEN_ASSERT_TYPE(XEN_Modifiers_P(arg3), arg3, 3, "XtTranslateKeycode", "Modifiers");
  XtTranslateKeycode(XEN_TO_C_Display(arg1), XEN_TO_C_KeyCode(arg2), XEN_TO_C_Modifiers(arg3), &m, &k);
  return(XEN_LIST_2(C_TO_XEN_Modifiers(m),
		    C_TO_XEN_KeySym(k)));
}

static XEN gxm_XtGetActionKeysym(XEN arg1)
{
  #define H_XtGetActionKeysym "KeySym XtGetActionKeysym(event) -> (keysym modifiers)"
  /* DIFF: XtGetActionKeysym omit and rtn last arg
   */
  Modifiers m;
  KeySym k;
  XEN_ASSERT_TYPE(XEN_XEvent_P(arg1), arg1, 1, "XtGetActionKeysym", "XEvent*");
  k = XtGetActionKeysym(XEN_TO_C_XEvent(arg1), &m);
  return(XEN_LIST_2(C_TO_XEN_KeySym(k),
		    C_TO_XEN_Modifiers(m)));
}

static XEN gxm_XtGetResourceList(XEN widget_class)
{
  #define H_XtGetResourceList "XtGetResourceList(widget-class) returns the widget class's resource list"
  XEN lst = XEN_EMPTY_LIST;
  Cardinal len = 0;
  int i;
  XtResourceList resources;
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(widget_class), widget_class, 1, "XtGetResourceList", "WidgetClass");
  XtGetResourceList(XEN_TO_C_WidgetClass(widget_class), &resources, &len);
  if (len > 0)
    {
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(XEN_LIST_7(C_TO_XEN_STRING(resources[i].resource_name),
				  C_TO_XEN_STRING(resources[i].resource_class),
				  C_TO_XEN_STRING(resources[i].resource_type),
				  C_TO_XEN_INT((int)(resources[i].resource_size)),
				  C_TO_XEN_INT((int)(resources[i].resource_offset)),
				  C_TO_XEN_STRING(resources[i].default_type),
				  C_TO_XEN_ULONG(resources[i].default_addr)),
		       lst);
      XtFree((char *)resources);
    }
  return(lst);
}

static XEN gxm_XtGetMultiClickTime(XEN arg1)
{
  #define H_XtGetMultiClickTime "int XtGetMultiClickTime(display) returns the time in milliseconds that the translation manager uses to \
determine if multiple events are to be interpreted as a repeated event "
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtGetMultiClickTime", "Display*");
  return(C_TO_XEN_INT(XtGetMultiClickTime(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XtSetMultiClickTime(XEN arg1, XEN arg2)
{
  #define H_XtSetMultiClickTime "void XtSetMultiClickTime(display, time) sets the time interval used by the translation manager to determine \
when multiple events are interpreted as a repeated event."
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtSetMultiClickTime", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XtSetMultiClickTime", "int");
  XtSetMultiClickTime(XEN_TO_C_Display(arg1), XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN register_proc = XEN_FALSE;
static void gxm_XtRegisterGrabActionProc(Widget w, XEvent* e, String* s, Cardinal* c) 
{
  XEN_CALL_3(register_proc,
	     C_TO_XEN_Widget(w),
	     C_TO_XEN_XEvent(e),
	     C_TO_XEN_Strings(s, *c),
	     c__FUNCTION__);
}

static XEN gxm_XtRegisterGrabAction(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtRegisterGrabAction "void XtRegisterGrabAction(action_proc, owner_events, event_mask, pointer_mode, keyboard_mode) adds the \
specified action_proc to a list known to the translation manager."
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg1) && (XEN_REQUIRED_ARGS_OK(arg1, 3)), arg1, 1, "XtRegisterGrabAction", "XtActionProc");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XtRegisterGrabAction", "boolean");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XtRegisterGrabAction", "unsigned int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XtRegisterGrabAction", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XtRegisterGrabAction", "int");
  xm_protect(arg1);
  register_proc = arg1;
  XtRegisterGrabAction(gxm_XtRegisterGrabActionProc, XEN_TO_C_BOOLEAN(arg2), XEN_TO_C_ULONG(arg3), XEN_TO_C_INT(arg4), XEN_TO_C_INT(arg5));
  return(XEN_FALSE);
}

static XEN gxm_XtCallActionProc(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XtCallActionProc "void XtCallActionProc(widget, action, event, params, num_params) searches for the named action routine in the \
same manner and order as translation tables are bound. If found, the action routine is invoked with the specified widget, event pointer, \
and parameters."
  char **params = NULL;
  int i, len = 0;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtCallActionProc", "Widget");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XtCallActionProc", "char*");
  XEN_ASSERT_TYPE(XEN_XEvent_P(arg3) || XEN_FALSE_P(arg3), arg3, 3, "XtCallActionProc", "XEvent*");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg4) || XEN_FALSE_P(arg4), arg4, 4, "XtCallActionProc", "list of String");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg5), arg5, 5, "XtCallActionProc", "int");
  if (XEN_LIST_P(arg4))
    {
      if (XEN_INTEGER_P(arg5)) 
	len = XEN_TO_C_INT(arg5); 
      else len = XEN_LIST_LENGTH(arg4);
    }
  if (len > 0) 
    {
      params = (char **)CALLOC(len, sizeof(char *));
      for (i = 0; i < len; i++)
	params[i] = strdup(XEN_TO_C_STRING(XEN_LIST_REF(arg4, i)));
    }
  XtCallActionProc(XEN_TO_C_Widget(arg1), 
		   XEN_TO_C_STRING(arg2), 
		   (XEN_XEvent_P(arg3)) ? XEN_TO_C_XEvent(arg3) : NULL,
		   params, len);
  if (params)
    {
      for (i = 0; i < len; i++)
	if (params[i]) free(params[i]);
      FREE(params);
    }
  return(xen_return_first(XEN_FALSE, arg4));
}

static XEN gxm_XtGetActionList(XEN arg1)
{
  #define H_XtGetActionList "void XtGetActionList(widget_class) -> list of actions"
  /* DIFF: XtGetActionList omit arg2 and 3, return list of lists
   */
  unsigned int len;
  XtActionList act;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg1), arg1, 1, "XtGetActionList", "WidgetClass");
  XtGetActionList(XEN_TO_C_WidgetClass(arg1), &act, &len);
  if (len > 0)
    {
      int i;
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(XEN_LIST_2(C_TO_XEN_STRING(act[i].string),
				  C_TO_XEN_ULONG(act[i].proc)),
		       lst);
      free(act);
    }
  return(lst);
}

#define C_TO_XEN_XM_ActionHook(Code, Context) \
  XEN_LIST_5(C_STRING_TO_XEN_SYMBOL("ActionHook"), Code, Context, XEN_ZERO, XEN_ZERO)
#define XM_ActionHook_P(Arg) WRAP_P("ActionHook", Arg)

static bool unprotect_actionhook(XEN val, int loc, unsigned long id)
{
  if ((XM_ActionHook_P(val)) &&
      ((XtActionHookId)XEN_TO_C_ULONG(XEN_LIST_REF(val, 4)) == (XtActionHookId)id))
    {
      xm_unprotect_at(loc);
      return(true);
    }
  return(false);
}

static XEN gxm_XtRemoveActionHook(XEN arg1)
{
  #define H_XtRemoveActionHook "void XtRemoveActionHook(idP) removes the specified action hook procedure from the list in which it was registered."
  XtActionHookId id;
  XEN_ASSERT_TYPE(XEN_XtActionHookId_P(arg1), arg1, 1, "XtRemoveActionHook", "XtActionHookId");
  id = XEN_TO_C_XtActionHookId(arg1);
  XtRemoveActionHook(id);
  map_over_protected_elements(unprotect_actionhook, (unsigned long)id);
  return(XEN_FALSE);
}

static void gxm_XtActionHookProc(Widget w, XtPointer p, String s, XEvent* e, String* s1, Cardinal* c) 
{
  /* DIFF: XtActionHookProc takes 5 args (last is string list)
   */
  XEN descr = (XEN)p;
  /* (list 'ActionHook function context gc-loc id) */
  XEN_CALL_5(XEN_CADR(descr), 
	     C_TO_XEN_Widget(w),
	     XEN_CADDR(descr),
	     C_TO_XEN_STRING(s),
	     C_TO_XEN_XEvent(e),
	     C_TO_XEN_Strings(s1, *c),
	     c__FUNCTION__);
}

static XEN gxm_XtAppAddActionHook(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtAppAddActionHook "XtActionHookId XtAppAddActionHook(app_context, proc, client_data) adds the specified procedure to the front of \
a list maintained in the application context."
  XtActionHookId id;
  int gc_loc;
  XEN descr;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppAddActionHook", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg2) && (XEN_REQUIRED_ARGS_OK(arg2, 5)), arg2, 2, "XtAppAddActionHook", "XtActionHookProc");
  descr = C_TO_XEN_XM_ActionHook(arg2, (XEN_BOUND_P(arg3)) ? arg3 : XEN_FALSE);
  gc_loc = xm_protect(descr);
  id = XtAppAddActionHook(XEN_TO_C_XtAppContext(arg1), gxm_XtActionHookProc, (XtPointer)descr);
  XEN_LIST_SET(descr, 3, C_TO_XEN_INT(gc_loc));
  XEN_LIST_SET(descr, 4, C_TO_XEN_ULONG(id));
  return(C_TO_XEN_XtActionHookId(id));
}


/* (72) this is part of an actionrec */
/* this code is a horrible kludge, but what can I do? -- to vector to the procedure corresponding to
 *   the action that triggered the action proc, I need to have some way to find that procedure,
 *   and there's none I can see -- the args incoming (widget event args argn) give me no way to
 *   trace back to the caller, or even to save that data at declaration time via clientData or
 *   its equivalent -- I don't even get to see the affected widget.  By the time install|augment|override
 *   translations is called, my procedure info has been buried (or at least compiled into some unknown form).
 *
 *   XtAppContext->action_table is a list of ActionLists
 *   Widget struct has: XtTMRec tm:
 *     XtTranslations  translations;
 *     XtBoundActions  proc_table;  -- list of XtActionProcs		
 *     struct _XtStateRec *current_state;  
 */

static XEN xtactionprocs[9];
static void gxm_XtActionProc0(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  XEN_CALL_3(xtactionprocs[0], C_TO_XEN_Widget(w), C_TO_XEN_XEvent(e), C_TO_XEN_Strings(args, *argn), c__FUNCTION__);
}
static void gxm_XtActionProc1(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  XEN_CALL_3(xtactionprocs[1], C_TO_XEN_Widget(w), C_TO_XEN_XEvent(e), C_TO_XEN_Strings(args, *argn), c__FUNCTION__);
}
static void gxm_XtActionProc2(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  XEN_CALL_3(xtactionprocs[2], C_TO_XEN_Widget(w), C_TO_XEN_XEvent(e), C_TO_XEN_Strings(args, *argn), c__FUNCTION__);
}
static void gxm_XtActionProc3(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  XEN_CALL_3(xtactionprocs[3], C_TO_XEN_Widget(w), C_TO_XEN_XEvent(e), C_TO_XEN_Strings(args, *argn), c__FUNCTION__);
}
static void gxm_XtActionProc4(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  XEN_CALL_3(xtactionprocs[4], C_TO_XEN_Widget(w), C_TO_XEN_XEvent(e), C_TO_XEN_Strings(args, *argn), c__FUNCTION__);
}
static void gxm_XtActionProc5(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  XEN_CALL_3(xtactionprocs[5], C_TO_XEN_Widget(w), C_TO_XEN_XEvent(e), C_TO_XEN_Strings(args, *argn), c__FUNCTION__);
}
static void gxm_XtActionProc6(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  XEN_CALL_3(xtactionprocs[6], C_TO_XEN_Widget(w), C_TO_XEN_XEvent(e), C_TO_XEN_Strings(args, *argn), c__FUNCTION__);
}
static void gxm_XtActionProc7(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  XEN_CALL_3(xtactionprocs[7], C_TO_XEN_Widget(w), C_TO_XEN_XEvent(e), C_TO_XEN_Strings(args, *argn), c__FUNCTION__);
}
static void gxm_XtActionProc8(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  fprintf(stderr, "too many actions");
}
static int xm_action_ctr = 0;

static XtActionsRec *make_action_rec(int len, XEN larg2)
{
  int i, gcloc;
  XtActionsRec *act;
  XEN arg2;
  arg2 = XEN_COPY_ARG(larg2);
  gcloc = xm_protect(arg2);
  act = (XtActionsRec *)CALLOC(len, sizeof(XtActionsRec));
  for (i = 0; i < len; i++, arg2 = XEN_CDR(arg2))
    {
      XEN pair;
      pair = XEN_CAR(arg2);
      act[i].string = (String)strdup(XEN_TO_C_STRING(XEN_CAR(pair)));
      if (xm_action_ctr >= 8)
	{
	  fprintf(stderr,"too many actions...");
	  act[i].proc = (XtActionProc)gxm_XtActionProc8;
	  xm_protect(XEN_CADR(pair));
	  xtactionprocs[8] = XEN_CADR(pair);
	}
      else
	{
	  switch (xm_action_ctr)
	    {
	    case 0: act[i].proc = (XtActionProc)gxm_XtActionProc0; break;
	    case 1: act[i].proc = (XtActionProc)gxm_XtActionProc1; break;
	    case 2: act[i].proc = (XtActionProc)gxm_XtActionProc2; break;
	    case 3: act[i].proc = (XtActionProc)gxm_XtActionProc3; break;
	    case 4: act[i].proc = (XtActionProc)gxm_XtActionProc4; break;
	    case 5: act[i].proc = (XtActionProc)gxm_XtActionProc5; break;
	    case 6: act[i].proc = (XtActionProc)gxm_XtActionProc6; break;
	    case 7: act[i].proc = (XtActionProc)gxm_XtActionProc7; break;
	    }
	  xm_protect(XEN_CADR(pair));
	  xtactionprocs[xm_action_ctr++] = XEN_CADR(pair);
	}
    }
  xm_unprotect_at(gcloc);
  return(act);
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtAddActions(XEN arg1)
{
  #define H_XtAddActions "void XtAddActions(actions, num_actions) has been replaced by XtAppAddActions."
  /* DIFF: XtAddActions takes list of lists for arg1 (name proc) pairs, not XtActionList, omits arg2 (pointless)
   *        and action proc itself takes 3 args (no need for trailing count)
   */
  XtActionsRec *act;
  int i, len;
  XEN_ASSERT_TYPE(XEN_LIST_P(arg1), arg1, 1, "XtAddActions", "list of XtActions");
  len = XEN_LIST_LENGTH(arg1);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg1, 1, "XtAddActions", "positive integer");
  act = make_action_rec(len, arg1);
  XtAddActions(act, len);
  for (i = 0; i < len; i++)
    if (act[i].string) free(act[i].string);
  FREE(act);
  return(arg1);
}
#endif

static XEN gxm_XtAppAddActions(XEN arg1, XEN arg2)
{
  #define H_XtAppAddActions "void XtAppAddActions(app_context, actions, num_actions) adds the specified action table and registers it \
with the translation manager."
  /* DIFF: XtAddAppActions takes list of lists for arg2 (name proc) pairs, not XtActionList, omits arg3 (pointless)
   *        and action proc itself takes 3 args (no need for trailing count)
   */
  XtActionsRec *act;
  int i, len;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppAddActions", "XtAppContext");
  XEN_ASSERT_TYPE(XEN_LIST_P(arg2), arg2, 2, "XtAppAddActions", "list of XtActions");
  len = XEN_LIST_LENGTH(arg2);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg2, 2, "XtAppAddActions", "positive integer");
  act = make_action_rec(len, arg2);
  XtAppAddActions(XEN_TO_C_XtAppContext(arg1), act, len);
  for (i = 0; i < len; i++)
    if (act[i].string) free(act[i].string);
  FREE(act);
  return(arg1);
}

static XEN gxm_XtUninstallTranslations(XEN arg1)
{
  #define H_XtUninstallTranslations "void XtUninstallTranslations(w) causes the entire translation table for widget to be removed."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtUninstallTranslations", "Widget");
  XtUninstallTranslations(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtInstallAllAccelerators(XEN arg1, XEN arg2)
{
  #define H_XtInstallAllAccelerators "void XtInstallAllAccelerators(destination, source)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtInstallAllAccelerators", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2), arg2, 2, "XtInstallAllAccelerators", "Widget");
  XtInstallAllAccelerators(XEN_TO_C_Widget(arg1), XEN_TO_C_Widget(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtInstallAccelerators(XEN arg1, XEN arg2)
{
  #define H_XtInstallAccelerators "void XtInstallAccelerators(destination, source) installs the accelerators from source onto destination by \
augmenting the destination translations with the source accelerators."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtInstallAccelerators", "Widget");
  XEN_ASSERT_TYPE(XEN_Widget_P(arg2), arg2, 2, "XtInstallAccelerators", "Widget");
  XtInstallAccelerators(XEN_TO_C_Widget(arg1), XEN_TO_C_Widget(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtAugmentTranslations(XEN arg1, XEN arg2)
{
  #define H_XtAugmentTranslations "void XtAugmentTranslations(w, translations) nondestructively merges the new translations into the existing \
widget translations."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtAugmentTranslations", "Widget");
  XEN_ASSERT_TYPE(XEN_XtTranslations_P(arg2), arg2, 2, "XtAugmentTranslations", "XtTranslations");
  XtAugmentTranslations(XEN_TO_C_Widget(arg1), XEN_TO_C_XtTranslations(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtOverrideTranslations(XEN arg1, XEN arg2)
{
  #define H_XtOverrideTranslations "void XtOverrideTranslations(w, translations) destructively merges the new translations into the existing \
widget translations."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtOverrideTranslations", "Widget");
  XEN_ASSERT_TYPE(XEN_XtTranslations_P(arg2), arg2, 2, "XtOverrideTranslations", "XtTranslations");
  XtOverrideTranslations(XEN_TO_C_Widget(arg1), XEN_TO_C_XtTranslations(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XtParseAcceleratorTable(XEN arg1)
{
  #define H_XtParseAcceleratorTable "XtAccelerators XtParseAcceleratorTable(const char *source) compiles source"
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtParseAcceleratorTable", "char*");
  return(C_TO_XEN_ULONG(XtParseAcceleratorTable(XEN_TO_C_STRING(arg1))));
}

static XEN gxm_XtParseTranslationTable(XEN arg1)
{
  #define H_XtParseTranslationTable "XtTranslations XtParseTranslationTable(table) compiles the translation table into the opaque internal \
representation of type XtTranslations."
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XtParseTranslationTable", "char*");
  return(C_TO_XEN_XtTranslations(XtParseTranslationTable(XEN_TO_C_STRING(arg1))));
}

static XEN gxm_XtKeysymToKeycodeList(XEN arg1, XEN arg2)
{
  #define H_XtKeysymToKeycodeList "void XtKeysymToKeycodeList(display, keysym) procedure returns all the \
KeyCodes that have keysym in their entry for the keyboard mapping table associated with display -> (keycodes)."
  /* DIFF: XtKeysymToKeycodeList final 2 args omit, returns list
   */
  unsigned int len;
  KeyCode *kr;
  XEN lst = XEN_EMPTY_LIST;
  int loc;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XtKeysymToKeycodeList", "Display*");
  XEN_ASSERT_TYPE(XEN_KeySym_P(arg2), arg2, 2, "XtKeysymToKeycodeList", "KeySym");
  XtKeysymToKeycodeList(XEN_TO_C_Display(arg1), XEN_TO_C_KeySym(arg2), &kr, &len);
  loc = xm_protect(lst);
  if (len > 0)
    {
      int i;
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_KeyCode(kr[i]), lst);
      free(kr);
    }
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XtTranslateCoords(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtTranslateCoords "void XtTranslateCoords(w, x, y) -> (root_x root_y)"
  /* DIFF: XtTranslateCoords omits and returns last 2 args
   */
  Position x, y;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtTranslateCoords", "Widget");
  XEN_ASSERT_TYPE(XEN_Position_P(arg2), arg2, 2, "XtTranslateCoords", "Position");
  XEN_ASSERT_TYPE(XEN_Position_P(arg3), arg3, 3, "XtTranslateCoords", "Position");
  XtTranslateCoords(XEN_TO_C_Widget(arg1), XEN_TO_C_Position(arg2), XEN_TO_C_Position(arg3), &x, &y);
  return(XEN_LIST_2(C_TO_XEN_Position(x),
		    C_TO_XEN_Position(y)));
}

static XEN gxm_XtMakeResizeRequest(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtMakeResizeRequest "XtGeometryResult XtMakeResizeRequest(w, width, height) -> (res width height)"
  /* DIFF: XtMakeResizeRequest w wid hgt [rtnw rtn] -> (list res w h)
   */
  Dimension w, h;
  int val;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtMakeResizeRequest", "Widget");
  XEN_ASSERT_TYPE(XEN_Dimension_P(arg2), arg2, 2, "XtMakeResizeRequest", "Dimension");
  XEN_ASSERT_TYPE(XEN_Dimension_P(arg3), arg3, 3, "XtMakeResizeRequest", "Dimension");
  val = XtMakeResizeRequest(XEN_TO_C_Widget(arg1), 
			    XEN_TO_C_Dimension(arg2), 
			    XEN_TO_C_Dimension(arg3), 
			    &w, &h);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_Dimension(w),
		    C_TO_XEN_Dimension(h)));
}

static XEN gxm_XtOwnSelectionIncremental(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6, XEN arg7, XEN arg8)
{
  #define H_XtOwnSelectionIncremental "Boolean XtOwnSelectionIncremental(w, selection, time, convert_callback, lose_callback, \
done_callback, cancel_callback, client_data)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtOwnSelectionIncremental", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtOwnSelectionIncremental", "Atom");
  XEN_ASSERT_TYPE(XEN_Time_P(arg3), arg3, 3, "XtOwnSelectionIncremental", "Time");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 10)), arg4, 4, "XtOwnSelectionIncremental", "XtConvertSelectionIncrProc");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg5) && (XEN_REQUIRED_ARGS_OK(arg5, 3)), arg5, 5, "XtOwnSelectionIncremental", "XtLoseSelectionIncrProc");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg6) && (XEN_REQUIRED_ARGS_OK(arg6, 5)), arg6, 6, "XtOwnSelectionIncremental", "XtSelectionDoneIncrProc");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg7) && (XEN_REQUIRED_ARGS_OK(arg7, 5)), arg7, 7, "XtOwnSelectionIncremental", "XtCancelConvertSelectionProc");
  add_selmap(XEN_TO_C_Widget(arg1), CONVERT_INCR, arg4);
  add_selmap(XEN_TO_C_Widget(arg1), LOSE_INCR, arg5);
  add_selmap(XEN_TO_C_Widget(arg1), DONE_INCR, arg6);
  add_selmap(XEN_TO_C_Widget(arg1), CANCEL_CONVERT, arg7);	     
  return(C_TO_XEN_BOOLEAN(XtOwnSelectionIncremental(XEN_TO_C_Widget(arg1), 
						    XEN_TO_C_Atom(arg2), 
						    XEN_TO_C_Time(arg3), 
						    (XtConvertSelectionIncrProc)gxm_XtConvertSelectionIncrProc, 
						    gxm_XtLoseSelectionIncrProc, 
						    gxm_XtSelectionDoneIncrProc, 
						    gxm_XtCancelConvertSelectionProc, 
						    (XtPointer)arg8)));
}

static XEN gxm_XtOwnSelection(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XtOwnSelection "Boolean XtOwnSelection(w, selection, time, convert_proc, lose_selection, done_proc) informs the selection \
mechanism that a widget believes it owns a selection."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtOwnSelection", "Widget");
  XEN_ASSERT_TYPE(XEN_Atom_P(arg2), arg2, 2, "XtOwnSelection", "Atom");
  XEN_ASSERT_TYPE(XEN_Time_P(arg3), arg3, 3, "XtOwnSelection", "Time");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 7)), arg4, 4, "XtOwnSelection", "XtConvertSelectionProc");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg5) && (XEN_REQUIRED_ARGS_OK(arg5, 2)), arg5, 5, "XtOwnSelection", "XtLoseSelectionProc");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg6) && (XEN_REQUIRED_ARGS_OK(arg6, 3)), arg6, 6, "XtOwnSelection", "XtSelectionDoneProc");
  add_selmap(XEN_TO_C_Widget(arg1), CONVERT, arg4);
  add_selmap(XEN_TO_C_Widget(arg1), LOSE, arg5);
  add_selmap(XEN_TO_C_Widget(arg1), DONE, arg6);	     
  return(C_TO_XEN_BOOLEAN(XtOwnSelection(XEN_TO_C_Widget(arg1), 
					 XEN_TO_C_Atom(arg2), 
					 XEN_TO_C_Time(arg3), 
					 (XtConvertSelectionProc)gxm_XtConvertSelectionProc, 
					 gxm_XtLoseSelectionProc, 
					 gxm_XtSelectionDoneProc)));
}

static XEN gxm_XtIsSensitive(XEN arg1)
{
  #define H_XtIsSensitive "Boolean XtIsSensitive(w) returns #t if user input events are being dispatched."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtIsSensitive", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsSensitive(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtIsRealized(XEN arg1)
{
  #define H_XtIsRealized "Boolean XtIsRealized(w) returns #t if the widget has been realized,"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtIsRealized", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsRealized(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtIsManaged(XEN arg1)
{
  #define H_XtIsManaged "Boolean XtIsManaged(widget) returns #t if the specified widget is of class RectObj or any subclass thereof and is managed."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtIsManaged", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsManaged(XEN_TO_C_Widget(arg1))));
}

static XEN gxm_XtIsObject(XEN arg1)
{
  #define H_XtIsObject "Boolean XtIsObject(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtIsObject", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsObject(XEN_TO_C_Widget(arg1))));
}

#if MOTIF_2
static XEN gxm_XtIsSubclass(XEN arg1, XEN arg2)
{
  #define H_XtIsSubclass "Boolean XtIsSubclass(w, widget_class) returns #t if the class of the specified widget is equal to or is a \
subclass of the specified class."
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtIsSubclass", "Widget");
  XEN_ASSERT_TYPE(XEN_WidgetClass_P(arg2), arg2, 2, "XtIsSubclass", "WidgetClass");
  return(C_TO_XEN_BOOLEAN(XtIsSubclass(XEN_TO_C_Widget(arg1), XEN_TO_C_WidgetClass(arg2))));
}
#endif

static XEN gxm_XtAppPeekEvent(XEN arg1)
{
  #define H_XtAppPeekEvent "Boolean XtAppPeekEvent(app_context) fills in the event and returns a nonzero value. If no X \
input is on the queue, XtAppPeekEvent flushes the output buffer and blocks until input is available."
  /* DIFF: XtAppPeekEvent app [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  XEN_ASSERT_TYPE(XEN_XtAppContext_P(arg1), arg1, 1, "XtAppPeekEvent", "XtAppContext");
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XtAppPeekEvent(XEN_TO_C_XtAppContext(arg1), e);
  return(XEN_LIST_2(C_TO_XEN_BOOLEAN(val), C_TO_XEN_XEvent_OBJ(e)));
}

#if (!XM_DISABLE_DEPRECATED)
static XEN gxm_XtPeekEvent(void)
{
  #define H_XtPeekEvent "Boolean XtPeekEvent() has been replaced by XtAppPeekEvent."
  /* DIFF: XtPeekEvent [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  e = (XEvent *)CALLOC(1, sizeof(XEvent));
  val = XtPeekEvent(e);
  return(XEN_LIST_2(C_TO_XEN_BOOLEAN(val), C_TO_XEN_XEvent_OBJ(e)));
}
#endif

static XEN gxm_XtCallAcceptFocus(XEN arg1, XEN arg2)
{
  #define H_XtCallAcceptFocus "Boolean XtCallAcceptFocus(w, time) calls the specified widget's accept_focus procedure, passing it the \
specified widget and time, and returns what the accept_focus procedure returns."
  /* DIFF: XtCallAcceptFocus takes Time arg (not Time*)
   */
  Time tm;
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtCallAcceptFocus", "Widget");
  XEN_ASSERT_TYPE(XEN_Time_P(arg2), arg2, 2, "XtCallAcceptFocus", "Time");
  tm = XEN_TO_C_Time(arg2);
  return(C_TO_XEN_BOOLEAN(XtCallAcceptFocus(XEN_TO_C_Widget(arg1), &tm)));
}

static XEN gxm_XtDispatchEvent(XEN arg1)
{
  #define H_XtDispatchEvent "Boolean XtDispatchEvent(event) sends those events to the event handler functions that have been previously \
registered with the dispatch routine."
  XEN_ASSERT_TYPE(XEN_XEvent_P(arg1), arg1, 1, "XtDispatchEvent", "XEvent*");
  return(C_TO_XEN_BOOLEAN(XtDispatchEvent(XEN_TO_C_XEvent(arg1))));
}

static XEN gxm_XtUnmanageChild(XEN arg1)
{
  #define H_XtUnmanageChild "void XtUnmanageChild(child)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtUnmanageChild", "Widget");
  XtUnmanageChild(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtUnmanageChildren(XEN arg1, XEN arg2)
{
  #define H_XtUnmanageChildren "void XtUnmanageChildren(children, num_children)"
  /* DIFF: XtUnmanageChildren arg1 is list of widgets
   */
  WidgetList ws1;
  int len;
  XEN_ASSERT_TYPE(XEN_LIST_P(arg1), arg1, 1, "XtUnmanageChildren", "WidgetList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg2), arg2, 2, "XtUnmanageChildren", "int");
  if (XEN_INTEGER_P(arg2)) len = XEN_TO_C_INT(arg2); else len = XEN_LIST_LENGTH(arg1);
  if (len > 0)
    {
      ws1 = XEN_TO_C_Widgets(arg1, len);
      XtUnmanageChildren(ws1, len);
      if (ws1) FREE(ws1);
    }
  return(XEN_FALSE);
}

static XEN gxm_XtManageChild(XEN arg1)
{
  #define H_XtManageChild "void XtManageChild(child)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg1), arg1, 1, "XtManageChild", "Widget");
  XtManageChild(XEN_TO_C_Widget(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XtManageChildren(XEN arg1, XEN arg2)
{
  #define H_XtManageChildren "void XtManageChildren(children, num_children)"
  /* DIFF: XtManageChildren arg1 is list of widgets
   */
  WidgetList ws1;
  int len;
  XEN_ASSERT_TYPE(XEN_LIST_P(arg1), arg1, 1, "XtManageChildren", "WidgetList");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(arg2), arg2, 2, "XtManageChildren", "int");
  if (XEN_INTEGER_P(arg2)) len = XEN_TO_C_INT(arg2); else len = XEN_LIST_LENGTH(arg1);
  if (len > 0)
    {
      ws1 = XEN_TO_C_Widgets(arg1, len);
      XtManageChildren(ws1, len);
      if (ws1) FREE(ws1);
    }
  return(XEN_FALSE);
}

static XEN gxm_XtIsRectObj(XEN arg)
{
  #define H_XtIsRectObj "Boolean XtIsRectObj(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsRectObj", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsRectObj(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtIsWidget(XEN arg)
{
  #define H_XtIsWidget "Boolean XtIsWidget(w)"
  return(C_TO_XEN_BOOLEAN(WRAP_P("Widget", arg) && 
			  (XtIsWidget(XEN_TO_C_Widget(arg)))));
}

static XEN gxm_XtIsComposite(XEN arg)
{
  #define H_XtIsComposite "Boolean XtIsComposite(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsComposite", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsComposite(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtIsConstraint(XEN arg)
{
  #define H_XtIsConstraint "Boolean XtIsConstraint(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsConstraint", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsConstraint(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtIsShell(XEN arg)
{
  #define H_XtIsShell "Boolean XtIsShell(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsShell(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtIsOverrideShell(XEN arg)
{
  #define H_XtIsOverrideShell "Boolean XtIsOverrideShell(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsOverrideShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsOverrideShell(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtIsWMShell(XEN arg)
{
  #define H_XtIsWMShell "Boolean XtIsWMShell(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsWMShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsWMShell(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtIsVendorShell(XEN arg)
{
  #define H_XtIsVendorShell "Boolean XtIsVendorShell(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsVendorShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsVendorShell(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtIsTransientShell(XEN arg)
{
  #define H_XtIsTransientShell "Boolean XtIsTransientShell(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsTransientShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsTransientShell(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtIsTopLevelShell(XEN arg)
{
  #define H_XtIsTopLevelShell "Boolean XtIsTopLevelShell(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsTopLevelShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsTopLevelShell(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtIsApplicationShell(XEN arg)
{
  #define H_XtIsApplicationShell "Boolean XtIsApplicationShell(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsApplicationShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsApplicationShell(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtIsSessionShell(XEN arg)
{
  #define H_XtIsSessionShell "Boolean XtIsSessionShell(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtIsSessionShell", "Widget");
  return(C_TO_XEN_BOOLEAN(XtIsSessionShell(XEN_TO_C_Widget(arg))));
}

static XEN gxm_XtMapWidget(XEN arg)
{
  #define H_XtMapWidget "XtMapWidget(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtMapWidget", "Widget");
  XtMapWidget(XEN_TO_C_Widget(arg));
  return(XEN_FALSE);
}

static XEN gxm_XtUnmapWidget(XEN arg)
{
  #define H_XtUnmapWidget "XtUnmapWidget(w)"
  XEN_ASSERT_TYPE(XEN_Widget_P(arg), arg, 0, "XtUnmapWidget", "Widget");
  XtUnmapWidget(XEN_TO_C_Widget(arg));
  return(XEN_FALSE);
}

static XEN gxm_XtSetArg(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XtSetArg "XtSetArg in xm is useless -- it returns its arguments as a list"
  return(XEN_LIST_3(arg1, arg2, arg3));
}


#endif
/* end HAVE_MOTIF */


/* ---------------------------------------------------------------------------------------------------- */
#if HAVE_XM_XP
static XEN gxm_XpCancelPage(XEN arg1, XEN arg2)
{
  #define H_XpCancelPage "void XpCancelPage(Display *display,Bool discard)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpCancelPage", "Display*");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XpCancelPage", "BOOLEAN");
  XpCancelPage(XEN_TO_C_Display(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XpEndPage(XEN arg1)
{
  #define H_XpEndPage "void XpEndPage(Display *display)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpEndPage", "Display*");
  XpEndPage(XEN_TO_C_Display(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XpStartPage(XEN arg1, XEN arg2)
{
  #define H_XpStartPage "void XpStartPage(Display *display,Window window)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpStartPage", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XpStartPage", "Window");
  XpStartPage(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XpCancelDoc(XEN arg1, XEN arg2)
{
  #define H_XpCancelDoc "void XpCancelDoc(Display *display,Bool discard)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpCancelDoc", "Display*");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XpCancelDoc", "BOOLEAN");
  XpCancelDoc(XEN_TO_C_Display(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XpEndDoc(XEN arg1)
{
  #define H_XpEndDoc "void XpEndDoc(Display *display)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpEndDoc", "Display*");
  XpEndDoc(XEN_TO_C_Display(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XpStartDoc(XEN arg1, XEN arg2)
{
  #define H_XpStartDoc "void XpStartDoc(Display *display, XPDocumentType type)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpStartDoc", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XpStartDoc", "XPDocumentType");
  XpStartDoc(XEN_TO_C_Display(arg1), (XPDocumentType)XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XpCancelJob(XEN arg1, XEN arg2)
{
  #define H_XpCancelJob "void XpCancelJob(Display *display,Bool discard)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpCancelJob", "Display*");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg2), arg2, 2, "XpCancelJob", "BOOLEAN");
  XpCancelJob(XEN_TO_C_Display(arg1), XEN_TO_C_BOOLEAN(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XpEndJob(XEN arg1)
{
  #define H_XpEndJob "void XpEndJob(Display *display)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpEndJob", "Display*");
  XpEndJob(XEN_TO_C_Display(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XpStartJob(XEN arg1, XEN arg2)
{
  #define H_XpStartJob "void XpStartJob(Display *display, XPSaveData save_data)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpStartJob", "Display*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg2), arg2, 2, "XpStartJob", "XPSaveData");
  XpStartJob(XEN_TO_C_Display(arg1), (XPSaveData)XEN_TO_C_INT(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XpRehashPrinterList(XEN arg1)
{
  #define H_XpRehashPrinterList "void XpRehashPrinterList(Display *display)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpRehashPrinterList", "Display*");
  XpRehashPrinterList(XEN_TO_C_Display(arg1));
  return(XEN_FALSE);
}

static XEN gxm_XpDestroyContext(XEN arg1, XEN arg2)
{
  #define H_XpDestroyContext "void XpDestroyContext(Display *display, XPContext print_context)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpDestroyContext", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpDestroyContext", "XPContext");
  XpDestroyContext(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XpGetContext(XEN arg1)
{
  #define H_XpGetContext "XPContext XpGetContext(Display *display)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpGetContext", "Display*");
  return(C_TO_XEN_XPContext(XpGetContext(XEN_TO_C_Display(arg1))));
}

static XEN gxm_XpSetContext(XEN arg1, XEN arg2)
{
  #define H_XpSetContext "void XpSetContext(Display *display, XPContext print_context)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpSetContext", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpSetContext", "XPContext");
  XpSetContext(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2));
  return(XEN_FALSE);
}

static XEN gxm_XpCreateContext(XEN arg1, XEN arg2)
{
  #define H_XpCreateContext "XPContext XpCreateContext(Display *display,char *printer_name)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpCreateContext", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XpCreateContext", "char*");
  return(C_TO_XEN_XPContext(XpCreateContext(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2))));
}

static XEN gxm_XpNotifyPdm(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XpNotifyPdm "char *XpNotifyPdm(Display *print_display,Window print_window, XPContext print_context, \
Display *video_display,Window video_window,Bool auth_flag)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpNotifyPdm", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XpNotifyPdm", "Window");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg3), arg3, 3, "XpNotifyPdm", "XPContext");
  XEN_ASSERT_TYPE(XEN_Display_P(arg4), arg4, 4, "XpNotifyPdm", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg5), arg5, 5, "XpNotifyPdm", "Window");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg6), arg6, 6, "XpNotifyPdm", "BOOLEAN");
  return(C_TO_XEN_STRING(XpNotifyPdm(XEN_TO_C_Display(arg1), 
				     XEN_TO_C_Window(arg2), 
				     XEN_TO_C_XPContext(arg3), 
				     XEN_TO_C_Display(arg4), 
				     XEN_TO_C_Window(arg5), 
				     XEN_TO_C_BOOLEAN(arg6))));
}

static XEN gxm_XpGetLocaleNetString(void)
{
  #define H_XpGetLocaleNetString "char *XpGetLocaleNetString(void)"
  return(C_TO_XEN_STRING(XpGetLocaleNetString()));
}

static XEN gxm_XpSendAuth(XEN arg1, XEN arg2)
{
  #define H_XpSendAuth "Status XpSendAuth(Display *display,Window window)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpSendAuth", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XpSendAuth", "Window");
  return(C_TO_XEN_INT(XpSendAuth(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2))));
}

static XEN gxm_XpGetImageResolution(XEN arg1, XEN arg2)
{
  #define H_XpGetImageResolution "int XpGetImageResolution(Display *display, XPContext print_context)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpGetImageResolution", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpGetImageResolution", "XPContext");
  return(C_TO_XEN_INT(XpGetImageResolution(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2))));
}

static XEN gxm_XpGetOneAttribute(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XpGetOneAttribute "char *XpGetOneAttribute(Display *display, XPContext print_context, XPAttributes type,char *attribute_name)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpGetOneAttribute", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpGetOneAttribute", "XPContext");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XpGetOneAttribute", "XPAttributes");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XpGetOneAttribute", "char*");
  return(C_TO_XEN_STRING(XpGetOneAttribute(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2), 
					   (XPAttributes)XEN_TO_C_INT(arg3), XEN_TO_C_STRING(arg4))));
}

static XEN gxm_XpSetAttributes(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XpSetAttributes "void XpSetAttributes(Display *display, XPContext print_context, XPAttributes type,char *pool, XPAttrReplacement replacement_rule)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpSetAttributes", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpSetAttributes", "XPContext");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XpSetAttributes", "XPAttributes");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg4), arg4, 4, "XpSetAttributes", "char*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg5), arg5, 5, "XpSetAttributes", "XPAttrReplacement");
  XpSetAttributes(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2), 
		  (XPAttributes)XEN_TO_C_INT(arg3), XEN_TO_C_STRING(arg4), 
		  (XPAttrReplacement)XEN_TO_C_INT(arg5));
  return(XEN_FALSE);
}

static XEN gxm_XpGetAttributes(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XpGetAttributes "char *XpGetAttributes(Display *display, XPContext print_context, XPAttributes type)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpGetAttributes", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpGetAttributes", "XPContext");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XpGetAttributes", "XPAttributes");
  return(C_TO_XEN_STRING(XpGetAttributes(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2), 
					 (XPAttributes)XEN_TO_C_INT(arg3))));
}

static XEN gxm_XpGetScreenOfContext(XEN arg1, XEN arg2)
{
  #define H_XpGetScreenOfContext "Screen *XpGetScreenOfContext(Display *display, XPContext print_context)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpGetScreenOfContext", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpGetScreenOfContext", "XPContext");
  return(C_TO_XEN_Screen(XpGetScreenOfContext(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2))));
}

static XEN gxm_XpSendOneTicket(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XpSendOneTicket "Status XpSendOneTicket(Display *display,Window window, Xauth *ticket,Bool more)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpSendOneTicket", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XpSendOneTicket", "Window");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XpSendOneTicket", "Xauth*"); /* opaque */
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(arg4), arg4, 4, "XpSendOneTicket", "BOOLEAN");
  return(C_TO_XEN_INT(XpSendOneTicket(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
				      (Xauth *)XEN_TO_C_ULONG(arg3), XEN_TO_C_BOOLEAN(arg4))));
}

static XEN gxm_XpGetAuthParams(XEN arg1, XEN arg2)
{
  #define H_XpGetAuthParams "Status XpGetAuthParams(Display *print_display,Display *video_display)"
  /* DIFF: XpGetAuthParams omits 3 ref args and returns them 
   */
  Atom a1, a2;
  Display *dp;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpGetAuthParams", "Display*");
  XEN_ASSERT_TYPE(XEN_Display_P(arg2), arg2, 2, "XpGetAuthParams", "Display*");
  val = XpGetAuthParams(XEN_TO_C_Display(arg1), XEN_TO_C_Display(arg2), &dp, &a1, &a2);
  return(XEN_LIST_4(C_TO_XEN_INT(val),
		    C_TO_XEN_Display(dp),
		    C_TO_XEN_Atom(a1),
		    C_TO_XEN_Atom(a2)));
}

static XEN gxm_XpGetPdmStartParams(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XpGetPdmStartParams "Status XpGetPdmStartParams(Display *print_display,Window print_window, XPContext print_context, \
Display *video_display,Window video_window)"
  /* DIFF: XpGetPdmStartParams trailing 6 ref args omitted and returned
   */
  int val, i1, i2;
  Atom a1, a2;
  Display *dp;
  unsigned char *pars;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpGetPdmStartParams", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XpGetPdmStartParams", "Window");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg3), arg3, 3, "XpGetPdmStartParams", "XPContext");
  XEN_ASSERT_TYPE(XEN_Display_P(arg4), arg4, 4, "XpGetPdmStartParams", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg5), arg5, 5, "XpGetPdmStartParams", "Window");
  val = XpGetPdmStartParams(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
			    XEN_TO_C_XPContext(arg3), XEN_TO_C_Display(arg4), 
			    XEN_TO_C_Window(arg5), 
			    &dp, &a1, &a2, &i1, &pars, &i2);
  return(XEN_LIST_7(C_TO_XEN_INT(val),
		    C_TO_XEN_Display(dp),
		    C_TO_XEN_Atom(a1),
		    C_TO_XEN_Atom(a2),
		    C_TO_XEN_INT(i1),
		    C_TO_XEN_STRING((const char *)pars),
		    C_TO_XEN_INT(i2)));
}

static XEN gxm_XpQueryScreens(XEN arg1)
{
  #define H_XpQueryScreens "Screen **XpQueryScreens(Display *display)"
  /* DIFF: XpQueryScreens omits last arg, returns list of Screens
   */
  int len, i, loc;
  Screen **val = NULL;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpQueryScreens", "Display*");
  val = XpQueryScreens(XEN_TO_C_Display(arg1), &len);
  loc = xm_protect(lst);
  if (val)
    {
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(C_TO_XEN_Screen(val[i]), lst);
    }
  xm_unprotect_at(loc);
  return(lst);
}

static XEN gxm_XpQueryExtension(XEN arg1)
{
  #define H_XpQueryExtension "Bool XpQueryExtension(Display *display)"
  /* DIFF: XpQueryExtension omits and rtns last 2 args
   */
  int val, i1, i2;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpQueryExtension", "Display*");
  val = (int)XpQueryExtension(XEN_TO_C_Display(arg1), &i1, &i2);
  return(XEN_LIST_3(C_TO_XEN_BOOLEAN(val),
		    C_TO_XEN_INT(i1),
		    C_TO_XEN_INT(i2)));
}

static XEN gxm_XpQueryVersion(XEN arg1)
{
  #define H_XpQueryVersion "Status XpQueryVersion(Display *display)"
  /* DIFF: XpQueryVersion omits and rtns last 2 args
   */
  int val;
  short i1, i2;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpQueryVersion", "Display*");
  val = XpQueryVersion(XEN_TO_C_Display(arg1), &i1, &i2);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_INT((int)i1),
		    C_TO_XEN_INT((int)i2)));
}

static XEN gxm_XpFreePrinterList(XEN arg1)
{
  #define H_XpFreePrinterList "void XpFreePrinterList(XPPrinterList printer_list)"
  /* DIFF: XpFreePrinterList is a no-op
   */
  return(XEN_FALSE);
}

static XEN gxm_XpGetPageDimensions(XEN arg1, XEN arg2)
{
  #define H_XpGetPageDimensions "Status XpGetPageDimensions(Display *display, XPContext print_context)"
  /* DIFF: XpGetPageDimensions omits and rtns last 3 args
   */
  unsigned short i1, i2;
  XRectangle *r;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpGetPageDimensions", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpGetPageDimensions", "XPContext");
  r = (XRectangle *)CALLOC(1, sizeof(XRectangle));
  val = XpGetPageDimensions(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2), &i1, &i2, r);
  return(XEN_LIST_4(C_TO_XEN_INT(val),
		    C_TO_XEN_INT((int)i1),
		    C_TO_XEN_INT((int)i2),
		    C_TO_XEN_XRectangle(r)));
}

static XEN gxm_XpSetImageResolution(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  #define H_XpSetImageResolution "Bool XpSetImageResolution(Display *display, XPContext print_context,int image_res,int *prev_res)"
  /* DIFF: XpSetImageResolution last arg is int, val is returned
   */
  int i, val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpSetImageResolution", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpSetImageResolution", "XPContext");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg3), arg3, 3, "XpSetImageResolution", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XpSetImageResolution", "int*");
  val = XpSetImageResolution(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2), XEN_TO_C_INT(arg3), &i);
  return(XEN_LIST_2(C_TO_XEN_BOOLEAN(val),
		    C_TO_XEN_INT(i)));
}

static XEN gxm_XpGetPrinterList(XEN arg1, XEN arg2)
{
  #define H_XpGetPrinterList "XPPrinterList XpGetPrinterList(Display *display,char *printer_name)"
  /* DIFF: XpGetPrinterList returns list of printers, omits arg 3
   */
  XPPrinterList xp;
  int len;
  XEN lst = XEN_EMPTY_LIST;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpGetPrinterList", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XpGetPrinterList", "char*");
  xp = XpGetPrinterList(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2), &len);
  if (xp)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(XEN_LIST_2(C_TO_XEN_STRING(xp[i].name),
				  C_TO_XEN_STRING(xp[i].desc)),
		       lst);
      XpFreePrinterList(xp);
      xm_unprotect_at(loc);
    }
  return(lst);
}

static XEN gxm_XpSelectInput(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XpSelectInput "void XpSelectInput(Display *display, XPContext print_context,unsigned long event_mask)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpSelectInput", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpSelectInput", "XPContext");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XpSelectInput", "ulong");
  XpSelectInput(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2), XEN_TO_C_ULONG(arg3));
  return(XEN_FALSE);
}

static XEN gxm_XpInputSelected(XEN arg1, XEN arg2, XEN arg3)
{
  #define H_XpInputSelected "unsigned long XpInputSelected(Display *display, XPContext print_context,unsigned long *all_events_mask)"
  /* DIFF: XpInputSelected arg3 is int, is returned
   */
  unsigned long val, i1;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpInputSelected", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpInputSelected", "XPContext");
  XEN_ASSERT_TYPE(XEN_ULONG_P(arg3), arg3, 3, "XpInputSelected", "ulong*");
  i1 = XEN_TO_C_ULONG(arg3);
  val = XpInputSelected(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2), &i1);
  return(XEN_LIST_2(C_TO_XEN_ULONG(val),
		    C_TO_XEN_ULONG(i1)));
}

static void gxm_XPSaveProc(Display *display, XPContext context, unsigned char *sdata, unsigned int data_len, XPointer client_data)
{
  XEN data = (XEN)client_data;
  XEN_CALL_5(XEN_CAR(data),
	     C_TO_XEN_Display(display),
	     C_TO_XEN_XPContext(context),
	     C_TO_XEN_STRING((const char *)sdata),
	     C_TO_XEN_INT(data_len),
	     XEN_CADDR(data),
	     c__FUNCTION__);
  xm_unprotect(data); /* not sure about this -- should it be deferred until the finish proc? */
}

static void gxm_XPFinishProc(Display *display, XPContext context, XPGetDocStatus status, XPointer client_data)
{
  XEN data = (XEN)client_data;
  XEN_CALL_4(XEN_CADR(data),
	     C_TO_XEN_Display(display),
	     C_TO_XEN_XPContext(context),
	     C_TO_XEN_INT(status),
	     XEN_CADDR(data),
	     c__FUNCTION__);
  xm_unprotect(data);
}

static XEN gxm_XpGetDocumentData(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  #define H_XpGetDocumentData "Status XpGetDocumentData(Display *display, XPContext context, XPSaveProc save_proc, XPFinishProc finish_proc, XPointer client_data)"
  XEN descr;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpGetDocumentData", "Display*");
  XEN_ASSERT_TYPE(XEN_XPContext_P(arg2), arg2, 2, "XpGetDocumentData", "XPContext");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg3) && (XEN_REQUIRED_ARGS_OK(arg3, 5)), arg3, 3, "XpGetDocumentData", "XPSaveProc");
  XEN_ASSERT_TYPE(XEN_PROCEDURE_P(arg4) && (XEN_REQUIRED_ARGS_OK(arg4, 4)), arg4, 4, "XpGetDocumentData", "XPFinishProc");
  descr = XEN_LIST_3(arg3, arg4, arg5);
  xm_protect(descr);
  return(C_TO_XEN_INT(XpGetDocumentData(XEN_TO_C_Display(arg1), XEN_TO_C_XPContext(arg2), 
					(XPSaveProc)gxm_XPSaveProc,
					(XPFinishProc)gxm_XPFinishProc,
					(char *)descr)));
}

static XEN gxm_XpPutDocumentData(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5, XEN arg6)
{
  #define H_XpPutDocumentData "void XpPutDocumentData(Display *display,Drawable drawable,unsigned char *data,int data_len,char *doc_fmt,char *options)"
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpPutDocumentData", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XpPutDocumentData", "Window");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XpPutDocumentData", "uchar*");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(arg4), arg4, 4, "XpPutDocumentData", "int");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg5), arg5, 5, "XpPutDocumentData", "char*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg6), arg6, 6, "XpPutDocumentData", "char*");
  XpPutDocumentData(XEN_TO_C_Display(arg1), XEN_TO_C_Window(arg2), 
		    (unsigned char *)XEN_TO_C_STRING(arg3), 
		    XEN_TO_C_INT(arg4), XEN_TO_C_STRING(arg5), XEN_TO_C_STRING(arg6));
  return(XEN_FALSE);
}
#endif


/* ---------------------------------------------------------------------------------------------------- */
#if HAVE_XPM

XM_TYPE_PTR_NO_C2X(XpmImage, XpmImage *)
XM_TYPE_PTR_NO_C2X(XpmAttributes, XpmAttributes *) /* _OBJ?? */
XM_TYPE_PTR_NO_C2X(XpmColorSymbol, XpmColorSymbol *)

#if HAVE_XPM_CREATE_XPM_IMAGE_FROM_PIXMAP
static XEN gxm_XpmCreateXpmImageFromPixmap(XEN arg1, XEN arg2, XEN arg3, XEN arg5)
{
  XpmImage *image;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpmCreateXpmImageFromPixmap", "Display*");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg2), arg2, 2, "XpmCreateXpmImageFromPixmap", "Pixmap");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg3), arg3, 3, "XpmCreateXpmImageFromPixmap", "Pixmap");
  XEN_ASSERT_TYPE(XEN_XpmAttributes_P(arg5) || XEN_FALSE_P(arg5), arg5, 5, "XpmCreateXpmImageFromPixmap", "XpmAttributes*");
  image = (XpmImage *)CALLOC(1, sizeof(XpmImage));
  val = XpmCreateXpmImageFromPixmap(XEN_TO_C_Display(arg1), 
				    XEN_TO_C_Pixmap(arg2), 
				    XEN_TO_C_Pixmap(arg3), 
				    image,
				    (XEN_FALSE_P(arg5)) ? NULL : XEN_TO_C_XpmAttributes(arg5));
  if (val == XpmSuccess)
    return(WRAP_FOR_XEN_OBJ("XpmImage", image));
  FREE(image);
  return(C_TO_XEN_INT(val));
}

static XEN gxm_XpmCreatePixmapFromXpmImage(XEN arg1, XEN arg2, XEN arg3, XEN arg4)
{
  /* DIFF: XpmCreatePixmapFromXpmImage omits and returns pixmap args
   */
  Pixmap p1, p2;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpmCreatePixmapFromXpmImage", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XpmCreatePixmapFromXpmImage", "Drawable");
  XEN_ASSERT_TYPE(XEN_XpmImage_P(arg3), arg3, 3, "XpmCreatePixmapFromXpmImage", "XpmImage*");
  XEN_ASSERT_TYPE(XEN_XpmAttributes_P(arg4) || XEN_FALSE_P(arg4), arg4, 4, "XpmCreatePixmapFromXpmImage", "XpmAttributes*");
  val = XpmCreatePixmapFromXpmImage(XEN_TO_C_Display(arg1), 
				    XEN_TO_C_Window(arg2), 
				    XEN_TO_C_XpmImage(arg3), 
				    &p1, &p2,
				    (XEN_FALSE_P(arg4)) ? NULL : XEN_TO_C_XpmAttributes(arg4));
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_Pixmap(p1),
		    C_TO_XEN_Pixmap(p2)));
}
#endif

static XEN gxm_XpmReadFileToPixmap(XEN arg1, XEN arg2, XEN arg3, XEN arg6)
{
  /* DIFF: XpmReadFileToPixmap omits and returns pixmap args
   */
  Pixmap p1, p2;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpmReadFileToPixmap", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XpmReadFileToPixmap", "Drawable");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg3), arg3, 3, "XpmReadFileToPixmap", "char*");
  XEN_ASSERT_TYPE(XEN_XpmAttributes_P(arg6) || XEN_FALSE_P(arg6), arg6, 6, "XpmReadFileToPixmap", "XpmAttributes*");
  val = XpmReadFileToPixmap(XEN_TO_C_Display(arg1), 
			    XEN_TO_C_Window(arg2), 
			    XEN_TO_C_STRING(arg3), 
			    &p1, &p2,
			    (XEN_FALSE_P(arg6)) ? NULL : XEN_TO_C_XpmAttributes(arg6));
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_Pixmap(p1),
		    C_TO_XEN_Pixmap(p2)));
}

static XEN gxm_XpmReadFileToXpmImage(XEN arg1)
{
  int val;
  XpmImage *image;
  XEN_ASSERT_TYPE(XEN_STRING_P(arg1), arg1, 1, "XpmReadFileToXpmImage", "char*");
  image = (XpmImage *)CALLOC(1, sizeof(XpmImage));
  val = XpmReadFileToXpmImage(XEN_TO_C_STRING(arg1), image, NULL);
  if (val == XpmSuccess)
    return(WRAP_FOR_XEN_OBJ("XpmImage", image));
  FREE(image);
  return(C_TO_XEN_INT(val));
}

static XEN gxm_XpmCreatePixmapFromData(XEN arg1, XEN arg2, XEN larg3, XEN arg6)
{
  /* DIFF: XpmCreatePixmapFromData omits and returns pixmap args, arg3 (bits) is list of strings
   */
  Pixmap p1, p2;
  int val, i, len;
  char **bits;
  XEN arg3;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpmCreatePixmapFromData", "Display*");
  XEN_ASSERT_TYPE(XEN_Window_P(arg2), arg2, 2, "XpmCreatePixmapFromData", "Drawable");
  XEN_ASSERT_TYPE(XEN_LIST_P(larg3), larg3, 3, "XpmCreatePixmapFromData", "list of char*");
  XEN_ASSERT_TYPE(XEN_XpmAttributes_P(arg6) || XEN_FALSE_P(arg6), arg6, 6, "XpmCreatePixmapFromData", "XpmAttributes*");
  arg3 = XEN_COPY_ARG(larg3);
  len = XEN_LIST_LENGTH(arg3);
  if (len <= 0) XEN_ASSERT_TYPE(0, arg3, 3, "XpmCreatePixmapFromData", "positive integer");
  bits = (char **)CALLOC(len, sizeof(char *));
  for (i = 0; i < len; i++, arg3 = XEN_CDR(arg3))
    bits[i] = strdup(XEN_TO_C_STRING(XEN_CAR(arg3)));
  val = XpmCreatePixmapFromData(XEN_TO_C_Display(arg1), 
				XEN_TO_C_Window(arg2), 
				bits,
				&p1, &p2,
				(XEN_FALSE_P(arg6)) ? NULL : XEN_TO_C_XpmAttributes(arg6));
  for (i = 0; i < len; i++)
    if (bits[i]) free(bits[i]);
  FREE(bits);
  return(XEN_LIST_3(C_TO_XEN_INT(val),
		    C_TO_XEN_Pixmap(p1),
		    C_TO_XEN_Pixmap(p2)));
}

static XEN gxm_XpmWriteFileFromPixmap(XEN arg1, XEN arg2, XEN arg3, XEN arg4, XEN arg5)
{
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpmWriteFileFromPixmap", "Display*");
  XEN_ASSERT_TYPE(XEN_STRING_P(arg2), arg2, 2, "XpmWriteFileFromPixmap", "char*");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg3), arg3, 3, "XpmWriteFileFromPixmap", "Pixmap");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg4), arg4, 4, "XpmWriteFileFromPixmap", "Pixmap");
  XEN_ASSERT_TYPE(XEN_XpmAttributes_P(arg5) || XEN_FALSE_P(arg5), arg5, 5, "XpmWriteFileFromPixmap", "XpmAttributes*");
  return(C_TO_XEN_INT(XpmWriteFileFromPixmap(XEN_TO_C_Display(arg1), XEN_TO_C_STRING(arg2), 
					     XEN_TO_C_Pixmap(arg3), XEN_TO_C_Pixmap(arg4), 
					     (XEN_FALSE_P(arg5)) ? NULL : XEN_TO_C_XpmAttributes(arg5))));
}

static XEN gxm_XpmCreateDataFromPixmap(XEN arg1, XEN arg3, XEN arg4, XEN arg5)
{
  /* DIFF: XpmCreateDataFromPixmap arg2 omitted and rtn'd
   */
  char **buf = NULL;
  int val;
  XEN_ASSERT_TYPE(XEN_Display_P(arg1), arg1, 1, "XpmCreateDataFromPixmap", "Display*");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg3), arg3, 3, "XpmCreateDataFromPixmap", "Pixmap");
  XEN_ASSERT_TYPE(XEN_Pixmap_P(arg4), arg4, 4, "XpmCreateDataFromPixmap", "Pixmap");
  XEN_ASSERT_TYPE(XEN_XpmAttributes_P(arg5) || XEN_FALSE_P(arg5), arg5, 5, "XpmCreateDataFromPixmap", "XpmAttributes*");
  val = XpmCreateDataFromPixmap(XEN_TO_C_Display(arg1), &buf, 
				XEN_TO_C_Pixmap(arg3), 
				XEN_TO_C_Pixmap(arg4), 
				(XEN_FALSE_P(arg5)) ? NULL : XEN_TO_C_XpmAttributes(arg5));
  return(XEN_LIST_2(C_TO_XEN_INT(val),
		    C_TO_XEN_ULONG(buf)));
}

#if HAVE_XPM_GET_ERROR_STRING
static XEN gxm_XpmGetErrorString(XEN err)
{
  #define H_XpmGetErrorString "(XpmGetErrorString err) -> string describing error"
  XEN_ASSERT_TYPE(XEN_INTEGER_P(err), err, XEN_ONLY_ARG, "XpmGetErrorString", "an Xpm status code");
  return(C_TO_XEN_STRING(XpmGetErrorString(XEN_TO_C_INT(err))));
}
#endif

static XEN gxm_XpmColorSymbol(XEN name, XEN value, XEN pixel)
{
  XpmColorSymbol *r;
  XEN_ASSERT_TYPE(XEN_STRING_P(name), name, 1, "XpmColorSymbol", "char*");
  XEN_ASSERT_TYPE(XEN_FALSE_P(value) || XEN_STRING_P(value), value, 2, "XpmColorSymbol", "char*");
  XEN_ASSERT_TYPE(XEN_Pixel_P(pixel), pixel, 3, "XpmColorSymbol", "Pixel");
  r = (XpmColorSymbol *)CALLOC(1, sizeof(XpmColorSymbol));
  r->name = XEN_TO_C_STRING(name);
  r->value = (XEN_FALSE_P(value)) ? NULL : XEN_TO_C_STRING(value);
  r->pixel = XEN_TO_C_Pixel(pixel);
  return(xen_return_first(WRAP_FOR_XEN_OBJ("XpmColorSymbol",r), name, value));
}

static XEN gxm_XpmImage(XEN width, XEN height, XEN cpp, XEN ncolors, XEN data)
{
  XpmImage *r;
  XEN_ASSERT_TYPE(XEN_ULONG_P(width), width, 1, "XpmImage", "ulong");
  XEN_ASSERT_TYPE(XEN_ULONG_P(height), height, 2, "XpmImage", "ulong");
  XEN_ASSERT_TYPE(XEN_ULONG_P(cpp), cpp, 3, "XpmImage", "ulong");
  XEN_ASSERT_TYPE(XEN_ULONG_P(ncolors), ncolors, 4, "XpmImage", "ulong");
  XEN_ASSERT_TYPE(XEN_ULONG_P(data), data, 5, "XpmImage", "ulong");
  r = (XpmImage *)CALLOC(1, sizeof(XpmImage));
  r->width = XEN_TO_C_ULONG(width);
  r->height = XEN_TO_C_ULONG(height);
  r->cpp = XEN_TO_C_ULONG(cpp);
  r->ncolors = XEN_TO_C_ULONG(ncolors);
  r->data = (unsigned int *)XEN_TO_C_ULONG(data);
  return(WRAP_FOR_XEN_OBJ("XpmImage", r));
}

static XEN gxm_XpmAttributes(void)
{
  return(WRAP_FOR_XEN_OBJ("XpmAttributes", 
			  (XpmAttributes *)CALLOC(1, sizeof(XpmAttributes))));
}

static XEN gxm_cpp(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE((XEN_XpmImage_P(ptr)) || (XEN_XpmAttributes_P(ptr)), ptr, XEN_ONLY_ARG, "cpp", "XpmImage");
  if (XEN_XpmImage_P(ptr)) return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XpmImage(ptr))->cpp)));
  return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XpmAttributes(ptr))->cpp)));
}

static XEN gxm_set_cpp(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE((XEN_XpmImage_P(ptr)) || (XEN_XpmAttributes_P(ptr)), ptr, XEN_ARG_1, "cpp", "XpmImage");
  if (XEN_XpmImage_P(ptr)) (XEN_TO_C_XpmImage(ptr))->cpp = XEN_TO_C_ULONG(val);
  (XEN_TO_C_XpmAttributes(ptr))->cpp = XEN_TO_C_ULONG(val);
  return(val);
}

static XEN gxm_ncolors(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE((XEN_XpmImage_P(ptr)) || (XEN_XpmAttributes_P(ptr)), ptr, XEN_ONLY_ARG, "ncolors", "XpmImage");
  if (XEN_XpmImage_P(ptr)) return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XpmImage(ptr))->ncolors)));
  return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XpmAttributes(ptr))->ncolors)));
}

static XEN gxm_set_ncolors(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE((XEN_XpmImage_P(ptr)) || (XEN_XpmAttributes_P(ptr)), ptr, XEN_ARG_1, "ncolors", "XpmImage");
  if (XEN_XpmImage_P(ptr)) (XEN_TO_C_XpmImage(ptr))->ncolors = XEN_TO_C_ULONG(val);
  else (XEN_TO_C_XpmAttributes(ptr))->ncolors = XEN_TO_C_ULONG(val);
  return(val);
}

#if 0
static XEN gxm_set_data(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XpmImage_P(ptr), ptr, XEN_ARG_1, "data", "XpmImage");
  (XEN_TO_C_XpmImage(ptr))->data = (unsigned int *)XEN_TO_C_ULONG(val);
  return(val);
}
#endif

static XEN gxm_valuemask(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ONLY_ARG, "valuemask", "XpmAttributes");
  return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XpmAttributes(ptr))->valuemask)));
}

static XEN gxm_set_valuemask(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ARG_1, "valuemask", "XpmAttributes");
  (XEN_TO_C_XpmAttributes(ptr))->valuemask = XEN_TO_C_ULONG(val);
  return(val);
}

static XEN gxm_x_hotspot(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ONLY_ARG, "x_hotspot", "XpmAttributes");
  return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XpmAttributes(ptr))->x_hotspot)));
}

static XEN gxm_set_x_hotspot(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ARG_1, "x_hotspot", "XpmAttributes");
  (XEN_TO_C_XpmAttributes(ptr))->x_hotspot = XEN_TO_C_ULONG(val);
  return(val);
}

static XEN gxm_y_hotspot(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ONLY_ARG, "y_hotspot", "XpmAttributes");
  return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XpmAttributes(ptr))->y_hotspot)));
}

static XEN gxm_set_y_hotspot(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ARG_1, "y_hotspot", "XpmAttributes");
  (XEN_TO_C_XpmAttributes(ptr))->y_hotspot = XEN_TO_C_ULONG(val);
  return(val);
}

static XEN gxm_colorsymbols(XEN ptr)
{
  XpmAttributes *atr;
  int i, len;
  XEN lst = XEN_EMPTY_LIST;
  XM_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ONLY_ARG, "colorsymbols", "XpmAttributes");
  atr = XEN_TO_C_XpmAttributes(ptr);
  len = atr->numsymbols;
  if (len > 0)
    {
      XpmColorSymbol *cols;
      cols = atr->colorsymbols;
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(WRAP_FOR_XEN("XpmColorSymbol", &(cols[i])), lst);
    }
  return(lst);
}

static XEN gxm_set_colorsymbols(XEN ptr, XEN vals)
{
  XpmAttributes *atr;
  int i, len;
  XEN lst;
  XM_SET_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ARG_1, "colorsymbols", "XpmAttributes");
  XM_SET_FIELD_ASSERT_TYPE(XEN_LIST_P(vals), vals, XEN_ARG_2, "colorsymbols", "list of XpmColorSymbols");
  atr = XEN_TO_C_XpmAttributes(ptr);
  len = XEN_LIST_LENGTH(vals);
  if (len > 0)
    {
      XpmColorSymbol *cols = NULL, *cur;
      cols = (XpmColorSymbol *)CALLOC(len, sizeof(XpmColorSymbol));
      for (lst = XEN_COPY_ARG(vals), i = 0; i < len; i++, lst = XEN_CDR(lst))
	{
	  cur = XEN_TO_C_XpmColorSymbol(XEN_CAR(lst));
	  if (cur->name) cols[i].name = strdup(cur->name);
	  if (cur->value) cols[i].value = strdup(cur->value);
	  cols[i].pixel = cur->pixel;
	}
      atr->colorsymbols = cols;
    }
  return(vals);
}

static XEN gxm_numsymbols(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ONLY_ARG, "numsymbols", "XpmAttributes");
  return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XpmAttributes(ptr))->numsymbols)));
}

static XEN gxm_set_numsymbols(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ARG_1, "numsymbols", "XpmAttributes");
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "numsymbols", "integer");
  (XEN_TO_C_XpmAttributes(ptr))->numsymbols = XEN_TO_C_INT(val);
  return(val);
}

/* pixels is the list -- not sure what good npixels is without pixels */
static XEN gxm_npixels(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ONLY_ARG, "npixels", "XpmAttributes");
  return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XpmAttributes(ptr))->npixels)));
}

static XEN gxm_set_npixels(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ARG_1, "npixels", "XpmAttributes");
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "npixels", "integer");
  (XEN_TO_C_XpmAttributes(ptr))->npixels = XEN_TO_C_INT(val);
  return(val);
}

#endif
/* HAVE_XPM */

#if WITH_EDITRES
static XEN gxm_XEditResCheckMessages(XEN widget, XEN data, XEN event, XEN cont)
{
  Boolean flag;
  XEN_ASSERT_TYPE(XEN_Widget_P(widget), widget, XEN_ARG_1, "widget", "_XEditResCheckMessages");
  XEN_ASSERT_TYPE(XEN_XEvent_P(event), event, XEN_ARG_3, "XEvent", "_XEditResCheckMessages");
  XEN_ASSERT_TYPE(XEN_BOOLEAN_P(cont), cont, XEN_ARG_4, "boolean", "_XEditResCheckMessages");
  flag = XEN_TO_C_BOOLEAN(cont);
  _XEditResCheckMessages(XEN_TO_C_Widget(widget),
			 (XtPointer)data,
			 XEN_TO_C_XEvent(event),
			 &flag);
  return(XEN_FALSE);
}
#endif



/* ---------------------------------------------------------------------------------------------------- */

#if HAVE_GUILE
static void define_procedures(void)
{
  #define XM_DEFINE_PROCEDURE(Name, Value, A1, A2, A3, Help) XEN_DEFINE_PROCEDURE(XM_PREFIX #Name XM_POSTFIX, Value, A1, A2, A3, Help)

  xm_gc_table = XEN_MAKE_VECTOR(1, XEN_FALSE);
  XEN_PROTECT_FROM_GC(xm_gc_table);
  xm_protected_size = 512;
  xm_protected = XEN_MAKE_VECTOR(xm_protected_size, XEN_FALSE);
  XEN_VECTOR_SET(xm_gc_table, 0, xm_protected);

#if HAVE_XM_XP
  XM_DEFINE_PROCEDURE(XpStartPage, gxm_XpStartPage, 2, 0, 0, H_XpStartPage);
  XM_DEFINE_PROCEDURE(XpEndPage, gxm_XpEndPage, 1, 0, 0, H_XpEndPage);
  XM_DEFINE_PROCEDURE(XpCancelPage, gxm_XpCancelPage, 2, 0, 0, H_XpCancelPage);
  XM_DEFINE_PROCEDURE(XpStartJob, gxm_XpStartJob, 2, 0, 0, H_XpStartJob);
  XM_DEFINE_PROCEDURE(XpEndJob, gxm_XpEndJob, 1, 0, 0, H_XpEndJob);
  XM_DEFINE_PROCEDURE(XpCancelJob, gxm_XpCancelJob, 2, 0, 0, H_XpCancelJob);
  XM_DEFINE_PROCEDURE(XpStartDoc, gxm_XpStartDoc, 2, 0, 0, H_XpStartDoc);
  XM_DEFINE_PROCEDURE(XpEndDoc, gxm_XpEndDoc, 1, 0, 0, H_XpEndDoc);
  XM_DEFINE_PROCEDURE(XpCancelDoc, gxm_XpCancelDoc, 2, 0, 0, H_XpCancelDoc);
  XM_DEFINE_PROCEDURE(XpRehashPrinterList, gxm_XpRehashPrinterList, 1, 0, 0, H_XpRehashPrinterList);
  XM_DEFINE_PROCEDURE(XpCreateContext, gxm_XpCreateContext, 2, 0, 0, H_XpCreateContext);
  XM_DEFINE_PROCEDURE(XpSetContext, gxm_XpSetContext, 2, 0, 0, H_XpSetContext);
  XM_DEFINE_PROCEDURE(XpGetContext, gxm_XpGetContext, 1, 0, 0, H_XpGetContext);
  XM_DEFINE_PROCEDURE(XpDestroyContext, gxm_XpDestroyContext, 2, 0, 0, H_XpDestroyContext);
  XM_DEFINE_PROCEDURE(XpGetLocaleNetString, gxm_XpGetLocaleNetString, 0, 0, 0, H_XpGetLocaleNetString);
  XM_DEFINE_PROCEDURE(XpNotifyPdm, gxm_XpNotifyPdm, 6, 0, 0, H_XpNotifyPdm);
  XM_DEFINE_PROCEDURE(XpSendAuth, gxm_XpSendAuth, 2, 0, 0, H_XpSendAuth);
  XM_DEFINE_PROCEDURE(XpGetImageResolution, gxm_XpGetImageResolution, 2, 0, 0, H_XpGetImageResolution);
  XM_DEFINE_PROCEDURE(XpGetAttributes, gxm_XpGetAttributes, 3, 0, 0, H_XpGetAttributes);
  XM_DEFINE_PROCEDURE(XpSetAttributes, gxm_XpSetAttributes, 5, 0, 0, H_XpSetAttributes);
  XM_DEFINE_PROCEDURE(XpGetOneAttribute, gxm_XpGetOneAttribute, 4, 0, 0, H_XpGetOneAttribute);
  XM_DEFINE_PROCEDURE(XpGetScreenOfContext, gxm_XpGetScreenOfContext, 2, 0, 0, H_XpGetScreenOfContext);
  XM_DEFINE_PROCEDURE(XpFreePrinterList, gxm_XpFreePrinterList, 1, 0, 0, H_XpFreePrinterList);
  XM_DEFINE_PROCEDURE(XpQueryVersion, gxm_XpQueryVersion, 1, 0, 0, H_XpQueryVersion);
  XM_DEFINE_PROCEDURE(XpQueryExtension, gxm_XpQueryExtension, 1, 0, 0, H_XpQueryExtension);
  XM_DEFINE_PROCEDURE(XpQueryScreens, gxm_XpQueryScreens, 1, 0, 0, H_XpQueryScreens);
  XM_DEFINE_PROCEDURE(XpGetPdmStartParams, gxm_XpGetPdmStartParams, 5, 0, 0, H_XpGetPdmStartParams);
  XM_DEFINE_PROCEDURE(XpGetAuthParams, gxm_XpGetAuthParams, 2, 0, 0, H_XpGetAuthParams);
  XM_DEFINE_PROCEDURE(XpSendOneTicket, gxm_XpSendOneTicket, 4, 0, 0, H_XpSendOneTicket);
  XM_DEFINE_PROCEDURE(XpGetPageDimensions, gxm_XpGetPageDimensions, 2, 0, 0, H_XpGetPageDimensions);
  XM_DEFINE_PROCEDURE(XpSetImageResolution, gxm_XpSetImageResolution, 4, 0, 0, H_XpSetImageResolution);
  XM_DEFINE_PROCEDURE(XpGetPrinterList, gxm_XpGetPrinterList, 2, 0, 0, H_XpGetPrinterList);
  XM_DEFINE_PROCEDURE(XpSelectInput, gxm_XpSelectInput, 3, 0, 0, H_XpSelectInput);
  XM_DEFINE_PROCEDURE(XpInputSelected, gxm_XpInputSelected, 3, 0, 0, H_XpInputSelected);
  XM_DEFINE_PROCEDURE(XpPutDocumentData, gxm_XpPutDocumentData, 6, 0, 0, H_XpPutDocumentData);
  XM_DEFINE_PROCEDURE(XpGetDocumentData, gxm_XpGetDocumentData, 5, 0, 0, H_XpGetDocumentData);
#endif
#if HAVE_MOTIF
  XM_DEFINE_PROCEDURE(XtSetArg, gxm_XtSetArg, 3, 0, 0, H_XtSetArg);
  XM_DEFINE_PROCEDURE(XtManageChildren, gxm_XtManageChildren, 1, 1, 0, H_XtManageChildren);
  XM_DEFINE_PROCEDURE(XtManageChild, gxm_XtManageChild, 1, 0, 0, H_XtManageChild);
  XM_DEFINE_PROCEDURE(XtUnmanageChildren, gxm_XtUnmanageChildren, 1, 1, 0, H_XtUnmanageChildren);
  XM_DEFINE_PROCEDURE(XtUnmanageChild, gxm_XtUnmanageChild, 1, 0, 0, H_XtUnmanageChild);
  XM_DEFINE_PROCEDURE(XtDispatchEvent, gxm_XtDispatchEvent, 1, 0, 0, H_XtDispatchEvent);
  XM_DEFINE_PROCEDURE(XtCallAcceptFocus, gxm_XtCallAcceptFocus, 2, 0, 0, H_XtCallAcceptFocus);
  XM_DEFINE_PROCEDURE(XtAppPeekEvent, gxm_XtAppPeekEvent, 1, 0, 0, H_XtAppPeekEvent);
#if MOTIF_2
  XM_DEFINE_PROCEDURE(XtIsSubclass, gxm_XtIsSubclass, 2, 0, 0, H_XtIsSubclass);
#endif
  XM_DEFINE_PROCEDURE(XtIsObject, gxm_XtIsObject, 1, 0, 0, H_XtIsObject);
  XM_DEFINE_PROCEDURE(XtIsManaged, gxm_XtIsManaged, 1, 0, 0, H_XtIsManaged);
  XM_DEFINE_PROCEDURE(XtIsRealized, gxm_XtIsRealized, 1, 0, 0, H_XtIsRealized);
  XM_DEFINE_PROCEDURE(XtIsSensitive, gxm_XtIsSensitive, 1, 0, 0, H_XtIsSensitive);
  XM_DEFINE_PROCEDURE(XtOwnSelection, gxm_XtOwnSelection, 6, 0, 0, H_XtOwnSelection);
  XM_DEFINE_PROCEDURE(XtOwnSelectionIncremental, gxm_XtOwnSelectionIncremental, 8, 0, 0, H_XtOwnSelectionIncremental);
  XM_DEFINE_PROCEDURE(XtMakeResizeRequest, gxm_XtMakeResizeRequest, 3, 0, 0, H_XtMakeResizeRequest);
  XM_DEFINE_PROCEDURE(XtTranslateCoords, gxm_XtTranslateCoords, 3, 0, 0, H_XtTranslateCoords);
  XM_DEFINE_PROCEDURE(XtKeysymToKeycodeList, gxm_XtKeysymToKeycodeList, 2, 0, 0, H_XtKeysymToKeycodeList);
  XM_DEFINE_PROCEDURE(XtParseTranslationTable, gxm_XtParseTranslationTable, 1, 0, 0, H_XtParseTranslationTable);
  XM_DEFINE_PROCEDURE(XtParseAcceleratorTable, gxm_XtParseAcceleratorTable, 1, 0, 0, H_XtParseAcceleratorTable);
  XM_DEFINE_PROCEDURE(XtOverrideTranslations, gxm_XtOverrideTranslations, 2, 0, 0, H_XtOverrideTranslations);
  XM_DEFINE_PROCEDURE(XtAugmentTranslations, gxm_XtAugmentTranslations, 2, 0, 0, H_XtAugmentTranslations);
  XM_DEFINE_PROCEDURE(XtInstallAccelerators, gxm_XtInstallAccelerators, 2, 0, 0, H_XtInstallAccelerators);
  XM_DEFINE_PROCEDURE(XtInstallAllAccelerators, gxm_XtInstallAllAccelerators, 2, 0, 0, H_XtInstallAllAccelerators);
  XM_DEFINE_PROCEDURE(XtUninstallTranslations, gxm_XtUninstallTranslations, 1, 0, 0, H_XtUninstallTranslations);
  XM_DEFINE_PROCEDURE(XtAppAddActions, gxm_XtAppAddActions, 2, 0, 0, H_XtAppAddActions);
  XM_DEFINE_PROCEDURE(XtAppAddActionHook, gxm_XtAppAddActionHook, 2, 1, 0, H_XtAppAddActionHook);
  XM_DEFINE_PROCEDURE(XtRemoveActionHook, gxm_XtRemoveActionHook, 1, 0, 0, H_XtRemoveActionHook);
  XM_DEFINE_PROCEDURE(XtGetActionList, gxm_XtGetActionList, 1, 0, 0, H_XtGetActionList);
  XM_DEFINE_PROCEDURE(XtCallActionProc, gxm_XtCallActionProc, 4, 1, 0, H_XtCallActionProc);
  XM_DEFINE_PROCEDURE(XtRegisterGrabAction, gxm_XtRegisterGrabAction, 5, 0, 0, H_XtRegisterGrabAction);
  XM_DEFINE_PROCEDURE(XtSetMultiClickTime, gxm_XtSetMultiClickTime, 2, 0, 0, H_XtSetMultiClickTime);
  XM_DEFINE_PROCEDURE(XtGetMultiClickTime, gxm_XtGetMultiClickTime, 1, 0, 0, H_XtGetMultiClickTime);
  XM_DEFINE_PROCEDURE(XtGetResourceList, gxm_XtGetResourceList, 1, 0, 0, H_XtGetResourceList);
  XM_DEFINE_PROCEDURE(XtGetActionKeysym, gxm_XtGetActionKeysym, 1, 0, 0, H_XtGetActionKeysym);
  XM_DEFINE_PROCEDURE(XtTranslateKeycode, gxm_XtTranslateKeycode, 3, 0, 0, H_XtTranslateKeycode);
  XM_DEFINE_PROCEDURE(XtTranslateKey, gxm_XtTranslateKey, 3, 0, 0, H_XtTranslateKey);
  XM_DEFINE_PROCEDURE(XtSetKeyTranslator, gxm_XtSetKeyTranslator, 2, 0, 0, H_XtSetKeyTranslator);
  XM_DEFINE_PROCEDURE(XtRegisterCaseConverter, gxm_XtRegisterCaseConverter, 4, 0, 0, H_XtRegisterCaseConverter);
  XM_DEFINE_PROCEDURE(XtConvertCase, gxm_XtConvertCase, 2, 0, 0, H_XtConvertCase);
  XM_DEFINE_PROCEDURE(XtAddEventHandler, gxm_XtAddEventHandler, 4, 1, 0, H_XtAddEventHandler);
  XM_DEFINE_PROCEDURE(XtRemoveEventHandler, gxm_XtRemoveEventHandler, 5, 0, 0, H_XtRemoveEventHandler);
  XM_DEFINE_PROCEDURE(XtAddRawEventHandler, gxm_XtAddRawEventHandler, 5, 0, 0, H_XtAddRawEventHandler);
  XM_DEFINE_PROCEDURE(XtRemoveRawEventHandler, gxm_XtRemoveRawEventHandler, 5, 0, 0, H_XtRemoveRawEventHandler);
  XM_DEFINE_PROCEDURE(XtInsertEventHandler, gxm_XtInsertEventHandler, 6, 0, 0, H_XtInsertEventHandler);
  XM_DEFINE_PROCEDURE(XtInsertRawEventHandler, gxm_XtInsertRawEventHandler, 6, 0, 0, H_XtInsertRawEventHandler);
  XM_DEFINE_PROCEDURE(XtDispatchEventToWidget, gxm_XtDispatchEventToWidget, 2, 0, 0, H_XtDispatchEventToWidget);
  XM_DEFINE_PROCEDURE(XtBuildEventMask, gxm_XtBuildEventMask, 1, 0, 0, H_XtBuildEventMask);
  XM_DEFINE_PROCEDURE(XtAddGrab, gxm_XtAddGrab, 3, 0, 0, H_XtAddGrab);
  XM_DEFINE_PROCEDURE(XtRemoveGrab, gxm_XtRemoveGrab, 1, 0, 0, H_XtRemoveGrab);
  XM_DEFINE_PROCEDURE(XtAppProcessEvent, gxm_XtAppProcessEvent, 2, 0, 0, H_XtAppProcessEvent);
  XM_DEFINE_PROCEDURE(XtAppMainLoop, gxm_XtAppMainLoop, 1, 0, 0, H_XtAppMainLoop);
  XM_DEFINE_PROCEDURE(XtAddExposureToRegion, gxm_XtAddExposureToRegion, 2, 0, 0, H_XtAddExposureToRegion);
  XM_DEFINE_PROCEDURE(XtSetKeyboardFocus, gxm_XtSetKeyboardFocus, 2, 0, 0, H_XtSetKeyboardFocus);
  XM_DEFINE_PROCEDURE(XtGetKeyboardFocusWidget, gxm_XtGetKeyboardFocusWidget, 1, 0, 0, H_XtGetKeyboardFocusWidget);
  XM_DEFINE_PROCEDURE(XtLastEventProcessed, gxm_XtLastEventProcessed, 1, 0, 0, H_XtLastEventProcessed);
  XM_DEFINE_PROCEDURE(XtLastTimestampProcessed, gxm_XtLastTimestampProcessed, 1, 0, 0, H_XtLastTimestampProcessed);
  XM_DEFINE_PROCEDURE(XtAppAddTimeOut, gxm_XtAppAddTimeOut, 3, 1, 0, H_XtAppAddTimeOut);
  XM_DEFINE_PROCEDURE(XtRemoveTimeOut, gxm_XtRemoveTimeOut, 1, 0, 0, H_XtRemoveTimeOut);
  XM_DEFINE_PROCEDURE(XtAppAddInput, gxm_XtAppAddInput, 4, 1, 0, H_XtAppAddInput);
  XM_DEFINE_PROCEDURE(XtRemoveInput, gxm_XtRemoveInput, 1, 0, 0, H_XtRemoveInput);
  XM_DEFINE_PROCEDURE(XtAppNextEvent, gxm_XtAppNextEvent, 1, 0, 0, H_XtAppNextEvent);
  XM_DEFINE_PROCEDURE(XtAppPending, gxm_XtAppPending, 1, 0, 0, H_XtAppPending);
  XM_DEFINE_PROCEDURE(XtRealizeWidget, gxm_XtRealizeWidget, 1, 0, 0, H_XtRealizeWidget);
  XM_DEFINE_PROCEDURE(XtUnrealizeWidget, gxm_XtUnrealizeWidget, 1, 0, 0, H_XtUnrealizeWidget);
  XM_DEFINE_PROCEDURE(XtDestroyWidget, gxm_XtDestroyWidget, 1, 0, 0, H_XtDestroyWidget);
  XM_DEFINE_PROCEDURE(XtSetSensitive, gxm_XtSetSensitive, 2, 0, 0, H_XtSetSensitive);
  XM_DEFINE_PROCEDURE(XtNameToWidget, gxm_XtNameToWidget, 2, 0, 0, H_XtNameToWidget);
  XM_DEFINE_PROCEDURE(XtWindowToWidget, gxm_XtWindowToWidget, 2, 0, 0, H_XtWindowToWidget);
  XM_DEFINE_PROCEDURE(XtMergeArgLists, gxm_XtMergeArgLists, 4, 0, 0, H_XtMergeArgLists);
  XM_DEFINE_PROCEDURE(XtVaCreateArgsList, gxm_XtVaCreateArgsList, 2, 0, 0, H_XtVaCreateArgsList);
  XM_DEFINE_PROCEDURE(XtDisplay, gxm_XtDisplay, 1, 0, 0, H_XtDisplay);
  XM_DEFINE_PROCEDURE(XtDisplayOfObject, gxm_XtDisplayOfObject, 1, 0, 0, H_XtDisplayOfObject);
  XM_DEFINE_PROCEDURE(XtScreen, gxm_XtScreen, 1, 0, 0, H_XtScreen);
  XM_DEFINE_PROCEDURE(XtScreenOfObject, gxm_XtScreenOfObject, 1, 0, 0, H_XtScreenOfObject);
  XM_DEFINE_PROCEDURE(XtWindow, gxm_XtWindow, 1, 0, 0, H_XtWindow);
  XM_DEFINE_PROCEDURE(XtWindowOfObject, gxm_XtWindowOfObject, 1, 0, 0, H_XtWindowOfObject);
  XM_DEFINE_PROCEDURE(XtName, gxm_XtName, 1, 0, 0, H_XtName);
  XM_DEFINE_PROCEDURE(XtSuperclass, gxm_XtSuperclass, 1, 0, 0, H_XtSuperclass);
  XM_DEFINE_PROCEDURE(XtClass, gxm_XtClass, 1, 0, 0, H_XtClass);
  XM_DEFINE_PROCEDURE(XtParent, gxm_XtParent, 1, 0, 0, H_XtParent);
  XM_DEFINE_PROCEDURE(XtAddCallback, gxm_XtAddCallback, 3, 1, 0, H_XtAddCallback);
  XM_DEFINE_PROCEDURE(XtRemoveCallback, gxm_XtRemoveCallback, 3, 0, 0, H_XtRemoveCallback);
  XM_DEFINE_PROCEDURE(XtAddCallbacks, gxm_XtAddCallbacks, 3, 0, 0, H_XtAddCallbacks);
  XM_DEFINE_PROCEDURE(XtRemoveCallbacks, gxm_XtRemoveCallbacks, 3, 0, 0, H_XtRemoveCallbacks);
  XM_DEFINE_PROCEDURE(XtRemoveAllCallbacks, gxm_XtRemoveAllCallbacks, 2, 0, 0, H_XtRemoveAllCallbacks);
  XM_DEFINE_PROCEDURE(XtCallCallbacks, gxm_XtCallCallbacks, 3, 0, 0, H_XtCallCallbacks);
  XM_DEFINE_PROCEDURE(XtHasCallbacks, gxm_XtHasCallbacks, 2, 0, 0, H_XtHasCallbacks);
  XM_DEFINE_PROCEDURE(XtCreatePopupShell, gxm_XtCreatePopupShell, 4, 1, 0, H_XtCreatePopupShell);
  XM_DEFINE_PROCEDURE(XtVaCreatePopupShell, gxm_XtVaCreatePopupShell, 4, 0, 0, H_XtVaCreatePopupShell);
  XM_DEFINE_PROCEDURE(XtPopup, gxm_XtPopup, 2, 0, 0, H_XtPopup);
  XM_DEFINE_PROCEDURE(XtPopupSpringLoaded, gxm_XtPopupSpringLoaded, 1, 0, 0, H_XtPopupSpringLoaded);
  XM_DEFINE_PROCEDURE(XtCallbackNone, gxm_XtCallbackNone, 3, 0, 0, H_XtCallbackNone);
  XM_DEFINE_PROCEDURE(XtCallbackNonexclusive, gxm_XtCallbackNonexclusive, 3, 0, 0, H_XtCallbackNonexclusive);
  XM_DEFINE_PROCEDURE(XtCallbackExclusive, gxm_XtCallbackExclusive, 3, 0, 0, H_XtCallbackExclusive);
  XM_DEFINE_PROCEDURE(XtPopdown, gxm_XtPopdown, 1, 0, 0, H_XtPopdown);
  XM_DEFINE_PROCEDURE(XtCallbackPopdown, gxm_XtCallbackPopdown, 3, 0, 0, H_XtCallbackPopdown);
  XM_DEFINE_PROCEDURE(XtCreateWidget, gxm_XtCreateWidget, 4, 1, 0, H_XtCreateWidget);
  XM_DEFINE_PROCEDURE(XtCreateManagedWidget, gxm_XtCreateManagedWidget, 4, 1, 0, H_XtCreateManagedWidget);
  XM_DEFINE_PROCEDURE(XtVaCreateWidget, gxm_XtVaCreateWidget, 4, 0, 0, H_XtVaCreateWidget);
  XM_DEFINE_PROCEDURE(XtVaCreateManagedWidget, gxm_XtVaCreateManagedWidget, 4, 0, 0, H_XtVaCreateManagedWidget);
  XM_DEFINE_PROCEDURE(XtAppCreateShell, gxm_XtAppCreateShell, 5, 1, 0, H_XtAppCreateShell);
  XM_DEFINE_PROCEDURE(XtVaAppCreateShell, gxm_XtVaAppCreateShell, 5, 0, 0, H_XtVaAppCreateShell);
  XM_DEFINE_PROCEDURE(XtToolkitInitialize, gxm_XtToolkitInitialize, 0, 0, 0, H_XtToolkitInitialize);
  XM_DEFINE_PROCEDURE(XtSetLanguageProc, gxm_XtSetLanguageProc, 3, 0, 0, H_XtSetLanguageProc);
  XM_DEFINE_PROCEDURE(XtDisplayInitialize, gxm_XtDisplayInitialize, 6, 0, 0, H_XtDisplayInitialize);
  XM_DEFINE_PROCEDURE(XtOpenApplication, gxm_XtOpenApplication, 5, 1, 0, H_XtOpenApplication);
  XM_DEFINE_PROCEDURE(XtVaOpenApplication, gxm_XtVaOpenApplication, 5, 1, 0, H_XtVaOpenApplication);
  XM_DEFINE_PROCEDURE(XtAppInitialize, gxm_XtAppInitialize, 4, 1, 0, H_XtAppInitialize);
  XM_DEFINE_PROCEDURE(XtVaAppInitialize, gxm_XtVaAppInitialize, 4, 1, 0, H_XtVaAppInitialize);
  XM_DEFINE_PROCEDURE(XtOpenDisplay, gxm_XtOpenDisplay, 6, 0, 0, H_XtOpenDisplay);
  XM_DEFINE_PROCEDURE(XtCreateApplicationContext, gxm_XtCreateApplicationContext, 0, 0, 0, H_XtCreateApplicationContext);
  XM_DEFINE_PROCEDURE(XtDestroyApplicationContext, gxm_XtDestroyApplicationContext, 1, 0, 0, H_XtDestroyApplicationContext);
  XM_DEFINE_PROCEDURE(XtInitializeWidgetClass, gxm_XtInitializeWidgetClass, 1, 0, 0, H_XtInitializeWidgetClass);
  XM_DEFINE_PROCEDURE(XtAppSetFallbackResources, gxm_XtAppSetFallbackResources, 2, 0, 0, H_XtAppSetFallbackResources);
  XM_DEFINE_PROCEDURE(XtWidgetToApplicationContext, gxm_XtWidgetToApplicationContext, 1, 0, 0, H_XtWidgetToApplicationContext);
  XM_DEFINE_PROCEDURE(XtDisplayToApplicationContext, gxm_XtDisplayToApplicationContext, 1, 0, 0, H_XtDisplayToApplicationContext);
  XM_DEFINE_PROCEDURE(XtCloseDisplay, gxm_XtCloseDisplay, 1, 0, 0, H_XtCloseDisplay);
  XM_DEFINE_PROCEDURE(XtSetValues, gxm_XtSetValues, 2, 1, 0, H_XtSetValues);
  XM_DEFINE_PROCEDURE(XtVaSetValues, gxm_XtVaSetValues, 2, 0, 0, H_XtVaSetValues);
  XM_DEFINE_PROCEDURE(XtGetValues, gxm_XtGetValues, 2, 1, 0, H_XtGetValues);
  XM_DEFINE_PROCEDURE(XtVaGetValues, gxm_XtVaGetValues, 2, 0, 0, H_XtVaGetValues);
  XM_DEFINE_PROCEDURE(XtAppSetErrorMsgHandler, gxm_XtAppSetErrorMsgHandler, 2, 0, 0, H_XtAppSetErrorMsgHandler);
  XM_DEFINE_PROCEDURE(XtAppSetWarningMsgHandler, gxm_XtAppSetWarningMsgHandler, 2, 0, 0, H_XtAppSetWarningMsgHandler);
  XM_DEFINE_PROCEDURE(XtAppErrorMsg, gxm_XtAppErrorMsg, 7, 0, 0, H_XtAppErrorMsg);
  XM_DEFINE_PROCEDURE(XtAppWarningMsg, gxm_XtAppWarningMsg, 7, 0, 0, H_XtAppWarningMsg);
  XM_DEFINE_PROCEDURE(XtAppSetErrorHandler, gxm_XtAppSetErrorHandler, 2, 0, 0, H_XtAppSetErrorHandler);
  XM_DEFINE_PROCEDURE(XtAppSetWarningHandler, gxm_XtAppSetWarningHandler, 2, 0, 0, H_XtAppSetWarningHandler);
  XM_DEFINE_PROCEDURE(XtAppError, gxm_XtAppError, 2, 0, 0, H_XtAppError);
  XM_DEFINE_PROCEDURE(XtMalloc, gxm_XtMalloc, 1, 0, 0, H_XtMalloc);
  XM_DEFINE_PROCEDURE(XtCalloc, gxm_XtCalloc, 2, 0, 0, H_XtCalloc);
  XM_DEFINE_PROCEDURE(XtRealloc, gxm_XtRealloc, 2, 0, 0, H_XtRealloc);
  XM_DEFINE_PROCEDURE(XtFree, gxm_XtFree, 1, 0, 0, H_XtFree);
  XM_DEFINE_PROCEDURE(XtAppAddWorkProc, gxm_XtAppAddWorkProc, 2, 1, 0, H_XtAppAddWorkProc);
  XM_DEFINE_PROCEDURE(XtRemoveWorkProc, gxm_XtRemoveWorkProc, 1, 0, 0, H_XtRemoveWorkProc);
  XM_DEFINE_PROCEDURE(XtGetGC, gxm_XtGetGC, 3, 0, 0, H_XtGetGC);
  XM_DEFINE_PROCEDURE(XtAllocateGC, gxm_XtAllocateGC, 6, 0, 0, H_XtAllocateGC);
  XM_DEFINE_PROCEDURE(XtDestroyGC, gxm_XtDestroyGC, 1, 0, 0, H_XtDestroyGC);
  XM_DEFINE_PROCEDURE(XtReleaseGC, gxm_XtReleaseGC, 2, 0, 0, H_XtReleaseGC);
  XM_DEFINE_PROCEDURE(XtFindFile, gxm_XtFindFile, 4, 0, 0, H_XtFindFile);
  XM_DEFINE_PROCEDURE(XtResolvePathname, gxm_XtResolvePathname, 8, 0, 0, H_XtResolvePathname);
  XM_DEFINE_PROCEDURE(XtDisownSelection, gxm_XtDisownSelection, 3, 0, 0, H_XtDisownSelection);
  XM_DEFINE_PROCEDURE(XtGetSelectionValue, gxm_XtGetSelectionValue, 6, 0, 0, H_XtGetSelectionValue);
  XM_DEFINE_PROCEDURE(XtGetSelectionValues, gxm_XtGetSelectionValues, 7, 0, 0, H_XtGetSelectionValues);
  XM_DEFINE_PROCEDURE(XtAppSetSelectionTimeout, gxm_XtAppSetSelectionTimeout, 2, 0, 0, H_XtAppSetSelectionTimeout);
  XM_DEFINE_PROCEDURE(XtAppGetSelectionTimeout, gxm_XtAppGetSelectionTimeout, 1, 0, 0, H_XtAppGetSelectionTimeout);
  XM_DEFINE_PROCEDURE(XtGetSelectionRequest, gxm_XtGetSelectionRequest, 3, 0, 0, H_XtGetSelectionRequest);
  XM_DEFINE_PROCEDURE(XtGetSelectionValueIncremental, gxm_XtGetSelectionValueIncremental, 6, 0, 0, H_XtGetSelectionValueIncremental);
  XM_DEFINE_PROCEDURE(XtGetSelectionValuesIncremental, gxm_XtGetSelectionValuesIncremental, 7, 0, 0, H_XtGetSelectionValuesIncremental);
  XM_DEFINE_PROCEDURE(XtCreateSelectionRequest, gxm_XtCreateSelectionRequest, 2, 0, 0, H_XtCreateSelectionRequest);
  XM_DEFINE_PROCEDURE(XtSendSelectionRequest, gxm_XtSendSelectionRequest, 3, 0, 0, H_XtSendSelectionRequest);
  XM_DEFINE_PROCEDURE(XtCancelSelectionRequest, gxm_XtCancelSelectionRequest, 2, 0, 0, H_XtCancelSelectionRequest);
  XM_DEFINE_PROCEDURE(XtGrabKey, gxm_XtGrabKey, 6, 0, 0, H_XtGrabKey);
  XM_DEFINE_PROCEDURE(XtUngrabKey, gxm_XtUngrabKey, 3, 0, 0, H_XtUngrabKey);
  XM_DEFINE_PROCEDURE(XtGrabKeyboard, gxm_XtGrabKeyboard, 5, 0, 0, H_XtGrabKeyboard);
  XM_DEFINE_PROCEDURE(XtUngrabKeyboard, gxm_XtUngrabKeyboard, 2, 0, 0, H_XtUngrabKeyboard);
  XM_DEFINE_PROCEDURE(XtGrabButton, gxm_XtGrabButton, 9, 0, 0, H_XtGrabButton);
  XM_DEFINE_PROCEDURE(XtUngrabButton, gxm_XtUngrabButton, 3, 0, 0, H_XtUngrabButton);
  XM_DEFINE_PROCEDURE(XtGrabPointer, gxm_XtGrabPointer, 8, 0, 0, H_XtGrabPointer);
  XM_DEFINE_PROCEDURE(XtUngrabPointer, gxm_XtUngrabPointer, 2, 0, 0, H_XtUngrabPointer);
  XM_DEFINE_PROCEDURE(XtGetApplicationNameAndClass, gxm_XtGetApplicationNameAndClass, 1, 0, 0, H_XtGetApplicationNameAndClass);
  XM_DEFINE_PROCEDURE(XtGetDisplays, gxm_XtGetDisplays, 1, 0, 0, H_XtGetDisplays);
  XM_DEFINE_PROCEDURE(XtToolkitThreadInitialize, gxm_XtToolkitThreadInitialize, 0, 0, 0, H_XtToolkitThreadInitialize);
  XM_DEFINE_PROCEDURE(XtAppLock, gxm_XtAppLock, 1, 0, 0, H_XtAppLock);
  XM_DEFINE_PROCEDURE(XtAppUnlock, gxm_XtAppUnlock, 1, 0, 0, H_XtAppUnlock);
  XM_DEFINE_PROCEDURE(XtIsRectObj, gxm_XtIsRectObj, 1, 0, 0, H_XtIsRectObj);
  XM_DEFINE_PROCEDURE(XtIsWidget, gxm_XtIsWidget, 1, 0, 0, H_XtIsWidget);
  XM_DEFINE_PROCEDURE(XtIsComposite, gxm_XtIsComposite, 1, 0, 0, H_XtIsComposite);
  XM_DEFINE_PROCEDURE(XtIsConstraint, gxm_XtIsConstraint, 1, 0, 0, H_XtIsConstraint);
  XM_DEFINE_PROCEDURE(XtIsShell, gxm_XtIsShell, 1, 0, 0, H_XtIsShell);
  XM_DEFINE_PROCEDURE(XtIsOverrideShell, gxm_XtIsOverrideShell, 1, 0, 0, H_XtIsOverrideShell);
  XM_DEFINE_PROCEDURE(XtIsWMShell, gxm_XtIsWMShell, 1, 0, 0, H_XtIsWMShell);
  XM_DEFINE_PROCEDURE(XtIsVendorShell, gxm_XtIsVendorShell, 1, 0, 0, H_XtIsVendorShell);
  XM_DEFINE_PROCEDURE(XtIsTransientShell, gxm_XtIsTransientShell, 1, 0, 0, H_XtIsTransientShell);
  XM_DEFINE_PROCEDURE(XtIsTopLevelShell, gxm_XtIsTopLevelShell, 1, 0, 0, H_XtIsTopLevelShell);
  XM_DEFINE_PROCEDURE(XtIsApplicationShell, gxm_XtIsApplicationShell, 1, 0, 0, H_XtIsApplicationShell);
  XM_DEFINE_PROCEDURE(XtIsSessionShell, gxm_XtIsSessionShell, 1, 0, 0, H_XtIsSessionShell);
  XM_DEFINE_PROCEDURE(XtMapWidget, gxm_XtMapWidget, 1, 0, 0, H_XtMapWidget);
  XM_DEFINE_PROCEDURE(XtUnmapWidget, gxm_XtUnmapWidget, 1, 0, 0, H_XtUnmapWidget);
#endif
  XM_DEFINE_PROCEDURE(XLoadQueryFont, gxm_XLoadQueryFont, 2, 0, 0, H_XLoadQueryFont);
  XM_DEFINE_PROCEDURE(XQueryFont, gxm_XQueryFont, 2, 0, 0, H_XQueryFont);
  XM_DEFINE_PROCEDURE(XGetMotionEvents, gxm_XGetMotionEvents, 4, 0, 0, H_XGetMotionEvents);
  XM_DEFINE_PROCEDURE(XDeleteModifiermapEntry, gxm_XDeleteModifiermapEntry, 3, 0, 0, H_XDeleteModifiermapEntry);
  XM_DEFINE_PROCEDURE(XGetModifierMapping, gxm_XGetModifierMapping, 1, 0, 0, H_XGetModifierMapping);
  XM_DEFINE_PROCEDURE(XInsertModifiermapEntry, gxm_XInsertModifiermapEntry, 3, 0, 0, H_XInsertModifiermapEntry);
  XM_DEFINE_PROCEDURE(XNewModifiermap, gxm_XNewModifiermap, 1, 0, 0, H_XNewModifiermap);
  XM_DEFINE_PROCEDURE(XCreateImage, gxm_XCreateImage, 0, 0, 1, H_XCreateImage);
  XM_DEFINE_PROCEDURE(XGetImage, gxm_XGetImage, 8, 0, 0, H_XGetImage);
  XM_DEFINE_PROCEDURE(XGetSubImage, gxm_XGetSubImage, 0, 0, 1, H_XGetSubImage);
  XM_DEFINE_PROCEDURE(XOpenDisplay, gxm_XOpenDisplay, 1, 0, 0, H_XOpenDisplay);
  XM_DEFINE_PROCEDURE(XFetchBytes, gxm_XFetchBytes, 1, 0, 0, H_XFetchBytes);
  XM_DEFINE_PROCEDURE(XFetchBuffer, gxm_XFetchBuffer, 2, 0, 0, H_XFetchBuffer);
  XM_DEFINE_PROCEDURE(XGetAtomName, gxm_XGetAtomName, 2, 0, 0, H_XGetAtomName);
  XM_DEFINE_PROCEDURE(XDisplayName, gxm_XDisplayName, 1, 0, 0, H_XDisplayName);
  XM_DEFINE_PROCEDURE(XKeysymToString, gxm_XKeysymToString, 1, 0, 0, H_XKeysymToString);
  XM_DEFINE_PROCEDURE(XSynchronize, gxm_XSynchronize, 2, 0, 0, H_XSynchronize);
  XM_DEFINE_PROCEDURE(XSetAfterFunction, gxm_XSetAfterFunction, 2, 0, 0, H_XSetAfterFunction);
  XM_DEFINE_PROCEDURE(XInternAtom, gxm_XInternAtom, 3, 0, 0, H_XInternAtom);
  XM_DEFINE_PROCEDURE(XCopyColormapAndFree, gxm_XCopyColormapAndFree, 2, 0, 0, H_XCopyColormapAndFree);
  XM_DEFINE_PROCEDURE(XCreateColormap, gxm_XCreateColormap, 4, 0, 0, H_XCreateColormap);
  XM_DEFINE_PROCEDURE(XCreatePixmapCursor, gxm_XCreatePixmapCursor, 7, 0, 0, H_XCreatePixmapCursor);
  XM_DEFINE_PROCEDURE(XCreateGlyphCursor, gxm_XCreateGlyphCursor, 7, 0, 0, H_XCreateGlyphCursor);
  XM_DEFINE_PROCEDURE(XCreateFontCursor, gxm_XCreateFontCursor, 2, 0, 0, H_XCreateFontCursor);
  XM_DEFINE_PROCEDURE(XLoadFont, gxm_XLoadFont, 2, 0, 0, H_XLoadFont);
  XM_DEFINE_PROCEDURE(XCreateGC, gxm_XCreateGC, 4, 0, 0, H_XCreateGC);
  XM_DEFINE_PROCEDURE(XFlushGC, gxm_XFlushGC, 2, 0, 0, H_XFlushGC);
  XM_DEFINE_PROCEDURE(XCreatePixmap, gxm_XCreatePixmap, 5, 0, 0, H_XCreatePixmap);
  XM_DEFINE_PROCEDURE(XCreateBitmapFromData, gxm_XCreateBitmapFromData, 5, 0, 0, H_XCreateBitmapFromData);
  XM_DEFINE_PROCEDURE(XCreatePixmapFromBitmapData, gxm_XCreatePixmapFromBitmapData, 8, 0, 0, H_XCreatePixmapFromBitmapData);
  XM_DEFINE_PROCEDURE(XCreateSimpleWindow, gxm_XCreateSimpleWindow, 9, 0, 0, H_XCreateSimpleWindow);
  XM_DEFINE_PROCEDURE(XGetSelectionOwner, gxm_XGetSelectionOwner, 2, 0, 0, H_XGetSelectionOwner);
  XM_DEFINE_PROCEDURE(XCreateWindow, gxm_XCreateWindow, 0, 0, 1, H_XCreateWindow);
  XM_DEFINE_PROCEDURE(XListInstalledColormaps, gxm_XListInstalledColormaps, 2, 0, 0, H_XListInstalledColormaps);
  XM_DEFINE_PROCEDURE(XListFonts, gxm_XListFonts, 3, 0, 0, H_XListFonts);
  XM_DEFINE_PROCEDURE(XListFontsWithInfo, gxm_XListFontsWithInfo, 3, 0, 0, H_XListFontsWithInfo);
  XM_DEFINE_PROCEDURE(XGetFontPath, gxm_XGetFontPath, 1, 0, 0, H_XGetFontPath);
  XM_DEFINE_PROCEDURE(XListExtensions, gxm_XListExtensions, 1, 0, 0, H_XListExtensions);
  XM_DEFINE_PROCEDURE(XListProperties, gxm_XListProperties, 2, 0, 0, H_XListProperties);
  XM_DEFINE_PROCEDURE(XKeycodeToKeysym, gxm_XKeycodeToKeysym, 3, 0, 0, H_XKeycodeToKeysym);
  XM_DEFINE_PROCEDURE(XLookupKeysym, gxm_XLookupKeysym, 2, 0, 0, H_XLookupKeysym);
  XM_DEFINE_PROCEDURE(XGetKeyboardMapping, gxm_XGetKeyboardMapping, 3, 0, 0, H_XGetKeyboardMapping);
  XM_DEFINE_PROCEDURE(XStringToKeysym, gxm_XStringToKeysym, 1, 0, 0, H_XStringToKeysym);
  XM_DEFINE_PROCEDURE(XMaxRequestSize, gxm_XMaxRequestSize, 1, 0, 0, H_XMaxRequestSize);
  XM_DEFINE_PROCEDURE(XExtendedMaxRequestSize, gxm_XExtendedMaxRequestSize, 1, 0, 0, H_XExtendedMaxRequestSize);
  XM_DEFINE_PROCEDURE(XDisplayMotionBufferSize, gxm_XDisplayMotionBufferSize, 1, 0, 0, H_XDisplayMotionBufferSize);
  XM_DEFINE_PROCEDURE(XVisualIDFromVisual, gxm_XVisualIDFromVisual, 1, 0, 0, H_XVisualIDFromVisual);
  XM_DEFINE_PROCEDURE(XRootWindow, gxm_XRootWindow, 2, 0, 0, H_RootWindow);
  XM_DEFINE_PROCEDURE(XDefaultRootWindow, gxm_XDefaultRootWindow, 1, 0, 0, H_DefaultRootWindow);
  XM_DEFINE_PROCEDURE(XRootWindowOfScreen, gxm_XRootWindowOfScreen, 1, 0, 0, H_RootWindowOfScreen);
  XM_DEFINE_PROCEDURE(XDefaultVisual, gxm_XDefaultVisual, 2, 0, 0, H_DefaultVisual);
  XM_DEFINE_PROCEDURE(XDefaultVisualOfScreen, gxm_XDefaultVisualOfScreen, 1, 0, 0, H_DefaultVisualOfScreen);
  XM_DEFINE_PROCEDURE(XDefaultGC, gxm_XDefaultGC, 2, 0, 0, H_DefaultGC);
  XM_DEFINE_PROCEDURE(XDefaultGCOfScreen, gxm_XDefaultGCOfScreen, 1, 0, 0, H_DefaultGCOfScreen);
  XM_DEFINE_PROCEDURE(XBlackPixel, gxm_XBlackPixel, 2, 0, 0, H_BlackPixel);
  XM_DEFINE_PROCEDURE(XWhitePixel, gxm_XWhitePixel, 2, 0, 0, H_WhitePixel);
  XM_DEFINE_PROCEDURE(XAllPlanes, gxm_XAllPlanes, 0, 0, 0, H_AllPlanes);
  XM_DEFINE_PROCEDURE(XBlackPixelOfScreen, gxm_XBlackPixelOfScreen, 1, 0, 0, H_BlackPixelOfScreen);
  XM_DEFINE_PROCEDURE(XWhitePixelOfScreen, gxm_XWhitePixelOfScreen, 1, 0, 0, H_WhitePixelOfScreen);
  XM_DEFINE_PROCEDURE(XNextRequest, gxm_XNextRequest, 1, 0, 0, H_NextRequest);
  XM_DEFINE_PROCEDURE(XLastKnownRequestProcessed, gxm_XLastKnownRequestProcessed, 1, 0, 0, H_LastKnownRequestProcessed);
  XM_DEFINE_PROCEDURE(XServerVendor, gxm_XServerVendor, 1, 0, 0, H_ServerVendor);
  XM_DEFINE_PROCEDURE(XDisplayString, gxm_XDisplayString, 1, 0, 0, H_DisplayString);
  XM_DEFINE_PROCEDURE(XDefaultColormap, gxm_XDefaultColormap, 2, 0, 0, H_DefaultColormap);
  XM_DEFINE_PROCEDURE(XDefaultColormapOfScreen, gxm_XDefaultColormapOfScreen, 1, 0, 0, H_DefaultColormapOfScreen);
  XM_DEFINE_PROCEDURE(XDisplayOfScreen, gxm_XDisplayOfScreen, 1, 0, 0, H_DisplayOfScreen);
  XM_DEFINE_PROCEDURE(XScreenOfDisplay, gxm_XScreenOfDisplay, 2, 0, 0, H_ScreenOfDisplay);
  XM_DEFINE_PROCEDURE(XDefaultScreenOfDisplay, gxm_XDefaultScreenOfDisplay, 1, 0, 0, H_DefaultScreenOfDisplay);
  XM_DEFINE_PROCEDURE(XEventMaskOfScreen, gxm_XEventMaskOfScreen, 1, 0, 0, H_EventMaskOfScreen);
  XM_DEFINE_PROCEDURE(XScreenNumberOfScreen, gxm_XScreenNumberOfScreen, 1, 0, 0, H_XScreenNumberOfScreen);
  XM_DEFINE_PROCEDURE(XSetErrorHandler, gxm_XSetErrorHandler, 1, 0, 0, H_XSetErrorHandler);
  XM_DEFINE_PROCEDURE(XSetIOErrorHandler, gxm_XSetIOErrorHandler, 1, 0, 0, H_XSetIOErrorHandler);
  XM_DEFINE_PROCEDURE(XListPixmapFormats, gxm_XListPixmapFormats, 1, 0, 0, H_XListPixmapFormats);
  XM_DEFINE_PROCEDURE(XListDepths, gxm_XListDepths, 2, 0, 0, H_XListDepths);
  XM_DEFINE_PROCEDURE(XReconfigureWMWindow, gxm_XReconfigureWMWindow, 5, 0, 0, H_XReconfigureWMWindow);
  XM_DEFINE_PROCEDURE(XGetWMProtocols, gxm_XGetWMProtocols, 2, 0, 0, H_XGetWMProtocols);
  XM_DEFINE_PROCEDURE(XSetWMProtocols, gxm_XSetWMProtocols, 4, 0, 0, H_XSetWMProtocols);
  XM_DEFINE_PROCEDURE(XIconifyWindow, gxm_XIconifyWindow, 3, 0, 0, H_XIconifyWindow);
  XM_DEFINE_PROCEDURE(XWithdrawWindow, gxm_XWithdrawWindow, 3, 0, 0, H_XWithdrawWindow);
  XM_DEFINE_PROCEDURE(XGetCommand, gxm_XGetCommand, 2, 0, 0, H_XGetCommand);
  XM_DEFINE_PROCEDURE(XGetWMColormapWindows, gxm_XGetWMColormapWindows, 2, 0, 0, H_XGetWMColormapWindows);
  XM_DEFINE_PROCEDURE(XSetWMColormapWindows, gxm_XSetWMColormapWindows, 4, 0, 0, H_XSetWMColormapWindows);
  XM_DEFINE_PROCEDURE(XSetTransientForHint, gxm_XSetTransientForHint, 3, 0, 0, H_XSetTransientForHint);
  XM_DEFINE_PROCEDURE(XActivateScreenSaver, gxm_XActivateScreenSaver, 1, 0, 0, H_XActivateScreenSaver);
  XM_DEFINE_PROCEDURE(XAllocColor, gxm_XAllocColor, 3, 0, 0, H_XAllocColor);
  XM_DEFINE_PROCEDURE(XAllocColorCells, gxm_XAllocColorCells, 5, 0, 0, H_XAllocColorCells);
  XM_DEFINE_PROCEDURE(XAllocColorPlanes, gxm_XAllocColorPlanes, 0, 0, 1, H_XAllocColorPlanes);
  XM_DEFINE_PROCEDURE(XAllocNamedColor, gxm_XAllocNamedColor, 5, 0, 0, H_XAllocNamedColor);
  XM_DEFINE_PROCEDURE(XAllowEvents, gxm_XAllowEvents, 3, 0, 0, H_XAllowEvents);
  XM_DEFINE_PROCEDURE(XAutoRepeatOff, gxm_XAutoRepeatOff, 1, 0, 0, H_XAutoRepeatOff);
  XM_DEFINE_PROCEDURE(XAutoRepeatOn, gxm_XAutoRepeatOn, 1, 0, 0, H_XAutoRepeatOn);
  XM_DEFINE_PROCEDURE(XBell, gxm_XBell, 2, 0, 0, H_XBell);
  XM_DEFINE_PROCEDURE(XBitmapBitOrder, gxm_XBitmapBitOrder, 1, 0, 0, H_BitmapBitOrder);
  XM_DEFINE_PROCEDURE(XBitmapPad, gxm_XBitmapPad, 1, 0, 0, H_BitmapPad);
  XM_DEFINE_PROCEDURE(XBitmapUnit, gxm_XBitmapUnit, 1, 0, 0, H_BitmapUnit);
  XM_DEFINE_PROCEDURE(XCellsOfScreen, gxm_XCellsOfScreen, 1, 0, 0, H_CellsOfScreen);
  XM_DEFINE_PROCEDURE(XChangeActivePointerGrab, gxm_XChangeActivePointerGrab, 4, 0, 0, H_XChangeActivePointerGrab);
  XM_DEFINE_PROCEDURE(XChangeGC, gxm_XChangeGC, 4, 0, 0, H_XChangeGC);
  XM_DEFINE_PROCEDURE(XChangeKeyboardControl, gxm_XChangeKeyboardControl, 3, 0, 0, H_XChangeKeyboardControl);
  XM_DEFINE_PROCEDURE(XChangeKeyboardMapping, gxm_XChangeKeyboardMapping, 5, 0, 0, H_XChangeKeyboardMapping);
  XM_DEFINE_PROCEDURE(XChangePointerControl, gxm_XChangePointerControl, 6, 0, 0, H_XChangePointerControl);
  XM_DEFINE_PROCEDURE(XChangeProperty, gxm_XChangeProperty, 7, 1, 0, H_XChangeProperty);
  XM_DEFINE_PROCEDURE(XChangeWindowAttributes, gxm_XChangeWindowAttributes, 4, 0, 0, H_XChangeWindowAttributes);
  XM_DEFINE_PROCEDURE(XCheckIfEvent, gxm_XCheckIfEvent, 3, 0, 0, H_XCheckIfEvent);
  XM_DEFINE_PROCEDURE(XCheckMaskEvent, gxm_XCheckMaskEvent, 2, 0, 0, H_XCheckMaskEvent);
  XM_DEFINE_PROCEDURE(XCheckTypedEvent, gxm_XCheckTypedEvent, 2, 0, 0, H_XCheckTypedEvent);
  XM_DEFINE_PROCEDURE(XCheckTypedWindowEvent, gxm_XCheckTypedWindowEvent, 3, 0, 0, H_XCheckTypedWindowEvent);
  XM_DEFINE_PROCEDURE(XCheckWindowEvent, gxm_XCheckWindowEvent, 3, 0, 0, H_XCheckWindowEvent);
  XM_DEFINE_PROCEDURE(XCirculateSubwindows, gxm_XCirculateSubwindows, 3, 0, 0, H_XCirculateSubwindows);
  XM_DEFINE_PROCEDURE(XCirculateSubwindowsDown, gxm_XCirculateSubwindowsDown, 2, 0, 0, H_XCirculateSubwindowsDown);
  XM_DEFINE_PROCEDURE(XCirculateSubwindowsUp, gxm_XCirculateSubwindowsUp, 2, 0, 0, H_XCirculateSubwindowsUp);
  XM_DEFINE_PROCEDURE(XClearArea, gxm_XClearArea, 7, 0, 0, H_XClearArea);
  XM_DEFINE_PROCEDURE(XClearWindow, gxm_XClearWindow, 2, 0, 0, H_XClearWindow);
  XM_DEFINE_PROCEDURE(XCloseDisplay, gxm_XCloseDisplay, 1, 0, 0, H_XCloseDisplay);
  XM_DEFINE_PROCEDURE(XConfigureWindow, gxm_XConfigureWindow, 4, 0, 0, H_XConfigureWindow);
  XM_DEFINE_PROCEDURE(XConnectionNumber, gxm_XConnectionNumber, 1, 0, 0, H_XConnectionNumber);
  XM_DEFINE_PROCEDURE(XConvertSelection, gxm_XConvertSelection, 6, 0, 0, H_XConvertSelection);
  XM_DEFINE_PROCEDURE(XCopyArea, gxm_XCopyArea, 0, 0, 1, H_XCopyArea);
  XM_DEFINE_PROCEDURE(XCopyGC, gxm_XCopyGC, 4, 0, 0, H_XCopyGC);
  XM_DEFINE_PROCEDURE(XCopyPlane, gxm_XCopyPlane, 0, 0, 1, H_XCopyPlane);
  XM_DEFINE_PROCEDURE(XDefaultDepth, gxm_XDefaultDepth, 2, 0, 0, H_DefaultDepth);
  XM_DEFINE_PROCEDURE(XDefaultDepthOfScreen, gxm_XDefaultDepthOfScreen, 1, 0, 0, H_DefaultDepthOfScreen);
  XM_DEFINE_PROCEDURE(XDefaultScreen, gxm_XDefaultScreen, 1, 0, 0, H_DefaultScreen);
  XM_DEFINE_PROCEDURE(XDefineCursor, gxm_XDefineCursor, 3, 0, 0, H_XDefineCursor);
  XM_DEFINE_PROCEDURE(XDeleteProperty, gxm_XDeleteProperty, 3, 0, 0, H_XDeleteProperty);
  XM_DEFINE_PROCEDURE(XDestroyWindow, gxm_XDestroyWindow, 2, 0, 0, H_XDestroyWindow);
  XM_DEFINE_PROCEDURE(XDestroySubwindows, gxm_XDestroySubwindows, 2, 0, 0, H_XDestroySubwindows);
  XM_DEFINE_PROCEDURE(XDoesBackingStore, gxm_XDoesBackingStore, 1, 0, 0, H_XDoesBackingStore);
  XM_DEFINE_PROCEDURE(XDoesSaveUnders, gxm_XDoesSaveUnders, 1, 0, 0, H_XDoesSaveUnders);
  XM_DEFINE_PROCEDURE(XDisableAccessControl, gxm_XDisableAccessControl, 1, 0, 0, H_XDisableAccessControl);
  XM_DEFINE_PROCEDURE(XDisplayCells, gxm_XDisplayCells, 2, 0, 0, H_XDisplayCells);
  XM_DEFINE_PROCEDURE(XDisplayHeight, gxm_XDisplayHeight, 2, 0, 0, H_XDisplayHeight);
  XM_DEFINE_PROCEDURE(XDisplayHeightMM, gxm_XDisplayHeightMM, 2, 0, 0, H_XDisplayHeightMM);
  XM_DEFINE_PROCEDURE(XDisplayKeycodes, gxm_XDisplayKeycodes, 1, 0, 0, H_XDisplayKeycodes);
  XM_DEFINE_PROCEDURE(XDisplayPlanes, gxm_XDisplayPlanes, 2, 0, 0, H_XDisplayPlanes);
  XM_DEFINE_PROCEDURE(XDisplayWidth, gxm_XDisplayWidth, 2, 0, 0, H_XDisplayWidth);
  XM_DEFINE_PROCEDURE(XDisplayWidthMM, gxm_XDisplayWidthMM, 2, 0, 0, H_XDisplayWidthMM);
  XM_DEFINE_PROCEDURE(XDrawArc, gxm_XDrawArc, 9, 0, 0, H_XDrawArc);
  XM_DEFINE_PROCEDURE(XDrawArcs, gxm_XDrawArcs, 5, 0, 0, H_XDrawArcs);
  XM_DEFINE_PROCEDURE(XDrawImageString, gxm_XDrawImageString, 7, 0, 0, H_XDrawImageString);
  XM_DEFINE_PROCEDURE(XDrawLine, gxm_XDrawLine, 7, 0, 0, H_XDrawLine);
  XM_DEFINE_PROCEDURE(XDrawLines, gxm_XDrawLines, 6, 0, 0, H_XDrawLines);
  XM_DEFINE_PROCEDURE(XDrawLinesDirect, gxm_XDrawLinesDirect, 6, 0, 0, H_XDrawLinesDirect);
  XM_DEFINE_PROCEDURE(freeXPoints, gxm_FreeXPoints, 1, 0, 0, H_freeXPoints);
  XM_DEFINE_PROCEDURE(vector->XPoints, gxm_Vector2XPoints, 1, 0, 0, H_vector2XPoints);
  XM_DEFINE_PROCEDURE(XDrawPoint, gxm_XDrawPoint, 5, 0, 0, H_XDrawPoint);
  XM_DEFINE_PROCEDURE(XDrawPoints, gxm_XDrawPoints, 6, 0, 0, H_XDrawPoints);
  XM_DEFINE_PROCEDURE(XDrawRectangle, gxm_XDrawRectangle, 7, 0, 0, H_XDrawRectangle);
  XM_DEFINE_PROCEDURE(XDrawRectangles, gxm_XDrawRectangles, 5, 0, 0, H_XDrawRectangles);
  XM_DEFINE_PROCEDURE(XDrawSegments, gxm_XDrawSegments, 5, 0, 0, H_XDrawSegments);
  XM_DEFINE_PROCEDURE(XDrawString, gxm_XDrawString, 7, 0, 0, H_XDrawString);
  XM_DEFINE_PROCEDURE(XDrawText, gxm_XDrawText, 6, 1, 0, H_XDrawText);
  XM_DEFINE_PROCEDURE(XEnableAccessControl, gxm_XEnableAccessControl, 1, 0, 0, H_XEnableAccessControl);
  XM_DEFINE_PROCEDURE(XEventsQueued, gxm_XEventsQueued, 2, 0, 0, H_XEventsQueued);
  XM_DEFINE_PROCEDURE(XFetchName, gxm_XFetchName, 2, 0, 0, H_XFetchName);
  XM_DEFINE_PROCEDURE(XFillArc, gxm_XFillArc, 9, 0, 0, H_XFillArc);
  XM_DEFINE_PROCEDURE(XFillArcs, gxm_XFillArcs, 5, 0, 0, H_XFillArcs);
  XM_DEFINE_PROCEDURE(XFillPolygon, gxm_XFillPolygon, 7, 0, 0, H_XFillPolygon);
  XM_DEFINE_PROCEDURE(XFillRectangle, gxm_XFillRectangle, 7, 0, 0, H_XFillRectangle);
  XM_DEFINE_PROCEDURE(XFillRectangles, gxm_XFillRectangles, 5, 0, 0, H_XFillRectangles);
  XM_DEFINE_PROCEDURE(XFlush, gxm_XFlush, 1, 0, 0, H_XFlush);
  XM_DEFINE_PROCEDURE(XForceScreenSaver, gxm_XForceScreenSaver, 2, 0, 0, H_XForceScreenSaver);
  XM_DEFINE_PROCEDURE(XFree, gxm_XFree, 1, 0, 0, H_XFree);
  XM_DEFINE_PROCEDURE(XFreeColormap, gxm_XFreeColormap, 2, 0, 0, H_XFreeColormap);
  XM_DEFINE_PROCEDURE(XFreeColors, gxm_XFreeColors, 5, 0, 0, H_XFreeColors);
  XM_DEFINE_PROCEDURE(XFreeCursor, gxm_XFreeCursor, 2, 0, 0, H_XFreeCursor);
  XM_DEFINE_PROCEDURE(XFreeExtensionList, gxm_XFreeExtensionList, 1, 0, 0, "a no-op in xm");
  XM_DEFINE_PROCEDURE(XFreeFont, gxm_XFreeFont, 2, 0, 0, H_XFreeFont);
  XM_DEFINE_PROCEDURE(XFreeFontInfo, gxm_XFreeFontInfo, 3, 0, 0, H_XFreeFontInfo);
  XM_DEFINE_PROCEDURE(XFreeFontNames, gxm_XFreeFontNames, 1, 0, 0, H_XFreeFontNames);
  XM_DEFINE_PROCEDURE(XFreeFontPath, gxm_XFreeFontPath, 1, 0, 0, H_XFreeFontPath);
  XM_DEFINE_PROCEDURE(XFreeGC, gxm_XFreeGC, 2, 0, 0, H_XFreeGC);
  XM_DEFINE_PROCEDURE(XFreeModifiermap, gxm_XFreeModifiermap, 1, 0, 0, H_XFreeModifiermap);
  XM_DEFINE_PROCEDURE(XFreePixmap, gxm_XFreePixmap, 2, 0, 0, H_XFreePixmap);
  XM_DEFINE_PROCEDURE(XGeometry, gxm_XGeometry, 0, 0, 1, H_XGeometry);
  XM_DEFINE_PROCEDURE(XGetErrorText, gxm_XGetErrorText, 4, 0, 0, H_XGetErrorText);
  XM_DEFINE_PROCEDURE(XGetFontProperty, gxm_XGetFontProperty, 2, 0, 0, H_XGetFontProperty);
  XM_DEFINE_PROCEDURE(XGetGCValues, gxm_XGetGCValues, 3, 0, 0, H_XGetGCValues);
  XM_DEFINE_PROCEDURE(XGCValues, gxm_XGCValues, 0, 0, 0, "creates a new XGCValues object for xm");
  XM_DEFINE_PROCEDURE(XEvent, gxm_XEvent, 0, 1, 0, "creates a new XEvent object for xm");
  XM_DEFINE_PROCEDURE(XGetGeometry, gxm_XGetGeometry, 2, 0, 0, H_XGetGeometry);
  XM_DEFINE_PROCEDURE(XGetIconName, gxm_XGetIconName, 2, 0, 0, H_XGetIconName);
  XM_DEFINE_PROCEDURE(XGetInputFocus, gxm_XGetInputFocus, 1, 0, 0, H_XGetInputFocus);
  XM_DEFINE_PROCEDURE(XGetKeyboardControl, gxm_XGetKeyboardControl, 1, 0, 0, H_XGetKeyboardControl);
  XM_DEFINE_PROCEDURE(XGetPointerControl, gxm_XGetPointerControl, 1, 0, 0, H_XGetPointerControl);
  XM_DEFINE_PROCEDURE(XGetPointerMapping, gxm_XGetPointerMapping, 3, 0, 0, H_XGetPointerMapping);
  XM_DEFINE_PROCEDURE(XGetScreenSaver, gxm_XGetScreenSaver, 1, 0, 0, H_XGetScreenSaver);
  XM_DEFINE_PROCEDURE(XGetTransientForHint, gxm_XGetTransientForHint, 2, 0, 0, H_XGetTransientForHint);
  XM_DEFINE_PROCEDURE(XGetWindowProperty, gxm_XGetWindowProperty, 0, 0, 1, H_XGetWindowProperty);
  XM_DEFINE_PROCEDURE(XGetWindowAttributes, gxm_XGetWindowAttributes, 2, 0, 0, H_XGetWindowAttributes);
  XM_DEFINE_PROCEDURE(XGrabButton, gxm_XGrabButton, 0, 0, 1, H_XGrabButton);
  XM_DEFINE_PROCEDURE(XGrabKey, gxm_XGrabKey, 7, 0, 0, H_XGrabKey);
  XM_DEFINE_PROCEDURE(XGrabKeyboard, gxm_XGrabKeyboard, 6, 0, 0, H_XGrabKeyboard);
  XM_DEFINE_PROCEDURE(XGrabPointer, gxm_XGrabPointer, 9, 0, 0, H_XGrabPointer);
  XM_DEFINE_PROCEDURE(XGrabServer, gxm_XGrabServer, 1, 0, 0, H_XGrabServer);
  XM_DEFINE_PROCEDURE(XHeightMMOfScreen, gxm_XHeightMMOfScreen, 1, 0, 0, H_HeightMMOfScreen);
  XM_DEFINE_PROCEDURE(XHeightOfScreen, gxm_XHeightOfScreen, 1, 0, 0, H_HeightOfScreen);
  XM_DEFINE_PROCEDURE(XIfEvent, gxm_XIfEvent, 3, 0, 0, H_XIfEvent);
  XM_DEFINE_PROCEDURE(XImageByteOrder, gxm_XImageByteOrder, 1, 0, 0, H_ImageByteOrder);
  XM_DEFINE_PROCEDURE(XInstallColormap, gxm_XInstallColormap, 2, 0, 0, H_XInstallColormap);
  XM_DEFINE_PROCEDURE(XKeysymToKeycode, gxm_XKeysymToKeycode, 2, 0, 0, H_XKeysymToKeycode);
  XM_DEFINE_PROCEDURE(XKillClient, gxm_XKillClient, 2, 0, 0, H_XKillClient);
  XM_DEFINE_PROCEDURE(XLookupColor, gxm_XLookupColor, 5, 0, 0, H_XLookupColor);
  XM_DEFINE_PROCEDURE(XLowerWindow, gxm_XLowerWindow, 2, 0, 0, H_XLowerWindow);
  XM_DEFINE_PROCEDURE(XMapRaised, gxm_XMapRaised, 2, 0, 0, H_XMapRaised);
  XM_DEFINE_PROCEDURE(XMapSubwindows, gxm_XMapSubwindows, 2, 0, 0, H_XMapSubwindows);
  XM_DEFINE_PROCEDURE(XMapWindow, gxm_XMapWindow, 2, 0, 0, H_XMapWindow);
  XM_DEFINE_PROCEDURE(XMaskEvent, gxm_XMaskEvent, 2, 0, 0, H_XMaskEvent);
  XM_DEFINE_PROCEDURE(XMaxCmapsOfScreen, gxm_XMaxCmapsOfScreen, 1, 0, 0, H_MaxCmapsOfScreen);
  XM_DEFINE_PROCEDURE(XMinCmapsOfScreen, gxm_XMinCmapsOfScreen, 1, 0, 0, H_MinCmapsOfScreen);
  XM_DEFINE_PROCEDURE(XMoveResizeWindow, gxm_XMoveResizeWindow, 6, 0, 0, H_XMoveResizeWindow);
  XM_DEFINE_PROCEDURE(XMoveWindow, gxm_XMoveWindow, 4, 0, 0, H_XMoveWindow);
  XM_DEFINE_PROCEDURE(XNextEvent, gxm_XNextEvent, 1, 0, 0, H_XNextEvent);
  XM_DEFINE_PROCEDURE(XNoOp, gxm_XNoOp, 1, 0, 0, H_XNoOp);
  XM_DEFINE_PROCEDURE(XParseColor, gxm_XParseColor, 4, 0, 0, H_XParseColor);
  XM_DEFINE_PROCEDURE(XParseGeometry, gxm_XParseGeometry, 1, 0, 0, H_XParseGeometry);
  XM_DEFINE_PROCEDURE(XPeekEvent, gxm_XPeekEvent, 1, 0, 0, H_XPeekEvent);
  XM_DEFINE_PROCEDURE(XPeekIfEvent, gxm_XPeekIfEvent, 3, 0, 0, H_XPeekIfEvent);
  XM_DEFINE_PROCEDURE(XPending, gxm_XPending, 1, 0, 0, H_XPending);
  XM_DEFINE_PROCEDURE(XPlanesOfScreen, gxm_XPlanesOfScreen, 1, 0, 0, H_PlanesOfScreen);
  XM_DEFINE_PROCEDURE(XProtocolRevision, gxm_XProtocolRevision, 1, 0, 0, H_ProtocolRevision);
  XM_DEFINE_PROCEDURE(XProtocolVersion, gxm_XProtocolVersion, 1, 0, 0, H_ProtocolVersion);
  XM_DEFINE_PROCEDURE(XPutBackEvent, gxm_XPutBackEvent, 2, 0, 0, H_XPutBackEvent);
  XM_DEFINE_PROCEDURE(XPutImage, gxm_XPutImage, 0, 0, 1, H_XPutImage);
  XM_DEFINE_PROCEDURE(XQLength, gxm_XQLength, 1, 0, 0, H_QLength);
  XM_DEFINE_PROCEDURE(XQueryBestCursor, gxm_XQueryBestCursor, 4, 0, 0, H_XQueryBestCursor);
  XM_DEFINE_PROCEDURE(XQueryBestSize, gxm_XQueryBestSize, 5, 0, 0, H_XQueryBestSize);
  XM_DEFINE_PROCEDURE(XQueryBestStipple, gxm_XQueryBestStipple, 4, 0, 0, H_XQueryBestStipple);
  XM_DEFINE_PROCEDURE(XQueryBestTile, gxm_XQueryBestTile, 4, 0, 0, H_XQueryBestTile);
  XM_DEFINE_PROCEDURE(XQueryColor, gxm_XQueryColor, 3, 0, 0, H_XQueryColor);
  XM_DEFINE_PROCEDURE(XQueryColors, gxm_XQueryColors, 3, 1, 0, H_XQueryColors);
  XM_DEFINE_PROCEDURE(XQueryExtension, gxm_XQueryExtension, 2, 0, 0, H_XQueryExtension);
  XM_DEFINE_PROCEDURE(XQueryKeymap, gxm_XQueryKeymap, 1, 0, 0, H_XQueryKeymap);
  XM_DEFINE_PROCEDURE(XQueryPointer, gxm_XQueryPointer, 2, 0, 0, H_XQueryPointer);
  XM_DEFINE_PROCEDURE(XQueryTextExtents, gxm_XQueryTextExtents, 3, 0, 0, H_XQueryTextExtents);
  XM_DEFINE_PROCEDURE(XQueryTree, gxm_XQueryTree, 2, 0, 0, H_XQueryTree);
  XM_DEFINE_PROCEDURE(XRaiseWindow, gxm_XRaiseWindow, 2, 0, 0, H_XRaiseWindow);
  XM_DEFINE_PROCEDURE(XReadBitmapFile, gxm_XReadBitmapFile, 3, 0, 0, H_XReadBitmapFile);
  XM_DEFINE_PROCEDURE(XReadBitmapFileData, gxm_XReadBitmapFileData, 1, 0, 0, H_XReadBitmapFileData);
  XM_DEFINE_PROCEDURE(XRebindKeysym, gxm_XRebindKeysym, 6, 0, 0, H_XRebindKeysym);
  XM_DEFINE_PROCEDURE(XRecolorCursor, gxm_XRecolorCursor, 4, 0, 0, H_XRecolorCursor);
  XM_DEFINE_PROCEDURE(XRefreshKeyboardMapping, gxm_XRefreshKeyboardMapping, 1, 0, 0, H_XRefreshKeyboardMapping);
  XM_DEFINE_PROCEDURE(XReparentWindow, gxm_XReparentWindow, 5, 0, 0, H_XReparentWindow);
  XM_DEFINE_PROCEDURE(XResetScreenSaver, gxm_XResetScreenSaver, 1, 0, 0, H_XResetScreenSaver);
  XM_DEFINE_PROCEDURE(XResizeWindow, gxm_XResizeWindow, 4, 0, 0, H_XResizeWindow);
  XM_DEFINE_PROCEDURE(XRestackWindows, gxm_XRestackWindows, 3, 0, 0, H_XRestackWindows);
  XM_DEFINE_PROCEDURE(XRotateBuffers, gxm_XRotateBuffers, 2, 0, 0, H_XRotateBuffers);
  XM_DEFINE_PROCEDURE(XRotateWindowProperties, gxm_XRotateWindowProperties, 5, 0, 0, H_XRotateWindowProperties);
  XM_DEFINE_PROCEDURE(XScreenCount, gxm_XScreenCount, 1, 0, 0, H_ScreenCount);
  XM_DEFINE_PROCEDURE(XSelectInput, gxm_XSelectInput, 3, 0, 0, H_XSelectInput);
  XM_DEFINE_PROCEDURE(XSendEvent, gxm_XSendEvent, 5, 0, 0, H_XSendEvent);
  XM_DEFINE_PROCEDURE(XSetAccessControl, gxm_XSetAccessControl, 2, 0, 0, H_XSetAccessControl);
  XM_DEFINE_PROCEDURE(XSetArcMode, gxm_XSetArcMode, 3, 0, 0, H_XSetArcMode);
  XM_DEFINE_PROCEDURE(XSetBackground, gxm_XSetBackground, 3, 0, 0, H_XSetBackground);
  XM_DEFINE_PROCEDURE(XSetClipMask, gxm_XSetClipMask, 3, 0, 0, H_XSetClipMask);
  XM_DEFINE_PROCEDURE(XSetClipOrigin, gxm_XSetClipOrigin, 4, 0, 0, H_XSetClipOrigin);
  XM_DEFINE_PROCEDURE(XSetClipRectangles, gxm_XSetClipRectangles, 7, 0, 0, H_XSetClipRectangles);
  XM_DEFINE_PROCEDURE(XSetCloseDownMode, gxm_XSetCloseDownMode, 2, 0, 0, H_XSetCloseDownMode);
  XM_DEFINE_PROCEDURE(XSetCommand, gxm_XSetCommand, 4, 0, 0, H_XSetCommand);
  XM_DEFINE_PROCEDURE(XSetDashes, gxm_XSetDashes, 4, 1, 0, H_XSetDashes);
  XM_DEFINE_PROCEDURE(XSetFillRule, gxm_XSetFillRule, 3, 0, 0, H_XSetFillRule);
  XM_DEFINE_PROCEDURE(XSetFillStyle, gxm_XSetFillStyle, 3, 0, 0, H_XSetFillStyle);
  XM_DEFINE_PROCEDURE(XSetFont, gxm_XSetFont, 3, 0, 0, H_XSetFont);
  XM_DEFINE_PROCEDURE(XSetFontPath, gxm_XSetFontPath, 3, 0, 0, H_XSetFontPath);
  XM_DEFINE_PROCEDURE(XSetForeground, gxm_XSetForeground, 3, 0, 0, H_XSetForeground);
  XM_DEFINE_PROCEDURE(XSetFunction, gxm_XSetFunction, 3, 0, 0, H_XSetFunction);
  XM_DEFINE_PROCEDURE(XSetGraphicsExposures, gxm_XSetGraphicsExposures, 3, 0, 0, H_XSetGraphicsExposures);
  XM_DEFINE_PROCEDURE(XSetIconName, gxm_XSetIconName, 3, 0, 0, H_XSetIconName);
  XM_DEFINE_PROCEDURE(XSetInputFocus, gxm_XSetInputFocus, 4, 0, 0, H_XSetInputFocus);
  XM_DEFINE_PROCEDURE(XSetLineAttributes, gxm_XSetLineAttributes, 6, 0, 0, H_XSetLineAttributes);
  XM_DEFINE_PROCEDURE(XSetModifierMapping, gxm_XSetModifierMapping, 2, 0, 0, H_XSetModifierMapping);
  XM_DEFINE_PROCEDURE(XSetPlaneMask, gxm_XSetPlaneMask, 3, 0, 0, H_XSetPlaneMask);
  XM_DEFINE_PROCEDURE(XSetPointerMapping, gxm_XSetPointerMapping, 2, 1, 0, H_XSetPointerMapping);
  XM_DEFINE_PROCEDURE(XSetScreenSaver, gxm_XSetScreenSaver, 5, 0, 0, H_XSetScreenSaver);
  XM_DEFINE_PROCEDURE(XSetSelectionOwner, gxm_XSetSelectionOwner, 4, 0, 0, H_XSetSelectionOwner);
  XM_DEFINE_PROCEDURE(XSetState, gxm_XSetState, 6, 0, 0, H_XSetState);
  XM_DEFINE_PROCEDURE(XSetStipple, gxm_XSetStipple, 3, 0, 0, H_XSetStipple);
  XM_DEFINE_PROCEDURE(XSetSubwindowMode, gxm_XSetSubwindowMode, 3, 0, 0, H_XSetSubwindowMode);
  XM_DEFINE_PROCEDURE(XSetTSOrigin, gxm_XSetTSOrigin, 4, 0, 0, H_XSetTSOrigin);
  XM_DEFINE_PROCEDURE(XSetTile, gxm_XSetTile, 3, 0, 0, H_XSetTile);
  XM_DEFINE_PROCEDURE(XSetWindowBackground, gxm_XSetWindowBackground, 3, 0, 0, H_XSetWindowBackground);
  XM_DEFINE_PROCEDURE(XSetWindowBackgroundPixmap, gxm_XSetWindowBackgroundPixmap, 3, 0, 0, H_XSetWindowBackgroundPixmap);
  XM_DEFINE_PROCEDURE(XSetWindowBorder, gxm_XSetWindowBorder, 3, 0, 0, H_XSetWindowBorder);
  XM_DEFINE_PROCEDURE(XSetWindowBorderPixmap, gxm_XSetWindowBorderPixmap, 3, 0, 0, H_XSetWindowBorderPixmap);
  XM_DEFINE_PROCEDURE(XSetWindowBorderWidth, gxm_XSetWindowBorderWidth, 3, 0, 0, H_XSetWindowBorderWidth);
  XM_DEFINE_PROCEDURE(XSetWindowColormap, gxm_XSetWindowColormap, 3, 0, 0, H_XSetWindowColormap);
  XM_DEFINE_PROCEDURE(XStoreBuffer, gxm_XStoreBuffer, 4, 0, 0, H_XStoreBuffer);
  XM_DEFINE_PROCEDURE(XStoreBytes, gxm_XStoreBytes, 3, 0, 0, H_XStoreBytes);
  XM_DEFINE_PROCEDURE(XStoreColor, gxm_XStoreColor, 3, 0, 0, H_XStoreColor);
  XM_DEFINE_PROCEDURE(XStoreColors, gxm_XStoreColors, 4, 0, 0, H_XStoreColors);
  XM_DEFINE_PROCEDURE(XStoreName, gxm_XStoreName, 3, 0, 0, H_XStoreName);
  XM_DEFINE_PROCEDURE(XStoreNamedColor, gxm_XStoreNamedColor, 5, 0, 0, H_XStoreNamedColor);
  XM_DEFINE_PROCEDURE(XSync, gxm_XSync, 2, 0, 0, H_XSync);
  XM_DEFINE_PROCEDURE(XTextExtents, gxm_XTextExtents, 3, 0, 0, H_XTextExtents);
  XM_DEFINE_PROCEDURE(XTextWidth, gxm_XTextWidth, 3, 0, 0, H_XTextWidth);
  XM_DEFINE_PROCEDURE(XTranslateCoordinates, gxm_XTranslateCoordinates, 5, 0, 0, H_XTranslateCoordinates);
  XM_DEFINE_PROCEDURE(XUndefineCursor, gxm_XUndefineCursor, 2, 0, 0, H_XUndefineCursor);
  XM_DEFINE_PROCEDURE(XUngrabButton, gxm_XUngrabButton, 4, 0, 0, H_XUngrabButton);
  XM_DEFINE_PROCEDURE(XUngrabKey, gxm_XUngrabKey, 4, 0, 0, H_XUngrabKey);
  XM_DEFINE_PROCEDURE(XUngrabKeyboard, gxm_XUngrabKeyboard, 2, 0, 0, H_XUngrabKeyboard);
  XM_DEFINE_PROCEDURE(XUngrabPointer, gxm_XUngrabPointer, 2, 0, 0, H_XUngrabPointer);
  XM_DEFINE_PROCEDURE(XUngrabServer, gxm_XUngrabServer, 1, 0, 0, H_XUngrabServer);
  XM_DEFINE_PROCEDURE(XUninstallColormap, gxm_XUninstallColormap, 2, 0, 0, H_XUninstallColormap);
  XM_DEFINE_PROCEDURE(XUnloadFont, gxm_XUnloadFont, 2, 0, 0, H_XUnloadFont);
  XM_DEFINE_PROCEDURE(XUnmapSubwindows, gxm_XUnmapSubwindows, 2, 0, 0, H_XUnmapSubwindows);
  XM_DEFINE_PROCEDURE(XUnmapWindow, gxm_XUnmapWindow, 2, 0, 0, H_XUnmapWindow);
  XM_DEFINE_PROCEDURE(XVendorRelease, gxm_XVendorRelease, 1, 0, 0, H_VendorRelease);
  XM_DEFINE_PROCEDURE(XWarpPointer, gxm_XWarpPointer, 9, 0, 0, H_XWarpPointer);
  XM_DEFINE_PROCEDURE(XWidthMMOfScreen, gxm_XWidthMMOfScreen, 1, 0, 0, H_WidthMMOfScreen);
  XM_DEFINE_PROCEDURE(XWidthOfScreen, gxm_XWidthOfScreen, 1, 0, 0, H_WidthOfScreen);
  XM_DEFINE_PROCEDURE(XWindowEvent, gxm_XWindowEvent, 3, 0, 0, H_XWindowEvent);
  XM_DEFINE_PROCEDURE(XWriteBitmapFile, gxm_XWriteBitmapFile, 7, 0, 0, H_XWriteBitmapFile);
  XM_DEFINE_PROCEDURE(XSupportsLocale, gxm_XSupportsLocale, 0, 0, 0, H_XSupportsLocale);
  XM_DEFINE_PROCEDURE(XSetLocaleModifiers, gxm_XSetLocaleModifiers, 1, 0, 0, H_XSetLocaleModifiers);
  XM_DEFINE_PROCEDURE(XCreateFontSet, gxm_XCreateFontSet, 2, 0, 0, H_XCreateFontSet);
  XM_DEFINE_PROCEDURE(XFreeFontSet, gxm_XFreeFontSet, 2, 0, 0, H_XFreeFontSet);
  XM_DEFINE_PROCEDURE(XFontsOfFontSet, gxm_XFontsOfFontSet, 1, 0, 0, H_XFontsOfFontSet);
  XM_DEFINE_PROCEDURE(XBaseFontNameListOfFontSet, gxm_XBaseFontNameListOfFontSet, 1, 0, 0, H_XBaseFontNameListOfFontSet);
  XM_DEFINE_PROCEDURE(XLocaleOfFontSet, gxm_XLocaleOfFontSet, 1, 0, 0, H_XLocaleOfFontSet);
  XM_DEFINE_PROCEDURE(XContextDependentDrawing, gxm_XContextDependentDrawing, 1, 0, 0, H_XContextDependentDrawing);
  XM_DEFINE_PROCEDURE(XDirectionalDependentDrawing, gxm_XDirectionalDependentDrawing, 1, 0, 0, H_XDirectionalDependentDrawing);
  XM_DEFINE_PROCEDURE(XContextualDrawing, gxm_XContextualDrawing, 1, 0, 0, H_XContextualDrawing);
  XM_DEFINE_PROCEDURE(XFilterEvent, gxm_XFilterEvent, 2, 0, 0, H_XFilterEvent);
  XM_DEFINE_PROCEDURE(XAllocIconSize, gxm_XAllocIconSize, 0, 0, 0, H_XAllocIconSize);
  XM_DEFINE_PROCEDURE(XAllocStandardColormap, gxm_XAllocStandardColormap, 0, 0, 0, H_XAllocStandardColormap);
  XM_DEFINE_PROCEDURE(XAllocWMHints, gxm_XAllocWMHints, 0, 0, 0, H_XAllocWMHints);
  XM_DEFINE_PROCEDURE(XClipBox, gxm_XClipBox, 1, 0, 0, H_XClipBox);
  XM_DEFINE_PROCEDURE(XCreateRegion, gxm_XCreateRegion, 0, 0, 0, H_XCreateRegion);
  XM_DEFINE_PROCEDURE(XDefaultString, gxm_XDefaultString, 0, 0, 0, H_XDefaultString);
  XM_DEFINE_PROCEDURE(XDeleteContext, gxm_XDeleteContext, 3, 0, 0, H_XDeleteContext);
  XM_DEFINE_PROCEDURE(XDestroyRegion, gxm_XDestroyRegion, 1, 0, 0, H_XDestroyRegion);
  XM_DEFINE_PROCEDURE(XEmptyRegion, gxm_XEmptyRegion, 1, 0, 0, H_XEmptyRegion);
  XM_DEFINE_PROCEDURE(XEqualRegion, gxm_XEqualRegion, 2, 0, 0, H_XEqualRegion);
  XM_DEFINE_PROCEDURE(XFindContext, gxm_XFindContext, 3, 0, 0, H_XFindContext);
  XM_DEFINE_PROCEDURE(XGetIconSizes, gxm_XGetIconSizes, 2, 0, 0, H_XGetIconSizes);
  XM_DEFINE_PROCEDURE(XGetRGBColormaps, gxm_XGetRGBColormaps, 3, 0, 0, H_XGetRGBColormaps);
#if (!XM_DISABLE_DEPRECATED)
  XM_DEFINE_PROCEDURE(XGetStandardColormap, gxm_XGetStandardColormap, 3, 0, 0, H_XGetStandardColormap);
#endif
  XM_DEFINE_PROCEDURE(XGetVisualInfo, gxm_XGetVisualInfo, 3, 0, 0, H_XGetVisualInfo);
  XM_DEFINE_PROCEDURE(XGetWMHints, gxm_XGetWMHints, 2, 0, 0, H_XGetWMHints);
  XM_DEFINE_PROCEDURE(XIntersectRegion, gxm_XIntersectRegion, 3, 0, 0, H_XIntersectRegion);
  XM_DEFINE_PROCEDURE(XConvertCase, gxm_XConvertCase, 1, 0, 0, H_XConvertCase);
  XM_DEFINE_PROCEDURE(XLookupString, gxm_XLookupString, 1, 0, 0, H_XLookupString);
  XM_DEFINE_PROCEDURE(XMatchVisualInfo, gxm_XMatchVisualInfo, 4, 0, 0, H_XMatchVisualInfo);
  XM_DEFINE_PROCEDURE(XOffsetRegion, gxm_XOffsetRegion, 3, 0, 0, H_XOffsetRegion);
  XM_DEFINE_PROCEDURE(XPointInRegion, gxm_XPointInRegion, 3, 0, 0, H_XPointInRegion);
  XM_DEFINE_PROCEDURE(XPolygonRegion, gxm_XPolygonRegion, 3, 0, 0, H_XPolygonRegion);
  XM_DEFINE_PROCEDURE(XRectInRegion, gxm_XRectInRegion, 5, 0, 0, H_XRectInRegion);
  XM_DEFINE_PROCEDURE(XSaveContext, gxm_XSaveContext, 4, 0, 0, H_XSaveContext);
  XM_DEFINE_PROCEDURE(XUniqueContext, gxm_XUniqueContext, 0, 0, 0, H_XUniqueContext);
  XM_DEFINE_PROCEDURE(XSetRGBColormaps, gxm_XSetRGBColormaps, 5, 0, 0, H_XSetRGBColormaps);
  XM_DEFINE_PROCEDURE(XSetWMHints, gxm_XSetWMHints, 3, 0, 0, H_XSetWMHints);
  XM_DEFINE_PROCEDURE(XSetRegion, gxm_XSetRegion, 3, 0, 0, H_XSetRegion);
#if (!XM_DISABLE_DEPRECATED)
  XM_DEFINE_PROCEDURE(XSetStandardColormap, gxm_XSetStandardColormap, 4, 0, 0, H_XSetStandardColormap);
  XM_DEFINE_PROCEDURE(XSetStandardProperties, gxm_XSetStandardProperties, 8, 0, 0, H_XSetStandardProperties);
#endif
  XM_DEFINE_PROCEDURE(XSetWMProperties, gxm_XSetWMProperties, 8, 0, 0, H_XSetWMProperties);

  XM_DEFINE_PROCEDURE(XShrinkRegion, gxm_XShrinkRegion, 3, 0, 0, H_XShrinkRegion);
  XM_DEFINE_PROCEDURE(XSubtractRegion, gxm_XSubtractRegion, 3, 0, 0, H_XSubtractRegion);
  XM_DEFINE_PROCEDURE(XUnionRectWithRegion, gxm_XUnionRectWithRegion, 3, 0, 0, H_XUnionRectWithRegion);
  XM_DEFINE_PROCEDURE(XUnionRegion, gxm_XUnionRegion, 3, 0, 0, H_XUnionRegion);
  XM_DEFINE_PROCEDURE(XXorRegion, gxm_XXorRegion, 3, 0, 0, H_XXorRegion);

  XM_DEFINE_PROCEDURE(DefaultScreen, gxm_DefaultScreen, 1, 0, 0, H_DefaultScreen);
  XM_DEFINE_PROCEDURE(DefaultRootWindow, gxm_DefaultRootWindow, 1, 0, 0, H_DefaultRootWindow);
  XM_DEFINE_PROCEDURE(QLength, gxm_QLength, 1, 0, 0, H_QLength);
  XM_DEFINE_PROCEDURE(ScreenCount, gxm_ScreenCount, 1, 0, 0, H_ScreenCount);
  XM_DEFINE_PROCEDURE(ServerVendor, gxm_ServerVendor, 1, 0, 0, H_ServerVendor);
  XM_DEFINE_PROCEDURE(ProtocolVersion, gxm_ProtocolVersion, 1, 0, 0, H_ProtocolVersion);
  XM_DEFINE_PROCEDURE(ProtocolRevision, gxm_ProtocolRevision, 1, 0, 0, H_ProtocolRevision);
  XM_DEFINE_PROCEDURE(VendorRelease, gxm_VendorRelease, 1, 0, 0, H_VendorRelease);
  XM_DEFINE_PROCEDURE(DisplayString, gxm_DisplayString, 1, 0, 0, H_DisplayString);
  XM_DEFINE_PROCEDURE(BitmapUnit, gxm_BitmapUnit, 1, 0, 0, H_BitmapUnit);
  XM_DEFINE_PROCEDURE(BitmapBitOrder, gxm_BitmapBitOrder, 1, 0, 0, H_BitmapBitOrder);
  XM_DEFINE_PROCEDURE(BitmapPad, gxm_BitmapPad, 1, 0, 0, H_BitmapPad);
  XM_DEFINE_PROCEDURE(ImageByteOrder, gxm_ImageByteOrder, 1, 0, 0, H_ImageByteOrder);
  XM_DEFINE_PROCEDURE(NextRequest, gxm_NextRequest, 1, 0, 0, H_NextRequest);
  XM_DEFINE_PROCEDURE(LastKnownRequestProcessed, gxm_LastKnownRequestProcessed, 1, 0, 0, H_LastKnownRequestProcessed);
  XM_DEFINE_PROCEDURE(DefaultScreenOfDisplay, gxm_DefaultScreenOfDisplay, 1, 0, 0, H_DefaultScreenOfDisplay);
  XM_DEFINE_PROCEDURE(DisplayOfScreen, gxm_DisplayOfScreen, 1, 0, 0, H_DisplayOfScreen);
  XM_DEFINE_PROCEDURE(RootWindowOfScreen, gxm_RootWindowOfScreen, 1, 0, 0, H_RootWindowOfScreen);
  XM_DEFINE_PROCEDURE(BlackPixelOfScreen, gxm_BlackPixelOfScreen, 1, 0, 0, H_BlackPixelOfScreen);
  XM_DEFINE_PROCEDURE(WhitePixelOfScreen, gxm_WhitePixelOfScreen, 1, 0, 0, H_WhitePixelOfScreen);
  XM_DEFINE_PROCEDURE(DefaultColormapOfScreen, gxm_DefaultColormapOfScreen, 1, 0, 0, H_DefaultColormapOfScreen);
  XM_DEFINE_PROCEDURE(DefaultDepthOfScreen, gxm_DefaultDepthOfScreen, 1, 0, 0, H_DefaultDepthOfScreen);
  XM_DEFINE_PROCEDURE(DefaultGCOfScreen, gxm_DefaultGCOfScreen, 1, 0, 0, H_DefaultGCOfScreen);
  XM_DEFINE_PROCEDURE(DefaultVisualOfScreen, gxm_DefaultVisualOfScreen, 1, 0, 0, H_DefaultVisualOfScreen);
  XM_DEFINE_PROCEDURE(WidthOfScreen, gxm_WidthOfScreen, 1, 0, 0, H_WidthOfScreen);
  XM_DEFINE_PROCEDURE(HeightOfScreen, gxm_HeightOfScreen, 1, 0, 0, H_HeightOfScreen);
  XM_DEFINE_PROCEDURE(WidthMMOfScreen, gxm_WidthMMOfScreen, 1, 0, 0, H_WidthMMOfScreen);
  XM_DEFINE_PROCEDURE(HeightMMOfScreen, gxm_HeightMMOfScreen, 1, 0, 0, H_HeightMMOfScreen);
  XM_DEFINE_PROCEDURE(PlanesOfScreen, gxm_PlanesOfScreen, 1, 0, 0, H_PlanesOfScreen);
  XM_DEFINE_PROCEDURE(CellsOfScreen, gxm_CellsOfScreen, 1, 0, 0, H_CellsOfScreen);
  XM_DEFINE_PROCEDURE(MinCmapsOfScreen, gxm_MinCmapsOfScreen, 1, 0, 0, H_MinCmapsOfScreen);
  XM_DEFINE_PROCEDURE(MaxCmapsOfScreen, gxm_MaxCmapsOfScreen, 1, 0, 0, H_MaxCmapsOfScreen);
  XM_DEFINE_PROCEDURE(DoesSaveUnders, gxm_DoesSaveUnders, 1, 0, 0, H_DoesSaveUnders);
  XM_DEFINE_PROCEDURE(DoesBackingStore, gxm_DoesBackingStore, 1, 0, 0, H_DoesBackingStore);
  XM_DEFINE_PROCEDURE(EventMaskOfScreen, gxm_EventMaskOfScreen, 1, 0, 0, H_EventMaskOfScreen);
  XM_DEFINE_PROCEDURE(RootWindow, gxm_RootWindow, 2, 0, 0, H_RootWindow);
  XM_DEFINE_PROCEDURE(DefaultVisual, gxm_DefaultVisual, 2, 0, 0, H_DefaultVisual);
  XM_DEFINE_PROCEDURE(DefaultGC, gxm_DefaultGC, 2, 0, 0, H_DefaultGC);
  XM_DEFINE_PROCEDURE(BlackPixel, gxm_BlackPixel, 2, 0, 0, H_BlackPixel);
  XM_DEFINE_PROCEDURE(WhitePixel, gxm_WhitePixel, 2, 0, 0, H_WhitePixel);
  XM_DEFINE_PROCEDURE(DisplayWidth, gxm_DisplayWidth, 2, 0, 0, H_DisplayWidth);
  XM_DEFINE_PROCEDURE(DisplayHeight, gxm_DisplayHeight, 2, 0, 0, H_DisplayHeight);
  XM_DEFINE_PROCEDURE(DisplayWidthMM, gxm_DisplayWidthMM, 2, 0, 0, H_DisplayWidthMM);
  XM_DEFINE_PROCEDURE(DisplayHeightMM, gxm_DisplayHeightMM, 2, 0, 0, H_DisplayHeightMM);
  XM_DEFINE_PROCEDURE(DisplayPlanes, gxm_DisplayPlanes, 2, 0, 0, H_DisplayPlanes);
  XM_DEFINE_PROCEDURE(DisplayCells, gxm_DisplayCells, 2, 0, 0, H_DisplayCells);
  XM_DEFINE_PROCEDURE(DefaultColormap, gxm_DefaultColormap, 2, 0, 0, H_DefaultColormap);
  XM_DEFINE_PROCEDURE(ScreenOfDisplay, gxm_ScreenOfDisplay, 2, 0, 0, H_ScreenOfDisplay);
  XM_DEFINE_PROCEDURE(DefaultDepth, gxm_DefaultDepth, 2, 0, 0, H_DefaultDepth);

  XM_DEFINE_PROCEDURE(IsKeypadKey, gxm_IsKeypadKey, 1, 0, 0, H_IsKeypadKey);
  XM_DEFINE_PROCEDURE(IsPrivateKeypadKey, gxm_IsPrivateKeypadKey, 1, 0, 0, H_IsPrivateKeypadKey);
  XM_DEFINE_PROCEDURE(IsCursorKey, gxm_IsCursorKey, 1, 0, 0, H_IsCursorKey);
  XM_DEFINE_PROCEDURE(IsPFKey, gxm_IsPFKey, 1, 0, 0, H_IsPFKey);
  XM_DEFINE_PROCEDURE(IsFunctionKey, gxm_IsFunctionKey, 1, 0, 0, H_IsFunctionKey);
  XM_DEFINE_PROCEDURE(IsMiscFunctionKey, gxm_IsMiscFunctionKey, 1, 0, 0, H_IsMiscFunctionKey);
  XM_DEFINE_PROCEDURE(IsModifierKey, gxm_IsModifierKey, 1, 0, 0, H_IsModifierKey);

#if HAVE_MOTIF
  XM_DEFINE_PROCEDURE(XmCreateMessageBox, gxm_XmCreateMessageBox, 3, 1, 0, H_XmCreateMessageBox);
  XM_DEFINE_PROCEDURE(XmCreateMessageDialog, gxm_XmCreateMessageDialog, 3, 1, 0, H_XmCreateMessageDialog);
  XM_DEFINE_PROCEDURE(XmCreateErrorDialog, gxm_XmCreateErrorDialog, 3, 1, 0, H_XmCreateErrorDialog);
  XM_DEFINE_PROCEDURE(XmCreateInformationDialog, gxm_XmCreateInformationDialog, 3, 1, 0, H_XmCreateInformationDialog);
  XM_DEFINE_PROCEDURE(XmCreateQuestionDialog, gxm_XmCreateQuestionDialog, 3, 1, 0, H_XmCreateQuestionDialog);
  XM_DEFINE_PROCEDURE(XmCreateWarningDialog, gxm_XmCreateWarningDialog, 3, 1, 0, H_XmCreateWarningDialog);
  XM_DEFINE_PROCEDURE(XmCreateWorkingDialog, gxm_XmCreateWorkingDialog, 3, 1, 0, H_XmCreateWorkingDialog);
  XM_DEFINE_PROCEDURE(XmCreateTemplateDialog, gxm_XmCreateTemplateDialog, 3, 1, 0, H_XmCreateTemplateDialog);
  XM_DEFINE_PROCEDURE(XmMessageBoxGetChild, gxm_XmMessageBoxGetChild, 2, 0, 0, H_XmMessageBoxGetChild);
  XM_DEFINE_PROCEDURE(XmCreateArrowButtonGadget, gxm_XmCreateArrowButtonGadget, 3, 1, 0, H_XmCreateArrowButtonGadget);
  XM_DEFINE_PROCEDURE(XmCreateArrowButton, gxm_XmCreateArrowButton, 3, 1, 0, H_XmCreateArrowButton);
#if MOTIF_2
  XM_DEFINE_PROCEDURE(XmCreateNotebook, gxm_XmCreateNotebook, 3, 1, 0, H_XmCreateNotebook);
  XM_DEFINE_PROCEDURE(XmNotebookGetPageInfo, gxm_XmNotebookGetPageInfo, 2, 0, 0, H_XmNotebookGetPageInfo);
#if HAVE_XM_XP
  XM_DEFINE_PROCEDURE(XmPrintSetup, gxm_XmPrintSetup, 4, 1, 0, H_XmPrintSetup);
  XM_DEFINE_PROCEDURE(XmPrintToFile, gxm_XmPrintToFile, 4, 0, 0, H_XmPrintToFile);
  XM_DEFINE_PROCEDURE(XmPrintPopupPDM, gxm_XmPrintPopupPDM, 2, 0, 0, H_XmPrintPopupPDM);
  XM_DEFINE_PROCEDURE(XmRedisplayWidget, gxm_XmRedisplayWidget, 1, 0, 0, H_XmRedisplayWidget);
#endif
  XM_DEFINE_PROCEDURE(XmTransferSetParameters, gxm_XmTransferSetParameters, 5, 0, 0, H_XmTransferSetParameters);
  XM_DEFINE_PROCEDURE(XmTransferDone, gxm_XmTransferDone, 2, 0, 0, H_XmTransferDone);
  XM_DEFINE_PROCEDURE(XmTransferValue, gxm_XmTransferValue, 5, 0, 0, H_XmTransferValue);
  XM_DEFINE_PROCEDURE(XmTransferStartRequest, gxm_XmTransferStartRequest, 1, 0, 0, H_XmTransferStartRequest);
  XM_DEFINE_PROCEDURE(XmTransferSendRequest, gxm_XmTransferSendRequest, 2, 0, 0, H_XmTransferSendRequest);
  XM_DEFINE_PROCEDURE(XmCreateComboBox, gxm_XmCreateComboBox, 3, 1, 0, H_XmCreateComboBox);
  XM_DEFINE_PROCEDURE(XmCreateDropDownComboBox, gxm_XmCreateDropDownComboBox, 3, 1, 0, H_XmCreateDropDownComboBox);
  XM_DEFINE_PROCEDURE(XmCreateDropDownList, gxm_XmCreateDropDownList, 3, 1, 0, H_XmCreateDropDownList);
  XM_DEFINE_PROCEDURE(XmComboBoxAddItem, gxm_XmComboBoxAddItem, 4, 0, 0, H_XmComboBoxAddItem);
  XM_DEFINE_PROCEDURE(XmComboBoxDeletePos, gxm_XmComboBoxDeletePos, 2, 0, 0, H_XmComboBoxDeletePos);
  XM_DEFINE_PROCEDURE(XmComboBoxSelectItem, gxm_XmComboBoxSelectItem, 2, 0, 0, H_XmComboBoxSelectItem);
  XM_DEFINE_PROCEDURE(XmComboBoxSetItem, gxm_XmComboBoxSetItem, 2, 0, 0, H_XmComboBoxSetItem);
  XM_DEFINE_PROCEDURE(XmComboBoxUpdate, gxm_XmComboBoxUpdate, 1, 0, 0, H_XmComboBoxUpdate);
  XM_DEFINE_PROCEDURE(XmCreateContainer, gxm_XmCreateContainer, 3, 1, 0, H_XmCreateContainer);
  XM_DEFINE_PROCEDURE(XmContainerGetItemChildren, gxm_XmContainerGetItemChildren, 2, 0, 0, H_XmContainerGetItemChildren);
  XM_DEFINE_PROCEDURE(XmContainerRelayout, gxm_XmContainerRelayout, 1, 0, 0, H_XmContainerRelayout);
  XM_DEFINE_PROCEDURE(XmContainerReorder, gxm_XmContainerReorder, 3, 0, 0, H_XmContainerReorder);
  XM_DEFINE_PROCEDURE(XmContainerCut, gxm_XmContainerCut, 2, 0, 0, H_XmContainerCut);
  XM_DEFINE_PROCEDURE(XmContainerCopy, gxm_XmContainerCopy, 2, 0, 0, H_XmContainerCopy);
  XM_DEFINE_PROCEDURE(XmContainerPaste, gxm_XmContainerPaste, 1, 0, 0, H_XmContainerPaste);
  XM_DEFINE_PROCEDURE(XmContainerCopyLink, gxm_XmContainerCopyLink, 2, 0, 0, H_XmContainerCopyLink);
  XM_DEFINE_PROCEDURE(XmContainerPasteLink, gxm_XmContainerPasteLink, 1, 0, 0, H_XmContainerPasteLink);
  XM_DEFINE_PROCEDURE(XmCreateSpinBox, gxm_XmCreateSpinBox, 3, 1, 0, H_XmCreateSpinBox);
  XM_DEFINE_PROCEDURE(XmSpinBoxValidatePosition, gxm_XmSpinBoxValidatePosition, 1, 0, 0, H_XmSpinBoxValidatePosition);
  XM_DEFINE_PROCEDURE(XmCreateSimpleSpinBox, gxm_XmCreateSimpleSpinBox, 3, 1, 0, H_XmCreateSimpleSpinBox);
  XM_DEFINE_PROCEDURE(XmSimpleSpinBoxAddItem, gxm_XmSimpleSpinBoxAddItem, 3, 0, 0, H_XmSimpleSpinBoxAddItem);
  XM_DEFINE_PROCEDURE(XmSimpleSpinBoxDeletePos, gxm_XmSimpleSpinBoxDeletePos, 2, 0, 0, H_XmSimpleSpinBoxDeletePos);
  XM_DEFINE_PROCEDURE(XmSimpleSpinBoxSetItem, gxm_XmSimpleSpinBoxSetItem, 2, 0, 0, H_XmSimpleSpinBoxSetItem);
  XM_DEFINE_PROCEDURE(XmDropSiteRegistered, gxm_XmDropSiteRegistered, 1, 0, 0, H_XmDropSiteRegistered);
  XM_DEFINE_PROCEDURE(XmTextFieldCopyLink, gxm_XmTextFieldCopyLink, 2, 0, 0, H_XmTextFieldCopyLink);
  XM_DEFINE_PROCEDURE(XmCreateIconGadget, gxm_XmCreateIconGadget, 3, 1, 0, H_XmCreateIconGadget);
#ifndef LESSTIF_VERSION
  XM_DEFINE_PROCEDURE(XmTextFieldPasteLink, gxm_XmTextFieldPasteLink, 1, 0, 0, H_XmTextFieldPasteLink);
  XM_DEFINE_PROCEDURE(XmTextGetCenterline, gxm_XmTextGetCenterline, 1, 0, 0, H_XmTextGetCenterline);
  XM_DEFINE_PROCEDURE(XmToggleButtonGadgetSetValue, gxm_XmToggleButtonGadgetSetValue, 3, 0, 0, H_XmToggleButtonGadgetSetValue);
  XM_DEFINE_PROCEDURE(XmCreateIconHeader, gxm_XmCreateIconHeader, 3, 1, 0, H_XmCreateIconHeader);
#endif
  XM_DEFINE_PROCEDURE(XmObjectAtPoint, gxm_XmObjectAtPoint, 3, 0, 0, H_XmObjectAtPoint);
  XM_DEFINE_PROCEDURE(XmConvertStringToUnits, gxm_XmConvertStringToUnits, 4, 0, 0, H_XmConvertStringToUnits);
  XM_DEFINE_PROCEDURE(XmCreateGrabShell, gxm_XmCreateGrabShell, 3, 1, 0, H_XmCreateGrabShell);
#ifndef LESSTIF_VERSION
  XM_DEFINE_PROCEDURE(XmToggleButtonSetValue, gxm_XmToggleButtonSetValue, 3, 0, 0, H_XmToggleButtonSetValue);
  XM_DEFINE_PROCEDURE(XmTextPasteLink, gxm_XmTextPasteLink, 1, 0, 0, H_XmTextPasteLink);
  XM_DEFINE_PROCEDURE(XmScaleSetTicks, gxm_XmScaleSetTicks, 7, 0, 0, H_XmScaleSetTicks);
#endif
  XM_DEFINE_PROCEDURE(XmTextCopyLink, gxm_XmTextCopyLink, 2, 0, 0, H_XmTextCopyLink);
#endif
  XM_DEFINE_PROCEDURE(XmInternAtom, gxm_XmInternAtom, 3, 0, 0, H_XmInternAtom);
  XM_DEFINE_PROCEDURE(XmGetAtomName, gxm_XmGetAtomName, 2, 0, 0, H_XmGetAtomName);
  XM_DEFINE_PROCEDURE(XmCreatePanedWindow, gxm_XmCreatePanedWindow, 3, 1, 0, H_XmCreatePanedWindow);
  XM_DEFINE_PROCEDURE(XmCreateBulletinBoard, gxm_XmCreateBulletinBoard, 3, 1, 0, H_XmCreateBulletinBoard);
  XM_DEFINE_PROCEDURE(XmCreateBulletinBoardDialog, gxm_XmCreateBulletinBoardDialog, 3, 1, 0, H_XmCreateBulletinBoardDialog);
  XM_DEFINE_PROCEDURE(XmCreateCascadeButtonGadget, gxm_XmCreateCascadeButtonGadget, 3, 1, 0, H_XmCreateCascadeButtonGadget);
  XM_DEFINE_PROCEDURE(XmCascadeButtonGadgetHighlight, gxm_XmCascadeButtonGadgetHighlight, 2, 0, 0, H_XmCascadeButtonGadgetHighlight);
  XM_DEFINE_PROCEDURE(XmAddProtocols, gxm_XmAddProtocols, 3, 1, 0, H_XmAddProtocols);
  XM_DEFINE_PROCEDURE(XmRemoveProtocols, gxm_XmRemoveProtocols, 3, 1, 0, H_XmRemoveProtocols);
  XM_DEFINE_PROCEDURE(XmAddProtocolCallback, gxm_XmAddProtocolCallback, 5, 0, 0, H_XmAddProtocolCallback);
  XM_DEFINE_PROCEDURE(XmRemoveProtocolCallback, gxm_XmRemoveProtocolCallback, 5, 0, 0, H_XmRemoveProtocolCallback);
  XM_DEFINE_PROCEDURE(XmActivateProtocol, gxm_XmActivateProtocol, 3, 0, 0, H_XmActivateProtocol);
  XM_DEFINE_PROCEDURE(XmDeactivateProtocol, gxm_XmDeactivateProtocol, 3, 0, 0, H_XmDeactivateProtocol);
  XM_DEFINE_PROCEDURE(XmSetProtocolHooks, gxm_XmSetProtocolHooks, 7, 0, 0, H_XmSetProtocolHooks);
  XM_DEFINE_PROCEDURE(XmCreateCascadeButton, gxm_XmCreateCascadeButton, 3, 1, 0, H_XmCreateCascadeButton);
  XM_DEFINE_PROCEDURE(XmCascadeButtonHighlight, gxm_XmCascadeButtonHighlight, 2, 0, 0, H_XmCascadeButtonHighlight);
  XM_DEFINE_PROCEDURE(XmCreatePushButtonGadget, gxm_XmCreatePushButtonGadget, 3, 1, 0, H_XmCreatePushButtonGadget);
  XM_DEFINE_PROCEDURE(XmCreatePushButton, gxm_XmCreatePushButton, 3, 1, 0, H_XmCreatePushButton);
  XM_DEFINE_PROCEDURE(XmCreateCommand, gxm_XmCreateCommand, 3, 1, 0, H_XmCreateCommand);
  XM_DEFINE_PROCEDURE(XmCommandGetChild, gxm_XmCommandGetChild, 2, 0, 0, H_XmCommandGetChild);
  XM_DEFINE_PROCEDURE(XmCommandSetValue, gxm_XmCommandSetValue, 2, 0, 0, H_XmCommandSetValue);
  XM_DEFINE_PROCEDURE(XmCommandAppendValue, gxm_XmCommandAppendValue, 2, 0, 0, H_XmCommandAppendValue);
  XM_DEFINE_PROCEDURE(XmCommandError, gxm_XmCommandError, 2, 0, 0, H_XmCommandError);
  XM_DEFINE_PROCEDURE(XmCreateCommandDialog, gxm_XmCreateCommandDialog, 3, 1, 0, H_XmCreateCommandDialog);
  XM_DEFINE_PROCEDURE(XmMenuPosition, gxm_XmMenuPosition, 2, 0, 0, H_XmMenuPosition);
  XM_DEFINE_PROCEDURE(XmCreateRowColumn, gxm_XmCreateRowColumn, 3, 1, 0, H_XmCreateRowColumn);
  XM_DEFINE_PROCEDURE(XmCreateWorkArea, gxm_XmCreateWorkArea, 3, 1, 0, H_XmCreateWorkArea);
  XM_DEFINE_PROCEDURE(XmCreateRadioBox, gxm_XmCreateRadioBox, 3, 1, 0, H_XmCreateRadioBox);
  XM_DEFINE_PROCEDURE(XmCreateOptionMenu, gxm_XmCreateOptionMenu, 3, 1, 0, H_XmCreateOptionMenu);
  XM_DEFINE_PROCEDURE(XmOptionLabelGadget, gxm_XmOptionLabelGadget, 1, 0, 0, H_XmOptionLabelGadget);
  XM_DEFINE_PROCEDURE(XmOptionButtonGadget, gxm_XmOptionButtonGadget, 1, 0, 0, H_XmOptionButtonGadget);
  XM_DEFINE_PROCEDURE(XmCreateMenuBar, gxm_XmCreateMenuBar, 3, 1, 0, H_XmCreateMenuBar);
  XM_DEFINE_PROCEDURE(XmCreatePopupMenu, gxm_XmCreatePopupMenu, 3, 1, 0, H_XmCreatePopupMenu);
  XM_DEFINE_PROCEDURE(XmCreatePulldownMenu, gxm_XmCreatePulldownMenu, 3, 1, 0, H_XmCreatePulldownMenu);
  XM_DEFINE_PROCEDURE(XmGetPostedFromWidget, gxm_XmGetPostedFromWidget, 1, 0, 0, H_XmGetPostedFromWidget);
  XM_DEFINE_PROCEDURE(XmGetTearOffControl, gxm_XmGetTearOffControl, 1, 0, 0, H_XmGetTearOffControl);
#if (!XM_DISABLE_DEPRECATED)
  XM_DEFINE_PROCEDURE(XmAddToPostFromList, gxm_XmAddToPostFromList, 2, 0, 0, H_XmAddToPostFromList);
  XM_DEFINE_PROCEDURE(XmRemoveFromPostFromList, gxm_XmRemoveFromPostFromList, 2, 0, 0, H_XmRemoveFromPostFromList);
#endif
  XM_DEFINE_PROCEDURE(XmScaleSetValue, gxm_XmScaleSetValue, 2, 0, 0, H_XmScaleSetValue);
  XM_DEFINE_PROCEDURE(XmScaleGetValue, gxm_XmScaleGetValue, 1, 0, 0, H_XmScaleGetValue);
  XM_DEFINE_PROCEDURE(XmCreateScale, gxm_XmCreateScale, 3, 1, 0, H_XmCreateScale);
#ifndef LESSTIF_VERSION
  XM_DEFINE_PROCEDURE(XmClipboardBeginCopy, gxm_XmClipboardBeginCopy, 5, 0, 0, H_XmClipboardBeginCopy);
#endif
  XM_DEFINE_PROCEDURE(XmClipboardStartCopy, gxm_XmClipboardStartCopy, 6, 0, 0, H_XmClipboardStartCopy);
  XM_DEFINE_PROCEDURE(XmClipboardCopy, gxm_XmClipboardCopy, 7, 0, 0, H_XmClipboardCopy);
  XM_DEFINE_PROCEDURE(XmClipboardEndCopy, gxm_XmClipboardEndCopy, 3, 0, 0, H_XmClipboardEndCopy);
  XM_DEFINE_PROCEDURE(XmClipboardCancelCopy, gxm_XmClipboardCancelCopy, 3, 0, 0, H_XmClipboardCancelCopy);
  XM_DEFINE_PROCEDURE(XmClipboardWithdrawFormat, gxm_XmClipboardWithdrawFormat, 3, 0, 0, H_XmClipboardWithdrawFormat);
  XM_DEFINE_PROCEDURE(XmClipboardCopyByName, gxm_XmClipboardCopyByName, 6, 0, 0, H_XmClipboardCopyByName);
  XM_DEFINE_PROCEDURE(XmClipboardUndoCopy, gxm_XmClipboardUndoCopy, 2, 0, 0, H_XmClipboardUndoCopy);
  XM_DEFINE_PROCEDURE(XmClipboardLock, gxm_XmClipboardLock, 2, 0, 0, H_XmClipboardLock);
  XM_DEFINE_PROCEDURE(XmClipboardUnlock, gxm_XmClipboardUnlock, 3, 0, 0, H_XmClipboardUnlock);
  XM_DEFINE_PROCEDURE(XmClipboardStartRetrieve, gxm_XmClipboardStartRetrieve, 3, 0, 0, H_XmClipboardStartRetrieve);
  XM_DEFINE_PROCEDURE(XmClipboardEndRetrieve, gxm_XmClipboardEndRetrieve, 2, 0, 0, H_XmClipboardEndRetrieve);
  XM_DEFINE_PROCEDURE(XmClipboardRetrieve, gxm_XmClipboardRetrieve, 4, 0, 0, H_XmClipboardRetrieve);
  XM_DEFINE_PROCEDURE(XmClipboardInquireCount, gxm_XmClipboardInquireCount, 2, 0, 0, H_XmClipboardInquireCount);
  XM_DEFINE_PROCEDURE(XmClipboardInquireFormat, gxm_XmClipboardInquireFormat, 4, 0, 0, H_XmClipboardInquireFormat);
  XM_DEFINE_PROCEDURE(XmClipboardInquireLength, gxm_XmClipboardInquireLength, 3, 0, 0, H_XmClipboardInquireLength);
  XM_DEFINE_PROCEDURE(XmClipboardInquirePendingItems, gxm_XmClipboardInquirePendingItems, 3, 0, 0, H_XmClipboardInquirePendingItems);
  XM_DEFINE_PROCEDURE(XmClipboardRegisterFormat, gxm_XmClipboardRegisterFormat, 3, 0, 0, H_XmClipboardRegisterFormat);
#if MOTIF_2
#ifndef LESSTIF_VERSION
  XM_DEFINE_PROCEDURE(XmGetXmScreen, gxm_XmGetXmScreen, 1, 0, 0, H_XmGetXmScreen);
#endif
#endif
  XM_DEFINE_PROCEDURE(XmCreateScrollBar, gxm_XmCreateScrollBar, 3, 1, 0, H_XmCreateScrollBar);
  XM_DEFINE_PROCEDURE(XmScrollBarGetValues, gxm_XmScrollBarGetValues, 1, 0, 0, H_XmScrollBarGetValues);
  XM_DEFINE_PROCEDURE(XmScrollBarSetValues, gxm_XmScrollBarSetValues, 6, 0, 0, H_XmScrollBarSetValues);
  XM_DEFINE_PROCEDURE(XmCreateDialogShell, gxm_XmCreateDialogShell, 3, 1, 0, H_XmCreateDialogShell);
  XM_DEFINE_PROCEDURE(XmCreateScrolledWindow, gxm_XmCreateScrolledWindow, 3, 1, 0, H_XmCreateScrolledWindow);
  XM_DEFINE_PROCEDURE(XmScrollVisible, gxm_XmScrollVisible, 4, 0, 0, H_XmScrollVisible);
  XM_DEFINE_PROCEDURE(XmGetDragContext, gxm_XmGetDragContext, 2, 0, 0, H_XmGetDragContext);
  XM_DEFINE_PROCEDURE(XmGetXmDisplay, gxm_XmGetXmDisplay, 1, 0, 0, H_XmGetXmDisplay);
  XM_DEFINE_PROCEDURE(XmSelectionBoxGetChild, gxm_XmSelectionBoxGetChild, 2, 0, 0, H_XmSelectionBoxGetChild);
  XM_DEFINE_PROCEDURE(XmCreateSelectionBox, gxm_XmCreateSelectionBox, 3, 1, 0, H_XmCreateSelectionBox);
  XM_DEFINE_PROCEDURE(XmCreateSelectionDialog, gxm_XmCreateSelectionDialog, 3, 1, 0, H_XmCreateSelectionDialog);
  XM_DEFINE_PROCEDURE(XmCreatePromptDialog, gxm_XmCreatePromptDialog, 3, 1, 0, H_XmCreatePromptDialog);
  XM_DEFINE_PROCEDURE(XmDragStart, gxm_XmDragStart, 3, 1, 0, H_XmDragStart);
  XM_DEFINE_PROCEDURE(XmDragCancel, gxm_XmDragCancel, 1, 0, 0, H_XmDragCancel);
  XM_DEFINE_PROCEDURE(XmTargetsAreCompatible, gxm_XmTargetsAreCompatible, 5, 0, 0, H_XmTargetsAreCompatible);
  XM_DEFINE_PROCEDURE(XmCreateSeparatorGadget, gxm_XmCreateSeparatorGadget, 3, 1, 0, H_XmCreateSeparatorGadget);
  XM_DEFINE_PROCEDURE(XmCreateDragIcon, gxm_XmCreateDragIcon, 3, 1, 0, H_XmCreateDragIcon);
  XM_DEFINE_PROCEDURE(XmCreateSeparator, gxm_XmCreateSeparator, 3, 1, 0, H_XmCreateSeparator);
  XM_DEFINE_PROCEDURE(XmCreateDrawingArea, gxm_XmCreateDrawingArea, 3, 1, 0, H_XmCreateDrawingArea);
  XM_DEFINE_PROCEDURE(XmCreateDrawnButton, gxm_XmCreateDrawnButton, 3, 1, 0, H_XmCreateDrawnButton);
  XM_DEFINE_PROCEDURE(XmDropSiteRegister, gxm_XmDropSiteRegister, 2, 1, 0, H_XmDropSiteRegister);
  XM_DEFINE_PROCEDURE(XmDropSiteUnregister, gxm_XmDropSiteUnregister, 1, 0, 0, H_XmDropSiteUnregister);
  XM_DEFINE_PROCEDURE(XmDropSiteStartUpdate, gxm_XmDropSiteStartUpdate, 1, 0, 0, H_XmDropSiteStartUpdate);
  XM_DEFINE_PROCEDURE(XmDropSiteUpdate, gxm_XmDropSiteUpdate, 2, 1, 0, H_XmDropSiteUpdate);
  XM_DEFINE_PROCEDURE(XmDropSiteEndUpdate, gxm_XmDropSiteEndUpdate, 1, 0, 0, H_XmDropSiteEndUpdate);
  XM_DEFINE_PROCEDURE(XmDropSiteRetrieve, gxm_XmDropSiteRetrieve, 2, 1, 0, H_XmDropSiteRetrieve);
  XM_DEFINE_PROCEDURE(XmDropSiteQueryStackingOrder, gxm_XmDropSiteQueryStackingOrder, 1, 0, 0, H_XmDropSiteQueryStackingOrder);
  XM_DEFINE_PROCEDURE(XmDropSiteConfigureStackingOrder, gxm_XmDropSiteConfigureStackingOrder, 3, 0, 0, H_XmDropSiteConfigureStackingOrder);
  XM_DEFINE_PROCEDURE(XmDropTransferStart, gxm_XmDropTransferStart, 2, 1, 0, H_XmDropTransferStart);
  XM_DEFINE_PROCEDURE(XmDropTransferAdd, gxm_XmDropTransferAdd, 2, 0, 0, H_XmDropTransferAdd);
  XM_DEFINE_PROCEDURE(XmTextFieldGetString, gxm_XmTextFieldGetString, 1, 0, 0, H_XmTextFieldGetString);
  XM_DEFINE_PROCEDURE(XmTextFieldGetSubstring, gxm_XmTextFieldGetSubstring, 3, 0, 0, H_XmTextFieldGetSubstring);
  XM_DEFINE_PROCEDURE(XmTextFieldGetLastPosition, gxm_XmTextFieldGetLastPosition, 1, 0, 0, H_XmTextFieldGetLastPosition);
  XM_DEFINE_PROCEDURE(XmTextFieldSetString, gxm_XmTextFieldSetString, 2, 0, 0, H_XmTextFieldSetString);
  XM_DEFINE_PROCEDURE(XmTextFieldReplace, gxm_XmTextFieldReplace, 4, 0, 0, H_XmTextFieldReplace);
  XM_DEFINE_PROCEDURE(XmTextFieldInsert, gxm_XmTextFieldInsert, 3, 0, 0, H_XmTextFieldInsert);
#if MOTIF_2
  XM_DEFINE_PROCEDURE(XmTextFieldSetAddMode, gxm_XmTextFieldSetAddMode, 2, 0, 0, H_XmTextFieldSetAddMode);
#ifndef LESSTIF_VERSION
  XM_DEFINE_PROCEDURE(XmTextFieldGetAddMode, gxm_XmTextFieldGetAddMode, 1, 0, 0, H_XmTextFieldGetAddMode);
#endif
#endif
  XM_DEFINE_PROCEDURE(XmTextFieldGetEditable, gxm_XmTextFieldGetEditable, 1, 0, 0, H_XmTextFieldGetEditable);
  XM_DEFINE_PROCEDURE(XmTextFieldSetEditable, gxm_XmTextFieldSetEditable, 2, 0, 0, H_XmTextFieldSetEditable);
  XM_DEFINE_PROCEDURE(XmTextFieldGetMaxLength, gxm_XmTextFieldGetMaxLength, 1, 0, 0, H_XmTextFieldGetMaxLength);
  XM_DEFINE_PROCEDURE(XmTextFieldSetMaxLength, gxm_XmTextFieldSetMaxLength, 2, 0, 0, H_XmTextFieldSetMaxLength);
  XM_DEFINE_PROCEDURE(XmTextFieldGetCursorPosition, gxm_XmTextFieldGetCursorPosition, 1, 0, 0, H_XmTextFieldGetCursorPosition);
  XM_DEFINE_PROCEDURE(XmTextFieldGetInsertionPosition, gxm_XmTextFieldGetInsertionPosition, 1, 0, 0, H_XmTextFieldGetInsertionPosition);
  XM_DEFINE_PROCEDURE(XmTextFieldSetCursorPosition, gxm_XmTextFieldSetCursorPosition, 2, 0, 0, H_XmTextFieldSetCursorPosition);
  XM_DEFINE_PROCEDURE(XmTextFieldSetInsertionPosition, gxm_XmTextFieldSetInsertionPosition, 2, 0, 0, H_XmTextFieldSetInsertionPosition);
  XM_DEFINE_PROCEDURE(XmTextFieldGetSelectionPosition, gxm_XmTextFieldGetSelectionPosition, 1, 0, 0, H_XmTextFieldGetSelectionPosition);
  XM_DEFINE_PROCEDURE(XmTextFieldGetSelection, gxm_XmTextFieldGetSelection, 1, 0, 0, H_XmTextFieldGetSelection);
  XM_DEFINE_PROCEDURE(XmTextFieldRemove, gxm_XmTextFieldRemove, 1, 0, 0, H_XmTextFieldRemove);
  XM_DEFINE_PROCEDURE(XmTextFieldCopy, gxm_XmTextFieldCopy, 2, 0, 0, H_XmTextFieldCopy);
  XM_DEFINE_PROCEDURE(XmTextFieldCut, gxm_XmTextFieldCut, 2, 0, 0, H_XmTextFieldCut);
  XM_DEFINE_PROCEDURE(XmTextFieldPaste, gxm_XmTextFieldPaste, 1, 0, 0, H_XmTextFieldPaste);
  XM_DEFINE_PROCEDURE(XmTextFieldClearSelection, gxm_XmTextFieldClearSelection, 2, 0, 0, H_XmTextFieldClearSelection);
  XM_DEFINE_PROCEDURE(XmTextFieldSetSelection, gxm_XmTextFieldSetSelection, 4, 0, 0, H_XmTextFieldSetSelection);
  XM_DEFINE_PROCEDURE(XmTextFieldXYToPos, gxm_XmTextFieldXYToPos, 3, 0, 0, H_XmTextFieldXYToPos);
  XM_DEFINE_PROCEDURE(XmTextFieldPosToXY, gxm_XmTextFieldPosToXY, 2, 0, 0, H_XmTextFieldPosToXY);
  XM_DEFINE_PROCEDURE(XmTextFieldShowPosition, gxm_XmTextFieldShowPosition, 2, 0, 0, H_XmTextFieldShowPosition);
  XM_DEFINE_PROCEDURE(XmTextFieldSetHighlight, gxm_XmTextFieldSetHighlight, 4, 0, 0, H_XmTextFieldSetHighlight);
  XM_DEFINE_PROCEDURE(XmTextFieldGetBaseline, gxm_XmTextFieldGetBaseline, 1, 0, 0, H_XmTextFieldGetBaseline);
  XM_DEFINE_PROCEDURE(XmCreateTextField, gxm_XmCreateTextField, 3, 1, 0, H_XmCreateTextField);
  XM_DEFINE_PROCEDURE(XmFileSelectionBoxGetChild, gxm_XmFileSelectionBoxGetChild, 2, 0, 0, H_XmFileSelectionBoxGetChild);
  XM_DEFINE_PROCEDURE(XmFileSelectionDoSearch, gxm_XmFileSelectionDoSearch, 2, 0, 0, H_XmFileSelectionDoSearch);
  XM_DEFINE_PROCEDURE(XmCreateFileSelectionBox, gxm_XmCreateFileSelectionBox, 3, 1, 0, H_XmCreateFileSelectionBox);
  XM_DEFINE_PROCEDURE(XmCreateFileSelectionDialog, gxm_XmCreateFileSelectionDialog, 3, 1, 0, H_XmCreateFileSelectionDialog);
  XM_DEFINE_PROCEDURE(XmTextSetHighlight, gxm_XmTextSetHighlight, 4, 0, 0, H_XmTextSetHighlight);
  XM_DEFINE_PROCEDURE(XmCreateScrolledText, gxm_XmCreateScrolledText, 3, 1, 0, H_XmCreateScrolledText);
  XM_DEFINE_PROCEDURE(XmCreateText, gxm_XmCreateText, 3, 1, 0, H_XmCreateText);
  XM_DEFINE_PROCEDURE(XmTextGetSubstring, gxm_XmTextGetSubstring, 3, 0, 0, H_XmTextGetSubstring);
  XM_DEFINE_PROCEDURE(XmTextGetString, gxm_XmTextGetString, 1, 0, 0, H_XmTextGetString);
  XM_DEFINE_PROCEDURE(XmTextGetLastPosition, gxm_XmTextGetLastPosition, 1, 0, 0, H_XmTextGetLastPosition);
  XM_DEFINE_PROCEDURE(XmTextSetString, gxm_XmTextSetString, 2, 0, 0, H_XmTextSetString);
  XM_DEFINE_PROCEDURE(XmTextReplace, gxm_XmTextReplace, 4, 0, 0, H_XmTextReplace);
  XM_DEFINE_PROCEDURE(XmTextInsert, gxm_XmTextInsert, 3, 0, 0, H_XmTextInsert);
#if MOTIF_2
  XM_DEFINE_PROCEDURE(XmTextSetAddMode, gxm_XmTextSetAddMode, 2, 0, 0, H_XmTextSetAddMode);
#ifndef LESSTIF_VERSION
  XM_DEFINE_PROCEDURE(XmTextGetAddMode, gxm_XmTextGetAddMode, 1, 0, 0, H_XmTextGetAddMode);
#endif
#endif
  XM_DEFINE_PROCEDURE(XmTextGetEditable, gxm_XmTextGetEditable, 1, 0, 0, H_XmTextGetEditable);
  XM_DEFINE_PROCEDURE(XmTextSetEditable, gxm_XmTextSetEditable, 2, 0, 0, H_XmTextSetEditable);
  XM_DEFINE_PROCEDURE(XmTextGetMaxLength, gxm_XmTextGetMaxLength, 1, 0, 0, H_XmTextGetMaxLength);
  XM_DEFINE_PROCEDURE(XmTextSetMaxLength, gxm_XmTextSetMaxLength, 2, 0, 0, H_XmTextSetMaxLength);
  XM_DEFINE_PROCEDURE(XmTextGetTopCharacter, gxm_XmTextGetTopCharacter, 1, 0, 0, H_XmTextGetTopCharacter);
  XM_DEFINE_PROCEDURE(XmTextSetTopCharacter, gxm_XmTextSetTopCharacter, 2, 0, 0, H_XmTextSetTopCharacter);
  XM_DEFINE_PROCEDURE(XmTextGetCursorPosition, gxm_XmTextGetCursorPosition, 1, 0, 0, H_XmTextGetCursorPosition);
  XM_DEFINE_PROCEDURE(XmTextGetInsertionPosition, gxm_XmTextGetInsertionPosition, 1, 0, 0, H_XmTextGetInsertionPosition);
  XM_DEFINE_PROCEDURE(XmTextSetInsertionPosition, gxm_XmTextSetInsertionPosition, 2, 0, 0, H_XmTextSetInsertionPosition);
  XM_DEFINE_PROCEDURE(XmTextSetCursorPosition, gxm_XmTextSetCursorPosition, 2, 0, 0, H_XmTextSetCursorPosition);
  XM_DEFINE_PROCEDURE(XmTextRemove, gxm_XmTextRemove, 1, 0, 0, H_XmTextRemove);
  XM_DEFINE_PROCEDURE(XmTextCopy, gxm_XmTextCopy, 2, 0, 0, H_XmTextCopy);
  XM_DEFINE_PROCEDURE(XmTextCut, gxm_XmTextCut, 2, 0, 0, H_XmTextCut);
  XM_DEFINE_PROCEDURE(XmTextPaste, gxm_XmTextPaste, 1, 0, 0, H_XmTextPaste);
  XM_DEFINE_PROCEDURE(XmTextGetSelection, gxm_XmTextGetSelection, 1, 0, 0, H_XmTextGetSelection);
  XM_DEFINE_PROCEDURE(XmTextSetSelection, gxm_XmTextSetSelection, 4, 0, 0, H_XmTextSetSelection);
  XM_DEFINE_PROCEDURE(XmTextClearSelection, gxm_XmTextClearSelection, 2, 0, 0, H_XmTextClearSelection);
  XM_DEFINE_PROCEDURE(XmTextGetSelectionPosition, gxm_XmTextGetSelectionPosition, 1, 0, 0, H_XmTextGetSelectionPosition);
  XM_DEFINE_PROCEDURE(XmTextXYToPos, gxm_XmTextXYToPos, 3, 0, 0, H_XmTextXYToPos);
  XM_DEFINE_PROCEDURE(XmTextPosToXY, gxm_XmTextPosToXY, 2, 0, 0, H_XmTextPosToXY);
  XM_DEFINE_PROCEDURE(XmTextGetSource, gxm_XmTextGetSource, 1, 0, 0, H_XmTextGetSource);
  XM_DEFINE_PROCEDURE(XmTextSetSource, gxm_XmTextSetSource, 4, 0, 0, H_XmTextSetSource);
  XM_DEFINE_PROCEDURE(XmTextShowPosition, gxm_XmTextShowPosition, 2, 0, 0, H_XmTextShowPosition);
  XM_DEFINE_PROCEDURE(XmTextScroll, gxm_XmTextScroll, 2, 0, 0, H_XmTextScroll);
  XM_DEFINE_PROCEDURE(XmTextGetBaseline, gxm_XmTextGetBaseline, 1, 0, 0, H_XmTextGetBaseline);
  XM_DEFINE_PROCEDURE(XmTextDisableRedisplay, gxm_XmTextDisableRedisplay, 1, 0, 0, H_XmTextDisableRedisplay);
  XM_DEFINE_PROCEDURE(XmTextEnableRedisplay, gxm_XmTextEnableRedisplay, 1, 0, 0, H_XmTextEnableRedisplay);
  XM_DEFINE_PROCEDURE(XmTextFindString, gxm_XmTextFindString, 4, 0, 0, H_XmTextFindString);
  XM_DEFINE_PROCEDURE(XmCreateForm, gxm_XmCreateForm, 3, 1, 0, H_XmCreateForm);
  XM_DEFINE_PROCEDURE(XmCreateFormDialog, gxm_XmCreateFormDialog, 3, 1, 0, H_XmCreateFormDialog);
  XM_DEFINE_PROCEDURE(XmCreateFrame, gxm_XmCreateFrame, 3, 1, 0, H_XmCreateFrame);
  XM_DEFINE_PROCEDURE(XmToggleButtonGadgetGetState, gxm_XmToggleButtonGadgetGetState, 1, 0, 0, H_XmToggleButtonGadgetGetState);
  XM_DEFINE_PROCEDURE(XmToggleButtonGadgetSetState, gxm_XmToggleButtonGadgetSetState, 3, 0, 0, H_XmToggleButtonGadgetSetState);
  XM_DEFINE_PROCEDURE(XmCreateToggleButtonGadget, gxm_XmCreateToggleButtonGadget, 3, 1, 0, H_XmCreateToggleButtonGadget);
  XM_DEFINE_PROCEDURE(XmToggleButtonGetState, gxm_XmToggleButtonGetState, 1, 0, 0, H_XmToggleButtonGetState);
  XM_DEFINE_PROCEDURE(XmToggleButtonSetState, gxm_XmToggleButtonSetState, 3, 0, 0, H_XmToggleButtonSetState);
  XM_DEFINE_PROCEDURE(XmCreateToggleButton, gxm_XmCreateToggleButton, 3, 1, 0, H_XmCreateToggleButton);
  XM_DEFINE_PROCEDURE(XmCreateLabelGadget, gxm_XmCreateLabelGadget, 3, 1, 0, H_XmCreateLabelGadget);
  XM_DEFINE_PROCEDURE(XmCreateLabel, gxm_XmCreateLabel, 3, 1, 0, H_XmCreateLabel);
  XM_DEFINE_PROCEDURE(XmIsMotifWMRunning, gxm_XmIsMotifWMRunning, 1, 0, 0, H_XmIsMotifWMRunning);
#if HAVE_XmToolTipGetLabel
  XM_DEFINE_PROCEDURE(XmToolTipGetLabel, gxm_XmToolTipGetLabel, 1, 0, 0, H_XmToolTipGetLabel);
#endif
#if HAVE_XmCreateButtonBox
  XM_DEFINE_PROCEDURE(XmCreateButtonBox, gxm_XmCreateButtonBox, 3, 1, 0, H_XmCreateButtonBox);
#endif
#if HAVE_XmCreateDataField
  XM_DEFINE_PROCEDURE(XmCreateDataField, gxm_XmCreateDataField, 3, 1, 0, H_XmCreateDataField);
  XM_DEFINE_PROCEDURE(XmDataFieldSetString, gxm_XmDataFieldSetString, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldGetString, gxm_XmDataFieldGetString, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldSetHighlight, gxm_XmDataFieldSetHighlight, 4, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldSetAddMode, gxm_XmDataFieldSetAddMode, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldGetSelection, gxm_XmDataFieldGetSelection, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldSetSelection, gxm_XmDataFieldSetSelection, 4, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldGetSelectionPosition, gxm_XmDataFieldGetSelectionPosition, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldXYToPos, gxm_XmDataFieldXYToPos, 3, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldShowPosition, gxm_XmDataFieldShowPosition, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldCut, gxm_XmDataFieldCut, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldCopy, gxm_XmDataFieldCopy, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldPaste, gxm_XmDataFieldPaste, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldSetEditable, gxm_XmDataFieldSetEditable, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDataFieldSetInsertionPosition, gxm_XmDataFieldSetInsertionPosition, 2, 0, 0, NULL);
#endif
#if HAVE_XmCreateMultiList
  XM_DEFINE_PROCEDURE(XmCreateMultiList, gxm_XmCreateMultiList, 3, 1, 0, H_XmCreateMultiList);
  XM_DEFINE_PROCEDURE(XmMultiListDeselectAllItems, gxm_XmMultiListDeselectAllItems, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListUnselectAllItems, gxm_XmMultiListUnselectAllItems, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListSelectAllItems, gxm_XmMultiListSelectAllItems, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListDeselectRow, gxm_XmMultiListDeselectRow, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListToggleRow, gxm_XmMultiListToggleRow, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListMakeRowVisible, gxm_XmMultiListMakeRowVisible, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListSelectRow, gxm_XmMultiListSelectRow, 3, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListUnselectItem, gxm_XmMultiListUnselectItem, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListDeselectItem, gxm_XmMultiListDeselectItem, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListDeselectItems, gxm_XmMultiListDeselectItems, 3, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListSelectItems, gxm_XmMultiListSelectItems, 4, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMultiListGetSelectedRows, gxm_XmMultiListGetSelectedRows, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateTabStack
  XM_DEFINE_PROCEDURE(XmTabStackGetSelectedTab, gxm_XmTabStackGetSelectedTab, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmTabStackSelectTab, gxm_XmTabStackSelectTab, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmCreateTabStack, gxm_XmCreateTabStack, 3, 1, 0, H_XmCreateTabStack);
#if HAVE_XmTabStackXYToWidget
  XM_DEFINE_PROCEDURE(XmTabStackIndexToWidget, gxm_XmTabStackIndexToWidget, 2, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmTabStackXYToWidget, gxm_XmTabStackXYToWidget, 3, 0, 0, NULL);
#endif
#endif
#if HAVE_XmCreateColumn
  XM_DEFINE_PROCEDURE(XmCreateColumn, gxm_XmCreateColumn, 3, 1, 0, H_XmCreateColumn);
#if HAVE_XmColumnGetChildLabel
  XM_DEFINE_PROCEDURE(XmColumnGetChildLabel, gxm_XmColumnGetChildLabel, 1, 0, 0, NULL);
#endif
#endif
#if HAVE_XmCreateDropDown
  XM_DEFINE_PROCEDURE(XmCreateDropDown, gxm_XmCreateDropDown, 3, 1, 0, H_XmCreateDropDown);
  XM_DEFINE_PROCEDURE(XmDropDownGetValue, gxm_XmDropDownGetValue, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDropDownGetList, gxm_XmDropDownGetList, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDropDownGetText, gxm_XmDropDownGetText, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDropDownGetArrow, gxm_XmDropDownGetArrow, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDropDownGetLabel, gxm_XmDropDownGetLabel, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateFontSelector
  XM_DEFINE_PROCEDURE(XmCreateFontSelector, gxm_XmCreateFontSelector, 3, 1, 0, H_XmCreateFontSelector);
#endif
#if HAVE_XmCreateColorSelector
  XM_DEFINE_PROCEDURE(XmCreateColorSelector, gxm_XmCreateColorSelector, 3, 1, 0, H_XmCreateColorSelector);
#endif
  XM_DEFINE_PROCEDURE(XmListAddItem, gxm_XmListAddItem, 3, 0, 0, H_XmListAddItem);
  XM_DEFINE_PROCEDURE(XmListAddItems, gxm_XmListAddItems, 4, 0, 0, H_XmListAddItems);
  XM_DEFINE_PROCEDURE(XmListAddItemsUnselected, gxm_XmListAddItemsUnselected, 4, 0, 0, H_XmListAddItemsUnselected);
  XM_DEFINE_PROCEDURE(XmListAddItemUnselected, gxm_XmListAddItemUnselected, 3, 0, 0, H_XmListAddItemUnselected);
  XM_DEFINE_PROCEDURE(XmListDeleteItem, gxm_XmListDeleteItem, 2, 0, 0, H_XmListDeleteItem);
  XM_DEFINE_PROCEDURE(XmListDeleteItems, gxm_XmListDeleteItems, 2, 1, 0, H_XmListDeleteItems);
  XM_DEFINE_PROCEDURE(XmListDeletePositions, gxm_XmListDeletePositions, 2, 1, 0, H_XmListDeletePositions);
  XM_DEFINE_PROCEDURE(XmListDeletePos, gxm_XmListDeletePos, 2, 0, 0, H_XmListDeletePos);
  XM_DEFINE_PROCEDURE(XmListDeleteItemsPos, gxm_XmListDeleteItemsPos, 3, 0, 0, H_XmListDeleteItemsPos);
  XM_DEFINE_PROCEDURE(XmListDeleteAllItems, gxm_XmListDeleteAllItems, 1, 0, 0, H_XmListDeleteAllItems);
  XM_DEFINE_PROCEDURE(XmListReplaceItems, gxm_XmListReplaceItems, 4, 0, 0, H_XmListReplaceItems);
  XM_DEFINE_PROCEDURE(XmListReplaceItemsPos, gxm_XmListReplaceItemsPos, 4, 0, 0, H_XmListReplaceItemsPos);
  XM_DEFINE_PROCEDURE(XmListReplaceItemsUnselected, gxm_XmListReplaceItemsUnselected, 4, 0, 0, H_XmListReplaceItemsUnselected);
  XM_DEFINE_PROCEDURE(XmListReplaceItemsPosUnselected, gxm_XmListReplaceItemsPosUnselected, 4, 0, 0, H_XmListReplaceItemsPosUnselected);
  XM_DEFINE_PROCEDURE(XmListReplacePositions, gxm_XmListReplacePositions, 3, 1, 0, H_XmListReplacePositions);
  XM_DEFINE_PROCEDURE(XmListSelectItem, gxm_XmListSelectItem, 3, 0, 0, H_XmListSelectItem);
  XM_DEFINE_PROCEDURE(XmListSelectPos, gxm_XmListSelectPos, 3, 0, 0, H_XmListSelectPos);
  XM_DEFINE_PROCEDURE(XmListDeselectItem, gxm_XmListDeselectItem, 2, 0, 0, H_XmListDeselectItem);
  XM_DEFINE_PROCEDURE(XmListDeselectPos, gxm_XmListDeselectPos, 2, 0, 0, H_XmListDeselectPos);
  XM_DEFINE_PROCEDURE(XmListDeselectAllItems, gxm_XmListDeselectAllItems, 1, 0, 0, H_XmListDeselectAllItems);
  XM_DEFINE_PROCEDURE(XmListSetPos, gxm_XmListSetPos, 2, 0, 0, H_XmListSetPos);
  XM_DEFINE_PROCEDURE(XmListSetBottomPos, gxm_XmListSetBottomPos, 2, 0, 0, H_XmListSetBottomPos);
  XM_DEFINE_PROCEDURE(XmListSetItem, gxm_XmListSetItem, 2, 0, 0, H_XmListSetItem);
  XM_DEFINE_PROCEDURE(XmListSetBottomItem, gxm_XmListSetBottomItem, 2, 0, 0, H_XmListSetBottomItem);
  XM_DEFINE_PROCEDURE(XmListSetAddMode, gxm_XmListSetAddMode, 2, 0, 0, H_XmListSetAddMode);
  XM_DEFINE_PROCEDURE(XmListItemExists, gxm_XmListItemExists, 2, 0, 0, H_XmListItemExists);
  XM_DEFINE_PROCEDURE(XmListItemPos, gxm_XmListItemPos, 2, 0, 0, H_XmListItemPos);
  XM_DEFINE_PROCEDURE(XmListGetKbdItemPos, gxm_XmListGetKbdItemPos, 1, 0, 0, H_XmListGetKbdItemPos);
  XM_DEFINE_PROCEDURE(XmListSetKbdItemPos, gxm_XmListSetKbdItemPos, 2, 0, 0, H_XmListSetKbdItemPos);
  XM_DEFINE_PROCEDURE(XmListYToPos, gxm_XmListYToPos, 2, 0, 0, H_XmListYToPos);
  XM_DEFINE_PROCEDURE(XmListPosToBounds, gxm_XmListPosToBounds, 2, 0, 0, H_XmListPosToBounds);
  XM_DEFINE_PROCEDURE(XmListGetMatchPos, gxm_XmListGetMatchPos, 2, 0, 0, H_XmListGetMatchPos);
  XM_DEFINE_PROCEDURE(XmListSetHorizPos, gxm_XmListSetHorizPos, 2, 0, 0, H_XmListSetHorizPos);
  XM_DEFINE_PROCEDURE(XmListUpdateSelectedList, gxm_XmListUpdateSelectedList, 1, 0, 0, H_XmListUpdateSelectedList);
  XM_DEFINE_PROCEDURE(XmListPosSelected, gxm_XmListPosSelected, 2, 0, 0, H_XmListPosSelected);
  XM_DEFINE_PROCEDURE(XmCreateList, gxm_XmCreateList, 3, 1, 0, H_XmCreateList);
  XM_DEFINE_PROCEDURE(XmCreateScrolledList, gxm_XmCreateScrolledList, 3, 1, 0, H_XmCreateScrolledList);
  XM_DEFINE_PROCEDURE(XmTranslateKey, gxm_XmTranslateKey, 3, 0, 0, H_XmTranslateKey);
  XM_DEFINE_PROCEDURE(XmCreateMainWindow, gxm_XmCreateMainWindow, 3, 1, 0, H_XmCreateMainWindow);
  XM_DEFINE_PROCEDURE(XmInstallImage, gxm_XmInstallImage, 2, 0, 0, H_XmInstallImage);
  XM_DEFINE_PROCEDURE(XmUninstallImage, gxm_XmUninstallImage, 1, 0, 0, H_XmUninstallImage);
  XM_DEFINE_PROCEDURE(XmGetPixmap, gxm_XmGetPixmap, 4, 0, 0, H_XmGetPixmap);
  XM_DEFINE_PROCEDURE(XmGetPixmapByDepth, gxm_XmGetPixmapByDepth, 5, 0, 0, H_XmGetPixmapByDepth);
  XM_DEFINE_PROCEDURE(XmDestroyPixmap, gxm_XmDestroyPixmap, 2, 0, 0, H_XmDestroyPixmap);
  XM_DEFINE_PROCEDURE(XmUpdateDisplay, gxm_XmUpdateDisplay, 1, 0, 0, H_XmUpdateDisplay);
  XM_DEFINE_PROCEDURE(XmWidgetGetBaselines, gxm_XmWidgetGetBaselines, 1, 0, 0, H_XmWidgetGetBaselines);
  XM_DEFINE_PROCEDURE(XmWidgetGetDisplayRect, gxm_XmWidgetGetDisplayRect, 1, 0, 0, H_XmWidgetGetDisplayRect);
  XM_DEFINE_PROCEDURE(XmRegisterSegmentEncoding, gxm_XmRegisterSegmentEncoding, 2, 0, 0, H_XmRegisterSegmentEncoding);
  XM_DEFINE_PROCEDURE(XmMapSegmentEncoding, gxm_XmMapSegmentEncoding, 1, 0, 0, H_XmMapSegmentEncoding);
  XM_DEFINE_PROCEDURE(XmCvtCTToXmString, gxm_XmCvtCTToXmString, 1, 0, 0, H_XmCvtCTToXmString);
  XM_DEFINE_PROCEDURE(XmCvtXmStringToCT, gxm_XmCvtXmStringToCT, 1, 0, 0, H_XmCvtXmStringToCT);
  XM_DEFINE_PROCEDURE(XmConvertUnits, gxm_XmConvertUnits, 5, 0, 0, H_XmConvertUnits);
  XM_DEFINE_PROCEDURE(XmCreateSimpleMenuBar, gxm_XmCreateSimpleMenuBar, 3, 1, 0, H_XmCreateSimpleMenuBar);
  XM_DEFINE_PROCEDURE(XmCreateSimplePopupMenu, gxm_XmCreateSimplePopupMenu, 3, 1, 0, H_XmCreateSimplePopupMenu);
  XM_DEFINE_PROCEDURE(XmCreateSimplePulldownMenu, gxm_XmCreateSimplePulldownMenu, 3, 1, 0, H_XmCreateSimplePulldownMenu);
  XM_DEFINE_PROCEDURE(XmCreateSimpleOptionMenu, gxm_XmCreateSimpleOptionMenu, 3, 1, 0, H_XmCreateSimpleOptionMenu);
  XM_DEFINE_PROCEDURE(XmCreateSimpleRadioBox, gxm_XmCreateSimpleRadioBox, 3, 1, 0, H_XmCreateSimpleRadioBox);
  XM_DEFINE_PROCEDURE(XmCreateSimpleCheckBox, gxm_XmCreateSimpleCheckBox, 3, 1, 0, H_XmCreateSimpleCheckBox);
  XM_DEFINE_PROCEDURE(XmVaCreateSimpleMenuBar, gxm_XmVaCreateSimpleMenuBar, 3, 0, 0, H_XmVaCreateSimpleMenuBar);
  XM_DEFINE_PROCEDURE(XmVaCreateSimplePopupMenu, gxm_XmVaCreateSimplePopupMenu, 4, 0, 0, H_XmVaCreateSimplePopupMenu);
  XM_DEFINE_PROCEDURE(XmVaCreateSimplePulldownMenu, gxm_XmVaCreateSimplePulldownMenu, 5, 0, 0, H_XmVaCreateSimplePulldownMenu);
  XM_DEFINE_PROCEDURE(XmVaCreateSimpleOptionMenu, gxm_XmVaCreateSimpleOptionMenu, 7, 0, 0, H_XmVaCreateSimpleOptionMenu);
  XM_DEFINE_PROCEDURE(XmVaCreateSimpleRadioBox, gxm_XmVaCreateSimpleRadioBox, 5, 0, 0, H_XmVaCreateSimpleRadioBox);
  XM_DEFINE_PROCEDURE(XmVaCreateSimpleCheckBox, gxm_XmVaCreateSimpleCheckBox, 4, 0, 0, H_XmVaCreateSimpleCheckBox);
  XM_DEFINE_PROCEDURE(XmTrackingEvent, gxm_XmTrackingEvent, 3, 0, 0, H_XmTrackingEvent);
  XM_DEFINE_PROCEDURE(XmSetColorCalculation, gxm_XmSetColorCalculation, 1, 0, 0, H_XmSetColorCalculation);
  XM_DEFINE_PROCEDURE(XmGetColorCalculation, gxm_XmGetColorCalculation, 0, 0, 0, H_XmGetColorCalculation);
  XM_DEFINE_PROCEDURE(XmGetColors, gxm_XmGetColors, 3, 0, 0, H_XmGetColors);
  XM_DEFINE_PROCEDURE(XmChangeColor, gxm_XmChangeColor, 2, 0, 0, H_XmChangeColor);
  XM_DEFINE_PROCEDURE(XmStringCreate, gxm_XmStringCreate, 2, 0, 0, H_XmStringCreate);
  XM_DEFINE_PROCEDURE(XmStringCreateLocalized, gxm_XmStringCreateLocalized, 1, 0, 0, H_XmStringCreateLocalized);
  XM_DEFINE_PROCEDURE(XmStringDirectionCreate, gxm_XmStringDirectionCreate, 1, 0, 0, H_XmStringDirectionCreate);
  XM_DEFINE_PROCEDURE(XmStringSeparatorCreate, gxm_XmStringSeparatorCreate, 0, 0, 0, H_XmStringSeparatorCreate);
  XM_DEFINE_PROCEDURE(XmStringInitContext, gxm_XmStringInitContext, 1, 0, 0, H_XmStringInitContext);
  XM_DEFINE_PROCEDURE(XmStringFreeContext, gxm_XmStringFreeContext, 1, 0, 0, H_XmStringFreeContext);
#if MOTIF_2
  XM_DEFINE_PROCEDURE(XmStringConcatAndFree, gxm_XmStringConcatAndFree, 2, 0, 0, H_XmStringConcatAndFree);
  XM_DEFINE_PROCEDURE(XmStringIsVoid, gxm_XmStringIsVoid, 1, 0, 0, H_XmStringIsVoid);
  XM_DEFINE_PROCEDURE(XmStringPeekNextTriple, gxm_XmStringPeekNextTriple, 1, 0, 0, H_XmStringPeekNextTriple);
  XM_DEFINE_PROCEDURE(XmStringGetNextTriple, gxm_XmStringGetNextTriple, 1, 0, 0, H_XmStringGetNextTriple);
  XM_DEFINE_PROCEDURE(XmStringComponentCreate, gxm_XmStringComponentCreate, 3, 0, 0, H_XmStringComponentCreate);
  XM_DEFINE_PROCEDURE(XmStringUnparse, gxm_XmStringUnparse, 7, 0, 0, H_XmStringUnparse);
  XM_DEFINE_PROCEDURE(XmStringParseText, gxm_XmStringParseText, 7, 0, 0, H_XmStringParseText);
  XM_DEFINE_PROCEDURE(XmStringToXmStringTable, gxm_XmStringToXmStringTable, 2, 0, 0, H_XmStringToXmStringTable);
  XM_DEFINE_PROCEDURE(XmStringTableToXmString, gxm_XmStringTableToXmString, 3, 0, 0, H_XmStringTableToXmString);
  XM_DEFINE_PROCEDURE(XmStringTableUnparse, gxm_XmStringTableUnparse, 8, 0, 0, H_XmStringTableUnparse);
  XM_DEFINE_PROCEDURE(XmStringTableParseStringArray, gxm_XmStringTableParseStringArray, 7, 0, 0, H_XmStringTableParseStringArray);
  XM_DEFINE_PROCEDURE(XmDirectionToStringDirection, gxm_XmDirectionToStringDirection, 1, 0, 0, H_XmDirectionToStringDirection);
  XM_DEFINE_PROCEDURE(XmStringDirectionToDirection, gxm_XmStringDirectionToDirection, 1, 0, 0, H_XmStringDirectionToDirection);
  XM_DEFINE_PROCEDURE(XmStringGenerate, gxm_XmStringGenerate, 4, 0, 0, H_XmStringGenerate);
  XM_DEFINE_PROCEDURE(XmStringPutRendition, gxm_XmStringPutRendition, 2, 0, 0, H_XmStringPutRendition);
  XM_DEFINE_PROCEDURE(XmParseMappingCreate, gxm_XmParseMappingCreate, 1, 1, 0, H_XmParseMappingCreate);
  XM_DEFINE_PROCEDURE(XmParseMappingSetValues, gxm_XmParseMappingSetValues, 2, 1, 0, H_XmParseMappingSetValues);
  XM_DEFINE_PROCEDURE(XmParseMappingGetValues, gxm_XmParseMappingGetValues, 2, 1, 0, H_XmParseMappingGetValues);
  XM_DEFINE_PROCEDURE(XmParseMappingFree, gxm_XmParseMappingFree, 1, 0, 0, H_XmParseMappingFree);
  XM_DEFINE_PROCEDURE(XmParseTableFree, gxm_XmParseTableFree, 1, 1, 0, H_XmParseTableFree);
  XM_DEFINE_PROCEDURE(XmStringTableProposeTablist, gxm_XmStringTableProposeTablist, 5, 0, 0, H_XmStringTableProposeTablist);
  XM_DEFINE_PROCEDURE(XmTabSetValue, gxm_XmTabSetValue, 2, 0, 0, H_XmTabSetValue);
  XM_DEFINE_PROCEDURE(XmTabGetValues, gxm_XmTabGetValues, 1, 0, 0, H_XmTabGetValues);
  XM_DEFINE_PROCEDURE(XmTabFree, gxm_XmTabFree, 1, 0, 0, H_XmTabFree);
  XM_DEFINE_PROCEDURE(XmTabListFree, gxm_XmTabListFree, 1, 0, 0, H_XmTabListFree);
  XM_DEFINE_PROCEDURE(XmTabCreate, gxm_XmTabCreate, 5, 0, 0, H_XmTabCreate);
  XM_DEFINE_PROCEDURE(XmTabListTabCount, gxm_XmTabListTabCount, 1, 0, 0, H_XmTabListTabCount);
  XM_DEFINE_PROCEDURE(XmTabListRemoveTabs, gxm_XmTabListRemoveTabs, 2, 1, 0, H_XmTabListRemoveTabs);
  XM_DEFINE_PROCEDURE(XmTabListReplacePositions, gxm_XmTabListReplacePositions, 3, 1, 0, H_XmTabListReplacePositions);
  XM_DEFINE_PROCEDURE(XmTabListGetTab, gxm_XmTabListGetTab, 2, 0, 0, H_XmTabListGetTab);
  XM_DEFINE_PROCEDURE(XmTabListCopy, gxm_XmTabListCopy, 3, 0, 0, H_XmTabListCopy);
  XM_DEFINE_PROCEDURE(XmTabListInsertTabs, gxm_XmTabListInsertTabs, 4, 0, 0, H_XmTabListInsertTabs);
  XM_DEFINE_PROCEDURE(XmRenderTableCvtFromProp, gxm_XmRenderTableCvtFromProp, 3, 0, 0, H_XmRenderTableCvtFromProp);
  XM_DEFINE_PROCEDURE(XmRenderTableCvtToProp, gxm_XmRenderTableCvtToProp, 2, 0, 0, H_XmRenderTableCvtToProp);
  XM_DEFINE_PROCEDURE(XmRenditionUpdate, gxm_XmRenditionUpdate, 2, 1, 0, H_XmRenditionUpdate);
  XM_DEFINE_PROCEDURE(XmRenditionRetrieve, gxm_XmRenditionRetrieve, 2, 1, 0, H_XmRenditionRetrieve);
  XM_DEFINE_PROCEDURE(XmRenditionFree, gxm_XmRenditionFree, 1, 0, 0, H_XmRenditionFree);
  XM_DEFINE_PROCEDURE(XmRenditionCreate, gxm_XmRenditionCreate, 3, 1, 0, H_XmRenditionCreate);
  XM_DEFINE_PROCEDURE(XmRenderTableGetRenditions, gxm_XmRenderTableGetRenditions, 0, 3, 0, H_XmRenderTableGetRenditions);
  XM_DEFINE_PROCEDURE(XmRenderTableGetRendition, gxm_XmRenderTableGetRendition, 2, 0, 0, H_XmRenderTableGetRendition);
  XM_DEFINE_PROCEDURE(XmRenderTableGetTags, gxm_XmRenderTableGetTags, 1, 0, 0, H_XmRenderTableGetTags);
  XM_DEFINE_PROCEDURE(XmRenderTableFree, gxm_XmRenderTableFree, 1, 0, 0, H_XmRenderTableFree);
  XM_DEFINE_PROCEDURE(XmRenderTableCopy, gxm_XmRenderTableCopy, 0, 3, 0, H_XmRenderTableCopy);
  XM_DEFINE_PROCEDURE(XmRenderTableRemoveRenditions, gxm_XmRenderTableRemoveRenditions, 0, 3, 0, H_XmRenderTableRemoveRenditions);
  XM_DEFINE_PROCEDURE(XmRenderTableAddRenditions, gxm_XmRenderTableAddRenditions, 4, 0, 0, H_XmRenderTableAddRenditions);
#endif
  XM_DEFINE_PROCEDURE(XmStringConcat, gxm_XmStringConcat, 2, 0, 0, H_XmStringConcat);
  XM_DEFINE_PROCEDURE(XmStringCopy, gxm_XmStringCopy, 1, 0, 0, H_XmStringCopy);
  XM_DEFINE_PROCEDURE(XmStringCompare, gxm_XmStringCompare, 2, 0, 0, H_XmStringCompare);
  XM_DEFINE_PROCEDURE(XmStringEmpty, gxm_XmStringEmpty, 1, 0, 0, H_XmStringEmpty);
  XM_DEFINE_PROCEDURE(XmStringHasSubstring, gxm_XmStringHasSubstring, 2, 0, 0, H_XmStringHasSubstring);
  XM_DEFINE_PROCEDURE(XmStringFree, gxm_XmStringFree, 1, 0, 0, H_XmStringFree);
  XM_DEFINE_PROCEDURE(XmStringBaseline, gxm_XmStringBaseline, 2, 0, 0, H_XmStringBaseline);
  XM_DEFINE_PROCEDURE(XmStringWidth, gxm_XmStringWidth, 2, 0, 0, H_XmStringWidth);
  XM_DEFINE_PROCEDURE(XmStringHeight, gxm_XmStringHeight, 2, 0, 0, H_XmStringHeight);
  XM_DEFINE_PROCEDURE(XmStringExtent, gxm_XmStringExtent, 2, 0, 0, H_XmStringExtent);
  XM_DEFINE_PROCEDURE(XmStringLineCount, gxm_XmStringLineCount, 1, 0, 0, H_XmStringLineCount);
  XM_DEFINE_PROCEDURE(XmStringDraw, gxm_XmStringDraw, 0, 0, 1, H_XmStringDraw);
  XM_DEFINE_PROCEDURE(XmStringDrawImage, gxm_XmStringDrawImage, 0, 0, 1, H_XmStringDrawImage);
  XM_DEFINE_PROCEDURE(XmStringDrawUnderline, gxm_XmStringDrawUnderline, 0, 0, 1, H_XmStringDrawUnderline);
  XM_DEFINE_PROCEDURE(XmGetDestination, gxm_XmGetDestination, 1, 0, 0, H_XmGetDestination);
  XM_DEFINE_PROCEDURE(XmIsTraversable, gxm_XmIsTraversable, 1, 0, 0, H_XmIsTraversable);
  XM_DEFINE_PROCEDURE(XmGetVisibility, gxm_XmGetVisibility, 1, 0, 0, H_XmGetVisibility);
  XM_DEFINE_PROCEDURE(XmGetTabGroup, gxm_XmGetTabGroup, 1, 0, 0, H_XmGetTabGroup);
  XM_DEFINE_PROCEDURE(XmGetFocusWidget, gxm_XmGetFocusWidget, 1, 0, 0, H_XmGetFocusWidget);
  XM_DEFINE_PROCEDURE(XmProcessTraversal, gxm_XmProcessTraversal, 2, 0, 0, H_XmProcessTraversal);
  XM_DEFINE_PROCEDURE(XmCreateMenuShell, gxm_XmCreateMenuShell, 3, 1, 0, H_XmCreateMenuShell);

  XM_DEFINE_PROCEDURE(XmIsMessageBox, gxm_XmIsMessageBox, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsArrowButtonGadget, gxm_XmIsArrowButtonGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsArrowButton, gxm_XmIsArrowButton, 1, 0, 0, NULL);
#if MOTIF_2
  XM_DEFINE_PROCEDURE(XmCvtXmStringToByteStream, gxm_XmCvtXmStringToByteStream, 1, 0, 0, H_XmCvtXmStringToByteStream);
  XM_DEFINE_PROCEDURE(XmCvtByteStreamToXmString, gxm_XmCvtByteStreamToXmString, 1, 0, 0, H_XmCvtByteStreamToXmString);
  XM_DEFINE_PROCEDURE(XmStringByteStreamLength, gxm_XmStringByteStreamLength, 1, 0, 0, H_XmStringByteStreamLength);

  XM_DEFINE_PROCEDURE(XmIsNotebook, gxm_XmIsNotebook, 1, 0, 0, NULL);
#if HAVE_XM_XP
  XM_DEFINE_PROCEDURE(XmIsPrintShell, gxm_XmIsPrintShell, 1, 0, 0, NULL);
#endif
  XM_DEFINE_PROCEDURE(XmIsComboBox, gxm_XmIsComboBox, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsContainer, gxm_XmIsContainer, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsGrabShell, gxm_XmIsGrabShell, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsIconGadget, gxm_XmIsIconGadget, 1, 0, 0, NULL);
#ifndef LESSTIF_VERSION
  XM_DEFINE_PROCEDURE(XmIsIconHeader, gxm_XmIsIconHeader, 1, 0, 0, NULL);
#endif
#endif
  XM_DEFINE_PROCEDURE(XmIsPanedWindow, gxm_XmIsPanedWindow, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsBulletinBoard, gxm_XmIsBulletinBoard, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsPrimitive, gxm_XmIsPrimitive, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsCascadeButtonGadget, gxm_XmIsCascadeButtonGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsCascadeButton, gxm_XmIsCascadeButton, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsPushButtonGadget, gxm_XmIsPushButtonGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsPushButton, gxm_XmIsPushButton, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsCommand, gxm_XmIsCommand, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsRowColumn, gxm_XmIsRowColumn, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsScale, gxm_XmIsScale, 1, 0, 0, NULL);
#if MOTIF_2
  XM_DEFINE_PROCEDURE(XmIsScreen, gxm_XmIsScreen, 1, 0, 0, NULL);
#endif
  XM_DEFINE_PROCEDURE(XmIsScrollBar, gxm_XmIsScrollBar, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsDialogShell, gxm_XmIsDialogShell, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsScrolledWindow, gxm_XmIsScrolledWindow, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsDisplay, gxm_XmIsDisplay, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsSelectionBox, gxm_XmIsSelectionBox, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsDragContext, gxm_XmIsDragContext, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsSeparatorGadget, gxm_XmIsSeparatorGadget, 1, 0, 0, NULL);
#if MOTIF_2
#ifndef LESSTIF_VERSION
  XM_DEFINE_PROCEDURE(XmIsDragIconObjectClass, gxm_XmIsDragIconObjectClass, 1, 0, 0, NULL);
#endif
#endif
  XM_DEFINE_PROCEDURE(XmIsSeparator, gxm_XmIsSeparator, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsDrawingArea, gxm_XmIsDrawingArea, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsDrawnButton, gxm_XmIsDrawnButton, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsDropSiteManager, gxm_XmIsDropSiteManager, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsDropTransfer, gxm_XmIsDropTransfer, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsTextField, gxm_XmIsTextField, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsFileSelectionBox, gxm_XmIsFileSelectionBox, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsText, gxm_XmIsText, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsForm, gxm_XmIsForm, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsFrame, gxm_XmIsFrame, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsGadget, gxm_XmIsGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsToggleButtonGadget, gxm_XmIsToggleButtonGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsToggleButton, gxm_XmIsToggleButton, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsLabelGadget, gxm_XmIsLabelGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsLabel, gxm_XmIsLabel, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsVendorShell, gxm_XmIsVendorShell, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsList, gxm_XmIsList, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsMainWindow, gxm_XmIsMainWindow, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsManager, gxm_XmIsManager, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIsMenuShell, gxm_XmIsMenuShell, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmListGetSelectedPos, gxm_XmListGetSelectedPos, 1, 0, 0, H_XmListGetSelectedPos);
#if HAVE_XmCreateButtonBox
  XM_DEFINE_PROCEDURE(XmIsButtonBox, gxm_XmIsButtonBox, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateMultiList
  XM_DEFINE_PROCEDURE(XmIsMultiList, gxm_XmIsMultiList, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateTabStack
  XM_DEFINE_PROCEDURE(XmIsTabStack, gxm_XmIsTabStack, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateDataField
  XM_DEFINE_PROCEDURE(XmIsDataField, gxm_XmIsDataField, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateDropDown
  XM_DEFINE_PROCEDURE(XmIsDropDown, gxm_XmIsDropDown, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateColumn
  XM_DEFINE_PROCEDURE(XmIsColumn, gxm_XmIsColumn, 1, 0, 0, NULL);
#endif

#if (!XM_DISABLE_DEPRECATED)
  XM_DEFINE_PROCEDURE(XtWarning, gxm_XtWarning, 1, 0, 0, H_XtWarning);
  XM_DEFINE_PROCEDURE(XtAppWarning, gxm_XtAppWarning, 2, 0, 0, H_XtAppWarning);
  XM_DEFINE_PROCEDURE(XtSetWarningMsgHandler, gxm_XtSetWarningMsgHandler, 1, 0, 0, H_XtSetWarningMsgHandler);
  XM_DEFINE_PROCEDURE(XtSetWarningHandler, gxm_XtSetWarningHandler, 1, 0, 0, H_XtSetWarningHandler);
  XM_DEFINE_PROCEDURE(XtWarningMsg, gxm_XtWarningMsg, 6, 0, 0, H_XtWarningMsg);
  XM_DEFINE_PROCEDURE(XtErrorMsg, gxm_XtErrorMsg, 6, 0, 0, H_XtErrorMsg);
  XM_DEFINE_PROCEDURE(XtError, gxm_XtError, 1, 0, 0, H_XtError);
  XM_DEFINE_PROCEDURE(XtSetErrorMsgHandler, gxm_XtSetErrorMsgHandler, 1, 0, 0, H_XtSetErrorMsgHandler);
  XM_DEFINE_PROCEDURE(XtSetErrorHandler, gxm_XtSetErrorHandler, 1, 0, 0, H_XtSetErrorHandler);
  XM_DEFINE_PROCEDURE(XtInitialize, gxm_XtInitialize, 4, 0, 0, H_XtInitialize);
  XM_DEFINE_PROCEDURE(XtCreateApplicationShell, gxm_XtCreateApplicationShell, 3, 1, 0, H_XtCreateApplicationShell);
  XM_DEFINE_PROCEDURE(XtSetSelectionTimeout, gxm_XtSetSelectionTimeout, 1, 0, 0, H_XtSetSelectionTimeout);
  XM_DEFINE_PROCEDURE(XtGetSelectionTimeout, gxm_XtGetSelectionTimeout, 0, 0, 0, H_XtGetSelectionTimeout);
  XM_DEFINE_PROCEDURE(XtAddActions, gxm_XtAddActions, 1, 0, 0, H_XtAddActions);
  XM_DEFINE_PROCEDURE(XtAddInput, gxm_XtAddInput, 3, 1, 0, H_XtAddInput);
  XM_DEFINE_PROCEDURE(XtAddWorkProc, gxm_XtAddWorkProc, 1, 1, 0, H_XtAddWorkProc);
  XM_DEFINE_PROCEDURE(XtAddTimeOut, gxm_XtAddTimeOut, 2, 1, 0, H_XtAddTimeOut);
  XM_DEFINE_PROCEDURE(XtMainLoop, gxm_XtMainLoop, 0, 0, 0, H_XtMainLoop);
  XM_DEFINE_PROCEDURE(XtProcessEvent, gxm_XtProcessEvent, 1, 0, 0, H_XtProcessEvent);
  XM_DEFINE_PROCEDURE(XtPending, gxm_XtPending, 0, 0, 0, H_XtPending);
  XM_DEFINE_PROCEDURE(XtNextEvent, gxm_XtNextEvent, 0, 0, 0, H_XtNextEvent);
  XM_DEFINE_PROCEDURE(XtPeekEvent, gxm_XtPeekEvent, 0, 0, 0, H_XtPeekEvent);

  XM_DEFINE_PROCEDURE(XmFontListEntryCreate, gxm_XmFontListEntryCreate, 3, 0, 0, H_XmFontListEntryCreate);
  XM_DEFINE_PROCEDURE(XmFontListEntryFree, gxm_XmFontListEntryFree, 1, 0, 0, H_XmFontListEntryFree);
  XM_DEFINE_PROCEDURE(XmFontListEntryGetFont, gxm_XmFontListEntryGetFont, 1, 0, 0, H_XmFontListEntryGetFont);
  XM_DEFINE_PROCEDURE(XmFontListEntryGetTag, gxm_XmFontListEntryGetTag, 1, 0, 0, H_XmFontListEntryGetTag);
  XM_DEFINE_PROCEDURE(XmFontListAppendEntry, gxm_XmFontListAppendEntry, 2, 0, 0, H_XmFontListAppendEntry);
  XM_DEFINE_PROCEDURE(XmFontListNextEntry, gxm_XmFontListNextEntry, 1, 0, 0, H_XmFontListNextEntry);
  XM_DEFINE_PROCEDURE(XmFontListRemoveEntry, gxm_XmFontListRemoveEntry, 2, 0, 0, H_XmFontListRemoveEntry);
  XM_DEFINE_PROCEDURE(XmFontListEntryLoad, gxm_XmFontListEntryLoad, 4, 0, 0, H_XmFontListEntryLoad);
  XM_DEFINE_PROCEDURE(XmFontListFree, gxm_XmFontListFree, 1, 0, 0, H_XmFontListFree);
  XM_DEFINE_PROCEDURE(XmFontListCopy, gxm_XmFontListCopy, 1, 0, 0, H_XmFontListCopy);
  XM_DEFINE_PROCEDURE(XmFontListInitFontContext, gxm_XmFontListInitFontContext, 1, 0, 0, H_XmFontListInitFontContext);
  XM_DEFINE_PROCEDURE(XmFontListFreeFontContext, gxm_XmFontListFreeFontContext, 1, 0, 0, H_XmFontListFreeFontContext);

  XM_DEFINE_PROCEDURE(XmFontContext?, XEN_XmFontContext_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmFontListEntry?, XEN_XmFontListEntry_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmFontList?, XEN_XmFontList_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmFontListCreate, gxm_XmFontListCreate, 2, 0, 0, H_XmFontListCreate);
  XM_DEFINE_PROCEDURE(XmTrackingLocate, gxm_XmTrackingLocate, 3, 0, 0, H_XmTrackingLocate);
  XM_DEFINE_PROCEDURE(XmFontListGetNextFont, gxm_XmFontListGetNextFont, 1, 0, 0, H_XmFontListGetNextFont);
  XM_DEFINE_PROCEDURE(XmStringByteCompare, gxm_XmStringByteCompare, 2, 0, 0, H_XmStringByteCompare);
  XM_DEFINE_PROCEDURE(XmStringCreateLtoR, gxm_XmStringCreateLtoR, 2, 0, 0, H_XmStringCreateLtoR);
  XM_DEFINE_PROCEDURE(XmStringCreateSimple, gxm_XmStringCreateSimple, 1, 0, 0, H_XmStringCreateSimple);
  XM_DEFINE_PROCEDURE(XmStringGetLtoR, gxm_XmStringGetLtoR, 2, 0, 0, H_XmStringGetLtoR);
  XM_DEFINE_PROCEDURE(XmStringGetNextSegment, gxm_XmStringGetNextSegment, 1, 0, 0, H_XmStringGetNextSegment);
  XM_DEFINE_PROCEDURE(XmStringSegmentCreate, gxm_XmStringSegmentCreate, 4, 0, 0, H_XmStringSegmentCreate);
  XM_DEFINE_PROCEDURE(XmStringPeekNextComponent, gxm_XmStringPeekNextComponent, 1, 0, 0, H_XmStringPeekNextComponent);
  XM_DEFINE_PROCEDURE(XmStringGetNextComponent, gxm_XmStringGetNextComponent, 1, 0, 0, H_XmStringGetNextComponent);
  XM_DEFINE_PROCEDURE(XmFontListAdd, gxm_XmFontListAdd, 3, 0, 0, H_XmFontListAdd);
  XM_DEFINE_PROCEDURE(XmStringLength, gxm_XmStringLength, 1, 0, 0, H_XmStringLength);
  XM_DEFINE_PROCEDURE(XmStringNConcat, gxm_XmStringNConcat, 3, 0, 0, H_XmStringNConcat);
  XM_DEFINE_PROCEDURE(XmStringNCopy, gxm_XmStringNCopy, 2, 0, 0, H_XmStringNCopy);
  XM_DEFINE_PROCEDURE(XmScrolledWindowSetAreas, gxm_XmScrolledWindowSetAreas, 4, 0, 0, H_XmScrolledWindowSetAreas);
  XM_DEFINE_PROCEDURE(XmSetFontUnits, gxm_XmSetFontUnits, 3, 0, 0, H_XmSetFontUnits);
  XM_DEFINE_PROCEDURE(XmSetFontUnit, gxm_XmSetFontUnit, 2, 0, 0, H_XmSetFontUnit);
  XM_DEFINE_PROCEDURE(XmGetMenuCursor, gxm_XmGetMenuCursor, 1, 0, 0, H_XmGetMenuCursor);
  XM_DEFINE_PROCEDURE(XmSetMenuCursor, gxm_XmSetMenuCursor, 2, 0, 0, H_XmSetMenuCursor);
  XM_DEFINE_PROCEDURE(XmMainWindowSetAreas, gxm_XmMainWindowSetAreas, 6, 0, 0, H_XmMainWindowSetAreas);
  XM_DEFINE_PROCEDURE(XmMainWindowSep1, gxm_XmMainWindowSep1, 1, 0, 0, H_XmMainWindowSep1);
  XM_DEFINE_PROCEDURE(XmMainWindowSep2, gxm_XmMainWindowSep2, 1, 0, 0, H_XmMainWindowSep2);
  XM_DEFINE_PROCEDURE(XmMainWindowSep3, gxm_XmMainWindowSep3, 1, 0, 0, H_XmMainWindowSep3);
#endif

#endif

#if HAVE_XPM
  XM_DEFINE_PROCEDURE(XpmCreatePixmapFromData, gxm_XpmCreatePixmapFromData, 4, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XpmCreateDataFromPixmap, gxm_XpmCreateDataFromPixmap, 4, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XpmReadFileToPixmap, gxm_XpmReadFileToPixmap, 4, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XpmReadFileToXpmImage, gxm_XpmReadFileToXpmImage, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XpmReadPixmapFile, gxm_XpmReadFileToPixmap, 4, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XpmWriteFileFromPixmap, gxm_XpmWriteFileFromPixmap, 5, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XpmWritePixmapFile, gxm_XpmWriteFileFromPixmap, 5, 0, 0, NULL);
#if HAVE_XPM_GET_ERROR_STRING
  XM_DEFINE_PROCEDURE(XpmGetErrorString, gxm_XpmGetErrorString, 1, 0, 0, H_XpmGetErrorString);
#endif
#if HAVE_XPM_CREATE_XPM_IMAGE_FROM_PIXMAP
  XM_DEFINE_PROCEDURE(XpmCreatePixmapFromXpmImage, gxm_XpmCreatePixmapFromXpmImage, 4, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XpmCreateXpmImageFromPixmap, gxm_XpmCreateXpmImageFromPixmap, 4, 0, 0, NULL);
#endif
#endif
  XM_DEFINE_PROCEDURE(XGetPixel, gxm_XGetPixel, 3, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XDestroyImage, gxm_XDestroyImage, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XPutPixel, gxm_XPutPixel, 4, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XSubImage, gxm_XSubImage, 5, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XAddPixel, gxm_XAddPixel, 2, 0, 0, NULL);

#if HAVE_MOTIF
  XM_DEFINE_PROCEDURE(XtAppContext?, XEN_XtAppContext_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XtRequestId?, XEN_XtRequestId_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XtWorkProcId?, XEN_XtWorkProcId_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XtInputId?, XEN_XtInputId_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XtIntervalId?, XEN_XtIntervalId_p, 1, 0, 0, NULL);
#endif
  XM_DEFINE_PROCEDURE(Screen?, XEN_Screen_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XEvent?, XEN_XEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XRectangle?, XEN_XRectangle_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XArc?, XEN_XArc_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XPoint?, XEN_XPoint_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XSegment?, XEN_XSegment_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XColor?, XEN_XColor_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Atom?, XEN_Atom_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Colormap?, XEN_Colormap_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XModifierKeymap?, XEN_XModifierKeymap_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Depth?, XEN_Depth_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Display?, XEN_Display_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Drawable?, XEN_Window_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Font?, XEN_Font_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(GC?, XEN_GC_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(KeySym?, XEN_KeySym_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Pixel?, XEN_Pixel_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Pixmap?, XEN_Pixmap_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Region?, XEN_Region_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Time?, XEN_Time_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Visual?, XEN_Visual_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Window?, XEN_Window_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XFontProp?, XEN_XFontProp_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XFontSet?, XEN_XFontSet_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XFontStruct?, XEN_XFontStruct_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XGCValues?, XEN_XGCValues_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XImage?, XEN_XImage_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XVisualInfo?, XEN_XVisualInfo_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XWMHints?, XEN_XWMHints_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XWindowAttributes?, XEN_XWindowAttributes_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XWindowChanges?, XEN_XWindowChanges_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(KeyCode?, XEN_KeyCode_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XContext?, XEN_XContext_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XCharStruct?, XEN_XCharStruct_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XTextItem?, XEN_XTextItem_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XStandardColormap?, XEN_XStandardColormap_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(Cursor?, XEN_Cursor_p, 1, 0, 0, NULL);
#if HAVE_XM_XP
  XM_DEFINE_PROCEDURE(XPContext?, XEN_XPContext_p, 1, 0, 0, NULL);
#endif
#if HAVE_MOTIF
  XM_DEFINE_PROCEDURE(Widget?, XEN_Widget_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmStringContext?, XEN_XmStringContext_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(WidgetClass?, XEN_WidgetClass_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmString?, XEN_XmString_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmToggleButton?, gxm_XmIsToggleButton, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDrawingArea?, gxm_XmIsDrawingArea, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmPushButton?, gxm_XmIsPushButton, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmTextField?, gxm_XmIsTextField, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmFileSelectionBox?, gxm_XmIsFileSelectionBox, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmText?, gxm_XmIsText, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmFrame?, gxm_XmIsFrame, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmLabel?, gxm_XmIsLabel, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmList?, gxm_XmIsList, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmArrowButton?, gxm_XmIsArrowButton, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmScrollBar?, gxm_XmIsScrollBar, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmCommand?, gxm_XmIsCommand, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmScale?, gxm_XmIsScale, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmRowColumn?, gxm_XmIsRowColumn, 1, 0, 0, NULL);
#if MOTIF_2
  XM_DEFINE_PROCEDURE(XmTab?, XEN_XmTab_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmNotebook?, gxm_XmIsNotebook, 1, 0, 0, NULL);
#if HAVE_XM_XP
  XM_DEFINE_PROCEDURE(XmPrintShell?, gxm_XmIsPrintShell, 1, 0, 0, NULL);
#endif
  XM_DEFINE_PROCEDURE(XmComboBox?, gxm_XmIsComboBox, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmContainer?, gxm_XmIsContainer, 1, 0, 0, NULL);
#ifndef LESSTIF_VERSION
  XM_DEFINE_PROCEDURE(XmIconHeader?, gxm_XmIsIconHeader, 1, 0, 0, NULL);
#endif
  XM_DEFINE_PROCEDURE(XmGrabShell?, gxm_XmIsGrabShell, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmRendition?, XEN_XmRendition_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmRenderTable?, XEN_XmRenderTable_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmIconGadget?, gxm_XmIsIconGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmTabList?, XEN_XmTabList_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmParseMapping?, XEN_XmParseMapping_p, 1, 0, 0, NULL);
#endif
  XM_DEFINE_PROCEDURE(XmPanedWindow?, gxm_XmIsPanedWindow, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmScrolledWindow?, gxm_XmIsScrolledWindow, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmCascadeButton?, gxm_XmIsCascadeButton, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmForm?, gxm_XmIsForm, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmBulletinBoard?, gxm_XmIsBulletinBoard, 1, 0, 0, NULL);
#if MOTIF_2
  XM_DEFINE_PROCEDURE(XmScreen?, gxm_XmIsScreen, 1, 0, 0, NULL);
#endif
  XM_DEFINE_PROCEDURE(XmDialogShell?, gxm_XmIsDialogShell, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDisplay?, gxm_XmIsDisplay, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmSelectionBox?, gxm_XmIsSelectionBox, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDragContext?, gxm_XmIsDragContext, 1, 0, 0, NULL);
#if MOTIF_2
#ifndef LESSTIF_VERSION
  XM_DEFINE_PROCEDURE(XmDragIconObjectClass?, gxm_XmIsDragIconObjectClass, 1, 0, 0, NULL);
#endif
#endif
  XM_DEFINE_PROCEDURE(XmSeparator?, gxm_XmIsSeparator, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDropSiteManager?, gxm_XmIsDropSiteManager, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDropTransfer?, gxm_XmIsDropTransfer, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmVendorShell?, gxm_XmIsVendorShell, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMainWindow?, gxm_XmIsMainWindow, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMessageBox?, gxm_XmIsMessageBox, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmManager?, gxm_XmIsManager, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmMenuShell?, gxm_XmIsMenuShell, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmLabelGadget?, gxm_XmIsLabelGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmPushButtonGadget?, gxm_XmIsPushButtonGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmSeparatorGadget?, gxm_XmIsSeparatorGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmArrowButtonGadget?, gxm_XmIsArrowButtonGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmCascadeButtonGadget?, gxm_XmIsCascadeButtonGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmToggleButtonGadget?, gxm_XmIsToggleButtonGadget, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmDrawnButton?, gxm_XmIsDrawnButton, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmPrimitive?, gxm_XmIsPrimitive, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XmTextSource?, XEN_XmTextSource_p, 1, 0, 0, NULL);
#if HAVE_XmCreateButtonBox
  XM_DEFINE_PROCEDURE(XmButtonBox?, gxm_XmIsButtonBox, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateMultiList
  XM_DEFINE_PROCEDURE(XmMultiList?, gxm_XmIsMultiList, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateTabStack
  XM_DEFINE_PROCEDURE(XmTabStack?, gxm_XmIsTabStack, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateDataField
  XM_DEFINE_PROCEDURE(XmDataField?, gxm_XmIsDataField, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateDropDown
  XM_DEFINE_PROCEDURE(XmDropDown?, gxm_XmIsDropDown, 1, 0, 0, NULL);
#endif
#if HAVE_XmCreateColumn
  XM_DEFINE_PROCEDURE(XmColumn?, gxm_XmIsColumn, 1, 0, 0, NULL);
#endif
#endif
#if HAVE_XPM
  XM_DEFINE_PROCEDURE(XpmAttributes?, XEN_XpmAttributes_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XpmImage?, XEN_XpmImage_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XpmColorSymbol?, XEN_XpmColorSymbol_p, 1, 0, 0, NULL);
#endif

  XM_DEFINE_PROCEDURE(XButtonEvent?, XEN_XButtonEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XCirculateEvent?, XEN_XCirculateEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XCirculateRequestEvent?, XEN_XCirculateRequestEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XClientMessageEvent?, XEN_XClientMessageEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XColormapEvent?, XEN_XColormapEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XConfigureEvent?, XEN_XConfigureEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XConfigureRequestEvent?, XEN_XConfigureRequestEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XCreateWindowEvent?, XEN_XCreateWindowEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XCrossingEvent?, XEN_XCrossingEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XDestroyWindowEvent?, XEN_XDestroyWindowEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XErrorEvent?, XEN_XErrorEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XExposeEvent?, XEN_XExposeEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XFocusChangeEvent?, XEN_XFocusChangeEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XGraphicsExposeEvent?, XEN_XGraphicsExposeEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XGravityEvent?, XEN_XGravityEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XKeyEvent?, XEN_XKeyEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XKeymapEvent?, XEN_XKeymapEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XMapEvent?, XEN_XMapEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XMapRequestEvent?, XEN_XMapRequestEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XMappingEvent?, XEN_XMappingEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XMotionEvent?, XEN_XMotionEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XNoExposeEvent?, XEN_XNoExposeEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XPropertyEvent?, XEN_XPropertyEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XReparentEvent?, XEN_XReparentEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XResizeRequestEvent?, XEN_XResizeRequestEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XSelectionClearEvent?, XEN_XSelectionClearEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XSelectionEvent?, XEN_XSelectionEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XSelectionRequestEvent?, XEN_XSelectionRequestEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XSetWindowAttributes?, XEN_XSetWindowAttributes_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XUnmapEvent?, XEN_XUnmapEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XVisibilityEvent?, XEN_XVisibilityEvent_p, 1, 0, 0, NULL);
  XM_DEFINE_PROCEDURE(XIconSize?, XEN_XIconSize_p, 1, 0, 0, NULL);

  XEN_DEFINE_PROCEDURE("->string", c_to_xen_string, 1, 0, 0, NULL);
  XEN_DEFINE_PROCEDURE("->strings", c_to_xen_strings, 2, 0, 0, NULL);
  XEN_DEFINE_PROCEDURE("->ints", c_to_xen_ints, 2, 0, 0, NULL);
  XEN_DEFINE_PROCEDURE("->Atoms", c_to_xen_atoms, 2, 0, 0, NULL);
  XEN_DEFINE_PROCEDURE("->XRectangles", c_to_xen_xrectangles, 2, 0, 0, NULL);

#if WITH_EDITRES
  XEN_DEFINE_PROCEDURE("_XEditResCheckMessages", gxm_XEditResCheckMessages, 4, 0, 0, NULL);
#endif

#if HAVE_XSHAPEQUERYEXTENSION
  XM_DEFINE_PROCEDURE(XShapeQueryExtension, gxm_XShapeQueryExtension, 1, 0, 0, H_XShapeQueryExtension);
  XM_DEFINE_PROCEDURE(XShapeQueryVersion, gxm_XShapeQueryVersion, 1, 0, 0, H_XShapeQueryVersion);
  XM_DEFINE_PROCEDURE(XShapeQueryExtents, gxm_XShapeQueryExtents, 2, 0, 0, H_XShapeQueryExtents);
  XM_DEFINE_PROCEDURE(XShapeGetRectangles, gxm_XShapeGetRectangles, 3, 0, 0, H_XShapeGetRectangles);
  XM_DEFINE_PROCEDURE(XShapeOffsetShape, gxm_XShapeOffsetShape, 5, 0, 0, H_XShapeOffsetShape);
  XM_DEFINE_PROCEDURE(XShapeCombineRegion, gxm_XShapeCombineRegion, 7, 0, 0, H_XShapeCombineRegion);
  XM_DEFINE_PROCEDURE(XShapeCombineMask, gxm_XShapeCombineMask, 7, 0, 0, H_XShapeCombineMask);
  XM_DEFINE_PROCEDURE(XShapeCombineShape, gxm_XShapeCombineShape, 8, 0, 0, H_XShapeCombineShape);
  XM_DEFINE_PROCEDURE(XShapeCombineRectangles, gxm_XShapeCombineRectangles, 9, 0, 0, H_XShapeCombineRectangles);
#endif

}
#endif



/* -------------------------------- struct handlers -------------------------------- */

/* XRectangle */

static XEN gxm_XRectangle(XEN x, XEN y, XEN width, XEN height)
{
  #define H_XRectangle "(XRectangle x y width height) returns the given XRectangle"
  XRectangle *r;
  XEN_ASSERT_TYPE(XEN_INTEGER_P(x), x, 1, "XRectangle", "short");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(y), y, 2, "XRectangle", "short");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(width), width, 3, "XRectangle", "INT");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(height), height, 4, "XRectangle", "INT");
  r = (XRectangle *)CALLOC(1, sizeof(XRectangle));
  r->x = (short)XEN_TO_C_INT(x);
  r->y = (short)XEN_TO_C_INT(y);
  r->width = XEN_TO_C_INT(width);
  r->height = XEN_TO_C_INT(height);
  return(C_TO_XEN_XRectangle(r));
}

/* XSegment */

static XEN gxm_XSegment(XEN x1, XEN y1, XEN x2, XEN y2)
{
  XSegment *r;
  XEN_ASSERT_TYPE(XEN_INTEGER_P(x1), x1, 1, "XSegment", "short");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(y1), y1, 2, "XSegment", "short");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(x2), x2, 3, "XSegment", "short");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(y2), y2, 4, "XSegment", "short");
  r = (XSegment *)CALLOC(1, sizeof(XSegment));
  r->x1 = (short)XEN_TO_C_INT(x1);
  r->y1 = (short)XEN_TO_C_INT(y1);
  r->x2 = (short)XEN_TO_C_INT(x2);
  r->y2 = (short)XEN_TO_C_INT(y2);
  return(C_TO_XEN_XSegment(r));
}

static XEN gxm_y2(XEN ptr)
{
  if (XEN_XSegment_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XSegment(ptr))->y2)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "y2", "XSegment");
  return(XEN_FALSE);
}

static XEN gxm_set_y2(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_1, "y2", "an integer");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XSegment_P(ptr), ptr, XEN_ARG_2, "y2", "XSegment");
  (XEN_TO_C_XSegment(ptr))->y2 = (short)XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_x2(XEN ptr)
{
  if (XEN_XSegment_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XSegment(ptr))->x2)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "x2", "XSegment");
  return(XEN_FALSE);
}

static XEN gxm_set_x2(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_1, "x2", "an integer");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XSegment_P(ptr), ptr, XEN_ARG_2, "x2", "XSegment");
  (XEN_TO_C_XSegment(ptr))->x2 = (short)XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_y1(XEN ptr)
{
  if (XEN_XSegment_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XSegment(ptr))->y1)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "y1", "XSegment");
  return(XEN_FALSE);
}

static XEN gxm_set_y1(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_1, "y1", "an integer");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XSegment_P(ptr), ptr, XEN_ARG_2, "y1", "XSegment");
  (XEN_TO_C_XSegment(ptr))->y1 = (short)XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_x1(XEN ptr)
{
  if (XEN_XSegment_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XSegment(ptr))->x1)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "x1", "XSegment");
  return(XEN_FALSE);
}

static XEN gxm_set_x1(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_1, "x1", "an integer");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XSegment_P(ptr), ptr, XEN_ARG_2, "x1", "XSegment");
  (XEN_TO_C_XSegment(ptr))->x1 = (short)XEN_TO_C_INT(val);
  return(val);
}

/* XPoint */

static XEN gxm_XPoint(XEN x, XEN y)
{
  XPoint *r;
  XEN_ASSERT_TYPE(XEN_INTEGER_P(x), x, 1, "XPoint", "short");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(y), y, 2, "XPoint", "short");
  r = (XPoint *)CALLOC(1, sizeof(XPoint));
  r->x = (short)XEN_TO_C_INT(x);
  r->y = (short)XEN_TO_C_INT(y);
  return(C_TO_XEN_XPoint(r));
}

/* XArc */

static XEN gxm_XArc(XEN x, XEN y, XEN width, XEN height, XEN angle1, XEN angle2)
{
  XArc *r;
  XEN_ASSERT_TYPE(XEN_INTEGER_P(x), x, 1, "XArc", "short");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(y), y, 2, "XArc", "short");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(width), width, 3, "XArc", "INT");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(height), height, 4, "XArc", "INT");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(angle1), angle1, 5, "XArc", "short");
  XEN_ASSERT_TYPE(XEN_INTEGER_P(angle2), angle2, 6, "XArc", "short");
  r = (XArc *)CALLOC(1, sizeof(XArc));
  r->x = (short)XEN_TO_C_INT(x);
  r->y = (short)XEN_TO_C_INT(y);
  r->width = XEN_TO_C_INT(width);
  r->height = XEN_TO_C_INT(height);
  r->angle1 = (short)XEN_TO_C_INT(angle1);
  r->angle2 = (short)XEN_TO_C_INT(angle2);
  return(C_TO_XEN_XArc(r));
}

static XEN gxm_angle2(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XArc_P(ptr), ptr, XEN_ONLY_ARG, "angle2", "XArc");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XArc(ptr))->angle2)));
}

static XEN gxm_set_angle2(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_1, "angle2", "an integer");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XArc_P(ptr), ptr, XEN_ARG_2, "angle2", "XArc");
  (XEN_TO_C_XArc(ptr))->angle2 = (short)XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_angle1(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XArc_P(ptr), ptr, XEN_ONLY_ARG, "angle1", "XArc");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XArc(ptr))->angle1)));
}

static XEN gxm_set_angle1(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_1, "angle1", "an integer");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XArc_P(ptr), ptr, XEN_ARG_2, "angle1", "XArc");
  (XEN_TO_C_XArc(ptr))->angle1 = (short)XEN_TO_C_INT(val);
  return(val);
}

/* XColor */

static XEN gxm_XColor(XEN pixel, XEN red, XEN green, XEN blue, XEN flags, XEN pad)
{
  XColor *r;
  XEN_ASSERT_TYPE(XEN_ULONG_IF_BOUND_P(pixel), pixel, 1, "XColor", "ulong");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(red), red, 2, "XColor", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(green), green, 3, "XColor", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(blue), blue, 4, "XColor", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(flags), flags, 5, "XColor", "char");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(pad), pad, 6, "XColor", "char");
  r = (XColor *)CALLOC(1, sizeof(XColor));
  if (XEN_BOUND_P(pixel)) r->pixel = XEN_TO_C_ULONG(pixel);
  if (XEN_BOUND_P(red)) r->red = XEN_TO_C_INT(red);
  if (XEN_BOUND_P(green)) r->green = XEN_TO_C_INT(green);
  if (XEN_BOUND_P(blue)) r->blue = XEN_TO_C_INT(blue);
  if (XEN_BOUND_P(flags)) r->flags = (char)XEN_TO_C_INT(flags);
  if (XEN_BOUND_P(pad)) r->pad = (char)XEN_TO_C_INT(pad);
  return(C_TO_XEN_XColor(r));
}

static XEN gxm_pad(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XColor_P(ptr), ptr, XEN_ONLY_ARG, "pad", "XColor");
  return(C_TO_XEN_char((char)((XEN_TO_C_XColor(ptr))->pad)));
}

static XEN gxm_set_pad(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XColor_P(ptr), ptr, XEN_ARG_1, "pad", "XColor");
  XM_SET_FIELD_ASSERT_TYPE(XEN_char_P(val), val, XEN_ARG_2, "pad", "char");
  (XEN_TO_C_XColor(ptr))->pad = XEN_TO_C_char(val);
  return(val);
}

static XEN gxm_blue(XEN ptr)
{
  if (XEN_XColor_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XColor(ptr))->blue)));
#if WITH_GTK_AND_X11
  return(gxg_blue(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "blue", "XColor");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_blue(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "blue", "an integer");
  if (XEN_XColor_P(ptr)) (XEN_TO_C_XColor(ptr))->blue = XEN_TO_C_INT(val);
#if WITH_GTK_AND_X11
  else return(gxg_set_blue(ptr, val));
#else
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "blue", "XColor");
#endif
  return(val);
}

static XEN gxm_green(XEN ptr)
{
  if (XEN_XColor_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XColor(ptr))->green)));
#if WITH_GTK_AND_X11
  return(gxg_green(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "green", "XColor");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_green(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "green", "an integer");
  if (XEN_XColor_P(ptr)) (XEN_TO_C_XColor(ptr))->green = XEN_TO_C_INT(val);
#if WITH_GTK_AND_X11
  else return(gxg_set_green(ptr, val));
#else
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "green", "XColor");
#endif
  return(val);
}

static XEN gxm_red(XEN ptr)
{
  if (XEN_XColor_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XColor(ptr))->red)));
#if WITH_GTK_AND_X11
  return(gxg_red(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "red", "XColor");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_red(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "red", "an integer");
  if (XEN_XColor_P(ptr)) (XEN_TO_C_XColor(ptr))->red = XEN_TO_C_INT(val);
#if WITH_GTK_AND_X11
  else return(gxg_set_red(ptr, val));
#else
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "red", "XColor");
#endif
  return(val);
}

static XEN gxm_XWindowChanges(XEN x, XEN y, XEN width, XEN height, XEN border_width, XEN sibling, XEN stack_mode)
{
  XWindowChanges *r;
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(x), x, 1, "XWindowChanges", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(y), y, 2, "XWindowChanges", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(width), width, 3, "XWindowChanges", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(height), height, 4, "XWindowChanges", "int");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(border_width), border_width, 5, "XWindowChanges", "int");
  XEN_ASSERT_TYPE(XEN_Window_P(sibling) || XEN_NOT_BOUND_P(sibling), sibling, 6, "XWindowChanges", "Window");
  XEN_ASSERT_TYPE(XEN_INTEGER_IF_BOUND_P(stack_mode), stack_mode, 7, "XWindowChanges", "int");
  r = (XWindowChanges *)CALLOC(1, sizeof(XWindowChanges));
  if (XEN_BOUND_P(x)) r->x = XEN_TO_C_INT(x);
  if (XEN_BOUND_P(y)) r->y = XEN_TO_C_INT(y);
  if (XEN_BOUND_P(width)) r->width = XEN_TO_C_INT(width);
  if (XEN_BOUND_P(height)) r->height = XEN_TO_C_INT(height);
  if (XEN_BOUND_P(border_width)) r->border_width = XEN_TO_C_INT(border_width);
  if (XEN_BOUND_P(sibling)) r->sibling = XEN_TO_C_Window(sibling);
  if (XEN_BOUND_P(stack_mode)) r->stack_mode = XEN_TO_C_INT(stack_mode);
  return(C_TO_XEN_XWindowChanges(r));
}

static XEN gxm_XSetWindowAttributes(XEN arglist)
{
  int len;
  XSetWindowAttributes *r;
  r = (XSetWindowAttributes *)CALLOC(1, sizeof(XSetWindowAttributes));
  len = XEN_LIST_LENGTH(arglist);
  if ((len > 0) && (XEN_Pixmap_P(XEN_LIST_REF(arglist, 0)))) r->background_pixmap = XEN_TO_C_Pixmap(XEN_LIST_REF(arglist, 0));
  if ((len > 1) && (XEN_Pixel_P(XEN_LIST_REF(arglist, 1)))) r->background_pixel = XEN_TO_C_Pixel(XEN_LIST_REF(arglist, 1));
  if ((len > 2) && (XEN_Pixmap_P(XEN_LIST_REF(arglist, 2)))) r->border_pixmap = XEN_TO_C_Pixmap(XEN_LIST_REF(arglist, 2));
  if ((len > 3) && (XEN_Pixel_P(XEN_LIST_REF(arglist, 3)))) r->border_pixel = XEN_TO_C_Pixel(XEN_LIST_REF(arglist, 3));
  if ((len > 4) && (XEN_INTEGER_P(XEN_LIST_REF(arglist, 4)))) r->bit_gravity = XEN_TO_C_INT(XEN_LIST_REF(arglist, 4));
  if ((len > 5) && (XEN_INTEGER_P(XEN_LIST_REF(arglist, 5)))) r->win_gravity = XEN_TO_C_INT(XEN_LIST_REF(arglist, 5));
  if ((len > 6) && (XEN_INTEGER_P(XEN_LIST_REF(arglist, 6)))) r->backing_store = XEN_TO_C_INT(XEN_LIST_REF(arglist, 6));
  if ((len > 7) && (XEN_ULONG_P(XEN_LIST_REF(arglist, 7)))) r->backing_planes = XEN_TO_C_ULONG(XEN_LIST_REF(arglist, 7));
  if ((len > 8) && (XEN_Pixel_P(XEN_LIST_REF(arglist, 8)))) r->backing_pixel = XEN_TO_C_Pixel(XEN_LIST_REF(arglist, 8));
  if ((len > 9) && (XEN_BOOLEAN_P(XEN_LIST_REF(arglist, 9)))) r->save_under = XEN_TO_C_BOOLEAN(XEN_LIST_REF(arglist, 9));
  if ((len > 10) && (XEN_INTEGER_P(XEN_LIST_REF(arglist, 10)))) r->event_mask = XEN_TO_C_INT(XEN_LIST_REF(arglist, 10));
  if ((len > 11) && (XEN_INTEGER_P(XEN_LIST_REF(arglist, 11)))) r->do_not_propagate_mask = XEN_TO_C_INT(XEN_LIST_REF(arglist, 11));
  if ((len > 12) && (XEN_BOOLEAN_P(XEN_LIST_REF(arglist, 12)))) r->override_redirect = XEN_TO_C_BOOLEAN(XEN_LIST_REF(arglist, 12));
  if ((len > 13) && (XEN_Colormap_P(XEN_LIST_REF(arglist, 13)))) r->colormap = XEN_TO_C_Colormap(XEN_LIST_REF(arglist, 13));
  if ((len > 14) && (XEN_Cursor_P(XEN_LIST_REF(arglist, 14)))) r->cursor = XEN_TO_C_Cursor(XEN_LIST_REF(arglist, 14));
  return(C_TO_XEN_XSetWindowAttributes(r));
}

static XEN gxm_stack_mode(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowChanges_P(ptr), ptr, XEN_ONLY_ARG, "stack_mode", "XWindowChanges");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XWindowChanges(ptr))->stack_mode)));
}

static XEN gxm_sibling(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowChanges_P(ptr), ptr, XEN_ONLY_ARG, "sibling", "XWindowChanges");
  return(C_TO_XEN_Window((Window)((XEN_TO_C_XWindowChanges(ptr))->sibling)));
}

static XEN gxm_obdata(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XImage_P(ptr), ptr, XEN_ONLY_ARG, "obdata", "XImage");
  return(C_TO_XEN_ULONG((XPointer)((XEN_TO_C_XImage(ptr))->obdata)));
}

static XEN gxm_bytes_per_line(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XImage_P(ptr), ptr, XEN_ONLY_ARG, "bytes_per_line", "XImage");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XImage(ptr))->bytes_per_line)));
}

static XEN gxm_bitmap_pad(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XImage_P(ptr), ptr, XEN_ONLY_ARG, "bitmap_pad", "XImage");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XImage(ptr))->bitmap_pad)));
}

static XEN gxm_bitmap_bit_order(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XImage_P(ptr), ptr, XEN_ONLY_ARG, "bitmap_bit_order", "XImage");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XImage(ptr))->bitmap_bit_order)));
}

static XEN gxm_bitmap_unit(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XImage_P(ptr), ptr, XEN_ONLY_ARG, "bitmap_unit", "XImage");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XImage(ptr))->bitmap_unit)));
}

static XEN gxm_byte_order(XEN ptr)
{
  if (XEN_XImage_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XImage(ptr))->byte_order)));
#if WITH_GTK_AND_X11
  return(gxg_byte_order(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "byte_order", "XImage");
  return(XEN_FALSE);
#endif
}

static XEN gxm_xoffset(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XImage_P(ptr), ptr, XEN_ONLY_ARG, "xoffset", "XImage");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XImage(ptr))->xoffset)));
}

static XEN gxm_screen(XEN ptr)
{
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_Screen((Screen *)((XEN_TO_C_XWindowAttributes(ptr))->screen)));
#if MOTIF_2
  if (XEN_XmTopLevelEnterCallbackStruct_P(ptr)) return(C_TO_XEN_Screen((Screen *)((XEN_TO_C_XmTopLevelEnterCallbackStruct(ptr))->screen)));
  if (XEN_XmTopLevelLeaveCallbackStruct_P(ptr)) return(C_TO_XEN_Screen((Screen *)((XEN_TO_C_XmTopLevelLeaveCallbackStruct(ptr))->screen)));
#endif
  if (XEN_XVisualInfo_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XVisualInfo(ptr))->screen)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "screen", "a struct with a screen field");
  return(XEN_FALSE);
}

static XEN gxm_your_event_mask(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "your_event_mask", "XWindowAttributes");
  return(C_TO_XEN_INT((long)((XEN_TO_C_XWindowAttributes(ptr))->your_event_mask)));
}

static XEN gxm_all_event_masks(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "all_event_masks", "XWindowAttributes");
  return(C_TO_XEN_INT((long)((XEN_TO_C_XWindowAttributes(ptr))->all_event_masks)));
}

static XEN gxm_map_state(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "map_state", "XWindowAttributes");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XWindowAttributes(ptr))->map_state)));
}

static XEN gxm_map_installed(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "map_installed", "XWindowAttributes");
  return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XWindowAttributes(ptr))->map_installed)));
}

static XEN gxm_set_visual(XEN ptr, XEN val)
{
#if HAVE_XPM
  XM_SET_FIELD_ASSERT_TYPE(XEN_Visual_P(val), val, XEN_ARG_2, "visual", "Visual");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ARG_1, "visual", "Visual");
  (XEN_TO_C_XpmAttributes(ptr))->visual = (Visual *)XEN_TO_C_Visual(val);
#endif
  return(val);
}

static XEN gxm_visual(XEN ptr)
{
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_Visual((Visual *)((XEN_TO_C_XWindowAttributes(ptr))->visual)));
#if HAVE_XPM
  if (XEN_XpmAttributes_P(ptr)) return(C_TO_XEN_Visual((Visual *)((XEN_TO_C_XpmAttributes(ptr))->visual)));
#endif
#if WITH_GTK_AND_X11
  return(gxg_visual(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "visual", "a struct with a visual field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_cursor(XEN ptr)
{
  if (XEN_XSetWindowAttributes_P(ptr)) return(C_TO_XEN_Cursor((Cursor)((XEN_TO_C_XSetWindowAttributes(ptr))->cursor)));
#if WITH_GTK_AND_X11
  return(gxg_cursor(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "cursor", "XSetWindowAttributes");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_cursor(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ARG_1, "cursor", "XSetWindowAttributes");
  XM_SET_FIELD_ASSERT_TYPE(XEN_Cursor_P(val), val, XEN_ARG_2, "cursor", "a Cursor");
  (XEN_TO_C_XSetWindowAttributes(ptr))->cursor = XEN_TO_C_Cursor(val);
  return(val);
}

static XEN gxm_do_not_propagate_mask(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr) || XEN_XSetWindowAttributes_P(ptr), 
		       ptr, XEN_ONLY_ARG, "do_not_propagate_mask", "a struct with a do_not_propagate_mask field");
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_INT((long)((XEN_TO_C_XWindowAttributes(ptr))->do_not_propagate_mask)));
  return(C_TO_XEN_INT((long)((XEN_TO_C_XSetWindowAttributes(ptr))->do_not_propagate_mask)));
}

static XEN gxm_event_mask(XEN ptr)
{
  if (XEN_XSetWindowAttributes_P(ptr)) return(C_TO_XEN_INT((long)((XEN_TO_C_XSetWindowAttributes(ptr))->event_mask)));
#if WITH_GTK_AND_X11
  return(gxg_event_mask(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "event_mask", "XSetWindowAttributes");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_event_mask(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ARG_1, "event_mask", "XSetWindowAttributes");
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "event_mask", "an integer");
  (XEN_TO_C_XSetWindowAttributes(ptr))->event_mask = XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_save_under(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr) || XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "save_under", "a struct with a save_under field");
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XWindowAttributes(ptr))->save_under)));
  return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XSetWindowAttributes(ptr))->save_under)));
}

static XEN gxm_set_save_under(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr) || XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ARG_1, "save_under", "a struct with a save_under field");
  XM_SET_FIELD_ASSERT_TYPE(XEN_BOOLEAN_P(val), val, XEN_ARG_2, "save_under", "a Boolean");
  if (XEN_XWindowAttributes_P(ptr))
      (XEN_TO_C_XWindowAttributes(ptr))->save_under = XEN_TO_C_BOOLEAN(val);
  else
      (XEN_TO_C_XSetWindowAttributes(ptr))->save_under = XEN_TO_C_BOOLEAN(val);
  return(val);
}

static XEN gxm_backing_pixel(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr) || XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "backing_pixel", "a struct with a backing_pixel field");
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_Pixel((unsigned long)((XEN_TO_C_XWindowAttributes(ptr))->backing_pixel)));
  return(C_TO_XEN_Pixel((unsigned long)((XEN_TO_C_XSetWindowAttributes(ptr))->backing_pixel)));
}

static XEN gxm_backing_planes(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr) || XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "backing_planes", "a struct with a backing_planes field");
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XWindowAttributes(ptr))->backing_planes)));
  return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XSetWindowAttributes(ptr))->backing_planes)));
}

static XEN gxm_win_gravity(XEN ptr)
{
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XWindowAttributes(ptr))->win_gravity)));
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XSetWindowAttributes(ptr))->win_gravity)));
#if WITH_GTK_AND_X11
  return(gxg_win_gravity(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "win_gravity", "a struct with a win_gravity field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_bit_gravity(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr) || XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "bit_gravity", "a struct with a bit_gravity field");
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XWindowAttributes(ptr))->bit_gravity)));
  return(C_TO_XEN_INT((int)((XEN_TO_C_XSetWindowAttributes(ptr))->bit_gravity)));
}

static XEN gxm_set_bit_gravity(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XWindowAttributes_P(ptr) || XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ARG_1, "bit_gravity", "a struct with a bit_gravity field");
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "bit_gravity", "an integer");
  if (XEN_XWindowAttributes_P(ptr))
      (XEN_TO_C_XWindowAttributes(ptr))->bit_gravity = XEN_TO_C_INT(val);
  else
      (XEN_TO_C_XSetWindowAttributes(ptr))->bit_gravity = XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_border_pixel(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "border_pixel", "XSetWindowAttributes");
  return(C_TO_XEN_Pixel((unsigned long)((XEN_TO_C_XSetWindowAttributes(ptr))->border_pixel)));
}

static XEN gxm_set_border_pixel(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ARG_1, "border_pixel", "XSetWindowAttributes");
  XM_SET_FIELD_ASSERT_TYPE(XEN_Pixel_P(val), val, XEN_ARG_2, "border_pixel", "a Pixel");
  (XEN_TO_C_XSetWindowAttributes(ptr))->border_pixel = XEN_TO_C_Pixel(val);
  return(val);
}

static XEN gxm_border_pixmap(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "border_pixmap", "XSetWindowAttributes");
  return(C_TO_XEN_Pixmap((Pixmap)((XEN_TO_C_XSetWindowAttributes(ptr))->border_pixmap)));
}

static XEN gxm_background_pixel(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "background_pixel", "XSetWindowAttributes");
  return(C_TO_XEN_Pixel((unsigned long)((XEN_TO_C_XSetWindowAttributes(ptr))->background_pixel)));
}

static XEN gxm_set_background_pixel(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ARG_1, "background_pixel", "XSetWindowAttributes");
  XM_SET_FIELD_ASSERT_TYPE(XEN_Pixel_P(val), val, XEN_ARG_2, "background_pixel", "a Pixel");
  (XEN_TO_C_XSetWindowAttributes(ptr))->background_pixel = XEN_TO_C_Pixel(val);
  return(val);
}

static XEN gxm_background_pixmap(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XSetWindowAttributes_P(ptr), ptr, XEN_ONLY_ARG, "background_pixmap", "XSetWindowAttributes");
  return(C_TO_XEN_Pixmap((Pixmap)((XEN_TO_C_XSetWindowAttributes(ptr))->background_pixmap)));
}

static XEN gxm_bits_per_pixel(XEN ptr)
{
  if (XEN_XImage_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XImage(ptr))->bits_per_pixel)));
#if WITH_GTK_AND_X11
  return(gxg_bits_per_pixel(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "bits_per_pixel", "XImage");
  return(XEN_FALSE);
#endif
}

static XEN gxm_visuals(XEN ptr)
{
  Depth *dps;
  Visual *vs;
  int i, len;
  XEN lst = XEN_EMPTY_LIST;
  XM_FIELD_ASSERT_TYPE(XEN_Depth_P(ptr), ptr, XEN_ONLY_ARG, "visuals", "Depth");
  dps = XEN_TO_C_Depth(ptr);
  len = dps->nvisuals;
  if (len > 0)
    {
      vs = dps->visuals;
      for (i = len - 1; i >= 0; i--)
	lst = XEN_CONS(WRAP_FOR_XEN("Visual", &(vs[i])), lst);
    }
  return(lst);
}

static XEN gxm_nvisuals(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_Depth_P(ptr), ptr, XEN_ONLY_ARG, "nvisuals", "Depth");
  return(C_TO_XEN_INT((int)((XEN_TO_C_Depth(ptr))->nvisuals)));
}

static XEN gxm_set_depth(XEN ptr, XEN val)
{
#if HAVE_XPM
  XM_SET_FIELD_ASSERT_TYPE(XEN_XpmAttributes_P(ptr), ptr, XEN_ARG_1, "depth", "XpmAttributes");
  (XEN_TO_C_XpmAttributes(ptr))->depth = XEN_TO_C_ULONG(val);
#endif
  return(val);
}

static XEN gxm_depth(XEN ptr)
{
  if (XEN_XImage_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XImage(ptr))->depth)));
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XWindowAttributes(ptr))->depth)));
  if (XEN_Depth_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_Depth(ptr))->depth)));
  if (XEN_XVisualInfo_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XVisualInfo(ptr))->depth)));
#if HAVE_XPM
  if (XEN_XpmAttributes_P(ptr)) return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XpmAttributes(ptr))->depth)));
#endif
#if WITH_GTK_AND_X11
  return(gxg_depth(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "depth", "a struct with a depth field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_map_entries(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_Visual_P(ptr), ptr, XEN_ONLY_ARG, "map_entries", "Visual");
  return(C_TO_XEN_INT((int)((XEN_TO_C_Visual(ptr))->map_entries)));
}

static XEN gxm_bits_per_rgb(XEN ptr)
{
  if (XEN_XVisualInfo_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XVisualInfo(ptr))->bits_per_rgb)));
  if (XEN_Visual_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_Visual(ptr))->bits_per_rgb)));
#if WITH_GTK_AND_X11
  return(gxg_bits_per_rgb(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "bits_per_rgb", "Visual or XVisualInfo");
  return(XEN_FALSE);
#endif
}

static XEN gxm_blue_mask(XEN ptr)
{
  if (XEN_XImage_P(ptr)) return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XImage(ptr))->blue_mask)));
  if (XEN_XVisualInfo_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XVisualInfo(ptr))->blue_mask)));
  if (XEN_Visual_P(ptr)) return(C_TO_XEN_INT((long)((XEN_TO_C_Visual(ptr))->blue_mask)));
#if WITH_GTK_AND_X11
  return(gxg_blue_mask(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "blue_mask", "a struct with a blue_mask field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_green_mask(XEN ptr)
{
  if (XEN_XImage_P(ptr)) return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XImage(ptr))->green_mask)));
  if (XEN_XVisualInfo_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XVisualInfo(ptr))->green_mask)));
  if (XEN_Visual_P(ptr)) return(C_TO_XEN_INT((long)((XEN_TO_C_Visual(ptr))->green_mask)));
#if WITH_GTK_AND_X11
  return(gxg_green_mask(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "green_mask", "a struct with a green_mask field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_red_mask(XEN ptr)
{
  if (XEN_XImage_P(ptr)) return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XImage(ptr))->red_mask)));
  if (XEN_XVisualInfo_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XVisualInfo(ptr))->red_mask)));
  if (XEN_Visual_P(ptr)) return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_Visual(ptr))->red_mask)));
#if WITH_GTK_AND_X11
  return(gxg_red_mask(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "red_mask", "a struct with a red_mask field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_colormap_size(XEN ptr)
{
  if (XEN_XVisualInfo_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XVisualInfo(ptr))->colormap_size)));
#if WITH_GTK_AND_X11
  return(gxg_colormap_size(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "colormap_size", "XVisualInfo*");
  return(XEN_FALSE);
#endif
}

static XEN gxm_class(XEN ptr)
{
#ifndef __cplusplus
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XWindowAttributes(ptr))->class)));
  if (XEN_Visual_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_Visual(ptr))->class)));
  if (XEN_XVisualInfo_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XVisualInfo(ptr))->class)));
#endif
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "class", "a struct with a class field");
  return(XEN_FALSE);
}

static XEN gxm_visualid(XEN ptr)
{
  if (XEN_Visual_P(ptr)) return(C_TO_XEN_ULONG((VisualID)((XEN_TO_C_Visual(ptr))->visualid)));
  if (XEN_XStandardColormap_P(ptr)) return(C_TO_XEN_ULONG((VisualID)((XEN_TO_C_XStandardColormap(ptr))->visualid)));
  if (XEN_XVisualInfo_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XVisualInfo(ptr))->visualid)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "visualid", "a struct with a visualid field");
  return(XEN_FALSE);
}

static XEN gxm_set_pixel(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Pixel_P(val), val, XEN_ARG_2, "pixel", "a Pixel");  
  if (XEN_XColor_P(ptr)) (XEN_TO_C_XColor(ptr))->pixel = XEN_TO_C_Pixel(val);
#if HAVE_XPM
  else if (XEN_XpmColorSymbol_P(ptr)) (XEN_TO_C_XpmColorSymbol(ptr))->pixel = XEN_TO_C_Pixel(val);
#endif
#if WITH_GTK_AND_X11
  else return(gxg_set_pixel(ptr, val));
#endif
  return(val);
}

static XEN gxm_pixel(XEN ptr)
{
  if (XEN_XColor_P(ptr)) return(C_TO_XEN_Pixel((Pixel)((XEN_TO_C_XColor(ptr))->pixel)));
#if HAVE_MOTIF
  if (XEN_XmScrollBarCallbackStruct_P(ptr)) return(C_TO_XEN_Pixel((Pixel)((XEN_TO_C_XmScrollBarCallbackStruct(ptr))->pixel)));
#endif
#if HAVE_XPM
  if (XEN_XpmColorSymbol_P(ptr)) return(C_TO_XEN_Pixel((Pixel)((XEN_TO_C_XpmColorSymbol(ptr))->pixel)));
#endif
#if WITH_GTK_AND_X11
  return(gxg_pixel(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "pixel", "a struct with a pixel field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_window_group(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWMHints_P(ptr), ptr, XEN_ONLY_ARG, "window_group", "XWMHints");
  return(C_TO_XEN_ULONG((XID)((XEN_TO_C_XWMHints(ptr))->window_group)));
}

static XEN gxm_icon_mask(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWMHints_P(ptr), ptr, XEN_ONLY_ARG, "icon_mask", "XWMHints");
  return(C_TO_XEN_Pixmap((Pixmap)((XEN_TO_C_XWMHints(ptr))->icon_mask)));
}

static XEN gxm_icon_y(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWMHints_P(ptr), ptr, XEN_ONLY_ARG, "icon_y", "XWMHints");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XWMHints(ptr))->icon_y)));
}

static XEN gxm_icon_x(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWMHints_P(ptr), ptr, XEN_ONLY_ARG, "icon_x", "XWMHints");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XWMHints(ptr))->icon_x)));
}

static XEN gxm_icon_window(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWMHints_P(ptr), ptr, XEN_ONLY_ARG, "icon_window", "XWMHints");
  return(C_TO_XEN_Window((Window)((XEN_TO_C_XWMHints(ptr))->icon_window)));
}

static XEN gxm_icon_pixmap(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWMHints_P(ptr), ptr, XEN_ONLY_ARG, "icon_pixmap", "XWMHints");
  return(C_TO_XEN_Pixmap((Pixmap)((XEN_TO_C_XWMHints(ptr))->icon_pixmap)));
}

static XEN gxm_initial_state(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWMHints_P(ptr), ptr, XEN_ONLY_ARG, "initial_state", "XWMHints");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XWMHints(ptr))->initial_state)));
}

static XEN gxm_set_initial_state(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XWMHints_P(ptr), ptr, XEN_ARG_1, "initial_state", "XWMHints");
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "initial_state", "an integer");
  (XEN_TO_C_XWMHints(ptr))->initial_state = XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_input(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XWMHints_P(ptr), ptr, XEN_ONLY_ARG, "input", "XWMHints");
  return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XWMHints(ptr))->input)));
}

static XEN gxm_set_input(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XWMHints_P(ptr), ptr, XEN_ARG_1, "input", "XWMHints");
  XM_SET_FIELD_ASSERT_TYPE(XEN_BOOLEAN_P(val), val, XEN_ARG_2, "input", "a boolean");
  (XEN_TO_C_XWMHints(ptr))->input = XEN_TO_C_BOOLEAN(val);
  return(val);
}

static XEN gxm_flags(XEN ptr)
{
  if (XEN_XColor_P(ptr)) return(C_TO_XEN_char((char)((XEN_TO_C_XColor(ptr))->flags)));
#if MOTIF_2
  if (XEN_XmSelectionCallbackStruct_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XmSelectionCallbackStruct(ptr))->flags)));
  if (XEN_XmDestinationCallbackStruct_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XmDestinationCallbackStruct(ptr))->flags)));
  if (XEN_XmConvertCallbackStruct_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XmConvertCallbackStruct(ptr))->flags)));
#endif
  if (XEN_XWMHints_P(ptr)) return(C_TO_XEN_INT((long)((XEN_TO_C_XWMHints(ptr))->flags)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "flags", "a struct with a flags field");
  return(XEN_FALSE);
}

static XEN gxm_set_flags(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_XColor_P(ptr) || XEN_XWMHints_P(ptr), ptr, XEN_ARG_1, "flags", "XColor or XWMHints");
  XM_SET_FIELD_ASSERT_TYPE(XEN_char_P(val), val, XEN_ARG_2, "input", "a char (int)");
  if (XEN_XColor_P(ptr)) (XEN_TO_C_XColor(ptr))->flags = XEN_TO_C_char(val);
  else if (XEN_XWMHints_P(ptr)) (XEN_TO_C_XWMHints(ptr))->flags = XEN_TO_C_char(val);
  return(val);
}

static XEN gxm_per_char(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XFontStruct_P(ptr), ptr, XEN_ONLY_ARG, "per_char", "XFontStruct");
  return(C_TO_XEN_XCharStruct((XCharStruct *)((XEN_TO_C_XFontStruct(ptr))->per_char)));
}

static XEN gxm_max_bounds(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XFontStruct_P(ptr), ptr, XEN_ONLY_ARG, "max_bounds", "XFontStruct");
  return(C_TO_XEN_XCharStruct((XCharStruct *)(&(XEN_TO_C_XFontStruct(ptr))->max_bounds)));
}

static XEN gxm_min_bounds(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XFontStruct_P(ptr), ptr, XEN_ONLY_ARG, "min_bounds", "XFontStruct");
  return(C_TO_XEN_XCharStruct((XCharStruct *)(&(XEN_TO_C_XFontStruct(ptr))->min_bounds)));
}

static XEN gxm_min_height(XEN ptr)
{
  if (XEN_XIconSize_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XIconSize(ptr))->min_height)));
#if WITH_GTK_AND_X11
  return(gxg_min_height(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "min_height", "XIconSize");
  return(XEN_FALSE);
#endif
}

static XEN gxm_min_width(XEN ptr)
{
  if (XEN_XIconSize_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XIconSize(ptr))->min_width)));
#if WITH_GTK_AND_X11
  return(gxg_min_width(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "min_width", "XIconSize");
  return(XEN_FALSE);
#endif
}

static XEN gxm_max_height(XEN ptr)
{
  if (XEN_XIconSize_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XIconSize(ptr))->max_height)));
#if WITH_GTK_AND_X11
  return(gxg_max_height(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "max_height", "XIconSize");
  return(XEN_FALSE);
#endif
}

static XEN gxm_max_width(XEN ptr)
{
  if (XEN_XIconSize_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XIconSize(ptr))->max_width)));
#if WITH_GTK_AND_X11
  return(gxg_max_width(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "max_width", "XIconSize");
  return(XEN_FALSE);
#endif
}

static XEN gxm_height_inc(XEN ptr)
{
  if (XEN_XIconSize_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XIconSize(ptr))->height_inc)));
#if WITH_GTK_AND_X11
  return(gxg_height_inc(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "height_inc", "XIconSize");
  return(XEN_FALSE);
#endif
}

static XEN gxm_width_inc(XEN ptr)
{
  if (XEN_XIconSize_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XIconSize(ptr))->width_inc)));
#if WITH_GTK_AND_X11
  return(gxg_width_inc(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "width_inc", "XIconSize");
  return(XEN_FALSE);
#endif
}

static XEN gxm_properties(XEN ptr)
{
  int i, len;
  XFontStruct *fs;
  XFontProp *props;
  XEN lst = XEN_EMPTY_LIST;
  XM_FIELD_ASSERT_TYPE(XEN_XFontStruct_P(ptr), ptr, XEN_ONLY_ARG, "properties", "XFontStruct");
  fs = XEN_TO_C_XFontStruct(ptr);
  len = fs->n_properties;
  props = fs->properties;
  for (i = 0; i < len; i++)
    lst = XEN_CONS(C_TO_XEN_XFontProp(&(props[i])), lst);
  return(lst);
}

static XEN gxm_fid(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XFontStruct_P(ptr), ptr, XEN_ONLY_ARG, "fid", "XFontStruct");
  return(C_TO_XEN_Font((Font)((XEN_TO_C_XFontStruct(ptr))->fid)));
}

static XEN gxm_card32(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XFontProp_P(ptr), ptr, XEN_ONLY_ARG, "card32", "XFontProp");
  return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XFontProp(ptr))->card32)));
}

static XEN gxm_set_name(XEN ptr, XEN val)
{
#if HAVE_XPM
  XM_SET_FIELD_ASSERT_TYPE(XEN_STRING_P(val), val, XEN_ARG_2, "name", "a string");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XpmColorSymbol_P(ptr), ptr, XEN_ARG_1, "name", "XpmColorSymbol");
  (XEN_TO_C_XpmColorSymbol(ptr))->name = strdup(XEN_TO_C_STRING(val));
#endif
  return(val);
}

static XEN gxm_name(XEN ptr)
{
  if (XEN_XFontProp_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XFontProp(ptr))->name)));
#if HAVE_XPM
  if (XEN_XpmColorSymbol_P(ptr)) return(C_TO_XEN_STRING((char *)((XEN_TO_C_XpmColorSymbol(ptr))->name)));
#endif
#if WITH_GTK_AND_X11
  return(gxg_name(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "name", "a struct with a name field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_attributes(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XCharStruct_P(ptr), ptr, XEN_ONLY_ARG, "attributes", "XCharStruct");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XCharStruct(ptr))->attributes)));
}

static XEN gxm_descent(XEN ptr)
{
  if (XEN_XFontStruct_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XFontStruct(ptr))->descent)));
  if (XEN_XCharStruct_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XCharStruct(ptr))->descent)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "descent", "a struct with a descent field");
  return(XEN_FALSE);
}

static XEN gxm_ascent(XEN ptr)
{
  if (XEN_XFontStruct_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XFontStruct(ptr))->ascent)));
  if (XEN_XCharStruct_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XCharStruct(ptr))->ascent)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "ascent", "a struct with an ascent field");
  return(XEN_FALSE);
}

static XEN gxm_rbearing(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XCharStruct_P(ptr), ptr, XEN_ONLY_ARG, "rbearing", "XCharStruct");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XCharStruct(ptr))->rbearing)));
}

static XEN gxm_lbearing(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XCharStruct_P(ptr), ptr, XEN_ONLY_ARG, "lbearing", "XCharStruct");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XCharStruct(ptr))->lbearing)));
}

static XEN gxm_request_code(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XErrorEvent_P(ptr), ptr, XEN_ONLY_ARG, "request_code", "XErrorEvent");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XErrorEvent(ptr))->request_code)));
}

static XEN gxm_set_request_code(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "request_code", "an integer");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XErrorEvent_P(ptr), ptr, XEN_ARG_1, "request_code", "XErrorEvent");
  (XEN_TO_C_XErrorEvent(ptr))->request_code = XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_error_code(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XErrorEvent_P(ptr), ptr, XEN_ONLY_ARG, "error_code", "XErrorEvent");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XErrorEvent(ptr))->error_code)));
}

static XEN gxm_set_error_code(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "error_code", "an integer");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XErrorEvent_P(ptr), ptr, XEN_ARG_1, "error_code", "XErrorEvent");
  (XEN_TO_C_XErrorEvent(ptr))->error_code = XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_resourceid(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XErrorEvent_P(ptr), ptr, XEN_ONLY_ARG, "resourceid", "XErrorEvent");
  return(C_TO_XEN_ULONG((XID)((XEN_TO_C_XErrorEvent(ptr))->resourceid)));
}

static XEN gxm_set_resourceid(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_ULONG_P(val), val, XEN_ARG_2, "resourceid", "an XID");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XErrorEvent_P(ptr), ptr, XEN_ARG_1, "resourceid", "XErrorEvent");
  (XEN_TO_C_XErrorEvent(ptr))->resourceid = (XID)XEN_TO_C_ULONG(val);
  return(val);
}

static XEN gxm_first_keycode(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XMappingEvent_P(ptr), ptr, XEN_ONLY_ARG, "first_keycode", "XMappingEvent");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XMappingEvent(ptr))->first_keycode)));
}

static XEN gxm_set_first_keycode(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "first_keycode", "an integer");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XMappingEvent_P(ptr), ptr, XEN_ARG_1, "first_keycode", "XMappingEvent");
  (XEN_TO_C_XMappingEvent(ptr))->first_keycode = XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_request(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XMappingEvent_P(ptr), ptr, XEN_ONLY_ARG, "request", "XMappingEvent");
  return(C_TO_XEN_INT((int)((XEN_TO_C_XMappingEvent(ptr))->request)));
}

static XEN gxm_set_request(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "request", "an integer");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XMappingEvent_P(ptr), ptr, XEN_ARG_1, "request", "XMappingEvent");
  (XEN_TO_C_XMappingEvent(ptr))->request = XEN_TO_C_INT(val);
  return(val);
}

static XEN gxm_format(XEN ptr)
{
  if (XEN_XImage_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XImage(ptr))->format)));
#if MOTIF_2
  if (XEN_XmSelectionCallbackStruct_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XmSelectionCallbackStruct(ptr))->format)));
  if (XEN_XmConvertCallbackStruct_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XmConvertCallbackStruct(ptr))->format)));
#endif
#if HAVE_MOTIF
  if (XEN_XmTextBlock_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XmTextBlock(ptr))->format)));
#endif
  if (XEN_XClientMessageEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XClientMessageEvent(ptr))->format)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "format", "XClientMessageEvent");
  return(XEN_FALSE);
}

static XEN gxm_set_format(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "format", "an integer");
  if (XEN_XClientMessageEvent_P(ptr)) (XEN_TO_C_XClientMessageEvent(ptr))->format = XEN_TO_C_INT(val);
#if HAVE_MOTIF
  else if (XEN_XmTextBlock_P(ptr)) (XEN_TO_C_XmTextBlock(ptr))->format = XEN_TO_C_INT(val);
#endif
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "format", "XClientMessageEvent or XmTextBlock");
  return(val);
}

static XEN gxm_message_type(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XClientMessageEvent_P(ptr), ptr, XEN_ONLY_ARG, "message_type", "XClientMessageEvent");
  return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XClientMessageEvent(ptr))->message_type)));
}

static XEN gxm_set_message_type(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Atom_P(val), val, XEN_ARG_2, "message_type", "an Atom");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XClientMessageEvent_P(ptr), ptr, XEN_ARG_1, "message_type", "XClientMessageEvent");
  (XEN_TO_C_XClientMessageEvent(ptr))->message_type = XEN_TO_C_Atom(val);
  return(val);
}

static XEN gxm_new(XEN ptr)
{
#ifndef __cplusplus
  XM_FIELD_ASSERT_TYPE(XEN_XColormapEvent_P(ptr), ptr, XEN_ONLY_ARG, "new", "XColormapEvent");
  return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XColormapEvent(ptr))->new)));
#endif
  return(XEN_FALSE);
}

static XEN gxm_set_new(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_BOOLEAN_P(val), val, XEN_ARG_2, "new", "a boolean");
#ifndef __cplusplus
  if (XEN_XColormapEvent_P(ptr)) (XEN_TO_C_XColormapEvent(ptr))->new = XEN_TO_C_BOOLEAN(val);
  else 
#endif
    XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "new", "XColormapEvent");
  return(val);
}

static XEN gxm_colormap(XEN ptr)
{
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_Colormap((Colormap)((XEN_TO_C_XWindowAttributes(ptr))->colormap)));
  if (XEN_XSetWindowAttributes_P(ptr)) return(C_TO_XEN_Colormap((Colormap)((XEN_TO_C_XSetWindowAttributes(ptr))->colormap)));
  if (XEN_XColormapEvent_P(ptr)) return(C_TO_XEN_Colormap((Colormap)((XEN_TO_C_XColormapEvent(ptr))->colormap)));
  if (XEN_XStandardColormap_P(ptr)) return(C_TO_XEN_Colormap((Colormap)((XEN_TO_C_XStandardColormap(ptr))->colormap)));
#if HAVE_XPM
  if (XEN_XpmAttributes_P(ptr)) return(C_TO_XEN_Colormap((Colormap)((XEN_TO_C_XpmAttributes(ptr))->colormap)));
#endif
#if WITH_GTK_AND_X11
  return(gxg_colormap(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "colormap", "a struct with a colormap field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_colormap(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Colormap_P(val), val, XEN_ARG_2, "colormap", "a Colormap");
#if HAVE_XPM
  if (XEN_XpmAttributes_P(ptr)) (XEN_TO_C_XpmAttributes(ptr))->colormap = XEN_TO_C_Colormap(val);
  else 
#endif
    if (XEN_XColormapEvent_P(ptr)) (XEN_TO_C_XColormapEvent(ptr))->colormap = XEN_TO_C_Colormap(val);
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "colormap", "XpmAttributes or XColormapEvent");
  return(val);
}

static XEN gxm_property(XEN ptr)
{
  if (XEN_XSelectionEvent_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XSelectionEvent(ptr))->property)));
  if (XEN_XSelectionRequestEvent_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XSelectionRequestEvent(ptr))->property)));
#if WITH_GTK_AND_X11
  return(gxg_property(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "property", "a struct with a property field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_property(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Atom_P(val), val, XEN_ARG_2, "property", "an Atom");
  if (XEN_XSelectionEvent_P(ptr)) (XEN_TO_C_XSelectionEvent(ptr))->property = XEN_TO_C_Atom(val);
  else if (XEN_XSelectionRequestEvent_P(ptr)) (XEN_TO_C_XSelectionRequestEvent(ptr))->property = XEN_TO_C_Atom(val);
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "property", "XSelection(Request)Event");
  return(val);
}

static XEN gxm_target(XEN ptr)
{
#if MOTIF_2
  if (XEN_XmPopupHandlerCallbackStruct_P(ptr)) return(C_TO_XEN_Widget((Widget)((XEN_TO_C_XmPopupHandlerCallbackStruct(ptr))->target)));
  if (XEN_XmSelectionCallbackStruct_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XmSelectionCallbackStruct(ptr))->target)));
  if (XEN_XmConvertCallbackStruct_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XmConvertCallbackStruct(ptr))->target)));
#endif
  if (XEN_XSelectionEvent_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XSelectionEvent(ptr))->target)));
  if (XEN_XSelectionRequestEvent_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XSelectionRequestEvent(ptr))->target)));
#if WITH_GTK_AND_X11
  return(gxg_target(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "target", "a struct with a target field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_target(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Atom_P(val), val, XEN_ARG_2, "target", "an Atom");
  if (XEN_XSelectionEvent_P(ptr)) (XEN_TO_C_XSelectionEvent(ptr))->target = XEN_TO_C_Atom(val);
  else if (XEN_XSelectionRequestEvent_P(ptr)) (XEN_TO_C_XSelectionRequestEvent(ptr))->target = XEN_TO_C_Atom(val);
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "target", "XSelection(Request)Event");
  return(val);
}

static XEN gxm_requestor(XEN ptr)
{
  if (XEN_XSelectionEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XSelectionEvent(ptr))->requestor)));
  if (XEN_XSelectionRequestEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XSelectionRequestEvent(ptr))->requestor)));
#if WITH_GTK_AND_X11
  return(gxg_requestor(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "requestor", "a struct with a requestor field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_requestor(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Window_P(val), val, XEN_ARG_2, "requestor", "a Window");
  if (XEN_XSelectionEvent_P(ptr)) (XEN_TO_C_XSelectionEvent(ptr))->requestor = XEN_TO_C_Window(val);
  else if (XEN_XSelectionRequestEvent_P(ptr)) (XEN_TO_C_XSelectionRequestEvent(ptr))->requestor = XEN_TO_C_Window(val);
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "requestor", "XSelection(Request)Event");
  return(val);
}

static XEN gxm_owner(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XSelectionRequestEvent_P(ptr), ptr, XEN_ONLY_ARG, "owner", "XSelectionRequestEvent");
  return(C_TO_XEN_Window((Window)((XEN_TO_C_XSelectionRequestEvent(ptr))->owner)));
}

static XEN gxm_set_owner(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Window_P(val), val, XEN_ARG_2, "owner", "a Window");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XSelectionRequestEvent_P(ptr), ptr, XEN_ARG_1, "owner", "XSelectionRequestEvent");
  (XEN_TO_C_XSelectionRequestEvent(ptr))->owner = XEN_TO_C_Window(val);
  return(val);
}

static XEN gxm_selection(XEN ptr)
{
#if MOTIF_2
  if (XEN_XmTransferDoneCallbackStruct_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XmTransferDoneCallbackStruct(ptr))->selection)));
  if (XEN_XmSelectionCallbackStruct_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XmSelectionCallbackStruct(ptr))->selection)));
  if (XEN_XmDestinationCallbackStruct_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XmDestinationCallbackStruct(ptr))->selection)));
  if (XEN_XmConvertCallbackStruct_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XmConvertCallbackStruct(ptr))->selection)));
#endif
  if (XEN_XSelectionEvent_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XSelectionEvent(ptr))->selection)));
  if (XEN_XSelectionRequestEvent_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XSelectionRequestEvent(ptr))->selection)));
  if (XEN_XSelectionClearEvent_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XSelectionClearEvent(ptr))->selection)));
#if WITH_GTK_AND_X11
  return(gxg_selection(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "selection", "a struct with a selection field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_selection(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Atom_P(val), val, XEN_ARG_2, "selection", "an Atom");
  if (XEN_XSelectionEvent_P(ptr)) (XEN_TO_C_XSelectionEvent(ptr))->selection = XEN_TO_C_Atom(val);
  else if (XEN_XSelectionRequestEvent_P(ptr)) (XEN_TO_C_XSelectionRequestEvent(ptr))->selection = XEN_TO_C_Atom(val);
  else if (XEN_XSelectionClearEvent_P(ptr)) (XEN_TO_C_XSelectionClearEvent(ptr))->selection = XEN_TO_C_Atom(val);
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "selection", "XSelection(Request|Clear)Event");
  return(val);
}

static XEN gxm_atom(XEN ptr)
{
  if (XEN_XPropertyEvent_P(ptr)) return(C_TO_XEN_Atom((Atom)((XEN_TO_C_XPropertyEvent(ptr))->atom)));
#if WITH_GTK_AND_X11
  return(gxg_atom(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "atom", "XPropertyEvent");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_atom(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Atom_P(val), val, XEN_ARG_2, "atom", "an Atom");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XPropertyEvent_P(ptr), ptr, XEN_ARG_1, "atom", "XPropertyEvent");
  (XEN_TO_C_XPropertyEvent(ptr))->atom = XEN_TO_C_Atom(val);
  return(val);
}

static XEN gxm_place(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XCirculateRequestEvent_P(ptr) || XEN_XCirculateEvent_P(ptr), ptr, XEN_ONLY_ARG, "place", "a struct with a place field");
  if (XEN_XCirculateRequestEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XCirculateRequestEvent(ptr))->place)));
  return(C_TO_XEN_INT((int)((XEN_TO_C_XCirculateEvent(ptr))->place)));
}

static XEN gxm_set_place(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "place", "an integer");
  if (XEN_XCirculateRequestEvent_P(ptr)) (XEN_TO_C_XCirculateRequestEvent(ptr))->place = XEN_TO_C_INT(val);
  else if (XEN_XCirculateEvent_P(ptr)) (XEN_TO_C_XCirculateEvent(ptr))->place = XEN_TO_C_INT(val);
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "place", "a struct with a place field");
  return(val);
}

static XEN gxm_value_mask(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XConfigureRequestEvent_P(ptr), ptr, XEN_ONLY_ARG, "value_mask", "XConfigureRequestEvent");
  return(C_TO_XEN_ULONG((unsigned long)((XEN_TO_C_XConfigureRequestEvent(ptr))->value_mask)));
}

static XEN gxm_set_value_mask(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_ULONG_P(val), val, XEN_ARG_2, "value_mask", "an unsigned long");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XConfigureRequestEvent_P(ptr), ptr, XEN_ARG_1, "value_mask", "XConfigureRequestEvent");
  (XEN_TO_C_XConfigureRequestEvent(ptr))->value_mask = XEN_TO_C_ULONG(val);
  return(val);
}

static XEN gxm_above(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XConfigureRequestEvent_P(ptr) || XEN_XConfigureEvent_P(ptr), ptr, XEN_ONLY_ARG, "above", "a struct with an above field");
  if (XEN_XConfigureRequestEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XConfigureRequestEvent(ptr))->above)));
  return(C_TO_XEN_Window((Window)((XEN_TO_C_XConfigureEvent(ptr))->above)));
}

static XEN gxm_set_above(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Window_P(val), val, XEN_ARG_2, "above", "a Window");
  if (XEN_XConfigureRequestEvent_P(ptr)) (XEN_TO_C_XConfigureRequestEvent(ptr))->above = XEN_TO_C_Window(val);
  else if (XEN_XConfigureEvent_P(ptr)) (XEN_TO_C_XConfigureEvent(ptr))->above = XEN_TO_C_Window(val);
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "above", "a struct with an above field");
  return(val);
}

static XEN gxm_from_configure(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XUnmapEvent_P(ptr), ptr, XEN_ONLY_ARG, "from_configure", "XUnmapEvent");
  return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XUnmapEvent(ptr))->from_configure)));
}

static XEN gxm_set_from_configure(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_BOOLEAN_P(val), val, XEN_ARG_2, "from_configure", "a boolean");
  XM_SET_FIELD_ASSERT_TYPE(XEN_XUnmapEvent_P(ptr), ptr, XEN_ARG_1, "from_configure", "XUnmapEvent");
  (XEN_TO_C_XUnmapEvent(ptr))->from_configure = XEN_TO_C_BOOLEAN(val);
  return(val);
}


static XEN gxm_event(XEN ptr)
{
  /* Xlib.h says event is a window in these cases -- kinda strange looking */
  if (XEN_XCirculateEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XCirculateEvent(ptr))->event)));
  if (XEN_XGravityEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XGravityEvent(ptr))->event)));
  if (XEN_XConfigureEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XConfigureEvent(ptr))->event)));
  if (XEN_XReparentEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XReparentEvent(ptr))->event)));
  if (XEN_XMapEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XMapEvent(ptr))->event)));
  if (XEN_XUnmapEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XUnmapEvent(ptr))->event)));
  if (XEN_XDestroyWindowEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XDestroyWindowEvent(ptr))->event)));
#if HAVE_MOTIF
  if (XEN_AnyCallbackStruct_P(ptr)) return(C_TO_XEN_XEvent((XEvent *)((XEN_TO_C_XmAnyCallbackStruct(ptr))->event)));
#endif
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "event", "a struct with an event field");
  return(XEN_FALSE);
}

static XEN gxm_set_event(XEN ptr, XEN val)
{
  if (XEN_XCirculateEvent_P(ptr)) (XEN_TO_C_XCirculateEvent(ptr))->event = XEN_TO_C_Window(val); else
  if (XEN_XGravityEvent_P(ptr)) (XEN_TO_C_XGravityEvent(ptr))->event = XEN_TO_C_Window(val); else
  if (XEN_XConfigureEvent_P(ptr)) (XEN_TO_C_XConfigureEvent(ptr))->event = XEN_TO_C_Window(val); else
  if (XEN_XReparentEvent_P(ptr)) (XEN_TO_C_XReparentEvent(ptr))->event = XEN_TO_C_Window(val); else
  if (XEN_XMapEvent_P(ptr)) (XEN_TO_C_XMapEvent(ptr))->event = XEN_TO_C_Window(val); else
  if (XEN_XUnmapEvent_P(ptr)) (XEN_TO_C_XUnmapEvent(ptr))->event = XEN_TO_C_Window(val); else
  if (XEN_XDestroyWindowEvent_P(ptr)) (XEN_TO_C_XDestroyWindowEvent(ptr))->event = XEN_TO_C_Window(val); else
#if HAVE_MOTIF
  if (XEN_AnyCallbackStruct_P(ptr)) (XEN_TO_C_XmAnyCallbackStruct(ptr))->event = XEN_TO_C_XEvent(val); else
#endif
  XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "event", "a struct with an event field");
  return(val);
}

static XEN gxm_override_redirect(XEN ptr)
{
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XWindowAttributes(ptr))->override_redirect)));
  if (XEN_XSetWindowAttributes_P(ptr)) return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XSetWindowAttributes(ptr))->override_redirect)));
  if (XEN_XConfigureEvent_P(ptr)) return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XConfigureEvent(ptr))->override_redirect)));
  if (XEN_XReparentEvent_P(ptr)) return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XReparentEvent(ptr))->override_redirect)));
  if (XEN_XMapEvent_P(ptr)) return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XMapEvent(ptr))->override_redirect)));
  if (XEN_XCreateWindowEvent_P(ptr)) return(C_TO_XEN_BOOLEAN((Bool)((XEN_TO_C_XCreateWindowEvent(ptr))->override_redirect)));
#if WITH_GTK_AND_X11
  return(gxg_override_redirect(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "override_redirect", "a struct with an override_redirect field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_override_redirect(XEN ptr, XEN val)
{
  Bool b;
  XM_SET_FIELD_ASSERT_TYPE(XEN_BOOLEAN_P(val), val, XEN_ARG_2, "override_redirect", "a boolean");
  b = (Bool)XEN_TO_C_BOOLEAN(val);
  if (XEN_XConfigureEvent_P(ptr)) (XEN_TO_C_XConfigureEvent(ptr))->override_redirect = b;
  else if (XEN_XReparentEvent_P(ptr)) (XEN_TO_C_XReparentEvent(ptr))->override_redirect = b;
  else if (XEN_XMapEvent_P(ptr)) (XEN_TO_C_XMapEvent(ptr))->override_redirect = b;
  else if (XEN_XCreateWindowEvent_P(ptr)) (XEN_TO_C_XCreateWindowEvent(ptr))->override_redirect = b;
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "override_redirect", "a struct with an override_redirect field");
  return(val);
}

static XEN gxm_border_width(XEN ptr)
{
  if (XEN_XWindowChanges_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XWindowChanges(ptr))->border_width)));
  if (XEN_XWindowAttributes_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XWindowAttributes(ptr))->border_width)));
  if (XEN_XConfigureRequestEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XConfigureRequestEvent(ptr))->border_width)));
  if (XEN_XConfigureEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XConfigureEvent(ptr))->border_width)));
  if (XEN_XCreateWindowEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XCreateWindowEvent(ptr))->border_width)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "border_width", "a struct with a border_width field");
  return(XEN_FALSE);
}

static XEN gxm_set_border_width(XEN ptr, XEN val)
{
  int wid;
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "border_width", "an integer");
  wid = XEN_TO_C_INT(val);
  if (XEN_XConfigureRequestEvent_P(ptr)) (XEN_TO_C_XConfigureRequestEvent(ptr))->border_width = wid;
  else if (XEN_XConfigureEvent_P(ptr)) (XEN_TO_C_XConfigureEvent(ptr))->border_width = wid;
  else if (XEN_XCreateWindowEvent_P(ptr)) (XEN_TO_C_XCreateWindowEvent(ptr))->border_width = wid;
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "border_width", "a struct with a border_width field");
  return(val);
}

static XEN gxm_parent(XEN ptr)
{
  if (XEN_XCirculateRequestEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XCirculateRequestEvent(ptr))->parent)));
  if (XEN_XConfigureRequestEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XConfigureRequestEvent(ptr))->parent)));
  if (XEN_XReparentEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XReparentEvent(ptr))->parent)));
  if (XEN_XMapRequestEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XMapRequestEvent(ptr))->parent)));
  if (XEN_XCreateWindowEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XCreateWindowEvent(ptr))->parent)));
#if WITH_GTK_AND_X11
  return(gxg_parent(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "parent", "a struct with a parent field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_parent(XEN ptr, XEN val)
{
  Window w;
  XM_SET_FIELD_ASSERT_TYPE(XEN_Window_P(val), val, XEN_ARG_2, "parent", "a Window");
  w = XEN_TO_C_Window(val);
  if (XEN_XCirculateRequestEvent_P(ptr)) (XEN_TO_C_XCirculateRequestEvent(ptr))->parent = w;
  else if (XEN_XConfigureRequestEvent_P(ptr)) (XEN_TO_C_XConfigureRequestEvent(ptr))->parent = w;
  else if (XEN_XReparentEvent_P(ptr)) (XEN_TO_C_XReparentEvent(ptr))->parent = w;
  else if (XEN_XMapRequestEvent_P(ptr)) (XEN_TO_C_XMapRequestEvent(ptr))->parent = w;
  else if (XEN_XCreateWindowEvent_P(ptr)) (XEN_TO_C_XCreateWindowEvent(ptr))->parent= w;
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "parent", "a struct with a parent field");
  return(val);
}

static XEN gxm_minor_code(XEN ptr)
{
  if (XEN_XErrorEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XErrorEvent(ptr))->minor_code)));
  if (XEN_XNoExposeEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XNoExposeEvent(ptr))->minor_code)));
  if (XEN_XGraphicsExposeEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XGraphicsExposeEvent(ptr))->minor_code)));
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "minor_code", "a struct with a minor_code field");
  return(XEN_FALSE);
}

static XEN gxm_set_minor_code(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "minor_code", "an integer");
  if (XEN_XErrorEvent_P(ptr)) (XEN_TO_C_XErrorEvent(ptr))->minor_code = XEN_TO_C_INT(val);
  else if (XEN_XNoExposeEvent_P(ptr)) (XEN_TO_C_XNoExposeEvent(ptr))->minor_code = XEN_TO_C_INT(val);
  else if (XEN_XGraphicsExposeEvent_P(ptr)) (XEN_TO_C_XGraphicsExposeEvent(ptr))->minor_code = XEN_TO_C_INT(val);
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "minor_code", "a struct with a minor_code field");
  return(val);
}

static XEN gxm_major_code(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XNoExposeEvent_P(ptr) || XEN_XGraphicsExposeEvent_P(ptr), ptr, XEN_ONLY_ARG, "major_code", "a struct with a major_code field");
  if (XEN_XNoExposeEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XNoExposeEvent(ptr))->major_code)));
  return(C_TO_XEN_INT((int)((XEN_TO_C_XGraphicsExposeEvent(ptr))->major_code)));
}

static XEN gxm_set_major_code(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN_ARG_2, "major_code", "an integer");
  if (XEN_XNoExposeEvent_P(ptr)) (XEN_TO_C_XNoExposeEvent(ptr))->major_code = XEN_TO_C_INT(val);
  else if (XEN_XGraphicsExposeEvent_P(ptr)) (XEN_TO_C_XGraphicsExposeEvent(ptr))->major_code = XEN_TO_C_INT(val);
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "major_code", "a struct with a major_code field");
  return(val);
}

static XEN gxm_drawable(XEN ptr)
{
  XM_FIELD_ASSERT_TYPE(XEN_XNoExposeEvent_P(ptr) || XEN_XGraphicsExposeEvent_P(ptr), ptr, XEN_ONLY_ARG, "drawable", "a struct with a drawable field");
  if (XEN_XNoExposeEvent_P(ptr)) return(C_TO_XEN_Window((Window)((XEN_TO_C_XNoExposeEvent(ptr))->drawable)));
  return(C_TO_XEN_Window((Window)((XEN_TO_C_XGraphicsExposeEvent(ptr))->drawable)));
}

static XEN gxm_set_drawable(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_Window_P(val), val, XEN_ARG_2, "drawable", "a Window");
  if (XEN_XNoExposeEvent_P(ptr)) (XEN_TO_C_XNoExposeEvent(ptr))->drawable = XEN_TO_C_Window(val);
  else if (XEN_XGraphicsExposeEvent_P(ptr)) (XEN_TO_C_XGraphicsExposeEvent(ptr))->drawable = XEN_TO_C_Window(val);
  else XM_SET_FIELD_ASSERT_TYPE(0, ptr, XEN_ARG_1, "drawable", "a struct with a drawable field");
  return(val);
}

static XEN gxm_count(XEN ptr)
{
  if (XEN_XMappingEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XMappingEvent(ptr))->count)));
  if (XEN_XGraphicsExposeEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XGraphicsExposeEvent(ptr))->count)));
  if (XEN_XExposeEvent_P(ptr)) return(C_TO_XEN_INT((int)((XEN_TO_C_XExposeEvent(ptr))->count)));
#if WITH_GTK_AND_X11
  return(gxg_count(ptr));
#else
  XM_FIELD_ASSERT_TYPE(0, ptr, XEN_ONLY_ARG, "count", "a struct with a count field");
  return(XEN_FALSE);
#endif
}

static XEN gxm_set_count(XEN ptr, XEN val)
{
  XM_SET_FIELD_ASSERT_TYPE(XEN_INTEGER_P(val), val, XEN