/*
 * Copyright(c) 1997 by Jun-ichiro Itoh. All rights reserved.
 * Freely redistributable.  Absolutely no warranty.
 *
 * author contact: Jun-ichiro itojun Itoh <itojun@mt.cs.keio.ac.jp>
 * $Id: log.c,v 1.1.1.1 1998/02/23 14:18:30 itojun Exp $
 */
#include "common.h"

#if HAVE_SYS_IOCTL_H
# include <sys/ioctl.h>
#endif
#include <assert.h>

static int log_fd = -1;
#define LOGBUFSIZ	16*1024
static char log_buf[LOGBUFSIZ];
static int log_idx;
static char log_prevfrom;
static char log_prevto;

static void log_put __P((char *, int));

/*
 * PUBLIC_ void log_flush __P((void));
 */
void
log_flush()
{
	if (log_fd < 0)
		return;
	write(log_fd, log_buf, log_idx);
	log_idx = 0;
}

static void
log_put(buf, len)
	char *buf;
	int len;
{
	assert(len < sizeof(log_buf));
	if (sizeof(log_buf) < log_idx + len)
		log_flush();

	memcpy(&log_buf[log_idx], buf, len);
	log_idx += len;
}

/*
 * PUBLIC: void log_end __P((void));
 */
void
log_end()
{
	if (log_fd < 0)
		return;

	log_flush();
	close(log_fd);
	log_fd = -1;
}

/*
 * PUBLIC: int log_init __P((char *));
 */
int
log_init(name)
	char *name;
{
	if (0 <= log_fd)
		log_end();

	log_fd = open(name, O_WRONLY|O_CREAT|O_APPEND, 0644);
	log_prevfrom = log_prevto = '\0';
	log_idx = 0;
	return log_fd;
}

/*
 * PUBLIC: void log_write __P((int, int, char *, int));
 */
void
log_write(from, to, buf, len)
	int from;
	int to;
	char *buf;
	int len;
{
	size_t i;
	size_t adv;
	char tbuf[10];

	if (log_fd < 0)
		return;

	i = 0;
	if (log_prevfrom != from || log_prevto != to) {
		sprintf(tbuf, "\n%c>%c: ", from, to);
		log_put(tbuf, strlen(tbuf));
		log_prevfrom = from;
		log_prevto = to;
	}
	while (i < len) {
		if (isprint(buf[i])) {
			adv = i;
			while (isprint(buf[adv]) && adv < len)
				adv++;
			log_put(&buf[i], adv - i);
			i = adv;
		} else {
			sprintf(tbuf, "\\x%02x", buf[i] & 0xff);
			log_put(tbuf, 4);
			i++;
		}
	}
}

/*
 * PUBLIC: void log_status __P((int, int, char *));
 */
void
log_status(line, state, name)
	int line;
	int state;
	char *name;
{
	char tbuf[20];
#define	SHOW(str, bit)	\
    {					\
	if (state & bit)		\
		log_put("*", 1);	\
	log_put(str, strlen(str));	\
	log_put(" ", 1);		\
    }

	if (log_fd < 0)
		return;

	sprintf(tbuf, "\nline %d: ", line);
	log_put(tbuf, strlen(tbuf));

	SHOW("LE", TIOCM_LE);
	SHOW("DTR", TIOCM_DTR);
	SHOW("RTS", TIOCM_RTS);
	SHOW("ST", TIOCM_ST);
	SHOW("SR", TIOCM_SR);
	SHOW("CTS", TIOCM_CTS);
	SHOW("CD", TIOCM_CD);
	SHOW("RI", TIOCM_RI);
	SHOW("DSR", TIOCM_DSR);

	log_prevfrom = log_prevto = '\0';
#undef	SHOW()
}

/*
 * PUBLIC: void log_memo __P((char *));
 */
void
log_memo(memo)
	char *memo;
{
	time_t t;
	char tbuf[100];
	size_t len;

	if (log_fd < 0)
		return;

	sprintf(tbuf, "\nmemo at ");
	log_put(tbuf, strlen(tbuf));

	t = time(NULL);
	strcpy(tbuf, ctime(&t));
	len = strlen(tbuf);
	if (tbuf[len - 1] == '\n')
		tbuf[--len] = '\0';
	log_put(tbuf, len);

	log_put(": ", 2);
	log_put(memo, strlen(memo));

	log_prevfrom = log_prevto = '\0';
}
