/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include <somaplayer.h>
#include <string.h>

#ifdef ENABLE_GLIB
#include <glib.h>
#endif

#include <stdlib.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <dirent.h>

char *
__get_tmp_dir (void)
{
#ifdef ENABLE_GLIB
  return (char *) g_get_tmp_dir ();
#else
  return "/tmp/";
#endif
}

/* Open function */
somaplayer_controller *
somaplayer_open_unix (char *server)
{
  somaplayer_controller *s;

  if (!
      (s = (somaplayer_controller *) malloc (sizeof (somaplayer_controller))))
    return NULL;

  s->type = 0;

  if (server)
    s->server = strdup (server);
  else
    {
      int len = 0;
      DIR *d;
      struct dirent *dir;

      if (!(d = opendir (__get_tmp_dir ())))
	{
	  free (s);
	  return NULL;
	}

      while ((dir = readdir (d)))
	{

	  if (!strncmp (dir->d_name, "somaplayer-", 11)
	      && !strcmp (dir->d_name + strlen (dir->d_name) - 5, ".sock"))
	    {
	      len = strlen (__get_tmp_dir ()) + strlen (dir->d_name) + 2;

	      if (!(s->server = (char *) malloc (sizeof (char) * len)))
		{
		  free (s);
		  return NULL;
		}

	      snprintf (s->server, len, "%s/%s", __get_tmp_dir (),
			dir->d_name);

	      break;
	    }
	}

      if (!len)
	{
	  free (s);
	  return NULL;
	}
    }

  return s;
}

somaplayer_controller *
somaplayer_open_tcp (char *server, int port)
{
  somaplayer_controller *s;

  if (!
      (s = (somaplayer_controller *) malloc (sizeof (somaplayer_controller))))
    return NULL;

  s->type = 1;

  if (server)
    s->server = strdup (server);
  else
    {
      if (!(s->server = strdup ("127.0.0.1")))
	{
	  free (s);
	  return NULL;
	}
    }

  if (port)
    s->port = port;
  else
    s->port = SOMAPLAYER_PORT;

  return s;
}

int
__somaplayer_open (somaplayer_controller * sc)
{

  if (!sc)
    return -1;

  if (!sc->type)
    {
      int fd;
      struct sockaddr_un saddr;

      if ((fd = socket (AF_UNIX, SOCK_STREAM, 0)) < 0)
	return -1;

      saddr.sun_family = AF_UNIX;
      strncpy (saddr.sun_path, sc->server, 108);

      if (connect (fd, (struct sockaddr *) &saddr, sizeof (saddr)) == -1)
	{
	  close (fd);
	  return -1;
	}

      return fd;
    }
  else
    {
      int fd;
      struct sockaddr_in sock;
      struct hostent *hp;

      if ((fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	return -1;

      if (!(hp = gethostbyname (sc->server)))
	return -1;

      memset ((void *) &sock, 0, sizeof (sock));
      sock.sin_family = AF_INET;
      sock.sin_addr.s_addr = ((struct in_addr *) (hp->h_addr))->s_addr;

      sock.sin_port = htons (sc->port);

      if (connect (fd, (struct sockaddr *) &sock, sizeof (sock)) < 0)
	{
	  close (fd);
	  return -1;
	}

      return fd;
    }
}

void
__somaplayer_close (somaplayer_controller * s)
{
  close (s->fd);
}

int
somaplayer_close (somaplayer_controller * s)
{
  if (!s)
    return 1;

  free (s->server);
  free (s);

  return 0;
}

int
somaplayer_next (somaplayer_controller * s)
{
  if (!s)
    return 1;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      write (s->fd, "next\n", 5);
      __somaplayer_close (s);

      return 0;
    }

  return 1;
}

int
somaplayer_prev (somaplayer_controller * s)
{
  if (!s)
    return 1;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      write (s->fd, "prev\n", 5);
      __somaplayer_close (s);

      return 0;
    }

  return 1;
}

int
somaplayer_quit (somaplayer_controller * s)
{
  if (!s)
    return 1;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      write (s->fd, "quit\n", 5);
      __somaplayer_close (s);

      return 0;
    }

  return 1;
}

int
somaplayer_pause (somaplayer_controller * s)
{
  if (!s)
    return 1;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      write (s->fd, "pause\n", 6);
      __somaplayer_close (s);

      return 0;
    }

  return 1;
}

int
somaplayer_set_volume (somaplayer_controller * s, int v)
{
  char buf[1024];
  int len;

  if (!s)
    return 1;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      len = snprintf (buf, 1024, "set_volume %d\n", v);
      write (s->fd, buf, len);
      __somaplayer_close (s);

      return 0;
    }

  return 1;
}

int
somaplayer_set_balance (somaplayer_controller * s, int v)
{
  char buf[1024];
  int len;

  if (!s)
    return 1;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      len = snprintf (buf, 1024, "set_balance %d\n", v);
      write (s->fd, buf, len);
      __somaplayer_close (s);

      return 0;
    }

  return 1;
}

int
somaplayer_get_volume (somaplayer_controller * s, int *v)
{
  char buf[1024];
  int len;

  if (!s)
    return 1;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      write (s->fd, "get_volume\n", 11);
      len = read (s->fd, buf, 1024);
      buf[len] = 0;

      for (len--; len; len--)
	if (buf[len] == '\n')
	  {
	    buf[len] = 0;
	    break;
	  }
      __somaplayer_close (s);

      *v = atoi (buf);
      return 0;
    }

  return 1;
}

int
somaplayer_get_balance (somaplayer_controller * s, int *b)
{
  char buf[1024];
  int len;

  if (!s)
    return 1;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      write (s->fd, "get_balance\n", 12);
      len = read (s->fd, buf, 1024);
      buf[len] = 0;

      for (len--; len; len--)
	if (buf[len] == '\n')
	  {
	    buf[len] = 0;
	    break;
	  }
      __somaplayer_close (s);

      *b = atoi (buf);
      return 0;
    }

  return 1;
}

char *
somaplayer_this (somaplayer_controller * s)
{
  char buf[1024];
  int len;

  if (!s)
    return NULL;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      write (s->fd, "this\n", 11);

      len = read (s->fd, buf, 1024);
      if(len==1) len--;
      buf[len] = 0;

      for (; len; len--)
	if (buf[len] == '\n')
	  {
	    buf[len] = 0;
	    break;
	  }

      __somaplayer_close (s);

      return strdup (buf);
    }

  return NULL;
}

char **
somaplayer_get_playlist (somaplayer_controller * s)
{
  struct l
  {
    char *file;
    struct l *next;
  } *list = NULL, *old = NULL;

  char buf[1024];
  char **ret;
  int i = 0;

  if (!s)
    return NULL;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      write (s->fd, "get_playlist\n", 13);

      while (read (s->fd, buf + i, 1) > 0)
	{
	  if (buf[i] == '\n')
	    {
	      buf[i] = 0;

	      if (!list)
		{
		  if (!(list = (struct l *) malloc (sizeof (struct l))))
		    return NULL;
		  memset (list, 0, sizeof (struct l));

		  old = list;
		}
	      else
		{
		  if (!(old->next = (struct l *) malloc (sizeof (struct l))))
		    return NULL;
		  memset (old->next, 0, sizeof (struct l));
		  old = old->next;
		}

	      if (!(old->file = strdup (buf)))
		return NULL;

	      i = 0;
	    }

	  else
	  i++;
	}

      __somaplayer_close (s);
    }

  if (!list)
    return NULL;

  old = list;
  i = 0;
  while (old)
    {
      i++;
      old = old->next;
    }

  if (!(ret = (char **) malloc (sizeof (char *) * (i + 1))))
    return NULL;

  old = list;
  i = 0;
  while (old)
    {
      ret[i] = old->file;
      old = old->next;
      i++;
    }
  ret[i] = NULL;

  while (list)
    {
      old = list;
      list = list->next;
      free (old);
    }

  return ret;
}

void
somaplayer_del_playlist (somaplayer_controller * s, int n) {

  if(!s) return;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      char buf[1024];
      int len;

      len=snprintf(buf,sizeof(buf), "del_playlist %d\n",n);
      write (s->fd, buf, len);
    }
}

void
somaplayer_move_playlist (somaplayer_controller * s, int n, int before) {
  if(!s) return;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      char buf[1024];
      int len;

      len=snprintf(buf,sizeof(buf), "move_playlist %d %d\n",n, before);
      write (s->fd, buf, len);
    }
}

void
somaplayer_add_playlist (somaplayer_controller * s, char *what) {

  if(!s) return;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      char buf[1024];
      int len;

      len=snprintf(buf,sizeof(buf), "add_playlist %s\n",what);
      write (s->fd, buf, len);
    }
}

void
somaplayer_play_playlist (somaplayer_controller * s, int what) {

  if(!s) return;

  if ((s->fd = __somaplayer_open (s)) > 0)
    {
      char buf[1024];
      int len;

      len=snprintf(buf,sizeof(buf), "play_playlist %d\n",what);
      write (s->fd, buf, len);
    }
}

/* EOF */
