import NetworkConstants
import Event
import URLFetch
import PollManager
import ItemStore
import utils
import error
import Config

STATUS_IMAGE_WAITING = None
STATUS_IMAGE_BROKEN = None

class CacheEntry(object):
    def __init__(self, image, count, pollstopper = None):
        self._image = image
        self._count = count
        self._pollstopper = pollstopper

    def get_image(self): return self._image
    def set_image(self, image): self._image = image
    image = property(get_image, set_image)

    def get_count(self): return self._count
    def set_count(self, count): self._count = count
    count = property(get_count, set_count)

    def incr_count(self): self._count += 1
    def decr_count(self): self._count -= 1

    def get_pollstopper(self): return self._pollstopper
    def set_pollstopper(self, pollstopper):
        self._pollstopper = pollstopper
        if pollstopper is not None:
            pollstopper.signal_connect(Event.PollingStoppedSignal,
                                       self._polling_stopped)
    pollstopper = property(get_pollstopper, set_pollstopper)

    def _polling_stopped(self):
        self._pollstopper = None

class Cache(Event.SignalEmitter):
    """Image cache with explicit reference counting."""
    def __init__(self):
        Event.SignalEmitter.__init__(self)
        self.initialize_slots(Event.ImageUpdatedSignal)
        # self.__cache contains items of the form [Image, refcount]
        self.__cache = {}

    def __getitem__(self, key):
        return self.__cache[key].image
     
    def add_refer(self, key, restore = False, item = None):
        if key not in self.__cache:
            if not restore:
                # fetch image
                image = Image(key, Image.WAITING)
                ic = ImageConsumer(image)
                headers = {}
                if item and item.feed:
                    headers['Referer'] = item.feed.location
                try:
                    stopper = URLFetch.connection_manager.request(
                        key, ic,
                        priority=NetworkConstants.PRIORITY_IMAGE,
                        headers=headers)
                except URLFetch.RequestSchemeException, e:
                    ic.http_failed(e)
                except Exception, e:
                    error.logtb("ImageCache.add_refer:  ", str(e))
                    ic.http_failed(e)
                self.__cache[key] = CacheEntry(
                    image, 1, PollManager.PollStopper(stopper, image))
            else:
                image = Image(key, Image.DATA_IN_DB)
                self.__cache[key] = CacheEntry(image, 1)
        else:
            self.__cache[key].incr_count()

    def remove_refer(self, key):
        self.__cache[key].decr_count()
        if self.__cache[key].count == 0:
            del self.__cache[key]
            self.emit_signal(Event.ImageUpdatedSignal(self, key, None))

    def image_updated(self, url, data):
        self.__cache[url].pollstopper = None
        self.emit_signal(Event.ImageUpdatedSignal(self, url, data))

    def set_image(self, url, image):
        if self.__cache.has_key(url):
            self.__cache[url].incr_count()
        else:
            self.__cache[url] = CacheEntry(image, 1)

    def stop_transfer(self, url):
        if self.__cache[url].pollstopper is not None:
            self.__cache[url].pollstopper.stop()
            self.__cache[url].pollstopper = None

class Image:
    WAITING = 1
    DATA_IN_DB = 2
    FAILED = 3

    def __init__(self, url, status = DATA_IN_DB):
        self.url = url
        self.status = status

    def get_data(self):
        if self.status == self.WAITING:
            return STATUS_IMAGE_WAITING
        elif self.status == self.DATA_IN_DB:
            data = self.read_data()
            if data is None:
                self.status = self.FAILED
                return STATUS_IMAGE_BROKEN
            return data
        else:
            return STATUS_IMAGE_BROKEN

    def set_data(self, data):
        cache.image_updated(self.url, data)
        self.status = self.DATA_IN_DB

    def set_failed(self):
        self.status = self.FAILED

    def read_data(self):
        c = Config.get_instance()
        return ItemStore.get_instance(c.straw_dir).read_image(self.url)

    def _return_id(self):
        return "%d %s" % (id(self), self.url)

cache = Cache()

class ImageConsumer:
    def __init__(self, imobj):
        self._imobj = imobj
    
    def http_results(self, status, headers, data):
        if status[1] == 200:
            self._imobj.set_data(data)
        else:
            self._imobj.set_failed()

    def http_failed(self, exception):
        self._imobj.set_failed()

    def operation_stopped(self):
        self._imobj.set_failed()

def initialize():
    global STATUS_IMAGE_WAITING
    global STATUS_IMAGE_BROKEN
    libdir = utils.find_lib_dir()
    STATUS_IMAGE_WAITING = open(
        libdir + "/image-waiting.png").read()
    STATUS_IMAGE_BROKEN = open(
        libdir + "/image-broken.png").read()
