/* ==================================================== ======== ======= *
 *
 *  ubrowser.hpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _ubrowser_hpp_
#define _ubrowser_hpp_
//pragma ident	"@(#)ubrowser.hpp	ubit:03.06.03"
#include <ubit/ubox.hpp>
#include <ubit/ext/udoc.hpp>
#include <ubit/ext/uicon.hpp>

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

class BrwDir;

class UBrowser : public UBox{
public:

  struct Options : UGroup {
    friend class UBrowser;
    bool          show_icon_previews;
    UBackground   background;
    UColor        fgcolor;
    
    UWidth        controls_info_width;
    UHeight       controls_info_height;
    UWidth        main_panel_width;
    UHeight       main_panel_height;
    UWidth        doc_width;
    UHeight       doc_height;
    UWidth        icon_width;
    UHspacing     iconbox_hspacing;
    UVspacing     iconbox_vspacing;
    UWidth        icon_content_width;
    UHeight       icon_content_height;
    UWidth        preview_content_width;
    UHeight       preview_content_height;
    UWidth        clone_frame_width;
    UHeight       clone_frame_height;
    Options();
  };

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // public API

  UBrowser(const UStr& pathname = ".");
  virtual ~UBrowser();

  virtual int open(const UStr& pathname);
  virtual int open(UStr& pathname);
  ///< opens a directory or a file (the second variant normalizes pathname).

  virtual int open(UFilebox&);
  ///< opens the file given by a UFilebox.

  virtual void openParent();
  virtual void openEntry();  
  virtual void nextEntry();
  virtual void previousEntry();

  Options& getOptions() {return opts;}
  /**< retrieves the options of the UBrowser.
    *  options can be changed while the UBrowser is running.
    */

  class UFilebox*    getOpenBox()      {return &open_box;}
  class UMenu*       getOpenMenu()     {return &open_menu;}

  class UDocbox*     getDocbox()       {return &docbox;}
  class UBox*        getDocPanel()     {return &doc_panel;}
  class UScrollpane* getDocSpane()     {return &doc_spane;}

  class UBox*        getFinder()       {return &finder;}
  class UScrollpane* getFinderSpane();

  class UIconbox*    getSelectedIconbox() const;
  class UIcon*       getSelectedEntry()   const;

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual UItem& addHost(UStr* hostname);
  virtual void addHosts(const std::vector<UStr*>& hostnames);
  ///< strings are *not* duplicated and should not be destroyed.

  virtual void addHosts(const char* hostnames[]);
  ///< strings are duplicated and can be destroyed.

  virtual bool createClone(const UStr& hostname);
  ///< creates a synchronized clone on this host.

  UBox& getBigCustomControls();
  UBox& getSmallCustomControls();
  void showControlPanel(bool state);

  void showAlert(const UStr& msg);
  void showAlert(bool state);
  ///< shows/hides an alert message.
  
  void alert(const UStr& msg) {showAlert(msg);}
  ///< OBSOLETE synonym of showAlert()

  enum Action {
    DocRequest = 1, DocOpen, DocShow, DirRequest, DirOpen, DirShow,
    LinkEnter, LinkLeave, LinkPress, LinkRelease, LinkAction
  };
  
  virtual void actionNotify(Action, const UStr& path, class UDoc*) {}
  /**< notification of various actions.
    * can be subclassed by clients for appropriate behavior.
    */

  virtual void processMessage(UEvent&);
  /**< reads and processes an UMessage.
    * can be subclassed by clients for adding actions.
    */
  
  virtual bool readBookmarks(const char* filename, bool from_home_dir);
  virtual bool writeBookmarks(const char* filename, bool from_home_dir);
  /**< bookmark management.
   * can be called by clients for reading and saving bookmarks.
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
protected:   // private API

  virtual void addToHistory(BrwDir*);
  virtual void addHostCB(const UStr& hostname);
  virtual void createCloneRequest(const UStr& hostname);
  virtual bool createClone(UItem* host_item, UStr* hostname);
  virtual void deleteClone(UWin* clone_win, UItem* host_item);
  virtual UFrame* createCloneFrame(const UStr& title);
  ///< can be subclassed by clients for appropriate behavior.
  
  virtual void removeFromHistory(BrwDir*);
  virtual BrwDir* findDir(const UStr& pathname);
  virtual void showDirInfo(BrwDir*);
  virtual void removeIconbox(BrwDir*, bool update);
  virtual void iconifyIconbox(BrwDir*);
  virtual void iconSelectCB(UIconbox*);
  virtual void iconActionCB(UIconbox*);
  
  virtual int  openDoc(const UStr& pathname);
  virtual int  openDir(const UStr& pathname, bool reload);
  virtual int  openDir(BrwDir*, bool reload);  
  virtual void showDoc(const UStr& pathname, class UDoc*);
  virtual void showDir(const UStr& pathname);
  virtual void showPreview(class UIcon*);
  virtual void showPreviewRequest(class UIcon*);
  virtual void showIconPreviews(class UTimer*, class UIconbox*);
  virtual void showFullWin();
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

private:
  enum {NoMode, DirMode, DocMode} mode;
  bool                    open_in_fullwin;
  Options                 opts;
  UFilebox                open_box;
  UMenu                   open_menu;
  UVbox                   main_panel;
  UDocbox                 docbox;
  UVbox                   finder;
  UVbox                   doc_panel;
  uptr<class UScrollpane> doc_spane;
  uptr<class BrwControls> ctrls;
  uptr<class BrwFullWin>  fullwin;
  uptr<class UAlertbox>   alertbox;
  UDialog                 ask_dialog;
  UStr ask_dialog_msg;
  class BrwLinkCallbacks* link_callbacks;
  UFlagdef                title_location;
    
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // current selection
  uptr<UIconbox>  selected_iconbox;
  uptr<UIcon>     selected_fullwin;
  uptr<BrwDir>    last_direntry;
  UIcon*          last_preview_request;
  UIcon*          last_preview;
  uptr<UIconbox>  kill_ring;
  UListPos        pos;  // !!!!

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // friends
  friend class BrwDir;
  friend class BrwFullWin;
  friend class BrwControls;
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
