/* ==================================================== ======== ======= *
 *
 *  umenu.hpp
 *  Ubit Project [Elc::2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _umenu_hpp_
#define	_umenu_hpp_
//pragma ident	"@(#)umenu.hpp	ubit:03.06.03"
#include <ubit/uwin.hpp>

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Menu bar.
 */
class UMenubar: public UBar {
  void enterChild(UEvent&);
  void leaveChild(UEvent&);
  void relaxChild(UEvent&);

public:
  static UStyle *style;

  UMenubar(const UArgs& a = UArgs::none);

  friend UMenubar& umenubar(const UArgs& a = UArgs::none)
  {return *new UMenubar(a);}
  ///< creator shortcut that is equivalent to: *new UMenubar().

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

  virtual UGroup* getBrowsingGroup() {return this;}
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** UMenu: pulldown or cascaded menu. 
 *
 * Notes:
 *  - UMenu objects are AUTOMATICALLY OPENED by their parent(s)
 *  - UMenu parents should generally be buttons (see example below)
 *  - if you need contextual menus, use class: UPopmenu
 *  - as other Ubit UBricks, menus MUST HAVE at least one parent
 * 
 * See also classes: UWin, UMenubar, UPopmenu.
 *
 * Example: <pre>
 *   umenubar( ubutton("Menu 1"
 *                     + umenu( ubutton("aaa") + ubutton("bbb") ))
 *            + ubutton("Menu 2"
 *                     + umenu( ulabel("xxx") + ucheckbox("ccc") ))
 *           )
 * </pre>
 */
class UMenu : public UWin {
public:
  static UStyle *style;

  UMenu(const UArgs& a = UArgs::none);
  friend UMenu& umenu(const UArgs& a = UArgs::none) {return *new UMenu(a);}
  /**< creator shortcut that is equivalent to: *new UMenu().*/

  virtual ~UMenu();

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

  virtual UGroup* getBrowsingGroup();
  ///< [impl].

  virtual void open();
  /**< opens the menu (use this method instead of show(true) in the general case).
   * Notes:
   * - there is usually no need to call this method explicitely when using
   *   UMenu objects (but it must be called when using UPopmenu objects)
   * - this method implicitely calls the (inherited) show() method
   *
   * Details: the open() method:
   * - grabs the mouse 
   * - closes menus that are already opened (except if they are cascaded 
   *   parents of this menu)
   * - enforces an autoclose behavior for this menu (it will be automatically
   *  closed when a button child is clicked)
    */

  virtual void close(int status = 0);
  /**< closes the menu (use this method instead of show(false) in the general case).
   * There is usually no need to call this function explicitely as menus are 
   * automatically closed
   * <p>the close() method 1) ungrabs the mouse, 2) hides this menu and its
   * cascaded children
   * <p>See also: UWin::close()
   */

  virtual void setPlacement(const class UPlacement*);
  virtual void setPlacement(const class UPlacement&);
  /**< specifies automatic placement rules (see UPlacement).
   * Details:
   * - null argument --> default placement rules
   * - data given as an argument can be freed at any time by client
   * - update() must be called to enforce new rules
    */

  virtual bool realize();

#ifndef NO_DOC
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  virtual void addingTo(class ULink *selflink, UGroup *parent);
  virtual void removingFrom(class ULink *selflink, UGroup *parent);
  ///< NOTE that this function require a specific destructor.

private:
  friend class UAppli;
  friend class UMenuCtrl;

  UGroup *menuBrowsingGroup;
  UCall *enter_opener, *arm_opener, *disarm_opener;
  class UPlacement *placement;

  void enterChild(UEvent&);
  void leaveChild(UEvent&);
  void relaxChild(UEvent&);
  void enterOpener(UEvent&);
  void armOpener(UEvent&);
  void disarmOpener(UEvent&);

  /// internal function for opening menus
  virtual void openImpl(class UMenuCtrl&, UView *opener, 
			UGroup *menugroup, bool auto_place);
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** UPopmenu: Contextual Menu.
 *
 * UPopmenu objects are NOT automatically OPENED by their parent(s).
 * There a two ways for opening Popmenus:
 * - 1) by using the UPopmenu::autoOpens() method
 * - 2) by adding appropriate callbacks to menu openers (see example below)
 * 
 * Note: as other Ubit UBricks, menus MUST HAVE at least one parent.
 * 
 * See also classes: UWin, UMenu.
 *
 * Example (see also method: autoOpens()) :
 * <pre>
 *  opener.addlist(menu + UOn::mpress / ucall(&menu, openMenu))
 *
 *  void openMenu(UEvent& e, UPopmenu* pop) {
 *       popmenu->move(e, 0, 0);  // move the menu to Event location
 *       popmenu->open();         // open the menu
 *   }
 * </pre>
 */
class UPopmenu : public UMenu {
public:
  struct InputCond {
    u_id mods;
    u_id buttons;
    u_id keysym;
    InputCond(u_id _mods, u_id _buttons, u_id _keysym) {
      mods = _mods; buttons = _buttons; keysym = _keysym;
    }
  };
  
  static UStyle *style;

  UPopmenu(const UArgs& a = UArgs::none);

  friend UPopmenu& upopmenu(const UArgs& a = UArgs::none) {return *new UPopmenu(a);}
  ///< creator shortcut that is equivalent to: *new UPopmenu().

  virtual void autoOpens(UBox& opener, u_id button_mask, u_id keysym,
                         bool children_open_menu = false);
  virtual void autoOpens(UBox& opener,
                         const std::vector<InputCond>&,
                         bool children_open_menu);
  
  /**< this 'opener' will automatically open the menu.
   * Arguments:
   * - 'button_mask' indicate which mouse buttons open the menu when pressed. 
   *   It's an ORed combination of UEvent::MButton1, UEvent::MButton2, etc.
   * - if 'keysym' is not null it indicates which X keysym (= which logical
   *   key on the keyboard) opens the menu when pressed.
   * - 'children_open_menu' means that the menu is always opened, even if 
   *   the event is located inside a child of the opener (default is false).
   */

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

protected:
  std::vector<InputCond> input_conds;
  virtual bool autoOpenCheck(UEvent&);
  virtual void autoOpenImpl(UEvent&);
  ///< function called by autoOpens() when the menu pops up.
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */


