/* ==================================================== ======== ======= *
 *
 *  uufont.cpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//#pragma ident	"@(#)uufont.cpp	ubit:03.06.02"
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <ucontext.hpp>
#include <uctrl.hpp>
#include <ufont.hpp>
#include <uappli.hpp>
#include <ubox.hpp>
#include <update.hpp>

// 4 variants: plain, bold, italic, boldItalic
//const int UFont::FONTMAP_SIZE  = 4 * (UFont::MAX_LSIZE + 1);

static const UFont::FontSize _FONT_SIZES[UFont::MAX_LSIZE] = {
  {1,2,"2"}, 
  {2,4,"4"}, 
  {3,6,"6"},     // XX_SMALL
  {4,7,"7"}, 
  {5,8,"8"},     // X_SMALL
  {6,9,"9"}, 
  {7,10,"10"},   // SMALL
  {8,12,"12"},   //MEDIUM_LSIZE = 8 :: 12pt
  {9,14,"14"},  // LARGE
  {10,15,"15"}, 
  {11,16,"16"},  // X_LARGE
  {12,18,"18"},
  {13,20,"20"},  // XX_LARGE
  {14,22,"22"},
  {15,24,"24"},
  {16,26,"26"},
  {17,28,"28"},
  {18,30,"30"},
  {19,32,"32"},
  {20,34,"34"},
  {21,36,"36"},
  {22,38,"38"},
  {23,40,"40"}
};

const UFont::FontSize*
UFontFamily::FONT_SIZES = _FONT_SIZES;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// *** Predefined FontFamilies ***

//Problemes:
// -- la specif 'normal' deconne sous certains Linux
// -- italique est tantot specifie par 'o' ou par 'i'
// -- fixed n'a pas forcement d'italique (ou alors d'une taille delirante)

UFontFamily 
UFontFamily::helvetica; //("helvetica", "medium", "bold", "o", "normal", "1");
UFontFamily
UFontFamily::courier; //("courier", "medium", "bold", "o", "normal","1");
UFontFamily 
UFontFamily::times; //("times", "medium", "bold", "i", "normal","1");

UFontFamily 
UFontFamily::fixed; //("fixed", "medium", "bold", "o", "*", "1");

// The Standard Font Family. Should always be available and preloaded
UFontFamily 
UFontFamily::standard;// = helvetica;

// The "Any" Font Family. 
// Mean 'any font family' in UFont() specifications (see below)
UFontFamily 
UFontFamily::any;// = standard;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UFontFamily::family_count = 0;

// Creates a new FontFamily from a native system font name
void UFontFamily::init(const char *_name,
			 const char *_normal_style,
			 const char *_bold_style, 
			 const char *_italic_style,
			 const char *_compression,
			 const char *_encoding,
			 const UFont::FontSize *_font_sizes) {
  index       = family_count++;     // !ATT: reference count
  name	      = _name;
  normal_weight = _normal_style;
  bold_weight   = _bold_style;
  slant  = _italic_style;
  compression = _compression;
  encoding    = _encoding;
  if (_font_sizes) font_sizes = _font_sizes;
  else font_sizes = FONT_SIZES;
}

const UFont::FontSize* UFontFamily::getFontSizes() {
  return FONT_SIZES;
}

#define FSIZE(X)  FONT_SIZES[(X)-1].ptsize

float UFontFamily::getXYScale(short lscale) {
  int fsize = lscale + UFont::MEDIUM_LSIZE;

  if (fsize > UFont::MAX_LSIZE) {
    return (1.75 * fsize) / FSIZE(UFont::MEDIUM_LSIZE);
  }
  else if (fsize < UFont::MIN_LSIZE) {
    return (float)FSIZE(UFont::MIN_LSIZE) / FSIZE(UFont::MEDIUM_LSIZE);
  }
  else {
    return (float)FSIZE(fsize) / FSIZE(UFont::MEDIUM_LSIZE);
  }
}


// there is usually no need to call this function except if you want
// to change the default font sizes, in which case it must be called
// at initialization time; Argument is an array of MAX_LSIZE
/***  VOIR UConfig
void UFontFamily::setDefaultFontSizes(const UFont::FontSize* fs) {
  DEFAULT_FONT_SIZES = fs;
}
***/

int UFontFamily::lsizeToPtsize(int _lsize) const { // LOGICAL size 
  if (_lsize <= 0) return 0;
  else if (_lsize > UFont::MAX_LSIZE) _lsize = UFont::MAX_LSIZE;
  return font_sizes[_lsize-1].ptsize;
}

int UFontFamily::ptsizeToLsize(int _ptsize) const {
  if (_ptsize <= 0) return 0;
  int mind = 99999;
  int mink = 0;
  for (int k = 0; k < UFont::MAX_LSIZE; k++) {
    int d = _ptsize - font_sizes[k].ptsize;
    if (d < 0) d = -d; //abs
    if (d > mind) return font_sizes[mink].logsize;
    else {mind = d; mink = k;} 
  }

  return font_sizes[mink].logsize; // securite
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// *** Fonts ***

// NOT bold nor Italic
UFont UFont::normal(UFontFamily::any,  -BOLD|-ITALIC, 0, UMode::UCONST);
UFont UFont::bold(UFontFamily::any,      BOLD,        0, UMode::UCONST);
UFont UFont::italic(UFontFamily::any,    ITALIC,      0, UMode::UCONST);
UFont UFont::fill(UFontFamily::any,      FILL,        0, UMode::UCONST);
UFont UFont::underline(UFontFamily::any, UNDERLINE,   0, UMode::UCONST);
UFont UFont::overline(UFontFamily::any,  OVERLINE,    0, UMode::UCONST);
UFont UFont::strikethrough(UFontFamily::any, STRIKETHROUGH, 0, UMode::UCONST);

// NOT bold, italic, etc
UFont UFont::_bold(UFontFamily::any,     -BOLD,        0, UMode::UCONST);
UFont UFont::_italic(UFontFamily::any,   -ITALIC,      0, UMode::UCONST);
UFont UFont::_fill(UFontFamily::any,     -FILL,        0, UMode::UCONST);
UFont UFont::_underline(UFontFamily::any,-UNDERLINE,   0, UMode::UCONST);
UFont UFont::_overline(UFontFamily::any, -OVERLINE,    0, UMode::UCONST);
UFont UFont::_strikethrough(UFontFamily::any, -STRIKETHROUGH, 0, UMode::UCONST);

// sizes
UFont UFont::xx_small(UFontFamily::any,  0, 3, UMode::UCONST);
UFont UFont::x_small(UFontFamily::any,   0, 5, UMode::UCONST);
UFont UFont::small(UFontFamily::any,     0, 7, UMode::UCONST);
// Default size is MEDIUM_LSIZE
UFont UFont::medium(UFontFamily::any,    0, MEDIUM_LSIZE,   UMode::UCONST);
UFont UFont::large(UFontFamily::any,     0, 9, UMode::UCONST);
UFont UFont::x_large(UFontFamily::any,   0, 11, UMode::UCONST);
UFont UFont::xx_large(UFontFamily::any,  0, 13, UMode::UCONST);

// 'inherit' means that the font is inherited from the parent box
UFont UFont::inherit(UFontFamily::standard,    0,0, UMode::UCONST);
// standard imposes the default size
UFont UFont::standard(UFontFamily::standard,   0, MEDIUM_LSIZE, UMode::UCONST);

UFont UFont::helvetica(UFontFamily::helvetica, 0,0, UMode::UCONST);
UFont UFont::times(UFontFamily::times,         0,0, UMode::UCONST);
UFont UFont::courier(UFontFamily::courier,     0,0, UMode::UCONST);
UFont UFont::fixed(UFontFamily::fixed,     0,0, UMode::UCONST);

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// Arguments:
// - family is a previously created font family (see above)
// - style is an integer bitmask of BOLD, ITALIC, etc. 0 means default
// - lsize is the LOGICAL size: from MIN_LSIZE to MAX_LSIZE
//   * Default size is MEDIUM_LSIZE
//   * 0 means "unspecified, don't impose a specific size".
//
UFont::UFont(const UFontFamily& ff, int _styles, int _lsize, u_modes m) 
  : UProp(m) {
  family = &ff;
  setLsize(_lsize, false);		// NB: 0 means default
  setStyle(_styles, false);
}

UFont::UFont(const UFont& f) : UProp() {
  family   = f.family;
  lsize    = f.lsize;
  onStyles = f.onStyles;
  offStyles= f.offStyles;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UFont& UFont::operator=(const UFont&f) {
  set(f);
  return *this;
}

void UFont::set(const UFont&f) {
  if (equals(f)) return;
  family    = f.family;
  lsize     = f.lsize;
  onStyles  = f.onStyles;
  offStyles = f.offStyles;
  changed(true);
}

void UFont::setImpl(const UFontDesc*f, bool upd) {
  family   = f->family;
  lsize    = f->lsize;
  onStyles = f->styles;
  offStyles= 0;
  changed(upd);
}

void UFont::merge(const UFont&f) {
  if (f.family != &UFontFamily::any)  family = f.family;
  // 0 means don't impose size
  if (f.lsize > 0) lsize = f.lsize;

  // combines styles
  onStyles = onStyles | f.onStyles;
  offStyles = offStyles | f.offStyles;
  changed(true);
}

bool UFont::equals(const UFont&f) const {
  if (this == &f) return true;
  else return (family == f.family
	       && lsize == f.lsize
	       && onStyles == f.onStyles
	       && offStyles == f.offStyles);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UFont::setFamily(const UFont &f) {
  family = f.family;
  changed(true);
}

void UFont::setFamily(const UFontFamily &ff) {
  family = &ff;
  changed(true);
}

void UFont::setStyle(const UFont &f) {
  onStyles  = f.onStyles;
  offStyles = f.offStyles;
  changed(true);
}

void UFont::setStyle(signed int _styles, bool upd) {
  if (_styles > 0) {
    onStyles  = _styles;
    offStyles = 0;
  }
  else {
    onStyles  = 0;
    offStyles = -_styles;
  }
  changed(upd);
}

void UFont::setSize(const UFont &f) {
  lsize  = f.lsize;
  changed(true);
}

// LOGICAL size NOT point size; 0 = default
void UFont::setLsize(int _lsize, bool upd) {
  // 0 means don't impose a specific size
  if (_lsize <= 0) _lsize = 0;
  else if (_lsize > MAX_LSIZE) _lsize = MAX_LSIZE;
  lsize = _lsize;
  changed(upd);
}

int UFont::getLsize()     const {return lsize;}

void UFont::setPointSize(int _ptsize, bool upd) {
  setLsize(family->ptsizeToLsize(_ptsize), upd);
}
void UFont::setPtsize(int _ptsize, bool upd) {
  setLsize(family->ptsizeToLsize(_ptsize), upd);
}

int UFont::getPointSize() const {return family->lsizeToPtsize(lsize);}
int UFont::getPtsize()    const {return family->lsizeToPtsize(lsize);}

bool UFont::isBold()      const {return (onStyles & BOLD) != 0;}
bool UFont::isItalic()    const {return (onStyles & ITALIC) != 0;}

bool UFont::isFilled()     const {return (onStyles & FILL) != 0;}
bool UFont::isUnderlined() const {return (onStyles & UNDERLINE) != 0;}
bool UFont::isOverlined()  const {return (onStyles & OVERLINE) != 0;}
bool UFont::isStrikethrough() const {return (onStyles & STRIKETHROUGH) != 0;}


const UFontFamily* UFont::getFamily() const {return family;}

void UFont::update() {
  // size changed in both directions
  parents.updateParents(UUpdate::all);
}

// This method initializes a UFont for a given UDisplay (or UAppli)
// It returns true if the font could be found and false otherwise
// (a default font will be used in this case)
// NOTE: it is not necessary to call this function explicitely since
// version 99.10 (it will automatically be called the first time 
// this UFont is used for drawing a string on the screen)
//
bool UFont::realize(UDisp *d) {
  return d->realizeFont(*this);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UFont::putProp(UContext *props, UCtrl *state) {
  UFontDesc &df = props->fontdesc;
  if (family != &UFontFamily::any) {
    df.family = family;
  }
  // ajouter les 'onStyles' et enlever les 'offStyles'
  df.styles = (df.styles | onStyles) & ~offStyles;

  //NB: lsize = 0 means "any size" and thus, does not change df.size 
  if (lsize > 0) {
    df.lsize = lsize + props->lscale;   //NB: lscale = LOGICAL scale

    if (df.lsize < MIN_LSIZE) df.lsize = MIN_LSIZE;
    else if (df.lsize > MAX_LSIZE) df.lsize = MAX_LSIZE;
  }

  df.font_ix = (df.lsize << 2) + (df.styles & BOLD) + (df.styles & ITALIC);
}

/* ==================================================== ======== ======= */

UFontDesc::UFontDesc(const UFont& f) {
  set(f);
}

void UFontDesc::set(const class UFont& f) {
  family  = f.family;
  styles  = f.onStyles;
  lsize   = f.lsize;
  font_ix = (lsize << 2) + (styles & UFont::BOLD) + (styles & UFont::ITALIC);
}

void UFontDesc::set(const class UFont& f, int delta_scale) {
  family  = f.family;
  styles  = f.onStyles;
  lsize   = f.lsize + delta_scale;

  if (lsize < UFont::MIN_LSIZE) lsize = UFont::MIN_LSIZE;
  else if (lsize > UFont::MAX_LSIZE) lsize = UFont::MAX_LSIZE;
 
  font_ix = (lsize << 2) + (styles & UFont::BOLD) + (styles & UFont::ITALIC);
}

void UFontDesc::merge(const class UFont& f, int delta_scale) {
  if (f.family != &UFontFamily::any)  family = f.family;
  if (f.lsize > 0) lsize = f.lsize;
  lsize += delta_scale;

  if (lsize < UFont::MIN_LSIZE) lsize = UFont::MIN_LSIZE;
  else if (lsize > UFont::MAX_LSIZE) lsize = UFont::MAX_LSIZE;

  styles = styles | f.onStyles;     // combine styles
  font_ix = (lsize << 2) + (styles & UFont::BOLD) + (styles & UFont::ITALIC);
}

void UFontDesc::incrScale(int delta_scale) {
  lsize += delta_scale;

  if (lsize < 1) lsize = 1;
  else if (lsize > UFont::MAX_LSIZE) lsize = UFont::MAX_LSIZE;

  font_ix = (lsize << 2) + (styles & UFont::BOLD) + (styles & UFont::ITALIC);
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
