/* ==================================================== ======== ======= *
 *
 *  uuscrollbar.cpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuscrollbar.cpp	ubit:03.06.03"
#include <ubrick.hpp>
#include <ucond.hpp>
#include <ucall.hpp>
#include <ucolor.hpp>
#include <usymbol.hpp>
#include <uborder.hpp>
#include <uevent.hpp>
#include <ubox.hpp>
#include <uboxImpl.hpp>
#include <uscrollbar.hpp>
#include <upane.hpp>
#include <ustyle.hpp>
#include <ugraph.hpp>
#include <uview.hpp>
#include <uviewImpl.hpp>
#include <ugadgets.hpp>
#include <ucall.hpp>
#include <update.hpp>

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */
// LessScroller

UStyle *UScrollbar::LessScroller::style;

const UStyle& UScrollbar::LessScroller::getStyle(UContext*) const{
  static UGroup *v_content, *h_content;

  if (!style) {
    style = new UStyle(UBox::makeStyle());  

    style->orient = UOrient::INHERIT;
    style->local.halign = UHalign::CENTER;
    style->local.valign = UValign::CENTER;
    //style->local.border = &UBorder::shadowIn;
    style->local.padding.set(0,0);

    v_content = &ugroup(USymbol::up);
    h_content = &ugroup(USymbol::left);
  }

  if (isBmode(UMode::IS_VERTICAL)) {
    style->local.content = v_content;
  }
  else {
    style->local.content = h_content;
  }
  return *style;
}

UScrollbar::LessScroller::LessScroller(const UArgs& a) : UButton(a) {}

/* ==================================================== ======== ======= */
/* ==================================================== ======== ======= */
// MoreScroller

UStyle *UScrollbar::MoreScroller::style;

const UStyle& UScrollbar::MoreScroller::getStyle(UContext*) const{
  static UGroup *v_content, *h_content;

  if (!style) {
    style = new UStyle(UBox::makeStyle());  

    style->orient = UOrient::INHERIT;
    style->local.halign = UHalign::CENTER;
    style->local.valign = UValign::CENTER;
    //style->local.border = &UBorder::shadowIn;
    style->local.padding.set(0,0);

    v_content = &ugroup(USymbol::down);
    h_content = &ugroup(USymbol::right);
  }

  if (isBmode(UMode::IS_VERTICAL)) {
    style->local.content = v_content;
  }
  else {
    style->local.content = h_content;
  }
  return *style;
}

UScrollbar::MoreScroller::MoreScroller(const UArgs& a) : UButton(a) {}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

class UScrollbarStyle : public UStyle {
public:
  class UBgcolor *idle_bgcolor, *entered_bgcolor, *pressed_bgcolor;
  class UBgcolor *slider_bgcolor, *rail_bgcolor;
  float slider_alpha, rail_alpha;
  class UWidth   *slider_width;
  class UHeight  *slider_height;
  class UBorder  *slider_border, *rail_border;
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

UScrollbarStyle *UScrollbar::style;

const UStyle& UScrollbar::getStyle(UContext*) const {
  if (!style) {
    style = new UScrollbarStyle();
    style->viewStyle = &UScrollbarView::style;
    // pour eviter updates inutiles (cf calcul de layoutView)
    style->local.height = UHeight::KEEP_SIZE;
    style->local.width  = UWidth::KEEP_SIZE;
    style->local.hspacing = 0;
    style->local.vspacing = 0;
  }

  static UAlpha alpha03(0.35), alpha05(0.55), alpha10(1.0);

  if (isTransparent()) {
    //style->local.alpha = 0.4;
    style->idle_bgcolor    = &UBgcolor::none;
    //style->entered_bgcolor = &UBgcolor::orange;
    style->entered_bgcolor = &UBgcolor::blue;
    style->pressed_bgcolor = &UBgcolor::red;
    style->local.border    = &UBorder::none;

    //style->rail_bgcolor    = &UBgcolor::yellow;
    style->rail_bgcolor    = &UBgcolor::darkgrey;
    style->rail_alpha      = 0.35;
    style->rail_border     = &UBorder::none;

    //style->slider_bgcolor  = &UBgcolor::orange;
    style->slider_bgcolor  = &UBgcolor::blue;
    style->slider_alpha    = 0.55;
    style->slider_border   = &UBorder::shadowOut;
  } 
  else {
    style->local.alpha = 1.0;
    style->idle_bgcolor    = &UBgcolor::grey;
    style->entered_bgcolor = &UBgcolor::lightgrey;
    style->pressed_bgcolor = &UBgcolor::lightgrey;
    //style->local.border = &UBorder::shadowIn;  // &UBorder::etchedIn;

    style->rail_bgcolor    = &UBgcolor::darkgrey;  // pas assez sombre
    style->rail_alpha      = 1.0;
    style->rail_border     = &UBorder::shadowIn;

    style->slider_bgcolor  = &UBgcolor::darkgrey; // grey
    style->slider_alpha    = 1.0;
    style->slider_border   = &UBorder::shadowOut;
  }

  static UHeight height35(30), height10(10);
  static UWidth  width35(30),  width10(10);

  if (isVertical()) {
    style->orient = UOrient::VERTICAL;
    style->local.halign = UHalign::FLEX;
    style->local.valign = UValign::TOP;

    style->slider_height = &height35;
    style->slider_width  = &width10;
  }
  else {
    style->orient = UOrient::HORIZONTAL;
    style->local.halign = UHalign::LEFT;
    style->local.valign = UValign::FLEX;

    style->slider_width  = &width35;
    style->slider_height = &height10;
  }

  return *style;
}

/* ==================================================== ======== ======= */

UBox& UScrollbar::createRail(const UArgs& a) {
  UBox* btn = new UBox();
  btn->setCmodes(UMode::CAN_ARM | UMode::ENTER_HIGHLIGHT, false);
  btn->setCmodes(UMode::HAS_CLOSE_MENU_MODE, true);
  btn->setCmodes(UMode::CLOSE_MENU_MODE, false);
  return *btn;
}

UBox& UScrollbar::createSlider(const UArgs& a) {
  UBox* btn = new UButton();
  btn->setCmodes(UMode::CAN_ARM | UMode::ENTER_HIGHLIGHT, false);
  btn->setCmodes(UMode::HAS_CLOSE_MENU_MODE, true);
  btn->setCmodes(UMode::CLOSE_MENU_MODE, false);
  return *btn;
}

UBox& UScrollbar::createLessScroller(const UArgs& a) {
  UBox* btn = new UScrollbar::LessScroller(a);
  btn->setCmodes(UMode::HAS_CLOSE_MENU_MODE, true);
  btn->setCmodes(UMode::CLOSE_MENU_MODE, false);
  return *btn;
}

UBox& UScrollbar::createMoreScroller(const UArgs& a) {
  UBox* btn = new UScrollbar::MoreScroller(a);
  btn->setCmodes(UMode::HAS_CLOSE_MENU_MODE, true);
  btn->setCmodes(UMode::CLOSE_MENU_MODE, false);
  return *btn;
}

/* ==================================================== ======== ======= */

UScrollbar& uscrollbar(const UOrient &o, const UArgs& a) {
  return *(new UScrollbar(o, a));
}
UScrollbar& uhscrollbar(const UArgs& a) {
  return *(new UScrollbar(UOrient::horizontal, a));
}
UScrollbar& uvscrollbar(const UArgs& a) {
  return *(new UScrollbar(UOrient::vertical, a));
}

UScrollbar::UScrollbar(const UArgs& a) 
  : UBox(a), orient(UOrient::vertical), pvalue(new UFloat(0.0)) { 
  constructs(); 
}

UScrollbar::UScrollbar(const UOrient &_orient, const UArgs& a) 
  : UBox(a), orient(_orient), pvalue(new UFloat(0.0)) { 
  constructs(); 
}

UScrollbar::UScrollbar(UFloat& _f, const UOrient &_orient, const UArgs& a) 
  : UBox(a), orient(_orient), pvalue(_f) {
  constructs(); 
}

void UScrollbar::constructs() {

  scroller_increment = 0.5;
  backgd_increment   = 10.0;
  // pvalue init. in constructors
  ppane       = null;

  pless_btn   = createLessScroller();
  pmore_btn   = createMoreScroller();
  pslider_btn = createSlider(); 
  pslider_box = new UBox();
  rail_border = new UBorder();

  pressed = entered = false;
  with_rail   = true;
  transparent = false;
  enter_feedback = drag_feedback = false;
  incremental_mode = true;
  delta_mouse = 0;

  // depend de l'orient qui ne peut plus changer...
  getStyle(null); // inits var 'style'

  bgcolor.set(*style->idle_bgcolor);

  setTransparent(false);

  pslider_btn->addlist
    (
     slider_pos 
     + style->slider_width + style->slider_height
     + slider_bgcolor + slider_alpha
     + UOn::mdrag    / ucall(this, &UScrollbar::dragSlider)
     + UOn::mpress   / ucall(this, &UScrollbar::pressSlider)
     + UOn::mrelease / ucall(this, &UScrollbar::releaseSlider)
     + UOn::enter / ucall(this, &UScrollbar::enterSlider)
     + UOn::leave / ucall(this, &UScrollbar::leaveSlider)
     );

  pless_btn->add(UOn::mpress / ucall(this, -1, &UScrollbar::pressScroller));
  pmore_btn->add(UOn::mpress / ucall(this, +1, &UScrollbar::pressScroller));

  if (with_rail) {
    UBox* railbox = &ubox // createRail
      (
       UMode::ignoreEvents
       + rail_bgcolor + rail_alpha + *rail_border  
       );

    if (isHorizontal()) {
      railbox->add(uheight(6));
      pslider_box->addlist(uhflex() + uvcenter() + uhmargin(4) + railbox);
    }
    else {
      railbox->add(uwidth(6));
      pslider_box->addlist(uvflex() + uhcenter() + uvmargin(4) + railbox);
    }
  }

  pslider_box->addlist
    (
     UOn::mpress  / ucall(this, &UScrollbar::pressSliderBox)
     + UOn::mdrag / ucall(this, &UScrollbar::dragSliderBox)
     + UOn::mrelease / ucall(this, &UScrollbar::releaseSliderBox)
     + UOn::enter / ucall(this, &UScrollbar::enterSliderBox)
     + UOn::leave / ucall(this, &UScrollbar::leaveSliderBox)
     + *pslider_btn
     );

  this->addAttr(orient);

  if (isHorizontal())
    this->addlist(uhflex() + uvflex() + *pslider_box
		  + uright() + *pless_btn + *pmore_btn);
  else
    this->addlist(uhflex() + uvflex() + *pslider_box 
		  + ubottom() + *pless_btn + *pmore_btn);
}

void UScrollbar::setTransparent(bool state) {
  transparent = state;
  getStyle(null); // inits var 'style'

  slider_bgcolor = *style->slider_bgcolor;
  slider_alpha.set(style->slider_alpha, true);

  rail_bgcolor = *style->rail_bgcolor;
  rail_alpha.set(style->rail_alpha, true);
  *rail_border = *style->rail_border;
}

void UScrollbar::setUpdateOnDrag(bool state) {
  incremental_mode = state;
}

/*
void UScrollbar::setModes(int smodes) {
  if (smodes & STANDARD)      {with_rail = true; transparent = false;}
  if (smodes & TRANSP)        {with_rail = true; transparent = true;}
  if (smodes & WITH_RAIL)     {with_rail = true;}
  if (smodes & WITH_FEEDBACK) {enter_feedback = drag_feedback = true;}
  //if (smodes & COMPACT_EVENTS) ...;
}
*/

UScrollbar::~UScrollbar() {
  if (ppane) {
    if (this == ppane->getVScrollbar()) ppane->setVScrollbarImpl(null);
    if (this == ppane->getHScrollbar()) ppane->setHScrollbarImpl(null);
    ppane = null;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UScrollbar::setPane(class UPane *p) {ppane = p;}

bool UScrollbar::isHorizontal() const 
{return orient.get() == UOrient::HORIZONTAL;}

bool UScrollbar::isVertical() const 
{return orient.get() == UOrient::VERTICAL;}

bool UScrollbar::isTransparent() const 
{return transparent;}


/* ==================================================== ======== ======= */

float UScrollbar::getScrollerIncrement() const 
{return scroller_increment;}

void  UScrollbar::setScrollerIncrement(float percent) 
{scroller_increment = percent;}

float UScrollbar::getBackgroundIncrement() const 
{return backgd_increment;}

void  UScrollbar::setBackgroundIncrement(float percent) 
{backgd_increment = percent;}

float UScrollbar::getValue() const 
{return pvalue->getValue();}

void UScrollbar::setValue(const UFloat& _percent) {
  setValue(_percent.getValue());
}

void UScrollbar::setValue(float newval, bool update_pane) {
  if (ppane && update_pane) {
    // if a pane is attached to this scrollbar, recompute xscroll, yscroll
    // before moving the scrollbar
    if (orient.get() == UOrient::VERTICAL)  ppane->setYScroll(newval);
    else ppane->setXScroll(newval);
  }
  else setValueImpl(newval, true);
}

/* ==================================================== ======== ======= */

void UScrollbar::setValueImpl(float newval, bool upd) {
  if (newval < 0) newval = 0;
  else if (newval > 100) newval = 100.0;

  //if (newval == value) return;   //!Faux: marche pas quand ajout
  *pvalue = newval;

  // move the scrollbar, updates display, calls callbacks
  if (upd) {

    //if (transparent) {
      // reaffichage complet du scrollbar qui est transparent
      // le doublebuf est necessaire pour eviter flicking (en fait il devrait
      // etre fait plus tot !!
      UUpdate upd_dbl = UUpdate::all;

      if (transparent 
	  // serait inutile dans ce cas si on reaffichait pas l'autre scrollbar
	  || with_rail)
	upd_dbl.paintDoubleBuffered();
      update(upd_dbl);
      //}

      /*
    else {
      // MOVE au lieu d'un reaffichage complet du scrollbar,
      // plus economique en terme d'affichage, mais necessite qq precautions
      // (en particulier d'etre sur que le scrollbar ne bouge pas
      // entre-temps via une operation indirecte de reaffichage)

      for (ULinkLink *ll = parents.first(); ll; ll = ll->next()) {
	UBoxLink *link = static_cast<UBoxLink*>(ll->link()); // par construction

	for (UView* view = link->views(); view != null; view = view->getNext()) {
	  UScrollbarView *sbar_view = dynamic_cast<UScrollbarView*>(view);
	  // NB: UPDATE fait par cette fct (via un MOVE, d'ou efficacite)
	  if (sbar_view) sbar_view->setScroll(value);
	}
      }
    }
    */

    // call callbacks
    UEvent e(UEvent::change,  null, null, NULL);
    e.setSource(this);
    fire(e, UOn::change);
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

float UScrollbar::getPercent(UEvent& e, 
			     UView *slider_btn_view, UView *slider_box_view,
			     u_dim _delta_mouse) {
  if (isVertical()) {
    int ipos = e.getYwin() - slider_box_view->getYwin() - _delta_mouse;
    int remaining_h = slider_box_view->getHeight() - slider_btn_view->getHeight();
    return ((float) ipos) / remaining_h * 100.0;
  }
  else {
    int ipos = e.getXwin() - slider_box_view->getXwin() - _delta_mouse;
    int remaining_w = slider_box_view->getWidth() - slider_btn_view->getWidth();
    return ((float) ipos) / remaining_w * 100.0;
 }
}

void UScrollbar::pressSlider(UEvent& e) {
  pressed = true;
  if (drag_feedback) bgcolor.set(*style->pressed_bgcolor);

  // delta_mouse : decalage pointe par rapport a slider
  if (isVertical()) delta_mouse = e.getY();
  else delta_mouse = e.getX();
}
 
void UScrollbar::dragSlider(UEvent& e) {  
  UView *slider_btn_view = e.getView();
  UView *slider_box_view = slider_btn_view->getParentView();
  float newval = getPercent(e, slider_btn_view, slider_box_view, delta_mouse);

  // ne rien faire si on sort du range (cad si la souris sort du scrollbar)
  // chinat: o n'arrive pas a mettre le scrollbar a 0 ou a 100
  // if (newval >= 0 && newval <= 100)  setValue(newval);
  setValue(newval, incremental_mode);
}

void UScrollbar::releaseSlider(UEvent& e) {
  // dragSlider(e); plutot penible car bouge le slider qunad on clique
  // dessus sans modifier sa position (a cause de la maniere dont sont
  // calculees les positions)
  // MAIS necessaire dans le cas non incremental
  
  if (!incremental_mode) {
    UView *slider_btn_view = e.getView();
    UView *slider_box_view = slider_btn_view->getParentView();
    float newval = getPercent(e, slider_btn_view, slider_box_view, delta_mouse);
    setValue(newval, true);
  }
  
  pressed = false;
  if (drag_feedback)
    bgcolor.set(entered ? *style->entered_bgcolor : *style->idle_bgcolor);

  //effet: le slider se synchronise avec la pos ou on a lache' la souris
  // meme si on est sorti de la fenetre (car dans ce cas, a moisn qu'il
  // n'y ait un xgrab, le scrollbar ne suit plus la souris)

}

void UScrollbar::enterSlider(UEvent& e) {
  entered = true;
  if (enter_feedback && !pressed) bgcolor.set(*style->entered_bgcolor);
}

void UScrollbar::leaveSlider(UEvent& e) {
  entered = false;
  if (enter_feedback && !pressed) bgcolor.set(*style->idle_bgcolor);
}
 
/* ==================================================== ======== ======= */

void UScrollbar::enterSliderBox(UEvent& e) {
  entered = true;
  if (enter_feedback && !pressed) 
    bgcolor.set(*style->entered_bgcolor);
}

void UScrollbar::leaveSliderBox(UEvent& e) {
  entered = false;
  if (enter_feedback && !pressed) bgcolor.set(*style->idle_bgcolor);
}

void UScrollbar::releaseSliderBox(UEvent& e) {
  pressed = false;
  if (drag_feedback) 
    bgcolor.set(entered ? *style->entered_bgcolor : *style->idle_bgcolor);
}

void UScrollbar::dragSliderBox(UEvent& e) {
  UView *slider_box_view = e.getView();
  UView *slider_btn_view = pslider_btn->getFirstViewInside(slider_box_view);
  float newval = getPercent(e, slider_btn_view, slider_box_view, delta_mouse);
  setValue(newval, incremental_mode);
}

void UScrollbar::pressSliderBox(UEvent& e) {
  pressed = true;
  if (drag_feedback) bgcolor.set(*style->pressed_bgcolor);

  UView *slider_box_view = e.getView();
  UView *slider_btn_view = pslider_btn->getFirstViewInside(slider_box_view);
  float value = pvalue->getValue();

  if (e.getButtons() == UEvent::MButton1) {
    float pressed_val = getPercent(e, slider_btn_view, slider_box_view, 0);

    if (pressed_val > value) {
      if (value + backgd_increment > pressed_val)
        setValue(pressed_val, true);
      else setValue(value + backgd_increment, true);
    }
    else {
      if (value - backgd_increment < pressed_val)
        setValue(pressed_val, true);
      else setValue(value - backgd_increment, true);
    }
  }
  else if (e.getButtons() == UEvent::MButton2) {
    setValue(getPercent(e, slider_btn_view, slider_box_view, 0), true);
  }
}

// the more or less Scroller was pressed
void UScrollbar::pressScroller(UEvent&, int dir) {
  // faire en sorte qu'on arrive toujours au bords en cliquant sur les
  // more & less buttons meme si la valeut tombe pas juste et sort du 
  // range 0,100
  float newval = pvalue->getValue() + scroller_increment * dir;
  if (newval < 0) newval = 0;
  else if (newval > 100) newval = 100.0;
  setValue(newval, true);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// UScrollbarView
/* ==================================================== ======== ======= */

UViewStyle UScrollbarView::style(&UScrollbarView::makeView, UMode::UCONST);

UScrollbarView::UScrollbarView(UBoxLink *box_link, UView* par_view, 
			       UWinGraph *wgraph)
  : UView(box_link, par_view, wgraph) 
{
  //scrollmin = scrollmax = 0;
}

// "static" constructor used by UViewStyle to make a new view
UView* UScrollbarView::makeView(UBoxLink*bl, UView* parview, UWinGraph *wgraph) {
  return new UScrollbarView(bl, parview, wgraph);
}

// Attention: cette fonction fait un update implicite!
// (-> ne pas l'appler dans doLayout ou doUpdate!) 

void UScrollbarView::setScroll(float _value) {
  UScrollbar *scr = (UScrollbar*)getBox();
  UView *slider_box_view = scr->pslider_box->getFirstViewInside(this);
  UView *slider_btn_view = scr->pslider_btn->getFirstViewInside(slider_box_view);

  //float val = scrollmin + _value * (scrollmax - scrollmin) / 100.0;

  if (scr->isVertical()) {
    int remaining_h = slider_box_view->getHeight() - slider_btn_view->getHeight();
    float val = _value * remaining_h / 100.0;
    if (val >= 0.0/* && val <= 100.0*/)
      scr->slider_pos.setY(int(val));   // NB: fait update
  }
  else {
    int remaining_w = slider_box_view->getWidth() - slider_btn_view->getWidth();
    float val = _value * remaining_w / 100.0;
    if (val >= 0.0/* && val <= 100.0*/)
    scr->slider_pos.setX(int(val));   // NB: fait update
  }
}

bool UScrollbarView::doLayout(UContext &parp, UViewLayout &vl) {
  return UView::doLayout(parp, vl); 
}

void UScrollbarView::doUpdate(UContext &parp, URegion r, URegion clip, 
			      UViewUpdate &vup) {
  // l'intersection de clip et de r est affectee dans clip
  if (clip.setInter(r) == 0) return;

  UScrollbar *scr = (UScrollbar*)getBox();

  // il faut appeler DEUX fois doUpdate:   !!!!!!
  // -- la premiere pour calculer correctement la taille du UScrollbarView
  //    (car doLayout ne suffit pas)
  // -- la seconde vraiment pour tout afficher correctement 
  //    a la bonne position (c'est a dire APRES avoir bouge le slider)

  /// if (scr->isTransparent()) {
    // ne rien reafficher juste mettre a jour data
    UViewUpdate updata(UViewUpdate::UPDATE_DATA);
    UView::doUpdate(parp, r, clip, updata);
    /// }
  // il faut reafficher dans ce cas  (for some reason...)
  /// else UView::doUpdate(parp, r, clip, vup);

  UView *slider_box_view = scr->pslider_box->getFirstViewInside(this);
  UView *slider_btn_view = scr->pslider_btn->getFirstViewInside(slider_box_view);

  if (scr->isVertical()) {
    int remaining_h = slider_box_view->getHeight() - slider_btn_view->getHeight();

    // repositionner le slider
    // ATT: set(,,false) ==> no update; surtout ne pas appeler move car 
    // ca rappellerait cette meme fonction => boucle infinie
    // NB: centrer horizontalement
    scr->slider_pos.set((getWidth() - slider_btn_view->getWidth()) / 2, 
			int(scr->getValue() * remaining_h / 100.0),
			false);
  }

  else {  // horizontal
    int remaining_w = slider_box_view->getWidth() - slider_btn_view->getWidth();
    scr->slider_pos.set(int(scr->getValue() * remaining_w / 100.0),
			(getHeight() - slider_btn_view->getHeight()) / 2,
			false);
  }

  // voir remarque ci-dessus sur le fait qu'il faut appeler 2 FOIS doUpdate
  UView::doUpdate(parp, r, clip, vup);
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
